import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Check, ChevronsUpDown, Search, Shirt, X, BookOpen, Package } from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import {
  Command,
  CommandEmpty,
  CommandGroup,
  CommandInput,
  CommandItem,
  CommandList,
} from "@/components/ui/command";
import {
  Popover,
  PopoverContent,
  PopoverTrigger,
} from "@/components/ui/popover";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface Fabric {
  id: number;
  name: string;
  internal_code: string;
  primary_image_url: string | null;
  group_name: string;
  source?: 'dictionary' | 'warehouse';
}

interface FabricSelectorProps {
  orderId: string;
  itemId: string;
  orderNumber: string;
  currentValue?: string;
  onSelect?: (fabric: Fabric) => void;
  componentOnly?: boolean;
}

export function FabricSelector({ orderId, itemId, orderNumber, currentValue, onSelect, componentOnly = false }: FabricSelectorProps) {
  const [open, setOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const { toast } = useToast();

  const { data: fabrics = [], isLoading } = useQuery<Fabric[]>({
    queryKey: ["/api/warehouse/fabrics", searchQuery],
    queryFn: async () => {
      const params = new URLSearchParams();
      if (searchQuery) params.set("search", searchQuery);
      const res = await fetch(`/api/warehouse/fabrics?${params.toString()}`);
      if (!res.ok) throw new Error("Failed to fetch fabrics");
      return res.json();
    },
  });

  const updateMutation = useMutation({
    mutationFn: async (fabric: Fabric) => {
      return apiRequest(
        "PATCH",
        `/api/orders/${orderId}/items/${itemId}/upholstery`,
        { upholsteryType: fabric.name }
      );
    },
    onSuccess: (_, fabric) => {
      queryClient.invalidateQueries({ queryKey: [`/api/orders/by-number/${orderNumber}`] });
      toast({
        title: "Tapicerka zaktualizowana",
        description: `Wybrano: ${fabric.name}`,
      });
      onSelect?.(fabric);
      setOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować tapicerki",
        variant: "destructive",
      });
    },
  });

  const selectedFabric = fabrics.find(f => f.name === currentValue);

  return (
    <Popover open={open} onOpenChange={setOpen}>
      <PopoverTrigger asChild>
        <Button
          variant="outline"
          role="combobox"
          aria-expanded={open}
          className="w-full justify-between h-auto min-h-9 py-1.5"
          data-testid={`button-fabric-selector-${itemId}`}
        >
          <div className="flex items-center gap-2 min-w-0">
            {selectedFabric?.primary_image_url ? (
              <img
                src={selectedFabric.primary_image_url}
                alt={selectedFabric.name}
                className="w-6 h-6 rounded object-cover flex-shrink-0"
              />
            ) : (
              <Shirt className="w-4 h-4 text-muted-foreground flex-shrink-0" />
            )}
            <span className="truncate text-sm">
              {currentValue || "Brak"}
            </span>
          </div>
          <ChevronsUpDown className="ml-2 h-4 w-4 shrink-0 opacity-50" />
        </Button>
      </PopoverTrigger>
      <PopoverContent className="w-80 p-0" align="start">
        <Command shouldFilter={false}>
          <CommandInput
            placeholder="Szukaj tkaniny..."
            value={searchQuery}
            onValueChange={setSearchQuery}
            data-testid="input-fabric-search"
          />
          <CommandList>
            {isLoading ? (
              <div className="py-6 text-center text-sm text-muted-foreground">
                Ładowanie...
              </div>
            ) : fabrics.length === 0 ? (
              <CommandEmpty>Nie znaleziono tkanin.</CommandEmpty>
            ) : (
              <>
                <CommandGroup heading="Opcje">
                  <CommandItem
                    key="none"
                    value="brak"
                    onSelect={() => {
                      const noneFabric = { id: 0, name: '', internal_code: '', primary_image_url: null, group_name: '' };
                      if (componentOnly && onSelect) {
                        onSelect(noneFabric);
                        setOpen(false);
                        toast({
                          title: "Tapicerka usunięta",
                          description: "Produkt nie ma tapicerki",
                        });
                      } else {
                        updateMutation.mutate(noneFabric);
                      }
                    }}
                    className="flex items-center gap-3 py-2"
                    data-testid="fabric-option-none"
                  >
                    <div className="w-10 h-10 rounded-md overflow-hidden bg-muted flex-shrink-0 border flex items-center justify-center">
                      <X className="w-5 h-5 text-muted-foreground" />
                    </div>
                    <div className="flex-1 min-w-0">
                      <div className="font-medium text-sm">Brak</div>
                      <div className="text-xs text-muted-foreground">Produkt bez tapicerki</div>
                    </div>
                    <Check
                      className={cn(
                        "h-4 w-4 flex-shrink-0",
                        !currentValue ? "opacity-100" : "opacity-0"
                      )}
                    />
                  </CommandItem>
                </CommandGroup>
                {fabrics.filter(f => f.source === 'dictionary').length > 0 && (
                  <CommandGroup heading={<span className="flex items-center gap-1.5"><BookOpen className="w-3 h-3" />Słownik (ogólne typy)</span>}>
                    {fabrics.filter(f => f.source === 'dictionary').map((fabric) => (
                      <CommandItem
                        key={`dict-${fabric.id}`}
                        value={fabric.name}
                        onSelect={() => {
                          if (componentOnly && onSelect) {
                            onSelect(fabric);
                            setOpen(false);
                            toast({
                              title: "Tapicerka zaktualizowana",
                              description: `Wybrano: ${fabric.name}`,
                            });
                          } else {
                            updateMutation.mutate(fabric);
                          }
                        }}
                        className="flex items-center gap-3 py-2"
                        data-testid={`fabric-option-dict-${fabric.id}`}
                      >
                        <div className="w-10 h-10 rounded-md overflow-hidden bg-muted flex-shrink-0 border">
                          {fabric.primary_image_url ? (
                            <img
                              src={fabric.primary_image_url}
                              alt={fabric.name}
                              className="w-full h-full object-cover"
                            />
                          ) : (
                            <div className="w-full h-full flex items-center justify-center">
                              <Shirt className="w-4 h-4 text-muted-foreground" />
                            </div>
                          )}
                        </div>
                        <div className="flex-1 min-w-0">
                          <div className="font-medium text-sm truncate">{fabric.name}</div>
                          <div className="text-xs text-muted-foreground truncate">
                            {fabric.internal_code}
                          </div>
                        </div>
                        <Check
                          className={cn(
                            "h-4 w-4 flex-shrink-0",
                            currentValue === fabric.name ? "opacity-100" : "opacity-0"
                          )}
                        />
                      </CommandItem>
                    ))}
                  </CommandGroup>
                )}
                {fabrics.filter(f => f.source === 'warehouse').length > 0 && (
                  <CommandGroup heading={<span className="flex items-center gap-1.5"><Package className="w-3 h-3" />Magazyn (konkretne)</span>}>
                    {fabrics.filter(f => f.source === 'warehouse').map((fabric) => (
                      <CommandItem
                        key={`wh-${fabric.id}`}
                        value={fabric.name}
                        onSelect={() => {
                          if (componentOnly && onSelect) {
                            onSelect(fabric);
                            setOpen(false);
                            toast({
                              title: "Tapicerka zaktualizowana",
                              description: `Wybrano: ${fabric.name}`,
                            });
                          } else {
                            updateMutation.mutate(fabric);
                          }
                        }}
                        className="flex items-center gap-3 py-2"
                        data-testid={`fabric-option-wh-${fabric.id}`}
                      >
                        <div className="w-10 h-10 rounded-md overflow-hidden bg-muted flex-shrink-0 border">
                          {fabric.primary_image_url ? (
                            <img
                              src={fabric.primary_image_url}
                              alt={fabric.name}
                              className="w-full h-full object-cover"
                            />
                          ) : (
                            <div className="w-full h-full flex items-center justify-center">
                              <Shirt className="w-4 h-4 text-muted-foreground" />
                            </div>
                          )}
                        </div>
                        <div className="flex-1 min-w-0">
                          <div className="font-medium text-sm truncate">{fabric.name}</div>
                          <div className="text-xs text-muted-foreground truncate">
                            {fabric.group_name} • {fabric.internal_code}
                          </div>
                        </div>
                        <Check
                          className={cn(
                            "h-4 w-4 flex-shrink-0",
                            currentValue === fabric.name ? "opacity-100" : "opacity-0"
                          )}
                        />
                      </CommandItem>
                    ))}
                  </CommandGroup>
                )}
              </>
            )}
          </CommandList>
        </Command>
      </PopoverContent>
    </Popover>
  );
}
