import { pool } from './server/postgres';

async function checkProductionFlow() {
  console.log('\n🏭 === ANALIZA FLOW PRODUKCJI ZLP-0010 ===\n');
  
  // 1. Podsumowanie ZLP
  const ordersQuery = await pool.query(`
    SELECT 
      po.order_number,
      po.color_code,
      COUNT(DISTINCT pwo.id) as work_orders,
      COUNT(DISTINCT pobi.id) as bom_items,
      SUM(pobi.quantity) as total_qty
    FROM production.production_orders po
    LEFT JOIN production.production_work_orders pwo ON pwo.production_order_id = po.id
    LEFT JOIN production.production_order_boms pob ON pob.production_order_id = po.id
    LEFT JOIN production.production_order_bom_items pobi ON pobi.production_order_bom_id = pob.id
    WHERE po.order_number LIKE 'ZLP-0010-%'
    GROUP BY po.id, po.order_number, po.color_code
    ORDER BY po.order_number
  `);
  
  console.log('📦 Wygenerowane ZLP:\n');
  ordersQuery.rows.forEach(row => {
    console.log(`  ${row.order_number}:`);
    console.log(`    - BOM Items: ${row.bom_items} rekordów (każdy qty=1)`);
    console.log(`    - Work Orders: ${row.work_orders} operacji`);
    console.log('');
  });
  
  // 2. Szczegóły routing resolution
  console.log('🔀 ROUTING RESOLUTION:\n');
  
  const routingQuery = await pool.query(`
    SELECT 
      po.order_number,
      rv.variant_code,
      rv.variant_name,
      pobi.required_operations,
      COUNT(*) as komponentow
    FROM production.production_order_bom_items pobi
    JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
    JOIN production.production_orders po ON po.id = pob.production_order_id
    LEFT JOIN production.production_routing_variants rv ON rv.id = pobi.routing_variant_id
    WHERE po.order_number LIKE 'ZLP-0010-%'
    GROUP BY po.order_number, rv.variant_code, rv.variant_name, pobi.required_operations
    ORDER BY po.order_number, rv.variant_code
  `);
  
  let currentOrder = '';
  routingQuery.rows.forEach(row => {
    if (row.order_number !== currentOrder) {
      console.log(`\n  ${row.order_number}:`);
      currentOrder = row.order_number;
    }
    console.log(`    - Variant ${row.variant_code}: ${row.komponentow} komponentów → ${row.required_operations}`);
  });
  
  // 3. Work Orders szczegóły
  console.log('\n\n⚙️ WORK ORDERS (operacje do wykonania):\n');
  
  const woQuery = await pool.query(`
    SELECT 
      po.order_number,
      pwo.work_order_number,
      pwo.sequence,
      pwo.quantity_planned,
      pwo.status
    FROM production.production_work_orders pwo
    JOIN production.production_orders po ON po.id = pwo.production_order_id
    WHERE po.order_number LIKE 'ZLP-0010-%'
    ORDER BY po.order_number, pwo.sequence
  `);
  
  currentOrder = '';
  woQuery.rows.forEach(row => {
    if (row.order_number !== currentOrder) {
      console.log(`\n  ${row.order_number}:`);
      currentOrder = row.order_number;
    }
    console.log(`    ${row.sequence}. ${row.work_order_number}: ${row.quantity_planned} szt - ${row.status}`);
  });
  
  // 4. Przykład: BOM items dla SUROWA (tapicerka)
  console.log('\n\n🪑 TAPICERKA (ZLP-0010-SUROWA) - szczegóły:\n');
  
  const suryQuery = await pool.query(`
    SELECT 
      pobi.component_name,
      rv.variant_code,
      rv.variant_name,
      pobi.required_operations,
      pobi.source_furniture_reference as zamowienie,
      pobi.quantity
    FROM production.production_order_bom_items pobi
    JOIN production.production_order_boms pob ON pob.id = pobi.production_order_bom_id
    JOIN production.production_orders po ON po.id = pob.production_order_id
    LEFT JOIN production.production_routing_variants rv ON rv.id = pobi.routing_variant_id
    WHERE po.order_number = 'ZLP-0010-SUROWA'
    ORDER BY pobi.id
  `);
  
  suryQuery.rows.forEach(row => {
    console.log(`  ${row.component_name}:`);
    console.log(`    - Variant: ${row.variant_code} (${row.variant_name})`);
    console.log(`    - Operacje: ${row.required_operations}`);
    console.log(`    - Zamówienie: ${row.zamowienie}`);
    console.log(`    - Quantity: ${row.quantity}`);
    console.log('');
  });
  
  // 5. Buffer system integration check
  console.log('\n📦 BUFFER SYSTEM - konfiguracja operacji:\n');
  
  const bufferOpsQuery = await pool.query(`
    SELECT 
      pro.id,
      pro.name,
      pro.creates_buffer,
      pro.buffer_location_id,
      pl.name as location_name
    FROM production.production_routing_operations pro
    LEFT JOIN production.production_locations pl ON pl.id = pro.buffer_location_id
    WHERE pro.creates_buffer = true
    LIMIT 10
  `);
  
  if (bufferOpsQuery.rows.length > 0) {
    bufferOpsQuery.rows.forEach(row => {
      console.log(`  ✓ Operacja: "${row.name}" → tworzy bufor w lokalizacji: ${row.location_name || 'BRAK'}`);
    });
  } else {
    console.log('  ⚠️ Brak operacji z flagą creates_buffer=true');
    console.log('  💡 Aby włączyć system buforów, zaznacz operacje które tworzą półprodukty (np. "Cięcie HDF")');
  }
  
  await pool.end();
  console.log('\n✨ Analiza zakończona!\n');
}

checkProductionFlow().catch(err => {
  console.error('❌ Błąd:', err);
  process.exit(1);
});
