import { ChevronDown, ChevronUp, SortAsc, SortDesc } from "lucide-react";
import { Button } from "@/components/ui/button";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";

interface OrdersControlBarProps {
  viewMode: "flat" | "grouped";
  allExpanded: boolean;
  onExpandAll: () => void;
  onCollapseAll: () => void;
  pageSize: number;
  onPageSizeChange: (size: number) => void;
  sortBy: 'order_date' | 'order_number' | 'buyer_name' | 'total_amount' | 'product_sku';
  sortOrder: 'asc' | 'desc';
  onSortByChange: (sortBy: 'order_date' | 'order_number' | 'buyer_name' | 'total_amount' | 'product_sku') => void;
  onSortOrderToggle: () => void;
}

const sortByLabels: Record<string, string> = {
  order_date: 'Data zamówienia',
  order_number: 'Numer zamówienia',
  buyer_name: 'Nazwisko klienta',
  total_amount: 'Kwota zamówienia',
  product_sku: 'SKU produktu',
};

export function OrdersControlBar({
  viewMode,
  allExpanded,
  onExpandAll,
  onCollapseAll,
  pageSize,
  onPageSizeChange,
  sortBy,
  sortOrder,
  onSortByChange,
  onSortOrderToggle,
}: OrdersControlBarProps) {
  return (
    <div 
      className="flex-shrink-0 flex items-center gap-2 min-h-[25px]" 
      data-testid="orders-control-bar"
    >
      {viewMode === "grouped" && (
        <Button
          variant="outline"
          size="sm"
          onClick={allExpanded ? onCollapseAll : onExpandAll}
          className="h-[25px] px-2 text-xs"
          data-testid="button-toggle-expand-all"
        >
          {allExpanded ? (
            <>
              <ChevronUp className="h-3 w-3 mr-1" />
              Zwiń wszystko
            </>
          ) : (
            <>
              <ChevronDown className="h-3 w-3 mr-1" />
              Rozwiń wszystko
            </>
          )}
        </Button>
      )}

      <div className="flex items-center gap-1 text-xs text-muted-foreground">
        <span>Pokaż:</span>
        <Select
          value={pageSize.toString()}
          onValueChange={(value) => onPageSizeChange(parseInt(value))}
        >
          <SelectTrigger className="h-[25px] w-[70px] text-xs" data-testid="select-page-size">
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="25">25</SelectItem>
            <SelectItem value="50">50</SelectItem>
            <SelectItem value="100">100</SelectItem>
            <SelectItem value="200">200</SelectItem>
          </SelectContent>
        </Select>
      </div>

      <div className="flex items-center gap-1 text-xs text-muted-foreground">
        <span>Sortuj:</span>
        <Select
          value={sortBy}
          onValueChange={(value) => onSortByChange(value as typeof sortBy)}
        >
          <SelectTrigger className="h-[25px] w-[160px] text-xs" data-testid="select-sort-by">
            <SelectValue />
          </SelectTrigger>
          <SelectContent>
            {Object.entries(sortByLabels).map(([key, label]) => (
              <SelectItem key={key} value={key}>
                {label}
              </SelectItem>
            ))}
          </SelectContent>
        </Select>

        <Button
          variant="outline"
          size="sm"
          onClick={onSortOrderToggle}
          className="h-[25px] w-[25px] p-0"
          data-testid="button-toggle-sort-order"
        >
          {sortOrder === 'asc' ? (
            <SortAsc className="h-3 w-3" />
          ) : (
            <SortDesc className="h-3 w-3" />
          )}
        </Button>
      </div>
    </div>
  );
}
