import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Copy } from "lucide-react";
import { useToast } from "@/hooks/use-toast";

export function TemplateVariablesHelp() {
  const { toast } = useToast();

  const copyToClipboard = (text: string) => {
    navigator.clipboard.writeText(text);
    toast({ title: "Skopiowano do schowka", description: text });
  };

  const productVariables = [
    { name: "{{title}}", description: "Tytuł produktu" },
    { name: "{{sku}}", description: "SKU produktu" },
    { name: "{{short_description}}", description: "Krótki opis" },
    { name: "{{length}}", description: "Długość (cm)" },
    { name: "{{width}}", description: "Szerokość (cm)" },
    { name: "{{height}}", description: "Wysokość (cm)" },
    { name: "{{weight}}", description: "Waga (kg)" },
    { name: "{{color}}", description: "Kolor" },
    { name: "{{material}}", description: "Materiał" },
    { name: "{{base_price}}", description: "Cena bazowa" },
    { name: "{{currency}}", description: "Waluta (PLN, EUR, USD)" },
  ];

  const addonVariables = [
    { name: "{{#addons}}...{{/addons}}", description: "Pętla po dodatkach" },
    { name: "{{@index}}", description: "Numer indeksu (0, 1, 2...)" },
    { name: "{{@first}}", description: "true dla pierwszego elementu" },
    { name: "{{@last}}", description: "true dla ostatniego elementu" },
    { name: "{{name}}", description: "Nazwa dodatku" },
    { name: "{{code}}", description: "Kod dodatku" },
    { name: "{{description}}", description: "Opis dodatku" },
    { name: "{{addon_type}}", description: "Typ (fabric/board/certificate...)" },
  ];

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base">Dostępne zmienne</CardTitle>
        <CardDescription>
          Kliknij aby skopiować zmienną do schowka
        </CardDescription>
      </CardHeader>
      <CardContent>
        <ScrollArea className="h-[400px] w-full pr-4">
          <div className="space-y-4">
            {/* Zmienne produktu */}
            <div>
              <h4 className="font-medium text-sm mb-2">Dane produktu</h4>
              <div className="space-y-2">
                {productVariables.map((variable) => (
                  <div
                    key={variable.name}
                    className="flex items-center justify-between p-2 rounded-md hover:bg-accent cursor-pointer transition-colors"
                    onClick={() => copyToClipboard(variable.name)}
                    data-testid={`variable-${variable.name.replace(/[{}]/g, "")}`}
                  >
                    <div className="flex-1">
                      <code className="text-xs font-mono bg-muted px-2 py-1 rounded">
                        {variable.name}
                      </code>
                      <p className="text-xs text-muted-foreground mt-1">
                        {variable.description}
                      </p>
                    </div>
                    <Copy className="h-3 w-3 text-muted-foreground ml-2" />
                  </div>
                ))}
              </div>
            </div>

            {/* Zmienne pętli dodatków */}
            <div>
              <h4 className="font-medium text-sm mb-2">Pętla dodatków</h4>
              <div className="space-y-2">
                {addonVariables.map((variable) => (
                  <div
                    key={variable.name}
                    className="flex items-center justify-between p-2 rounded-md hover:bg-accent cursor-pointer transition-colors"
                    onClick={() => copyToClipboard(variable.name)}
                    data-testid={`variable-addon-${variable.name.replace(/[{}#@/]/g, "")}`}
                  >
                    <div className="flex-1">
                      <code className="text-xs font-mono bg-muted px-2 py-1 rounded">
                        {variable.name}
                      </code>
                      <p className="text-xs text-muted-foreground mt-1">
                        {variable.description}
                      </p>
                    </div>
                    <Copy className="h-3 w-3 text-muted-foreground ml-2" />
                  </div>
                ))}
              </div>
            </div>

            {/* Przykład użycia */}
            <div>
              <h4 className="font-medium text-sm mb-2">Przykład użycia w tabeli</h4>
              <div className="bg-muted p-3 rounded-md">
                <pre className="text-xs font-mono whitespace-pre-wrap">
{`<table>
  <tr>
    <th>Parametr</th>
    <th>Wartość</th>
  </tr>
  <tr>
    <td>Długość</td>
    <td>{{length}} cm</td>
  </tr>
  <tr>
    <td>Szerokość</td>
    <td>{{width}} cm</td>
  </tr>
  <tr>
    <td>Wysokość</td>
    <td>{{height}} cm</td>
  </tr>
  <tr>
    <td>Kolor</td>
    <td>{{color}}</td>
  </tr>
  <tr>
    <td>Materiał</td>
    <td>{{material}}</td>
  </tr>
</table>

<h3>Dostępne dodatki:</h3>
{{#addons}}
  <p>{{@index}}. {{name}} ({{code}})</p>
  {{#if description}}
    <p>{{description}}</p>
  {{/if}}
{{/addons}}`}
                </pre>
              </div>
            </div>
          </div>
        </ScrollArea>
      </CardContent>
    </Card>
  );
}
