import { useState } from "react";
import { useMutation } from "@tanstack/react-query";
import { Trash2, RefreshCw, ExternalLink, Package } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";

interface ManageLinksDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  links: Array<{
    id: number;
    platform: string;
    externalId: string;
    marketplaceSku?: string;
    marketplaceName?: string;
    productId?: number;
  }>;
  catalogProductId?: number;
  marketplaceSku?: string;
  onAddNew?: () => void;
}

export function ManageLinksDialog({
  open,
  onOpenChange,
  links,
  catalogProductId,
  marketplaceSku,
  onAddNew,
}: ManageLinksDialogProps) {
  const { toast } = useToast();
  const [confirmDelete, setConfirmDelete] = useState<number | null>(null);
  const [deletingLinkId, setDeletingLinkId] = useState<number | null>(null);

  // Mutation to delete link
  const deleteLinkMutation = useMutation({
    mutationFn: async (linkId: number) => {
      return apiRequest("DELETE", `/api/catalog/products/platform-links/${linkId}`, {});
    },
    onSuccess: () => {
      // Invalidate queries for both catalog and marketplace
      if (catalogProductId) {
        queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${catalogProductId}`] });
      }
      if (marketplaceSku) {
        queryClient.invalidateQueries({ queryKey: [`/api/marketplace-products/${marketplaceSku}`] });
      }
      
      toast({
        title: "✅ Połączenie usunięte",
        description: "Produkt został odłączony od marketplace",
      });
      
      setConfirmDelete(null);
      setDeletingLinkId(null);
    },
    onError: (error: any) => {
      toast({
        title: "❌ Błąd usuwania",
        description: error.message || "Nie udało się usunąć połączenia",
        variant: "destructive",
      });
      setDeletingLinkId(null);
    },
  });

  const handleDelete = (linkId: number) => {
    setDeletingLinkId(linkId);
    deleteLinkMutation.mutate(linkId);
  };

  const handleChange = (linkId: number) => {
    // First, delete the existing link
    setDeletingLinkId(linkId);
    deleteLinkMutation.mutate(linkId, {
      onSuccess: () => {
        // After deletion, open the add dialog
        onOpenChange(false);
        setTimeout(() => {
          onAddNew?.();
        }, 100);
      },
    });
  };

  return (
    <>
      <Dialog open={open} onOpenChange={onOpenChange}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Zarządzaj połączeniami</DialogTitle>
            <DialogDescription>
              Przeglądaj, usuwaj lub zmieniaj połączenia z marketplace
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            {/* Existing links */}
            {links.length > 0 ? (
              <div className="space-y-2">
                <div className="text-sm font-medium">Aktualne połączenia ({links.length}):</div>
                {links.map((link) => (
                  <div
                    key={link.id}
                    className="flex items-center gap-3 p-3 border rounded-lg bg-muted/30"
                    data-testid={`link-item-${link.id}`}
                  >
                    <Badge variant="secondary" className="shrink-0">
                      {link.platform === 'allegro' ? '🔴 Allegro' : '🟢 Shoper'}
                    </Badge>
                    <div className="flex-1">
                      <div className="font-medium text-sm">
                        {link.marketplaceName || 'Produkt marketplace'}
                      </div>
                      <div className="text-xs text-muted-foreground">
                        SKU: {link.marketplaceSku || link.externalId}
                      </div>
                    </div>
                    <div className="flex items-center gap-1">
                      <Button
                        size="sm"
                        variant="ghost"
                        onClick={() => setConfirmDelete(link.id)}
                        disabled={deletingLinkId === link.id}
                        data-testid={`button-delete-${link.id}`}
                        title="Usuń połączenie"
                        className="h-8 px-2 text-destructive hover:text-destructive hover:bg-destructive/10"
                      >
                        <Trash2 className="h-4 w-4" />
                      </Button>
                      <Button
                        size="sm"
                        variant="ghost"
                        onClick={() => handleChange(link.id)}
                        disabled={deletingLinkId === link.id}
                        data-testid={`button-change-${link.id}`}
                        title="Zmień połączenie (usuń i dodaj nowe)"
                        className="h-8 px-2 text-primary hover:text-primary hover:bg-primary/10"
                      >
                        <RefreshCw className="h-4 w-4" />
                      </Button>
                    </div>
                  </div>
                ))}
              </div>
            ) : (
              <div className="text-center p-6 text-muted-foreground">
                <Package className="h-12 w-12 mx-auto mb-2 opacity-50" />
                <p>Brak połączeń</p>
                <p className="text-sm mt-2">Dodaj nowe połączenie klikając przycisk poniżej</p>
              </div>
            )}

            {/* Add new button */}
            {onAddNew && (
              <div className="border-t pt-4">
                <Button
                  onClick={() => {
                    onOpenChange(false);
                    setTimeout(() => {
                      onAddNew();
                    }, 100);
                  }}
                  variant="outline"
                  className="w-full"
                  data-testid="button-add-new-link"
                >
                  <ExternalLink className="h-4 w-4 mr-2" />
                  Dodaj nowe połączenie
                </Button>
              </div>
            )}

            {/* Close button */}
            <div className="flex justify-end border-t pt-4">
              <Button
                variant="outline"
                onClick={() => onOpenChange(false)}
                data-testid="button-close"
              >
                Zamknij
              </Button>
            </div>
          </div>
        </DialogContent>
      </Dialog>

      {/* Confirm delete dialog */}
      <AlertDialog open={confirmDelete !== null} onOpenChange={(open) => !open && setConfirmDelete(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno chcesz usunąć to połączenie?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta operacja jest nieodwracalna. Połączenie z produktem marketplace zostanie usunięte.
              Synchronizacja zamówień nie będzie już działać dla tego produktu.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => confirmDelete && handleDelete(confirmDelete)}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              data-testid="button-confirm-delete"
            >
              Usuń połączenie
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </>
  );
}
