import type { Pool, PoolClient } from "pg";
import dayjs from "dayjs";
import type { 
  ProductionPlanNameTemplate, 
  InsertProductionPlanNameTemplate 
} from "@shared/schema";

interface PatternToken {
  type: "literal" | "date" | "datetime" | "seq";
  value: string;
  format?: string;
  width?: number;
}

interface ParsedPattern {
  tokens: PatternToken[];
  hasSeq: boolean;
  hasDate: boolean;
}

export function parsePattern(pattern: string): ParsedPattern {
  const tokens: PatternToken[] = [];
  const validTokenRegex = /\{(date|datetime|seq)\|([^}]+)\}/g;
  const unknownTokenRegex = /\{([^}]*)\}/g;
  
  const unknownTokens: string[] = [];
  let match;
  
  while ((match = unknownTokenRegex.exec(pattern)) !== null) {
    const fullToken = match[0];
    if (!/^\{(date|datetime|seq)\|[^}]+\}$/.test(fullToken)) {
      unknownTokens.push(fullToken);
    }
  }
  
  if (unknownTokens.length > 0) {
    throw new Error(
      `Invalid token(s) in pattern: ${unknownTokens.join(", ")}. ` +
      `Valid formats: {date|FORMAT}, {datetime|FORMAT}, {seq|WIDTH}`
    );
  }
  
  const openBraces = (pattern.match(/\{/g) || []).length;
  const closeBraces = (pattern.match(/\}/g) || []).length;
  if (openBraces !== closeBraces) {
    throw new Error(`Unmatched braces in pattern: ${openBraces} opening, ${closeBraces} closing`);
  }

  let hasSeq = false;
  let hasDate = false;
  let lastIndex = 0;

  validTokenRegex.lastIndex = 0;
  
  while ((match = validTokenRegex.exec(pattern)) !== null) {
    if (match.index > lastIndex) {
      tokens.push({ type: "literal", value: pattern.slice(lastIndex, match.index) });
    }

    const tokenType = match[1];
    const tokenParam = match[2];

    if (tokenType === "seq") {
      hasSeq = true;
      const width = parseInt(tokenParam, 10);
      if (isNaN(width) || width < 1 || width > 10) {
        throw new Error(`Invalid sequence width: ${tokenParam}. Must be between 1 and 10.`);
      }
      tokens.push({ type: "seq", value: tokenParam, width });
    } else if (tokenType === "date") {
      hasDate = true;
      tokens.push({ type: "date", value: tokenParam, format: tokenParam });
    } else if (tokenType === "datetime") {
      hasDate = true;
      tokens.push({ type: "datetime", value: tokenParam, format: tokenParam });
    }

    lastIndex = validTokenRegex.lastIndex;
  }

  if (lastIndex < pattern.length) {
    tokens.push({ type: "literal", value: pattern.slice(lastIndex) });
  }

  if (tokens.length === 0) {
    throw new Error("Pattern must contain at least one literal or token");
  }

  return { tokens, hasSeq, hasDate };
}

export async function generatePlanName(
  {
    templateId,
    dateOverride,
  }: {
    templateId: number;
    dateOverride?: Date;
  },
  client: PoolClient
): Promise<{ generatedName: string; counter: number; tokens: PatternToken[] }> {
  const currentDate = dateOverride || new Date();
  const dateStr = dayjs(currentDate).format("YYYY-MM-DD");

  const templateQuery = `
    SELECT id, template_name, pattern, is_active 
    FROM production.plan_name_templates 
    WHERE id = $1
  `;
  const templateResult = await client.query(templateQuery, [templateId]);

  if (templateResult.rows.length === 0) {
    throw new Error(`Template with id ${templateId} not found`);
  }

  const template = templateResult.rows[0];

  if (!template.is_active) {
    throw new Error(`Template "${template.template_name}" is not active`);
  }

  const parsed = parsePattern(template.pattern);
  let counter = 1;

  if (parsed.hasSeq) {
    const counterQuery = `
      SELECT next_value 
      FROM production.plan_name_counters 
      WHERE template_id = $1 AND counter_date = $2
      FOR UPDATE
    `;
    const counterResult = await client.query(counterQuery, [templateId, dateStr]);

    if (counterResult.rows.length > 0) {
      counter = counterResult.rows[0].next_value;
      
      const updateQuery = `
        UPDATE production.plan_name_counters 
        SET next_value = next_value + 1, updated_at = NOW()
        WHERE template_id = $1 AND counter_date = $2
      `;
      await client.query(updateQuery, [templateId, dateStr]);
    } else {
      const insertQuery = `
        INSERT INTO production.plan_name_counters (template_id, counter_date, next_value)
        VALUES ($1, $2, 2)
      `;
      await client.query(insertQuery, [templateId, dateStr]);
    }
  }

  let generatedName = "";
  for (const token of parsed.tokens) {
    if (token.type === "literal") {
      generatedName += token.value;
    } else if (token.type === "date" && token.format) {
      generatedName += dayjs(currentDate).format(token.format);
    } else if (token.type === "datetime" && token.format) {
      generatedName += dayjs(currentDate).format(token.format);
    } else if (token.type === "seq" && token.width) {
      generatedName += String(counter).padStart(token.width, "0");
    }
  }

  return { generatedName, counter, tokens: parsed.tokens };
}

export async function seedDefaultPlanNameTemplates(pool: Pool): Promise<{ inserted: number; skipped: number }> {
  const defaultTemplates = [
    {
      templateName: "Zapas",
      pattern: "Zapas-{date|DD.MM}-{seq|2}",
      description: "Plan zapasów z datą i numerem kolejnym (np. Zapas-12.11-01)",
      sortOrder: 1,
    },
    {
      templateName: "Zamówienia",
      pattern: "Zamówienia-{date|DD.MM.YYYY}-{seq|2}",
      description: "Plan zamówień klientów z pełną datą i numerem (np. Zamówienia-12.11.2025-01)",
      sortOrder: 2,
    },
    {
      templateName: "Rozkroje",
      pattern: "Rozkroje-{datetime|DD.MM_HH:mm}-{seq|2}",
      description: "Plan rozkrojów z datą, czasem i numerem (np. Rozkroje-12.11_14:30-01)",
      sortOrder: 3,
    },
  ];

  let inserted = 0;
  let skipped = 0;

  for (const template of defaultTemplates) {
    try {
      // Check if template with this name already exists
      const checkQuery = `
        SELECT id FROM production.plan_name_templates 
        WHERE template_name = $1 
        LIMIT 1
      `;
      const existing = await pool.query(checkQuery, [template.templateName]);

      if (existing.rows.length > 0) {
        skipped++;
        continue;
      }

      // Insert template
      const insertQuery = `
        INSERT INTO production.plan_name_templates (template_name, pattern, description, is_active, sort_order)
        VALUES ($1, $2, $3, true, $4)
      `;
      await pool.query(insertQuery, [
        template.templateName,
        template.pattern,
        template.description,
        template.sortOrder,
      ]);
      inserted++;
    } catch (error) {
      console.error(`Failed to seed template ${template.templateName}:`, error);
      skipped++;
    }
  }

  return { inserted, skipped };
}

export async function getAllTemplates(pool: Pool): Promise<ProductionPlanNameTemplate[]> {
  const query = `
    SELECT 
      id, 
      template_name as "templateName", 
      pattern, 
      description, 
      is_active as "isActive", 
      sort_order as "sortOrder",
      created_at as "createdAt",
      updated_at as "updatedAt"
    FROM production.plan_name_templates 
    WHERE is_active = true
    ORDER BY sort_order ASC, template_name ASC
  `;
  const result = await pool.query(query);
  return result.rows;
}

export async function getTemplateById(pool: Pool, id: number): Promise<ProductionPlanNameTemplate | null> {
  const query = `
    SELECT 
      id, 
      template_name as "templateName", 
      pattern, 
      description, 
      is_active as "isActive", 
      sort_order as "sortOrder",
      created_at as "createdAt",
      updated_at as "updatedAt"
    FROM production.plan_name_templates 
    WHERE id = $1
  `;
  const result = await pool.query(query, [id]);
  return result.rows[0] || null;
}

export async function createTemplate(
  pool: Pool,
  data: InsertProductionPlanNameTemplate
): Promise<ProductionPlanNameTemplate> {
  parsePattern(data.pattern);

  const query = `
    INSERT INTO production.plan_name_templates (template_name, pattern, description, is_active, sort_order)
    VALUES ($1, $2, $3, $4, $5)
    RETURNING 
      id, 
      template_name as "templateName", 
      pattern, 
      description, 
      is_active as "isActive", 
      sort_order as "sortOrder",
      created_at as "createdAt",
      updated_at as "updatedAt"
  `;
  const result = await pool.query(query, [
    data.templateName,
    data.pattern,
    data.description || null,
    data.isActive ?? true,
    data.sortOrder ?? 0,
  ]);
  return result.rows[0];
}

export async function updateTemplate(
  pool: Pool,
  id: number,
  data: Partial<InsertProductionPlanNameTemplate>
): Promise<ProductionPlanNameTemplate> {
  if (data.pattern) {
    parsePattern(data.pattern);
  }

  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.templateName !== undefined) {
    updates.push(`template_name = $${paramIndex++}`);
    values.push(data.templateName);
  }
  if (data.pattern !== undefined) {
    updates.push(`pattern = $${paramIndex++}`);
    values.push(data.pattern);
  }
  if (data.description !== undefined) {
    updates.push(`description = $${paramIndex++}`);
    values.push(data.description);
  }
  if (data.isActive !== undefined) {
    updates.push(`is_active = $${paramIndex++}`);
    values.push(data.isActive);
  }
  if (data.sortOrder !== undefined) {
    updates.push(`sort_order = $${paramIndex++}`);
    values.push(data.sortOrder);
  }

  updates.push(`updated_at = NOW()`);
  values.push(id);

  const query = `
    UPDATE production.plan_name_templates 
    SET ${updates.join(", ")}
    WHERE id = $${paramIndex}
    RETURNING 
      id, 
      template_name as "templateName", 
      pattern, 
      description, 
      is_active as "isActive", 
      sort_order as "sortOrder",
      created_at as "createdAt",
      updated_at as "updatedAt"
  `;
  const result = await pool.query(query, values);
  
  if (result.rows.length === 0) {
    throw new Error(`Template with id ${id} not found`);
  }
  
  return result.rows[0];
}

export async function deleteTemplate(pool: Pool, id: number): Promise<void> {
  const query = `
    UPDATE production.plan_name_templates 
    SET is_active = false, updated_at = NOW()
    WHERE id = $1
  `;
  await pool.query(query, [id]);
}
