// Parser i matcher dla produktów z Airtable

interface AirtableProduct {
  sku: string;
  name: string;
  length: number | null;
  width: number | null;
  color: string | null;
  doors: string | null;
  legs: string | null;
}

interface ParsedProduct {
  sku: string;
  length: number | null;
  width: number | null;
  color: string | null;
  doors: string | null;
  legs: string | null;
}

/**
 * Parsuje nazwę produktu VB w celu wyciągnięcia parametrów
 * Przykład: "Szafka na buty VB60x36D3N3 CZARNY" => { length: 60, width: 36, doors: "D3", legs: "N3", color: "CZARNY" }
 */
export function parseVBProductName(name: string): Partial<ParsedProduct> {
  const result: Partial<ParsedProduct> = {};
  
  // Regex dla pełnego formatu VB: VB{length}x{width}D{door}N{leg}
  // Akceptuje opcjonalne separatory (spacje, podkreślenia, myślniki) między tokenami
  // Przykłady: VB60x36D3N3, VB 60x36 D3 N3, VB60x36_D3_N3, VB60x36-D3-N3
  // Używa lookahead (?![0-9]) aby zapobiec dopasowaniu "VB60x36D3N300" jako D3N3
  const vbMatch = name.match(/VB\s*(\d+)[xX](\d+)[\s_\-]*D([1-4])(?![0-9])[\s_\-]*N([1-4])(?![0-9])/i);
  if (vbMatch) {
    result.length = parseInt(vbMatch[1], 10);
    result.width = parseInt(vbMatch[2], 10);
    result.doors = 'D' + vbMatch[3];
    result.legs = 'N' + vbMatch[4];
  }
  
  // Regex dla koloru - ostatnie słowo po dopasowanym tokenie VB
  // Usuń cały dopasowany token VB (np. "VB60x36D3N3") aby nie usunąć legitymnych części jak "S1"
  let colorPart = name;
  if (vbMatch) {
    colorPart = name.replace(vbMatch[0], '').trim();
  }
  
  // Weź ostatnie słowo (kolor)
  const words = colorPart.split(/\s+/);
  const potentialColor = words[words.length - 1];
  
  // Lista znanych kolorów
  const knownColors = [
    'CZARNY', 'BIALY', 'BIAŁY', 'WOTAN', 'ARTISAN', 'SONOMA', 
    'KASZMIR', 'OLEJOWANY', 'ANTRACYT', 'LANCELOT', 'ESTANA'
  ];
  
  if (potentialColor && knownColors.some(c => potentialColor.toUpperCase().includes(c))) {
    result.color = potentialColor.toUpperCase()
      .replace(/[ŁĄĆĘŃÓŚŹŻłąćęńóśźż]/g, (c: string) => {
        const map: Record<string, string> = {
          'Ł': 'L', 'Ą': 'A', 'Ć': 'C', 'Ę': 'E', 'Ń': 'N', 'Ó': 'O', 'Ś': 'S', 'Ź': 'Z', 'Ż': 'Z',
          'ł': 'L', 'ą': 'A', 'ć': 'C', 'ę': 'E', 'ń': 'N', 'ó': 'O', 'ś': 'S', 'ź': 'Z', 'ż': 'Z'
        };
        return map[c] || c;
      });
  }
  
  return result;
}

/**
 * Parsuje CSV z Airtable do tablicy produktów
 */
export function parseAirtableCSV(csvText: string): AirtableProduct[] {
  const lines = csvText.split('\n').filter(line => line.trim());
  if (lines.length === 0) return [];
  
  // Pierwsza linia to nagłówki
  const headers = lines[0].split(',');
  const skuIndex = headers.findIndex(h => h.toLowerCase().includes('sku'));
  const nameIndex = headers.findIndex(h => h.toLowerCase().includes('nazwa'));
  const lengthIndex = headers.findIndex(h => h.toLowerCase().includes('długość') || h.toLowerCase().includes('dlugosc'));
  const widthIndex = headers.findIndex(h => h.toLowerCase().includes('szerokość') || h.toLowerCase().includes('szerokosc'));
  
  const products: AirtableProduct[] = [];
  
  for (let i = 1; i < lines.length; i++) {
    const fields = parseCSVLine(lines[i]);
    if (fields.length === 0) continue;
    
    const sku = fields[skuIndex] || '';
    const name = fields[nameIndex] || '';
    
    if (!sku || !name) continue;
    
    // Parsuj nazwę produktu
    const parsed = parseVBProductName(name);
    
    // Użyj wartości z kolumn jeśli są dostępne, w przeciwnym razie z parsowanej nazwy
    const length = lengthIndex >= 0 && fields[lengthIndex] 
      ? parseFloat(fields[lengthIndex]) 
      : parsed.length;
    
    const width = widthIndex >= 0 && fields[widthIndex] 
      ? parseFloat(fields[widthIndex]) 
      : parsed.width;
    
    products.push({
      sku,
      name,
      length: length || null,
      width: width || null,
      color: parsed.color || null,
      doors: parsed.doors || null,
      legs: parsed.legs || null,
    });
  }
  
  return products;
}

/**
 * Parsuje pojedynczą linię CSV (obsługuje cudzysłowy i przecinki wewnątrz pól)
 */
function parseCSVLine(line: string): string[] {
  const result: string[] = [];
  let current = '';
  let inQuotes = false;
  
  for (let i = 0; i < line.length; i++) {
    const char = line[i];
    
    if (char === '"') {
      inQuotes = !inQuotes;
    } else if (char === ',' && !inQuotes) {
      result.push(current.trim());
      current = '';
    } else {
      current += char;
    }
  }
  
  result.push(current.trim());
  return result;
}

/**
 * Interfejs dla produktu w naszym systemie
 */
export interface SystemProduct {
  id: number;
  sku: string;
  title: string;
  length: string | null;
  width: string | null;
  color: string | null;
  doors: string | null;
  legs: string | null;
  skuAirtable: string | null;
}

/**
 * Wynik matchowania
 */
export interface MatchResult {
  airtableProduct: AirtableProduct;
  systemProduct: SystemProduct | null;
  matchScore: number; // 0-100
  matchDetails: string[];
}

/**
 * Matchuje produkty z Airtable do produktów w systemie
 */
export function matchProducts(
  airtableProducts: AirtableProduct[],
  systemProducts: SystemProduct[]
): MatchResult[] {
  const results: MatchResult[] = [];
  
  for (const airtable of airtableProducts) {
    let bestMatch: SystemProduct | null = null;
    let bestScore = 0;
    let bestDetails: string[] = [];
    
    for (const system of systemProducts) {
      const { score, details } = calculateMatchScore(airtable, system);
      
      if (score > bestScore) {
        bestScore = score;
        bestMatch = system;
        bestDetails = details;
      }
    }
    
    // Akceptuj tylko dopasowania powyżej 70% (3 z 5 parametrów)
    if (bestScore < 60) {
      bestMatch = null;
      bestDetails = ['Brak dopasowania - za mało wspólnych parametrów'];
    }
    
    results.push({
      airtableProduct: airtable,
      systemProduct: bestMatch,
      matchScore: bestScore,
      matchDetails: bestDetails,
    });
  }
  
  return results;
}

/**
 * Oblicza score dopasowania (0-100) i zwraca szczegóły
 */
function calculateMatchScore(
  airtable: AirtableProduct,
  system: SystemProduct
): { score: number; details: string[] } {
  let score = 0;
  let maxScore = 0;
  const details: string[] = [];
  
  // Długość (20 punktów)
  if (airtable.length !== null && system.length !== null) {
    maxScore += 20;
    const systemLength = parseFloat(system.length);
    if (Math.abs(airtable.length - systemLength) < 0.1) {
      score += 20;
      details.push(`✓ Długość: ${airtable.length} cm`);
    } else {
      details.push(`✗ Długość: ${airtable.length} cm ≠ ${systemLength} cm`);
    }
  }
  
  // Szerokość (20 punktów)
  if (airtable.width !== null && system.width !== null) {
    maxScore += 20;
    const systemWidth = parseFloat(system.width);
    if (Math.abs(airtable.width - systemWidth) < 0.1) {
      score += 20;
      details.push(`✓ Szerokość: ${airtable.width} cm`);
    } else {
      details.push(`✗ Szerokość: ${airtable.width} cm ≠ ${systemWidth} cm`);
    }
  }
  
  // Kolor (20 punktów)
  if (airtable.color && system.color) {
    maxScore += 20;
    if (normalizeColor(airtable.color) === normalizeColor(system.color)) {
      score += 20;
      details.push(`✓ Kolor: ${airtable.color}`);
    } else {
      details.push(`✗ Kolor: ${airtable.color} ≠ ${system.color}`);
    }
  }
  
  // Drzwi (20 punktów)
  if (airtable.doors && system.doors) {
    maxScore += 20;
    if (airtable.doors.toUpperCase() === system.doors.toUpperCase()) {
      score += 20;
      details.push(`✓ Drzwi: ${airtable.doors}`);
    } else {
      details.push(`✗ Drzwi: ${airtable.doors} ≠ ${system.doors}`);
    }
  }
  
  // Nogi (20 punktów)
  if (airtable.legs && system.legs) {
    maxScore += 20;
    if (airtable.legs.toUpperCase() === system.legs.toUpperCase()) {
      score += 20;
      details.push(`✓ Nogi: ${airtable.legs}`);
    } else {
      details.push(`✗ Nogi: ${airtable.legs} ≠ ${system.legs}`);
    }
  }
  
  // Oblicz procent
  const percentage = maxScore > 0 ? Math.round((score / maxScore) * 100) : 0;
  
  return { score: percentage, details };
}

/**
 * Normalizuje kolor do porównania (uppercase, bez polskich znaków)
 */
function normalizeColor(color: string): string {
  return color.toUpperCase()
    .replace(/[ŁĄĆĘŃÓŚŹŻłąćęńóśźż]/g, (c: string) => {
      const map: Record<string, string> = {
        'Ł': 'L', 'Ą': 'A', 'Ć': 'C', 'Ę': 'E', 'Ń': 'N', 'Ó': 'O', 'Ś': 'S', 'Ź': 'Z', 'Ż': 'Z',
        'ł': 'L', 'ą': 'A', 'ć': 'C', 'ę': 'E', 'ń': 'N', 'ó': 'O', 'ś': 'S', 'ź': 'Z', 'ż': 'Z'
      };
      return map[c] || c;
    })
    .trim();
}
