import { UseFormReturn } from "react-hook-form";
import { UseMutationResult } from "@tanstack/react-query";
import { Loader2 } from "lucide-react";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import { Button } from "@/components/ui/button";
import type { ProductionCarrierGroup } from "@shared/schema";
import { z } from "zod";

export const carrierFormSchema = z.object({
  carrierGroupId: z.coerce.number().min(1, "Grupa jest wymagana"),
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  barcode: z.string().optional(),
  status: z.string().default("available"),
  capacity: z.coerce.number().optional().nullable(),
  capacityUnit: z.string().optional(),
  notes: z.string().optional(),
  isActive: z.boolean().default(true),
});

export type CarrierFormData = z.infer<typeof carrierFormSchema>;

interface ProductionCarrierFormProps {
  form: UseFormReturn<CarrierFormData>;
  onSubmit: (data: CarrierFormData) => void;
  mutation: UseMutationResult<any, any, any, any>;
  groups: ProductionCarrierGroup[];
  mode: "create" | "edit";
  onCancel?: () => void;
}

export function ProductionCarrierForm({
  form,
  onSubmit,
  mutation,
  groups,
  mode,
  onCancel,
}: ProductionCarrierFormProps) {
  return (
    <Form {...form}>
      <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
        <FormField
          control={form.control}
          name="carrierGroupId"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Grupa nośnika</FormLabel>
              <Select
                onValueChange={(value) => field.onChange(parseInt(value))}
                value={field.value?.toString() ?? ""}
              >
                <FormControl>
                  <SelectTrigger data-testid="select-carrier-group">
                    <SelectValue placeholder="Wybierz grupę" />
                  </SelectTrigger>
                </FormControl>
                <SelectContent>
                  {groups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <FormMessage />
            </FormItem>
          )}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="code"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Kod</FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    placeholder="np. PAL-001"
                    data-testid="input-carrier-code"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="name"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Nazwa</FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    placeholder="Nazwa nośnika"
                    data-testid="input-carrier-name"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <FormField
          control={form.control}
          name="barcode"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Kod kreskowy</FormLabel>
              <FormControl>
                <Input
                  {...field}
                  value={field.value ?? ""}
                  placeholder="Kod kreskowy..."
                  data-testid="input-carrier-barcode"
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="status"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Status</FormLabel>
              <Select onValueChange={field.onChange} value={field.value}>
                <FormControl>
                  <SelectTrigger data-testid="select-carrier-status">
                    <SelectValue placeholder="Wybierz status" />
                  </SelectTrigger>
                </FormControl>
                <SelectContent>
                  <SelectItem value="available">Dostępny</SelectItem>
                  <SelectItem value="in_use">W użyciu</SelectItem>
                  <SelectItem value="maintenance">Konserwacja</SelectItem>
                  <SelectItem value="damaged">Uszkodzony</SelectItem>
                  <SelectItem value="archived">Zarchiwizowany</SelectItem>
                </SelectContent>
              </Select>
              <FormMessage />
            </FormItem>
          )}
        />

        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
          <FormField
            control={form.control}
            name="capacity"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Pojemność</FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    type="number"
                    step="0.01"
                    value={field.value ?? ""}
                    onChange={(e) => field.onChange(e.target.value === "" ? null : parseFloat(e.target.value))}
                    placeholder="0"
                    data-testid="input-carrier-capacity"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="capacityUnit"
            render={({ field }) => (
              <FormItem>
                <FormLabel>Jednostka pojemności</FormLabel>
                <FormControl>
                  <Input
                    {...field}
                    value={field.value ?? ""}
                    placeholder="np. szt, kg"
                    data-testid="input-carrier-unit"
                  />
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />
        </div>

        <FormField
          control={form.control}
          name="notes"
          render={({ field }) => (
            <FormItem>
              <FormLabel>Notatki</FormLabel>
              <FormControl>
                <Textarea
                  {...field}
                  value={field.value ?? ""}
                  placeholder="Dodatkowe notatki..."
                  data-testid="input-carrier-notes"
                />
              </FormControl>
              <FormMessage />
            </FormItem>
          )}
        />

        <FormField
          control={form.control}
          name="isActive"
          render={({ field }) => (
            <FormItem className="flex flex-row items-start space-x-3 space-y-0">
              <FormControl>
                <Checkbox
                  checked={field.value ?? true}
                  onCheckedChange={field.onChange}
                  data-testid="checkbox-carrier-active"
                />
              </FormControl>
              <div className="space-y-1 leading-none">
                <FormLabel>Aktywny</FormLabel>
              </div>
            </FormItem>
          )}
        />

        <div className="flex gap-2 justify-end pt-4">
          {onCancel && (
            <Button
              type="button"
              variant="outline"
              onClick={onCancel}
              disabled={mutation.isPending}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
          )}
          <Button
            type="submit"
            disabled={mutation.isPending}
            data-testid="button-submit-carrier"
          >
            {mutation.isPending && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
            {mode === "create" ? "Utwórz nośnik" : "Zapisz zmiany"}
          </Button>
        </div>
      </form>
    </Form>
  );
}
