import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { ArrowLeft, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { ProductionWorkCenterForm, workCenterFormSchema, type WorkCenterFormData } from "@/features/production-work-centers/production-work-center-form";
import type { ProductionWorkCenter, ProductionLocation } from "@shared/schema";
import { useEffect } from "react";

export default function ProductionWorkCenterEdit() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const { data: workCenter, isLoading } = useQuery<ProductionWorkCenter>({
    queryKey: ["/api/production/work-centers", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/work-centers/${id}`);
      if (!res.ok) throw new Error("Failed to fetch work center");
      return res.json();
    },
  });

  const { data: locations = [] } = useQuery<ProductionLocation[]>({
    queryKey: ["/api/production/locations"],
  });

  const form = useForm<WorkCenterFormData>({
    resolver: zodResolver(workCenterFormSchema),
    defaultValues: {
      code: "",
      name: "",
      status: "available",
      isActive: true,
      capabilities: [],
    },
  });

  useEffect(() => {
    if (workCenter) {
      form.reset({
        code: workCenter.code,
        name: workCenter.name,
        description: workCenter.description ?? "",
        locationId: workCenter.locationId ?? null,
        capabilities: workCenter.capabilities ?? [],
        status: workCenter.status,
        throughputPerHour: workCenter.throughputPerHour ? Number(workCenter.throughputPerHour) : null,
        throughputUnit: workCenter.throughputUnit ?? "",
        operatingCostPerHour: workCenter.operatingCostPerHour ? Number(workCenter.operatingCostPerHour) : null,
        oeeTarget: workCenter.oeeTarget ? Number(workCenter.oeeTarget) : null,
        currentOee: workCenter.currentOee ? Number(workCenter.currentOee) : null,
        isActive: workCenter.isActive ?? true,
        notes: workCenter.notes ?? "",
      });
    }
  }, [workCenter, form]);

  const updateMutation = useMutation({
    mutationFn: (data: WorkCenterFormData) => {
      const normalizedData = {
        ...data,
        locationId: data.locationId && data.locationId !== 0 ? data.locationId : null,
      };
      return apiRequest(`/api/production/work-centers/${id}`, "PATCH", normalizedData);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/work-centers"] });
      queryClient.invalidateQueries({ queryKey: ["/api/production/work-centers", id] });
      toast({ title: "Gniazdo robocze zostało zaktualizowane" });
      navigate(`/production/work-centers/${id}`);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować gniazda roboczego",
        variant: "destructive",
      });
    },
  });

  const handleCancel = () => {
    navigate(`/production/work-centers/${id}`);
  };

  if (isLoading) {
    return (
      <div className="container mx-auto p-6">
        <div className="flex justify-center p-8">
          <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-workcenter-edit" />
        </div>
      </div>
    );
  }

  if (!workCenter) {
    return (
      <div className="container mx-auto p-6">
        <Card>
          <CardContent className="p-8 text-center text-muted-foreground">
            Nie znaleziono gniazda roboczego
          </CardContent>
        </Card>
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center gap-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={handleCancel}
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-page-title">
            Edycja gniazda roboczego
          </h1>
          <p className="text-muted-foreground font-mono">
            {workCenter.code}
          </p>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane gniazda roboczego</CardTitle>
        </CardHeader>
        <CardContent>
          <ProductionWorkCenterForm
            form={form}
            onSubmit={(data) => updateMutation.mutate(data)}
            mutation={updateMutation}
            locations={locations}
            mode="edit"
            onCancel={handleCancel}
          />
        </CardContent>
      </Card>
    </div>
  );
}
