import React, { useState, useEffect, useMemo } from "react";
import { useRoute, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { MatrixLayout } from "@/features/matrix/matrix-layout";
import { Card, CardContent } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { Loader2, Plus, Copy, Trash2, Save, ArrowDown, Settings2, ListTree, ChevronRight, ChevronDown, GitBranch } from "lucide-react";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Checkbox } from "@/components/ui/checkbox";
import {
  Tooltip,
  TooltipContent,
  TooltipProvider,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import { Combobox, ComboboxOption } from "@/components/ui/combobox";
import { DimensionVariantsDialog } from "@/components/dimension-variants-dialog";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
} from "@/components/ui/dialog";
import { Label } from "@/components/ui/label";

interface DimensionVariant {
  dimension: number;
  value: number | null;
  formula: string | null;
}

interface ComponentTemplate {
  id: number;
  parentId: number | null; // Hierarchia parent-branch (automatyczna przez duplikowanie)
  furniture_type: string;
  cz1: string;
  cz2: string | null;
  door: string | null;
  leg: string | null;
  base_length: number | null;
  base_width: number | null;
  thickness: number | null;
  color: string | null;
  edge1: boolean;
  edge2: boolean;
  edge3: boolean;
  edge4: boolean;
  plate_type: string | null;
  edging_material: string | null;
  edging_pattern: string | null; // Deprecated - kept for backward compatibility
  furniture_length_condition: number | null;
  furniture_width_condition: number | null;
  alternative_lengths: DimensionVariant[];
  alternative_widths: DimensionVariant[];
  drilling_required: boolean;
  no_color_change: boolean;
  exclude_from_cutting: boolean;
  apply_by_length: boolean;
  apply_by_width: boolean;
  half_plate: boolean; // Deprecated - kept for backward compatibility
  production_category: string | null;
  unit_price: number | null;
  is_active: boolean;
}

interface TemplatesResponse {
  templates: ComponentTemplate[];
  total: number;
  limit: number;
  offset: number;
}

interface Dictionary {
  id: number;
  code: string;
  shortName: string | null;
  readableName: string | null;
  dictionaryType: string;
}

// Color mapping for visual backgrounds - prostokąty z kolorowymi tłami
const COLOR_BACKGROUNDS: Record<string, string> = {
  'BIALY': 'bg-white text-black border-2 border-gray-400',
  'CZARNY': 'bg-gray-900 text-white border-2 border-gray-700',
  'SZARY': 'bg-gray-500 text-white border-2 border-gray-600',
  'SUROWA': 'bg-amber-200 text-black border-2 border-amber-400',
  'HDF-BIALY': 'bg-gray-200 text-black border-2 border-gray-500',
  'HDF-CZARNY': 'bg-gray-800 text-white border-2 border-gray-900',
  'BRZOZA': 'bg-yellow-100 text-black border-2 border-yellow-300',
  'DĄB': 'bg-amber-600 text-white border-2 border-amber-700',
};

// Helper function to convert legacy edging_pattern to edge1-4 booleans
const convertEdgingPatternToEdges = (pattern: string | null): { edge1: boolean; edge2: boolean; edge3: boolean; edge4: boolean } => {
  if (!pattern || pattern.length !== 4) {
    return { edge1: false, edge2: false, edge3: false, edge4: false };
  }
  return {
    edge1: pattern[0] === 'T',
    edge2: pattern[1] === 'T',
    edge3: pattern[2] === 'T',
    edge4: pattern[3] === 'T',
  };
};

// Helper function to convert edge1-4 booleans back to legacy edging_pattern
const convertEdgesToPattern = (edge1: boolean, edge2: boolean, edge3: boolean, edge4: boolean): string => {
  return `${edge1 ? 'T' : 'N'}${edge2 ? 'T' : 'N'}${edge3 ? 'T' : 'N'}${edge4 ? 'T' : 'N'}`;
};

export default function MatrixTemplatesPage() {
  const { toast } = useToast();
  const [, params] = useRoute("/matrix/:furnitureType?");
  const [, setLocation] = useLocation();
  const furnitureType = params?.furnitureType || null;
  
  // Read templateId from query params
  const urlParams = new URLSearchParams(window.location.search);
  const highlightTemplateId = urlParams.get('templateId') ? parseInt(urlParams.get('templateId')!) : null;

  const [templates, setTemplates] = useState<ComponentTemplate[]>([]);
  const [editedRows, setEditedRows] = useState<Set<number>>(new Set());
  const [deletingRows, setDeletingRows] = useState<Set<number>>(new Set()); // Wiersze w trakcie usuwania - czerwone tło
  // Track expanded cells for plate_type and edging_material (key: `${templateId}-${field}`)
  const [expandedCells, setExpandedCells] = useState<Set<string>>(new Set());
  // Track expanded rows to show dimension variants inline
  const [expandedRows, setExpandedRowsState] = useState<Set<number>>(() => {
    try {
      const stored = localStorage.getItem('matrix-expandedRows');
      return stored ? new Set(JSON.parse(stored)) : new Set();
    } catch {
      return new Set();
    }
  });

  // Helper to update expandedRows with localStorage persistence
  const setExpandedRows = (value: Set<number> | ((prev: Set<number>) => Set<number>)) => {
    setExpandedRowsState(prev => {
      const newValue = typeof value === 'function' ? value(prev) : value;
      try {
        localStorage.setItem('matrix-expandedRows', JSON.stringify(Array.from(newValue)));
      } catch (e) {
        console.error('Failed to save expandedRows:', e);
      }
      return newValue;
    });
  };
  
  // State for creating new dictionary entries
  const [showNewDictionaryDialog, setShowNewDictionaryDialog] = useState(false);
  const [newDictionaryType, setNewDictionaryType] = useState<'component_cz1' | 'product_type'>('component_cz1');
  const [newDictionaryCode, setNewDictionaryCode] = useState('');
  const [newDictionaryName, setNewDictionaryName] = useState('');
  
  // State for creating variant
  const [showVariantDialog, setShowVariantDialog] = useState(false);
  const [variantSourceTemplate, setVariantSourceTemplate] = useState<ComponentTemplate | null>(null);
  const [variantCz2, setVariantCz2] = useState('');
  const [variantAlternativeLengths, setVariantAlternativeLengths] = useState<DimensionVariant[]>([]);
  const [variantAlternativeWidths, setVariantAlternativeWidths] = useState<DimensionVariant[]>([]);
  
  // Bulk edit for door/leg - using simple prompt
  
  // Load settings from localStorage
  const loadSetting = <T,>(key: string, defaultValue: T): T => {
    try {
      const stored = localStorage.getItem(`matrix-${key}`);
      return stored ? JSON.parse(stored) : defaultValue;
    } catch {
      return defaultValue;
    }
  };

  const saveSetting = <T,>(key: string, value: T) => {
    try {
      localStorage.setItem(`matrix-${key}`, JSON.stringify(value));
    } catch (e) {
      console.error('Failed to save setting:', e);
    }
  };

  // Settings with localStorage
  const [groupingMode, setGroupingModeState] = useState<'none' | 'cz1' | 'cz1_cz2'>(() => 
    loadSetting('groupingMode', 'cz1_cz2')
  );
  const [showPreviewName, setShowPreviewNameState] = useState(() => 
    loadSetting('showPreviewName', true)
  );
  const [showBaseTable, setShowBaseTableState] = useState(() => 
    loadSetting('showBaseTable', true)
  );

  const setGroupingMode = (value: 'none' | 'cz1' | 'cz1_cz2') => {
    setGroupingModeState(value);
    saveSetting('groupingMode', value);
  };

  const setShowPreviewName = (value: boolean) => {
    setShowPreviewNameState(value);
    saveSetting('showPreviewName', value);
  };

  const setShowBaseTable = (value: boolean) => {
    setShowBaseTableState(value);
    saveSetting('showBaseTable', value);
  };

  // Fetch available furniture types
  interface FurnitureType {
    furniture_type: string;
    template_count: number;
  }
  
  const { data: furnitureTypes } = useQuery<FurnitureType[]>({
    queryKey: ['/api/component-templates/furniture-types'],
  });

  // Auto-redirect to last selected or first furniture type if none selected
  useEffect(() => {
    if (!furnitureType && furnitureTypes && furnitureTypes.length > 0) {
      // Pobierz ostatnio wybrany typ z localStorage
      const lastSelectedType = loadSetting<string | null>('lastFurnitureType', null);
      
      // Sprawdź czy ostatnio wybrany typ nadal istnieje
      const targetType = lastSelectedType && furnitureTypes.some(ft => ft.furniture_type === lastSelectedType)
        ? lastSelectedType
        : furnitureTypes[0].furniture_type;
      
      // Przekieruj do wybranego typu
      setLocation(`/matrix/${encodeURIComponent(targetType)}`);
    }
  }, [furnitureType, furnitureTypes, setLocation]);

  // Zapisz ostatnio wybrany typ w localStorage
  useEffect(() => {
    if (furnitureType) {
      saveSetting('lastFurnitureType', furnitureType);
    }
  }, [furnitureType]);

  // Build query URL with params - tylko jeśli mamy wybrany furnitureType
  const queryUrl = furnitureType
    ? `/api/component-templates?furnitureType=${encodeURIComponent(furnitureType)}&limit=1000&offset=0`
    : null;

  const { data, isLoading } = useQuery<TemplatesResponse>({
    queryKey: queryUrl ? [queryUrl] : ['skip'],
    enabled: !!queryUrl, // Nie wykonuj query jeśli nie ma furnitureType
  });

  // Fetch dictionaries for cz1, cz2
  const { data: dictionaries } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries"],
  });

  // Fetch door and leg dictionaries for variant configuration
  const { data: doorDictionaries } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries?type=door"],
  });

  const { data: legDictionaries } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries?type=leg"],
  });

  // Fetch warehouse materials for plates and edging
  const { data: plateMaterials } = useQuery<any[]>({
    queryKey: [`/api/warehouse/materials?category=plyty`],
    queryFn: async () => {
      const response = await fetch('/api/warehouse/materials?category=plyty');
      if (!response.ok) throw new Error('Failed to fetch plate materials');
      return response.json();
    }
  });

  const { data: edgingMaterials } = useQuery<any[]>({
    queryKey: [`/api/warehouse/materials?category=obrzeza`],
    queryFn: async () => {
      const response = await fetch('/api/warehouse/materials?category=obrzeza');
      if (!response.ok) throw new Error('Failed to fetch edging materials');
      return response.json();
    }
  });

  useEffect(() => {
    if (data?.templates) {
      // Convert legacy edging_pattern to edge1-4 for templates that don't have edge values set
      const convertedTemplates = data.templates.map(template => {
        // If all edges are false but edging_pattern exists, convert it
        if (!template.edge1 && !template.edge2 && !template.edge3 && !template.edge4 && template.edging_pattern) {
          const edges = convertEdgingPatternToEdges(template.edging_pattern);
          return { ...template, ...edges };
        }
        return template;
      });
      setTemplates(convertedTemplates);
      setEditedRows(new Set()); // Reset edited rows when data changes
    }
  }, [data]);
  
  // Auto-expand and scroll to template from query params
  useEffect(() => {
    if (highlightTemplateId && templates.length > 0) {
      const templateExists = templates.some(t => t.id === highlightTemplateId);
      if (templateExists) {
        // Auto-expand the row
        setExpandedRows(prev => {
          const newSet = new Set(prev);
          newSet.add(highlightTemplateId);
          return newSet;
        });
        
        // Scroll to the element after a short delay to ensure rendering
        setTimeout(() => {
          const element = document.getElementById(`template-row-${highlightTemplateId}`);
          if (element) {
            element.scrollIntoView({ behavior: 'smooth', block: 'center' });
            // Flash highlight effect
            element.classList.add('ring-2', 'ring-primary', 'ring-offset-2');
            setTimeout(() => {
              element.classList.remove('ring-2', 'ring-primary', 'ring-offset-2');
            }, 2000);
          }
        }, 300);
      }
    }
  }, [highlightTemplateId, templates]);

  const total = data?.total || 0;

  // Prepare combobox options
  const cz1Options: ComboboxOption[] = dictionaries
    ?.filter(d => d.dictionaryType === 'component_cz1')
    .map(d => ({
      value: d.code,
      label: d.shortName || d.readableName || d.code,
    })) || [];

  const cz2Options: ComboboxOption[] = dictionaries
    ?.filter(d => d.dictionaryType === 'component_cz2')
    .map(d => ({
      value: d.code,
      label: d.shortName || d.readableName || d.code,
    })) || [];

  const doorOptions: ComboboxOption[] = [
    { value: '', label: '---' },
    ...(doorDictionaries?.map(d => ({
      value: d.code,
      label: d.shortName || d.readableName || d.code,
    })) || [])
  ];

  const legOptions: ComboboxOption[] = [
    { value: '', label: '---' },
    ...(legDictionaries?.map(d => ({
      value: d.code,
      label: d.shortName || d.readableName || d.code,
    })) || [])
  ];

  const colorOptions: ComboboxOption[] = dictionaries
    ?.filter(d => d.dictionaryType === 'color')
    .map(d => ({
      value: d.code,
      label: d.shortName || d.readableName || d.code,
    })) || [];

  const plateOptions: ComboboxOption[] = [
    { value: '', label: '---' },
    ...(plateMaterials?.map(m => ({
      value: m.internalCode,
      label: m.name,
    })) || [])
  ];

  const edgingOptions: ComboboxOption[] = edgingMaterials
    ?.map(m => ({
      value: m.internalCode,
      label: m.name,
    })) || [];

  // Grupowanie po cz1 i cz2
  interface GroupedTemplates {
    cz1: string;
    cz2: string | null;
    templates: ComponentTemplate[];
  }

  // Hierarchiczna struktura dla wariantów cz2
  interface TemplateWithVariants {
    parent: ComponentTemplate;
    variants: ComponentTemplate[]; // Warianty z różnymi cz2
  }

  // Stan dla rozwijania wariantów cz2
  const [expandedVariants, setExpandedVariants] = useState<Set<number>>(new Set());
  
  const toggleVariantExpansion = (templateId: number) => {
    setExpandedVariants(prev => {
      const newSet = new Set(prev);
      if (newSet.has(templateId)) {
        newSet.delete(templateId);
      } else {
        newSet.add(templateId);
      }
      return newSet;
    });
  };

  const isVariantExpanded = (templateId: number) => expandedVariants.has(templateId);

  // Funkcja do budowania hierarchii parent-children na podstawie parent_id
  // PROSTA LOGIKA: tylko parent_id decyduje kto jest rodzicem i dzieckiem
  const buildVariantHierarchy = (templates: ComponentTemplate[]): TemplateWithVariants[] => {
    const hierarchy: TemplateWithVariants[] = [];
    const processedIds = new Set<number>();

    templates.forEach(template => {
      if (processedIds.has(template.id)) return;

      const isParent = template.parentId === null; // Parent = parent_id NULL
      
      if (isParent) {
        // To jest parent - znajdź jego dzieci
        const variants = templates.filter(t => 
          !processedIds.has(t.id) &&
          t.parentId === template.id
        );
        
        hierarchy.push({
          parent: template,
          variants: variants
        });
        
        // Oznacz rodzica i wszystkie dzieci jako przetworzone
        processedIds.add(template.id);
        variants.forEach(v => processedIds.add(v.id));
      }
      // Jeśli to dziecko (ma parent_id), ale rodzic nie jest na liście - POMIŃ!
      // NIE pokazuj dziecka jako osobnego rodzica
    });

    return hierarchy;
  };

  const groupedTemplates: GroupedTemplates[] = useMemo(() => {
    if (groupingMode === 'none') {
      // No grouping - return all templates as single group
      return [{
        cz1: '',
        cz2: null,
        templates: templates,
      }];
    } else if (groupingMode === 'cz1') {
      // Group by cz1 only
      const groups = templates.reduce((acc: GroupedTemplates[], template) => {
        const existing = acc.find(g => g.cz1 === template.cz1);
        if (existing) {
          existing.templates.push(template);
        } else {
          acc.push({
            cz1: template.cz1,
            cz2: null,
            templates: [template],
          });
        }
        return acc;
      }, []);
      return groups;
    } else {
      // Group by cz1 + cz2
      const groups = templates.reduce((acc: GroupedTemplates[], template) => {
        const existing = acc.find(g => g.cz1 === template.cz1 && g.cz2 === template.cz2);
        if (existing) {
          existing.templates.push(template);
        } else {
          acc.push({
            cz1: template.cz1,
            cz2: template.cz2,
            templates: [template],
          });
        }
        return acc;
      }, []);
      return groups;
    }
  }, [templates, groupingMode]);

  // Generate partial component name for inline variants (only known data)
  const generatePartialVariantName = (template: ComponentTemplate, type: 'length' | 'width', dimension: number): string => {
    const parts = [];
    
    // cz2 (product type)
    if (template.cz2) {
      parts.push(template.cz2);
    }
    
    // cz1 (component type)
    if (template.cz1) {
      parts.push(template.cz1);
    }
    
    // dimension
    if (type === 'length') {
      // For length: VB-BOK-L-390x (x at end = unknown width)
      parts.push(`${dimension}x`);
    } else {
      // For width: VB-BOK-L-x280 (x at start = unknown length)
      parts.push(`x${dimension}`);
    }
    
    return parts.join('-');
  };

  // Generate component name preview
  const generateComponentName = (template: ComponentTemplate): string => {
    if (!template.cz1) return '';

    let name = template.cz1;

    // Dodaj cz2 jeśli istnieje
    if (template.cz2 && template.cz2.length > 0) {
      name += `-${template.cz2}`;
    }

    // Dodaj wymiary formatki jeśli są
    if (template.base_length && template.base_width) {
      name += `-${template.base_length}x${template.base_width}`;
    } else if (template.base_length) {
      name += `-${template.base_length}`;
    } else if (template.base_width) {
      name += `-${template.base_width}`;
    }

    // Dodaj grubość jeśli jest
    if (template.thickness) {
      name += `-${template.thickness}mm`;
    }

    // Dodaj kolor tylko jeśli istnieje
    if (template.color && template.color.length > 0) {
      name += `-${template.color}`;
    }

    return name;
  };

  // Generate component name with specific variant dimensions
  // Format: cz1-cz2-dimension (np. BOK-L-ALTUS-760)
  const generateVariantComponentName = (
    template: ComponentTemplate, 
    lengthMm?: number,  // wymiar w mm (np. 300, 500, 600) - dla wariantów długości
    widthMm?: number    // wymiar w mm (np. 300, 360, 460) - dla wariantów szerokości
  ): string => {
    if (!template.cz1) return '';

    let name = template.cz1;

    // Dodaj cz2 jeśli istnieje
    if (template.cz2 && template.cz2.length > 0) {
      name += `-${template.cz2}`;
    }

    // Dodaj TYLKO wymiar wariantu (długość LUB szerokość, nie oba)
    if (lengthMm !== undefined && lengthMm !== (template.base_length ?? undefined)) {
      // To jest wariant długości - dodaj tylko długość
      name += `-${lengthMm}`;
    } else if (widthMm !== undefined && widthMm !== (template.base_width ?? undefined)) {
      // To jest wariant szerokości - dodaj tylko szerokość
      name += `-${widthMm}`;
    }

    return name;
  };

  // Calculate formula value for a variant
  // dimensionCm: wymiar mebla w CM (np. 30, 50, 60)
  // Returns: wymiar formatki w MM (np. 300, 500, 600)
  const calculateFormulaValue = (formula: string | null, dimensionCm: number): number | null => {
    if (!formula) return null;
    
    try {
      const trimmedFormula = formula.trim();
      
      // Sprawdź czy to prosta liczba (np. "390") - to jest w MM
      const numberMatch = trimmedFormula.match(/^(\d+(?:\.\d+)?)$/);
      if (numberMatch) {
        return parseFloat(numberMatch[1]);
      }
      
      // Sprawdź czy to proste wyrażenie matematyczne (np. "280 - 2", "300 + 50")
      const simpleExprMatch = trimmedFormula.match(/^(\d+(?:\.\d+)?)\s*([+\-*/])\s*(\d+(?:\.\d+)?)$/);
      if (simpleExprMatch) {
        const [, num1Str, operator, num2Str] = simpleExprMatch;
        const num1 = parseFloat(num1Str);
        const num2 = parseFloat(num2Str);
        let result = num1;
        switch (operator) {
          case '+': result = num1 + num2; break;
          case '-': result = num1 - num2; break;
          case '*': result = num1 * num2; break;
          case '/': result = num1 / num2; break;
        }
        return Math.round(result);
      }
      
      // Sprawdź czy to wzór z operatorem (np. "dlMeb - 38")
      const formulaMatch = trimmedFormula.match(/^(dlMeb|szMeb)\s*([+\-*/])\s*(\d+(?:\.\d+)?)$/);
      
      if (!formulaMatch) {
        return null;
      }

      const [, variable, operator, operandStr] = formulaMatch;
      const operand = parseFloat(operandStr);
      
      // dlMeb/szMeb to wymiar MEBLA w MM
      // dimensionCm to cm (30, 50, 60), więc konwertujemy na mm
      const baseValueMm = dimensionCm * 10;  // 30 cm = 300 mm
      
      // Wykonaj obliczenie w MM (operand też jest w MM)
      let resultMm = baseValueMm;
      switch (operator) {
        case '+': resultMm = baseValueMm + operand; break;  // 300 + 50 = 350 mm
        case '-': resultMm = baseValueMm - operand; break;  // 300 - 38 = 262 mm
        case '*': resultMm = baseValueMm * operand; break;
        case '/': resultMm = baseValueMm / operand; break;
      }

      return Math.round(resultMm);
    } catch (error) {
      return null;
    }
  };

  // Toggle row expansion to show/hide dimension variants
  const toggleRowExpansion = (templateId: number) => {
    setExpandedRows(prev => {
      const next = new Set(prev);
      if (next.has(templateId)) {
        next.delete(templateId);
      } else {
        next.add(templateId);
      }
      return next;
    });
  };

  // Always expand row (for auto-expand after save)
  const expandRow = (templateId: number) => {
    setExpandedRows(prev => {
      const next = new Set(prev);
      next.add(templateId);
      return next;
    });
  };

  const isRowExpanded = (templateId: number): boolean => {
    return expandedRows.has(templateId);
  };

  // Toggle cell expansion for plate_type and edging_material
  const toggleCellExpansion = (templateId: number, field: string) => {
    const key = `${templateId}-${field}`;
    setExpandedCells(prev => {
      const next = new Set(prev);
      if (next.has(key)) {
        next.delete(key);
      } else {
        next.add(key);
      }
      return next;
    });
  };

  const isCellExpanded = (templateId: number, field: string): boolean => {
    return expandedCells.has(`${templateId}-${field}`);
  };

  // Render collapsed/expanded value with toggle button
  const renderCollapsibleValue = (
    value: string | null,
    templateId: number,
    field: 'plate_type' | 'edging_material',
    options: ComboboxOption[]
  ) => {
    if (!value) return null;
    
    const isExpanded = isCellExpanded(templateId, field);
    const option = options.find(o => o.value === value);
    const label = option?.label || value;
    
    // If label is short (<=3 chars), no need for collapsing
    if (label.length <= 3) {
      return (
        <Tooltip>
          <TooltipTrigger asChild>
            <span className="text-xs">{label}</span>
          </TooltipTrigger>
          <TooltipContent>{label}</TooltipContent>
        </Tooltip>
      );
    }
    
    const displayText = isExpanded ? label : label.substring(0, 3);
    
    return (
      <div className="flex items-center gap-1">
        <Tooltip>
          <TooltipTrigger asChild>
            <span className={`text-xs ${isExpanded ? '' : 'font-mono'}`}>
              {displayText}
            </span>
          </TooltipTrigger>
          <TooltipContent>{label}</TooltipContent>
        </Tooltip>
        <Button
          variant="ghost"
          size="icon"
          className="h-5 w-5 shrink-0"
          onClick={() => toggleCellExpansion(templateId, field)}
          data-testid={`button-toggle-${field}-${templateId}`}
        >
          {isExpanded ? '-' : '+'}
        </Button>
      </div>
    );
  };

  // Update template field
  const updateField = (id: number, field: keyof ComponentTemplate, value: any) => {
    setTemplates(prev => prev.map(t => 
      t.id === id ? { ...t, [field]: value } : t
    ));
    setEditedRows(prev => new Set(prev).add(id));
  };

  // Bulk update door/leg for all parents
  const handleBulkDoorEdit = () => {
    const value = window.prompt('Wpisz wartość Door dla wszystkich formatek rodziców (zostaw puste aby wyczyścić):') || '';
    const parentTemplates = templates.filter(t => t.parentId === null);
    
    parentTemplates.forEach(t => {
      updateField(t.id, 'door', value || null);
    });
    
    toast({
      title: `Zaktualizowano ${parentTemplates.length} ${parentTemplates.length === 1 ? 'formatkę' : 'formatek'}`,
      description: `Ustawiono Door = "${value || '(puste)'}"`
    });
  };

  const handleBulkLegEdit = () => {
    const value = window.prompt('Wpisz wartość Leg dla wszystkich formatek rodziców (zostaw puste aby wyczyścić):') || '';
    const parentTemplates = templates.filter(t => t.parentId === null);
    
    parentTemplates.forEach(t => {
      updateField(t.id, 'leg', value || null);
    });
    
    toast({
      title: `Zaktualizowano ${parentTemplates.length} ${parentTemplates.length === 1 ? 'formatkę' : 'formatek'}`,
      description: `Ustawiono Leg = "${value || '(puste)'}"`
    });
  };

  // Save mutation - bulk save all edited rows
  const saveMutation = useMutation({
    mutationFn: async (triggerTemplate: ComponentTemplate) => {
      // Zbierz wszystkie edytowane rekordy (w tym ten który wywołał zapis)
      const editedTemplates = templates.filter(t => editedRows.has(t.id));
      
      // Oddziel nowe (temp ID > 1000000000) od istniejących
      const newTemplates = editedTemplates.filter(t => t.id > 1000000000);
      const existingTemplates = editedTemplates.filter(t => t.id <= 1000000000);
      
      const results: any[] = [];
      
      // Zapisz nowe formatki (POST) pojedynczo
      for (const template of newTemplates) {
        try {
          const templateWithLegacyFields = {
            ...template,
            edging_pattern: convertEdgesToPattern(template.edge1, template.edge2, template.edge3, template.edge4),
          };
          const { id, ...rest } = templateWithLegacyFields;
          const result = await apiRequest('POST', '/api/component-templates', rest);
          results.push({ id: template.id, success: true, data: result });
        } catch (error: any) {
          results.push({ id: template.id, success: false, error: error.message });
        }
      }
      
      // Zapisz istniejące formatki (bulk update)
      if (existingTemplates.length > 0) {
        const templatesWithLegacyFields = existingTemplates.map(t => ({
          ...t,
          edging_pattern: convertEdgesToPattern(t.edge1, t.edge2, t.edge3, t.edge4),
        }));
        
        const response = await apiRequest('POST', '/api/component-templates/bulk-update', {
          templates: templatesWithLegacyFields
        });
        const bulkResult = await response.json();
        
        results.push(...bulkResult.results);
      }
      
      return {
        results,
        summary: {
          total: editedTemplates.length,
          success: results.filter((r: any) => r.success).length,
          failed: results.filter((r: any) => !r.success).length
        }
      };
    },
    onSuccess: (data) => {
      const { summary, results } = data;
      
      // Pokaż komunikat o zapisanych zmianach
      if (summary.failed === 0) {
        toast({ 
          title: `Zapisano ${summary.success} ${summary.success === 1 ? 'formatkę' : 'formatki'}` 
        });
      } else {
        toast({ 
          title: `Zapisano ${summary.success} z ${summary.total}`,
          description: `${summary.failed} ${summary.failed === 1 ? 'formatka nie została zapisana' : 'formatek nie zostało zapisanych'}`,
          variant: "destructive"
        });
      }
      
      // Usuń tylko pomyślnie zapisane wiersze z editedRows
      setEditedRows(prev => {
        const newSet = new Set(prev);
        results.forEach((result: any) => {
          if (result.success) {
            newSet.delete(result.id);
          }
        });
        return newSet;
      });
      
      // Invalidate queries to refresh data
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0]?.toString() || '';
          return key.startsWith('/api/component-templates');
        }
      });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd zapisu", 
        description: error.message,
        variant: "destructive" 
      });
    },
  });

  // Duplicate template mutation - immediately saves copy to database  
  const duplicateMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest('POST', `/api/component-templates/${id}/duplicate`);
    },
    onSuccess: () => {
      toast({ 
        title: "Skopiowano formatkę", 
        description: "Kopia została zapisana" 
      });
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0]?.toString() || '';
          return key.startsWith('/api/component-templates');
        }
      });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd kopiowania", 
        description: error.message,
        variant: "destructive" 
      });
    },
  });

  // Delete template
  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest('DELETE', `/api/component-templates/${id}`);
    },
    onMutate: async (id: number) => {
      // Oznacz wiersz jako usuwany - czerwone tło
      setDeletingRows(prev => new Set(prev).add(id));
    },
    onSuccess: () => {
      toast({ title: "Usunięto formatkę" });
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0]?.toString() || '';
          return key.startsWith('/api/component-templates');
        }
      });
      // Wiersz zniknie po refetch, więc nie trzeba usuwać z deletingRows
    },
    onError: (error: any, id: number) => {
      // W razie błędu - usuń z deletingRows
      setDeletingRows(prev => {
        const next = new Set(prev);
        next.delete(id);
        return next;
      });
      toast({ 
        title: "Błąd usuwania", 
        description: error.message,
        variant: "destructive" 
      });
    },
  });

  // Create variant template
  const createVariantMutation = useMutation({
    mutationFn: async ({ id, cz2, alternativeLengths, alternativeWidths }: { 
      id: number; 
      cz2: string;
      alternativeLengths: DimensionVariant[];
      alternativeWidths: DimensionVariant[];
    }) => {
      return apiRequest('POST', `/api/component-templates/${id}/variant`, { 
        cz2,
        alternative_lengths: alternativeLengths,
        alternative_widths: alternativeWidths
      });
    },
    onSuccess: () => {
      toast({ title: "Utworzono wariant formatki" });
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0]?.toString() || '';
          return key.startsWith('/api/component-templates');
        }
      });
      setShowVariantDialog(false);
      setVariantSourceTemplate(null);
      setVariantCz2('');
      setVariantAlternativeLengths([]);
      setVariantAlternativeWidths([]);
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd tworzenia wariantu", 
        description: error.message,
        variant: "destructive" 
      });
    },
  });

  // Create new dictionary entry
  const createDictionaryMutation = useMutation({
    mutationFn: async (data: { dictionaryType: string; code: string; name: string; shortName: string; readableName: string }) => {
      return apiRequest('POST', '/api/dictionaries', data);
    },
    onSuccess: () => {
      toast({ title: "Dodano nowy wpis słownika" });
      queryClient.invalidateQueries({ queryKey: ['/api/dictionaries'] });
      setShowNewDictionaryDialog(false);
      setNewDictionaryCode('');
      setNewDictionaryName('');
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się dodać wpisu",
        variant: "destructive" 
      });
    },
  });

  // Open dialog to create new cz1 entry
  const handleCreateNewCz1 = () => {
    setNewDictionaryType('component_cz1');
    setShowNewDictionaryDialog(true);
  };

  // Open dialog to create new cz2 (product_type) entry
  const handleCreateNewCz2 = () => {
    setNewDictionaryType('product_type');
    setShowNewDictionaryDialog(true);
  };

  // Save new dictionary entry
  const handleSaveNewDictionary = () => {
    if (!newDictionaryCode || !newDictionaryName) {
      toast({ 
        title: "Błąd", 
        description: "Wypełnij wszystkie pola",
        variant: "destructive" 
      });
      return;
    }

    const code = newDictionaryCode.toUpperCase();
    createDictionaryMutation.mutate({
      dictionaryType: newDictionaryType,
      code,
      name: newDictionaryName,
      shortName: code,
      readableName: newDictionaryName,
    });
  };

  // Add new row
  const addNewRow = () => {
    const newTemplate: ComponentTemplate = {
      id: Date.now(), // Temporary ID
      furniture_type: furnitureType || 'NOWY',
      cz1: '',
      cz2: '',
      door: null,
      leg: null,
      base_length: null,
      base_width: null,
      thickness: null,
      color: null,
      edge1: false,
      edge2: false,
      edge3: false,
      edge4: false,
      plate_type: null,
      edging_material: null,
      edging_pattern: null, // Deprecated
      furniture_length_condition: null,
      furniture_width_condition: null,
      alternative_lengths: [],
      alternative_widths: [],
      drilling_required: false,
      no_color_change: false,
      exclude_from_cutting: false,
      apply_by_length: false,
      apply_by_width: false,
      half_plate: false, // Deprecated
      production_category: null,
      unit_price: null,
      is_active: true,
    };
    setTemplates(prev => [...prev, newTemplate]);
    setEditedRows(prev => new Set(prev).add(newTemplate.id));
  };

  // Duplicate parent or branch (copy entire template with deep clone of arrays)
  // Duplicate as branch mutation - saves to database with parent_id set
  const duplicateBranchMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest('POST', `/api/component-templates/${id}/duplicate-as-branch`);
    },
    onSuccess: (newBranch: ComponentTemplate) => {
      toast({ 
        title: "Zduplikowano formatkę jako branch", 
        description: `Utworzono branch z parent_id=${newBranch.parentId}`
      });
      
      // Automatycznie rozwiń rodzica aby pokazać nowe dziecko
      if (newBranch.parentId) {
        setExpandedVariants(prev => {
          const next = new Set(prev);
          next.add(newBranch.parentId!);
          return next;
        });
      }
      
      queryClient.invalidateQueries({ 
        predicate: (query) => {
          const key = query.queryKey[0]?.toString() || '';
          return key.startsWith('/api/component-templates');
        }
      });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd duplikowania branch", 
        description: error.message,
        variant: "destructive" 
      });
    },
  });

  const duplicateBranch = (template: ComponentTemplate) => {
    duplicateBranchMutation.mutate(template.id);
  };

  // Copy column value down to next row only
  const copyColumnDown = (field: keyof ComponentTemplate, fromIndex: number) => {
    const value = templates[fromIndex][field];
    const nextIndex = fromIndex + 1;
    
    if (nextIndex >= templates.length) {
      toast({ title: "Brak wiersza poniżej do skopiowania", variant: "destructive" });
      return;
    }
    
    setTemplates(prev => prev.map((t, idx) => 
      idx === nextIndex ? { ...t, [field]: value } : t
    ));
    
    // Mark the next row as edited
    setEditedRows(prev => new Set(prev).add(templates[nextIndex].id));
    toast({ title: `Skopiowano wartość do następnego wiersza` });
  };

  // State for dimension variants table
  const [selectedTemplateForVariants, setSelectedTemplateForVariants] = useState<number | null>(null);
  const [showVariantsTable, setShowVariantsTableState] = useState(() => 
    loadSetting('showVariantsTable', false)
  );

  const setShowVariantsTable = (value: boolean) => {
    setShowVariantsTableState(value);
    saveSetting('showVariantsTable', value);
  };

  return (
    <MatrixLayout furnitureType={furnitureType}>
      <div className="p-4 md:p-6 space-y-6">
        <div className="flex flex-col gap-4">
          <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
            <div>
              <h1 className="text-2xl md:text-3xl font-bold">
                {furnitureType ? `Formatki: ${furnitureType}` : "Wszystkie formatki"}
              </h1>
              <p className="text-sm text-muted-foreground mt-1">
                {total} {total === 1 ? "szablon" : "szablonów"} • {editedRows.size} edytowanych
              </p>
            </div>
            <div className="flex gap-2">
              <Button 
                variant="outline"
                size="sm"
                onClick={() => setShowBaseTable(!showBaseTable)}
                data-testid="button-toggle-base-table"
                className={`${showBaseTable ? 'bg-blue-500/10 border-blue-500/50 text-blue-600 dark:text-blue-400' : ''}`}
              >
                <Settings2 className="w-3 h-3 mr-1.5" />
                {showBaseTable ? "Ukryj" : "Pokaż"} parametry
              </Button>
              <Button 
                variant="outline"
                size="sm"
                onClick={() => setShowVariantsTable(!showVariantsTable)}
                data-testid="button-toggle-variants"
                className={`${showVariantsTable ? 'bg-purple-500/10 border-purple-500/50 text-purple-600 dark:text-purple-400' : ''}`}
              >
                <ListTree className="w-3 h-3 mr-1.5" />
                {showVariantsTable ? "Ukryj" : "Pokaż"} warianty
              </Button>
            </div>
          </div>

          {/* Ustawienia wyświetlania */}
          <div className="flex flex-wrap items-center gap-4 text-sm">
            <label className="flex items-center gap-2 cursor-pointer">
              <Checkbox
                checked={showPreviewName}
                onCheckedChange={setShowPreviewName}
                data-testid="checkbox-show-preview-name"
              />
              <span>Pokaż podgląd nazwy</span>
            </label>
          </div>
        </div>

        {/* Tabela 1: Komponenty bazowe */}
        {showBaseTable && (
          <Card>
            <CardContent className="p-4 pb-0">
              <div className="flex items-center justify-between mb-4">
                <div className="flex items-center gap-2">
                  <h2 className="text-lg font-semibold">Komponenty bazowe</h2>
                </div>
                <div className="flex items-center gap-2">
                  <label className="text-sm font-medium">Grupowanie:</label>
                  <Select value={groupingMode} onValueChange={(value: 'none' | 'cz1' | 'cz1_cz2') => setGroupingMode(value)}>
                    <SelectTrigger className="w-[180px]" data-testid="select-grouping-mode">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="none">Brak grupowania</SelectItem>
                      <SelectItem value="cz1">Grupuj po Cz1</SelectItem>
                      <SelectItem value="cz1_cz2">Grupuj po Cz1 + Cz2</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
              </div>
            </CardContent>
            <CardContent className="p-0">
            {isLoading ? (
              <div className="flex items-center justify-center py-12">
                <Loader2 className="w-8 h-8 animate-spin text-muted-foreground" />
              </div>
            ) : templates.length === 0 ? (
              <div className="text-center py-12 text-muted-foreground">
                Brak szablonów formatek
              </div>
            ) : (
              <div className="overflow-x-auto">
                <TooltipProvider>
                  <Table>
                    <TableHeader>
                      {/* Nagłówek sekcji */}
                      <TableRow className="border-b-2">
                        <TableHead colSpan={showPreviewName ? 14 : 13} className="text-center font-bold text-sm bg-muted/30">
                          WYMIARY I PARAMETRY
                        </TableHead>
                        <TableHead colSpan={2} className="text-center font-bold text-sm bg-accent/30 sticky right-0 z-10">
                          ZARZĄDZANIE
                        </TableHead>
                      </TableRow>
                      {/* Nagłówki kolumn */}
                      <TableRow>
                        <TableHead className="w-10 sticky left-0 bg-background z-10"></TableHead>
                        <TableHead className="w-16 sticky left-10 bg-background z-10">ID</TableHead>
                        {showPreviewName && <TableHead className="min-w-[200px]">Podgląd nazwy</TableHead>}
                        <TableHead className="min-w-[120px]">cz1</TableHead>
                        <TableHead className="min-w-[120px]">cz2</TableHead>
                        <TableHead className="min-w-[100px]">
                          <div className="flex items-center gap-1">
                            <span>Door</span>
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  className="h-5 w-5 shrink-0"
                                  onClick={handleBulkDoorEdit}
                                  data-testid="button-bulk-door"
                                >
                                  <Copy className="w-3 h-3" />
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Ustaw dla wszystkich rodziców</TooltipContent>
                            </Tooltip>
                          </div>
                        </TableHead>
                        <TableHead className="min-w-[100px]">
                          <div className="flex items-center gap-1">
                            <span>Leg</span>
                            <Tooltip>
                              <TooltipTrigger asChild>
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  className="h-5 w-5 shrink-0"
                                  onClick={handleBulkLegEdit}
                                  data-testid="button-bulk-leg"
                                >
                                  <Copy className="w-3 h-3" />
                                </Button>
                              </TooltipTrigger>
                              <TooltipContent>Ustaw dla wszystkich rodziców</TooltipContent>
                            </Tooltip>
                          </div>
                        </TableHead>
                        <TableHead className="min-w-[100px] text-right">Dł.</TableHead>
                        <TableHead className="min-w-[100px] text-right">Szer.</TableHead>
                        <TableHead className="min-w-[100px] text-right">Grub.</TableHead>
                        <TableHead className="min-w-[120px]">Stała płyta</TableHead>
                        <TableHead className="min-w-[180px]">Obrzeża</TableHead>
                        <TableHead className="min-w-[100px]">Opcje</TableHead>
                        <TableHead className="min-w-[100px] text-center">Użycia</TableHead>
                        <TableHead className="min-w-[140px] sticky right-0 bg-background z-10">Akcje</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {groupedTemplates.map((group, groupIndex) => (
                        <React.Fragment key={`group-${group.cz1}-${group.cz2}`}>
                          {/* Nagłówek grupy - tylko gdy jest grupowanie */}
                          {groupingMode !== 'none' && (
                            <TableRow className="bg-muted/50 hover:bg-muted/50">
                              <TableCell colSpan={showPreviewName ? 15 : 14} className="font-semibold sticky left-0 z-10 bg-inherit">
                                <div className="flex items-center gap-2">
                                  <Badge variant="secondary">{group.cz1}</Badge>
                                  {groupingMode === 'cz1_cz2' && group.cz2 && <Badge variant="outline">{group.cz2}</Badge>}
                                  <span className="text-muted-foreground text-sm ml-auto">
                                    {group.templates.length} {group.templates.length === 1 ? 'formatka' : 'formatek'}
                                  </span>
                                </div>
                              </TableCell>
                            </TableRow>
                          )}
                          {/* Szablony w grupie - z hierarchią parent-branch (parent_id) */}
                          {buildVariantHierarchy(group.templates).map((node) => {
                            const template = node.parent;
                            const rowIndex = templates.indexOf(template);
                            const hasVariants = (template.alternative_lengths?.length ?? 0) > 0 || 
                                               (template.alternative_widths?.length ?? 0) > 0;
                            const isExpanded = isRowExpanded(template.id);
                            const hasBranches = node.variants.length > 0;
                            const isBranchExpanded = expandedVariants.has(template.id);
                            
                            return (
                              <React.Fragment key={template.id}>
                                {/* PARENT FORMATKA - niebieskie tło TYLKO gdy jest rozwinięty (pokazuje branches) */}
                                <TableRow 
                                  id={`template-row-${template.id}`}
                                  className={
                                    deletingRows.has(template.id)
                                      ? "bg-red-200 dark:bg-red-900/50 transition-colors duration-300" 
                                      : isBranchExpanded
                                        ? "bg-blue-100 dark:bg-blue-900/30" 
                                        : editedRows.has(template.id) 
                                          ? "bg-yellow-200 dark:bg-yellow-900/40" 
                                          : ""
                                  }
                                  data-testid={`row-template-${template.id}`}
                                >
                                  <TableCell className="sticky left-0 bg-inherit z-10 p-0">
                                    <div className="flex items-center gap-1">
                                      {/* NAJPIERW: Niebieski przycisk wariantów - liczba wewnątrz */}
                                      {hasVariants && (
                                        <Button
                                          variant="ghost"
                                          size="sm"
                                          className="h-8 px-2 shrink-0 bg-blue-500/80 hover:bg-blue-600 text-white"
                                          onClick={() => toggleRowExpansion(template.id)}
                                          data-testid={`button-expand-${template.id}`}
                                        >
                                          <div className="flex items-center gap-1">
                                            {isExpanded ? (
                                              <ChevronDown className="w-3 h-3" />
                                            ) : (
                                              <ChevronRight className="w-3 h-3" />
                                            )}
                                            <span className="text-xs font-semibold">
                                              {(template.alternative_lengths?.length || 0) + (template.alternative_widths?.length || 0)}
                                            </span>
                                          </div>
                                        </Button>
                                      )}
                                      {/* POTEM: Fioletowy przycisk dzieci (branches) - taka sama wysokość jak niebieski */}
                                      {hasBranches && (
                                        <Button
                                          variant="ghost"
                                          size="sm"
                                          className="h-8 px-2 shrink-0 bg-purple-500/80 hover:bg-purple-600 text-white"
                                          onClick={() => {
                                            setExpandedVariants(prev => {
                                              const next = new Set(prev);
                                              if (next.has(template.id)) {
                                                next.delete(template.id);
                                              } else {
                                                next.add(template.id);
                                              }
                                              return next;
                                            });
                                          }}
                                          data-testid={`button-expand-branch-${template.id}`}
                                        >
                                          <div className="flex items-center gap-1">
                                            {isBranchExpanded ? (
                                              <ChevronDown className="w-3 h-3" />
                                            ) : (
                                              <ChevronRight className="w-3 h-3" />
                                            )}
                                            <span className="text-xs font-semibold">{node.variants.length}</span>
                                          </div>
                                        </Button>
                                      )}
                                    </div>
                                  </TableCell>
                                  <TableCell className="font-mono text-xs sticky left-10 bg-inherit z-10">
                                    {template.id}
                                  </TableCell>
                                
                                {/* Podgląd nazwy */}
                                {showPreviewName && (
                                  <TableCell>
                                    <div className="font-mono text-sm font-semibold text-primary">
                                      {generateComponentName(template) || <span className="text-muted-foreground italic">brak nazwy</span>}
                                    </div>
                                  </TableCell>
                                )}
                                
                                {/* cz1 */}
                                <TableCell>
                            <div className="flex items-center gap-1">
                              <Combobox
                                options={cz1Options}
                                  value={template.cz1}
                                  onChange={(value) => updateField(template.id, 'cz1', value)}
                                  placeholder="Wybierz cz1..."
                                  searchPlaceholder="Szukaj..."
                                  className="h-8 text-sm min-w-[100px] bg-green-500/80 hover:bg-green-600 active:bg-green-700 text-white border-green-600"
                                  onCreateNew={handleCreateNewCz1}
                                  createNewLabel="+ Dodaj nowy element"
                                />
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6 shrink-0"
                                      onClick={() => copyColumnDown('cz1', rowIndex)}
                                      data-testid={`button-copy-cz1-${template.id}`}
                                    >
                                      <ArrowDown className="w-3 h-3" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kopiuj poniżej</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* cz2 */}
                            <TableCell>
                              <div className="flex items-center gap-1">
                                <Combobox
                                  options={cz2Options}
                                  value={template.cz2 || ''}
                                  onChange={(value) => updateField(template.id, 'cz2', value)}
                                  placeholder="Wybierz cz2..."
                                  searchPlaceholder="Szukaj..."
                                  className="h-8 text-sm min-w-[100px]"
                                  onCreateNew={handleCreateNewCz2}
                                  createNewLabel="+ Dodaj nowy produkt"
                                />
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6 shrink-0"
                                      onClick={() => copyColumnDown('cz2', rowIndex)}
                                    >
                                      <ArrowDown className="w-3 h-3" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kopiuj poniżej</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* Door */}
                            <TableCell>
                              <div className="flex items-center gap-1">
                                <Combobox
                                  options={doorOptions}
                                  value={template.door || ''}
                                  onChange={(value) => updateField(template.id, 'door', value || null)}
                                  placeholder="Door..."
                                  searchPlaceholder="Szukaj..."
                                  className="h-8 text-sm min-w-[80px] bg-purple-500/80 hover:bg-purple-600 text-white border-purple-600"
                                />
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6 shrink-0"
                                      onClick={() => copyColumnDown('door', rowIndex)}
                                    >
                                      <ArrowDown className="w-3 h-3" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kopiuj poniżej</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* Leg */}
                            <TableCell>
                              <div className="flex items-center gap-1">
                                <Combobox
                                  options={legOptions}
                                  value={template.leg || ''}
                                  onChange={(value) => updateField(template.id, 'leg', value || null)}
                                  placeholder="Leg..."
                                  searchPlaceholder="Szukaj..."
                                  className="h-8 text-sm min-w-[80px] bg-purple-600/80 hover:bg-purple-700 text-white border-purple-700"
                                />
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6 shrink-0"
                                      onClick={() => copyColumnDown('leg', rowIndex)}
                                    >
                                      <ArrowDown className="w-3 h-3" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kopiuj poniżej</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* Długość */}
                            <TableCell>
                              <div className="flex items-center gap-1">
                                <DimensionVariantsDialog
                                  type="length"
                                  variants={template.alternative_lengths || []}
                                  onChange={(variants) => updateField(template.id, 'alternative_lengths', variants)}
                                  furnitureType={template.furniture_type}
                                  furnitureLength={template.base_length}
                                  furnitureWidth={template.base_width}
                                  furnitureLengthCondition={template.furniture_length_condition}
                                  furnitureWidthCondition={template.furniture_width_condition}
                                  cz1={template.cz1}
                                  cz2={template.cz2}
                                  onExpandAfterSave={() => expandRow(template.id)}
                                  triggerButton={
                                    <Button 
                                      variant="ghost" 
                                      className="h-8 px-3 bg-blue-500/80 hover:bg-blue-600 text-white w-full justify-start gap-2"
                                      data-testid={`button-variants-length-${template.id}`}
                                    >
                                      <Settings2 className="w-4 h-4" />
                                      <span className="flex-1 text-left">Generuj długość ({template.alternative_lengths?.length || 0})</span>
                                    </Button>
                                  }
                                />
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6 shrink-0"
                                      onClick={() => copyColumnDown('base_length', rowIndex)}
                                    >
                                      <ArrowDown className="w-3 h-3" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kopiuj poniżej</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* Szerokość */}
                            <TableCell>
                              <div className="flex items-center gap-1">
                                <DimensionVariantsDialog
                                  type="width"
                                  variants={template.alternative_widths || []}
                                  onChange={(variants) => updateField(template.id, 'alternative_widths', variants)}
                                  furnitureType={template.furniture_type}
                                  furnitureLength={template.base_length}
                                  furnitureWidth={template.base_width}
                                  furnitureLengthCondition={template.furniture_length_condition}
                                  furnitureWidthCondition={template.furniture_width_condition}
                                  cz1={template.cz1}
                                  cz2={template.cz2}
                                  onExpandAfterSave={() => expandRow(template.id)}
                                  triggerButton={
                                    <Button 
                                      variant="ghost" 
                                      className="h-8 px-3 bg-blue-500/80 hover:bg-blue-600 text-white w-full justify-start gap-2"
                                      data-testid={`button-variants-width-${template.id}`}
                                    >
                                      <Settings2 className="w-4 h-4" />
                                      <span className="flex-1 text-left">Generuj szerokość ({template.alternative_widths?.length || 0})</span>
                                    </Button>
                                  }
                                />
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6 shrink-0"
                                      onClick={() => copyColumnDown('base_width', rowIndex)}
                                    >
                                      <ArrowDown className="w-3 h-3" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kopiuj poniżej</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* Grubość */}
                            <TableCell>
                              <div className="flex items-center gap-1">
                                <Input
                                  type="number"
                                  value={template.thickness || ''}
                                  onChange={(e) => updateField(template.id, 'thickness', e.target.value ? parseFloat(e.target.value) : null)}
                                  className="h-8 text-sm text-right min-w-[80px]"
                                  data-testid={`input-thickness-${template.id}`}
                                />
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6 shrink-0"
                                      onClick={() => copyColumnDown('thickness', rowIndex)}
                                    >
                                      <ArrowDown className="w-3 h-3" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kopiuj poniżej</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* Stała płyta */}
                            <TableCell>
                              <div className="flex items-center gap-1">
                                <Combobox
                                  options={plateOptions}
                                  value={template.plate_type || ''}
                                  onChange={(value) => updateField(template.id, 'plate_type', value)}
                                  placeholder="Wybierz płytę..."
                                  searchPlaceholder="Szukaj płyty..."
                                  className="h-8 text-sm min-w-[120px]"
                                />
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-6 w-6 shrink-0"
                                      onClick={() => copyColumnDown('plate_type', rowIndex)}
                                    >
                                      <ArrowDown className="w-3 h-3" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Kopiuj poniżej</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* Obrzeża - 4 checkboxes */}
                            <TableCell>
                              <div className="flex items-center gap-2">
                                <div className="grid grid-cols-2 gap-1">
                                  <label className="flex items-center gap-1 cursor-pointer">
                                    <Checkbox
                                      checked={template.edge1}
                                      onCheckedChange={(checked) => updateField(template.id, 'edge1', checked)}
                                      className="h-3.5 w-3.5"
                                      data-testid={`checkbox-edge1-${template.id}`}
                                    />
                                    <span className="text-xs">O1</span>
                                  </label>
                                  <label className="flex items-center gap-1 cursor-pointer">
                                    <Checkbox
                                      checked={template.edge2}
                                      onCheckedChange={(checked) => updateField(template.id, 'edge2', checked)}
                                      className="h-3.5 w-3.5"
                                      data-testid={`checkbox-edge2-${template.id}`}
                                    />
                                    <span className="text-xs">O2</span>
                                  </label>
                                  <label className="flex items-center gap-1 cursor-pointer">
                                    <Checkbox
                                      checked={template.edge3}
                                      onCheckedChange={(checked) => updateField(template.id, 'edge3', checked)}
                                      className="h-3.5 w-3.5"
                                      data-testid={`checkbox-edge3-${template.id}`}
                                    />
                                    <span className="text-xs">O3</span>
                                  </label>
                                  <label className="flex items-center gap-1 cursor-pointer">
                                    <Checkbox
                                      checked={template.edge4}
                                      onCheckedChange={(checked) => updateField(template.id, 'edge4', checked)}
                                      className="h-3.5 w-3.5"
                                      data-testid={`checkbox-edge4-${template.id}`}
                                    />
                                    <span className="text-xs">O4</span>
                                  </label>
                                </div>
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="sm"
                                      className="h-6 px-1.5 text-xs shrink-0"
                                      onClick={() => {
                                        const allChecked = template.edge1 && template.edge2 && template.edge3 && template.edge4;
                                        updateField(template.id, 'edge1', !allChecked);
                                        updateField(template.id, 'edge2', !allChecked);
                                        updateField(template.id, 'edge3', !allChecked);
                                        updateField(template.id, 'edge4', !allChecked);
                                      }}
                                      data-testid={`button-toggle-all-edges-${template.id}`}
                                    >
                                      {template.edge1 && template.edge2 && template.edge3 && template.edge4 ? 'X' : '✓'}
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Zaznacz/odznacz wszystkie obrzeża</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>

                            {/* Opcje */}
                            <TableCell>
                              <div className="flex flex-col gap-1">
                                <label className="flex items-center gap-1 cursor-pointer">
                                  <Checkbox
                                    checked={template.drilling_required}
                                    onCheckedChange={(checked) => updateField(template.id, 'drilling_required', checked)}
                                    className="h-3.5 w-3.5"
                                    data-testid={`checkbox-drilling-${template.id}`}
                                  />
                                  <span className="text-xs">Wierc.</span>
                                </label>
                                <label className="flex items-center gap-1 cursor-pointer">
                                  <input
                                    type="radio"
                                    checked={template.apply_by_length}
                                    onChange={() => {
                                      updateField(template.id, 'apply_by_length', true);
                                      updateField(template.id, 'apply_by_width', false);
                                    }}
                                    className="h-3.5 w-3.5"
                                    data-testid={`radio-apply-length-${template.id}`}
                                  />
                                  <span className="text-xs">Dł.</span>
                                </label>
                                <label className="flex items-center gap-1 cursor-pointer">
                                  <input
                                    type="radio"
                                    checked={template.apply_by_width}
                                    onChange={() => {
                                      updateField(template.id, 'apply_by_length', false);
                                      updateField(template.id, 'apply_by_width', true);
                                    }}
                                    className="h-3.5 w-3.5"
                                    data-testid={`radio-apply-width-${template.id}`}
                                  />
                                  <span className="text-xs">Szer.</span>
                                </label>
                              </div>
                            </TableCell>

                            {/* Użycia */}
                            <TableCell className="text-center">
                              <TemplateUsageBadge templateId={template.id} />
                            </TableCell>

                            {/* Akcje */}
                            <TableCell className="sticky right-0 bg-inherit z-10">
                              <div className="flex items-center gap-1">
                                {editedRows.has(template.id) && (
                                  <Tooltip>
                                    <TooltipTrigger asChild>
                                      <Button
                                        variant="ghost"
                                        size="icon"
                                        className="h-8 w-8 text-green-600 shrink-0"
                                        onClick={() => saveMutation.mutate(template)}
                                        disabled={saveMutation.isPending}
                                        data-testid={`button-save-${template.id}`}
                                      >
                                        {saveMutation.isPending ? (
                                          <Loader2 className="w-4 h-4 animate-spin" />
                                        ) : (
                                          <Save className="w-4 h-4" />
                                        )}
                                      </Button>
                                    </TooltipTrigger>
                                    <TooltipContent>Zapisz</TooltipContent>
                                  </Tooltip>
                                )}
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-8 w-8 shrink-0"
                                      onClick={() => duplicateMutation.mutate(template.id)}
                                      data-testid={`button-duplicate-${template.id}`}
                                    >
                                      <Copy className="w-4 h-4" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Duplikuj (niezależna kopia)</TooltipContent>
                                </Tooltip>
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-8 w-8 shrink-0 text-blue-600"
                                      onClick={() => duplicateBranch(template)}
                                      data-testid={`button-branch-${template.id}`}
                                    >
                                      <GitBranch className="w-4 h-4" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Duplikuj jako branch</TooltipContent>
                                </Tooltip>
                                <Tooltip>
                                  <TooltipTrigger asChild>
                                    <Button
                                      variant="ghost"
                                      size="icon"
                                      className="h-8 w-8 text-destructive shrink-0"
                                      onClick={() => {
                                        if (template.id > 1000000000) {
                                          // Remove from local state if not saved yet
                                          setTemplates(prev => prev.filter(t => t.id !== template.id));
                                          setEditedRows(prev => {
                                            const next = new Set(prev);
                                            next.delete(template.id);
                                            return next;
                                          });
                                        } else {
                                          deleteMutation.mutate(template.id);
                                        }
                                      }}
                                      data-testid={`button-delete-${template.id}`}
                                    >
                                      <Trash2 className="w-4 h-4" />
                                    </Button>
                                  </TooltipTrigger>
                                  <TooltipContent>Usuń</TooltipContent>
                                </Tooltip>
                              </div>
                            </TableCell>
                        </TableRow>

                        {/* Expanded variants - shown when row is expanded */}
                        {isExpanded && hasVariants && (
                          <TableRow className="bg-muted/10">
                            <TableCell colSpan={showPreviewName ? 13 : 12} className="p-4">
                              <div className="space-y-4">
                                {/* Length variants */}
                                {template.alternative_lengths && template.alternative_lengths.length > 0 && (
                                  <div>
                                    <h4 className="font-semibold text-sm mb-2 text-muted-foreground">
                                      Warianty długości ({template.alternative_lengths.length})
                                    </h4>
                                    <div className="grid grid-cols-1 gap-2">
                                      {template.alternative_lengths.map((variant, idx) => (
                                        <div 
                                          key={idx}
                                          className="flex items-center gap-2 p-2 bg-background rounded border text-sm"
                                        >
                                          <span className="text-center text-muted-foreground text-xs font-semibold w-6 shrink-0">
                                            {idx + 1}.
                                          </span>
                                          <Badge variant="outline" className="font-mono text-xs px-2">
                                            {generatePartialVariantName(template, 'length', variant.dimension * 10)}
                                          </Badge>
                                          <span className="text-muted-foreground text-xs font-medium">Długość mebla (cm):</span>
                                          <span className="font-semibold w-12 text-center">{variant.dimension}</span>
                                          <span className="text-muted-foreground text-xs font-medium">Wzór:</span>
                                          <Input
                                            value={variant.formula || ''}
                                            onChange={(e) => {
                                              const newFormula = e.target.value || null;
                                              const calculatedValue = calculateFormulaValue(newFormula, variant.dimension);
                                              const updated = [...(template.alternative_lengths || [])];
                                              updated[idx] = { 
                                                ...updated[idx], 
                                                formula: newFormula,
                                                value: calculatedValue !== null ? calculatedValue : updated[idx].value
                                              };
                                              updateField(template.id, 'alternative_lengths', updated);
                                            }}
                                            className="h-7 w-32 text-xs font-mono bg-muted/40 text-muted-foreground"
                                            placeholder="np. dlMeb + 50"
                                          />
                                          <span className="text-muted-foreground text-xs font-medium">Długość formatki (mm):</span>
                                          <Input
                                            type="number"
                                            value={variant.value ?? ''}
                                            onChange={(e) => {
                                              const updated = [...(template.alternative_lengths || [])];
                                              updated[idx] = { ...updated[idx], value: e.target.value ? parseFloat(e.target.value) : null };
                                              updateField(template.id, 'alternative_lengths', updated);
                                            }}
                                            className="h-7 w-24 text-sm text-right"
                                            placeholder="--"
                                          />
                                          <Tooltip>
                                            <TooltipTrigger asChild>
                                              <Button
                                                variant="ghost"
                                                size="icon"
                                                className="h-7 w-7 text-destructive"
                                                onClick={() => {
                                                  const updated = (template.alternative_lengths || []).filter((_, i) => i !== idx);
                                                  updateField(template.id, 'alternative_lengths', updated);
                                                }}
                                              >
                                                <Trash2 className="w-3 h-3" />
                                              </Button>
                                            </TooltipTrigger>
                                            <TooltipContent>Usuń wariant</TooltipContent>
                                          </Tooltip>
                                        </div>
                                      ))}
                                    </div>
                                  </div>
                                )}

                                {/* Width variants */}
                                {template.alternative_widths && template.alternative_widths.length > 0 && (
                                  <div>
                                    <h4 className="font-semibold text-sm mb-2 text-muted-foreground">
                                      Warianty szerokości ({template.alternative_widths.length})
                                    </h4>
                                    <div className="grid grid-cols-1 gap-2">
                                      {template.alternative_widths.map((variant, idx) => (
                                        <div 
                                          key={idx}
                                          className="flex items-center gap-2 p-2 bg-background rounded border text-sm"
                                        >
                                          <span className="text-center text-muted-foreground text-xs font-semibold w-6 shrink-0">
                                            {idx + 1}.
                                          </span>
                                          <Badge variant="outline" className="font-mono text-xs px-2">
                                            {generatePartialVariantName(template, 'width', variant.dimension * 10)}
                                          </Badge>
                                          <span className="text-muted-foreground text-xs font-medium">Szerokość mebla (cm):</span>
                                          <span className="font-semibold w-12 text-center">{variant.dimension}</span>
                                          <span className="text-muted-foreground text-xs font-medium">Wzór:</span>
                                          <Input
                                            value={variant.formula || ''}
                                            onChange={(e) => {
                                              const newFormula = e.target.value || null;
                                              const calculatedValue = calculateFormulaValue(newFormula, variant.dimension);
                                              const updated = [...(template.alternative_widths || [])];
                                              updated[idx] = { 
                                                ...updated[idx], 
                                                formula: newFormula,
                                                value: calculatedValue !== null ? calculatedValue : updated[idx].value
                                              };
                                              updateField(template.id, 'alternative_widths', updated);
                                            }}
                                            className="h-7 w-32 text-xs font-mono bg-muted/40 text-muted-foreground"
                                            placeholder="np. szMeb + 50"
                                          />
                                          <span className="text-muted-foreground text-xs font-medium">Szerokość formatki (mm):</span>
                                          <Input
                                            type="number"
                                            value={variant.value ?? ''}
                                            onChange={(e) => {
                                              const updated = [...(template.alternative_widths || [])];
                                              updated[idx] = { ...updated[idx], value: e.target.value ? parseFloat(e.target.value) : null };
                                              updateField(template.id, 'alternative_widths', updated);
                                            }}
                                            className="h-7 w-24 text-sm text-right"
                                            placeholder="--"
                                          />
                                          <Tooltip>
                                            <TooltipTrigger asChild>
                                              <Button
                                                variant="ghost"
                                                size="icon"
                                                className="h-7 w-7 text-destructive"
                                                onClick={() => {
                                                  const updated = (template.alternative_widths || []).filter((_, i) => i !== idx);
                                                  updateField(template.id, 'alternative_widths', updated);
                                                }}
                                              >
                                                <Trash2 className="w-3 h-3" />
                                              </Button>
                                            </TooltipTrigger>
                                            <TooltipContent>Usuń wariant</TooltipContent>
                                          </Tooltip>
                                        </div>
                                      ))}
                                    </div>
                                  </div>
                                )}
                              </div>
                            </TableCell>
                          </TableRow>
                        )}

                        {/* Branchy - pokazane gdy parent jest rozwinięty - PEŁNY WIERSZ */}
                        {isBranchExpanded && hasBranches && node.variants.map((variantTemplate) => {
                          const childTemplate = variantTemplate;
                          const childRowIndex = templates.indexOf(childTemplate);
                          const childHasVariants = (childTemplate.alternative_lengths?.length || 0) > 0 || (childTemplate.alternative_widths?.length || 0) > 0;
                          const isChildExpanded = isRowExpanded(childTemplate.id);
                          
                          return (
                            <React.Fragment key={childTemplate.id}>
                              <TableRow 
                                className={
                                  deletingRows.has(childTemplate.id)
                                    ? "bg-red-200 dark:bg-red-900/50 transition-colors duration-300"
                                    : editedRows.has(childTemplate.id)
                                      ? "bg-yellow-200 dark:bg-yellow-900/40"
                                      : "bg-blue-100 dark:bg-blue-900/30"
                                }
                                data-testid={`row-variant-${childTemplate.id}`}
                              >
                                <TableCell className="sticky left-0 bg-inherit z-10 p-0">
                                  <div className="flex items-center gap-1 pl-12">
                                    <GitBranch className="w-3 h-3 text-muted-foreground" />
                                    {childHasVariants && (
                                      <Button
                                        variant="ghost"
                                        size="sm"
                                        className="h-8 px-2 shrink-0 bg-blue-500/80 hover:bg-blue-600 text-white"
                                        onClick={() => toggleRowExpansion(childTemplate.id)}
                                        data-testid={`button-expand-${childTemplate.id}`}
                                      >
                                        <div className="flex items-center gap-1">
                                          {isChildExpanded ? (
                                            <ChevronDown className="w-3 h-3" />
                                          ) : (
                                            <ChevronRight className="w-3 h-3" />
                                          )}
                                          <span className="text-xs font-semibold">
                                            {(childTemplate.alternative_lengths?.length || 0) + (childTemplate.alternative_widths?.length || 0)}
                                          </span>
                                        </div>
                                      </Button>
                                    )}
                                  </div>
                                </TableCell>
                                <TableCell className="font-mono text-xs sticky left-10 bg-inherit z-10">
                                  {childTemplate.id}
                                </TableCell>
                                
                                {showPreviewName && (
                                  <TableCell>
                                    <div className="font-mono text-sm font-semibold text-primary">
                                      {generateComponentName(childTemplate) || <span className="text-muted-foreground italic">brak nazwy</span>}
                                    </div>
                                  </TableCell>
                                )}
                                
                                <TableCell>
                                  <div className="flex items-center gap-1">
                                    <Combobox
                                      options={cz1Options}
                                      value={childTemplate.cz1}
                                      onChange={(value) => updateField(childTemplate.id, 'cz1', value)}
                                      placeholder="Wybierz cz1..."
                                      searchPlaceholder="Szukaj..."
                                      className="h-8 text-sm min-w-[100px] bg-green-500/80 hover:bg-green-600 active:bg-green-700 text-white border-green-600"
                                      onCreateNew={handleCreateNewCz1}
                                      createNewLabel="+ Dodaj nowy element"
                                    />
                                  </div>
                                </TableCell>

                                <TableCell>
                                  <div className="flex items-center gap-1">
                                    <Combobox
                                      options={cz2Options}
                                      value={childTemplate.cz2 || ''}
                                      onChange={(value) => updateField(childTemplate.id, 'cz2', value)}
                                      placeholder="Wybierz cz2..."
                                      searchPlaceholder="Szukaj..."
                                      className="h-8 text-sm min-w-[100px] bg-green-600/90 hover:bg-green-700/90 active:bg-green-800/90 text-white border-green-700"
                                      onCreateNew={handleCreateNewCz2}
                                      createNewLabel="+ Dodaj nowy produkt"
                                    />
                                  </div>
                                </TableCell>

                                {/* Door */}
                                <TableCell>
                                  <div className="flex items-center gap-1">
                                    <Combobox
                                      options={doorOptions}
                                      value={childTemplate.door || ''}
                                      onChange={(value) => updateField(childTemplate.id, 'door', value || null)}
                                      placeholder="Door..."
                                      searchPlaceholder="Szukaj..."
                                      className="h-8 text-sm min-w-[80px] bg-purple-500/80 hover:bg-purple-600 text-white border-purple-600"
                                    />
                                  </div>
                                </TableCell>

                                {/* Leg */}
                                <TableCell>
                                  <div className="flex items-center gap-1">
                                    <Combobox
                                      options={legOptions}
                                      value={childTemplate.leg || ''}
                                      onChange={(value) => updateField(childTemplate.id, 'leg', value || null)}
                                      placeholder="Leg..."
                                      searchPlaceholder="Szukaj..."
                                      className="h-8 text-sm min-w-[80px] bg-purple-600/80 hover:bg-purple-700 text-white border-purple-700"
                                    />
                                  </div>
                                </TableCell>

                                {/* Długość */}
                                <TableCell>
                                  <div className="flex items-center gap-1">
                                    <DimensionVariantsDialog
                                      type="length"
                                      variants={childTemplate.alternative_lengths || []}
                                      onChange={(variants) => updateField(childTemplate.id, 'alternative_lengths', variants)}
                                      furnitureType={childTemplate.furniture_type}
                                      furnitureLength={childTemplate.base_length}
                                      furnitureWidth={childTemplate.base_width}
                                      furnitureLengthCondition={childTemplate.furniture_length_condition}
                                      furnitureWidthCondition={childTemplate.furniture_width_condition}
                                      cz1={childTemplate.cz1}
                                      cz2={childTemplate.cz2}
                                      onExpandAfterSave={() => expandRow(childTemplate.id)}
                                      triggerButton={
                                        <Button 
                                          variant="ghost" 
                                          className="h-8 px-3 bg-blue-500/80 hover:bg-blue-600 text-white w-full justify-start gap-2"
                                          data-testid={`button-variants-length-${childTemplate.id}`}
                                        >
                                          <Settings2 className="w-4 h-4" />
                                          <span className="flex-1 text-left">Generuj długość ({childTemplate.alternative_lengths?.length || 0})</span>
                                        </Button>
                                      }
                                    />
                                  </div>
                                </TableCell>

                                {/* Szerokość */}
                                <TableCell>
                                  <div className="flex items-center gap-1">
                                    <DimensionVariantsDialog
                                      type="width"
                                      variants={childTemplate.alternative_widths || []}
                                      onChange={(variants) => updateField(childTemplate.id, 'alternative_widths', variants)}
                                      furnitureType={childTemplate.furniture_type}
                                      furnitureLength={childTemplate.base_length}
                                      furnitureWidth={childTemplate.base_width}
                                      furnitureLengthCondition={childTemplate.furniture_length_condition}
                                      furnitureWidthCondition={childTemplate.furniture_width_condition}
                                      cz1={childTemplate.cz1}
                                      cz2={childTemplate.cz2}
                                      onExpandAfterSave={() => expandRow(childTemplate.id)}
                                      triggerButton={
                                        <Button 
                                          variant="ghost" 
                                          className="h-8 px-3 bg-blue-500/80 hover:bg-blue-600 text-white w-full justify-start gap-2"
                                          data-testid={`button-variants-width-${childTemplate.id}`}
                                        >
                                          <Settings2 className="w-4 h-4" />
                                          <span className="flex-1 text-left">Generuj szerokość ({childTemplate.alternative_widths?.length || 0})</span>
                                        </Button>
                                      }
                                    />
                                  </div>
                                </TableCell>

                                {/* Grubość */}
                                <TableCell>
                                  <div className="flex items-center gap-1">
                                    <Input
                                      type="number"
                                      value={childTemplate.thickness || ''}
                                      onChange={(e) => updateField(childTemplate.id, 'thickness', e.target.value ? parseFloat(e.target.value) : null)}
                                      className="h-8 text-sm text-right min-w-[80px]"
                                      data-testid={`input-thickness-${childTemplate.id}`}
                                    />
                                  </div>
                                </TableCell>

                                {/* Stała płyta */}
                                <TableCell>
                                  <div className="flex items-center gap-1">
                                    <Combobox
                                      options={plateOptions}
                                      value={childTemplate.plate_type || ''}
                                      onChange={(value) => updateField(childTemplate.id, 'plate_type', value)}
                                      placeholder="Wybierz płytę..."
                                      searchPlaceholder="Szukaj płyty..."
                                      className="h-8 text-sm min-w-[120px]"
                                    />
                                  </div>
                                </TableCell>

                                {/* Akcje */}
                                <TableCell className="sticky right-0 bg-inherit z-10">
                                  <div className="flex items-center gap-1">
                                    {editedRows.has(childTemplate.id) && (
                                      <Tooltip>
                                        <TooltipTrigger asChild>
                                          <Button
                                            variant="ghost"
                                            size="icon"
                                            className="h-8 w-8 text-green-600 shrink-0"
                                            onClick={() => saveMutation.mutate(childTemplate)}
                                            disabled={saveMutation.isPending}
                                            data-testid={`button-save-${childTemplate.id}`}
                                          >
                                            {saveMutation.isPending ? (
                                              <Loader2 className="w-4 h-4 animate-spin" />
                                            ) : (
                                              <Save className="w-4 h-4" />
                                            )}
                                          </Button>
                                        </TooltipTrigger>
                                        <TooltipContent>Zapisz</TooltipContent>
                                      </Tooltip>
                                    )}
                                    <Tooltip>
                                      <TooltipTrigger asChild>
                                        <Button
                                          variant="ghost"
                                          size="icon"
                                          className="h-8 w-8 text-blue-600 shrink-0"
                                          onClick={() => duplicateMutation.mutate(childTemplate.id)}
                                          data-testid={`button-duplicate-${childTemplate.id}`}
                                        >
                                          <Copy className="w-4 h-4" />
                                        </Button>
                                      </TooltipTrigger>
                                      <TooltipContent>Duplikuj (niezależna kopia)</TooltipContent>
                                    </Tooltip>
                                    <Tooltip>
                                      <TooltipTrigger asChild>
                                        <Button
                                          variant="ghost"
                                          size="icon"
                                          className="h-8 w-8 text-destructive shrink-0"
                                          onClick={() => {
                                            if (childTemplate.id > 1000000000) {
                                              setTemplates(prev => prev.filter(t => t.id !== childTemplate.id));
                                              setEditedRows(prev => {
                                                const next = new Set(prev);
                                                next.delete(childTemplate.id);
                                                return next;
                                              });
                                            } else {
                                              deleteMutation.mutate(childTemplate.id);
                                            }
                                          }}
                                          data-testid={`button-delete-${childTemplate.id}`}
                                        >
                                          <Trash2 className="w-4 h-4" />
                                        </Button>
                                      </TooltipTrigger>
                                      <TooltipContent>Usuń wariant</TooltipContent>
                                    </Tooltip>
                                  </div>
                                </TableCell>
                              </TableRow>

                              {/* Expanded variants dla child template */}
                              {isChildExpanded && childHasVariants && (
                                <TableRow className="bg-blue-50/50 dark:bg-blue-950/10">
                                  <TableCell colSpan={showPreviewName ? 15 : 14} className="p-4">
                                    <div className="space-y-4 pl-12">
                                      {/* Length variants for child */}
                                      {childTemplate.alternative_lengths && childTemplate.alternative_lengths.length > 0 && (
                                        <div>
                                          <h4 className="font-semibold text-sm mb-2 text-muted-foreground">
                                            Warianty długości ({childTemplate.alternative_lengths.length})
                                          </h4>
                                          <div className="grid grid-cols-1 gap-2">
                                            {childTemplate.alternative_lengths.map((variant, idx) => (
                                              <div 
                                                key={idx}
                                                className="flex items-center gap-2 p-2 bg-background rounded border text-sm"
                                              >
                                                <span className="text-center text-muted-foreground text-xs font-semibold w-6 shrink-0">
                                                  {idx + 1}.
                                                </span>
                                                <Badge variant="outline" className="font-mono text-xs px-2">
                                                  {generatePartialVariantName(childTemplate, 'length', variant.dimension * 10)}
                                                </Badge>
                                                <span className="text-muted-foreground text-xs font-medium">Długość mebla (cm):</span>
                                                <span className="font-semibold w-12 text-center">{variant.dimension}</span>
                                                <span className="text-muted-foreground text-xs font-medium">Wzór:</span>
                                                <span className="font-mono text-xs bg-muted/40 px-2 py-1 rounded">{variant.formula || '---'}</span>
                                                <span className="text-muted-foreground text-xs font-medium">Długość formatki (mm):</span>
                                                <span className="font-semibold">{variant.value || '---'}</span>
                                              </div>
                                            ))}
                                          </div>
                                        </div>
                                      )}

                                      {/* Width variants for child */}
                                      {childTemplate.alternative_widths && childTemplate.alternative_widths.length > 0 && (
                                        <div>
                                          <h4 className="font-semibold text-sm mb-2 text-muted-foreground">
                                            Warianty szerokości ({childTemplate.alternative_widths.length})
                                          </h4>
                                          <div className="grid grid-cols-1 gap-2">
                                            {childTemplate.alternative_widths.map((variant, idx) => (
                                              <div 
                                                key={idx}
                                                className="flex items-center gap-2 p-2 bg-background rounded border text-sm"
                                              >
                                                <span className="text-center text-muted-foreground text-xs font-semibold w-6 shrink-0">
                                                  {idx + 1}.
                                                </span>
                                                <Badge variant="outline" className="font-mono text-xs px-2">
                                                  {generatePartialVariantName(childTemplate, 'width', variant.dimension * 10)}
                                                </Badge>
                                                <span className="text-muted-foreground text-xs font-medium">Szerokość mebla (cm):</span>
                                                <span className="font-semibold w-12 text-center">{variant.dimension}</span>
                                                <span className="text-muted-foreground text-xs font-medium">Wzór:</span>
                                                <span className="font-mono text-xs bg-muted/40 px-2 py-1 rounded">{variant.formula || '---'}</span>
                                                <span className="text-muted-foreground text-xs font-medium">Szerokość formatki (mm):</span>
                                                <span className="font-semibold">{variant.value || '---'}</span>
                                              </div>
                                            ))}
                                          </div>
                                        </div>
                                      )}
                                    </div>
                                  </TableCell>
                                </TableRow>
                              )}
                            </React.Fragment>
                          );
                        })}
                      </React.Fragment>
                    );
                  })}
                </React.Fragment>
              ))}
                    </TableBody>
                  </Table>
                </TooltipProvider>
              </div>
            )}
          </CardContent>
          <CardContent className="p-4 pt-2">
            <Button 
              onClick={addNewRow}
              size="sm"
              data-testid="button-add-template"
              className="bg-green-600 hover:bg-green-700 text-white"
            >
              <Plus className="w-3 h-3 mr-1.5" />
              Dodaj wiersz
            </Button>
          </CardContent>
          </Card>
        )}

        {/* Tabela 2a: Warianty długości */}
        {showVariantsTable && (
          <Card>
            <CardContent className="p-4 pb-0">
              <div className="flex items-center justify-between mb-4">
                <h2 className="text-2xl font-bold">
                  Długość ({templates.reduce((sum, t) => sum + (t.alternative_lengths?.length || 0), 0)})
                </h2>
              </div>
            </CardContent>
            <CardContent className="p-0">
              <div className="overflow-x-auto">
                <TooltipProvider>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="w-12">Lp.</TableHead>
                        <TableHead className="w-16">ID</TableHead>
                        <TableHead className="min-w-[200px]">Komponent</TableHead>
                        <TableHead className="w-24">Wymiar</TableHead>
                        <TableHead className="w-32">Wartość</TableHead>
                        <TableHead className="min-w-[200px]">Formuła</TableHead>
                        <TableHead className="w-24">Akcje</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {templates.flatMap(template => {
                        const lengthVariants = (template.alternative_lengths || []).map(v => ({
                          templateId: template.id,
                          templateName: generateVariantComponentName(template, v.dimension * 10, template.base_width ?? undefined),
                          variant: v
                        }));
                        return lengthVariants;
                      }).map((item, idx) => (
                        <TableRow key={`${item.templateId}-length-${item.variant.dimension}-${idx}`}>
                          <TableCell className="text-center text-muted-foreground text-xs font-semibold">
                            {idx + 1}
                          </TableCell>
                          <TableCell className="font-mono text-xs">
                            {item.templateId}
                          </TableCell>
                          <TableCell className="text-sm">
                            {item.templateName}
                          </TableCell>
                          <TableCell className="font-semibold">
                            {item.variant.dimension}
                          </TableCell>
                          <TableCell>
                            <Input
                              type="number"
                              step="0.01"
                              value={item.variant.value || ''}
                              onChange={(e) => {
                                const newValue = e.target.value ? parseFloat(e.target.value) : null;
                                const template = templates.find(t => t.id === item.templateId);
                                if (!template) return;
                                
                                const updatedVariants = (template.alternative_lengths || []).map(v =>
                                  v.dimension === item.variant.dimension ? { ...v, value: newValue } : v
                                );
                                updateField(item.templateId, 'alternative_lengths', updatedVariants);
                              }}
                              className="h-8 text-sm text-right"
                              data-testid={`input-variant-value-${item.templateId}-length-${item.variant.dimension}`}
                            />
                          </TableCell>
                          <TableCell>
                            <Input
                              type="text"
                              value={item.variant.formula || ''}
                              onChange={(e) => {
                                const newFormula = e.target.value || null;
                                const template = templates.find(t => t.id === item.templateId);
                                if (!template) return;
                                
                                const updatedVariants = (template.alternative_lengths || []).map(v =>
                                  v.dimension === item.variant.dimension ? { ...v, formula: newFormula } : v
                                );
                                updateField(item.templateId, 'alternative_lengths', updatedVariants);
                              }}
                              className="h-8 text-sm font-mono"
                              placeholder="np. dlSzaf - 38"
                              data-testid={`input-variant-formula-${item.templateId}-length-${item.variant.dimension}`}
                            />
                          </TableCell>
                          <TableCell>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-8 w-8 text-destructive"
                              onClick={() => {
                                const template = templates.find(t => t.id === item.templateId);
                                if (!template) return;
                                
                                const updatedVariants = (template.alternative_lengths || []).filter(v =>
                                  v.dimension !== item.variant.dimension
                                );
                                updateField(item.templateId, 'alternative_lengths', updatedVariants);
                                toast({ title: "Usunięto wariant długości" });
                              }}
                              data-testid={`button-delete-variant-${item.templateId}-length-${item.variant.dimension}`}
                            >
                              <Trash2 className="w-4 h-4" />
                            </Button>
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </TooltipProvider>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Tabela 2b: Warianty szerokości */}
        {showVariantsTable && (
          <Card>
            <CardContent className="p-4 pb-0">
              <div className="flex items-center justify-between mb-4">
                <h2 className="text-2xl font-bold">
                  Szerokość ({templates.reduce((sum, t) => sum + (t.alternative_widths?.length || 0), 0)})
                </h2>
              </div>
            </CardContent>
            <CardContent className="p-0">
              <div className="overflow-x-auto">
                <TooltipProvider>
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead className="w-12">Lp.</TableHead>
                        <TableHead className="w-16">ID</TableHead>
                        <TableHead className="min-w-[200px]">Komponent</TableHead>
                        <TableHead className="w-24">Wymiar</TableHead>
                        <TableHead className="w-32">Wartość</TableHead>
                        <TableHead className="min-w-[200px]">Formuła</TableHead>
                        <TableHead className="w-24">Akcje</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {templates.flatMap(template => {
                        const widthVariants = (template.alternative_widths || []).map(v => ({
                          templateId: template.id,
                          templateName: generateVariantComponentName(template, template.base_length ?? undefined, v.dimension * 10),
                          variant: v
                        }));
                        return widthVariants;
                      }).map((item, idx) => (
                        <TableRow key={`${item.templateId}-width-${item.variant.dimension}-${idx}`}>
                          <TableCell className="text-center text-muted-foreground text-xs font-semibold">
                            {idx + 1}
                          </TableCell>
                          <TableCell className="font-mono text-xs">
                            {item.templateId}
                          </TableCell>
                          <TableCell className="text-sm">
                            {item.templateName}
                          </TableCell>
                          <TableCell className="font-semibold">
                            {item.variant.dimension}
                          </TableCell>
                          <TableCell>
                            <Input
                              type="number"
                              step="0.01"
                              value={item.variant.value || ''}
                              onChange={(e) => {
                                const newValue = e.target.value ? parseFloat(e.target.value) : null;
                                const template = templates.find(t => t.id === item.templateId);
                                if (!template) return;
                                
                                const updatedVariants = (template.alternative_widths || []).map(v =>
                                  v.dimension === item.variant.dimension ? { ...v, value: newValue } : v
                                );
                                updateField(item.templateId, 'alternative_widths', updatedVariants);
                              }}
                              className="h-8 text-sm text-right"
                              data-testid={`input-variant-value-${item.templateId}-width-${item.variant.dimension}`}
                            />
                          </TableCell>
                          <TableCell>
                            <Input
                              type="text"
                              value={item.variant.formula || ''}
                              onChange={(e) => {
                                const newFormula = e.target.value || null;
                                const template = templates.find(t => t.id === item.templateId);
                                if (!template) return;
                                
                                const updatedVariants = (template.alternative_widths || []).map(v =>
                                  v.dimension === item.variant.dimension ? { ...v, formula: newFormula } : v
                                );
                                updateField(item.templateId, 'alternative_widths', updatedVariants);
                              }}
                              className="h-8 text-sm font-mono"
                              placeholder="np. szSzaf + 20"
                              data-testid={`input-variant-formula-${item.templateId}-width-${item.variant.dimension}`}
                            />
                          </TableCell>
                          <TableCell>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-8 w-8 text-destructive"
                              onClick={() => {
                                const template = templates.find(t => t.id === item.templateId);
                                if (!template) return;
                                
                                const updatedVariants = (template.alternative_widths || []).filter(v =>
                                  v.dimension !== item.variant.dimension
                                );
                                updateField(item.templateId, 'alternative_widths', updatedVariants);
                                toast({ title: "Usunięto wariant szerokości" });
                              }}
                              data-testid={`button-delete-variant-${item.templateId}-width-${item.variant.dimension}`}
                            >
                              <Trash2 className="w-4 h-4" />
                            </Button>
                          </TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </TooltipProvider>
              </div>
            </CardContent>
          </Card>
        )}
      </div>

      {/* Dialog do dodawania nowego wpisu słownika */}
      <Dialog open={showNewDictionaryDialog} onOpenChange={setShowNewDictionaryDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>
              {newDictionaryType === 'component_cz1' ? 'Dodaj nowy element (cz1)' : 'Dodaj nowy rodzaj produktu (cz2)'}
            </DialogTitle>
            <DialogDescription>
              Utwórz nowy wpis w słowniku {newDictionaryType === 'component_cz1' ? 'elementów' : 'rodzajów produktów'}
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            <div>
              <Label htmlFor="dict-code">Kod (wielkie litery, bez polskich znaków)</Label>
              <Input
                id="dict-code"
                value={newDictionaryCode}
                onChange={(e) => setNewDictionaryCode(e.target.value.toUpperCase())}
                placeholder={newDictionaryType === 'component_cz1' ? 'np. BOK-L' : 'np. ALTUS'}
                className="mt-1"
              />
            </div>

            <div>
              <Label htmlFor="dict-name">Nazwa (czytelna)</Label>
              <Input
                id="dict-name"
                value={newDictionaryName}
                onChange={(e) => setNewDictionaryName(e.target.value)}
                placeholder={newDictionaryType === 'component_cz1' ? 'np. Bok lewy' : 'np. Altus'}
                className="mt-1"
              />
            </div>
          </div>

          <DialogFooter>
            <Button 
              variant="outline" 
              onClick={() => {
                setShowNewDictionaryDialog(false);
                setNewDictionaryCode('');
                setNewDictionaryName('');
              }}
            >
              Anuluj
            </Button>
            <Button 
              onClick={handleSaveNewDictionary}
              disabled={createDictionaryMutation.isPending}
            >
              {createDictionaryMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Zapisywanie...
                </>
              ) : (
                'Zapisz'
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      {/* Dialog do tworzenia wariantu formatki */}
      <Dialog open={showVariantDialog} onOpenChange={setShowVariantDialog}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Utwórz wariant formatki</DialogTitle>
            <DialogDescription>
              Wariant skopiuje wszystkie parametry z formatki "{variantSourceTemplate?.cz1}" 
              i pozwoli ustawić nową wartość cz2 (np. D2, VB-D3, VB-N3)
            </DialogDescription>
          </DialogHeader>

          <div className="space-y-4">
            <div>
              <Label htmlFor="variant-cz2">Wariant cz2 *</Label>
              <Combobox
                options={cz2Options}
                value={variantCz2}
                onChange={setVariantCz2}
                placeholder="Wybierz wariant cz2..."
                searchPlaceholder="Szukaj wariantu..."
                emptyText="Nie znaleziono wariantu"
                className="mt-1"
              />
              <p className="text-xs text-muted-foreground mt-1">
                Wartość cz2 określa wariant (np. D2 dla drzwi z uchwytem, VB-D3 dla VB-D3, VB-N3 dla nóżek)
              </p>
            </div>

            {variantSourceTemplate && (
              <div className="space-y-3">
                <div className="bg-muted p-3 rounded-md text-sm">
                  <p className="font-semibold mb-2">Parametry które zostaną skopiowane:</p>
                  <ul className="space-y-1 text-xs">
                    <li><span className="text-muted-foreground">Rodzaj:</span> {variantSourceTemplate.furniture_type}</li>
                    <li><span className="text-muted-foreground">Cz1:</span> {variantSourceTemplate.cz1}</li>
                    <li><span className="text-muted-foreground">Cz2 (oryginał):</span> {variantSourceTemplate.cz2 || '---'}</li>
                    <li><span className="text-muted-foreground">Grubość:</span> {variantSourceTemplate.thickness}mm</li>
                    <li><span className="text-muted-foreground">Obrzeża:</span> {[
                      variantSourceTemplate.edge1 && 'Góra',
                      variantSourceTemplate.edge2 && 'Prawo',
                      variantSourceTemplate.edge3 && 'Dół',
                      variantSourceTemplate.edge4 && 'Lewo'
                    ].filter(Boolean).join(', ') || 'Brak'}</li>
                  </ul>
                </div>

                <div className="border rounded-md p-3 space-y-2">
                  <Label className="text-sm font-semibold">Wymiary wariantu</Label>
                  <div className="grid grid-cols-2 gap-2">
                    <DimensionVariantsDialog
                      type="length"
                      variants={variantAlternativeLengths}
                      onChange={setVariantAlternativeLengths}
                      furnitureType={variantSourceTemplate.furniture_type}
                      furnitureLength={variantSourceTemplate.base_length}
                      furnitureWidth={variantSourceTemplate.base_width}
                      furnitureLengthCondition={variantSourceTemplate.furniture_length_condition}
                      furnitureWidthCondition={variantSourceTemplate.furniture_width_condition}
                      cz1={variantSourceTemplate.cz1}
                      cz2={variantCz2 || variantSourceTemplate.cz2}
                      triggerButton={
                        <Button 
                          variant="outline" 
                          className="w-full justify-start gap-2 text-sm"
                          type="button"
                        >
                          <Settings2 className="w-4 h-4" />
                          <span className="flex-1 text-left">Długość ({variantAlternativeLengths.length})</span>
                        </Button>
                      }
                    />
                    <DimensionVariantsDialog
                      type="width"
                      variants={variantAlternativeWidths}
                      onChange={setVariantAlternativeWidths}
                      furnitureType={variantSourceTemplate.furniture_type}
                      furnitureLength={variantSourceTemplate.base_length}
                      furnitureWidth={variantSourceTemplate.base_width}
                      furnitureLengthCondition={variantSourceTemplate.furniture_length_condition}
                      furnitureWidthCondition={variantSourceTemplate.furniture_width_condition}
                      cz1={variantSourceTemplate.cz1}
                      cz2={variantCz2 || variantSourceTemplate.cz2}
                      triggerButton={
                        <Button 
                          variant="outline" 
                          className="w-full justify-start gap-2 text-sm"
                          type="button"
                        >
                          <Settings2 className="w-4 h-4" />
                          <span className="flex-1 text-left">Szerokość ({variantAlternativeWidths.length})</span>
                        </Button>
                      }
                    />
                  </div>
                  <p className="text-xs text-muted-foreground">
                    Możesz edytować wymiary dla tego wariantu lub zostawić je takie same jak oryginał
                  </p>
                </div>
              </div>
            )}
          </div>

          <DialogFooter>
            <Button 
              variant="outline" 
              onClick={() => {
                setShowVariantDialog(false);
                setVariantSourceTemplate(null);
                setVariantCz2('');
              }}
            >
              Anuluj
            </Button>
            <Button 
              onClick={() => {
                if (!variantCz2) {
                  toast({
                    title: "Błąd",
                    description: "Wybierz wartość cz2 dla wariantu",
                    variant: "destructive"
                  });
                  return;
                }
                if (variantSourceTemplate) {
                  createVariantMutation.mutate({
                    id: variantSourceTemplate.id,
                    cz2: variantCz2,
                    alternativeLengths: variantAlternativeLengths,
                    alternativeWidths: variantAlternativeWidths
                  });
                }
              }}
              disabled={createVariantMutation.isPending || !variantCz2}
            >
              {createVariantMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Tworzenie...
                </>
              ) : (
                'Utwórz wariant'
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </MatrixLayout>
  );
}

// Template Usage Badge Component
interface TemplateUsage {
  productId: number;
  productSku: string;
  productTitle: string;
  bomId: number;
}

function TemplateUsageBadge({ templateId }: { templateId: number }) {
  const [isDialogOpen, setIsDialogOpen] = useState(false);

  const { data: usages = [], isLoading } = useQuery<TemplateUsage[]>({
    queryKey: [`/api/component-templates/${templateId}/usages`],
    enabled: isDialogOpen, // Only fetch when dialog is open
  });

  if (isLoading) {
    return (
      <Badge variant="outline" className="cursor-pointer">
        <Loader2 className="w-3 h-3 animate-spin" />
      </Badge>
    );
  }

  const count = usages.length;

  return (
    <div className="inline-flex items-center">
      <Button
        variant="ghost"
        size="sm"
        className="h-6 px-2"
        onClick={() => setIsDialogOpen(true)}
        data-testid={`button-usages-${templateId}`}
      >
        {count > 0 ? (
          <Badge variant="secondary" className="gap-1">
            <GitBranch className="w-3 h-3" />
            {count}
          </Badge>
        ) : (
          <Badge variant="outline" className="text-muted-foreground">
            0
          </Badge>
        )}
      </Button>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[80vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle>Produkty używające formatki #{templateId}</DialogTitle>
            <DialogDescription>
              Lista produktów katalogowych które mają tę formatkę w swoim BOM
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-2">
            {count === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                Brak produktów używających tej formatki
              </div>
            ) : (
              <div className="space-y-2">
                {usages.map((usage) => (
                  <Card key={usage.productId} className="hover-elevate">
                    <CardContent className="p-4">
                      <div className="flex items-center justify-between">
                        <div className="flex-1">
                          <div className="font-medium">{usage.productTitle}</div>
                          <div className="text-sm text-muted-foreground">
                            SKU: {usage.productSku}
                          </div>
                        </div>
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => {
                            window.open(`/catalog-products/${usage.productId}`, '_blank');
                          }}
                          data-testid={`button-view-product-${usage.productId}`}
                        >
                          Otwórz
                        </Button>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            )}
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setIsDialogOpen(false)}
              data-testid="button-close-usages"
            >
              Zamknij
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

    </div>
  );
}
