import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { useLocation } from "wouter";
import { FileText, Banknote, Package, MessageSquare, Truck, ArrowUpDown, ArrowUp, ArrowDown, RefreshCw, ExternalLink, Copy, Check } from "lucide-react";
import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { apiRequest, queryClient } from "@/lib/queryClient";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import { Progress } from "@/components/ui/progress";
import { Badge } from "@/components/ui/badge";
import { StatusBadge } from "@/components/status-badge";
import { useToast } from "@/hooks/use-toast";
import type { AllegroOrder, OrderStatus, FulfillmentStatus } from "@shared/schema";

interface OrdersTableProps {
  orders: AllegroOrder[];
  isLoading?: boolean;
  sortBy?: string;
  sortOrder?: "ASC" | "DESC";
  onSort?: (column: string) => void;
}

export function OrdersTable({ 
  orders, 
  isLoading = false,
  sortBy,
  sortOrder,
  onSort
}: OrdersTableProps) {
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const [syncingOrderId, setSyncingOrderId] = useState<string | null>(null);
  const [copiedOrderId, setCopiedOrderId] = useState<string | null>(null);

  const copyOrderNumber = async (e: React.MouseEvent, orderNumber: string, orderId: string) => {
    e.stopPropagation();
    
    try {
      await navigator.clipboard.writeText(orderNumber);
      setCopiedOrderId(orderId);
      
      setTimeout(() => {
        setCopiedOrderId(null);
      }, 2000);
    } catch (err) {
      console.error('Błąd kopiowania:', err);
    }
  };

  // Pobierz konfigurację Odoo dla URL-i
  const { data: odooConfig } = useQuery<{
    configured: boolean;
    url?: string;
    user_url?: string;
  }>({
    queryKey: ["/api/odoo/config"],
  });

  // Pobierz konfigurację Shoper dla URL-i
  const { data: shoperConfig } = useQuery<{
    orders_url?: string;
    products_url?: string;
  }>({
    queryKey: ["/api/shoper/config"],
  });

  // Oblicz maksymalną wartość zamówienia dla paska postępu
  const maxOrderValue = orders.length > 0 
    ? Math.max(...orders.map(o => parseFloat((o as any).summary?.totalToPay?.amount || "0")))
    : 0;

  // Mutacja do synchronizacji zamówienia z Odoo
  const syncToOdooMutation = useMutation({
    mutationFn: async (orderId: string) => {
      const order = orders.find(o => o.id === orderId);
      if (!order) throw new Error("Zamówienie nie znalezione");
      
      return await apiRequest("POST", `/api/orders/${order.orderNumber}/sync-to-odoo`);
    },
    onSuccess: () => {
      toast({
        title: "Synchronizacja rozpoczęta",
        description: "Zamówienie zostało dodane do kolejki synchronizacji Odoo.",
      });
      queryClient.invalidateQueries({ queryKey: ["/api/orders"] });
      setSyncingOrderId(null);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd synchronizacji",
        description: error.message || "Nie udało się rozpocząć synchronizacji.",
        variant: "destructive",
      });
      setSyncingOrderId(null);
    },
  });

  const handleSyncToOdoo = (e: React.MouseEvent, orderId: string) => {
    e.stopPropagation();
    setSyncingOrderId(orderId);
    syncToOdooMutation.mutate(orderId);
  };

  const getOdooOrderUrl = (odooOrderId: number): string | null => {
    const odooUserUrl = odooConfig?.user_url || odooConfig?.url;
    if (!odooUserUrl || !odooOrderId) return null;
    const baseUrl = odooUserUrl.replace(/\/$/, '');
    return `${baseUrl}/sales/${odooOrderId}`;
  };

  const getPlatformUrl = (order: any): string | null => {
    if (order.source === 'ALLEGRO' && order.allegroOrderId) {
      return `https://allegro.pl/moje-allegro/sprzedaz/zamowienia/${order.allegroOrderId}`;
    } else if (order.source === 'SHOPER' && order.shoperOrderId) {
      const baseUrl = (shoperConfig?.orders_url || 'https://alpmeb.pl/admin/orders/').replace(/\/$/, '');
      return `${baseUrl}/${order.shoperOrderId}`;
    }
    return null;
  };

  const SortableHeader = ({ column, children, className = "" }: { column: string; children: React.ReactNode; className?: string }) => {
    const isSorted = sortBy === column;
    return (
      <TableHead 
        className={`cursor-pointer hover-elevate ${className}`}
        onClick={() => onSort?.(column)}
      >
        <div className="flex items-center gap-1">
          {children}
          {isSorted ? (
            sortOrder === 'ASC' ? <ArrowUp className="h-3 w-3" /> : <ArrowDown className="h-3 w-3" />
          ) : (
            <ArrowUpDown className="h-3 w-3 opacity-30" />
          )}
        </div>
      </TableHead>
    );
  };

  if (isLoading) {
    return (
      <div className="rounded-md border">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead className="w-[100px]">Zdjęcia</TableHead>
              <TableHead className="w-[100px]">Numer</TableHead>
              <TableHead className="w-[140px]">Data</TableHead>
              <TableHead className="w-[50px] text-center">Platforma</TableHead>
              <TableHead className="w-[100px]">Odoo #</TableHead>
              <TableHead>Kupujący</TableHead>
              <TableHead className="w-[60px] text-center">
                <Package className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[120px] text-right">Kwota</TableHead>
              <TableHead className="w-[180px]">Wartość</TableHead>
              <TableHead className="w-[140px]">Status płatności</TableHead>
              <TableHead className="w-[150px]">Status realizacji</TableHead>
              <TableHead className="w-[50px] text-center">
                <Banknote className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[50px] text-center">
                <FileText className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[50px] text-center">
                <Truck className="h-4 w-4 mx-auto" />
              </TableHead>
              <TableHead className="w-[50px] text-center">
                <MessageSquare className="h-4 w-4 mx-auto" />
              </TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {[1, 2, 3, 4, 5].map((i) => (
              <TableRow key={i}>
                <TableCell>
                  <div className="h-10 w-10 bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-4 w-32 bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-4 w-24 bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-4 w-8 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-4 w-20 bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-4 w-24 bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-8 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-right">
                  <div className="h-4 w-20 ml-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-2 w-full bg-muted rounded-full animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-6 w-24 bg-muted rounded-full animate-pulse" />
                </TableCell>
                <TableCell>
                  <div className="h-6 w-28 bg-muted rounded-full animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-4 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-4 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-4 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
                <TableCell className="text-center">
                  <div className="h-4 w-4 mx-auto bg-muted rounded animate-pulse" />
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </div>
    );
  }

  if (orders.length === 0) {
    return (
      <div className="rounded-md border p-12 text-center">
        <div className="mx-auto w-24 h-24 mb-4 rounded-full bg-muted flex items-center justify-center">
          <svg
            className="w-12 h-12 text-muted-foreground"
            fill="none"
            viewBox="0 0 24 24"
            stroke="currentColor"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              strokeWidth={1.5}
              d="M20 13V6a2 2 0 00-2-2H6a2 2 0 00-2 2v7m16 0v5a2 2 0 01-2 2H6a2 2 0 01-2-2v-5m16 0h-2.586a1 1 0 00-.707.293l-2.414 2.414a1 1 0 01-.707.293h-3.172a1 1 0 01-.707-.293l-2.414-2.414A1 1 0 006.586 13H4"
            />
          </svg>
        </div>
        <h3 className="text-lg font-semibold mb-2">Brak zamówień</h3>
        <p className="text-sm text-muted-foreground max-w-sm mx-auto">
          Nie znaleziono żadnych zamówień. Upewnij się, że połączenie z Allegro jest
          aktywne i zsynchronizuj dane.
        </p>
      </div>
    );
  }

  return (
    <div className="rounded-md border">
      <Table>
        <TableHeader>
          <TableRow>
            <TableHead className="w-[100px]">Zdjęcia</TableHead>
            <SortableHeader column="order_number" className="w-[140px]">Numer</SortableHeader>
            <SortableHeader column="order_date" className="w-[140px]">Data</SortableHeader>
            <SortableHeader column="total_to_pay_amount" className="w-[120px] text-right">Kwota</SortableHeader>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <div className="cursor-help text-xs font-semibold">Platforma</div>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Kliknij aby przejść do zamówienia na platformie</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <TableHead className="w-[100px]">Odoo #</TableHead>
            <SortableHeader column="buyer_login">Kupujący</SortableHeader>
            <TableHead className="w-[60px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Package className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Liczba produktów</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <TableHead className="w-[180px]">Wartość</TableHead>
            <SortableHeader column="payment_status" className="w-[140px]">Status płatności</SortableHeader>
            <SortableHeader column="status" className="w-[150px]">Status realizacji</SortableHeader>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Banknote className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Metoda płatności</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <FileText className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Faktura</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <Truck className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Przesyłka</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
            <TableHead className="w-[50px] text-center">
              <Tooltip>
                <TooltipTrigger asChild>
                  <MessageSquare className="h-4 w-4 mx-auto" />
                </TooltipTrigger>
                <TooltipContent>
                  <p>Wiadomość</p>
                </TooltipContent>
              </Tooltip>
            </TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {orders.map((order) => {
            const o = order as any;
            const buyerFirstName = o.buyer?.firstName || "";
            const buyerLastName = o.buyer?.lastName || "";
            const buyerLogin = o.buyer?.login || "";
            const fullName = buyerFirstName && buyerLastName 
              ? `${buyerFirstName} ${buyerLastName}` 
              : buyerLogin || "—";
            
            const lineItems = o.lineItems || [];
            const hasMessage = o.messageToSeller && o.messageToSeller.trim().length > 0;
            const isSynced = !!o.odooOrderId;
            const platformUrl = getPlatformUrl(o);
            const odooUrl = isSynced ? getOdooOrderUrl(o.odooOrderId) : null;
            
            return (
              <TableRow
                key={order.id}
                className="hover-elevate cursor-pointer"
                onClick={() => navigate(`/order/${order.orderNumber || order.id}`)}
                data-testid={`row-order-${order.orderNumber}`}
              >
                <TableCell>
                  <div className="flex gap-1">
                    {lineItems.slice(0, 3).map((item: any, idx: number) => (
                      <div key={idx} className="relative">
                        {item.imageUrl ? (
                          <img 
                            src={item.imageUrl}
                            alt={item.offer?.name || 'Produkt'}
                            className="h-10 w-10 object-cover rounded border"
                            onError={(e) => {
                              (e.target as HTMLImageElement).src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PHJlY3Qgd2lkdGg9IjQwIiBoZWlnaHQ9IjQwIiBmaWxsPSIjZjNmNGY2Ii8+PC9zdmc+';
                            }}
                          />
                        ) : (
                          <div className="h-10 w-10 bg-muted rounded border flex items-center justify-center">
                            <Package className="h-4 w-4 text-muted-foreground" />
                          </div>
                        )}
                      </div>
                    ))}
                    {lineItems.length > 3 && (
                      <div className="h-10 w-10 bg-muted rounded border flex items-center justify-center text-xs font-medium">
                        +{lineItems.length - 3}
                      </div>
                    )}
                  </div>
                </TableCell>
                <TableCell>
                  <Tooltip>
                    <TooltipTrigger asChild>
                      <div className="inline-block">
                        <Badge
                          className="cursor-pointer hover-elevate active-elevate-2 inline-flex items-center gap-2 bg-slate-600 dark:bg-slate-700 text-white font-bold border-0 text-lg"
                          onClick={(e) => copyOrderNumber(e, String(order.orderNumber || order.id), order.id)}
                          data-testid={`badge-order-number-${order.orderNumber}`}
                        >
                          <span>#{order.orderNumber || order.id}</span>
                          {copiedOrderId === order.id ? (
                            <Check className="h-4 w-4" />
                          ) : (
                            <Copy className="h-4 w-4" />
                          )}
                        </Badge>
                      </div>
                    </TooltipTrigger>
                    <TooltipContent>Kliknij aby skopiować numer zamówienia</TooltipContent>
                  </Tooltip>
                </TableCell>
                <TableCell>
                  <div className="flex flex-col">
                    <span className="text-sm font-medium">
                      {format(new Date(o.boughtAt || o.orderDate), "dd MMM yyyy", {
                        locale: pl,
                      })}
                    </span>
                    <span className="text-xs text-muted-foreground">
                      {format(new Date(o.boughtAt || o.orderDate), "HH:mm", {
                        locale: pl,
                      })}
                    </span>
                  </div>
                </TableCell>
                <TableCell className="text-right font-medium tabular-nums">
                  {(() => {
                    const paidAmount = o.payment?.paidAmount?.amount;
                    const isPaid = paidAmount && parseFloat(paidAmount) > 0;
                    const isCashOnDelivery = o.payment?.type === 'CASH_ON_DELIVERY';
                    const colorClass = isPaid 
                      ? 'text-green-600 dark:text-green-400' 
                      : isCashOnDelivery 
                        ? 'text-orange-600 dark:text-orange-400'
                        : 'text-red-600 dark:text-red-400';
                    return (
                      <span className={colorClass}>
                        {parseFloat(o.summary?.totalToPay?.amount || "0").toFixed(2)} {o.summary?.totalToPay?.currency || "PLN"}
                      </span>
                    );
                  })()}
                </TableCell>
                <TableCell className="text-center">
                  {platformUrl ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <a
                          href={platformUrl}
                          target="_blank"
                          rel="noopener noreferrer"
                          onClick={(e) => e.stopPropagation()}
                          className={`inline-flex items-center justify-center w-7 h-7 rounded-md text-xs font-bold hover-elevate active-elevate-2 cursor-pointer transition-colors ${
                            order.source === 'SHOPER' 
                              ? 'bg-blue-500/10 text-blue-600 dark:text-blue-400' 
                              : 'bg-orange-500/10 text-orange-600 dark:text-orange-400'
                          }`}
                          data-testid={`link-platform-${order.orderNumber}`}
                        >
                          {order.source === 'SHOPER' ? 'S' : 'A'}
                        </a>
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Otwórz zamówienie w {order.source === 'SHOPER' ? 'Shoper' : 'Allegro'}</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <span className={`inline-flex items-center justify-center w-7 h-7 rounded-md text-xs font-bold ${
                          order.source === 'SHOPER' 
                            ? 'bg-blue-500/10 text-blue-600 dark:text-blue-400' 
                            : 'bg-orange-500/10 text-orange-600 dark:text-orange-400'
                        }`}>
                          {order.source === 'SHOPER' ? 'S' : 'A'}
                        </span>
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Skonfiguruj URL {order.source === 'SHOPER' ? 'Shoper' : 'Allegro'} w ustawieniach</p>
                      </TooltipContent>
                    </Tooltip>
                  )}
                </TableCell>
                <TableCell>
                  {isSynced ? (
                    odooUrl ? (
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <a
                            href={odooUrl}
                            target="_blank"
                            rel="noopener noreferrer"
                            onClick={(e) => e.stopPropagation()}
                            className="inline-flex items-center gap-1 text-purple-600 dark:text-purple-400 hover:underline font-mono text-sm"
                            data-testid={`link-odoo-order-${order.orderNumber}`}
                          >
                            #{o.odooOrderId}
                            <ExternalLink className="h-3 w-3" />
                          </a>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p>Otwórz zamówienie w Odoo</p>
                        </TooltipContent>
                      </Tooltip>
                    ) : (
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <span className="inline-flex items-center gap-1 text-purple-600 dark:text-purple-400 font-mono text-sm cursor-help">
                            #{o.odooOrderId}
                          </span>
                        </TooltipTrigger>
                        <TooltipContent>
                          <p>Skonfiguruj URL Odoo aby otworzyć zamówienie</p>
                        </TooltipContent>
                      </Tooltip>
                    )
                  ) : (
                    <span className="text-muted-foreground text-sm">—</span>
                  )}
                </TableCell>
                <TableCell>
                  <div>
                    <div
                      className="font-medium"
                      data-testid={`text-buyer-${order.allegroOrderId}`}
                    >
                      {fullName}
                    </div>
                    {buyerLogin && buyerLogin !== fullName && (
                      <div className="text-xs text-muted-foreground">
                        {buyerLogin}
                      </div>
                    )}
                  </div>
                </TableCell>
                <TableCell className="text-center tabular-nums">
                  {lineItems.length || 0}
                </TableCell>
                <TableCell>
                  {(() => {
                    const currentValue = parseFloat(o.summary?.totalToPay?.amount || "0");
                    const percentage = maxOrderValue > 0 ? (currentValue / maxOrderValue) * 100 : 0;
                    return (
                      <div className="space-y-1">
                        <Progress 
                          value={percentage} 
                          className="h-2"
                          data-testid={`progress-order-value-${order.allegroOrderId}`}
                        />
                        <div className="text-xs text-muted-foreground text-center">
                          {percentage.toFixed(0)}%
                        </div>
                      </div>
                    );
                  })()}
                </TableCell>
                <TableCell>
                  <StatusBadge
                    type="payment"
                    status={(() => {
                      const paidAmount = o.payment?.paidAmount?.amount;
                      const isPaid = paidAmount && parseFloat(paidAmount) > 0;
                      const isCashOnDelivery = o.payment?.type === 'CASH_ON_DELIVERY';
                      const refundAmount = parseFloat(o.payment?.refundAmount || "0");
                      const hasRefund = refundAmount > 0;
                      const paidAmountValue = isPaid ? parseFloat(paidAmount) : 0;
                      const netPaidAmount = paidAmountValue - refundAmount;
                      
                      if (hasRefund && netPaidAmount === 0) return 'REFUNDED';
                      if (isPaid) return 'PAID';
                      if (isCashOnDelivery) return 'CASH_ON_DELIVERY';
                      return 'UNPAID';
                    })() as OrderStatus}
                  />
                </TableCell>
                <TableCell>
                  <StatusBadge
                    type="fulfillment"
                    status={(o.status?.toUpperCase() || 'NEW') as FulfillmentStatus}
                  />
                </TableCell>
                <TableCell className="text-center">
                  {o.payment?.type === "CASH_ON_DELIVERY" ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Banknote className="h-4 w-4 mx-auto text-amber-600" data-testid={`icon-payment-cod-${order.allegroOrderId}`} />
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Pobranie</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : o.payment?.type === "ONLINE" ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Banknote className="h-4 w-4 mx-auto text-green-600" data-testid={`icon-payment-online-${order.allegroOrderId}`} />
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Płatność online</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <span className="text-muted-foreground">—</span>
                  )}
                </TableCell>
                <TableCell className="text-center">
                  {o.invoice?.required ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <FileText className="h-4 w-4 mx-auto text-blue-600" data-testid={`icon-invoice-${order.allegroOrderId}`} />
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Z fakturą</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <span className="text-muted-foreground">—</span>
                  )}
                </TableCell>
                <TableCell className="text-center">
                  {(() => {
                    const shipmentList = typeof o.delivery?.shipments === 'string' 
                      ? (o.delivery.shipments ? JSON.parse(o.delivery.shipments) : [])
                      : (o.delivery?.shipments || []);
                    const hasTracking = shipmentList.some((s: any) => s.waybill);
                    const trackingNumbers = shipmentList.filter((s: any) => s.waybill).map((s: any) => s.waybill).join(', ');
                    
                    if (hasTracking) {
                      return (
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Truck className="h-4 w-4 mx-auto text-green-600" data-testid={`icon-tracking-${order.allegroOrderId}`} />
                          </TooltipTrigger>
                          <TooltipContent>
                            <p>Nr przesyłki: {trackingNumbers}</p>
                          </TooltipContent>
                        </Tooltip>
                      );
                    } else if (o.delivery?.smart) {
                      return (
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Truck className="h-4 w-4 mx-auto text-muted-foreground" data-testid={`icon-smart-${order.allegroOrderId}`} />
                          </TooltipTrigger>
                          <TooltipContent>
                            <p>Smart! - {o.delivery?.method?.name || "Allegro Delivery"}</p>
                          </TooltipContent>
                        </Tooltip>
                      );
                    } else if (o.delivery?.method?.name) {
                      return (
                        <Tooltip>
                          <TooltipTrigger asChild>
                            <Truck className="h-4 w-4 mx-auto text-muted-foreground" />
                          </TooltipTrigger>
                          <TooltipContent>
                            <p>{o.delivery.method.name}</p>
                          </TooltipContent>
                        </Tooltip>
                      );
                    } else {
                      return <span className="text-muted-foreground">—</span>;
                    }
                  })()}
                </TableCell>
                <TableCell className="text-center">
                  {hasMessage ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <MessageSquare className="h-4 w-4 mx-auto text-blue-600" data-testid={`icon-message-${order.allegroOrderId}`} />
                      </TooltipTrigger>
                      <TooltipContent className="max-w-xs">
                        <p className="text-sm">{o.messageToSeller}</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <span className="text-muted-foreground">—</span>
                  )}
                </TableCell>
              </TableRow>
            );
          })}
        </TableBody>
      </Table>
    </div>
  );
}
