import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Search, Link as LinkIcon, Loader2, Package, X, Copy, Unlink, Boxes, ExternalLink } from "lucide-react";
import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";

interface MarketplaceLinkDialogProps {
  open: boolean;
  onOpenChange: (open: boolean) => void;
  mode: 'catalog-to-marketplace' | 'marketplace-to-catalog';
  catalogProductId?: number;
  marketplaceProduct?: {
    id: number;
    externalId: string;
    platform: 'allegro' | 'shoper';
    sku: string;
    name: string;
  };
  existingLinks?: Array<{
    id: number;
    platform: string;
    externalId: string;
    marketplaceSku?: string;
    marketplaceName?: string;
    linkType?: 'product' | 'set';
    catalogProductId?: number | null;
    catalogSetId?: number | null;
  }>;
  onSuccess?: () => void;
}

interface MarketplaceSuggestion {
  id: number;
  external_id: string;
  platform: string;
  sku: string;
  name: string;
}

interface CatalogSuggestion {
  id: number;
  sku: string;
  title: string;
  color?: string | null;
}

interface SetSuggestion {
  id: number;
  sku: string;
  title: string;
  color?: string | null;
  panel_count?: number | null;
  hook_length?: string | null;
  base_price?: string | null;
  calculated_price?: string | null;
  images?: string[];
}

export function MarketplaceLinkDialog({
  open,
  onOpenChange,
  mode,
  catalogProductId,
  marketplaceProduct,
  existingLinks = [],
  onSuccess,
}: MarketplaceLinkDialogProps) {
  const { toast } = useToast();
  const [activeTab, setActiveTab] = useState<'products' | 'sets'>('products');
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedProduct, setSelectedProduct] = useState<{
    id: number;
    externalId: string;
    platform: string;
    sku: string;
    name: string;
    linkType?: 'product' | 'set';
  } | null>(null);
  const [confirmDelete, setConfirmDelete] = useState<number | null>(null);
  const [deletingLinkId, setDeletingLinkId] = useState<number | null>(null);

  // Reset state when dialog closes
  useEffect(() => {
    if (!open) {
      setSearchQuery("");
      setSelectedProduct(null);
    }
  }, [open]);

  // Query for marketplace products (catalog → marketplace mode)
  const { data: marketplaceSuggestions = [], isLoading: marketplaceLoading } = useQuery<MarketplaceSuggestion[]>({
    queryKey: ["/api/commerce/marketplace-products", searchQuery],
    queryFn: async () => {
      if (!searchQuery || searchQuery.length < 2) return [];
      const res = await fetch(`/api/commerce/marketplace-products?search=${encodeURIComponent(searchQuery)}`);
      if (!res.ok) throw new Error('Failed to fetch marketplace products');
      return res.json();
    },
    enabled: mode === 'catalog-to-marketplace' && !!searchQuery && searchQuery.length >= 2 && open,
  });

  // Query for catalog products (marketplace → catalog mode, products tab)
  const { data: catalogSuggestions = [], isLoading: catalogLoading } = useQuery<CatalogSuggestion[]>({
    queryKey: ["/api/catalog/products/suggest-by-sku", searchQuery],
    queryFn: async () => {
      if (!searchQuery || searchQuery.length < 2) return [];
      const res = await fetch(`/api/catalog/products/suggest-by-sku?sku=${encodeURIComponent(searchQuery)}&limit=20`);
      if (!res.ok) throw new Error('Failed to fetch catalog products');
      return res.json();
    },
    enabled: mode === 'marketplace-to-catalog' && activeTab === 'products' && !!searchQuery && searchQuery.length >= 2 && open,
  });

  // Query for catalog sets (marketplace → catalog mode, sets tab)
  const { data: setSuggestions = [], isLoading: setsLoading } = useQuery<SetSuggestion[]>({
    queryKey: ["/api/catalog/product-sets/suggest", searchQuery],
    queryFn: async () => {
      if (!searchQuery || searchQuery.length < 2) return [];
      const res = await fetch(`/api/catalog/product-sets/suggest?query=${encodeURIComponent(searchQuery)}&limit=20`);
      if (!res.ok) throw new Error('Failed to fetch catalog sets');
      return res.json();
    },
    enabled: mode === 'marketplace-to-catalog' && activeTab === 'sets' && !!searchQuery && searchQuery.length >= 2 && open,
  });

  // Query for existing platform links (catalog → marketplace mode)
  const { data: fetchedLinks } = useQuery<Array<{
    id: number;
    platform: string;
    externalId: string;
    marketplaceSku?: string;
    marketplaceName?: string;
    linkType?: 'product' | 'set';
    catalogProductId?: number | null;
    catalogSetId?: number | null;
  }>>({
    queryKey: [`/api/catalog-products/${catalogProductId}`],
    queryFn: async () => {
      if (!catalogProductId) return [];
      const res = await fetch(`/api/catalog-products/${catalogProductId}`);
      if (!res.ok) return [];
      const data = await res.json();
      // Transform platformData to existingLinks format with link type and catalog IDs
      return (data.platformData || []).map((pd: any) => ({
        id: pd.id,
        platform: pd.platform,
        externalId: pd.externalId,
        marketplaceSku: pd.marketplaceSku,
        marketplaceName: pd.marketplaceName,
        linkType: pd.linkType || 'product',
        catalogProductId: pd.catalogProductId || null,
        catalogSetId: pd.catalogSetId || null,
      }));
    },
    enabled: mode === 'catalog-to-marketplace' && !!catalogProductId && open,
  });

  // Use fetched links or passed links - ensure it's always an array
  const activeLinks = (mode === 'catalog-to-marketplace' ? fetchedLinks : existingLinks) || [];
  
  // DEBUG: Log activeLinks to verify data structure
  console.log('🔍 [MarketplaceLinkDialog] activeLinks:', JSON.stringify(activeLinks, null, 2));

  // Mutation to delete link
  const deleteLinkMutation = useMutation({
    mutationFn: async (linkId: number) => {
      return apiRequest("DELETE", `/api/catalog-products/platform-links/${linkId}`, {});
    },
    onSuccess: () => {
      // Invalidate queries for both catalog and marketplace
      if (catalogProductId) {
        queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${catalogProductId}`] });
        queryClient.invalidateQueries({ queryKey: ["/api/catalog-products"] });
      }
      if (marketplaceProduct) {
        queryClient.invalidateQueries({ queryKey: [`/api/marketplace-products/${marketplaceProduct.sku}`] });
        queryClient.invalidateQueries({ queryKey: ["/api/products"] });
      }
      
      toast({
        title: "✅ Połączenie usunięte",
        description: "Produkt został odłączony od marketplace",
      });
      
      setConfirmDelete(null);
      setDeletingLinkId(null);
      onSuccess?.();
    },
    onError: (error: any) => {
      toast({
        title: "❌ Błąd usuwania",
        description: error.message || "Nie udało się usunąć połączenia",
        variant: "destructive",
      });
      setDeletingLinkId(null);
    },
  });

  // Mutation to link products or sets
  const linkMutation = useMutation({
    mutationFn: async () => {
      if (!selectedProduct) {
        throw new Error("No product/set selected");
      }

      if (mode === 'catalog-to-marketplace') {
        if (!catalogProductId) throw new Error("Catalog product ID is required");
        return apiRequest("POST", '/api/catalog/products/link-marketplace', {
          catalogProductId,
          externalId: selectedProduct.externalId,
          platform: selectedProduct.platform.toLowerCase(),
          linkType: 'product',
        });
      } else {
        if (!marketplaceProduct) throw new Error("Marketplace product is required");
        
        // Determine link type based on active tab
        const linkType = selectedProduct.linkType || (activeTab === 'sets' ? 'set' : 'product');
        
        const payload: any = {
          externalId: marketplaceProduct.externalId,
          platform: marketplaceProduct.platform.toLowerCase(),
          linkType,
        };
        
        // Add either catalogProductId or catalogSetId based on link type
        if (linkType === 'set') {
          payload.catalogSetId = selectedProduct.id;
        } else {
          payload.catalogProductId = selectedProduct.id;
        }
        
        return apiRequest("POST", '/api/catalog/products/link-marketplace', payload);
      }
    },
    onSuccess: () => {
      // Invalidate both catalog and marketplace queries
      if (catalogProductId) {
        queryClient.invalidateQueries({ queryKey: [`/api/catalog-products/${catalogProductId}`] });
        queryClient.invalidateQueries({ queryKey: ["/api/catalog-products"] });
      }
      if (marketplaceProduct) {
        queryClient.invalidateQueries({ queryKey: [`/api/marketplace-products/${marketplaceProduct.sku}`] });
        queryClient.invalidateQueries({ queryKey: ["/api/products"] });
      }
      
      const entityType = activeTab === 'sets' ? 'Zestaw katalogowy' : 'Produkt katalogowy';
      toast({
        title: "✅ Połączenie utworzone",
        description: mode === 'catalog-to-marketplace' 
          ? "Produkt marketplace został połączony z produktem katalogowym."
          : `${entityType} został połączony z produktem marketplace.`,
      });
      
      onOpenChange(false);
      onSuccess?.();
    },
    onError: (error: any) => {
      toast({
        title: "❌ Błąd połączenia",
        description: error.message || "Nie udało się połączyć produktów.",
        variant: "destructive",
      });
    },
  });

  const handleDelete = (linkId: number) => {
    setDeletingLinkId(linkId);
    deleteLinkMutation.mutate(linkId);
  };

  // Determine current suggestions and loading state based on mode and active tab
  const suggestions = mode === 'catalog-to-marketplace' 
    ? marketplaceSuggestions 
    : (activeTab === 'sets' ? setSuggestions : catalogSuggestions);
  
  const isLoading = mode === 'catalog-to-marketplace' 
    ? marketplaceLoading 
    : (activeTab === 'sets' ? setsLoading : catalogLoading);

  // Parse active filters from search query
  const activeFilters = searchQuery
    ? searchQuery.includes(';')
      ? searchQuery.split(';').map(t => t.trim()).filter(t => t)
      : searchQuery.includes(',')
      ? searchQuery.split(',').map(t => t.trim()).filter(t => t)
      : [searchQuery.trim()]
    : [];

  return (
    <>
    <Dialog open={open} onOpenChange={onOpenChange}>
      <DialogContent className="max-w-2xl">
        <DialogHeader>
          <DialogTitle>
            {mode === 'catalog-to-marketplace' 
              ? `Powiązania marketplace (${activeLinks.length})`
              : 'Połącz z produktem katalogowym'}
          </DialogTitle>
          <DialogDescription>
            {mode === 'catalog-to-marketplace'
              ? 'Zarządzaj połączeniami z ofertami marketplace (Allegro/Shoper)'
              : 'Znajdź i połącz ten produkt marketplace z produktem katalogowym'}
          </DialogDescription>
        </DialogHeader>

        <div className="space-y-4">
          {/* Existing links (only for catalog → marketplace) */}
          {mode === 'catalog-to-marketplace' && activeLinks.length > 0 && (
            <div className="border rounded-lg p-4 bg-muted/30">
              <h4 className="text-sm font-semibold mb-3">Aktualne połączenia:</h4>
              <div className="space-y-2">
                {activeLinks.map((link) => {
                  // Determine catalog link href based on linkType
                  const catalogHref = link.linkType === 'set' && link.catalogSetId
                    ? `/catalog-sets/${link.catalogSetId}`
                    : link.linkType === 'product' && link.catalogProductId
                    ? `/catalog-products/${link.catalogProductId}`
                    : null;

                  return (
                    <div 
                      key={`${link.platform}-${link.externalId}`}
                      className="flex items-center gap-2 p-2 bg-background rounded border"
                      data-testid={`marketplace-link-${link.platform}`}
                    >
                      <Badge variant="secondary" className="shrink-0">
                        {link.platform === 'allegro' ? 'Allegro' : link.platform === 'shoper' ? 'Shoper' : link.platform}
                      </Badge>
                      {link.linkType && (
                        <Badge variant="outline" className="shrink-0 text-xs">
                          {link.linkType === 'set' ? <Boxes className="h-3 w-3 mr-1" /> : <Package className="h-3 w-3 mr-1" />}
                          {link.linkType === 'set' ? 'Zestaw' : 'Produkt'}
                        </Badge>
                      )}
                      <div className="flex-1 text-sm">
                        <span className="font-medium">SKU: {link.marketplaceSku || link.externalId}</span>
                        {link.marketplaceName && (
                          <span className="text-muted-foreground ml-2">- {link.marketplaceName}</span>
                        )}
                      </div>
                      {catalogHref && (
                        <Link href={catalogHref} asChild>
                          <Button
                            size="sm"
                            variant="ghost"
                            title={`Otwórz ${link.linkType === 'set' ? 'zestaw' : 'produkt'} katalogowy`}
                            data-testid={`button-view-catalog-${link.id}`}
                            className="h-8 px-2"
                          >
                            <ExternalLink className="h-4 w-4" />
                          </Button>
                        </Link>
                      )}
                      <Button
                        size="sm"
                        variant="ghost"
                        onClick={() => setConfirmDelete(link.id)}
                        disabled={deletingLinkId === link.id}
                        data-testid={`button-unlink-${link.id}`}
                        title="Rozłącz"
                        className="h-8 px-2 text-destructive hover:text-destructive hover:bg-destructive/10"
                      >
                        <Unlink className="h-4 w-4" />
                      </Button>
                    </div>
                  );
                })}
              </div>
            </div>
          )}

          {/* Search input for catalog → marketplace mode */}
          {mode === 'catalog-to-marketplace' && (
            <div className="border-t pt-4">
              <label className="text-sm font-medium mb-2 block">
                Szukaj i dodaj nowe połączenie
              </label>
              <Input
                placeholder="Wpisz SKU lub nazwę produktu marketplace..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                data-testid="input-search-product"
              />
              <p className="text-xs text-muted-foreground mt-1.5">
                Wskazówka: Użyj <code className="px-1 py-0.5 bg-muted rounded text-xs">;</code> dla filtrowania AND
              </p>
            </div>
          )}

          {/* Tabs for marketplace → catalog mode */}
          {mode === 'marketplace-to-catalog' && (
            <Tabs value={activeTab} onValueChange={(v) => setActiveTab(v as 'products' | 'sets')} className="border-t pt-4">
              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="products" data-testid="tab-products">
                  <Package className="h-4 w-4 mr-2" />
                  Produkty
                </TabsTrigger>
                <TabsTrigger value="sets" data-testid="tab-sets">
                  <Boxes className="h-4 w-4 mr-2" />
                  Zestawy
                </TabsTrigger>
              </TabsList>

              {/* Products tab content */}
              <TabsContent value="products" className="space-y-4 mt-4">
                <div>
                  <label className="text-sm font-medium mb-2 block">
                    Szukaj produkt katalogowy po SKU lub nazwie
                  </label>
                  <Input
                    placeholder="Wpisz SKU lub nazwę produktu katalogowego..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    data-testid="input-search-product"
                  />
                  <p className="text-xs text-muted-foreground mt-1.5">
                    Wskazówka: Użyj <code className="px-1 py-0.5 bg-muted rounded text-xs">;</code> dla filtrowania AND
                  </p>

                  {/* Active filters badges */}
                  {activeFilters.length > 0 && searchQuery.includes(';') && (
                    <div className="flex flex-wrap gap-1.5 mt-2">
                      <span className="text-xs text-muted-foreground">Aktywne filtry (AND):</span>
                      {activeFilters.map((filter, index) => (
                        <Badge key={index} variant="secondary" className="text-xs">
                          {filter}
                        </Badge>
                      ))}
                    </div>
                  )}
                  {activeFilters.length > 0 && searchQuery.includes(',') && (
                    <div className="flex flex-wrap gap-1.5 mt-2">
                      <span className="text-xs text-muted-foreground">Aktywne filtry (OR):</span>
                      {activeFilters.map((filter, index) => (
                        <Badge key={index} variant="outline" className="text-xs">
                          {filter}
                        </Badge>
                      ))}
                    </div>
                  )}
                </div>
              </TabsContent>

              {/* Sets tab content */}
              <TabsContent value="sets" className="space-y-4 mt-4">
                <div>
                  <label className="text-sm font-medium mb-2 block">
                    Szukaj zestaw katalogowy po SKU lub nazwie
                  </label>
                  <Input
                    placeholder="Wpisz SKU lub nazwę zestawu katalogowego..."
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                    data-testid="input-search-set"
                  />
                  <p className="text-xs text-muted-foreground mt-1.5">
                    Wskazówka: Użyj <code className="px-1 py-0.5 bg-muted rounded text-xs">;</code> dla filtrowania AND
                  </p>

                  {/* Active filters badges */}
                  {activeFilters.length > 0 && searchQuery.includes(';') && (
                    <div className="flex flex-wrap gap-1.5 mt-2">
                      <span className="text-xs text-muted-foreground">Aktywne filtry (AND):</span>
                      {activeFilters.map((filter, index) => (
                        <Badge key={index} variant="secondary" className="text-xs">
                          {filter}
                        </Badge>
                      ))}
                    </div>
                  )}
                  {activeFilters.length > 0 && searchQuery.includes(',') && (
                    <div className="flex flex-wrap gap-1.5 mt-2">
                      <span className="text-xs text-muted-foreground">Aktywne filtry (OR):</span>
                      {activeFilters.map((filter, index) => (
                        <Badge key={index} variant="outline" className="text-xs">
                          {filter}
                        </Badge>
                      ))}
                    </div>
                  )}
                </div>
              </TabsContent>
            </Tabs>
          )}

          {/* Loading state */}
          {isLoading && (
            <div className="flex justify-center p-4">
              <Loader2 className="h-6 w-6 animate-spin text-muted-foreground" />
            </div>
          )}

          {/* Marketplace suggestions (catalog → marketplace) */}
          {mode === 'catalog-to-marketplace' && !isLoading && searchQuery && suggestions.length > 0 && (
            <div className="space-y-2 max-h-64 overflow-y-auto border-t pt-4">
              <div className="text-sm font-medium mb-2">Wyniki wyszukiwania ({suggestions.length}):</div>
              {(suggestions as MarketplaceSuggestion[]).map((suggestion) => (
                <div
                  key={`${suggestion.platform}-${suggestion.external_id}`}
                  className={`p-3 border rounded cursor-pointer hover-elevate ${
                    selectedProduct?.id === suggestion.id
                      ? 'border-primary bg-primary/5'
                      : ''
                  }`}
                  onClick={() => setSelectedProduct({
                    id: suggestion.id,
                    externalId: suggestion.external_id,
                    platform: suggestion.platform,
                    sku: suggestion.sku,
                    name: suggestion.name,
                  })}
                  data-testid={`suggestion-${suggestion.id}`}
                >
                  <div className="flex items-start gap-2 mb-1">
                    <Badge variant="secondary" className="mt-0.5">
                      {suggestion.platform === 'allegro' ? 'Allegro' : suggestion.platform === 'shoper' ? 'Shoper' : suggestion.platform}
                    </Badge>
                    <div className="flex-1">
                      <div className="flex items-start gap-2 mb-1">
                        <div className="flex-1 font-medium text-sm line-clamp-2">
                          {suggestion.name}
                        </div>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={(e) => {
                            e.stopPropagation();
                            setSearchQuery(suggestion.name);
                          }}
                          className="h-6 w-6 p-0 shrink-0"
                          title="Kopiuj nazwę do wyszukiwarki"
                        >
                          <Copy className="h-3 w-3" />
                        </Button>
                      </div>
                      <div className="flex items-center gap-2">
                        <button
                          onClick={(e) => {
                            e.stopPropagation();
                            setSearchQuery(suggestion.sku);
                          }}
                          className="text-xs text-muted-foreground hover:text-primary hover:underline"
                          title="Kopiuj SKU do wyszukiwarki"
                        >
                          SKU: {suggestion.sku}
                        </button>
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}

          {/* Catalog products suggestions (marketplace → catalog, products tab) */}
          {mode === 'marketplace-to-catalog' && activeTab === 'products' && !isLoading && searchQuery && suggestions.length > 0 && (
            <div className="space-y-2 max-h-64 overflow-y-auto border-t pt-4">
              <div className="text-sm font-medium mb-2">Produkty ({suggestions.length}):</div>
              {(suggestions as CatalogSuggestion[]).map((suggestion) => (
                <div
                  key={suggestion.id}
                  className={`p-3 border rounded cursor-pointer hover-elevate ${
                    selectedProduct?.id === suggestion.id
                      ? 'border-primary bg-primary/5'
                      : ''
                  }`}
                  onClick={() => setSelectedProduct({
                    id: suggestion.id,
                    externalId: '', // Not needed for catalog products
                    platform: '', // Not needed for catalog products
                    sku: suggestion.sku,
                    name: suggestion.title,
                    linkType: 'product', // Mark as product
                  })}
                  data-testid={`suggestion-${suggestion.id}`}
                >
                  <div className="flex items-start gap-2 mb-1">
                    <Package className="h-4 w-4 text-muted-foreground mt-0.5" />
                    <div className="flex-1">
                      <div className="flex items-start gap-2 mb-1">
                        <div className="flex-1 font-medium text-sm line-clamp-2">
                          {suggestion.title}
                        </div>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={(e) => {
                            e.stopPropagation();
                            setSearchQuery(suggestion.title);
                          }}
                          className="h-6 w-6 p-0 shrink-0"
                          title="Kopiuj nazwę do wyszukiwarki"
                        >
                          <Copy className="h-3 w-3" />
                        </Button>
                      </div>
                      <div className="flex items-center gap-2">
                        <button
                          onClick={(e) => {
                            e.stopPropagation();
                            setSearchQuery(suggestion.sku);
                          }}
                          className="text-xs text-muted-foreground hover:text-primary hover:underline"
                          title="Kopiuj SKU do wyszukiwarki"
                        >
                          SKU: {suggestion.sku}
                        </button>
                        {suggestion.color && (
                          <Badge variant="outline" className="text-xs">
                            {suggestion.color}
                          </Badge>
                        )}
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}

          {/* Catalog sets suggestions (marketplace → catalog, sets tab) */}
          {mode === 'marketplace-to-catalog' && activeTab === 'sets' && !isLoading && searchQuery && suggestions.length > 0 && (
            <div className="space-y-2 max-h-64 overflow-y-auto border-t pt-4">
              <div className="text-sm font-medium mb-2">Zestawy ({suggestions.length}):</div>
              {(suggestions as SetSuggestion[]).map((suggestion) => (
                <div
                  key={suggestion.id}
                  className={`p-3 border rounded cursor-pointer hover-elevate ${
                    selectedProduct?.id === suggestion.id
                      ? 'border-primary bg-primary/5'
                      : ''
                  }`}
                  onClick={() => setSelectedProduct({
                    id: suggestion.id,
                    externalId: '', // Not needed for sets
                    platform: '', // Not needed for sets
                    sku: suggestion.sku,
                    name: suggestion.title,
                    linkType: 'set', // Mark as set
                  })}
                  data-testid={`suggestion-set-${suggestion.id}`}
                >
                  <div className="flex items-start gap-2 mb-1">
                    <Boxes className="h-4 w-4 text-muted-foreground mt-0.5" />
                    <div className="flex-1">
                      <div className="flex items-start gap-2 mb-1">
                        <div className="flex-1 font-medium text-sm line-clamp-2">
                          {suggestion.title}
                        </div>
                        <Button
                          size="sm"
                          variant="ghost"
                          onClick={(e) => {
                            e.stopPropagation();
                            setSearchQuery(suggestion.title);
                          }}
                          className="h-6 w-6 p-0 shrink-0"
                          title="Kopiuj nazwę do wyszukiwarki"
                        >
                          <Copy className="h-3 w-3" />
                        </Button>
                      </div>
                      <div className="flex items-center gap-2 flex-wrap">
                        <button
                          onClick={(e) => {
                            e.stopPropagation();
                            setSearchQuery(suggestion.sku);
                          }}
                          className="text-xs text-muted-foreground hover:text-primary hover:underline"
                          title="Kopiuj SKU do wyszukiwarki"
                        >
                          SKU: {suggestion.sku}
                        </button>
                        {suggestion.panel_count && (
                          <Badge variant="outline" className="text-xs">
                            {suggestion.panel_count} paneli
                          </Badge>
                        )}
                        {suggestion.color && (
                          <Badge variant="outline" className="text-xs">
                            {suggestion.color}
                          </Badge>
                        )}
                        {suggestion.base_price && (
                          <Badge variant="secondary" className="text-xs">
                            {Number(suggestion.base_price).toFixed(2)} zł
                          </Badge>
                        )}
                      </div>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          )}

          {/* Empty state */}
          {!isLoading && searchQuery && suggestions.length === 0 && searchQuery.length >= 2 && (
            <div className="text-center p-6 text-muted-foreground">
              <Search className="h-12 w-12 mx-auto mb-2 opacity-50" />
              <p>
                {mode === 'catalog-to-marketplace'
                  ? 'Nie znaleziono produktów marketplace'
                  : activeTab === 'sets'
                  ? 'Nie znaleziono zestawów katalogowych'
                  : 'Nie znaleziono produktów katalogowych'}
              </p>
              <p className="text-sm mt-2">Spróbuj innej frazy wyszukiwania</p>
            </div>
          )}

          {/* Action buttons */}
          <div className="flex justify-end gap-2 pt-4 border-t">
            <Button
              variant="outline"
              onClick={() => onOpenChange(false)}
              data-testid="button-cancel"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => linkMutation.mutate()}
              disabled={!selectedProduct || linkMutation.isPending}
              data-testid="button-confirm"
            >
              {linkMutation.isPending && (
                <Loader2 className="mr-2 h-4 w-4 animate-spin" />
              )}
              <LinkIcon className="mr-2 h-4 w-4" />
              Połącz
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>

    {/* Confirm delete dialog */}
    <AlertDialog open={confirmDelete !== null} onOpenChange={(open) => !open && setConfirmDelete(null)}>
      <AlertDialogContent>
        <AlertDialogHeader>
          <AlertDialogTitle>Czy na pewno chcesz rozłączyć te produkty?</AlertDialogTitle>
          <AlertDialogDescription>
            Ta operacja usunie połączenie między produktem katalogowym a produktem marketplace.
            Synchronizacja zamówień nie będzie już działać dla tego produktu.
          </AlertDialogDescription>
        </AlertDialogHeader>
        <AlertDialogFooter>
          <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
          <AlertDialogAction
            onClick={() => confirmDelete && handleDelete(confirmDelete)}
            className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
            data-testid="button-confirm-delete"
          >
            Rozłącz
          </AlertDialogAction>
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  </>
  );
}
