import { Pool } from "pg";
import type { 
  ProductionCarrierGroup, 
  InsertProductionCarrierGroup,
  ProductionCarrier,
  InsertProductionCarrier
} from "@shared/schema";

// Carrier Groups
export async function getCarrierGroups(pool: Pool): Promise<ProductionCarrierGroup[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_carrier_groups
    ORDER BY sort_order, name ASC
  `);
  
  return result.rows.map(row => ({
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    defaultCapacity: row.default_capacity,
    capacityUnit: row.capacity_unit,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getCarrierGroupById(pool: Pool, id: number): Promise<ProductionCarrierGroup | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_carrier_groups
    WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    defaultCapacity: row.default_capacity,
    capacityUnit: row.capacity_unit,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createCarrierGroup(pool: Pool, data: InsertProductionCarrierGroup): Promise<ProductionCarrierGroup> {
  const result = await pool.query(`
    INSERT INTO production.production_carrier_groups 
    (code, name, description, default_capacity, capacity_unit, is_active, sort_order)
    VALUES ($1, $2, $3, $4, $5, $6, $7)
    RETURNING *
  `, [
    data.code,
    data.name,
    data.description || null,
    data.defaultCapacity || null,
    data.capacityUnit || null,
    data.isActive ?? true,
    data.sortOrder ?? 0,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    defaultCapacity: row.default_capacity,
    capacityUnit: row.capacity_unit,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateCarrierGroup(pool: Pool, id: number, data: Partial<InsertProductionCarrierGroup>): Promise<ProductionCarrierGroup | null> {
  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.code !== undefined) {
    updates.push(`code = $${paramIndex++}`);
    values.push(data.code);
  }
  if (data.name !== undefined) {
    updates.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }
  if (data.description !== undefined) {
    updates.push(`description = $${paramIndex++}`);
    values.push(data.description);
  }
  if (data.defaultCapacity !== undefined) {
    updates.push(`default_capacity = $${paramIndex++}`);
    values.push(data.defaultCapacity);
  }
  if (data.capacityUnit !== undefined) {
    updates.push(`capacity_unit = $${paramIndex++}`);
    values.push(data.capacityUnit);
  }
  if (data.isActive !== undefined) {
    updates.push(`is_active = $${paramIndex++}`);
    values.push(data.isActive);
  }
  if (data.sortOrder !== undefined) {
    updates.push(`sort_order = $${paramIndex++}`);
    values.push(data.sortOrder);
  }

  if (updates.length === 0) return getCarrierGroupById(pool, id);

  updates.push(`updated_at = NOW()`);
  values.push(id);

  const result = await pool.query(`
    UPDATE production.production_carrier_groups
    SET ${updates.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);

  if (result.rows.length === 0) return null;

  const row = result.rows[0];
  return {
    id: row.id,
    code: row.code,
    name: row.name,
    description: row.description,
    defaultCapacity: row.default_capacity,
    capacityUnit: row.capacity_unit,
    isActive: row.is_active,
    sortOrder: row.sort_order,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteCarrierGroup(pool: Pool, id: number): Promise<void> {
  await pool.query(`
    DELETE FROM production.production_carrier_groups
    WHERE id = $1
  `, [id]);
}

// Carriers
export async function getCarriers(pool: Pool, groupId?: number): Promise<ProductionCarrier[]> {
  const query = groupId
    ? `SELECT * FROM production.production_carriers WHERE carrier_group_id = $1 ORDER BY code ASC`
    : `SELECT * FROM production.production_carriers ORDER BY code ASC`;
  
  const result = groupId 
    ? await pool.query(query, [groupId])
    : await pool.query(query);
  
  return result.rows.map(row => ({
    id: row.id,
    carrierGroupId: row.carrier_group_id,
    code: row.code,
    name: row.name,
    barcode: row.barcode,
    status: row.status,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    weight: row.weight,
    currentLocationId: row.current_location_id,
    defaultLocationId: row.default_location_id,
    lastMaintenanceDate: row.last_maintenance_date,
    nextMaintenanceDate: row.next_maintenance_date,
    notes: row.notes,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getCarrierById(pool: Pool, id: number): Promise<ProductionCarrier | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_carriers
    WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    carrierGroupId: row.carrier_group_id,
    code: row.code,
    name: row.name,
    barcode: row.barcode,
    status: row.status,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    weight: row.weight,
    currentLocationId: row.current_location_id,
    defaultLocationId: row.default_location_id,
    lastMaintenanceDate: row.last_maintenance_date,
    nextMaintenanceDate: row.next_maintenance_date,
    notes: row.notes,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createCarrier(pool: Pool, data: InsertProductionCarrier): Promise<ProductionCarrier> {
  const result = await pool.query(`
    INSERT INTO production.production_carriers 
    (carrier_group_id, code, name, barcode, status, capacity, capacity_unit, 
     current_load, dimensions, weight, current_location_id, default_location_id, 
     last_maintenance_date, next_maintenance_date, notes, is_active)
    VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16)
    RETURNING *
  `, [
    data.carrierGroupId,
    data.code,
    data.name,
    data.barcode || null,
    data.status || 'available',
    data.capacity || null,
    data.capacityUnit || null,
    data.currentLoad || 0,
    data.dimensions ? JSON.stringify(data.dimensions) : null,
    data.weight || null,
    data.currentLocationId || null,
    data.defaultLocationId || null,
    data.lastMaintenanceDate || null,
    data.nextMaintenanceDate || null,
    data.notes || null,
    data.isActive ?? true,
  ]);
  
  const row = result.rows[0];
  return {
    id: row.id,
    carrierGroupId: row.carrier_group_id,
    code: row.code,
    name: row.name,
    barcode: row.barcode,
    status: row.status,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    weight: row.weight,
    currentLocationId: row.current_location_id,
    defaultLocationId: row.default_location_id,
    lastMaintenanceDate: row.last_maintenance_date,
    nextMaintenanceDate: row.next_maintenance_date,
    notes: row.notes,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function updateCarrier(pool: Pool, id: number, data: Partial<InsertProductionCarrier>): Promise<ProductionCarrier | null> {
  const updates: string[] = [];
  const values: any[] = [];
  let paramIndex = 1;

  if (data.carrierGroupId !== undefined) {
    updates.push(`carrier_group_id = $${paramIndex++}`);
    values.push(data.carrierGroupId);
  }
  if (data.code !== undefined) {
    updates.push(`code = $${paramIndex++}`);
    values.push(data.code);
  }
  if (data.name !== undefined) {
    updates.push(`name = $${paramIndex++}`);
    values.push(data.name);
  }
  if (data.barcode !== undefined) {
    updates.push(`barcode = $${paramIndex++}`);
    values.push(data.barcode);
  }
  if (data.status !== undefined) {
    updates.push(`status = $${paramIndex++}`);
    values.push(data.status);
  }
  if (data.capacity !== undefined) {
    updates.push(`capacity = $${paramIndex++}`);
    values.push(data.capacity);
  }
  if (data.capacityUnit !== undefined) {
    updates.push(`capacity_unit = $${paramIndex++}`);
    values.push(data.capacityUnit);
  }
  if (data.currentLoad !== undefined) {
    updates.push(`current_load = $${paramIndex++}`);
    values.push(data.currentLoad);
  }
  if (data.dimensions !== undefined) {
    updates.push(`dimensions = $${paramIndex++}`);
    values.push(data.dimensions ? JSON.stringify(data.dimensions) : null);
  }
  if (data.weight !== undefined) {
    updates.push(`weight = $${paramIndex++}`);
    values.push(data.weight);
  }
  if (data.currentLocationId !== undefined) {
    updates.push(`current_location_id = $${paramIndex++}`);
    values.push(data.currentLocationId);
  }
  if (data.defaultLocationId !== undefined) {
    updates.push(`default_location_id = $${paramIndex++}`);
    values.push(data.defaultLocationId);
  }
  if (data.lastMaintenanceDate !== undefined) {
    updates.push(`last_maintenance_date = $${paramIndex++}`);
    values.push(data.lastMaintenanceDate);
  }
  if (data.nextMaintenanceDate !== undefined) {
    updates.push(`next_maintenance_date = $${paramIndex++}`);
    values.push(data.nextMaintenanceDate);
  }
  if (data.notes !== undefined) {
    updates.push(`notes = $${paramIndex++}`);
    values.push(data.notes);
  }
  if (data.isActive !== undefined) {
    updates.push(`is_active = $${paramIndex++}`);
    values.push(data.isActive);
  }

  if (updates.length === 0) return getCarrierById(pool, id);

  updates.push(`updated_at = NOW()`);
  values.push(id);

  const result = await pool.query(`
    UPDATE production.production_carriers
    SET ${updates.join(', ')}
    WHERE id = $${paramIndex}
    RETURNING *
  `, values);

  if (result.rows.length === 0) return null;

  const row = result.rows[0];
  return {
    id: row.id,
    carrierGroupId: row.carrier_group_id,
    code: row.code,
    name: row.name,
    barcode: row.barcode,
    status: row.status,
    capacity: row.capacity,
    capacityUnit: row.capacity_unit,
    currentLoad: row.current_load,
    dimensions: row.dimensions,
    weight: row.weight,
    currentLocationId: row.current_location_id,
    defaultLocationId: row.default_location_id,
    lastMaintenanceDate: row.last_maintenance_date,
    nextMaintenanceDate: row.next_maintenance_date,
    notes: row.notes,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function deleteCarrier(pool: Pool, id: number): Promise<void> {
  await pool.query(`
    DELETE FROM production.production_carriers
    WHERE id = $1
  `, [id]);
}

export async function seedDefaultCarriers(pool: Pool): Promise<{ groupsInserted: number; carriersInserted: number }> {
  // Step 1: Seed carrier groups
  const groupResult = await pool.query(`
    INSERT INTO production.production_carrier_groups 
    (code, name, description, default_capacity, capacity_unit, is_active, sort_order)
    VALUES 
      ('PALLET-EURO', 'Palety Euro', 'Palety europejskie 1200x800mm', 500.00, 'kg', true, 10),
      ('PALLET-STANDARD', 'Palety Standardowe', 'Palety standardowe 1200x1000mm', 600.00, 'kg', true, 20),
      ('CART', 'Wózki Transportowe', 'Wózki do transportu wewnętrznego', 200.00, 'kg', true, 30)
    ON CONFLICT (code) DO NOTHING
    RETURNING id, code
  `);

  const groupsInserted = groupResult.rowCount || 0;

  // Step 2: Get carrier group IDs for seeding carriers
  const groupsMap = await pool.query(`
    SELECT id, code FROM production.production_carrier_groups 
    WHERE code IN ('PALLET-EURO', 'PALLET-STANDARD', 'CART')
  `);

  const euroGroupId = groupsMap.rows.find(g => g.code === 'PALLET-EURO')?.id;
  const standardGroupId = groupsMap.rows.find(g => g.code === 'PALLET-STANDARD')?.id;
  const cartGroupId = groupsMap.rows.find(g => g.code === 'CART')?.id;

  let carriersInserted = 0;

  // Seed Euro Pallets (10 pallets)
  if (euroGroupId) {
    const euroResult = await pool.query(`
      INSERT INTO production.production_carriers 
      (carrier_group_id, code, name, status, capacity, capacity_unit, dimensions, weight, is_active)
      VALUES 
        ($1, 'EURO-001', 'Paleta Euro #001', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-002', 'Paleta Euro #002', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-003', 'Paleta Euro #003', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-004', 'Paleta Euro #004', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-005', 'Paleta Euro #005', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-006', 'Paleta Euro #006', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-007', 'Paleta Euro #007', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-008', 'Paleta Euro #008', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-009', 'Paleta Euro #009', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true),
        ($1, 'EURO-010', 'Paleta Euro #010', 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true)
      ON CONFLICT (carrier_group_id, code) DO NOTHING
    `, [euroGroupId]);
    carriersInserted += euroResult.rowCount || 0;
  }

  // Seed Standard Pallets (5 pallets)
  if (standardGroupId) {
    const standardResult = await pool.query(`
      INSERT INTO production.production_carriers 
      (carrier_group_id, code, name, status, capacity, capacity_unit, dimensions, weight, is_active)
      VALUES 
        ($1, 'STD-001', 'Paleta Standardowa #001', 'available', 600.00, 'kg', '{"length": 1200, "width": 1000, "height": 144, "unit": "mm"}'::jsonb, 30.00, true),
        ($1, 'STD-002', 'Paleta Standardowa #002', 'available', 600.00, 'kg', '{"length": 1200, "width": 1000, "height": 144, "unit": "mm"}'::jsonb, 30.00, true),
        ($1, 'STD-003', 'Paleta Standardowa #003', 'available', 600.00, 'kg', '{"length": 1200, "width": 1000, "height": 144, "unit": "mm"}'::jsonb, 30.00, true),
        ($1, 'STD-004', 'Paleta Standardowa #004', 'available', 600.00, 'kg', '{"length": 1200, "width": 1000, "height": 144, "unit": "mm"}'::jsonb, 30.00, true),
        ($1, 'STD-005', 'Paleta Standardowa #005', 'available', 600.00, 'kg', '{"length": 1200, "width": 1000, "height": 144, "unit": "mm"}'::jsonb, 30.00, true)
      ON CONFLICT (carrier_group_id, code) DO NOTHING
    `, [standardGroupId]);
    carriersInserted += standardResult.rowCount || 0;
  }

  // Seed Transport Carts (3 carts)
  if (cartGroupId) {
    const cartResult = await pool.query(`
      INSERT INTO production.production_carriers 
      (carrier_group_id, code, name, status, capacity, capacity_unit, dimensions, weight, is_active)
      VALUES 
        ($1, 'CART-001', 'Wózek Transportowy #001', 'available', 200.00, 'kg', '{"length": 1000, "width": 600, "height": 800, "unit": "mm"}'::jsonb, 15.00, true),
        ($1, 'CART-002', 'Wózek Transportowy #002', 'available', 200.00, 'kg', '{"length": 1000, "width": 600, "height": 800, "unit": "mm"}'::jsonb, 15.00, true),
        ($1, 'CART-003', 'Wózek Transportowy #003', 'available', 200.00, 'kg', '{"length": 1000, "width": 600, "height": 800, "unit": "mm"}'::jsonb, 15.00, true)
      ON CONFLICT (carrier_group_id, code) DO NOTHING
    `, [cartGroupId]);
    carriersInserted += cartResult.rowCount || 0;
  }

  return { groupsInserted, carriersInserted };
}
