import type { Express } from "express";
import { pool } from "../../postgres";
import { requirePermission } from "../../auth";
import multer from "multer";
import Papa from "papaparse";
import { Buffer } from "buffer";

// Configure multer for memory storage
const upload = multer({ storage: multer.memoryStorage() });

export function registerCuttingPatternRoutes(app: Express) {
  // GET /api/production/cut-patterns - Lista wszystkich rozkrojów
  app.get("/api/production/cut-patterns", requirePermission("view_production"), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          cp.id,
          cp.code,
          cp.board_length,
          cp.board_width,
          cp.board_thickness,
          cp.kerf,
          cp.image_url,
          cp.opis,
          cp.status,
          cp.notes,
          cp.is_active,
          cp.created_at,
          cp.updated_at,
          COALESCE(COUNT(cpi.id), 0) as items_count,
          COALESCE(SUM(cpi.quantity_requested), 0) as total_quantity,
          COALESCE(SUM(CASE WHEN cpi.status = 'zaplanowana' THEN cpi.quantity_requested ELSE 0 END), 0) as planned_quantity,
          COALESCE(SUM(CASE WHEN cpi.status = 'w_produkcji' THEN cpi.quantity_requested ELSE 0 END), 0) as in_production_quantity,
          COALESCE(SUM(CASE WHEN cpi.status = 'wykonana' THEN cpi.quantity_requested ELSE 0 END), 0) as completed_quantity
        FROM production.cut_patterns cp
        LEFT JOIN production.cut_pattern_items cpi ON cpi.pattern_id = cp.id AND cpi.is_active = true
        WHERE cp.is_active = true
        GROUP BY cp.id
        ORDER BY cp.code ASC
      `);
      
      res.json(result.rows);
    } catch (error: any) {
      console.error("❌ Error fetching cutting patterns:", error);
      res.status(500).json({ 
        error: "Błąd pobierania rozkrojów",
        details: error.message 
      });
    }
  });

  // GET /api/production/cut-patterns/:id - Szczegóły rozkroju z listą formatek
  app.get("/api/production/cut-patterns/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID rozkroju" });
      }
      
      // Pobierz dane rozkroju
      const patternResult = await pool.query(`
        SELECT 
          cp.id,
          cp.code,
          cp.board_length,
          cp.board_width,
          cp.board_thickness,
          cp.kerf,
          cp.image_url,
          cp.opis,
          cp.status,
          cp.notes,
          cp.is_active,
          cp.created_at,
          cp.updated_at
        FROM production.cut_patterns cp
        WHERE cp.id = $1 AND cp.is_active = true
      `, [id]);
      
      if (patternResult.rows.length === 0) {
        return res.status(404).json({ error: "Rozkrój nie znaleziony" });
      }
      
      const pattern = patternResult.rows[0];
      
      // Pobierz listę formatek dla tego rozkroju
      const itemsResult = await pool.query(`
        SELECT 
          id,
          item_name,
          length,
          width,
          thickness,
          board_material_id,
          edging_material_id,
          quantity_requested,
          quantity_completed,
          status,
          notes,
          is_active,
          created_at,
          updated_at
        FROM production.cut_pattern_items
        WHERE pattern_id = $1 AND is_active = true
        ORDER BY item_name ASC
      `, [id]);
      
      res.json({
        ...pattern,
        items: itemsResult.rows,
        items_count: itemsResult.rows.length,
        total_quantity: itemsResult.rows.reduce((sum, item) => sum + (item.quantity_requested || 0), 0),
        completed_quantity: itemsResult.rows.reduce((sum, item) => sum + (item.quantity_completed || 0), 0)
      });
    } catch (error: any) {
      console.error("❌ Error fetching cutting pattern details:", error);
      res.status(500).json({ 
        error: "Błąd pobierania szczegółów rozkroju",
        details: error.message 
      });
    }
  });

  // POST /api/production/cut-patterns/:id/start - Rozpocznij produkcję rozkroju
  app.post("/api/production/cut-patterns/:id/start", requirePermission("manage_production"), async (req, res) => {
    const client = await pool.connect();
    
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID rozkroju" });
      }
      
      await client.query('BEGIN');
      
      // Sprawdź czy rozkrój istnieje i ma status 'planowany'
      const checkResult = await client.query(`
        SELECT id, status FROM production.cut_patterns
        WHERE id = $1 AND is_active = true
      `, [id]);
      
      if (checkResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Rozkrój nie znaleziony" });
      }
      
      if (checkResult.rows[0].status !== 'planowany') {
        await client.query('ROLLBACK');
        return res.status(400).json({ 
          error: "Nie można rozpocząć produkcji",
          details: `Rozkrój ma status: ${checkResult.rows[0].status}` 
        });
      }
      
      // Zmień status rozkroju
      const patternUpdate = await client.query(`
        UPDATE production.cut_patterns
        SET status = 'w_produkcji', updated_at = NOW()
        WHERE id = $1 AND is_active = true
        RETURNING id
      `, [id]);
      
      if (patternUpdate.rowCount === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Nie udało się zaktualizować rozkroju" });
      }
      
      // Zmień status wszystkich formatek
      await client.query(`
        UPDATE production.cut_pattern_items
        SET status = 'w_produkcji', updated_at = NOW()
        WHERE pattern_id = $1 AND status = 'zaplanowana' AND is_active = true
      `, [id]);
      
      await client.query('COMMIT');
      res.json({ success: true, message: "Produkcja rozkroju rozpoczęta" });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error starting production:", error);
      res.status(500).json({ 
        error: "Błąd rozpoczynania produkcji",
        details: error.message 
      });
    } finally {
      client.release();
    }
  });

  // POST /api/production/cut-patterns/:id/complete - Zakończ produkcję rozkroju
  app.post("/api/production/cut-patterns/:id/complete", requirePermission("manage_production"), async (req, res) => {
    const client = await pool.connect();
    
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID rozkroju" });
      }
      
      await client.query('BEGIN');
      
      // Sprawdź czy rozkrój istnieje i ma status 'w_produkcji'
      const checkResult = await client.query(`
        SELECT id, status FROM production.cut_patterns
        WHERE id = $1 AND is_active = true
      `, [id]);
      
      if (checkResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Rozkrój nie znaleziony" });
      }
      
      if (checkResult.rows[0].status !== 'w_produkcji') {
        await client.query('ROLLBACK');
        return res.status(400).json({ 
          error: "Nie można zakończyć produkcji",
          details: `Rozkrój ma status: ${checkResult.rows[0].status}` 
        });
      }
      
      // Zmień status rozkroju
      const patternUpdate = await client.query(`
        UPDATE production.cut_patterns
        SET status = 'wykonany', updated_at = NOW()
        WHERE id = $1 AND is_active = true
        RETURNING id
      `, [id]);
      
      if (patternUpdate.rowCount === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Nie udało się zaktualizować rozkroju" });
      }
      
      // Zmień status wszystkich formatek i ustaw quantity_completed
      await client.query(`
        UPDATE production.cut_pattern_items
        SET 
          status = 'wykonana',
          quantity_completed = quantity_requested,
          updated_at = NOW()
        WHERE pattern_id = $1 AND status = 'w_produkcji' AND is_active = true
      `, [id]);
      
      await client.query('COMMIT');
      res.json({ success: true, message: "Produkcja rozkroju zakończona" });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error completing production:", error);
      res.status(500).json({ 
        error: "Błąd kończenia produkcji",
        details: error.message 
      });
    } finally {
      client.release();
    }
  });

  // POST /api/production/cut-patterns/:id/import-formatki - Import formatek z CSV
  app.post("/api/production/cut-patterns/:id/import-formatki", 
    requirePermission("manage_production"), 
    upload.single('file'), 
    async (req, res) => {
      const client = await pool.connect();
      
      try {
        const patternId = parseInt(req.params.id);
        
        if (isNaN(patternId)) {
          return res.status(400).json({ error: "Nieprawidłowe ID rozkroju" });
        }
        
        if (!req.file) {
          return res.status(400).json({ error: "Brak pliku CSV" });
        }
        
        // Sprawdź czy rozkrój istnieje
        const patternCheck = await client.query(`
          SELECT id, code FROM production.cut_patterns
          WHERE id = $1 AND is_active = true
        `, [patternId]);
        
        if (patternCheck.rows.length === 0) {
          return res.status(404).json({ error: "Rozkrój nie znaleziony" });
        }
        
        // Convert from Windows-1250 to UTF-8
        const iconv = await import('iconv-lite');
        const csvContent = iconv.decode(req.file.buffer, 'windows-1250');
        
        // Parse CSV
        const parseResult = Papa.parse(csvContent, {
          header: true,
          skipEmptyLines: true,
          delimiter: ',',
        });
        
        if (parseResult.errors.length > 0) {
          console.error("CSV parsing errors:", parseResult.errors);
          return res.status(400).json({ 
            error: "Błąd parsowania CSV",
            details: parseResult.errors[0].message 
          });
        }
        
        const rows = parseResult.data as any[];
        
        if (rows.length === 0) {
          return res.status(400).json({ error: "Plik CSV jest pusty" });
        }
        
        await client.query('BEGIN');
        
        let importedCount = 0;
        
        for (const row of rows) {
          // Mapowanie kolumn z CSV (sprawdź różne warianty nazw)
          const material = row['Materiał'] || row['Material'] || row['material'] || '';
          const length = parseFloat(
            row['Długość przygotówki'] || 
            row['Dlugosc przygotowki'] || 
            row['Długość'] || 
            row['Dlugosc'] ||
            row['Length'] || 
            '0'
          );
          const width = parseFloat(
            row['Szerokość przygotówki'] || 
            row['Szerokosc przygotowki'] || 
            row['Szerokość'] || 
            row['Szerokosc'] ||
            row['Width'] || 
            '0'
          );
          const quantity = parseInt(
            row['Ilość'] || 
            row['Ilosc'] || 
            row['Quantity'] || 
            '1'
          );
          const feature = row['Cecha'] || row['Feature'] || '';
          const description = row['Opis'] || row['Description'] || '';
          
          // Okleinowania - obsługa zarówno pojedynczej kolumny jak i 4 osobnych
          const edgingUnified = row['Okleinowania'] || row['Edging'] || '';
          const edgingFront = row['Przednia okleina'] || row['Front'] || '';
          const edgingRight = row['Prawa okleina'] || row['Right'] || '';
          const edgingBack = row['Tylna okleina'] || row['Back'] || '';
          const edgingLeft = row['LEWA_OKLEINA'] || row['Lewa okleina'] || row['Left'] || '';
          
          if (!material || length === 0 || width === 0) {
            console.warn(`⚠️  Pomijam niepełny wiersz - Material: "${material}", Length: ${length}, Width: ${width}`);
            console.warn(`    Dostępne kolumny: ${Object.keys(row).join(', ')}`);
            continue;
          }
          
          // Pobierz thickness z materiału (np. "18_WOTAN" -> 18)
          const thicknessMatch = material.match(/^(\d+)_/);
          const thickness = thicknessMatch ? parseFloat(thicknessMatch[1]) : 18;
          
          // Konstruuj nazwę formatki
          const itemName = feature || description || `${material} ${length}x${width}`;
          
          // Konstruuj notatki z informacjami o okleinowaniu
          const edgingInfo = [];
          
          // Jeśli jest unified kolumna Okleinowania, użyj jej
          if (edgingUnified && edgingUnified !== '0') {
            edgingInfo.push(`Okleinowanie: ${edgingUnified}`);
          } else {
            // W przeciwnym razie użyj osobnych kolumn
            if (edgingFront && edgingFront !== '0') edgingInfo.push(`Przód: ${edgingFront}`);
            if (edgingRight && edgingRight !== '0') edgingInfo.push(`Prawo: ${edgingRight}`);
            if (edgingBack && edgingBack !== '0') edgingInfo.push(`Tył: ${edgingBack}`);
            if (edgingLeft && edgingLeft !== '0') edgingInfo.push(`Lewo: ${edgingLeft}`);
          }
          
          const notes = edgingInfo.length > 0 ? edgingInfo.join(', ') : null;
          
          // Dodaj formatkę do rozkroju
          await client.query(`
            INSERT INTO production.cut_pattern_items (
              pattern_id,
              item_name,
              length,
              width,
              thickness,
              quantity_requested,
              quantity_completed,
              status,
              notes,
              is_active
            ) VALUES ($1, $2, $3, $4, $5, $6, 0, 'zaplanowana', $7, true)
          `, [
            patternId,
            itemName,
            length.toString(),
            width.toString(),
            thickness.toString(),
            quantity,
            notes
          ]);
          
          importedCount++;
        }
        
        if (importedCount === 0) {
          await client.query('ROLLBACK');
          console.error(`❌ Nie zaimportowano żadnej formatki z pliku CSV`);
          console.error(`   Sprawdź czy plik ma poprawne nagłówki kolumn i dane`);
          console.error(`   Oczekiwane kolumny: Materiał, Długość przygotówki, Szerokość przygotówki, Ilość, Cecha, Opis`);
          
          return res.status(400).json({ 
            error: "Nie udało się zaimportować formatek",
            details: `Wszystkie wiersze zostały pominięte. Sprawdź czy plik ma poprawne nagłówki kolumn (Materiał, Długość przygotówki, Szerokość przygotówki) i niepuste wartości.`,
            totalRows: rows.length,
            imported: 0
          });
        }
        
        await client.query('COMMIT');
        
        console.log(`✅ Zaimportowano ${importedCount} formatek do rozkroju ${patternCheck.rows[0].code}`);
        
        res.json({ 
          success: true, 
          message: `Zaimportowano ${importedCount} formatek`,
          imported: importedCount,
          totalRows: rows.length
        });
      } catch (error: any) {
        await client.query('ROLLBACK');
        console.error("❌ Error importing formatki:", error);
        res.status(500).json({ 
          error: "Błąd importu formatek",
          details: error.message 
        });
      } finally {
        client.release();
      }
    }
  );
}
