import type { Express } from "express";
import { pool } from "../../postgres";
import { requirePermission } from "../../auth";
import multer from "multer";
import Papa from "papaparse";
import { Buffer } from "buffer";
import iconv from "iconv-lite";

const upload = multer({ storage: multer.memoryStorage() });

export function registerCutPatternTemplateRoutes(app: Express) {
  // GET /api/production/cut-pattern-templates - Lista szablonów rozkrojów
  app.get("/api/production/cut-pattern-templates", requirePermission("view_production"), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          cpt.id,
          cpt.code,
          cpt.name,
          cpt.description,
          cpt.board_type_code,
          cpt.board_length,
          cpt.board_width,
          cpt.board_thickness,
          cpt.kerf,
          cpt.image_url,
          cpt.opis,
          cpt.usage_count,
          cpt.last_used_at,
          cpt.notes,
          cpt.metadata,
          cpt.is_active,
          cpt.created_at,
          cpt.updated_at,
          COALESCE(COUNT(cpti.id), 0) as items_count,
          COALESCE(SUM(cpti.quantity_default), 0) as total_quantity
        FROM production.cut_pattern_templates cpt
        LEFT JOIN production.cut_pattern_template_items cpti ON cpti.template_id = cpt.id AND cpti.is_active = true
        WHERE cpt.is_active = true
        GROUP BY cpt.id
        ORDER BY cpt.usage_count DESC, cpt.code ASC
      `);
      
      res.json(result.rows);
    } catch (error: any) {
      console.error("❌ Error fetching cut pattern templates:", error);
      res.status(500).json({ 
        error: "Błąd pobierania szablonów rozkrojów",
        details: error.message 
      });
    }
  });

  // GET /api/production/cut-pattern-templates/:id - Szczegóły szablonu z formatkami
  app.get("/api/production/cut-pattern-templates/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID szablonu" });
      }
      
      const templateResult = await pool.query(`
        SELECT 
          cpt.id,
          cpt.code,
          cpt.name,
          cpt.description,
          cpt.board_type_code,
          cpt.board_length,
          cpt.board_width,
          cpt.board_thickness,
          cpt.kerf,
          cpt.image_url,
          cpt.opis,
          cpt.usage_count,
          cpt.last_used_at,
          cpt.notes,
          cpt.metadata,
          cpt.is_active,
          cpt.created_at,
          cpt.updated_at
        FROM production.cut_pattern_templates cpt
        WHERE cpt.id = $1 AND cpt.is_active = true
      `, [id]);
      
      if (templateResult.rows.length === 0) {
        return res.status(404).json({ error: "Szablon nie znaleziony" });
      }
      
      const template = templateResult.rows[0];
      
      const itemsResult = await pool.query(`
        SELECT 
          id,
          item_name,
          length,
          width,
          thickness,
          board_material_id,
          edging_material_id,
          board_code,
          edging_code,
          color_code,
          edge1,
          edge2,
          edge3,
          edge4,
          quantity_default,
          position_metadata,
          notes,
          sort_order,
          is_active,
          created_at,
          updated_at
        FROM production.cut_pattern_template_items
        WHERE template_id = $1 AND is_active = true
        ORDER BY sort_order ASC, item_name ASC
      `, [id]);
      
      res.json({
        ...template,
        items: itemsResult.rows,
        items_count: itemsResult.rows.length,
        total_quantity: itemsResult.rows.reduce((sum, item) => sum + (item.quantity_default || 0), 0)
      });
    } catch (error: any) {
      console.error("❌ Error fetching template details:", error);
      res.status(500).json({ 
        error: "Błąd pobierania szczegółów szablonu",
        details: error.message 
      });
    }
  });

  // POST /api/production/cut-pattern-templates - Utwórz nowy szablon
  app.post("/api/production/cut-pattern-templates", requirePermission("manage_production"), async (req, res) => {
    const client = await pool.connect();
    
    try {
      let {
        code,
        name,
        description,
        boardTypeCode,
        boardLength,
        boardWidth,
        boardThickness = 18,
        kerf = 4,
        imageUrl,
        opis,
        notes,
        metadata,
        items = []
      } = req.body;
      
      if (!name || !boardLength || !boardWidth) {
        return res.status(400).json({ 
          error: "Wymagane pola: name, boardLength, boardWidth" 
        });
      }
      
      await client.query('BEGIN');
      
      // Auto-generuj kod ROZ-XXX jeśli nie podano (race-safe)
      if (!code) {
        await client.query('SELECT pg_advisory_xact_lock(123456789)');
        
        const lastCodeResult = await client.query(`
          SELECT code FROM production.cut_pattern_templates 
          WHERE code LIKE 'ROZ-%' AND is_active = true
          ORDER BY 
            CASE 
              WHEN code ~ '^ROZ-[0-9]+$' THEN CAST(SUBSTRING(code FROM 5) AS INTEGER)
              ELSE 0
            END DESC
          LIMIT 1
        `);
        
        let nextNumber = 1;
        if (lastCodeResult.rows.length > 0) {
          const lastCode = lastCodeResult.rows[0].code;
          const match = lastCode.match(/^ROZ-(\d+)$/);
          if (match) {
            nextNumber = parseInt(match[1]) + 1;
          }
        }
        
        code = `ROZ-${nextNumber}`;
      } else {
        // Sprawdź czy ręcznie podany kod jest unikalny
        const codeCheck = await client.query(
          `SELECT id FROM production.cut_pattern_templates WHERE code = $1 AND is_active = true`,
          [code]
        );
        
        if (codeCheck.rows.length > 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ error: `Szablon o kodzie ${code} już istnieje` });
        }
      }
      
      // Utwórz szablon
      const templateResult = await client.query(`
        INSERT INTO production.cut_pattern_templates (
          code, name, description, board_type_code, board_length, board_width,
          board_thickness, kerf, image_url, opis, notes, metadata, created_by
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
        RETURNING *
      `, [
        code, name, description, boardTypeCode, boardLength, boardWidth,
        boardThickness, kerf, imageUrl, opis, notes, 
        metadata ? JSON.stringify(metadata) : null,
        req.user?.id || null
      ]);
      
      const templateId = templateResult.rows[0].id;
      
      // Dodaj formatki jeśli podano
      if (items && items.length > 0) {
        for (let i = 0; i < items.length; i++) {
          const item = items[i];
          await client.query(`
            INSERT INTO production.cut_pattern_template_items (
              template_id, item_name, length, width, thickness,
              board_material_id, edging_material_id, board_code, edging_code, color_code,
              edge1, edge2, edge3, edge4, quantity_default, position_metadata, notes, sort_order
            ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18)
          `, [
            templateId, item.itemName, item.length, item.width, item.thickness || boardThickness,
            item.boardMaterialId, item.edgingMaterialId, item.boardCode, item.edgingCode, item.colorCode,
            item.edge1 || false, item.edge2 || false, item.edge3 || false, item.edge4 || false,
            item.quantityDefault || 1, item.positionMetadata ? JSON.stringify(item.positionMetadata) : null,
            item.notes, i
          ]);
        }
      }
      
      await client.query('COMMIT');
      
      console.log(`✅ Created template ${code} with ${items.length} items`);
      
      res.json({ 
        success: true, 
        message: `Szablon ${code} został utworzony`,
        templateId 
      });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error creating template:", error);
      res.status(500).json({ 
        error: "Błąd tworzenia szablonu",
        details: error.message 
      });
    } finally {
      client.release();
    }
  });

  // PATCH /api/production/cut-pattern-templates/:id - Edytuj szablon
  app.patch("/api/production/cut-pattern-templates/:id", requirePermission("manage_production"), async (req, res) => {
    const client = await pool.connect();
    
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID szablonu" });
      }
      
      const {
        name,
        description,
        boardTypeCode,
        boardLength,
        boardWidth,
        boardThickness,
        kerf,
        imageUrl,
        opis,
        notes,
        metadata
      } = req.body;
      
      await client.query('BEGIN');
      
      // Sprawdź czy szablon istnieje
      const check = await client.query(
        `SELECT id FROM production.cut_pattern_templates WHERE id = $1 AND is_active = true`,
        [id]
      );
      
      if (check.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Szablon nie znaleziony" });
      }
      
      // Aktualizuj szablon
      await client.query(`
        UPDATE production.cut_pattern_templates
        SET 
          name = COALESCE($1, name),
          description = COALESCE($2, description),
          board_type_code = COALESCE($3, board_type_code),
          board_length = COALESCE($4, board_length),
          board_width = COALESCE($5, board_width),
          board_thickness = COALESCE($6, board_thickness),
          kerf = COALESCE($7, kerf),
          image_url = COALESCE($8, image_url),
          opis = COALESCE($9, opis),
          notes = COALESCE($10, notes),
          metadata = COALESCE($11, metadata),
          updated_at = NOW()
        WHERE id = $12
      `, [
        name, description, boardTypeCode, boardLength, boardWidth,
        boardThickness, kerf, imageUrl, opis, notes,
        metadata ? JSON.stringify(metadata) : null,
        id
      ]);
      
      await client.query('COMMIT');
      
      console.log(`✅ Updated template ID ${id}`);
      
      res.json({ 
        success: true, 
        message: "Szablon został zaktualizowany"
      });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error updating template:", error);
      res.status(500).json({ 
        error: "Błąd aktualizacji szablonu",
        details: error.message 
      });
    } finally {
      client.release();
    }
  });

  // DELETE /api/production/cut-pattern-templates/:id - Usuń szablon (soft delete)
  app.delete("/api/production/cut-pattern-templates/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      
      if (isNaN(id)) {
        return res.status(400).json({ error: "Nieprawidłowe ID szablonu" });
      }
      
      const result = await pool.query(`
        UPDATE production.cut_pattern_templates
        SET is_active = false, updated_at = NOW()
        WHERE id = $1 AND is_active = true
        RETURNING code
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Szablon nie znaleziony" });
      }
      
      console.log(`✅ Deleted template ${result.rows[0].code}`);
      
      res.json({ 
        success: true, 
        message: "Szablon został usunięty"
      });
    } catch (error: any) {
      console.error("❌ Error deleting template:", error);
      res.status(500).json({ 
        error: "Błąd usuwania szablonu",
        details: error.message 
      });
    }
  });

  // POST /api/production/cut-pattern-templates/:id/items - Dodaj formatkę do szablonu
  app.post("/api/production/cut-pattern-templates/:id/items", requirePermission("manage_production"), async (req, res) => {
    try {
      const templateId = parseInt(req.params.id);
      
      if (isNaN(templateId)) {
        return res.status(400).json({ error: "Nieprawidłowe ID szablonu" });
      }
      
      const {
        itemName,
        length,
        width,
        thickness,
        boardMaterialId,
        edgingMaterialId,
        boardCode,
        edgingCode,
        colorCode,
        edge1,
        edge2,
        edge3,
        edge4,
        quantityDefault = 1,
        positionMetadata,
        notes
      } = req.body;
      
      if (!length || !width) {
        return res.status(400).json({ error: "Wymagane pola: length, width" });
      }
      
      // Pobierz najwyższy sort_order
      const sortResult = await pool.query(
        `SELECT COALESCE(MAX(sort_order), -1) as max_sort FROM production.cut_pattern_template_items WHERE template_id = $1`,
        [templateId]
      );
      const nextSort = sortResult.rows[0].max_sort + 1;
      
      const result = await pool.query(`
        INSERT INTO production.cut_pattern_template_items (
          template_id, item_name, length, width, thickness,
          board_material_id, edging_material_id, board_code, edging_code, color_code,
          edge1, edge2, edge3, edge4, quantity_default, position_metadata, notes, sort_order
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18)
        RETURNING id
      `, [
        templateId, itemName, length, width, thickness,
        boardMaterialId, edgingMaterialId, boardCode, edgingCode, colorCode,
        edge1 || false, edge2 || false, edge3 || false, edge4 || false,
        quantityDefault, positionMetadata ? JSON.stringify(positionMetadata) : null,
        notes, nextSort
      ]);
      
      console.log(`✅ Added item to template ${templateId}`);
      
      res.json({ 
        success: true, 
        message: "Formatka dodana do szablonu",
        itemId: result.rows[0].id
      });
    } catch (error: any) {
      console.error("❌ Error adding item to template:", error);
      res.status(500).json({ 
        error: "Błąd dodawania formatki",
        details: error.message 
      });
    }
  });

  // DELETE /api/production/cut-pattern-templates/:id/items/:itemId - Usuń formatkę z szablonu
  app.delete("/api/production/cut-pattern-templates/:id/items/:itemId", requirePermission("manage_production"), async (req, res) => {
    try {
      const templateId = parseInt(req.params.id);
      const itemId = parseInt(req.params.itemId);
      
      if (isNaN(templateId) || isNaN(itemId)) {
        return res.status(400).json({ error: "Nieprawidłowe ID" });
      }
      
      const result = await pool.query(`
        UPDATE production.cut_pattern_template_items
        SET is_active = false, updated_at = NOW()
        WHERE id = $1 AND template_id = $2 AND is_active = true
      `, [itemId, templateId]);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ error: "Formatka nie znaleziona" });
      }
      
      console.log(`✅ Removed item ${itemId} from template ${templateId}`);
      
      res.json({ 
        success: true, 
        message: "Formatka usunięta z szablonu"
      });
    } catch (error: any) {
      console.error("❌ Error removing item from template:", error);
      res.status(500).json({ 
        error: "Błąd usuwania formatki",
        details: error.message 
      });
    }
  });

  // POST /api/production/cut-pattern-templates/import-csv - Import CSV do nowego szablonu
  app.post("/api/production/cut-pattern-templates/import-csv", requirePermission("manage_production"), upload.single('file'), async (req, res) => {
    const client = await pool.connect();
    
    try {
      if (!req.file) {
        return res.status(400).json({ error: "Nie przesłano pliku" });
      }
      
      const { name, description } = req.body;
      
      if (!name) {
        return res.status(400).json({ error: "Wymagane pole: name" });
      }
      
      // Konwertuj Windows-1250 → UTF-8
      const utf8Content = iconv.decode(Buffer.from(req.file.buffer), 'windows-1250');
      
      // Parsuj CSV
      const parsed = Papa.parse<Record<string, string>>(utf8Content, {
        header: true,
        skipEmptyLines: true,
        dynamicTyping: false
      });
      
      if (!parsed.data || parsed.data.length === 0) {
        return res.status(400).json({ error: "Plik CSV jest pusty lub nieprawidłowy" });
      }
      
      const rows = parsed.data;
      
      await client.query('BEGIN');
      
      // Automatycznie generuj kod ROZ-XXX w ramach transakcji (race-safe)
      // Użyj advisory lock aby zapewnić atomowość generowania kodu
      await client.query('SELECT pg_advisory_xact_lock(123456789)');
      
      const lastCodeResult = await client.query(`
        SELECT code FROM production.cut_pattern_templates 
        WHERE code LIKE 'ROZ-%' AND is_active = true
        ORDER BY 
          CASE 
            WHEN code ~ '^ROZ-[0-9]+$' THEN CAST(SUBSTRING(code FROM 5) AS INTEGER)
            ELSE 0
          END DESC
        LIMIT 1
      `);
      
      let nextNumber = 1;
      if (lastCodeResult.rows.length > 0) {
        const lastCode = lastCodeResult.rows[0].code;
        const match = lastCode.match(/^ROZ-(\d+)$/);
        if (match) {
          nextNumber = parseInt(match[1]) + 1;
        }
      }
      
      const code = `ROZ-${nextNumber}`;
      console.log(`📥 Parsing ${rows.length} rows from CSV for template ${code}`);
      
      // Określ wymiary płyty z pierwszego wiersza
      const firstRow = rows[0];
      const boardLength = 2800; // Domyślne wymiary
      const boardWidth = 2070;
      const boardThickness = 18;
      
      // Utwórz szablon
      const templateResult = await client.query(`
        INSERT INTO production.cut_pattern_templates (
          code, name, description, board_length, board_width, board_thickness, kerf, created_by
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        RETURNING id
      `, [code, name, description, boardLength, boardWidth, boardThickness, 4, req.user?.id || null]);
      
      const templateId = templateResult.rows[0].id;
      let importedCount = 0;
      
      for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        
        const material = row['Materiał'] || row['Material'] || row['material'] || '';
        const length = parseFloat(
          row['Długość przygotówki'] || row['Dlugosc przygotowki'] || 
          row['Length'] || row['length'] || '0'
        );
        const width = parseFloat(
          row['Szerokość przygotówki'] || row['Szerokosc przygotowki'] || 
          row['Width'] || row['width'] || '0'
        );
        const quantity = parseInt(
          row['Ilość'] || row['Ilosc'] || row['Quantity'] || '1'
        );
        const feature = row['Cecha'] || row['Feature'] || '';
        const description = row['Opis'] || row['Description'] || '';
        
        const edgingUnified = row['Okleinowania'] || row['Edging'] || '';
        const edgingFront = row['Przednia okleina'] || row['Front'] || '';
        const edgingRight = row['Prawa okleina'] || row['Right'] || '';
        const edgingBack = row['Tylna okleina'] || row['Back'] || '';
        const edgingLeft = row['LEWA_OKLEINA'] || row['Lewa okleina'] || row['Left'] || '';
        
        if (!material || length === 0 || width === 0) {
          console.warn(`⚠️  Pomijam niepełny wiersz - Material: "${material}", Length: ${length}, Width: ${width}`);
          continue;
        }
        
        // Parsing thickness z materiału (np. "18_WOTAN" → 18)
        const thicknessMatch = material.match(/^(\d+)_/);
        const thickness = thicknessMatch ? parseFloat(thicknessMatch[1]) : boardThickness;
        
        const itemName = feature || description || `${material} ${length}x${width}`;
        
        // Konstruuj notatki z okleinowaniami
        const edgingInfo = [];
        if (edgingUnified && edgingUnified !== '0') {
          edgingInfo.push(`Okleinowanie: ${edgingUnified}`);
        } else {
          if (edgingFront && edgingFront !== '0') edgingInfo.push(`Przód: ${edgingFront}`);
          if (edgingRight && edgingRight !== '0') edgingInfo.push(`Prawo: ${edgingRight}`);
          if (edgingBack && edgingBack !== '0') edgingInfo.push(`Tył: ${edgingBack}`);
          if (edgingLeft && edgingLeft !== '0') edgingInfo.push(`Lewo: ${edgingLeft}`);
        }
        
        const notes = edgingInfo.length > 0 ? edgingInfo.join(', ') : null;
        
        await client.query(`
          INSERT INTO production.cut_pattern_template_items (
            template_id, item_name, length, width, thickness, board_code,
            quantity_default, notes, sort_order
          ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)
        `, [templateId, itemName, length, width, thickness, material, quantity, notes, i]);
        
        importedCount++;
      }
      
      if (importedCount === 0) {
        await client.query('ROLLBACK');
        console.error(`❌ Nie zaimportowano żadnej formatki`);
        return res.status(400).json({ 
          error: "Nie udało się zaimportować formatek",
          details: `Wszystkie wiersze zostały pominięte. Sprawdź poprawność danych CSV.`,
          totalRows: rows.length,
          imported: 0
        });
      }
      
      await client.query('COMMIT');
      
      console.log(`✅ Created template ${code} from CSV with ${importedCount} items`);
      
      res.json({ 
        success: true, 
        message: `Szablon ${code} utworzony z ${importedCount} formatkami`,
        templateId,
        imported: importedCount,
        totalRows: rows.length
      });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error importing CSV template:", error);
      res.status(500).json({ 
        error: "Błąd importu CSV",
        details: error.message 
      });
    } finally {
      client.release();
    }
  });
}
