import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Link, useLocation } from "wouter";
import { Plus, Edit, Trash2, Loader2, Copy } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { insertProductionCarrierGroupSchema, insertProductionCarrierSchema } from "@shared/schema";
import type { ProductionCarrierGroup, ProductionCarrier } from "@shared/schema";
import { z } from "zod";

const partialCarrierGroupSchema = insertProductionCarrierGroupSchema.partial().extend({
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
});

const carrierFormSchema = z.object({
  carrierGroupId: z.coerce.number().min(1, "Grupa jest wymagana"),
  code: z.string().min(1, "Kod jest wymagany"),
  name: z.string().min(1, "Nazwa jest wymagana"),
  barcode: z.string().optional(),
  status: z.string().default("available"),
  capacity: z.coerce.number().optional(),
  capacityUnit: z.string().optional(),
  notes: z.string().optional(),
  isActive: z.boolean().default(true),
});

export default function ProductionCarriers() {
  // Check for edit query param and switch to carriers tab
  const urlParams = new URLSearchParams(window.location.search);
  const editId = urlParams.get('edit');
  const [activeTab, setActiveTab] = useState(editId ? "carriers" : "groups");

  return (
    <div className="container mx-auto p-6 space-y-6">
      <h1 className="text-3xl font-bold" data-testid="text-page-title">Nośniki Produkcyjne</h1>
      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList data-testid="tabs-carrier-sections">
          <TabsTrigger value="groups" data-testid="tab-groups">Grupy Nośników</TabsTrigger>
          <TabsTrigger value="carriers" data-testid="tab-carriers">Nośniki</TabsTrigger>
        </TabsList>
        <TabsContent value="groups"><CarrierGroupsTab /></TabsContent>
        <TabsContent value="carriers"><CarriersTab /></TabsContent>
      </Tabs>
    </div>
  );
}

function CarrierGroupsTab() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingGroup, setEditingGroup] = useState<ProductionCarrierGroup | null>(null);
  const { toast } = useToast();

  type CarrierGroupFormData = z.infer<typeof partialCarrierGroupSchema>;
  const form = useForm<CarrierGroupFormData>({
    resolver: zodResolver(partialCarrierGroupSchema),
    defaultValues: { code: "", name: "", description: "", isActive: true },
  });

  const { data: groups = [], isLoading } = useQuery<ProductionCarrierGroup[]>({
    queryKey: ["/api/production/carriers/groups"],
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => apiRequest("/api/production/carriers/groups", "POST", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers/groups"] });
      setIsDialogOpen(false);
      form.reset();
      toast({ title: "Grupa nośników została utworzona" });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) =>
      apiRequest(`/api/production/carriers/groups/${id}`, "PATCH", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers/groups"] });
      setIsDialogOpen(false);
      setEditingGroup(null);
      form.reset();
      toast({ title: "Grupa nośników została zaktualizowana" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest(`/api/production/carriers/groups/${id}`, "DELETE"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers/groups"] });
      toast({ title: "Grupa nośników została usunięta" });
    },
  });

  useEffect(() => {
    if (editingGroup) {
      form.reset({
        code: editingGroup.code,
        name: editingGroup.name,
        description: editingGroup.description || "",
        isActive: editingGroup.isActive,
      });
    } else {
      form.reset({ code: "", name: "", description: "", isActive: true });
    }
  }, [editingGroup, form]);

  const onSubmit = (data: CarrierGroupFormData) => {
    if (editingGroup) {
      updateMutation.mutate({ id: editingGroup.id, data });
    } else {
      createMutation.mutate(data);
    }
  };

  return (
    <>
      <Card>
        <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-4">
          <CardTitle>Grupy Nośników</CardTitle>
          <Button onClick={() => { setEditingGroup(null); setIsDialogOpen(true); }} data-testid="button-add-group">
            <Plus className="w-4 h-4 mr-2" />Dodaj Grupę
          </Button>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex items-center justify-center p-8" data-testid="loading-groups">
              <Loader2 className="w-8 h-8 animate-spin text-muted-foreground" />
            </div>
          ) : groups.length === 0 ? (
            <p className="text-muted-foreground text-center p-8" data-testid="empty-groups">Brak grup nośników</p>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Kod</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Opis</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {groups.map((group) => (
                  <TableRow key={group.id} data-testid={`row-group-${group.id}`}>
                    <TableCell className="font-mono" data-testid={`text-group-code-${group.id}`}>{group.code}</TableCell>
                    <TableCell data-testid={`text-group-name-${group.id}`}>{group.name}</TableCell>
                    <TableCell className="text-muted-foreground" data-testid={`text-group-desc-${group.id}`}>{group.description || "-"}</TableCell>
                    <TableCell data-testid={`text-group-status-${group.id}`}>
                      <span className={group.isActive ? "text-green-600" : "text-gray-400"}>
                        {group.isActive ? "Aktywna" : "Nieaktywna"}
                      </span>
                    </TableCell>
                    <TableCell className="text-right space-x-2">
                      <Button size="icon" variant="ghost" onClick={() => { setEditingGroup(group); setIsDialogOpen(true); }} data-testid={`button-edit-group-${group.id}`}>
                        <Edit className="w-4 h-4" />
                      </Button>
                      <Button size="icon" variant="ghost" onClick={() => deleteMutation.mutate(group.id)} data-testid={`button-delete-group-${group.id}`}>
                        <Trash2 className="w-4 h-4" />
                      </Button>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={(open) => { setIsDialogOpen(open); if (!open) { setEditingGroup(null); form.reset(); } }}>
        <DialogContent data-testid="dialog-carrier-group">
          <DialogHeader>
            <DialogTitle>{editingGroup ? "Edytuj Grupę Nośników" : "Dodaj Grupę Nośników"}</DialogTitle>
            <DialogDescription>
              {editingGroup ? "Zaktualizuj informacje o grupie nośników" : "Utwórz nową grupę nośników (palety, wózki, etc.)"}
            </DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <FormField control={form.control} name="code" render={({ field }) => (
                <FormItem>
                  <FormLabel>Kod*</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder="np. PALLET, CART" data-testid="input-group-code" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )} />
              <FormField control={form.control} name="name" render={({ field }) => (
                <FormItem>
                  <FormLabel>Nazwa*</FormLabel>
                  <FormControl>
                    <Input {...field} placeholder="np. Palety, Wózki" data-testid="input-group-name" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )} />
              <FormField control={form.control} name="description" render={({ field }) => (
                <FormItem>
                  <FormLabel>Opis</FormLabel>
                  <FormControl>
                    <Textarea {...field} value={field.value || ""} placeholder="Opcjonalny opis grupy" data-testid="input-group-description" />
                  </FormControl>
                  <FormMessage />
                </FormItem>
              )} />
              <DialogFooter>
                <Button type="button" variant="outline" onClick={() => { setIsDialogOpen(false); setEditingGroup(null); form.reset(); }} data-testid="button-cancel-group">
                  Anuluj
                </Button>
                <Button type="submit" disabled={createMutation.isPending || updateMutation.isPending} data-testid="button-submit-group">
                  {(createMutation.isPending || updateMutation.isPending) && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
                  {editingGroup ? "Zapisz" : "Utwórz"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </>
  );
}

function CarriersTab() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingCarrier, setEditingCarrier] = useState<ProductionCarrier | null>(null);
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  
  // Check for edit query param
  const urlParams = new URLSearchParams(window.location.search);
  const editId = urlParams.get('edit');

  type CarrierFormData = z.infer<typeof carrierFormSchema>;
  const form = useForm<CarrierFormData>({
    resolver: zodResolver(carrierFormSchema),
    defaultValues: { carrierGroupId: 0, code: "", name: "", barcode: "", status: "available", capacity: 0, capacityUnit: "", notes: "", isActive: true },
  });

  const { data: groups = [] } = useQuery<ProductionCarrierGroup[]>({
    queryKey: ["/api/production/carriers/groups"],
  });

  const { data: carriers = [], isLoading } = useQuery<ProductionCarrier[]>({
    queryKey: ["/api/production/carriers"],
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => apiRequest("/api/production/carriers", "POST", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers"] });
      setIsDialogOpen(false);
      form.reset();
      toast({ title: "Nośnik został utworzony" });
    },
  });

  const updateMutation = useMutation({
    mutationFn: ({ id, data }: { id: number; data: any }) =>
      apiRequest(`/api/production/carriers/${id}`, "PATCH", data),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers"] });
      setIsDialogOpen(false);
      setEditingCarrier(null);
      form.reset();
      toast({ title: "Nośnik został zaktualizowany" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest(`/api/production/carriers/${id}`, "DELETE"),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/carriers"] });
      toast({ title: "Nośnik został usunięty" });
    },
  });

  const handleDuplicate = (carrier: ProductionCarrier) => {
    const timestamp = new Date().getTime().toString().slice(-6);
    const duplicatedData: CarrierFormData = {
      carrierGroupId: carrier.carrierGroupId,
      code: `${carrier.code}-${timestamp}`,
      name: `${carrier.name} (kopia)`,
      barcode: carrier.barcode ?? undefined,
      status: carrier.status || "available",
      capacity: carrier.capacity ?? undefined,
      capacityUnit: carrier.capacityUnit ?? undefined,
      notes: carrier.notes ?? undefined,
      isActive: carrier.isActive ?? true,
    };
    
    createMutation.mutate(duplicatedData);
  };

  useEffect(() => {
    if (editingCarrier) {
      form.reset({
        carrierGroupId: editingCarrier.carrierGroupId,
        code: editingCarrier.code,
        name: editingCarrier.name,
        barcode: editingCarrier.barcode ?? undefined,
        status: editingCarrier.status || "available",
        capacity: editingCarrier.capacity ?? undefined,
        capacityUnit: editingCarrier.capacityUnit ?? undefined,
        notes: editingCarrier.notes ?? undefined,
        isActive: editingCarrier.isActive ?? true,
      });
    } else {
      form.reset({ carrierGroupId: 0, code: "", name: "", barcode: undefined, status: "available", capacity: undefined, capacityUnit: undefined, notes: undefined, isActive: true });
    }
  }, [editingCarrier, form]);

  // Handle edit query param
  useEffect(() => {
    if (editId && carriers.length > 0 && !isDialogOpen) {
      const carrierToEdit = carriers.find(c => c.id === parseInt(editId));
      if (carrierToEdit) {
        setEditingCarrier(carrierToEdit);
        setIsDialogOpen(true);
        // Clear query param
        setLocation('/production/carriers');
      }
    }
  }, [editId, carriers, isDialogOpen, setLocation]);

  const onSubmit = (data: CarrierFormData) => {
    if (editingCarrier) {
      updateMutation.mutate({ id: editingCarrier.id, data });
    } else {
      createMutation.mutate(data);
    }
  };

  const getGroupName = (groupId: number) => groups.find(g => g.id === groupId)?.name || "Nieznana grupa";

  return (
    <>
      <Card>
        <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-4">
          <CardTitle>Nośniki</CardTitle>
          <Button onClick={() => { setEditingCarrier(null); setIsDialogOpen(true); }} data-testid="button-add-carrier">
            <Plus className="w-4 h-4 mr-2" />Dodaj Nośnik
          </Button>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex items-center justify-center p-8" data-testid="loading-carriers">
              <Loader2 className="w-8 h-8 animate-spin text-muted-foreground" />
            </div>
          ) : carriers.length === 0 ? (
            <p className="text-muted-foreground text-center p-8" data-testid="empty-carriers">Brak nośników</p>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Kod</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Grupa</TableHead>
                  <TableHead>Kod kreskowy</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Pojemność</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {carriers.map((carrier) => (
                  <TableRow key={carrier.id} data-testid={`row-carrier-${carrier.id}`}>
                    <TableCell className="font-mono" data-testid={`text-carrier-code-${carrier.id}`}>
                      <Link href={`/production/carriers/${carrier.id}`}>
                        <span className="text-primary hover:underline cursor-pointer">
                          {carrier.code}
                        </span>
                      </Link>
                    </TableCell>
                    <TableCell data-testid={`text-carrier-name-${carrier.id}`}>
                      <Link href={`/production/carriers/${carrier.id}`}>
                        <span className="text-primary hover:underline cursor-pointer">
                          {carrier.name}
                        </span>
                      </Link>
                    </TableCell>
                    <TableCell data-testid={`text-carrier-group-${carrier.id}`}>{getGroupName(carrier.carrierGroupId)}</TableCell>
                    <TableCell className="font-mono text-muted-foreground" data-testid={`text-carrier-barcode-${carrier.id}`}>{carrier.barcode || "-"}</TableCell>
                    <TableCell data-testid={`text-carrier-status-${carrier.id}`}>
                      <span className={carrier.status === "available" ? "text-green-600" : carrier.status === "in_use" ? "text-blue-600" : carrier.status === "maintenance" ? "text-orange-600" : "text-gray-400"}>
                        {carrier.status === "available" ? "Dostępny" : carrier.status === "in_use" ? "W użyciu" : carrier.status === "maintenance" ? "Konserwacja" : carrier.status === "damaged" ? "Uszkodzony" : "Zarchiwizowany"}
                      </span>
                    </TableCell>
                    <TableCell data-testid={`text-carrier-capacity-${carrier.id}`}>{carrier.capacity ? `${carrier.capacity} ${carrier.capacityUnit || ""}` : "-"}</TableCell>
                    <TableCell className="text-right">
                      <div className="flex gap-1 justify-end">
                        <Button size="icon" variant="ghost" onClick={() => { setEditingCarrier(carrier); setIsDialogOpen(true); }} data-testid={`button-edit-carrier-${carrier.id}`}>
                          <Edit className="w-4 h-4" />
                        </Button>
                        <Button size="icon" variant="ghost" onClick={() => handleDuplicate(carrier)} disabled={createMutation.isPending} data-testid={`button-duplicate-carrier-${carrier.id}`}>
                          <Copy className="w-4 h-4" />
                        </Button>
                        <Button size="icon" variant="ghost" onClick={() => deleteMutation.mutate(carrier.id)} data-testid={`button-delete-carrier-${carrier.id}`}>
                          <Trash2 className="w-4 h-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={(open) => { setIsDialogOpen(open); if (!open) { setEditingCarrier(null); form.reset(); } }}>
        <DialogContent className="max-w-2xl" data-testid="dialog-carrier">
          <DialogHeader>
            <DialogTitle>{editingCarrier ? "Edytuj Nośnik" : "Dodaj Nośnik"}</DialogTitle>
            <DialogDescription>
              {editingCarrier ? "Zaktualizuj informacje o nośniku" : "Utwórz nowy nośnik (konkretną paletę, wózek, etc.)"}
            </DialogDescription>
          </DialogHeader>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <FormField control={form.control} name="carrierGroupId" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Grupa*</FormLabel>
                    <Select onValueChange={(value) => field.onChange(parseInt(value))} value={field.value?.toString() || ""}>
                      <FormControl>
                        <SelectTrigger data-testid="select-carrier-group">
                          <SelectValue placeholder="Wybierz grupę" />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        {groups.map(group => (
                          <SelectItem key={group.id} value={group.id.toString()}>{group.name}</SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="code" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod*</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. PAL-001" data-testid="input-carrier-code" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="name" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa*</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="np. Paleta EUR 001" data-testid="input-carrier-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="barcode" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kod kreskowy</FormLabel>
                    <FormControl>
                      <Input {...field} value={field.value || ""} placeholder="np. 1234567890123" data-testid="input-carrier-barcode" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="status" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Status*</FormLabel>
                    <Select onValueChange={field.onChange} value={field.value || "available"}>
                      <FormControl>
                        <SelectTrigger data-testid="select-carrier-status">
                          <SelectValue />
                        </SelectTrigger>
                      </FormControl>
                      <SelectContent>
                        <SelectItem value="available">Dostępny</SelectItem>
                        <SelectItem value="in_use">W użyciu</SelectItem>
                        <SelectItem value="maintenance">Konserwacja</SelectItem>
                        <SelectItem value="damaged">Uszkodzony</SelectItem>
                        <SelectItem value="archived">Zarchiwizowany</SelectItem>
                      </SelectContent>
                    </Select>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="capacity" render={({ field }) => (
                  <FormItem>
                    <FormLabel>Pojemność</FormLabel>
                    <FormControl>
                      <Input {...field} value={field.value ?? ""} type="number" step="0.01" placeholder="np. 1000" onChange={field.onChange} data-testid="input-carrier-capacity" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
                <FormField control={form.control} name="notes" render={({ field }) => (
                  <FormItem className="col-span-2">
                    <FormLabel>Uwagi</FormLabel>
                    <FormControl>
                      <Textarea {...field} value={field.value || ""} placeholder="Dodatkowe informacje o nośniku" data-testid="input-carrier-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )} />
              </div>
              <DialogFooter>
                <Button type="button" variant="outline" onClick={() => { setIsDialogOpen(false); setEditingCarrier(null); form.reset(); }} data-testid="button-cancel-carrier">
                  Anuluj
                </Button>
                <Button type="submit" disabled={createMutation.isPending || updateMutation.isPending} data-testid="button-submit-carrier">
                  {(createMutation.isPending || updateMutation.isPending) && <Loader2 className="w-4 h-4 mr-2 animate-spin" />}
                  {editingCarrier ? "Zapisz" : "Utwórz"}
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </>
  );
}
