import { useQuery } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Package, Search, ChevronDown, ChevronUp, Image as ImageIcon, ArrowUpDown, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, ExternalLink } from "lucide-react";
import { useState, useEffect, Fragment } from "react";
import { Skeleton } from "@/components/ui/skeleton";
import { useLocation, Link } from "wouter";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { ComponentTemplateThumbnail } from "@/components/component-template-thumbnail";

interface BomComponent {
  id: number;
  product_bom_id: number;
  component_template_id: number;
  template_id: number | null;
  generated_name: string;
  quantity: number;
  component_type: string;
  template_cz1: string | null;
  template_cz2: string | null;
  furniture_type: string;
  template_base_length: number;
  template_base_width: number;
  thickness: number;
  edge1: boolean;
  edge2: boolean;
  edge3: boolean;
  edge4: boolean;
  calculated_length: number;
  calculated_width: number;
  calculated_color: string;
  color_hex: string | null;
  // Metrics fields
  board_area_m2?: number | null;
  edging_length_m?: number | null;
  glue_amount_g?: number | null;
  board_material_id?: number | null;
  edging_material_id?: number | null;
  glue_material_id?: number | null;
  board_cost?: number | null;
  edging_cost?: number | null;
  glue_cost?: number | null;
  total_material_cost?: number | null;
  board_material_name?: string | null;
  edging_material_name?: string | null;
  glue_material_name?: string | null;
  // Product fields
  product_id: number;
  product_sku: string;
  product_title: string;
  product_type: string;
  product_color: string;
  product_length: number;
  product_width: number;
  product_image_url: string | null;
  odoo_product_id: number | null;
  created_at?: string;
  updated_at?: string;
}

const STORAGE_KEY = 'bom-components-filters';

// Helper function to determine text color based on background luminance
function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}

// Helper function to determine if color needs visible border
function needsVisibleBorder(hexColor: string | null | undefined): boolean {
  if (!hexColor || !hexColor.startsWith('#')) return false;
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance < 30 || luminance > 230;
}

export default function BomComponentsPage() {
  const [, setLocation] = useLocation();
  
  // Load filters from localStorage
  const loadFilters = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return {};
  };

  const savedFilters = loadFilters();
  
  const [searchQuery, setSearchQuery] = useState(savedFilters.searchQuery || "");
  const [filtersExpanded, setFiltersExpanded] = useState(false);

  // Pagination and sorting
  const [page, setPage] = useState(savedFilters.page || 1);
  const [limit, setLimit] = useState(savedFilters.limit || 50);
  const [sortBy, setSortBy] = useState(savedFilters.sortBy || 'generated_name');
  const [sortOrder, setSortOrder] = useState<'ASC' | 'DESC'>(savedFilters.sortOrder || 'ASC');
  const [compactMode, setCompactMode] = useState(savedFilters.compactMode ?? true);
  const [groupBy, setGroupBy] = useState<string | null>(savedFilters.groupBy || null);
  const [expandedGroups, setExpandedGroups] = useState<Set<string>>(
    new Set(savedFilters.expandedGroups || [])
  );

  // Save filters to localStorage
  useEffect(() => {
    const filtersToSave = {
      searchQuery,
      page,
      limit,
      sortBy,
      sortOrder,
      compactMode,
      groupBy,
      expandedGroups: Array.from(expandedGroups)
    };
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(filtersToSave));
    } catch (error) {
      console.error('Error saving filters to localStorage:', error);
    }
  }, [searchQuery, page, limit, sortBy, sortOrder, compactMode, groupBy, expandedGroups]);

  // Fetch Odoo config
  const { data: odooConfig } = useQuery<{ url: string | null; userUrl: string | null }>({
    queryKey: ['/api/odoo/url'],
  });

  // Fetch components
  const { data, isLoading } = useQuery<{ components: BomComponent[]; total: number }>({
    queryKey: ['/api/bom-components', { 
      search: searchQuery, 
      limit, 
      offset: (page - 1) * limit,
      sortBy,
      sortOrder,
      groupBy
    }],
  });

  const components = data?.components || [];
  const total = data?.total || 0;
  const totalPages = Math.ceil(total / limit);

  // Group components if groupBy is set
  const groupedComponents = () => {
    if (!groupBy) return null;

    const groups: Record<string, BomComponent[]> = {};
    components.forEach((comp) => {
      let groupKey = '';
      switch (groupBy) {
        case 'product_type':
          groupKey = comp.product_type || 'Bez typu';
          break;
        case 'product_sku':
          groupKey = comp.product_sku || 'Bez SKU';
          break;
        case 'calculated_length':
          groupKey = `Długość ${comp.calculated_length}mm`;
          break;
        case 'calculated_width':
          groupKey = `Szerokość ${comp.calculated_width}mm`;
          break;
        case 'calculated_color':
          groupKey = comp.calculated_color || 'Bez koloru';
          break;
        default:
          groupKey = 'Inne';
      }
      if (!groups[groupKey]) {
        groups[groupKey] = [];
      }
      groups[groupKey].push(comp);
    });
    return groups;
  };

  const toggleGroup = (groupKey: string) => {
    setExpandedGroups(prev => {
      const newSet = new Set(prev);
      if (newSet.has(groupKey)) {
        newSet.delete(groupKey);
      } else {
        newSet.add(groupKey);
      }
      return newSet;
    });
  };

  const toggleAllGroups = () => {
    const groups = groupedComponents();
    if (!groups) return;
    
    const allKeys = Object.keys(groups);
    if (expandedGroups.size === allKeys.length) {
      setExpandedGroups(new Set());
    } else {
      setExpandedGroups(new Set(allKeys));
    }
  };

  // Pagination component
  const PaginationControls = () => {
    const startItem = ((page - 1) * limit) + 1;
    const endItem = Math.min(page * limit, total);
    
    return (
      <div className="flex flex-col md:flex-row items-center justify-between gap-2 md:gap-4">
        <div className="text-xs md:text-sm text-muted-foreground order-2 md:order-1">
          <span className="hidden md:inline">
            Wyniki {startItem} - {endItem} z {total}
          </span>
          <span className="md:hidden">
            {startItem}-{endItem} / {total}
          </span>
        </div>
        <div className="flex items-center gap-1 md:gap-2 order-1 md:order-2">
          <Button
            variant="outline"
            size="sm"
            onClick={() => setPage(1)}
            disabled={page === 1}
            data-testid="button-first-page"
            className="h-8 w-8 p-0"
          >
            <ChevronsLeft className="h-4 w-4" />
          </Button>
          <Button
            variant="outline"
            size="sm"
            onClick={() => setPage((p: number) => Math.max(1, p - 1))}
            disabled={page === 1}
            data-testid="button-prev-page"
            className="h-8 w-8 p-0"
          >
            <ChevronLeft className="h-4 w-4" />
          </Button>
          <span className="text-sm px-2 md:px-4">
            {page} / {totalPages}
          </span>
          <Button
            variant="outline"
            size="sm"
            onClick={() => setPage((p: number) => Math.min(totalPages, p + 1))}
            disabled={page === totalPages}
            data-testid="button-next-page"
            className="h-8 w-8 p-0"
          >
            <ChevronRight className="h-4 w-4" />
          </Button>
          <Button
            variant="outline"
            size="sm"
            onClick={() => setPage(totalPages)}
            disabled={page === totalPages}
            data-testid="button-last-page"
            className="h-8 w-8 p-0"
          >
            <ChevronsRight className="h-4 w-4" />
          </Button>
        </div>
      </div>
    );
  };

  return (
    <div className="flex flex-col h-full w-full max-w-full overflow-x-hidden">
      {/* Header */}
      <Card className="rounded-none border-x-0 border-t-0">
        <CardHeader className="h-10 px-3 py-0 flex items-center">
          <div className="flex items-center gap-2 w-full">
            <Package className="h-4 w-4 text-primary flex-shrink-0" />
            <CardTitle className="text-base flex-1" data-testid="title-bom-components">
              Formatki BOM
            </CardTitle>
            <span className="text-xs text-muted-foreground">
              Wszystkie komponenty z list materiałowych
            </span>
          </div>
        </CardHeader>
      </Card>

      <div className="flex-1 overflow-auto">
        <div className="px-3 md:px-6 py-4 space-y-4 max-w-full">
          {/* Search and filters */}
          <div className="flex flex-col gap-2">
            <div className="flex gap-2">
              <div className="relative flex-1">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Szukaj... (użyj ; dla AND, , dla OR)"
                  value={searchQuery}
                  onChange={(e) => {
                    setSearchQuery(e.target.value);
                    setPage(1);
                  }}
                  className="pl-10"
                  data-testid="input-search"
                />
              </div>
              <Button
                variant="outline"
                size="icon"
                onClick={() => setFiltersExpanded(!filtersExpanded)}
                data-testid="button-toggle-filters"
              >
                {filtersExpanded ? <ChevronUp className="h-4 w-4" /> : <ChevronDown className="h-4 w-4" />}
              </Button>
            </div>

            {/* Active Filters Display */}
            {searchQuery && (
              <div className="flex flex-wrap gap-2 items-center">
                <span className="text-xs text-muted-foreground">Aktywne filtry:</span>
                {searchQuery.split(';').map((term: string, index: number) => {
                  const trimmedTerm = term.trim();
                  if (!trimmedTerm) return null;
                  
                  return (
                    <Badge
                      key={index}
                      variant="secondary"
                      className="gap-1 pr-1"
                      data-testid={`badge-active-filter-${index}`}
                    >
                      <span className="text-xs">{trimmedTerm}</span>
                      <Button
                        variant="ghost"
                        size="sm"
                        className="h-4 w-4 p-0 hover:bg-transparent"
                        onClick={() => {
                          // Remove this specific term from search
                          const terms = searchQuery.split(';')
                            .map((t: string) => t.trim())
                            .filter((t: string, i: number) => i !== index && t !== '');
                          setSearchQuery(terms.join(';'));
                          setPage(1);
                        }}
                        data-testid={`button-clear-filter-${index}`}
                      >
                        ×
                      </Button>
                    </Badge>
                  );
                })}
              </div>
            )}
          </div>
        </div>

        {/* Filters Card */}
        {filtersExpanded && (
          <Card>
            <CardHeader className="pb-3">
              <CardTitle className="text-base">Filtry i widok</CardTitle>
            </CardHeader>
            <CardContent className="grid grid-cols-2 md:grid-cols-4 gap-3">
              <div className="space-y-0.5">
                <Label className="text-[10px]">Tryb kompaktowy</Label>
                <div className="flex items-center space-x-2 h-7">
                  <Switch
                    checked={compactMode}
                    onCheckedChange={setCompactMode}
                    data-testid="switch-compact-mode"
                  />
                  <Label className="text-xs">{compactMode ? 'Tak' : 'Nie'}</Label>
                </div>
              </div>

              <div className="space-y-0.5">
                <Label className="text-[10px]">Grupuj po</Label>
                <Select value={groupBy || 'none'} onValueChange={(val) => setGroupBy(val === 'none' ? null : val)}>
                  <SelectTrigger data-testid="select-group-by" className="h-7 text-xs">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="none">Brak</SelectItem>
                    <SelectItem value="product_sku">SKU produktu</SelectItem>
                    <SelectItem value="product_type">Typ mebla</SelectItem>
                    <SelectItem value="calculated_length">Długość</SelectItem>
                    <SelectItem value="calculated_width">Szerokość</SelectItem>
                    <SelectItem value="calculated_color">Kolor</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-0.5">
                <Label className="text-[10px]">Sortuj po</Label>
                <Select value={sortBy} onValueChange={setSortBy}>
                  <SelectTrigger data-testid="select-sort-by" className="h-7 text-xs">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="generated_name">Nazwa</SelectItem>
                    <SelectItem value="calculated_length">Długość</SelectItem>
                    <SelectItem value="calculated_width">Szerokość</SelectItem>
                    <SelectItem value="calculated_color">Kolor</SelectItem>
                    <SelectItem value="product_type">Typ</SelectItem>
                    <SelectItem value="thickness">Grubość</SelectItem>
                    <SelectItem value="created_at">Data utworzenia</SelectItem>
                    <SelectItem value="updated_at">Data modyfikacji</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-0.5">
                <Label className="text-[10px]">Kolejność</Label>
                <Select value={sortOrder} onValueChange={(val) => setSortOrder(val as 'ASC' | 'DESC')}>
                  <SelectTrigger data-testid="select-sort-order" className="h-7 text-xs">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="ASC">Rosnąco</SelectItem>
                    <SelectItem value="DESC">Malejąco</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-0.5">
                <Label className="text-[10px]">Wierszy na stronie</Label>
                <Select value={limit.toString()} onValueChange={(val) => { setLimit(parseInt(val)); setPage(1); }}>
                  <SelectTrigger data-testid="select-limit" className="h-7 text-xs">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="25">25</SelectItem>
                    <SelectItem value="50">50</SelectItem>
                    <SelectItem value="100">100</SelectItem>
                    <SelectItem value="200">200</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </CardContent>
          </Card>
        )}

        {/* Table */}
        <Card>
          <CardContent className="pt-6">
            {isLoading ? (
              <div className="space-y-2">
                {[...Array(10)].map((_, i) => (
                  <Skeleton key={i} className="h-16 w-full" />
                ))}
              </div>
            ) : components.length === 0 ? (
              <div className="text-center py-12">
                <Package className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                <h3 className="text-lg font-medium mb-2">Brak formatek</h3>
                <p className="text-muted-foreground">
                  {searchQuery
                    ? "Nie znaleziono formatek pasujących do wyszukiwania"
                    : "Brak wygenerowanych BOM w systemie"}
                </p>
              </div>
            ) : (
              <>
                <div className="mb-4">
                  <PaginationControls />
                </div>

                <div className="rounded-md border overflow-x-auto">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        {/* Sticky columns */}
                        <TableHead className="sticky left-0 z-20 bg-background border-r w-16">#</TableHead>
                        <TableHead className="sticky left-16 z-20 bg-background border-r w-20 font-mono text-xs">ID</TableHead>
                        
                        {/* Scrollable columns */}
                        <TableHead className="w-20">Zdjęcie</TableHead>
                        <TableHead className="min-w-[200px]">Nazwa</TableHead>
                        <TableHead className="w-24">Kolor</TableHead>
                        <TableHead className="w-20 text-right">CZ1</TableHead>
                        <TableHead className="w-20 text-right">CZ2</TableHead>
                        <TableHead className="w-20 text-right">L</TableHead>
                        <TableHead className="w-20 text-right">W</TableHead>
                        <TableHead className="w-20 text-right">H</TableHead>
                        <TableHead className="w-24">Typ</TableHead>
                        <TableHead className="min-w-[150px]">Płyta</TableHead>
                        <TableHead className="min-w-[150px]">Obrzeże</TableHead>
                        <TableHead className="w-24">Krawędzie</TableHead>
                        <TableHead className="w-24 text-right">Koszt</TableHead>
                        <TableHead className="min-w-[150px]">Produkt</TableHead>
                        <TableHead className="w-20">Zdjęcie prod.</TableHead>
                        <TableHead className="w-32">Utworzono</TableHead>
                        <TableHead className="w-32">Zaktualizowano</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {groupBy ? (
                        (() => {
                          const groups = groupedComponents();
                          if (!groups) return null;

                          return Object.entries(groups).map(([groupKey, groupComponents]) => (
                            <Fragment key={groupKey}>
                              {/* Group Header */}
                              <TableRow
                                className="bg-muted/30 font-semibold hover-elevate cursor-pointer"
                                onClick={() => toggleGroup(groupKey)}
                                data-testid={`row-group-${groupKey}`}
                              >
                                <TableCell colSpan={17} className="py-2">
                                  <div className="flex items-center gap-2">
                                    {expandedGroups.has(groupKey) ? (
                                      <ChevronDown className="h-4 w-4" />
                                    ) : (
                                      <ChevronUp className="h-4 w-4" />
                                    )}
                                    <span>{groupKey}</span>
                                    <Badge variant="secondary" className="ml-2">{groupComponents.length}</Badge>
                                  </div>
                                </TableCell>
                              </TableRow>

                              {/* Group Rows */}
                              {expandedGroups.has(groupKey) && groupComponents.map((comp, idx) => (
                                <ComponentRow key={comp.id} component={comp} compactMode={compactMode} odooConfig={odooConfig} index={idx} />
                              ))}
                            </Fragment>
                          ));
                        })()
                      ) : (
                        components.map((comp, idx) => (
                          <ComponentRow key={comp.id} component={comp} compactMode={compactMode} odooConfig={odooConfig} index={idx} />
                        ))
                      )}
                    </TableBody>
                  </Table>
                </div>

                <div className="mt-4">
                  <PaginationControls />
                </div>
              </>
            )}
          </CardContent>
        </Card>
      </div>
    </div>
  );
}

function ComponentRow({ 
  component, 
  compactMode, 
  odooConfig,
  index
}: { 
  component: BomComponent; 
  compactMode: boolean;
  odooConfig?: { url: string | null; userUrl: string | null } | null;
  index: number;
}) {
  const [, setLocation] = useLocation();

  return (
    <TableRow 
      className={`cursor-pointer hover-elevate ${compactMode ? 'h-8' : ''}`}
      onClick={() => setLocation(`/bom-components/${component.id}`)}
      data-testid={`row-component-${component.id}`}
    >
      {/* #1 - Position (sticky) */}
      <TableCell className="sticky left-0 z-10 bg-background border-r text-center font-mono text-sm text-muted-foreground">
        {index + 1}
      </TableCell>

      {/* #2 - ID (sticky) */}
      <TableCell className="sticky left-16 z-10 bg-background border-r text-center font-mono text-xs text-muted-foreground">
        {component.id}
      </TableCell>

      {/* #3 - Zdjęcie formatki */}
      <TableCell>
        <ComponentTemplateThumbnail
          baseLength={component.template_base_length || 100}
          baseWidth={component.template_base_width || 100}
          edge1={component.edge1 || false}
          edge2={component.edge2 || false}
          edge3={component.edge3 || false}
          edge4={component.edge4 || false}
          size="sm"
        />
      </TableCell>

      {/* #4 - Nazwa */}
      <TableCell className="font-mono font-medium text-sm">
        <div className="flex items-center gap-2">
          <span>{component.generated_name}</span>
          {component.odoo_product_id && odooConfig?.userUrl && (
            <a
              href={`${odooConfig.userUrl.replace(/\/$/, '')}/products/${component.odoo_product_id}`}
              target="_blank"
              rel="noopener noreferrer"
              onClick={(e) => e.stopPropagation()}
              className="text-purple-600 dark:text-purple-400 hover:text-purple-700 dark:hover:text-purple-300"
              data-testid={`link-formatka-odoo-${component.id}`}
            >
              <ExternalLink className="w-3.5 h-3.5" />
            </a>
          )}
        </div>
      </TableCell>

      {/* #5 - Kolor */}
      <TableCell>
        {component.calculated_color ? (() => {
          const bgColor = component.color_hex || '#94A3B8';
          const textColor = getTextColorForBackground(bgColor);
          const hasBorder = needsVisibleBorder(bgColor);
          return (
            <Badge 
              variant="outline" 
              className={`text-xs h-5 px-1.5 ${hasBorder ? '' : 'border-transparent'}`}
              style={{ backgroundColor: bgColor, color: textColor }}
            >
              {component.calculated_color}
            </Badge>
          );
        })() : '-'}
      </TableCell>

      {/* #6 - CZ1 */}
      <TableCell className="text-right font-mono text-sm">
        {component.template_cz1 || '-'}
      </TableCell>

      {/* #7 - CZ2 */}
      <TableCell className="text-right font-mono text-sm">
        {component.template_cz2 || '-'}
      </TableCell>

      {/* #8 - L */}
      <TableCell className="text-right font-mono text-sm">
        {component.calculated_length ?? '-'}
      </TableCell>

      {/* #9 - W */}
      <TableCell className="text-right font-mono text-sm">
        {component.calculated_width ?? '-'}
      </TableCell>

      {/* #10 - H */}
      <TableCell className="text-right font-mono text-sm">
        {component.thickness ?? '-'}
      </TableCell>

      {/* #11 - Typ */}
      <TableCell>
        {component.component_type ? (
          <Badge variant="outline" className="text-xs">
            {component.component_type}
          </Badge>
        ) : '-'}
      </TableCell>

      {/* #12 - Płyta */}
      <TableCell className="text-sm">
        {component.board_material_name || '-'}
      </TableCell>

      {/* #13 - Obrzeże */}
      <TableCell className="text-sm">
        {component.edging_material_name || '-'}
      </TableCell>

      {/* #14 - Krawędzie */}
      <TableCell className="text-center">
        {(component.edge1 || component.edge2 || component.edge3 || component.edge4) ? (
          <Badge variant="secondary" className="text-xs font-mono">
            {[
              component.edge1 && '1',
              component.edge2 && '2',
              component.edge3 && '3',
              component.edge4 && '4'
            ].filter(Boolean).join(',')}
          </Badge>
        ) : '-'}
      </TableCell>

      {/* #15 - Koszt */}
      <TableCell className="text-right font-mono text-sm">
        {component.total_material_cost ? 
          `${component.total_material_cost.toFixed(2)} zł` : 
          '-'
        }
      </TableCell>

      {/* #16 - Produkt */}
      <TableCell>
        <Link 
          href={`/catalog-products/${component.product_id}`}
          onClick={(e) => e.stopPropagation()}
        >
          <div className="hover:underline text-primary flex items-center gap-1">
            <span className="font-mono text-xs">{component.product_sku}</span>
            <ExternalLink className="h-3 w-3" />
          </div>
        </Link>
        <div className="text-xs text-muted-foreground truncate max-w-[200px]">
          {component.product_title}
        </div>
      </TableCell>

      {/* #17 - Zdjęcie prod. */}
      <TableCell>
        {component.product_image_url ? (
          <img 
            src={component.product_image_url} 
            alt={component.product_title}
            className="w-[40px] h-[40px] object-cover rounded"
            data-testid={`img-product-${component.product_id}`}
          />
        ) : (
          <div className="w-[40px] h-[40px] bg-muted rounded flex items-center justify-center">
            <ImageIcon className="h-4 w-4 text-muted-foreground" />
          </div>
        )}
      </TableCell>

      {/* #18 - Created At */}
      <TableCell className="text-xs text-muted-foreground">
        {component.created_at ? new Date(component.created_at).toLocaleDateString('pl-PL', { 
          year: '2-digit', month: '2-digit', day: '2-digit', 
          hour: '2-digit', minute: '2-digit' 
        }) : '-'}
      </TableCell>

      {/* #19 - Updated At */}
      <TableCell className="text-xs text-muted-foreground">
        {component.updated_at ? new Date(component.updated_at).toLocaleDateString('pl-PL', { 
          year: '2-digit', month: '2-digit', day: '2-digit', 
          hour: '2-digit', minute: '2-digit' 
        }) : '-'}
      </TableCell>
    </TableRow>
  );
}
