import { Button } from "@/components/ui/button";
import { Calendar } from "lucide-react";
import { cn } from "@/lib/utils";

export type DateRangePreset = "today" | "yesterday" | "dayBeforeYesterday" | "last2days" | "last3days" | "last7days" | "last14days" | "last30days" | "last90days";

interface DateRangeFilterProps {
  value: DateRangePreset;
  onChange: (preset: DateRangePreset) => void;
}

const PRESETS: { value: DateRangePreset; label: string }[] = [
  { value: "today", label: "Dzisiaj" },
  { value: "yesterday", label: "Wczoraj" },
  { value: "dayBeforeYesterday", label: "Przedwczoraj" },
  { value: "last2days", label: "Ostatnie 2 dni" },
  { value: "last3days", label: "Ostatnie 3 dni" },
  { value: "last7days", label: "Tydzień" },
  { value: "last14days", label: "14 dni" },
  { value: "last30days", label: "Miesiąc" },
  { value: "last90days", label: "90 dni" },
];

export function DateRangeFilter({ value, onChange }: DateRangeFilterProps) {
  return (
    <div className="flex flex-wrap items-center gap-2">
      <div className="flex items-center gap-2 text-sm text-muted-foreground">
        <Calendar className="h-4 w-4" />
        <span>Zakres:</span>
      </div>
      <div className="flex flex-wrap gap-2">
        {PRESETS.map((preset) => (
          <Button
            key={preset.value}
            size="sm"
            variant={value === preset.value ? "default" : "outline"}
            onClick={() => onChange(preset.value)}
            data-testid={`button-daterange-${preset.value}`}
            className="h-8"
          >
            {preset.label}
          </Button>
        ))}
      </div>
    </div>
  );
}

// Helper function to convert preset to days count
export function presetToDays(preset: DateRangePreset): number {
  switch (preset) {
    case "today": return 1;
    case "yesterday": return 1;
    case "dayBeforeYesterday": return 1;
    case "last2days": return 2;
    case "last3days": return 3;
    case "last7days": return 7;
    case "last14days": return 14;
    case "last30days": return 30;
    case "last90days": return 90;
  }
}

// Helper function to get date range from preset
export function getDateRangeFromPreset(preset: DateRangePreset): { from: string; to: string } {
  const today = new Date();
  today.setHours(0, 0, 0, 0);
  
  const to = new Date(today);
  const from = new Date(today);

  switch (preset) {
    case "today":
      // Same day
      break;
    case "yesterday":
      from.setDate(from.getDate() - 1);
      to.setDate(to.getDate() - 1);
      break;
    case "dayBeforeYesterday":
      from.setDate(from.getDate() - 2);
      to.setDate(to.getDate() - 2);
      break;
    case "last2days":
      from.setDate(from.getDate() - 1);
      break;
    case "last3days":
      from.setDate(from.getDate() - 2);
      break;
    case "last7days":
      from.setDate(from.getDate() - 6);
      break;
    case "last14days":
      from.setDate(from.getDate() - 13);
      break;
    case "last30days":
      from.setDate(from.getDate() - 29);
      break;
    case "last90days":
      from.setDate(from.getDate() - 89);
      break;
  }

  return {
    from: from.toISOString().split('T')[0],
    to: to.toISOString().split('T')[0],
  };
}
