from odoo import _, api, fields, models
from odoo.exceptions import UserError

from ..models.allegro_client import AllegroError


class AllegroAccountPair(models.TransientModel):
    _name = 'trilab.allegro.pair'
    _description = 'Pair with Allegro user account'

    overwrite = fields.Boolean(
        'Overwrite Existing Pair', help='If you check this box, your new pair will overwrite saved one.'
    )

    error_message = fields.Char('Error Message')

    state = fields.Selection(
        [
            ('init', 'init'),
            ('already_paired', 'already_paired'),
            ('processing', 'processing'),
            ('error', 'error'),
            ('done', 'done'),
        ],
        string='Status',
        default='init',
    )

    account_id = fields.Many2one('trilab.market.account', string='Account', required=True, readonly=True)
    user_data = fields.Char(string='User Data')

    pair_code = fields.Char(string='Pair Code', store=True)
    pair_url = fields.Char(string='Pair URL', store=True)
    device_code = fields.Char(string='Device Code', store=True)

    def get_pair_data(self, client=None) -> dict:
        self.ensure_one()

        try:
            response = self.account_id.get_client(client).get_pairing_data()
        except AllegroError as error:
            raise UserError(_('Invalid pairing data: %s', str(error)))

        return {
            'pair_code': response.get('user_code'),
            'pair_url': response.get('verification_uri_complete'),
            'device_code': response.get('device_code'),
        }

    @api.model_create_multi
    def create(self, vals_list: list):
        result = super().create(vals_list)

        for wizard_id in result.filtered(lambda w_id: w_id.state == 'init'):
            client = wizard_id.account_id.get_client()

            response = {}

            if wizard_id.account_id.state == 'paired':
                try:
                    response = client.me()
                except AllegroError:
                    pass

            if response and response.get('id'):
                # paired and got valid data
                wizard_id.user_data = _('id: %s, name: %s', response.get('id'), response.get('login'))
                wizard_id.state = 'already_paired'

            else:
                # not paired, no data or error
                wizard_id.write(
                    {'state': 'processing', 'user_data': _('*unknown*')} | wizard_id.get_pair_data(client=client)
                )

        return result

    def reload_dialog(self):
        return {
            'res_id': self.id,
            'res_model': 'trilab.allegro.pair',
            'type': 'ir.actions.act_window',
            'target': 'new',
            'view_mode': 'form',
        }

    def action_pair_again(self):
        self.state = 'processing'

        # remove old settings
        self.account_id.reset_state()

        # update pairing data
        self.update(self.get_pair_data())

        return self.reload_dialog()

    def action_verify(self):
        for wizard_id in self:
            if wizard_id.state == 'processing':
                try:
                    wizard_id.account_id.get_client().get_token(device_code=self.device_code)
                except AllegroError as error:
                    raise UserError(str(error))

                wizard_id.account_id.allegro_check_account_options()

                wizard_id.state = 'done'

        return self.reload_dialog()
