import { Pool, PoolClient } from "pg";
import type { ProductionRoutingOperation, InsertProductionRoutingOperation } from "@shared/schema";
import { recalculateTotalTime } from "./routings";

// Transaction helper
export async function withTransaction<T>(
  pool: Pool,
  callback: (client: PoolClient) => Promise<T>
): Promise<T> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');
    const result = await callback(client);
    await client.query('COMMIT');
    return result;
  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

export async function getOperationsByRoutingId(pool: Pool, routingId: number): Promise<ProductionRoutingOperation[]> {
  const result = await pool.query(`
    SELECT * FROM production.production_routing_operations 
    WHERE routing_id = $1
    ORDER BY sequence ASC
  `, [routingId]);
  
  return result.rows.map(row => ({
    id: row.id,
    routingId: row.routing_id,
    sequence: row.sequence,
    code: row.code,
    name: row.name,
    description: row.description,
    workCenterId: row.work_center_id,
    normId: row.norm_id,
    estimatedTime: row.estimated_time,
    setupTime: row.setup_time,
    allowParallel: row.allow_parallel,
    prerequisiteOperationIds: row.prerequisite_operation_ids,
    instructions: row.instructions,
    qualityCheckRequired: row.quality_check_required,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  }));
}

export async function getOperationById(pool: Pool, id: number): Promise<ProductionRoutingOperation | null> {
  const result = await pool.query(`
    SELECT * FROM production.production_routing_operations WHERE id = $1
  `, [id]);
  
  if (result.rows.length === 0) return null;
  
  const row = result.rows[0];
  return {
    id: row.id,
    routingId: row.routing_id,
    sequence: row.sequence,
    code: row.code,
    name: row.name,
    description: row.description,
    workCenterId: row.work_center_id,
    normId: row.norm_id,
    estimatedTime: row.estimated_time,
    setupTime: row.setup_time,
    allowParallel: row.allow_parallel,
    prerequisiteOperationIds: row.prerequisite_operation_ids,
    instructions: row.instructions,
    qualityCheckRequired: row.quality_check_required,
    isActive: row.is_active,
    createdAt: row.created_at,
    updatedAt: row.updated_at,
  };
}

export async function createOperation(pool: Pool, data: InsertProductionRoutingOperation): Promise<ProductionRoutingOperation> {
  return withTransaction(pool, async (client) => {
    // Get next sequence number if not provided
    let sequence = data.sequence;
    if (sequence === undefined || sequence === null) {
      const maxSeqResult = await client.query(`
        SELECT COALESCE(MAX(sequence), 0) + 1 AS next_seq
        FROM production.production_routing_operations
        WHERE routing_id = $1
      `, [data.routingId]);
      sequence = maxSeqResult.rows[0].next_seq;
    }

    const result = await client.query(`
      INSERT INTO production.production_routing_operations 
      (routing_id, sequence, code, name, description, work_center_id, norm_id, 
       estimated_time, setup_time, allow_parallel, prerequisite_operation_ids, 
       instructions, quality_check_required, is_active)
      VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14)
      RETURNING *
    `, [
      data.routingId,
      sequence,
      data.code,
      data.name,
      data.description ?? null,
      data.workCenterId ?? null,
      data.normId ?? null,
      data.estimatedTime ?? 0,
      data.setupTime ?? 0,
      data.allowParallel ?? false,
      data.prerequisiteOperationIds ? JSON.stringify(data.prerequisiteOperationIds) : null,
      data.instructions ?? null,
      data.qualityCheckRequired ?? false,
      data.isActive ?? true,
    ]);

    const row = result.rows[0];
    
    // Recalculate total time (using transaction client)
    await recalculateTotalTime(client, data.routingId);
    
    return {
      id: row.id,
      routingId: row.routing_id,
      sequence: row.sequence,
      code: row.code,
      name: row.name,
      description: row.description,
      workCenterId: row.work_center_id,
      normId: row.norm_id,
      estimatedTime: row.estimated_time,
      setupTime: row.setup_time,
      allowParallel: row.allow_parallel,
      prerequisiteOperationIds: row.prerequisite_operation_ids,
      instructions: row.instructions,
      qualityCheckRequired: row.quality_check_required,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  });
}

export async function updateOperation(pool: Pool, id: number, data: Partial<InsertProductionRoutingOperation>): Promise<ProductionRoutingOperation | null> {
  return withTransaction(pool, async (client) => {
    const updates: string[] = [];
    const values: any[] = [];
    let paramIndex = 1;

    if (data.sequence !== undefined) {
      updates.push(`sequence = $${paramIndex++}`);
      values.push(data.sequence);
    }
    if (data.code !== undefined) {
      updates.push(`code = $${paramIndex++}`);
      values.push(data.code);
    }
    if (data.name !== undefined) {
      updates.push(`name = $${paramIndex++}`);
      values.push(data.name);
    }
    if (data.description !== undefined) {
      updates.push(`description = $${paramIndex++}`);
      values.push(data.description);
    }
    if (data.workCenterId !== undefined) {
      updates.push(`work_center_id = $${paramIndex++}`);
      values.push(data.workCenterId);
    }
    if (data.normId !== undefined) {
      updates.push(`norm_id = $${paramIndex++}`);
      values.push(data.normId);
    }
    if (data.estimatedTime !== undefined) {
      updates.push(`estimated_time = $${paramIndex++}`);
      values.push(data.estimatedTime);
    }
    if (data.setupTime !== undefined) {
      updates.push(`setup_time = $${paramIndex++}`);
      values.push(data.setupTime);
    }
    if (data.allowParallel !== undefined) {
      updates.push(`allow_parallel = $${paramIndex++}`);
      values.push(data.allowParallel);
    }
    if (data.prerequisiteOperationIds !== undefined) {
      updates.push(`prerequisite_operation_ids = $${paramIndex++}`);
      values.push(data.prerequisiteOperationIds ? JSON.stringify(data.prerequisiteOperationIds) : null);
    }
    if (data.instructions !== undefined) {
      updates.push(`instructions = $${paramIndex++}`);
      values.push(data.instructions);
    }
    if (data.qualityCheckRequired !== undefined) {
      updates.push(`quality_check_required = $${paramIndex++}`);
      values.push(data.qualityCheckRequired);
    }
    if (data.isActive !== undefined) {
      updates.push(`is_active = $${paramIndex++}`);
      values.push(data.isActive);
    }

    if (updates.length === 0) {
      return await getOperationById(pool, id);
    }

    updates.push(`updated_at = NOW()`);
    values.push(id);

    const result = await client.query(`
      UPDATE production.production_routing_operations 
      SET ${updates.join(", ")}
      WHERE id = $${paramIndex}
      RETURNING *
    `, values);

    if (result.rows.length === 0) return null;

    const row = result.rows[0];
    
    // Recalculate total time if time-related fields changed (using transaction client)
    if (data.estimatedTime !== undefined || data.setupTime !== undefined) {
      await recalculateTotalTime(client, row.routing_id);
    }
    
    return {
      id: row.id,
      routingId: row.routing_id,
      sequence: row.sequence,
      code: row.code,
      name: row.name,
      description: row.description,
      workCenterId: row.work_center_id,
      normId: row.norm_id,
      estimatedTime: row.estimated_time,
      setupTime: row.setup_time,
      allowParallel: row.allow_parallel,
      prerequisiteOperationIds: row.prerequisite_operation_ids,
      instructions: row.instructions,
      qualityCheckRequired: row.quality_check_required,
      isActive: row.is_active,
      createdAt: row.created_at,
      updatedAt: row.updated_at,
    };
  });
}

export async function deleteOperation(pool: Pool, id: number): Promise<boolean> {
  return withTransaction(pool, async (client) => {
    // Get routing ID before deleting
    const opResult = await client.query(`
      SELECT routing_id FROM production.production_routing_operations WHERE id = $1
    `, [id]);
    
    if (opResult.rows.length === 0) return false;
    
    const routingId = opResult.rows[0].routing_id;
    
    const result = await client.query(`
      DELETE FROM production.production_routing_operations WHERE id = $1
    `, [id]);
    
    const deleted = result.rowCount !== null && result.rowCount > 0;
    
    if (deleted) {
      // Recalculate total time (using transaction client)
      await recalculateTotalTime(client, routingId);
    }
    
    return deleted;
  });
}

export async function resequenceOperations(pool: Pool, routingId: number, orderedIds: number[]): Promise<boolean> {
  return withTransaction(pool, async (client) => {
    // Lock rows for update
    await client.query(`
      SELECT id FROM production.production_routing_operations 
      WHERE routing_id = $1 
      FOR UPDATE
    `, [routingId]);
    
    // Update sequence for each operation
    for (let i = 0; i < orderedIds.length; i++) {
      await client.query(`
        UPDATE production.production_routing_operations
        SET sequence = $1, updated_at = NOW()
        WHERE id = $2 AND routing_id = $3
      `, [i + 1, orderedIds[i], routingId]);
    }
    
    return true;
  });
}
