import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Package, Search, X, Store, Download } from "lucide-react";
import { useState, useMemo } from "react";
import { Skeleton } from "@/components/ui/skeleton";
import { useLocation } from "wouter";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { ConnectionStatusIcon } from "@/components/connection-status-icon";
import { MarketplaceLinkDialog } from "@/components/marketplace-catalog-link-dialog";

interface MarketplaceProduct {
  id: number;
  offer_external_id: string;
  source: 'allegro' | 'shoper';
  name: string;
  image_url: string | null;
  last_sold_at: string | null;
  times_sold: number;
  total_quantity_sold: number;
  avg_unit_price: string | null;
  created_at: string;
  updated_at: string;
  linkType: 'product' | 'set' | null;
  catalogProductId: number | null;
  catalogProductSku: string | null;
  catalogSetId: number | null;
  catalogSetSku: string | null;
  isLinked: boolean;
}

export default function ProductsPage() {
  const [searchQuery, setSearchQuery] = useState("");
  const [sourceFilter, setSourceFilter] = useState<"all" | "allegro" | "shoper">("all");
  const [linkedFilter, setLinkedFilter] = useState<"all" | "linked" | "unlinked">("all");
  const [linkDialogOpen, setLinkDialogOpen] = useState(false);
  const [selectedProductForLinking, setSelectedProductForLinking] = useState<MarketplaceProduct | null>(null);
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const queryParams = useMemo(() => {
    const params: Record<string, string> = {};
    if (searchQuery) params.search = searchQuery;
    if (sourceFilter !== "all") params.source = sourceFilter;
    if (linkedFilter !== "all") params.linked = linkedFilter;
    return params;
  }, [searchQuery, sourceFilter, linkedFilter]);

  const queryKey = useMemo(() => ['/api/products', queryParams], [queryParams]);

  const { data: products = [], isLoading } = useQuery<MarketplaceProduct[]>({
    queryKey,
  });

  const syncImageMutation = useMutation({
    mutationFn: async (externalId: string) => {
      const response = await fetch(`/api/marketplace-products/${externalId}/sync-image`, {
        method: 'POST',
        credentials: 'include',
      });
      if (!response.ok) {
        const error = await response.json();
        throw new Error(error.error || 'Failed to sync image');
      }
      return response.json();
    },
    onSuccess: (data, externalId) => {
      toast({
        title: "✅ Zdjęcie zsynchronizowane",
        description: "Zdjęcie produktu zostało pobrane z Allegro",
      });
      queryClient.invalidateQueries({ queryKey: ['/api/products'] });
    },
    onError: (error: any, externalId) => {
      toast({
        title: "❌ Błąd synchronizacji",
        description: error.message || "Nie udało się pobrać zdjęcia",
        variant: "destructive",
      });
    },
  });

  const clearAllFilters = () => {
    setSearchQuery("");
    setSourceFilter("all");
    setLinkedFilter("all");
  };

  const hasActiveFilters = searchQuery || sourceFilter !== "all" || linkedFilter !== "all";

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div>
        <h1 className="text-3xl font-bold mb-2">Produkty Marketplace</h1>
        <p className="text-muted-foreground">
          Produkty sprzedane na Allegro i Shoper (agregowane z zamówień)
        </p>
      </div>

      {/* Wyszukiwarka */}
      <Card className="border-2 border-primary/20 bg-primary/5">
        <CardContent className="pt-6">
          <div className="space-y-2">
            <label className="text-sm font-medium text-primary flex items-center gap-2">
              <Search className="h-4 w-4" />
              Wyszukaj produkt
            </label>
            <div className="relative">
              <Input
                placeholder="Wpisz nazwę produktu lub SKU..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="text-lg h-12 bg-background"
                data-testid="input-search-main"
              />
              {searchQuery && (
                <Button
                  variant="ghost"
                  size="sm"
                  onClick={() => setSearchQuery("")}
                  className="absolute right-2 top-1/2 -translate-y-1/2"
                  data-testid="button-clear-search"
                >
                  <X className="h-4 w-4" />
                </Button>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Filtry */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Store className="h-5 w-5" />
            Filtry
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="space-y-2">
              <label className="text-sm font-medium">Źródło</label>
              <Select value={sourceFilter} onValueChange={(val: "all" | "allegro" | "shoper") => setSourceFilter(val)}>
                <SelectTrigger data-testid="select-source">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie platformy</SelectItem>
                  <SelectItem value="allegro">Allegro</SelectItem>
                  <SelectItem value="shoper">Shoper</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <label className="text-sm font-medium">Status połączenia</label>
              <Select value={linkedFilter} onValueChange={(val: "all" | "linked" | "unlinked") => setLinkedFilter(val)}>
                <SelectTrigger data-testid="select-linked">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="linked">Połączone</SelectItem>
                  <SelectItem value="unlinked">Niepołączone</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Aktywne filtry */}
      {hasActiveFilters && (
        <div className="flex flex-wrap items-center gap-2 p-4 bg-muted/50 rounded-lg border">
          <span className="text-sm font-medium text-muted-foreground">Aktywne filtry:</span>
          
          {searchQuery && (
            <Badge
              variant="default"
              className="gap-1 bg-blue-600 hover:bg-blue-700 cursor-pointer"
              onClick={() => setSearchQuery("")}
              data-testid="badge-search"
            >
              {searchQuery}
              <X className="h-3 w-3" />
            </Badge>
          )}

          {sourceFilter !== "all" && (
            <Badge
              variant="default"
              className="gap-1 bg-blue-600 hover:bg-blue-700 cursor-pointer"
              onClick={() => setSourceFilter("all")}
              data-testid="badge-source"
            >
              {sourceFilter === "allegro" ? "Allegro" : "Shoper"}
              <X className="h-3 w-3" />
            </Badge>
          )}

          {linkedFilter !== "all" && (
            <Badge
              variant="default"
              className="gap-1 bg-emerald-600 hover:bg-emerald-700 cursor-pointer"
              onClick={() => setLinkedFilter("all")}
              data-testid="badge-linked"
            >
              {linkedFilter === "linked" ? "Połączone" : "Niepołączone"}
              <X className="h-3 w-3" />
            </Badge>
          )}

          <Button
            variant="ghost"
            size="sm"
            onClick={clearAllFilters}
            className="ml-auto"
            data-testid="button-clear-all-filters"
          >
            Wyczyść wszystko
          </Button>
        </div>
      )}

      {/* Tabela produktów */}
      <Card>
        <CardHeader>
          <CardTitle>Produkty ({products.length})</CardTitle>
          <CardDescription>
            Unikalne produkty sprzedane na marketplace (Allegro + Shoper)
          </CardDescription>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="space-y-2">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          ) : products.length === 0 ? (
            <div className="text-center py-12">
              <Package className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <p className="text-muted-foreground">Nie znaleziono produktów</p>
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-20">Zdjęcie</TableHead>
                  <TableHead>SKU</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Źródło</TableHead>
                  <TableHead>Sprzedane</TableHead>
                  <TableHead>Śr. cena</TableHead>
                  <TableHead>Ostatnia sprzedaż</TableHead>
                  <TableHead className="text-center">Połączenie</TableHead>
                  <TableHead className="w-24">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {products.map((product) => (
                  <TableRow 
                    key={product.id} 
                    data-testid={`row-product-${product.offer_external_id}`}
                  >
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}`)}
                    >
                      {product.image_url ? (
                        <img
                          src={product.image_url}
                          alt={product.name}
                          className="h-20 w-20 object-cover rounded border"
                          onError={(e) => {
                            (e.target as HTMLImageElement).src = 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iODAiIGhlaWdodD0iODAiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyI+PHJlY3Qgd2lkdGg9IjgwIiBoZWlnaHQ9IjgwIiBmaWxsPSIjZjNmNGY2Ii8+PC9zdmc+';
                          }}
                        />
                      ) : (
                        <div className="h-20 w-20 bg-muted rounded border flex items-center justify-center">
                          <Package className="h-10 w-10 text-muted-foreground" />
                        </div>
                      )}
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}`)}
                    >
                      <span className="font-mono text-sm" data-testid={`text-sku-${product.offer_external_id}`}>
                        {product.offer_external_id}
                      </span>
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}`)}
                    >
                      <div className="max-w-md">
                        <div className="font-medium line-clamp-2">{product.name}</div>
                      </div>
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}`)}
                    >
                      <Badge variant={product.source === 'allegro' ? 'default' : 'secondary'}>
                        {product.source === 'allegro' ? 'Allegro' : 'Shoper'}
                      </Badge>
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}`)}
                    >
                      <div className="text-sm">
                        <div className="font-medium">{product.total_quantity_sold} szt.</div>
                        <div className="text-xs text-muted-foreground">{product.times_sold} zam.</div>
                      </div>
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}`)}
                    >
                      {product.avg_unit_price ? (
                        <div className="font-medium">
                          {parseFloat(product.avg_unit_price).toFixed(2)} PLN
                        </div>
                      ) : (
                        <span className="text-muted-foreground">—</span>
                      )}
                    </TableCell>
                    <TableCell 
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product/${product.offer_external_id}`)}
                    >
                      <div className="text-sm text-muted-foreground">
                        {product.last_sold_at
                          ? new Date(product.last_sold_at).toLocaleDateString('pl-PL')
                          : '—'}
                      </div>
                    </TableCell>
                    <TableCell className="text-center" onClick={(e) => e.stopPropagation()}>
                      <ConnectionStatusIcon
                        isLinked={product.isLinked}
                        targetPath={
                          product.linkType === 'product' && product.catalogProductId
                            ? `/catalog-products/${product.catalogProductId}`
                            : product.linkType === 'set' && product.catalogSetId
                            ? `/catalog-sets/${product.catalogSetId}`
                            : null
                        }
                        targetSku={product.catalogProductSku || product.catalogSetSku}
                        productId={product.id}
                        onManageLinks={() => {
                          setSelectedProductForLinking(product);
                          setLinkDialogOpen(true);
                        }}
                      />
                    </TableCell>
                    <TableCell>
                      {(!product.image_url || product.image_url === '') && product.source.toUpperCase() === 'ALLEGRO' && (
                        <Button
                          size="sm"
                          variant="outline"
                          onClick={(e) => {
                            e.stopPropagation();
                            syncImageMutation.mutate(product.offer_external_id);
                          }}
                          disabled={syncImageMutation.isPending}
                          data-testid={`button-sync-image-${product.offer_external_id}`}
                        >
                          <Download className="h-4 w-4 mr-2" />
                          {syncImageMutation.isPending ? 'Pobieranie...' : 'Pobierz zdjęcie'}
                        </Button>
                      )}
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Marketplace Link Management Dialog */}
      {selectedProductForLinking && (
        <MarketplaceLinkDialog
          open={linkDialogOpen}
          onOpenChange={setLinkDialogOpen}
          mode="marketplace-to-catalog"
          marketplaceProduct={{
            id: selectedProductForLinking.id,
            externalId: selectedProductForLinking.offer_external_id,
            platform: selectedProductForLinking.source,
            sku: selectedProductForLinking.catalogProductSku || selectedProductForLinking.offer_external_id,
            name: selectedProductForLinking.name,
          }}
          onSuccess={() => {
            queryClient.invalidateQueries({ queryKey: ['/api/products'] });
          }}
        />
      )}
    </div>
  );
}
