import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { Plus, Pencil, Trash2, Play, Loader2, X, Copy, Sparkles, Search, ChevronLeft, ChevronRight } from "lucide-react";
import { RichTextEditor } from "@/components/rich-text-editor";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
} from "@/components/ui/dialog";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Textarea } from "@/components/ui/textarea";

interface ProductMatrix {
  id: number;
  name: string;
  namePrefix: string | null;
  nameSuffix: string | null;
  productType: string | null;
  doors: string | null;
  legs: string | null;
  lengths: string[];
  widths: string[];
  heights: string[];
  productGroups: string[];
  colors: string[];
  descriptionHtml: string | null;
  descriptionDoc: any | null;
  useAiGeneration: boolean;
  aiPrompt: string | null;
  defaultPrice: string | null;
  createdAt: Date | null;
  updatedAt: Date | null;
}

interface ProductCreatorDictionary {
  id: number;
  dictionaryType: string;
  code: string;
  name: string;
  readableName: string | null;
  isActive: boolean;
}

// Funkcje pomocnicze do wyciągania wartości ze słowników
const getDimensionValues = (options: ProductCreatorDictionary[] | undefined): string[] => {
  if (!options) return [];
  return options
    .filter(opt => opt.isActive)
    .map(opt => opt.name || opt.code)
    .sort((a, b) => {
      const numA = parseInt(a);
      const numB = parseInt(b);
      if (!isNaN(numA) && !isNaN(numB)) return numA - numB;
      return a.localeCompare(b);
    });
};

export default function ProductMatrices() {
  const { toast } = useToast();
  const [, navigate] = useLocation();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [isDeleteDialogOpen, setIsDeleteDialogOpen] = useState(false);
  const [selectedMatrix, setSelectedMatrix] = useState<ProductMatrix | null>(null);
  const [matrixToDelete, setMatrixToDelete] = useState<number | null>(null);
  const [activeTab, setActiveTab] = useState("podstawowe");

  // Form state
  const [name, setName] = useState("");
  const [namePrefix, setNamePrefix] = useState("");
  const [nameSuffix, setNameSuffix] = useState("");
  const [productType, setProductType] = useState("");
  const [doors, setDoors] = useState("");
  const [legs, setLegs] = useState("");
  const [lengths, setLengths] = useState<string[]>([]);
  const [widths, setWidths] = useState<string[]>([]);
  const [heights, setHeights] = useState<string[]>([]);
  const [productGroupsArray, setProductGroupsArray] = useState<string[]>([]);
  const [colors, setColors] = useState<string[]>([]);
  const [descriptionHtml, setDescriptionHtml] = useState("");
  const [descriptionDoc, setDescriptionDoc] = useState<any>(null);
  const [useAiGeneration, setUseAiGeneration] = useState(false);
  const [aiPrompt, setAiPrompt] = useState("");
  const [defaultPrice, setDefaultPrice] = useState("");

  // Temporary inputs for adding dimensions and colors
  const [newLengthInput, setNewLengthInput] = useState("");
  const [newWidthInput, setNewWidthInput] = useState("");
  const [newHeightInput, setNewHeightInput] = useState("");
  const [newColorInput, setNewColorInput] = useState("");

  // Load settings from localStorage
  const getInitialSettings = () => {
    const saved = localStorage.getItem('productMatricesSettings');
    if (saved) {
      try {
        return JSON.parse(saved);
      } catch {
        return {};
      }
    }
    return {};
  };

  const initialSettings = getInitialSettings();

  // Sorting, filtering, and pagination state
  const [sort, setSort] = useState<string>(initialSettings.sort || 'updatedAt');
  const [order, setOrder] = useState<'asc' | 'desc'>(initialSettings.order || 'desc');
  const [searchInput, setSearchInput] = useState<string>(initialSettings.search || '');
  const [debouncedSearch, setDebouncedSearch] = useState<string>(initialSettings.search || '');
  const [productTypeFilter, setProductTypeFilter] = useState<string>(initialSettings.productTypeFilter || '');
  const [page, setPage] = useState<number>(initialSettings.page || 0);
  const [pageSize, setPageSize] = useState<number>(initialSettings.pageSize || 25);

  // Debounce search input (500ms delay)
  useEffect(() => {
    const timer = setTimeout(() => {
      setDebouncedSearch(searchInput);
      setPage(0); // Reset to first page when search changes
    }, 500);

    return () => clearTimeout(timer);
  }, [searchInput]);

  // Save settings to localStorage when they change
  useEffect(() => {
    const settings = { sort, order, search: searchInput, productTypeFilter, page, pageSize };
    localStorage.setItem('productMatricesSettings', JSON.stringify(settings));
  }, [sort, order, searchInput, productTypeFilter, page, pageSize]);

  // Fetch matrices with pagination and filtering
  const { data: matricesData, isLoading } = useQuery<{ matrices: ProductMatrix[], total: number, limit: number, offset: number }>({
    queryKey: ["/api/product-matrices", sort, order, debouncedSearch, productTypeFilter, pageSize, page * pageSize],
    queryFn: async () => {
      const params = new URLSearchParams({
        sort,
        order,
        search: debouncedSearch,
        productType: productTypeFilter || '', // Convert empty/null to empty string
        limit: pageSize.toString(),
        offset: (page * pageSize).toString()
      });
      const response = await fetch(`/api/product-matrices?${params}`, {
        credentials: 'include'
      });
      if (!response.ok) throw new Error('Failed to fetch matrices');
      return response.json();
    }
  });

  const matrices = matricesData?.matrices || [];
  const totalMatrices = matricesData?.total || 0;
  const totalPages = Math.ceil(totalMatrices / pageSize);

  // Fetch dictionaries for dropdown options
  const { data: productTypes } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_type"],
  });
  const { data: productGroups } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=product_group"],
  });
  const { data: doorsOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=door"],
  });
  const { data: legsOptions} = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=leg"],
  });
  const { data: colorOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });
  const { data: dimensionLengthOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_length"],
  });
  const { data: dimensionWidthOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_width"],
  });
  const { data: dimensionHeightOptions } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=dimension_height"],
  });

  // Create/Update mutation
  const saveMutation = useMutation({
    mutationFn: async (data: Partial<ProductMatrix>) => {
      if (selectedMatrix) {
        return apiRequest("PUT", `/api/product-matrices/${selectedMatrix.id}`, data);
      } else {
        return apiRequest("POST", "/api/product-matrices", data);
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/product-matrices"] });
      toast({
        title: selectedMatrix ? "Zaktualizowano matrycę" : "Utworzono matrycę",
        description: "Matryca produktów została zapisana pomyślnie.",
      });
      handleCloseDialog();
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać matryce.",
        variant: "destructive",
      });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/product-matrices/${id}`, undefined);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/product-matrices"] });
      toast({
        title: "Usunięto matrycę",
        description: "Matryca produktów została usunięta.",
      });
      setIsDeleteDialogOpen(false);
      setMatrixToDelete(null);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się usunąć matryce.",
        variant: "destructive",
      });
    },
  });

  // Generate AI description mutation
  const generateAiDescriptionMutation = useMutation({
    mutationFn: async (matrixId: number) => {
      return apiRequest("POST", `/api/product-matrices/${matrixId}/ai-generate-description`, {
        aiPrompt,
      });
    },
    onSuccess: (data: any) => {
      if (data.description) {
        // Set HTML directly - RichTextEditor will handle it
        setDescriptionHtml(data.description);
        // Set doc to null to force re-initialization from HTML
        setDescriptionDoc(null);
        toast({
          title: "Opis wygenerowany",
          description: `Wygenerowano opis przy użyciu AI (koszt: ${data.cost?.toFixed(4)} USD)`,
        });
      }
    },
    onError: () => {
      toast({
        variant: "destructive",
        title: "Błąd generowania",
        description: "Nie udało się wygenerować opisu przez AI.",
      });
    },
  });

  // Generate products mutation
  const generateMutation = useMutation({
    mutationFn: async (matrixId: number) => {
      return apiRequest("POST", `/api/product-matrices/${matrixId}/generate`, {});
    },
    onSuccess: (data: any) => {
      toast({
        title: "Produkty wygenerowane",
        description: `Pomyślnie wygenerowano ${data.productsGenerated} produktów.`,
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd generowania",
        description: error.message || "Nie udało się wygenerować produktów.",
        variant: "destructive",
      });
    },
  });

  const handleOpenDialog = (matrix?: ProductMatrix) => {
    if (matrix) {
      setSelectedMatrix(matrix);
      setName(matrix.name);
      setNamePrefix(matrix.namePrefix || "");
      setNameSuffix(matrix.nameSuffix || "");
      setProductType(matrix.productType || "");
      setDoors(matrix.doors || "");
      setLegs(matrix.legs || "");
      setLengths(matrix.lengths || []);
      setWidths(matrix.widths || []);
      setHeights(matrix.heights || []);
      setProductGroupsArray(matrix.productGroups || []);
      setColors(matrix.colors || []);
      setDescriptionHtml(matrix.descriptionHtml || "");
      setDescriptionDoc(matrix.descriptionDoc);
      setUseAiGeneration(matrix.useAiGeneration || false);
      setAiPrompt(matrix.aiPrompt || "");
      setDefaultPrice(matrix.defaultPrice || "");
    } else {
      setSelectedMatrix(null);
      resetForm();
    }
    setActiveTab("podstawowe");
    setIsDialogOpen(true);
  };

  const handleCloseDialog = () => {
    setIsDialogOpen(false);
    setSelectedMatrix(null);
    resetForm();
  };

  const resetForm = () => {
    setName("");
    setNamePrefix("");
    setNameSuffix("");
    setProductType("");
    setDoors("");
    setLegs("");
    setLengths([]);
    setWidths([]);
    setHeights([]);
    setProductGroupsArray([]);
    setColors([]);
    setDescriptionHtml("");
    setDescriptionDoc(null);
    setUseAiGeneration(false);
    setAiPrompt("");
    setDefaultPrice("");
    setNewLengthInput("");
    setNewWidthInput("");
    setNewHeightInput("");
    setNewColorInput("");
  };

  const handleSave = () => {
    if (!name.trim()) {
      toast({
        title: "Błąd walidacji",
        description: "Nazwa matryce jest wymagana.",
        variant: "destructive",
      });
      return;
    }

    console.log("🔍 handleSave - Before filtering:", {
      lengths,
      widths,
      heights,
      productGroupsArray,
      colors
    });

    // Filter out empty dimensions and their corresponding product groups
    // Keep only indices where at least one dimension value is non-empty
    const cleanLengths: string[] = [];
    const cleanWidths: string[] = [];
    const cleanHeights: string[] = [];
    const cleanProductGroups: string[] = [];

    const maxLength = Math.max(lengths.length, widths.length, heights.length);
    for (let i = 0; i < maxLength; i++) {
      const length = lengths[i];
      const width = widths[i];
      const height = heights[i];
      
      // Keep this index if any dimension has a value
      if (length || width || height) {
        cleanLengths.push(length || "");
        cleanWidths.push(width || "");
        cleanHeights.push(height || "");
        cleanProductGroups.push(productGroupsArray[i] || "");
      }
    }

    console.log("🔍 handleSave - After filtering:", {
      cleanLengths,
      cleanWidths,
      cleanHeights,
      cleanProductGroups,
      colors
    });

    const data: Partial<ProductMatrix> = {
      name,
      namePrefix: namePrefix || null,
      nameSuffix: nameSuffix || null,
      productType: productType || null,
      doors: doors || null,
      legs: legs || null,
      lengths: cleanLengths,
      widths: cleanWidths,
      heights: cleanHeights,
      productGroups: cleanProductGroups,
      colors,
      descriptionHtml: descriptionHtml || null,
      descriptionDoc,
      useAiGeneration,
      aiPrompt: aiPrompt || null,
      defaultPrice: defaultPrice || null,
    };

    console.log("🔍 handleSave - Sending data:", JSON.stringify(data, null, 2));

    saveMutation.mutate(data);
  };

  const handleDelete = (id: number) => {
    setMatrixToDelete(id);
    setIsDeleteDialogOpen(true);
  };

  const confirmDelete = () => {
    if (matrixToDelete !== null) {
      deleteMutation.mutate(matrixToDelete);
    }
  };

  const handleGenerate = (matrixId: number) => {
    generateMutation.mutate(matrixId);
  };

  const handleDuplicate = (matrix: ProductMatrix) => {
    // Otwórz dialog z danymi skopiowanymi z matryce
    setSelectedMatrix(null); // Nowa matryca (bez ID)
    setName(matrix.name + " (kopia)");
    setNamePrefix(matrix.namePrefix || "");
    setNameSuffix(matrix.nameSuffix || "");
    setProductType(matrix.productType || "");
    setDoors(matrix.doors || "");
    setLegs(matrix.legs || "");
    setLengths(matrix.lengths || []);
    setWidths(matrix.widths || []);
    setHeights(matrix.heights || []);
    setProductGroupsArray(matrix.productGroups || []);
    setColors(matrix.colors || []);
    setDescriptionHtml(matrix.descriptionHtml || "");
    setDescriptionDoc(matrix.descriptionDoc);
    setUseAiGeneration(matrix.useAiGeneration || false);
    setAiPrompt(matrix.aiPrompt || "");
    setDefaultPrice(matrix.defaultPrice || "");
    setActiveTab("podstawowe");
    setIsDialogOpen(true);
  };

  // Dimension management
  const handleAddDimension = () => {
    // Dodaj wymiary tylko jeśli przynajmniej jedno pole jest wypełnione
    if (newLengthInput || newWidthInput || newHeightInput) {
      setLengths([...lengths, newLengthInput.trim()]);
      setWidths([...widths, newWidthInput.trim()]);
      setHeights([...heights, newHeightInput.trim()]);
      setProductGroupsArray([...productGroupsArray, ""]); // Dodaj pustą grupę
      setNewLengthInput("");
      setNewWidthInput("");
      setNewHeightInput("");
    }
  };

  const handleRemoveDimension = (index: number) => {
    setLengths(lengths.filter((_, i) => i !== index));
    setWidths(widths.filter((_, i) => i !== index));
    setHeights(heights.filter((_, i) => i !== index));
    setProductGroupsArray(productGroupsArray.filter((_, i) => i !== index));
  };

  const handleCopyDimension = (index: number) => {
    setNewLengthInput(lengths[index] || "");
    setNewWidthInput(widths[index] || "");
    setNewHeightInput(heights[index] || "");
    toast({
      title: "Rozmiar skopiowany",
      description: "Wymiary zostały skopiowane do pól edycji.",
    });
  };

  const handleAddAllCombinations = (height: string) => {
    const newLengths: string[] = [];
    const newWidths: string[] = [];
    const newHeights: string[] = [];
    const newGroups: string[] = [];

    // Pobierz wartości ze słowników
    const lengthValues = getDimensionValues(dimensionLengthOptions);
    const widthValues = getDimensionValues(dimensionWidthOptions);

    // Generuj wszystkie kombinacje długość × szerokość dla danej wysokości
    lengthValues.forEach(length => {
      widthValues.forEach(width => {
        newLengths.push(length);
        newWidths.push(width);
        newHeights.push(height);
        newGroups.push(""); // Pusta grupa dla nowego rozmiaru
      });
    });

    // Dodaj do istniejących rozmiarów
    setLengths([...lengths, ...newLengths]);
    setWidths([...widths, ...newWidths]);
    setHeights([...heights, ...newHeights]);
    setProductGroupsArray([...productGroupsArray, ...newGroups]);

    toast({
      title: "Kombinacje dodane",
      description: `Dodano ${newLengths.length} kombinacji dla wysokości ${height}mm.`,
    });
  };

  // Color management
  const handleAddColor = () => {
    if (newColorInput.trim()) {
      setColors([...colors, newColorInput.trim()]);
      setNewColorInput("");
    }
  };

  const handleRemoveColor = (index: number) => {
    setColors(colors.filter((_, i) => i !== index));
  };

  const loadAllColorsFromDict = () => {
    if (colorOptions) {
      const allColors = colorOptions
        .filter(c => c.isActive)
        .map(c => c.name || c.code);
      setColors(allColors);
      toast({
        title: "Kolory załadowane",
        description: `Dodano ${allColors.length} kolorów ze słownika.`,
      });
    }
  };

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-page-title">Matryce Produktów</h1>
          <p className="text-muted-foreground" data-testid="text-page-description">
            Zarządzaj matrycami do generowania wariantów produktów
          </p>
        </div>
        <Button onClick={() => navigate("/product-matrices/new/edit")} data-testid="button-new-matrix">
          <Plus className="h-4 w-4 mr-2" />
          Nowa Matryca
        </Button>
      </div>

      {isLoading ? (
        <div className="flex justify-center items-center h-64">
          <Loader2 className="h-8 w-8 animate-spin" data-testid="loader-matrices" />
        </div>
      ) : (
        <Card>
          <CardHeader>
            <CardTitle>Lista Matryce</CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {/* Filters and Controls */}
            <div className="flex flex-wrap gap-4 items-end">
              {/* Search input */}
              <div className="flex-1 min-w-[200px]">
                <Label htmlFor="search">Szukaj</Label>
                <div className="relative">
                  <Search className="absolute left-2 top-1/2 transform -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                  <Input
                    id="search"
                    type="text"
                    placeholder="Nazwa matryce..."
                    value={searchInput}
                    onChange={(e) => {
                      setSearchInput(e.target.value);
                    }}
                    className="pl-8"
                    data-testid="input-search"
                  />
                </div>
              </div>

              {/* Product Type Filter */}
              <div className="w-[200px]">
                <Label htmlFor="product-type-filter">Typ produktu</Label>
                <Select value={productTypeFilter || "all"} onValueChange={(value) => {
                  setProductTypeFilter(value === "all" ? "" : value);
                  setPage(0); // Reset to first page when filtering
                }}>
                  <SelectTrigger id="product-type-filter" data-testid="select-product-type-filter">
                    <SelectValue placeholder="Wszystkie" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie</SelectItem>
                    {productTypes?.filter(opt => opt.isActive).map((opt) => (
                      <SelectItem key={opt.id} value={opt.name || opt.code}>
                        {opt.readableName || opt.name || opt.code}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {/* Sort Field */}
              <div className="w-[180px]">
                <Label htmlFor="sort-field">Sortuj według</Label>
                <Select value={sort} onValueChange={(value) => setSort(value)}>
                  <SelectTrigger id="sort-field" data-testid="select-sort">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="updatedAt">Data modyfikacji</SelectItem>
                    <SelectItem value="createdAt">Data utworzenia</SelectItem>
                    <SelectItem value="name">Nazwa</SelectItem>
                    <SelectItem value="productType">Typ produktu</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {/* Sort Order */}
              <div className="w-[120px]">
                <Label htmlFor="sort-order">Kolejność</Label>
                <Select value={order} onValueChange={(value: 'asc' | 'desc') => setOrder(value)}>
                  <SelectTrigger id="sort-order" data-testid="select-order">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="desc">Malejąco</SelectItem>
                    <SelectItem value="asc">Rosnąco</SelectItem>
                  </SelectContent>
                </Select>
              </div>

              {/* Page Size */}
              <div className="w-[120px]">
                <Label htmlFor="page-size">Na stronie</Label>
                <Select value={pageSize.toString()} onValueChange={(value) => {
                  setPageSize(parseInt(value, 10));
                  setPage(0); // Reset to first page when changing page size
                }}>
                  <SelectTrigger id="page-size" data-testid="select-page-size">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="25">25</SelectItem>
                    <SelectItem value="50">50</SelectItem>
                    <SelectItem value="100">100</SelectItem>
                    <SelectItem value="200">200</SelectItem>
                    <SelectItem value="500">500</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            {/* Top Pagination */}
            <div className="flex items-center justify-between">
              <div className="text-sm text-muted-foreground">
                Wyświetlanie {matrices.length === 0 ? 0 : page * pageSize + 1} - {Math.min((page + 1) * pageSize, totalMatrices)} z {totalMatrices} matryce
              </div>
              <div className="flex items-center gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setPage(Math.max(0, page - 1))}
                  disabled={page === 0}
                  data-testid="button-prev-page-top"
                >
                  <ChevronLeft className="h-4 w-4" />
                  Poprzednia
                </Button>
                <div className="text-sm font-medium">
                  Strona {page + 1} z {Math.max(1, totalPages)}
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setPage(Math.min(totalPages - 1, page + 1))}
                  disabled={page >= totalPages - 1}
                  data-testid="button-next-page-top"
                >
                  Następna
                  <ChevronRight className="h-4 w-4" />
                </Button>
              </div>
            </div>

            {/* Lista matryc - responsywna tabela */}
            <div className="border rounded-lg overflow-x-auto">
              <Table className="min-w-[380px]">
                <TableHeader>
                  <TableRow>
                    <TableHead className="min-w-[100px]">Nazwa</TableHead>
                    <TableHead className="min-w-[70px] max-w-[90px]">
                      <span className="hidden md:inline">Typ produktu</span>
                      <span className="md:hidden">Typ</span>
                    </TableHead>
                    <TableHead className="hidden md:table-cell min-w-[60px]">Rozm.</TableHead>
                    <TableHead className="hidden md:table-cell min-w-[60px]">Kol.</TableHead>
                    <TableHead className="min-w-[60px]">
                      <span className="hidden md:inline">Produkty</span>
                      <span className="md:hidden">Prod.</span>
                    </TableHead>
                    <TableHead className="text-right min-w-[180px] md:min-w-[280px]">Akcje</TableHead>
                  </TableRow>
                </TableHeader>
              <TableBody>
                {matrices && matrices.length > 0 ? (
                  matrices.map((matrix) => (
                    <TableRow 
                      key={matrix.id} 
                      data-testid={`row-matrix-${matrix.id}`}
                      className="cursor-pointer hover-elevate"
                      onClick={() => navigate(`/product-matrices/${matrix.id}/edit`)}
                    >
                      <TableCell className="font-medium" data-testid={`text-matrix-name-${matrix.id}`}>
                        {matrix.name}
                      </TableCell>
                      <TableCell data-testid={`text-product-type-${matrix.id}`}>
                        {matrix.productType || "-"}
                      </TableCell>
                      <TableCell className="hidden md:table-cell" data-testid={`text-sizes-count-${matrix.id}`}>
                        <Badge variant="secondary" className="whitespace-nowrap">
                          {Math.max(
                            matrix.lengths?.length || 0,
                            matrix.widths?.length || 0,
                            matrix.heights?.length || 0
                          )}
                        </Badge>
                      </TableCell>
                      <TableCell className="hidden md:table-cell" data-testid={`text-colors-count-${matrix.id}`}>
                        <Badge variant="secondary" className="whitespace-nowrap">{matrix.colors?.length || 0}</Badge>
                      </TableCell>
                      <TableCell data-testid={`text-products-count-${matrix.id}`}>
                        <Badge className="whitespace-nowrap">
                          {Math.max(1, 
                            matrix.lengths?.length || 0,
                            matrix.widths?.length || 0,
                            matrix.heights?.length || 0
                          ) * Math.max(1, matrix.colors?.length || 0)}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-right" onClick={(e) => e.stopPropagation()}>
                        <div className="flex justify-end gap-1 md:gap-2 flex-wrap">
                          <Button
                            size="sm"
                            onClick={(e) => {
                              e.stopPropagation();
                              handleGenerate(matrix.id);
                            }}
                            disabled={generateMutation.isPending && generateMutation.variables === matrix.id}
                            data-testid={`button-generate-${matrix.id}`}
                            title="Generuj produkty z tej matryce"
                            className="whitespace-nowrap"
                          >
                            {generateMutation.isPending && generateMutation.variables === matrix.id ? (
                              <Loader2 className="h-4 w-4 md:mr-2 animate-spin" />
                            ) : (
                              <Play className="h-4 w-4 md:mr-2" />
                            )}
                            <span className="hidden md:inline">Generuj produkty</span>
                          </Button>
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={(e) => {
                              e.stopPropagation();
                              handleDuplicate(matrix);
                            }}
                            data-testid={`button-duplicate-${matrix.id}`}
                            title="Duplikuj matrycę"
                          >
                            <Copy className="h-4 w-4" />
                          </Button>
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={(e) => {
                              e.stopPropagation();
                              navigate(`/product-matrices/${matrix.id}/edit`);
                            }}
                            data-testid={`button-edit-${matrix.id}`}
                            title="Edytuj matrycę"
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            size="sm"
                            variant="outline"
                            onClick={(e) => {
                              e.stopPropagation();
                              handleDelete(matrix.id);
                            }}
                            data-testid={`button-delete-${matrix.id}`}
                            title="Usuń matrycę"
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </TableCell>
                    </TableRow>
                  ))
                ) : (
                  <TableRow>
                    <TableCell colSpan={6} className="text-center text-muted-foreground">
                      Brak matryce. Utwórz pierwszą matrycę klikając przycisk "Nowa Matryca".
                    </TableCell>
                  </TableRow>
                )}
              </TableBody>
            </Table>
            </div>

            {/* Bottom Pagination */}
            <div className="flex items-center justify-between pt-4">
              <div className="text-sm text-muted-foreground">
                Wyświetlanie {matrices.length === 0 ? 0 : page * pageSize + 1} - {Math.min((page + 1) * pageSize, totalMatrices)} z {totalMatrices} matryce
              </div>
              <div className="flex items-center gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setPage(Math.max(0, page - 1))}
                  disabled={page === 0}
                  data-testid="button-prev-page-bottom"
                >
                  <ChevronLeft className="h-4 w-4" />
                  Poprzednia
                </Button>
                <div className="text-sm font-medium">
                  Strona {page + 1} z {Math.max(1, totalPages)}
                </div>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => setPage(Math.min(totalPages - 1, page + 1))}
                  disabled={page >= totalPages - 1}
                  data-testid="button-next-page-bottom"
                >
                  Następna
                  <ChevronRight className="h-4 w-4" />
                </Button>
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {/* Create/Edit Dialog */}
      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-4xl max-h-[90vh] overflow-y-auto">
          <DialogHeader>
            <DialogTitle data-testid="text-dialog-title">
              {selectedMatrix ? "Edytuj Matrycę" : "Nowa Matryca"}
            </DialogTitle>
            <DialogDescription>
              Skonfiguruj parametry matryce do generowania produktów
            </DialogDescription>
          </DialogHeader>

          <Tabs value={activeTab} onValueChange={setActiveTab}>
            <TabsList className="grid w-full grid-cols-4">
              <TabsTrigger value="podstawowe" data-testid="tab-podstawowe">Podstawowe</TabsTrigger>
              <TabsTrigger value="rozmiary" data-testid="tab-rozmiary">Rozmiary</TabsTrigger>
              <TabsTrigger value="kolory" data-testid="tab-kolory">Kolory</TabsTrigger>
              <TabsTrigger value="opis" data-testid="tab-opis">Opis</TabsTrigger>
            </TabsList>

            {/* Tab: Podstawowe */}
            <TabsContent value="podstawowe" className="space-y-4">
              <div className="space-y-4">
                {/* Nazwa + Prefix + Suffix w jednym rzędzie */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div>
                    <Label htmlFor="name">Nazwa matryce *</Label>
                    <Input
                      id="name"
                      value={name}
                      onChange={(e) => setName(e.target.value)}
                      placeholder="np. Szafy SUPRA"
                      data-testid="input-name"
                    />
                  </div>
                  <div>
                    <Label htmlFor="namePrefix">Prefix nazwy</Label>
                    <Input
                      id="namePrefix"
                      value={namePrefix}
                      onChange={(e) => setNamePrefix(e.target.value)}
                      placeholder="np. Szafa"
                      data-testid="input-name-prefix"
                    />
                  </div>
                  <div>
                    <Label htmlFor="nameSuffix">Suffix nazwy</Label>
                    <Input
                      id="nameSuffix"
                      value={nameSuffix}
                      onChange={(e) => setNameSuffix(e.target.value)}
                      placeholder="np. Premium"
                      data-testid="input-name-suffix"
                    />
                  </div>
                </div>

                {/* Parametry dictionary w jednym rzędzie - kompaktowo */}
                <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-3 gap-4">
                  <div>
                    <Label htmlFor="productType">Typ produktu</Label>
                    <Select value={productType} onValueChange={setProductType}>
                      <SelectTrigger data-testid="select-product-type">
                        <SelectValue placeholder="Wybierz typ" />
                      </SelectTrigger>
                      <SelectContent>
                        {productTypes?.map((pt) => (
                          <SelectItem key={pt.id} value={pt.code}>
                            {pt.code}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <Label htmlFor="doors">Drzwi</Label>
                    <Select value={doors} onValueChange={setDoors}>
                      <SelectTrigger data-testid="select-doors">
                        <SelectValue placeholder="Wybierz" />
                      </SelectTrigger>
                      <SelectContent>
                        {doorsOptions?.map((d) => (
                          <SelectItem key={d.id} value={d.name || d.code}>
                            {d.readableName || d.name || d.code}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <Label htmlFor="legs">Nóżki</Label>
                    <Select value={legs} onValueChange={setLegs}>
                      <SelectTrigger data-testid="select-legs">
                        <SelectValue placeholder="Wybierz" />
                      </SelectTrigger>
                      <SelectContent>
                        {legsOptions?.map((l) => (
                          <SelectItem key={l.id} value={l.name || l.code}>
                            {l.readableName || l.name || l.code}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                </div>

                {/* Cena domyślna */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                  <div>
                    <Label htmlFor="defaultPrice">Cena domyślna (PLN)</Label>
                    <Input
                      id="defaultPrice"
                      type="number"
                      step="0.01"
                      value={defaultPrice}
                      onChange={(e) => setDefaultPrice(e.target.value)}
                      placeholder="np. 1299.99"
                      data-testid="input-default-price"
                    />
                  </div>
                </div>
              </div>
            </TabsContent>

            {/* Tab: Rozmiary */}
            <TabsContent value="rozmiary" className="space-y-4">
              <div className="space-y-4">
                {/* Szybkie dodawanie kombinacji */}
                <div className="space-y-2">
                  <Label>Szybkie dodawanie</Label>
                  <div className="p-4 border rounded-md bg-muted/50 space-y-3">
                    <p className="text-sm text-muted-foreground">
                      Dodaj wszystkie kombinacje predefiniowanych wymiarów dla wybranej wysokości:
                    </p>
                    <div className="flex gap-2 flex-wrap">
                      {getDimensionValues(dimensionHeightOptions).map((height) => (
                        <Button 
                          key={height}
                          onClick={() => handleAddAllCombinations(height)}
                          variant="secondary"
                          data-testid={`button-add-combinations-${height}`}
                        >
                          <Plus className="h-4 w-4 mr-2" />
                          Wszystkie kombinacje dla H={height}mm
                        </Button>
                      ))}
                    </div>
                    <div className="text-xs text-muted-foreground space-y-1">
                      <p>• Długości: {getDimensionValues(dimensionLengthOptions).join(', ')}mm</p>
                      <p>• Szerokości: {getDimensionValues(dimensionWidthOptions).join(', ')}mm</p>
                      <p>• Każdy przycisk doda {getDimensionValues(dimensionLengthOptions).length * getDimensionValues(dimensionWidthOptions).length} kombinacji</p>
                    </div>
                  </div>
                </div>

                {/* Ręczne dodawanie */}
                <div className="space-y-2">
                  <Label>Lub dodaj pojedynczy rozmiar ręcznie</Label>
                  <div className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-4 gap-2">
                    <div>
                      <Label htmlFor="newLength" className="text-xs">Długość (mm)</Label>
                      <Select value={newLengthInput} onValueChange={setNewLengthInput}>
                        <SelectTrigger data-testid="select-new-length">
                          <SelectValue placeholder="Wybierz" />
                        </SelectTrigger>
                        <SelectContent>
                          {getDimensionValues(dimensionLengthOptions).map((length) => (
                            <SelectItem key={length} value={length}>
                              {length}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label htmlFor="newWidth" className="text-xs">Szerokość (mm)</Label>
                      <Select value={newWidthInput} onValueChange={setNewWidthInput}>
                        <SelectTrigger data-testid="select-new-width">
                          <SelectValue placeholder="Wybierz" />
                        </SelectTrigger>
                        <SelectContent>
                          {getDimensionValues(dimensionWidthOptions).map((width) => (
                            <SelectItem key={width} value={width}>
                              {width}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div>
                      <Label htmlFor="newHeight" className="text-xs">Wysokość (mm)</Label>
                      <Select value={newHeightInput} onValueChange={setNewHeightInput}>
                        <SelectTrigger data-testid="select-new-height">
                          <SelectValue placeholder="Wybierz" />
                        </SelectTrigger>
                        <SelectContent>
                          {getDimensionValues(dimensionHeightOptions).map((height) => (
                            <SelectItem key={height} value={height}>
                              {height}
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                    </div>
                    <div className="flex items-end">
                      <Button 
                        onClick={handleAddDimension} 
                        className="w-full"
                        data-testid="button-add-dimension"
                      >
                        <Plus className="h-4 w-4 mr-2" />
                        Dodaj
                      </Button>
                    </div>
                  </div>
                </div>

                <div className="space-y-2">
                  <Label>Lista rozmiarów ({Math.max(lengths.length, widths.length, heights.length)})</Label>
                  <div className="border rounded-md p-4 max-h-96 overflow-y-auto">
                    {lengths.length > 0 || widths.length > 0 || heights.length > 0 ? (
                      <Table>
                        <TableHeader>
                          <TableRow>
                            <TableHead className="w-[100px]">Długość</TableHead>
                            <TableHead className="w-[100px]">Szerokość</TableHead>
                            <TableHead className="w-[100px]">Wysokość</TableHead>
                            <TableHead className="w-[180px]">Grupa produktu</TableHead>
                            <TableHead className="w-[100px]">Akcje</TableHead>
                          </TableRow>
                        </TableHeader>
                        <TableBody>
                          {Array.from({ length: Math.max(lengths.length, widths.length, heights.length) }).map((_, index) => (
                            <TableRow key={index} data-testid={`dimension-row-${index}`}>
                              <TableCell>
                                <Select 
                                  value={lengths[index] || ""} 
                                  onValueChange={(value) => {
                                    const newLengths = [...lengths];
                                    newLengths[index] = value;
                                    setLengths(newLengths);
                                  }}
                                >
                                  <SelectTrigger data-testid={`select-length-${index}`} className="h-8">
                                    <SelectValue placeholder="-" />
                                  </SelectTrigger>
                                  <SelectContent>
                                    {getDimensionValues(dimensionLengthOptions).map((length) => (
                                      <SelectItem key={length} value={length}>
                                        {length}
                                      </SelectItem>
                                    ))}
                                  </SelectContent>
                                </Select>
                              </TableCell>
                              <TableCell>
                                <Select 
                                  value={widths[index] || ""} 
                                  onValueChange={(value) => {
                                    const newWidths = [...widths];
                                    newWidths[index] = value;
                                    setWidths(newWidths);
                                  }}
                                >
                                  <SelectTrigger data-testid={`select-width-${index}`} className="h-8">
                                    <SelectValue placeholder="-" />
                                  </SelectTrigger>
                                  <SelectContent>
                                    {getDimensionValues(dimensionWidthOptions).map((width) => (
                                      <SelectItem key={width} value={width}>
                                        {width}
                                      </SelectItem>
                                    ))}
                                  </SelectContent>
                                </Select>
                              </TableCell>
                              <TableCell>
                                <Select 
                                  value={heights[index] || ""} 
                                  onValueChange={(value) => {
                                    const newHeights = [...heights];
                                    newHeights[index] = value;
                                    setHeights(newHeights);
                                  }}
                                >
                                  <SelectTrigger data-testid={`select-height-${index}`} className="h-8">
                                    <SelectValue placeholder="-" />
                                  </SelectTrigger>
                                  <SelectContent>
                                    {getDimensionValues(dimensionHeightOptions).map((height) => (
                                      <SelectItem key={height} value={height}>
                                        {height}
                                      </SelectItem>
                                    ))}
                                  </SelectContent>
                                </Select>
                              </TableCell>
                              <TableCell>
                                <Select 
                                  value={productGroupsArray[index] || ""} 
                                  onValueChange={(value) => {
                                    const newGroups = [...productGroupsArray];
                                    newGroups[index] = value;
                                    setProductGroupsArray(newGroups);
                                  }}
                                >
                                  <SelectTrigger data-testid={`select-product-group-${index}`} className="h-8">
                                    <SelectValue placeholder="Wybierz grupę" />
                                  </SelectTrigger>
                                  <SelectContent>
                                    {productGroups?.map((pg) => (
                                      <SelectItem key={pg.id} value={pg.name || pg.code}>
                                        {pg.readableName || pg.name || pg.code}
                                      </SelectItem>
                                    ))}
                                  </SelectContent>
                                </Select>
                              </TableCell>
                              <TableCell>
                                <div className="flex gap-1">
                                  <Button
                                    size="sm"
                                    variant="ghost"
                                    onClick={() => handleCopyDimension(index)}
                                    data-testid={`button-copy-dimension-${index}`}
                                    title="Kopiuj rozmiar"
                                  >
                                    <Copy className="h-4 w-4" />
                                  </Button>
                                  <Button
                                    size="sm"
                                    variant="ghost"
                                    onClick={() => handleRemoveDimension(index)}
                                    data-testid={`button-remove-dimension-${index}`}
                                    title="Usuń rozmiar"
                                  >
                                    <X className="h-4 w-4" />
                                  </Button>
                                </div>
                              </TableCell>
                            </TableRow>
                          ))}
                        </TableBody>
                      </Table>
                    ) : (
                      <p className="text-muted-foreground text-center py-4">
                        Brak rozmiarów. Dodaj rozmiary używając pól powyżej.
                      </p>
                    )}
                  </div>
                </div>
              </div>
            </TabsContent>

            {/* Tab: Kolory */}
            <TabsContent value="kolory" className="space-y-4">
              <div className="space-y-4">
                <div className="flex items-end gap-2">
                  <div className="flex-1">
                    <Label htmlFor="newColor">Nowy kolor</Label>
                    <Input
                      id="newColor"
                      value={newColorInput}
                      onChange={(e) => setNewColorInput(e.target.value)}
                      placeholder="np. BIAŁY"
                      onKeyDown={(e) => {
                        if (e.key === "Enter") {
                          e.preventDefault();
                          handleAddColor();
                        }
                      }}
                      data-testid="input-new-color"
                    />
                  </div>
                  <Button onClick={handleAddColor} data-testid="button-add-color">
                    <Plus className="h-4 w-4" />
                  </Button>
                  <Button onClick={loadAllColorsFromDict} variant="outline" data-testid="button-load-colors">
                    Załaduj wszystkie ze słownika
                  </Button>
                </div>

                <div className="space-y-2">
                  <Label>Lista kolorów ({colors.length})</Label>
                  <div className="border rounded-md p-4 space-y-2 max-h-96 overflow-y-auto">
                    {colors.length > 0 ? (
                      colors.map((color, index) => (
                        <div
                          key={index}
                          className="flex items-center justify-between p-2 bg-muted rounded hover-elevate"
                          data-testid={`color-item-${index}`}
                        >
                          <span>{color}</span>
                          <Button
                            size="sm"
                            variant="ghost"
                            onClick={() => handleRemoveColor(index)}
                            data-testid={`button-remove-color-${index}`}
                          >
                            <X className="h-4 w-4" />
                          </Button>
                        </div>
                      ))
                    ) : (
                      <p className="text-muted-foreground text-center py-4">
                        Brak kolorów. Dodaj kolory używając pola powyżej lub załaduj ze słownika.
                      </p>
                    )}
                  </div>
                </div>
              </div>
            </TabsContent>

            {/* Tab: Opis */}
            <TabsContent value="opis" className="space-y-4">
              <div className="space-y-4">
                <div>
                  <Label>Szablon opisu produktu</Label>
                  <p className="text-sm text-muted-foreground mb-2">
                    Możesz użyć zmiennych: {"{color}"}, {"{productType}"}, {"{length}"}, {"{width}"}, {"{height}"}, {"{size}"}
                  </p>
                  <RichTextEditor
                    content={descriptionDoc}
                    onContentChange={(html, doc) => {
                      setDescriptionHtml(html);
                      setDescriptionDoc(doc);
                    }}
                  />
                </div>

                <div className="space-y-2">
                  <div className="flex items-center gap-2">
                    <input
                      type="checkbox"
                      id="useAi"
                      checked={useAiGeneration}
                      onChange={(e) => setUseAiGeneration(e.target.checked)}
                      data-testid="checkbox-use-ai"
                    />
                    <Label htmlFor="useAi">Użyj AI do generowania opisów</Label>
                  </div>

                  {useAiGeneration && (
                    <div className="space-y-2">
                      <Label htmlFor="aiPrompt">Prompt dla AI</Label>
                      <Textarea
                        id="aiPrompt"
                        className="min-h-[100px] bg-background text-foreground"
                        value={aiPrompt}
                        onChange={(e) => setAiPrompt(e.target.value)}
                        placeholder="Wpisz instrukcje dla AI..."
                        data-testid="textarea-ai-prompt"
                      />
                      <div className="flex items-center gap-2">
                        <Button
                          type="button"
                          variant="secondary"
                          onClick={() => {
                            if (selectedMatrix?.id) {
                              generateAiDescriptionMutation.mutate(selectedMatrix.id);
                            }
                          }}
                          disabled={!selectedMatrix?.id || generateAiDescriptionMutation.isPending}
                          data-testid="button-generate-ai-description"
                        >
                          {generateAiDescriptionMutation.isPending ? (
                            <Loader2 className="h-4 w-4 animate-spin mr-2" />
                          ) : (
                            <Sparkles className="h-4 w-4 mr-2" />
                          )}
                          Generuj opis AI
                        </Button>
                        {!selectedMatrix?.id && (
                          <p className="text-xs text-muted-foreground">
                            Zapisz matrycę aby wygenerować opis AI
                          </p>
                        )}
                      </div>
                    </div>
                  )}
                </div>
              </div>
            </TabsContent>
          </Tabs>

          <div className="flex justify-end gap-2 mt-4">
            <Button variant="outline" onClick={handleCloseDialog} data-testid="button-cancel">
              Anuluj
            </Button>
            <Button onClick={handleSave} disabled={saveMutation.isPending} data-testid="button-save">
              {saveMutation.isPending ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : null}
              Zapisz
            </Button>
          </div>
        </DialogContent>
      </Dialog>

      {/* Delete Confirmation Dialog */}
      <AlertDialog open={isDeleteDialogOpen} onOpenChange={setIsDeleteDialogOpen}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno usunąć?</AlertDialogTitle>
            <AlertDialogDescription>
              Ta akcja nie może być cofnięta. Matryca zostanie trwale usunięta.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel data-testid="button-cancel-delete">Anuluj</AlertDialogCancel>
            <AlertDialogAction onClick={confirmDelete} data-testid="button-confirm-delete">
              {deleteMutation.isPending ? <Loader2 className="h-4 w-4 animate-spin mr-2" /> : null}
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
