import { useState, useCallback, useEffect } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Label } from "@/components/ui/label";
import { useToast } from "@/hooks/use-toast";
import { ArrowLeft, Save, Upload, X, ExternalLink, Package, Files, Loader2 } from "lucide-react";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useDropzone } from "react-dropzone";
import { AccessoryGroupComboboxWithAdd } from "@/components/accessory-group-combobox-with-add";
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Link } from "wouter";

type AccessoryGroup = {
  id: number;
  name: string;
  code: string;
};

type WarehouseMaterial = {
  id: number;
  name: string;
  internalCode: string | null;
  supplierCode: string | null;
  price: string | null;
  description: string | null;
  category: string;
  groupName: string | null;
  primaryImageUrl: string | null;
  gallery: string[];
};

type Accessory = {
  id: number;
  groupId: number | null;
  warehouseMaterialId: number | null;
  name: string;
  code: string;
  alpmaCode: string | null;
  description: string | null;
  price: string | null;
  imageUrl: string | null;
  displayOrder: number;
  isActive: boolean;
};

export default function AccessoryEditor() {
  const { id } = useParams();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const isNew = id === "new";

  const [duplicateDialogOpen, setDuplicateDialogOpen] = useState(false);
  const [duplicateTargetGroupId, setDuplicateTargetGroupId] = useState<string>("");

  const [formData, setFormData] = useState({
    groupId: "",
    warehouseMaterialId: "",
    name: "",
    code: "",
    alpmaCode: "",
    description: "",
    imageUrl: "",
    displayOrder: "0",
    price: "",
  });

  const [uploadedFile, setUploadedFile] = useState<File | null>(null);
  const [previewUrl, setPreviewUrl] = useState<string | null>(null);
  const [materialSearchOpen, setMaterialSearchOpen] = useState(false);
  const [isWarehouseMaterial, setIsWarehouseMaterial] = useState(true); // true = z magazynu, false = opisowe

  // Fetch groups
  const { data: groups = [] } = useQuery<AccessoryGroup[]>({
    queryKey: ["/api/accessory-groups"],
  });

  // Fetch warehouse materials
  const { data: materials = [] } = useQuery<WarehouseMaterial[]>({
    queryKey: ["/api/warehouse/materials/all"],
  });

  // Fetch accessory if editing
  const { data: accessory, isLoading } = useQuery<Accessory>({
    queryKey: [`/api/accessories/${id}`],
    enabled: !isNew && !!id,
  });

  // Fetch linked warehouse material details
  const { data: linkedMaterial } = useQuery<WarehouseMaterial>({
    queryKey: [`/api/warehouse/materials/${formData.warehouseMaterialId}`],
    enabled: !!formData.warehouseMaterialId && formData.warehouseMaterialId !== "",
  });

  // Load groupId from URL query params for new accessory
  useEffect(() => {
    if (isNew && groups.length > 0) {
      const params = new URLSearchParams(window.location.search);
      const groupCodeParam = params.get('groupId');
      if (groupCodeParam) {
        // Find group by code
        const group = groups.find(g => g.code === groupCodeParam);
        if (group) {
          setFormData(prev => ({ ...prev, groupId: group.id.toString() }));
        }
      }
    }
  }, [isNew, groups]);

  // Load data into form when editing
  useEffect(() => {
    if (accessory && !isNew) {
      setFormData({
        groupId: accessory.groupId?.toString() || "",
        warehouseMaterialId: accessory.warehouseMaterialId?.toString() || "",
        name: accessory.name || "",
        code: accessory.code || "",
        alpmaCode: accessory.alpmaCode || "",
        description: accessory.description || "",
        imageUrl: accessory.imageUrl || "",
        displayOrder: accessory.displayOrder?.toString() || "0",
        price: accessory.price || "",
      });
      // Set material type based on whether it has warehouse link
      setIsWarehouseMaterial(!!accessory.warehouseMaterialId);
      if (accessory.imageUrl) {
        setPreviewUrl(accessory.imageUrl);
      }
    }
  }, [accessory, isNew]);

  // Auto-copy material data when material is selected
  const handleMaterialSelect = (materialId: string) => {
    const material = materials.find((m) => m.id === parseInt(materialId));
    if (material) {
      // Get image URL from material (primaryImageUrl or first from gallery)
      const imageUrl = material.primaryImageUrl || (material.gallery && material.gallery.length > 0 ? material.gallery[0] : "");
      
      const updatedData = {
        ...formData,
        warehouseMaterialId: materialId,
        name: material.name,
        description: material.description || "",
        price: material.price || "",
        imageUrl: imageUrl,
      };
      setFormData(updatedData);
      
      // Update preview if image exists
      if (imageUrl) {
        setPreviewUrl(imageUrl);
      }
      
      // Auto-save after selecting material
      autoSaveMutation.mutate(updatedData);
    }
    setMaterialSearchOpen(false);
  };

  // Drag & drop for image upload
  const onDrop = useCallback((acceptedFiles: File[]) => {
    const file = acceptedFiles[0];
    if (file) {
      setUploadedFile(file);
      const reader = new FileReader();
      reader.onload = () => {
        setPreviewUrl(reader.result as string);
      };
      reader.readAsDataURL(file);
    }
  }, []);

  const { getRootProps, getInputProps, isDragActive } = useDropzone({
    onDrop,
    accept: {
      "image/*": [".png", ".jpg", ".jpeg", ".gif", ".webp"],
    },
    maxFiles: 1,
  });

  // Upload image mutation
  const uploadImageMutation = useMutation({
    mutationFn: async (file: File) => {
      const formData = new FormData();
      formData.append("image", file);
      const response = await fetch("/api/upload/accessory-image", {
        method: "POST",
        body: formData,
        credentials: "include",
      });
      if (!response.ok) throw new Error("Upload failed");
      return response.json();
    },
  });

  // Save mutation
  const saveMutation = useMutation({
    mutationFn: async (data: any) => {
      // Upload image first if there's a new file
      let imageUrl = formData.imageUrl;
      if (uploadedFile) {
        const uploadResult = await uploadImageMutation.mutateAsync(uploadedFile);
        imageUrl = uploadResult.url;
      }

      const payload = {
        ...data,
        imageUrl,
        groupId: data.groupId ? parseInt(data.groupId) : null,
        warehouseMaterialId: data.warehouseMaterialId ? parseInt(data.warehouseMaterialId) : null,
        displayOrder: parseInt(data.displayOrder) || 0,
        price: data.price || null,
      };

      const response = isNew
        ? await apiRequest("POST", "/api/accessories", payload)
        : await apiRequest("PUT", `/api/accessories/${id}`, payload);
      
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      queryClient.invalidateQueries({ queryKey: [`/api/accessories/${id}`] });
      
      toast({
        title: "Zapisano",
        description: isNew ? "Akcesorium utworzone" : "Zmiany zapisane",
      });
      
      // If it was a new accessory, navigate to edit mode
      if (isNew && data.id) {
        navigate(`/accessories/${data.id}/edit`);
      }
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zapisać akcesorium",
        variant: "destructive",
      });
    },
  });

  // Auto-save mutation (without image upload)
  const autoSaveMutation = useMutation({
    mutationFn: async (data: any) => {
      const payload = {
        ...data,
        groupId: data.groupId ? parseInt(data.groupId) : null,
        warehouseMaterialId: data.warehouseMaterialId ? parseInt(data.warehouseMaterialId) : null,
        displayOrder: parseInt(data.displayOrder) || 0,
        price: data.price || null,
      };

      const response = isNew
        ? await apiRequest("POST", "/api/accessories", payload)
        : await apiRequest("PUT", `/api/accessories/${id}`, payload);
      
      return await response.json();
    },
    onSuccess: (data) => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      queryClient.invalidateQueries({ queryKey: [`/api/accessories/${id}`] });
      
      toast({
        title: "Zapisano automatycznie",
        description: "Dane z magazynu zostały zapisane",
      });
      
      // If it was a new accessory, navigate to edit mode
      if (isNew && data.id) {
        navigate(`/accessories/${data.id}/edit`);
      }
    },
    onError: (error: any) => {
      toast({
        title: "Błąd auto-zapisu",
        description: error.message || "Nie udało się zapisać zmian",
        variant: "destructive",
      });
    },
  });

  // Duplicate mutation
  const duplicateMutation = useMutation({
    mutationFn: async ({ accessoryId, targetGroupId }: { accessoryId: number; targetGroupId: number }) => {
      return await apiRequest("POST", `/api/accessories/${accessoryId}/duplicate`, {
        targetGroupId,
      }) as unknown as { id: number };
    },
    onSuccess: (data: { id: number }) => {
      queryClient.invalidateQueries({ queryKey: ["/api/accessories"] });
      toast({
        title: "Sukces",
        description: "Akcesorium zostało zduplikowane",
      });
      setDuplicateDialogOpen(false);
      setDuplicateTargetGroupId("");
      // Navigate to the duplicated accessory
      navigate(`/accessories/${data.id}/edit`);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zduplikować akcesorium",
        variant: "destructive",
      });
    },
  });

  const handleDuplicateClick = () => {
    if (!accessory) return;
    setDuplicateTargetGroupId(accessory.groupId?.toString() || "");
    setDuplicateDialogOpen(true);
  };

  const handleDuplicateConfirm = () => {
    if (!accessory || !duplicateTargetGroupId) {
      toast({
        title: "Błąd",
        description: "Wybierz grupę docelową",
        variant: "destructive",
      });
      return;
    }

    duplicateMutation.mutate({
      accessoryId: accessory.id,
      targetGroupId: parseInt(duplicateTargetGroupId),
    });
  };

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault();
    saveMutation.mutate(formData);
  };

  const removeImage = () => {
    setUploadedFile(null);
    setPreviewUrl(null);
    setFormData({ ...formData, imageUrl: "" });
  };

  if (isLoading && !isNew) {
    return (
      <div className="p-6">
        <div className="text-sm text-muted-foreground">Ładowanie...</div>
      </div>
    );
  }

  const selectedMaterial = materials.find((m) => m.id === parseInt(formData.warehouseMaterialId));

  return (
    <div className="p-4 md:p-6 space-y-4">
      <div className="flex items-center gap-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={() => window.history.back()}
          data-testid="button-back"
        >
          <ArrowLeft className="h-4 w-4" />
        </Button>
        <div>
          <h1 className="text-xl md:text-2xl font-bold">
            {isNew ? "Nowe akcesorium" : "Edycja akcesorium"}
          </h1>
          <p className="text-xs md:text-sm text-muted-foreground">
            {isNew
              ? "Dodaj nowe akcesorium do katalogu"
              : "Edytuj dane akcesorium"}
          </p>
        </div>
      </div>

      <form onSubmit={handleSubmit}>
        <div className="grid grid-cols-1 lg:grid-cols-4 gap-4">
          {/* Main form - 3 columns */}
          <div className="lg:col-span-3">
            <Card>
              <CardHeader className="pb-3">
                <CardTitle className="text-base">Podstawowe informacje</CardTitle>
              </CardHeader>
              <CardContent className="space-y-3">
                {/* Type selection: warehouse material or descriptive */}
                <div className="space-y-2">
                  <Label>Typ akcesorium</Label>
                  <div className="flex gap-4">
                    <label className="flex items-center gap-2 cursor-pointer">
                      <input
                        type="radio"
                        checked={isWarehouseMaterial}
                        onChange={() => {
                          setIsWarehouseMaterial(true);
                        }}
                        className="rounded-full"
                        data-testid="radio-warehouse-material"
                      />
                      <span className="text-sm">Materiał z magazynu</span>
                    </label>
                    <label className="flex items-center gap-2 cursor-pointer">
                      <input
                        type="radio"
                        checked={!isWarehouseMaterial}
                        onChange={() => {
                          setIsWarehouseMaterial(false);
                          // Clear warehouse material and its image when switching to descriptive
                          setFormData({ 
                            ...formData, 
                            warehouseMaterialId: "",
                            imageUrl: "" 
                          });
                          setPreviewUrl(null);
                          setUploadedFile(null);
                        }}
                        className="rounded-full"
                        data-testid="radio-descriptive"
                      />
                      <span className="text-sm">Akcesorium opisowe (poza magazynem)</span>
                    </label>
                  </div>
                </div>

                {/* Material link from warehouse - only if warehouse material type */}
                {isWarehouseMaterial && (
                  <div className="space-y-2">
                    <Label>Materiał z magazynu</Label>
                    <Popover open={materialSearchOpen} onOpenChange={setMaterialSearchOpen}>
                    <PopoverTrigger asChild>
                      <Button
                        variant="outline"
                        role="combobox"
                        aria-expanded={materialSearchOpen}
                        className="w-full justify-between"
                        type="button"
                        data-testid="button-select-material"
                      >
                        {selectedMaterial ? (
                          <span className="truncate">
                            {selectedMaterial.name}
                            {selectedMaterial.internalCode && ` (${selectedMaterial.internalCode})`}
                          </span>
                        ) : (
                          <span className="text-muted-foreground">Wybierz materiał...</span>
                        )}
                        <Package className="ml-2 h-4 w-4 shrink-0 opacity-50" />
                      </Button>
                    </PopoverTrigger>
                    <PopoverContent className="w-[400px] p-0" align="start">
                      <Command>
                        <CommandInput placeholder="Szukaj materiału..." />
                        <CommandList>
                          <CommandEmpty>Nie znaleziono materiału.</CommandEmpty>
                          <CommandGroup>
                            {materials.map((material) => (
                              <CommandItem
                                key={material.id}
                                value={`${material.name} ${material.internalCode || ""} ${material.supplierCode || ""}`}
                                onSelect={() => handleMaterialSelect(material.id.toString())}
                              >
                                <div className="flex flex-col">
                                  <div className="font-medium">{material.name}</div>
                                  <div className="text-xs text-muted-foreground">
                                    {material.category} {material.groupName && `• ${material.groupName}`}
                                    {material.internalCode && ` • ${material.internalCode}`}
                                  </div>
                                </div>
                              </CommandItem>
                            ))}
                          </CommandGroup>
                        </CommandList>
                      </Command>
                    </PopoverContent>
                    </Popover>
                  
                    {/* Display linked material info */}
                  {linkedMaterial && (
                    <div className="p-2 border rounded-md bg-muted/50 space-y-1">
                      <div className="flex items-center justify-between gap-2">
                        <div className="text-xs font-medium">Połączony materiał:</div>
                        <Link href={`/warehouse/${linkedMaterial.category}?materialId=${linkedMaterial.id}`}>
                          <Button variant="ghost" size="sm" className="h-6 text-xs" type="button">
                            <ExternalLink className="h-3 w-3 mr-1" />
                            Otwórz w magazynie
                          </Button>
                        </Link>
                      </div>
                      <div className="text-xs text-muted-foreground">
                        Kategoria: {linkedMaterial.category}
                        {linkedMaterial.groupName && ` • ${linkedMaterial.groupName}`}
                      </div>
                      {linkedMaterial.price && (
                        <div className="text-xs font-medium">
                          Cena: {parseFloat(linkedMaterial.price).toFixed(2)} PLN
                        </div>
                      )}
                      {linkedMaterial.internalCode && (
                        <div className="text-xs text-muted-foreground">
                          Kod wewnętrzny: {linkedMaterial.internalCode}
                        </div>
                      )}
                    </div>
                  )}
                  </div>
                )}

                <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                  <div className="space-y-2">
                    <Label htmlFor="group">Grupa</Label>
                    <AccessoryGroupComboboxWithAdd
                      groups={groups}
                      value={formData.groupId}
                      onChange={(value) => setFormData({ ...formData, groupId: value })}
                      placeholder="Wybierz grupę"
                      searchPlaceholder="Szukaj grupy..."
                      emptyText="Nie znaleziono grup."
                      testId="select-group"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="displayOrder">Kolejność</Label>
                    <Input
                      id="displayOrder"
                      type="number"
                      value={formData.displayOrder}
                      onChange={(e) =>
                        setFormData({ ...formData, displayOrder: e.target.value })
                      }
                      data-testid="input-display-order"
                    />
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="name">
                    Nazwa <span className="text-destructive">*</span>
                  </Label>
                  <Input
                    id="name"
                    value={formData.name}
                    onChange={(e) =>
                      setFormData({ ...formData, name: e.target.value })
                    }
                    required
                    data-testid="input-name"
                  />
                </div>

                <div className="grid grid-cols-1 md:grid-cols-3 gap-3">
                  <div className="space-y-2">
                    <Label htmlFor="code">
                      Kod <span className="text-destructive">*</span>
                    </Label>
                    <Input
                      id="code"
                      value={formData.code}
                      onChange={(e) =>
                        setFormData({ ...formData, code: e.target.value })
                      }
                      required
                      data-testid="input-code"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="alpmaCode">Kod Alpma</Label>
                    <Input
                      id="alpmaCode"
                      value={formData.alpmaCode}
                      onChange={(e) =>
                        setFormData({ ...formData, alpmaCode: e.target.value })
                      }
                      placeholder="Własny kod produktu"
                      data-testid="input-alpma-code"
                    />
                  </div>

                  <div className="space-y-2">
                    <Label htmlFor="price">Cena (PLN)</Label>
                    <Input
                      id="price"
                      type="number"
                      step="0.01"
                      value={formData.price}
                      onChange={(e) =>
                        setFormData({ ...formData, price: e.target.value })
                      }
                      placeholder="0.00"
                      data-testid="input-price"
                    />
                  </div>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="description">Opis</Label>
                  <Textarea
                    id="description"
                    value={formData.description}
                    onChange={(e) =>
                      setFormData({ ...formData, description: e.target.value })
                    }
                    rows={3}
                    data-testid="textarea-description"
                  />
                </div>
              </CardContent>
            </Card>
          </div>

          {/* Image upload - compact, 1 column */}
          <div className="lg:col-span-1">
            <Card>
              <CardHeader className="pb-3">
                <CardTitle className="text-base">Zdjęcie</CardTitle>
              </CardHeader>
              <CardContent>
                {/* For warehouse materials: show linked image (read-only) */}
                {isWarehouseMaterial && formData.warehouseMaterialId && formData.imageUrl ? (
                  <div className="space-y-2">
                    <div className="relative aspect-square border overflow-hidden bg-muted rounded-md">
                      <img
                        src={formData.imageUrl}
                        alt="Zdjęcie z magazynu"
                        className="w-full h-full object-cover"
                      />
                      <div className="absolute bottom-0 left-0 right-0 bg-black/60 text-white text-xs p-1.5 text-center">
                        Zdjęcie z magazynu
                      </div>
                    </div>
                    <p className="text-xs text-muted-foreground text-center">
                      Zdjęcie pochodzi z połączonego materiału magazynowego
                    </p>
                  </div>
                ) : (
                  /* For descriptive accessories: allow image upload */
                  previewUrl || formData.imageUrl ? (
                    <div className="space-y-2">
                      <div className="relative aspect-square border overflow-hidden bg-muted rounded-md">
                        <img
                          src={previewUrl || formData.imageUrl}
                          alt="Podgląd"
                          className="w-full h-full object-cover"
                        />
                        <Button
                          type="button"
                          variant="destructive"
                          size="icon"
                          className="absolute top-1 right-1 h-7 w-7"
                          onClick={removeImage}
                          data-testid="button-remove-image"
                        >
                          <X className="h-3 w-3" />
                        </Button>
                      </div>
                    </div>
                  ) : (
                    <div
                      {...getRootProps()}
                      className={`border-2 border-dashed aspect-square rounded-md flex flex-col items-center justify-center cursor-pointer hover-elevate transition-colors p-2 ${
                        isDragActive ? "border-primary bg-accent" : ""
                      }`}
                      data-testid="dropzone-image"
                    >
                      <input {...getInputProps()} />
                      <Upload className="h-6 w-6 mb-1 text-muted-foreground" />
                      <p className="text-xs text-center text-muted-foreground px-2">
                        {isDragActive
                          ? "Upuść tutaj"
                          : "Przeciągnij lub kliknij aby dodać zdjęcie"}
                      </p>
                    </div>
                  )
                )}
              </CardContent>
            </Card>
          </div>
        </div>

        <div className="flex justify-end gap-2 mt-4">
          <Button
            type="button"
            variant="outline"
            onClick={() => navigate("/accessories")}
            data-testid="button-cancel"
          >
            Anuluj
          </Button>
          {!isNew && accessory && (
            <Button
              type="button"
              variant="outline"
              onClick={handleDuplicateClick}
              data-testid="button-duplicate"
            >
              <Files className="w-4 h-4 mr-2" />
              Duplikuj
            </Button>
          )}
          <Button
            type="submit"
            disabled={saveMutation.isPending || uploadImageMutation.isPending}
            data-testid="button-save"
          >
            <Save className="h-4 w-4 mr-2" />
            {saveMutation.isPending || uploadImageMutation.isPending
              ? "Zapisywanie..."
              : "Zapisz"}
          </Button>
        </div>
      </form>

      {/* Duplicate Dialog */}
      <Dialog open={duplicateDialogOpen} onOpenChange={setDuplicateDialogOpen}>
        <DialogContent data-testid="dialog-duplicate-accessory">
          <DialogHeader>
            <DialogTitle>Duplikuj akcesorium</DialogTitle>
            <DialogDescription>
              {accessory && `Duplikuj: ${accessory.name}`}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            <div className="space-y-2">
              <Label htmlFor="target-group">Grupa docelowa</Label>
              <Select value={duplicateTargetGroupId} onValueChange={setDuplicateTargetGroupId}>
                <SelectTrigger id="target-group" data-testid="select-target-group">
                  <SelectValue placeholder="Wybierz grupę" />
                </SelectTrigger>
                <SelectContent>
                  {groups.map((group) => (
                    <SelectItem key={group.id} value={group.id.toString()}>
                      {group.name}
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              <p className="text-xs text-muted-foreground">
                Kod akcesorium zostanie automatycznie dostosowany jeśli będzie konflikt
              </p>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setDuplicateDialogOpen(false)}
              data-testid="button-cancel-duplicate"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleDuplicateConfirm}
              disabled={duplicateMutation.isPending || !duplicateTargetGroupId}
              data-testid="button-confirm-duplicate"
            >
              {duplicateMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Duplikowanie...
                </>
              ) : (
                <>
                  <Files className="w-4 h-4 mr-2" />
                  Duplikuj
                </>
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
