import { Pool } from "pg";
import dotenv from "dotenv";

dotenv.config();

const pool = new Pool({
  connectionString: process.env.DATABASE_URL,
});

// Helper function to build stable matrix variant ID (same as in routes.ts)
function buildMatrixVariantId(
  matrixId: number,
  length: number | null,
  width: number | null,
  height: number | null,
  color: string | null,
  colorOptions?: string[]
): string {
  const dimParts = [
    length !== null ? length.toString() : 'null',
    width !== null ? width.toString() : 'null',
    height !== null ? height.toString() : 'null',
  ];
  
  let id = `${matrixId}-${dimParts.join('-')}-${color || 'no-color'}`;
  
  // Add color options if they exist (sorted for deterministic ID)
  if (colorOptions && colorOptions.length > 0) {
    const sortedOptions = colorOptions.slice().sort().join('-');
    id += `-${sortedOptions}`;
  }
  
  return id;
}

async function testMatrixVariantIdLookup() {
  console.log('🧪 === TEST: Matrix Variant ID Lookup ===\n');
  
  try {
    // Product 310 ma zmieniony SKU na "CUSTOM-SKU-TEST"
    // Powinien być znaleziony po matrix_variant_id, nie po SKU
    const testProductId = 310;
    
    console.log(`📦 Sprawdzam produkt ID ${testProductId}...\n`);
    
    // Get product details
    const productResult = await pool.query(
      `SELECT id, sku, matrix_id, length, width, height, color, color_options, matrix_variant_id
       FROM catalog.products
       WHERE id = $1`,
      [testProductId]
    );
    
    if (productResult.rows.length === 0) {
      console.error('❌ Produkt nie znaleziony!');
      return;
    }
    
    const product = productResult.rows[0];
    const colorOptions = product.color_options || [];
    
    console.log('📋 Dane produktu:');
    console.log(`   ID: ${product.id}`);
    console.log(`   SKU: ${product.sku}`);
    console.log(`   Matrix ID: ${product.matrix_id}`);
    console.log(`   Dimensions: ${product.length} × ${product.width} × ${product.height}`);
    console.log(`   Color: ${product.color || 'brak'}`);
    console.log(`   Color Options: ${colorOptions.length > 0 ? colorOptions.join(', ') : 'brak'}`);
    console.log(`   Matrix Variant ID (DB): ${product.matrix_variant_id}\n`);
    
    // Build expected matrix_variant_id from dimensions
    const expectedVariantId = buildMatrixVariantId(
      product.matrix_id,
      parseFloat(product.length),
      parseFloat(product.width),
      parseFloat(product.height),
      product.color,
      colorOptions
    );
    
    console.log(`🔑 Oczekiwane Matrix Variant ID: ${expectedVariantId}`);
    console.log(`🔑 Matrix Variant ID z bazy: ${product.matrix_variant_id}\n`);
    
    if (expectedVariantId !== product.matrix_variant_id) {
      console.error('❌ BŁĄD: Matrix Variant ID nie zgadza się!');
      console.error(`   Oczekiwano: ${expectedVariantId}`);
      console.error(`   Otrzymano: ${product.matrix_variant_id}`);
      return;
    }
    
    console.log('✅ Matrix Variant ID pasuje!\n');
    
    // Test 1: Lookup by matrix_variant_id (primary method)
    console.log('🔍 TEST 1: Szukanie po matrix_variant_id...');
    const lookupByVariantId = await pool.query(
      `SELECT id, sku FROM catalog.products 
       WHERE matrix_variant_id = $1`,
      [expectedVariantId]
    );
    
    if (lookupByVariantId.rows.length > 0) {
      const found = lookupByVariantId.rows[0];
      console.log(`   ✅ ZNALEZIONO: ID ${found.id}, SKU: ${found.sku}`);
      
      if (found.id === testProductId) {
        console.log(`   ✅ Poprawny produkt! (pomimo że SKU zmieniony z "SKU-310" na "${found.sku}")\n`);
      } else {
        console.error(`   ❌ BŁĄD: Znaleziono inny produkt! Oczekiwano ID ${testProductId}, otrzymano ${found.id}\n`);
      }
    } else {
      console.error('   ❌ BŁĄD: Nie znaleziono produktu po matrix_variant_id!\n');
    }
    
    // Test 2: Check for duplicates
    console.log('🔍 TEST 2: Sprawdzanie duplikatów...');
    const duplicatesCheck = await pool.query(
      `SELECT COUNT(*) as count
       FROM catalog.products
       WHERE matrix_id = $1 
         AND length = $2 
         AND width = $3 
         AND height = $4 
         AND (color = $5 OR (color IS NULL AND $5 IS NULL))`,
      [product.matrix_id, product.length, product.width, product.height, product.color]
    );
    
    const duplicateCount = parseInt(duplicatesCheck.rows[0].count);
    console.log(`   Produkty z tymi samymi wymiarami: ${duplicateCount}`);
    
    if (duplicateCount === 1) {
      console.log('   ✅ Brak duplikatów!\n');
    } else {
      console.warn(`   ⚠️  UWAGA: ${duplicateCount} produktów z tymi samymi wymiarami!\n`);
      
      // Show all matching products
      const allMatches = await pool.query(
        `SELECT id, sku, matrix_variant_id
         FROM catalog.products
         WHERE matrix_id = $1 
           AND length = $2 
           AND width = $3 
           AND height = $4 
           AND (color = $5 OR (color IS NULL AND $5 IS NULL))`,
        [product.matrix_id, product.length, product.width, product.height, product.color]
      );
      
      console.log('   Wszystkie pasujące produkty:');
      allMatches.rows.forEach(p => {
        console.log(`      - ID ${p.id}, SKU: ${p.sku}, matrix_variant_id: ${p.matrix_variant_id}`);
      });
      console.log();
    }
    
    // Summary
    console.log('✅ === TEST ZAKOŃCZONY POMYŚLNIE ===');
    console.log('   1. Matrix Variant ID buduje się prawidłowo');
    console.log('   2. Lookup po matrix_variant_id działa');
    console.log('   3. Znajdzie produkt pomimo zmienionego SKU');
    console.log('   4. Brak duplikatów dla tych samych wymiarów');
    
  } catch (error) {
    console.error('❌ Błąd podczas testu:', error);
    throw error;
  } finally {
    await pool.end();
  }
}

testMatrixVariantIdLookup()
  .then(() => {
    console.log('\n🎉 Test zakończony pomyślnie!');
    process.exit(0);
  })
  .catch((error) => {
    console.error('\n💥 Test zakończony z błędem:', error);
    process.exit(1);
  });
