import { pool } from '../../postgres';
import { resolveRoutingForPlan, type RoutingVariant, type PlanRoutingAssignment, type RoutingLogCallback } from './routing-variant-resolver';

/**
 * Pobiera wolny nośnik (paletę) lub tworzy nowy jeśli brak wolnych.
 * Nośnik jest "zajęty" jeśli jest przypisany do palety produkcyjnej, która nie jest ukończona.
 * 
 * @param client - Połączenie do bazy danych (transakcja)
 * @param usedCarrierIds - Zbiór ID nośników już przypisanych w tej sesji (żeby nie przypisać dwa razy)
 * @returns ID wolnego lub nowo utworzonego nośnika
 */
async function getOrCreateAvailableCarrier(
  client: any,
  usedCarrierIds: Set<number>
): Promise<number> {
  // 1. Znajdź grupę "PALLET-EURO" (domyślna dla palet produkcyjnych)
  const groupResult = await client.query(`
    SELECT id FROM production.production_carrier_groups
    WHERE code = 'PALLET-EURO' AND is_active = true
    LIMIT 1
  `);
  
  let carrierGroupId: number;
  
  if (groupResult.rows.length === 0) {
    // Utwórz grupę jeśli nie istnieje
    const newGroupResult = await client.query(`
      INSERT INTO production.production_carrier_groups 
      (code, name, description, default_capacity, capacity_unit, is_active, sort_order)
      VALUES ('PALLET-EURO', 'Palety Euro', 'Palety europejskie 1200x800mm', 500.00, 'kg', true, 10)
      ON CONFLICT (code) DO UPDATE SET code = 'PALLET-EURO'
      RETURNING id
    `);
    carrierGroupId = newGroupResult.rows[0].id;
  } else {
    carrierGroupId = groupResult.rows[0].id;
  }
  
  // 2. Znajdź wolny nośnik (nie przypisany do żadnej aktywnej palety i nie użyty w tej sesji)
  const usedIdsArray = Array.from(usedCarrierIds);
  const availableCarrierResult = await client.query(`
    SELECT c.id
    FROM production.production_carriers c
    WHERE c.carrier_group_id = $1
      AND c.is_active = true
      AND c.status = 'available'
      AND c.id NOT IN (
        SELECT DISTINCT pop.carrier_id 
        FROM production.production_order_pallets pop
        WHERE pop.carrier_id IS NOT NULL
          AND pop.status IN ('pending', 'in_progress')
      )
      ${usedIdsArray.length > 0 ? `AND c.id NOT IN (${usedIdsArray.join(',')})` : ''}
    ORDER BY c.code ASC
    LIMIT 1
  `, [carrierGroupId]);
  
  if (availableCarrierResult.rows.length > 0) {
    const carrierId = availableCarrierResult.rows[0].id;
    console.log(`[ZLP Generator] Przypisano wolny nośnik ID: ${carrierId}`);
    return carrierId;
  }
  
  // 3. Brak wolnych nośników - utwórz nowy
  // Znajdź najwyższy numer w grupie
  const maxCodeResult = await client.query(`
    SELECT MAX(
      CAST(
        NULLIF(REGEXP_REPLACE(code, '[^0-9]', '', 'g'), '') AS INTEGER
      )
    ) as max_num
    FROM production.production_carriers
    WHERE carrier_group_id = $1
  `, [carrierGroupId]);
  
  const nextNum = (maxCodeResult.rows[0].max_num || 0) + 1;
  const newCode = `Paleta Euro #${String(nextNum).padStart(3, '0')}`;
  
  const newCarrierResult = await client.query(`
    INSERT INTO production.production_carriers 
    (carrier_group_id, code, name, status, capacity, capacity_unit, dimensions, weight, is_active)
    VALUES ($1, $2, $3, 'available', 500.00, 'kg', '{"length": 1200, "width": 800, "height": 144, "unit": "mm"}'::jsonb, 25.00, true)
    RETURNING id
  `, [carrierGroupId, newCode, newCode]);
  
  const newCarrierId = newCarrierResult.rows[0].id;
  console.log(`[ZLP Generator] Utworzono nowy nośnik "${newCode}" (ID: ${newCarrierId})`);
  return newCarrierId;
}

interface ComponentAggregation {
  color: string;
  routingVariantCode: string | null;
  routingId: number | null;
  carrierId: number | null;
  components: ComponentForProduction[];
}

// New interface for color-based aggregation with multiple flows
interface ColorAggregation {
  color: string;
  flows: FlowGroup[];  // Multiple flows within the same color
  totalComponentCount: number;
}

interface FlowGroup {
  flowCode: string;  // OKLEJANIE-WIERCENIE, OKLEJANIE-GOTOWE, WIERCENIE, CIECIE-GOTOWE
  routingId: number | null;
  routingVariantCode: string | null;
  carrierId: number | null;
  splitAfterOperation?: string | null;  // cutting, edging - operacja po której paleta się rozdziela
  components: ComponentForProduction[];
}

interface ComponentForProduction {
  componentId: number;
  productId: number;
  planLineId: number;
  generatedName: string;
  componentType: string;
  color: string;
  materialType: string | null;
  length: number | null;
  width: number | null;
  thickness: number | null;
  quantity: number;
  edge1: boolean;
  edge2: boolean;
  edge3: boolean;
  edge4: boolean;
  routingVariant: RoutingVariant | null;
  planAssignment: PlanRoutingAssignment | null;
  sourceFurnitureReference: string | null;
}

interface GeneratedProductionOrder {
  orderNumber: string;
  colorCode: string;
  componentCount: number;
  totalQuantity: number;
  bomId: number;
  bomItemIds: number[];
  workOrderIds: number[];
}

interface PackedProductLineInfo {
  planLineId: number;
  productId: number;
  productName: string;
  productSku: string;
  quantity: number;
  reservedQuantity: number;
  packedProductId: number;
  packedProductSku: string;
  sourceReference: string | null;
  reservedItemIds: number[];
  reservedSerialNumbers: string[];
}

interface FormatkaReservationLineInfo {
  reservationId: number;
  planLineId: number | null;
  productSku: string;
  productName: string;
  quantityReserved: number;
  unitOfMeasure: string;
  locationId: number | null;
  locationName: string | null;
  notes: string | null;
}

interface GeneratedWarehouseDocument {
  docId: number;
  docNumber: string;
  docType: string;
  totalLines: number;
  totalQuantity: number;
}

export interface ProductionOrderGenerationResult {
  success: boolean;
  generatedOrders: GeneratedProductionOrder[];
  generatedDocuments: GeneratedWarehouseDocument[];
  errors: string[];
  summary: {
    planId: number;
    planNumber: string;
    totalOrders: number;
    totalComponents: number;
    totalPackedProducts: number;
    colorBreakdown: Record<string, number>;
  };
}

export interface ProgressCallback {
  (progress: {
    step: string;
    current: number;
    total: number;
    message: string;
    type: 'info' | 'success' | 'warning' | 'error';
  }): void;
}

/**
 * Generuje Production Orders (ZLP) z Production Plan z agregacją po kolorze.
 * 
 * Proces:
 * 1. Pobiera wszystkie linie z Production Plan
 * 2. Dla każdej linii pobiera komponenty produktu (formatki z bom.product_components)
 * 3. Resolve routing variant dla każdego komponentu
 * 4. Agreguje komponenty po color_code
 * 5. Dla każdego koloru tworzy Production Order z BOM i Work Orders
 * 
 * @param planId - ID Production Plan
 * @param onProgress - Optional callback for progress updates (SSE streaming)
 * @returns ProductionOrderGenerationResult z listą utworzonych zleceń
 */
export async function generateProductionOrdersFromPlan(
  planId: number,
  onProgress?: ProgressCallback
): Promise<ProductionOrderGenerationResult> {
  const emit = (step: string, current: number, total: number, message: string, type: 'info' | 'success' | 'warning' | 'error' = 'info') => {
    if (onProgress) {
      onProgress({ step, current, total, message, type });
    }
  };
  const client = await pool.connect();
  const errors: string[] = [];

  try {
    await client.query('BEGIN');
    emit('init', 0, 100, 'Rozpoczynam generowanie ZLP...', 'info');

    // 1. Pobierz plan i sprawdź status
    const planResult = await client.query(`
      SELECT id, plan_number, name, status, auto_assign_routings
      FROM production.production_plans
      WHERE id = $1
    `, [planId]);

    if (planResult.rows.length === 0) {
      emit('error', 0, 100, `Plan produkcyjny ${planId} nie znaleziony`, 'error');
      throw new Error(`Production Plan ${planId} not found`);
    }

    const plan = planResult.rows[0];
    const autoAssignRoutings = plan.auto_assign_routings ?? true; // Domyślnie true dla kompatybilności wstecznej
    emit('init', 5, 100, `Załadowano plan: ${plan.plan_number} (auto-assign: ${autoAssignRoutings ? 'ON' : 'OFF'})`, 'success');
    
    const allowedStatuses = ['draft', 'approved', 'in_progress', 'generated'];
    if (!allowedStatuses.includes(plan.status)) {
      emit('error', 5, 100, `Nieprawidłowy status planu: ${plan.status}`, 'error');
      throw new Error(`Cannot generate orders for plan with status: ${plan.status}`);
    }

    // 1.5. Check if there are existing ZLPs and clean them up (for regeneration)
    // ZLPs are linked via source_order_number = plan_number
    const existingZlpsResult = await client.query(`
      SELECT id FROM production.production_orders
      WHERE source_order_number = $1
    `, [plan.plan_number]);

    // Track if we need to reset line statuses
    let shouldResetLines = false;
    
    // 1.5a. Cleanup existing warehouse documents (WZ-SPAK, WZ-FORM) for this plan
    // Usuwamy tylko szkice (draft) - potwierdzone i zrealizowane dokumenty wymagają ręcznego anulowania
    const existingDraftDocsResult = await client.query(`
      SELECT id, doc_number, doc_type FROM warehouse.documents
      WHERE plan_id = $1 AND doc_type IN ('WZ-SPAK', 'WZ-FORM') AND status = 'draft'
    `, [planId]);
    
    // Sprawdź czy są nieprzetworzalne dokumenty (potwierdzone/zrealizowane)
    // Jeśli tak - BLOKUJEMY regenerację żeby uniknąć duplikatów
    const existingNonDraftDocsResult = await client.query(`
      SELECT id, doc_number, status, doc_type FROM warehouse.documents
      WHERE plan_id = $1 AND doc_type IN ('WZ-SPAK', 'WZ-FORM') AND status IN ('confirmed', 'printed', 'completed')
    `, [planId]);
    
    if (existingNonDraftDocsResult.rows.length > 0) {
      const docNumbers = existingNonDraftDocsResult.rows.map((r: any) => r.doc_number).join(', ');
      const errorMsg = `Nie można regenerować ZLP - istnieją potwierdzone dokumenty (${docNumbers}). Najpierw anuluj te dokumenty.`;
      emit('error', 5, 100, errorMsg, 'error');
      await client.query('ROLLBACK');
      throw new Error(errorMsg);
    }
    
    if (existingDraftDocsResult.rows.length > 0) {
      const docIds = existingDraftDocsResult.rows.map((r: any) => r.id);
      const wzSpakCount = existingDraftDocsResult.rows.filter((r: any) => r.doc_type === 'WZ-SPAK').length;
      const wzFormCount = existingDraftDocsResult.rows.filter((r: any) => r.doc_type === 'WZ-FORM').length;
      
      // Delete document lines first
      await client.query(`
        DELETE FROM warehouse.document_lines
        WHERE document_id = ANY($1::int[])
      `, [docIds]);
      
      // Delete documents
      await client.query(`
        DELETE FROM warehouse.documents
        WHERE id = ANY($1::int[])
      `, [docIds]);
      
      const deletedTypes = [];
      if (wzSpakCount > 0) deletedTypes.push(`${wzSpakCount} WZ-SPAK`);
      if (wzFormCount > 0) deletedTypes.push(`${wzFormCount} WZ-FORM`);
      emit('cleanup', 6, 100, `Usunięto szkice dokumentów: ${deletedTypes.join(', ')}`, 'info');
    }
    
    // 1.5b. UWAGA: NIE zwalniamy rezerwacji packed_product_items podczas regeneracji!
    // Rezerwacje są zachowywane - są powiązane z liniami planu i pozostają aktywne.
    // Rezerwacje są zwalniane TYLKO gdy linia planu jest usuwana (deletePlanLine w planning.ts)
    // Dzięki temu regeneracja ZLP nie powoduje utraty rezerwacji.

    if (existingZlpsResult.rows.length > 0) {
      emit('cleanup', 7, 100, `Usuwanie ${existingZlpsResult.rows.length} istniejących ZLP...`, 'info');
      
      // 1. Delete BOM items first (deepest child records)
      await client.query(`
        DELETE FROM production.production_order_bom_items
        WHERE production_order_bom_id IN (
          SELECT pob.id FROM production.production_order_boms pob
          JOIN production.production_orders po ON po.id = pob.production_order_id
          WHERE po.source_order_number = $1
        )
      `, [plan.plan_number]);
      
      // 1.5. Delete pallets (FK to production_orders, referenced by BOM items)
      await client.query(`
        DELETE FROM production.production_order_pallets
        WHERE production_order_id IN (
          SELECT id FROM production.production_orders WHERE source_order_number = $1
        )
      `, [plan.plan_number]);
      
      // 2. Delete BOMs
      await client.query(`
        DELETE FROM production.production_order_boms
        WHERE production_order_id IN (
          SELECT id FROM production.production_orders WHERE source_order_number = $1
        )
      `, [plan.plan_number]);
      
      // 3. Delete work orders
      await client.query(`
        DELETE FROM production.production_work_orders
        WHERE production_order_id IN (
          SELECT id FROM production.production_orders WHERE source_order_number = $1
        )
      `, [plan.plan_number]);
      
      // 4. Delete production orders
      await client.query(`
        DELETE FROM production.production_orders
        WHERE source_order_number = $1
      `, [plan.plan_number]);
      
      emit('cleanup', 9, 100, `Wyczyszczono istniejące ZLP, BOM`, 'success');
      
      // After deleting ZLPs, we need to reset line statuses
      shouldResetLines = true;
    } else {
      // No ZLPs exist - check if lines have non-pending status (manual deletion case)
      // Only reset if there are truly no ZLPs for this plan
      const nonPendingLinesResult = await client.query(`
        SELECT COUNT(*) as count FROM production.production_plan_lines
        WHERE plan_id = $1 AND status IN ('scheduled', 'in_progress', 'completed')
      `, [planId]);
      
      const nonPendingCount = parseInt(nonPendingLinesResult.rows[0].count);
      if (nonPendingCount > 0) {
        shouldResetLines = true;
      }
    }
    
    // Reset line statuses if needed (after ZLP deletion or when ZLPs were manually deleted)
    if (shouldResetLines) {
      const resetResult = await client.query(`
        UPDATE production.production_plan_lines
        SET status = 'pending'
        WHERE plan_id = $1 AND status IN ('scheduled', 'in_progress', 'completed')
        RETURNING id
      `, [planId]);
      
      if (resetResult.rowCount && resetResult.rowCount > 0) {
        emit('reset', 9, 100, `Zresetowano ${resetResult.rowCount} linii do statusu "pending"`, 'success');
      }
    }

    // 2. Pobierz wszystkie linie planu z produktami
    // UWAGA: Nie używamy LEFT JOIN z packed_products - rezerwacje sprawdzamy osobno w packed_product_items
    emit('lines', 10, 100, 'Pobieranie linii planu...', 'info');
    const linesResult = await client.query(`
      SELECT 
        ppl.id as line_id,
        ppl.product_id,
        ppl.quantity as line_quantity,
        ppl.reserved_quantity,
        ppl.source_reference,
        p.sku,
        p.name as product_name
      FROM production.production_plan_lines ppl
      JOIN catalog.products p ON p.id = ppl.product_id
      WHERE ppl.plan_id = $1
        AND ppl.status = 'pending'
        AND ppl.deleted_at IS NULL
      ORDER BY ppl.sequence, ppl.id
    `, [planId]);

    if (linesResult.rows.length === 0) {
      emit('complete', 100, 100, 'Brak oczekujących linii do przetworzenia', 'warning');
      return {
        success: true,
        generatedOrders: [],
        generatedDocuments: [],
        errors: ['No pending lines found in plan'],
        summary: {
          planId,
          planNumber: plan.plan_number,
          totalOrders: 0,
          totalComponents: 0,
          totalPackedProducts: 0,
          colorBreakdown: {},
        },
      };
    }

    emit('lines', 15, 100, `Znaleziono ${linesResult.rows.length} linii planu`, 'success');
    
    // 2.1. Oddziel linie z zarezerwowanymi produktami spakowanymi od linii do produkcji
    // WAŻNE: Nie polegamy na LEFT JOIN z packed_products - sprawdzamy faktyczne rezerwacje w packed_product_items
    const packedProductLines: PackedProductLineInfo[] = [];
    const productionLines: typeof linesResult.rows = [];
    
    for (const line of linesResult.rows) {
      // Sprawdź czy linia ma FAKTYCZNIE zarezerwowane produkty spakowane (w tabeli packed_product_items)
      // To jest źródło prawdy - nie polegamy na reserved_quantity z plan_lines ani na JOIN z packed_products
      // WAŻNE: Grupujemy według packed_product_id aby poprawnie obsłużyć przypadki gdy linia ma sztuki z różnych packed products
      const reservedItemsResult = await client.query(`
        SELECT 
          ppi.id, 
          ppi.serial_number,
          ppi.packed_product_id,
          pp.product_sku as packed_product_sku
        FROM warehouse.packed_product_items ppi
        JOIN warehouse.packed_products pp ON pp.id = ppi.packed_product_id
        WHERE ppi.reserved_for_plan_line_id = $1
          AND ppi.status = 'reserved'
        ORDER BY ppi.packed_product_id, ppi.packed_at ASC
      `, [line.line_id]);
      
      if (reservedItemsResult.rows.length > 0) {
        // Grupuj zarezerwowane sztuki według packed_product_id
        const groupedByPackedProduct = new Map<number, {
          packedProductSku: string;
          items: { id: number; serialNumber: string }[];
        }>();
        
        for (const item of reservedItemsResult.rows) {
          const ppId = item.packed_product_id;
          if (!groupedByPackedProduct.has(ppId)) {
            groupedByPackedProduct.set(ppId, {
              packedProductSku: item.packed_product_sku,
              items: [],
            });
          }
          groupedByPackedProduct.get(ppId)!.items.push({
            id: item.id,
            serialNumber: item.serial_number,
          });
        }
        
        // Utwórz wpisy dla każdej grupy packed products
        // W większości przypadków będzie tylko jeden packed_product per linia, ale obsługujemy edge case
        Array.from(groupedByPackedProduct.entries()).forEach(([ppId, group]) => {
          packedProductLines.push({
            planLineId: line.line_id,
            productId: line.product_id,
            productName: line.product_name,
            productSku: line.sku,
            quantity: parseInt(line.line_quantity),
            reservedQuantity: group.items.length,
            packedProductId: ppId,
            packedProductSku: group.packedProductSku,
            sourceReference: line.source_reference,
            reservedItemIds: group.items.map((i: { id: number; serialNumber: string }) => i.id),
            reservedSerialNumbers: group.items.map((i: { id: number; serialNumber: string }) => i.serialNumber),
          });
        });
        
        // Jeśli cała ilość jest zarezerwowana z magazynu, nie dodawaj do linii produkcyjnych
        if (reservedItemsResult.rows.length >= parseInt(line.line_quantity)) {
          emit('packed', 16, 100, `📦 Produkt spakowany: ${line.product_name} (${reservedItemsResult.rows.length}szt z magazynu)`, 'info');
          continue;
        }
      }
      
      // Linia do produkcji (BOM)
      productionLines.push(line);
    }
    
    emit('lines', 17, 100, `${packedProductLines.length} produktów spakowanych, ${productionLines.length} do produkcji`, 'success');
    
    // Track generated documents
    const generatedDocuments: GeneratedWarehouseDocument[] = [];

    // 2.2. Utwórz dokument WZ-SPAK dla produktów spakowanych (jeśli są)
    if (packedProductLines.length > 0) {
      emit('wzspak', 18, 100, 'Tworzenie dokumentu WZ-SPAK dla produktów spakowanych...', 'info');
      
      const wzSpakDoc = await createWarehouseDocumentForPackedProducts(
        client,
        planId,
        plan.plan_number,
        packedProductLines
      );
      
      generatedDocuments.push(wzSpakDoc);
      emit('wzspak', 20, 100, `Utworzono ${wzSpakDoc.docNumber} (${wzSpakDoc.totalQuantity} szt)`, 'success');
      
      // Oznacz linie z produktami spakowanymi jako scheduled
      // WAŻNE: Agregujemy sumę zarezerwowanych sztuk per planLineId (bo jeden planLineId może mieć wiele grup packed products)
      const reservedCountByPlanLine = new Map<number, { totalReserved: number; requiredQty: number }>();
      for (const p of packedProductLines) {
        const existing = reservedCountByPlanLine.get(p.planLineId) || { totalReserved: 0, requiredQty: p.quantity };
        existing.totalReserved += p.reservedQuantity;
        reservedCountByPlanLine.set(p.planLineId, existing);
      }
      
      const packedLineIds: number[] = [];
      Array.from(reservedCountByPlanLine.entries()).forEach(([planLineId, { totalReserved, requiredQty }]) => {
        if (totalReserved >= requiredQty) {
          packedLineIds.push(planLineId);
        }
      });
      
      if (packedLineIds.length > 0) {
        await client.query(`
          UPDATE production.production_plan_lines
          SET status = 'scheduled'
          WHERE id = ANY($1::int[])
        `, [packedLineIds]);
      }
    }
    
    // 2.3. Pobierz rezerwacje formatek z bufora produkcji dla tego planu i utwórz WZ-FORM
    const formatkaReservationsResult = await client.query(`
      SELECT 
        pbr.id as reservation_id,
        pbr.zlp_item_id as plan_line_id,
        pbr.product_sku,
        pbr.quantity_reserved,
        pbr.unit_of_measure,
        pbr.location_id,
        pl.name as location_name,
        pbr.notes,
        COALESCE(m.name, pbr.product_sku) as product_name
      FROM production.production_buffer_reservations pbr
      LEFT JOIN production.production_locations pl ON pl.id = pbr.location_id
      LEFT JOIN warehouse.materials m ON m.internal_code = pbr.product_sku
      WHERE pbr.zlp_item_id IN (
        SELECT ppl.id 
        FROM production.production_plan_lines ppl 
        WHERE ppl.plan_id = $1 AND ppl.deleted_at IS NULL
      )
      AND pbr.status = 'ACTIVE'
      ORDER BY pbr.id
    `, [planId]);
    
    if (formatkaReservationsResult.rows.length > 0) {
      emit('wzform', 21, 100, 'Tworzenie dokumentu WZ-FORM dla formatek z bufora...', 'info');
      
      const formatkaLines: FormatkaReservationLineInfo[] = formatkaReservationsResult.rows.map((row: any) => ({
        reservationId: row.reservation_id,
        planLineId: row.plan_line_id,
        productSku: row.product_sku,
        productName: row.product_name,
        quantityReserved: parseFloat(row.quantity_reserved),
        unitOfMeasure: row.unit_of_measure || 'szt',
        locationId: row.location_id,
        locationName: row.location_name,
        notes: row.notes,
      }));
      
      const wzFormDoc = await createWarehouseDocumentForFormatki(
        client,
        planId,
        plan.plan_number,
        formatkaLines
      );
      
      generatedDocuments.push(wzFormDoc);
      emit('wzform', 22, 100, `Utworzono ${wzFormDoc.docNumber} (${wzFormDoc.totalQuantity} szt)`, 'success');
    }
    
    // 3. Dla każdej linii produkcyjnej pobierz komponenty produktu (BOM)
    // WAŻNE: Pomijamy formatki które są zarezerwowane z bufora produkcyjnego
    
    // Funkcja normalizująca klucz - zamienia "×" (Unicode) na "x" (ASCII) i usuwa białe znaki
    const normalizeKey = (key: string): string => {
      return key.replace(/×/g, 'x').replace(/\s+/g, '').trim();
    };
    
    // Zbuduj mapę rezerwacji per planLineId -> product_sku (znormalizowany) -> ilość zarezerwowana
    // Rezerwacja ma product_sku np. "390x280-BIALY" lub "390×280-BIALY", musimy dopasować do generated_name
    const reservationsByLineId = new Map<number, Map<string, number>>();
    for (const res of formatkaReservationsResult.rows) {
      const lineId = res.plan_line_id;
      if (!reservationsByLineId.has(lineId)) {
        reservationsByLineId.set(lineId, new Map<string, number>());
      }
      const lineReservations = reservationsByLineId.get(lineId)!;
      // WAŻNE: Normalizuj klucz (zamień × na x) aby uniknąć problemów z różnymi znakami
      const normalizedSku = normalizeKey(res.product_sku);
      const currentQty = lineReservations.get(normalizedSku) || 0;
      lineReservations.set(normalizedSku, currentQty + parseFloat(res.quantity_reserved));
    }
    
    // Funkcja pomocnicza do ekstrakcji klucza z generated_name (wymiar-kolor)
    // np. "BOK-L-VB-390x280-BIALY" -> "390x280-BIALY"
    // np. "HDF-VB-796x337-HDF-BIALY" -> "796x337-HDF-BIALY"
    const extractDimensionColorKey = (generatedName: string, color: string): string => {
      // Szukamy wzorca wymiaru (liczba x/× liczba) w nazwie
      const dimensionMatch = generatedName.match(/(\d+)[x×](\d+)/);
      if (dimensionMatch) {
        // Zawsze używaj ASCII "x" w zwracanym kluczu (znormalizowane)
        const dimension = `${dimensionMatch[1]}x${dimensionMatch[2]}`;
        return `${dimension}-${color}`;
      }
      return `${generatedName}-${color}`;
    };
    
    const allComponents: ComponentForProduction[] = [];
    const totalProdLines = productionLines.length;
    let processedLines = 0;

    for (const line of productionLines) {
      processedLines++;
      const lineProgress = 20 + Math.floor((processedLines / Math.max(totalProdLines, 1)) * 25);
      emit('components', lineProgress, 100, `Przetwarzam produkt: ${line.product_name} (${processedLines}/${totalProdLines})`, 'info');
      // Najpierw pobierz BOM ID dla produktu
      const bomResult = await client.query(`
        SELECT id
        FROM bom.product_boms
        WHERE product_id = $1
        LIMIT 1
      `, [line.product_id]);

      if (bomResult.rows.length === 0) {
        errors.push(`No BOM found for product ${line.product_sku} (ID: ${line.product_id})`);
        continue;
      }

      const bomId = bomResult.rows[0].id;

      // Teraz pobierz komponenty z BOM
      const componentsResult = await client.query(`
        SELECT 
          pc.id as component_id,
          pc.generated_name,
          pc.component_type,
          pc.color,
          pc.calculated_length,
          pc.calculated_width,
          pc.thickness,
          pc.quantity,
          pc.edging_pattern,
          pc.board_type
        FROM bom.product_components pc
        WHERE pc.product_bom_id = $1
        ORDER BY pc.id
      `, [bomId]);

      // WAŻNE: Dla każdego produktu w linii planu tworzymy OSOBNE rekordy formatek
      // Zamiast mnożyć quantity, tworzymy indywidualne ComponentForProduction
      // Przykład: 20× VB-50, każdy ma 10 formatek → 200 osobnych rekordów w allComponents
      const lineQuantity = parseInt(line.line_quantity);
      
      // Pobierz mapę rezerwacji dla tej linii (używamy jej do pomijania zarezerwowanych formatek)
      const lineReservations = reservationsByLineId.get(line.line_id);
      // Kopia mapy aby śledzić zużyte rezerwacje w tej linii
      const usedReservations = new Map<string, number>();
      
      for (let productIndex = 0; productIndex < lineQuantity; productIndex++) {
        for (const comp of componentsResult.rows) {
          const componentQuantity = parseFloat(comp.quantity || '1');
          
          // Dla każdej formatki w BOM produktu tworzymy osobny rekord
          for (let compIndex = 0; compIndex < componentQuantity; compIndex++) {
            // Pobierz kolor BEZPOŚREDNIO z pola color w bazie
            // WAŻNE: HDF-BIALY i BIALY to RÓŻNE kolory!
            // Fallback: próbuj wyciągnąć kolor z nazwy generowanej (ostatnia część po myślniku)
            let color = comp.color;
            if (!color && comp.generated_name) {
              // Wyciągnij ostatnią część nazwy jako potencjalny kolor
              // np. "BOK-L-VB-390x280-WOTAN" → "WOTAN"
              // np. "PANEL-SCIENNY-500x300" → brak koloru (kończy się na wymiarach)
              const parts = comp.generated_name.split('-');
              const lastPart = parts[parts.length - 1];
              // Sprawdź czy ostatnia część nie jest wymiarem (nie zawiera x i nie jest liczbą)
              if (lastPart && !/^\d/.test(lastPart) && !lastPart.includes('x')) {
                color = lastPart;
              }
            }
            if (!color) color = 'BRAK-KOLORU';
            
            // Wyciągnij typ komponentu z nazwy (pierwsza część przed myślnikiem)
            // np. "BOK-L-VB-390x280-WOTAN" → "BOK"
            // np. "SIEDZISKO-VB-760x360-WOTAN" → "SIEDZISKO"
            const nameParts = comp.generated_name.split('-');
            const componentType = comp.component_type || (nameParts.length > 0 ? nameParts[0] : 'unknown');

            // Parse edging pattern (np. "TTTT" → edge1=T, edge2=T, edge3=T, edge4=T)
            const edgingPattern = comp.edging_pattern || '';
            const edge1 = edgingPattern.length > 0 && edgingPattern[0] !== 'F';
            const edge2 = edgingPattern.length > 1 && edgingPattern[1] !== 'F';
            const edge3 = edgingPattern.length > 2 && edgingPattern[2] !== 'F';
            const edge4 = edgingPattern.length > 3 && edgingPattern[3] !== 'F';

            // Określ typ materiału na podstawie grubości LUB nazwy komponentu
            const thickness = comp.thickness ? parseFloat(comp.thickness) : null;
            let materialType: string | null = comp.board_type || null;
            
            // 1. Spróbuj określić z grubości
            if (!materialType && thickness) {
              if (thickness === 18) materialType = 'plyta_18mm';
              else if (thickness === 25) materialType = 'plyta_25mm';
              else if (thickness === 3) materialType = 'hdf_3mm';
              else if (thickness === 8) materialType = 'hdf_8mm';
            }
            
            // 2. Fallback: określ z nazwy komponentu lub koloru
            if (!materialType && comp.generated_name) {
              const upperName = comp.generated_name.toUpperCase();
              // Nazwy zaczynające się od HDF lub zawierające HDF w nazwie
              if (upperName.startsWith('HDF') || upperName.includes('-HDF-')) {
                materialType = 'hdf_3mm';
              }
            }
            
            // 3. Fallback: sprawdź kolor (HDF-BIALY = hdf_3mm)
            if (!materialType && color) {
              if (color.toUpperCase().startsWith('HDF-')) {
                materialType = 'hdf_3mm';
              }
            }
            
            // 4. Domyślny fallback dla standardowych formatek - płyta 18mm
            if (!materialType) {
              // Wszystkie inne formatki są traktowane jako plyta_18mm
              materialType = 'plyta_18mm';
            }

            // SPRAWDŹ CZY FORMATKA JEST ZAREZERWOWANA Z BUFORA PRODUKCYJNEGO
            // Jeśli tak - pomijamy ją (nie dodajemy do produkcji, bo jest już w magazynie)
            if (lineReservations) {
              // WAŻNE: Normalizuj klucz wymiaru, bo rezerwacje też są znormalizowane
              // Obsługuje różne formaty: "×" vs "x", spacje w kolorach jak "DĄB ARTISAN" itp.
              const rawDimensionKey = extractDimensionColorKey(comp.generated_name, color);
              const dimensionKey = normalizeKey(rawDimensionKey);
              const reservedQty = lineReservations.get(dimensionKey) || 0;
              const usedQty = usedReservations.get(dimensionKey) || 0;
              
              if (usedQty < reservedQty) {
                // Ta formatka jest zarezerwowana z bufora - pomijamy ją
                usedReservations.set(dimensionKey, usedQty + 1);
                console.log(`📦 [ZLP] Pomijam formatkę zarezerwowaną z bufora: ${comp.generated_name} (${dimensionKey})`);
                continue; // Nie dodawaj do allComponents
              }
            }

            allComponents.push({
              componentId: comp.component_id,
              productId: line.product_id,
              planLineId: line.line_id,
              generatedName: comp.generated_name,
              componentType,
              color,
              materialType,
              length: comp.calculated_length ? parseFloat(comp.calculated_length) : null,
              width: comp.calculated_width ? parseFloat(comp.calculated_width) : null,
              thickness,
              quantity: 1, // ZAWSZE 1 - każda formatka to osobny rekord
              edge1,
              edge2,
              edge3,
              edge4,
              routingVariant: null, // Wypełnimy później
              planAssignment: null, // Wypełnimy później
              sourceFurnitureReference: line.source_reference,
            });
          }
        }
      }
    }

    emit('components', 40, 100, `Zebrano ${allComponents.length} komponentów do przetworzenia`, 'success');

    // 4. Resolve routing dla wszystkich komponentów (plan-level + global rules)
    emit('routing', 45, 100, 'Rozwiązywanie marszrut dla komponentów...', 'info');
    const totalComps = allComponents.length;
    let routedComps = 0;
    
    // Collect routing logs - we'll emit unique ones at the end
    const routingLogs: Array<{ message: string; level: string }> = [];
    const seenRoutingKeys = new Set<string>();
    
    // Create logger that collects routing decisions
    const routingLogger: RoutingLogCallback = (message, level) => {
      // Only collect 'match' and 'info' level logs to avoid flooding
      if (level === 'match' || level === 'info') {
        routingLogs.push({ message, level });
      }
    };
    
    for (const component of allComponents) {
      // Only enable detailed logging for first few components to show the decision process
      const enableDetailedLogging = routedComps < 3;
      
      const resolution = await resolveRoutingForPlan(
        planId,
        component.generatedName,
        component.color,
        component.materialType,
        client, // Przekazujemy istniejące połączenie zamiast tworzyć nowe
        enableDetailedLogging ? routingLogger : undefined,
        autoAssignRoutings // Przekazujemy flagę z planu
      );
      
      component.routingVariant = resolution.variant;
      component.planAssignment = resolution.planAssignment;
      
      // Track unique routing assignments for summary
      const routingKey = resolution.planAssignment 
        ? `plan:${resolution.planAssignment.routingCode}`
        : resolution.variant 
          ? `variant:${resolution.variant.variantCode}`
          : 'none';
      
      if (!seenRoutingKeys.has(routingKey) && routingKey !== 'none') {
        seenRoutingKeys.add(routingKey);
        const source = resolution.resolutionSource || 'unknown';
        const routingName = resolution.planAssignment?.routingName || resolution.variant?.variantName || 'brak';
        emit('routing_detail', 45, 100, `[+] Nowa marszruta: ${routingKey.split(':')[1]} (${source}) - ${routingName}`, 'info');
      }
      
      // Jeśli nie znaleziono ani variant ani plan assignment - to błąd
      if (!resolution.variant && !resolution.planAssignment) {
        errors.push(`No routing found for: ${component.generatedName} (${component.color}, ${component.materialType || 'unknown material'})`);
        emit('routing_error', 45, 100, `[!] Brak marszruty dla: ${component.generatedName} (${component.color})`, 'warning');
      }
      
      routedComps++;
      if (routedComps % 50 === 0 || routedComps === totalComps) {
        const routingProgress = 45 + Math.floor((routedComps / totalComps) * 15);
        emit('routing', routingProgress, 100, `Rozwiązano marszruty: ${routedComps}/${totalComps}`, 'info');
      }
    }
    
    // Emit collected routing logs (first few detailed decisions)
    for (const log of routingLogs.slice(0, 15)) {
      emit('routing_log', 58, 100, log.message, log.level === 'match' ? 'success' : 'info');
    }
    if (routingLogs.length > 15) {
      emit('routing_log', 59, 100, `... i ${routingLogs.length - 15} więcej wpisów`, 'info');
    }

    emit('routing', 60, 100, `Marszruty rozwiązane (${seenRoutingKeys.size} unikalnych)`, 'success');

    // 5. Agreguj komponenty TYLKO po kolorze (palety per przepływ wewnątrz koloru)
    emit('aggregate', 62, 100, 'Grupowanie komponentów według koloru...', 'info');
    const colorGroups = aggregateComponentsByColor(allComponents);
    
    // Oblicz łączną liczbę palet (przepływów)
    const totalPalletCount = colorGroups.reduce((sum, cg) => sum + cg.flows.length, 0);
    emit('aggregate', 63, 100, `Utworzono ${colorGroups.length} grup kolorów (${totalPalletCount} palet)`, 'success');
    
    // 6. Generuj Production Orders dla każdego koloru (ZLP per kolor, palety per przepływ)
    // Nośniki są automatycznie przypisywane z wolnych lub tworzone nowe podczas tworzenia palet
    emit('orders', 68, 100, 'Tworzenie zleceń produkcyjnych (ZLP) z automatycznym przypisaniem nośników...', 'info');
    const generatedOrders: GeneratedProductionOrder[] = [];
    const totalGroups = colorGroups.length;
    let processedGroups = 0;
    
    // Zbiór do śledzenia używanych nośników w tej sesji generowania
    const usedCarrierIds = new Set<number>();

    for (const colorGroup of colorGroups) {
      const orderResult = await createProductionOrderForColor(
        client,
        planId,
        plan.plan_number,
        colorGroup.color,
        colorGroup.flows,
        usedCarrierIds
      );

      generatedOrders.push(orderResult);
      processedGroups++;
      const orderProgress = 68 + Math.floor((processedGroups / totalGroups) * 25);
      const flowInfo = colorGroup.flows.map(f => f.flowCode).join(', ');
      emit('orders', orderProgress, 100, `Utworzono ZLP: ${orderResult.orderNumber} [${flowInfo}] (${colorGroup.totalComponentCount} komponentów, ${colorGroup.flows.length} palet)`, 'success');
    }

    // 7. Oznacz linie planu jako scheduled (jeśli wszystko poszło dobrze)
    if (errors.length === 0) {
      emit('finalize', 95, 100, 'Finalizowanie zmian w bazie danych...', 'info');
      await client.query(`
        UPDATE production.production_plan_lines
        SET status = 'scheduled'
        WHERE plan_id = $1 AND status = 'pending'
      `, [planId]);

      await client.query(`
        UPDATE production.production_plans
        SET status = 'generated'
        WHERE id = $1 AND status IN ('draft', 'approved', 'in_progress')
      `, [planId]);
      
      await client.query('COMMIT');
      emit('done', 100, 100, 'Generowanie ZLP zakończone pomyślnie!', 'success');
    } else {
      // Jeśli są błędy, wycofaj wszystkie zmiany
      emit('rollback', 95, 100, `Wycofuję zmiany - znaleziono ${errors.length} błędów`, 'warning');
      await client.query('ROLLBACK');
      emit('done', 100, 100, `Generowanie zakończone z ${errors.length} błędami`, 'warning');
      
      // WAŻNE: Po rollback zwracamy puste listy - dokumenty i zlecenia zostały wycofane!
      return {
        success: false,
        generatedOrders: [],
        generatedDocuments: [],
        errors,
        summary: {
          planId,
          planNumber: plan.plan_number,
          totalOrders: 0,
          totalComponents: 0,
          totalPackedProducts: 0,
          colorBreakdown: {},
        },
      };
    }

    // Oblicz podsumowanie (tylko gdy sukces)
    const colorBreakdown: Record<string, number> = {};
    let totalComponents = 0;
    
    for (const order of generatedOrders) {
      colorBreakdown[order.colorCode] = order.componentCount;
      totalComponents += order.componentCount;
    }
    
    const totalPackedProducts = packedProductLines.reduce((sum, p) => sum + p.reservedQuantity, 0);

    return {
      success: true,
      generatedOrders,
      generatedDocuments,
      errors: [],
      summary: {
        planId,
        planNumber: plan.plan_number,
        totalOrders: generatedOrders.length,
        totalComponents,
        totalPackedProducts,
        colorBreakdown,
      },
    };

  } catch (error: any) {
    emit('error', 0, 100, `Błąd krytyczny: ${error.message}`, 'error');
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}


/**
 * Tworzy dokument magazynowy WZ-SPAK dla zarezerwowanych produktów spakowanych.
 * Dokument jest tworzony w statusie 'draft' i zawiera pozycje dla każdego produktu spakowanego.
 */
async function createWarehouseDocumentForPackedProducts(
  client: any,
  planId: number,
  planNumber: string,
  packedProductLines: PackedProductLineInfo[]
): Promise<GeneratedWarehouseDocument> {
  const year = new Date().getFullYear();
  
  // Generuj numer dokumentu WZ-SPAK (używamy MAX zamiast COUNT żeby uniknąć konfliktów)
  const maxNumResult = await client.query(`
    SELECT COALESCE(
      MAX(
        CAST(
          NULLIF(SPLIT_PART(doc_number, '/', 3), '') AS INTEGER
        )
      ), 0
    ) + 1 as next_num 
    FROM warehouse.documents 
    WHERE doc_type = 'WZ-SPAK' 
      AND doc_number LIKE $1
  `, [`WZ-SPAK/${year}/%`]);
  
  const nextNum = maxNumResult.rows[0].next_num;
  const docNumber = `WZ-SPAK/${year}/${String(nextNum).padStart(4, '0')}`;
  
  // Oblicz sumy
  const totalQuantity = packedProductLines.reduce((sum, p) => sum + p.reservedQuantity, 0);
  const totalLines = packedProductLines.length;
  
  // Utwórz nagłówek dokumentu
  const docResult = await client.query(`
    INSERT INTO warehouse.documents (
      doc_number, doc_type, status, plan_id,
      remarks, total_lines, total_quantity,
      created_at, updated_at
    ) VALUES ($1, 'WZ-SPAK', 'draft', $2, $3, $4, $5, NOW(), NOW())
    RETURNING id
  `, [
    docNumber,
    planId,
    `Auto-wygenerowano z planu ${planNumber} - produkty spakowane z magazynu`,
    totalLines,
    totalQuantity,
  ]);
  
  const docId = docResult.rows[0].id;
  
  // Utwórz pozycje dokumentu
  for (let i = 0; i < packedProductLines.length; i++) {
    const line = packedProductLines[i];
    
    await client.query(`
      INSERT INTO warehouse.document_lines (
        document_id, line_number, source_type, source_id, catalog_product_id,
        sku, product_name, quantity_requested, unit,
        serial_numbers, plan_line_id, notes,
        created_at, updated_at
      ) VALUES ($1, $2, 'packed_product', $3, $4, $5, $6, $7, 'szt', $8::jsonb, $9, $10, NOW(), NOW())
    `, [
      docId,
      i + 1,
      line.packedProductId,
      line.productId,
      line.packedProductSku,
      line.productName,
      line.reservedQuantity,
      JSON.stringify(line.reservedSerialNumbers),
      line.planLineId,
      line.sourceReference ? `Zamówienie: ${line.sourceReference}` : null,
    ]);
  }
  
  console.log(`📋 [WZ-SPAK] Utworzono dokument ${docNumber} z ${totalLines} pozycjami (${totalQuantity} szt)`);
  
  return {
    docId,
    docNumber,
    docType: 'WZ-SPAK',
    totalLines,
    totalQuantity,
  };
}


/**
 * Tworzy dokument magazynowy WZ-FORM dla zarezerwowanych formatek z bufora produkcji.
 * Dokument jest tworzony w statusie 'draft' i zawiera pozycje dla każdej formatki.
 */
async function createWarehouseDocumentForFormatki(
  client: any,
  planId: number,
  planNumber: string,
  formatkaLines: FormatkaReservationLineInfo[]
): Promise<GeneratedWarehouseDocument> {
  const year = new Date().getFullYear();
  
  // Generuj numer dokumentu WZ-FORM (używamy MAX zamiast COUNT żeby uniknąć konfliktów)
  const maxNumResult = await client.query(`
    SELECT COALESCE(
      MAX(
        CAST(
          NULLIF(SPLIT_PART(doc_number, '/', 3), '') AS INTEGER
        )
      ), 0
    ) + 1 as next_num 
    FROM warehouse.documents 
    WHERE doc_type = 'WZ-FORM' 
      AND doc_number LIKE $1
  `, [`WZ-FORM/${year}/%`]);
  
  const nextNum = maxNumResult.rows[0].next_num;
  const docNumber = `WZ-FORM/${year}/${String(nextNum).padStart(4, '0')}`;
  
  // Oblicz sumy
  const totalQuantity = formatkaLines.reduce((sum, f) => sum + f.quantityReserved, 0);
  const totalLines = formatkaLines.length;
  
  // Utwórz nagłówek dokumentu
  const docResult = await client.query(`
    INSERT INTO warehouse.documents (
      doc_number, doc_type, status, plan_id,
      remarks, total_lines, total_quantity,
      created_at, updated_at
    ) VALUES ($1, 'WZ-FORM', 'draft', $2, $3, $4, $5, NOW(), NOW())
    RETURNING id
  `, [
    docNumber,
    planId,
    `Auto-wygenerowano z planu ${planNumber} - formatki z bufora produkcji`,
    totalLines,
    totalQuantity,
  ]);
  
  const docId = docResult.rows[0].id;
  
  // Utwórz pozycje dokumentu
  for (let i = 0; i < formatkaLines.length; i++) {
    const line = formatkaLines[i];
    
    await client.query(`
      INSERT INTO warehouse.document_lines (
        document_id, line_number, source_type, source_id,
        sku, product_name, quantity_requested, unit,
        plan_line_id, notes,
        created_at, updated_at
      ) VALUES ($1, $2, 'formatka', $3, $4, $5, $6, $7, $8, $9, NOW(), NOW())
    `, [
      docId,
      i + 1,
      line.reservationId,
      line.productSku,
      line.productName,
      line.quantityReserved,
      line.unitOfMeasure,
      line.planLineId,
      line.notes || `Rezerwacja z bufora: ${line.locationName || 'brak lokalizacji'}`,
    ]);
  }
  
  console.log(`📋 [WZ-FORM] Utworzono dokument ${docNumber} z ${totalLines} pozycjami (${totalQuantity} szt)`);
  
  return {
    docId,
    docNumber,
    docType: 'WZ-FORM',
    totalLines,
    totalQuantity,
  };
}


/**
 * Agreguje komponenty TYLKO według koloru.
 * Wewnątrz każdej grupy kolorów tworzone są podgrupy per ŚCIEŻKĘ PRZEPŁYWU.
 * 
 * NOWA LOGIKA (agregacja po zbieżności ścieżek):
 * - ZLP = grupowanie po kolorze (np. ZLP-0040-WOTAN)
 * - Palety = grupowanie po ŚCIEŻCE PRZEPŁYWU wewnątrz koloru:
 *   - OKLEJANIE: CO + COW + BOCZKI-TYL (wspólna ścieżka: cutting→edging)
 *   - WIERCENIE: CW (cutting→drilling, bez oklejania)
 *   - CIECIE: C (tylko cutting)
 * - Po cięciu palety się rozdzielają na podstawie następnej operacji
 * - Po oklejaniu CO (gotowe) się oddziela od COW (kontynuuje na wiercenie)
 */
function aggregateComponentsByColor(
  components: ComponentForProduction[]
): ColorAggregation[] {
  // Import funkcji z pallet-flow-aggregator
  const { getPostCuttingGroup, getPostEdgingGroup } = require('./pallet-flow-aggregator');
  
  // Klucz główny: tylko COLOR (np. "WOTAN")
  const colorGroups = new Map<string, {
    color: string;
    flowsMap: Map<string, {
      flowCode: string;
      routingId: number | null;
      routingVariantCode: string | null;
      splitAfterOperation: string | null;
      components: ComponentForProduction[];
    }>;
  }>();

  for (const comp of components) {
    const color = comp.color || 'BRAK-KOLORU';
    
    // Pobierz kod wariantu/marszruty i routing_id
    let variantCode: string | null = null;
    let routingCode: string | null = null;
    let routingId: number | null = null;
    
    // Priorytet: planAssignment > routingVariant
    if (comp.planAssignment) {
      routingCode = comp.planAssignment.routingCode;
      routingId = comp.planAssignment.routingId;
    } else if (comp.routingVariant) {
      variantCode = comp.routingVariant.variantCode;
      routingId = comp.routingVariant.routingIds?.[0] || null;
    }
    
    // Określ grupę przepływu na podstawie analizy ścieżki operacji
    const codeForPath = variantCode || routingCode || 'COW';
    const postCuttingGroup = getPostCuttingGroup(codeForPath);
    const postEdgingGroup = getPostEdgingGroup(codeForPath);
    
    // Generuj flowCode na podstawie grupy ścieżki (nie wariantu!)
    // To powoduje że CO i COW trafiają na tę samą paletę (do oklejania)
    let flowCode: string;
    let splitAfterOperation: string | null;
    
    if (postCuttingGroup === 'TO_EDGING') {
      // Formatki idące na oklejanie (CO, COW, BOCZKI-TYL)
      // Dzielimy dalej po oklejaniu
      if (postEdgingGroup === 'DONE') {
        flowCode = 'OKLEJANIE-GOTOWE'; // CO - gotowe po oklejaniu
        splitAfterOperation = 'edging';
      } else {
        flowCode = 'OKLEJANIE-WIERCENIE'; // COW, BOCZKI-TYL - kontynuują na wiercenie
        splitAfterOperation = 'edging';
      }
    } else if (postCuttingGroup === 'TO_DRILLING') {
      // Formatki idące bezpośrednio na wiercenie (CW - np. DNO)
      flowCode = 'WIERCENIE';
      splitAfterOperation = 'cutting';
    } else {
      // Formatki gotowe po cięciu (C - np. HDF)
      flowCode = 'CIECIE-GOTOWE';
      splitAfterOperation = 'cutting';
    }
    
    // Zachowaj oryginalny wariant dla referencji
    const routingVariantCode = variantCode || (routingCode ? `PLAN-${routingCode}` : null);
    
    // Utwórz grupę kolorów jeśli nie istnieje
    if (!colorGroups.has(color)) {
      colorGroups.set(color, {
        color,
        flowsMap: new Map(),
      });
    }
    
    const colorGroup = colorGroups.get(color)!;
    
    // Utwórz podgrupę przepływu jeśli nie istnieje
    if (!colorGroup.flowsMap.has(flowCode)) {
      colorGroup.flowsMap.set(flowCode, {
        flowCode,
        routingId,
        routingVariantCode,
        splitAfterOperation,
        components: [],
      });
    }
    
    colorGroup.flowsMap.get(flowCode)!.components.push(comp);
  }

  // Konwertuj do tablicy ColorAggregation
  return Array.from(colorGroups.values()).map(colorGroup => ({
    color: colorGroup.color,
    flows: Array.from(colorGroup.flowsMap.values()).map(flow => ({
      flowCode: flow.flowCode,
      routingId: flow.routingId,
      routingVariantCode: flow.routingVariantCode,
      carrierId: null, // Przypisywane później
      splitAfterOperation: flow.splitAfterOperation,
      components: flow.components,
    })),
    totalComponentCount: Array.from(colorGroup.flowsMap.values())
      .reduce((sum, flow) => sum + flow.components.length, 0),
  }));
}

/**
 * @deprecated Use aggregateComponentsByColor instead
 * Agreguje komponenty według koloru ORAZ wariantu routingu.
 * Dzięki temu formatki z różnymi ścieżkami produkcyjnymi (np. CO vs COW) 
 * trafiają na osobne palety/ZLP.
 */
function aggregateComponentsByColorAndVariant(
  components: ComponentForProduction[]
): ComponentAggregation[] {
  // Klucz: "COLOR|VARIANT_CODE" np. "WOTAN|COW" lub "WOTAN|CO"
  const groups = new Map<string, {
    color: string;
    variantCode: string | null;
    routingId: number | null;
    components: ComponentForProduction[];
  }>();

  for (const comp of components) {
    const color = comp.color || 'BRAK-KOLORU';
    
    // Pobierz kod wariantu i routing_id
    let variantCode: string | null = null;
    let routingId: number | null = null;
    
    // Priorytet: planAssignment > routingVariant
    if (comp.planAssignment) {
      variantCode = `PLAN-${comp.planAssignment.routingCode}`;
      routingId = comp.planAssignment.routingId;
    } else if (comp.routingVariant) {
      variantCode = comp.routingVariant.variantCode;
      routingId = comp.routingVariant.routingIds?.[0] || null;
    }
    
    // Klucz grupowania: kolor + wariant
    const groupKey = `${color}|${variantCode || 'BRAK'}`;
    
    if (!groups.has(groupKey)) {
      groups.set(groupKey, {
        color,
        variantCode,
        routingId,
        components: [],
      });
    }
    
    groups.get(groupKey)!.components.push(comp);
  }

  // Konwertuj do tablicy ComponentAggregation
  // Carrier zostanie przypisany później w głównej funkcji
  return Array.from(groups.values()).map(group => ({
    color: group.color,
    routingVariantCode: group.variantCode,
    routingId: group.routingId,
    carrierId: null, // Przypisywane później
    components: group.components,
  }));
}

/**
 * Tworzy Production Order dla danego koloru z wieloma przepływami.
 * 
 * Nowa logika:
 * - Jeden ZLP per kolor (np. ZLP-0040-WOTAN)
 * - Wiele palet per przepływ wewnątrz ZLP (np. WOTAN-CO, WOTAN-COW)
 * - Palety idą razem przez cięcie i oklejanie
 * - Po oklejaniu palety się rozdzielają na podstawie swojego przepływu
 * - Nośniki (palety fizyczne) są automatycznie przypisywane z wolnych lub tworzone nowe
 */
async function createProductionOrderForColor(
  client: any,
  planId: number,
  planNumber: string,
  colorCode: string,
  flows: FlowGroup[],
  usedCarrierIds: Set<number>
): Promise<GeneratedProductionOrder> {
  
  // 1. Generuj numer zlecenia ZLP tylko z kolorem (bez wariantu)
  const cleanPlanNumber = planNumber.replace(/^PLAN-/i, '');
  // Format: ZLP-0040-WOTAN (bez sufiksu wariantu)
  const orderNumber = `ZLP-${cleanPlanNumber}-${colorCode}`;
  
  // 2. Zbierz wszystkie komponenty ze wszystkich przepływów
  const allComponents: ComponentForProduction[] = [];
  for (const flow of flows) {
    allComponents.push(...flow.components);
  }
  
  // 3. Użyj routing_id z pierwszego przepływu (dla wspólnych operacji jak cięcie/oklejanie)
  // W nowej logice ZLP nie ma jednego routingu - każda paleta ma swój
  const primaryRoutingId = flows[0]?.routingId || null;
  
  // Pobierz location_id i planAssignmentId z pierwszego komponentu
  let locationId: number | null = null;
  let planAssignmentId: number | null = null;
  
  for (const comp of allComponents) {
    if (comp.planAssignment) {
      locationId = comp.planAssignment.locationId;
      planAssignmentId = comp.planAssignment.id;
      break;
    }
  }
  
  // Loguj info o utworzeniu ZLP
  const flowCodes = flows.map(f => f.flowCode).join(', ');
  console.log(`[ZLP Generator] Creating ${orderNumber}: flows=[${flowCodes}], ${allComponents.length} components, ${flows.length} pallets`);

  // 4. Utwórz Production Order (carrier_id = null - teraz jest na poziomie palety)
  const orderResult = await client.query(`
    INSERT INTO production.production_orders (
      order_number,
      product_id,
      routing_id,
      location_id,
      carrier_id,
      routing_variant_code,
      status,
      priority,
      quantity_planned,
      unit_of_measure,
      color_code,
      notes,
      source_order_number
    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
    RETURNING id
  `, [
    orderNumber,
    allComponents[0].productId,
    primaryRoutingId,  // Primary routing (wspólne operacje)
    locationId,
    null,  // carrier_id teraz jest na poziomie palety, nie ZLP
    null,  // routing_variant_code również jest na poziomie palety
    'draft',
    'normal',
    allComponents.length,
    'szt',
    colorCode,
    `Generated from Production Plan: ${planNumber} | Flows: ${flowCodes}`,
    planNumber,
  ]);

  const productionOrderId = orderResult.rows[0].id;

  // 5. Utwórz Production Order BOM
  const bomResult = await client.query(`
    INSERT INTO production.production_order_boms (
      production_order_id,
      source_plan_id,
      color_code,
      status
    ) VALUES ($1, $2, $3, $4)
    RETURNING id
  `, [productionOrderId, planId, colorCode, 'active']);

  const bomId = bomResult.rows[0].id;
  
  // 6. Utwórz palety dla każdego przepływu z automatycznym przypisaniem nośnika
  const palletIdByFlowCode = new Map<string, number>();
  let palletSequence = 1;
  
  for (const flow of flows) {
    const palletLabel = `${colorCode}-${flow.flowCode}-${palletSequence}`;
    
    // Automatycznie przypisz wolny nośnik lub utwórz nowy
    const carrierId = await getOrCreateAvailableCarrier(client, usedCarrierIds);
    usedCarrierIds.add(carrierId);
    
    // Określ splitAfterOperation z flow lub domyślnie 'edging'
    const splitAfterOp = flow.splitAfterOperation || 'edging';
    
    const palletResult = await client.query(`
      INSERT INTO production.production_order_pallets (
        production_order_id,
        pallet_label,
        sequence,
        flow_code,
        routing_id,
        routing_variant_code,
        carrier_id,
        status,
        split_after_operation,
        component_count,
        notes
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
      RETURNING id
    `, [
      productionOrderId,
      palletLabel,
      palletSequence,
      flow.flowCode,
      flow.routingId,
      flow.routingVariantCode,
      carrierId,
      'pending',
      splitAfterOp,  // Operacja po której paleta się rozdziela (cutting lub edging)
      flow.components.length,
      `Auto-generated pallet for flow ${flow.flowCode} (split after ${splitAfterOp})`,
    ]);
    
    palletIdByFlowCode.set(flow.flowCode, palletResult.rows[0].id);
    palletSequence++;
    
    console.log(`[ZLP Generator] Created pallet ${palletLabel} (ID: ${palletResult.rows[0].id}, carrier: ${carrierId}) for flow ${flow.flowCode}`);
  }

  // 7. Utwórz BOM Items dla wszystkich komponentów z przypisaniem do palet
  const bomItemIds: number[] = [];
  
  for (const flow of flows) {
    const palletId = palletIdByFlowCode.get(flow.flowCode);
    
    for (const comp of flow.components) {
      // Pobierz operacje z routingVariant LUB z planAssignment
      const operations = comp.routingVariant?.defaultOperations || comp.planAssignment?.defaultOperations || [];
      
      const itemResult = await client.query(`
        INSERT INTO production.production_order_bom_items (
          production_order_bom_id,
          pallet_id,
          component_name,
          component_type,
          quantity,
          unit_of_measure,
          routing_variant_id,
          required_operations,
          color_code,
          length,
          width,
          thickness,
          source_plan_line_id,
          source_product_id,
          source_furniture_reference,
          item_status,
          quantity_ordered
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17)
        RETURNING id
      `, [
        bomId,
        palletId,  // Link do palety
        comp.generatedName,
        comp.componentType,
        comp.quantity,
        'szt',
        comp.routingVariant?.id || null,
        operations.length > 0 ? JSON.stringify(operations) : null,
        comp.color,
        comp.length,
        comp.width,
        comp.thickness,
        comp.planLineId,
        comp.productId,
        comp.sourceFurnitureReference,
        'pending',
        comp.quantity,
      ]);

      bomItemIds.push(itemResult.rows[0].id);
    }
  }

  // 8. Generuj Work Orders dla WSZYSTKICH unikalnych routingów (nie tylko pierwszego!)
  // Każdy przepływ może mieć inny routing (np. POLKA -> CO, BOK -> COW, DNO -> CW)
  // Musimy wygenerować work orders dla każdego routingu
  
  // Zbierz unikalne routing IDs z wszystkich przepływów
  const uniqueRoutingIds = new Set<number>();
  for (const flow of flows) {
    if (flow.routingId) {
      uniqueRoutingIds.add(flow.routingId);
    }
  }
  
  const allWorkOrderIds: number[] = [];
  
  // Generuj work orders dla każdego unikalnego routingu
  // Używamy globalnego licznika sekwencji aby uniknąć duplikatów (production_order_id, sequence)
  let globalSequenceOffset = 0;
  
  for (const routingId of Array.from(uniqueRoutingIds)) {
    // Znajdź komponenty przypisane do tego routingu
    const componentsForRouting = flows
      .filter(f => f.routingId === routingId)
      .flatMap(f => f.components);
    
    // Znajdź planAssignmentId dla tego routingu (z pierwszego komponentu)
    let routingPlanAssignmentId: number | null = null;
    for (const comp of componentsForRouting) {
      if (comp.planAssignment) {
        routingPlanAssignmentId = comp.planAssignment.id;
        break;
      }
    }
    
    console.log(`[ZLP Generator] Generating work orders for routing ${routingId} (${componentsForRouting.length} components), sequence offset: ${globalSequenceOffset}`);
    
    const { workOrderIds, lastSequence } = await generateWorkOrdersForBom(
      client,
      productionOrderId,
      bomId,
      componentsForRouting,
      routingId,
      routingPlanAssignmentId,
      globalSequenceOffset
    );
    
    allWorkOrderIds.push(...workOrderIds);
    globalSequenceOffset = lastSequence;
  }
  
  // Fallback: jeśli żaden flow nie ma routingId, użyj primaryRoutingId
  if (uniqueRoutingIds.size === 0 && primaryRoutingId) {
    console.log(`[ZLP Generator] No routing in flows, using primary routing ${primaryRoutingId}`);
    const { workOrderIds } = await generateWorkOrdersForBom(
      client,
      productionOrderId,
      bomId,
      allComponents,
      primaryRoutingId,
      planAssignmentId,
      globalSequenceOffset
    );
    allWorkOrderIds.push(...workOrderIds);
  }

  return {
    orderNumber,
    colorCode,
    componentCount: allComponents.length,
    totalQuantity: allComponents.reduce((sum, c) => sum + c.quantity, 0),
    bomId,
    bomItemIds,
    workOrderIds: allWorkOrderIds,
  };
}

/**
 * @deprecated Use createProductionOrderForColor instead
 * Tworzy Production Order dla danej grupy kolorów i wariantów routingu.
 * Każda grupa (kolor + wariant) = osobne ZLP na osobnej palecie.
 */
async function createProductionOrderForColorAndVariant(
  client: any,
  planId: number,
  planNumber: string,
  colorCode: string,
  routingVariantCode: string | null,
  preResolvedRoutingId: number | null,
  carrierId: number | null,
  components: ComponentForProduction[]
): Promise<GeneratedProductionOrder> {
  
  // 1. Generuj numer zlecenia ZLP z kolorem i wariantem
  const cleanPlanNumber = planNumber.replace(/^PLAN-/i, '');
  // Format: ZLP-0040-WOTAN-COW lub ZLP-0040-WOTAN (gdy brak wariantu)
  const variantSuffix = routingVariantCode ? `-${routingVariantCode}` : '';
  const orderNumber = `ZLP-${cleanPlanNumber}-${colorCode}${variantSuffix}`;

  // 2. Użyj routing_id przekazanego z agregacji (już rozwiązanego)
  const routingId = preResolvedRoutingId;
  
  // Pobierz location_id i planAssignmentId z pierwszego komponentu (jeśli jest)
  let locationId: number | null = null;
  let planAssignmentId: number | null = null;
  
  for (const comp of components) {
    if (comp.planAssignment) {
      locationId = comp.planAssignment.locationId;
      planAssignmentId = comp.planAssignment.id;
      break;
    }
  }
  
  // Loguj info o utworzeniu ZLP
  console.log(`[ZLP Generator] Creating ${orderNumber}: routing=${routingId}, carrier=${carrierId}, variant=${routingVariantCode}`);

  // 3. Utwórz Production Order z carrier_id i routing_variant_code
  const orderResult = await client.query(`
    INSERT INTO production.production_orders (
      order_number,
      product_id,
      routing_id,
      location_id,
      carrier_id,
      routing_variant_code,
      status,
      priority,
      quantity_planned,
      unit_of_measure,
      color_code,
      notes,
      source_order_number
    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
    RETURNING id
  `, [
    orderNumber,
    components[0].productId,
    routingId,
    locationId,
    carrierId,
    routingVariantCode,
    'draft',
    'normal',
    components.length,
    'szt',
    colorCode,
    `Generated from Production Plan: ${planNumber}`,
    planNumber,
  ]);

  const productionOrderId = orderResult.rows[0].id;

  // 3. Utwórz Production Order BOM
  const bomResult = await client.query(`
    INSERT INTO production.production_order_boms (
      production_order_id,
      source_plan_id,
      color_code,
      status
    ) VALUES ($1, $2, $3, $4)
    RETURNING id
  `, [productionOrderId, planId, colorCode, 'active']);

  const bomId = bomResult.rows[0].id;

  // 4. Utwórz BOM Items dla wszystkich komponentów
  const bomItemIds: number[] = [];

  for (const comp of components) {
    // Pobierz operacje z routingVariant LUB z planAssignment
    const operations = comp.routingVariant?.defaultOperations || comp.planAssignment?.defaultOperations || [];
    
    const itemResult = await client.query(`
      INSERT INTO production.production_order_bom_items (
        production_order_bom_id,
        component_name,
        component_type,
        quantity,
        unit_of_measure,
        routing_variant_id,
        required_operations,
        color_code,
        length,
        width,
        thickness,
        source_plan_line_id,
        source_product_id,
        source_furniture_reference,
        item_status,
        quantity_ordered
      ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16)
      RETURNING id
    `, [
      bomId,
      comp.generatedName,
      comp.componentType,
      comp.quantity,
      'szt',
      comp.routingVariant?.id || null,
      operations.length > 0 ? JSON.stringify(operations) : null,
      comp.color,
      comp.length,
      comp.width,
      comp.thickness,
      comp.planLineId,
      comp.productId,
      comp.sourceFurnitureReference,
      'pending',
      comp.quantity,
    ]);

    bomItemIds.push(itemResult.rows[0].id);
  }

  // 5. Generuj Work Orders na podstawie unique operations (z uwzględnieniem nadpisań)
  const { workOrderIds } = await generateWorkOrdersForBom(
    client,
    productionOrderId,
    bomId,
    components,
    routingId,
    planAssignmentId,
    0
  );

  return {
    orderNumber,
    colorCode,
    componentCount: components.length,
    totalQuantity: components.reduce((sum, c) => sum + c.quantity, 0),
    bomId,
    bomItemIds,
    workOrderIds,
  };
}

/**
 * Generuje Work Orders dla Production Order na podstawie required_operations z BOM items.
 * Uwzględnia nadpisania operacji z plan_routing_operation_overrides.
 */
async function generateWorkOrdersForBom(
  client: any,
  productionOrderId: number,
  bomId: number,
  components: ComponentForProduction[],
  routingId: number | null,
  planAssignmentId: number | null = null,
  sequenceOffset: number = 0
): Promise<{ workOrderIds: number[], lastSequence: number }> {
  
  // Jeśli mamy routing_id, pobierz operacje bezpośrednio z bazy z ich work_center_id i buforami
  if (routingId) {
    // Pobierz operacje marszruty wraz z domyślnymi buforami
    const opsResult = await client.query(`
      SELECT id, sequence, code, name, work_center_id, buffer_before_id, buffer_after_id, is_active
      FROM production.production_routing_operations
      WHERE routing_id = $1 AND is_active = true
      ORDER BY sequence
    `, [routingId]) as { rows: { 
      id: number; 
      sequence: number; 
      code: string; 
      name: string; 
      work_center_id: number | null; 
      buffer_before_id: number | null;
      buffer_after_id: number | null;
      is_active: boolean; 
    }[] };
    
    // Pobierz nadpisania operacji jeśli mamy przypisanie planu
    let overridesMap: Map<number, { 
      workCenterId: number | null; 
      skipOperation: boolean;
      bufferBeforeId: number | null;
      bufferAfterId: number | null;
    }> = new Map();
    
    if (planAssignmentId) {
      const overridesResult = await client.query(`
        SELECT 
          routing_operation_id,
          override_work_center_id,
          override_buffer_before_id,
          override_buffer_after_id,
          skip_operation
        FROM production.plan_routing_operation_overrides
        WHERE plan_routing_assignment_id = $1
      `, [planAssignmentId]) as { rows: { 
        routing_operation_id: number; 
        override_work_center_id: number | null; 
        override_buffer_before_id: number | null;
        override_buffer_after_id: number | null;
        skip_operation: boolean; 
      }[] };
      
      for (const override of overridesResult.rows) {
        overridesMap.set(override.routing_operation_id, {
          workCenterId: override.override_work_center_id,
          skipOperation: override.skip_operation,
          bufferBeforeId: override.override_buffer_before_id,
          bufferAfterId: override.override_buffer_after_id,
        });
      }
    }
    
    const workOrderIds: number[] = [];
    
    // Pobierz operatorów przypisanych do gniazd roboczych (kolejka)
    const operatorsByWorkCenter = new Map<number, { id: number; fullName: string }[]>();
    const operatorsResult = await client.query(`
      SELECT 
        owc.work_center_id,
        po.id,
        po.full_name
      FROM production.operator_work_centers owc
      JOIN production.production_operators po ON owc.operator_id = po.id
      WHERE po.is_active = true
      ORDER BY owc.is_primary DESC, po.full_name ASC
    `);
    
    for (const row of operatorsResult.rows) {
      if (!operatorsByWorkCenter.has(row.work_center_id)) {
        operatorsByWorkCenter.set(row.work_center_id, []);
      }
      operatorsByWorkCenter.get(row.work_center_id)!.push({
        id: row.id,
        fullName: row.full_name,
      });
    }
    
    for (const op of opsResult.rows) {
      // Sprawdź nadpisanie dla tej operacji
      const override = overridesMap.get(op.id);
      
      // Pomiń operację jeśli ustawiono skip_operation
      if (override?.skipOperation) {
        continue;
      }
      
      // Użyj nadpisanego work_center_id lub domyślnego
      const workCenterId = override?.workCenterId !== undefined && override.workCenterId !== null 
        ? override.workCenterId 
        : op.work_center_id;
      
      // Użyj nadpisanych buforów lub domyślnych z operacji
      const bufferBeforeId = override?.bufferBeforeId !== undefined 
        ? override.bufferBeforeId 
        : op.buffer_before_id;
      const bufferAfterId = override?.bufferAfterId !== undefined 
        ? override.bufferAfterId 
        : op.buffer_after_id;
      
      // Pobierz domyślnego operatora z kolejki dla gniazda roboczego (round-robin)
      let operatorId: number | null = null;
      let operatorName: string | null = null;
      
      if (workCenterId) {
        const operators = operatorsByWorkCenter.get(workCenterId);
        if (operators && operators.length > 0) {
          // Pobierz pierwszego operatora z kolejki
          operatorId = operators[0].id;
          operatorName = operators[0].fullName;
          
          // Rotuj kolejkę - przesuń pierwszego operatora na koniec (round-robin)
          if (operators.length > 1) {
            const firstOperator = operators.shift()!;
            operators.push(firstOperator);
          }
        }
      }
      
      const adjustedSequence = sequenceOffset + op.sequence;
      const workOrderNumber = `WO-${productionOrderId}-R${routingId}-${adjustedSequence}`;
      
      const result = await client.query(`
        INSERT INTO production.production_work_orders (
          work_order_number,
          production_order_id,
          routing_operation_id,
          work_center_id,
          buffer_before_id,
          buffer_after_id,
          sequence,
          status,
          quantity_planned,
          operator_id,
          operator_name
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
        RETURNING id
      `, [
        workOrderNumber,
        productionOrderId,
        op.id,
        workCenterId,
        bufferBeforeId,
        bufferAfterId,
        adjustedSequence,
        'pending',
        components.length,
        operatorId,
        operatorName,
      ]);

      const workOrderId = result.rows[0].id;
      workOrderIds.push(workOrderId);
      
      // Dodaj rekord do tabeli work_order_operators (relacja wiele-do-wielu)
      if (operatorId) {
        await client.query(`
          INSERT INTO production.work_order_operators (work_order_id, operator_id, is_primary)
          VALUES ($1, $2, true)
        `, [workOrderId, operatorId]);
      }
    }

    // Oblicz ostatnią użytą sekwencję
    const maxOpSequence = opsResult.rows.length > 0 
      ? Math.max(...opsResult.rows.map(op => op.sequence))
      : 0;
    
    return { workOrderIds, lastSequence: sequenceOffset + maxOpSequence };
  }
  
  // Fallback: Zbierz wszystkie unikalne operacje z nazw (z routingVariant LUB planAssignment)
  const allOperations = new Set<string>();
  
  for (const comp of components) {
    const operations = comp.routingVariant?.defaultOperations || comp.planAssignment?.defaultOperations || [];
    for (const op of operations) {
      allOperations.add(op);
    }
  }

  // Sekwencja operacji (preferowane kolejność)
  const operationSequence = [
    'cutting',
    'edging',
    'drilling_holes',
    'upholstering',
    'drilling_mount',
    'drilling',
    'assembly',
    'packing',
  ];

  // Posortuj operacje według preferowanej sekwencji
  const sortedOperations = Array.from(allOperations).sort((a, b) => {
    const indexA = operationSequence.indexOf(a);
    const indexB = operationSequence.indexOf(b);
    
    if (indexA === -1 && indexB === -1) return 0;
    if (indexA === -1) return 1;
    if (indexB === -1) return -1;
    
    return indexA - indexB;
  });

  // Generuj Work Order dla każdej operacji
  // UWAGA: W fallback path operacje są tylko nazwami bez work_center_id,
  // więc nie można automatycznie przypisać operatora - musi być przypisany później w flow-tree
  const workOrderIds: number[] = [];
  let sequence = sequenceOffset + 1;

  for (const operation of sortedOperations) {
    const workOrderNumber = `WO-${productionOrderId}-${sequence}`;
    
    const result = await client.query(`
      INSERT INTO production.production_work_orders (
        work_order_number,
        production_order_id,
        sequence,
        status,
        quantity_planned
      ) VALUES ($1, $2, $3, $4, $5)
      RETURNING id
    `, [
      workOrderNumber,
      productionOrderId,
      sequence,
      'pending',
      components.length,
    ]);

    workOrderIds.push(result.rows[0].id);
    sequence++;
  }

  return { workOrderIds, lastSequence: sequence - 1 };
}

/**
 * Pobiera kolejny numer sekwencji dla ZLP (format ZLP-0001 do ZLP-9999).
 */
async function getNextZlpSequence(client: any): Promise<number> {
  const result = await client.query(`
    SELECT COALESCE(MAX(
      CAST(
        SUBSTRING(order_number FROM 'ZLP-([0-9]+)') AS INTEGER
      )
    ), 0) + 1 as next_seq
    FROM production.production_orders
    WHERE order_number LIKE 'ZLP-%'
  `);

  return result.rows[0].next_seq;
}
