import { Pool } from "pg";
import PDFDocument from "pdfkit";
import QRCode from "qrcode";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { isFormatka } from "@shared/schema";
import * as ordersService from "./orders";
import { execSync } from "child_process";
import path from "path";
import fs from "fs";

function getDejaVuFontPath(): string {
  try {
    const fontPath = execSync('fc-match "DejaVu Sans" -f "%{file}"', { encoding: 'utf-8' }).trim();
    if (fontPath && fs.existsSync(fontPath)) {
      return fontPath;
    }
  } catch (e) {
    console.warn('[PDF Export] Could not find DejaVu Sans font via fc-match');
  }
  return '';
}

function getDejaVuBoldFontPath(): string {
  try {
    const fontPath = execSync('fc-match "DejaVu Sans:bold" -f "%{file}"', { encoding: 'utf-8' }).trim();
    if (fontPath && fs.existsSync(fontPath)) {
      return fontPath;
    }
  } catch (e) {
    console.warn('[PDF Export] Could not find DejaVu Sans Bold font via fc-match');
  }
  return '';
}

function formatDatePL(date: Date | null | undefined): string {
  if (!date) return '-';
  return format(new Date(date), "dd.MM.yyyy HH:mm", { locale: pl });
}

function formatDateShortPL(date: Date | null | undefined): string {
  if (!date) return '-';
  return format(new Date(date), "dd.MM.yyyy", { locale: pl });
}

const statusLabels: Record<string, string> = {
  draft: "Projekt",
  confirmed: "Potwierdzone",
  planned: "Zaplanowane",
  in_progress: "W realizacji",
  paused: "Wstrzymane",
  done: "Zakończone",
  cancelled: "Anulowane",
  pending: "Oczekuje",
  completed: "Zakończone",
};

const priorityLabels: Record<string, string> = {
  low: "Niski",
  normal: "Normalny",
  high: "Wysoki",
  urgent: "Pilny",
};

export async function generateProductionCardPDF(pool: Pool, orderId: number): Promise<Buffer | null> {
  const order = await ordersService.getOrderById(pool, orderId);
  if (!order) {
    console.error(`[PDF Export] Order ${orderId} not found`);
    return null;
  }

  const workOrders = order.workOrders || [];
  const allComponents = order.components || [];
  
  const formatki = allComponents.filter((comp: any) => isFormatka(comp.generatedName));

  return new Promise(async (resolve, reject) => {
    try {
      const chunks: Buffer[] = [];
      const doc = new PDFDocument({
        size: 'A4',
        margin: 40,
        bufferPages: true,
      });

      const dejaVuPath = getDejaVuFontPath();
      const dejaVuBoldPath = getDejaVuBoldFontPath();
      
      if (dejaVuPath) {
        doc.registerFont('DejaVu', dejaVuPath);
      }
      if (dejaVuBoldPath) {
        doc.registerFont('DejaVu-Bold', dejaVuBoldPath);
      }
      
      const fontRegular = dejaVuPath ? 'DejaVu' : 'Helvetica';
      const fontBold = dejaVuBoldPath ? 'DejaVu-Bold' : 'Helvetica-Bold';

      doc.on('data', (chunk: Buffer) => chunks.push(chunk));
      doc.on('end', () => resolve(Buffer.concat(chunks)));
      doc.on('error', (err: Error) => {
        console.error(`[PDF Export] PDF generation error for order ${orderId}:`, err);
        reject(err);
      });

      const displayOrderNumber = (order.orderNumber && order.orderNumber !== 'undefined' && order.orderNumber !== 'null') 
        ? order.orderNumber 
        : `ZLP-${orderId}`;
      
      let qrBuffer: Buffer;
      try {
        const qrDataUrl = await QRCode.toDataURL(displayOrderNumber, {
          width: 100,
          margin: 1,
          errorCorrectionLevel: 'M',
        });
        qrBuffer = Buffer.from(qrDataUrl.split(',')[1], 'base64');
      } catch (qrErr) {
        console.error(`[PDF Export] QR code generation failed for order ${orderId}:`, qrErr);
        throw qrErr;
      }

      doc.font(fontBold).fontSize(24).text(displayOrderNumber, 50, 40, { continued: false });
      
      doc.image(qrBuffer, 460, 30, { width: 80, height: 80 });

      const colorText = order.colorCode ? `Kolor: ${order.colorCode}` : '';
      if (colorText) {
        doc.font(fontRegular).fontSize(14).text(colorText, 50, 70);
      }

      doc.fontSize(10).fillColor('#666666');
      doc.text(`Status: ${statusLabels[order.status] || order.status}`, 50, 90);
      doc.text(`Priorytet: ${priorityLabels[order.priority] || order.priority}`, 200, 90);
      doc.text(`Ilość: ${order.quantityPlanned}`, 350, 90);

      doc.moveTo(40, 115).lineTo(555, 115).stroke('#cccccc');

      let yPos = 130;
      doc.fillColor('#000000').font(fontBold).fontSize(12).text('DATY I TERMINY', 50, yPos);
      yPos += 20;
      
      doc.font(fontRegular).fontSize(10).fillColor('#333333');
      doc.text(`Utworzono: ${formatDatePL(order.createdAt)}`, 50, yPos);
      doc.text(`Plan start: ${formatDateShortPL(order.plannedStartDate)}`, 250, yPos);
      yPos += 15;
      doc.text(`Plan koniec: ${formatDateShortPL(order.plannedEndDate)}`, 250, yPos);
      if (order.actualStartDate) {
        doc.text(`Rzeczywisty start: ${formatDatePL(order.actualStartDate)}`, 50, yPos);
      }
      yPos += 15;
      if (order.actualEndDate) {
        doc.text(`Rzeczywisty koniec: ${formatDatePL(order.actualEndDate)}`, 50, yPos);
        yPos += 15;
      }

      doc.moveTo(40, yPos + 5).lineTo(555, yPos + 5).stroke('#cccccc');
      yPos += 20;

      if (workOrders.length > 0) {
        doc.fillColor('#000000').font(fontBold).fontSize(12).text('ŚCIEŻKA PRODUKCJI (ROUTING)', 50, yPos);
        yPos += 25;

        if (order.routingName) {
          doc.font(fontRegular).fontSize(10).fillColor('#666666').text(`Routing: ${order.routingName}`, 50, yPos);
          yPos += 20;
        }

        const tableTop = yPos;
        const col1 = 50;
        const col2 = 80;
        const col3 = 200;
        const col4 = 350;
        const col5 = 450;

        doc.font(fontBold).fontSize(9).fillColor('#000000');
        doc.text('Lp.', col1, tableTop);
        doc.text('Operacja', col2, tableTop);
        doc.text('Stanowisko', col3, tableTop);
        doc.text('Status', col4, tableTop);
        doc.text('Termin', col5, tableTop);

        doc.moveTo(40, tableTop + 15).lineTo(555, tableTop + 15).stroke('#dddddd');

        yPos = tableTop + 20;
        doc.font(fontRegular).fontSize(9).fillColor('#333333');

        for (const wo of workOrders) {
          if (yPos > 750) {
            doc.addPage();
            yPos = 50;
          }

          doc.text(`${wo.sequence}.`, col1, yPos);
          doc.text(wo.operationName || wo.operationCode || '-', col2, yPos, { width: 110 });
          doc.text(wo.workCenterName || '-', col3, yPos, { width: 140 });
          doc.text(statusLabels[wo.status] || wo.status, col4, yPos);
          doc.text(formatDateShortPL(wo.scheduledStartTime), col5, yPos);
          yPos += 18;
        }

        doc.moveTo(40, yPos + 5).lineTo(555, yPos + 5).stroke('#cccccc');
        yPos += 20;
      }

      if (formatki.length > 0) {
        if (yPos > 650) {
          doc.addPage();
          yPos = 50;
        }

        const aggregatedFormatki: Map<string, { name: string; dims: string; thick: string; color: string; qty: number }> = new Map();
        
        for (const item of formatki) {
          const length = item.calculatedLength ? parseFloat(item.calculatedLength) : null;
          const width = item.calculatedWidth ? parseFloat(item.calculatedWidth) : null;
          const thick = item.thickness ? parseFloat(item.thickness) : null;
          const dims = length && width ? `${Math.round(length)}x${Math.round(width)}` : '-';
          const thickStr = thick ? `${Math.round(thick)}` : '-';
          const color = item.color || '-';
          const name = item.generatedName || '-';
          const qty = item.quantity ? parseFloat(item.quantity) : 1;
          
          const key = `${name}|${dims}|${thickStr}|${color}`;
          
          if (aggregatedFormatki.has(key)) {
            const existing = aggregatedFormatki.get(key)!;
            existing.qty += qty;
          } else {
            aggregatedFormatki.set(key, { name, dims, thick: thickStr, color, qty });
          }
        }
        
        const aggregatedList = Array.from(aggregatedFormatki.values());
        const totalQty = aggregatedList.reduce((sum, f) => sum + f.qty, 0);

        doc.fillColor('#000000').font(fontBold).fontSize(12).text('FORMATKI DO PRODUKCJI', 50, yPos);
        yPos += 18;
        doc.font(fontRegular).fontSize(9).fillColor('#666666').text(`Łączna liczba unikalnych: ${aggregatedList.length} (szt. łącznie: ${Math.round(totalQty)})`, 50, yPos);
        yPos += 18;

        const tableTop = yPos;
        const fcol1 = 50;
        const fcol2 = 80;
        const fcol3 = 300;
        const fcol4 = 380;
        const fcol5 = 450;
        const fcol6 = 510;

        doc.font(fontBold).fontSize(8).fillColor('#000000');
        doc.text('Lp.', fcol1, tableTop);
        doc.text('Nazwa formatki', fcol2, tableTop);
        doc.text('Wymiary', fcol3, tableTop);
        doc.text('Grub.', fcol4, tableTop);
        doc.text('Kolor', fcol5, tableTop);
        doc.text('Ilość', fcol6, tableTop);

        doc.moveTo(40, tableTop + 12).lineTo(555, tableTop + 12).stroke('#dddddd');

        yPos = tableTop + 16;
        doc.font(fontRegular).fontSize(8).fillColor('#333333');

        aggregatedList.forEach((item, index) => {
          if (yPos > 780) {
            doc.addPage();
            yPos = 50;
            doc.font(fontBold).fontSize(8).fillColor('#000000');
            doc.text('Lp.', fcol1, yPos);
            doc.text('Nazwa formatki', fcol2, yPos);
            doc.text('Wymiary', fcol3, yPos);
            doc.text('Grub.', fcol4, yPos);
            doc.text('Kolor', fcol5, yPos);
            doc.text('Ilość', fcol6, yPos);
            doc.moveTo(40, yPos + 12).lineTo(555, yPos + 12).stroke('#dddddd');
            yPos += 16;
            doc.font(fontRegular).fontSize(8).fillColor('#333333');
          }

          doc.text(`${index + 1}.`, fcol1, yPos);
          doc.text(item.name, fcol2, yPos, { width: 210 });
          doc.text(item.dims, fcol3, yPos);
          doc.text(item.thick, fcol4, yPos);
          doc.text(item.color, fcol5, yPos, { width: 55 });
          doc.text(`${Math.round(item.qty)}`, fcol6, yPos);
          yPos += 14;
        });
      }

      if (order.notes) {
        if (yPos > 720) {
          doc.addPage();
          yPos = 50;
        }
        yPos += 10;
        doc.moveTo(40, yPos).lineTo(555, yPos).stroke('#cccccc');
        yPos += 15;
        doc.font(fontBold).fontSize(10).fillColor('#000000').text('UWAGI:', 50, yPos);
        yPos += 15;
        doc.font(fontRegular).fontSize(9).fillColor('#333333').text(order.notes, 50, yPos, { width: 500 });
      }

      const logoPath = path.join(process.cwd(), 'server/assets/alpma-logo.png');
      const generatedAt = formatDatePL(new Date());
      const productionDate = order.plannedStartDate ? formatDateShortPL(order.plannedStartDate) : '-';
      const totalPages = doc.bufferedPageRange().count;
      
      for (let i = 0; i < totalPages; i++) {
        doc.switchToPage(i);
        
        doc.moveTo(40, 800).lineTo(555, 800).stroke('#dddddd');
        
        if (fs.existsSync(logoPath)) {
          try {
            doc.image(logoPath, 40, 805, { width: 15, height: 15 });
          } catch (e) {
          }
        }
        
        doc.font(fontRegular).fontSize(7).fillColor('#999999');
        doc.text(displayOrderNumber, 60, 808, { continued: false, lineBreak: false });
        doc.text(`Produkcja: ${productionDate}`, 160, 808, { continued: false, lineBreak: false });
        doc.text(`Wygenerowano: ${generatedAt}`, 280, 808, { continued: false, lineBreak: false });
        doc.text(`Strona ${i + 1} z ${totalPages}`, 480, 808, { continued: false, lineBreak: false });
      }

      doc.end();
    } catch (err) {
      console.error(`[PDF Export] Unexpected error for order ${orderId}:`, err);
      reject(err);
    }
  });
}

export async function generateSawCSV(pool: Pool, orderId: number): Promise<string | null> {
  const order = await ordersService.getOrderById(pool, orderId);
  if (!order) {
    console.error(`[CSV Export] Order ${orderId} not found`);
    return null;
  }

  const bomItems = await ordersService.getOrderBomItems(pool, orderId);
  if (!bomItems || bomItems.length === 0) {
    console.warn(`[CSV Export] No BOM items found for order ${orderId}`);
    return '';
  }

  const formatki = bomItems.filter((item: any) => isFormatka(item.componentName));
  
  if (formatki.length === 0) {
    console.warn(`[CSV Export] No formatki found in order ${orderId}`);
    return '';
  }

  const productIds = formatki
    .map((f: any) => f.sourceProductId)
    .filter((id: number | null): id is number => id !== null);
  
  const productEans: Map<number, string> = new Map();
  
  if (productIds.length > 0) {
    try {
      const uniqueIds = Array.from(new Set(productIds));
      const eansResult = await pool.query(`
        SELECT id, ean 
        FROM catalog.products 
        WHERE id = ANY($1)
      `, [uniqueIds]);
      
      for (const row of eansResult.rows) {
        if (row.ean) {
          productEans.set(row.id, row.ean);
        }
      }
    } catch (err) {
      console.error(`[CSV Export] Failed to fetch product EANs:`, err);
    }
  }

  const lines: string[] = [];
  const orderColorCode = order.colorCode || '';
  const zlpNumber = order.orderNumber?.startsWith('ZLP-') 
    ? order.orderNumber 
    : `ZLP-${order.orderNumber || orderId}`;

  for (const item of formatki) {
    const colorCode = item.colorCode || orderColorCode || 'BRAK';
    const thickness = item.thickness ? Math.round(item.thickness) : 18;
    const normalizedColor = colorCode.replace(/\s+/g, '_').toUpperCase();
    const materialCode = `${thickness}_${normalizedColor}`;
    
    const length = item.length ? Math.round(item.length) : 0;
    const width = item.width ? Math.round(item.width) : 0;
    const quantity = Math.round(item.quantity || 1);
    
    const componentName = item.componentName || '';
    
    let baseName = componentName;
    if (normalizedColor && componentName.toUpperCase().endsWith(`-${normalizedColor}`)) {
      baseName = componentName.substring(0, componentName.length - normalizedColor.length - 1);
    } else if (colorCode && componentName.toUpperCase().endsWith(`-${colorCode.toUpperCase()}`)) {
      baseName = componentName.substring(0, componentName.length - colorCode.length - 1);
    }
    
    const itemId = `rec${item.id}`;
    const productId = item.sourceProductId || 0;
    const ean = item.sourceProductId ? (productEans.get(item.sourceProductId) || '') : '';

    const row = [
      materialCode,
      '0',
      length.toString(),
      width.toString(),
      quantity.toString(),
      componentName,
      zlpNumber,
      '',
      '',
      '',
      '',
      baseName,
      itemId,
      '',
      productId.toString(),
      ean,
      '',
      productId.toString()
    ];

    lines.push(row.join(','));
  }

  console.log(`[CSV Export] Generated CSV with ${lines.length} formatki for order ${orderId}`);
  return lines.join('\n');
}
