import { pool } from '../postgres';

interface RoutingVariant {
  id: number;
  variantCode: string;
  variantName: string;
  description: string;
  defaultOperations: string[];
  isActive: boolean;
}

async function seedRoutingVariants() {
  console.log("🌱 Seedowanie routing variants...");

  const client = await pool.connect();

  try {
    await client.query('BEGIN');

    const variants = [
      {
        variantCode: "CUT_EDGE_DRILL",
        variantName: "Cięcie + Oklejanie + Wiercenie",
        description: "Pełna obróbka: formatowanie, oklejanie krawędzi, wiercenie otworów (WD, WG, POLKA z otworami)",
        defaultOperations: JSON.stringify(["cutting", "edging", "drilling"]),
      },
      {
        variantCode: "CUT_EDGE",
        variantName: "Cięcie + Oklejanie",
        description: "Formatowanie i oklejanie bez wiercenia (POLKA, DNO bez otworów)",
        defaultOperations: JSON.stringify(["cutting", "edging"]),
      },
      {
        variantCode: "CUT_DRILL",
        variantName: "Cięcie + Wiercenie",
        description: "Formatowanie i wiercenie bez oklejania (HDF tył z otworami)",
        defaultOperations: JSON.stringify(["cutting", "drilling"]),
      },
      {
        variantCode: "CUT_ONLY",
        variantName: "Tylko cięcie",
        description: "Tylko formatowanie bez dalszej obróbki (HDF, elementy wewnętrzne)",
        defaultOperations: JSON.stringify(["cutting"]),
      },
      {
        variantCode: "CUT_DRILL_UPHOLSTER",
        variantName: "Cięcie + Wiercenie + Tapicernia",
        description: "Formatowanie, wiercenie przed tapicernią, tapicerowanie (SIEDZISKO SUROWA), wiercenie po tapicerni",
        defaultOperations: JSON.stringify(["cutting", "drilling_holes", "upholstering", "drilling_mount"]),
      },
    ];

    const insertedVariants: RoutingVariant[] = [];

    for (const variant of variants) {
      const result = await client.query(`
        INSERT INTO production.production_routing_variants 
          (variant_code, variant_name, description, default_operations, is_active)
        VALUES ($1, $2, $3, $4::jsonb, $5)
        ON CONFLICT (variant_code) 
        DO UPDATE SET
          variant_name = EXCLUDED.variant_name,
          description = EXCLUDED.description,
          default_operations = EXCLUDED.default_operations,
          is_active = EXCLUDED.is_active,
          updated_at = NOW()
        RETURNING id, variant_code, variant_name, description, default_operations, is_active
      `, [
        variant.variantCode,
        variant.variantName,
        variant.description,
        variant.defaultOperations,
        true,
      ]);

      const row = result.rows[0];
      insertedVariants.push({
        id: row.id,
        variantCode: row.variant_code,
        variantName: row.variant_name,
        description: row.description,
        defaultOperations: row.default_operations,
        isActive: row.is_active,
      });
    }

    console.log(`✅ Dodano ${insertedVariants.length} routing variants`);

    const variantMap: Record<string, number> = {};
    for (const v of insertedVariants) {
      variantMap[v.variantCode] = v.id;
    }

    console.log("\n🔗 Seedowanie routing variant rules...");

    await client.query('DELETE FROM production.production_routing_variant_rules');

    const rules = [
      // CUT_EDGE_DRILL - Elementy z pełną obróbką
      { variantCode: "CUT_EDGE_DRILL", namePattern: "WD-%", colorPattern: null, priority: 100 },
      { variantCode: "CUT_EDGE_DRILL", namePattern: "WG-%", colorPattern: null, priority: 100 },
      { variantCode: "CUT_EDGE_DRILL", namePattern: "POLKA-%", colorPattern: null, priority: 90 },

      // CUT_EDGE - Elementy z oklejaniem bez wiercenia
      { variantCode: "CUT_EDGE", namePattern: "DNO-%", colorPattern: null, priority: 80 },
      { variantCode: "CUT_EDGE", namePattern: "PLECY-%", colorPattern: null, priority: 80 },

      // CUT_DRILL - Elementy z wierceniem bez oklejania
      { variantCode: "CUT_DRILL", namePattern: "HDF TYŁ%", colorPattern: null, priority: 70 },

      // CUT_ONLY - Elementy tylko do cięcia
      { variantCode: "CUT_ONLY", namePattern: "HDF%", colorPattern: null, priority: 60 },
      { variantCode: "CUT_ONLY", namePattern: "FREZ%", colorPattern: null, priority: 60 },

      // CUT_DRILL_UPHOLSTER - Tapicernia (SIEDZISKO SUROWA)
      { variantCode: "CUT_DRILL_UPHOLSTER", namePattern: "SIEDZISKO%", colorPattern: "SUROWA", priority: 200 },
      { variantCode: "CUT_DRILL_UPHOLSTER", namePattern: "%SUROWA%", colorPattern: null, priority: 150 },
    ];

    let insertedRulesCount = 0;

    for (const rule of rules) {
      const variantId = variantMap[rule.variantCode];
      if (!variantId) {
        console.error(`⚠️  Nie znaleziono variantu: ${rule.variantCode}`);
        continue;
      }

      await client.query(`
        INSERT INTO production.production_routing_variant_rules
          (routing_variant_id, name_pattern, color_pattern, priority, is_active)
        VALUES ($1, $2, $3, $4, $5)
      `, [
        variantId,
        rule.namePattern,
        rule.colorPattern,
        rule.priority,
        true,
      ]);

      insertedRulesCount++;
    }

    console.log(`✅ Dodano ${insertedRulesCount} routing variant rules`);

    await client.query('COMMIT');

    console.log("\n📊 Podsumowanie:");
    console.log(`- ${insertedVariants.length} routing variants`);
    console.log(`- ${insertedRulesCount} routing variant rules`);

    console.log("\n🎯 Routing Variants:");
    for (const variant of insertedVariants) {
      const ruleCount = rules.filter((r) => r.variantCode === variant.variantCode).length;
      console.log(`  - ${variant.variantCode}: ${ruleCount} reguł`);
    }

  } catch (error) {
    await client.query('ROLLBACK');
    throw error;
  } finally {
    client.release();
  }
}

seedRoutingVariants()
  .then(() => {
    console.log("\n✅ Seed zakończony pomyślnie!");
    process.exit(0);
  })
  .catch((error) => {
    console.error("❌ Błąd podczas seedowania:", error);
    process.exit(1);
  });
