import { Express } from 'express';
import { Pool } from 'pg';
import { z } from 'zod';
import { generateWarehouseDocumentPDF } from '../../services/warehouse/document-exports';

const baseDocumentSchema = z.object({
  docType: z.enum(['WZ-SPAK', 'WZ-TAP', 'WZ-FORM', 'WZ-OKUC', 'WZ-OPAK', 'WZ-PROD', 'PZ-PROD', 'MM', 'WZ-HDF', 'WZ-SUR']),
  planId: z.number().optional().nullable(),
  workOrderId: z.number().optional().nullable(),
  operationId: z.number().optional().nullable(),
  sourceLocationId: z.number().optional().nullable(),
  sourceLocationName: z.string().optional().nullable(),
  targetDepartment: z.string().optional().nullable(),
  targetLocationId: z.number().optional().nullable(),
  remarks: z.string().optional().nullable(),
  lines: z.array(z.object({
    sourceType: z.enum(['packed_product', 'formatka', 'okucia', 'opakowania', 'tapicernia', 'hdf', 'surowiec', 'polprodukt', 'custom']),
    sourceId: z.number().optional().nullable(),
    catalogProductId: z.number().optional().nullable(),
    sku: z.string().optional().nullable(),
    productName: z.string(),
    quantityRequested: z.number(),
    unit: z.string().optional().default('szt'),
    warehouseLocationId: z.number().optional().nullable(),
    warehouseLocationName: z.string().optional().nullable(),
    planLineId: z.number().optional().nullable(),
    notes: z.string().optional().nullable(),
  })).min(1, "Dokument musi zawierać co najmniej jedną pozycję"),
});

const createDocumentSchema = baseDocumentSchema.refine(
  (data) => {
    if (data.docType === 'MM') {
      return data.sourceLocationId != null && data.sourceLocationName != null && data.sourceLocationName.trim() !== '';
    }
    return true;
  },
  {
    message: "Dokument MM wymaga lokalizacji źródłowej (sourceLocationId i sourceLocationName)",
    path: ["sourceLocationId"],
  }
);

const updateDocumentSchema = z.object({
  sourceLocationId: z.number().optional().nullable(),
  sourceLocationName: z.string().optional().nullable(),
  targetDepartment: z.string().optional().nullable(),
  targetLocationId: z.number().optional().nullable(),
  remarks: z.string().optional().nullable(),
});

const DOCUMENT_TYPE_NAMES: Record<string, string> = {
  'WZ-SPAK': 'Produkty spakowane',
  'WZ-TAP': 'Tapicernia',
  'WZ-FORM': 'Formatki',
  'WZ-OKUC': 'Okucia',
  'WZ-OPAK': 'Opakowania',
  'WZ-PROD': 'Wydanie na produkcję',
  'PZ-PROD': 'Przyjęcie z produkcji',
  'MM': 'Przesunięcie międzymagazynowe',
  'WZ-HDF': 'Płyty HDF',
  'WZ-SUR': 'Surowce',
};

const STATUS_NAMES: Record<string, string> = {
  'draft': 'Szkic',
  'confirmed': 'Potwierdzony',
  'printed': 'Wydrukowany',
  'completed': 'Zrealizowany',
  'cancelled': 'Anulowany',
};

export function registerWarehouseDocumentsRoutes(app: Express, pool: Pool, isAuthenticated: any) {

  async function generateDocNumber(pool: Pool, docType: string): Promise<string> {
    const year = new Date().getFullYear();
    const result = await pool.query(`
      SELECT COUNT(*) + 1 as next_num 
      FROM warehouse.documents 
      WHERE doc_type = $1 
        AND EXTRACT(YEAR FROM created_at) = $2
    `, [docType, year]);
    
    const nextNum = result.rows[0].next_num;
    return `${docType}/${year}/${String(nextNum).padStart(4, '0')}`;
  }

  app.get("/api/warehouse/documents", isAuthenticated, async (req, res) => {
    try {
      const { 
        docType, 
        status,
        statusTab,
        planId,
        search,
        limit = '50', 
        offset = '0' 
      } = req.query;

      let query = `
        SELECT 
          d.id,
          d.doc_number as "docNumber",
          d.doc_type as "docType",
          d.status,
          d.plan_id as "planId",
          d.work_order_id as "workOrderId",
          d.operation_id as "operationId",
          d.source_location_id as "sourceLocationId",
          d.source_location_name as "sourceLocationName",
          d.target_department as "targetDepartment",
          d.target_location_id as "targetLocationId",
          d.issued_at as "issuedAt",
          d.issued_by as "issuedBy",
          d.issued_by_name as "issuedByName",
          d.confirmed_at as "confirmedAt",
          d.confirmed_by as "confirmedBy",
          d.confirmed_by_name as "confirmedByName",
          d.printed_at as "printedAt",
          d.printed_by as "printedBy",
          d.pdf_path as "pdfPath",
          d.completed_at as "completedAt",
          d.completed_by as "completedBy",
          d.completed_by_name as "completedByName",
          d.remarks,
          d.total_lines as "totalLines",
          d.total_quantity as "totalQuantity",
          d.created_at as "createdAt",
          d.updated_at as "updatedAt",
          pp.name as "planName",
          pl.name as "targetLocationName",
          sl.name as "sourceLocationDisplayName"
        FROM warehouse.documents d
        LEFT JOIN production.production_plans pp ON pp.id = d.plan_id
        LEFT JOIN production.production_locations pl ON pl.id = d.target_location_id
        LEFT JOIN production.production_locations sl ON sl.id = d.source_location_id
        WHERE 1=1
      `;

      const params: any[] = [];
      let paramIndex = 1;

      if (docType) {
        query += ` AND d.doc_type = $${paramIndex++}`;
        params.push(docType);
      }

      if (status) {
        query += ` AND d.status = $${paramIndex++}`;
        params.push(status);
      }

      if (statusTab) {
        if (statusTab === 'active') {
          query += ` AND d.status IN ('draft', 'confirmed', 'printed')`;
        } else if (statusTab === 'completed') {
          query += ` AND d.status = 'completed'`;
        } else if (statusTab === 'cancelled') {
          query += ` AND d.status = 'cancelled'`;
        }
      }

      if (planId) {
        query += ` AND d.plan_id = $${paramIndex++}`;
        params.push(parseInt(planId as string));
      }

      if (search) {
        query += ` AND (d.doc_number ILIKE $${paramIndex++} OR d.target_department ILIKE $${paramIndex++})`;
        params.push(`%${search}%`, `%${search}%`);
        paramIndex++;
      }

      const countQuery = query.replace(/SELECT[\s\S]*?FROM/, 'SELECT COUNT(*) as total FROM');
      const countResult = await pool.query(countQuery, params);
      const total = parseInt(countResult.rows[0].total);

      query += ` ORDER BY d.created_at DESC LIMIT $${paramIndex++} OFFSET $${paramIndex++}`;
      params.push(parseInt(limit as string), parseInt(offset as string));

      const result = await pool.query(query, params);

      const statsQuery = `
        SELECT 
          doc_type as "docType",
          status,
          COUNT(*) as count
        FROM warehouse.documents
        GROUP BY doc_type, status
        ORDER BY doc_type, status
      `;
      const statsResult = await pool.query(statsQuery);

      const stats = {
        byType: {} as Record<string, number>,
        byStatus: {} as Record<string, number>,
        total: 0,
      };

      for (const row of statsResult.rows) {
        const count = parseInt(row.count);
        stats.byType[row.docType] = (stats.byType[row.docType] || 0) + count;
        stats.byStatus[row.status] = (stats.byStatus[row.status] || 0) + count;
        stats.total += count;
      }

      res.json({
        documents: result.rows,
        pagination: {
          limit: parseInt(limit as string),
          offset: parseInt(offset as string),
          total,
        },
        stats,
      });
    } catch (error) {
      console.error("❌ Error fetching warehouse documents:", error);
      res.status(500).json({ error: "Failed to fetch warehouse documents" });
    }
  });

  app.get("/api/warehouse/documents/:id", isAuthenticated, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid document ID" });
      }

      const docResult = await pool.query(`
        SELECT 
          d.id,
          d.doc_number as "docNumber",
          d.doc_type as "docType",
          d.status,
          d.plan_id as "planId",
          d.work_order_id as "workOrderId",
          d.operation_id as "operationId",
          d.source_location_id as "sourceLocationId",
          d.source_location_name as "sourceLocationName",
          d.target_department as "targetDepartment",
          d.target_location_id as "targetLocationId",
          d.issued_at as "issuedAt",
          d.issued_by as "issuedBy",
          d.issued_by_name as "issuedByName",
          d.confirmed_at as "confirmedAt",
          d.confirmed_by as "confirmedBy",
          d.confirmed_by_name as "confirmedByName",
          d.printed_at as "printedAt",
          d.printed_by as "printedBy",
          d.pdf_path as "pdfPath",
          d.completed_at as "completedAt",
          d.completed_by as "completedBy",
          d.completed_by_name as "completedByName",
          d.remarks,
          d.total_lines as "totalLines",
          d.total_quantity as "totalQuantity",
          d.created_at as "createdAt",
          d.updated_at as "updatedAt",
          pp.name as "planName",
          pl.name as "targetLocationName",
          sl.name as "sourceLocationDisplayName"
        FROM warehouse.documents d
        LEFT JOIN production.production_plans pp ON pp.id = d.plan_id
        LEFT JOIN production.production_locations pl ON pl.id = d.target_location_id
        LEFT JOIN production.production_locations sl ON sl.id = d.source_location_id
        WHERE d.id = $1
      `, [id]);

      if (docResult.rows.length === 0) {
        return res.status(404).json({ error: "Document not found" });
      }

      const linesResult = await pool.query(`
        SELECT 
          dl.id,
          dl.document_id as "documentId",
          dl.line_number as "lineNumber",
          dl.source_type as "sourceType",
          dl.source_id as "sourceId",
          dl.catalog_product_id as "catalogProductId",
          dl.sku,
          dl.product_name as "productName",
          dl.quantity_requested as "quantityRequested",
          dl.quantity_picked as "quantityPicked",
          dl.unit,
          dl.warehouse_location_id as "warehouseLocationId",
          dl.warehouse_location_name as "warehouseLocationName",
          dl.serial_numbers as "serialNumbers",
          dl.plan_line_id as "planLineId",
          dl.is_picked as "isPicked",
          dl.picked_at as "pickedAt",
          dl.picked_by as "pickedBy",
          dl.notes,
          dl.created_at as "createdAt",
          dl.updated_at as "updatedAt"
        FROM warehouse.document_lines dl
        WHERE dl.document_id = $1
        ORDER BY dl.line_number
      `, [id]);

      res.json({
        ...docResult.rows[0],
        lines: linesResult.rows,
      });
    } catch (error) {
      console.error("❌ Error fetching warehouse document:", error);
      res.status(500).json({ error: "Failed to fetch warehouse document" });
    }
  });

  app.post("/api/warehouse/documents", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    try {
      const parsed = createDocumentSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ error: "Invalid request body", details: parsed.error.errors });
      }

      const { docType, planId, workOrderId, operationId, sourceLocationId, sourceLocationName, targetDepartment, targetLocationId, remarks, lines } = parsed.data;
      const user = (req as any).user;
      const docNumber = await generateDocNumber(pool, docType);

      await client.query('BEGIN');

      const docResult = await client.query(`
        INSERT INTO warehouse.documents (
          doc_number, doc_type, status, plan_id, work_order_id, operation_id,
          source_location_id, source_location_name, target_department, target_location_id,
          issued_at, issued_by, issued_by_name, remarks, total_lines, total_quantity
        ) VALUES ($1, $2, 'draft', $3, $4, $5, $6, $7, $8, $9, NOW(), $10, $11, $12, $13, $14)
        RETURNING id
      `, [
        docNumber,
        docType,
        planId || null,
        workOrderId || null,
        operationId || null,
        sourceLocationId || null,
        sourceLocationName || null,
        targetDepartment || null,
        targetLocationId || null,
        user?.id || null,
        user?.username || null,
        remarks || null,
        lines.length,
        lines.reduce((sum, l) => sum + l.quantityRequested, 0),
      ]);

      const docId = docResult.rows[0].id;

      for (let i = 0; i < lines.length; i++) {
        const line = lines[i];
        await client.query(`
          INSERT INTO warehouse.document_lines (
            document_id, line_number, source_type, source_id, catalog_product_id,
            sku, product_name, quantity_requested, unit, 
            warehouse_location_id, warehouse_location_name, plan_line_id, notes
          ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
        `, [
          docId,
          i + 1,
          line.sourceType,
          line.sourceId || null,
          line.catalogProductId || null,
          line.sku || null,
          line.productName,
          line.quantityRequested,
          line.unit || 'szt',
          line.warehouseLocationId || null,
          line.warehouseLocationName || null,
          line.planLineId || null,
          line.notes || null,
        ]);
      }

      await client.query('COMMIT');

      console.log(`✅ Created warehouse document ${docNumber} with ${lines.length} lines`);

      res.status(201).json({ id: docId, docNumber });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error creating warehouse document:", error);
      res.status(500).json({ error: "Failed to create warehouse document" });
    } finally {
      client.release();
    }
  });

  app.patch("/api/warehouse/documents/:id", isAuthenticated, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid document ID" });
      }

      const parsed = updateDocumentSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ error: "Invalid request body", details: parsed.error.errors });
      }

      const { sourceLocationId, sourceLocationName, targetDepartment, targetLocationId, remarks } = parsed.data;

      const result = await pool.query(`
        UPDATE warehouse.documents
        SET 
          source_location_id = COALESCE($1, source_location_id),
          source_location_name = COALESCE($2, source_location_name),
          target_department = COALESCE($3, target_department),
          target_location_id = COALESCE($4, target_location_id),
          remarks = COALESCE($5, remarks),
          updated_at = NOW()
        WHERE id = $6 AND status = 'draft'
        RETURNING id, doc_number as "docNumber"
      `, [
        sourceLocationId ?? null,
        sourceLocationName ?? null,
        targetDepartment ?? null,
        targetLocationId ?? null,
        remarks ?? null,
        id,
      ]);

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Document not found or not in draft status" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating warehouse document:", error);
      res.status(500).json({ error: "Failed to update warehouse document" });
    }
  });

  app.post("/api/warehouse/documents/:id/confirm", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid document ID" });
      }

      const user = (req as any).user;

      await client.query('BEGIN');

      const docCheck = await client.query(
        'SELECT id, status, doc_number, plan_id, doc_type FROM warehouse.documents WHERE id = $1 FOR UPDATE',
        [id]
      );

      if (docCheck.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Document not found" });
      }

      if (docCheck.rows[0].status !== 'draft') {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Document must be in draft status to confirm" });
      }

      const hasPlanId = docCheck.rows[0].plan_id != null;
      const docType = docCheck.rows[0].doc_type;
      const skipReservation = hasPlanId && (docType === 'WZ-SPAK' || docType === 'WZ-FORM');
      
      if (skipReservation) {
        console.log(`📦 [DOC CONFIRM] Pomijam rezerwację dla ${docCheck.rows[0].doc_number} - produkty już zarezerwowane przez plan ${docCheck.rows[0].plan_id}`);
      }

      const linesResult = await client.query(`
        SELECT id, source_type, source_id, quantity_requested 
        FROM warehouse.document_lines 
        WHERE document_id = $1
      `, [id]);

      const reservationErrors: string[] = [];
      
      for (const line of linesResult.rows) {
        if (!line.source_id) continue;
        
        if (skipReservation) {
          continue;
        }
        
        let updateResult;
        
        switch (line.source_type) {
          case 'packed_product':
            updateResult = await client.query(`
              UPDATE warehouse.packed_products 
              SET reserved_quantity = reserved_quantity + $1,
                  updated_at = NOW()
              WHERE id = $2 AND quantity - reserved_quantity >= $1
              RETURNING id, quantity, reserved_quantity
            `, [line.quantity_requested, line.source_id]);
            
            if (updateResult.rowCount === 0) {
              const checkResult = await client.query(
                'SELECT id, quantity, reserved_quantity FROM warehouse.packed_products WHERE id = $1',
                [line.source_id]
              );
              if (checkResult.rows.length === 0) {
                reservationErrors.push(`Produkt spakowany ID ${line.source_id} nie istnieje`);
              } else {
                const available = checkResult.rows[0].quantity - checkResult.rows[0].reserved_quantity;
                reservationErrors.push(`Niewystarczająca ilość produktu ID ${line.source_id}: dostępne ${available}, wymagane ${line.quantity_requested}`);
              }
            }
            break;
            
          case 'formatka':
            updateResult = await client.query(`
              UPDATE warehouse.stock_panels 
              SET reserved_quantity = reserved_quantity + $1,
                  updated_at = NOW()
              WHERE id = $2 AND quantity - reserved_quantity >= $1
              RETURNING id, quantity, reserved_quantity
            `, [line.quantity_requested, line.source_id]);
            
            if (updateResult.rowCount === 0) {
              const checkResult = await client.query(
                'SELECT id, quantity, reserved_quantity FROM warehouse.stock_panels WHERE id = $1',
                [line.source_id]
              );
              if (checkResult.rows.length === 0) {
                reservationErrors.push(`Formatka ID ${line.source_id} nie istnieje`);
              } else {
                const available = Number(checkResult.rows[0].quantity) - Number(checkResult.rows[0].reserved_quantity);
                reservationErrors.push(`Niewystarczająca ilość formatki ID ${line.source_id}: dostępne ${available}, wymagane ${line.quantity_requested}`);
              }
            }
            break;
            
          case 'opakowania':
            updateResult = await client.query(`
              UPDATE warehouse.packaging_materials 
              SET reserved_quantity = reserved_quantity + $1,
                  updated_at = NOW()
              WHERE id = $2 AND quantity - reserved_quantity >= $1
              RETURNING id, quantity, reserved_quantity
            `, [line.quantity_requested, line.source_id]);
            
            if (updateResult.rowCount === 0) {
              const checkResult = await client.query(
                'SELECT id, quantity, reserved_quantity FROM warehouse.packaging_materials WHERE id = $1',
                [line.source_id]
              );
              if (checkResult.rows.length === 0) {
                reservationErrors.push(`Opakowanie ID ${line.source_id} nie istnieje`);
              } else {
                const available = Number(checkResult.rows[0].quantity) - Number(checkResult.rows[0].reserved_quantity);
                reservationErrors.push(`Niewystarczająca ilość opakowania ID ${line.source_id}: dostępne ${available}, wymagane ${line.quantity_requested}`);
              }
            }
            break;
            
          case 'okucia':
          case 'tapicernia':
            updateResult = await client.query(`
              UPDATE warehouse.materials 
              SET reserved_quantity = reserved_quantity + $1,
                  updated_at = NOW()
              WHERE id = $2 AND quantity - reserved_quantity >= $1
              RETURNING id, quantity, reserved_quantity
            `, [line.quantity_requested, line.source_id]);
            
            if (updateResult.rowCount === 0) {
              const checkResult = await client.query(
                'SELECT id, quantity, reserved_quantity FROM warehouse.materials WHERE id = $1',
                [line.source_id]
              );
              if (checkResult.rows.length === 0) {
                reservationErrors.push(`Materiał ID ${line.source_id} nie istnieje`);
              } else {
                const available = Number(checkResult.rows[0].quantity) - Number(checkResult.rows[0].reserved_quantity);
                reservationErrors.push(`Niewystarczająca ilość materiału ID ${line.source_id}: dostępne ${available}, wymagane ${line.quantity_requested}`);
              }
            }
            break;
        }
      }
      
      if (reservationErrors.length > 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({ 
          error: "Nie można potwierdzić dokumentu - błędy rezerwacji", 
          details: reservationErrors 
        });
      }

      const result = await client.query(`
        UPDATE warehouse.documents
        SET 
          status = 'confirmed',
          confirmed_at = NOW(),
          confirmed_by = $1,
          confirmed_by_name = $2,
          updated_at = NOW()
        WHERE id = $3
        RETURNING id, doc_number as "docNumber", doc_type as "docType", status, work_order_id as "workOrderId"
      `, [user?.id || null, user?.username || null, id]);

      const confirmedDoc = result.rows[0];

      if (confirmedDoc.workOrderId) {
        const docTypeToSourceType: Record<string, string> = {
          'WZ-HDF': 'hdf',
          'WZ-FORM': 'formatka',
          'WZ-OKUC': 'okucia',
          'WZ-OPAK': 'opakowania',
          'WZ-TAP': 'tapicernia',
          'WZ-SUR': 'surowiec',
          'WZ-SPAK': 'packed_product',
        };
        
        const sourceType = docTypeToSourceType[confirmedDoc.docType];
        
        if (sourceType) {
          for (const line of linesResult.rows) {
            const lineQuantity = Number(line.quantity_requested);
            
            const updateMaterialSource = await client.query(`
              UPDATE production.operation_material_sources
              SET 
                quantity_provided = LEAST(quantity_required, quantity_provided + $1),
                document_id = COALESCE(document_id, $2),
                document_number = COALESCE(document_number, $3),
                status = CASE 
                  WHEN LEAST(quantity_required, quantity_provided + $1) >= quantity_required THEN 'complete'
                  WHEN LEAST(quantity_required, quantity_provided + $1) > 0 THEN 'partial'
                  ELSE 'pending'
                END,
                updated_at = NOW()
              WHERE work_order_id = $4 
                AND source_type = $5 
                AND status != 'complete'
                AND id = (
                  SELECT id FROM production.operation_material_sources 
                  WHERE work_order_id = $4 AND source_type = $5 AND status != 'complete'
                  ORDER BY created_at ASC
                  LIMIT 1
                )
              RETURNING id, status, quantity_provided as "quantityProvided", quantity_required as "quantityRequired"
            `, [lineQuantity, id, confirmedDoc.docNumber, confirmedDoc.workOrderId, sourceType]);
            
            if (updateMaterialSource.rowCount && updateMaterialSource.rowCount > 0) {
              const updated = updateMaterialSource.rows[0];
              console.log(`📦 Updated material source ${updated.id}: ${updated.quantityProvided}/${updated.quantityRequired} (${updated.status})`);
            }
          }
        }
      }

      await client.query('COMMIT');

      console.log(`✅ Confirmed warehouse document ${confirmedDoc.docNumber} - reserved ${linesResult.rows.length} lines`);

      res.json(result.rows[0]);
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error confirming warehouse document:", error);
      res.status(500).json({ error: "Failed to confirm warehouse document" });
    } finally {
      client.release();
    }
  });

  app.post("/api/warehouse/documents/:id/complete", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid document ID" });
      }

      const user = (req as any).user;

      await client.query('BEGIN');

      const docCheck = await client.query(
        'SELECT id, status, doc_number FROM warehouse.documents WHERE id = $1 FOR UPDATE',
        [id]
      );

      if (docCheck.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Document not found" });
      }

      if (!['confirmed', 'printed'].includes(docCheck.rows[0].status)) {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Document must be in confirmed or printed status to complete" });
      }

      const linesResult = await client.query(`
        SELECT id, source_type, source_id, quantity_requested 
        FROM warehouse.document_lines 
        WHERE document_id = $1
      `, [id]);

      const deductionErrors: string[] = [];
      
      for (const line of linesResult.rows) {
        if (!line.source_id) continue;

        let updateResult;
        
        switch (line.source_type) {
          case 'packed_product':
            updateResult = await client.query(`
              UPDATE warehouse.packed_products 
              SET quantity = quantity - $1,
                  reserved_quantity = GREATEST(0, reserved_quantity - $1),
                  updated_at = NOW()
              WHERE id = $2 AND quantity >= $1 AND reserved_quantity >= $1
              RETURNING id, quantity, reserved_quantity
            `, [line.quantity_requested, line.source_id]);
            
            if (updateResult.rowCount === 0) {
              const checkResult = await client.query(
                'SELECT id, quantity, reserved_quantity FROM warehouse.packed_products WHERE id = $1',
                [line.source_id]
              );
              if (checkResult.rows.length === 0) {
                deductionErrors.push(`Produkt spakowany ID ${line.source_id} nie istnieje`);
              } else {
                deductionErrors.push(`Niewystarczająca ilość produktu ID ${line.source_id}: stan ${checkResult.rows[0].quantity}, zarezerwowane ${checkResult.rows[0].reserved_quantity}, wymagane ${line.quantity_requested}`);
              }
            }
            break;
            
          case 'formatka':
            updateResult = await client.query(`
              UPDATE warehouse.stock_panels 
              SET quantity = quantity - $1,
                  reserved_quantity = GREATEST(0, reserved_quantity - $1),
                  updated_at = NOW()
              WHERE id = $2 AND quantity >= $1 AND reserved_quantity >= $1
              RETURNING id, quantity, reserved_quantity
            `, [line.quantity_requested, line.source_id]);
            
            if (updateResult.rowCount === 0) {
              const checkResult = await client.query(
                'SELECT id, quantity, reserved_quantity FROM warehouse.stock_panels WHERE id = $1',
                [line.source_id]
              );
              if (checkResult.rows.length === 0) {
                deductionErrors.push(`Formatka ID ${line.source_id} nie istnieje`);
              } else {
                deductionErrors.push(`Niewystarczająca ilość formatki ID ${line.source_id}: stan ${checkResult.rows[0].quantity}, zarezerwowane ${checkResult.rows[0].reserved_quantity}, wymagane ${line.quantity_requested}`);
              }
            }
            break;
            
          case 'opakowania':
            updateResult = await client.query(`
              UPDATE warehouse.packaging_materials 
              SET quantity = quantity - $1,
                  reserved_quantity = GREATEST(0, reserved_quantity - $1),
                  updated_at = NOW()
              WHERE id = $2 AND quantity >= $1 AND reserved_quantity >= $1
              RETURNING id, quantity, reserved_quantity
            `, [line.quantity_requested, line.source_id]);
            
            if (updateResult.rowCount === 0) {
              const checkResult = await client.query(
                'SELECT id, quantity, reserved_quantity FROM warehouse.packaging_materials WHERE id = $1',
                [line.source_id]
              );
              if (checkResult.rows.length === 0) {
                deductionErrors.push(`Opakowanie ID ${line.source_id} nie istnieje`);
              } else {
                deductionErrors.push(`Niewystarczająca ilość opakowania ID ${line.source_id}: stan ${checkResult.rows[0].quantity}, zarezerwowane ${checkResult.rows[0].reserved_quantity}, wymagane ${line.quantity_requested}`);
              }
            }
            break;
            
          case 'okucia':
          case 'tapicernia':
            updateResult = await client.query(`
              UPDATE warehouse.materials 
              SET quantity = quantity - $1,
                  reserved_quantity = GREATEST(0, reserved_quantity - $1),
                  updated_at = NOW()
              WHERE id = $2 AND quantity >= $1 AND reserved_quantity >= $1
              RETURNING id, quantity, reserved_quantity
            `, [line.quantity_requested, line.source_id]);
            
            if (updateResult.rowCount === 0) {
              const checkResult = await client.query(
                'SELECT id, quantity, reserved_quantity FROM warehouse.materials WHERE id = $1',
                [line.source_id]
              );
              if (checkResult.rows.length === 0) {
                deductionErrors.push(`Materiał ID ${line.source_id} nie istnieje`);
              } else {
                deductionErrors.push(`Niewystarczająca ilość materiału ID ${line.source_id}: stan ${checkResult.rows[0].quantity}, zarezerwowane ${checkResult.rows[0].reserved_quantity}, wymagane ${line.quantity_requested}`);
              }
            }
            break;
        }
      }
      
      if (deductionErrors.length > 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({ 
          error: "Nie można zrealizować dokumentu - błędy wydania", 
          details: deductionErrors 
        });
      }

      const result = await client.query(`
        UPDATE warehouse.documents
        SET 
          status = 'completed',
          completed_at = NOW(),
          completed_by = $1,
          completed_by_name = $2,
          updated_at = NOW()
        WHERE id = $3
        RETURNING id, doc_number as "docNumber", status
      `, [user?.id || null, user?.username || null, id]);

      await client.query('COMMIT');

      console.log(`✅ Completed warehouse document ${result.rows[0].docNumber} - deducted ${linesResult.rows.length} lines from stock`);

      res.json(result.rows[0]);
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error completing warehouse document:", error);
      res.status(500).json({ error: "Failed to complete warehouse document" });
    } finally {
      client.release();
    }
  });

  app.post("/api/warehouse/documents/:id/cancel", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid document ID" });
      }

      const user = (req as any).user;

      await client.query('BEGIN');

      const docCheck = await client.query(
        'SELECT id, status, doc_number FROM warehouse.documents WHERE id = $1 FOR UPDATE',
        [id]
      );

      if (docCheck.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Document not found" });
      }

      if (docCheck.rows[0].status === 'completed') {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Cannot cancel completed documents" });
      }

      const wasConfirmed = ['confirmed', 'printed'].includes(docCheck.rows[0].status);
      
      if (wasConfirmed) {
        const linesResult = await client.query(`
          SELECT id, source_type, source_id, quantity_requested 
          FROM warehouse.document_lines 
          WHERE document_id = $1
        `, [id]);

        for (const line of linesResult.rows) {
          if (!line.source_id) continue;
          
          switch (line.source_type) {
            case 'packed_product':
              await client.query(`
                UPDATE warehouse.packed_products 
                SET reserved_quantity = GREATEST(0, reserved_quantity - $1),
                    updated_at = NOW()
                WHERE id = $2
              `, [line.quantity_requested, line.source_id]);
              break;
              
            case 'formatka':
              await client.query(`
                UPDATE warehouse.stock_panels 
                SET reserved_quantity = GREATEST(0, reserved_quantity - $1),
                    updated_at = NOW()
                WHERE id = $2
              `, [line.quantity_requested, line.source_id]);
              break;
              
            case 'opakowania':
              await client.query(`
                UPDATE warehouse.packaging_materials 
                SET reserved_quantity = GREATEST(0, reserved_quantity - $1),
                    updated_at = NOW()
                WHERE id = $2
              `, [line.quantity_requested, line.source_id]);
              break;
              
            case 'okucia':
            case 'tapicernia':
              await client.query(`
                UPDATE warehouse.materials 
                SET reserved_quantity = GREATEST(0, reserved_quantity - $1),
                    updated_at = NOW()
                WHERE id = $2
              `, [line.quantity_requested, line.source_id]);
              break;
          }
        }
      }

      const result = await client.query(`
        UPDATE warehouse.documents
        SET 
          status = 'cancelled',
          updated_at = NOW()
        WHERE id = $1
        RETURNING id, doc_number as "docNumber", status
      `, [id]);

      await client.query('COMMIT');

      console.log(`❌ Cancelled warehouse document ${result.rows[0].docNumber}${wasConfirmed ? ' - released reservations' : ''}`);

      res.json(result.rows[0]);
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error cancelling warehouse document:", error);
      res.status(500).json({ error: "Failed to cancel warehouse document" });
    } finally {
      client.release();
    }
  });

  app.delete("/api/warehouse/documents/:id", isAuthenticated, async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ error: "Invalid document ID" });
      }

      const result = await pool.query(`
        DELETE FROM warehouse.documents
        WHERE id = $1 AND status = 'draft'
        RETURNING id, doc_number as "docNumber"
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Document not found or not in draft status" });
      }

      console.log(`🗑️ Deleted warehouse document ${result.rows[0].docNumber}`);

      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting warehouse document:", error);
      res.status(500).json({ error: "Failed to delete warehouse document" });
    }
  });

  app.post("/api/warehouse/documents/:id/lines", isAuthenticated, async (req, res) => {
    try {
      const docId = parseInt(req.params.id);
      if (isNaN(docId)) {
        return res.status(400).json({ error: "Invalid document ID" });
      }

      const lineSchema = z.object({
        sourceType: z.enum(['packed_product', 'formatka', 'okucia', 'opakowania', 'tapicernia', 'custom']),
        sourceId: z.number().optional().nullable(),
        catalogProductId: z.number().optional().nullable(),
        sku: z.string().optional().nullable(),
        productName: z.string(),
        quantityRequested: z.number(),
        unit: z.string().optional().default('szt'),
        warehouseLocationId: z.number().optional().nullable(),
        warehouseLocationName: z.string().optional().nullable(),
        planLineId: z.number().optional().nullable(),
        notes: z.string().optional().nullable(),
      });

      const parsed = lineSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ error: "Invalid request body", details: parsed.error.errors });
      }

      const docCheck = await pool.query(
        'SELECT id, status FROM warehouse.documents WHERE id = $1',
        [docId]
      );

      if (docCheck.rows.length === 0) {
        return res.status(404).json({ error: "Document not found" });
      }

      if (docCheck.rows[0].status !== 'draft') {
        return res.status(400).json({ error: "Can only add lines to draft documents" });
      }

      const maxLineResult = await pool.query(
        'SELECT COALESCE(MAX(line_number), 0) + 1 as next_line FROM warehouse.document_lines WHERE document_id = $1',
        [docId]
      );
      const lineNumber = maxLineResult.rows[0].next_line;

      const line = parsed.data;
      const result = await pool.query(`
        INSERT INTO warehouse.document_lines (
          document_id, line_number, source_type, source_id, catalog_product_id,
          sku, product_name, quantity_requested, unit,
          warehouse_location_id, warehouse_location_name, plan_line_id, notes
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
        RETURNING id
      `, [
        docId,
        lineNumber,
        line.sourceType,
        line.sourceId || null,
        line.catalogProductId || null,
        line.sku || null,
        line.productName,
        line.quantityRequested,
        line.unit || 'szt',
        line.warehouseLocationId || null,
        line.warehouseLocationName || null,
        line.planLineId || null,
        line.notes || null,
      ]);

      await pool.query(`
        UPDATE warehouse.documents
        SET 
          total_lines = total_lines + 1,
          total_quantity = total_quantity + $1,
          updated_at = NOW()
        WHERE id = $2
      `, [line.quantityRequested, docId]);

      res.status(201).json({ id: result.rows[0].id, lineNumber });
    } catch (error) {
      console.error("❌ Error adding document line:", error);
      res.status(500).json({ error: "Failed to add document line" });
    }
  });

  app.delete("/api/warehouse/documents/:docId/lines/:lineId", isAuthenticated, async (req, res) => {
    try {
      const docId = parseInt(req.params.docId);
      const lineId = parseInt(req.params.lineId);
      
      if (isNaN(docId) || isNaN(lineId)) {
        return res.status(400).json({ error: "Invalid document or line ID" });
      }

      const docCheck = await pool.query(
        'SELECT id, status FROM warehouse.documents WHERE id = $1',
        [docId]
      );

      if (docCheck.rows.length === 0) {
        return res.status(404).json({ error: "Document not found" });
      }

      if (docCheck.rows[0].status !== 'draft') {
        return res.status(400).json({ error: "Can only delete lines from draft documents" });
      }

      const lineResult = await pool.query(
        'SELECT quantity_requested FROM warehouse.document_lines WHERE id = $1 AND document_id = $2',
        [lineId, docId]
      );

      if (lineResult.rows.length === 0) {
        return res.status(404).json({ error: "Line not found" });
      }

      await pool.query(
        'DELETE FROM warehouse.document_lines WHERE id = $1',
        [lineId]
      );

      await pool.query(`
        UPDATE warehouse.documents
        SET 
          total_lines = GREATEST(0, total_lines - 1),
          total_quantity = GREATEST(0, total_quantity - $1),
          updated_at = NOW()
        WHERE id = $2
      `, [lineResult.rows[0].quantity_requested, docId]);

      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting document line:", error);
      res.status(500).json({ error: "Failed to delete document line" });
    }
  });

  app.get("/api/warehouse/documents/meta/types", isAuthenticated, async (_req, res) => {
    res.json({
      documentTypes: Object.entries(DOCUMENT_TYPE_NAMES).map(([value, label]) => ({ value, label })),
      statusTypes: Object.entries(STATUS_NAMES).map(([value, label]) => ({ value, label })),
      lineSourceTypes: [
        { value: 'packed_product', label: 'Produkty spakowane' },
        { value: 'formatka', label: 'Formatki' },
        { value: 'okucia', label: 'Okucia' },
        { value: 'opakowania', label: 'Opakowania' },
        { value: 'tapicernia', label: 'Tapicernia' },
        { value: 'custom', label: 'Inne' },
      ],
    });
  });

  app.get("/api/warehouse/documents/:id/pdf", isAuthenticated, async (req, res) => {
    try {
      const documentId = parseInt(req.params.id);
      if (isNaN(documentId)) {
        return res.status(400).json({ error: "Invalid document ID" });
      }

      const pdfBuffer = await generateWarehouseDocumentPDF(pool, documentId);
      
      if (!pdfBuffer) {
        return res.status(404).json({ error: "Document not found" });
      }

      const docResult = await pool.query(
        'SELECT doc_number FROM warehouse.documents WHERE id = $1',
        [documentId]
      );
      const docNumber = docResult.rows[0]?.doc_number || `DOC-${documentId}`;
      const safeFileName = docNumber.replace(/\//g, '-');

      res.setHeader('Content-Type', 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="${safeFileName}.pdf"`);
      res.setHeader('Content-Length', pdfBuffer.length);
      res.send(pdfBuffer);
    } catch (error) {
      console.error("❌ Error generating document PDF:", error);
      res.status(500).json({ error: "Failed to generate PDF" });
    }
  });
}
