import type { Express } from "express";
import { requirePermission } from "../../auth";
import { z } from "zod";
import { pool } from "../../postgres";

const variantFormSchema = z.object({
  variantCode: z.string().min(1),
  variantName: z.string().min(1),
  description: z.string().nullable().optional(),
  defaultOperations: z.array(z.string()).min(1, "At least one operation is required"),
  routingIds: z.array(z.number()).default([]),
  isActive: z.boolean().default(true),
});

const ruleFormSchema = z.object({
  routingVariantId: z.number(),
  namePattern: z.string().nullable().optional(),
  formatkaCodes: z.array(z.string()).default([]),
  colorPattern: z.string().nullable().optional(),
  priority: z.number().default(0),
  isActive: z.boolean().default(true),
});

export function registerRoutingVariantRoutes(app: Express) {
  // GET /api/production/routing-variants - Get all routing variants with rules count and linked routings
  app.get("/api/production/routing-variants", requirePermission("view_production"), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          v.*,
          (SELECT COUNT(*) FROM production.production_routing_variant_rules vr WHERE vr.routing_variant_id = v.id) as rules_count
        FROM production.production_routing_variants v
        ORDER BY v.variant_code
      `);
      
      // Fetch linked routings for each variant
      const routingIds: number[] = [];
      result.rows.forEach(row => {
        const ids = row.routing_ids || [];
        ids.forEach((id: number) => {
          if (!routingIds.includes(id)) routingIds.push(id);
        });
      });

      let routingsMap: Record<number, { code: string; name: string }> = {};
      if (routingIds.length > 0) {
        const routingsResult = await pool.query(`
          SELECT id, code, name FROM production.production_routings WHERE id = ANY($1)
        `, [routingIds]);
        routingsResult.rows.forEach(r => {
          routingsMap[r.id] = { code: r.code, name: r.name };
        });
      }
      
      const variants = result.rows.map(row => ({
        id: row.id,
        variantCode: row.variant_code,
        variantName: row.variant_name,
        description: row.description,
        defaultOperations: row.default_operations,
        routingIds: row.routing_ids || [],
        linkedRoutings: (row.routing_ids || []).map((id: number) => ({
          id,
          code: routingsMap[id]?.code || '',
          name: routingsMap[id]?.name || '',
        })).filter((r: any) => r.code),
        isActive: row.is_active,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
        rulesCount: parseInt(row.rules_count),
      }));
      
      res.json(variants);
    } catch (error) {
      console.error("Error fetching routing variants:", error);
      res.status(500).json({ message: "Failed to fetch routing variants" });
    }
  });

  // GET /api/production/routing-variants/:id - Get routing variant by ID with rules
  app.get("/api/production/routing-variants/:id", requirePermission("view_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid variant ID" });
      }

      const variantResult = await pool.query(`
        SELECT * FROM production.production_routing_variants WHERE id = $1
      `, [id]);

      if (variantResult.rows.length === 0) {
        return res.status(404).json({ message: "Routing variant not found" });
      }

      const row = variantResult.rows[0];
      
      // Fetch linked routings
      const routingIds = row.routing_ids || [];
      let linkedRoutings: any[] = [];
      if (routingIds.length > 0) {
        const routingsResult = await pool.query(`
          SELECT id, code, name FROM production.production_routings WHERE id = ANY($1)
        `, [routingIds]);
        linkedRoutings = routingsResult.rows.map(r => ({
          id: r.id,
          code: r.code,
          name: r.name,
        }));
      }

      const variant = {
        id: row.id,
        variantCode: row.variant_code,
        variantName: row.variant_name,
        description: row.description,
        defaultOperations: row.default_operations,
        routingIds: row.routing_ids || [],
        linkedRoutings,
        isActive: row.is_active,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      };

      res.json(variant);
    } catch (error) {
      console.error("Error fetching routing variant:", error);
      res.status(500).json({ message: "Failed to fetch routing variant" });
    }
  });

  // POST /api/production/routing-variants - Create new routing variant
  app.post("/api/production/routing-variants", requirePermission("manage_production"), async (req, res) => {
    try {
      const data = variantFormSchema.parse(req.body);
      
      const result = await pool.query(`
        INSERT INTO production.production_routing_variants 
          (variant_code, variant_name, description, default_operations, routing_ids, is_active)
        VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING *
      `, [
        data.variantCode,
        data.variantName,
        data.description || null,
        JSON.stringify(data.defaultOperations),
        JSON.stringify(data.routingIds),
        data.isActive ?? true,
      ]);

      const row = result.rows[0];
      res.status(201).json({
        id: row.id,
        variantCode: row.variant_code,
        variantName: row.variant_name,
        description: row.description,
        defaultOperations: row.default_operations,
        routingIds: row.routing_ids || [],
        isActive: row.is_active,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Variant with this code already exists" });
      }
      console.error("Error creating routing variant:", error);
      res.status(500).json({ message: "Failed to create routing variant" });
    }
  });

  // PATCH /api/production/routing-variants/:id - Update routing variant
  app.patch("/api/production/routing-variants/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid variant ID" });
      }

      const data = variantFormSchema.partial().parse(req.body);
      
      const setClauses: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      if (data.variantCode !== undefined) {
        setClauses.push(`variant_code = $${paramIndex++}`);
        values.push(data.variantCode);
      }
      if (data.variantName !== undefined) {
        setClauses.push(`variant_name = $${paramIndex++}`);
        values.push(data.variantName);
      }
      if (data.description !== undefined) {
        setClauses.push(`description = $${paramIndex++}`);
        values.push(data.description);
      }
      if (data.defaultOperations !== undefined) {
        setClauses.push(`default_operations = $${paramIndex++}`);
        values.push(JSON.stringify(data.defaultOperations));
      }
      if (data.routingIds !== undefined) {
        setClauses.push(`routing_ids = $${paramIndex++}`);
        values.push(JSON.stringify(data.routingIds));
      }
      if (data.isActive !== undefined) {
        setClauses.push(`is_active = $${paramIndex++}`);
        values.push(data.isActive);
      }

      if (setClauses.length === 0) {
        return res.status(400).json({ message: "No fields to update" });
      }

      setClauses.push(`updated_at = NOW()`);
      values.push(id);

      const result = await pool.query(`
        UPDATE production.production_routing_variants 
        SET ${setClauses.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING *
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Routing variant not found" });
      }

      const row = result.rows[0];
      res.json({
        id: row.id,
        variantCode: row.variant_code,
        variantName: row.variant_name,
        description: row.description,
        defaultOperations: row.default_operations,
        routingIds: row.routing_ids || [],
        isActive: row.is_active,
        createdAt: row.created_at,
        updatedAt: row.updated_at,
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Variant with this code already exists" });
      }
      console.error("Error updating routing variant:", error);
      res.status(500).json({ message: "Failed to update routing variant" });
    }
  });

  // DELETE /api/production/routing-variants/:id - Delete routing variant
  app.delete("/api/production/routing-variants/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid variant ID" });
      }

      const result = await pool.query(`
        DELETE FROM production.production_routing_variants WHERE id = $1 RETURNING id
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Routing variant not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting routing variant:", error);
      res.status(500).json({ message: "Failed to delete routing variant" });
    }
  });

  // =============== RULES ENDPOINTS ===============

  // GET /api/production/routing-variants/:variantId/rules - Get rules for variant
  app.get("/api/production/routing-variants/:variantId/rules", requirePermission("view_production"), async (req, res) => {
    try {
      const variantId = parseInt(req.params.variantId);
      if (isNaN(variantId)) {
        return res.status(400).json({ message: "Invalid variant ID" });
      }

      const result = await pool.query(`
        SELECT * FROM production.production_routing_variant_rules
        WHERE routing_variant_id = $1
        ORDER BY priority DESC, id ASC
      `, [variantId]);

      const rules = result.rows.map(row => ({
        id: row.id,
        routingVariantId: row.routing_variant_id,
        namePattern: row.name_pattern,
        formatkaCodes: row.formatka_codes || [],
        colorPattern: row.color_pattern,
        priority: row.priority,
        isActive: row.is_active,
        createdAt: row.created_at,
      }));

      res.json(rules);
    } catch (error) {
      console.error("Error fetching rules:", error);
      res.status(500).json({ message: "Failed to fetch rules" });
    }
  });

  // GET /api/production/routing-variant-rules - Get all rules (for overview)
  app.get("/api/production/routing-variant-rules", requirePermission("view_production"), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          r.*,
          v.variant_code,
          v.variant_name
        FROM production.production_routing_variant_rules r
        JOIN production.production_routing_variants v ON v.id = r.routing_variant_id
        ORDER BY r.priority DESC, r.id ASC
      `);

      const rules = result.rows.map(row => ({
        id: row.id,
        routingVariantId: row.routing_variant_id,
        variantCode: row.variant_code,
        variantName: row.variant_name,
        namePattern: row.name_pattern,
        formatkaCodes: row.formatka_codes || [],
        colorPattern: row.color_pattern,
        priority: row.priority,
        isActive: row.is_active,
        createdAt: row.created_at,
      }));

      res.json(rules);
    } catch (error) {
      console.error("Error fetching rules:", error);
      res.status(500).json({ message: "Failed to fetch rules" });
    }
  });

  // POST /api/production/routing-variant-rules - Create new rule
  app.post("/api/production/routing-variant-rules", requirePermission("manage_production"), async (req, res) => {
    try {
      const data = ruleFormSchema.parse(req.body);
      
      const result = await pool.query(`
        INSERT INTO production.production_routing_variant_rules 
          (routing_variant_id, name_pattern, formatka_codes, color_pattern, priority, is_active)
        VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING *
      `, [
        data.routingVariantId,
        data.namePattern || null,
        JSON.stringify(data.formatkaCodes || []),
        data.colorPattern || null,
        data.priority ?? 0,
        data.isActive ?? true,
      ]);

      const row = result.rows[0];
      res.status(201).json({
        id: row.id,
        routingVariantId: row.routing_variant_id,
        namePattern: row.name_pattern,
        formatkaCodes: row.formatka_codes || [],
        colorPattern: row.color_pattern,
        priority: row.priority,
        isActive: row.is_active,
        createdAt: row.created_at,
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error creating rule:", error);
      res.status(500).json({ message: "Failed to create rule" });
    }
  });

  // PATCH /api/production/routing-variant-rules/:id - Update rule
  app.patch("/api/production/routing-variant-rules/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid rule ID" });
      }

      const data = ruleFormSchema.partial().parse(req.body);
      
      const setClauses: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      if (data.routingVariantId !== undefined) {
        setClauses.push(`routing_variant_id = $${paramIndex++}`);
        values.push(data.routingVariantId);
      }
      if (data.namePattern !== undefined) {
        setClauses.push(`name_pattern = $${paramIndex++}`);
        values.push(data.namePattern);
      }
      if (data.formatkaCodes !== undefined) {
        setClauses.push(`formatka_codes = $${paramIndex++}`);
        values.push(JSON.stringify(data.formatkaCodes));
      }
      if (data.colorPattern !== undefined) {
        setClauses.push(`color_pattern = $${paramIndex++}`);
        values.push(data.colorPattern);
      }
      if (data.priority !== undefined) {
        setClauses.push(`priority = $${paramIndex++}`);
        values.push(data.priority);
      }
      if (data.isActive !== undefined) {
        setClauses.push(`is_active = $${paramIndex++}`);
        values.push(data.isActive);
      }

      if (setClauses.length === 0) {
        return res.status(400).json({ message: "No fields to update" });
      }

      values.push(id);

      const result = await pool.query(`
        UPDATE production.production_routing_variant_rules 
        SET ${setClauses.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING *
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Rule not found" });
      }

      const row = result.rows[0];
      res.json({
        id: row.id,
        routingVariantId: row.routing_variant_id,
        namePattern: row.name_pattern,
        formatkaCodes: row.formatka_codes || [],
        colorPattern: row.color_pattern,
        priority: row.priority,
        isActive: row.is_active,
        createdAt: row.created_at,
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      console.error("Error updating rule:", error);
      res.status(500).json({ message: "Failed to update rule" });
    }
  });

  // DELETE /api/production/routing-variant-rules/:id - Delete rule
  app.delete("/api/production/routing-variant-rules/:id", requirePermission("manage_production"), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid rule ID" });
      }

      const result = await pool.query(`
        DELETE FROM production.production_routing_variant_rules WHERE id = $1 RETURNING id
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Rule not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting rule:", error);
      res.status(500).json({ message: "Failed to delete rule" });
    }
  });

  // POST /api/production/routing-variant-rules/test - Test pattern matching
  app.post("/api/production/routing-variant-rules/test", requirePermission("view_production"), async (req, res) => {
    try {
      const { componentName, color } = req.body;
      
      if (!componentName) {
        return res.status(400).json({ message: "componentName is required" });
      }

      // Wyodrębnij kod formatki z nazwy komponentu (usuń rozmiar z końca - zachowaj myślniki)
      const sizePattern = /[\s-]?\d+x\d+.*$/i;
      const formatkaCode = componentName.replace(sizePattern, '').trim().toUpperCase();

      // Najpierw sprawdź formatkaCodes (dokładne dopasowanie kodu formatki)
      const formatkaResult = await pool.query(`
        SELECT 
          r.id as rule_id,
          v.id as variant_id,
          v.variant_code,
          v.variant_name,
          v.default_operations,
          v.routing_ids,
          r.name_pattern,
          r.formatka_codes,
          r.color_pattern,
          r.priority
        FROM production.production_routing_variant_rules r
        JOIN production.production_routing_variants v ON v.id = r.routing_variant_id
        WHERE r.is_active = true 
          AND v.is_active = true
          AND r.formatka_codes @> $1::jsonb
          AND (r.color_pattern = $2 OR r.color_pattern IS NULL)
        ORDER BY r.priority DESC, r.id ASC
        LIMIT 1
      `, [JSON.stringify([formatkaCode]), color || null]);

      if (formatkaResult.rows.length > 0) {
        const row = formatkaResult.rows[0];
        const routingIds = row.routing_ids || [];
        let linkedRoutings: any[] = [];
        if (routingIds.length > 0) {
          const routingsResult = await pool.query(`
            SELECT id, code, name FROM production.production_routings WHERE id = ANY($1)
          `, [routingIds]);
          linkedRoutings = routingsResult.rows.map(r => ({
            id: r.id,
            code: r.code,
            name: r.name,
          }));
        }

        return res.json({
          matched: true,
          matchType: 'formatkaCode',
          componentName,
          formatkaCode,
          color,
          matchedRule: {
            ruleId: row.rule_id,
            variantId: row.variant_id,
            variantCode: row.variant_code,
            variantName: row.variant_name,
            defaultOperations: row.default_operations,
            routingIds: routingIds,
            linkedRoutings,
            formatkaCodes: row.formatka_codes || [],
            namePattern: row.name_pattern,
            colorPattern: row.color_pattern,
            priority: row.priority,
          },
        });
      }

      // Fallback do name_pattern (SQL LIKE)
      const result = await pool.query(`
        SELECT 
          r.id as rule_id,
          v.id as variant_id,
          v.variant_code,
          v.variant_name,
          v.default_operations,
          v.routing_ids,
          r.name_pattern,
          r.formatka_codes,
          r.color_pattern,
          r.priority
        FROM production.production_routing_variant_rules r
        JOIN production.production_routing_variants v ON v.id = r.routing_variant_id
        WHERE r.is_active = true 
          AND v.is_active = true
          AND ($1 LIKE r.name_pattern OR r.name_pattern IS NULL)
          AND (r.color_pattern = $2 OR r.color_pattern IS NULL)
        ORDER BY r.priority DESC, r.id ASC
        LIMIT 1
      `, [componentName, color || null]);

      if (result.rows.length === 0) {
        return res.json({ 
          matched: false, 
          message: "No matching rule found",
          componentName,
          formatkaCode,
          color,
        });
      }

      const row = result.rows[0];
      
      // Fetch linked routings details
      const routingIds = row.routing_ids || [];
      let linkedRoutings: any[] = [];
      if (routingIds.length > 0) {
        const routingsResult = await pool.query(`
          SELECT id, code, name FROM production.production_routings WHERE id = ANY($1)
        `, [routingIds]);
        linkedRoutings = routingsResult.rows.map(r => ({
          id: r.id,
          code: r.code,
          name: r.name,
        }));
      }

      res.json({
        matched: true,
        matchType: 'namePattern',
        componentName,
        formatkaCode,
        color,
        matchedRule: {
          ruleId: row.rule_id,
          variantId: row.variant_id,
          variantCode: row.variant_code,
          variantName: row.variant_name,
          defaultOperations: row.default_operations,
          routingIds: routingIds,
          linkedRoutings,
          formatkaCodes: row.formatka_codes || [],
          namePattern: row.name_pattern,
          colorPattern: row.color_pattern,
          priority: row.priority,
        },
      });
    } catch (error) {
      console.error("Error testing pattern:", error);
      res.status(500).json({ message: "Failed to test pattern" });
    }
  });
}
