import type { Express } from "express";
import { z } from "zod";
import { pool } from "../../postgres";
import * as ordersService from "../../services/production/orders";
import { generateProductionCardPDF, generateSawCSV } from "../../services/production/order-exports";
import { requirePermission } from "../../auth";

const createOrderSchema = z.object({
  orderNumber: z.string().optional(),
  productId: z.number(),
  bomId: z.number().optional().nullable(),
  routingId: z.number().optional().nullable(),
  status: z.enum(['draft', 'confirmed', 'planned', 'in_progress', 'paused', 'done', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  quantityPlanned: z.number().positive(),
  unitOfMeasure: z.string().optional(),
  plannedStartDate: z.string().optional().nullable(),
  plannedEndDate: z.string().optional().nullable(),
  responsibleUserId: z.number().optional().nullable(),
  sourceOrderNumber: z.string().optional().nullable(),
  locationId: z.number().optional().nullable(),
  notes: z.string().optional().nullable(),
});

const filtersSchema = z.object({
  status: z.string().optional(),
  workflowStage: z.string().optional(),
  priority: z.string().optional(),
  productId: z.coerce.number().optional(),
  responsibleUserId: z.coerce.number().optional(),
  startDate: z.string().optional(),
  endDate: z.string().optional(),
  search: z.string().optional(),
  limit: z.coerce.number().optional(),
  offset: z.coerce.number().optional(),
});

const updateWorkflowStageSchema = z.object({
  workflowStage: z.string(),
});

const updateDamageSchema = z.object({
  isDamaged: z.boolean(),
  damageType: z.string().optional().nullable(),
  damageNotes: z.string().optional().nullable(),
}).refine(
  (data) => !data.isDamaged || (data.isDamaged && data.damageType),
  {
    message: "damageType is required when isDamaged is true",
    path: ["damageType"],
  }
);

export function registerOrderRoutes(app: Express) {
  // GET /api/production/orders - Get all orders with filtering
  app.get("/api/production/orders", requirePermission('view_production'), async (req, res) => {
    try {
      const filters = filtersSchema.parse(req.query);
      
      // Convert string dates to Date objects
      const processedFilters = {
        ...filters,
        startDate: filters.startDate ? new Date(filters.startDate) : undefined,
        endDate: filters.endDate ? new Date(filters.endDate) : undefined,
      };

      const orders = await ordersService.getOrders(pool, processedFilters);
      res.json(orders);
    } catch (error) {
      console.error("Error fetching production orders:", error);
      res.status(500).json({ message: "Failed to fetch production orders" });
    }
  });

  // GET /api/production/orders/:id - Get order by ID
  app.get("/api/production/orders/:id", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.getOrderById(pool, id);
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error fetching production order:", error);
      res.status(500).json({ message: "Failed to fetch production order" });
    }
  });

  // GET /api/production/orders/:id/items - Get BOM items (formatki/komponenty) for a ZLP
  app.get("/api/production/orders/:id/items", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const items = await ordersService.getOrderBomItems(pool, id);
      res.json(items);
    } catch (error) {
      console.error("Error fetching order BOM items:", error);
      res.status(500).json({ message: "Failed to fetch order items" });
    }
  });

  // POST /api/production/orders - Create new order
  app.post("/api/production/orders", requirePermission('manage_production'), async (req, res) => {
    try {
      const data = createOrderSchema.parse(req.body);
      
      // Auto-generate order number if not provided
      const orderNumber = data.orderNumber || await ordersService.generateOrderNumber(pool);
      
      const order = await ordersService.createOrder(pool, {
        ...data,
        orderNumber,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : null,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : null,
      });
      
      res.status(201).json(order);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Order with this number already exists" });
      }
      console.error("Error creating production order:", error);
      res.status(500).json({ message: "Failed to create production order" });
    }
  });

  // PATCH /api/production/orders/:id - Update order
  app.patch("/api/production/orders/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const data = createOrderSchema.partial().parse(req.body);
      const order = await ordersService.updateOrder(pool, id, {
        ...data,
        plannedStartDate: data.plannedStartDate ? new Date(data.plannedStartDate) : undefined,
        plannedEndDate: data.plannedEndDate ? new Date(data.plannedEndDate) : undefined,
      });
      
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.json(order);
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      if ((error as any).code === '23505') {
        return res.status(409).json({ message: "Order with this number already exists" });
      }
      console.error("Error updating production order:", error);
      res.status(500).json({ message: "Failed to update production order" });
    }
  });

  // DELETE /api/production/orders/:id - Delete order
  app.delete("/api/production/orders/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const success = await ordersService.deleteOrder(pool, id);
      if (!success) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.status(204).send();
    } catch (error) {
      console.error("Error deleting production order:", error);
      res.status(500).json({ message: "Failed to delete production order" });
    }
  });

  // POST /api/production/orders/:id/start - Start order
  app.post("/api/production/orders/:id/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.startOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot start order - invalid status or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error starting production order:", error);
      res.status(500).json({ message: "Failed to start production order" });
    }
  });

  // POST /api/production/orders/:id/complete - Complete order
  app.post("/api/production/orders/:id/complete", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.completeOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot complete order - must be in_progress or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error completing production order:", error);
      res.status(500).json({ message: "Failed to complete production order" });
    }
  });

  // POST /api/production/orders/:id/cancel - Cancel order
  app.post("/api/production/orders/:id/cancel", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const order = await ordersService.cancelOrder(pool, id);
      if (!order) {
        return res.status(400).json({ message: "Cannot cancel order - already done/cancelled or order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error cancelling production order:", error);
      res.status(500).json({ message: "Failed to cancel production order" });
    }
  });

  // PATCH /api/production/orders/:id/workflow-stage - Update workflow stage
  app.patch("/api/production/orders/:id/workflow-stage", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const { workflowStage } = updateWorkflowStageSchema.parse(req.body);
      
      // Get user ID from session (if authenticated)
      const userId = (req as any).user?.id;

      const order = await ordersService.updateWorkflowStage(pool, id, workflowStage, userId);
      if (!order) {
        return res.status(404).json({ message: "Production order not found" });
      }

      res.json(order);
    } catch (error) {
      console.error("Error updating workflow stage:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      res.status(500).json({ message: "Failed to update workflow stage" });
    }
  });

  // PATCH /api/production/order-bom-items/:id/damage - Mark/unmark component as damaged
  app.patch("/api/production/order-bom-items/:id/damage", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid BOM item ID" });
      }

      const data = updateDamageSchema.parse(req.body);
      
      // Get user ID from session (if authenticated)
      const userId = (req as any).user?.id;

      const item = await ordersService.updateBomItemDamage(pool, id, {
        isDamaged: data.isDamaged,
        damageType: data.damageType,
        damageNotes: data.damageNotes,
        damagedBy: userId,
      });
      
      if (!item) {
        return res.status(404).json({ message: "BOM item not found" });
      }

      res.json(item);
    } catch (error) {
      console.error("Error updating BOM item damage status:", error);
      if (error instanceof z.ZodError) {
        return res.status(400).json({ message: "Validation error", errors: error.errors });
      }
      res.status(500).json({ message: "Failed to update damage status" });
    }
  });

  // ====== WORK ORDER ROUTES ======

  // POST /api/production/work-orders/:id/start - Start work order
  app.post("/api/production/work-orders/:id/start", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const result = await ordersService.startWorkOrder(pool, id);
      
      if (result.error) {
        const statusMap: Record<string, number> = {
          'not_found': 404,
          'invalid_status': 400,
          'invalid_order_status': 400,
          'sequence_error': 400,
          'concurrent_update': 409,
        };
        return res.status(statusMap[result.error] || 400).json({ message: result.message });
      }
      
      res.json(result);
    } catch (error: any) {
      console.error("Error starting work order:", error);
      res.status(500).json({ message: "Internal server error while starting work order" });
    }
  });

  // POST /api/production/work-orders/:id/complete - Complete work order
  app.post("/api/production/work-orders/:id/complete", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const options: ordersService.CompleteWorkOrderOptions = {};
      
      if (req.body.quantityProduced !== undefined) {
        options.quantityProduced = parseFloat(req.body.quantityProduced);
      }
      if (req.body.quantityScrap !== undefined) {
        options.quantityScrap = parseFloat(req.body.quantityScrap);
      }
      if (req.body.qualityCheckPassed !== undefined) {
        options.qualityCheckPassed = req.body.qualityCheckPassed;
      }
      if (req.body.qualityCheckNotes !== undefined) {
        options.qualityCheckNotes = req.body.qualityCheckNotes;
      }
      if (req.body.notes !== undefined) {
        options.notes = req.body.notes;
      }

      const result = await ordersService.completeWorkOrder(pool, id, options);
      
      if (result.error) {
        const statusMap: Record<string, number> = {
          'not_found': 404,
          'invalid_status': 400,
          'concurrent_update': 409,
        };
        return res.status(statusMap[result.error] || 400).json({ message: result.message });
      }
      
      res.json(result);
    } catch (error: any) {
      console.error("Error completing work order:", error);
      res.status(500).json({ message: "Internal server error while completing work order" });
    }
  });

  // ========== Multi-Source Material APIs ==========

  const materialSourceSchema = z.object({
    workOrderId: z.number(),
    sourceType: z.enum(['hdf', 'formatka', 'okucia', 'opakowania', 'tapicernia', 'surowiec', 'polprodukt']),
    sourceLocationId: z.number().optional().nullable(),
    sourceLocationName: z.string().optional().nullable(),
    quantityRequired: z.number().positive(),
    notes: z.string().optional().nullable(),
  });

  const updateMaterialSourceSchema = z.object({
    quantityProvided: z.number().min(0).optional(),
    documentId: z.number().optional().nullable(),
    documentNumber: z.string().optional().nullable(),
    notes: z.string().optional().nullable(),
  });

  // GET /api/production/work-orders/:id/material-sources - Get material sources for work order
  app.get("/api/production/work-orders/:id/material-sources", requirePermission('view_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const result = await pool.query(`
        SELECT 
          oms.id,
          oms.work_order_id as "workOrderId",
          oms.source_type as "sourceType",
          oms.source_location_id as "sourceLocationId",
          oms.source_location_name as "sourceLocationName",
          oms.quantity_required as "quantityRequired",
          oms.quantity_provided as "quantityProvided",
          oms.document_id as "documentId",
          oms.document_number as "documentNumber",
          oms.status,
          oms.notes,
          oms.created_at as "createdAt",
          oms.updated_at as "updatedAt",
          pl.name as "locationName",
          pl.code as "locationCode"
        FROM production.operation_material_sources oms
        LEFT JOIN production.production_locations pl ON oms.source_location_id = pl.id
        WHERE oms.work_order_id = $1
        ORDER BY oms.source_type, oms.created_at
      `, [workOrderId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching material sources:", error);
      res.status(500).json({ message: "Failed to fetch material sources" });
    }
  });

  // POST /api/production/work-orders/material-sources - Add material source requirement
  app.post("/api/production/work-orders/material-sources", requirePermission('manage_production'), async (req, res) => {
    try {
      const parsed = materialSourceSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid request body", errors: parsed.error.errors });
      }

      const { workOrderId, sourceType, sourceLocationId, sourceLocationName, quantityRequired, notes } = parsed.data;

      const result = await pool.query(`
        INSERT INTO production.operation_material_sources (
          work_order_id, source_type, source_location_id, source_location_name,
          quantity_required, notes
        ) VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING 
          id,
          work_order_id as "workOrderId",
          source_type as "sourceType",
          source_location_id as "sourceLocationId",
          source_location_name as "sourceLocationName",
          quantity_required as "quantityRequired",
          quantity_provided as "quantityProvided",
          status,
          notes,
          created_at as "createdAt"
      `, [workOrderId, sourceType, sourceLocationId || null, sourceLocationName || null, quantityRequired, notes || null]);

      res.status(201).json(result.rows[0]);
    } catch (error) {
      console.error("Error adding material source:", error);
      res.status(500).json({ message: "Failed to add material source" });
    }
  });

  // PATCH /api/production/work-orders/material-sources/:id - Update material source (quantity provided, document link)
  app.patch("/api/production/work-orders/material-sources/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid material source ID" });
      }

      const parsed = updateMaterialSourceSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid request body", errors: parsed.error.errors });
      }

      const updates: string[] = ['updated_at = NOW()'];
      const values: any[] = [];
      let paramIndex = 1;

      const { quantityProvided, documentId, documentNumber, notes } = parsed.data;

      if (quantityProvided !== undefined) {
        updates.push(`quantity_provided = LEAST(quantity_required, $${paramIndex++})`);
        values.push(quantityProvided);
        updates.push(`status = CASE 
          WHEN LEAST(quantity_required, $${paramIndex - 1}) >= quantity_required THEN 'complete'
          WHEN LEAST(quantity_required, $${paramIndex - 1}) > 0 THEN 'partial'
          ELSE 'pending'
        END`);
      }
      if (documentId !== undefined) {
        updates.push(`document_id = $${paramIndex++}`);
        values.push(documentId);
      }
      if (documentNumber !== undefined) {
        updates.push(`document_number = $${paramIndex++}`);
        values.push(documentNumber);
      }
      if (notes !== undefined) {
        updates.push(`notes = $${paramIndex++}`);
        values.push(notes);
      }

      values.push(id);

      const result = await pool.query(`
        UPDATE production.operation_material_sources
        SET ${updates.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING 
          id,
          work_order_id as "workOrderId",
          source_type as "sourceType",
          quantity_required as "quantityRequired",
          quantity_provided as "quantityProvided",
          document_id as "documentId",
          document_number as "documentNumber",
          status,
          notes,
          updated_at as "updatedAt"
      `, values);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Material source not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error updating material source:", error);
      res.status(500).json({ message: "Failed to update material source" });
    }
  });

  // DELETE /api/production/work-orders/material-sources/:id - Remove material source requirement
  app.delete("/api/production/work-orders/material-sources/:id", requirePermission('manage_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid material source ID" });
      }

      const result = await pool.query(`
        DELETE FROM production.operation_material_sources
        WHERE id = $1
        RETURNING id
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Material source not found" });
      }

      res.json({ message: "Material source removed", id });
    } catch (error) {
      console.error("Error removing material source:", error);
      res.status(500).json({ message: "Failed to remove material source" });
    }
  });

  // GET /api/production/work-orders/:id/material-readiness - Check if all material sources are ready
  app.get("/api/production/work-orders/:id/material-readiness", requirePermission('view_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const result = await pool.query(`
        SELECT 
          COUNT(*) as "totalSources",
          COUNT(CASE WHEN status = 'complete' THEN 1 END) as "completeSources",
          COUNT(CASE WHEN status = 'partial' THEN 1 END) as "partialSources",
          COUNT(CASE WHEN status = 'pending' THEN 1 END) as "pendingSources",
          SUM(quantity_required) as "totalRequired",
          SUM(quantity_provided) as "totalProvided",
          CASE 
            WHEN COUNT(*) = 0 THEN true
            WHEN COUNT(*) = COUNT(CASE WHEN status = 'complete' THEN 1 END) THEN true
            ELSE false
          END as "isReady"
        FROM production.operation_material_sources
        WHERE work_order_id = $1
      `, [workOrderId]);

      const readiness = result.rows[0];
      res.json({
        workOrderId,
        totalSources: parseInt(readiness.totalSources) || 0,
        completeSources: parseInt(readiness.completeSources) || 0,
        partialSources: parseInt(readiness.partialSources) || 0,
        pendingSources: parseInt(readiness.pendingSources) || 0,
        totalRequired: parseFloat(readiness.totalRequired) || 0,
        totalProvided: parseFloat(readiness.totalProvided) || 0,
        isReady: readiness.isReady,
      });
    } catch (error) {
      console.error("Error checking material readiness:", error);
      res.status(500).json({ message: "Failed to check material readiness" });
    }
  });

  // PATCH /api/production/work-orders/:id/operator - Update operator assignment for a work order (legacy - single operator)
  app.patch("/api/production/work-orders/:id/operator", requirePermission('manage_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const schema = z.object({
        operatorId: z.number().nullable(),
        operatorName: z.string().nullable().optional(),
      });

      const parsed = schema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid request body", errors: parsed.error.errors });
      }

      const { operatorId } = parsed.data;

      // Walidacja: sprawdź czy operator istnieje (jeśli podano ID)
      let finalOperatorId: number | null = null;
      let finalOperatorName: string | null = null;
      
      if (operatorId !== null) {
        const opResult = await pool.query(
          `SELECT id, full_name FROM production.production_operators WHERE id = $1`,
          [operatorId]
        );
        if (opResult.rows.length === 0) {
          return res.status(400).json({ message: "Operator not found", operatorId });
        }
        finalOperatorId = opResult.rows[0].id;
        finalOperatorName = opResult.rows[0].full_name;
      }

      const result = await pool.query(`
        UPDATE production.production_work_orders
        SET operator_id = $1, operator_name = $2, updated_at = NOW()
        WHERE id = $3
        RETURNING 
          id,
          operator_id as "operatorId",
          operator_name as "operatorName"
      `, [finalOperatorId, finalOperatorName, workOrderId]);

      if (result.rows.length === 0) {
        return res.status(404).json({ message: "Work order not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Error updating work order operator:", error);
      res.status(500).json({ message: "Failed to update work order operator" });
    }
  });

  // GET /api/production/work-orders/:id/operators - Get all operators assigned to a work order
  app.get("/api/production/work-orders/:id/operators", requirePermission('view_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const result = await pool.query(`
        SELECT 
          woo.id,
          woo.work_order_id as "workOrderId",
          woo.operator_id as "operatorId",
          woo.is_primary as "isPrimary",
          woo.assigned_at as "assignedAt",
          woo.notes,
          po.full_name as "operatorName",
          po.short_code as "operatorCode"
        FROM production.work_order_operators woo
        JOIN production.production_operators po ON woo.operator_id = po.id
        WHERE woo.work_order_id = $1
        ORDER BY woo.is_primary DESC, po.full_name
      `, [workOrderId]);

      res.json(result.rows);
    } catch (error) {
      console.error("Error fetching work order operators:", error);
      res.status(500).json({ message: "Failed to fetch work order operators" });
    }
  });

  // PUT /api/production/work-orders/:id/operators - Update all operators for a work order
  app.put("/api/production/work-orders/:id/operators", requirePermission('manage_production'), async (req, res) => {
    try {
      const workOrderId = parseInt(req.params.id);
      if (isNaN(workOrderId)) {
        return res.status(400).json({ message: "Invalid work order ID" });
      }

      const schema = z.object({
        operatorIds: z.array(z.number()),
        primaryOperatorId: z.number().nullable().optional(),
      });

      const parsed = schema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: "Invalid request body", errors: parsed.error.errors });
      }

      const { operatorIds, primaryOperatorId } = parsed.data;

      // Sprawdź czy work order istnieje
      const woCheck = await pool.query(
        `SELECT id FROM production.production_work_orders WHERE id = $1`,
        [workOrderId]
      );
      if (woCheck.rows.length === 0) {
        return res.status(404).json({ message: "Work order not found" });
      }

      // Walidacja: sprawdź czy wszyscy operatorzy istnieją
      if (operatorIds.length > 0) {
        const opResult = await pool.query(
          `SELECT id, full_name FROM production.production_operators WHERE id = ANY($1)`,
          [operatorIds]
        );
        if (opResult.rows.length !== operatorIds.length) {
          const foundIds = opResult.rows.map(r => r.id);
          const missingIds = operatorIds.filter(id => !foundIds.includes(id));
          return res.status(400).json({ message: "Some operators not found", missingIds });
        }
      }

      const client = await pool.connect();
      try {
        await client.query('BEGIN');

        // Usuń wszystkie obecne przypisania
        await client.query(
          `DELETE FROM production.work_order_operators WHERE work_order_id = $1`,
          [workOrderId]
        );

        // Dodaj nowe przypisania
        for (const opId of operatorIds) {
          const isPrimary = opId === primaryOperatorId;
          await client.query(`
            INSERT INTO production.work_order_operators (work_order_id, operator_id, is_primary)
            VALUES ($1, $2, $3)
          `, [workOrderId, opId, isPrimary]);
        }

        // Zaktualizuj legacy pola w work_orders dla kompatybilności
        if (primaryOperatorId) {
          const primaryOp = await client.query(
            `SELECT full_name FROM production.production_operators WHERE id = $1`,
            [primaryOperatorId]
          );
          await client.query(`
            UPDATE production.production_work_orders
            SET operator_id = $1, operator_name = $2, updated_at = NOW()
            WHERE id = $3
          `, [primaryOperatorId, primaryOp.rows[0]?.full_name || null, workOrderId]);
        } else if (operatorIds.length > 0) {
          // Pierwszy operator jako główny jeśli nie określono
          const firstOp = await client.query(
            `SELECT full_name FROM production.production_operators WHERE id = $1`,
            [operatorIds[0]]
          );
          await client.query(`
            UPDATE production.production_work_orders
            SET operator_id = $1, operator_name = $2, updated_at = NOW()
            WHERE id = $3
          `, [operatorIds[0], firstOp.rows[0]?.full_name || null, workOrderId]);
        } else {
          // Brak operatorów - wyczyść legacy pola
          await client.query(`
            UPDATE production.production_work_orders
            SET operator_id = NULL, operator_name = NULL, updated_at = NOW()
            WHERE id = $1
          `, [workOrderId]);
        }

        await client.query('COMMIT');

        // Pobierz zaktualizowaną listę operatorów
        const result = await client.query(`
          SELECT 
            woo.id,
            woo.work_order_id as "workOrderId",
            woo.operator_id as "operatorId",
            woo.is_primary as "isPrimary",
            woo.assigned_at as "assignedAt",
            woo.notes,
            po.full_name as "operatorName",
            po.short_code as "operatorCode"
          FROM production.work_order_operators woo
          JOIN production.production_operators po ON woo.operator_id = po.id
          WHERE woo.work_order_id = $1
          ORDER BY woo.is_primary DESC, po.full_name
        `, [workOrderId]);

        res.json(result.rows);
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Error updating work order operators:", error);
      res.status(500).json({ message: "Failed to update work order operators" });
    }
  });

  // POST /api/production/work-orders/bulk-assign - Bulk assign operators to work orders by operation type
  app.post("/api/production/work-orders/bulk-assign", requirePermission('manage_production'), async (req, res) => {
    try {
      const schema = z.object({
        planId: z.number().int().positive(),
        operationCode: z.string().optional().nullable(),
        workCenterId: z.number().int().positive().optional().nullable(),
        colorCode: z.string().optional().nullable(),
        operatorIds: z.array(z.number().int().positive()).min(1, "Musisz wybrać co najmniej jednego operatora"),
        primaryOperatorId: z.number().int().positive().optional().nullable(),
        mode: z.enum(['overwrite', 'fill_empty']).default('overwrite'),
      }).refine(
        (data) => !data.primaryOperatorId || data.operatorIds.includes(data.primaryOperatorId),
        { message: "Główny operator musi być na liście wybranych operatorów" }
      );

      const parsed = schema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({ message: parsed.error.errors[0]?.message || "Invalid request body", errors: parsed.error.errors });
      }

      const { planId, operationCode, workCenterId, colorCode, operatorIds, primaryOperatorId, mode } = parsed.data;

      // Build query to find matching work orders
      let query = `
        SELECT pwo.id
        FROM production.production_work_orders pwo
        JOIN production.production_orders po ON pwo.production_order_id = po.id
        WHERE po.plan_id = $1
      `;
      const params: any[] = [planId];
      let paramIndex = 2;

      if (operationCode) {
        query += ` AND pwo.operation_code = $${paramIndex++}`;
        params.push(operationCode);
      }
      if (workCenterId) {
        query += ` AND pwo.work_center_id = $${paramIndex++}`;
        params.push(workCenterId);
      }
      if (colorCode) {
        query += ` AND po.color_code = $${paramIndex++}`;
        params.push(colorCode);
      }

      // If mode is 'fill_empty', only get work orders without any operators
      if (mode === 'fill_empty') {
        query += ` AND NOT EXISTS (
          SELECT 1 FROM production.work_order_operators woo WHERE woo.work_order_id = pwo.id
        )`;
      }

      const workOrdersResult = await pool.query(query, params);
      const workOrderIds = workOrdersResult.rows.map(r => r.id);

      if (workOrderIds.length === 0) {
        return res.json({ 
          success: true, 
          message: "Nie znaleziono zleceń do przypisania", 
          assignedCount: 0 
        });
      }

      // Validate operators exist
      if (operatorIds.length > 0) {
        const opResult = await pool.query(
          `SELECT id FROM production.production_operators WHERE id = ANY($1)`,
          [operatorIds]
        );
        if (opResult.rows.length !== operatorIds.length) {
          return res.status(400).json({ message: "Niektórzy operatorzy nie zostali znalezieni" });
        }
      }

      const client = await pool.connect();
      try {
        await client.query('BEGIN');

        for (const workOrderId of workOrderIds) {
          // Delete existing assignments
          await client.query(
            `DELETE FROM production.work_order_operators WHERE work_order_id = $1`,
            [workOrderId]
          );

          // Add new assignments
          for (const opId of operatorIds) {
            const isPrimary = opId === primaryOperatorId || (operatorIds.length === 1 && !primaryOperatorId);
            await client.query(`
              INSERT INTO production.work_order_operators (work_order_id, operator_id, is_primary)
              VALUES ($1, $2, $3)
            `, [workOrderId, opId, isPrimary]);
          }

          // Update legacy fields
          const primaryId = primaryOperatorId || (operatorIds.length > 0 ? operatorIds[0] : null);
          if (primaryId) {
            const primaryOp = await client.query(
              `SELECT full_name FROM production.production_operators WHERE id = $1`,
              [primaryId]
            );
            await client.query(`
              UPDATE production.production_work_orders
              SET operator_id = $1, operator_name = $2, updated_at = NOW()
              WHERE id = $3
            `, [primaryId, primaryOp.rows[0]?.full_name || null, workOrderId]);
          } else {
            await client.query(`
              UPDATE production.production_work_orders
              SET operator_id = NULL, operator_name = NULL, updated_at = NOW()
              WHERE id = $1
            `, [workOrderId]);
          }
        }

        await client.query('COMMIT');

        res.json({ 
          success: true, 
          message: `Przypisano operatorów do ${workOrderIds.length} zleceń`,
          assignedCount: workOrderIds.length 
        });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Error bulk assigning operators:", error);
      res.status(500).json({ message: "Failed to bulk assign operators" });
    }
  });

  // GET /api/production/orders/:id/pdf - Generate PDF production card
  app.get("/api/production/orders/:id/pdf", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const pdfBuffer = await generateProductionCardPDF(pool, id);
      if (!pdfBuffer) {
        return res.status(404).json({ message: "Production order not found" });
      }

      // Get order number for filename
      const order = await ordersService.getOrderById(pool, id);
      const filename = order?.orderNumber 
        ? `${order.orderNumber.replace(/[^a-zA-Z0-9-]/g, '_')}_karta.pdf`
        : `ZLP-${id}_karta.pdf`;

      res.setHeader('Content-Type', 'application/pdf');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      res.setHeader('Content-Length', pdfBuffer.length);
      res.send(pdfBuffer);
    } catch (error) {
      console.error("Error generating production card PDF:", error);
      res.status(500).json({ message: "Failed to generate PDF" });
    }
  });

  // GET /api/production/orders/:id/csv - Generate CSV for saw machine
  app.get("/api/production/orders/:id/csv", requirePermission('view_production'), async (req, res) => {
    try {
      const id = parseInt(req.params.id);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid order ID" });
      }

      const csvContent = await generateSawCSV(pool, id);
      if (csvContent === null) {
        return res.status(404).json({ message: "Production order not found" });
      }

      // Get order details for filename
      const order = await ordersService.getOrderById(pool, id);
      const colorCode = order?.colorCode ? `_${order.colorCode.replace(/\s+/g, '_').toUpperCase()}` : '';
      const filename = order?.orderNumber 
        ? `${order.orderNumber.replace(/[^a-zA-Z0-9-]/g, '_')}${colorCode}.csv`
        : `ZLP-${id}.csv`;

      res.setHeader('Content-Type', 'text/csv; charset=utf-8');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      res.send(csvContent);
    } catch (error) {
      console.error("Error generating saw CSV:", error);
      res.status(500).json({ message: "Failed to generate CSV" });
    }
  });

}
