import type { Express } from "express";
import { createServer, type Server } from "http";
import { EventEmitter } from 'events';
import { storage } from "./storage";
import axios from "axios";
import { randomBytes } from "crypto";
import path from "path";
import fs from "fs/promises";
import multer from "multer";
import Papa from "papaparse";
import PDFDocument from "pdfkit";
import { logError, logAllegroError, logShoperError, logOdooError, logSyncError, logWebhookError, logAuthError, cleanupOldErrorLogs } from "./error-logger";
import { pool, saveOrderToPostgres, testConnection, getOrdersFromPostgres, getOrderFromPostgres, getOrderByNumberFromPostgres, createSyncLog, completeSyncLog, getSyncLogs, saveOrderToCommerce, saveShoperOrderToPostgres, getRecentlyUpdatedOrders, getRecentOrderChanges, logOrderChange, getSyncSettingsFromPostgres, createOrUpdateSyncSettingsInPostgres, getOrderStatistics, getOrdersChartData, getOrdersPeriodSummary, getTodayDetailedStats, getAllegroConnectionFromPostgres, saveAllegroConnectionToPostgres, updateAllegroConnectionTokens, getUserByUsername, getUserByEmail, createUser, createPasswordResetToken, getPasswordResetToken, markTokenAsUsed, updateUserPassword, getAllUsers, updateUserRole, updateUserPermissions, updateUserStatus, deleteUser, getAllTemplateCategories, getTemplateCategoryById, createTemplateCategory, updateTemplateCategory, deleteTemplateCategory, getAllDescriptionTemplates, getDescriptionTemplateById, createDescriptionTemplate, updateDescriptionTemplate, deleteDescriptionTemplate, getProductTemplates, assignTemplateToProduct, updateProductTemplateConfig, removeTemplateFromProduct, createAiGenerationRequest, updateAiGenerationRequest, getAiGenerationHistory } from "./postgres";
import { setupAuth, isAuthenticated, hashPassword, comparePasswords } from "./auth";
import { hasPermission, type Permission } from "../shared/permissions";
import passport from "passport";
import { requireApiToken, logApiRequest, generateApiToken, hashToken } from "./api-auth";
import { getWebhookLogs, triggerOrderWebhooks } from "./webhooks";
import swaggerUi from "swagger-ui-express";
import { swaggerSpec } from "./swagger";
import type { DictionaryType, ProductCreatorDictionary } from "../shared/schema";
import { insertUserSavedFilterSchema, insertPackagingMaterialSchema } from "../shared/schema";
import { z } from "zod";
import { renderAccessoryGridAsJPG } from "./accessory-grid-renderer";

const ALLEGRO_AUTH_URL = "https://allegro.pl/auth/oauth/authorize";
const ALLEGRO_TOKEN_URL = "https://allegro.pl/auth/oauth/token";
const ALLEGRO_API_URL = "https://api.allegro.pl";

interface AllegroTokenResponse {
  access_token: string;
  refresh_token: string;
  expires_in: number;
  token_type: string;
}

interface AllegroOrderItem {
  id: string;
  buyer: {
    login: string;
    email?: string;
  };
  payment: {
    paidAmount?: {
      amount: string;
      currency: string;
    };
    status?: string;
  };
  fulfillment?: {
    status?: string;
  };
  lineItems: Array<{ id: string }>;
  updatedAt: string;
  summary?: {
    totalToPay?: {
      amount: string;
      currency: string;
    };
  };
}

let syncInterval: NodeJS.Timeout | null = null;
let isCurrentlySyncing = false;
let shoperSyncInterval: NodeJS.Timeout | null = null;
let recentUpdatesInterval: NodeJS.Timeout | null = null;
let odooSyncInterval: NodeJS.Timeout | null = null;
let feesSyncInterval: NodeJS.Timeout | null = null;
let errorLogsCleanupInterval: NodeJS.Timeout | null = null;
let autoHealingInterval: NodeJS.Timeout | null = null;
const oauthStates = new Map<string, { timestamp: number }>();

// Generation logs streaming with SSE
const logStreamEmitter = new EventEmitter();
logStreamEmitter.setMaxListeners(100); // Increase max listeners for multiple concurrent generations

// Helper function to emit logs to SSE clients
function emitGenerationLog(sessionId: string, message: string, type: 'info' | 'success' | 'error' | 'warning' = 'info') {
  logStreamEmitter.emit(sessionId, {
    timestamp: new Date().toISOString(),
    message,
    type,
  });
}

// Helper function to extract accessory tags from template HTML
function extractAccessoryTags(templateHtml: string): string[] {
  // Updated regex to support optional :gridN or :gridN- parameter
  // Matches: {{akcesorium-CODE}}, {{akcesorium-CODE:grid4}}, {{akcesorium-CODE:grid3-}}, etc.
  const tagRegex = /\{\{(?:akcesorium|okucia|akcesoria)-([a-zA-Z0-9_-]+)(?::grid\d+-?)?\}\}/gi;
  const matches = Array.from(templateHtml.matchAll(tagRegex));
  const codes = new Set<string>();
  
  for (const match of matches) {
    if (match[1]) {
      // Keep original case - codes in database are lowercase
      codes.add(match[1].toLowerCase());
    }
  }
  
  return Array.from(codes);
}

// Helper function to extract image tags from template HTML
// Matches: {img1 w=100}, {img2 w=75}, {img3 w=50}, {img4 w=25}, etc.
function extractImageTags(templateHtml: string): Array<{ index: number; width: number }> {
  const tagRegex = /\{img(\d+)\s+w=(\d+)\}/gi;
  const matches = Array.from(templateHtml.matchAll(tagRegex));
  const tags: Array<{ index: number; width: number }> = [];
  
  for (const match of matches) {
    if (match[1] && match[2]) {
      const index = parseInt(match[1], 10);
      const width = parseInt(match[2], 10);
      // Validate width is one of the allowed values: 100, 75, 50, 25
      if ([100, 75, 50, 25].includes(width)) {
        tags.push({ index, width });
      }
    }
  }
  
  return tags;
}

// Helper function to safely parse numeric values, returning 0 for NaN/invalid values
function safeParseNumber(value: string | number | null | undefined, defaultValue: number = 0): number {
  if (value === null || value === undefined || value === '') {
    return defaultValue;
  }
  const parsed = typeof value === 'number' ? value : parseFloat(value.toString());
  return isNaN(parsed) ? defaultValue : parsed;
}

// Helper function to process accessory tags and render them as HTML or JPG
async function processAccessoryTags(
  html: string, 
  templateContext: any,
  sessionId?: string
): Promise<string> {
  const accessoryTagRegex = /\{\{(?:akcesorium|okucia|akcesoria)-([a-zA-Z0-9_-]+)(?::([a-zA-Z0-9-]+))?\}\}/g;
  const matches = Array.from(html.matchAll(accessoryTagRegex));
  
  if (matches.length === 0) {
    return html;
  }
  
  let processedHtml = html;
  
  // Process each tag
  for (const match of matches) {
    const [fullMatch, groupCode, gridParam] = match;
    
    console.log(`🔍 Processing accessory tag: ${fullMatch}`);
    
    const group = templateContext.accessories.groups.find(
      (g: any) => g.groupCode === groupCode || g.code === groupCode
    );
    
    if (!group || !group.items || group.items.length === 0) {
      console.log(`⚠️  Accessory group "${groupCode}" not found or empty`);
      processedHtml = processedHtml.replace(fullMatch, '');
      continue;
    }
    
    // JPG rendering for accessory grids (enabled)
    const useJpgRendering = true;
    
    // Check if grid parameter exists (grid3, grid4, grid2-2, etc.)
    if (gridParam && gridParam.startsWith('grid') && useJpgRendering) {
      // Calculate how many items we need for this grid
      let itemCount = 0;
      if (gridParam.endsWith('-')) {
        // Infinite grid - use all items
        itemCount = group.items.length;
      } else if (gridParam.includes('-')) {
        // Format: grid2-2 (rows-cols)
        const [rows, cols] = gridParam.replace('grid', '').split('-').map(Number);
        itemCount = rows * cols;
      } else {
        // Format: grid3, grid4 (single row with N columns)
        itemCount = parseInt(gridParam.replace('grid', ''));
      }
      
      // Take only the items we need for this grid
      const itemsForGrid = group.items.slice(0, itemCount);
      
      console.log(`📸 Rendering accessory grid as JPG: ${groupCode}:${gridParam} (${itemsForGrid.length} of ${group.items.length} items)`);
      if (sessionId) {
        emitGenerationLog(sessionId, `   📸 Renderowanie gridu JPG: ${groupCode}:${gridParam} (${itemsForGrid.length} itemów)`, 'info');
      }
      
      try {
        // Render grid as JPG with only the items we need
        const imageUrl = await renderAccessoryGridAsJPG(itemsForGrid, gridParam);
        
        // Replace tag with <img> tag
        const imgTag = `<img src="${imageUrl}" alt="${group.groupName || group.name} - ${gridParam}" style="width: 100%; max-width: 800px; height: auto; display: block; margin: 20px auto; border-radius: 4px;" />`;
        processedHtml = processedHtml.replace(fullMatch, imgTag);
        
        console.log(`✅ Grid JPG rendered: ${imageUrl}`);
        if (sessionId) {
          emitGenerationLog(sessionId, `   ✅ Grid JPG wygenerowany`, 'success');
        }
      } catch (error) {
        console.error(`❌ Failed to render grid JPG for ${groupCode}:${gridParam}:`, error);
        if (sessionId) {
          emitGenerationLog(sessionId, `   ❌ Błąd renderowania gridu: ${error instanceof Error ? error.message : 'Nieznany błąd'}`, 'error');
        }
        processedHtml = processedHtml.replace(fullMatch, '');
      }
    } else {
      // No JPG rendering - render HTML (grid or list)
      console.log(`📝 Rendering accessory HTML: ${groupCode}${gridParam ? ` (${gridParam})` : ''}`);
      if (sessionId) {
        emitGenerationLog(sessionId, `   📝 Renderowanie HTML: ${groupCode}${gridParam ? ` (${gridParam})` : ''}`, 'info');
      }
      
      // Parse grid parameter to determine columns
      let columns = 1;
      let itemsToShow = group.items;
      
      if (gridParam && gridParam.startsWith('grid')) {
        if (gridParam.endsWith('-')) {
          // Infinite grid - show all items, parse columns from grid format
          const gridNum = gridParam.replace('grid', '').replace('-', '');
          columns = gridNum ? parseInt(gridNum, 10) : 3; // default to 3 columns if not specified
        } else if (gridParam.includes('-')) {
          // Format: grid2-2 (rows-cols)
          const [rows, cols] = gridParam.replace('grid', '').split('-').map(Number);
          columns = cols;
          const itemCount = rows * cols;
          itemsToShow = group.items.slice(0, itemCount);
        } else {
          // Format: grid3, grid4 (single row with N columns)
          columns = parseInt(gridParam.replace('grid', ''));
          itemsToShow = group.items.slice(0, columns);
        }
      }
      
      const itemWidth = columns === 1 ? '100%' : `${(100 / columns) - 2}%`;
      
      const htmlContent = `
        <div class="accessory-group" data-group="${groupCode}" style="margin: 20px 0;">
          <h3 style="margin-bottom: 15px; font-size: 1.2em;">${group.groupName || group.name}</h3>
          <div class="accessory-items" style="display: flex; flex-wrap: wrap; gap: 10px; margin: 0 -5px;">
            ${itemsToShow.map((item: any) => `
              <div class="accessory-item" style="width: ${itemWidth}; min-width: ${columns > 3 ? '150px' : '200px'}; box-sizing: border-box; padding: 10px; border: 1px solid #e0e0e0; border-radius: 4px; background: #fafafa;">
                ${item.imageUrl ? `
                  <img src="${item.imageUrl}" alt="${item.name}" class="accessory-image" style="width: 100%; height: auto; display: block; margin-bottom: 10px; border-radius: 4px; object-fit: contain; max-height: 300px;" />
                ` : ''}
                <div style="text-align: center;">
                  <h4 style="margin: 0 0 5px 0; font-size: 0.9em; font-weight: 600;">${item.name}</h4>
                  ${item.code ? `<p style="margin: 0 0 5px 0; font-size: 0.8em; color: #666;">(${item.code})</p>` : ''}
                  ${item.description ? `<p style="margin: 0; font-size: 0.8em; color: #888;">${item.description}</p>` : ''}
                </div>
              </div>
            `).join('')}
          </div>
        </div>
      `.trim();
      
      processedHtml = processedHtml.replace(fullMatch, htmlContent);
    }
  }
  
  return processedHtml;
}

// Helper function to build template context with product data, accessories, AI descriptions, and size tables
async function buildTemplateContext(matrix: any, variant: {
  length: number | null;
  width: number | null;
  height: number | null;
  color: string;
  productName: string;
  productGroup: string | null;
  finalPrice: number | null;
  sizeStr: string;
}, templateHtml?: string, sessionId?: string): Promise<any> {
  const { generateProductDescription } = await import('./ai-service');
  
  // 1. Build base product data
  const productData = {
    alpmaCode: '', // Will be set after product creation
    sku: '', // Will be set after product creation
    title: variant.productName,
    color: variant.color || undefined,
    dimensions: {
      width: variant.width !== null ? variant.width : undefined,
      depth: variant.length !== null ? variant.length : undefined,
      height: variant.height !== null ? variant.height : undefined,
    },
    price: variant.finalPrice || matrix.defaultPrice,
    productType: matrix.productType || undefined,
    productGroup: variant.productGroup || undefined,
    doors: matrix.doors || undefined,
    legs: matrix.legs || undefined,
    material: matrix.material || undefined,
  };
  
  // 2. Build size table from matrix dimensions
  const lengths = matrix.lengths || [];
  const widths = matrix.widths || [];
  const heights = matrix.heights || [];
  
  const sizeTableRows = [];
  const maxRows = Math.max(lengths.length, widths.length, heights.length);
  
  for (let i = 0; i < maxRows; i++) {
    const l = lengths[i] ? parseFloat(lengths[i]) : null;
    const w = widths[i] ? parseFloat(widths[i]) : null;
    const h = heights[i] ? parseFloat(heights[i]) : null;
    
    if (l !== null || w !== null || h !== null) {
      sizeTableRows.push({
        variant: `Wariant ${i + 1}`,
        values: [
          l ? `${l / 10}cm` : '-',
          w ? `${w / 10}cm` : '-',
          h ? `${h / 10}cm` : '-',
        ]
      });
    }
  }
  
  const sizeTable = {
    headers: ['Długość', 'Szerokość', 'Wysokość'],
    rows: sizeTableRows
  };
  
  // 3. Extract accessory tags from template HTML and fetch corresponding groups
  let accessoriesData: { groups: any[] } = { groups: [] };
  
  if (templateHtml) {
    try {
      const accessoryCodes = extractAccessoryTags(templateHtml);
      
      if (accessoryCodes.length > 0) {
        console.log(`🏷️  Found ${accessoryCodes.length} accessory tags in template: ${accessoryCodes.join(', ')}`);
        
        // Fetch accessory groups by codes found in template
        const result = await pool.query(`
          SELECT 
            ag.id as group_id,
            ag.name as group_name,
            ag.code as group_code,
            ag.category as group_category,
            ag.description as group_description,
            json_agg(
              json_build_object(
                'id', a.id,
                'name', a.name,
                'code', a.code,
                'alpmaCode', a.alpma_code,
                'description', a.description,
                'imageUrl', a.image_url,
                'displayOrder', a.display_order
              ) ORDER BY a.display_order
            ) FILTER (WHERE a.id IS NOT NULL) as items
          FROM catalog.accessory_groups ag
          LEFT JOIN catalog.accessories a ON a.group_id = ag.id AND a.is_active = true
          WHERE ag.is_active = true AND LOWER(ag.code) = ANY($1::text[])
          GROUP BY ag.id, ag.name, ag.code, ag.category, ag.description, ag.display_order
          ORDER BY ag.display_order
        `, [accessoryCodes]);
        
        accessoriesData.groups = result.rows.map((row: any) => ({
          groupId: row.group_id,
          groupName: row.group_name,
          groupCode: row.group_code,
          code: row.group_code, // Alias for compatibility
          name: row.group_name, // Alias for compatibility
          category: row.group_category,
          description: row.group_description, // Group description for AI context
          items: row.items || []
        }));
        
        console.log(`📦 Loaded ${accessoriesData.groups.length} accessory groups from template tags`);
      } else {
        console.log(`ℹ️  No accessory tags found in template`);
      }
    } catch (error) {
      console.error('❌ Error fetching accessories from template tags:', error);
    }
  }
  
  // 4. Generate AI descriptions - CALL ONCE for all sections, not separately!
  const aiDescriptions: any = {};
  
  if (matrix.aiConfig && typeof matrix.aiConfig === 'object') {
    const aiConfig = matrix.aiConfig;
    
    // Check if ANY section is enabled
    const hasEnabledSections = Object.entries(aiConfig).some(([sectionKey, sectionConfig]) => {
      const config = sectionConfig as any;
      // Skip warranty if disabled
      if (sectionKey === 'warranty' && matrix.hasWarranty === false) return false;
      return config.enabled && config.prompt;
    });
    
    if (hasEnabledSections) {
      try {
        console.log(`🤖 Generating AI description for all sections (single call)`);
        if (sessionId) {
          const enabledCount = Object.values(aiConfig).filter((c: any) => c.enabled && c.prompt).length;
          emitGenerationLog(sessionId, `   🤖 Generowanie AI (${enabledCount} sekcji)...`, 'info');
        }
        
        // Call AI ONCE - it will generate ALL sections in one response
        const result = await generateProductDescription({
          productData,
          templateHtml: '', // Don't use templateHtml here - AI uses global prompts from settings
          tone: 'professional',
          language: 'pl',
          maxLength: 500,
          accessories: accessoriesData,
          customPrompts: aiConfig, // Pass custom prompts from matrix
        });
        
        // Parse the result to extract individual sections
        // AI returns HTML with all sections in one response
        const fullDescription = result.description;
        
        console.log(`✅ AI description generated (${result.totalTokens} tokens)`);
        console.log(`📄 Full description length: ${fullDescription.length} chars`);
        console.log(`📋 Full description preview:\n${fullDescription.substring(0, 500)}...`);
        if (sessionId) {
          emitGenerationLog(sessionId, `   ✅ AI wygenerowane (${result.totalTokens} tokenów, ${fullDescription.length} znaków)`, 'success');
        }
        
        // CRITICAL: Extract individual sections from the full HTML
        // AI generates sections with specific headers like:
        // <div><h2>Wprowadzenie produktu</h2>...</div>
        // <div><h2>Cechy i zalety</h2>...</div>
        // etc.
        
        // Map section keys to likely header patterns in AI output
        // More flexible patterns that handle various HTML structures
        const sectionPatterns: Record<string, RegExp[]> = {
          intro: [
            /<h2[^>]*>Wprowadzenie[^<]*<\/h2>([\s\S]*?)(?=<h2|$)/i,
            /<div[^>]*>\s*<h2[^>]*>Wprowadzenie[^<]*<\/h2>([\s\S]*?)<\/div>/i,
            /<p[^>]*class="[^"]*intro[^"]*"[^>]*>([\s\S]*?)<\/p>/i,
          ],
          features: [
            /<h2[^>]*>Cechy[^<]*<\/h2>([\s\S]*?)(?=<h2|$)/i,
            /<div[^>]*>\s*<h2[^>]*>Cechy[^<]*<\/h2>([\s\S]*?)<\/div>/i,
            /<ul[^>]*>([\s\S]*?)<\/ul>/i,
          ],
          safety: [
            /<h2[^>]*>Bezpieczeństwo[^<]*<\/h2>([\s\S]*?)(?=<h2|$)/i,
            /<div[^>]*>\s*<h2[^>]*>Bezpieczeństwo[^<]*<\/h2>([\s\S]*?)<\/div>/i,
            /<p[^>]*class="[^"]*safety[^"]*"[^>]*>([\s\S]*?)<\/p>/i,
          ],
          care: [
            /<h2[^>]*>Pielęgnacja[^<]*<\/h2>([\s\S]*?)(?=<h2|$)/i,
            /<div[^>]*>\s*<h2[^>]*>Pielęgnacja[^<]*<\/h2>([\s\S]*?)<\/div>/i,
            /<p[^>]*class="[^"]*care[^"]*"[^>]*>([\s\S]*?)<\/p>/i,
          ],
          warranty: [
            /<h2[^>]*>Gwarancja[^<]*<\/h2>([\s\S]*?)(?=<h2|$)/i,
            /<div[^>]*>\s*<h2[^>]*>Gwarancja[^<]*<\/h2>([\s\S]*?)<\/div>/i,
            /<p[^>]*class="[^"]*warranty[^"]*"[^>]*>([\s\S]*?)<\/p>/i,
          ],
        };
        
        // Try to extract each section
        for (const [sectionKey, sectionConfig] of Object.entries(aiConfig)) {
          const config = sectionConfig as any;
          
          // Skip warranty if disabled
          if (sectionKey === 'warranty' && matrix.hasWarranty === false) {
            console.log(`⏭️  Skipping warranty section (hasWarranty=false)`);
            continue;
          }
          
          if (config.enabled && config.prompt) {
            const patterns = sectionPatterns[sectionKey];
            let extracted = '';
            
            if (patterns) {
              for (const pattern of patterns) {
                const match = fullDescription.match(pattern);
                if (match && match[1]) {
                  extracted = match[1].trim();
                  console.log(`✂️  Extracted section "${sectionKey}" (${extracted.length} chars): ${extracted.substring(0, 100)}...`);
                  break;
                }
              }
            }
            
            // If extraction failed, show debug info and use empty string instead of full description
            if (!extracted) {
              console.warn(`⚠️  Could not extract section "${sectionKey}" from AI response`);
              console.warn(`🔍  Tested ${patterns?.length || 0} patterns without match`);
              console.warn(`📝  Full description structure check - looking for headers...`);
              // Extract all h2 headers from fullDescription for debugging
              const h2Matches = fullDescription.match(/<h2[^>]*>([^<]+)<\/h2>/gi);
              if (h2Matches) {
                console.warn(`📑  Found ${h2Matches.length} h2 headers:`, h2Matches.map(h => h.replace(/<[^>]*>/g, '')));
              } else {
                console.warn(`📑  No h2 headers found in AI response`);
              }
              extracted = `<!-- ${sectionKey} section not generated by AI -->`;
            }
            
            aiDescriptions[sectionKey] = extracted;
          }
        }
      } catch (error) {
        console.error(`❌ Error generating AI descriptions:`, error);
        // Set empty strings for all sections
        for (const [sectionKey, sectionConfig] of Object.entries(aiConfig)) {
          const config = sectionConfig as any;
          if (config.enabled) {
            aiDescriptions[sectionKey] = '';
          }
        }
      }
    }
  }
  
  // 5. Build complete template context with FLAT AI tags (using hyphens)
  const flatContext: any = {
    product: productData,
    variant: {
      length: variant.length,
      width: variant.width,
      height: variant.height,
      basePrice: matrix.defaultPrice ? parseFloat(matrix.defaultPrice) : variant.finalPrice,
    },
    color: variant.color,
    dimensions: variant.sizeStr,
    price: variant.finalPrice,
    accessories: accessoriesData,
    sizeTable,
    // Legacy compatibility
    ai_description: aiDescriptions.main || aiDescriptions.intro || '',
  };
  
  // Add AI sections as flat properties with hyphen notation: ai-intro, ai-features, etc.
  for (const [sectionKey, sectionValue] of Object.entries(aiDescriptions)) {
    flatContext[`ai-${sectionKey}`] = sectionValue;
  }
  
  return flatContext;
}

// Helper function to build combination key for product deduplication
// Format: "productType:length×width×height:color" (no options) or "productType:length×width×height:color:option1,option2,..." (with options)
// Ensures consistent key generation with deterministic handling of missing values
function buildCombinationKey(
  productType: string | null,
  length: number | null,
  width: number | null,
  height: number | null,
  color: string | null,
  colorOptions: string[]
): string {
  const comboParts = [
    length !== null ? length.toString() : 'null',
    width !== null ? width.toString() : 'null',
    height !== null ? height.toString() : 'null',
  ];
  
  // Build base key
  let key = `${productType || 'NO_TYPE'}:${comboParts.join('×')}:${color || 'no-color'}`;
  
  // Add color options ONLY if they exist (non-empty array)
  if (colorOptions && colorOptions.length > 0) {
    // Sort color options for deterministic key (e.g., ["UNIT1", "UNIT2"] always same order)
    const sortedOptions = colorOptions.slice().sort().join(',');
    key += `:${sortedOptions}`;
  }
  
  return key;
}

// Helper function to build legacy combination key for backwards compatibility
// Format: "length×width×height:color:colorIndex"
// Used for fallback lookup of existing products created with old key format
function buildLegacyCombinationKey(
  length: number | null,
  width: number | null,
  height: number | null,
  color: string | null,
  colorIndex: number
): string {
  const comboParts = [
    length !== null ? length.toString() : 'null',
    width !== null ? width.toString() : 'null',
    height !== null ? height.toString() : 'null',
  ];
  return `${comboParts.join('×')}:${color || 'no-color'}:${colorIndex}`;
}

// Helper function to build stable matrix variant ID
// Format: "matrixId-length-width-height-color" or "matrixId-length-width-height-color-option1-option2-..."
// This ID is stable and survives SKU changes, dimension reordering, and other edits
// Used as primary lookup key for product deduplication during regeneration
function buildMatrixVariantId(
  matrixId: number,
  length: number | null,
  width: number | null,
  height: number | null,
  color: string | null,
  colorOptions?: string[]
): string {
  const dimParts = [
    length !== null ? length.toString() : 'null',
    width !== null ? width.toString() : 'null',
    height !== null ? height.toString() : 'null',
  ];
  
  let id = `${matrixId}-${dimParts.join('-')}-${color || 'no-color'}`;
  
  // Add color options if they exist (sorted for deterministic ID)
  if (colorOptions && colorOptions.length > 0) {
    const sortedOptions = colorOptions.slice().sort().join('-');
    id += `-${sortedOptions}`;
  }
  
  return id;
}

// Helper function to generate products from matrix
async function generateProductsFromMatrix(matrix: any, sessionId?: string): Promise<any[]> {
  const { generateProductDescription } = await import('./ai-service');
  
  const lengths = matrix.lengths || [];
  const widths = matrix.widths || [];
  const heights = matrix.heights || [];
  const colors = matrix.colors || [];
  const colorImages = matrix.colorImages || {};
  const productGroups = matrix.productGroups || [];
  const priceModifiers = matrix.priceModifiers || [];
  const priceModifierTypes = (matrix as any).priceModifierTypes || [];
  
  const generatedProducts: any[] = [];
  
  // Parse and deduplicate dimensions
  const uniqueLengths: (number | null)[] = lengths.length > 0 
    ? Array.from(new Set(lengths.map((l: string) => parseFloat(l)).filter((l: number) => !isNaN(l))))
    : [null];
  const uniqueWidths: (number | null)[] = widths.length > 0
    ? Array.from(new Set(widths.map((w: string) => parseFloat(w)).filter((w: number) => !isNaN(w))))
    : [null];
  const uniqueHeights: (number | null)[] = heights.length > 0
    ? Array.from(new Set(heights.map((h: string) => parseFloat(h)).filter((h: number) => !isNaN(h))))
    : [null];
  
  // Use selected colors (INDICES) if available, otherwise use all color indices
  // selectedColors now contains indices as strings (e.g., ["0", "3", "5"])
  const selectedColorsArray = matrix.selectedColors || [];
  const colorIndices: number[] = selectedColorsArray.length > 0
    ? selectedColorsArray.map((s: string) => parseInt(s, 10)).filter((n: number) => !isNaN(n) && n >= 0 && n < colors.length)
    : (colors.length > 0 ? colors.map((_: any, idx: number) => idx) : []);
  
  const totalCombinations = uniqueLengths.length * uniqueWidths.length * uniqueHeights.length * (colorIndices.length || 1);
  console.log(`📐 Generating Cartesian product: ${uniqueLengths.length} lengths × ${uniqueWidths.length} widths × ${uniqueHeights.length} heights × ${colorIndices.length || 1} colors`);
  console.log(`  Lengths: [${uniqueLengths.join(', ')}]`);
  console.log(`  Widths: [${uniqueWidths.join(', ')}]`);
  console.log(`  Heights: [${uniqueHeights.join(', ')}]`);
  console.log(`  Color indices to generate: [${colorIndices.join(', ')}]`);
  console.log(`  Color names: [${colorIndices.map(idx => colors[idx]).join(', ')}]`);
  if (selectedColorsArray.length > 0) {
    console.log(`  ✓ Using ${colorIndices.length} selected color indices (total available: ${colors.length})`);
  } else {
    console.log(`  ✓ Using all ${colors.length} colors (no selection filter)`);
  }
  
  if (sessionId) {
    emitGenerationLog(sessionId, `📐 Rozpoczynanie generowania ${totalCombinations} kombinacji produktów...`, 'info');
    emitGenerationLog(sessionId, `   ${uniqueLengths.length} długości × ${uniqueWidths.length} szerokości × ${uniqueHeights.length} wysokości × ${colorIndices.length || 1} kolorów`, 'info');
    if (selectedColorsArray.length > 0) {
      emitGenerationLog(sessionId, `   ✓ Generowanie tylko dla ${colorIndices.length} zaznaczonych kolorów`, 'info');
    }
  }
  
  // Fetch shortNames for doors and legs from dictionaries
  const shortNameMap: Record<string, string> = {};
  const codesToFetch: string[] = [];
  
  if (matrix.doors) {
    codesToFetch.push(matrix.doors);
  }
  if (matrix.legs) {
    codesToFetch.push(matrix.legs);
  }
  
  if (codesToFetch.length > 0) {
    const result = await pool.query(
      `SELECT code, short_name 
       FROM product_creator.dictionaries 
       WHERE code = ANY($1)`,
      [codesToFetch]
    );
    result.rows.forEach((row: any) => {
      shortNameMap[row.code] = row.short_name || row.code;
    });
  }
  
  // Helper function to find matching row index for dimension combination
  // Normalizes and compares dimensions to handle edge cases like empty strings, whitespace, etc.
  const findRowIndex = (length: number | null, width: number | null, height: number | null): number | null => {
    const normalizeValue = (val: string | number | null | undefined): number | null => {
      if (val === null || val === undefined) return null;
      if (typeof val === 'number') return val;
      if (typeof val === 'string') {
        if (val.trim() === '') return null;
        const num = parseFloat(val.trim());
        return isNaN(num) ? null : num;
      }
      return null;
    };
    
    const compareValues = (a: number | null, b: number | null): boolean => {
      if (a === null && b === null) return true;
      if (a === null || b === null) return false;
      return Math.abs(a - b) < 0.001; // Use epsilon comparison for floats
    };
    
    for (let i = 0; i < Math.max(lengths.length, widths.length, heights.length); i++) {
      const rowLength = normalizeValue(lengths[i]);
      const rowWidth = normalizeValue(widths[i]);
      const rowHeight = normalizeValue(heights[i]);
      
      if (compareValues(rowLength, length) && 
          compareValues(rowWidth, width) && 
          compareValues(rowHeight, height)) {
        return i;
      }
    }
    return null;
  };
  
  // Generate products for Cartesian product: length × width × height × color
  for (const length of uniqueLengths) {
    for (const width of uniqueWidths) {
      for (const height of uniqueHeights) {
        // Build a size string for display in cm (e.g., "60×30×124 cm")
        // Convert from mm to cm by dividing by 10, remove trailing zeros
        const formatCm = (mm: number) => {
          const cm = mm / 10;
          // Use toFixed(1) then remove .0 if integer
          return cm % 1 === 0 ? cm.toFixed(0) : cm.toFixed(1);
        };
        
        const sizeParts = [];
        if (length) sizeParts.push(formatCm(length));
        if (width) sizeParts.push(formatCm(width));
        if (height) sizeParts.push(formatCm(height));
        const sizeStr = sizeParts.length > 0 ? `${sizeParts.join('×')} cm` : '';
        
        // Find matching row index to get productGroup and priceModifier
        const rowIndex = findRowIndex(length, width, height);
        const productGroup = rowIndex !== null && productGroups[rowIndex] 
          ? productGroups[rowIndex] 
          : null;
        const priceModifier = rowIndex !== null && priceModifiers[rowIndex]
          ? parseFloat(priceModifiers[rowIndex])
          : 0;
        const priceModifierType = rowIndex !== null && priceModifierTypes[rowIndex]
          ? priceModifierTypes[rowIndex]
          : 'percent';
        
        for (const colorIndex of colorIndices) {
          // Get color name from index
          const color = colors[colorIndex];
          
          // Get images for this color from colorImages mapping (using index)
          const imagesForColor = colorImages[colorIndex.toString()] || [];
          
          // Build product name: [prefix] [productType] [size in cm] [suffix] [D{X}N{Y}] [color]
          // Note: PANEL (tapicerowany) nie ma koloru - będzie zdefiniowany później przez klienta
          const nameParts: string[] = [];
          
          if (matrix.namePrefix) {
            nameParts.push(matrix.namePrefix);
          }
          
          if (matrix.productType) {
            nameParts.push(matrix.productType);
          }
          
          if (sizeStr) {
            nameParts.push(sizeStr);
          }
          
          if (matrix.nameSuffix) {
            nameParts.push(matrix.nameSuffix);
          }
          
          // Dodaj drzwi i nogi w formacie D{X}N{Y} przed kolorem (dla oszczędności miejsca)
          const doorLegParts: string[] = [];
          if (matrix.doors) {
            const doorShortName = shortNameMap[matrix.doors] || matrix.doors;
            doorLegParts.push(doorShortName);
          }
          if (matrix.legs) {
            const legShortName = shortNameMap[matrix.legs] || matrix.legs;
            doorLegParts.push(legShortName);
          }
          if (doorLegParts.length > 0) {
            // Połącz drzwi i nogi bez spacji, np. D1N1
            nameParts.push(doorLegParts.join(''));
          }
          
          // Dla PANEL (panel tapicerowany) pomijamy kolor - będzie zdefiniowany przez klienta
          if (color && matrix.productType !== 'PANEL') {
            nameParts.push(color);
          }
          
          // Add color options (UNIT-COLOR combinations) if available (using index)
          const colorOptions = (matrix as any).colorOptions || {};
          const optionsForColor = colorOptions[colorIndex.toString()] || [];
          if (optionsForColor.length > 0) {
            nameParts.push(...optionsForColor);
          }
          
          const productName = nameParts.filter(Boolean).join(' ');
          
          // Generate temporary SKU
          const tempSku = `TEMP-${matrix.productType || 'PRODUCT'}-${color || 'NO_COLOR'}-${sizeStr || 'NO_SIZE'}-${Date.now()}`;
          
          // Calculate price with modifier (needed for template context)
          let finalPrice = matrix.defaultPrice ? parseFloat(matrix.defaultPrice) : null;
          
          if (finalPrice && priceModifier !== 0) {
            if (priceModifierType === 'fixed') {
              // Fixed amount modifier (in PLN)
              finalPrice = finalPrice + priceModifier;
            } else {
              // Percentage modifier (default)
              finalPrice = finalPrice + (finalPrice * priceModifier / 100);
            }
          }
          
          // Initialize description variables
          let descriptionHtml = '';
          let descriptionDoc = null;
          
          // NEW: Generate description using template system if templateId is set
          if (matrix.templateId) {
            try {
              console.log(`📝 Using template ID ${matrix.templateId} for product: ${productName}`);
              if (sessionId) {
                emitGenerationLog(sessionId, `📝 Przetwarzanie: ${productName}`, 'info');
                emitGenerationLog(sessionId, `   Wymiary: ${sizeStr || 'brak'}, Kolor: ${color || 'brak'}`, 'info');
              }
              
              // Fetch template from database
              const template = await getDescriptionTemplateById(matrix.templateId);
              if (sessionId && template) {
                emitGenerationLog(sessionId, `   Pobrano szablon ID: ${matrix.templateId}`, 'info');
              }
              
              if (template && template.htmlContent) {
                // Build template context with all data (pass template HTML to extract accessory tags)
                const templateContext = await buildTemplateContext(matrix, {
                  length,
                  width,
                  height,
                  color,
                  productName,
                  productGroup,
                  finalPrice,
                  sizeStr
                }, template.htmlContent, sessionId);
                
                // Decode HTML entities (Tiptap encodes HTML tags as text)
                const decodeHtmlEntities = (html: string): string => {
                  return html
                    .replace(/&lt;/g, '<')
                    .replace(/&gt;/g, '>')
                    .replace(/&amp;/g, '&')
                    .replace(/&quot;/g, '"')
                    .replace(/&#39;/g, "'");
                };
                
                // Import template renderer from client-side (we'll need to make it work server-side)
                // For now, we'll do simple string replacement similar to the renderer
                let renderedHtml = decodeHtmlEntities(template.htmlContent);
                
                // Replace AI section variables using HYPHEN notation: {{ai-intro}}, {{ai-features}}, etc.
                for (const [key, value] of Object.entries(templateContext)) {
                  if (key.startsWith('ai-')) {
                    const pattern = new RegExp(`\\{\\{${key}\\}\\}`, 'g');
                    renderedHtml = renderedHtml.replace(pattern, value as string);
                  }
                }
                
                // Replace accessory tags: {{akcesorium-GROUP_CODE:gridN}} or {{akcesorium-GROUP_CODE}}
                // If :gridN parameter present (grid3, grid4, grid2-2, grid3-, grid4-) -> render as JPG
                // If no grid parameter -> render traditional HTML list
                if (sessionId) {
                  const accessoryMatches = renderedHtml.match(/\{\{(?:akcesorium|okucia|akcesoria)-([a-zA-Z0-9_-]+)(?::([a-zA-Z0-9-]+))?\}\}/g);
                  if (accessoryMatches && accessoryMatches.length > 0) {
                    emitGenerationLog(sessionId, `   🔧 Przetwarzanie ${accessoryMatches.length} tagów akcesoriów...`, 'info');
                  }
                }
                renderedHtml = await processAccessoryTags(renderedHtml, templateContext, sessionId);
                
                // Replace image tags: {img1 w=100}, {img2 w=75}, {img3 w=50}, {img4 w=25}
                // Get images for this color from colorImages mapping (using index already calculated above)
                // imagesForColor is already defined above at line 685
                const imageTagRegex = /\{img(\d+)\s+w=(\d+)\}/gi;
                renderedHtml = renderedHtml.replace(imageTagRegex, (_match: string, imgIndex: string, widthParam: string) => {
                  const index = parseInt(imgIndex, 10) - 1; // Convert to 0-based index
                  const width = parseInt(widthParam, 10);
                  
                  // Validate width is one of the allowed values
                  if (![100, 75, 50, 25].includes(width)) {
                    console.warn(`⚠️  Invalid width parameter: ${widthParam}. Allowed: 100, 75, 50, 25`);
                    return '';
                  }
                  
                  // Check if image exists at this index
                  if (index < 0 || index >= imagesForColor.length) {
                    console.log(`⚠️  Image ${imgIndex} not found for color "${color}" (has ${imagesForColor.length} images)`);
                    return '';
                  }
                  
                  const imageEntry = imagesForColor[index];
                  
                  // Handle new format: {url, thumbnailUrl, mediumUrl}
                  let imageUrl: string;
                  if (typeof imageEntry === 'object' && imageEntry.url) {
                    // Use full quality images for template images
                    imageUrl = imageEntry.url;
                  } else {
                    // Old format: plain string URL
                    imageUrl = imageEntry as string;
                  }
                  
                  console.log(`🖼️  Replacing {img${imgIndex} w=${width}} with image: ${imageUrl}`);
                  
                  // Generate responsive HTML with inline CSS for Allegro/Shoper compatibility
                  // IMPORTANT: No div wrapper - TipTap removes it during editing
                  // All styling must be on <img> tag directly
                  // Use inline-block for horizontal layout when multiple images with small widths (25%, 50%, 75%)
                  const display = width === 100 ? 'block' : 'inline-block';
                  const margin = width === 100 ? '20px auto' : '10px 5px';
                  return `<img src="${imageUrl}" alt="${productName} - zdjęcie ${imgIndex}" class="resizable-image" style="width: ${width}%; max-width: ${width}%; height: auto; display: ${display}; border-radius: 4px; margin: ${margin}; vertical-align: top;" />`;
                });
                
                // Replace simple variables
                renderedHtml = renderedHtml
                  .replace(/\{\{product\.alpmaCode\}\}/g, templateContext.product.alpmaCode || '')
                  .replace(/\{\{product\.title\}\}/g, templateContext.product.title || '')
                  .replace(/\{\{product\.color\}\}/g, templateContext.product.color || '')
                  .replace(/\{\{product\.price\}\}/g, String(templateContext.product.price || ''))
                  .replace(/\{\{product\.productType\}\}/g, templateContext.product.productType || '')
                  .replace(/\{\{title\}\}/g, templateContext.product.title || '')
                  .replace(/\{\{color\}\}/g, templateContext.color || '')
                  .replace(/\{\{dimensions\}\}/g, templateContext.dimensions || '')
                  .replace(/\{\{price\}\}/g, String(templateContext.price || ''))
                  .replace(/\{\{basePrice\}\}/g, String(templateContext.variant?.basePrice || templateContext.price || ''))
                  .replace(/\{\{length\}\}/g, String(templateContext.variant?.length || ''))
                  .replace(/\{\{width\}\}/g, String(templateContext.variant?.width || ''))
                  .replace(/\{\{height\}\}/g, String(templateContext.variant?.height || ''));
                
                descriptionHtml = renderedHtml;
                
                // DON'T convert template HTML to Tiptap JSON - keep it as HTML
                // Templates are already properly formatted HTML with structure
                descriptionDoc = null;
                
                console.log(`✅ Description generated using template for: ${productName}`);
              } else {
                console.warn(`⚠️  Template ${matrix.templateId} not found, falling back to legacy method`);
              }
            } catch (templateError) {
              console.error('❌ Error using template:', templateError);
              // Fall through to legacy method
            }
          }
          
          // LEGACY: Fall back to old description generation if no template
          if (!descriptionHtml) {
            descriptionHtml = matrix.descriptionHtml || '';
            descriptionDoc = matrix.descriptionDoc;
            
            // Replace variables in HTML description
            if (descriptionHtml) {
              descriptionHtml = descriptionHtml
                .replace(/\{length\}/gi, String(length || ''))
                .replace(/\{width\}/gi, String(width || ''))
                .replace(/\{height\}/gi, String(height || ''))
                .replace(/\{color\}/gi, color || '')
                .replace(/\{productType\}/gi, matrix.productType || '')
                .replace(/\{productGroup\}/gi, productGroup || '')
                .replace(/\{doors\}/gi, matrix.doors || '')
                .replace(/\{legs\}/gi, matrix.legs || '')
                .replace(/\{size\}/gi, sizeStr || '')
                .replace(/\{name\}/gi, productName);
            }
            
            // Generate AI description if enabled (legacy)
            if (matrix.useAiGeneration && matrix.aiPrompt) {
              try {
                let processedPrompt = matrix.aiPrompt
                  .replace(/\{length\}/gi, String(length || ''))
                  .replace(/\{width\}/gi, String(width || ''))
                  .replace(/\{height\}/gi, String(height || ''))
                  .replace(/\{color\}/gi, color || '')
                  .replace(/\{productType\}/gi, matrix.productType || '')
                  .replace(/\{productGroup\}/gi, productGroup || '')
                  .replace(/\{doors\}/gi, matrix.doors || '')
                  .replace(/\{legs\}/gi, matrix.legs || '')
                  .replace(/\{size\}/gi, sizeStr || '')
                  .replace(/\{name\}/gi, productName);
                
                const productData = {
                  title: productName,
                  color: color || undefined,
                  dimensions: {
                    length: length !== null ? length : undefined,
                    width: width !== null ? width : undefined,
                    height: height !== null ? height : undefined,
                  },
                  productType: matrix.productType || undefined,
                  productGroup: productGroup || undefined,
                  doors: matrix.doors || undefined,
                  legs: matrix.legs || undefined,
                };
                
                console.log(`🤖 Generating AI description (legacy) for: ${productName}`);
                
                const result = await generateProductDescription({
                  productData,
                  templateHtml: processedPrompt,
                  tone: 'professional',
                  language: 'pl',
                  maxLength: 500,
                });
                
                console.log(`✅ AI description generated (${result.totalTokens} tokens, ${result.cost.toFixed(4)} PLN)`);
                
                descriptionHtml = result.description;
                
                const { htmlToTiptapJson } = await import('./ai-service');
                descriptionDoc = htmlToTiptapJson(descriptionHtml);
              } catch (aiError) {
                console.error('❌ Error generating AI description:', aiError);
              }
            }
          }
          
          // Create combination key for idempotent generation
          // NEW format: "productType:length×width×height:color:option1,option2,..."
          // Includes productType and colorOptions for accurate deduplication
          const combinationKey = buildCombinationKey(
            matrix.productType,
            length,
            width,
            height,
            color,
            optionsForColor || []
          );
          
          // Legacy key for backwards compatibility (fallback lookup)
          const legacyCombinationKey = buildLegacyCombinationKey(
            length,
            width,
            height,
            color,
            colorIndex
          );
          
          // Build stable matrix variant ID (survives SKU changes and dimension reordering)
          const matrixVariantId = buildMatrixVariantId(
            matrix.id,
            length,
            width,
            height,
            color,
            optionsForColor || []
          );
          
          // Create product in database (upsert based on matrix_variant_id)
          const productData = {
            sku: tempSku,
            title: productName,
            shortDescription: null,
            longDescriptionHtml: descriptionHtml,
            longDescriptionDoc: descriptionDoc,
            length: length,
            width: width,
            height: height,
            weight: null,
            color: color || null,
            colorOptions: optionsForColor || [],
            material: matrix.material || null,
            productType: matrix.productType || null,
            productGroup: productGroup,
            doors: matrix.doors || null,
            legs: matrix.legs || null,
            basePrice: finalPrice,
            currency: 'PLN',
            isActive: true,
            matrixId: matrix.id,
            combinationKey: combinationKey,
            matrixVariantId: matrixVariantId,
            generatedFromMatrix: true,
          };
          
          try {
            // Try to find existing product with matrix_variant_id FIRST (most reliable - survives SKU changes)
            let existingResult = await pool.query(
              `SELECT id, sku FROM catalog.products 
               WHERE matrix_variant_id = $1`,
              [matrixVariantId]
            );
            
            // If not found by variant ID, try combination_key for backwards compatibility
            if (existingResult.rows.length === 0) {
              existingResult = await pool.query(
                `SELECT id, sku FROM catalog.products 
                 WHERE matrix_id = $1 AND combination_key = $2`,
                [matrix.id, combinationKey]
              );
              
              // If found via combination_key, migrate to matrix_variant_id
              if (existingResult.rows.length > 0) {
                const productId = existingResult.rows[0].id;
                console.log(`🔄 Migrating product ID ${productId} to matrix_variant_id system`);
                
                await pool.query(
                  `UPDATE catalog.products 
                   SET matrix_variant_id = $1, updated_at = CURRENT_TIMESTAMP
                   WHERE id = $2`,
                  [matrixVariantId, productId]
                );
                
                if (sessionId) {
                  emitGenerationLog(sessionId, `🔄 Migracja produktu do matrix_variant_id`, 'info');
                }
              }
            }
            
            // If STILL not found, try LEGACY keys for backwards compatibility
            if (existingResult.rows.length === 0) {
              existingResult = await pool.query(
                `SELECT id, sku FROM catalog.products 
                 WHERE matrix_id = $1 AND combination_key = $2`,
                [matrix.id, legacyCombinationKey]
              );
              
              // If found with legacy key, migrate to matrix_variant_id
              if (existingResult.rows.length > 0) {
                const legacyProductId = existingResult.rows[0].id;
                console.log(`🔄 Migrating product ID ${legacyProductId} from legacy combination_key to matrix_variant_id`);
                
                await pool.query(
                  `UPDATE catalog.products 
                   SET combination_key = $1, matrix_variant_id = $2, updated_at = CURRENT_TIMESTAMP
                   WHERE id = $3`,
                  [combinationKey, matrixVariantId, legacyProductId]
                );
                
                if (sessionId) {
                  emitGenerationLog(sessionId, `🔄 Migracja produktu ze starego formatu`, 'info');
                }
              } else {
                // If still not found, try VERY OLD format without productType and without colorIndex
                // Format: "length×width×height:color"
                const veryOldKey = `${length !== null ? length : 'null'}×${width !== null ? width : 'null'}×${height !== null ? height : 'null'}:${color || 'no-color'}`;
                existingResult = await pool.query(
                  `SELECT id, sku FROM catalog.products 
                   WHERE matrix_id = $1 AND combination_key = $2`,
                  [matrix.id, veryOldKey]
                );
                
                // If found with very old key, migrate to new key format + matrix_variant_id
                if (existingResult.rows.length > 0) {
                  const veryOldProductId = existingResult.rows[0].id;
                  console.log(`🔄 Migrating product ID ${veryOldProductId} from VERY OLD key (without colorIndex) to new format + matrix_variant_id`);
                  console.log(`   Old key: ${veryOldKey}`);
                  console.log(`   New key: ${combinationKey}`);
                  console.log(`   Matrix variant ID: ${matrixVariantId}`);
                  
                  await pool.query(
                    `UPDATE catalog.products 
                     SET combination_key = $1, matrix_variant_id = $2, updated_at = CURRENT_TIMESTAMP
                     WHERE id = $3`,
                    [combinationKey, matrixVariantId, veryOldProductId]
                  );
                  
                  if (sessionId) {
                    emitGenerationLog(sessionId, `🔄 Migracja produktu ze starego formatu + matrix_variant_id`, 'info');
                  }
                }
              }
            }
            
            let productId: number;
            let finalSku: string;
            
            if (existingResult.rows.length > 0) {
              // UPDATE existing product
              const existing = existingResult.rows[0];
              productId = existing.id;
              finalSku = existing.sku; // Keep existing SKU
              
              await pool.query(
                `UPDATE catalog.products SET
                  name = $1,
                  title = $2,
                  description = $3,
                  short_description = $4,
                  long_description_html = $5,
                  long_description_doc = $6,
                  length = $7,
                  width = $8,
                  height = $9,
                  weight = $10,
                  color = $11,
                  color_options = $12,
                  material = $13,
                  product_type = $14,
                  product_group = $15,
                  doors = $16,
                  legs = $17,
                  base_price = $18,
                  currency = $19,
                  is_active = $20,
                  external_symbol = $21,
                  generated_from_matrix = $22,
                  matrix_variant_id = $23,
                  updated_at = CURRENT_TIMESTAMP
                 WHERE id = $24`,
                [
                  productData.title, // name
                  productData.title,
                  productData.longDescriptionHtml, // description
                  productData.shortDescription,
                  productData.longDescriptionHtml,
                  productData.longDescriptionDoc ? JSON.stringify(productData.longDescriptionDoc) : null,
                  productData.length,
                  productData.width,
                  productData.height,
                  productData.weight,
                  productData.color,
                  productData.colorOptions,
                  productData.material,
                  productData.productType,
                  productData.productGroup,
                  productData.doors,
                  productData.legs,
                  productData.basePrice,
                  productData.currency,
                  productData.isActive,
                  productData.externalSymbol,
                  productData.generatedFromMatrix,
                  productData.matrixVariantId,
                  productId,
                ]
              );
              
              console.log(`🔄 Updated existing product: ${productName} (${finalSku})`);
              if (sessionId) {
                emitGenerationLog(sessionId, `🔄 Zaktualizowano produkt: ${productName}`, 'success');
              }
            } else {
              // INSERT new product
              const result = await pool.query(
                `INSERT INTO catalog.products (
                  name, sku, title, description, short_description, long_description_html, long_description_doc,
                  length, width, height, weight, color, color_options, material, 
                  product_type, product_group, doors, legs,
                  base_price, currency, is_active, external_symbol,
                  matrix_id, combination_key, matrix_variant_id, generated_from_matrix
                ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21, $22, $23, $24, $25, $26)
                RETURNING id, sku`,
                [
                  productData.title, // name (required NOT NULL)
                  productData.sku,
                  productData.title,
                  productData.longDescriptionHtml, // description (old column)
                  productData.shortDescription,
                  productData.longDescriptionHtml,
                  productData.longDescriptionDoc ? JSON.stringify(productData.longDescriptionDoc) : null,
                  productData.length,
                  productData.width,
                  productData.height,
                  productData.weight,
                  productData.color,
                  productData.colorOptions,
                  productData.material,
                  productData.productType,
                  productData.productGroup,
                  productData.doors,
                  productData.legs,
                  productData.basePrice,
                  productData.currency,
                  productData.isActive,
                  productData.externalSymbol,
                  productData.matrixId,
                  productData.combinationKey,
                  productData.matrixVariantId,
                  productData.generatedFromMatrix,
                ]
              );
              
              const createdProduct = result.rows[0];
              productId = createdProduct.id;
              
              // Update SKU with product ID
              finalSku = `SKU-${productId}`;
              await pool.query(
                'UPDATE catalog.products SET sku = $1 WHERE id = $2',
                [finalSku, productId]
              );
              
              console.log(`✅ Created new product: ${productName} (${finalSku})`);
              if (sessionId) {
                emitGenerationLog(sessionId, `✅ Utworzono nowy produkt: ${productName}`, 'success');
              }
            }
            
            // Insert images from color gallery to product images
            if (imagesForColor && imagesForColor.length > 0) {
              console.log(`📸 Adding ${imagesForColor.length} images to product ${productId}`);
              
              // During regeneration, delete old images first to avoid duplicate key errors
              if (existingResult.rows.length > 0) {
                console.log(`🗑️ Removing old images for product ${productId}`);
                await pool.query(
                  'DELETE FROM catalog.product_images WHERE product_id = $1',
                  [productId]
                );
              }
              
              for (let i = 0; i < imagesForColor.length; i++) {
                const imageEntry = imagesForColor[i];
                
                // Support both old format (string) and new format (object with thumbnails)
                // For legacy strings, use original URL as fallback for thumbnails (better than NULL)
                const imageUrl = typeof imageEntry === 'string' ? imageEntry : imageEntry.url;
                const thumbnailUrl = typeof imageEntry === 'string' ? imageEntry : (imageEntry.thumbnailUrl || imageEntry.url);
                const mediumUrl = typeof imageEntry === 'string' ? imageEntry : (imageEntry.mediumUrl || imageEntry.url);
                
                const filename = imageUrl.split('/').pop() || `image-${i + 1}.jpg`;
                const imageType = i === 0 ? 'packshot' : 'visualization'; // First image as packshot, rest as visualization
                const isPrimary = i === 0; // First image is the primary image
                
                await pool.query(
                  `INSERT INTO catalog.product_images (product_id, url, filename, thumbnail_url, medium_url, image_type, alt_text, sort_order, is_primary)
                   VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9)`,
                  [
                    productId,
                    imageUrl,
                    filename,
                    thumbnailUrl,
                    mediumUrl,
                    imageType,
                    `${productName} - zdjęcie ${i + 1}`,
                    i,
                    isPrimary
                  ]
                );
              }
              
              console.log(`✅ Added ${imagesForColor.length} images to product ${productId} (first marked as primary)`);
            }
            
            generatedProducts.push({
              ...productData,
              id: productId,
              sku: finalSku,
            });
          } catch (dbError) {
            console.error(`❌ Error creating product ${productName}:`, dbError);
            if (sessionId) {
              emitGenerationLog(sessionId, `❌ Błąd podczas tworzenia produktu ${productName}: ${dbError instanceof Error ? dbError.message : 'Nieznany błąd'}`, 'error');
            }
          }
        } // end color loop
      } // end height loop
    } // end width loop
  } // end length loop
  
  return generatedProducts;
}

// Helper function to generate set name from matrix
async function generateSetName(matrix: any, color: string | null, colorOptions?: string[]): Promise<string> {
  const nameParts: string[] = [];
  
  // Fetch shortNames from dictionaries for all codes we need
  const codesToFetch: string[] = [];
  const codeTypes: Record<string, string> = {};
  
  if (matrix.productGroup) {
    codesToFetch.push(matrix.productGroup);
    codeTypes[matrix.productGroup] = 'product_group';
  }
  if (matrix.doors) {
    codesToFetch.push(matrix.doors);
    codeTypes[matrix.doors] = 'door';
  }
  if (matrix.legs) {
    codesToFetch.push(matrix.legs);
    codeTypes[matrix.legs] = 'leg';
  }
  
  // Also fetch component types and color options
  const components = matrix.components || [];
  components.forEach((comp: any) => {
    if (comp.componentType && comp.componentType !== 'PANEL') {
      codesToFetch.push(comp.componentType);
      codeTypes[comp.componentType] = 'product_type';
    }
  });
  
  // Add color options to fetch list
  if (colorOptions && colorOptions.length > 0) {
    colorOptions.forEach((option: string) => {
      codesToFetch.push(option);
      codeTypes[option] = 'color_option';
    });
  }
  
  // Fetch all shortNames in one query
  const shortNameMap: Record<string, string> = {};
  if (codesToFetch.length > 0) {
    const result = await pool.query(
      `SELECT code, short_name, dictionary_type 
       FROM product_creator.dictionaries 
       WHERE code = ANY($1)`,
      [codesToFetch]
    );
    result.rows.forEach((row: any) => {
      shortNameMap[row.code] = row.short_name || row.code;
    });
  }
  
  // 1. Prefix
  if (matrix.namePrefix) {
    nameParts.push(matrix.namePrefix);
  }
  
  // 2. Grupa produktów (use shortName)
  if (matrix.productGroup) {
    nameParts.push(shortNameMap[matrix.productGroup] || matrix.productGroup);
  }
  
  // 3. Długość × Szerokość (konwersja mm -> cm)
  if (matrix.length && matrix.width) {
    const lengthCm = Math.round(parseFloat(matrix.length) / 10);
    const widthCm = Math.round(parseFloat(matrix.width) / 10);
    nameParts.push(`${lengthCm}x${widthCm}cm`);
  }
  
  // 4. Rodzaje produktów z komponentów (np. VB60, Supra30) - use shortName
  const productTypes: string[] = [];
  let panelCount = 0;
  
  components.forEach((comp: any) => {
    const compType = comp.componentType || '';
    const compLength = comp.length ? Math.round(parseFloat(comp.length) / 10) : null;
    
    if (compType === 'PANEL') {
      panelCount += (comp.quantity || 0);
    } else if (compType && compLength) {
      const shortName = shortNameMap[compType] || compType;
      productTypes.push(`${shortName}${compLength}`);
    } else if (compType) {
      const shortName = shortNameMap[compType] || compType;
      productTypes.push(shortName);
    }
  });
  
  productTypes.forEach(type => nameParts.push(type));
  
  // 5. Liczba paneli
  if (panelCount > 0) {
    nameParts.push(`${panelCount} Panel${panelCount > 1 ? 'e' : ''}`);
  }
  
  // 6. Drzwi (use shortName)
  if (matrix.doors) {
    nameParts.push(shortNameMap[matrix.doors] || matrix.doors);
  }
  
  // 7. Nogi (use shortName)
  if (matrix.legs) {
    nameParts.push(shortNameMap[matrix.legs] || matrix.legs);
  }
  
  // 8. Kolor
  if (color) {
    nameParts.push(color);
  }
  
  // 9. Opcje koloru (use shortName)
  if (colorOptions && colorOptions.length > 0) {
    colorOptions.forEach((option: string) => {
      nameParts.push(shortNameMap[option] || option);
    });
  }
  
  // 10. Suffix
  if (matrix.nameSuffix) {
    nameParts.push(matrix.nameSuffix);
  }
  
  return nameParts.filter(Boolean).join(' ').trim();
}

// Helper function to extract sections from HTML description
function extractDescriptionSections(html: string): {
  intro: string;
  features: string;
  safety: string;
  accessories: string;
  care: string;
} {
  const sections = {
    intro: '',
    features: '',
    safety: '',
    accessories: '',
    care: ''
  };
  
  if (!html) return sections;
  
  // Extract intro - everything between </h2> and first <h3>
  // First try to find content after </h2> closing tag
  let introMatch = html.match(/<\/h2>([\s\S]*?)(?=<h3|$)/i);
  if (introMatch && introMatch[1]) {
    sections.intro = introMatch[1].trim();
  } else {
    // If no </h2> found, extract from start to first <h3>
    introMatch = html.match(/^([\s\S]*?)(?=<h3|$)/i);
    if (introMatch && introMatch[1]) {
      sections.intro = introMatch[1].trim();
    }
  }
  
  // Extract features section
  const featuresMatch = html.match(/<h3>Cechy produktu<\/h3>([\s\S]*?)(<h3|<div|$)/i);
  if (featuresMatch) {
    sections.features = featuresMatch[1].trim();
  }
  
  // Extract safety section
  const safetyMatch = html.match(/<h3>Bezpieczeństwo użytkowania<\/h3>([\s\S]*?)(<h3|<div|$)/i);
  if (safetyMatch) {
    sections.safety = safetyMatch[1].trim();
  }
  
  // Extract accessories section
  const accessoriesMatch = html.match(/<h3>Dostępne akcesoria i okucia:<\/h3>([\s\S]*?)(<h3|<div|$)/i);
  if (accessoriesMatch) {
    sections.accessories = accessoriesMatch[1].trim();
  }
  
  // Extract care section
  const careMatch = html.match(/<h3>Pielęgnacja<\/h3>([\s\S]*?)(<h3|<div|$)/i);
  if (careMatch) {
    sections.care = careMatch[1].trim();
  }
  
  return sections;
}

// Helper function to generate enhanced description for a set
async function generateEnhancedSetDescription(
  matrix: any, 
  color: string | null, 
  colorIndex: number,
  components: any[], 
  setName: string,
  sessionId?: string
): Promise<{ shortDescription: string; fullDescription: string }> {
  try {
    console.log(`📝 Generating enhanced description for set: ${setName}`);
    
    // 1. Get set dimensions from matrix
    const setLength = matrix.length ? parseFloat(matrix.length) : null;
    const setWidth = matrix.width ? parseFloat(matrix.width) : null;
    const setHeight = matrix.height ? parseFloat(matrix.height) : null;
    
    // 2. Build set parameters section
    const setDimensions = [];
    if (setLength) setDimensions.push(`Długość: ${Math.round(setLength / 10)} cm (${setLength} mm)`);
    if (setWidth) setDimensions.push(`Szerokość: ${Math.round(setWidth / 10)} cm (${setWidth} mm)`);
    if (setHeight) setDimensions.push(`Wysokość: ${Math.round(setHeight / 10)} cm (${setHeight} mm)`);
    if (matrix.defaultDepth) setDimensions.push(`Głębokość: ${Math.round(parseFloat(matrix.defaultDepth) / 10)} cm`);
    
    // 3. Fetch product_type dictionaries to get readable names
    const dictQuery = `
      SELECT code, readable_name 
      FROM product_creator.dictionaries 
      WHERE dictionary_type = 'product_type' AND is_active = true
    `;
    const dictResult = await pool.query(dictQuery);
    const productTypeNames = new Map(dictResult.rows.map(row => [row.code, row.readable_name]));
    
    // 4. Find matching products for components and fetch their AI descriptions
    const componentDescriptions: any[] = [];
    
    for (const comp of components) {
      const compType = comp.componentType;
      const compLength = comp.length ? parseFloat(comp.length) : null;
      const compWidth = comp.width ? parseFloat(comp.width) : null;
      const compQuantity = comp.quantity || 1;
      
      // Try to find matching product in catalog
      let matchingProduct = null;
      
      if (compType && compLength) {
        // Build query to find matching product
        const queryConditions: string[] = [];
        const queryParams: any[] = [];
        let paramIndex = 1;
        
        // Match by product type
        queryConditions.push(`product_type = $${paramIndex}`);
        queryParams.push(compType);
        paramIndex++;
        
        // Match by dimensions (with tolerance of ±5mm)
        queryConditions.push(`ABS(CAST(length AS NUMERIC) - $${paramIndex}) <= 5`);
        queryParams.push(compLength);
        paramIndex++;
        
        if (compWidth) {
          queryConditions.push(`ABS(CAST(width AS NUMERIC) - $${paramIndex}) <= 5`);
          queryParams.push(compWidth);
          paramIndex++;
        }
        
        // Match by color if specified
        if (color) {
          queryConditions.push(`color = $${paramIndex}`);
          queryParams.push(color);
          paramIndex++;
        }
        
        const productQuery = `
          SELECT id, sku, title, short_description, long_description_html, 
                 length, width, height, color, product_type, gallery
          FROM catalog.products
          WHERE ${queryConditions.join(' AND ')}
            AND is_active = true
          ORDER BY created_at DESC
          LIMIT 1
        `;
        
        const productResult = await pool.query(productQuery, queryParams);
        
        if (productResult.rows.length > 0) {
          matchingProduct = productResult.rows[0];
          console.log(`✅ Found matching product for component ${compType}: ${matchingProduct.sku}`);
        } else {
          console.log(`⚠️ No matching product found for component ${compType} (${compLength}mm${compWidth ? ` × ${compWidth}mm` : ''})`);
        }
      }
      
      // Build component description entry and extract sections
      const sections = matchingProduct?.long_description_html 
        ? extractDescriptionSections(matchingProduct.long_description_html)
        : { intro: '', features: '', safety: '', accessories: '', care: '' };
      
      // Build component name: "Nazwa czytelna - KOD (wymiary) × ilość"
      const readableName = productTypeNames.get(compType) || compType;
      const dims = [compLength, compWidth].filter((v): v is number => v !== null).map(v => `${Math.round(v / 10)} cm`).join(' × ');
      let componentName = readableName ? `${readableName} - ${compType}` : compType;
      if (dims) {
        componentName += ` (${dims})`;
      }
      if (compQuantity > 1) {
        componentName += ` × ${compQuantity} szt.`;
      }
      
      const compEntry: any = {
        name: componentName,
        type: compType,
        length: compLength,
        width: compWidth,
        quantity: compQuantity,
        product: matchingProduct,
        sections: sections
      };
      
      componentDescriptions.push(compEntry);
    }
    
    // 5. Build short description with components
    const componentsList = componentDescriptions.map(c => c.name).join(', ');
    const shortDescription = `Zestaw ${matrix.name}${color ? ` w kolorze ${color}` : ''} składający się z: ${componentsList}`;
    
    // 6. Build full description HTML
    const descriptionParts: string[] = [];
    
    // Color image at the very top (if available) - display second image only
    if (color && colorIndex !== -1 && matrix.colorImages && matrix.colorImages[colorIndex.toString()]) {
      const colorImagesData = matrix.colorImages[colorIndex.toString()];
      // colorImages[index] can be either array or string
      let secondImageUrl: string | null = null;
      
      if (Array.isArray(colorImagesData)) {
        secondImageUrl = colorImagesData[1] || null; // Take second image (index 1)
      } else if (typeof colorImagesData === 'string') {
        const urlsArray = colorImagesData.includes(',') 
          ? colorImagesData.split(',').map(url => url.trim())
          : [colorImagesData];
        secondImageUrl = urlsArray[1] || null; // Take second image
      }
      
      if (secondImageUrl) {
        descriptionParts.push(`<div class="set-color-image" style="text-align: center; margin-bottom: 20px;">`);
        descriptionParts.push(`<img src="${secondImageUrl}" alt="${color}" style="width: auto; height: auto;" />`);
        descriptionParts.push(`</div>`);
      }
    }
    
    // Header with set name
    descriptionParts.push(`<h2>${setName}</h2>`);
    
    // Set parameters table
    if (setDimensions.length > 0) {
      descriptionParts.push(`<h3>Parametry zestawu</h3>`);
      descriptionParts.push(`<table class="set-parameters">`);
      descriptionParts.push(`<tbody>`);
      setDimensions.forEach(dim => {
        const [label, value] = dim.split(': ');
        descriptionParts.push(`<tr><td><strong>${label}:</strong></td><td>${value}</td></tr>`);
      });
      if (color) {
        descriptionParts.push(`<tr><td><strong>Kolor:</strong></td><td>${color}</td></tr>`);
      }
      descriptionParts.push(`<tr><td><strong>Liczba elementów:</strong></td><td>${components.length}</td></tr>`);
      descriptionParts.push(`</tbody>`);
      descriptionParts.push(`</table>`);
    }
    
    // 7. Build enhanced introduction by combining intros from all components
    descriptionParts.push(`<h3>Opis zestawu</h3>`);
    descriptionParts.push(`<p>${setName} to kompletne rozwiązanie składające się z następujących elementów:</p>`);
    descriptionParts.push(`<ul>`);
    
    for (const compDesc of componentDescriptions) {
      // compDesc.name already contains dimensions and quantity, so just use it directly
      descriptionParts.push(`<li><strong>${compDesc.name}</strong></li>`);
    }
    
    descriptionParts.push(`</ul>`);
    
    // Combine all component intros into one enhanced description
    const componentIntros = componentDescriptions
      .filter(c => c.sections.intro)
      .map(c => {
        // Clean HTML tags from intro but keep content
        const cleanIntro = c.sections.intro
          .replace(/<[^>]+>/g, ' ')
          .replace(/\s+/g, ' ')
          .trim();
        return cleanIntro;
      });
    
    if (componentIntros.length > 0) {
      const combinedIntro = componentIntros.join(' ');
      descriptionParts.push(`<p>${combinedIntro}</p>`);
    } else {
      descriptionParts.push(`<p>Zestaw został zaprojektowany z myślą o funkcjonalności i estetyce, łącząc wysokiej jakości komponenty w spójną całość.</p>`);
    }
    
    // 7. Detailed components section with all sections
    descriptionParts.push(`<h3>Szczegółowy opis komponentów</h3>`);
    
    for (const compDesc of componentDescriptions) {
      const dims = [compDesc.length, compDesc.width].filter(Boolean).map(v => `${Math.round(v / 10)} cm`).join(' × ');
      
      descriptionParts.push(`<div class="component-section">`);
      
      // H3 header with full product title (generated name with all parameters)
      if (compDesc.product?.title) {
        descriptionParts.push(`<h3>${compDesc.product.title}</h3>`);
      } else {
        descriptionParts.push(`<h3>${compDesc.name}${dims ? ` ${dims}` : ''}</h3>`);
      }
      
      if (compDesc.product) {
        // Format: "Nazwa czytelna - Nazwa" below the H3 header
        descriptionParts.push(`<p><strong>${compDesc.name}</strong></p>`);
        descriptionParts.push(`<p><strong>SKU produktu:</strong> ${compDesc.product.sku} | <strong>Ilość:</strong> ${compDesc.quantity} szt.</p>`);
        
        // Add product image - display ONLY second image from gallery (index 1) with original proportions
        if (compDesc.product.gallery) {
          let secondImageUrl: string | null = null;
          
          // gallery can be array or JSON string
          if (Array.isArray(compDesc.product.gallery)) {
            secondImageUrl = compDesc.product.gallery[1] || null;
          } else if (typeof compDesc.product.gallery === 'string') {
            try {
              const galleryArray = JSON.parse(compDesc.product.gallery);
              secondImageUrl = Array.isArray(galleryArray) ? galleryArray[1] || null : null;
            } catch {
              // Not JSON, might be comma-separated
              const urlsArray = compDesc.product.gallery.includes(',') 
                ? compDesc.product.gallery.split(',').map((url: string) => url.trim())
                : [compDesc.product.gallery];
              secondImageUrl = urlsArray[1] || null;
            }
          }
          
          if (secondImageUrl) {
            descriptionParts.push(`<div class="component-image" style="text-align: center; margin: 15px 0;">`);
            descriptionParts.push(`<img src="${secondImageUrl}" alt="${compDesc.name}" style="width: auto; height: auto;" />`);
            descriptionParts.push(`</div>`);
          }
        }
        
        // Add introduction
        if (compDesc.sections.intro) {
          descriptionParts.push(compDesc.sections.intro);
        }
        
        // Add features section
        if (compDesc.sections.features) {
          descriptionParts.push(`<h4>Cechy produktu</h4>`);
          descriptionParts.push(compDesc.sections.features);
        }
        
        // Add safety section
        if (compDesc.sections.safety) {
          descriptionParts.push(`<h4>Bezpieczeństwo użytkowania</h4>`);
          descriptionParts.push(compDesc.sections.safety);
        }
        
        // Add accessories section
        if (compDesc.sections.accessories) {
          descriptionParts.push(`<h4>Dostępne akcesoria i okucia</h4>`);
          descriptionParts.push(compDesc.sections.accessories);
        }
        
        // Add care section
        if (compDesc.sections.care) {
          descriptionParts.push(`<h4>Pielęgnacja</h4>`);
          descriptionParts.push(compDesc.sections.care);
        }
      } else {
        // No matching product found
        descriptionParts.push(`<p>Komponent typu ${compDesc.type}${dims ? ` o wymiarach ${dims}` : ''}. Ilość: ${compDesc.quantity} szt.</p>`);
      }
      
      descriptionParts.push(`</div>`);
    }
    
    // Technical specifications table
    descriptionParts.push(`<h3>Specyfikacja techniczna komponentów</h3>`);
    descriptionParts.push(`<table class="component-specs">`);
    descriptionParts.push(`<thead>`);
    descriptionParts.push(`<tr><th>Komponent</th><th>Typ</th><th>Wymiary (cm)</th><th>Ilość</th></tr>`);
    descriptionParts.push(`</thead>`);
    descriptionParts.push(`<tbody>`);
    
    for (const compDesc of componentDescriptions) {
      const dims = [compDesc.length, compDesc.width, compDesc.product?.height]
        .filter(Boolean)
        .map(v => Math.round(parseFloat(v) / 10))
        .join(' × ');
      
      descriptionParts.push(`<tr>`);
      descriptionParts.push(`<td>${compDesc.name}</td>`);
      descriptionParts.push(`<td>${compDesc.type || '-'}</td>`);
      descriptionParts.push(`<td>${dims || '-'}</td>`);
      descriptionParts.push(`<td>${compDesc.quantity}</td>`);
      descriptionParts.push(`</tr>`);
    }
    
    descriptionParts.push(`</tbody>`);
    descriptionParts.push(`</table>`);
    
    // Closing paragraph
    descriptionParts.push(`<p><strong>Uwaga:</strong> Zestaw wymaga montażu. Wszystkie niezbędne elementy montażowe są dołączone do poszczególnych komponentów.</p>`);
    
    const fullDescription = descriptionParts.join('\n');
    
    console.log(`✅ Enhanced description generated (${fullDescription.length} characters)`);
    if (sessionId) {
      emitGenerationLog(sessionId, `   📝 Wygenerowano rozszerzony opis (${componentDescriptions.filter(c => c.product).length}/${components.length} komponentów dopasowanych)`, 'info');
    }
    
    return {
      shortDescription,
      fullDescription
    };
  } catch (error) {
    console.error('❌ Error generating enhanced description:', error);
    
    // Fallback to simple description
    const shortDescription = `Zestaw ${matrix.name}${color ? ` w kolorze ${color}` : ''} składający się z ${components.length} komponentów`;
    const fullDescription = `
      <h2>${setName}</h2>
      <p>${shortDescription}</p>
      <h3>Komponenty zestawu:</h3>
      <ul>
        ${components.map((comp: any, idx: number) => {
          const compName = comp.name || `Komponent ${idx + 1}`;
          const compType = comp.componentType || '';
          const dims = [comp.length, comp.width].filter(Boolean).map(v => `${v}mm`).join(' × ');
          const qty = comp.quantity || 1;
          return `<li>${compName}${compType ? ` (${compType})` : ''}${dims ? ` - ${dims}` : ''} × ${qty} szt.</li>`;
        }).join('\n        ')}
      </ul>
    `.trim();
    
    return { shortDescription, fullDescription };
  }
}

// Helper function to filter color options based on component's relevant prefixes
// FEATURE DISABLED: relevantOptionPrefixes is not used in current generation logic
// Returns original colorOptions without any filtering
function filterColorOptionsByRelevantPrefixes(
  colorOptions: string[],
  relevantPrefixes: string[] | undefined
): string[] {
  // FEATURE DISABLED - always return all options without filtering
  // Original filtering logic preserved below as comment for future reference:
  /*
  // If no relevantPrefixes defined, return all options (backward compatibility)
  if (!relevantPrefixes || relevantPrefixes.length === 0) {
    return colorOptions;
  }
  
  // Filter options to only those matching relevant prefixes
  return colorOptions.filter(option => {
    return relevantPrefixes.some(prefix => 
      option.toUpperCase().startsWith(prefix.toUpperCase())
    );
  });
  */
  return colorOptions;
}

// Helper function to generate sets from set matrix
async function generateSetsFromMatrix(matrix: any, sessionId?: string): Promise<any[]> {
  const colors = matrix.colors || [];
  const components = (matrix.components || []) as any[];
  const colorImages = matrix.colorImages || {};
  const componentProductOverrides = matrix.componentProductOverrides || {};
  
  const generatedSets: any[] = [];
  
  // Use selected colors if available, otherwise use all colors
  const selectedColorsArray = matrix.selectedColors || [];
  const colorsToUse = selectedColorsArray.length > 0 
    ? selectedColorsArray 
    : (colors.length > 0 ? colors : [null]);
  
  const totalSets = colorsToUse.length;
  console.log(`📦 Generating ${totalSets} sets from matrix: ${matrix.name}`);
  console.log(`  Components: ${components.length}`);
  console.log(`  Colors to generate: ${colorsToUse.length > 0 ? colorsToUse.join(', ') : 'No color'}`);
  
  if (sessionId) {
    emitGenerationLog(sessionId, `📦 Rozpoczynanie generowania ${totalSets} zestawów...`, 'info');
    emitGenerationLog(sessionId, `   ${components.length} komponentów × ${colorsToUse.length} kolorów`, 'info');
    if (selectedColorsArray.length > 0) {
      emitGenerationLog(sessionId, `   ✓ Generowanie tylko dla ${selectedColorsArray.length} zaznaczonych kolorów`, 'info');
    }
  }
  
  // Generate sets for each color
  for (const color of colorsToUse) {
    try {
      // Normalize color: convert numeric strings to integers (selectedColors may contain "3", "03", " 3 " etc.)
      let normalizedColor: string | number | null;
      if (typeof color === 'string') {
        const trimmed = color.trim();
        const parsed = Number.parseInt(trimmed, 10);
        // Check if the entire string is a valid number (no remaining non-digit chars)
        normalizedColor = !Number.isNaN(parsed) && /^\d+$/.test(trimmed) ? parsed : color;
      } else {
        normalizedColor = color;
      }
      
      // Find index of this color in the original colors array
      // Handle both cases: normalizedColor can be NUMBER (index) or STRING (color name)
      let colorIndex: number;
      if (typeof normalizedColor === 'number') {
        // normalizedColor is an index from selectedColors - validate bounds
        if (normalizedColor >= 0 && normalizedColor < matrix.colors.length) {
          colorIndex = normalizedColor;
        } else {
          // Out of bounds - log error and skip overrides for this color
          console.error(`❌ Color index ${normalizedColor} out of bounds (colors.length = ${matrix.colors.length})`);
          if (sessionId) {
            emitGenerationLog(sessionId, `❌ Nieprawidłowy indeks koloru: ${normalizedColor}`, 'error');
          }
          colorIndex = -1;
        }
      } else if (normalizedColor) {
        // normalizedColor is a name, find its index
        colorIndex = matrix.colors.indexOf(normalizedColor);
      } else {
        // no color
        colorIndex = -1;
      }
      
      // Get actual color name for display/SKU (resolve index to name if needed)
      const actualColorName = typeof normalizedColor === 'number' && colorIndex >= 0
        ? (matrix.colors[colorIndex] || `Index-${colorIndex}`) 
        : normalizedColor;
      
      // Calculate price
      const basePrice = parseFloat(matrix.basePriceModifier as string) || 0;
      const initialPrice = basePrice; // Can be enhanced with component prices later
      
      // Helper to get value from colorImages/colorOptions with fallback strategy
      // Priority: index-based (new format) → name-based (backward compatibility)
      const getColorValue = (obj: any, colorName: string | null, index: number): any => {
        if (!obj) return undefined;
        
        // Try multiple lookup strategies (index-based first for new format)
        const lookupKeys: string[] = [
          index.toString(),                        // Index-based (new format, priority)
          colorName ? colorName.trim() : '',              // Normalized (trimmed)
          colorName || '',                            // Original
          colorName ? colorName.trim().toLowerCase() : '' // Lowercase normalized
        ].filter(k => k !== '');
        
        for (const key of lookupKeys) {
          if (obj[key]) return obj[key];
        }
        return undefined;
      };
      
      // Get images for this color (with multi-strategy fallback)
      // Note: imagesForColor may be array of strings (old format) or array of objects (new format with thumbnails)
      const imagesForColor: any[] = getColorValue(colorImages, actualColorName, colorIndex) || [];
      
      // Extract URLs for backward compatibility with product_sets.images (text[] column)
      const imageUrls: string[] = imagesForColor.map((img: any) => 
        typeof img === 'string' ? img : img.url
      );
      
      // Get color options for this color (with multi-strategy fallback)
      // If matrix has no colorOptions at all, treat as "any options OK"
      const colorOptionsForColor: string[] = getColorValue(matrix.colorOptions, actualColorName, colorIndex) || [];
      console.log(`🎨 Color options for ${actualColorName} (index ${colorIndex}):`, colorOptionsForColor);
      
      // Build SKU (must be unique for each configuration)
      const skuParts = [
        'SET',
        matrix.name.replace(/\s+/g, '-').toUpperCase(),
        actualColorName ? actualColorName.toString().replace(/\s+/g, '-').toUpperCase() : 'BASE',
        // Add color options to SKU for uniqueness (use first option if available)
        colorOptionsForColor.length > 0 ? colorOptionsForColor[0].replace(/\s+/g, '-').toUpperCase() : null,
        // Add doors/legs if they differ from defaults
        matrix.doors && matrix.doors !== 'none' ? matrix.doors.replace(/\s+/g, '-').toUpperCase().substring(0, 10) : null,
        matrix.legs && matrix.legs !== 'none' ? matrix.legs.replace(/\s+/g, '-').toUpperCase().substring(0, 10) : null
      ].filter(Boolean);
      const sku = skuParts.join('-');
      
      // Build combination key for uniqueness - must include ALL configuration parameters
      // that make sets different (color, doors, legs, colorOptions, depth, panel count, hook length)
      const combinationParts = [
        actualColorName || 'no-color',
        matrix.doors || 'no-doors',
        matrix.legs || 'no-legs',
        colorOptionsForColor.length > 0 ? `opts-${colorOptionsForColor.sort().join('-')}` : '',
        matrix.defaultDepth ? `depth-${matrix.defaultDepth}` : '',
        matrix.panelCount ? `panels-${matrix.panelCount}` : '',
        matrix.hookLength ? `hook-${matrix.hookLength}` : ''
      ].filter(Boolean);
      const combinationKey = combinationParts.join('|');
      
      // Build set name using new generator function (with colorOptions)
      const setName = await generateSetName(matrix, actualColorName, colorOptionsForColor);
      
      // Generate description - enhanced or simple
      let shortDescription: string;
      let fullDescription: string;
      
      if (matrix.useEnhancedDescription) {
        // Use enhanced description with component product details
        const enhancedDesc = await generateEnhancedSetDescription(matrix, color, colorIndex, components, setName, sessionId);
        shortDescription = enhancedDesc.shortDescription;
        fullDescription = enhancedDesc.fullDescription;
      } else {
        // Use simple description
        shortDescription = `Zestaw ${matrix.name}${color ? ` w kolorze ${color}` : ''} składający się z ${components.length} komponentów`;
        fullDescription = `
          <h2>${setName}</h2>
          <p>${shortDescription}</p>
          <h3>Komponenty zestawu:</h3>
          <ul>
            ${components.map((comp: any, idx: number) => {
              const compName = comp.name || `Komponent ${idx + 1}`;
              const compType = comp.componentType || '';
              const dims = [comp.length, comp.width].filter(Boolean).map(v => `${v}mm`).join(' × ');
              const qty = comp.quantity || 1;
              return `<li>${compName}${compType ? ` (${compType})` : ''}${dims ? ` - ${dims}` : ''} × ${qty} szt.</li>`;
            }).join('\n            ')}
          </ul>
        `.trim();
      }
      
      console.log(`📦 Creating set: ${setName} (${sku})`);
      if (sessionId) {
        emitGenerationLog(sessionId, `📦 Tworzenie zestawu: ${setName}`, 'info');
      }
      
      // Check if set already exists with this matrix_id + combination_key
      const existingResult = await pool.query(
        `SELECT id, sku FROM product_creator.product_sets 
         WHERE set_matrix_id = $1 AND combination_key = $2`,
        [matrix.id, combinationKey]
      );
      
      let setId: number;
      let finalSku: string;
      
      if (existingResult.rows.length > 0) {
        // UPDATE existing set
        const existing = existingResult.rows[0];
        setId = existing.id;
        finalSku = existing.sku; // Keep existing SKU
        
        await pool.query(
          `UPDATE product_creator.product_sets SET
            title = $1,
            short_description = $2,
            full_description = $3,
            color = $4,
            color_options = $5,
            base_price = $6,
            calculated_price = $7,
            images = $8,
            depth = $9,
            panel_count = $10,
            hook_length = $11,
            updated_at = CURRENT_TIMESTAMP
           WHERE id = $12`,
          [
            setName,
            shortDescription,
            fullDescription,
            actualColorName || null,
            colorOptionsForColor || [],
            basePrice,
            initialPrice,
            imageUrls,
            matrix.defaultDepth || null,
            matrix.panelCount || null,
            matrix.hookLength || null,
            setId
          ]
        );
        
        console.log(`🔄 Updated existing set: ${setName} (${finalSku})`);
        if (sessionId) {
          emitGenerationLog(sessionId, `🔄 Zaktualizowano zestaw: ${setName}`, 'success');
        }
      } else {
        // INSERT new set
        console.log(`💾 INSERT set with color_options:`, colorOptionsForColor);
        const result = await pool.query(
          `INSERT INTO product_creator.product_sets (
            set_matrix_id, sku, title, short_description, full_description,
            color, color_options, base_price, calculated_price, images,
            depth, panel_count, hook_length,
            generated_from_matrix, combination_key, is_active
          ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16)
          RETURNING id`,
          [
            matrix.id,
            sku,
            setName,
            shortDescription,
            fullDescription,
            actualColorName || null,
            colorOptionsForColor || [],
            basePrice,
            initialPrice,
            imageUrls,
            matrix.defaultDepth || null,
            matrix.panelCount || null,
            matrix.hookLength || null,
            true, // generated_from_matrix
            combinationKey,
            true // is_active
          ]
        );
        
        setId = result.rows[0].id;
        finalSku = sku;
        
        console.log(`✅ Created new set: ${setName} (${finalSku})`);
        if (sessionId) {
          emitGenerationLog(sessionId, `✅ Utworzono zestaw: ${setName}`, 'success');
        }
      }
      
      // Insert images from color gallery to set_images with optimized versions
      if (imagesForColor && imagesForColor.length > 0) {
        console.log(`📸 Adding ${imagesForColor.length} images to set ${setId}`);
        
        // During regeneration, delete old images first
        if (existingResult.rows.length > 0) {
          await pool.query(
            'DELETE FROM catalog.set_images WHERE set_id = $1',
            [setId]
          );
        }
        
        for (let i = 0; i < imagesForColor.length; i++) {
          const imageEntry = imagesForColor[i];
          
          // Support both old format (string) and new format (object with thumbnails)
          // For legacy strings, use original URL as fallback for thumbnails (better than NULL)
          const imageUrl = typeof imageEntry === 'string' ? imageEntry : imageEntry.url;
          const thumbnailUrl = typeof imageEntry === 'string' ? imageEntry : (imageEntry.thumbnailUrl || imageEntry.url);
          const mediumUrl = typeof imageEntry === 'string' ? imageEntry : (imageEntry.mediumUrl || imageEntry.url);
          
          const filename = imageUrl.split('/').pop() || `image-${i + 1}.jpg`;
          const imageType = i === 0 ? 'packshot' : 'visualization';
          const isPrimary = i === 0;
          
          await pool.query(
            `INSERT INTO catalog.set_images (set_id, url, filename, thumbnail_url, medium_url, image_type, alt_text, sort_order, is_primary)
             VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, true, $10)`,
            [
              setId,
              imageUrl,
              filename,
              thumbnailUrl,
              mediumUrl,
              imageType,
              `${setName} - zdjęcie ${i + 1}`,
              i,
              isPrimary
            ]
          );
        }
        
        console.log(`✅ Added ${imagesForColor.length} images to set ${setId} (first marked as primary)`);
      }
      
      // Create/update component links
      // First, delete existing links for this set
      await pool.query(
        'DELETE FROM product_creator.set_product_links WHERE set_id = $1',
        [setId]
      );
      
      // Then create new links for each component
      for (let i = 0; i < components.length; i++) {
        const comp = components[i];
        const componentType = comp.componentType || comp.name || `Component ${i + 1}`;
        
        // Try to find matching product from catalog
        let productId: number | null = null;
        let productSource: 'override' | 'auto' = 'auto';
        
        // 🎯 PRIORITY 1: Check for component product override
        // Use colorIndex as key (skip for invalid or no-color case: colorIndex < 0)
        if (colorIndex >= 0 && colorIndex < matrix.colors.length) {
          const colorIndexKey = colorIndex.toString();
          // Safe access with optional chaining (componentProductOverrides may be sparse)
          const overrideForColor = componentProductOverrides?.[colorIndexKey];
          const overrideProductId = overrideForColor?.[componentType]?.productId;
          
          if (overrideProductId) {
            // Override found - validate product exists before using
            const productCheckResult = await pool.query(
              'SELECT id, name, sku FROM catalog.products WHERE id = $1',
              [overrideProductId]
            );
            
            if (productCheckResult.rows.length > 0) {
              // Override product exists - use it
              productId = overrideProductId;
              productSource = 'override';
              const product = productCheckResult.rows[0];
              console.log(`🎯 Using override for ${componentType} in color ${actualColorName} (index ${colorIndex}): product ${productId} (${product.name})`);
              if (sessionId) {
                emitGenerationLog(sessionId, `   🎯 Nadpisanie: ${componentType} → ${product.name} (${product.sku})`, 'info');
              }
            } else {
              // Override product doesn't exist - log warning and fall back to auto-matching
              console.warn(`⚠️ Override product ${overrideProductId} for ${componentType} not found - falling back to auto-matching`);
              if (sessionId) {
                emitGenerationLog(sessionId, `   ⚠️ Produkt nadpisania ${overrideProductId} nie istnieje - automatyczne wyszukiwanie`, 'warning');
              }
            }
          }
        }
        
        // 🔍 PRIORITY 2: Auto-match product if no override was used
        if (productId === null && comp.componentType) {
          // Build query dynamically based on available dimensions
          let query = `
            SELECT id 
            FROM catalog.products 
            WHERE product_type = $1
          `;
          const params: any[] = [comp.componentType];
          let paramIndex = 2;
          
          // Add length condition if specified
          if (comp.length !== null && comp.length !== undefined) {
            query += ` AND length = $${paramIndex}`;
            params.push(String(comp.length));
            paramIndex++;
          }
          
          // Add width condition if specified
          if (comp.width !== null && comp.width !== undefined) {
            query += ` AND width = $${paramIndex}`;
            params.push(String(comp.width));
            paramIndex++;
          }
          
          // Add doors condition from set matrix if specified
          // Match products with specific doors OR products without doors (NULL or empty string)
          if (matrix.doors && matrix.doors !== 'none') {
            query += ` AND (doors = $${paramIndex} OR doors IS NULL OR doors = '')`;
            params.push(matrix.doors);
            paramIndex++;
            console.log(`  🚪 Matching doors: ${matrix.doors} (or NULL/empty)`);
          }
          
          // Add legs condition from set matrix if specified
          // Match products with specific legs OR products without legs (NULL or empty string)
          if (matrix.legs && matrix.legs !== 'none') {
            query += ` AND (legs = $${paramIndex} OR legs IS NULL OR legs = '')`;
            params.push(matrix.legs);
            paramIndex++;
            console.log(`  🦵 Matching legs: ${matrix.legs} (or NULL/empty)`);
          }
          
          // For non-PANEL components, also match color
          // BUT: if component has ignoreColorOptions flag, skip color matching entirely
          if (!comp.ignoreColorOptions && actualColorName && comp.componentType !== 'PANEL') {
            query += ` AND color = $${paramIndex}`;
            params.push(actualColorName);
            paramIndex++;
          }
          
          // Match color_options based on set matrix colorOptions
          // BUT: if component has ignoreColorOptions flag, skip color_options matching entirely
          if (comp.ignoreColorOptions) {
            // Component IGNORES color options AND color (e.g., hanger, upholstered panel)
            // Don't add ANY color/color_options filter - match products regardless of their color
            console.log(`  🔧 Component ${componentType} has ignoreColorOptions=true → no filter on color or color_options`);
          } else if (colorOptionsForColor.length > 0) {
            // Filter color options by component's relevant prefixes (if defined)
            // Example: VB has only DRZWI (no FRONT), so filter ["DRZWI-WOTAN", "FRONT-WOTAN"] → ["DRZWI-WOTAN"]
            const filteredColorOptions = filterColorOptionsByRelevantPrefixes(
              colorOptionsForColor,
              comp.relevantOptionPrefixes
            );
            
            if (filteredColorOptions.length > 0) {
              // Set matrix HAS color options (after filtering) → match products that have ALL these options
              console.log(`  🎨 Matching products WITH color_options: [${filteredColorOptions.join(', ')}]${comp.relevantOptionPrefixes ? ` (filtered by prefixes: [${comp.relevantOptionPrefixes.join(', ')}])` : ''}`);
              query += ` AND color_options @> $${paramIndex}::text[]`;
              params.push(filteredColorOptions);
              paramIndex++;
            } else {
              // All options were filtered out → match products WITHOUT options
              console.log(`  ⚪ All color options filtered out by relevantOptionPrefixes → matching products WITHOUT color_options`);
              query += ` AND (color_options IS NULL OR cardinality(color_options) = 0)`;
            }
          } else {
            // Set matrix has NO options → match ONLY products WITHOUT options
            // This ensures we don't match products with incompatible options
            console.log(`  ⚪ Set matrix has NO color options → matching products WITHOUT color_options`);
            query += ` AND (color_options IS NULL OR cardinality(color_options) = 0)`;
          }
          
          query += ` LIMIT 1`;
          
          const productResult = await pool.query(query, params);
          
          if (productResult.rows.length > 0) {
            productId = productResult.rows[0].id;
            console.log(`✓ Found product ${productId} for component ${componentType}`);
            if (sessionId) {
              emitGenerationLog(sessionId, `   ✓ Znaleziono produkt dla ${componentType}`, 'info');
            }
          } else {
            const dimensions = [
              comp.length !== null && comp.length !== undefined ? `${comp.length}` : 'null',
              comp.width !== null && comp.width !== undefined ? `${comp.width}` : 'null'
            ].join('×');
            const colorOptionsInfo = colorOptionsForColor.length > 0 
              ? ` z opcjami [${colorOptionsForColor.join(', ')}]`
              : ' bez opcji kolorystycznych';
            const doorsInfo = matrix.doors && matrix.doors !== 'none' ? ` doors=${matrix.doors}` : '';
            const legsInfo = matrix.legs && matrix.legs !== 'none' ? ` legs=${matrix.legs}` : '';
            console.log(`⚠️  No product found for component ${componentType} (${dimensions}, color: ${color || 'N/A'}${colorOptionsInfo}${doorsInfo}${legsInfo})`);
            if (sessionId) {
              emitGenerationLog(sessionId, `   ⚠️  Brak produktu dla ${componentType} (${dimensions})${colorOptionsInfo}${doorsInfo}${legsInfo}`, 'warning');
            }
          }
        }
        
        await pool.query(
          `INSERT INTO product_creator.set_product_links (
            set_id, product_id, component_type, quantity, position
          ) VALUES ($1, $2, $3, $4, $5)`,
          [
            setId,
            productId, // Now links to actual product if found
            componentType,
            comp.quantity || 1,
            i
          ]
        );
      }
      
      console.log(`🔗 Created ${components.length} component links for set ${setId}`);
      if (sessionId) {
        emitGenerationLog(sessionId, `🔗 Dodano ${components.length} komponentów do zestawu`, 'info');
      }
      
      // Recalculate set price from component costs
      const calculatedPrice = await recalculateSetPrice(setId);
      if (sessionId) {
        emitGenerationLog(sessionId, `💰 Obliczona cena: ${calculatedPrice.toFixed(2)} PLN`, 'info');
      }
      
      generatedSets.push({
        id: setId,
        sku: finalSku,
        title: setName,
        color: actualColorName || null,
        componentsCount: components.length
      });
    } catch (error) {
      console.error(`❌ Error creating set for color ${color}:`, error);
      if (sessionId) {
        emitGenerationLog(sessionId, `❌ Błąd podczas tworzenia zestawu dla koloru ${color}: ${error instanceof Error ? error.message : 'Nieznany błąd'}`, 'error');
      }
    }
  }
  
  return generatedSets;
}

// Helper function to recalculate set price from component costs and apply modifier
async function recalculateSetPrice(setId: number): Promise<number> {
  // Calculate component total
  const result = await pool.query(`
    SELECT COALESCE(SUM(
      COALESCE(p.base_price, 0) * spl.quantity
    ), 0) as total_price
    FROM product_creator.set_product_links spl
    LEFT JOIN catalog.products p ON p.id = spl.product_id
    WHERE spl.set_id = $1
  `, [setId]);
  
  const calculatedPrice = parseFloat(result.rows[0].total_price) || 0;
  
  // Get modifier settings from set_matrix
  const modifierResult = await pool.query(`
    SELECT 
      sm.modifier_type,
      sm.modifier_operation,
      sm.modifier_value
    FROM product_creator.product_sets ps
    LEFT JOIN product_creator.set_matrices sm ON sm.id = ps.set_matrix_id
    WHERE ps.id = $1
  `, [setId]);
  
  let finalPrice = calculatedPrice;
  
  if (modifierResult.rows.length > 0) {
    const { modifier_type, modifier_operation, modifier_value } = modifierResult.rows[0];
    const modifierValue = parseFloat(modifier_value) || 0;
    
    if (modifierValue !== 0) {
      if (modifier_type === 'percentage') {
        // Apply percentage modifier
        const multiplier = modifier_operation === 'subtract' 
          ? (1 - modifierValue / 100) 
          : (1 + modifierValue / 100);
        finalPrice = calculatedPrice * multiplier;
      } else {
        // Apply amount modifier
        finalPrice = modifier_operation === 'subtract'
          ? calculatedPrice - modifierValue
          : calculatedPrice + modifierValue;
      }
      
      // Clamp to prevent negative prices
      if (finalPrice < 0) {
        console.warn(`⚠️  Set ${setId}: Modifier would result in negative price (${finalPrice.toFixed(2)} PLN), clamping to 0`);
        finalPrice = 0;
      }
    }
  }
  
  // Update both calculated_price and final_price
  await pool.query(`
    UPDATE product_creator.product_sets 
    SET 
      calculated_price = $1, 
      final_price = $2,
      updated_at = CURRENT_TIMESTAMP
    WHERE id = $3
  `, [calculatedPrice, finalPrice, setId]);
  
  console.log(`💰 Set ${setId} - Components: ${calculatedPrice.toFixed(2)} PLN → Final: ${finalPrice.toFixed(2)} PLN`);
  return finalPrice;
}

// Helper function to trigger webhooks after order save
async function triggerWebhooksForOrder(
  source: 'ALLEGRO' | 'SHOPER',
  sourceOrderId: string,
  isNew: boolean
): Promise<void> {
  try {
    // Fetch full order data from database
    const orderResult = await pool.query(`
      SELECT 
        o.*,
        COALESCE(
          json_agg(
            json_build_object(
              'id', oi.id,
              'name', oi.name,
              'offer_external_id', oi.offer_external_id,
              'quantity', oi.quantity,
              'image_url', oi.image_url,
              'returns_quantity', oi.returns_quantity
            ) ORDER BY oi.id
          ) FILTER (WHERE oi.id IS NOT NULL),
          '[]'::json
        ) as items
      FROM commerce.orders o
      LEFT JOIN commerce.order_items oi ON o.id = oi.order_id
      WHERE o.source_order_id = $1 AND o.source = $2
      GROUP BY o.id
    `, [sourceOrderId, source]);

    if (orderResult.rows.length > 0) {
      const fullOrderData = orderResult.rows[0];
      
      // Determine event type based on whether it's new
      const eventType = isNew ? 'order.created' : 'order.updated';
      
      // Add to Odoo sync queue
      if (fullOrderData.order_number) {
        try {
          const { addOrderToSyncQueue } = await import('./odoo-sync.js');
          await addOrderToSyncQueue(
            fullOrderData.order_number,
            fullOrderData.source,
            isNew ? 'create' : 'update'
          );
        } catch (odooError) {
          console.error('❌ Failed to add #' + fullOrderData.order_number, 'to sync queue:', odooError);
        }
      } else {
        console.warn(`⚠️ Skipping Odoo sync for order without order_number (source: ${source}, id: ${sourceOrderId})`);
      }
      
      // Check if payment status is PAID (trigger order.paid)
      if (!isNew && fullOrderData.payment_status === 'PAID') {
        await triggerOrderWebhooks('order.paid', fullOrderData);
      }
      
      // Check if order has tracking numbers (trigger order.shipped)
      const trackingNumbers = fullOrderData.tracking_numbers;
      if (!isNew && trackingNumbers && (
        (typeof trackingNumbers === 'string' && trackingNumbers.trim() !== '') ||
        (Array.isArray(trackingNumbers) && trackingNumbers.length > 0)
      )) {
        await triggerOrderWebhooks('order.shipped', fullOrderData);
      }
      
      // Always trigger main event (created or updated)
      await triggerOrderWebhooks(eventType, fullOrderData);
    }
  } catch (webhookError) {
    console.error('⚠️ Error triggering webhooks:', webhookError);
    // Don't fail if webhooks fail
  }
}

// Helper: Exponential backoff retry
async function retryWithBackoff<T>(
  fn: () => Promise<T>,
  maxRetries: number = 3,
  initialDelay: number = 1000
): Promise<T> {
  let lastError: any;
  
  for (let attempt = 0; attempt < maxRetries; attempt++) {
    try {
      return await fn();
    } catch (error: any) {
      lastError = error;
      
      // Don't retry on 4xx errors (auth issues) - tylko na problemy sieciowe
      if (error?.response?.status && error.response.status >= 400 && error.response.status < 500) {
        throw error;
      }
      
      if (attempt < maxRetries - 1) {
        const delay = initialDelay * Math.pow(2, attempt);
        console.log(`⚠️ Retry ${attempt + 1}/${maxRetries} after ${delay}ms...`);
        await new Promise(resolve => setTimeout(resolve, delay));
      }
    }
  }
  
  throw lastError;
}

async function refreshAccessToken(
  clientId: string,
  clientSecret: string,
  refreshToken: string
): Promise<AllegroTokenResponse> {
  const credentials = Buffer.from(`${clientId}:${clientSecret}`).toString(
    "base64"
  );

  // Używamy retry mechanism dla stabilności (tylko na błędy sieciowe, nie auth)
  return await retryWithBackoff(async () => {
    const response = await axios.post<AllegroTokenResponse>(
      ALLEGRO_TOKEN_URL,
      new URLSearchParams({
        grant_type: "refresh_token",
        refresh_token: refreshToken,
      }),
      {
        headers: {
          Authorization: `Basic ${credentials}`,
          "Content-Type": "application/x-www-form-urlencoded",
        },
      }
    );

    return response.data;
  });
}

async function fetchOrdersFromAllegro(accessToken: string) {
  try {
    const allOrders: AllegroOrderItem[] = [];
    let offset = 0;
    const limit = 100;
    let hasMore = true;

    console.log(`🔄 Starting paginated fetch from Allegro (limit: ${limit} per page)...`);

    while (hasMore) {
      const response = await axios.get<{ checkoutForms: AllegroOrderItem[] }>(
        `${ALLEGRO_API_URL}/order/checkout-forms`,
        {
          headers: {
            Authorization: `Bearer ${accessToken}`,
            Accept: "application/vnd.allegro.public.v1+json",
          },
          params: {
            limit,
            offset,
          },
        }
      );

      const orders = response.data.checkoutForms || [];
      allOrders.push(...orders);

      console.log(`📦 Fetched ${orders.length} orders (offset: ${offset}, total so far: ${allOrders.length})`);

      if (orders.length < limit) {
        hasMore = false;
        console.log(`✅ Finished fetching - got ${allOrders.length} total orders from Allegro`);
      } else {
        offset += limit;
      }
    }

    return allOrders;
  } catch (error) {
    console.error("Error fetching orders from Allegro:", error);
    throw error;
  }
}

async function fetchOrdersFromAllegroWithFilters(
  accessToken: string,
  limit: number = 100,
  offset: number = 0,
  filters?: Record<string, string>
): Promise<AllegroOrderItem[]> {
  try {
    const params: any = { limit, offset };
    if (filters) {
      Object.assign(params, filters);
    }

    const response = await axios.get<{ checkoutForms: AllegroOrderItem[] }>(
      `${ALLEGRO_API_URL}/order/checkout-forms`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: "application/vnd.allegro.public.v1+json",
        },
        params,
      }
    );

    return response.data.checkoutForms || [];
  } catch (error) {
    console.error("Error fetching orders from Allegro with filters:", error);
    throw error;
  }
}

async function fetchRawOrderFromAllegro(accessToken: string, orderId: string) {
  try {
    const response = await axios.get(
      `${ALLEGRO_API_URL}/order/checkout-forms/${orderId}`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: "application/vnd.allegro.public.v1+json",
        },
      }
    );

    return response.data;
  } catch (error) {
    console.error("Error fetching raw order from Allegro:", error);
    throw error;
  }
}

async function fetchShipmentsFromAllegro(accessToken: string, orderId: string) {
  try {
    const response = await axios.get(
      `${ALLEGRO_API_URL}/order/checkout-forms/${orderId}/shipments`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: "application/vnd.allegro.public.v1+json",
        },
      }
    );

    return response.data.shipments || [];
  } catch (error) {
    // Jeśli nie ma przesyłek lub błąd 404, zwróć pustą tablicę
    if (axios.isAxiosError(error) && error.response?.status === 404) {
      return [];
    }
    console.error(`Error fetching shipments for order ${orderId}:`, error);
    return [];
  }
}

async function fetchCustomerReturnsFromAllegro(accessToken: string, orderId: string) {
  try {
    console.log(`🔍 DEBUG: Fetching customer returns for order ${orderId}`);
    const response = await axios.get(
      `${ALLEGRO_API_URL}/order/customer-returns`,
      {
        headers: {
          Authorization: `Bearer ${accessToken}`,
          Accept: "application/vnd.allegro.beta.v1+json",
        },
        params: {
          'checkout-form.id': orderId,
        },
      }
    );

    const customerReturns = response.data.customerReturns || [];
    console.log(`📦 DEBUG: Found ${customerReturns.length} customer returns for order ${orderId}`);
    if (customerReturns.length > 0) {
      console.log(`📦 DEBUG: Returns data:`, JSON.stringify(customerReturns, null, 2));
    }
    
    return customerReturns;
  } catch (error) {
    // Jeśli nie ma zwrotów lub błąd 404, zwróć pustą tablicę
    if (axios.isAxiosError(error) && error.response?.status === 404) {
      console.log(`⚠️ DEBUG: No customer returns found (404) for order ${orderId}`);
      return [];
    }
    console.error(`Error fetching customer returns for order ${orderId}:`, error);
    return [];
  }
}

async function syncOrders() {
  if (isCurrentlySyncing) {
    console.log("Sync already in progress, skipping...");
    return;
  }

  isCurrentlySyncing = true;
  const syncStartedAt = new Date();
  let logId: number | null = null;
  let ordersNew = 0;
  let ordersUpdated = 0;
  let ordersFetched = 0;
  
  try {
    logId = await createSyncLog(syncStartedAt);
    
    const connection = await storage.getAllegroConnection();
    if (!connection || !connection.accessToken || !connection.isActive) {
      console.log("No active connection, skipping sync");
      if (logId) {
        await completeSyncLog(logId, 'ERROR', 0, 0, 0, 'No active connection');
      }
      return;
    }

    // 🔧 ULEPSZENIE: Użyj middleware do zapewnienia ważnego tokenu
    const accessToken = await ensureValidAllegroToken();
    
    if (!accessToken) {
      throw new Error("No valid Allegro token - please re-authenticate");
    }

    // Synchronizacja z pełnym oknem czasowym lub przyrostowa
    const currentSettings = await getSyncSettingsFromPostgres();
    const fullSyncEnabled = currentSettings?.fullSyncEnabled ?? true;
    const fullSyncWindowHours = parseInt(currentSettings?.fullSyncWindowHours || '2', 10);
    const lastSyncAt = currentSettings?.lastSyncAt;
    
    let allegroOrders: AllegroOrderItem[] = [];
    let cutoffTime: string | undefined;
    
    // Pierwsza synchronizacja - zawsze pobierz WSZYSTKIE zamówienia
    if (!lastSyncAt) {
      console.log(`📦 First sync - fetching all Allegro orders (no time filter)`);
      allegroOrders = await fetchOrdersFromAllegro(accessToken);
    } else if (fullSyncEnabled) {
      // Pełna synchronizacja okna czasowego - pobierz zamówienia z ostatnich X godzin
      const windowStart = new Date();
      windowStart.setHours(windowStart.getHours() - fullSyncWindowHours);
      cutoffTime = windowStart.toISOString();
      console.log(`📦 Full window sync - fetching Allegro orders updated in last ${fullSyncWindowHours}h (after ${cutoffTime})`);
    } else {
      // Przyrostowa synchronizacja - tylko od ostatniej synchronizacji
      cutoffTime = new Date(lastSyncAt).toISOString();
      console.log(`📦 Incremental sync - fetching Allegro orders updated after ${cutoffTime}`);
    }
    
    // Pobierz zamówienia z paginacją (jeśli jest cutoffTime i jeszcze nie ma zamówień)
    if (cutoffTime && allegroOrders.length === 0) {
      let offset = 0;
      const limit = 100;
      let hasMore = true;

      while (hasMore) {
        const response = await axios.get<{ checkoutForms: AllegroOrderItem[] }>(
          `${ALLEGRO_API_URL}/order/checkout-forms`,
          {
            headers: {
              Authorization: `Bearer ${accessToken}`,
              Accept: "application/vnd.allegro.public.v1+json",
            },
            params: {
              "updatedAt.gte": cutoffTime,
              limit,
              offset,
            },
          }
        );
        
        const orders = response.data.checkoutForms || [];
        allegroOrders.push(...orders);
        
        console.log(`📦 Fetched ${orders.length} orders (offset: ${offset}, total: ${allegroOrders.length})`);
        
        if (orders.length < limit) {
          hasMore = false;
        } else {
          offset += limit;
        }
      }
    }
    
    ordersFetched = allegroOrders.length;

    for (const allegroOrder of allegroOrders) {
      const paymentStatus =
        allegroOrder.payment?.status?.toUpperCase() || "PENDING";
      const totalAmount =
        allegroOrder.summary?.totalToPay?.amount ||
        allegroOrder.payment?.paidAmount?.amount ||
        "0";

      await storage.createOrUpdateOrder({
        id: allegroOrder.id,
        allegroOrderId: allegroOrder.id,
        buyerLogin: allegroOrder.buyer.login,
        buyerEmail: allegroOrder.buyer.email,
        totalAmount,
        currency:
          allegroOrder.summary?.totalToPay?.currency ||
          allegroOrder.payment?.paidAmount?.currency ||
          "PLN",
        paymentStatus,
        fulfillmentStatus:
          allegroOrder.fulfillment?.status?.toUpperCase() || "NEW",
        itemsCount: allegroOrder.lineItems.length.toString(),
        orderDate: new Date(allegroOrder.updatedAt),
        paymentDate:
          paymentStatus === "PAID" ? new Date(allegroOrder.updatedAt) : null,
        rawData: allegroOrder,
      });

      try {
        // Pobierz dane o przesyłkach dla tego zamówienia
        const shipments = await fetchShipmentsFromAllegro(accessToken, allegroOrder.id);
        const customerReturns = await fetchCustomerReturnsFromAllegro(accessToken, allegroOrder.id);
        
        await saveOrderToPostgres(allegroOrder, shipments, customerReturns, accessToken);
        await enrichAllegroOrderWithImages(allegroOrder, allegroOrder.id);
        
        const result = await saveOrderToCommerce(
          'ALLEGRO',
          allegroOrder.id,
          allegroOrder,
          shipments,
          customerReturns,
          accessToken
        );
        
        // Trigger webhooks after successful save
        await triggerWebhooksForOrder('ALLEGRO', allegroOrder.id, result.isNew);
        
        if (result.isNew) {
          ordersNew++;
        } else {
          ordersUpdated++;
        }
        
        console.log(`✅ Saved Allegro order ${allegroOrder.id} to commerce.orders and allegro.orders (${shipments.length} shipments)`);
      } catch (error) {
        console.error(`Failed to save order ${allegroOrder.id} to PostgreSQL:`, error);
      }
    }

    const settings = await getSyncSettingsFromPostgres();
    await createOrUpdateSyncSettingsInPostgres({
      autoRefreshEnabled: settings?.autoRefreshEnabled ?? true,
      refreshIntervalMinutes: settings?.refreshIntervalMinutes ?? "3",
      lastSyncAt: new Date(),
    });

    // Renumeracja wyłączona - luki w ID są normalną częścią PostgreSQL SERIAL
    // ID: 1,2,3,99,104... są POPRAWNE (luki powstają przez failed transactions)

    console.log(`Successfully synced ${allegroOrders.length} orders from Allegro (${ordersNew} new, ${ordersUpdated} updated)`);
    
    if (logId) {
      await completeSyncLog(logId, 'SUCCESS', ordersFetched, ordersNew, ordersUpdated);
    }
  } catch (error) {
    console.error("Sync error:", error);
    if (logId) {
      await completeSyncLog(logId, 'ERROR', ordersFetched, ordersNew, ordersUpdated, error instanceof Error ? error.message : 'Unknown error');
    }
    throw error;
  } finally {
    isCurrentlySyncing = false;
  }
}

function startSyncScheduler(intervalMinutes: number) {
  if (syncInterval) {
    clearInterval(syncInterval);
    syncInterval = null;
  }

  console.log(`Starting auto-sync scheduler with ${intervalMinutes} minute interval`);
  
  syncInterval = setInterval(
    () => {
      console.log("Running scheduled sync...");
      syncOrders().catch(err => console.error("Scheduled sync failed:", err));
    },
    intervalMinutes * 60 * 1000
  );
}

function stopSyncScheduler() {
  if (syncInterval) {
    clearInterval(syncInterval);
    syncInterval = null;
    console.log("Stopped auto-sync scheduler");
  }
}

function startShoperSyncScheduler(intervalMinutes: number) {
  if (shoperSyncInterval) {
    clearInterval(shoperSyncInterval);
    shoperSyncInterval = null;
  }

  console.log(`Starting Shoper auto-sync scheduler with ${intervalMinutes} minute interval`);
  
  shoperSyncInterval = setInterval(
    () => {
      console.log("Running scheduled Shoper sync...");
      syncShoperOrders().catch(err => console.error("Scheduled Shoper sync failed:", err));
    },
    intervalMinutes * 60 * 1000
  );
}

function stopShoperSyncScheduler() {
  if (shoperSyncInterval) {
    clearInterval(shoperSyncInterval);
    shoperSyncInterval = null;
    console.log("Stopped Shoper auto-sync scheduler");
  }
}

function startOdooSyncScheduler(intervalMinutes: number) {
  if (odooSyncInterval) {
    clearInterval(odooSyncInterval);
    odooSyncInterval = null;
  }

  console.log(`Starting Odoo sync scheduler with ${intervalMinutes} minute interval`);
  
  odooSyncInterval = setInterval(
    async () => {
      console.log("Running scheduled Odoo sync...");
      const { syncRecentOrdersToOdoo, processSyncQueue } = await import('./odoo-sync.js');
      console.log('🔧 Imported Odoo sync functions');
      // Najpierw dodaj nowe zamówienia do kolejki
      await syncRecentOrdersToOdoo(intervalMinutes).catch(err => console.error("Scheduled Odoo recent orders sync failed:", err));
      console.log('🔧 syncRecentOrdersToOdoo completed');
      // Potem przetwórz kolejkę (syncRecentOrdersToOdoo JUŻ wywołuje processSyncQueue, więc NIE wywołuj go ponownie!)
      // await processSyncQueue().catch(err => console.error("Scheduled Odoo queue processing failed:", err));
      console.log('🔧 Scheduled Odoo sync completed');
    },
    intervalMinutes * 60 * 1000
  );
}

function stopOdooSyncScheduler() {
  if (odooSyncInterval) {
    clearInterval(odooSyncInterval);
    odooSyncInterval = null;
    console.log("Stopped Odoo sync scheduler");
  }
}

function startFeesSyncScheduler(intervalMinutes: number) {
  if (feesSyncInterval) {
    clearInterval(feesSyncInterval);
    feesSyncInterval = null;
  }

  console.log(`Starting Allegro fees sync scheduler with ${intervalMinutes} minute interval`);
  
  feesSyncInterval = setInterval(
    async () => {
      console.log("Running scheduled Allegro fees sync...");
      const { fetchAllegroFees, getAllegroToken } = await import('./allegro-api.js');
      try {
        const token = await getAllegroToken();
        await fetchAllegroFees(token, 7);
        console.log('✅ Scheduled Allegro fees sync completed');
      } catch (err) {
        console.error("Scheduled Allegro fees sync failed:", err);
      }
    },
    intervalMinutes * 60 * 1000
  );
}

function stopFeesSyncScheduler() {
  if (feesSyncInterval) {
    clearInterval(feesSyncInterval);
    feesSyncInterval = null;
    console.log("Stopped Allegro fees sync scheduler");
  }
}

function startErrorLogsCleanupScheduler() {
  if (errorLogsCleanupInterval) {
    clearInterval(errorLogsCleanupInterval);
    errorLogsCleanupInterval = null;
  }

  // Run cleanup daily (every 24 hours)
  const intervalMinutes = 1440; // 24 hours
  console.log(`Starting error logs cleanup scheduler (runs daily to clean logs older than 30 days)`);
  
  errorLogsCleanupInterval = setInterval(
    async () => {
      console.log("Running scheduled error logs cleanup...");
      try {
        const deletedCount = await cleanupOldErrorLogs(30);
        console.log(`✅ Cleaned up ${deletedCount} old error logs`);
      } catch (err) {
        console.error("Scheduled error logs cleanup failed:", err);
      }
    },
    intervalMinutes * 60 * 1000
  );
  
  // Also run immediately on startup
  console.log("Running initial error logs cleanup...");
  cleanupOldErrorLogs(30)
    .then(count => console.log(`✅ Initial cleanup: removed ${count} old error logs`))
    .catch(err => console.error("Initial error logs cleanup failed:", err));
}

function stopErrorLogsCleanupScheduler() {
  if (errorLogsCleanupInterval) {
    clearInterval(errorLogsCleanupInterval);
    errorLogsCleanupInterval = null;
    console.log("Stopped error logs cleanup scheduler");
  }
}

/**
 * Nocna pełna synchronizacja - uruchamia się raz dziennie o określonej godzinie
 * Pobiera wszystkie zamówienia z ostatnich N dni (domyślnie 7) aby złapać wszelkie zmiany
 */
async function runNightlyFullSync() {
  try {
    console.log("🌙 [NIGHTLY SYNC] Starting nightly full synchronization...");

    // Sprawdź czy funkcja jest włączona
    const settingsResult = await pool.query(`
      SELECT 
        nightly_sync_enabled,
        nightly_sync_hour,
        nightly_sync_window_days,
        last_nightly_sync_at
      FROM sync_settings
      LIMIT 1
    `);

    if (settingsResult.rows.length === 0) {
      console.log("⚠️  [NIGHTLY SYNC] No sync settings found - skipping");
      return;
    }

    const settings = settingsResult.rows[0];

    if (!settings.nightly_sync_enabled) {
      console.log("⚠️  [NIGHTLY SYNC] Nightly sync is disabled - skipping");
      return;
    }

    // Sprawdź czy już była synchronizacja dzisiaj
    if (settings.last_nightly_sync_at) {
      const lastSync = new Date(settings.last_nightly_sync_at);
      const now = new Date();
      const hoursSinceLastSync = (now.getTime() - lastSync.getTime()) / (1000 * 60 * 60);
      
      if (hoursSinceLastSync < 20) {
        console.log(`⏭️  [NIGHTLY SYNC] Last sync was ${Math.round(hoursSinceLastSync)}h ago - skipping`);
        return;
      }
    }

    // Pobierz connection
    const connection = await getAllegroConnectionFromPostgres();
    if (!connection || !connection.accessToken) {
      console.log("❌ [NIGHTLY SYNC] No Allegro connection found - skipping");
      return;
    }

    const windowDays = settings.nightly_sync_window_days || 7;
    const startDate = new Date();
    startDate.setDate(startDate.getDate() - windowDays);
    const endDate = new Date();

    console.log(`📅 [NIGHTLY SYNC] Fetching orders from ${startDate.toISOString()} to ${endDate.toISOString()} (${windowDays} days window)`);

    // Pobierz zamówienia z paginacją
    let allOrders: AllegroOrderItem[] = [];
    let offset = 0;
    const limit = 100;
    let hasMore = true;
    let pageCount = 0;

    while (hasMore) {
      const orders = await fetchOrdersFromAllegroWithFilters(
        connection.accessToken,
        limit,
        offset,
        {
          'updatedAt.gte': startDate.toISOString(),
          'updatedAt.lte': endDate.toISOString(),
        }
      );

      allOrders.push(...orders);
      pageCount++;
      
      console.log(`📦 [NIGHTLY SYNC] Page ${pageCount}: fetched ${orders.length} orders (total: ${allOrders.length})`);

      if (orders.length < limit) {
        hasMore = false;
      } else {
        offset += limit;
      }
    }

    console.log(`✅ [NIGHTLY SYNC] Fetched ${allOrders.length} orders from Allegro`);

    // Przetwórz zamówienia
    let newCount = 0;
    let updatedCount = 0;
    let errorCount = 0;

    const { saveOrderToPostgres } = await import('./postgres.js');
    
    for (const order of allOrders) {
      try {
        const result = await saveOrderToPostgres(order, [], [], connection.accessToken);
        if (result.isNew) {
          newCount++;
        } else {
          updatedCount++;
        }
      } catch (error: any) {
        errorCount++;
        console.error(`❌ [NIGHTLY SYNC] Error processing order ${order.id}:`, error.message);
      }
    }

    // Zaktualizuj timestamp ostatniej synchronizacji
    await pool.query(`
      UPDATE sync_settings 
      SET 
        last_nightly_sync_at = NOW(),
        updated_at = NOW()
    `);

    console.log(`✅ [NIGHTLY SYNC] Completed: ${newCount} new, ${updatedCount} updated, ${errorCount} errors out of ${allOrders.length} total`);

  } catch (error: any) {
    console.error("❌ [NIGHTLY SYNC] Error during nightly sync:", error);
    
    await logError({
      type: "sync",
      message: `Nightly full sync error: ${error.message}`,
      error,
      context: { component: "nightly-sync" },
    });
  }
}

let nightlySyncInterval: NodeJS.Timeout | null = null;

function startNightlySyncScheduler() {
  if (nightlySyncInterval) {
    clearInterval(nightlySyncInterval);
    nightlySyncInterval = null;
  }

  // Sprawdzaj co godzinę czy pora na nocną synchronizację
  console.log(`Starting nightly sync scheduler (checks hourly for configured time)`);
  
  nightlySyncInterval = setInterval(
    async () => {
      const now = new Date();
      const currentHour = now.getHours();

      // Pobierz ustawioną godzinę z bazy
      const settingsResult = await pool.query(`
        SELECT nightly_sync_enabled, nightly_sync_hour
        FROM sync_settings
        LIMIT 1
      `);

      if (settingsResult.rows.length === 0) {
        return;
      }

      const settings = settingsResult.rows[0];
      const targetHour = settings.nightly_sync_hour || 2;

      // Uruchom sync tylko o określonej godzinie
      if (settings.nightly_sync_enabled && currentHour === targetHour) {
        console.log(`🌙 [NIGHTLY SYNC] It's ${currentHour}:00 - running nightly sync...`);
        await runNightlyFullSync();
      }
    },
    60 * 60 * 1000 // Co godzinę
  );
}

function stopNightlySyncScheduler() {
  if (nightlySyncInterval) {
    clearInterval(nightlySyncInterval);
    nightlySyncInterval = null;
    console.log("Stopped nightly sync scheduler");
  }
}

function startAutoHealingScheduler() {
  if (autoHealingInterval) {
    clearInterval(autoHealingInterval);
    autoHealingInterval = null;
  }

  // Run auto-healing every 15 minutes
  const intervalMinutes = 15;
  console.log(`Starting Auto-Healing scheduler (runs every ${intervalMinutes} min)`);
  
  autoHealingInterval = setInterval(
    async () => {
      console.log("Running scheduled auto-healing cycle...");
      const { runAutoHealing } = await import('./auto-healing.js');
      try {
        await runAutoHealing();
        console.log('✅ Scheduled auto-healing cycle completed');
      } catch (err) {
        console.error("Scheduled auto-healing cycle failed:", err);
      }
    },
    intervalMinutes * 60 * 1000
  );
  
  // Also run once immediately on startup (after 30 seconds to let system stabilize)
  console.log("Scheduling initial auto-healing in 30 seconds...");
  setTimeout(async () => {
    const { runAutoHealing } = await import('./auto-healing.js');
    try {
      console.log("Running initial auto-healing cycle...");
      await runAutoHealing();
      console.log('✅ Initial auto-healing cycle completed');
    } catch (err) {
      console.error("Initial auto-healing cycle failed:", err);
    }
  }, 30000);
}

function stopAutoHealingScheduler() {
  if (autoHealingInterval) {
    clearInterval(autoHealingInterval);
    autoHealingInterval = null;
    console.log("Stopped Auto-Healing scheduler");
  }
}

// Recent updates checker removed - Allegro now uses incremental sync like Shoper

async function getCredentials() {
  const clientId = process.env.ALLEGRO_CLIENT_ID;
  const clientSecret = process.env.ALLEGRO_CLIENT_SECRET;
  
  if (clientId && clientSecret) {
    return { clientId, clientSecret, fromEnv: true };
  }
  
  const connection = await storage.getAllegroConnection();
  if (connection) {
    return { 
      clientId: connection.clientId, 
      clientSecret: connection.clientSecret,
      fromEnv: false 
    };
  }
  
  return null;
}

// 🔧 MIDDLEWARE: Ensure valid Allegro token before API calls
async function ensureValidAllegroToken(): Promise<string | null> {
  try {
    const connection = await getAllegroConnectionFromPostgres();
    
    if (!connection || !connection.accessToken || !connection.refreshToken) {
      return null;
    }
    
    // Sprawdź czy token wygasa w ciągu 15 minut
    const TOKEN_REFRESH_BUFFER_MS = 15 * 60 * 1000;
    const now = new Date().getTime();
    const expiresAt = connection.tokenExpiresAt ? new Date(connection.tokenExpiresAt).getTime() : 0;
    const timeUntilExpiry = expiresAt - now;
    
    // Jeśli token wygasł lub wygasa wkrótce - odśwież go
    if (!connection.tokenExpiresAt || timeUntilExpiry < TOKEN_REFRESH_BUFFER_MS) {
      console.log(`🔄 [Middleware] Token needs refresh (expires in ${Math.round(timeUntilExpiry / 60000)} min)`);
      
      const credentials = await getCredentials();
      if (!credentials) {
        console.error("❌ [Middleware] No credentials for token refresh");
        return null;
      }
      
      const tokenData = await refreshAccessToken(
        credentials.clientId,
        credentials.clientSecret,
        connection.refreshToken
      );
      
      const newExpiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
      
      await updateAllegroConnectionTokens(
        tokenData.access_token,
        tokenData.refresh_token,
        newExpiresAt
      );
      
      await storage.updateConnectionTokens(
        connection.id,
        tokenData.access_token,
        tokenData.refresh_token,
        newExpiresAt
      );
      
      console.log(`✅ [Middleware] Token refreshed - valid for ${Math.round(tokenData.expires_in / 3600)}h`);
      return tokenData.access_token;
    }
    
    // Token jest ważny
    return connection.accessToken;
  } catch (error) {
    console.error("❌ [Middleware] Token validation failed:", error);
    return null;
  }
}

export async function autoConnectAllegro(): Promise<boolean> {
  try {
    const connection = await getAllegroConnectionFromPostgres();
    
    if (!connection) {
      console.log("No Allegro connection found - skipping auto-connect");
      return false;
    }
    
    if (!connection.accessToken || !connection.refreshToken) {
      console.log("No tokens available - manual OAuth required");
      return false;
    }
    
    // 🔧 ULEPSZENIE: Proaktywne odświeżanie - 15 minut przed wygaśnięciem
    const TOKEN_REFRESH_BUFFER_MS = 15 * 60 * 1000; // 15 minut
    const now = new Date().getTime();
    const expiresAt = connection.tokenExpiresAt ? new Date(connection.tokenExpiresAt).getTime() : 0;
    const timeUntilExpiry = expiresAt - now;
    
    const needsRefresh = !connection.tokenExpiresAt || timeUntilExpiry < TOKEN_REFRESH_BUFFER_MS;
    
    if (needsRefresh) {
      if (timeUntilExpiry < 0) {
        console.log("🔄 Allegro token expired - refreshing...");
      } else {
        console.log(`🔄 Allegro token expires soon (${Math.round(timeUntilExpiry / 60000)} min) - proactive refresh...`);
      }
      
      const credentials = await getCredentials();
      if (!credentials) {
        console.log("❌ No credentials available for token refresh");
        return false;
      }
      
      const tokenData = await refreshAccessToken(
        credentials.clientId,
        credentials.clientSecret,
        connection.refreshToken
      );
      
      const newExpiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
      
      await updateAllegroConnectionTokens(
        tokenData.access_token,
        tokenData.refresh_token,
        newExpiresAt
      );
      
      // Also update MemStorage for immediate use
      await storage.updateConnectionTokens(
        connection.id,
        tokenData.access_token,
        tokenData.refresh_token,
        newExpiresAt
      );
      
      // Clear any previous errors
      await pool.query(`
        UPDATE allegro_connections 
        SET connection_error = NULL, 
            last_error_at = NULL, 
            requires_reauth = false 
        WHERE id = $1
      `, [connection.id]);
      
      console.log(`✅ Allegro token refreshed - expires in ${Math.round(tokenData.expires_in / 3600)} hours`);
    } else {
      console.log(`✅ Allegro token valid - expires in ${Math.round(timeUntilExpiry / 60000)} minutes`);
      
      // Clear any previous errors if token is valid
      await pool.query(`
        UPDATE allegro_connections 
        SET connection_error = NULL, 
            last_error_at = NULL, 
            requires_reauth = false,
            is_active = true
        WHERE id = $1
      `, [connection.id]);
    }
    
    // Load connection into MemStorage for use during this session
    await storage.createOrUpdateConnection({
      clientId: connection.clientId,
      clientSecret: connection.clientSecret,
      accessToken: connection.accessToken,
      refreshToken: connection.refreshToken,
      tokenExpiresAt: connection.tokenExpiresAt ?? undefined,
      isActive: true,
    });
    
    console.log("✅ Allegro connection loaded and ready");
    
    return true;
  } catch (error: any) {
    console.error("❌ Auto-connect to Allegro failed:", error);
    
    // Check if this is a token refresh error
    const isTokenError = error?.response?.data?.error === 'invalid_grant' || 
                        error?.response?.status === 400 ||
                        error?.response?.status === 401;
    
    if (isTokenError) {
      let errorMessage = error?.response?.data?.error_description || 
                          error?.response?.data?.error || 
                          'Token refresh failed - manual re-authentication required';
      
      console.error("🔴 Token invalid - requires manual re-authentication");
      console.error("Error details:", errorMessage);
      
      // Truncate error message if it contains a long JWT token
      // "Invalid refresh token: eyJhbGc..." -> "Invalid refresh token"
      if (errorMessage.includes('Invalid refresh token:')) {
        errorMessage = 'Invalid refresh token - token wygasł lub został unieważniony';
      } else if (errorMessage.length > 200) {
        // Limit other error messages to 200 characters
        errorMessage = errorMessage.substring(0, 200) + '...';
      }
      
      // Save error to database
      try {
        await pool.query(`
          UPDATE allegro_connections 
          SET connection_error = $1, 
              last_error_at = NOW(), 
              requires_reauth = true,
              is_active = false
          WHERE id = (SELECT id FROM allegro_connections LIMIT 1)
        `, [errorMessage]);
        
        console.log("✅ Error status saved to database");
        
        // Log to centralized error logs
        await logAllegroError(
          "Token refresh failed - manual re-authentication required",
          error,
          { errorType: error?.response?.data?.error, originalMessage: errorMessage }
        );
      } catch (dbError) {
        console.error("❌ Failed to save error status:", dbError);
      }
    }
    
    return false;
  }
}

let isShoperSyncing = false;

async function enrichShoperOrderWithImages(shoperOrder: any, orderId: string): Promise<void> {
  console.log(`📸 enrichShoperOrderWithImages() for order ${orderId}`);
  
  // Get product data with SKU from raw_data or catalog.products
  const productsFromDb = await pool.query(`
    SELECT 
      soi.shoper_product_id, 
      soi.product_name, 
      soi.image_url,
      COALESCE(soi.raw_data->>'code', cp.external_id) as sku
    FROM shoper.order_items soi
    LEFT JOIN catalog.products cp ON cp.external_id = soi.raw_data->>'code'
    WHERE soi.order_id = (SELECT id FROM shoper.orders WHERE shoper_order_id = $1)
  `, [orderId]);
  
  console.log(`📸 Found ${productsFromDb.rows.length} products from DB for order ${orderId}`);
  
  if (shoperOrder.products_data && productsFromDb.rows.length > 0) {
    const fs = await import('fs/promises');
    const path = await import('path');
    const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
    const adapter = await getFileStorageAdapter();
    
    const productsDir = path.join(process.cwd(), "attached_assets", "products");
    
    shoperOrder.products_data = await Promise.all(shoperOrder.products_data.map(async (product: any) => {
      const dbProduct = productsFromDb.rows.find(
        (p: any) => p.shoper_product_id === (product.product_id || product.id)?.toString()
      );
      
      if (dbProduct) {
        // Try to find image using SKU
        const sku = dbProduct.sku || product.stock?.code || product.code;
        if (sku) {
          const extensions = ['.jpg', '.jpeg', '.png', '.webp'];
          let imageFound = false;
          
          for (const ext of extensions) {
            const filename = `${sku}_1${ext}`;
            const sftpUrl = adapter.getUrl(`products/images/${filename}`);
            
            // Check if exists in SFTP
            if (await adapter.exists(sftpUrl)) {
              product.image_url = sftpUrl;
              console.log(`📸 Using SFTP image for product ${sku}: ${sftpUrl}`);
              imageFound = true;
              break;
            }
            
            // Check if exists locally
            const localImagePath = path.join(productsDir, filename);
            try {
              const fileBuffer = await fs.readFile(localImagePath);
              // Migrate to SFTP
              const mimeType = ext === '.png' ? 'image/png' : ext === '.webp' ? 'image/webp' : 'image/jpeg';
              const uploadedUrl = await adapter.upload({
                filename,
                buffer: fileBuffer,
                subfolder: 'products/images',
                mimetype: mimeType
              });
              product.image_url = uploadedUrl;
              console.log(`📸 Migrated local image to SFTP for ${sku}: ${uploadedUrl}`);
              imageFound = true;
              break;
            } catch {
              // No local image with this extension, continue
            }
          }
          
          // If no image found, try to download from platform URL
          if (!imageFound && dbProduct.image_url) {
            console.log(`📸 No image found for ${sku}, downloading from: ${dbProduct.image_url}`);
            try {
              const { downloadShoperProductImage } = await import('./shoper-api.js');
              const productId = dbProduct.shoper_product_id;
              const filename = await downloadShoperProductImage(dbProduct.image_url, productId);
              
              if (filename) {
                // Check if downloaded file exists and upload to SFTP
                const localPath = path.join(productsDir, filename);
                try {
                  const fileBuffer = await fs.readFile(localPath);
                  const ext = path.extname(filename).toLowerCase();
                  const mimeType = ext === '.png' ? 'image/png' : ext === '.webp' ? 'image/webp' : 'image/jpeg';
                  const uploadedUrl = await adapter.upload({
                    filename,
                    buffer: fileBuffer,
                    subfolder: 'products/images',
                    mimetype: mimeType
                  });
                  product.image_url = uploadedUrl;
                  console.log(`✅ Downloaded and uploaded to SFTP: ${uploadedUrl}`);
                } catch {
                  // Fallback to platform URL
                  product.image_url = dbProduct.image_url;
                  console.log(`⚠️ Upload failed, using platform URL: ${dbProduct.image_url}`);
                }
              } else {
                // Download failed - use platform URL
                product.image_url = dbProduct.image_url;
                console.log(`⚠️ Download failed, using platform URL: ${dbProduct.image_url}`);
              }
            } catch (downloadError) {
              console.error(`❌ Error downloading Shoper image for ${dbProduct.shoper_product_id}:`, downloadError);
              // Fallback to platform URL
              product.image_url = dbProduct.image_url;
            }
          }
        } else if (dbProduct.image_url) {
          // No SKU - fallback to platform URL
          console.log(`📸 No SKU for product ${product.product_id}, using platform URL`);
          product.image_url = dbProduct.image_url;
        }
      }
      return product;
    }));
  }
  console.log(`📸 enrichShoperOrderWithImages() completed`);
}

async function enrichAllegroOrderWithImages(allegroOrder: any, orderId: string): Promise<void> {
  console.log(`📸 enrichAllegroOrderWithImages() for order ${orderId}`);
  
  // Pobierz SKU produktów z zamówienia
  const productsFromDb = await pool.query(`
    SELECT oi.offer_external_id as external_id
    FROM commerce.order_items oi
    INNER JOIN commerce.orders o ON o.id = oi.order_id
    WHERE o.source = 'ALLEGRO' AND o.source_order_id = $1
  `, [orderId]);
  
  console.log(`📸 Found ${productsFromDb.rows.length} Allegro products for order ${orderId}`);
  
  if (allegroOrder.lineItems && productsFromDb.rows.length > 0) {
    const fs = await import('fs/promises');
    const path = await import('path');
    const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
    const adapter = await getFileStorageAdapter();
    const productsDir = path.join(process.cwd(), "attached_assets", "products");
    
    allegroOrder.lineItems = await Promise.all(allegroOrder.lineItems.map(async (lineItem: any) => {
      // Use external.id if available, otherwise use offer.id as fallback
      const externalId = lineItem.offer?.external?.id || lineItem.offer?.id;
      if (externalId) {
        const dbProduct = productsFromDb.rows.find((p: any) => p.external_id === externalId);
        if (dbProduct) {
          const extensions = ['.jpg', '.jpeg', '.png', '.webp'];
          
          for (const ext of extensions) {
            const filename = `${externalId}_1${ext}`;
            const sftpUrl = adapter.getUrl(`products/images/${filename}`);
            
            // Check if exists in SFTP
            if (await adapter.exists(sftpUrl)) {
              lineItem.imageUrl = sftpUrl;
              console.log(`📸 Using SFTP image for Allegro product ${externalId}: ${sftpUrl}`);
              break;
            }
            
            // Check if exists locally
            const localImagePath = path.join(productsDir, filename);
            try {
              const fileBuffer = await fs.readFile(localImagePath);
              // Migrate to SFTP
              const mimeType = ext === '.png' ? 'image/png' : ext === '.webp' ? 'image/webp' : 'image/jpeg';
              const uploadedUrl = await adapter.upload({
                filename,
                buffer: fileBuffer,
                subfolder: 'products/images',
                mimetype: mimeType
              });
              lineItem.imageUrl = uploadedUrl;
              console.log(`📸 Migrated local image to SFTP for Allegro product ${externalId}: ${uploadedUrl}`);
              break;
            } catch {
              // No local image with this extension, continue
            }
          }
        }
      }
      return lineItem;
    }));
  }
  console.log(`📸 enrichAllegroOrderWithImages() completed`);
}

async function syncShoperOrders() {
  if (isShoperSyncing) {
    console.log("Shoper sync already in progress, skipping...");
    return;
  }

  isShoperSyncing = true;
  const syncStartedAt = new Date();
  let ordersNew = 0;
  let ordersUpdated = 0;
  let ordersFetched = 0;
  let ordersFailed = 0;
  
  try {
    const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
    const { saveOrderToCommerce, saveShoperOrderToPostgres } = await import('./postgres.js');

    console.log("🛒 Starting Shoper orders sync...");
    
    // Pobierz słowniki metod płatności i dostaw
    const payments = await getShoperPayments();
    const deliveries = await getShoperDeliveries();
    
    // Use translated title (pl_PL) for payment method name, fallback to technical name
    const paymentMap = new Map(payments.map((p: any) => [p.payment_id, p.translations?.pl_PL?.title || p.name]));
    const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id, d.translations?.pl_PL?.name || d.name]));
    
    console.log(`📋 Loaded ${payments.length} payment methods and ${deliveries.length} delivery methods`);
    console.log(`💳 Payment methods:`, Array.from(paymentMap.entries()));
    console.log(`🚚 Delivery methods:`, Array.from(deliveryMap.entries()));
    
    const settings = await getSyncSettingsFromPostgres();
    const shoperFullSyncEnabled = settings?.shoperFullSyncEnabled ?? true;
    const shoperFullSyncWindowHours = parseInt(settings?.shoperFullSyncWindowHours || '2', 10);
    const lastShoperSyncAt = settings?.lastShoperSyncAt;
    
    let filters: Record<string, any> | undefined;
    
    // Pierwsza synchronizacja - zawsze pobierz WSZYSTKIE zamówienia
    if (!lastShoperSyncAt) {
      console.log("📦 First sync - fetching all Shoper orders (no time filter)");
      filters = undefined;
    } else if (shoperFullSyncEnabled) {
      // Pełna synchronizacja okna czasowego - pobierz zamówienia z ostatnich X godzin
      const windowStart = new Date();
      windowStart.setHours(windowStart.getHours() - shoperFullSyncWindowHours);
      const filterDate = windowStart.toISOString().replace('T', ' ').replace('Z', '');
      filters = {
        status_date: {
          ">": filterDate
        }
      };
      console.log(`🔍 Full window sync - fetching Shoper orders with status_date in last ${shoperFullSyncWindowHours}h (after ${filterDate})`);
    } else {
      // Przyrostowa synchronizacja - tylko od ostatniej synchronizacji
      const filterDate = new Date(lastShoperSyncAt).toISOString().replace('T', ' ').replace('Z', '');
      filters = {
        status_date: {
          ">": filterDate
        }
      };
      console.log(`🔍 Incremental sync - fetching Shoper orders with status_date after ${filterDate}`);
    }
    
    const shoperOrders = await getShoperOrders(100, filters);
    ordersFetched = shoperOrders.length;

    console.log(`📦 Fetched ${ordersFetched} orders from Shoper API`);

    for (const shoperOrder of shoperOrders) {
      try {
        const orderId = shoperOrder.order_id || shoperOrder.id;
        
        console.log(`📦 Processing Shoper order ${orderId}...`);
        
        // Wzbogać zamówienie o nazwy metod płatności i dostaw
        if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
          shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
        } else if (shoperOrder.payment_id) {
          console.log(`⚠️ Payment ID ${shoperOrder.payment_id} not found in paymentMap`);
        }
        if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
          shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
        }
        
        try {
          const orderProducts = await getShoperOrderProducts(orderId);
          shoperOrder.products_data = orderProducts;
          console.log(`✅ Fetched ${orderProducts.length} products for order ${orderId}`);
        } catch (productError) {
          console.warn(`⚠️ Could not fetch products for order ${orderId}, will save order without line items:`, productError);
          shoperOrder.products_data = [];
        }
        
        // Pobierz przesyłki dla zamówienia
        let parcels: any[] = [];
        try {
          parcels = await getShoperParcels(orderId);
          console.log(`✅ Fetched ${parcels.length} parcels for order ${orderId}`);
        } catch (parcelError) {
          console.warn(`⚠️ Could not fetch parcels for order ${orderId}:`, parcelError);
        }
        
        await saveShoperOrderToPostgres(shoperOrder, parcels);
        await enrichShoperOrderWithImages(shoperOrder, orderId?.toString());
        
        const result = await saveOrderToCommerce(
          'SHOPER',
          orderId?.toString(),
          shoperOrder,
          parcels,
          [] // Shoper nie ma zwrotów
        );
        
        // Trigger webhooks after successful save
        await triggerWebhooksForOrder('SHOPER', orderId?.toString(), result.isNew);
        
        if (result.isNew) {
          ordersNew++;
        } else {
          ordersUpdated++;
        }
        
        console.log(`✅ Saved Shoper order ${orderId} to commerce.orders and shoper.orders (${parcels.length} parcels)`);
      } catch (error) {
        ordersFailed++;
        console.error(`❌ Failed to process Shoper order ${shoperOrder.order_id}:`, error);
      }
    }

    await createOrUpdateSyncSettingsInPostgres({
      autoRefreshEnabled: settings?.autoRefreshEnabled ?? true,
      refreshIntervalMinutes: settings?.refreshIntervalMinutes ?? "3",
      lastSyncAt: settings?.lastSyncAt || null,
      shoperAutoRefreshEnabled: settings?.shoperAutoRefreshEnabled ?? true,
      shoperRefreshIntervalMinutes: settings?.shoperRefreshIntervalMinutes ?? "5",
      lastShoperSyncAt: new Date(),
    });

    // Renumeracja wyłączona - luki w ID są normalną częścią PostgreSQL SERIAL
    // ID: 1,2,3,99,104... są POPRAWNE (luki powstają przez failed transactions)

    const message = `Successfully synced Shoper orders: ${ordersNew} new, ${ordersUpdated} updated, ${ordersFailed} failed out of ${ordersFetched} total`;
    console.log(`✅ ${message}`);
    
    return { 
      success: true, 
      ordersFetched, 
      ordersNew, 
      ordersUpdated,
      ordersFailed
    };
  } catch (error) {
    console.error("❌ Shoper sync error:", error);
    throw error;
  } finally {
    isShoperSyncing = false;
  }
}

export async function registerRoutes(app: Express, skipDatabase: boolean = false): Promise<Server> {
  // Setup authentication (with or without database-backed sessions)
  setupAuth(app, skipDatabase);

  // ==================== FILE UPLOAD CONFIGURATION ====================
  
  // Helper functions for generating filenames
  function generateProductImageFilename(productId: string, ext: string): string {
    return `${productId}_1${ext}`;
  }

  function generateCatalogImageFilename(ext: string): string {
    const uniqueSuffix = `${Date.now()}-${randomBytes(6).toString('hex')}`;
    return `product-${uniqueSuffix}${ext}`;
  }

  function generateAccessoryImageFilename(ext: string): string {
    const uniqueSuffix = `${Date.now()}-${randomBytes(6).toString('hex')}`;
    return `accessory-${uniqueSuffix}${ext}`;
  }

  function generateAddonImageFilename(ext: string): string {
    const uniqueSuffix = `${Date.now()}-${randomBytes(6).toString('hex')}`;
    return `addon-${uniqueSuffix}${ext}`;
  }

  function generateWarehouseMaterialImageFilename(ext: string): string {
    const uniqueSuffix = `${Date.now()}-${randomBytes(6).toString('hex')}`;
    return `material-${uniqueSuffix}${ext}`;
  }

  // Shared multer instances with memory storage (files funneled through FileStorageAdapter)
  const uploadProductImage = multer({
    storage: multer.memoryStorage(),
    limits: { fileSize: 5 * 1024 * 1024 }, // 5MB
    fileFilter: (req, file, cb) => {
      const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp'];
      if (allowedTypes.includes(file.mimetype)) {
        cb(null, true);
      } else {
        cb(new Error('Invalid file type. Only JPEG, PNG and WebP are allowed.'));
      }
    }
  });

  const uploadCatalogImage = multer({
    storage: multer.memoryStorage(),
    limits: { fileSize: 10 * 1024 * 1024 }, // 10MB
    fileFilter: (req, file, cb) => {
      const allowedMimes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
      if (allowedMimes.includes(file.mimetype)) {
        cb(null, true);
      } else {
        cb(new Error('Invalid file type. Only JPEG, PNG, GIF and WebP are allowed.'));
      }
    }
  });

  const uploadAccessoryImage = multer({
    storage: multer.memoryStorage(),
    limits: { fileSize: 10 * 1024 * 1024 }, // 10MB
    fileFilter: (req, file, cb) => {
      const allowedMimes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
      if (allowedMimes.includes(file.mimetype)) {
        cb(null, true);
      } else {
        cb(new Error('Invalid file type. Only JPEG, PNG, GIF and WebP are allowed.'));
      }
    }
  });

  const uploadAddonImage = multer({
    storage: multer.memoryStorage(),
    limits: { fileSize: 10 * 1024 * 1024 }, // 10MB
    fileFilter: (req, file, cb) => {
      const allowedMimes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
      if (allowedMimes.includes(file.mimetype)) {
        cb(null, true);
      } else {
        cb(new Error('Invalid file type. Only JPEG, PNG, GIF and WebP are allowed.'));
      }
    }
  });

  const uploadWarehouseMaterialImage = multer({
    storage: multer.memoryStorage(),
    limits: { fileSize: 10 * 1024 * 1024 }, // 10MB
    fileFilter: (req, file, cb) => {
      const allowedMimes = ['image/jpeg', 'image/jpg', 'image/png', 'image/gif', 'image/webp'];
      if (allowedMimes.includes(file.mimetype)) {
        cb(null, true);
      } else {
        cb(new Error('Invalid file type. Only JPEG, PNG, GIF and WebP are allowed.'));
      }
    }
  });

  // Generic upload middleware for other file types
  const upload = multer({
    storage: multer.memoryStorage(),
    limits: { fileSize: 50 * 1024 * 1024 }, // 50MB limit
  });

  // SSE endpoint for streaming generation logs
  app.get('/api/generation-logs/stream/:sessionId', isAuthenticated, (req, res) => {
    const { sessionId } = req.params;
    
    // Set headers for Server-Sent Events
    res.setHeader('Content-Type', 'text/event-stream');
    res.setHeader('Cache-Control', 'no-cache');
    res.setHeader('Connection', 'keep-alive');
    res.setHeader('X-Accel-Buffering', 'no'); // Disable nginx buffering
    
    // Send initial connection message
    res.write(`data: ${JSON.stringify({ type: 'info', message: 'Połączono z streamem logów', timestamp: new Date().toISOString() })}\n\n`);
    
    // Create listener for this session
    const logListener = (logData: any) => {
      res.write(`data: ${JSON.stringify(logData)}\n\n`);
    };
    
    logStreamEmitter.on(sessionId, logListener);
    
    // Cleanup on client disconnect
    req.on('close', () => {
      logStreamEmitter.removeListener(sessionId, logListener);
      console.log(`📡 SSE client disconnected for session: ${sessionId}`);
    });
    
    console.log(`📡 SSE client connected for session: ${sessionId}`);
  });

  // Swagger API Documentation (accessible to authenticated users)
  app.use('/api-docs', swaggerUi.serve);
  app.get('/api-docs', swaggerUi.setup(swaggerSpec, {
    customCss: '.swagger-ui .topbar { display: none }',
    customSiteTitle: 'Alpma OMS API Documentation',
  }));

  // Auth routes
  app.post("/api/register", async (req, res, next) => {
    try {
      const { username, email, password, firstName, lastName } = req.body;

      // Check if username already exists
      const existingUser = await getUserByUsername(username);
      if (existingUser) {
        return res.status(400).json({ message: "Nazwa użytkownika jest już zajęta" });
      }

      // Check if email already exists
      const existingEmail = await getUserByEmail(email);
      if (existingEmail) {
        return res.status(400).json({ message: "Email jest już zarejestrowany" });
      }

      // Hash password and create user
      const hashedPassword = await hashPassword(password);
      const user = await createUser({
        username,
        email,
        password: hashedPassword,
        firstName,
        lastName,
      });

      // Log in the user after registration
      req.login(user, (err) => {
        if (err) return next(err);
        const { password: _, ...userWithoutPassword } = user;
        res.status(201).json(userWithoutPassword);
      });
    } catch (error: any) {
      console.error("Registration error:", error);
      res.status(500).json({ message: "Błąd podczas rejestracji" });
    }
  });

  app.post("/api/login", (req, res, next) => {
    passport.authenticate("local", (err: any, user: any, info: any) => {
      if (err) {
        return next(err);
      }
      if (!user) {
        return res.status(401).json({ message: info?.message || "Nieprawidłowe dane logowania" });
      }
      req.login(user, (err) => {
        if (err) {
          return next(err);
        }
        res.status(200).json(user);
      });
    })(req, res, next);
  });

  app.post("/api/logout", (req, res, next) => {
    req.logout((err) => {
      if (err) return next(err);
      res.sendStatus(200);
    });
  });

  app.get("/api/user", (req, res) => {
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Unauthorized" });
    }
    res.json(req.user);
  });

  // GET /api/stats/counts - Pobierz liczniki dla menu sidebara
  app.get("/api/stats/counts", isAuthenticated, async (req, res) => {
    try {
      // Produkty - katalog produktów
      const catalogProductsCount = await pool.query(`
        SELECT COUNT(*)::int as count FROM catalog.products WHERE is_active = true
      `);

      // Produkty - katalog zestawów
      const catalogSetsCount = await pool.query(`
        SELECT COUNT(*)::int as count FROM product_creator.product_sets WHERE is_active = true
      `);

      // Produkty - matryce produktów
      const productMatricesCount = await pool.query(`
        SELECT COUNT(*)::int as count FROM product_creator.product_matrices
      `);

      // Produkty - matryce zestawów
      const setMatricesCount = await pool.query(`
        SELECT COUNT(*)::int as count FROM product_creator.set_matrices WHERE is_active = true
      `);

      // Produkty - akcesoria
      const accessoriesCount = await pool.query(`
        SELECT COUNT(*)::int as count FROM catalog.accessories WHERE is_active = true
      `);

      // Magazyn - liczby materiałów w każdej kategorii
      const warehouseCategoryCounts = await pool.query(`
        SELECT 
          mg.category,
          COUNT(m.id)::int as count
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        WHERE m.is_active = true
        GROUP BY mg.category
      `);

      // Magazyn - liczby materiałów w każdej grupie
      const warehouseGroupCounts = await pool.query(`
        SELECT 
          mg.id,
          mg.code,
          mg.name,
          mg.category,
          COUNT(m.id)::int as count
        FROM warehouse.material_groups mg
        LEFT JOIN warehouse.materials m ON m.group_id = mg.id AND m.is_active = true
        WHERE mg.is_active = true
        GROUP BY mg.id, mg.code, mg.name, mg.category
        ORDER BY mg.display_order, mg.name
      `);

      // Akcesoria - liczby akcesoriów w każdej grupie
      const accessoryGroupCounts = await pool.query(`
        SELECT 
          ag.id,
          ag.code,
          ag.name,
          ag.category,
          COUNT(a.id)::int as count
        FROM catalog.accessory_groups ag
        LEFT JOIN catalog.accessories a ON a.group_id = ag.id AND a.is_active = true
        WHERE ag.is_active = true
        GROUP BY ag.id, ag.code, ag.name, ag.category
        ORDER BY ag.display_order, ag.name
      `);

      // Przekształć wyniki na obiekt
      const categoryCounts: Record<string, number> = {};
      warehouseCategoryCounts.rows.forEach((row: any) => {
        if (row.category) {
          categoryCounts[row.category] = row.count;
        }
      });

      res.json({
        products: {
          catalogProducts: catalogProductsCount.rows[0].count,
          catalogSets: catalogSetsCount.rows[0].count,
          productMatrices: productMatricesCount.rows[0].count,
          setMatrices: setMatricesCount.rows[0].count,
          accessories: accessoriesCount.rows[0].count,
        },
        warehouse: {
          categories: categoryCounts,
          groups: warehouseGroupCounts.rows,
        },
        accessories: {
          groups: accessoryGroupCounts.rows,
        },
      });
    } catch (error) {
      console.error("❌ Error fetching stats counts:", error);
      res.status(500).json({ error: "Failed to fetch stats counts" });
    }
  });

  // Password reset routes
  app.post("/api/password-reset/request", async (req, res) => {
    try {
      const { email } = req.body;
      const user = await getUserByEmail(email);
      
      if (!user) {
        // Don't reveal if email exists or not for security
        return res.status(200).json({ 
          message: "Jeśli email istnieje w systemie, otrzymasz link do resetowania hasła" 
        });
      }

      // Generate reset token
      const resetToken = randomBytes(32).toString("hex");
      const expiresAt = new Date(Date.now() + 3600000); // 1 hour
      
      await createPasswordResetToken(user.id, resetToken, expiresAt);

      // In production, send email with reset link here
      // For now, just return the token (development only)
      res.status(200).json({ 
        message: "Link do resetowania hasła został wysłany na podany email",
        resetToken, // Remove in production!
      });
    } catch (error) {
      console.error("Password reset request error:", error);
      res.status(500).json({ message: "Błąd podczas żądania resetowania hasła" });
    }
  });

  app.post("/api/password-reset/verify", async (req, res) => {
    try {
      const { token } = req.body;
      const resetToken = await getPasswordResetToken(token);
      
      if (!resetToken) {
        return res.status(400).json({ message: "Nieprawidłowy lub wygasły token" });
      }

      res.status(200).json({ message: "Token jest prawidłowy" });
    } catch (error) {
      console.error("Token verification error:", error);
      res.status(500).json({ message: "Błąd podczas weryfikacji tokenu" });
    }
  });

  app.post("/api/password-reset/reset", async (req, res) => {
    try {
      const { token, newPassword } = req.body;
      const resetToken = await getPasswordResetToken(token);
      
      if (!resetToken) {
        return res.status(400).json({ message: "Nieprawidłowy lub wygasły token" });
      }

      // Hash new password and update
      const hashedPassword = await hashPassword(newPassword);
      await updateUserPassword(resetToken.user_id, hashedPassword);
      await markTokenAsUsed(token);

      res.status(200).json({ message: "Hasło zostało pomyślnie zresetowane" });
    } catch (error) {
      console.error("Password reset error:", error);
      res.status(500).json({ message: "Błąd podczas resetowania hasła" });
    }
  });

  // Permission middleware
  const requirePermission = (permission: Permission) => {
    return (req: any, res: any, next: any) => {
      if (!req.isAuthenticated()) {
        return res.status(401).json({ message: "Unauthorized" });
      }
      
      const userRole = req.user.role;
      if (!hasPermission(userRole, permission)) {
        return res.status(403).json({ 
          message: "Forbidden - Insufficient permissions",
          required: permission,
          role: userRole
        });
      }
      
      next();
    };
  };

  // User management routes (admin only)
  const requireAdmin = (req: any, res: any, next: any) => {
    if (!req.isAuthenticated()) {
      return res.status(401).json({ message: "Unauthorized" });
    }
    if (req.user.role !== 'admin') {
      return res.status(403).json({ message: "Forbidden - Admin access required" });
    }
    next();
  };

  app.get("/api/users", requireAdmin, async (req, res) => {
    try {
      const users = await getAllUsers();
      // Don't send passwords to frontend
      const usersWithoutPasswords = users.map(({ password, ...user }: any) => user);
      res.json(usersWithoutPasswords);
    } catch (error) {
      console.error("Get users error:", error);
      res.status(500).json({ message: "Błąd podczas pobierania użytkowników" });
    }
  });

  app.post("/api/users", requireAdmin, async (req, res) => {
    try {
      const { username, email, password, firstName, lastName, role } = req.body;

      if (!username || !email || !password) {
        return res.status(400).json({ message: "Nazwa użytkownika, email i hasło są wymagane" });
      }

      if (password.length < 6) {
        return res.status(400).json({ message: "Hasło musi mieć minimum 6 znaków" });
      }

      const existingUser = await getUserByUsername(username);
      if (existingUser) {
        return res.status(400).json({ message: "Użytkownik o tej nazwie już istnieje" });
      }

      const hashedPwd = await hashPassword(password);
      const newUser = await createUser({
        username,
        email,
        password: hashedPwd,
        firstName: firstName || undefined,
        lastName: lastName || undefined,
      });

      if (role && role !== "user") {
        await updateUserRole(newUser.id, role);
      }

      const { password: _, ...userWithoutPassword } = newUser;
      res.status(201).json(userWithoutPassword);
    } catch (error) {
      console.error("Create user error:", error);
      res.status(500).json({ message: "Błąd podczas tworzenia użytkownika" });
    }
  });

  app.put("/api/users/:id/role", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id, 10);
      const { role } = req.body;
      
      if (!['admin', 'manager', 'user'].includes(role)) {
        return res.status(400).json({ message: "Nieprawidłowa rola" });
      }

      const user = await updateUserRole(userId, role);
      const { password, ...userWithoutPassword } = user;
      res.json(userWithoutPassword);
    } catch (error) {
      console.error("Update user role error:", error);
      res.status(500).json({ message: "Błąd podczas aktualizacji roli" });
    }
  });

  app.put("/api/users/:id/permissions", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id, 10);
      const { permissions } = req.body;
      
      const user = await updateUserPermissions(userId, permissions);
      const { password, ...userWithoutPassword } = user;
      res.json(userWithoutPassword);
    } catch (error) {
      console.error("Update user permissions error:", error);
      res.status(500).json({ message: "Błąd podczas aktualizacji uprawnień" });
    }
  });

  app.put("/api/users/:id/status", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id, 10);
      const { isActive } = req.body;
      
      const user = await updateUserStatus(userId, isActive);
      const { password, ...userWithoutPassword } = user;
      res.json(userWithoutPassword);
    } catch (error) {
      console.error("Update user status error:", error);
      res.status(500).json({ message: "Błąd podczas aktualizacji statusu" });
    }
  });

  app.put("/api/users/:id/password", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id, 10);
      const { newPassword } = req.body;
      
      if (!newPassword || newPassword.length < 6) {
        return res.status(400).json({ message: "Hasło musi mieć minimum 6 znaków" });
      }

      const hashedPassword = await hashPassword(newPassword);
      await updateUserPassword(userId, hashedPassword);
      
      res.json({ message: "Hasło zostało zmienione" });
    } catch (error) {
      console.error("Update user password error:", error);
      res.status(500).json({ message: "Błąd podczas zmiany hasła" });
    }
  });

  app.delete("/api/users/:id", requireAdmin, async (req, res) => {
    try {
      const userId = parseInt(req.params.id, 10);
      
      // Prevent deleting yourself
      if (req.user?.id === userId) {
        return res.status(400).json({ message: "Nie możesz usunąć własnego konta" });
      }

      await deleteUser(userId);
      res.status(200).json({ message: "Użytkownik został usunięty" });
    } catch (error) {
      console.error("Delete user error:", error);
      res.status(500).json({ message: "Błąd podczas usuwania użytkownika" });
    }
  });

  // =====================================================
  // User Page Settings (filters, column visibility, column order)
  // =====================================================
  
  app.get("/api/user-settings/:pageSlug", isAuthenticated, async (req, res) => {
    try {
      const userId = req.user!.id;
      const { pageSlug } = req.params;
      
      const settings = await storage.getUserPageSettings(userId, pageSlug);
      
      if (!settings) {
        return res.json({
          filters: {},
          columnVisibility: {},
          columnWidths: {},
          columnOrder: [],
        });
      }
      
      res.json(settings);
    } catch (error) {
      console.error("Get user page settings error:", error);
      res.status(500).json({ message: "Błąd podczas pobierania ustawień strony" });
    }
  });
  
  app.put("/api/user-settings/:pageSlug", isAuthenticated, async (req, res) => {
    try {
      const userId = req.user!.id;
      const { pageSlug } = req.params;
      const { filters, columnVisibility, columnOrder, columnWidths } = req.body;
      
      await storage.upsertUserPageSettings(userId, pageSlug, {
        filters,
        columnVisibility,
        columnOrder,
        columnWidths,
      });
      
      res.json({ message: "Ustawienia zostały zapisane" });
    } catch (error) {
      console.error("Update user page settings error:", error);
      res.status(500).json({ message: "Błąd podczas zapisywania ustawień strony" });
    }
  });

  // ==================== Error Logs API ====================

  app.get("/api/error-logs", requireAdmin, async (req, res) => {
    try {
      const { type, severity, resolved, limit = "100", offset = "0", search } = req.query;
      
      let query = `
        SELECT id, type, message, stack_trace, context, severity, timestamp, resolved_at, resolved_by, created_at
        FROM error_logs
        WHERE 1=1
      `;
      const params: any[] = [];
      let paramIndex = 1;

      // Filtrowanie po typie
      if (type && type !== 'all') {
        query += ` AND type = $${paramIndex}`;
        params.push(type);
        paramIndex++;
      }

      // Filtrowanie po severity
      if (severity && severity !== 'all') {
        query += ` AND severity = $${paramIndex}`;
        params.push(severity);
        paramIndex++;
      }

      // Filtrowanie po statusie resolved
      if (resolved === 'true') {
        query += ` AND resolved_at IS NOT NULL`;
      } else if (resolved === 'false') {
        query += ` AND resolved_at IS NULL`;
      }

      // Wyszukiwanie w message
      if (search && typeof search === 'string' && search.trim()) {
        query += ` AND message ILIKE $${paramIndex}`;
        params.push(`%${search}%`);
        paramIndex++;
      }

      // Sortowanie i paginacja
      query += ` ORDER BY timestamp DESC LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`;
      params.push(parseInt(limit as string, 10), parseInt(offset as string, 10));

      const result = await pool.query(query, params);

      // Pobierz całkowitą liczbę dla paginacji
      let countQuery = `SELECT COUNT(*) FROM error_logs WHERE 1=1`;
      const countParams: any[] = [];
      let countParamIndex = 1;

      if (type && type !== 'all') {
        countQuery += ` AND type = $${countParamIndex}`;
        countParams.push(type);
        countParamIndex++;
      }

      if (severity && severity !== 'all') {
        countQuery += ` AND severity = $${countParamIndex}`;
        countParams.push(severity);
        countParamIndex++;
      }

      if (resolved === 'true') {
        countQuery += ` AND resolved_at IS NOT NULL`;
      } else if (resolved === 'false') {
        countQuery += ` AND resolved_at IS NULL`;
      }

      if (search && typeof search === 'string' && search.trim()) {
        countQuery += ` AND message ILIKE $${countParamIndex}`;
        countParams.push(`%${search}%`);
      }

      const countResult = await pool.query(countQuery, countParams);
      const total = parseInt(countResult.rows[0].count, 10);

      res.json({
        logs: result.rows,
        total,
        limit: parseInt(limit as string, 10),
        offset: parseInt(offset as string, 10),
      });
    } catch (error) {
      console.error("Get error logs error:", error);
      res.status(500).json({ error: "Failed to get error logs" });
    }
  });

  app.patch("/api/error-logs/:id/resolve", requireAdmin, async (req, res) => {
    try {
      const logId = parseInt(req.params.id, 10);
      const userId = req.user?.id;

      await pool.query(`
        UPDATE error_logs
        SET resolved_at = NOW(), resolved_by = $1
        WHERE id = $2
      `, [userId, logId]);

      res.json({ message: "Error log marked as resolved" });
    } catch (error) {
      console.error("Resolve error log error:", error);
      res.status(500).json({ error: "Failed to resolve error log" });
    }
  });

  app.delete("/api/error-logs/:id", requireAdmin, async (req, res) => {
    try {
      const logId = parseInt(req.params.id, 10);

      await pool.query(`DELETE FROM error_logs WHERE id = $1`, [logId]);

      res.json({ message: "Error log deleted" });
    } catch (error) {
      console.error("Delete error log error:", error);
      res.status(500).json({ error: "Failed to delete error log" });
    }
  });

  app.post("/api/error-logs/cleanup", requireAdmin, async (req, res) => {
    try {
      const { days = 30 } = req.body;
      const deletedCount = await cleanupOldErrorLogs(days);

      res.json({ 
        message: `Cleaned up ${deletedCount} old error logs`,
        deletedCount 
      });
    } catch (error) {
      console.error("Cleanup error logs error:", error);
      res.status(500).json({ error: "Failed to cleanup error logs" });
    }
  });

  // ==================== End Error Logs API ====================

  app.get("/api/allegro/auth", requirePermission('manage_credentials'), async (req, res) => {
    try {
      const credentials = await getCredentials();
      if (!credentials) {
        return res.status(400).json({ error: "No credentials configured" });
      }

      const state = randomBytes(32).toString("hex");
      oauthStates.set(state, { timestamp: Date.now() });
      
      setTimeout(() => oauthStates.delete(state), 10 * 60 * 1000);

      const redirectUri = `https://${req.get("host")}/api/allegro/callback`;
      
      const authUrl = `${ALLEGRO_AUTH_URL}?response_type=code&client_id=${credentials.clientId}&redirect_uri=${encodeURIComponent(redirectUri)}&state=${state}`;

      res.redirect(authUrl);
    } catch (error) {
      console.error("Auth initiation error:", error);
      res.status(500).json({ error: "Auth initiation failed" });
    }
  });

  app.get("/api/allegro/callback", async (req, res) => {
    try {
      const { code, state } = req.query;
      
      if (!code || typeof code !== "string") {
        return res.redirect("/settings?error=no_code");
      }

      if (!state || typeof state !== "string" || !oauthStates.has(state)) {
        return res.redirect("/settings?error=invalid_state");
      }

      oauthStates.delete(state);

      const credentials = await getCredentials();
      if (!credentials) {
        return res.redirect("/settings?error=no_credentials");
      }

      const redirectUri = `https://${req.get("host")}/api/allegro/callback`;
      const authHeader = Buffer.from(
        `${credentials.clientId}:${credentials.clientSecret}`
      ).toString("base64");

      const response = await axios.post<AllegroTokenResponse>(
        ALLEGRO_TOKEN_URL,
        new URLSearchParams({
          grant_type: "authorization_code",
          code,
          redirect_uri: redirectUri,
        }),
        {
          headers: {
            Authorization: `Basic ${authHeader}`,
            "Content-Type": "application/x-www-form-urlencoded",
          },
        }
      );

      const { access_token, refresh_token, expires_in } = response.data;
      const expiresAt = new Date(Date.now() + expires_in * 1000);

      // Zapisz token do PostgreSQL (persystencja po restarcie)
      await saveAllegroConnectionToPostgres({
        clientId: credentials.clientId,
        clientSecret: credentials.clientSecret,
        accessToken: access_token,
        refreshToken: refresh_token,
        tokenExpiresAt: expiresAt,
        isActive: true,
      });

      // Zapisz też do MemStorage (do użycia w tej sesji)
      await storage.createOrUpdateConnection({
        clientId: credentials.clientId,
        clientSecret: credentials.clientSecret,
        accessToken: access_token,
        refreshToken: refresh_token,
        tokenExpiresAt: expiresAt,
        isActive: true,
      });

      await syncOrders();

      const settings = await getSyncSettingsFromPostgres();
      if (settings?.autoRefreshEnabled) {
        const interval = parseInt(settings.refreshIntervalMinutes || "3", 10);
        startSyncScheduler(interval);
      }

      res.redirect("/settings?connected=true");
    } catch (error) {
      console.error("OAuth callback error:", error);
      res.redirect("/settings?error=auth_failed");
    }
  });

  app.get("/api/allegro/connection", async (req, res) => {
    try {
      const credentials = await getCredentials();
      if (!credentials) {
        return res.json({ 
          isActive: false,
          hasCredentials: false,
          fromEnv: false
        });
      }

      // Read from PostgreSQL instead of MemStorage to persist across restarts
      const result = await pool.query(`
        SELECT is_active, requires_reauth, connection_error
        FROM allegro_connections
        LIMIT 1
      `);
      
      const isActive = result.rows.length > 0 ? result.rows[0].is_active : false;
      
      res.json({
        isActive,
        hasCredentials: true,
        fromEnv: credentials.fromEnv,
        clientId: credentials.clientId,
        requiresReauth: result.rows[0]?.requires_reauth || false,
        error: result.rows[0]?.connection_error || null,
      });
    } catch (error) {
      console.error("Get connection error:", error);
      res.status(500).json({ error: "Failed to get connection status" });
    }
  });

  // New endpoint to check connection status including errors
  app.get("/api/allegro/connection-status", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          is_active,
          connection_error,
          last_error_at,
          requires_reauth,
          token_expires_at
        FROM allegro_connections
        LIMIT 1
      `);

      if (result.rows.length === 0) {
        return res.json({
          connected: false,
          requiresReauth: false,
          error: null,
        });
      }

      const conn = result.rows[0];
      
      res.json({
        connected: conn.is_active,
        requiresReauth: conn.requires_reauth || false,
        error: conn.connection_error,
        lastErrorAt: conn.last_error_at,
        tokenExpiresAt: conn.token_expires_at,
      });
    } catch (error) {
      console.error("Get connection status error:", error);
      res.status(500).json({ error: "Failed to get connection status" });
    }
  });

  app.get("/api/allegro/balance", isAuthenticated, async (req, res) => {
    try {
      const connection = await storage.getAllegroConnection();
      
      if (!connection || !connection.accessToken) {
        return res.status(401).json({ error: "No active Allegro connection" });
      }

      const { fetchAllegroBalance } = await import('./allegro-api.js');
      const balance = await fetchAllegroBalance(connection.accessToken);
      
      res.json(balance);
    } catch (error) {
      console.error("Get Allegro balance error:", error);
      res.status(500).json({ error: "Failed to fetch Allegro balance" });
    }
  });

  // Sync fees from Allegro API and save to database
  app.post("/api/allegro/fees/sync", isAuthenticated, async (req, res) => {
    try {
      const connection = await storage.getAllegroConnection();
      
      if (!connection || !connection.accessToken) {
        return res.status(401).json({ error: "No active Allegro connection" });
      }

      const days = parseInt(req.body.days as string, 10) || 30;
      
      // Fetch fees from Allegro API
      const { fetchAllegroFees } = await import('./allegro-api.js');
      const axios = await import('axios');
      
      const endDate = new Date();
      const startDate = new Date();
      startDate.setDate(startDate.getDate() - days);

      let allEntries: any[] = [];
      let offset = 0;
      const limit = 100;
      let hasMore = true;

      while (hasMore) {
        const response = await axios.default.get(
          `https://api.allegro.pl/billing/billing-entries`,
          {
            headers: {
              Authorization: `Bearer ${connection.accessToken}`,
              Accept: 'application/vnd.allegro.public.v1+json',
            },
            params: {
              'occurredAt.gte': startDate.toISOString(),
              'occurredAt.lte': endDate.toISOString(),
              limit,
              offset
            }
          }
        );

        const entries = response.data.billingEntries || [];
        allEntries = allEntries.concat(entries);
        
        if (entries.length < limit) {
          hasMore = false;
        } else {
          offset += limit;
        }
      }

      // Save to database
      const { saveAllegroFeesToDatabase, updateAllegroFeeSummary } = await import('./postgres.js');
      const savedCount = await saveAllegroFeesToDatabase(allEntries);

      // Update daily summaries for all affected days
      const daysToUpdate = new Set<string>();
      for (const entry of allEntries) {
        if (entry.occurredAt) {
          const date = new Date(entry.occurredAt);
          daysToUpdate.add(date.toDateString());
        }
      }

      for (const dayStr of Array.from(daysToUpdate)) {
        await updateAllegroFeeSummary(new Date(dayStr));
      }

      res.json({
        success: true,
        totalFetched: allEntries.length,
        totalSaved: savedCount,
        daysUpdated: daysToUpdate.size
      });
    } catch (error) {
      console.error("Sync Allegro fees error:", error);
      res.status(500).json({ error: "Failed to sync Allegro fees" });
    }
  });

  // Get fees for display (current endpoint - kept for widget)
  app.get("/api/allegro/fees", isAuthenticated, async (req, res) => {
    try {
      const connection = await storage.getAllegroConnection();
      
      if (!connection || !connection.accessToken) {
        return res.status(401).json({ error: "No active Allegro connection" });
      }

      const days = parseInt(req.query.days as string, 10) || 30;
      const todayOnly = req.query.todayOnly === 'true'; // Optymalizacja - pobierz tylko dzisiejsze
      const { fetchAllegroFees } = await import('./allegro-api.js');
      const fees = await fetchAllegroFees(connection.accessToken, days, todayOnly);
      
      res.json(fees);
    } catch (error) {
      console.error("Get Allegro fees error:", error);
      res.status(500).json({ error: "Failed to fetch Allegro fees" });
    }
  });

  // Get fees history from database with advanced filtering and sorting
  app.get("/api/allegro/fees/history", isAuthenticated, async (req, res) => {
    try {
      const { getAllegroFeesHistory } = await import('./postgres.js');
      
      let startDate: Date | undefined;
      let endDate: Date | undefined;
      
      const dateRange = req.query.dateRange as string;
      const now = new Date();
      
      if (dateRange === 'today') {
        startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRange === 'yesterday') {
        const yesterday = new Date(now);
        yesterday.setDate(yesterday.getDate() - 1);
        startDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 0, 0, 0);
        endDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 23, 59, 59);
      } else if (dateRange === 'day-before-yesterday') {
        const dayBefore = new Date(now);
        dayBefore.setDate(dayBefore.getDate() - 2);
        startDate = new Date(dayBefore.getFullYear(), dayBefore.getMonth(), dayBefore.getDate(), 0, 0, 0);
        endDate = new Date(dayBefore.getFullYear(), dayBefore.getMonth(), dayBefore.getDate(), 23, 59, 59);
      } else if (dateRange === 'custom' || req.query.startDate || req.query.endDate) {
        startDate = req.query.startDate ? new Date(req.query.startDate as string) : undefined;
        endDate = req.query.endDate ? new Date(req.query.endDate as string) : undefined;
        if (endDate) {
          endDate.setHours(23, 59, 59, 999);
        }
      }
      
      const category = req.query.category as string | undefined;
      const sortBy = req.query.sortBy as string || 'occurred_at';
      const sortOrder = req.query.sortOrder as string || 'DESC';
      const limit = parseInt(req.query.limit as string, 10) || 50;
      const offset = parseInt(req.query.offset as string, 10) || 0;

      const result = await getAllegroFeesHistory(startDate, endDate, category, limit, offset, sortBy, sortOrder);
      res.json(result);
    } catch (error) {
      console.error("Get fees history error:", error);
      res.status(500).json({ error: "Failed to fetch fees history" });
    }
  });

  // Get daily summaries from database
  app.get("/api/allegro/fees/summaries", isAuthenticated, async (req, res) => {
    try {
      const { getAllegroFeeSummaries } = await import('./postgres.js');
      const from = req.query.from as string;
      const to = req.query.to as string;
      
      let days = 30;
      if (from && to) {
        const fromDate = new Date(from);
        const toDate = new Date(to);
        days = Math.ceil((toDate.getTime() - fromDate.getTime()) / (1000 * 60 * 60 * 24)) + 1;
      } else if (req.query.days) {
        days = parseInt(req.query.days as string, 10);
      }
      
      const summaries = await getAllegroFeeSummaries(days);
      res.json(summaries);
    } catch (error) {
      console.error("Get fee summaries error:", error);
      res.status(500).json({ error: "Failed to fetch fee summaries" });
    }
  });

  // Get grouped fees by category and type
  app.get("/api/allegro/fees/grouped", isAuthenticated, async (req, res) => {
    try {
      const { getAllegroFeesGrouped } = await import('./postgres.js');
      const from = req.query.from as string;
      const to = req.query.to as string;
      const startDate = from ? new Date(from) : undefined;
      const endDate = to ? new Date(to) : undefined;
      const grouped = await getAllegroFeesGrouped(startDate, endDate);
      res.json(grouped);
    } catch (error) {
      console.error("Get grouped fees error:", error);
      res.status(500).json({ error: "Failed to fetch grouped fees" });
    }
  });

  // Bardziej specyficzne ścieżki muszą być PRZED ogólnymi
  app.get("/api/orders/statistics", async (req, res) => {
    try {
      const stats = await getOrderStatistics();
      res.json(stats);
    } catch (error) {
      console.error("Get order statistics error:", error);
      res.status(500).json({ error: "Failed to fetch statistics" });
    }
  });

  app.get("/api/orders/chart", async (req, res) => {
    try {
      const days = parseInt(req.query.days as string, 10) || 30;
      const todayOnly = req.query.todayOnly === 'true'; // Optymalizacja - pobierz tylko dzisiejsze
      const chartData = await getOrdersChartData(days);
      res.json(chartData);
    } catch (error) {
      console.error("Get chart data error:", error);
      res.status(500).json({ error: "Failed to fetch chart data" });
    }
  });

  app.get("/api/orders/period-summary", async (req, res) => {
    try {
      const days = parseInt(req.query.days as string, 10) || 30;
      const todayOnly = req.query.todayOnly === 'true'; // Optymalizacja - pobierz tylko dzisiejsze
      const summary = await getOrdersPeriodSummary(days);
      res.json(summary);
    } catch (error) {
      console.error("Get period summary error:", error);
      res.status(500).json({ error: "Failed to fetch period summary" });
    }
  });

  app.get("/api/orders/today-stats", async (req, res) => {
    try {
      const stats = await getTodayDetailedStats();
      res.json(stats);
    } catch (error) {
      console.error("Get today stats error:", error);
      res.status(500).json({ error: "Failed to fetch today stats" });
    }
  });

  app.get("/api/orders/count", async (req, res) => {
    try {
      const { getOrdersCount } = await import('./postgres.js');
      const search = typeof req.query.search === 'string' ? req.query.search : undefined;
      const sourceFilter = typeof req.query.sourceFilter === 'string' ? req.query.sourceFilter : undefined;
      const dateRangeType = typeof req.query.dateRangeType === 'string' ? req.query.dateRangeType : undefined;
      const customDateFrom = typeof req.query.customDateFrom === 'string' ? req.query.customDateFrom : undefined;
      const customDateTo = typeof req.query.customDateTo === 'string' ? req.query.customDateTo : undefined;
      const paymentFilter = typeof req.query.paymentFilter === 'string' ? req.query.paymentFilter : undefined;
      const fulfillmentFilter = typeof req.query.fulfillmentFilter === 'string' ? req.query.fulfillmentFilter : undefined;
      const invoiceFilter = typeof req.query.invoiceFilter === 'string' ? req.query.invoiceFilter : undefined;
      const paymentTypeFilter = typeof req.query.paymentTypeFilter === 'string' ? req.query.paymentTypeFilter : undefined;
      
      const count = await getOrdersCount(
        search,
        sourceFilter,
        dateRangeType,
        customDateFrom,
        customDateTo,
        paymentFilter,
        fulfillmentFilter,
        invoiceFilter,
        paymentTypeFilter
      );
      res.json({ count });
    } catch (error) {
      console.error("Get orders count error:", error);
      res.status(500).json({ error: "Failed to fetch orders count" });
    }
  });

  app.get("/api/orders/payment-summary", isAuthenticated, async (req, res) => {
    try {
      const search = typeof req.query.search === 'string' ? req.query.search : undefined;
      const paymentFilter = typeof req.query.paymentFilter === 'string' ? req.query.paymentFilter : undefined;
      const fulfillmentFilter = typeof req.query.fulfillmentFilter === 'string' ? req.query.fulfillmentFilter : undefined;
      const invoiceFilter = typeof req.query.invoiceFilter === 'string' ? req.query.invoiceFilter : undefined;
      const paymentTypeFilter = typeof req.query.paymentTypeFilter === 'string' ? req.query.paymentTypeFilter : undefined;
      const sourceFilter = typeof req.query.sourceFilter === 'string' ? req.query.sourceFilter : undefined;
      const dateRangeType = typeof req.query.dateRangeType === 'string' ? req.query.dateRangeType : undefined;
      const customDateFrom = typeof req.query.customDateFrom === 'string' ? req.query.customDateFrom : undefined;
      const customDateTo = typeof req.query.customDateTo === 'string' ? req.query.customDateTo : undefined;

      const conditions: string[] = [];
      const params: any[] = [];
      let paramIndex = 1;

      // Search filter
      if (search) {
        conditions.push(`(
          o.order_number::text ILIKE $${paramIndex} OR
          o.buyer_email ILIKE $${paramIndex} OR
          o.buyer_first_name ILIKE $${paramIndex} OR
          o.buyer_last_name ILIKE $${paramIndex}
        )`);
        params.push(`%${search}%`);
        paramIndex++;
      }

      // Payment filter
      if (paymentFilter && paymentFilter !== 'Wszystkie') {
        conditions.push(`o.payment_status = $${paramIndex}`);
        params.push(paymentFilter);
        paramIndex++;
      }

      // Fulfillment filter
      if (fulfillmentFilter && fulfillmentFilter !== 'Wszystkie') {
        conditions.push(`o.status = $${paramIndex}`);
        params.push(fulfillmentFilter);
        paramIndex++;
      }

      // Invoice filter
      if (invoiceFilter && invoiceFilter !== 'Wszystkie') {
        const hasInvoice = invoiceFilter === 'Tak';
        conditions.push(`o.has_invoice = $${paramIndex}`);
        params.push(hasInvoice);
        paramIndex++;
      }

      // Payment type filter
      if (paymentTypeFilter && paymentTypeFilter !== 'Wszystkie') {
        conditions.push(`o.payment_type = $${paramIndex}`);
        params.push(paymentTypeFilter);
        paramIndex++;
      }

      // Source filter
      if (sourceFilter && sourceFilter !== 'Wszystkie') {
        conditions.push(`o.source = $${paramIndex}`);
        params.push(sourceFilter);
        paramIndex++;
      }

      // Date range filter
      if (dateRangeType && dateRangeType !== 'all' && dateRangeType !== 'Wszystkie') {
        const now = new Date();
        let startDate: Date | null = null;
        let endDate: Date | null = null;

        if (dateRangeType === 'today') {
          startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 0, 0, 0);
          endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
        } else if (dateRangeType === 'yesterday') {
          const yesterday = new Date(now);
          yesterday.setDate(yesterday.getDate() - 1);
          startDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 0, 0, 0);
          endDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 23, 59, 59);
        } else if (dateRangeType === '7days') {
          startDate = new Date(now);
          startDate.setDate(startDate.getDate() - 6);
          startDate.setHours(0, 0, 0, 0);
          endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
        } else if (dateRangeType === '30days') {
          startDate = new Date(now);
          startDate.setDate(startDate.getDate() - 29);
          startDate.setHours(0, 0, 0, 0);
          endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
        } else if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
          startDate = new Date(customDateFrom);
          endDate = new Date(customDateTo);
          endDate.setHours(23, 59, 59, 999);
        }

        if (startDate && endDate) {
          conditions.push(`o.order_date >= $${paramIndex}`);
          params.push(startDate.toISOString());
          paramIndex++;
          conditions.push(`o.order_date <= $${paramIndex}`);
          params.push(endDate.toISOString());
          paramIndex++;
        }
      }

      const whereClause = conditions.length > 0 ? `WHERE ${conditions.join(' AND ')}` : '';

      const result = await pool.query(`
        SELECT 
          COALESCE(SUM(CASE WHEN o.payment_status = 'PAID' THEN o.payment_amount ELSE 0 END), 0) as paid_amount,
          COUNT(CASE WHEN o.payment_status = 'PAID' THEN 1 END) as paid_count,
          COALESCE(SUM(CASE WHEN o.payment_status IN ('UNPAID', 'PENDING') AND o.payment_type != 'CASH_ON_DELIVERY' THEN o.total_to_pay_amount ELSE 0 END), 0) as unpaid_amount,
          COUNT(CASE WHEN o.payment_status IN ('UNPAID', 'PENDING') AND o.payment_type != 'CASH_ON_DELIVERY' THEN 1 END) as unpaid_count,
          COALESCE(SUM(CASE WHEN o.payment_type = 'CASH_ON_DELIVERY' THEN o.total_to_pay_amount ELSE 0 END), 0) as cod_amount,
          COUNT(CASE WHEN o.payment_type = 'CASH_ON_DELIVERY' THEN 1 END) as cod_count
        FROM commerce.orders o
        ${whereClause}
      `, params);

      const summary = result.rows[0];

      res.json({
        paid: {
          amount: parseFloat(summary.paid_amount) || 0,
          count: parseInt(summary.paid_count, 10) || 0
        },
        unpaid: {
          amount: parseFloat(summary.unpaid_amount) || 0,
          count: parseInt(summary.unpaid_count, 10) || 0
        },
        cod: {
          amount: parseFloat(summary.cod_amount) || 0,
          count: parseInt(summary.cod_count, 10) || 0
        }
      });
    } catch (error) {
      console.error("Get payment summary error:", error);
      res.status(500).json({ error: "Failed to fetch payment summary" });
    }
  });

  // Endpoint do matchowania zamówień między platformami (MUSI być PRZED /api/orders/:id)
  app.get("/api/orders/match", async (req, res) => {
    try {
      const { 
        platform, 
        dateRange, 
        startDate: customStartDate, 
        endDate: customEndDate,
        errorFilter = 'all',
        sortBy = 'date_desc'
      } = req.query;
      
      if (!platform || !['allegro', 'shoper', 'odoo'].includes(platform as string)) {
        return res.status(400).json({ error: "Invalid platform. Use: allegro, shoper, or odoo" });
      }
      
      if (!dateRange || !['today', 'yesterday', 'day-before-yesterday', 'last-7-days', 'custom'].includes(dateRange as string)) {
        return res.status(400).json({ error: "Invalid dateRange. Use: today, yesterday, day-before-yesterday, last-7-days, or custom" });
      }

      if (!['all', 'payment_mismatch', 'fulfillment_mismatch', 'missing_in_oms', 'missing_in_odoo'].includes(errorFilter as string)) {
        return res.status(400).json({ error: "Invalid errorFilter. Use: all, payment_mismatch, fulfillment_mismatch, missing_in_oms, missing_in_odoo" });
      }

      if (!['errors_first', 'errors_last', 'date_asc', 'date_desc'].includes(sortBy as string)) {
        return res.status(400).json({ error: "Invalid sortBy. Use: errors_first, errors_last, date_asc, date_desc" });
      }

      const now = new Date();
      let startDate: Date, endDate: Date;

      if (dateRange === 'today') {
        startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRange === 'yesterday') {
        const yesterday = new Date(now);
        yesterday.setDate(yesterday.getDate() - 1);
        startDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 0, 0, 0);
        endDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 23, 59, 59);
      } else if (dateRange === 'day-before-yesterday') {
        const dayBefore = new Date(now);
        dayBefore.setDate(dayBefore.getDate() - 2);
        startDate = new Date(dayBefore.getFullYear(), dayBefore.getMonth(), dayBefore.getDate(), 0, 0, 0);
        endDate = new Date(dayBefore.getFullYear(), dayBefore.getMonth(), dayBefore.getDate(), 23, 59, 59);
      } else if (dateRange === 'last-7-days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 7);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now);
        endDate.setHours(23, 59, 59, 999);
      } else if (dateRange === 'custom') {
        if (!customStartDate || !customEndDate) {
          return res.status(400).json({ error: "Custom date range requires startDate and endDate" });
        }
        startDate = new Date(customStartDate as string);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(customEndDate as string);
        endDate.setHours(23, 59, 59, 999);
      } else {
        return res.status(400).json({ error: "Invalid dateRange" });
      }

      const results: any[] = [];

      // Przygotuj klienta Odoo dla weryfikacji
      const { createOdooClient } = await import('./odoo-client.js');
      const odooConfig = await pool.query(`SELECT * FROM odoo_config LIMIT 1`);
      
      let odooClient = null;
      if (odooConfig.rows.length > 0) {
        try {
          odooClient = await createOdooClient();
        } catch (error) {
          console.error('Failed to create Odoo client for verification:', error);
        }
      }

      if (platform === 'allegro') {
        // Pobierz zamówienia z Allegro API
        const connection = await storage.getAllegroConnection();
        if (!connection?.accessToken) {
          return res.status(400).json({ error: "No Allegro connection" });
        }

        const allegroOrders = await fetchOrdersFromAllegroWithFilters(connection.accessToken, 100, 0, {
          'updatedAt.gte': startDate.toISOString(),
          'updatedAt.lte': endDate.toISOString()
        });

        for (const allegroOrder of allegroOrders) {
          const sourceOrderId = allegroOrder.id;
          
          // Oblicz payment_status z Allegro z dokładnym porównaniem kwot
          const paidAmount = parseFloat(allegroOrder.payment?.paidAmount?.amount || '0');
          const totalToPay = parseFloat(allegroOrder.summary?.totalToPay?.amount || '0');
          const hasFinishedAt = !!((allegroOrder.payment as any)?.finishedAt);
          
          let platformPaymentStatus = 'UNPAID';
          if (paidAmount >= totalToPay - 0.01 && totalToPay > 0) {
            platformPaymentStatus = 'PAID';
          } else if (paidAmount > 0) {
            platformPaymentStatus = 'PARTIAL';
          } else if (hasFinishedAt && paidAmount === 0) {
            platformPaymentStatus = 'PAID';
          }
          
          // Pobierz status realizacji z Allegro
          const platformFulfillmentStatus = allegroOrder.fulfillment?.status || 'UNKNOWN';
          
          // Sprawdź czy jest w OMS
          const omsCheck = await pool.query(`
            SELECT id, order_number, odoo_order_id, payment_status, status FROM commerce.orders 
            WHERE source = 'ALLEGRO' AND source_order_id = $1
          `, [sourceOrderId]);
          
          const inOMS = omsCheck.rows.length > 0;
          const omsOrderId = omsCheck.rows[0]?.id;
          const omsOrderNumber = omsCheck.rows[0]?.order_number;
          const omsPaymentStatus = omsCheck.rows[0]?.payment_status;
          const omsFulfillmentStatus = omsCheck.rows[0]?.status;
          let odooOrderId = omsCheck.rows[0]?.odoo_order_id;
          let actuallyInOdoo = !!odooOrderId;

          // Weryfikuj w Odoo przez API (jeśli zamówienie jest w OMS)
          if (inOMS && omsOrderNumber && odooClient) {
            try {
              const searchResults = await odooClient.execute('sale.order', 'search_read', [
                ['|',
                  ['client_order_ref', '=', `#${omsOrderNumber}`],
                  ['x_studio_numer_w_oms', '=', omsOrderNumber]
                ],
                ['id']
              ]);

          results.success++;

              if (searchResults && searchResults.length > 0) {
                actuallyInOdoo = true;
                odooOrderId = searchResults[0].id;
              } else {
                actuallyInOdoo = false;
              }
            } catch (error) {
              console.error(`Failed to verify Allegro order ${omsOrderNumber} in Odoo:`, error);
              // Fallback: zaufaj odoo_order_id z bazy
              actuallyInOdoo = !!odooOrderId;
            }
          }
          
          results.push({
            sourceOrderId,
            platform: 'ALLEGRO',
            orderDate: allegroOrder.updatedAt,
            inOMS,
            inOdoo: actuallyInOdoo,
            omsOrderId,
            omsOrderNumber,
            odooOrderId,
            verified: odooClient !== null && inOMS,
            platformPaymentStatus,
            omsPaymentStatus,
            paymentMismatch: inOMS && platformPaymentStatus !== omsPaymentStatus,
            platformPaymentAmount: allegroOrder.summary?.totalToPay?.amount,
            omsPaymentAmount: omsCheck.rows[0]?.payment_amount,
            platformFulfillmentStatus,
            omsFulfillmentStatus,
            fulfillmentMismatch: inOMS && platformFulfillmentStatus !== omsFulfillmentStatus
          });
        }
      } else if (platform === 'shoper') {
        // Pobierz zamówienia ze Shoper API
        const { getShoperOrders } = await import('./shoper-api.js');
        
        const filters = {
          updated_at: {
            ">=": startDate.toISOString().replace('T', ' ').replace('Z', ''),
            "<=": endDate.toISOString().replace('T', ' ').replace('Z', '')
          }
        };
        
        const shoperOrders = await getShoperOrders(100, filters);

        for (const shoperOrder of shoperOrders) {
          const sourceOrderId = shoperOrder.order_id?.toString() || shoperOrder.id?.toString();
          
          // Oblicz payment_status z Shoper (tak jak w postgres.ts)
          const paidAmount = parseFloat(shoperOrder.paid || '0');
          const totalAmount = parseFloat(shoperOrder.sum || '0');
          const hasPaidDate = !!(shoperOrder.paid_date && shoperOrder.paid_date !== '0000-00-00 00:00:00');
          
          let platformPaymentStatus = 'PENDING';
          if (hasPaidDate || shoperOrder.is_paid || paidAmount >= totalAmount - 0.01) {
            platformPaymentStatus = 'PAID';
          } else if (paidAmount > 0) {
            platformPaymentStatus = 'PARTIAL';
          } else {
            platformPaymentStatus = 'UNPAID';
          }
          
          // Pobierz status realizacji z Shoper
          const platformFulfillmentStatus = shoperOrder.status_id || 'UNKNOWN';
          
          // Sprawdź czy jest w OMS
          const omsCheck = await pool.query(`
            SELECT id, order_number, odoo_order_id, payment_status, status FROM commerce.orders 
            WHERE source = 'SHOPER' AND source_order_id = $1
          `, [sourceOrderId]);
          
          const inOMS = omsCheck.rows.length > 0;
          const omsOrderId = omsCheck.rows[0]?.id;
          const omsOrderNumber = omsCheck.rows[0]?.order_number;
          const omsPaymentStatus = omsCheck.rows[0]?.payment_status;
          const omsFulfillmentStatus = omsCheck.rows[0]?.status;
          let odooOrderId = omsCheck.rows[0]?.odoo_order_id;
          let actuallyInOdoo = !!odooOrderId;

          // Weryfikuj w Odoo przez API (jeśli zamówienie jest w OMS)
          if (inOMS && omsOrderNumber && odooClient) {
            try {
              const searchResults = await odooClient.execute('sale.order', 'search_read', [
                ['|',
                  ['client_order_ref', '=', `#${omsOrderNumber}`],
                  ['x_studio_numer_w_oms', '=', omsOrderNumber]
                ],
                ['id']
              ]);

          results.success++;

              if (searchResults && searchResults.length > 0) {
                actuallyInOdoo = true;
                odooOrderId = searchResults[0].id;
              } else {
                actuallyInOdoo = false;
              }
            } catch (error) {
              console.error(`Failed to verify Shoper order ${omsOrderNumber} in Odoo:`, error);
              // Fallback: zaufaj odoo_order_id z bazy
              actuallyInOdoo = !!odooOrderId;
            }
          }
          
          results.push({
            sourceOrderId,
            platform: 'SHOPER',
            orderDate: shoperOrder.date || shoperOrder.updated_at,
            inOMS,
            inOdoo: actuallyInOdoo,
            omsOrderId,
            omsOrderNumber,
            odooOrderId,
            verified: odooClient !== null && inOMS,
            platformPaymentStatus,
            omsPaymentStatus,
            paymentMismatch: inOMS && platformPaymentStatus !== omsPaymentStatus,
            platformPaymentAmount: shoperOrder.sum,
            omsPaymentAmount: omsCheck.rows[0]?.payment_amount,
            platformFulfillmentStatus,
            omsFulfillmentStatus,
            fulfillmentMismatch: inOMS && platformFulfillmentStatus !== omsFulfillmentStatus
          });
        }
      } else if (platform === 'odoo') {
        // Pobierz wszystkie zamówienia z OMS (z datą) i zweryfikuj w Odoo
        const omsOrders = await pool.query(`
          SELECT 
            o.id,
            o.order_number,
            o.odoo_order_id,
            o.source,
            o.source_order_id,
            o.order_date
          FROM commerce.orders o
          WHERE o.order_date >= $1
            AND o.order_date <= $2
          ORDER BY o.order_date DESC
        `, [startDate, endDate]);

        // Pobierz klienta Odoo
        const { createOdooClient } = await import('./odoo-client.js');
        const odooConfig = await pool.query(`SELECT * FROM odoo_config LIMIT 1`);
        
        let odooClient = null;
        if (odooConfig.rows.length > 0) {
          try {
            odooClient = await createOdooClient();
          } catch (error) {
            console.error('Failed to create Odoo client for verification:', error);
          }
        }

        for (const order of omsOrders.rows) {
          let actuallyInOdoo = false;
          let verifiedOdooId = order.odoo_order_id;

          // Weryfikuj w Odoo przez API
          if (odooClient) {
            try {
              // Szukaj po client_order_ref (#orderNumber) i x_studio_numer_w_oms
              const searchResults = await odooClient.execute('sale.order', 'search_read', [
                ['|',
                  ['client_order_ref', '=', `#${order.order_number}`],
                  ['x_studio_numer_w_oms', '=', order.order_number]
                ],
                ['id', 'name', 'client_order_ref', 'x_studio_numer_w_oms']
              ]);

          results.success++;

              if (searchResults && searchResults.length > 0) {
                actuallyInOdoo = true;
                verifiedOdooId = searchResults[0].id;
              }
            } catch (error) {
              console.error(`Failed to verify order ${order.order_number} in Odoo:`, error);
              // Fallback: zaufaj odoo_order_id z bazy
              actuallyInOdoo = !!order.odoo_order_id;
            }
          } else {
            // Brak połączenia z Odoo - zaufaj odoo_order_id z bazy
            actuallyInOdoo = !!order.odoo_order_id;
          }

          results.push({
            sourceOrderId: order.source_order_id,
            platform: 'ODOO',
            orderDate: order.order_date,
            inOMS: true,
            inOdoo: actuallyInOdoo,
            omsOrderId: order.id,
            omsOrderNumber: order.order_number,
            odooOrderId: verifiedOdooId,
            originalPlatform: order.source,
            verified: odooClient !== null // czy została zweryfikowana w Odoo
          });
        }
      }

      // Filtrowanie wyników po stronie serwera
      let filteredResults = results;
      
      if (errorFilter !== 'all') {
        switch (errorFilter) {
          case 'payment_mismatch':
            filteredResults = results.filter(r => r.paymentMismatch);
            break;
          case 'fulfillment_mismatch':
            filteredResults = results.filter(r => r.fulfillmentMismatch);
            break;
          case 'missing_in_oms':
            filteredResults = results.filter(r => !r.inOMS);
            break;
          case 'missing_in_odoo':
            filteredResults = results.filter(r => r.inOMS && !r.inOdoo);
            break;
        }
      }

      // Sortowanie wyników
      switch (sortBy) {
        case 'errors_first':
          filteredResults.sort((a, b) => {
            const aHasError = a.paymentMismatch || a.fulfillmentMismatch || !a.inOMS || (a.inOMS && !a.inOdoo);
            const bHasError = b.paymentMismatch || b.fulfillmentMismatch || !b.inOMS || (b.inOMS && !b.inOdoo);
            if (aHasError && !bHasError) return -1;
            if (!aHasError && bHasError) return 1;
            return new Date(b.orderDate).getTime() - new Date(a.orderDate).getTime();
          });
          break;
        case 'errors_last':
          filteredResults.sort((a, b) => {
            const aHasError = a.paymentMismatch || a.fulfillmentMismatch || !a.inOMS || (a.inOMS && !a.inOdoo);
            const bHasError = b.paymentMismatch || b.fulfillmentMismatch || !b.inOMS || (b.inOMS && !b.inOdoo);
            if (!aHasError && bHasError) return -1;
            if (aHasError && !bHasError) return 1;
            return new Date(b.orderDate).getTime() - new Date(a.orderDate).getTime();
          });
          break;
        case 'date_asc':
          filteredResults.sort((a, b) => new Date(a.orderDate).getTime() - new Date(b.orderDate).getTime());
          break;
        case 'date_desc':
        default:
          filteredResults.sort((a, b) => new Date(b.orderDate).getTime() - new Date(a.orderDate).getTime());
          break;
      }

      // Ogranicz do 100 wyników (po filtrowaniu i sortowaniu)
      const limitedResults = filteredResults.slice(0, 100);

      // Podsumowanie (bazuje na oryginalnych wynikach przed filtrowaniem)
      const summary = {
        total: results.length,
        inOMS: results.filter(r => r.inOMS).length,
        notInOMS: results.filter(r => !r.inOMS).length,
        inOdoo: results.filter(r => r.inOdoo).length,
        notInOdoo: results.filter(r => !r.inOdoo).length,
        mismatches: results.filter(r => r.inOMS && !r.inOdoo).length,
        paymentMismatches: results.filter(r => r.paymentMismatch).length,
        fulfillmentMismatches: results.filter(r => r.fulfillmentMismatch).length,
        filtered: limitedResults.length,
        truncated: filteredResults.length > 100
      };

      res.json({
        platform,
        dateRange,
        startDate,
        endDate,
        summary,
        orders: limitedResults,
        filterApplied: errorFilter,
        sortApplied: sortBy
      });
    } catch (error: any) {
      console.error("Order matching error:", error);
      res.status(500).json({ error: error.message || "Failed to match orders" });
    }
  });

  // Endpoint do weryfikacji zamówień po dacie ostatniej aktualizacji w marketplace
  app.get("/api/orders/recent-updates", async (req, res) => {
    try {
      const { 
        platform, 
        dateRange, 
        startDate: customStartDate, 
        endDate: customEndDate
      } = req.query;
      
      if (!platform || !['allegro', 'shoper'].includes(platform as string)) {
        return res.status(400).json({ error: "Invalid platform. Use: allegro or shoper" });
      }
      
      if (!dateRange || !['today', 'yesterday', 'last-7-days', 'last-30-days', 'custom'].includes(dateRange as string)) {
        return res.status(400).json({ error: "Invalid dateRange. Use: today, yesterday, last-7-days, last-30-days, or custom" });
      }

      const now = new Date();
      let startDate: Date, endDate: Date;

      if (dateRange === 'today') {
        startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 0, 0, 0);
        endDate = new Date(now.getFullYear(), now.getMonth(), now.getDate(), 23, 59, 59);
      } else if (dateRange === 'yesterday') {
        const yesterday = new Date(now);
        yesterday.setDate(yesterday.getDate() - 1);
        startDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 0, 0, 0);
        endDate = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate(), 23, 59, 59);
      } else if (dateRange === 'last-7-days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 7);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now);
        endDate.setHours(23, 59, 59, 999);
      } else if (dateRange === 'last-30-days') {
        startDate = new Date(now);
        startDate.setDate(startDate.getDate() - 30);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(now);
        endDate.setHours(23, 59, 59, 999);
      } else if (dateRange === 'custom') {
        if (!customStartDate || !customEndDate) {
          return res.status(400).json({ error: "Custom date range requires startDate and endDate" });
        }
        startDate = new Date(customStartDate as string);
        startDate.setHours(0, 0, 0, 0);
        endDate = new Date(customEndDate as string);
        endDate.setHours(23, 59, 59, 999);
      } else {
        return res.status(400).json({ error: "Invalid dateRange" });
      }

      const platformUpper = (platform as string).toUpperCase();
      
      // Wybierz odpowiednie pole dla marketplace timestamp w zależności od platformy
      // Allegro: raw_payload->>'updatedAt'
      // Shoper: raw_payload->>'status_date'
      const timestampField = platformUpper === 'ALLEGRO' 
        ? "raw_payload->>'updatedAt'" 
        : "raw_payload->>'status_date'";
      
      // Pobierz zamówienia z commerce.orders filtrując po marketplace updatedAt
      const result = await pool.query(`
        SELECT 
          id,
          source,
          source_order_id,
          order_number,
          order_date,
          status as fulfillment_status,
          payment_status,
          buyer_login,
          buyer_email,
          total_to_pay_amount,
          total_to_pay_currency,
          odoo_order_id,
          updated_at,
          (${timestampField})::timestamp as marketplace_updated_at
        FROM commerce.orders
        WHERE source = $1
          AND (${timestampField})::timestamp >= $2::timestamp
          AND (${timestampField})::timestamp <= $3::timestamp
        ORDER BY (${timestampField})::timestamp DESC
        LIMIT 200
      `, [platformUpper, startDate, endDate]);

      const orders = result.rows.map(row => ({
        id: row.id,
        source: row.source,
        sourceOrderId: row.source_order_id,
        orderNumber: row.order_number,
        orderDate: row.order_date,
        marketplaceUpdatedAt: row.marketplace_updated_at,
        lastUpdatedAt: row.updated_at,
        fulfillmentStatus: row.fulfillment_status,
        paymentStatus: row.payment_status,
        buyerLogin: row.buyer_login,
        buyerEmail: row.buyer_email,
        totalAmount: row.total_to_pay_amount ? parseFloat(row.total_to_pay_amount) : 0,
        currency: row.total_to_pay_currency,
        odooOrderId: row.odoo_order_id,
        inOdoo: !!row.odoo_order_id
      }));

      const summary = {
        total: orders.length,
        inOdoo: orders.filter(o => o.inOdoo).length,
        notInOdoo: orders.filter(o => !o.inOdoo).length,
        paid: orders.filter(o => o.paymentStatus === 'PAID').length,
        unpaid: orders.filter(o => o.paymentStatus === 'UNPAID').length
      };

      res.json({
        platform,
        dateRange,
        startDate,
        endDate,
        summary,
        orders
      });
    } catch (error: any) {
      console.error("Recent updates error:", error);
      res.status(500).json({ error: error.message || "Failed to fetch recent updates" });
    }
  });

  // Endpoint do sprawdzania luk w numeracji zamówień
  app.get("/api/orders/numbering/gaps", async (req, res) => {
    try {
      const { checkOrderNumberGaps } = await import('./order-numbering.js');
      const gaps = await checkOrderNumberGaps();
      res.json({ gaps });
    } catch (error: any) {
      console.error("Check gaps error:", error);
      res.status(500).json({ error: error.message || "Failed to check gaps" });
    }
  });

  // Endpoint do sprawdzania statystyk numeracji
  app.get("/api/orders/numbering/stats", async (req, res) => {
    try {
      const { getOrderNumberingStats } = await import('./order-numbering.js');
      const stats = await getOrderNumberingStats();
      res.json(stats);
    } catch (error: any) {
      console.error("Get numbering stats error:", error);
      res.status(500).json({ error: error.message || "Failed to get stats" });
    }
  });

  app.get("/api/orders/by-number/:orderNumber", async (req, res) => {
    try {
      let { orderNumber } = req.params;
      const { getOrderByNumberFromPostgres } = await import('./postgres.js');
      
      // Obsługa numeru z # (np. "#00001" → "00001") lub bez wiodących zer ("1" → "00001")
      orderNumber = orderNumber.replace('#', '').padStart(5, '0');
      
      // Pobierz zamówienie z transformacją danych
      const order = await getOrderByNumberFromPostgres(orderNumber);
      
      if (!order) {
        return res.status(404).json({ error: "Order not found" });
      }
      
      res.json(order);
    } catch (error) {
      console.error("Get order by number error:", error);
      res.status(500).json({ error: "Failed to fetch order" });
    }
  });

  app.get("/api/orders/:id", async (req, res) => {
    try {
      const { id } = req.params;
      const { getOrderFromPostgres } = await import('./postgres.js');
      
      // Sprawdź czy id to liczba (numeric order ID)
      const numericId = parseInt(id, 10);
      if (isNaN(numericId)) {
        return res.status(400).json({ error: "Invalid order ID format" });
      }
      
      // Pobierz zamówienie z transformacją danych
      const order = await getOrderFromPostgres(numericId);
      
      if (!order) {
        return res.status(404).json({ error: "Order not found" });
      }
      
      res.json(order);
    } catch (error) {
      console.error("Get order by ID error:", error);
      res.status(500).json({ error: "Failed to fetch order" });
    }
  });

  app.get("/api/orders/:allegroOrderId/raw", async (req, res) => {
    try {
      const { allegroOrderId } = req.params;
      const connection = await storage.getAllegroConnection();
      
      if (!connection || !connection.accessToken) {
        return res.status(401).json({ error: "No active Allegro connection" });
      }

      const rawOrder = await fetchRawOrderFromAllegro(connection.accessToken, allegroOrderId);
      res.json(rawOrder);
    } catch (error: any) {
      console.error("Get raw order error:", error);
      res.status(500).json({ error: error.message || "Failed to fetch raw order" });
    }
  });

  app.get("/api/orders", async (req, res) => {
    try {
      const sortBy = typeof req.query.sortBy === 'string' ? req.query.sortBy : 'order_date';
      const sortOrder = (typeof req.query.sortOrder === 'string' && req.query.sortOrder.toUpperCase() === 'ASC') ? 'ASC' : 'DESC';
      const limit = typeof req.query.limit === 'string' ? parseInt(req.query.limit, 10) : 50;
      const offset = typeof req.query.offset === 'string' ? parseInt(req.query.offset, 10) : 0;
      const search = typeof req.query.search === 'string' ? req.query.search : undefined;
      const sourceFilter = typeof req.query.sourceFilter === 'string' ? req.query.sourceFilter : undefined;
      const dateRangeType = typeof req.query.dateRangeType === 'string' ? req.query.dateRangeType : undefined;
      const customDateFrom = typeof req.query.customDateFrom === 'string' ? req.query.customDateFrom : undefined;
      const customDateTo = typeof req.query.customDateTo === 'string' ? req.query.customDateTo : undefined;
      const paymentFilter = typeof req.query.paymentFilter === 'string' ? req.query.paymentFilter : undefined;
      const fulfillmentFilter = typeof req.query.fulfillmentFilter === 'string' ? req.query.fulfillmentFilter : undefined;
      const invoiceFilter = typeof req.query.invoiceFilter === 'string' ? req.query.invoiceFilter : undefined;
      const paymentTypeFilter = typeof req.query.paymentTypeFilter === 'string' ? req.query.paymentTypeFilter : undefined;
      
      const orders = await getOrdersFromPostgres(
        sortBy, 
        sortOrder, 
        limit, 
        offset, 
        search,
        sourceFilter,
        dateRangeType,
        customDateFrom,
        customDateTo,
        paymentFilter,
        fulfillmentFilter,
        invoiceFilter,
        paymentTypeFilter
      );
      res.json(orders);
    } catch (error) {
      console.error("Get orders error:", error);
      res.status(500).json({ error: "Failed to fetch orders" });
    }
  });

  // 📸 Concurrency guard dla pobierania zdjęć
  let isDownloadingImages = false;

  // 📸 Endpoint do pobierania brakujących zdjęć produktów (z external URLs)
  app.post("/api/products/download-missing-images", async (req, res) => {
    if (isDownloadingImages) {
      return res.status(409).json({ 
        error: "Pobieranie zdjęć już trwa",
        inProgress: true 
      });
    }

    isDownloadingImages = true;
    try {
      const { downloadImageFromUrl } = await import('./allegro-api.js');
      
      // Pobierz produkty które mają external URL ale nie mają lokalnego pliku
      const result = await pool.query(`
        SELECT 
          p.id,
          p.external_id,
          p.name,
          p.image_url
        FROM catalog.products p
        WHERE p.image_url IS NOT NULL
          AND p.image_url LIKE 'http%'
        LIMIT 50
      `);
      
      let downloaded = 0;
      let failed = 0;
      
      for (const product of result.rows) {
        try {
          const localFilename = await downloadImageFromUrl(product.image_url, product.external_id);
          if (localFilename) {
            // Aktualizuj URL w bazie
            await pool.query(`
              UPDATE catalog.products
              SET image_url = $1, updated_at = NOW()
              WHERE id = $2
            `, [localFilename, product.id]);
            
            await pool.query(`
              UPDATE commerce.order_items
              SET image_url = $1
              WHERE offer_external_id = $2
            `, [localFilename, product.external_id]);
            
            downloaded++;
            console.log(`📸 Downloaded image for product ${product.external_id}: ${localFilename}`);
          } else {
            failed++;
          }
          
          // Throttle requests
          await new Promise(resolve => setTimeout(resolve, 300));
        } catch (error) {
          console.error(`❌ Failed to download image for ${product.external_id}:`, error);
          failed++;
        }
      }
      
      res.json({
        success: true,
        downloaded,
        failed,
        total: result.rows.length,
        message: `Pobrano ${downloaded} zdjęć, ${failed} niepowodzeń`
      });
    } catch (error: any) {
      console.error("Download missing images error:", error);
      res.status(500).json({ error: error.message });
    } finally {
      isDownloadingImages = false;
    }
  });

  app.post("/api/products/download-images", async (req, res) => {
    try {
      const { downloadProductImages } = await import('./allegro-api.js');
      const { getUniqueProductsFromPostgres } = await import('./postgres.js');
      
      const connection = await storage.getAllegroConnection();
      if (!connection || !connection.accessToken) {
        return res.status(401).json({ error: "Brak połączenia z Allegro" });
      }
      
      const products = await getUniqueProductsFromPostgres();
      
      console.log('🔍 DEBUG - First product from DB:', products[0]);
      
      if (products.length === 0) {
        return res.json({ success: 0, failed: 0, message: "Nie znaleziono produktów" });
      }

      const items = products.map((p: any) => ({
        productName: p.name,
        externalId: p.offer_external_id
      }));

      console.log('🔍 DEBUG - First mapped item:', items[0]);
      console.log(`📦 Starting download of ${items.length} product images by searching product names`);
      const result = await downloadProductImages(items, connection.accessToken);
      res.json({ 
        success: result.success, 
        failed: result.failed,
        total: items.length,
        message: `Pobrano ${result.success}/${items.length} zdjęć produktów`
      });
    } catch (error: any) {
      console.error("Download images error:", error);
      res.status(500).json({ error: error.message || "Nie udało się pobrać zdjęć" });
    }
  });

  app.get("/api/sync/status", async (req, res) => {
    try {
      const settings = await getSyncSettingsFromPostgres();
      res.json({
        lastSyncAt: settings?.lastSyncAt || null,
        isRefreshing: isCurrentlySyncing,
      });
    } catch (error) {
      console.error("Get sync status error:", error);
      res.status(500).json({ error: "Failed to get sync status" });
    }
  });

  app.post("/api/sync/settings", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { autoRefreshEnabled, refreshIntervalMinutes, fullSyncEnabled, fullSyncWindowHours } = req.body;

      const currentSettings = await getSyncSettingsFromPostgres();
      const settings = await createOrUpdateSyncSettingsInPostgres({
        autoRefreshEnabled,
        refreshIntervalMinutes,
        fullSyncEnabled: fullSyncEnabled ?? true,
        fullSyncWindowHours: fullSyncWindowHours ?? '2',
        lastSyncAt: currentSettings?.lastSyncAt || null,
      });

      if (autoRefreshEnabled) {
        const interval = parseInt(refreshIntervalMinutes, 10);
        startSyncScheduler(interval);
      } else {
        stopSyncScheduler();
      }

      res.json(settings);
    } catch (error) {
      console.error("Save sync settings error:", error);
      res.status(500).json({ error: "Failed to save sync settings" });
    }
  });

  app.get("/api/sync/settings", requirePermission('manage_settings'), async (req, res) => {
    try {
      const settings = await getSyncSettingsFromPostgres();
      if (!settings) {
        return res.json({
          autoRefreshEnabled: true,
          refreshIntervalMinutes: "3",
        });
      }
      res.json(settings);
    } catch (error) {
      console.error("Get sync settings error:", error);
      res.status(500).json({ error: "Failed to get sync settings" });
    }
  });

  app.get("/api/sync/logs", requirePermission('view_sync_logs'), async (req, res) => {
    try {
      const logs = await getSyncLogs(50);
      res.json(logs);
    } catch (error) {
      console.error("Get sync logs error:", error);
      res.status(500).json({ error: "Failed to fetch sync logs" });
    }
  });

  app.get("/api/sync/logs/unified", requirePermission('view_sync_logs'), async (req, res) => {
    try {
      const { source, limit = 100 } = req.query;
      const client = await pool.connect();
      
      try {
        let query = '';
        const params: any[] = [];
        
        if (source === 'ALLEGRO') {
          query = `
            SELECT 
              'ALLEGRO' as source,
              id,
              sync_started_at as timestamp,
              status,
              orders_fetched,
              orders_new,
              orders_updated,
              sync_duration_ms as duration_ms,
              error_message,
              NULL::text as order_codes
            FROM allegro.sync_logs
            ORDER BY sync_started_at DESC
            LIMIT $1
          `;
          params.push(limit);
        } else if (source === 'SHOPER') {
          query = `
            SELECT 
              'SHOPER' as source,
              id,
              sync_started_at as timestamp,
              CASE WHEN success THEN 'SUCCESS' ELSE 'ERROR' END as status,
              orders_count as orders_fetched,
              0 as orders_new,
              orders_count as orders_updated,
              EXTRACT(EPOCH FROM (sync_finished_at - sync_started_at)) * 1000 as duration_ms,
              error_message,
              NULL::text as order_codes
            FROM shoper.sync_logs
            ORDER BY sync_started_at DESC
            LIMIT $1
          `;
          params.push(limit);
        } else if (source === 'ODOO') {
          query = `
            SELECT 
              'ODOO' as source,
              id,
              created_at as timestamp,
              status,
              0 as orders_fetched,
              CASE WHEN operation = 'create' THEN 1 ELSE 0 END as orders_new,
              CASE WHEN operation = 'update' THEN 1 ELSE 0 END as orders_updated,
              duration as duration_ms,
              message as error_message,
              order_number::text as order_codes
            FROM public.odoo_sync_logs
            ORDER BY created_at DESC
            LIMIT $1
          `;
          params.push(limit);
        } else {
          // Wszystkie razem
          query = `
            SELECT * FROM (
              SELECT 
                'ALLEGRO' as source,
                id,
                sync_started_at as timestamp,
                status,
                orders_fetched,
                orders_new,
                orders_updated,
                sync_duration_ms as duration_ms,
                error_message,
                NULL::text as order_codes
              FROM allegro.sync_logs
              
              UNION ALL
              
              SELECT 
                'SHOPER' as source,
                id,
                sync_started_at as timestamp,
                CASE WHEN success THEN 'SUCCESS' ELSE 'ERROR' END as status,
                orders_count as orders_fetched,
                0 as orders_new,
                orders_count as orders_updated,
                EXTRACT(EPOCH FROM (sync_finished_at - sync_started_at)) * 1000 as duration_ms,
                error_message,
                NULL::text as order_codes
              FROM shoper.sync_logs
              
              UNION ALL
              
              SELECT 
                'ODOO' as source,
                id,
                created_at as timestamp,
                status,
                0 as orders_fetched,
                CASE WHEN operation = 'create' THEN 1 ELSE 0 END as orders_new,
                CASE WHEN operation = 'update' THEN 1 ELSE 0 END as orders_updated,
                duration as duration_ms,
                message as error_message,
                order_number::text as order_codes
              FROM public.odoo_sync_logs
            ) unified_logs
            ORDER BY timestamp DESC
            LIMIT $1
          `;
          params.push(limit);
        }
        
        const result = await client.query(query, params);
        res.json(result.rows);
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("Get unified sync logs error:", error);
      res.status(500).json({ error: "Failed to fetch unified sync logs" });
    }
  });

  app.post("/api/sync/manual", requirePermission('manage_sync'), async (req, res) => {
    try {
      await syncOrders();
      res.json({ success: true });
    } catch (error) {
      console.error("Manual sync error:", error);
      res.status(500).json({ error: "Failed to sync orders" });
    }
  });

  app.post("/api/products/download-images", requirePermission('manage_sync'), async (req, res) => {
    try {
      // Pobierz access token z połączenia Allegro
      const connection = await storage.getAllegroConnection();
      if (!connection || !connection.accessToken || !connection.isActive) {
        return res.status(400).json({ error: "No active Allegro connection" });
      }

      let accessToken = connection.accessToken;
      
      // Sprawdź czy token nie wygasł
      if (connection.tokenExpiresAt && new Date(connection.tokenExpiresAt) < new Date()) {
        if (connection.refreshToken) {
          const credentials = await getCredentials();
          if (!credentials) {
            return res.status(400).json({ error: "No credentials available for token refresh" });
          }
          
          const tokenData = await refreshAccessToken(
            credentials.clientId,
            credentials.clientSecret,
            connection.refreshToken
          );
          
          const expiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
          
          await storage.updateConnectionTokens(
            connection.id,
            tokenData.access_token,
            tokenData.refresh_token,
            expiresAt
          );
          
          accessToken = tokenData.access_token;
        } else {
          return res.status(400).json({ error: "Token expired and no refresh token available" });
        }
      }

      // Pobierz wszystkie produkty Allegro bez zdjęć
      const productsResult = await pool.query(`
        SELECT product_id, external_id, name
        FROM allegro.products
        WHERE image_url IS NULL OR image_url = ''
        ORDER BY id
        LIMIT 100
      `);

      console.log(`📸 Starting download of images for ${productsResult.rows.length} Allegro products...`);
      
      let downloaded = 0;
      let failed = 0;

      const { getAllegroOfferDetails } = await import('./allegro-api.js');
      
      for (const product of productsResult.rows) {
        try {
          const offerDetails = await getAllegroOfferDetails(product.product_id, accessToken);
          
          if (offerDetails?.images && offerDetails.images.length > 0) {
            const imageUrl = offerDetails.images[0].url;
            
            // Zaktualizuj product w bazie
            await pool.query(`
              UPDATE allegro.products
              SET image_url = $1, updated_at = NOW()
              WHERE product_id = $2
            `, [imageUrl,
        unit, product.product_id]);
            
            // Zaktualizuj również order_items
            await pool.query(`
              UPDATE allegro.order_items
              SET image_url = $1
              WHERE offer_id = $2
            `, [imageUrl,
        unit, product.product_id]);
            
            downloaded++;
            console.log(`✅ Downloaded image for product ${product.product_id}: ${product.name}`);
          } else {
            failed++;
            console.log(`⚠️  No image found for product ${product.product_id}: ${product.name}`);
          }
          
          // Poczekaj 100ms między requestami żeby nie przeciążyć API
          await new Promise(resolve => setTimeout(resolve, 100));
        } catch (error) {
          failed++;
          console.error(`❌ Failed to download image for product ${product.product_id}:`, error);
        }
      }

      console.log(`📸 Image download completed: ${downloaded} downloaded, ${failed} failed`);
      res.json({ 
        success: true, 
        total: productsResult.rows.length,
        downloaded, 
        failed 
      });
    } catch (error) {
      console.error("Download images error:", error);
      res.status(500).json({ error: "Failed to download images" });
    }
  });

  app.post("/api/sync/date-range", requirePermission('manage_sync'), async (req, res) => {
    try {
      const { startDate, endDate } = req.body;
      
      if (!startDate || !endDate) {
        return res.status(400).json({ error: "startDate and endDate are required" });
      }
      
      const connection = await storage.getAllegroConnection();
      if (!connection || !connection.accessToken || !connection.isActive) {
        return res.status(400).json({ error: "No active Allegro connection" });
      }

      let accessToken = connection.accessToken;
      
      if (
        connection.tokenExpiresAt &&
        new Date(connection.tokenExpiresAt) < new Date()
      ) {
        if (connection.refreshToken) {
          const credentials = await getCredentials();
          if (!credentials) {
            return res.status(400).json({ error: "No credentials available for token refresh" });
          }
          
          const tokenData = await refreshAccessToken(
            credentials.clientId,
            credentials.clientSecret,
            connection.refreshToken
          );
          
          const expiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
          
          await storage.updateConnectionTokens(
            connection.id,
            tokenData.access_token,
            tokenData.refresh_token,
            expiresAt
          );
          
          accessToken = tokenData.access_token;
        } else {
          return res.status(400).json({ error: "Token expired and no refresh token available" });
        }
      }

      const startDateTime = new Date(startDate);
      startDateTime.setHours(0, 0, 0, 0);
      const start = startDateTime.toISOString();
      
      const endDateTime = new Date(endDate);
      endDateTime.setHours(23, 59, 59, 999);
      const end = endDateTime.toISOString();
      
      console.log(`📅 Fetching Allegro orders from ${start} to ${end}`);
      
      let allegroOrders: AllegroOrderItem[] = [];
      let offset = 0;
      const limit = 100;
      let hasMore = true;
      
      while (hasMore) {
        const response = await axios.get<{ checkoutForms: AllegroOrderItem[] }>(
          `${ALLEGRO_API_URL}/order/checkout-forms`,
          {
            headers: {
              Authorization: `Bearer ${accessToken}`,
              Accept: "application/vnd.allegro.public.v1+json",
            },
            params: {
              "updatedAt.gte": start,
              "updatedAt.lte": end,
              limit,
              offset,
            },
          }
        );
        
        const fetchedOrders = response.data.checkoutForms || [];
        allegroOrders = allegroOrders.concat(fetchedOrders);
        
        if (fetchedOrders.length < limit) {
          hasMore = false;
        } else {
          offset += limit;
        }
      }
      
      console.log(`📦 Fetched ${allegroOrders.length} total orders from Allegro`);
      let ordersNew = 0;
      let ordersUpdated = 0;
      
      for (const allegroOrder of allegroOrders) {
        try {
          const shipments = await fetchShipmentsFromAllegro(accessToken, allegroOrder.id);
          const customerReturns = await fetchCustomerReturnsFromAllegro(accessToken, allegroOrder.id);
          await saveOrderToPostgres(allegroOrder, shipments, customerReturns, accessToken);
          await enrichAllegroOrderWithImages(allegroOrder, allegroOrder.id);
          
          const result = await saveOrderToCommerce(
            'ALLEGRO',
            allegroOrder.id,
            allegroOrder,
            shipments,
            customerReturns,
            accessToken
          );
          
          if (result.isNew) {
            ordersNew++;
          } else {
            ordersUpdated++;
          }
        } catch (error) {
          console.error(`Failed to save order ${allegroOrder.id}:`, error);
        }
      }
      
      res.json({ 
        success: true, 
        ordersFetched: allegroOrders.length,
        ordersNew,
        ordersUpdated
      });
    } catch (error: any) {
      console.error("Allegro date range sync error:", error);
      res.status(500).json({ error: error.message || "Failed to sync Allegro orders by date range" });
    }
  });

  // Endpoint do wymuszenia aktualizacji zamówienia z platformy (payment status fix)
  app.post("/api/orders/force-update", async (req, res) => {
    try {
      const { sourceOrderId, platform } = req.body;
      
      if (!sourceOrderId || !platform) {
        return res.status(400).json({ error: "Missing sourceOrderId or platform" });
      }

      if (platform === 'ALLEGRO') {
        const connection = await storage.getAllegroConnection();
        if (!connection?.accessToken) {
          return res.status(400).json({ error: "No Allegro connection" });
        }

        const { fetchPaymentRefunds } = await import('./allegro-api.js');
        
        const allegroOrder = await fetchRawOrderFromAllegro(connection.accessToken, sourceOrderId);
        const shipments = await fetchShipmentsFromAllegro(connection.accessToken, sourceOrderId);
        const customerReturns = await fetchCustomerReturnsFromAllegro(connection.accessToken, sourceOrderId);
        const refunds = await fetchPaymentRefunds(sourceOrderId, connection.accessToken);
        
        if (refunds && refunds.length > 0 && allegroOrder.payment) {
          allegroOrder.payment.refundReconciliation = refunds;
          allegroOrder.payment.refundAmount = refunds[0]?.totalValue?.amount || 0;
          allegroOrder.payment.refundDate = refunds[0]?.createdAt || null;
        }
        
        const { saveOrderToCommerce } = await import('./postgres.js');
        await saveOrderToCommerce('ALLEGRO', sourceOrderId, allegroOrder, shipments, customerReturns, connection.accessToken);
        
        res.json({ success: true, message: "Zaktualizowano z Allegro" });
      } else if (platform === 'SHOPER') {
        const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
        
        const payments = await getShoperPayments();
        const deliveries = await getShoperDeliveries();
        
        const paymentMap = new Map(payments.map((p: any) => [p.payment_id?.toString(), p.translations?.pl_PL?.title || p.name]));
        const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id?.toString(), d.name]));
        
        const shoperOrders = await getShoperOrders(1, { order_id: sourceOrderId });
        
        if (!shoperOrders || shoperOrders.length === 0) {
          return res.status(404).json({ error: "Order not found on Shoper platform" });
        }
        
        const shoperOrder = shoperOrders[0];
        
        if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
          shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
        }
        if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
          shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
        }
        
        const orderProducts = await getShoperOrderProducts(sourceOrderId);
        shoperOrder.products_data = orderProducts;
        
        let parcels: any[] = [];
        try {
          parcels = await getShoperParcels(sourceOrderId);
        } catch (parcelError) {
          console.warn(`⚠️ Could not fetch parcels for order ${sourceOrderId}`);
        }
        
        const { saveOrderToCommerce } = await import('./postgres.js');
        await saveOrderToCommerce('SHOPER', sourceOrderId, shoperOrder, parcels, []);
        
        res.json({ success: true, message: "Zaktualizowano z Shoper" });
      } else {
        return res.status(400).json({ error: "Invalid platform" });
      }
    } catch (error: any) {
      console.error("Force update order error:", error);
      res.status(500).json({ error: error.message || "Failed to update order" });
    }
  });

  // Nowy endpoint do pobierania zamówienia z platformy po source_order_id
  app.post("/api/orders/fetch-from-platform", async (req, res) => {
    try {
      const { sourceOrderId, platform } = req.body;
      
      if (!sourceOrderId || !platform) {
        return res.status(400).json({ error: "sourceOrderId and platform are required" });
      }
      
      const source = platform.toUpperCase();
      const source_order_id = sourceOrderId;
      
      if (source === 'ALLEGRO') {
        const connection = await storage.getAllegroConnection();
        if (!connection || !connection.accessToken || !connection.isActive) {
          return res.status(400).json({ error: "No active Allegro connection" });
        }

        let accessToken = connection.accessToken;
        
        if (
          connection.tokenExpiresAt &&
          new Date(connection.tokenExpiresAt) < new Date()
        ) {
          if (connection.refreshToken) {
            const credentials = await getCredentials();
            if (!credentials) {
              return res.status(400).json({ error: "No credentials available for token refresh" });
            }
            
            const tokenData = await refreshAccessToken(
              credentials.clientId,
              credentials.clientSecret,
              connection.refreshToken
            );
            
            const expiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
            
            await storage.updateConnectionTokens(
              connection.id,
              tokenData.access_token,
              tokenData.refresh_token,
              expiresAt
            );
            
            accessToken = tokenData.access_token;
          } else {
            return res.status(400).json({ error: "Token expired and no refresh token available" });
          }
        }
        
        const allegroOrder = await fetchRawOrderFromAllegro(accessToken, source_order_id);
        const shipments = await fetchShipmentsFromAllegro(accessToken, source_order_id);
        const customerReturns = await fetchCustomerReturnsFromAllegro(accessToken, source_order_id);
        
        const { fetchPaymentRefunds } = await import('./allegro-api.js');
        const refunds = await fetchPaymentRefunds(source_order_id, accessToken);
        
        if (refunds && refunds.length > 0 && allegroOrder.payment) {
          allegroOrder.payment.refundReconciliation = refunds;
          allegroOrder.payment.refundAmount = refunds[0]?.totalValue?.amount || 0;
          allegroOrder.payment.refundDate = refunds[0]?.createdAt || null;
        }
        
        await saveOrderToPostgres(allegroOrder, shipments, customerReturns, accessToken);
        await enrichAllegroOrderWithImages(allegroOrder, source_order_id);
        await saveOrderToCommerce('ALLEGRO', source_order_id, allegroOrder, shipments, customerReturns, accessToken);
        
        res.json({ success: true, platform: 'ALLEGRO' });
      } else if (source === 'SHOPER') {
        const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
        
        const payments = await getShoperPayments();
        const deliveries = await getShoperDeliveries();
        
        const paymentMap = new Map(payments.map((p: any) => [p.payment_id, p.translations?.pl_PL?.title || p.name]));
        const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id, d.name]));
        
        const shoperOrders = await getShoperOrders(1, { order_id: source_order_id });
        
        if (shoperOrders.length === 0) {
          return res.status(404).json({ error: "Order not found on Shoper platform" });
        }
        
        const shoperOrder = shoperOrders[0];
        
        if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
          shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
        }
        if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
          shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
        }
        
        const orderProducts = await getShoperOrderProducts(source_order_id);
        shoperOrder.products_data = orderProducts;
        
        let parcels: any[] = [];
        try {
          parcels = await getShoperParcels(source_order_id);
        } catch (parcelError) {
          console.warn(`⚠️ Could not fetch parcels for order ${source_order_id}`);
        }
        
        await saveShoperOrderToPostgres(shoperOrder, parcels);
        await saveOrderToCommerce('SHOPER', source_order_id, shoperOrder, parcels, []);
        
        res.json({ success: true, platform: 'SHOPER' });
      } else {
        return res.status(400).json({ error: "Unknown platform" });
      }
    } catch (error: any) {
      console.error("Fetch from platform error:", error);
      res.status(500).json({ error: error.message || "Failed to fetch order from platform" });
    }
  });

  app.post("/api/orders/:orderId/refresh", async (req, res) => {
    try {
      const { orderId } = req.params;
      
      const orderResult = await pool.query(`
        SELECT id, source, source_order_id 
        FROM commerce.orders 
        WHERE id = $1
      `, [orderId]);
      
      if (orderResult.rows.length === 0) {
        return res.status(404).json({ error: "Order not found" });
      }
      
      const order = orderResult.rows[0];
      const { source, source_order_id } = order;

      if (source === 'ALLEGRO') {
        const connection = await storage.getAllegroConnection();
        if (!connection || !connection.accessToken || !connection.isActive) {
          return res.status(400).json({ error: "No active Allegro connection" });
        }

        let accessToken = connection.accessToken;
        
        if (
          connection.tokenExpiresAt &&
          new Date(connection.tokenExpiresAt) < new Date()
        ) {
          if (connection.refreshToken) {
            const credentials = await getCredentials();
            if (!credentials) {
              return res.status(400).json({ error: "No credentials available for token refresh" });
            }
            
            const tokenData = await refreshAccessToken(
              credentials.clientId,
              credentials.clientSecret,
              connection.refreshToken
            );
            
            const expiresAt = new Date(Date.now() + tokenData.expires_in * 1000);
            
            await storage.updateConnectionTokens(
              connection.id,
              tokenData.access_token,
              tokenData.refresh_token,
              expiresAt
            );
            
            accessToken = tokenData.access_token;
          } else {
            return res.status(400).json({ error: "Token expired and no refresh token available" });
          }
        }
        
        const allegroOrder = await fetchRawOrderFromAllegro(accessToken, source_order_id);
        const shipments = await fetchShipmentsFromAllegro(accessToken, source_order_id);
        const customerReturns = await fetchCustomerReturnsFromAllegro(accessToken, source_order_id);
        
        // Pobierz zwroty płatności (stabilniejsze dane niż customerReturns)
        const { fetchPaymentRefunds } = await import('./allegro-api.js');
        const refunds = await fetchPaymentRefunds(source_order_id, accessToken);
        
        // Dodaj refund_reconciliation do allegroOrder.payment
        if (refunds && refunds.length > 0 && allegroOrder.payment) {
          allegroOrder.payment.refundReconciliation = refunds;
          allegroOrder.payment.refundAmount = refunds[0]?.totalValue?.amount || 0;
          allegroOrder.payment.refundDate = refunds[0]?.createdAt || null;
        }
        
        await saveOrderToPostgres(allegroOrder, shipments, customerReturns, accessToken);
        await enrichAllegroOrderWithImages(allegroOrder, source_order_id);
        await saveOrderToCommerce('ALLEGRO', source_order_id, allegroOrder, shipments, customerReturns, accessToken);
        
        res.json({ success: true, platform: 'ALLEGRO' });
      } else if (source === 'SHOPER') {
        const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
        
        const payments = await getShoperPayments();
        const deliveries = await getShoperDeliveries();
        
        // Use translated title (pl_PL) for payment method name, fallback to technical name
        const paymentMap = new Map(payments.map((p: any) => [p.payment_id, p.translations?.pl_PL?.title || p.name]));
        const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id, d.name]));
        
        const shoperOrders = await getShoperOrders(1, { order_id: source_order_id });
        
        if (shoperOrders.length === 0) {
          return res.status(404).json({ error: "Order not found on Shoper platform" });
        }
        
        const shoperOrder = shoperOrders[0];
        
        if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
          shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
        }
        if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
          shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
        }
        
        const orderProducts = await getShoperOrderProducts(source_order_id);
        shoperOrder.products_data = orderProducts;
        
        let parcels: any[] = [];
        try {
          parcels = await getShoperParcels(source_order_id);
        } catch (parcelError) {
          console.warn(`⚠️ Could not fetch parcels for order ${source_order_id}`);
        }
        
        await saveShoperOrderToPostgres(shoperOrder, parcels);
        await saveOrderToCommerce('SHOPER', source_order_id, shoperOrder, parcels, []);
        
        res.json({ success: true, platform: 'SHOPER' });
      } else {
        return res.status(400).json({ error: "Unknown platform" });
      }
    } catch (error: any) {
      console.error("Refresh order error:", error);
      res.status(500).json({ error: error.message || "Failed to refresh order" });
    }
  });

  // Endpoint do naprawy zamówień bez numerów
  app.post("/api/orders/fix-numbers", async (req, res) => {
    try {
      const { fixOrdersWithoutNumbers } = await import('./postgres.js');
      const fixed = await fixOrdersWithoutNumbers();
      
      res.json({ 
        success: true, 
        fixed,
        message: fixed > 0 ? `Naprawiono ${fixed} zamówień` : 'Wszystkie zamówienia mają numery'
      });
    } catch (error: any) {
      console.error("Fix order numbers error:", error);
      res.status(500).json({ error: error.message || "Nie udało się naprawić numerów zamówień" });
    }
  });

  // Endpoint do ręcznej synchronizacji zamówienia z Odoo
  app.post("/api/orders/:orderId/sync-to-odoo", async (req, res) => {
    try {
      let { orderId } = req.params;
      
      // Obsługa numeru z # (np. "#1" → "00001") lub bez wiodących zer ("1" → "00001")
      orderId = orderId.replace('#', '').padStart(5, '0');
      
      const orderResult = await pool.query(`
        SELECT order_number, source 
        FROM commerce.orders 
        WHERE order_number = $1
      `, [orderId]);
      
      if (orderResult.rows.length === 0) {
        return res.status(404).json({ error: "Order not found" });
      }
      
      const order = orderResult.rows[0];
      const { order_number, source } = order;
      
      // Dodaj zamówienie do kolejki synchronizacji Odoo
      const { addOrderToSyncQueue } = await import('./odoo-sync.js');
      
      // Sprawdź czy zamówienie ma już ID Odoo (wtedy UPDATE, w przeciwnym razie CREATE)
      const odooCheckResult = await pool.query(`
        SELECT odoo_order_id FROM commerce.orders WHERE order_number = $1
      `, [orderId]);
      
      const operation = odooCheckResult.rows[0]?.odoo_order_id ? 'update' : 'create';
      
      await addOrderToSyncQueue(order_number, source, operation);
      
      res.json({ 
        success: true, 
        orderNumber: order_number,
        operation,
        message: `Zamówienie #${order_number} dodane do kolejki synchronizacji Odoo (${operation})`
      });
    } catch (error: any) {
      console.error("Manual Odoo sync error:", error);
      res.status(500).json({ error: error.message || "Failed to sync order to Odoo" });
    }
  });

  app.post("/api/shoper/sync", requirePermission('manage_sync'), async (req, res) => {
    try {
      const result = await syncShoperOrders();
      res.json(result);
    } catch (error: any) {
      console.error("Shoper sync error:", error);
      res.status(500).json({ error: error.message || "Failed to sync Shoper orders" });
    }
  });

  app.post("/api/shoper/sync/date-range", requirePermission('manage_sync'), async (req, res) => {
    try {
      const { startDate, endDate } = req.body;
      
      if (!startDate || !endDate) {
        return res.status(400).json({ error: "startDate and endDate are required" });
      }
      
      const { getShoperOrders, getShoperOrderProducts, getShoperParcels, getShoperPayments, getShoperDeliveries } = await import('./shoper-api.js');
      const { saveOrderToCommerce, saveShoperOrderToPostgres } = await import('./postgres.js');

      const startDateTime = new Date(startDate);
      startDateTime.setHours(0, 0, 0, 0);
      const start = startDateTime.toISOString().replace('T', ' ').replace('Z', '');
      
      const endDateTime = new Date(endDate);
      endDateTime.setHours(23, 59, 59, 999);
      const end = endDateTime.toISOString().replace('T', ' ').replace('Z', '');
      
      console.log(`📅 Fetching Shoper orders from ${start} to ${end}`);
      
      const payments = await getShoperPayments();
      const deliveries = await getShoperDeliveries();
      
      // Use translated title (pl_PL) for payment method name, fallback to technical name
      const paymentMap = new Map(payments.map((p: any) => [p.payment_id, p.translations?.pl_PL?.title || p.name]));
      const deliveryMap = new Map(deliveries.map((d: any) => [d.delivery_id, d.name]));
      
      const filters = {
        updated_at: {
          ">=": start,
          "<=": end
        }
      };
      
      const shoperOrders = await getShoperOrders(100, filters);
      let ordersNew = 0;
      let ordersUpdated = 0;
      let ordersFailed = 0;

      console.log(`📦 Fetched ${shoperOrders.length} orders from Shoper API`);

      for (const shoperOrder of shoperOrders) {
        try {
          const orderId = shoperOrder.order_id || shoperOrder.id;
          
          if (shoperOrder.payment_id && paymentMap.has(shoperOrder.payment_id.toString())) {
            shoperOrder.payment_method_name = paymentMap.get(shoperOrder.payment_id.toString());
          }
          if (shoperOrder.shipping_id && deliveryMap.has(shoperOrder.shipping_id.toString())) {
            shoperOrder.delivery_method_name = deliveryMap.get(shoperOrder.shipping_id.toString());
          }
          
          try {
            const orderProducts = await getShoperOrderProducts(orderId);
            shoperOrder.products_data = orderProducts;
          } catch (productError) {
            console.warn(`⚠️ Could not fetch products for order ${orderId}`);
            shoperOrder.products_data = [];
          }
          
          let parcels: any[] = [];
          try {
            parcels = await getShoperParcels(orderId);
          } catch (parcelError) {
            console.warn(`⚠️ Could not fetch parcels for order ${orderId}`);
          }
          
          await saveShoperOrderToPostgres(shoperOrder, parcels);
          await enrichShoperOrderWithImages(shoperOrder, orderId?.toString());
          
          const result = await saveOrderToCommerce(
            'SHOPER',
            orderId?.toString(),
            shoperOrder,
            parcels,
            [] // Shoper nie ma zwrotów
          );
          
          if (result.isNew) {
            ordersNew++;
          } else {
            ordersUpdated++;
          }
        } catch (error) {
          ordersFailed++;
          console.error(`❌ Failed to process Shoper order ${shoperOrder.order_id}:`, error);
        }
      }
      
      res.json({ 
        success: true, 
        ordersFetched: shoperOrders.length,
        ordersNew,
        ordersUpdated,
        ordersFailed
      });
    } catch (error: any) {
      console.error("Shoper date range sync error:", error);
      res.status(500).json({ error: error.message || "Failed to sync Shoper orders by date range" });
    }
  });

  app.get("/api/shoper/sync/status", async (req, res) => {
    try {
      const settings = await getSyncSettingsFromPostgres();
      res.json({
        lastShoperSyncAt: settings?.lastShoperSyncAt || null,
        isRefreshing: isShoperSyncing,
      });
    } catch (error) {
      console.error("Get Shoper sync status error:", error);
      res.status(500).json({ error: "Failed to get Shoper sync status" });
    }
  });

  app.post("/api/shoper/sync/settings", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { shoperAutoRefreshEnabled, shoperRefreshIntervalMinutes, shoperFullSyncEnabled, shoperFullSyncWindowHours } = req.body;

      const currentSettings = await getSyncSettingsFromPostgres();
      const settings = await createOrUpdateSyncSettingsInPostgres({
        autoRefreshEnabled: currentSettings?.autoRefreshEnabled ?? true,
        refreshIntervalMinutes: currentSettings?.refreshIntervalMinutes ?? "3",
        lastSyncAt: currentSettings?.lastSyncAt || null,
        shoperAutoRefreshEnabled,
        shoperRefreshIntervalMinutes,
        shoperFullSyncEnabled: shoperFullSyncEnabled ?? true,
        shoperFullSyncWindowHours: shoperFullSyncWindowHours ?? '2',
        lastShoperSyncAt: currentSettings?.lastShoperSyncAt || null,
      });

      if (shoperAutoRefreshEnabled) {
        const interval = parseInt(shoperRefreshIntervalMinutes, 10);
        startShoperSyncScheduler(interval);
      } else {
        stopShoperSyncScheduler();
      }

      res.json(settings);
    } catch (error) {
      console.error("Save Shoper sync settings error:", error);
      res.status(500).json({ error: "Failed to save Shoper sync settings" });
    }
  });

  app.get("/api/shoper/sync/settings", requirePermission('manage_settings'), async (req, res) => {
    try {
      const settings = await getSyncSettingsFromPostgres();
      if (!settings) {
        return res.json({
          shoperAutoRefreshEnabled: true,
          shoperRefreshIntervalMinutes: "5",
        });
      }
      res.json({
        shoperAutoRefreshEnabled: settings.shoperAutoRefreshEnabled,
        shoperRefreshIntervalMinutes: settings.shoperRefreshIntervalMinutes,
        lastShoperSyncAt: settings.lastShoperSyncAt,
      });
    } catch (error) {
      console.error("Get Shoper sync settings error:", error);
      res.status(500).json({ error: "Failed to get Shoper sync settings" });
    }
  });

  // Recent Updates - pokazuje ostatnio zmodyfikowane zamówienia
  app.get("/api/recent-updates", async (req, res) => {
    try {
      // Pobierz zamówienia zmodyfikowane w ciągu ostatnich 24h
      const recentOrders = await getRecentlyUpdatedOrders(1440); // 24 godziny
      res.json(recentOrders);
    } catch (error) {
      console.error("Get recent updates error:", error);
      res.status(500).json({ error: "Failed to fetch recent updates" });
    }
  });

  // Order Changes - historia zmian w zamówieniach
  app.get("/api/order-changes", async (req, res) => {
    try {
      const { orderNumber, limit = 100 } = req.query;
      
      let query = `
        SELECT 
          id, order_id, order_number, source, change_type,
          field_changed, old_value, new_value, detected_at
        FROM public.order_changes
      `;
      
      const params: any[] = [];
      if (orderNumber) {
        query += ` WHERE order_number = $1`;
        params.push(parseInt(orderNumber as string, 10));
        query += ` ORDER BY detected_at DESC LIMIT $2`;
        params.push(limit);
      } else {
        query += ` ORDER BY detected_at DESC LIMIT $1`;
        params.push(limit);
      }
      
      const result = await pool.query(query, params);
      res.json(result.rows);
    } catch (error) {
      console.error("Get order changes error:", error);
      res.status(500).json({ error: "Failed to fetch order changes" });
    }
  });

  app.get("/api/products/images/:filename", async (req, res) => {
    try {
      const { filename } = req.params;
      const imagePath = path.join(process.cwd(), "attached_assets", "products", filename);
      
      const fileExists = await fs.access(imagePath).then(() => true).catch(() => false);
      
      if (!fileExists) {
        return res.status(404).json({ error: "Image not found" });
      }
      
      const ext = path.extname(filename).toLowerCase();
      const contentType = ext === '.jpg' || ext === '.jpeg' ? 'image/jpeg' : 
                         ext === '.png' ? 'image/png' : 
                         ext === '.webp' ? 'image/webp' : 'image/jpeg';
      
      res.setHeader('Content-Type', contentType);
      res.setHeader('Cache-Control', 'public, max-age=31536000');
      
      const imageBuffer = await fs.readFile(imagePath);
      res.send(imageBuffer);
    } catch (error) {
      console.error("Error serving product image:", error);
      res.status(500).json({ error: "Failed to serve image" });
    }
  });


  // POST /api/marketplace-products/:externalId/sync-image - Synchronizuj zdjęcie pojedynczego produktu marketplace
  app.post("/api/marketplace-products/:externalId/sync-image", isAuthenticated, async (req, res) => {
    try {
      const { externalId } = req.params;
      
      // Pobierz produkt z bazy
      const productResult = await pool.query(
        `SELECT offer_external_id, name, source, image_url
         FROM commerce.marketplace_products
         WHERE offer_external_id = $1`,
        [externalId]
      );

      if (productResult.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }

      const product = productResult.rows[0];

      // Tylko dla produktów Allegro
      if (product.source.toUpperCase() !== 'ALLEGRO') {
        return res.status(400).json({ error: "Image sync is only supported for Allegro products" });
      }

      // Pobierz access token
      const tokenResult = await pool.query('SELECT access_token FROM allegro_connections WHERE is_active = true LIMIT 1');
      const userToken = tokenResult.rows[0]?.access_token;

      if (!userToken) {
        return res.status(400).json({ error: "Allegro token not available. Please authorize Allegro first." });
      }

      // Pobierz zdjęcie z Allegro API
      const { downloadProductImage } = await import('./allegro-api.js');
      const filename = await downloadProductImage(product.name, externalId, userToken);

      if (filename) {
        // Zaktualizuj image_url w bazie
        const localPath = `/api/products/images/${filename}`;
        await pool.query(
          `UPDATE commerce.marketplace_products 
           SET image_url = $1, updated_at = NOW()
           WHERE offer_external_id = $2`,
          [localPath, externalId]
        );

        // Zaktualizuj też wszystkie order_items z tym produktem
        await pool.query(
          `UPDATE commerce.order_items 
           SET image_url = $1
           WHERE offer_external_id = $2`,
          [localPath, externalId]
        );

        res.json({ 
          success: true, 
          filename,
          localPath,
          message: "Image synchronized successfully" 
        });
      } else {
        res.status(500).json({ error: "Failed to download image from Allegro" });
      }
    } catch (error) {
      console.error("Sync marketplace product image error:", error);
      res.status(500).json({ error: "Failed to sync product image" });
    }
  });

  // GET /api/products - Lista marketplace products (z Allegro/Shoper)
  app.get("/api/products", isAuthenticated, async (req, res) => {
    try {
      const {
        search,           // Wyszukiwarka główna (nazwa produktu)
        source,           // Filtr: źródło (ALLEGRO/SHOPER)
        linked,           // Filtr: status połączenia (linked/unlinked)
      } = req.query;

      const params: any[] = [];
      const conditions: string[] = [];

      // Wyszukiwanie po nazwie produktu, SKU, lub offer_external_id
      if (search && typeof search === 'string' && search.trim()) {
        const searchTerm = search.trim();
        
        // Wykryj operator AND (;) lub OR (,)
        if (searchTerm.includes(';')) {
          // AND - wszystkie tokeny muszą być obecne
          const tokens = searchTerm.split(';').map(t => t.trim()).filter(t => t);
          const andConditions = tokens.map((token) => {
            params.push(`%${token}%`);
            params.push(`%${token}%`);
            params.push(`%${token}%`);
            return `(p.name ILIKE $${params.length - 2} OR p.sku ILIKE $${params.length - 1} OR p.offer_external_id ILIKE $${params.length})`;
          });
          conditions.push(`(${andConditions.join(' AND ')})`);
        } else if (searchTerm.includes(',')) {
          // OR - przynajmniej jeden token musi być obecny
          const tokens = searchTerm.split(',').map(t => t.trim()).filter(t => t);
          const orConditions = tokens.map((token) => {
            params.push(`%${token}%`);
            params.push(`%${token}%`);
            params.push(`%${token}%`);
            return `(p.name ILIKE $${params.length - 2} OR p.sku ILIKE $${params.length - 1} OR p.offer_external_id ILIKE $${params.length})`;
          });
          conditions.push(`(${orConditions.join(' OR ')})`);
        } else {
          // Pojedynczy token - szukaj w nazwie, SKU lub offer_external_id
          params.push(`%${searchTerm}%`);
          params.push(`%${searchTerm}%`);
          params.push(`%${searchTerm}%`);
          conditions.push(`(p.name ILIKE $${params.length - 2} OR p.sku ILIKE $${params.length - 1} OR p.offer_external_id ILIKE $${params.length})`);
        }
      }

      // Filtr źródła
      if (source && typeof source === 'string') {
        params.push(source);
        conditions.push(`p.source = $${params.length}`);
      }

      // Filtr statusu połączenia (catalog product OR catalog set)
      if (linked && typeof linked === 'string') {
        if (linked === 'linked') {
          conditions.push(`(cp.id IS NOT NULL OR ps.id IS NOT NULL)`);
        } else if (linked === 'unlinked') {
          conditions.push(`(cp.id IS NULL AND ps.id IS NULL)`);
        }
      }

      const whereClause = conditions.length > 0 
        ? `WHERE ${conditions.join(' AND ')}`
        : '';

      const result = await pool.query(`
        SELECT 
          p.id,
          p.offer_external_id,
          p.source,
          p.name,
          p.image_url,
          p.last_sold_at,
          p.times_sold,
          p.total_quantity_sold,
          p.created_at,
          p.updated_at,
          ppd.link_type AS "linkType",
          cp.id AS "catalogProductId",
          cp.sku AS "catalogProductSku",
          ps.id AS "catalogSetId",
          ps.sku AS "catalogSetSku",
          CASE WHEN (cp.id IS NOT NULL OR ps.id IS NOT NULL) THEN true ELSE false END AS "isLinked"
        FROM commerce.marketplace_products p
        LEFT JOIN catalog.product_platform_data ppd 
          ON LOWER(ppd.platform) = LOWER(p.source) 
          AND ppd.external_id = p.offer_external_id
        LEFT JOIN catalog.products cp ON ppd.product_id = cp.id AND ppd.link_type = 'product'
        LEFT JOIN product_creator.product_sets ps ON ppd.set_id = ps.id AND ppd.link_type = 'set'
        ${whereClause}
        ORDER BY p.last_sold_at DESC NULLS LAST, p.name ASC
      `, params);

      // Convert local URLs to SFTP URLs if needed
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      const products = result.rows.map((product: any) => {
        if (product.image_url && !product.image_url.startsWith('http')) {
          const filename = product.image_url.split('/').pop();
          product.image_url = adapter.getUrl(`products/images/${filename}`);
        }
        return product;
      });

      res.json(products);
    } catch (error) {
      console.error("Get marketplace products error:", error);
      res.status(500).json({ error: "Failed to fetch marketplace products" });
    }
  });

  // GET /api/products/filter-values - Pobierz dostępne wartości filtrów dla marketplace products
  app.get("/api/products/filter-values", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT DISTINCT source
        FROM commerce.marketplace_products
        ORDER BY source
      `);

      res.json({
        sources: result.rows.map(r => r.source).filter(Boolean),
      });
    } catch (error) {
      console.error("Get filter values error:", error);
      res.status(500).json({ error: "Failed to fetch filter values" });
    }
  });

  // GET /api/marketplace-products/:sku - Pobierz szczegóły produktu marketplace z listą zamówień
  // Search marketplace products by SKU or name (for linking dialog)
  // Supports semicolon (;) separated AND filtering and comma (,) separated OR filtering
  app.get("/api/commerce/marketplace-products", isAuthenticated, async (req, res) => {
    try {
      const { search } = req.query;

      // Validate search query
      if (!search || typeof search !== 'string' || search.length < 2) {
        return res.status(400).json({ 
          error: "Search query must be at least 2 characters long" 
        });
      }

      const params: any[] = [];
      const conditions: string[] = [];
      
      // Detect operator: semicolon (;) for AND, comma (,) for OR
      if (search.includes(';')) {
        // AND - all tokens must be present
        const tokens = search.split(';').map(t => t.trim()).filter(t => t);
        const andConditions = tokens.map((token) => {
          params.push(`%${token}%`);
          params.push(`%${token}%`);
          params.push(`%${token}%`);
          return `(sku ILIKE $${params.length - 2} OR offer_external_id ILIKE $${params.length - 1} OR name ILIKE $${params.length})`;
        });
        conditions.push(`(${andConditions.join(' AND ')})`);
      } else if (search.includes(',')) {
        // OR - at least one token must be present
        const tokens = search.split(',').map(t => t.trim()).filter(t => t);
        const orConditions = tokens.map((token) => {
          params.push(`%${token}%`);
          params.push(`%${token}%`);
          params.push(`%${token}%`);
          return `(sku ILIKE $${params.length - 2} OR offer_external_id ILIKE $${params.length - 1} OR name ILIKE $${params.length})`;
        });
        conditions.push(`(${orConditions.join(' OR ')})`);
      } else {
        // Single token - search in sku, offer_external_id, or name
        params.push(`%${search}%`);
        params.push(`%${search}%`);
        params.push(`%${search}%`);
        conditions.push(`(sku ILIKE $${params.length - 2} OR offer_external_id ILIKE $${params.length - 1} OR name ILIKE $${params.length})`);
      }

      const whereClause = conditions.length > 0 
        ? `WHERE ${conditions.join(' AND ')}`
        : '';
      
      // Add exact match prioritization params
      params.push(`${search}%`);
      const exactMatchParam = params.length;

      // Search by SKU or name with prioritization for exact SKU matches
      const result = await pool.query(`
        SELECT 
          id,
          offer_external_id AS external_id,
          source AS platform,
          COALESCE(sku, offer_external_id) AS sku,
          name
        FROM commerce.marketplace_products
        ${whereClause}
        ORDER BY 
          CASE WHEN sku ILIKE $${exactMatchParam} OR offer_external_id ILIKE $${exactMatchParam} THEN 0 ELSE 1 END,
          sku
        LIMIT 20
      `, params);

      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error searching marketplace products:", error);
      res.status(500).json({ error: "Failed to search marketplace products" });
    }
  });

  app.get("/api/marketplace-products/:sku", isAuthenticated, async (req, res) => {
    try {
      const { sku } = req.params;
      
      // Pobierz produkt marketplace
      const productResult = await pool.query(`
        SELECT 
          id,
          offer_external_id,
          source,
          name,
          image_url,
          last_sold_at,
          times_sold,
          total_quantity_sold,
          created_at,
          updated_at
        FROM commerce.marketplace_products
        WHERE offer_external_id = $1
      `, [sku]);

      if (productResult.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }

      const product = productResult.rows[0];
      
      // Convert local URL to SFTP URL if needed
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      if (product.image_url && !product.image_url.startsWith('http')) {
        const filename = product.image_url.split('/').pop();
        product.image_url = adapter.getUrl(`products/images/${filename}`);
      }

      // Pobierz zamówienia w których sprzedano ten produkt
      const ordersResult = await pool.query(`
        SELECT 
          o.id,
          o.order_number,
          o.source,
          o.source_order_id,
          o.order_date,
          o.payment_status,
          o.payment_amount,
          o.buyer_first_name,
          o.buyer_last_name,
          o.buyer_login,
          oi.quantity,
          oi.price as total_price
        FROM commerce.order_items oi
        INNER JOIN commerce.orders o ON o.id = oi.order_id
        WHERE oi.offer_external_id = $1
        ORDER BY o.order_date DESC
      `, [sku]);

      // Pobierz powiązany produkt katalogowy LUB zestaw (jeśli istnieje)
      const catalogLinkResult = await pool.query(`
        SELECT 
          ppd.id as link_id,
          ppd.link_type,
          p.id as product_id,
          p.sku as product_sku,
          p.title as product_title,
          ps.id as set_id,
          ps.sku as set_sku,
          ps.title as set_title
        FROM catalog.product_platform_data ppd
        LEFT JOIN catalog.products p ON p.id = ppd.product_id AND ppd.link_type = 'product'
        LEFT JOIN product_creator.product_sets ps ON ps.id = ppd.set_id AND ppd.link_type = 'set'
        WHERE ppd.external_id = $1
          AND LOWER(ppd.platform) = LOWER($2)
        LIMIT 1
      `, [sku, product.source]);

      // Build catalog entity based on link_type
      let catalogProduct = null;
      if (catalogLinkResult.rows.length > 0) {
        const row = catalogLinkResult.rows[0];
        if (row.link_type === 'set') {
          catalogProduct = {
            id: row.set_id,
            sku: row.set_sku,
            title: row.set_title,
            linkType: 'set',
            linkId: row.link_id,
          };
        } else {
          catalogProduct = {
            id: row.product_id,
            sku: row.product_sku,
            title: row.product_title,
            linkType: 'product',
            linkId: row.link_id,
          };
        }
      }

      res.json({
        product,
        orders: ordersResult.rows,
        catalogProduct,
      });
    } catch (error) {
      console.error("Get marketplace product details error:", error);
      res.status(500).json({ error: "Failed to fetch product details" });
    }
  });

  // GET /api/products/by-sku/:sku - Pobierz produkt po SKU (external_id) lub product_id
  app.get("/api/products/by-sku/:sku", isAuthenticated, async (req, res) => {
    try {
      const { sku } = req.params;
      
      const result = await pool.query(`
        SELECT 
          p.id,
          p.product_id,
          p.external_id,
          p.name,
          p.description,
          p.image_url,
          p.category,
          p.raw_data,
          p.created_at,
          p.updated_at,
          COUNT(DISTINCT oi.order_id) as orders_count,
          SUM(oi.quantity) as total_quantity_sold
        FROM catalog.products p
        LEFT JOIN commerce.order_items oi ON oi.catalog_product_id = p.id
        WHERE p.external_id = $1 OR p.product_id = $1
        GROUP BY p.id
      `, [sku]);

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }

      const product = result.rows[0];

      // Check for local image
      const extensions = ['.jpg', '.jpeg', '.png', '.webp'];
      let hasLocalImage = false;
      let localImageFilename = null;

      for (const ext of extensions) {
        const filename = `${product.product_id}_1${ext}`;
        const imagePath = path.join(process.cwd(), "attached_assets", "products", filename);
        const exists = await fs.access(imagePath).then(() => true).catch(() => false);
        
        if (exists) {
          hasLocalImage = true;
          localImageFilename = filename;
          break;
        }
      }

      res.json({
        ...product,
        hasLocalImage,
        localImageFilename
      });
    } catch (error) {
      console.error("Get product by SKU error:", error);
      res.status(500).json({ error: "Failed to fetch product" });
    }
  });

  // POST /api/products/:productId/image - Upload obrazka produktu
  app.post("/api/products/:productId/image", isAuthenticated, async (req, res) => {
    try {
      const { productId } = req.params;
      
      // Sanitize productId to prevent path traversal
      if (!productId || !/^[A-Za-z0-9_-]+$/.test(productId)) {
        return res.status(400).json({ error: "Invalid product ID" });
      }

      // Use multer middleware after validation
      uploadProductImage.single('image')(req, res, async (err) => {
        if (err) {
          console.error("Multer upload error:", err);
          return res.status(400).json({ error: err.message });
        }

        if (!req.file) {
          return res.status(400).json({ error: "No image file provided" });
        }

        const filename = req.file.filename;

        res.json({ 
          success: true, 
          filename,
          message: "Image uploaded successfully" 
        });
      });
    } catch (error) {
      console.error("Upload product image error:", error);
      res.status(500).json({ error: "Failed to upload image" });
    }
  });

  // DELETE /api/products/:productId/image - Usunięcie obrazka produktu
  app.delete("/api/products/:productId/image", isAuthenticated, async (req, res) => {
    try {
      const { productId } = req.params;
      
      // Sanitize productId to prevent path traversal
      if (!productId || !/^[A-Za-z0-9_-]+$/.test(productId)) {
        return res.status(400).json({ error: "Invalid product ID" });
      }

      const productsDir = path.join(process.cwd(), "attached_assets", "products");
      
      // Try to delete all possible image formats
      const extensions = ['.jpg', '.jpeg', '.png', '.webp'];
      let deleted = false;

      for (const ext of extensions) {
        const filename = `${productId}_1${ext}`;
        const filepath = path.join(productsDir, filename);
        
        try {
          await fs.unlink(filepath);
          deleted = true;
          console.log(`✅ Deleted image: ${filename}`);
        } catch (err) {
          // File doesn't exist, continue
        }
      }

      if (deleted) {
        res.json({ success: true, message: "Image deleted successfully" });
      } else {
        res.status(404).json({ error: "No image found for this product" });
      }
    } catch (error) {
      console.error("Delete product image error:", error);
      res.status(500).json({ error: "Failed to delete image" });
    }
  });

  // POST /api/products/:productId/fetch-from-platform - Pobierz zdjęcie z platformy (Allegro/Shoper)
  app.post("/api/products/:productId/fetch-from-platform", isAuthenticated, async (req, res) => {
    try {
      const { productId } = req.params;
      
      // Sanitize productId to prevent path traversal
      if (!productId || !/^[A-Za-z0-9_-]+$/.test(productId)) {
        return res.status(400).json({ error: "Invalid product ID" });
      }

      // Get product details from database
      const result = await pool.query(
        `SELECT product_id, external_id, name, image_url
         FROM catalog.products
         WHERE product_id = $1`,
        [productId]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }

      const product = result.rows[0];

      // Determine platform and download image
      let filename: string | null = null;

      if (product.external_id && product.external_id.startsWith('shoper_')) {
        // Shoper product
        if (!product.image_url) {
          return res.status(400).json({ error: "No image URL available for this Shoper product" });
        }

        const { downloadShoperProductImage } = await import('./shoper-api.js');
        const shoperId = product.external_id.replace('shoper_', '');
        filename = await downloadShoperProductImage(product.image_url, shoperId);
      } else {
        // Allegro product
        const allegroConfig = await pool.query('SELECT access_token FROM allegro_connections LIMIT 1');
        const userToken = allegroConfig.rows[0]?.access_token;

        if (!userToken) {
          return res.status(400).json({ error: "Allegro token not available. Please authorize Allegro first." });
        }

        const { downloadProductImage } = await import('./allegro-api.js');
        filename = await downloadProductImage(product.name, product.product_id, userToken);
      }

      if (filename) {
        res.json({ 
          success: true, 
          filename,
          message: "Image downloaded successfully from platform" 
        });
      } else {
        res.status(500).json({ error: "Failed to download image from platform" });
      }
    } catch (error) {
      console.error("Fetch product image from platform error:", error);
      res.status(500).json({ error: "Failed to fetch image from platform" });
    }
  });

  // POST /api/products/bulk-fetch-images - Masowe pobieranie zdjęć z platform
  app.post("/api/products/bulk-fetch-images", isAuthenticated, async (req, res) => {
    try {
      // Get all products without local images
      const result = await pool.query(
        `SELECT product_id, external_id, name, image_url
         FROM catalog.products
         WHERE image_url IS NOT NULL
         ORDER BY product_id`
      );

      const products = result.rows;
      const productsDir = path.join(process.cwd(), "attached_assets", "products");
      const extensions = ['.jpg', '.jpeg', '.png', '.webp'];
      
      // Filter products that don't have local images
      const productsToFetch = [];
      for (const product of products) {
        let hasLocalImage = false;
        
        for (const ext of extensions) {
          const filepath = path.join(productsDir, `${product.product_id}_1${ext}`);
          if (await fs.access(filepath).then(() => true).catch(() => false)) {
            hasLocalImage = true;
            break;
          }
        }
        
        if (!hasLocalImage) {
          productsToFetch.push(product);
        }
      }

      if (productsToFetch.length === 0) {
        return res.json({ 
          success: true, 
          message: "All products already have local images",
          total: 0,
          downloaded: 0,
          failed: 0
        });
      }

      // Get Allegro token once
      const allegroConfig = await pool.query('SELECT access_token FROM allegro_connections LIMIT 1');
      const allegroToken = allegroConfig.rows[0]?.access_token;

      const results = {
        total: productsToFetch.length,
        downloaded: 0,
        failed: 0,
        errors: [] as Array<{ productId: string, error: string }>
      };

      // Download images
      for (const product of productsToFetch) {
        try {
          let filename: string | null = null;

          if (product.external_id && product.external_id.startsWith('shoper_')) {
            const { downloadShoperProductImage } = await import('./shoper-api.js');
            const shoperId = product.external_id.replace('shoper_', '');
            filename = await downloadShoperProductImage(product.image_url, shoperId);
          } else {
            if (!allegroToken) {
              results.errors.push({ 
                productId: product.product_id, 
                error: "Allegro token not available" 
              });
              results.failed++;
              continue;
            }

            const { downloadProductImage } = await import('./allegro-api.js');
            filename = await downloadProductImage(product.name, product.product_id, allegroToken);
          }

          if (filename) {
            results.downloaded++;
            console.log(`✅ Downloaded image for product ${product.product_id}`);
          } else {
            results.failed++;
            results.errors.push({ 
              productId: product.product_id, 
              error: "Failed to download image" 
            });
          }
        } catch (error: any) {
          results.failed++;
          results.errors.push({ 
            productId: product.product_id, 
            error: error.message || "Unknown error" 
          });
          console.error(`❌ Failed to download image for product ${product.product_id}:`, error);
        }
      }

      res.json({
        success: true,
        message: `Downloaded ${results.downloaded} of ${results.total} images`,
        ...results
      });
    } catch (error) {
      console.error("Bulk fetch images error:", error);
      res.status(500).json({ error: "Failed to fetch images" });
    }
  });

  // ============================================================================
  // API TOKENS MANAGEMENT (for admins)
  // ============================================================================
  
  // GET /api/api-tokens - List all API tokens (admin only)
  app.get("/api/api-tokens", requireAdmin, async (req, res) => {
    try {
      const result = await pool.query(
        `SELECT 
          id,
          name,
          description,
          token_prefix,
          is_active,
          expires_at,
          last_used_at,
          created_by,
          created_at
        FROM api_tokens
        ORDER BY created_at DESC`
      );

      res.json(result.rows);
    } catch (error) {
      console.error("Get API tokens error:", error);
      res.status(500).json({ error: "Failed to fetch API tokens" });
    }
  });

  // POST /api/api-tokens - Create new API token (admin only)
  app.post("/api/api-tokens", requireAdmin, async (req, res) => {
    try {
      const { name, description, expiresInDays } = req.body;
      
      if (!name) {
        return res.status(400).json({ error: "Token name is required" });
      }

      // Generate token
      const token = generateApiToken();
      const hashedToken = hashToken(token);
      const tokenPrefix = token.substring(0, 8);

      // Calculate expiry date
      let expiresAt = null;
      if (expiresInDays && expiresInDays > 0) {
        expiresAt = new Date();
        expiresAt.setDate(expiresAt.getDate() + parseInt(expiresInDays, 10));
      }

      const userId = (req.user as any)?.id;

      const result = await pool.query(
        `INSERT INTO api_tokens (
          name, 
          description, 
          token, 
          token_prefix, 
          is_active, 
          expires_at, 
          created_by,
          created_at
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        RETURNING id, name, description, token_prefix, is_active, expires_at, created_at`,
        [name, description, hashedToken, tokenPrefix, true, expiresAt, userId, new Date()]
      );

      // Return token ONLY once during creation
      res.json({
        ...result.rows[0],
        token, // Full token returned only on creation
      });
    } catch (error) {
      console.error("Create API token error:", error);
      res.status(500).json({ error: "Failed to create API token" });
    }
  });

  // PUT /api/api-tokens/:id - Update API token (activate/deactivate)
  app.put("/api/api-tokens/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const { isActive, name, description } = req.body;

      const updates: string[] = [];
      const params: any[] = [];
      let paramIndex = 1;

      if (typeof isActive === 'boolean') {
        updates.push(`is_active = $${paramIndex}`);
        params.push(isActive);
        paramIndex++;
      }

      if (name !== undefined) {
        updates.push(`name = $${paramIndex}`);
        params.push(name);
        paramIndex++;
      }

      if (description !== undefined) {
        updates.push(`description = $${paramIndex}`);
        params.push(description);
        paramIndex++;
      }

      if (updates.length === 0) {
        return res.status(400).json({ error: "No updates provided" });
      }

      params.push(id);
      const result = await pool.query(
        `UPDATE api_tokens 
         SET ${updates.join(', ')}
         WHERE id = $${paramIndex}
         RETURNING id, name, description, token_prefix, is_active, expires_at, last_used_at, created_at`,
        params
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "API token not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Update API token error:", error);
      res.status(500).json({ error: "Failed to update API token" });
    }
  });

  // DELETE /api/api-tokens/:id - Delete API token
  app.delete("/api/api-tokens/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;

      const result = await pool.query(
        'DELETE FROM api_tokens WHERE id = $1 RETURNING id',
        [id]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "API token not found" });
      }

      res.json({ success: true, message: "API token deleted successfully" });
    } catch (error) {
      console.error("Delete API token error:", error);
      res.status(500).json({ error: "Failed to delete API token" });
    }
  });

  // GET /api/api-tokens/logs - Get API request logs (admin only)
  app.get("/api/api-tokens/logs", requireAdmin, async (req, res) => {
    try {
      const limit = parseInt(req.query.limit as string, 10) || 100;
      const tokenId = req.query.token_id as string;

      let query = `
        SELECT 
          l.id,
          l.token_id,
          l.method,
          l.path,
          l.status_code,
          l.response_time,
          l.ip_address,
          l.user_agent,
          l.created_at,
          t.name as token_name
        FROM api_request_logs l
        LEFT JOIN api_tokens t ON l.token_id = t.id
        ${tokenId ? 'WHERE l.token_id = $1' : ''}
        ORDER BY l.created_at DESC
        LIMIT ${tokenId ? '$2' : '$1'}
      `;

      const params = tokenId ? [tokenId, limit] : [limit];
      const result = await pool.query(query, params);

      res.json(result.rows);
    } catch (error) {
      console.error("Get API logs error:", error);
      res.status(500).json({ error: "Failed to fetch API logs" });
    }
  });

  // ============================================================================
  // WEBHOOKS MANAGEMENT (for admins)
  // ============================================================================
  
  // GET /api/webhooks - List all webhooks (admin only)
  app.get("/api/webhooks", requireAdmin, async (req, res) => {
    try {
      const result = await pool.query(
        `SELECT 
          id,
          name,
          url,
          events,
          is_active,
          retry_attempts,
          last_triggered_at,
          created_by,
          created_at,
          updated_at
        FROM webhook_configs
        ORDER BY created_at DESC`
      );

      res.json(result.rows);
    } catch (error) {
      console.error("Get webhooks error:", error);
      res.status(500).json({ error: "Failed to fetch webhooks" });
    }
  });

  // POST /api/webhooks - Create new webhook (admin only)
  app.post("/api/webhooks", requireAdmin, async (req, res) => {
    try {
      const { name, url, secret, events, retryAttempts } = req.body;
      
      if (!name || !url || !events) {
        return res.status(400).json({ error: "Name, URL, and events are required" });
      }

      const userId = (req.user as any)?.id;
      const webhookSecret = secret || randomBytes(32).toString('hex');

      const result = await pool.query(
        `INSERT INTO webhook_configs (
          name, 
          url, 
          secret, 
          events, 
          is_active, 
          retry_attempts,
          created_by,
          created_at
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        RETURNING id, name, url, events, is_active, retry_attempts, created_at`,
        [name, url, webhookSecret, JSON.stringify(events), true, retryAttempts || 3, userId, new Date()]
      );

      res.json({
        ...result.rows[0],
        secret: webhookSecret, // Return secret only on creation
      });
    } catch (error) {
      console.error("Create webhook error:", error);
      res.status(500).json({ error: "Failed to create webhook" });
    }
  });

  // PUT /api/webhooks/:id - Update webhook (admin only)
  app.put("/api/webhooks/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const { isActive, name, url, events, retryAttempts } = req.body;

      const updates: string[] = [];
      const params: any[] = [];
      let paramIndex = 1;

      if (typeof isActive === 'boolean') {
        updates.push(`is_active = $${paramIndex}`);
        params.push(isActive);
        paramIndex++;
      }

      if (name !== undefined) {
        updates.push(`name = $${paramIndex}`);
        params.push(name);
        paramIndex++;
      }

      if (url !== undefined) {
        updates.push(`url = $${paramIndex}`);
        params.push(url);
        paramIndex++;
      }

      if (events !== undefined) {
        updates.push(`events = $${paramIndex}`);
        params.push(JSON.stringify(events));
        paramIndex++;
      }

      if (retryAttempts !== undefined) {
        updates.push(`retry_attempts = $${paramIndex}`);
        params.push(retryAttempts);
        paramIndex++;
      }

      if (updates.length === 0) {
        return res.status(400).json({ error: "No updates provided" });
      }

      updates.push(`updated_at = $${paramIndex}`);
      params.push(new Date());
      paramIndex++;

      params.push(id);
      const result = await pool.query(
        `UPDATE webhook_configs 
         SET ${updates.join(', ')}
         WHERE id = $${paramIndex}
         RETURNING id, name, url, events, is_active, retry_attempts, last_triggered_at, created_at, updated_at`,
        params
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Webhook not found" });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Update webhook error:", error);
      res.status(500).json({ error: "Failed to update webhook" });
    }
  });

  // DELETE /api/webhooks/:id - Delete webhook (admin only)
  app.delete("/api/webhooks/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;

      const result = await pool.query(
        'DELETE FROM webhook_configs WHERE id = $1 RETURNING id',
        [id]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Webhook not found" });
      }

      res.json({ success: true, message: "Webhook deleted successfully" });
    } catch (error) {
      console.error("Delete webhook error:", error);
      res.status(500).json({ error: "Failed to delete webhook" });
    }
  });

  // GET /api/webhooks/logs - Get webhook delivery logs (admin only)
  app.get("/api/webhooks/logs", requireAdmin, async (req, res) => {
    try {
      const limit = parseInt(req.query.limit as string, 10) || 100;
      const webhookId = req.query.webhook_id ? parseInt(req.query.webhook_id as string, 10) : undefined;

      const logs = await getWebhookLogs(webhookId, limit);
      res.json(logs);
    } catch (error) {
      console.error("Get webhook logs error:", error);
      res.status(500).json({ error: "Failed to fetch webhook logs" });
    }
  });

  // POST /api/webhooks/:id/test - Send test webhook (admin only)
  app.post("/api/webhooks/:id/test", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const { sendWebhook } = await import('./webhooks.js');

      // Get webhook config to check events
      const webhookResult = await pool.query(
        'SELECT * FROM webhook_configs WHERE id = $1',
        [id]
      );

      if (webhookResult.rows.length === 0) {
        return res.status(404).json({ error: "Webhook not found" });
      }

      const webhook = webhookResult.rows[0];
      const events = webhook.events || [];

      // Select first configured event, or default to order.created
      const testEvent = events[0] || 'order.created';

      // Create test payload based on event type
      let testPayload: any = {
        order_number: 1,
        source: 'ALLEGRO',
        source_order_id: 'test-order-id-123',
        buyer_email: 'test@example.com',
        buyer_first_name: 'Jan',
        buyer_last_name: 'Testowy',
        total_amount: '999.99',
        payment_amount: '999.99',
        currency: 'PLN',
        payment_status: 'PENDING',
        payment_type: 'ONLINE',
        delivery_method: 'DPD Kurier',
        tracking_numbers: [],
        created_at: new Date().toISOString(),
        items: [
          {
            product_name: 'Testowy Produkt',
            quantity: 1,
            unit_price: '999.99',
            total_price: '999.99'
          }
        ]
      };

      // Adjust payload based on event type
      if (testEvent === 'order.paid') {
        testPayload.payment_status = 'PAID';
        testPayload.payment_date = new Date().toISOString();
      } else if (testEvent === 'order.shipped') {
        testPayload.tracking_numbers = ['TEST123456789'];
        testPayload.shipments = [{
          carrier: 'DPD',
          tracking_number: 'TEST123456789',
          shipped_at: new Date().toISOString()
        }];
      }

      // Send test webhook
      const success = await sendWebhook(parseInt(id, 10), testEvent, testPayload);

      if (success) {
        res.json({ 
          success: true, 
          message: `Test webhook wysłany pomyślnie (event: ${testEvent})` 
        });
      } else {
        res.status(500).json({ 
          success: false, 
          error: "Nie udało się wysłać testowego webhooka" 
        });
      }
    } catch (error) {
      console.error("Send test webhook error:", error);
      res.status(500).json({ error: "Failed to send test webhook" });
    }
  });

  // ============================================================================
  // ODOO INTEGRATION ENDPOINTS
  // ============================================================================

  // Get Odoo URL (accessible to all authenticated users)
  app.get("/api/odoo/url", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT url, user_url FROM odoo_config WHERE is_active = true ORDER BY id DESC LIMIT 1
      `);

      const url = result.rows.length > 0 
        ? result.rows[0].url 
        : process.env.ODOO_URL || null;
      
      const userUrl = result.rows.length > 0 
        ? result.rows[0].user_url 
        : null;

      res.json({ url, userUrl });
    } catch (error) {
      console.error("Get Odoo URL error:", error);
      res.status(500).json({ error: "Failed to get Odoo URL" });
    }
  });

  // Test Odoo connection
  app.post("/api/odoo/test-connection", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { createOdooClient } = await import('./odoo-client.js');
      const client = await createOdooClient();
      
      if (!client) {
        return res.status(400).json({ 
          success: false, 
          error: "Odoo credentials not configured" 
        });
      }

      const result = await client.testConnection();
      res.json(result);
    } catch (error) {
      console.error("Test Odoo connection error:", error);
      res.status(500).json({ 
        success: false, 
        error: error instanceof Error ? error.message : "Unknown error" 
      });
    }
  });

  // Get Odoo config
  app.get("/api/odoo/config", requirePermission('manage_settings'), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT * FROM odoo_config WHERE is_active = true ORDER BY id DESC LIMIT 1
      `);

      if (result.rows.length === 0) {
        return res.json({ 
          configured: false,
          url: process.env.ODOO_URL || null,
          database: process.env.ODOO_DATABASE || null,
          username: process.env.ODOO_USERNAME || null
        });
      }

      res.json({ 
        configured: true,
        ...result.rows[0]
      });
    } catch (error) {
      console.error("Get Odoo config error:", error);
      res.status(500).json({ error: "Failed to get Odoo config" });
    }
  });

  // Update Odoo config
  app.post("/api/odoo/config", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { syncIntervalMinutes, isActive, autoConfirmOrders, url, userUrl, omsDomain } = req.body;

      const existingConfig = await pool.query(`
        SELECT * FROM odoo_config WHERE is_active = true ORDER BY id DESC LIMIT 1
      `);

      if (existingConfig.rows.length > 0) {
        await pool.query(`
          UPDATE odoo_config
          SET 
            url = $1,
            user_url = $2,
            oms_domain = $3,
            sync_interval_minutes = $4,
            is_active = $5,
            auto_confirm_orders = $6,
            updated_at = NOW()
          WHERE id = $7
        `, [
          url || existingConfig.rows[0].url,
          userUrl || existingConfig.rows[0].user_url,
          omsDomain || existingConfig.rows[0].oms_domain || 'alp-oms.replit.app',
          syncIntervalMinutes || 3,
          isActive !== false,
          autoConfirmOrders || false,
          existingConfig.rows[0].id
        ]);
      } else {
        await pool.query(`
          INSERT INTO odoo_config (
            url, user_url, oms_domain, database, username, sync_interval_minutes, is_active, auto_confirm_orders
          ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        `, [
          url || process.env.ODOO_URL,
          userUrl || null,
          omsDomain || 'alp-oms.replit.app',
          process.env.ODOO_DATABASE,
          process.env.ODOO_USERNAME,
          syncIntervalMinutes || 3,
          isActive !== false,
          autoConfirmOrders || false
        ]);
      }

      if (isActive !== false) {
        const interval = syncIntervalMinutes || 3;
        startOdooSyncScheduler(interval);
        console.log(`✅ Odoo sync scheduler restarted (${interval} min interval)`);
      } else {
        stopOdooSyncScheduler();
        console.log('⏸️  Odoo sync scheduler stopped');
      }

      res.json({ success: true });
    } catch (error) {
      console.error("Update Odoo config error:", error);
      res.status(500).json({ error: "Failed to update Odoo config" });
    }
  });

  // Get Shoper config
  // Get Shoper custom statuses
  app.get("/api/shoper/statuses", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { getShoperStatuses } = await import('./shoper-api.js');
      const statuses = await getShoperStatuses();
      
      console.log('📋 Shoper statuses:', JSON.stringify(statuses, null, 2));
      res.json({ statuses });
    } catch (error: any) {
      console.error("Get Shoper statuses error:", error);
      res.status(500).json({ error: error.message || "Failed to get Shoper statuses" });
    }
  });

  app.get("/api/shoper/config", requirePermission('manage_settings'), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT * FROM shoper_config ORDER BY id DESC LIMIT 1
      `);

      if (result.rows.length === 0) {
        return res.json({ 
          configured: false,
          ordersUrl: 'https://alpmeb.pl/admin/orders/',
          productsUrl: 'https://sklep378098.shoparena.pl/admin/products/edit/id/'
        });
      }

      res.json({ 
        configured: true,
        ...result.rows[0]
      });
    } catch (error) {
      console.error("Get Shoper config error:", error);
      res.status(500).json({ error: "Failed to get Shoper config" });
    }
  });

  // Update Shoper config
  app.post("/api/shoper/config", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { ordersUrl, productsUrl } = req.body;

      const existingConfig = await pool.query(`
        SELECT * FROM shoper_config ORDER BY id DESC LIMIT 1
      `);

      if (existingConfig.rows.length > 0) {
        await pool.query(`
          UPDATE shoper_config
          SET 
            orders_url = $1,
            products_url = $2,
            updated_at = NOW()
          WHERE id = $3
        `, [
          ordersUrl || existingConfig.rows[0].orders_url,
          productsUrl || existingConfig.rows[0].products_url,
          existingConfig.rows[0].id
        ]);
      } else {
        await pool.query(`
          INSERT INTO shoper_config (orders_url, products_url)
          VALUES ($1, $2)
        `, [
          ordersUrl || 'https://alpmeb.pl/admin/orders/',
          productsUrl || 'https://sklep378098.shoparena.pl/admin/products/edit/id/'
        ]);
      }

      res.json({ success: true });
    } catch (error) {
      console.error("Update Shoper config error:", error);
      res.status(500).json({ error: "Failed to update Shoper config" });
    }
  });

  // Get Odoo sync status
  app.get("/api/odoo/sync/status", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { getOdooSyncStatus } = await import('./odoo-sync.js');
      const status = await getOdooSyncStatus();
      res.json(status);
    } catch (error) {
      console.error("Get Odoo sync status error:", error);
      res.status(500).json({ error: "Failed to get sync status" });
    }
  });

  // Manual Odoo sync
  app.post("/api/odoo/sync/manual", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { processSyncQueue } = await import('./odoo-sync.js');
      const result = await processSyncQueue();
      res.json(result);
    } catch (error) {
      console.error("Manual Odoo sync error:", error);
      res.status(500).json({ error: "Failed to run manual sync" });
    }
  });

  // Get Odoo sync logs
  app.get("/api/odoo/sync/logs", requirePermission('manage_settings'), async (req, res) => {
    try {
      const limit = parseInt(req.query.limit as string, 10) || 100;
      const { getOdooSyncLogs } = await import('./odoo-sync.js');
      const logs = await getOdooSyncLogs(limit);
      res.json(logs);
    } catch (error) {
      console.error("Get Odoo sync logs error:", error);
      res.status(500).json({ error: "Failed to get sync logs" });
    }
  });

  // Get sync health status for all platforms
  app.get("/api/sync/health", requirePermission('manage_settings'), async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          sync_type as platform,
          health_status as status,
          last_sync_at,
          total_synced_24h as total_orders,
          queue_failed as failed_orders,
          last_error,
          updated_at
        FROM sync_health_stats
        ORDER BY sync_type
      `);
      
      // Calculate overall health
      const stats = result.rows;
      const overallStatus = stats.some(s => s.status === 'critical') ? 'critical' :
                           stats.some(s => s.status === 'warning') ? 'warning' : 
                           'healthy';
      
      res.json({
        overallStatus,
        platforms: stats,
        lastUpdate: stats.length > 0 ? stats[0].updated_at : null
      });
    } catch (error) {
      console.error("Get sync health error:", error);
      res.status(500).json({ error: "Failed to get sync health status" });
    }
  });

  // Manually trigger Auto-Healing
  app.post("/api/sync/auto-heal", requirePermission('manage_settings'), async (req, res) => {
    try {
      console.log("Manual Auto-Healing triggered by user");
      const { runAutoHealing } = await import('./auto-healing.js');
      await runAutoHealing();
      res.json({ 
        success: true,
        message: "Auto-healing completed successfully"
      });
    } catch (error) {
      console.error("Manual Auto-Healing error:", error);
      res.status(500).json({ error: "Failed to run Auto-Healing" });
    }
  });

  // Manually trigger Nightly Full Sync
  app.post("/api/sync/nightly-manual", requirePermission('manage_settings'), async (req, res) => {
    try {
      console.log("🌙 Manual nightly sync triggered by user");
      await runNightlyFullSync();
      res.json({ 
        success: true,
        message: "Nightly full sync completed successfully"
      });
    } catch (error: any) {
      console.error("❌ Manual nightly sync error:", error);
      res.status(500).json({ error: "Failed to run nightly sync" });
    }
  });

  // Retry failed Odoo sync
  app.post("/api/odoo/sync/retry/:id", requirePermission('manage_settings'), async (req, res) => {
    try {
      const { id } = req.params;
      const { retryFailedSync, processSyncQueue } = await import('./odoo-sync.js');
      
      await retryFailedSync(parseInt(id, 10));
      await processSyncQueue();
      
      res.json({ success: true });
    } catch (error) {
      console.error("Retry Odoo sync error:", error);
      res.status(500).json({ error: "Failed to retry sync" });
    }
  });

  // ============================================================================
  // EXTERNAL API ENDPOINTS (for Odoo and other integrations)
  // ============================================================================
  
  // GET /api/external/orders - List all orders with pagination and filtering
  app.get("/api/external/orders", requireApiToken, logApiRequest, async (req, res) => {
    try {
      const page = parseInt(req.query.page as string, 10) || 1;
      const limit = parseInt(req.query.limit as string, 10) || 50;
      const offset = (page - 1) * limit;
      
      // Optional filters
      const source = req.query.source as string; // 'allegro' or 'shoper'
      const paymentStatus = req.query.payment_status as string;
      const dateFrom = req.query.date_from as string;
      const dateTo = req.query.date_to as string;

      // Build WHERE clause
      let whereClauses: string[] = [];
      let params: any[] = [];
      let paramIndex = 1;

      if (source) {
        whereClauses.push(`source = $${paramIndex}`);
        params.push(source);
        paramIndex++;
      }

      if (paymentStatus) {
        whereClauses.push(`payment_status = $${paramIndex}`);
        params.push(paymentStatus);
        paramIndex++;
      }

      if (dateFrom) {
        whereClauses.push(`order_date >= $${paramIndex}`);
        params.push(dateFrom);
        paramIndex++;
      }

      if (dateTo) {
        whereClauses.push(`order_date <= $${paramIndex}`);
        params.push(dateTo);
        paramIndex++;
      }

      const whereClause = whereClauses.length > 0 ? `WHERE ${whereClauses.join(' AND ')}` : '';

      // Get total count
      const countResult = await pool.query(
        `SELECT COUNT(*) as total FROM commerce.orders ${whereClause}`,
        params
      );
      const total = parseInt(countResult.rows[0].total, 10);

      // Get orders with pagination
      const ordersResult = await pool.query(
        `SELECT 
          order_number,
          source_order_id,
          source,
          order_date,
          buyer_first_name,
          buyer_last_name,
          buyer_email,
          buyer_phone,
          buyer_login,
          delivery_method,
          delivery_amount,
          delivery_address,
          invoice_required,
          buyer_company_name,
          tax_id,
          billing_address,
          total_to_pay_amount as total_amount,
          total_to_pay_currency as currency,
          payment_status,
          payment_type,
          payment_amount,
          tracking_numbers,
          buyer_notes,
          has_returns,
          refund_amount,
          refund_date,
          created_at,
          updated_at
        FROM commerce.orders 
        ${whereClause}
        ORDER BY order_date DESC
        LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`,
        [...params, limit, offset]
      );

      res.json({
        success: true,
        data: ordersResult.rows,
        pagination: {
          page,
          limit,
          total,
          totalPages: Math.ceil(total / limit),
        },
      });
    } catch (error) {
      console.error("External API - Get orders error:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch orders" 
      });
    }
  });

  // GET /api/external/orders/:id - Get order details by order_number (123)
  app.get("/api/external/orders/:orderNumber", requireApiToken, logApiRequest, async (req, res) => {
    try {
      const { orderNumber } = req.params;

      const result = await pool.query(
        `SELECT 
          order_number,
          source_order_id,
          source,
          order_date,
          buyer_first_name,
          buyer_last_name,
          buyer_email,
          buyer_phone,
          buyer_login,
          delivery_method,
          delivery_amount,
          delivery_address,
          invoice_required,
          buyer_company_name,
          tax_id,
          billing_address,
          total_to_pay_amount as total_amount,
          total_to_pay_currency as currency,
          payment_status,
          payment_type,
          payment_amount,
          tracking_numbers,
          buyer_notes,
          shipments,
          has_returns,
          refund_amount,
          refund_date,
          created_at,
          updated_at
        FROM commerce.orders 
        WHERE order_number = $1`,
        [parseInt(orderNumber, 10)]
      );

      if (result.rows.length === 0) {
        return res.status(404).json({ 
          success: false, 
          error: "Order not found" 
        });
      }

      res.json({
        success: true,
        data: result.rows[0],
      });
    } catch (error) {
      console.error("External API - Get order details error:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch order details" 
      });
    }
  });

  // GET /api/external/products - Get unique products list
  app.get("/api/external/products", requireApiToken, logApiRequest, async (req, res) => {
    try {
      const result = await pool.query(
        `SELECT 
          id,
          product_id,
          external_id,
          name,
          description,
          image_url,
          source,
          category,
          created_at,
          updated_at
        FROM catalog.products
        ORDER BY name ASC`
      );

      res.json({
        success: true,
        data: result.rows,
        total: result.rows.length,
      });
    } catch (error) {
      console.error("External API - Get products error:", error);
      res.status(500).json({ 
        success: false, 
        error: "Failed to fetch products" 
      });
    }
  });

  // Skip database connections if requested
  let connection = null;
  let settings = null;
  
  if (!skipDatabase) {
    connection = await getAllegroConnectionFromPostgres();
    settings = await getSyncSettingsFromPostgres();
  }
  
  // Sprawdź czy auto-sync jest włączony dla tego środowiska
  // ENABLE_AUTO_SYNC: domyślnie false w dev, true w production/deploy
  const isProduction = process.env.REPLIT_DEPLOYMENT === '1' || process.env.NODE_ENV === 'production';
  const enableAutoSync = process.env.ENABLE_AUTO_SYNC === 'true' || isProduction;
  
  console.log('🔍 Auto-sync environment check:', {
    isProduction,
    ENABLE_AUTO_SYNC: process.env.ENABLE_AUTO_SYNC,
    enableAutoSync,
    REPLIT_DEPLOYMENT: process.env.REPLIT_DEPLOYMENT
  });
  
  console.log('🔍 Allegro auto-sync check:', {
    hasConnection: !!connection,
    isActive: connection?.isActive,
    autoRefreshEnabled: settings?.autoRefreshEnabled,
    interval: settings?.refreshIntervalMinutes,
    environmentAllowsSync: enableAutoSync
  });
  
  if (enableAutoSync && connection?.isActive && settings?.autoRefreshEnabled) {
    const interval = parseInt(settings.refreshIntervalMinutes || "3", 10);
    console.log(`✅ Allegro auto-sync initialized (${interval} min interval)`);
    startSyncScheduler(interval);
  } else {
    console.log('⚠️  Allegro auto-sync NOT started:', {
      reason: !enableAutoSync ? 'Environment does not allow auto-sync (set ENABLE_AUTO_SYNC=true to override)' :
              !connection ? 'No connection' : 
              !connection.isActive ? 'Connection not active' :
              !settings?.autoRefreshEnabled ? 'Auto-refresh disabled' : 'Unknown'
    });
  }

  if (enableAutoSync && settings?.shoperAutoRefreshEnabled) {
    const interval = parseInt(settings.shoperRefreshIntervalMinutes || "5", 10);
    startShoperSyncScheduler(interval);
    console.log("✅ Shoper auto-sync initialized");
  } else if (!enableAutoSync && settings?.shoperAutoRefreshEnabled) {
    console.log('⚠️  Shoper auto-sync NOT started: Environment does not allow auto-sync');
  }

  // Initialize Odoo sync scheduler
  if (enableAutoSync) {
    try {
      const odooConfigResult = await pool.query(`
        SELECT * FROM odoo_config WHERE is_active = true ORDER BY id DESC LIMIT 1
      `);
      
      if (odooConfigResult.rows.length > 0) {
        const odooConfig = odooConfigResult.rows[0];
        const interval = odooConfig.sync_interval_minutes || 3;
        startOdooSyncScheduler(interval);
        console.log(`✅ Odoo sync scheduler initialized (${interval} min interval)`);
      }
    } catch (error) {
      console.log('⚠️  Odoo config not found, scheduler not started');
    }
  } else {
    console.log('⚠️  Odoo auto-sync NOT started: Environment does not allow auto-sync');
  }

  // Initialize Allegro fees sync scheduler
  if (enableAutoSync && connection?.isActive) {
    const feesInterval = 1440;
    startFeesSyncScheduler(feesInterval);
    console.log(`✅ Allegro fees sync scheduler initialized (${feesInterval} min interval = daily)`);
  }

  // Initialize error logs cleanup scheduler (only if database is available)
  if (!skipDatabase) {
    startErrorLogsCleanupScheduler();
    console.log("✅ Error logs cleanup scheduler initialized (runs daily)");
  } else {
    console.log('⚠️  Error logs cleanup NOT started: Database disabled');
  }

  // Initialize Auto-Healing scheduler (only if database is available)
  if (!skipDatabase && enableAutoSync) {
    startAutoHealingScheduler();
    console.log("✅ Auto-Healing scheduler initialized (runs every 15 min)");
  } else if (skipDatabase) {
    console.log('⚠️  Auto-Healing NOT started: Database disabled');
  } else {
    console.log('⚠️  Auto-Healing NOT started: Environment does not allow auto-sync');
  }

  // Initialize Nightly Full Sync scheduler (runs once daily at configured hour)
  if (enableAutoSync) {
    startNightlySyncScheduler();
    console.log("✅ Nightly full sync scheduler initialized (checks hourly for configured time)");
  }

  // Product Catalog API Endpoints (centralna baza produktów do eksportu)
  
  // Product Images API
  app.get("/api/product-images", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          id,
          url,
          alt_text AS "altText",
          image_type AS "imageType",
          filename
        FROM catalog.product_images
        ORDER BY created_at DESC
      `);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching product images:", error);
      res.status(500).json({ error: "Failed to fetch product images" });
    }
  });
  
  // AI Prompt Settings API
  app.get("/api/ai-settings", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          id,
          system_instructions AS "systemInstructions",
          product_material_info AS "productMaterialInfo",
          edge_band_info AS "edgeBandInfo",
          include_product_table AS "includeProductTable",
          product_table_header AS "productTableHeader",
          top_banner_image_id AS "topBannerImageId",
          bottom_banner_image_id AS "bottomBannerImageId",
          custom_instructions AS "customInstructions",
          prompt_intro AS "promptIntro",
          prompt_features AS "promptFeatures",
          prompt_safety AS "promptSafety",
          prompt_care AS "promptCare",
          prompt_warranty AS "promptWarranty",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
        FROM ai_prompt_settings
        LIMIT 1
      `);
      
      if (result.rows.length === 0) {
        // Create default settings if none exist
        const insertResult = await pool.query(`
          INSERT INTO ai_prompt_settings (system_instructions, product_material_info, edge_band_info, custom_instructions)
          VALUES ($1, $2, $3, $4)
          RETURNING 
            id,
            system_instructions AS "systemInstructions",
            product_material_info AS "productMaterialInfo",
            edge_band_info AS "edgeBandInfo",
            include_product_table AS "includeProductTable",
            product_table_header AS "productTableHeader",
            top_banner_image_id AS "topBannerImageId",
            bottom_banner_image_id AS "bottomBannerImageId",
            custom_instructions AS "customInstructions",
            prompt_intro AS "promptIntro",
            prompt_features AS "promptFeatures",
            prompt_safety AS "promptSafety",
            prompt_care AS "promptCare",
            prompt_warranty AS "promptWarranty",
            created_at AS "createdAt",
            updated_at AS "updatedAt"
        `, [
          'Jesteś ekspertem w tworzeniu profesjonalnych opisów produktów meblarskich.',
          'płyta meblowa 18mm renomowanych polskich firm',
          'obrzeże ABS 0.8mm',
          'Pamiętaj: DŁUGOŚĆ to wymiar poziomy, SZEROKOŚĆ to wymiar głębokości, WYSOKOŚĆ to wymiar pionowy.'
        ]);
        return res.json(insertResult.rows[0]);
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error fetching AI settings:", error);
      res.status(500).json({ error: "Failed to fetch AI settings" });
    }
  });

  app.put("/api/ai-settings", isAuthenticated, async (req, res) => {
    try {
      const {
        systemInstructions,
        productMaterialInfo,
        edgeBandInfo,
        includeProductTable,
        productTableHeader,
        topBannerImageId,
        bottomBannerImageId,
        customInstructions,
        promptIntro,
        promptFeatures,
        promptSafety,
        promptCare,
        promptWarranty,
      } = req.body;

      // Validate and coerce types
      const validatedData = {
        systemInstructions: String(systemInstructions || ''),
        productMaterialInfo: String(productMaterialInfo || ''),
        edgeBandInfo: String(edgeBandInfo || ''),
        includeProductTable: Boolean(includeProductTable),
        productTableHeader: String(productTableHeader || 'Wymiary produktu'),
        topBannerImageId: topBannerImageId ? parseInt(String(topBannerImageId), 10) : null,
        bottomBannerImageId: bottomBannerImageId ? parseInt(String(bottomBannerImageId), 10) : null,
        customInstructions: String(customInstructions || ''),
        promptIntro: String(promptIntro || ''),
        promptFeatures: String(promptFeatures || ''),
        promptSafety: String(promptSafety || ''),
        promptCare: String(promptCare || ''),
        promptWarranty: String(promptWarranty || ''),
      };

      // Get current settings ID (or create if doesn't exist)
      const currentResult = await pool.query(`SELECT id FROM ai_prompt_settings LIMIT 1`);
      
      let result;
      if (currentResult.rows.length === 0) {
        // Insert new
        result = await pool.query(`
          INSERT INTO ai_prompt_settings (
            system_instructions, product_material_info, edge_band_info,
            include_product_table, product_table_header,
            top_banner_image_id, bottom_banner_image_id, custom_instructions,
            prompt_intro, prompt_features, prompt_safety, prompt_care, prompt_warranty
          )
          VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
          RETURNING 
            id,
            system_instructions AS "systemInstructions",
            product_material_info AS "productMaterialInfo",
            edge_band_info AS "edgeBandInfo",
            include_product_table AS "includeProductTable",
            product_table_header AS "productTableHeader",
            top_banner_image_id AS "topBannerImageId",
            bottom_banner_image_id AS "bottomBannerImageId",
            custom_instructions AS "customInstructions",
            prompt_intro AS "promptIntro",
            prompt_features AS "promptFeatures",
            prompt_safety AS "promptSafety",
            prompt_care AS "promptCare",
            prompt_warranty AS "promptWarranty",
            created_at AS "createdAt",
            updated_at AS "updatedAt"
        `, [
          validatedData.systemInstructions,
          validatedData.productMaterialInfo,
          validatedData.edgeBandInfo,
          validatedData.includeProductTable,
          validatedData.productTableHeader,
          validatedData.topBannerImageId,
          validatedData.bottomBannerImageId,
          validatedData.customInstructions,
          validatedData.promptIntro,
          validatedData.promptFeatures,
          validatedData.promptSafety,
          validatedData.promptCare,
          validatedData.promptWarranty
        ]);
      } else {
        // Update existing
        const id = currentResult.rows[0].id;
        result = await pool.query(`
          UPDATE ai_prompt_settings
          SET
            system_instructions = $1,
            product_material_info = $2,
            edge_band_info = $3,
            include_product_table = $4,
            product_table_header = $5,
            top_banner_image_id = $6,
            bottom_banner_image_id = $7,
            custom_instructions = $8,
            prompt_intro = $9,
            prompt_features = $10,
            prompt_safety = $11,
            prompt_care = $12,
            prompt_warranty = $13,
            updated_at = NOW()
          WHERE id = $14
          RETURNING 
            id,
            system_instructions AS "systemInstructions",
            product_material_info AS "productMaterialInfo",
            edge_band_info AS "edgeBandInfo",
            include_product_table AS "includeProductTable",
            product_table_header AS "productTableHeader",
            top_banner_image_id AS "topBannerImageId",
            bottom_banner_image_id AS "bottomBannerImageId",
            custom_instructions AS "customInstructions",
            prompt_intro AS "promptIntro",
            prompt_features AS "promptFeatures",
            prompt_safety AS "promptSafety",
            prompt_care AS "promptCare",
            prompt_warranty AS "promptWarranty",
            created_at AS "createdAt",
            updated_at AS "updatedAt"
        `, [
          validatedData.systemInstructions,
          validatedData.productMaterialInfo,
          validatedData.edgeBandInfo,
          validatedData.includeProductTable,
          validatedData.productTableHeader,
          validatedData.topBannerImageId,
          validatedData.bottomBannerImageId,
          validatedData.customInstructions,
          validatedData.promptIntro,
          validatedData.promptFeatures,
          validatedData.promptSafety,
          validatedData.promptCare,
          validatedData.promptWarranty,
          id
        ]);
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating AI settings:", error);
      res.status(500).json({ error: "Failed to update AI settings" });
    }
  });

  // Generate product description using AI
  app.post("/api/ai/generate-description", isAuthenticated, async (req, res) => {
    try {
      const { productData } = req.body;

      if (!productData) {
        return res.status(400).json({ error: "Product data is required" });
      }

      // Import AI service
      const { generateProductDescription } = await import('./ai-service');

      // Generate description using AI service
      const result = await generateProductDescription({
        productData: {
          title: productData.title || productData.name,
          color: productData.color,
          material: productData.material,
          dimensions: {
            length: productData.length ? parseFloat(productData.length) : undefined,
            width: productData.width ? parseFloat(productData.width) : undefined,
            height: productData.height ? parseFloat(productData.height) : undefined,
          },
          price: productData.price ? parseFloat(productData.price) : undefined,
          shortDescription: productData.shortDescription,
        },
        tone: 'professional',
        language: 'pl',
      });

      res.json({
        descriptionHtml: result.description,
        tokens: {
          prompt: result.promptTokens,
          completion: result.completionTokens,
          total: result.totalTokens,
        },
        model: result.model,
        cost: result.cost,
      });
    } catch (error) {
      console.error("❌ Error generating AI description:", error);
      res.status(500).json({ error: "Failed to generate description" });
    }
  });
  
  // Lista produktów z paginacją i sortowaniem
  app.get("/api/catalog-products", isAuthenticated, async (req, res) => {
    try {
      const idsParam = req.query.ids as string | undefined;
      
      // Bulk fetch mode: GET /api/catalog-products?ids=1,2,3
      if (idsParam) {
        const ids = idsParam.split(',').map(id => parseInt(id.trim())).filter(id => !isNaN(id));
        
        if (ids.length === 0) {
          return res.json({});
        }
        
        const result = await pool.query(`
          SELECT 
            p.id,
            p.sku,
            p.title,
            p.color
          FROM catalog.products p
          WHERE p.id = ANY($1)
        `, [ids]);
        
        const productsMap: Record<number, { id: number; sku: string; title: string; color: string }> = {};
        result.rows.forEach((row: any) => {
          productsMap[row.id] = {
            id: row.id,
            sku: row.sku,
            title: row.title,
            color: row.color,
          };
        });
        
        return res.json(productsMap);
      }
      
      const page = parseInt(req.query.page as string, 10) || 1;
      const limit = parseInt(req.query.limit as string, 10) || 25;
      const sortBy = (req.query.sortBy as string) || 'created_at';
      const sortOrder = (req.query.sortOrder as string) || 'DESC';
      const search = req.query.search as string | undefined;
      const filterBomStatus = req.query.filterBomStatus as string | undefined;
      const componentType = req.query.componentType as string | undefined;
      
      // DEBUG: Log search parameter to see if semicolon arrives
      if (search) {
        console.log('🔍 [CATALOG-PRODUCTS] Search parameter:', JSON.stringify(search), {
          hasSemicolon: search.includes(';'),
          hasComma: search.includes(','),
          length: search.length
        });
      }
      
      const offset = (page - 1) * limit;
      
      // Map frontend column names to database column names - whitelist only
      const columnMap: Record<string, string> = {
        'sku': 'p.sku',
        'title': 'p.title',
        'color': 'p.color',
        'material': 'p.material',
        'productType': 'p.product_type',
        'productGroup': 'p.product_group',
        'basePrice': 'p.base_price',
        'isActive': 'p.is_active',
        'length': 'p.length',
        'width': 'p.width',
        'height': 'p.height',
        'doors': 'p.doors',
        'legs': 'p.legs',
        'created_at': 'p.created_at',
        'createdAt': 'p.created_at',
      };
      
      const dbColumn = columnMap[sortBy] || 'p.created_at';
      const validOrder = sortOrder.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';
      
      // Build search WHERE clause (semicolon = AND, comma = OR)
      const searchParams: string[] = [];
      const searchConditions: string[] = [];
      let paramIndex = 1;
      
      // Add componentType filter if provided
      if (componentType && componentType.trim()) {
        searchConditions.push(`p.product_type = $${paramIndex}`);
        searchParams.push(componentType.trim());
        paramIndex++;
      }
      
      if (search && search.trim()) {
        // Split by semicolon for AND groups
        const andTokens = search.split(';').map(t => t.trim()).filter(Boolean);
        
        andTokens.forEach(andToken => {
          // Split by comma for OR within each AND group
          const orTokens = andToken.split(',').map(t => t.trim()).filter(Boolean);
          
          if (orTokens.length === 1) {
            // Single term - search across all fields with OR
            const term = orTokens[0];
            searchConditions.push(`(
              p.title ILIKE $${paramIndex} OR
              p.sku ILIKE $${paramIndex} OR
              p.product_type ILIKE $${paramIndex} OR
              p.color ILIKE $${paramIndex} OR
              EXISTS (SELECT 1 FROM unnest(p.color_options) opt WHERE opt ILIKE $${paramIndex})
            )`);
            searchParams.push(`%${term}%`);
            paramIndex++;
          } else {
            // Multiple OR terms - each term searches across all fields
            const orConditions = orTokens.map(term => {
              const condition = `(
                p.title ILIKE $${paramIndex} OR
                p.sku ILIKE $${paramIndex} OR
                p.product_type ILIKE $${paramIndex} OR
                p.color ILIKE $${paramIndex} OR
                EXISTS (SELECT 1 FROM unnest(p.color_options) opt WHERE opt ILIKE $${paramIndex})
              )`;
              searchParams.push(`%${term}%`);
              paramIndex++;
              return condition;
            });
            searchConditions.push(`(${orConditions.join(' OR ')})`);
          }
        });
      }
      
      const whereClause = searchConditions.length > 0 
        ? `WHERE ${searchConditions.join(' AND ')}` 
        : '';
      
      // Build HAVING clause for BOM filter
      let havingClause = '';
      if (filterBomStatus === 'has_bom') {
        havingClause = 'HAVING COUNT(DISTINCT pc.id) > 0';
      } else if (filterBomStatus === 'no_bom') {
        havingClause = 'HAVING COUNT(DISTINCT pc.id) = 0';
      }
      
      // Get total count with search filter and BOM filter
      const countResult = await pool.query(`
        SELECT COUNT(*) as total
        FROM (
          SELECT p.id
          FROM catalog.products p
          LEFT JOIN bom.product_boms pb ON p.id = pb.product_id
          LEFT JOIN bom.product_components pc ON pb.id = pc.product_bom_id
          ${whereClause}
          GROUP BY p.id
          ${havingClause}
        ) as filtered_products
      `, searchParams);
      const total = parseInt(countResult.rows[0].total, 10);
      
      // Build ORDER BY clause safely (dbColumn is whitelisted, validOrder is validated)
      const orderByClause = `ORDER BY ${dbColumn} ${validOrder}`;
      
      // Prepare params for SELECT query (search params + limit + offset)
      const selectParams = [...searchParams, limit, offset];
      const limitParamIndex = searchParams.length + 1;
      const offsetParamIndex = searchParams.length + 2;
      
      // Get paginated results with search filter
      const result = await pool.query(`
        SELECT 
          p.id,
          p.sku,
          p.sku_airtable AS "skuAirtable",
          p.title,
          p.short_description AS "shortDescription",
          p.long_description_html AS "longDescriptionHtml",
          p.long_description_doc AS "longDescriptionDoc",
          p.length,
          p.width,
          p.height,
          p.weight,
          p.color,
          p.color_options AS "colorOptions",
          p.material,
          p.product_type AS "productType",
          p.product_group AS "productGroup",
          p.doors,
          p.legs,
          p.base_price AS "basePrice",
          p.currency,
          p.is_active AS "isActive",
          p.odoo_product_id AS "odooProductId",
          p.odoo_exported_at AS "odooExportedAt",
          p.odoo_export_error AS "odooExportError",
          p.created_at AS "createdAt",
          p.updated_at AS "updatedAt",
          COUNT(DISTINCT pi.id) AS "imagesCount",
          COUNT(DISTINCT paa.id) AS "addonsCount",
          COALESCE(COUNT(DISTINCT pc.id), 0) AS "bomCount",
          MAX(CASE WHEN pi.is_primary = true THEN pi.url END) AS "primaryImageUrl",
          MAX(CASE WHEN pi.is_primary = true THEN pi.thumbnail_url END) AS "primaryImageThumbnailUrl",
          COUNT(DISTINCT CASE WHEN mp.id IS NOT NULL THEN ppd.id END) AS "marketplaceLinksCount",
          MIN(mp.offer_external_id) AS "firstMarketplaceSku",
          MAX(pp.id) AS "packedProductId",
          MAX(pp.quantity) AS "stockQuantity",
          MAX(pp.reserved_quantity) AS "reservedQuantity"
        FROM catalog.products p
        LEFT JOIN catalog.product_images pi ON p.id = pi.product_id
        LEFT JOIN catalog.product_addon_assignments paa ON p.id = paa.product_id
        LEFT JOIN bom.product_boms pb ON p.id = pb.product_id
        LEFT JOIN bom.product_components pc ON pb.id = pc.product_bom_id
        LEFT JOIN catalog.product_platform_data ppd ON p.id = ppd.product_id
        LEFT JOIN commerce.marketplace_products mp 
          ON LOWER(ppd.platform) = LOWER(mp.source) 
          AND ppd.external_id = mp.offer_external_id
        LEFT JOIN warehouse.packed_products pp ON p.id = pp.catalog_product_id AND pp.is_active = true AND pp.is_archived = false
        ${whereClause}
        GROUP BY p.id
        ${havingClause}
        ${orderByClause}
        LIMIT $${limitParamIndex} OFFSET $${offsetParamIndex}
      `, selectParams);
      
      // Convert local URLs to SFTP URLs if needed
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      const products = result.rows.map((product: any) => {
        if (product.primaryImageUrl && !product.primaryImageUrl.startsWith('http')) {
          const filename = product.primaryImageUrl.split('/').pop();
          product.primaryImageUrl = adapter.getUrl(`products/images/${filename}`);
        }
        // Parse count fields as numbers
        product.imagesCount = parseInt(product.imagesCount, 10) || 0;
        product.addonsCount = parseInt(product.addonsCount, 10) || 0;
        product.bomCount = parseInt(product.bomCount, 10) || 0;
        product.marketplaceLinksCount = parseInt(product.marketplaceLinksCount, 10) || 0;
        // Parse stock quantities as numbers
        if (product.stockQuantity !== null) {
          product.stockQuantity = parseInt(product.stockQuantity, 10) || 0;
        }
        if (product.reservedQuantity !== null) {
          product.reservedQuantity = parseInt(product.reservedQuantity, 10) || 0;
        }
        return product;
      });
      
      res.json({
        products,
        pagination: {
          page,
          limit,
          total,
          totalPages: Math.ceil(total / limit),
        },
      });
    } catch (error) {
      console.error("❌ Error fetching catalog products:", error);
      res.status(500).json({ error: "Failed to fetch catalog products" });
    }
  });

  // Szczegóły produktu
  app.get("/api/catalog-products/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Produkt z danymi platform (z aliasami camelCase)
      const productResult = await pool.query(`
        SELECT 
          p.id,
          p.sku,
          p.title,
          p.short_description AS "shortDescription",
          p.long_description_html AS "longDescriptionHtml",
          p.long_description_doc AS "longDescriptionDoc",
          p.length,
          p.width,
          p.height,
          p.weight,
          p.color,
          p.material,
          p.product_type AS "productType",
          p.product_group AS "productGroup",
          p.doors,
          p.legs,
          p.base_price AS "basePrice",
          p.currency,
          p.is_active AS "isActive",
          p.matrix_id AS "matrixId",
          p.matrix_variant_id AS "matrixVariantId",
          p.combination_key AS "combinationKey",
          p.generated_from_matrix AS "generatedFromMatrix",
          p.color_options AS "colorOptions",
          p.odoo_product_id AS "odooProductId",
          p.odoo_exported_at AS "odooExportedAt",
          p.created_at AS "createdAt",
          p.updated_at AS "updatedAt"
        FROM catalog.products p 
        WHERE p.id = $1
      `, [id]);
      
      if (productResult.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }
      
      const product = productResult.rows[0];
      
      // Dane platform (with marketplace info and link type)
      const platformDataResult = await pool.query(`
        SELECT 
          ppd.id,
          ppd.product_id AS "productId",
          ppd.link_type AS "linkType",
          ppd.product_id AS "catalogProductId",
          ppd.set_id AS "catalogSetId",
          ppd.platform,
          ppd.external_id AS "externalId",
          ppd.custom_title AS "customTitle",
          ppd.category_id AS "categoryId",
          ppd.category_name AS "categoryName",
          ppd.description,
          ppd.custom_price AS "customPrice",
          ppd.is_published AS "isPublished",
          ppd.created_at AS "createdAt",
          ppd.updated_at AS "updatedAt",
          mp.id AS "marketplaceProductId",
          mp.name AS "marketplaceName",
          mp.offer_external_id AS "marketplaceSku"
        FROM catalog.product_platform_data ppd
        LEFT JOIN commerce.marketplace_products mp 
          ON LOWER(mp.source) = LOWER(ppd.platform) AND mp.offer_external_id = ppd.external_id
        WHERE ppd.product_id = $1
      `, [id]);
      
      // Zdjęcia
      const imagesResult = await pool.query(`
        SELECT * FROM catalog.product_images WHERE product_id = $1 ORDER BY sort_order, id
      `, [id]);
      
      // Convert local URLs to SFTP URLs if needed
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      const images = imagesResult.rows.map((image: any) => {
        if (image.url && !image.url.startsWith('http')) {
          const filename = image.url.split('/').pop();
          image.url = adapter.getUrl(`products/images/${filename}`);
        }
        return image;
      });
      
      // Przypisane dodatki
      const addonsResult = await pool.query(`
        SELECT 
          paa.*,
          pa.name as addon_name,
          pa.addon_type,
          pa.description as addon_description,
          pa.html_template
        FROM catalog.product_addon_assignments paa
        JOIN catalog.product_addons pa ON paa.addon_id = pa.id
        WHERE paa.product_id = $1
        ORDER BY paa.sort_order, paa.id
      `, [id]);
      
      
      // Packed product data (for warehouse link)
      const packedProductResult = await pool.query(`
        SELECT 
          MAX(pp.id) AS "packedProductId",
          MAX(pp.quantity) AS "stockQuantity",
          MAX(pp.reserved_quantity) AS "reservedQuantity"
        FROM warehouse.packed_products pp
        WHERE pp.catalog_product_id = $1 AND pp.is_active = true AND pp.is_archived = false
        LIMIT 1
      `, [id]);
      const packedProduct = packedProductResult.rows[0] || null;
      res.json({
        ...product,
        platformData: platformDataResult.rows,
        images,
        addons: addonsResult.rows,
        packedProduct,
      });
    } catch (error) {
      console.error("❌ Error fetching product:", error);
      res.status(500).json({ error: "Failed to fetch product" });
    }
  });

  // Tworzenie produktu
  app.post("/api/catalog-products", isAuthenticated, async (req, res) => {
    try {
      const {
        sku,
        title,
        shortDescription,
        longDescriptionHtml,
        longDescriptionDoc,
        length,
        width,
        height,
        weight,
        color,
        material,
        productType,
        basePrice,
        currency,
      } = req.body;
      
      const result = await pool.query(`
        INSERT INTO catalog.products (
          sku, title, short_description, long_description_html, long_description_doc,
          length, width, height, weight,
          color, material, product_type, base_price, currency
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14)
        RETURNING *
      `, [sku, title, shortDescription, longDescriptionHtml || null, longDescriptionDoc ? JSON.stringify(longDescriptionDoc) : null, length, width, height, weight, color, material, productType, basePrice, currency || 'PLN']);
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating product:", error);
      res.status(500).json({ error: "Failed to create product" });
    }
  });

  // Aktualizacja produktu (PUT - pełna aktualizacja z danymi platformowymi)
  app.put("/api/catalog-products/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const {
        sku, title, shortDescription, longDescriptionHtml, longDescriptionDoc, length, width, height, weight,
        color, material, productType, productGroup, doors, legs, basePrice, currency, isActive,
        allegro, shoper, odoo
      } = req.body;
      
      console.log(`📝 PUT /api/catalog-products/${id} - Received data:`, {
        sku, title, shortDescription, 
        productType, productGroup, doors, legs,
        longDescriptionHtml: longDescriptionHtml ? 'present' : 'null/undefined'
      });
      
      // Update main product
      const productResult = await pool.query(`
        UPDATE catalog.products
        SET 
          sku = $1, title = $2, short_description = $3, long_description_html = $4, long_description_doc = $5,
          length = $6, width = $7, height = $8, weight = $9,
          color = $10, material = $11, product_type = $12, product_group = $13, doors = $14, legs = $15,
          base_price = $16, currency = $17, is_active = $18, updated_at = NOW()
        WHERE id = $19
        RETURNING *
      `, [sku, title, shortDescription, longDescriptionHtml || null, longDescriptionDoc ? JSON.stringify(longDescriptionDoc) : null, length, width, height, weight, color, material, productType, productGroup, doors, legs, basePrice, currency || 'PLN', isActive ?? true, id]);
      
      if (productResult.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }
      
      // Update platform data for Allegro
      if (allegro) {
        await pool.query(`
          INSERT INTO catalog.product_platform_data (
            product_id, platform, custom_title, category_id, category_name,
            description, custom_price, is_published
          )
          VALUES ($1, 'allegro', $2, $3, $4, $5, $6, $7)
          ON CONFLICT (product_id, platform) 
          DO UPDATE SET
            custom_title = EXCLUDED.custom_title,
            category_id = EXCLUDED.category_id,
            category_name = EXCLUDED.category_name,
            description = EXCLUDED.description,
            is_published = EXCLUDED.is_published,
            updated_at = NOW()
        `, [id, allegro.customTitle, allegro.categoryId, allegro.categoryName, allegro.description, allegro.customPrice, allegro.isPublished ?? false]);
      }
      
      // Update platform data for Shoper
      if (shoper) {
        await pool.query(`
          INSERT INTO catalog.product_platform_data (
            product_id, platform, custom_title, category_id, category_name,
            description, custom_price, is_published
          )
          VALUES ($1, 'shoper', $2, $3, $4, $5, $6, $7)
          ON CONFLICT (product_id, platform) 
          DO UPDATE SET
            custom_title = EXCLUDED.custom_title,
            category_id = EXCLUDED.category_id,
            category_name = EXCLUDED.category_name,
            description = EXCLUDED.description,
            is_published = EXCLUDED.is_published,
            updated_at = NOW()
        `, [id, shoper.customTitle, shoper.categoryId, shoper.categoryName, shoper.description, shoper.customPrice, shoper.isPublished ?? false]);
      }
      
      // Update platform data for Odoo
      if (odoo) {
        await pool.query(`
          INSERT INTO catalog.product_platform_data (
            product_id, platform, custom_title, category_id, category_name,
            description, custom_price, is_published
          )
          VALUES ($1, 'odoo', $2, $3, $4, $5, $6, $7)
          ON CONFLICT (product_id, platform) 
          DO UPDATE SET
            custom_title = EXCLUDED.custom_title,
            category_id = EXCLUDED.category_id,
            category_name = EXCLUDED.category_name,
            description = EXCLUDED.description,
            is_published = EXCLUDED.is_published,
            updated_at = NOW()
        `, [id, odoo.customTitle, odoo.categoryId, odoo.categoryName, odoo.description, odoo.customPrice, odoo.isPublished ?? false]);
      }
      
      res.json(productResult.rows[0]);
    } catch (error) {
      console.error("❌ Error updating product:", error);
      res.status(500).json({ error: "Failed to update product" });
    }
  });

  // Aktualizacja produktu (PATCH - częściowa aktualizacja)
  app.patch("/api/catalog-products/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const updates = req.body;
      
      const fields = [];
      const values = [];
      let paramCount = 1;
      
      for (const [key, value] of Object.entries(updates)) {
        if (key !== 'id' && key !== 'createdAt' && key !== 'updatedAt') {
          const snakeKey = key.replace(/[A-Z]/g, letter => `_${letter.toLowerCase()}`);
          fields.push(`${snakeKey} = $${paramCount}`);
          values.push(value);
          paramCount++;
        }
      }
      
      // Sprawdź czy są jakieś pola do aktualizacji
      if (fields.length === 0) {
        return res.status(400).json({ error: "No fields to update" });
      }
      
      values.push(id);
      
      const result = await pool.query(`
        UPDATE catalog.products
        SET ${fields.join(', ')}, updated_at = NOW()
        WHERE id = $${paramCount}
        RETURNING *
      `, values);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating product:", error);
      res.status(500).json({ error: "Failed to update product" });
    }
  });

  // Pobierz zestawy, w których występuje produkt
  app.get("/api/catalog-products/:id/sets", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT 
          ps.id,
          ps.sku,
          ps.title,
          ps.set_matrix_id AS "setMatrixId",
          spl.quantity,
          spl.component_type AS "componentType"
        FROM product_creator.set_product_links spl
        JOIN product_creator.product_sets ps ON ps.id = spl.set_id
        WHERE spl.product_id = $1
        ORDER BY ps.title
      `, [id]);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching product sets:", error);
      res.status(500).json({ error: "Failed to fetch product sets" });
    }
  });

  // Usuwanie produktu
  app.delete("/api/catalog-products/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Sprawdź czy produkt jest używany w zestawach
      const setsCheck = await pool.query(`
        SELECT COUNT(*) as count
        FROM product_creator.set_product_links
        WHERE product_id = $1
      `, [id]);
      
      const setsCount = parseInt(setsCheck.rows[0].count, 10);
      
      if (setsCount > 0) {
        return res.status(400).json({ 
          error: `Nie można usunąć produktu. Jest używany w ${setsCount} zestawie/zestawach. Usuń produkt z zestawów przed usunięciem.` 
        });
      }
      
      // Usuń BOM i komponenty (jeśli istnieją) przed usunięciem produktu
      await pool.query(`
        DELETE FROM bom.product_components 
        WHERE product_bom_id IN (SELECT id FROM bom.product_boms WHERE product_id = $1)
      `, [id]);
      
      await pool.query(`DELETE FROM bom.product_boms WHERE product_id = $1`, [id]);
      
      console.log(`🗑️ Deleted BOM for product ${id} (if existed)`);
      
      // Usuń produkt
      await pool.query(`DELETE FROM catalog.products WHERE id = $1`, [id]);
      
      console.log(`✅ Deleted product ${id}`);
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting product:", error);
      res.status(500).json({ error: "Failed to delete product" });
    }
  });

  // Bulk update products
  app.post("/api/catalog-products/bulk-update", isAuthenticated, async (req, res) => {
    try {
      const { productIds, updates } = req.body;

      if (!productIds || !Array.isArray(productIds) || productIds.length === 0) {
        return res.status(400).json({ error: "Product IDs array is required" });
      }

      if (!updates || Object.keys(updates).length === 0) {
        return res.status(400).json({ error: "Updates object is required" });
      }

      const fields: string[] = [];
      const values: any[] = [];
      let paramCount = 1;

      // Convert camelCase to snake_case and build update fields
      for (const [key, value] of Object.entries(updates)) {
        if (key !== 'id' && key !== 'createdAt' && key !== 'updatedAt') {
          if (value !== undefined && value !== null && value !== '') {
            const snakeKey = key.replace(/[A-Z]/g, letter => `_${letter.toLowerCase()}`);
            fields.push(`${snakeKey} = $${paramCount}`);
            values.push(value);
            paramCount++;
          }
        }
      }

      if (fields.length === 0) {
        return res.status(400).json({ error: "No valid fields to update" });
      }

      // Add product IDs as the last parameter
      values.push(productIds);

      const result = await pool.query(`
        UPDATE catalog.products
        SET ${fields.join(', ')}, updated_at = NOW()
        WHERE id = ANY($${paramCount})
        RETURNING id
      `, values);

      console.log(`✅ Bulk updated ${result.rowCount} products`);

      res.json({ 
        success: true, 
        updatedCount: result.rowCount,
        productIds: result.rows.map(r => r.id)
      });
    } catch (error) {
      console.error("❌ Error bulk updating products:", error);
      res.status(500).json({ error: "Failed to bulk update products" });
    }
  });

  // Bulk delete products
  app.post("/api/catalog-products/bulk-delete", requireAdmin, async (req, res) => {
    try {
      const { productIds } = req.body;

      if (!productIds || !Array.isArray(productIds) || productIds.length === 0) {
        return res.status(400).json({ error: "Product IDs array is required" });
      }

      // Sprawdź czy któryś produkt jest używany w zestawach
      const setsCheck = await pool.query(`
        SELECT product_id, COUNT(*) as set_count
        FROM product_creator.set_product_links
        WHERE product_id = ANY($1)
        GROUP BY product_id
      `, [productIds]);
      
      if (setsCheck.rows.length > 0) {
        const blockedProducts = setsCheck.rows.map(r => 
          `ID ${r.product_id} (${r.set_count} zestawów)`
        ).join(', ');
        
        return res.status(400).json({ 
          error: `Nie można usunąć produktów powiązanych z zestawami: ${blockedProducts}. Usuń najpierw powiązania z zestawów.` 
        });
      }

      // Usuń BOM i komponenty dla wszystkich produktów
      await pool.query(`
        DELETE FROM bom.product_components 
        WHERE product_bom_id IN (
          SELECT id FROM bom.product_boms WHERE product_id = ANY($1)
        )
      `, [productIds]);
      
      const bomResult = await pool.query(`
        DELETE FROM bom.product_boms WHERE product_id = ANY($1) RETURNING id
      `, [productIds]);
      
      console.log(`🗑️ Deleted ${bomResult.rowCount} BOMs for bulk delete`);

      // Usuń produkty
      const result = await pool.query(
        `DELETE FROM catalog.products WHERE id = ANY($1) RETURNING id`,
        [productIds]
      );

      console.log(`✅ Bulk deleted ${result.rowCount} products`);

      res.json({ 
        success: true, 
        deletedCount: result.rowCount,
        productIds: result.rows.map(r => r.id)
      });
    } catch (error) {
      console.error("❌ Error bulk deleting products:", error);
      res.status(500).json({ error: "Failed to bulk delete products" });
    }
  });

  // Bulk BOM generation
  app.post("/api/catalog-products/bulk-bom", isAuthenticated, async (req, res) => {
    try {
      const { productIds } = req.body;

      if (!productIds || !Array.isArray(productIds) || productIds.length === 0) {
        return res.status(400).json({ error: "Product IDs array is required" });
      }

      console.log(`🔨 [BULK BOM] Starting bulk BOM generation for ${productIds.length} products`);

      const results = {
        successful: 0,
        failed: 0,
        errors: [] as string[]
      };

      // NOWY SYSTEM: Generuj BOM bez renderingu formatek (formatki będą renderowane w tle)
      for (const productId of productIds) {
        try {
          // Wywołaj endpoint generate-bom z skipFormatkaRendering=true
          const response = await fetch(`http://localhost:${process.env.PORT || 5000}/api/catalog-products/${productId}/generate-bom`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'Cookie': req.headers.cookie || ''
            },
            body: JSON.stringify({ skipFormatkaRendering: true })
          });

          if (response.ok) {
            results.successful++;
          } else {
            results.failed++;
            const errorData = await response.json().catch(() => ({ error: response.statusText }));
            results.errors.push(`Product ${productId}: ${errorData.error || response.statusText}`);
          }
        } catch (error: any) {
          results.failed++;
          results.errors.push(`Product ${productId}: ${error.message}`);
        }
      }

      console.log(`✅ [BULK BOM] Completed ${results.successful} BOMs, ${results.failed} failed`);

      res.json({
        ...results,
        message: `Generated ${results.successful} BOMs. Formatka visualizations will be rendered in the background.`
      });
    } catch (error) {
      console.error("❌ Error bulk generating BOM:", error);
      res.status(500).json({ error: "Failed to bulk generate BOM" });
    }
  });

  // Bulk BOM deletion
  app.delete("/api/catalog-products/bulk-bom", isAuthenticated, async (req, res) => {
    try {
      const { productIds } = req.body;

      if (!productIds || !Array.isArray(productIds) || productIds.length === 0) {
        return res.status(400).json({ error: "Product IDs array is required" });
      }

      console.log(`🗑️ [BULK BOM DELETE] Starting bulk BOM deletion for ${productIds.length} products`);

      const results = {
        successful: 0,
        failed: 0,
        errors: [] as string[]
      };

      for (const productId of productIds) {
        try {
          // Call the same endpoint logic as single BOM deletion
          const response = await fetch(`http://localhost:${process.env.PORT || 5000}/api/catalog-products/${productId}/bom`, {
            method: 'DELETE',
            headers: {
              'Content-Type': 'application/json',
              'Cookie': req.headers.cookie || ''
            }
          });

          if (response.ok) {
            results.successful++;
          } else if (response.status === 404) {
            // BOM nie istnieje - traktujemy jako sukces
            results.successful++;
          } else {
            results.failed++;
            results.errors.push(`Product ${productId}: ${response.statusText}`);
          }
        } catch (error: any) {
          results.failed++;
          results.errors.push(`Product ${productId}: ${error.message}`);
        }
      }

      console.log(`✅ [BULK BOM DELETE] Completed: ${results.successful} successful, ${results.failed} failed`);

      res.json(results);
    } catch (error) {
      console.error("❌ Error bulk deleting BOM:", error);
      res.status(500).json({ error: "Failed to bulk delete BOM" });
    }
  });

  // Bulk BOM export to Odoo
  app.post("/api/catalog-products/bulk-bom-export", isAuthenticated, async (req, res) => {
    try {
      const { productIds } = req.body;

      if (!productIds || !Array.isArray(productIds) || productIds.length === 0) {
        return res.status(400).json({ error: "Product IDs array is required" });
      }

      console.log(`📤 [BULK BOM EXPORT] Starting bulk BOM export to Odoo for ${productIds.length} products`);

      const results = {
        successful: 0,
        failed: 0,
        errors: [] as string[]
      };

      for (const productId of productIds) {
        try {
          // Call the same endpoint logic as single BOM export
          const response = await fetch(`http://localhost:${process.env.PORT || 5000}/api/catalog-products/${productId}/export-bom-to-odoo`, {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              'Cookie': req.headers.cookie || ''
            }
          });

          if (response.ok) {
            results.successful++;
          } else {
            const errorData = await response.json();
            results.failed++;
            results.errors.push(`Product ${productId}: ${errorData.error || response.statusText}`);
          }
        } catch (error: any) {
          results.failed++;
          results.errors.push(`Product ${productId}: ${error.message}`);
        }
      }

      console.log(`✅ [BULK BOM EXPORT] Completed: ${results.successful} successful, ${results.failed} failed`);

      res.json(results);
    } catch (error) {
      console.error("❌ Error bulk exporting BOM to Odoo:", error);
      res.status(500).json({ error: "Failed to bulk export BOM to Odoo" });
    }
  });

  // Export single product (HTML, CSV, PDF)
  app.get("/api/catalog-products/:id/export", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const format = req.query.format as string;

      if (!['html', 'csv', 'pdf'].includes(format)) {
        return res.status(400).json({ error: "Invalid format. Use: html, csv, pdf" });
      }

      // Fetch product
      const productResult = await pool.query(`
        SELECT * FROM catalog.products WHERE id = $1
      `, [id]);

      if (productResult.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }

      const product = productResult.rows[0];

      // Fetch images separately
      const imagesResult = await pool.query(`
        SELECT id, url, is_primary, sort_order
        FROM catalog.product_images
        WHERE product_id = $1
        ORDER BY is_primary DESC, sort_order ASC
      `, [id]);

      // Attach images to product
      product.images = imagesResult.rows.map(img => ({
        id: img.id,
        imageUrl: img.url,
        isPrimary: img.is_primary,
        sortOrder: img.sort_order
      }));

      if (format === 'html') {
        // Export as HTML
        const html = `
<!DOCTYPE html>
<html lang="pl">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>${product.title}</title>
  <style>
    body { font-family: Arial, sans-serif; max-width: 800px; margin: 40px auto; padding: 20px; }
    h1 { color: #333; }
    .section { margin: 20px 0; }
    .label { font-weight: bold; color: #666; }
    .description { margin: 15px 0; line-height: 1.6; }
    table { width: 100%; border-collapse: collapse; margin: 20px 0; }
    th, td { padding: 10px; text-align: left; border-bottom: 1px solid #ddd; }
    th { background-color: #f5f5f5; }
  </style>
</head>
<body>
  <h1>${product.title}</h1>
  <div class="section">
    <span class="label">SKU:</span> ${product.sku}
  </div>
  ${product.short_description ? `<div class="section"><span class="label">Krótki opis:</span> ${product.short_description}</div>` : ''}
  
  <table>
    <tr><th>Właściwość</th><th>Wartość</th></tr>
    ${product.length ? `<tr><td>Długość</td><td>${product.length} mm</td></tr>` : ''}
    ${product.width ? `<tr><td>Szerokość</td><td>${product.width} mm</td></tr>` : ''}
    ${product.height ? `<tr><td>Wysokość</td><td>${product.height} mm</td></tr>` : ''}
    ${product.weight ? `<tr><td>Waga</td><td>${product.weight} kg</td></tr>` : ''}
    ${product.color ? `<tr><td>Kolor</td><td>${product.color}</td></tr>` : ''}
    ${product.material ? `<tr><td>Materiał</td><td>${product.material}</td></tr>` : ''}
    ${product.product_type ? `<tr><td>Typ</td><td>${product.product_type}</td></tr>` : ''}
    ${product.product_group ? `<tr><td>Grupa</td><td>${product.product_group}</td></tr>` : ''}
    ${product.base_price ? `<tr><td>Cena bazowa</td><td>${product.base_price} ${product.currency}</td></tr>` : ''}
  </table>

  ${product.long_description_html ? `<div class="section"><h2>Pełny opis</h2><div class="description">${product.long_description_html}</div></div>` : ''}
</body>
</html>`;

        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
        const filename = product.sku + '-' + timestamp + '.html';
        res.setHeader('Content-Type', 'text/html; charset=utf-8');
        res.setHeader('Content-Disposition', 'attachment; filename=' + filename);
        res.send(html);

      } else if (format === 'csv') {
        // Export as CSV
        const csvData = {
          SKU: product.sku,
          Tytuł: product.title,
          'Krótki opis': product.short_description || '',
          'Długość (mm)': product.length || '',
          'Szerokość (mm)': product.width || '',
          'Wysokość (mm)': product.height || '',
          'Waga (kg)': product.weight || '',
          Kolor: product.color || '',
          Materiał: product.material || '',
          Typ: product.product_type || '',
          Grupa: product.product_group || '',
          'Cena bazowa': product.base_price || '',
          Waluta: product.currency || 'PLN',
          Aktywny: product.is_active ? 'Tak' : 'Nie',
        };

        const csv = Papa.unparse([csvData], {
          header: true,
          delimiter: ';',
        });

        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
        const filename = `${product.sku}-${timestamp}.csv`;
        res.setHeader('Content-Type', 'text/csv; charset=utf-8');
        res.setHeader('Content-Disposition', `attachment; filename=${filename}`);
        res.send('\ufeff' + csv); // UTF-8 BOM for Excel

      } else if (format === 'pdf') {
        // Export as PDF with UTF-8 support
        const doc = new PDFDocument({ margin: 50 });
        
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-').slice(0, -5);
        const filename = `${product.sku}-${timestamp}.pdf`;
        res.setHeader('Content-Type', 'application/pdf');
        res.setHeader('Content-Disposition', `attachment; filename=${filename}`);
        
        doc.pipe(res);

        // Use DejaVu Sans font for proper Polish character support
        doc.font('/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf');

        // Title
        doc.fontSize(20).text(product.title, { align: 'left' });
        doc.fontSize(10).text(`SKU: ${product.sku}`, { align: 'left' });
        doc.moveDown();

        // Primary image
        if (product.images && Array.isArray(product.images) && product.images.length > 0) {
          const primaryImage = product.images.find((img: any) => img.isPrimary) || product.images[0];
          if (primaryImage && primaryImage.imageUrl) {
            try {
              // Handle local uploads - remove leading slash for proper path.join
              if (primaryImage.imageUrl.startsWith('/uploads')) {
                const imageUrl = path.join(process.cwd(), primaryImage.imageUrl.replace(/^\//, ''));
                try {
                  await fs.access(imageUrl);
                  doc.image(imageUrl,
        unit, { fit: [250, 250], align: 'center' });
                  doc.moveDown();
                } catch (err) {
                  console.log('Image file not found, skipping:', imageUrl);
                }
              }
            } catch (err) {
              console.log('Error adding image to PDF:', err);
            }
          }
        }

        // Short description
        if (product.short_description) {
          doc.fontSize(12).text(product.short_description);
          doc.moveDown();
        }

        // Properties table
        doc.fontSize(14).text('Właściwości produktu', { underline: true });
        doc.moveDown(0.5);
        doc.fontSize(10);

        const properties = [
          { label: 'Długość', value: product.length ? `${product.length} mm` : '-' },
          { label: 'Szerokość', value: product.width ? `${product.width} mm` : '-' },
          { label: 'Wysokość', value: product.height ? `${product.height} mm` : '-' },
          { label: 'Waga', value: product.weight ? `${product.weight} kg` : '-' },
          { label: 'Kolor', value: product.color || '-' },
          { label: 'Materiał', value: product.material || '-' },
          { label: 'Typ produktu', value: product.product_type || '-' },
          { label: 'Grupa', value: product.product_group || '-' },
          { label: 'Cena bazowa', value: product.base_price ? `${product.base_price} ${product.currency}` : '-' },
        ];

        properties.forEach(prop => {
          doc.text(`${prop.label}: ${prop.value}`);
        });

        // Long description (strip HTML tags)
        if (product.long_description_html) {
          doc.moveDown();
          doc.fontSize(14).text('Pełny opis produktu', { underline: true });
          doc.moveDown(0.5);
          doc.fontSize(10);
          
          // Strip HTML tags and decode entities for plain text
          const plainText = product.long_description_html
            .replace(/<style[^>]*>[\s\S]*?<\/style>/gi, '')
            .replace(/<script[^>]*>[\s\S]*?<\/script>/gi, '')
            .replace(/<br\s*\/?>/gi, '\n')
            .replace(/<\/p>/gi, '\n\n')
            .replace(/<[^>]+>/g, ' ')
            .replace(/&nbsp;/g, ' ')
            .replace(/&amp;/g, '&')
            .replace(/&lt;/g, '<')
            .replace(/&gt;/g, '>')
            .replace(/&quot;/g, '"')
            .replace(/&#39;/g, "'")
            .replace(/&apos;/g, "'")
            .replace(/&oacute;/g, 'ó')
            .replace(/&oacute;/gi, 'ó')
            .replace(/&Oacute;/g, 'Ó')
            .replace(/&aacute;/g, 'á')
            .replace(/&Aacute;/g, 'Á')
            .replace(/&eacute;/g, 'é')
            .replace(/&Eacute;/g, 'É')
            .replace(/&iacute;/g, 'í')
            .replace(/&Iacute;/g, 'Í')
            .replace(/&uacute;/g, 'ú')
            .replace(/&Uacute;/g, 'Ú')
            .replace(/&#8211;/g, '-')
            .replace(/&#8212;/g, '-')
            .replace(/&#8216;/g, "'")
            .replace(/&#8217;/g, "'")
            .replace(/&#8220;/g, '"')
            .replace(/&#8221;/g, '"')
            .replace(/\s+/g, ' ')
            .trim();
          
          doc.text(plainText, { align: 'left', lineGap: 2 });
        }

        doc.end();
      }

    } catch (error) {
      console.error("❌ Error exporting product:", error);
      res.status(500).json({ error: "Failed to export product" });
    }
  });

  // Get orders that contain this catalog product (via marketplace links - unified commerce layer)
  app.get("/api/catalog-products/:id/ordered-by", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;

      // Find orders through marketplace product links in unified commerce layer
      // 1. Get SKU (external_id) from catalog.product_platform_data
      // 2. Match SKU with commerce.order_items.offer_external_id (works for all marketplaces)
      const result = await pool.query(`
        SELECT 
          o.order_number as order_id,
          o.buyer_login,
          o.buyer_email,
          COALESCE(o.buyer_first_name || ' ' || o.buyer_last_name, o.buyer_login) as buyer_full_name,
          o.order_date,
          o.source as marketplace,
          o.payment_status,
          COALESCE(o.total_to_pay_amount, o.payment_amount) as total_amount,
          COALESCE(o.total_to_pay_currency, o.payment_currency) as currency,
          oi.quantity,
          oi.name as item_name,
          oi.offer_external_id as offer_id,
          oi.image_url as item_image_url
        FROM commerce.order_items oi
        INNER JOIN commerce.orders o ON oi.order_id = o.id
        WHERE oi.offer_external_id IN (
          SELECT external_id 
          FROM catalog.product_platform_data 
          WHERE product_id = $1 
            AND external_id IS NOT NULL
        )
        ORDER BY o.order_date DESC
        LIMIT 100
      `, [id]);

      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching ordered-by data:", error);
      res.status(500).json({ error: "Failed to fetch orders for product" });
    }
  });

  // Batch export products (CSV, PDF)
  app.post("/api/catalog-products/batch-export", isAuthenticated, async (req, res) => {
    try {
      const { productIds, format } = req.body;

      if (!productIds || !Array.isArray(productIds) || productIds.length === 0) {
        return res.status(400).json({ error: "Product IDs array is required" });
      }

      if (!['csv', 'pdf'].includes(format)) {
        return res.status(400).json({ error: "Invalid format. Use: csv, pdf" });
      }

      // Fetch products
      const result = await pool.query(`
        SELECT * FROM catalog.products
        WHERE id = ANY($1)
        ORDER BY sku
      `, [productIds]);

      const products = result.rows;

      if (products.length === 0) {
        return res.status(404).json({ error: "No products found" });
      }

      if (format === 'csv') {
        // Batch export as CSV
        const csvData = products.map(p => ({
          SKU: p.sku,
          Tytuł: p.title,
          'Krótki opis': p.short_description || '',
          'Długość (mm)': p.length || '',
          'Szerokość (mm)': p.width || '',
          'Wysokość (mm)': p.height || '',
          'Waga (kg)': p.weight || '',
          Kolor: p.color || '',
          Materiał: p.material || '',
          Typ: p.product_type || '',
          Grupa: p.product_group || '',
          'Cena bazowa': p.base_price || '',
          Waluta: p.currency || 'PLN',
          Aktywny: p.is_active ? 'Tak' : 'Nie',
        }));

        const csv = Papa.unparse(csvData, {
          header: true,
          delimiter: ';',
        });

        // Generate timestamp filename
        const now = new Date();
        const timestamp = now.toISOString()
          .replace(/T/, '_')
          .replace(/:/g, '-')
          .split('.')[0]; // Format: YYYY-MM-DD_HH-MM-SS
        const filename = `products-export-${timestamp}.csv`;
        res.setHeader('Content-Type', 'text/csv; charset=utf-8');
        res.setHeader('Content-Disposition', `attachment; filename=${filename}`);
        res.send('\ufeff' + csv); // UTF-8 BOM for Excel

      } else if (format === 'pdf') {
        // Batch export as PDF with UTF-8 support
        const doc = new PDFDocument({ margin: 50 });
        
        // Generate timestamp filename
        const now = new Date();
        const timestamp = now.toISOString()
          .replace(/T/, '_')
          .replace(/:/g, '-')
          .split('.')[0]; // Format: YYYY-MM-DD_HH-MM-SS
        const filename = `products-export-${timestamp}.pdf`;
        res.setHeader('Content-Type', 'application/pdf');
        res.setHeader('Content-Disposition', `attachment; filename=${filename}`);
        
        doc.pipe(res);

        // Use DejaVu Sans font for proper Polish character support
        doc.font('/usr/share/fonts/truetype/dejavu/DejaVuSans.ttf');

        // Title
        doc.fontSize(18).text('Eksport produktów', { align: 'center' });
        doc.fontSize(10).text(`Data: ${new Date().toLocaleDateString('pl-PL')}`, { align: 'center' });
        doc.fontSize(10).text(`Liczba produktów: ${products.length}`, { align: 'center' });
        doc.moveDown(2);

        // Fetch images for all products
        const productImagesResult = await pool.query(`
          SELECT product_id, json_agg(json_build_object('imageUrl', url, 'isPrimary', is_primary) ORDER BY is_primary DESC, sort_order ASC) as images
          FROM catalog.product_images
          WHERE product_id = ANY($1)
          GROUP BY product_id
        `, [productIds]);

        const imagesByProductId: Record<number, any[]> = {};
        productImagesResult.rows.forEach((row: any) => {
          imagesByProductId[row.product_id] = row.images;
        });

        // Products
        for (let index = 0; index < products.length; index++) {
          const product = products[index];
          
          if (index > 0) {
            doc.addPage();
          }

          doc.fontSize(16).text(product.title, { underline: true });
          doc.fontSize(10).text(`SKU: ${product.sku}`);
          doc.moveDown();

          // Add primary image if available
          const productImages = imagesByProductId[product.id];
          if (productImages && productImages.length > 0) {
            const primaryImage = productImages.find((img: any) => img.isPrimary) || productImages[0];
            if (primaryImage && primaryImage.imageUrl && primaryImage.imageUrl.startsWith('/uploads')) {
              try {
                // Remove leading slash for proper path.join
                const imageUrl = path.join(process.cwd(), primaryImage.imageUrl.replace(/^\//, ''));
                await fs.access(imageUrl);
                doc.image(imageUrl,
        unit, { fit: [200, 200], align: 'center' });
                doc.moveDown();
              } catch (err) {
                // Image not found, skip silently
              }
            }
          }

          if (product.short_description) {
            doc.fontSize(10).text(product.short_description);
            doc.moveDown();
          }

          doc.fontSize(10);
          const props = [
            { label: 'Długość', value: product.length ? `${product.length} mm` : '-' },
            { label: 'Szerokość', value: product.width ? `${product.width} mm` : '-' },
            { label: 'Wysokość', value: product.height ? `${product.height} mm` : '-' },
            { label: 'Waga', value: product.weight ? `${product.weight} kg` : '-' },
            { label: 'Kolor', value: product.color || '-' },
            { label: 'Materiał', value: product.material || '-' },
            { label: 'Typ', value: product.product_type || '-' },
            { label: 'Grupa', value: product.product_group || '-' },
            { label: 'Cena', value: product.base_price ? `${product.base_price} ${product.currency}` : '-' },
          ];

          props.forEach(prop => {
            doc.text(`${prop.label}: ${prop.value}`);
          });

          // Long description (truncated for batch export)
          if (product.long_description_html) {
            doc.moveDown();
            doc.fontSize(12).text('Pełny opis:', { underline: true });
            doc.moveDown(0.5);
            doc.fontSize(9);
            
            // Strip HTML tags, decode entities and truncate for batch export
            const plainText = product.long_description_html
              .replace(/<style[^>]*>[\s\S]*?<\/style>/gi, '')
              .replace(/<script[^>]*>[\s\S]*?<\/script>/gi, '')
              .replace(/<br\s*\/?>/gi, ' ')
              .replace(/<\/p>/gi, ' ')
              .replace(/<[^>]+>/g, ' ')
              .replace(/&nbsp;/g, ' ')
              .replace(/&amp;/g, '&')
              .replace(/&lt;/g, '<')
              .replace(/&gt;/g, '>')
              .replace(/&quot;/g, '"')
              .replace(/&#39;/g, "'")
              .replace(/&apos;/g, "'")
              .replace(/&oacute;/gi, 'ó')
              .replace(/&Oacute;/g, 'Ó')
              .replace(/&aacute;/g, 'á')
              .replace(/&Aacute;/g, 'Á')
              .replace(/&eacute;/g, 'é')
              .replace(/&Eacute;/g, 'É')
              .replace(/&iacute;/g, 'í')
              .replace(/&Iacute;/g, 'Í')
              .replace(/&uacute;/g, 'ú')
              .replace(/&Uacute;/g, 'Ú')
              .replace(/&#8211;/g, '-')
              .replace(/&#8212;/g, '-')
              .replace(/&#8216;/g, "'")
              .replace(/&#8217;/g, "'")
              .replace(/&#8220;/g, '"')
              .replace(/&#8221;/g, '"')
              .replace(/\s+/g, ' ')
              .trim();
            
            // Truncate to 500 characters at last whitespace for better readability
            let truncated = plainText;
            if (plainText.length > 500) {
              truncated = plainText.substring(0, 500);
              const lastSpace = truncated.lastIndexOf(' ');
              if (lastSpace > 400) { // Only trim at space if it's not too far back
                truncated = truncated.substring(0, lastSpace);
              }
              truncated += '...';
            }
            doc.text(truncated, { align: 'left' });
          }
        }

        doc.end();
      }

    } catch (error) {
      console.error("❌ Error batch exporting products:", error);
      res.status(500).json({ error: "Failed to batch export products" });
    }
  });

  // Product Platform Data
  app.post("/api/catalog-products/:id/platform-data", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { platform, customTitle, categoryId, categoryName, description, customPrice, customAttributes } = req.body;
      
      const result = await pool.query(`
        INSERT INTO product_platform_data (
          product_id, platform, custom_title, category_id, category_name,
          description, custom_price, custom_attributes
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        ON CONFLICT (product_id, platform) 
        DO UPDATE SET
          custom_title = EXCLUDED.custom_title,
          category_id = EXCLUDED.category_id,
          category_name = EXCLUDED.category_name,
          description = EXCLUDED.description,
          custom_attributes = EXCLUDED.custom_attributes,
          updated_at = NOW()
        RETURNING *
      `, [id, platform, customTitle, categoryId, categoryName, description, customPrice, JSON.stringify(customAttributes)]);
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error saving platform data:", error);
      res.status(500).json({ error: "Failed to save platform data" });
    }
  });


  // Product Images
  
  // Get all product images (for gallery picker)
  app.get("/api/all-product-images", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          pi.*,
          p.title as product_title
        FROM catalog.product_images pi
        JOIN products p ON pi.product_id = p.id
        ORDER BY pi.created_at DESC
        LIMIT 500
      `);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching all product images:", error);
      res.status(500).json({ error: "Failed to fetch images" });
    }
  });
  
  // Get all images for a product
  app.get("/api/catalog-products/:id/images", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT * FROM catalog.product_images
        WHERE product_id = $1
        ORDER BY sort_order ASC, created_at ASC
      `, [id]);
      
      // Convert local URLs to SFTP URLs if needed
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      const images = result.rows.map((image: any) => {
        // Check if URL is local (starts with /uploads or is just a filename)
        if (image.url && !image.url.startsWith('http')) {
          // Extract filename and convert to SFTP URL
          const filename = image.url.split('/').pop();
          image.url = adapter.getUrl(`products/images/${filename}`);
        }
        return image;
      });
      
      res.json(images);
    } catch (error) {
      console.error("❌ Error fetching product images:", error);
      res.status(500).json({ error: "Failed to fetch product images" });
    }
  });

  // Upload a new image
  app.post("/api/catalog-products/:id/images", isAuthenticated, uploadCatalogImage.single('image'), async (req, res) => {
    try {
      const { id } = req.params;
      const { imageType, altText, title: imageTitle, isPrimary } = req.body;
      
      if (!req.file) {
        return res.status(400).json({ error: "No image file provided" });
      }

      // Verify product exists
      const productCheck = await pool.query(`SELECT id FROM catalog.products WHERE id = $1`, [id]);
      if (productCheck.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }

      // Generate base filename (without extension)
      const timestamp = Date.now();
      const randomString = Math.random().toString(36).substring(2, 8);
      const baseFilename = `catalog_${id}_${timestamp}_${randomString}`;
      
      // Process image to generate optimized versions
      const { processImage, generateImageVersionFilenames } = await import('./image-processor.js');
      const processed = await processImage(req.file.buffer, {
        thumbnailSize: 80,
        mediumSize: 400,
        quality: 85,
        format: 'webp'
      });
      
      // Generate filenames for all versions
      const filenames = generateImageVersionFilenames(baseFilename, processed.format);
      
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      // Upload all three versions to separate subdirectories under products/images/
      const [originalUrl, thumbnailUrl, mediumUrl] = await Promise.all([
        adapter.upload({
          filename: filenames.original,
          buffer: processed.buffers.original,
          mimetype: `image/${processed.format}`,
          subfolder: 'products/images'
        }),
        adapter.upload({
          filename: filenames.thumbnail,
          buffer: processed.buffers.thumbnail,
          mimetype: `image/${processed.format}`,
          subfolder: 'products/images/thumbnails'
        }),
        adapter.upload({
          filename: filenames.medium,
          buffer: processed.buffers.medium,
          mimetype: `image/${processed.format}`,
          subfolder: 'products/images/medium'
        })
      ]);

      console.log('✅ Uploaded optimized images:');
      console.log('  - Original:', originalUrl);
      console.log('  - Thumbnail (80x80):', thumbnailUrl);
      console.log('  - Medium (400x400):', mediumUrl);

      // If this is set as primary, unset other primary images
      if (isPrimary === 'true' || isPrimary === true) {
        await pool.query(`
          UPDATE catalog.product_images
          SET is_primary = false
          WHERE product_id = $1
        `, [id]);
      }

      // Get next sort order
      const sortResult = await pool.query(`
        SELECT COALESCE(MAX(sort_order), -1) + 1 as next_order
        FROM catalog.product_images
        WHERE product_id = $1
      `, [id]);
      const nextSortOrder = sortResult.rows[0].next_order;

      // First image (sortOrder 0) is always primary
      const isFirstImage = nextSortOrder === 0;
      const shouldBePrimary = isFirstImage || isPrimary === 'true' || isPrimary === true;

      // Save image record with all versions
      const result = await pool.query(`
        INSERT INTO catalog.product_images (
          product_id, url, filename, thumbnail_url, medium_url, 
          image_type, alt_text, title, is_primary, sort_order
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING *
      `, [
        id,
        originalUrl,
        filenames.original,
        thumbnailUrl,
        mediumUrl,
        imageType || 'packshot',
        altText || '',
        imageTitle || '',
        shouldBePrimary,
        nextSortOrder
      ]);
      
      res.json(result.rows[0]);
    } catch (error: any) {
      console.error("❌ Error uploading product image:", error);
      res.status(500).json({ error: "Failed to upload image", details: error.message });
    }
  });

  // Update image metadata
  app.patch("/api/catalog-products/:id/images/:imageId", isAuthenticated, async (req, res) => {
    try {
      const { id, imageId } = req.params;
      const { imageType, altText, title: imageTitle, isPrimary, sortOrder, tags } = req.body;
      
      // Build update query dynamically
      const updates: string[] = [];
      const values: any[] = [];
      let paramCount = 1;

      if (imageType !== undefined) {
        updates.push(`image_type = $${paramCount++}`);
        values.push(imageType);
      }
      if (altText !== undefined) {
        updates.push(`alt_text = $${paramCount++}`);
        values.push(altText);
      }
      if (imageTitle !== undefined) {
        updates.push(`title = $${paramCount++}`);
        values.push(imageTitle);
      }
      if (tags !== undefined) {
        updates.push(`tags = $${paramCount++}`);
        values.push(tags);
      }
      if (sortOrder !== undefined) {
        updates.push(`sort_order = $${paramCount++}`);
        values.push(sortOrder);
      }
      if (isPrimary !== undefined) {
        // If setting as primary, unset others first
        if (isPrimary) {
          await pool.query(`
            UPDATE catalog.product_images
            SET is_primary = false
            WHERE product_id = $1 AND id != $2
          `, [id, imageId]);
        }
        updates.push(`is_primary = $${paramCount++}`);
        values.push(isPrimary);
      }

      if (updates.length === 0) {
        return res.status(400).json({ error: "No fields to update" });
      }

      updates.push(`updated_at = NOW()`);
      values.push(id, imageId);

      const result = await pool.query(`
        UPDATE catalog.product_images
        SET ${updates.join(', ')}
        WHERE product_id = $${paramCount++} AND id = $${paramCount++}
        RETURNING *
      `, values);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Image not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating product image:", error);
      res.status(500).json({ error: "Failed to update image" });
    }
  });

  // Delete image
  app.delete("/api/catalog-products/:id/images/:imageId", isAuthenticated, async (req, res) => {
    try {
      const { id, imageId } = req.params;
      
      // Get image info before deleting
      const imageResult = await pool.query(`
        SELECT * FROM catalog.product_images
        WHERE product_id = $1 AND id = $2
      `, [id, imageId]);
      
      if (imageResult.rows.length === 0) {
        return res.status(404).json({ error: "Image not found" });
      }

      const image = imageResult.rows[0];
      
      // Delete from database
      await pool.query(`DELETE FROM catalog.product_images WHERE id = $1`, [imageId]);
      
      // Delete file from SFTP storage
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      try {
        // Extract filename from URL (handle both local and SFTP URLs)
        const filename = image.filename || image.url.split('/').pop();
        await adapter.delete(`products/images/${filename}`);
        console.log(`✅ Deleted product image from SFTP: ${filename}`);
      } catch (fileError) {
        console.error("⚠️ Error deleting image file from SFTP (continuing):", fileError);
        // Continue even if file deletion fails
      }
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting product image:", error);
      res.status(500).json({ error: "Failed to delete image" });
    }
  });

  // Reorder images
  app.patch("/api/catalog-products/:id/images/reorder", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { imageIds } = req.body; // Array of image IDs in new order
      
      if (!Array.isArray(imageIds)) {
        return res.status(400).json({ error: "imageIds must be an array" });
      }

      // Update sort_order for each image
      for (let i = 0; i < imageIds.length; i++) {
        await pool.query(`
          UPDATE catalog.product_images
          SET sort_order = $1, updated_at = NOW()
          WHERE product_id = $2 AND id = $3
        `, [i, id, imageIds[i]]);
      }
      
      // Return updated images
      const result = await pool.query(`
        SELECT * FROM catalog.product_images
        WHERE product_id = $1
        ORDER BY sort_order ASC
      `, [id]);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error reordering images:", error);
      res.status(500).json({ error: "Failed to reorder images" });
    }
  });

  // Product Addons
  app.get("/api/product-addons", isAuthenticated, async (req, res) => {
    try {
      const { type } = req.query;
      
      let query = `
        SELECT 
          pa.id,
          pa.addon_type AS "addonType",
          pa.name,
          pa.code,
          pa.description,
          pa.html_template AS "htmlTemplate",
          pa.attributes,
          pa.is_active AS "isActive",
          pa.created_at AS "createdAt",
          pa.updated_at AS "updatedAt",
          COUNT(DISTINCT ai.id) as "imagesCount"
        FROM catalog.product_addons pa
        LEFT JOIN catalog.addon_images ai ON pa.id = ai.addon_id
      `;
      
      const params = [];
      if (type) {
        query += ` WHERE pa.addon_type = $1`;
        params.push(type);
      }
      
      query += ` GROUP BY pa.id ORDER BY pa.created_at DESC`;
      
      const result = await pool.query(query, params);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching addons:", error);
      res.status(500).json({ error: "Failed to fetch addons" });
    }
  });

  app.post("/api/product-addons", isAuthenticated, async (req, res) => {
    try {
      const { addonType, name, code, description, htmlTemplate, attributes, isActive } = req.body;
      
      const result = await pool.query(`
        INSERT INTO catalog.product_addons (addon_type, name, code, description, html_template, attributes, is_active)
        VALUES ($1, $2, $3, $4, $5, $6, $7)
        RETURNING 
          id,
          addon_type AS "addonType",
          name,
          code,
          description,
          html_template AS "htmlTemplate",
          attributes,
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [addonType, name, code, description, htmlTemplate, JSON.stringify(attributes || {}), isActive !== undefined ? isActive : true]);
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating addon:", error);
      res.status(500).json({ error: "Failed to create addon" });
    }
  });

  app.post("/api/product-addons/import-csv", isAuthenticated, async (req, res) => {
    try {
      const { csvData } = req.body;
      
      if (!csvData || typeof csvData !== 'string') {
        return res.status(400).json({ error: "CSV data is required" });
      }

      // Parse CSV
      const lines = csvData.trim().split('\n');
      if (lines.length < 2) {
        return res.status(400).json({ error: "CSV file is empty or invalid" });
      }

      // Auto-detect separator (comma or semicolon)
      const firstLine = lines[0];
      const separator = firstLine.includes(';') ? ';' : ',';
      
      // Parse header
      const header = firstLine.split(separator).map(h => h.trim());
      
      // Validate required columns
      const requiredColumns = ['addonType', 'name'];
      const missingColumns = requiredColumns.filter(col => !header.includes(col));
      if (missingColumns.length > 0) {
        return res.status(400).json({ 
          error: `Missing required columns: ${missingColumns.join(', ')}` 
        });
      }

      const results = {
        success: 0,
        failed: 0,
        errors: [] as Array<{ row: number; error: string; data?: any }>
      };

      // Process each row
      for (let i = 1; i < lines.length; i++) {
        const line = lines[i].trim();
        if (!line) continue; // Skip empty lines

        try {
          // Simple CSV parsing (handles quoted values)
          const values: string[] = [];
          let currentValue = '';
          let insideQuotes = false;
          
          for (let j = 0; j < line.length; j++) {
            const char = line[j];
            if (char === '"') {
              insideQuotes = !insideQuotes;
            } else if (char === separator && !insideQuotes) {
              values.push(currentValue.trim());
              currentValue = '';
            } else {
              currentValue += char;
            }
          }
          values.push(currentValue.trim());

          // Map values to columns
          const row: any = {};
          header.forEach((col, index) => {
            row[col] = values[index] || null;
          });

          // Validate addon type
          const validTypes = ['fabric', 'board', 'certificate', 'accessory', 'component'];
          if (!row.addonType || !validTypes.includes(row.addonType)) {
            results.errors.push({
              row: i + 1,
              error: `Invalid addon type: ${row.addonType}. Must be one of: ${validTypes.join(', ')}`,
              data: row
            });
            results.failed++;
            continue;
          }

          // Validate name
          if (!row.name || row.name.trim() === '') {
            results.errors.push({
              row: i + 1,
              error: 'Name is required',
              data: row
            });
            results.failed++;
            continue;
          }

          // Parse isActive
          let isActive = true;
          if (row.isActive !== null && row.isActive !== undefined) {
            const activeStr = row.isActive.toLowerCase();
            isActive = activeStr === 'true' || activeStr === '1' || activeStr === 'yes';
          }

          // Insert into database
          await pool.query(`
            INSERT INTO catalog.product_addons (
              addon_type, name, code, description, html_template, attributes, is_active
            )
            VALUES ($1, $2, $3, $4, $5, $6, $7)
          `, [
            row.addonType,
            row.name,
            row.code || null,
            row.description || null,
            row.htmlTemplate || null,
            JSON.stringify({}),
            isActive
          ]);

          results.success++;

          results.success++;
        } catch (error) {
          console.error(`❌ Error importing row ${i + 1}:`, error);
          results.errors.push({
            row: i + 1,
            error: error instanceof Error ? error.message : 'Unknown error',
            data: line
          });
          results.failed++;
        }
      }

      res.json(results);
    } catch (error) {
      console.error("❌ Error importing CSV:", error);
      res.status(500).json({ error: "Failed to import CSV" });
    }
  });

  app.get("/api/product-addons/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT 
          pa.*,
          COALESCE(
            json_agg(
              json_build_object(
                'id', ai.id,
                'url', ai.url,
                'filename', ai.filename,
                'altText', ai.alt_text,
                'title', ai.title,
                'sortOrder', ai.sort_order
              ) ORDER BY ai.sort_order
            ) FILTER (WHERE ai.id IS NOT NULL),
            '[]'
          ) as images
        FROM catalog.product_addons pa
        LEFT JOIN catalog.addon_images ai ON pa.id = ai.addon_id
        WHERE pa.id = $1
        GROUP BY pa.id
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Addon not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error fetching addon:", error);
      res.status(500).json({ error: "Failed to fetch addon" });
    }
  });

  app.patch("/api/product-addons/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { addonType, name, code, description, htmlTemplate, attributes, isActive } = req.body;
      
      const updates = [];
      const values = [];
      let paramIndex = 1;
      
      if (addonType !== undefined) {
        updates.push(`addon_type = $${paramIndex++}`);
        values.push(addonType);
      }
      if (name !== undefined) {
        updates.push(`name = $${paramIndex++}`);
        values.push(name);
      }
      if (code !== undefined) {
        updates.push(`code = $${paramIndex++}`);
        values.push(code);
      }
      if (description !== undefined) {
        updates.push(`description = $${paramIndex++}`);
        values.push(description);
      }
      if (htmlTemplate !== undefined) {
        updates.push(`html_template = $${paramIndex++}`);
        values.push(htmlTemplate);
      }
      if (attributes !== undefined) {
        updates.push(`attributes = $${paramIndex++}`);
        values.push(JSON.stringify(attributes));
      }
      if (isActive !== undefined) {
        updates.push(`is_active = $${paramIndex++}`);
        values.push(isActive);
      }
      
      if (updates.length === 0) {
        return res.status(400).json({ error: "No fields to update" });
      }
      
      updates.push(`updated_at = NOW()`);
      values.push(id);
      
      const result = await pool.query(`
        UPDATE catalog.product_addons
        SET ${updates.join(", ")}
        WHERE id = $${paramIndex}
        RETURNING 
          id,
          addon_type AS "addonType",
          name,
          code,
          description,
          html_template AS "htmlTemplate",
          attributes,
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, values);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Addon not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating addon:", error);
      res.status(500).json({ error: "Failed to update addon" });
    }
  });

  app.delete("/api/product-addons/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        DELETE FROM catalog.product_addons WHERE id = $1 RETURNING *
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Addon not found" });
      }
      
      res.json({ success: true, deleted: result.rows[0] });
    } catch (error) {
      console.error("❌ Error deleting addon:", error);
      res.status(500).json({ error: "Failed to delete addon" });
    }
  });

  // Addon Images
  app.get("/api/product-addons/:id/images", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT 
          id,
          addon_id AS "addonId",
          url,
          filename,
          alt_text AS "altText",
          title,
          sort_order AS "sortOrder",
          created_at AS "createdAt"
        FROM catalog.addon_images
        WHERE addon_id = $1
        ORDER BY sort_order ASC
      `, [id]);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching addon images:", error);
      res.status(500).json({ error: "Failed to fetch addon images" });
    }
  });

  app.post("/api/product-addons/:id/images", isAuthenticated, uploadAddonImage.single('image'), async (req, res) => {
    try {
      const { id } = req.params;
      const { altText, title: imageTitle } = req.body;
      
      if (!req.file) {
        return res.status(400).json({ error: "No image file provided" });
      }

      // Verify addon exists
      const addonCheck = await pool.query(`SELECT id FROM catalog.product_addons WHERE id = $1`, [id]);
      if (addonCheck.rows.length === 0) {
        // Delete uploaded file if addon doesn't exist
        await fs.unlink(req.file.path);
        return res.status(404).json({ error: "Addon not found" });
      }

      // Get next sort order
      const sortResult = await pool.query(`
        SELECT COALESCE(MAX(sort_order), -1) + 1 as next_order
        FROM catalog.addon_images
        WHERE addon_id = $1
      `, [id]);
      const nextSortOrder = sortResult.rows[0].next_order;

      // Save image record
      const imageUrl = `/uploads/addons/${req.file.filename}`;
      const result = await pool.query(`
        INSERT INTO catalog.addon_images (addon_id, url, filename, alt_text, title, sort_order)
        VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING 
          id,
          addon_id AS "addonId",
          url,
          filename,
          alt_text AS "altText",
          title,
          sort_order AS "sortOrder",
          created_at AS "createdAt"
      `, [id, imageUrl,
        unit, req.file.filename, altText || '', imageTitle || '', nextSortOrder]);

      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error uploading addon image:", error);
      res.status(500).json({ error: "Failed to upload image" });
    }
  });

  app.delete("/api/product-addons/:id/images/:imageId", isAuthenticated, async (req, res) => {
    try {
      const { id, imageId } = req.params;
      
      // Get image info before deletion
      const imageResult = await pool.query(`
        SELECT * FROM catalog.addon_images WHERE id = $1 AND addon_id = $2
      `, [imageId, id]);
      
      if (imageResult.rows.length === 0) {
        return res.status(404).json({ error: "Image not found" });
      }
      
      const image = imageResult.rows[0];
      
      // Delete from database
      await pool.query(`DELETE FROM catalog.addon_images WHERE id = $1`, [imageId]);
      
      // Delete file from disk
      try {
        const filePath = path.join(process.cwd(), 'uploads', 'addons', image.filename);
        await fs.unlink(filePath);
      } catch (fsError) {
        console.error("Error deleting file:", fsError);
        // Continue even if file deletion fails
      }
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting addon image:", error);
      res.status(500).json({ error: "Failed to delete image" });
    }
  });

  // Product Addon Assignments
  app.get("/api/catalog-products/:id/addons", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT 
          paa.*,
          pa.addon_type,
          pa.name,
          pa.code,
          pa.description,
          pa.html_template,
          pa.attributes,
          pa.is_active
        FROM catalog.product_addon_assignments paa
        JOIN catalog.product_addons pa ON paa.addon_id = pa.id
        WHERE paa.product_id = $1
        ORDER BY paa.sort_order ASC
      `, [id]);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching product addons:", error);
      res.status(500).json({ error: "Failed to fetch product addons" });
    }
  });

  app.post("/api/catalog-products/:id/addons", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { addonId, sortOrder, customAttributes } = req.body;
      
      const result = await pool.query(`
        INSERT INTO catalog.product_addon_assignments (product_id, addon_id, sort_order, custom_attributes)
        VALUES ($1, $2, $3, $4)
        RETURNING *
      `, [id, addonId, sortOrder || 0, JSON.stringify(customAttributes || {})]);
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error assigning addon:", error);
      res.status(500).json({ error: "Failed to assign addon" });
    }
  });

  app.delete("/api/catalog-products/:id/addons/:assignmentId", isAuthenticated, async (req, res) => {
    try {
      const { id, assignmentId } = req.params;
      
      const result = await pool.query(`
        DELETE FROM catalog.product_addon_assignments 
        WHERE product_id = $1 AND id = $2
        RETURNING *
      `, [id, assignmentId]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Assignment not found" });
      }
      
      res.json({ success: true, deleted: result.rows[0] });
    } catch (error) {
      console.error("❌ Error removing addon assignment:", error);
      res.status(500).json({ error: "Failed to remove addon assignment" });
    }
  });

  // ==================== TEMPLATE CATEGORIES ====================
  
  app.get("/api/template-categories", isAuthenticated, async (req, res) => {
    try {
      const categories = await getAllTemplateCategories();
      res.json(categories);
    } catch (error) {
      console.error("❌ Error fetching template categories:", error);
      res.status(500).json({ error: "Failed to fetch template categories" });
    }
  });

  app.get("/api/template-categories/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const category = await getTemplateCategoryById(parseInt(id, 10));
      
      if (!category) {
        return res.status(404).json({ error: "Template category not found" });
      }
      
      res.json(category);
    } catch (error) {
      console.error("❌ Error fetching template category:", error);
      res.status(500).json({ error: "Failed to fetch template category" });
    }
  });

  app.post("/api/template-categories", requireAdmin, async (req, res) => {
    try {
      const { name, description, sortOrder } = req.body;
      
      if (!name) {
        return res.status(400).json({ error: "Name is required" });
      }
      
      const category = await createTemplateCategory({ name, description, sortOrder });
      res.json(category);
    } catch (error) {
      console.error("❌ Error creating template category:", error);
      res.status(500).json({ error: "Failed to create template category" });
    }
  });

  app.put("/api/template-categories/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const { name, description, sortOrder } = req.body;
      
      const category = await updateTemplateCategory(parseInt(id, 10), { name, description, sortOrder });
      
      if (!category) {
        return res.status(404).json({ error: "Template category not found" });
      }
      
      res.json(category);
    } catch (error) {
      console.error("❌ Error updating template category:", error);
      res.status(500).json({ error: "Failed to update template category" });
    }
  });

  app.delete("/api/template-categories/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const deleted = await deleteTemplateCategory(parseInt(id, 10));
      
      if (!deleted) {
        return res.status(404).json({ error: "Template category not found" });
      }
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting template category:", error);
      res.status(500).json({ error: "Failed to delete template category" });
    }
  });

  // ==================== DESCRIPTION TEMPLATES ====================
  
  app.get("/api/description-templates", isAuthenticated, async (req, res) => {
    try {
      const { categoryId } = req.query;
      const templates = await getAllDescriptionTemplates(
        categoryId ? parseInt(categoryId as string, 10) : undefined
      );
      res.json(templates);
    } catch (error) {
      console.error("❌ Error fetching templates:", error);
      res.status(500).json({ error: "Failed to fetch templates" });
    }
  });

  app.get("/api/description-templates/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const template = await getDescriptionTemplateById(parseInt(id, 10));
      
      if (!template) {
        return res.status(404).json({ error: "Template not found" });
      }
      
      res.json(template);
    } catch (error) {
      console.error("❌ Error fetching template:", error);
      res.status(500).json({ error: "Failed to fetch template" });
    }
  });

  app.post("/api/description-templates", isAuthenticated, async (req, res) => {
    try {
      const { name, categoryId, templateType, contentDoc, htmlContent, variables, isGlobal, isActive } = req.body;
      
      // Walidacja wymaganych pól
      if (!name || typeof name !== 'string' || name.trim() === '') {
        return res.status(400).json({ error: "Name is required and must be a non-empty string" });
      }
      if (!templateType || typeof templateType !== 'string') {
        return res.status(400).json({ error: "templateType is required and must be a string" });
      }
      if (!htmlContent || typeof htmlContent !== 'string') {
        return res.status(400).json({ error: "htmlContent is required and must be a string" });
      }
      
      // Walidacja typów opcjonalnych pól
      if (contentDoc !== undefined && (typeof contentDoc !== 'object' || contentDoc === null || Array.isArray(contentDoc))) {
        return res.status(400).json({ error: "contentDoc must be an object (not array or null) if provided" });
      }
      if (variables !== undefined && (typeof variables !== 'object' || variables === null || Array.isArray(variables))) {
        return res.status(400).json({ error: "variables must be an object (not array or null) if provided" });
      }
      if (categoryId !== undefined && (typeof categoryId !== 'number' || categoryId < 1)) {
        return res.status(400).json({ error: "categoryId must be a positive number if provided" });
      }
      
      const userId = (req.user as any)?.id;
      
      const template = await createDescriptionTemplate({
        name: name.trim(),
        categoryId,
        templateType,
        contentDoc,
        htmlContent,
        variables,
        createdBy: userId,
        isGlobal,
        isActive,
      });
      
      res.json(template);
    } catch (error) {
      console.error("❌ Error creating template:", error);
      res.status(500).json({ error: "Failed to create template" });
    }
  });

  app.put("/api/description-templates/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { name, categoryId, templateType, contentDoc, htmlContent, variables, isGlobal, isActive } = req.body;
      
      // Walidacja typów jeśli wartości są podane
      if (name !== undefined && (typeof name !== 'string' || name.trim() === '')) {
        return res.status(400).json({ error: "name must be a non-empty string if provided" });
      }
      if (templateType !== undefined && typeof templateType !== 'string') {
        return res.status(400).json({ error: "templateType must be a string if provided" });
      }
      if (htmlContent !== undefined && typeof htmlContent !== 'string') {
        return res.status(400).json({ error: "htmlContent must be a string if provided" });
      }
      if (contentDoc !== undefined && (typeof contentDoc !== 'object' || contentDoc === null || Array.isArray(contentDoc))) {
        return res.status(400).json({ error: "contentDoc must be an object (not array or null) if provided" });
      }
      if (variables !== undefined && (typeof variables !== 'object' || variables === null || Array.isArray(variables))) {
        return res.status(400).json({ error: "variables must be an object (not array or null) if provided" });
      }
      if (categoryId !== undefined && categoryId !== null && (typeof categoryId !== 'number' || categoryId < 1)) {
        return res.status(400).json({ error: "categoryId must be a positive number or null if provided" });
      }
      
      const template = await updateDescriptionTemplate(parseInt(id, 10), {
        name: name ? name.trim() : undefined,
        categoryId,
        templateType,
        contentDoc,
        htmlContent,
        variables,
        isGlobal,
        isActive,
      });
      
      if (!template) {
        return res.status(404).json({ error: "Template not found" });
      }
      
      res.json(template);
    } catch (error) {
      console.error("❌ Error updating template:", error);
      res.status(500).json({ error: "Failed to update template" });
    }
  });

  app.delete("/api/description-templates/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      const deleted = await deleteDescriptionTemplate(parseInt(id, 10));
      
      if (!deleted) {
        return res.status(404).json({ error: "Template not found" });
      }
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting template:", error);
      res.status(500).json({ error: "Failed to delete template" });
    }
  });

  app.post("/api/description-templates/:id/duplicate", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const userId = (req.user as any)?.id;
      
      // Pobierz oryginalny szablon
      const result = await pool.query(`
        SELECT * FROM catalog.description_templates 
        WHERE id = $1
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Template not found" });
      }
      
      const original = result.rows[0];
      
      // Utwórz kopię z nową nazwą
      const duplicateName = `${original.name} (kopia)`;
      
      const duplicateResult = await pool.query(`
        INSERT INTO catalog.description_templates 
          (category_id, name, template_type, content_doc, html_content, variables, is_global, is_active, created_by)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, true, $10)
        RETURNING *
      `, [
        original.category_id,
        duplicateName,
        original.template_type,
        original.content_doc,
        original.html_content,
        original.variables,
        original.is_global,
        original.is_active,
        userId
      ]);
      
      console.log(`✅ Template duplicated: ${original.name} -> ${duplicateName}`);
      res.json(duplicateResult.rows[0]);
    } catch (error) {
      console.error("❌ Error duplicating template:", error);
      res.status(500).json({ error: "Failed to duplicate template" });
    }
  });

  // ==================== PRODUCT-TEMPLATE ASSIGNMENTS ====================
  
  app.get("/api/catalog-products/:id/templates", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const templates = await getProductTemplates(parseInt(id, 10));
      res.json(templates);
    } catch (error) {
      console.error("❌ Error fetching product templates:", error);
      res.status(500).json({ error: "Failed to fetch product templates" });
    }
  });

  app.post("/api/catalog-products/:id/templates", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { templateId, renderConfig } = req.body;
      
      // Walidacja wymaganych pól
      if (!templateId || typeof templateId !== 'number' || templateId < 1) {
        return res.status(400).json({ error: "templateId is required and must be a positive number" });
      }
      
      // Walidacja opcjonalnych pól
      if (renderConfig !== undefined && (typeof renderConfig !== 'object' || renderConfig === null || Array.isArray(renderConfig))) {
        return res.status(400).json({ error: "renderConfig must be an object (not array or null) if provided" });
      }
      
      const assignment = await assignTemplateToProduct(parseInt(id, 10), templateId, renderConfig);
      res.json(assignment);
    } catch (error) {
      console.error("❌ Error assigning template:", error);
      res.status(500).json({ error: "Failed to assign template" });
    }
  });

  app.put("/api/product-templates/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { renderConfig } = req.body;
      
      // Walidacja renderConfig
      if (renderConfig === undefined || renderConfig === null || typeof renderConfig !== 'object' || Array.isArray(renderConfig)) {
        return res.status(400).json({ error: "renderConfig is required and must be an object (not array or null)" });
      }
      
      // Sprawdzenie czy assignment istnieje i czy należy do zalogowanego użytkownika (poprzez produkt)
      const existingCheck = await pool.query(`
        SELECT pt.id, pt.product_id, p.user_id 
        FROM catalog.product_description_templates pt
        JOIN products p ON pt.product_id = p.id
        WHERE pt.id = $1
      `, [id]);
      
      if (existingCheck.rows.length === 0) {
        return res.status(404).json({ error: "Product template assignment not found" });
      }
      
      // Tylko admin lub właściciel produktu może edytować
      const userId = (req.user as any)?.id;
      const userRole = (req.user as any)?.role;
      if (userRole !== 'admin' && existingCheck.rows[0].user_id !== userId) {
        return res.status(403).json({ error: "You don't have permission to edit this product template assignment" });
      }
      
      const assignment = await updateProductTemplateConfig(parseInt(id, 10), renderConfig);
      
      res.json(assignment);
    } catch (error) {
      console.error("❌ Error updating product template:", error);
      res.status(500).json({ error: "Failed to update product template" });
    }
  });

  app.delete("/api/product-templates/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const deleted = await removeTemplateFromProduct(parseInt(id, 10));
      
      if (!deleted) {
        return res.status(404).json({ error: "Product template assignment not found" });
      }
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error removing product template:", error);
      res.status(500).json({ error: "Failed to remove product template" });
    }
  });

  // ==================== AI GENERATION REQUESTS ====================
  
  app.post("/api/catalog-products/:id/ai-generate", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { templateId, tone = 'professional', language = 'pl', maxLength = 500 } = req.body;
      const userId = (req.user as any)?.id;
      const productId = parseInt(id, 10);
      
      // Pobierz dane produktu
      const productResult = await pool.query(`SELECT * FROM catalog.products WHERE id = $1`, [productId]);
      if (productResult.rows.length === 0) {
        return res.status(404).json({ error: "Product not found" });
      }
      const product = productResult.rows[0];
      
      // Pobierz zdjęcia produktu
      const imagesResult = await pool.query(`
        SELECT url, is_primary, sort_order
        FROM catalog.product_images
        WHERE product_id = $1
        ORDER BY sort_order ASC, id ASC
      `, [productId]);
      
      // Pobierz dane matrycy jeśli produkt ma matrixId (aby dostać hasWarranty)
      let hasWarranty = true; // Domyślnie true
      if (product.matrix_id) {
        const matrixResult = await pool.query(`
          SELECT has_warranty
          FROM product_creator.product_matrices
          WHERE id = $1
        `, [product.matrix_id]);
        
        if (matrixResult.rows.length > 0) {
          hasWarranty = matrixResult.rows[0].has_warranty ?? true;
        }
      }
      
      // Pobierz szablon jeśli podano
      let templateHtml = '';
      let templateName = '';
      let template: any = null;
      if (templateId) {
        template = await getDescriptionTemplateById(templateId);
        if (template) {
          templateHtml = template.htmlContent || '';
          templateName = template.name || '';
        }
      }
      
      // Utwórz request w bazie (status: pending -> processing)
      const request = await createAiGenerationRequest({
        productId,
        templateId,
        prompt: `Generating description for ${product.title}`,
        createdBy: userId,
      });
      
      // Aktualizuj status na processing
      await updateAiGenerationRequest(request.id, { status: 'processing' });
      
      try {
        let finalDescription = '';
        
        // CASE 1: Template provided - render template with product data, AI (if needed), images, and accessories
        if (templateId && templateHtml) {
          console.log(`📝 Using template "${templateName}" for product: ${product.title}`);
          
          // 1. Extract accessory tags from template and fetch groups
          let accessoriesData: { groups: any[] } = { groups: [] };
          
          try {
            const accessoryCodes = extractAccessoryTags(templateHtml);
            
            if (accessoryCodes.length > 0) {
              console.log(`🏷️  Found ${accessoryCodes.length} accessory tags in template: ${accessoryCodes.join(', ')}`);
              
              // Fetch accessory groups by codes found in template
              const accessoriesResult = await pool.query(`
                SELECT 
                  ag.id as group_id,
                  ag.name as group_name,
                  ag.code as group_code,
                  ag.category as group_category,
                  ag.description as group_description,
                  json_agg(
                    json_build_object(
                      'id', a.id,
                      'name', a.name,
                      'code', a.code,
                      'alpmaCode', a.alpma_code,
                      'description', a.description,
                      'imageUrl', a.image_url,
                      'displayOrder', a.display_order
                    ) ORDER BY a.display_order
                  ) FILTER (WHERE a.id IS NOT NULL) as items
                FROM catalog.accessory_groups ag
                LEFT JOIN catalog.accessories a ON a.group_id = ag.id AND a.is_active = true
                WHERE ag.is_active = true AND LOWER(ag.code) = ANY($1::text[])
                GROUP BY ag.id, ag.name, ag.code, ag.category, ag.description, ag.display_order
                ORDER BY ag.display_order
              `, [accessoryCodes]);
              
              accessoriesData.groups = accessoriesResult.rows.map((row: any) => ({
                groupId: row.group_id,
                groupName: row.group_name,
                groupCode: row.group_code,
                code: row.group_code,
                name: row.group_name,
                category: row.group_category,
                description: row.group_description,
                items: row.items || []
              }));
              
              console.log(`📦 Loaded ${accessoriesData.groups.length} accessory groups from template tags`);
            }
          } catch (error) {
            console.error('❌ Error fetching accessories from template tags:', error);
          }
          
          // 2. Check if template uses AI tags
          const aiTagPattern = /\{\{ai-(intro|features|safety|care|warranty)\}\}/i;
          const hasAiTags = aiTagPattern.test(templateHtml);
          
          const aiSections: Record<string, string> = {};
          
          if (hasAiTags) {
            console.log(`🤖 Template uses AI tags - generating AI content`);
            
            // Prepare product data for AI
            const { generateProductDescription } = await import('./ai-service');
            const productData = {
              sku: product.sku,
              title: product.title,
              shortDescription: product.short_description,
              color: product.color,
              material: product.material,
              dimensions: {
                width: product.width,
                height: product.height,
                depth: product.length,
              },
              hasWarranty: hasWarranty,
            };
            
            // Generate AI descriptions for all sections (single call)
            const aiResult = await generateProductDescription({
              productData,
              templateHtml: '', // Don't use templateHtml as context - AI uses global prompts
              tone,
              language,
              maxLength,
              accessories: accessoriesData,
            });
            
            console.log(`✅ AI generated ${aiResult.totalTokens} tokens`);
            
            // Parse AI response to extract individual sections
            const fullAiDescription = aiResult.description;
            
            // Extract sections using patterns
            const sectionPatterns: Record<string, RegExp[]> = {
              'ai-intro': [
                /<h2[^>]*>Wprowadzenie[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
                /<div[^>]*>\s*<h2[^>]*>Wprowadzenie[^<]*<\/h2>([\s\S]*?)<\/div>/i,
              ],
              'ai-features': [
                /<h2[^>]*>Cechy[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
                /<div[^>]*>\s*<h2[^>]*>Cechy[^<]*<\/h2>([\s\S]*?)<\/div>/i,
              ],
              'ai-safety': [
                /<h2[^>]*>Bezpieczeństwo[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
                /<div[^>]*>\s*<h2[^>]*>Bezpieczeństwo[^<]*<\/h2>([\s\S]*?)<\/div>/i,
              ],
              'ai-care': [
                /<h2[^>]*>Pielęgnacja[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
                /<div[^>]*>\s*<h2[^>]*>Pielęgnacja[^<]*<\/h2>([\s\S]*?)<\/div>/i,
              ],
              'ai-warranty': [
                /<h2[^>]*>Gwarancja[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
                /<div[^>]*>\s*<h2[^>]*>Gwarancja[^<]*<\/h2>([\s\S]*?)<\/div>/i,
              ],
            };
            
            for (const [sectionKey, patterns] of Object.entries(sectionPatterns)) {
              // Skip warranty if disabled
              if (sectionKey === 'ai-warranty' && !hasWarranty) {
                console.log(`⏭️  Skipping warranty section (hasWarranty=false)`);
                continue;
              }
              
              let extracted = '';
              for (const pattern of patterns) {
                const match = fullAiDescription.match(pattern);
                if (match && match[1]) {
                  extracted = `<div><h2>${sectionKey.replace('ai-', '').charAt(0).toUpperCase() + sectionKey.replace('ai-', '').slice(1)}</h2>${match[1].trim()}</div>`;
                  console.log(`✂️  Extracted section "${sectionKey}" (${extracted.length} chars)`);
                  break;
                }
              }
              
              if (extracted) {
                aiSections[sectionKey] = extracted;
              } else {
                console.warn(`⚠️  Could not extract section "${sectionKey}" - using empty`);
                aiSections[sectionKey] = '';
              }
            }
          } else {
            console.log(`ℹ️  Template doesn't use AI tags - skipping AI generation (saving tokens!)`);
          }
          
          // 5. Render template with all tags replaced
          const decodeHtmlEntities = (html: string): string => {
            return html
              .replace(/&lt;/g, '<')
              .replace(/&gt;/g, '>')
              .replace(/&amp;/g, '&')
              .replace(/&quot;/g, '"')
              .replace(/&#39;/g, "'");
          };
          
          let renderedHtml = decodeHtmlEntities(templateHtml);
          
          // Replace basic product tags: {{title}}, {{width}}, {{height}}, {{length}}, {{color}}, etc.
          const formatDimension = (value: number | null): string => {
            return value !== null ? `${value} cm` : '-';
          };
          
          const basicTags: Record<string, string> = {
            'title': product.title || '',
            'sku': product.sku || '',
            'color': product.color || '',
            'material': product.material || '',
            'width': formatDimension(product.width),
            'height': formatDimension(product.height),
            'length': formatDimension(product.length),
            'depth': formatDimension(product.length), // alias for length
            'price': product.base_price ? `${product.base_price} PLN` : '',
            'dimensions': [
              product.length ? `${product.length} cm` : null,
              product.width ? `${product.width} cm` : null,
              product.height ? `${product.height} cm` : null
            ].filter(Boolean).join(' × '),
          };
          
          // Replace basic tags
          for (const [key, value] of Object.entries(basicTags)) {
            const pattern = new RegExp(`\\{\\{${key}\\}\\}`, 'gi');
            renderedHtml = renderedHtml.replace(pattern, value);
          }
          
          // Replace AI section tags: {{ai-intro}}, {{ai-features}}, etc.
          for (const [key, value] of Object.entries(aiSections)) {
            const pattern = new RegExp(`\\{\\{${key}\\}\\}`, 'g');
            renderedHtml = renderedHtml.replace(pattern, value);
          }
          
          // Replace accessory tags: {{akcesorium-GROUP_CODE:gridN}}
          const accessoryTagRegex = /\{\{(?:akcesorium|okucia|akcesoria)-([a-zA-Z0-9_-]+)(?::grid(\d+))?\}\}/g;
          renderedHtml = renderedHtml.replace(accessoryTagRegex, (_match: string, groupCode: string, gridCols?: string) => {
            const group = accessoriesData.groups.find(
              (g: any) => g.groupCode === groupCode || g.code === groupCode
            );
            
            if (!group || !group.items || group.items.length === 0) {
              console.log(`⚠️  Accessory group "${groupCode}" not found or empty`);
              return '';
            }
            
            const columns = gridCols ? parseInt(gridCols, 10) : 1;
            const itemWidth = columns === 1 ? '100%' : `${(100 / columns) - 2}%`;
            
            // IMPORTANT: Don't use position: absolute for images - TipTap removes wrapper divs
            return `
              <div class="accessory-group" data-group="${groupCode}" style="margin: 20px 0;">
                <h3 style="margin-bottom: 15px; font-size: 1.2em;">${group.groupName || group.name}</h3>
                <div class="accessory-items" style="display: flex; flex-wrap: wrap; gap: 10px; margin: 0 -5px;">
                  ${group.items.map((item: any) => `
                    <div class="accessory-item" style="width: ${itemWidth}; min-width: ${columns > 3 ? '150px' : '200px'}; box-sizing: border-box; padding: 10px; border: 1px solid #e0e0e0; border-radius: 4px; background: #fafafa;">
                      ${item.imageUrl ? `
                        <img src="${item.imageUrl}" alt="${item.name}" class="accessory-image" style="width: 100%; height: auto; display: block; margin-bottom: 10px; border-radius: 4px; object-fit: contain; max-height: 300px;" />
                      ` : ''}
                      <div style="text-align: center;">
                        <h4 style="margin: 0 0 5px 0; font-size: 0.9em; font-weight: 600;">${item.name}</h4>
                        ${item.code ? `<p style="margin: 0 0 5px 0; font-size: 0.8em; color: #666;">(${item.code})</p>` : ''}
                        ${item.description ? `<p style="margin: 0; font-size: 0.8em; color: #888;">${item.description}</p>` : ''}
                      </div>
                    </div>
                  `).join('')}
                </div>
              </div>
            `.trim();
          });
          
          // Replace image tags: {img1 w=100}, {img2 w=75}, {img3 w=50}, {img4 w=25}
          const productImages = imagesResult.rows.map((img: any) => img.url);
          const imageTagRegex = /\{img(\d+)\s+w=(\d+)\}/gi;
          renderedHtml = renderedHtml.replace(imageTagRegex, (_match: string, imgIndex: string, widthParam: string) => {
            const index = parseInt(imgIndex, 10) - 1; // 0-based
            const width = parseInt(widthParam, 10);
            
            if (![100, 75, 50, 25].includes(width)) {
              console.warn(`⚠️  Invalid width parameter: ${widthParam}`);
              return '';
            }
            
            if (index < 0 || index >= productImages.length) {
              console.log(`⚠️  Image ${imgIndex} not found (has ${productImages.length} images)`);
              return '';
            }
            
            const imageEntry = productImages[index];
            
            // Handle potential new format: {url, thumbnailUrl, mediumUrl}
            let imageUrl: string;
            if (typeof imageEntry === 'object' && imageEntry.url) {
              // Use medium size for template images (better quality than thumbnail)
              imageUrl = imageEntry.mediumUrl || imageEntry.url;
            } else {
              // Current format: plain string URL
              imageUrl = imageEntry as string;
            }
            
            const widthPercent = width;
            
            return `
              <div style="width: ${widthPercent}%; margin: 15px auto;">
                <img src="${imageUrl}" alt="Product image ${imgIndex}" style="width: 100%; height: auto; display: block; border-radius: 4px;" />
              </div>
            `.trim();
          });
          
          finalDescription = renderedHtml;
          
        } else {
          // CASE 2: No template - generate plain AI description (legacy mode)
          console.log(`🤖 Generating plain AI description (no template)`);
          
          const { generateProductDescription } = await import('./ai-service');
          const productData = {
            sku: product.sku,
            title: product.title,
            shortDescription: product.short_description,
            color: product.color,
            material: product.material,
            dimensions: {
              width: product.width,
              height: product.height,
              depth: product.length,
            },
            hasWarranty: hasWarranty,
          };
          
          const aiResult = await generateProductDescription({
            productData,
            templateHtml: '',
            tone,
            language,
            maxLength,
          });
          
          finalDescription = aiResult.description;
        }
        
        // Aktualizuj request z wynikiem
        const updatedRequest = await updateAiGenerationRequest(request.id, {
          status: 'completed',
          response: finalDescription,
          costMetadata: {
            model: 'gpt-4',
            promptTokens: 0,
            completionTokens: 0,
            totalTokens: 0,
            cost: 0,
            templateName,
            tone,
            language,
          },
        });
        
        res.json(updatedRequest);
      } catch (aiError) {
        console.error("❌ AI generation failed:", aiError);
        
        await updateAiGenerationRequest(request.id, {
          status: 'failed',
          response: `Error: ${aiError instanceof Error ? aiError.message : 'Unknown error'}`,
        });
        
        res.status(500).json({ 
          error: "AI generation failed", 
          details: aiError instanceof Error ? aiError.message : 'Unknown error' 
        });
      }
    } catch (error) {
      console.error("❌ Error in AI generation endpoint:", error);
      res.status(500).json({ error: "Failed to generate description" });
    }
  });

  app.get("/api/catalog-products/:id/ai-history", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { limit } = req.query;
      
      const history = await getAiGenerationHistory(
        parseInt(id, 10),
        limit ? parseInt(limit as string, 10) : 10
      );
      
      res.json(history);
    } catch (error) {
      console.error("❌ Error fetching AI generation history:", error);
      res.status(500).json({ error: "Failed to fetch AI generation history" });
    }
  });

  // ==================== Product Creator Dictionaries ====================
  
  // GET /api/dictionaries - Lista wszystkich pozycji słowników (z opcjonalnym filtrowaniem po typie)
  app.get("/api/dictionaries", isAuthenticated, async (req, res) => {
    try {
      const { type } = req.query;
      const dictionaryType = type as DictionaryType | undefined;
      
      const dictionaries = await storage.getDictionaries(dictionaryType);
      res.json(dictionaries);
    } catch (error) {
      console.error("❌ Error fetching dictionaries:", error);
      res.status(500).json({ error: "Failed to fetch dictionaries" });
    }
  });

  // GET /api/dictionaries/:id - Pobierz pojedynczą pozycję słownika
  app.get("/api/dictionaries/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const dictionary = await storage.getDictionaryById(parseInt(id, 10));
      
      if (!dictionary) {
        return res.status(404).json({ error: "Dictionary entry not found" });
      }
      
      res.json(dictionary);
    } catch (error) {
      console.error("❌ Error fetching dictionary entry:", error);
      res.status(500).json({ error: "Failed to fetch dictionary entry" });
    }
  });

  // POST /api/dictionaries/init - Automatyczna inicjalizacja słowników
  app.post("/api/dictionaries/init", isAuthenticated, async (req, res) => {
    try {
      console.log("🚀 Starting dictionary initialization...");
      
      const initData = [
        // Component CZ1 (główne typy elementów)
        { dictionaryType: 'component_cz1', code: 'BOK-L', name: 'BOK-L', readableName: 'Bok lewy', shortName: 'BOK-L', isActive: true },
        { dictionaryType: 'component_cz1', code: 'BOK-P', name: 'BOK-P', readableName: 'Bok prawy', shortName: 'BOK-P', isActive: true },
        { dictionaryType: 'component_cz1', code: 'POLKA', name: 'POLKA', readableName: 'Półka', shortName: 'POLKA', isActive: true },
        { dictionaryType: 'component_cz1', code: 'DRZWI', name: 'DRZWI', readableName: 'Drzwi', shortName: 'DRZWI', isActive: true },
        { dictionaryType: 'component_cz1', code: 'FRONT', name: 'FRONT', readableName: 'Front', shortName: 'FRONT', isActive: true },
        { dictionaryType: 'component_cz1', code: 'PLECY', name: 'PLECY', readableName: 'Plecy', shortName: 'PLECY', isActive: true },
        { dictionaryType: 'component_cz1', code: 'WG', name: 'WG', readableName: 'Wieszak górny', shortName: 'WG', isActive: true },
        { dictionaryType: 'component_cz1', code: 'WD', name: 'WD', readableName: 'Wieszak dolny', shortName: 'WD', isActive: true },
        
        // Kolory podstawowe
        { dictionaryType: 'color', code: 'BIALY', name: 'Biały', readableName: 'Biały', shortName: 'BIAŁY', isActive: true },
        { dictionaryType: 'color', code: 'KASZMIR', name: 'Kaszmir', readableName: 'Kaszmir', shortName: 'KASZMIR', isActive: true },
        { dictionaryType: 'color', code: 'WOTAN', name: 'Wotan', readableName: 'Wotan', shortName: 'WOTAN', isActive: true },
        { dictionaryType: 'color', code: 'CZARNY', name: 'Czarny', readableName: 'Czarny', shortName: 'CZARNY', isActive: true },
        { dictionaryType: 'color', code: 'TABOR', name: 'Tabor', readableName: 'Tabor', shortName: 'TABOR', isActive: true },
        { dictionaryType: 'color', code: 'DEB', name: 'Dąb', readableName: 'Dąb', shortName: 'DĄB', isActive: true },
        { dictionaryType: 'color', code: 'ORZECH', name: 'Orzech', readableName: 'Orzech', shortName: 'ORZECH', isActive: true },
        { dictionaryType: 'color', code: 'SZARY', name: 'Szary', readableName: 'Szary', shortName: 'SZARY', isActive: true },

        // Powody złomowania - produkty
        { dictionaryType: 'scrap_reason_product', code: 'USZKODZONY_TRANSPORT', name: 'Uszkodzony transport', readableName: 'Uszkodzony podczas transportu', shortName: 'Transport', isActive: true },
        { dictionaryType: 'scrap_reason_product', code: 'BLAD_PRODUKCJI', name: 'Błąd produkcji', readableName: 'Błąd podczas produkcji', shortName: 'Produkcja', isActive: true },
        { dictionaryType: 'scrap_reason_product', code: 'WADA_MATERIALU', name: 'Wada materiału', readableName: 'Wada materiału źródłowego', shortName: 'Materiał', isActive: true },
        { dictionaryType: 'scrap_reason_product', code: 'USZKODZENIE_MAGAZYN', name: 'Uszkodzenie w magazynie', readableName: 'Uszkodzony w magazynie', shortName: 'Magazyn', isActive: true },
        { dictionaryType: 'scrap_reason_product', code: 'ZWROT_ZNISZCZONY', name: 'Zwrot zniszczony', readableName: 'Zniszczony przy zwrocie', shortName: 'Zwrot', isActive: true },
        { dictionaryType: 'scrap_reason_product', code: 'WADA_OKUCIA', name: 'Wada okucia', readableName: 'Wadliwe okucie', shortName: 'Okucie', isActive: true },
        { dictionaryType: 'scrap_reason_product', code: 'INNE', name: 'Inne', readableName: 'Inny powód', shortName: 'Inne', isActive: true },

        // Powody złomowania - formatki
        { dictionaryType: 'scrap_reason_cutting', code: 'BLAD_CIECIA', name: 'Błąd cięcia', readableName: 'Błąd podczas cięcia', shortName: 'Cięcie', isActive: true },
        { dictionaryType: 'scrap_reason_cutting', code: 'WADA_PLYTY', name: 'Wada płyty', readableName: 'Wada w płycie meblowej', shortName: 'Płyta', isActive: true },
        { dictionaryType: 'scrap_reason_cutting', code: 'USZKODZENIE', name: 'Uszkodzenie', readableName: 'Uszkodzenie mechaniczne', shortName: 'Uszkodz.', isActive: true },
        { dictionaryType: 'scrap_reason_cutting', code: 'ZLY_WYMIAR', name: 'Zły wymiar', readableName: 'Nieprawidłowy wymiar', shortName: 'Wymiar', isActive: true },
        { dictionaryType: 'scrap_reason_cutting', code: 'ODPRYSK', name: 'Odprysk', readableName: 'Odpryski na krawędzi', shortName: 'Odprysk', isActive: true },
        { dictionaryType: 'scrap_reason_cutting', code: 'PRZEBARWIENIE', name: 'Przebarwienie', readableName: 'Przebarwienie powierzchni', shortName: 'Kolor', isActive: true },
        { dictionaryType: 'scrap_reason_cutting', code: 'INNE', name: 'Inne', readableName: 'Inny powód', shortName: 'Inne', isActive: true },
      ];
      
      const created: ProductCreatorDictionary[] = [];
      const skipped: string[] = [];
      
      for (const item of initData) {
        try {
          const newDict = await storage.createDictionary(item);
          created.push(newDict);
          console.log(`✅ Created: ${item.dictionaryType} - ${item.code}`);
        } catch (error: any) {
          // Skip duplicates
          if (error.code === '23505') {
            skipped.push(`${item.dictionaryType}:${item.code}`);
            console.log(`⏭️  Skipped (already exists): ${item.dictionaryType} - ${item.code}`);
          } else {
            throw error;
          }
        }
      }
      
      console.log(`✅ Dictionary initialization complete: ${created.length} created, ${skipped.length} skipped`);
      res.json({ 
        success: true, 
        created: created.length, 
        skipped: skipped.length,
        items: created 
      });
    } catch (error) {
      console.error("❌ Error initializing dictionaries:", error);
      res.status(500).json({ error: "Failed to initialize dictionaries" });
    }
  });

  // POST /api/dictionaries - Dodaj nową pozycję słownika
  app.post("/api/dictionaries", isAuthenticated, async (req, res) => {
    try {
      console.log("📝 Creating dictionary entry with data:", JSON.stringify(req.body, null, 2));
      const newDictionary = await storage.createDictionary(req.body);
      console.log("✅ Dictionary entry created successfully:", newDictionary.id);
      res.status(201).json(newDictionary);
    } catch (error: any) {
      console.error("❌ Error creating dictionary entry:", error);
      console.error("❌ Request body was:", JSON.stringify(req.body, null, 2));
      
      // Handle duplicate key error
      if (error.code === '23505') {
        const code = req.body.code || '';
        return res.status(409).json({ 
          error: `Kod "${code}" już istnieje w tej kategorii słownika` 
        });
      }
      
      res.status(500).json({ error: "Nie udało się utworzyć wpisu słownika" });
    }
  });

  // PATCH /api/dictionaries/:id - Aktualizuj pozycję słownika
  app.patch("/api/dictionaries/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const updated = await storage.updateDictionary(parseInt(id, 10), req.body);
      res.json(updated);
    } catch (error) {
      console.error("❌ Error updating dictionary entry:", error);
      
      if (error instanceof Error && error.message.includes('not found')) {
        return res.status(404).json({ error: error.message });
      }
      
      res.status(500).json({ error: "Failed to update dictionary entry" });
    }
  });

  // DELETE /api/dictionaries/:id - Usuń pozycję słownika
  app.delete("/api/dictionaries/:id", requireAdmin, async (req, res) => {
    try {
      const { id } = req.params;
      await storage.deleteDictionary(parseInt(id, 10));
      res.status(204).send();
    } catch (error) {
      console.error("❌ Error deleting dictionary entry:", error);
      res.status(500).json({ error: "Failed to delete dictionary entry" });
    }
  });

  // Helper function to parse CSV line (handles quoted fields)
  function parseCSVLine(line: string): string[] {
    const result: string[] = [];
    let current = '';
    let inQuotes = false;
    
    for (let i = 0; i < line.length; i++) {
      const char = line[i];
      
      if (char === '"') {
        inQuotes = !inQuotes;
      } else if (char === ',' && !inQuotes) {
        result.push(current.trim());
        current = '';
      } else {
        current += char;
      }
    }
    
    result.push(current.trim());
    return result.map(val => {
      // Remove surrounding quotes
      if (val.startsWith('"') && val.endsWith('"')) {
        val = val.slice(1, -1);
      }
      // Handle NULL values
      if (val === 'NULL' || val === 'null' || val === '') {
        return null as any;
      }
      return val;
    });
  }

  // POST /api/dictionaries/import-csv - Import pozycji słownika z CSV
  app.post("/api/dictionaries/import-csv", isAuthenticated, async (req, res) => {
    try {
      const { dictionaryType, csvData } = req.body;
      
      if (!dictionaryType || !csvData) {
        return res.status(400).json({ error: "dictionaryType and csvData are required" });
      }
      
      const lines = csvData.trim().split('\n').filter((l: string) => l.trim());
      if (lines.length === 0) {
        return res.status(400).json({ error: "CSV data is empty" });
      }
      
      const imported: ProductCreatorDictionary[] = [];
      const errors: string[] = [];
      
      // Check if first line is a header
      const firstLine = parseCSVLine(lines[0]);
      const hasHeader = firstLine.some((val: string) => 
        val && ['id', 'nazwa', 'name', 'code', 'opis', 'description'].includes(val.toLowerCase())
      );
      
      let headers: string[] = [];
      let startIndex = 0;
      
      if (hasHeader) {
        headers = firstLine.map(h => h?.toLowerCase() || '');
        startIndex = 1;
      }
      
      for (let i = startIndex; i < lines.length; i++) {
        const line = lines[i].trim();
        if (!line) continue;
        
        const parts = parseCSVLine(line);
        
        try {
          let code = '';
          let name = '';
          let readableName: string | null = null;
          let description: string | null = null;
          let colorHex: string | null = null;
          let category: string | null = null;
          let isActive = true;
          
          if (hasHeader) {
            // Map columns based on headers
            const codeIdx = headers.findIndex(h => ['code', 'kod', 'nazwa'].includes(h));
            const nameIdx = headers.findIndex(h => ['name', 'nazwa'].includes(h));
            const descIdx = headers.findIndex(h => ['description', 'opis'].includes(h));
            const activeIdx = headers.findIndex(h => ['active', 'aktywny', 'is_active'].includes(h));
            const readableIdx = headers.findIndex(h => ['readable_name', 'readable', 'czytelna_nazwa'].includes(h));
            const categoryIdx = headers.findIndex(h => ['category', 'kategoria'].includes(h));
            const colorIdx = headers.findIndex(h => ['color', 'color_hex', 'kolor'].includes(h));
            
            code = parts[codeIdx] || parts[nameIdx] || '';
            name = parts[nameIdx] || parts[codeIdx] || '';
            description = parts[descIdx] || null;
            readableName = parts[readableIdx] || null;
            category = parts[categoryIdx] || null;
            colorHex = parts[colorIdx] || null;
            
            if (activeIdx >= 0 && parts[activeIdx]) {
              const activeVal = parts[activeIdx].toLowerCase();
              isActive = activeVal === 'true' || activeVal === '1' || activeVal === 'yes' || activeVal === 'tak';
            }
          } else {
            // Simple format: code, name, readableName, description, colorHex, category
            code = parts[0] || '';
            name = parts[1] || parts[0];
            readableName = parts[2] || null;
            description = parts[3] || null;
            colorHex = parts[4] || null;
            category = parts[5] || null;
          }
          
          if (!code || !name) {
            errors.push(`Line ${i + 1}: Missing code or name`);
            continue;
          }
          
          const dictionary = await storage.createDictionary({
            dictionaryType: dictionaryType as DictionaryType,
            code,
            name,
            readableName,
            description,
            color: colorHex,
            category,
            sortOrder: i - startIndex,
            isActive,
          });
          imported.push(dictionary);
        } catch (error) {
          errors.push(`Line ${i + 1}: ${error instanceof Error ? error.message : 'Unknown error'}`);
        }
      }
      
      res.json({
        imported: imported.length,
        errors,
        data: imported,
      });
    } catch (error) {
      console.error("❌ Error importing CSV:", error);
      res.status(500).json({ error: "Failed to import CSV data" });
    }
  });

  // Multer configuration for dictionary images
  const dictionaryImageStorage = multer.diskStorage({
    destination: async (req, file, cb) => {
      const uploadDir = path.join(process.cwd(), 'uploads', 'dictionaries');
      await fs.mkdir(uploadDir, { recursive: true });
      cb(null, uploadDir);
    },
    filename: (req, file, cb) => {
      const uniqueSuffix = `${Date.now()}-${Math.round(Math.random() * 1E9)}`;
      cb(null, `${uniqueSuffix}-${file.originalname}`);
    }
  });

  const uploadDictionaryImage = multer({
    storage: dictionaryImageStorage,
    limits: {
      fileSize: 10 * 1024 * 1024, // 10MB limit
    },
    fileFilter: (req, file, cb) => {
      const allowedTypes = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
      if (allowedTypes.includes(file.mimetype)) {
        cb(null, true);
      } else {
        cb(new Error('Invalid file type. Only JPEG, PNG, GIF, and WebP are allowed.'));
      }
    }
  });

  // GET /api/dictionaries/:id/images - Pobierz obrazy słownika
  app.get("/api/dictionaries/:id/images", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(
        `SELECT id, dictionary_id, filename, url, alt_text, title, created_at 
         FROM product_creator.dictionary_images 
         WHERE dictionary_id = $1 
         ORDER BY created_at DESC`,
        [id]
      );
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching dictionary images:", error);
      res.status(500).json({ error: "Failed to fetch dictionary images" });
    }
  });

  // POST /api/dictionaries/:id/images - Upload obrazu do słownika
  app.post("/api/dictionaries/:id/images", isAuthenticated, uploadDictionaryImage.single('image'), async (req, res) => {
    try {
      const { id } = req.params;
      const { altText, title: imageTitle } = req.body;

      if (!req.file) {
        return res.status(400).json({ error: "No image file provided" });
      }

      const dictionary = await storage.getDictionaryById(parseInt(id, 10));
      if (!dictionary) {
        // Clean up uploaded file
        await fs.unlink(req.file.path);
        return res.status(404).json({ error: "Dictionary entry not found" });
      }

      const imageUrl = `/uploads/dictionaries/${req.file.filename}`;

      const result = await pool.query(
        `INSERT INTO product_creator.dictionary_images (dictionary_id, filename, url, alt_text, title, created_at)
         VALUES ($1, $2, $3, $4, $5, $6)
         RETURNING id, dictionary_id, filename, url, alt_text, title, created_at`,
        [id, req.file.filename, imageUrl,
        unit, altText || null, imageTitle || null, new Date()]
      );

      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error uploading dictionary image:", error);
      // Try to clean up uploaded file on error
      if (req.file) {
        try {
          await fs.unlink(req.file.path);
        } catch (unlinkError) {
          console.error("❌ Error deleting file after upload failure:", unlinkError);
        }
      }
      res.status(500).json({ error: "Failed to upload image" });
    }
  });

  // DELETE /api/dictionaries/:id/images/:imageId - Usuń obraz słownika
  app.delete("/api/dictionaries/:id/images/:imageId", isAuthenticated, async (req, res) => {
    try {
      const { imageId } = req.params;

      // Get image info
      const imageResult = await pool.query(
        `SELECT filename FROM product_creator.dictionary_images WHERE id = $1`,
        [imageId]
      );

      if (imageResult.rows.length === 0) {
        return res.status(404).json({ error: "Image not found" });
      }

      const image = imageResult.rows[0];

      // Delete from database
      await pool.query(
        `DELETE FROM product_creator.dictionary_images WHERE id = $1`,
        [imageId]
      );

      // Delete file from disk
      try {
        const filePath = path.join(process.cwd(), 'uploads', 'dictionaries', image.filename);
        await fs.unlink(filePath);
      } catch (fsError) {
        console.error("Error deleting file:", fsError);
      }

      res.status(204).send();
    } catch (error) {
      console.error("❌ Error deleting dictionary image:", error);
      res.status(500).json({ error: "Failed to delete image" });
    }
  });

  // GET /api/dictionaries/formatki-prefixes - Zwróć dynamiczną listę prefiksów formatek ze słownika component_cz1
  app.get("/api/dictionaries/formatki-prefixes", isAuthenticated, async (req, res) => {
    try {
      // Fetch active component_cz1 entries
      const result = await pool.query(
        `SELECT code FROM product_creator.dictionaries
         WHERE dictionary_type = 'component_cz1'
         AND is_active = true
         ORDER BY code`
      );

      // Convert codes to prefixes by appending '-' to each code
      const prefixes = result.rows.map((row: any) => {
        const code = row.code.trim();
        // Add '-' only if it's not already there
        return code.endsWith('-') ? code : `${code}-`;
      });

      // Return unique prefixes
      const uniquePrefixes = Array.from(new Set(prefixes));
      
      res.json({ prefixes: uniquePrefixes });
    } catch (error) {
      console.error("❌ Error fetching formatki prefixes:", error);
      res.status(500).json({ error: "Failed to fetch formatki prefixes" });
    }
  });

  // ==================== Product Matrices ====================
  
  // GET /api/product-matrices - Lista wszystkich matryce produktów z sortowaniem, filtrowaniem i paginacją
  app.get("/api/product-matrices", isAuthenticated, async (req, res) => {
    try {
      const { 
        sort = 'updatedAt', 
        order = 'desc', 
        search = '', 
        productType = '', 
        limit = '25', 
        offset = '0' 
      } = req.query;

      // Validate and sanitize order parameter to prevent SQL injection
      const validatedOrder: 'asc' | 'desc' = (order as string).toLowerCase() === 'asc' ? 'asc' : 'desc';

      // Validate sort field (whitelist)
      const allowedSortFields = ['name', 'productType', 'updatedAt', 'createdAt'];
      const validatedSort = allowedSortFields.includes(sort as string) ? (sort as string) : 'updatedAt';

      const matrices = await storage.getProductMatrices({
        sort: validatedSort,
        order: validatedOrder,
        search: search as string,
        productType: productType as string,
        limit: parseInt(limit as string, 10),
        offset: parseInt(offset as string, 10)
      });

      // Convert colorImages URLs to SFTP for all matrices
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      const matricesWithConvertedUrls = matrices.map(matrix => {
        if (matrix.colorImages && typeof matrix.colorImages === 'object') {
          const colorImages: Record<string, any[]> = {};
          
          for (const [colorName, urls] of Object.entries(matrix.colorImages)) {
            if (Array.isArray(urls)) {
              colorImages[colorName] = urls.map((entry: any) => {
                // Handle new format: {url, thumbnailUrl, mediumUrl}
                if (typeof entry === 'object' && entry.url) {
                  return entry; // Already in correct format, return as-is
                }
                
                // Handle old format: plain string URL
                const url = entry as string;
                if (!url.startsWith('http')) {
                  const filename = url.split('/').pop();
                  return adapter.getUrl(`products/images/${filename}`);
                }
                return url;
              });
            }
          }
          
          return { ...matrix, colorImages };
        }
        return matrix;
      });

      // Get total count for pagination
      const totalCount = await storage.getProductMatricesCount({
        search: search as string,
        productType: productType as string
      });

      res.json({
        matrices: matricesWithConvertedUrls,
        total: totalCount,
        limit: parseInt(limit as string, 10),
        offset: parseInt(offset as string, 10)
      });
    } catch (error) {
      console.error("❌ Error fetching product matrices:", error);
      res.status(500).json({ error: "Failed to fetch product matrices" });
    }
  });

  // GET /api/product-matrices/:id - Pobierz pojedynczą matrycę
  app.get("/api/product-matrices/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const matrix = await storage.getProductMatrixById(parseInt(id, 10));
      
      if (!matrix) {
        return res.status(404).json({ error: "Product matrix not found" });
      }
      
      // Convert local colorImages URLs to SFTP URLs
      if (matrix.colorImages && typeof matrix.colorImages === 'object') {
        const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
        const adapter = await getFileStorageAdapter();
        
        const colorImages: Record<string, any[]> = {};
        
        for (const [colorName, urls] of Object.entries(matrix.colorImages)) {
          if (Array.isArray(urls)) {
            colorImages[colorName] = urls.map((entry: any) => {
              // Handle new format: {url, thumbnailUrl, mediumUrl}
              if (typeof entry === 'object' && entry.url) {
                return entry; // Already in correct format, return as-is
              }
              
              // Handle old format: plain string URL
              const url = entry as string;
              if (!url.startsWith('http')) {
                const filename = url.split('/').pop();
                return adapter.getUrl(`products/images/${filename}`);
              }
              return url;
            });
          }
        }
        
        matrix.colorImages = colorImages;
      }
      
      res.json(matrix);
    } catch (error) {
      console.error("❌ Error fetching product matrix:", error);
      res.status(500).json({ error: "Failed to fetch product matrix" });
    }
  });

  // POST /api/product-matrices - Dodaj nową matrycę
  app.post("/api/product-matrices", isAuthenticated, async (req, res) => {
    try {
      console.log("📝 Creating product matrix with data:", JSON.stringify(req.body, null, 2));
      const userId = (req.user as any)?.id;
      const matrixData = {
        ...req.body,
        createdBy: userId || null,
      };
      const newMatrix = await storage.createProductMatrix(matrixData);
      console.log("✅ Product matrix created successfully:", newMatrix.id);
      res.status(201).json(newMatrix);
    } catch (error) {
      console.error("❌ Error creating product matrix:", error);
      res.status(500).json({ error: "Failed to create product matrix" });
    }
  });

  // PUT /api/product-matrices/:id - Edytuj matrycę (pełna aktualizacja)
  app.put("/api/product-matrices/:id", isAuthenticated, async (req, res) => {
    console.log("🟢 PUT /api/product-matrices/:id ENDPOINT CALLED!");
    console.log("🟢 Request params:", req.params);
    console.log("🟢 Request body:", req.body);
    try {
      const { id } = req.params;
      console.log(`📝 Updating product matrix ${id} with data:`, JSON.stringify(req.body, null, 2));
      const updatedMatrix = await storage.updateProductMatrix(parseInt(id, 10), req.body);
      console.log("✅ Product matrix updated successfully:", updatedMatrix.id);
      console.log("✅ Updated data from DB:", JSON.stringify(updatedMatrix, null, 2));
      res.json(updatedMatrix);
    } catch (error) {
      console.error("❌ Error updating product matrix:", error);
      res.status(500).json({ error: "Failed to update product matrix" });
    }
  });

  // PATCH /api/product-matrices/:id - Edytuj matrycę (częściowa aktualizacja)
  app.patch("/api/product-matrices/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      console.log(`📝 Updating product matrix ${id} with data:`, JSON.stringify(req.body, null, 2));
      const updatedMatrix = await storage.updateProductMatrix(parseInt(id, 10), req.body);
      console.log("✅ Product matrix updated successfully:", updatedMatrix.id);
      res.json(updatedMatrix);
    } catch (error) {
      console.error("❌ Error updating product matrix:", error);
      res.status(500).json({ error: "Failed to update product matrix" });
    }
  });

  // DELETE /api/product-matrices/:id - Usuń matrycę
  app.delete("/api/product-matrices/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      await storage.deleteProductMatrix(parseInt(id, 10));
      console.log("✅ Product matrix deleted successfully:", id);
      res.status(204).send();
    } catch (error) {
      console.error("❌ Error deleting product matrix:", error);
      res.status(500).json({ error: "Failed to delete product matrix" });
    }
  });

  // GET /api/product-matrices/:id/products - Pobierz produkty wygenerowane z matrycy
  app.get("/api/product-matrices/:id/products", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(
        `SELECT id, sku, title, combination_key, base_price, is_active
         FROM catalog.products
         WHERE matrix_id = $1
         ORDER BY combination_key`,
        [parseInt(id, 10)]
      );
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching products from matrix:", error);
      res.status(500).json({ error: "Failed to fetch products from matrix" });
    }
  });

  // POST /api/product-matrices/:id/generate - Generuj produkty z matryce
  app.post("/api/product-matrices/:id/generate", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { sessionId } = req.body; // Get sessionId from request body
      
      console.log(`🔧 Generating products from matrix ID: ${id}`);
      if (sessionId) {
        emitGenerationLog(sessionId, `🔧 Rozpoczynanie generowania produktów z matrycy ID: ${id}`, 'info');
      }
      
      const matrix = await storage.getProductMatrixById(parseInt(id, 10));
      
      if (!matrix) {
        console.log(`❌ Matrix not found: ${id}`);
        if (sessionId) {
          emitGenerationLog(sessionId, `❌ Nie znaleziono matrycy: ${id}`, 'error');
        }
        return res.status(404).json({ error: "Product matrix not found" });
      }

      console.log(`📋 Matrix found: ${matrix.name}`);
      if (sessionId) {
        emitGenerationLog(sessionId, `📋 Znaleziono matrycę: ${matrix.name}`, 'success');
        emitGenerationLog(sessionId, `📊 Wymiary: ${matrix.lengths?.length || 0} długości × ${matrix.widths?.length || 0} szerokości × ${matrix.heights?.length || 0} wysokości`, 'info');
        emitGenerationLog(sessionId, `🎨 Kolory: ${Array.isArray(matrix.colors) ? matrix.colors.length : 0}`, 'info');
      }

      // Generate products based on matrix with sessionId for logs
      const generatedProducts = await generateProductsFromMatrix(matrix, sessionId);
      
      console.log(`✅ Generated ${generatedProducts?.length || 0} products`);
      if (sessionId) {
        emitGenerationLog(sessionId, `✅ Wygenerowano ${generatedProducts?.length || 0} produktów`, 'success');
      }
      
      // Update colorLastGenerated timestamps for generated colors
      const selectedColorsArray = matrix.selectedColors || [];
      const colorsToUse = selectedColorsArray.length > 0 
        ? selectedColorsArray 
        : (matrix.colors || []);
      
      if (colorsToUse.length > 0) {
        const currentTimestamp = new Date().toISOString();
        const colorLastGenerated: Record<string, string> = (matrix.colorLastGenerated as Record<string, string>) || {};
        
        // Update timestamp for each generated color
        colorsToUse.forEach((color: string) => {
          colorLastGenerated[color] = currentTimestamp;
        });
        
        // Save updated timestamps to database
        await pool.query(
          `UPDATE product_creator.product_matrices 
           SET color_last_generated = $1, updated_at = NOW()
           WHERE id = $2`,
          [JSON.stringify(colorLastGenerated), parseInt(id, 10)]
        );
        
        console.log(`📅 Updated generation timestamps for ${colorsToUse.length} colors`);
        if (sessionId) {
          emitGenerationLog(sessionId, `📅 Zaktualizowano daty generacji dla ${colorsToUse.length} kolorów`, 'info');
        }
      }
      
      res.json({
        success: true,
        productsGenerated: generatedProducts?.length || 0,
        products: generatedProducts || []
      });
    } catch (error) {
      console.error("❌ Error generating products from matrix:", error);
      const { sessionId } = req.body;
      if (sessionId) {
        emitGenerationLog(sessionId, `❌ Błąd generowania produktów: ${error instanceof Error ? error.message : 'Nieznany błąd'}`, 'error');
      }
      res.status(500).json({ error: "Failed to generate products" });
    }
  });

  // POST /api/product-matrices/:id/regenerate-descriptions - Regeneruj opisy dla wszystkich produktów z matrycy
  app.post("/api/product-matrices/:id/regenerate-descriptions", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { sessionId } = req.body; // Get sessionId for log streaming
      
      console.log(`🔄 Regenerating descriptions for all products from matrix ID: ${id}`);
      if (sessionId) {
        emitGenerationLog(sessionId, `🔄 Rozpoczynanie regeneracji opisów dla matrycy ID: ${id}`, 'info');
      }
      
      const matrix = await storage.getProductMatrixById(parseInt(id, 10));
      
      if (!matrix) {
        console.log(`❌ Matrix not found: ${id}`);
        if (sessionId) {
          emitGenerationLog(sessionId, `❌ Nie znaleziono matrycy: ${id}`, 'error');
        }
        return res.status(404).json({ error: "Product matrix not found" });
      }

      if (!matrix.templateId) {
        if (sessionId) {
          emitGenerationLog(sessionId, `❌ Matryca nie ma przypisanego szablonu`, 'error');
        }
        return res.status(400).json({ error: "Matrix does not have a template assigned" });
      }

      console.log(`📋 Matrix found: ${matrix.name}, Template ID: ${matrix.templateId}`);
      if (sessionId) {
        emitGenerationLog(sessionId, `📋 Znaleziono matrycę: ${matrix.name}`, 'success');
        emitGenerationLog(sessionId, `📝 Szablon ID: ${matrix.templateId}`, 'info');
      }
      
      // Fetch all products generated from this matrix
      const result = await pool.query(`
        SELECT * FROM catalog.products
        WHERE matrix_id = $1 AND generated_from_matrix = true
        ORDER BY id
      `, [id]);
      
      const products = result.rows;
      console.log(`📦 Found ${products.length} products to regenerate`);
      if (sessionId) {
        emitGenerationLog(sessionId, `📦 Znaleziono ${products.length} produktów do regeneracji`, 'info');
      }
      
      if (products.length === 0) {
        return res.json({
          success: true,
          message: "No products found to regenerate",
          productsUpdated: 0
        });
      }

      // Fetch template
      const template = await getDescriptionTemplateById(matrix.templateId);
      
      if (!template || !template.htmlContent) {
        return res.status(404).json({ error: "Template not found or has no content" });
      }

      let updatedCount = 0;
      const errors: any[] = [];
      
      // Process each product
      for (const product of products) {
        try {
          // Extract dimension and color info from product
          const length = product.length ? parseFloat(product.length) : null;
          const width = product.width ? parseFloat(product.width) : null;
          const height = product.height ? parseFloat(product.height) : null;
          const color = product.color || '';
          
          // Find index of this color in the original colors array
          const colorIndex = color ? matrix.colors.indexOf(color) : -1;
          
          // Build size string
          const formatCm = (mm: number | null) => {
            if (!mm) return null;
            const cm = mm / 10;
            return cm % 1 === 0 ? cm.toFixed(0) : cm.toFixed(1);
          };
          
          const sizeParts = [];
          if (length) sizeParts.push(formatCm(length));
          if (width) sizeParts.push(formatCm(width));
          if (height) sizeParts.push(formatCm(height));
          const sizeStr = sizeParts.length > 0 ? `${sizeParts.join('×')} cm` : '';
          
          // Build template context (pass template HTML to extract accessory tags)
          const templateContext = await buildTemplateContext(matrix, {
            length,
            width,
            height,
            color,
            productName: product.title,
            productGroup: product.product_group,
            finalPrice: product.base_price ? parseFloat(product.base_price) : null,
            sizeStr
          }, template.htmlContent);
          
          // Update context with actual product data
          templateContext.product.alpmaCode = product.sku;
          templateContext.product.sku = product.sku;
          
          // Decode HTML entities (Tiptap encodes HTML tags as text)
          const decodeHtmlEntities = (html: string): string => {
            return html
              .replace(/&lt;/g, '<')
              .replace(/&gt;/g, '>')
              .replace(/&amp;/g, '&')
              .replace(/&quot;/g, '"')
              .replace(/&#39;/g, "'");
          };
          
          // Render template
          let renderedHtml = decodeHtmlEntities(template.htmlContent);
          
          // Replace AI section variables using HYPHEN notation: {{ai-intro}}, {{ai-features}}, etc.
          for (const [key, value] of Object.entries(templateContext)) {
            if (key.startsWith('ai-')) {
              const pattern = new RegExp(`\\{\\{${key}\\}\\}`, 'g');
              renderedHtml = renderedHtml.replace(pattern, value as string);
            }
          }
          
          // Replace accessory tags with grid support: {{akcesorium-CODE}} or {{akcesorium-CODE:grid3}}
          const accessoryTagRegex = /\{\{(?:akcesorium|okucia|akcesoria)-([a-zA-Z0-9_-]+)(?::grid(\d+))?\}\}/g;
          renderedHtml = renderedHtml.replace(accessoryTagRegex, (_match: string, groupCode: string, gridParam?: string) => {
            console.log(`🔍 Looking for accessory group: "${groupCode}" (grid: ${gridParam || 'none'})`);
            console.log(`📦 Available groups: ${templateContext.accessories.groups.map((g: any) => g.groupCode || g.code).join(', ')}`);
            
            const group = templateContext.accessories.groups.find(
              (g: any) => g.groupCode === groupCode || g.code === groupCode
            );
            
            if (!group || !group.items || group.items.length === 0) {
              console.log(`⚠️  Accessory group "${groupCode}" not found or empty`);
              return '';
            }
            
            console.log(`✅ Found accessory group "${groupCode}" with ${group.items.length} items`);
            
            // Handle grid layout parameter
            const columns = gridParam ? parseInt(gridParam, 10) : 1;
            const itemWidth = columns === 1 ? '100%' : `${(100 / columns) - 2}%`; // -2% for gap
            
            // Generate responsive grid with inline CSS for Allegro/Shoper compatibility
            // IMPORTANT: Don't use position: absolute for images - TipTap removes wrapper divs
            return `
              <div class="accessory-group" data-group="${groupCode}" style="margin: 20px 0;">
                <h3 style="margin-bottom: 15px; font-size: 1.2em;">${group.groupName || group.name}</h3>
                <div class="accessory-items" style="display: flex; flex-wrap: wrap; gap: 10px; margin: 0 -5px;">
                  ${group.items.map((item: any) => `
                    <div class="accessory-item" style="width: ${itemWidth}; min-width: ${columns > 3 ? '150px' : '200px'}; box-sizing: border-box; padding: 10px; border: 1px solid #e0e0e0; border-radius: 4px; background: #fafafa;">
                      ${item.imageUrl ? `
                        <img src="${item.imageUrl}" alt="${item.name}" class="accessory-image" style="width: 100%; height: auto; display: block; margin-bottom: 10px; border-radius: 4px; object-fit: contain; max-height: 300px;" />
                      ` : ''}
                      <div style="text-align: center;">
                        <h4 style="margin: 0 0 5px 0; font-size: 0.9em; font-weight: 600;">${item.name}</h4>
                        ${item.code ? `<p style="margin: 0 0 5px 0; font-size: 0.8em; color: #666;">(${item.code})</p>` : ''}
                        ${item.description ? `<p style="margin: 0; font-size: 0.8em; color: #888;">${item.description}</p>` : ''}
                      </div>
                    </div>
                  `).join('')}
                </div>
              </div>
            `.trim();
          });
          
          // Replace image tags: {img1 w=100}, {img2 w=75}, {img3 w=50}, {img4 w=25}
          // Get images for this color from matrix colorImages mapping (using index)
          const colorImages = (matrix as any).colorImages || {};
          const imagesForColor = colorIndex !== -1 ? (colorImages[colorIndex.toString()] || []) : [];
          const imageTagRegex = /\{img(\d+)\s+w=(\d+)\}/gi;
          renderedHtml = renderedHtml.replace(imageTagRegex, (_match: string, imgIndex: string, widthParam: string) => {
            const index = parseInt(imgIndex, 10) - 1; // Convert to 0-based index
            const width = parseInt(widthParam, 10);
            
            // Validate width is one of the allowed values
            if (![100, 75, 50, 25].includes(width)) {
              console.warn(`⚠️  Invalid width parameter: ${widthParam}. Allowed: 100, 75, 50, 25`);
              return '';
            }
            
            // Check if image exists at this index
            if (index < 0 || index >= imagesForColor.length) {
              console.log(`⚠️  Image ${imgIndex} not found for color "${color}" (has ${imagesForColor.length} images)`);
              return '';
            }
            
            const imageEntry = imagesForColor[index];
            
            // Handle new format: {url, thumbnailUrl, mediumUrl}
            let imageUrl: string;
            if (typeof imageEntry === 'object' && imageEntry.url) {
              // Use full quality images for template images
              imageUrl = imageEntry.url;
            } else {
              // Old format: plain string URL
              imageUrl = imageEntry as string;
            }
            
            console.log(`🖼️  Replacing {img${imgIndex} w=${width}} with image: ${imageUrl}`);
            
            // Generate responsive HTML with inline CSS for Allegro/Shoper compatibility
            // IMPORTANT: No div wrapper - TipTap removes it during editing
            // All styling must be on <img> tag directly
            // Use inline-block for horizontal layout when multiple images with small widths (25%, 50%, 75%)
            const display = width === 100 ? 'block' : 'inline-block';
            const margin = width === 100 ? '20px auto' : '10px 5px';
            return `<img src="${imageUrl}" alt="${product.title} - zdjęcie ${imgIndex}" class="resizable-image" style="width: ${width}%; max-width: ${width}%; height: auto; display: ${display}; border-radius: 4px; margin: ${margin}; vertical-align: top;" />`;
          });
          
          // Replace simple variables
          renderedHtml = renderedHtml
            .replace(/\{\{product\.alpmaCode\}\}/g, templateContext.product.alpmaCode || '')
            .replace(/\{\{product\.title\}\}/g, templateContext.product.title || '')
            .replace(/\{\{product\.color\}\}/g, templateContext.product.color || '')
            .replace(/\{\{product\.price\}\}/g, String(templateContext.product.price || ''))
            .replace(/\{\{product\.productType\}\}/g, templateContext.product.productType || '')
            .replace(/\{\{title\}\}/g, templateContext.product.title || '')
            .replace(/\{\{color\}\}/g, templateContext.color || '')
            .replace(/\{\{dimensions\}\}/g, templateContext.dimensions || '')
            .replace(/\{\{price\}\}/g, String(templateContext.price || ''))
            .replace(/\{\{basePrice\}\}/g, String(templateContext.variant?.basePrice || templateContext.price || ''))
            .replace(/\{\{length\}\}/g, String(templateContext.variant?.length || ''))
            .replace(/\{\{width\}\}/g, String(templateContext.variant?.width || ''))
            .replace(/\{\{height\}\}/g, String(templateContext.variant?.height || ''));
          
          // Convert to Tiptap JSON
          const { htmlToTiptapJson } = await import('./ai-service');
          const descriptionDoc = htmlToTiptapJson(renderedHtml);
          
          // Update product in database
          await pool.query(`
            UPDATE catalog.products
            SET 
              long_description_html = $1,
              long_description_doc = $2,
              updated_at = NOW()
            WHERE id = $3
          `, [renderedHtml, JSON.stringify(descriptionDoc), product.id]);
          
          updatedCount++;
          console.log(`✅ Updated product ${product.id}: ${product.title}`);
          if (sessionId) {
            emitGenerationLog(sessionId, `✅ Zaktualizowano produkt: ${product.title}`, 'success');
          }
        } catch (productError) {
          console.error(`❌ Error regenerating product ${product.id}:`, productError);
          if (sessionId) {
            emitGenerationLog(sessionId, `❌ Błąd regeneracji produktu ${product.title}: ${productError instanceof Error ? productError.message : 'Nieznany błąd'}`, 'error');
          }
          errors.push({
            productId: product.id,
            productTitle: product.title,
            error: productError instanceof Error ? productError.message : String(productError)
          });
        }
      }
      
      console.log(`🎉 Regeneration complete: ${updatedCount}/${products.length} products updated`);
      if (sessionId) {
        emitGenerationLog(sessionId, `🎉 Regeneracja zakończona: ${updatedCount}/${products.length} produktów zaktualizowanych`, 'success');
        if (errors.length > 0) {
          emitGenerationLog(sessionId, `⚠️ Wystąpiły błędy przy ${errors.length} produktach`, 'warning');
        }
      }
      
      res.json({
        success: true,
        productsUpdated: updatedCount,
        totalProducts: products.length,
        errors: errors.length > 0 ? errors : undefined
      });
    } catch (error) {
      console.error("❌ Error regenerating descriptions:", error);
      const { sessionId } = req.body;
      if (sessionId) {
        emitGenerationLog(sessionId, `❌ Błąd regeneracji opisów: ${error instanceof Error ? error.message : 'Nieznany błąd'}`, 'error');
      }
      res.status(500).json({ error: "Failed to regenerate descriptions" });
    }
  });

  // POST /api/product-matrices/:id/preview-generation - Podgląd generowania dla jednego produktu z raportem
  app.post("/api/product-matrices/:id/preview-generation", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { variantIndex = 0 } = req.body; // Which variant to preview (default: first)
      
      console.log(`🔍 Generating preview for matrix ID: ${id}, variant: ${variantIndex}`);
      
      const matrix = await storage.getProductMatrixById(parseInt(id, 10));
      
      if (!matrix) {
        console.log(`❌ Matrix not found: ${id}`);
        return res.status(404).json({ error: "Product matrix not found" });
      }

      if (!matrix.templateId) {
        return res.status(400).json({ error: "Matrix does not have a template assigned" });
      }

      console.log(`📋 Matrix found: ${matrix.name}, Template ID: ${matrix.templateId}`);

      // Get dimensions and colors
      const lengths = matrix.lengths || [];
      const widths = matrix.widths || [];
      const heights = matrix.heights || [];
      const colors = matrix.colors || [];
      const colorImages: Record<string, string[]> = (matrix.colorImages as Record<string, string[]>) || {};
      const productGroups = matrix.productGroups || [];
      const priceModifiers = matrix.priceModifiers || [];
      const priceModifierTypes = (matrix as any).priceModifierTypes || [];

      // Parse dimensions
      const uniqueLengths: (number | null)[] = lengths.length > 0 
        ? Array.from(new Set(lengths.map((l: string) => parseFloat(l)).filter((l: number) => !isNaN(l))))
        : [null];
      const uniqueWidths: (number | null)[] = widths.length > 0
        ? Array.from(new Set(widths.map((w: string) => parseFloat(w)).filter((w: number) => !isNaN(w))))
        : [null];
      const uniqueHeights: (number | null)[] = heights.length > 0
        ? Array.from(new Set(heights.map((h: string) => parseFloat(h)).filter((h: number) => !isNaN(h))))
        : [null];

      const colorsToUse = colors.length > 0 ? colors : [''];

      // Select first variant to preview
      const length = uniqueLengths[0];
      const width = uniqueWidths[0];
      const height = uniqueHeights[0];
      const color = colorsToUse[0];
      
      // Find index of this color in the original colors array
      const colorIndex = color ? matrix.colors.indexOf(color) : -1;

      // Build size string
      const formatCm = (mm: number) => {
        const cm = mm / 10;
        return cm % 1 === 0 ? cm.toFixed(0) : cm.toFixed(1);
      };
      
      const sizeParts = [];
      if (length) sizeParts.push(formatCm(length));
      if (width) sizeParts.push(formatCm(width));
      if (height) sizeParts.push(formatCm(height));
      const sizeStr = sizeParts.length > 0 ? `${sizeParts.join('×')} cm` : '';

      // Find matching row index for dimensions
      const findRowIndex = (l: number | null, w: number | null, h: number | null): number | null => {
        const normalizeValue = (val: string | number | null | undefined): number | null => {
          if (val === null || val === undefined) return null;
          if (typeof val === 'number') return val;
          if (typeof val === 'string') {
            if (val.trim() === '') return null;
            const num = parseFloat(val.trim());
            return isNaN(num) ? null : num;
          }
          return null;
        };
        
        const compareValues = (a: number | null, b: number | null): boolean => {
          if (a === null && b === null) return true;
          if (a === null || b === null) return false;
          return Math.abs(a - b) < 0.001;
        };
        
        for (let i = 0; i < Math.max(lengths.length, widths.length, heights.length); i++) {
          const rowLength = normalizeValue(lengths[i]);
          const rowWidth = normalizeValue(widths[i]);
          const rowHeight = normalizeValue(heights[i]);
          
          if (compareValues(rowLength, l) && compareValues(rowWidth, w) && compareValues(rowHeight, h)) {
            return i;
          }
        }
        return null;
      };

      const rowIndex = findRowIndex(length, width, height);
      const productGroup = rowIndex !== null && productGroups[rowIndex] ? productGroups[rowIndex] : null;
      const priceModifier = rowIndex !== null && priceModifiers[rowIndex] ? parseFloat(priceModifiers[rowIndex]) : 0;
      const priceModifierType = rowIndex !== null && priceModifierTypes[rowIndex] ? priceModifierTypes[rowIndex] : 'percent';

      // Build product name
      // Note: PANEL (tapicerowany) nie ma koloru - będzie zdefiniowany później przez klienta
      const nameParts: string[] = [];
      if (matrix.namePrefix) nameParts.push(matrix.namePrefix);
      if (matrix.productType) nameParts.push(matrix.productType);
      if (sizeStr) nameParts.push(sizeStr);
      if (matrix.nameSuffix) nameParts.push(matrix.nameSuffix);
      // Dla PANEL (panel tapicerowany) pomijamy kolor - będzie zdefiniowany przez klienta
      if (color && matrix.productType !== 'PANEL') nameParts.push(color);
      const productName = nameParts.filter(Boolean).join(' ');

      // Calculate final price
      let finalPrice = matrix.defaultPrice ? parseFloat(matrix.defaultPrice) : null;
      if (finalPrice && priceModifier !== 0) {
        if (priceModifierType === 'fixed') {
          finalPrice = finalPrice + priceModifier;
        } else {
          finalPrice = finalPrice + (finalPrice * priceModifier / 100);
        }
      }

      // Fetch template
      const template = await getDescriptionTemplateById(matrix.templateId);
      if (!template || !template.htmlContent) {
        return res.status(400).json({ error: "Template not found or has no content" });
      }

      // Build template context
      const templateContext = await buildTemplateContext(matrix, {
        length,
        width,
        height,
        color,
        productName,
        productGroup,
        finalPrice,
        sizeStr
      }, template.htmlContent);

      // Decode HTML entities
      const decodeHtmlEntities = (html: string): string => {
        return html
          .replace(/&lt;/g, '<')
          .replace(/&gt;/g, '>')
          .replace(/&amp;/g, '&')
          .replace(/&quot;/g, '"')
          .replace(/&#39;/g, "'");
      };

      let renderedHtml = decodeHtmlEntities(template.htmlContent);

      // Replace AI section variables using HYPHEN notation: {{ai-intro}}, {{ai-features}}, etc.
      for (const [key, value] of Object.entries(templateContext)) {
        if (key.startsWith('ai-')) {
          const pattern = new RegExp(`\\{\\{${key}\\}\\}`, 'g');
          renderedHtml = renderedHtml.replace(pattern, value as string);
        }
      }

      // Replace accessory tags
      const accessoryTagRegex = /\{\{(?:akcesorium|okucia|akcesoria)-([a-zA-Z0-9_-]+)(?::grid(\d+))?\}\}/g;
      renderedHtml = renderedHtml.replace(accessoryTagRegex, (_match: string, groupCode: string, gridCols?: string) => {
        const group = templateContext.accessories.groups.find(
          (g: any) => g.groupCode === groupCode || g.code === groupCode
        );
        
        if (!group || !group.items || group.items.length === 0) {
          return '';
        }
        
        const columns = gridCols ? parseInt(gridCols, 10) : 1;
        const itemWidth = columns === 1 ? '100%' : `${(100 / columns) - 2}%`;
        
        // IMPORTANT: Don't use position: absolute for images - TipTap removes wrapper divs
        return `
          <div class="accessory-group" data-group="${groupCode}" style="margin: 20px 0;">
            <h3 style="margin-bottom: 15px; font-size: 1.2em;">${group.groupName || group.name}</h3>
            <div class="accessory-items" style="display: flex; flex-wrap: wrap; gap: 10px; margin: 0 -5px;">
              ${group.items.map((item: any) => `
                <div class="accessory-item" style="width: ${itemWidth}; min-width: ${columns > 3 ? '150px' : '200px'}; box-sizing: border-box; padding: 10px; border: 1px solid #e0e0e0; border-radius: 4px; background: #fafafa;">
                  ${item.imageUrl ? `
                    <img src="${item.imageUrl}" alt="${item.name}" class="accessory-image" style="width: 100%; height: auto; display: block; margin-bottom: 10px; border-radius: 4px; object-fit: contain; max-height: 300px;" />
                  ` : ''}
                  <div style="text-align: center;">
                    <h4 style="margin: 0 0 5px 0; font-size: 0.9em; font-weight: 600;">${item.name}</h4>
                    ${item.code ? `<p style="margin: 0 0 5px 0; font-size: 0.8em; color: #666;">(${item.code})</p>` : ''}
                    ${item.description ? `<p style="margin: 0; font-size: 0.8em; color: #888;">${item.description}</p>` : ''}
                  </div>
                </div>
              `).join('')}
            </div>
          </div>
        `.trim();
      });

      // Replace image tags (using index)
      const imagesForColor = colorIndex !== -1 ? (colorImages[colorIndex.toString()] || []) : [];
      const imageTagRegex = /\{img(\d+)\s+w=(\d+)\}/gi;
      renderedHtml = renderedHtml.replace(imageTagRegex, (_match: string, imgIndex: string, widthParam: string) => {
        const index = parseInt(imgIndex, 10) - 1;
        const width = parseInt(widthParam, 10);
        
        if (![100, 75, 50, 25].includes(width)) {
          console.warn(`⚠️  Invalid width parameter: ${widthParam}`);
          return '';
        }
        
        if (index < 0 || index >= imagesForColor.length) {
          console.log(`⚠️  Image ${imgIndex} not found for color "${color}"`);
          return '';
        }
        
        const imageUrl = imagesForColor[index];
        
        // IMPORTANT: No div wrapper - TipTap removes it during editing
        // All styling must be on <img> tag directly
        // Use inline-block for horizontal layout when multiple images with small widths (25%, 50%, 75%)
        const display = width === 100 ? 'block' : 'inline-block';
        const margin = width === 100 ? '20px auto' : '10px 5px';
        return `<img src="${imageUrl}" alt="${productName} - zdjęcie ${imgIndex}" class="resizable-image" style="width: ${width}%; max-width: ${width}%; height: auto; display: ${display}; border-radius: 4px; margin: ${margin}; vertical-align: top;" />`;
      });

      // Replace simple variables
      renderedHtml = renderedHtml
        .replace(/\{\{product\.alpmaCode\}\}/g, templateContext.product.alpmaCode || '')
        .replace(/\{\{product\.title\}\}/g, templateContext.product.title || '')
        .replace(/\{\{product\.color\}\}/g, templateContext.product.color || '')
        .replace(/\{\{product\.price\}\}/g, String(templateContext.product.price || ''))
        .replace(/\{\{product\.productType\}\}/g, templateContext.product.productType || '')
        .replace(/\{\{title\}\}/g, templateContext.product.title || '')
        .replace(/\{\{color\}\}/g, templateContext.color || '')
        .replace(/\{\{dimensions\}\}/g, templateContext.dimensions || '')
        .replace(/\{\{price\}\}/g, String(templateContext.price || ''))
        .replace(/\{\{basePrice\}\}/g, String(templateContext.variant?.basePrice || templateContext.price || ''))
        .replace(/\{\{length\}\}/g, String(templateContext.variant?.length || ''))
        .replace(/\{\{width\}\}/g, String(templateContext.variant?.width || ''))
        .replace(/\{\{height\}\}/g, String(templateContext.variant?.height || ''));

      // Build response with detailed report
      const report = {
        preview: {
          productName,
          dimensions: { length, width, height, sizeStr },
          color,
          price: finalPrice,
          productGroup,
          priceModifier: priceModifier !== 0 ? { value: priceModifier, type: priceModifierType } : null
        },
        aiPrompts: matrix.aiConfig || null,
        accessories: templateContext.accessories.groups.map((g: any) => ({
          groupCode: g.groupCode || g.code,
          groupName: g.groupName || g.name,
          itemCount: g.items.length,
          items: g.items.map((item: any) => ({
            name: item.name,
            code: item.code,
            description: item.description,
            hasImage: !!item.imageUrl
          }))
        })),
        images: imagesForColor.map((url: string, idx: number) => ({
          index: idx + 1,
          url,
          tag: `{img${idx + 1} w=100}`
        })),
        renderedHtml,
        template: {
          id: template.id,
          name: template.name,
          categoryName: template.categoryName
        }
      };

      console.log(`✅ Preview generated successfully for: ${productName}`);
      res.json(report);
    } catch (error) {
      console.error("❌ Error generating preview:", error);
      res.status(500).json({ error: "Failed to generate preview" });
    }
  });

  // POST /api/product-matrices/:id/ai-generate-description - Generuj opis AI dla matrycy
  app.post("/api/product-matrices/:id/ai-generate-description", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { aiPrompt, templateId } = req.body;
      const matrix = await storage.getProductMatrixById(parseInt(id, 10));
      
      if (!matrix) {
        return res.status(404).json({ error: "Product matrix not found" });
      }

      console.log(`🤖 Generating AI description for matrix: ${matrix.name}`);

      // Check if template provided
      let templateHtml = '';
      let templateName = '';
      if (templateId) {
        const template = await getDescriptionTemplateById(templateId);
        if (template) {
          templateHtml = template.htmlContent || '';
          templateName = template.name || '';
          console.log(`📝 Using template: ${templateName}`);
        }
      }

      // Extract accessory tags from template if provided
      let accessoriesData: { groups: any[] } = { groups: [] };
      if (templateHtml) {
        try {
          const accessoryCodes = extractAccessoryTags(templateHtml);
          
          if (accessoryCodes.length > 0) {
            console.log(`🏷️  Found ${accessoryCodes.length} accessory tags in template`);
            
            const accessoriesResult = await pool.query(`
              SELECT 
                ag.id as group_id,
                ag.name as group_name,
                ag.code as group_code,
                ag.category as group_category,
                ag.description as group_description,
                json_agg(
                  json_build_object(
                    'id', a.id,
                    'name', a.name,
                    'code', a.code,
                    'alpmaCode', a.alpma_code,
                    'description', a.description,
                    'imageUrl', a.image_url,
                    'displayOrder', a.display_order
                  ) ORDER BY a.display_order
                ) FILTER (WHERE a.id IS NOT NULL) as items
              FROM catalog.accessory_groups ag
              LEFT JOIN catalog.accessories a ON a.group_id = ag.id AND a.is_active = true
              WHERE ag.is_active = true AND LOWER(ag.code) = ANY($1::text[])
              GROUP BY ag.id, ag.name, ag.code, ag.category, ag.description, ag.display_order
              ORDER BY ag.display_order
            `, [accessoryCodes]);
            
            accessoriesData.groups = accessoriesResult.rows.map((row: any) => ({
              groupId: row.group_id,
              groupName: row.group_name,
              groupCode: row.group_code,
              code: row.group_code,
              name: row.group_name,
              category: row.group_category,
              description: row.group_description,
              items: row.items || []
            }));
          }
        } catch (error) {
          console.error('❌ Error fetching accessories:', error);
        }
      }

      // Check if template uses AI tags
      const aiTagPattern = /\{\{ai-(intro|features|safety|care|warranty)\}\}/i;
      const hasAiTags = templateHtml ? aiTagPattern.test(templateHtml) : true; // Default true if no template

      let finalDescription = '';
      let aiCost = 0;
      let aiTokens = { prompt: 0, completion: 0, total: 0 };
      let aiModel = '';

      if (hasAiTags || !templateHtml) {
        // Generate AI content
        const productData = {
          sku: `MATRIX-${matrix.id}`,
          title: matrix.name,
          shortDescription: `Matryca produktów: ${matrix.productType || 'typ nieznany'}`,
          price: matrix.defaultPrice ? parseFloat(matrix.defaultPrice.toString()) : undefined,
          color: matrix.colors && matrix.colors.length > 0 ? matrix.colors.join(', ') : undefined,
          material: matrix.material || undefined,
          dimensions: matrix.lengths && matrix.widths && matrix.heights && 
                      matrix.lengths.length > 0 && matrix.widths.length > 0 && matrix.heights.length > 0
            ? {
                width: parseFloat(matrix.widths[0]) / 10,
                depth: parseFloat(matrix.lengths[0]) / 10,
                height: parseFloat(matrix.heights[0]) / 10
              }
            : undefined,
        };

        const { generateProductDescription } = await import('./ai-service');
        const result = await generateProductDescription({
          productData,
          templateHtml: aiPrompt || '',
          tone: 'professional',
          language: 'pl',
          maxLength: 500,
          accessories: accessoriesData,
        });

        aiCost = result.cost;
        aiTokens = {
          prompt: result.promptTokens,
          completion: result.completionTokens,
          total: result.totalTokens,
        };
        aiModel = result.model;

        if (templateHtml) {
          // Render template with AI sections
          const aiSections: Record<string, string> = {};
          const fullAiDescription = result.description;
          
          const sectionPatterns: Record<string, RegExp[]> = {
            'ai-intro': [
              /<h2[^>]*>Wprowadzenie[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
              /<div[^>]*>\s*<h2[^>]*>Wprowadzenie[^<]*<\/h2>([\s\S]*?)<\/div>/i,
            ],
            'ai-features': [
              /<h2[^>]*>Cechy[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
              /<div[^>]*>\s*<h2[^>]*>Cechy[^<]*<\/h2>([\s\S]*?)<\/div>/i,
            ],
            'ai-safety': [
              /<h2[^>]*>Bezpieczeństwo[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
            ],
            'ai-care': [
              /<h2[^>]*>Pielęgnacja[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
            ],
            'ai-warranty': [
              /<h2[^>]*>Gwarancja[^<]*<\/h2>([\s\S]*?)(?=<h2|<\/div>|$)/i,
            ],
          };
          
          for (const [sectionKey, patterns] of Object.entries(sectionPatterns)) {
            let extracted = '';
            for (const pattern of patterns) {
              const match = fullAiDescription.match(pattern);
              if (match && match[1]) {
                extracted = `<div><h2>${sectionKey.replace('ai-', '').charAt(0).toUpperCase() + sectionKey.replace('ai-', '').slice(1)}</h2>${match[1].trim()}</div>`;
                break;
              }
            }
            aiSections[sectionKey] = extracted;
          }

          // Decode and render template
          const decodeHtmlEntities = (html: string): string => {
            return html
              .replace(/&lt;/g, '<')
              .replace(/&gt;/g, '>')
              .replace(/&amp;/g, '&')
              .replace(/&quot;/g, '"')
              .replace(/&#39;/g, "'");
          };

          let renderedHtml = decodeHtmlEntities(templateHtml);

          // Replace basic matrix tags
          const basicTags: Record<string, string> = {
            'title': matrix.name || '',
            'productType': matrix.productType || '',
            'material': matrix.material || '',
            'defaultPrice': matrix.defaultPrice ? `${matrix.defaultPrice} PLN` : '',
          };

          for (const [key, value] of Object.entries(basicTags)) {
            const pattern = new RegExp(`\\{\\{${key}\\}\\}`, 'gi');
            renderedHtml = renderedHtml.replace(pattern, value);
          }

          // Replace AI tags
          for (const [key, value] of Object.entries(aiSections)) {
            const pattern = new RegExp(`\\{\\{${key}\\}\\}`, 'g');
            renderedHtml = renderedHtml.replace(pattern, value);
          }

          // Replace accessory tags
          const accessoryTagRegex = /\{\{(?:akcesorium|okucia|akcesoria)-([a-zA-Z0-9_-]+)(?::grid(\d+))?\}\}/g;
          renderedHtml = renderedHtml.replace(accessoryTagRegex, (_match: string, groupCode: string, gridCols?: string) => {
            const group = accessoriesData.groups.find(
              (g: any) => g.groupCode === groupCode || g.code === groupCode
            );
            
            if (!group || !group.items || group.items.length === 0) {
              return '';
            }
            
            const columns = gridCols ? parseInt(gridCols, 10) : 1;
            const itemWidth = columns === 1 ? '100%' : `${(100 / columns) - 2}%`;
            
            // IMPORTANT: Don't use position: absolute for images - TipTap removes wrapper divs
            return `
              <div class="accessory-group" data-group="${groupCode}" style="margin: 20px 0;">
                <h3 style="margin-bottom: 15px; font-size: 1.2em;">${group.groupName || group.name}</h3>
                <div class="accessory-items" style="display: flex; flex-wrap: wrap; gap: 10px; margin: 0 -5px;">
                  ${group.items.map((item: any) => `
                    <div class="accessory-item" style="width: ${itemWidth}; min-width: ${columns > 3 ? '150px' : '200px'}; box-sizing: border-box; padding: 10px; border: 1px solid #e0e0e0; border-radius: 4px; background: #fafafa;">
                      ${item.imageUrl ? `
                        <img src="${item.imageUrl}" alt="${item.name}" class="accessory-image" style="width: 100%; height: auto; display: block; margin-bottom: 10px; border-radius: 4px; object-fit: contain; max-height: 300px;" />
                      ` : ''}
                      <div style="text-align: center;">
                        <h4 style="margin: 0 0 5px 0; font-size: 0.9em; font-weight: 600;">${item.name}</h4>
                        ${item.code ? `<p style="margin: 0 0 5px 0; font-size: 0.8em; color: #666;">(${item.code})</p>` : ''}
                        ${item.description ? `<p style="margin: 0; font-size: 0.8em; color: #888;">${item.description}</p>` : ''}
                      </div>
                    </div>
                  `).join('')}
                </div>
              </div>
            `.trim();
          });

          finalDescription = renderedHtml;
        } else {
          // No template - return plain AI
          finalDescription = result.description;
        }
      } else {
        // Template without AI tags - just render basic tags
        console.log(`ℹ️  Template doesn't use AI tags - skipping AI generation`);
        
        const decodeHtmlEntities = (html: string): string => {
          return html
            .replace(/&lt;/g, '<')
            .replace(/&gt;/g, '>')
            .replace(/&amp;/g, '&')
            .replace(/&quot;/g, '"')
            .replace(/&#39;/g, "'");
        };

        let renderedHtml = decodeHtmlEntities(templateHtml);

        // Replace basic tags only
        const basicTags: Record<string, string> = {
          'title': matrix.name || '',
          'productType': matrix.productType || '',
          'material': matrix.material || '',
          'defaultPrice': matrix.defaultPrice ? `${matrix.defaultPrice} PLN` : '',
        };

        for (const [key, value] of Object.entries(basicTags)) {
          const pattern = new RegExp(`\\{\\{${key}\\}\\}`, 'gi');
          renderedHtml = renderedHtml.replace(pattern, value);
        }

        finalDescription = renderedHtml;
      }

      console.log(`✅ Description generated successfully (cost: ${aiCost.toFixed(4)} USD)`);

      res.json({
        success: true,
        description: finalDescription,
        cost: aiCost,
        tokens: aiTokens,
        model: aiModel || 'none',
      });
    } catch (error) {
      console.error("❌ Error generating AI description for matrix:", error);
      res.status(500).json({ error: "Failed to generate AI description" });
    }
  });

  // Multer configuration for product matrix color images (memoryStorage for SFTP)
  const uploadMatrixColorImage = multer({
    storage: multer.memoryStorage(),
    limits: {
      fileSize: 10 * 1024 * 1024, // 10MB limit
    },
    fileFilter: (req, file, cb) => {
      const allowedTypes = ['image/jpeg', 'image/jpg', 'image/png', 'image/webp', 'image/gif'];
      if (allowedTypes.includes(file.mimetype)) {
        cb(null, true);
      } else {
        cb(new Error('Invalid file type. Only JPEG, PNG, WEBP, and GIF are allowed.'));
      }
    }
  });

  // POST /api/product-matrices/:id/color-images - Upload image for a color
  app.post("/api/product-matrices/:id/color-images", isAuthenticated, uploadMatrixColorImage.single('image'), async (req, res) => {
    try {
      const { id } = req.params;
      const { colorIndex } = req.body;

      if (!req.file) {
        return res.status(400).json({ error: "No image file provided" });
      }

      if (colorIndex === undefined || colorIndex === null) {
        return res.status(400).json({ error: "Color index is required" });
      }

      // Get matrix
      const matrix = await storage.getProductMatrixById(parseInt(id, 10));
      if (!matrix) {
        return res.status(404).json({ error: "Product matrix not found" });
      }

      // Generate base filename (without extension)
      const timestamp = Date.now();
      const randomString = Math.random().toString(36).substring(2, 8);
      const baseFilename = `color_${id}_${colorIndex}_${timestamp}_${randomString}`;
      
      // Process image to generate optimized versions (thumbnail, medium, original)
      const { processImage, generateImageVersionFilenames } = await import('./image-processor.js');
      const processed = await processImage(req.file.buffer, {
        thumbnailSize: 80,
        mediumSize: 400,
        quality: 85,
        format: 'webp'
      });
      
      // Generate filenames for all versions
      const filenames = generateImageVersionFilenames(baseFilename, processed.format);
      
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      // Upload all three versions to separate subdirectories under product-matrix-colors/
      const [originalUrl, thumbnailUrl, mediumUrl] = await Promise.all([
        adapter.upload({
          filename: filenames.original,
          buffer: processed.buffers.original,
          mimetype: `image/${processed.format}`,
          subfolder: 'product-matrix-colors'
        }),
        adapter.upload({
          filename: filenames.thumbnail,
          buffer: processed.buffers.thumbnail,
          mimetype: `image/${processed.format}`,
          subfolder: 'product-matrix-colors/thumbnails'
        }),
        adapter.upload({
          filename: filenames.medium,
          buffer: processed.buffers.medium,
          mimetype: `image/${processed.format}`,
          subfolder: 'product-matrix-colors/medium'
        })
      ]);

      // Get current colorImages object (using numeric indices as keys)
      // Structure: colorImages = { "0": [{url, thumbnailUrl, mediumUrl}, ...], "1": [...], ... }
      const colorImages = (matrix.colorImages as any) || {};
      
      // Add new image object with all three versions to the color's array
      const indexKey = colorIndex.toString();
      if (!colorImages[indexKey]) {
        colorImages[indexKey] = [];
      }
      
      // Store all three versions
      const imageEntry = {
        url: originalUrl,
        thumbnailUrl,
        mediumUrl
      };
      colorImages[indexKey].push(imageEntry);

      // Update database
      await pool.query(
        'UPDATE product_creator.product_matrices SET color_images = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
        [JSON.stringify(colorImages), id]
      );

      console.log(`✅ Uploaded optimized image (3 versions) for color index ${colorIndex} in product matrix ${id}`);

      res.json({
        success: true,
        imageUrl: originalUrl,
        thumbnailUrl,
        mediumUrl,
        colorIndex,
        colorImages
      });
    } catch (error) {
      console.error("❌ Error uploading product matrix color image:", error);
      res.status(500).json({ error: "Failed to upload color image" });
    }
  });

  // DELETE /api/product-matrices/:id/color-images/:colorIndex/:filename - Delete a color image
  app.delete("/api/product-matrices/:id/color-images/:colorIndex/:filename", isAuthenticated, async (req, res) => {
    try {
      const { id, colorIndex, filename } = req.params;

      // Get matrix
      const matrix = await storage.getProductMatrixById(parseInt(id, 10));
      if (!matrix) {
        return res.status(404).json({ error: "Product matrix not found" });
      }

      // Get current colorImages object (now uses numeric indices as keys)
      const colorImages = (matrix.colorImages as any) || {};
      
      const indexKey = colorIndex.toString();
      if (!colorImages[indexKey]) {
        return res.status(404).json({ error: "Color index not found" });
      }

      // Support both legacy format (string[]) and new format (object[])
      // Find image that contains this filename
      const imageIndex = colorImages[indexKey].findIndex((entry: any) => {
        const url = typeof entry === 'string' ? entry : entry.url;
        return url.includes(filename);
      });
      
      if (imageIndex === -1) {
        return res.status(404).json({ error: "Image not found for this color" });
      }

      // Get image entry before removing (to delete all versions)
      const imageEntry = colorImages[indexKey][imageIndex];
      const isObject = typeof imageEntry === 'object' && imageEntry !== null;
      
      colorImages[indexKey].splice(imageIndex, 1);

      // If no more images for this color index, remove the key
      if (colorImages[indexKey].length === 0) {
        delete colorImages[indexKey];
      }

      // Update database
      await pool.query(
        'UPDATE product_creator.product_matrices SET color_images = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
        [JSON.stringify(colorImages), id]
      );

      // Delete physical files from SFTP (all versions if object format)
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      try {
        if (isObject) {
          // Delete all three versions
          const promises = [
            adapter.delete(`product-matrix-colors/${filename}`),
            adapter.delete(`product-matrix-colors/thumbnails/${filename}`),
            adapter.delete(`product-matrix-colors/medium/${filename}`)
          ];
          await Promise.allSettled(promises);
          console.log(`✅ Deleted all versions of color image: ${filename}`);
        } else {
          // Legacy format - single file
          await adapter.delete(`products/images/${filename}`);
          console.log(`✅ Deleted legacy color image: ${filename}`);
        }
      } catch (deleteError) {
        console.warn(`⚠️ Could not delete file ${filename} from SFTP:`, deleteError);
      }

      res.json({
        success: true,
        colorImages
      });
    } catch (error) {
      console.error("❌ Error deleting matrix color image:", error);
      res.status(500).json({ error: "Failed to delete color image" });
    }
  });

  // ==================== PRODUCT SETS (ZESTAWY) SYSTEM ====================
  
  // GET /api/set-matrices - Lista wszystkich matryc zestawów
  app.get("/api/set-matrices", isAuthenticated, async (req, res) => {
    try {
      const matrices = await storage.getSetMatrices();
      res.json(matrices);
    } catch (error) {
      console.error("❌ Error fetching set matrices:", error);
      res.status(500).json({ error: "Failed to fetch set matrices" });
    }
  });

  // GET /api/set-matrices/:id - Pobierz pojedynczą matrycę zestawu
  app.get("/api/set-matrices/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const matrix = await storage.getSetMatrixById(parseInt(id, 10));
      
      if (!matrix) {
        return res.status(404).json({ error: "Set matrix not found" });
      }
      
      res.json(matrix);
    } catch (error) {
      console.error("❌ Error fetching set matrix:", error);
      res.status(500).json({ error: "Failed to fetch set matrix" });
    }
  });

  // POST /api/set-matrices - Dodaj nową matrycę zestawu
  app.post("/api/set-matrices", isAuthenticated, async (req, res) => {
    try {
      const matrix = await storage.createSetMatrix(req.body);
      res.status(201).json(matrix);
    } catch (error) {
      console.error("❌ Error creating set matrix:", error);
      res.status(500).json({ error: "Failed to create set matrix" });
    }
  });

  // POST /api/set-matrices/:id/duplicate - Duplikuj matrycę wraz z komponentami
  app.post("/api/set-matrices/:id/duplicate", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const originalMatrix = await storage.getSetMatrixById(parseInt(id, 10));
      
      if (!originalMatrix) {
        return res.status(404).json({ error: "Set matrix not found" });
      }
      
      // Utwórz kopię z nową nazwą (bez timestamp - prostsze)
      // UWAGA: colors i selectedColors to tablice TEXT[], nie JSONB - przekazuj je bezpośrednio
      const duplicatedMatrix = await storage.createSetMatrix({
        name: `${originalMatrix.name} - Kopia`,
        productGroup: originalMatrix.productGroup,
        imageUrl: originalMatrix.imageUrl,
        namePrefix: originalMatrix.namePrefix,
        nameSuffix: originalMatrix.nameSuffix,
        defaultDepth: originalMatrix.defaultDepth,
        hookLength: originalMatrix.hookLength,
        components: originalMatrix.components as any,
        modifierValue: originalMatrix.modifierValue,
        modifierType: originalMatrix.modifierType,
        // selectedColors to tablica number[], która zostanie przekonwertowana na string[]
        selectedColors: Array.isArray(originalMatrix.selectedColors) ? originalMatrix.selectedColors : [],
        useAiGeneration: originalMatrix.useAiGeneration,
        aiPrompt: originalMatrix.aiPrompt,
        descriptionTemplateId: originalMatrix.descriptionTemplateId,
        isActive: originalMatrix.isActive,
        // Kopiuj wszystkie dodatkowe pola
        // colors to tablica string[] - NIE stringifikuj, przekaż bezpośrednio
        colors: Array.isArray(originalMatrix.colors) ? originalMatrix.colors : [],
        colorImages: originalMatrix.colorImages as any,
        colorOptions: originalMatrix.colorOptions as any,
        length: originalMatrix.length,
        width: originalMatrix.width,
        height: originalMatrix.height,
        doors: originalMatrix.doors,
        legs: originalMatrix.legs,
        useEnhancedDescription: originalMatrix.useEnhancedDescription,
      });
      
      console.log(`✅ Matrix duplicated: ${originalMatrix.name} -> ${duplicatedMatrix.name}`);
      res.status(201).json(duplicatedMatrix);
    } catch (error) {
      console.error("❌ Error duplicating set matrix:", error);
      res.status(500).json({ error: "Failed to duplicate set matrix" });
    }
  });

  // PUT /api/set-matrices/:id - Edytuj matrycę zestawu
  app.put("/api/set-matrices/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const matrix = await storage.updateSetMatrix(parseInt(id, 10), req.body);
      res.json(matrix);
    } catch (error) {
      console.error("❌ Error updating set matrix:", error);
      res.status(500).json({ error: "Failed to update set matrix" });
    }
  });

  // DELETE /api/set-matrices/:id - Usuń matrycę zestawu
  app.delete("/api/set-matrices/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      await storage.deleteSetMatrix(parseInt(id, 10));
      res.status(204).send();
    } catch (error) {
      console.error("❌ Error deleting set matrix:", error);
      res.status(500).json({ error: "Failed to delete set matrix" });
    }
  });

  // POST /api/set-matrices/:id/color-images - Upload image for a set matrix color
  app.post("/api/set-matrices/:id/color-images", isAuthenticated, uploadMatrixColorImage.single('image'), async (req, res) => {
    try {
      const { id } = req.params;
      const { colorIndex } = req.body;

      if (!req.file) {
        return res.status(400).json({ error: "No image file provided" });
      }

      if (colorIndex === undefined || colorIndex === null) {
        return res.status(400).json({ error: "Color index is required" });
      }

      // Get matrix
      const matrix = await storage.getSetMatrixById(parseInt(id, 10));
      if (!matrix) {
        return res.status(404).json({ error: "Set matrix not found" });
      }

      // Generate base filename (without extension)
      const timestamp = Date.now();
      const randomString = Math.random().toString(36).substring(2, 8);
      const baseFilename = `setcolor_${id}_${colorIndex}_${timestamp}_${randomString}`;
      
      // Process image to generate optimized versions (thumbnail, medium, original)
      const { processImage, generateImageVersionFilenames } = await import('./image-processor.js');
      const processed = await processImage(req.file.buffer, {
        thumbnailSize: 80,
        mediumSize: 400,
        quality: 85,
        format: 'webp'
      });
      
      // Generate filenames for all versions
      const filenames = generateImageVersionFilenames(baseFilename, processed.format);
      
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      // Upload all three versions to separate subdirectories under product-matrix-colors/
      const [originalUrl, thumbnailUrl, mediumUrl] = await Promise.all([
        adapter.upload({
          filename: filenames.original,
          buffer: processed.buffers.original,
          mimetype: `image/${processed.format}`,
          subfolder: 'product-matrix-colors'
        }),
        adapter.upload({
          filename: filenames.thumbnail,
          buffer: processed.buffers.thumbnail,
          mimetype: `image/${processed.format}`,
          subfolder: 'product-matrix-colors/thumbnails'
        }),
        adapter.upload({
          filename: filenames.medium,
          buffer: processed.buffers.medium,
          mimetype: `image/${processed.format}`,
          subfolder: 'product-matrix-colors/medium'
        })
      ]);

      // Get current colorImages object (using numeric indices as keys)
      // Structure: colorImages = { "0": [{url, thumbnailUrl, mediumUrl}, ...], "1": [...], ... }
      const colorImages = (matrix.colorImages as any) || {};
      
      // Add new image object with all three versions to the color's array
      const indexKey = colorIndex.toString();
      if (!colorImages[indexKey]) {
        colorImages[indexKey] = [];
      }
      
      // Store all three versions
      const imageEntry = {
        url: originalUrl,
        thumbnailUrl,
        mediumUrl
      };
      colorImages[indexKey].push(imageEntry);

      // Update database
      await pool.query(
        'UPDATE product_creator.set_matrices SET color_images = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
        [JSON.stringify(colorImages), id]
      );

      console.log(`✅ Uploaded optimized image (3 versions) for color index ${colorIndex} in set matrix ${id}`);

      res.json({
        success: true,
        imageUrl: originalUrl,
        thumbnailUrl,
        mediumUrl,
        colorIndex,
        colorImages
      });
    } catch (error) {
      console.error("❌ Error uploading set matrix color image:", error);
      res.status(500).json({ error: "Failed to upload color image" });
    }
  });

  // DELETE /api/set-matrices/:id/color-images/:colorIndex/:filename - Delete a set matrix color image
  app.delete("/api/set-matrices/:id/color-images/:colorIndex/:filename", isAuthenticated, async (req, res) => {
    try {
      const { id, colorIndex, filename } = req.params;

      // Get matrix
      const matrix = await storage.getSetMatrixById(parseInt(id, 10));
      if (!matrix) {
        return res.status(404).json({ error: "Set matrix not found" });
      }

      // Get current colorImages object (using numeric indices as keys)
      const colorImages = (matrix.colorImages as any) || {};
      
      const indexKey = colorIndex.toString();
      if (!colorImages[indexKey]) {
        return res.status(404).json({ error: "Color index not found" });
      }

      // Support both legacy format (string[]) and new format (object[])
      // Find image that contains this filename
      const imageIndex = colorImages[indexKey].findIndex((entry: any) => {
        const url = typeof entry === 'string' ? entry : entry.url;
        return url.includes(filename);
      });
      
      if (imageIndex === -1) {
        return res.status(404).json({ error: "Image not found for this color" });
      }

      // Get image entry before removing (to delete all versions)
      const imageEntry = colorImages[indexKey][imageIndex];
      const isObject = typeof imageEntry === 'object' && imageEntry !== null;
      
      colorImages[indexKey].splice(imageIndex, 1);

      // If no more images for this color, remove the key
      if (colorImages[indexKey].length === 0) {
        delete colorImages[indexKey];
      }

      // Update database
      await pool.query(
        'UPDATE product_creator.set_matrices SET color_images = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
        [JSON.stringify(colorImages), id]
      );

      // Delete physical files from SFTP (all versions if object format)
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      try {
        if (isObject) {
          // Delete all three versions
          const promises = [
            adapter.delete(`product-matrix-colors/${filename}`),
            adapter.delete(`product-matrix-colors/thumbnails/${filename}`),
            adapter.delete(`product-matrix-colors/medium/${filename}`)
          ];
          await Promise.allSettled(promises);
          console.log(`✅ Deleted all versions of set matrix color image: ${filename}`);
        } else {
          // Legacy format - single file
          await adapter.delete(`products/images/${filename}`);
          console.log(`✅ Deleted legacy set matrix color image: ${filename}`);
        }
      } catch (deleteError) {
        console.warn(`⚠️ Could not delete file ${filename} from SFTP:`, deleteError);
      }

      res.json({
        success: true,
        colorImages
      });
    } catch (error) {
      console.error("❌ Error deleting set matrix color image:", error);
      res.status(500).json({ error: "Failed to delete color image" });
    }
  });

  // POST /api/set-matrices/generate-batch - Generuj zestawy z wielu matryc
  app.post("/api/set-matrices/generate-batch", isAuthenticated, async (req, res) => {
    try {
      const { matrixIds, sessionId } = req.body as { matrixIds: number[]; sessionId?: string };

      if (!matrixIds || matrixIds.length === 0) {
        return res.status(400).json({ error: "matrixIds array is required" });
      }

      // Use provided sessionId or generate new one
      const batchSessionId = sessionId || randomBytes(16).toString('hex');

      // Start batch generation in background
      (async () => {
        const results: any[] = [];
        let successCount = 0;
        let failedCount = 0;

        emitGenerationLog(batchSessionId, `🚀 Rozpoczynanie batch generation dla ${matrixIds.length} matryc...`, 'info');

        for (let i = 0; i < matrixIds.length; i++) {
          const matrixId = matrixIds[i];
          const progress = Math.round(((i + 1) / matrixIds.length) * 100);

          try {
            emitGenerationLog(batchSessionId, `\n📦 [${i + 1}/${matrixIds.length}] Przetwarzanie matrycy ID: ${matrixId}`, 'info');

            const matrix = await storage.getSetMatrixById(matrixId);
            
            if (!matrix) {
              emitGenerationLog(batchSessionId, `❌ Matryca ${matrixId} nie znaleziona - pomijam`, 'error');
              failedCount++;
              
              // Update generation status
              await pool.query(
                `UPDATE product_creator.set_matrices 
                 SET last_generation_status = $1, last_generation_at = CURRENT_TIMESTAMP 
                 WHERE id = $2`,
                ['failed', matrixId]
              );

              results.push({ matrixId, status: 'failed', error: 'Matrix not found' });
              continue;
            }

            emitGenerationLog(batchSessionId, `   📋 Matryca: ${matrix.name}`, 'info');

            // Generate sets
            const generatedSets = await generateSetsFromMatrix(matrix, batchSessionId);
            
            if (generatedSets && generatedSets.length > 0) {
              successCount++;
              emitGenerationLog(batchSessionId, `   ✅ Wygenerowano ${generatedSets.length} zestawów`, 'success');

              // Update generation status - success
              await pool.query(
                `UPDATE product_creator.set_matrices 
                 SET last_generation_status = $1, last_generation_at = CURRENT_TIMESTAMP, last_missing_components = NULL 
                 WHERE id = $2`,
                ['success', matrixId]
              );

              results.push({ 
                matrixId, 
                status: 'success', 
                setsGenerated: generatedSets.length 
              });
            } else {
              failedCount++;
              emitGenerationLog(batchSessionId, `   ⚠️ Nie wygenerowano żadnych zestawów`, 'warning');

              // Update generation status - partial
              await pool.query(
                `UPDATE product_creator.set_matrices 
                 SET last_generation_status = $1, last_generation_at = CURRENT_TIMESTAMP 
                 WHERE id = $2`,
                ['partial', matrixId]
              );

              results.push({ 
                matrixId, 
                status: 'partial', 
                setsGenerated: 0 
              });
            }

            // Emit progress
            emitGenerationLog(batchSessionId, `📊 Progress: ${progress}%`, 'info');

          } catch (error: any) {
            failedCount++;
            console.error(`❌ Error generating from matrix ${matrixId}:`, error);
            emitGenerationLog(batchSessionId, `   ❌ Błąd: ${error.message}`, 'error');

            // Update generation status - failed with error details
            // Save error message in lastMissingComponents for debugging
            const errorInfo = [{
              error: true,
              message: error.message,
              timestamp: new Date().toISOString()
            }];

            await pool.query(
              `UPDATE product_creator.set_matrices 
               SET last_generation_status = $1, last_generation_at = CURRENT_TIMESTAMP, last_missing_components = $2
               WHERE id = $3`,
              ['failed', JSON.stringify(errorInfo), matrixId]
            );

            results.push({ 
              matrixId, 
              status: 'failed', 
              error: error.message 
            });
          }
        }

        // Final summary
        emitGenerationLog(batchSessionId, `\n✅ Batch generation zakończone!`, 'success');
        emitGenerationLog(batchSessionId, `   📊 Sukces: ${successCount}/${matrixIds.length}`, 'success');
        emitGenerationLog(batchSessionId, `   ❌ Błędy: ${failedCount}/${matrixIds.length}`, failedCount > 0 ? 'warning' : 'info');
        emitGenerationLog(batchSessionId, JSON.stringify({ results, summary: { success: successCount, failed: failedCount, total: matrixIds.length } }), 'info');

      })().catch(error => {
        console.error('❌ Batch generation error:', error);
        emitGenerationLog(batchSessionId, `❌ Krytyczny błąd batch generation: ${error.message}`, 'error');
      });

      // Return immediately with sessionId for SSE connection
      res.json({
        success: true,
        sessionId: batchSessionId,
        message: 'Batch generation started'
      });

    } catch (error: any) {
      console.error("❌ Error starting batch generation:", error);
      res.status(500).json({ error: error.message || "Failed to start batch generation" });
    }
  });

  // POST /api/set-matrices/:id/generate - Generuj zestawy z matrycy
  app.post("/api/set-matrices/:id/generate", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { sessionId } = req.body;
      
      console.log(`🔧 Generating sets from matrix ID: ${id}`);
      if (sessionId) {
        emitGenerationLog(sessionId, `🔧 Rozpoczynanie generowania zestawów z matrycy ID: ${id}`, 'info');
      }
      
      const matrix = await storage.getSetMatrixById(parseInt(id, 10));
      
      if (!matrix) {
        console.log(`❌ Matrix not found: ${id}`);
        if (sessionId) {
          emitGenerationLog(sessionId, `❌ Nie znaleziono matrycy: ${id}`, 'error');
        }
        return res.status(404).json({ error: "Set matrix not found" });
      }

      console.log(`📋 Matrix found: ${matrix.name}`);
      if (sessionId) {
        emitGenerationLog(sessionId, `📋 Znaleziono matrycę: ${matrix.name}`, 'success');
        emitGenerationLog(sessionId, `🔧 Komponenty: ${((matrix.components as any[]) || []).length}`, 'info');
        emitGenerationLog(sessionId, `🎨 Kolory: ${Array.isArray(matrix.colors) ? matrix.colors.length : 0}`, 'info');
      }

      // Generate sets from matrix with sessionId for logs
      const generatedSets = await generateSetsFromMatrix(matrix, sessionId);
      
      console.log(`✅ Generated ${generatedSets?.length || 0} sets`);
      if (sessionId) {
        emitGenerationLog(sessionId, `✅ Wygenerowano ${generatedSets?.length || 0} zestawów`, 'success');
      }

      res.json({
        success: true,
        setsGenerated: generatedSets?.length || 0,
        sets: generatedSets
      });
    } catch (error: any) {
      console.error("❌ Error generating sets:", error);
      res.status(500).json({ error: error.message || "Failed to generate sets" });
    }
  });

  // POST /api/set-matrices/validate-generation - Waliduj czy można wygenerować zestawy
  app.post("/api/set-matrices/validate-generation", isAuthenticated, async (req, res) => {
    try {
      const { matrixIds } = req.body as { matrixIds?: number[] };
      
      // If no matrixIds provided, validate all
      const idsToValidate = matrixIds || (await storage.getSetMatrices()).map(m => m.id);
      
      console.log(`🔍 [VALIDATE] Validating ${idsToValidate.length} matrices:`, idsToValidate);
      
      const variants: any[] = [];
      let totalReady = 0;
      let totalBlocked = 0;
      
      for (const matrixId of idsToValidate) {
        const matrix = await storage.getSetMatrixById(matrixId);
        if (!matrix) {
          console.log(`⚠️ [VALIDATE] Matrix ${matrixId} not found`);
          continue;
        }
        
        const components = (matrix.components as any[]) || [];
        const colors = matrix.colors || [];
        const selectedColors = matrix.selectedColors || [];
        const colorsToUse = selectedColors.length > 0 ? selectedColors : colors;
        const componentProductOverrides = (matrix.componentProductOverrides as any) || {};
        
        // Convert selectedColors indices to actual color names and track original indices
        const variantColorIndices: number[] = [];
        const variantColors = colorsToUse.length > 0 
          ? colorsToUse.map((colorOrIndex: any, idx: number) => {
              // If it's a number, it's an index - convert to color name
              if (typeof colorOrIndex === 'number') {
                variantColorIndices[idx] = colorOrIndex;
                return colors[colorOrIndex] || colorOrIndex;
              }
              // It's already a color name - find its index
              variantColorIndices[idx] = colors.indexOf(colorOrIndex);
              return colorOrIndex;
            })
          : [null];
        
        // Get color options and images (same logic as generateSetsFromMatrix)
        const colorOptions = (matrix as any).colorOptions || {};
        const colorImages = (matrix as any).colorImages || {};
        
        console.log(`🔍 [VALIDATE] Matrix ${matrixId} (${matrix.name}):`, {
          components: components.length,
          colors: colors.length,
          selectedColors: selectedColors.length,
          colorsToUse: colorsToUse.length,
          variantColors: variantColors.length,
          hasColorOptions: Object.keys(colorOptions).length > 0
        });
        
        // For each color variant, check component availability
        for (let loopIndex = 0; loopIndex < variantColors.length; loopIndex++) {
          const color = variantColors[loopIndex];
          // Get original index for colorOptions lookup
          const originalColorIndex = variantColorIndices[loopIndex] ?? loopIndex;
          
          // Get color options for this color (index-based lookup with fallback to name-based)
          const getColorValue = (obj: any, color: string | null, index: number): any => {
            if (!obj) return undefined;
            
            // 1. Try index-based lookup (preferred - supports duplicate color names)
            const byIndex = obj[index.toString()];
            if (byIndex !== undefined) return byIndex;
            
            // 2. Try name-based lookup (backward compatibility)
            if (color && obj[color] !== undefined) return obj[color];
            
            return undefined;
          };
          
          const colorOptionsForColor = getColorValue(colorOptions, color, originalColorIndex) || [];
          
          console.log(`  🎨 [VALIDATE] Variant #${loopIndex}: ${color} (index ${originalColorIndex}), options: [${colorOptionsForColor.join(', ')}]`);
          
          // Build variant display name with color options
          const variantNameParts = [matrix.name, color];
          if (colorOptionsForColor.length > 0) {
            variantNameParts.push(...colorOptionsForColor);
          }
          const variantName = variantNameParts.filter(Boolean).join(' + ');
          
          const variantComponents: any[] = [];
          let hasAllComponents = true;
          
          console.log(`    🔍 [VALIDATE] Processing ${components.length} components for variant ${color}`);
          
          for (const comp of components) {
            const componentType = comp.componentType;
            const length = comp.length;
            const width = comp.width;
            const quantity = comp.quantity || 1;
            const ignoreColorOptions = comp.ignoreColorOptions || false;
            
            console.log(`    ➡️ [VALIDATE] Component: ${componentType}, length=${length}, width=${width}, ignoreColorOptions=${ignoreColorOptions}`);
            
            // Build component display name
            const compName = [
              componentType,
              length && width ? `${length}x${width}` : null,
              componentType !== 'PANEL' ? color : null
            ].filter(Boolean).join(' ');
            
            // 🎯 PRIORITY 1: Check for component product override
            let productId: number | null = null;
            let productSource: 'override' | 'auto' = 'auto';
            
            if (originalColorIndex >= 0 && originalColorIndex < colors.length) {
              const colorIndexKey = originalColorIndex.toString();
              const overrideForColor = componentProductOverrides?.[colorIndexKey];
              const overrideProductId = overrideForColor?.[componentType]?.productId;
              
              if (overrideProductId) {
                // Override found - validate product exists
                const productCheckResult = await pool.query(
                  'SELECT id FROM catalog.products WHERE id = $1',
                  [overrideProductId]
                );
                
                if (productCheckResult.rows.length > 0) {
                  productId = overrideProductId;
                  productSource = 'override';
                  console.log(`    🎯 [VALIDATE] Override found for ${componentType}: product ${productId}`);
                } else {
                  console.warn(`    ⚠️ [VALIDATE] Override product ${overrideProductId} for ${componentType} not found`);
                }
              }
            }
            
            // 🔍 PRIORITY 2: Auto-match product if no override was used
            if (productId === null) {
              // Build query with EXACT same logic as buildCatalogSet (lines 2016-2083)
              let query = `SELECT id FROM catalog.products WHERE product_type = $1`;
              const params: any[] = [componentType];
              let paramIndex = 2;
              
              // Add length condition if specified
              if (length !== null && length !== undefined) {
                query += ` AND length = $${paramIndex}`;
                params.push(String(length));
                paramIndex++;
              }
              
              // Add width condition if specified
              if (width !== null && width !== undefined) {
                query += ` AND width = $${paramIndex}`;
                params.push(String(width));
                paramIndex++;
              }
              
              // Add doors condition from set matrix if specified
              // Match products with specific doors OR products without doors (NULL)
              if (matrix.doors && matrix.doors !== 'none') {
                query += ` AND (doors = $${paramIndex} OR doors IS NULL OR doors = '')`;
                params.push(matrix.doors);
                paramIndex++;
              }
              
              // Add legs condition from set matrix if specified
              // Match products with specific legs OR products without legs (NULL)
              if (matrix.legs && matrix.legs !== 'none') {
                query += ` AND (legs = $${paramIndex} OR legs IS NULL OR legs = '')`;
                params.push(matrix.legs);
                paramIndex++;
              }
              
              // For non-PANEL components, match color
              // BUT: if component has ignoreColorOptions flag, skip color matching entirely
              if (!ignoreColorOptions && color && componentType !== 'PANEL') {
                query += ` AND color = $${paramIndex}`;
                params.push(color);
                paramIndex++;
              }
              
              // Match color_options based on set matrix colorOptions (CRITICAL for accuracy)
              if (ignoreColorOptions) {
                // Component IGNORES color options AND color (e.g., hanger, upholstered panel)
                // Don't add ANY color/color_options filter - match products regardless of their color
                console.log(`    🔧 [VALIDATE] Component ${componentType} ignores color options → no filter on color or color_options`);
              } else if (colorOptionsForColor.length > 0) {
                // Filter color options by component's relevant prefixes (if defined)
                const filteredColorOptions = filterColorOptionsByRelevantPrefixes(
                  colorOptionsForColor,
                  comp.relevantOptionPrefixes
                );
                
                if (filteredColorOptions.length > 0) {
                  // Set matrix HAS color options (after filtering) → match products that have ALL these options
                  console.log(`    🎨 [VALIDATE] Matching WITH color_options: [${filteredColorOptions.join(', ')}]${comp.relevantOptionPrefixes ? ` (filtered by: [${comp.relevantOptionPrefixes.join(', ')}])` : ''}`);
                  query += ` AND color_options @> $${paramIndex}::text[]`;
                  params.push(filteredColorOptions);
                  paramIndex++;
                } else {
                  // All options were filtered out → match products WITHOUT options
                  console.log(`    ⚪ [VALIDATE] All color options filtered out → matching products WITHOUT color_options`);
                  query += ` AND (color_options IS NULL OR cardinality(color_options) = 0)`;
                }
              } else {
                // Set matrix has NO options → match ONLY products WITHOUT options
                query += ` AND (color_options IS NULL OR cardinality(color_options) = 0)`;
              }
              
              query += ` LIMIT 1`;
              
              console.log(`    📝 [VALIDATE] SQL for ${componentType}:`, query);
              console.log(`    📝 [VALIDATE] Params:`, params);
              
              const result = await pool.query(query, params);
              productId = result.rows[0]?.id || null; // Assign to existing productId variable
            }
            
            // Determine availability
            const isAvailable = !!productId;
            
            if (isAvailable) {
              console.log(`    ✅ [VALIDATE] Found: ${compName} → Product #${productId}${productSource === 'override' ? ' (override)' : ''}`);
            } else {
              hasAllComponents = false;
              console.log(`    ⚠️ [VALIDATE] Missing: ${compName}, options: [${colorOptionsForColor.join(', ')}]`);
            }
            
            variantComponents.push({
              name: compName,
              componentType,
              length,
              width,
              color,
              // Don't send colorOptions if component ignores them (e.g., hanger, upholstered panel)
              colorOptions: ignoreColorOptions ? [] : colorOptionsForColor,
              quantity,
              available: isAvailable,
              availableCount: isAvailable ? 1 : 0,
              productId: productId || null,
              productSource: productSource // 'override' | 'auto'
            });
          }
          
          if (hasAllComponents) {
            totalReady++;
          } else {
            totalBlocked++;
          }
          
          variants.push({
            matrixId: matrix.id,
            matrixName: matrix.name,
            variantName,
            color,
            colorOptions: colorOptionsForColor,
            components: variantComponents,
            canGenerate: hasAllComponents
          });
        }
      }
      
      console.log(`✅ [VALIDATE] Result:`, {
        totalVariants: variants.length,
        readyCount: totalReady,
        blockedCount: totalBlocked
      });
      
      res.json({
        variants,
        summary: {
          totalVariants: variants.length,
          readyCount: totalReady,
          blockedCount: totalBlocked
        }
      });
    } catch (error: any) {
      console.error("❌ Error validating generation:", error);
      res.status(500).json({ error: error.message || "Failed to validate generation" });
    }
  });

  // POST /api/set-matrices/:id/preview-generation - Podgląd generowania zestawu
  app.post("/api/set-matrices/:id/preview-generation", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const matrix = await storage.getSetMatrixById(parseInt(id, 10));
      
      if (!matrix) {
        return res.status(404).json({ error: "Set matrix not found" });
      }

      // Generate preview for first color or no color
      const colors = matrix.colors || [];
      const selectedColors = matrix.selectedColors || [];
      const colorsToUse = selectedColors.length > 0 ? selectedColors : colors;
      const previewColor = colorsToUse.length > 0 ? colorsToUse[0] : null;

      // Build preview set name
      const setName = [
        matrix.namePrefix,
        matrix.name,
        previewColor,
        matrix.nameSuffix
      ].filter(Boolean).join(' ').trim();

      // Build SKU
      const skuParts = [
        matrix.name.replace(/\s+/g, '-').toUpperCase(),
        previewColor ? previewColor.replace(/\s+/g, '-').toUpperCase() : 'BASE'
      ].filter(Boolean);
      const sku = skuParts.join('-');

      // Get component details
      const components = (matrix.components as any[]) || [];
      
      const preview = {
        setName,
        sku,
        color: previewColor,
        components: components.map((comp, idx) => ({
          position: idx + 1,
          componentType: comp.componentType || 'NIEZNANY',
          name: comp.name || `Komponent ${idx + 1}`,
          length: comp.length,
          width: comp.width,
          quantity: comp.quantity || 1
        })),
        totalComponents: components.length,
        estimatedPrice: matrix.modifierValue || 0
      };

      res.json({ preview });
    } catch (error) {
      console.error("❌ Error generating preview:", error);
      res.status(500).json({ error: "Failed to generate preview" });
    }
  });

  // GET /api/set-matrices/:id/sets - Pobierz zestawy wygenerowane z danej matrycy
  app.get("/api/set-matrices/:id/sets", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(
        `SELECT id, sku, title, color, color_options, base_price, calculated_price, is_active, created_at
         FROM product_creator.product_sets
         WHERE set_matrix_id = $1
         ORDER BY created_at DESC`,
        [parseInt(id, 10)]
      );
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching generated sets:", error);
      res.status(500).json({ error: "Failed to fetch generated sets" });
    }
  });

  // GET /api/product-sets - Lista wszystkich zestawów produktów z marketplace linkage
  app.get("/api/product-sets", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          ps.id,
          ps.set_matrix_id AS "setMatrixId",
          ps.sku,
          ps.title,
          ps.short_description AS "shortDescription",
          ps.full_description AS "fullDescription",
          ps.color,
          ps.color_options AS "colorOptions",
          ps.depth,
          ps.panel_count AS "panelCount",
          ps.hook_length AS "hookLength",
          ps.base_price AS "basePrice",
          ps.calculated_price AS "calculatedPrice",
          ps.final_price AS "finalPrice",
          ps.images,
          ps.generated_from_matrix AS "generatedFromMatrix",
          ps.combination_key AS "combinationKey",
          ps.is_active AS "isActive",
          ps.created_at AS "createdAt",
          ps.updated_at AS "updatedAt",
          sm.modifier_type AS "modifierType",
          sm.modifier_operation AS "modifierOperation",
          sm.modifier_value AS "modifierValue",
          mp.id AS "marketplaceProductId",
          mp.offer_external_id AS "marketplaceExternalId",
          mp.source AS "marketplacePlatform",
          CASE WHEN mp.id IS NOT NULL THEN true ELSE false END AS "isLinked",
          sm.length AS "matrixLength",
          sm.width AS "matrixWidth",
          sm.height AS "matrixHeight",
          sm.doors AS "matrixDoors",
          sm.legs AS "matrixLegs",
          sm.product_group AS "matrixProductGroup",
          sm.color_options AS "matrixColorOptions",
          sm.component_product_overrides AS "matrixComponentOverrides",
          si.url AS "primaryImageUrl",
          si.thumbnail_url AS "primaryImageThumbnailUrl",
          si.medium_url AS "primaryImageMediumUrl"
        FROM product_creator.product_sets ps
        LEFT JOIN catalog.product_platform_data ppd 
          ON ppd.set_id = ps.id 
          AND ppd.link_type = 'set'
        LEFT JOIN commerce.marketplace_products mp 
          ON LOWER(mp.source) = LOWER(ppd.platform) 
          AND mp.offer_external_id = ppd.external_id
        LEFT JOIN product_creator.set_matrices sm
          ON ps.set_matrix_id = sm.id
        LEFT JOIN catalog.set_images si
          ON si.set_id = ps.id
          AND si.is_primary = true
        ORDER BY ps.id DESC, si.is_primary DESC, si.sort_order ASC
      `);
      
      // Ensure images is always an array and process matrix data
      const sets = result.rows.map(row => ({
        ...row,
        images: row.images || [],
        matrixColorOptions: row.matrixColorOptions || null,
        matrixComponentOverrides: row.matrixComponentOverrides || null,
        // Add primary image data for thumbnail display
        primaryImage: row.primaryImageUrl ? {
          url: row.primaryImageUrl,
          thumbnailUrl: row.primaryImageThumbnailUrl || row.primaryImageUrl,
          mediumUrl: row.primaryImageMediumUrl || row.primaryImageUrl
        } : null
      }));
      
      res.json(sets);
    } catch (error) {
      console.error("❌ Error fetching product sets:", error);
      res.status(500).json({ error: "Failed to fetch product sets" });
    }
  });

  // GET /api/product-sets/:id - Pobierz pojedynczy zestaw
  app.get("/api/product-sets/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const set = await storage.getProductSetById(parseInt(id, 10));
      
      if (!set) {
        return res.status(404).json({ error: "Product set not found" });
      }
      
      res.json(set);
    } catch (error) {
      console.error("❌ Error fetching product set:", error);
      res.status(500).json({ error: "Failed to fetch product set" });
    }
  });

  // GET /api/product-sets/:id/products - Pobierz produkty w zestawie
  app.get("/api/product-sets/:id/products", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Pobierz produkty wraz z datą dodania i zdjęciem
      const result = await pool.query(`
        SELECT 
          spl.id AS link_id,
          spl.product_id,
          spl.component_type,
          spl.quantity,
          spl.position,
          spl.created_at,
          p.id,
          p.sku,
          p.title,
          p.base_price AS "basePrice",
          p.is_active AS "isActive",
          (
            SELECT json_build_object(
              'url', pi.url,
              'altText', pi.alt_text,
              'isPrimary', pi.is_primary
            )
            FROM catalog.product_images pi
            WHERE pi.product_id = p.id
            ORDER BY pi.is_primary DESC, pi.sort_order ASC
            LIMIT 1
          ) AS image
        FROM product_creator.set_product_links spl
        LEFT JOIN catalog.products p ON p.id = spl.product_id
        WHERE spl.set_id = $1
        ORDER BY spl.position ASC, spl.id ASC
      `, [id]);
      
      const productsWithLinks = result.rows.map(row => ({
        linkId: row.link_id,
        productId: row.product_id,
        componentType: row.component_type,
        quantity: row.quantity,
        position: row.position,
        createdAt: row.created_at,
        product: row.id ? {
          id: row.id,
          sku: row.sku,
          title: row.title,
          basePrice: row.basePrice,
          isActive: row.isActive,
          image: row.image
        } : null
      }));
      
      res.json(productsWithLinks);
    } catch (error) {
      console.error("❌ Error fetching set products:", error);
      res.status(500).json({ error: "Failed to fetch set products" });
    }
  });

  // POST /api/product-sets - Dodaj nowy zestaw
  app.post("/api/product-sets", isAuthenticated, async (req, res) => {
    try {
      const set = await storage.createProductSet(req.body);
      res.status(201).json(set);
    } catch (error) {
      console.error("❌ Error creating product set:", error);
      res.status(500).json({ error: "Failed to create product set" });
    }
  });

  // PUT /api/product-sets/:id - Edytuj zestaw
  app.put("/api/product-sets/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const set = await storage.updateProductSet(parseInt(id, 10), req.body);
      res.json(set);
    } catch (error) {
      console.error("❌ Error updating product set:", error);
      res.status(500).json({ error: "Failed to update product set" });
    }
  });

  // DELETE /api/product-sets/:id - Usuń zestaw
  app.delete("/api/product-sets/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      await storage.deleteProductSet(parseInt(id, 10));
      res.status(204).send();
    } catch (error) {
      console.error("❌ Error deleting product set:", error);
      res.status(500).json({ error: "Failed to delete product set" });
    }
  });

  // POST /api/product-sets/:id/recalculate - Przelicz cenę zestawu z komponentów
  app.post("/api/product-sets/:id/recalculate", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const totalPrice = await recalculateSetPrice(parseInt(id, 10));
      res.json({ 
        calculatedPrice: totalPrice,
        recalculatedAt: new Date().toISOString()
      });
    } catch (error) {
      console.error("❌ Error recalculating set price:", error);
      res.status(500).json({ error: "Failed to recalculate price" });
    }
  });

  // POST /api/product-sets/:id/products - Dodaj produkt do zestawu
  app.post("/api/product-sets/:id/products", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { productId, componentType, quantity, position } = req.body;
      
      const link = await storage.createSetProductLink({
        setId: parseInt(id, 10),
        productId,
        componentType,
        quantity,
        position
      });
      
      res.status(201).json(link);
    } catch (error) {
      console.error("❌ Error adding product to set:", error);
      res.status(500).json({ error: "Failed to add product to set" });
    }
  });

  // DELETE /api/set-product-links/:id - Usuń produkt z zestawu
  app.delete("/api/set-product-links/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      await storage.deleteSetProductLink(parseInt(id, 10));
      res.status(204).send();
    } catch (error) {
      console.error("❌ Error removing product from set:", error);
      res.status(500).json({ error: "Failed to remove product from set" });
    }
  });

  // ==================== PRODUCT ACCESSORIES SYSTEM ====================
  
  // GET /api/accessory-groups - Lista wszystkich grup dodatków
  app.get("/api/accessory-groups", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          id,
          name,
          code,
          category,
          description,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
        FROM catalog.accessory_groups
        WHERE is_active = true
        ORDER BY 
          CASE WHEN category IS NULL THEN 1 ELSE 0 END,
          category ASC,
          display_order ASC,
          name ASC
      `);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching accessory groups:", error);
      res.status(500).json({ error: "Failed to fetch accessory groups" });
    }
  });

  // GET /api/accessory-groups/codes - Lista kodów grup dla generowania tagów
  app.get("/api/accessory-groups/codes", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          code,
          name,
          category
        FROM catalog.accessory_groups
        WHERE is_active = true
        ORDER BY 
          CASE WHEN category IS NULL THEN 1 ELSE 0 END,
          category ASC,
          name ASC
      `);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching accessory group codes:", error);
      res.status(500).json({ error: "Failed to fetch accessory group codes" });
    }
  });

  // POST /api/accessory-groups - Utwórz nową grupę dodatków
  app.post("/api/accessory-groups", isAuthenticated, async (req, res) => {
    try {
      const { name, code, category, description, displayOrder } = req.body;
      
      if (!name || !code) {
        return res.status(400).json({ error: "Name and code are required" });
      }
      
      const result = await pool.query(`
        INSERT INTO catalog.accessory_groups (name, code, category, description, display_order)
        VALUES ($1, $2, $3, $4, $5)
        RETURNING 
          id,
          name,
          code,
          category,
          description,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [name, code, category || null, description, displayOrder || 0]);
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') { // Unique violation
        return res.status(400).json({ error: "Accessory group with this code already exists" });
      }
      console.error("❌ Error creating accessory group:", error);
      res.status(500).json({ error: "Failed to create accessory group" });
    }
  });

  // PUT /api/accessory-groups/:id - Aktualizuj grupę dodatków
  app.put("/api/accessory-groups/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { name, code, category, description, displayOrder, isActive } = req.body;
      
      const result = await pool.query(`
        UPDATE catalog.accessory_groups 
        SET 
          name = COALESCE($1, name),
          code = COALESCE($2, code),
          category = COALESCE($3, category),
          description = $4,
          display_order = COALESCE($5, display_order),
          is_active = COALESCE($6, is_active),
          updated_at = CURRENT_TIMESTAMP
        WHERE id = $7
        RETURNING 
          id,
          name,
          code,
          category,
          description,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [name, code, category, description, displayOrder, isActive, id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Accessory group not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Accessory group with this code already exists" });
      }
      console.error("❌ Error updating accessory group:", error);
      res.status(500).json({ error: "Failed to update accessory group" });
    }
  });

  // DELETE /api/accessory-groups/:id - Usuń grupę dodatków
  app.delete("/api/accessory-groups/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        DELETE FROM catalog.accessory_groups WHERE id = $1 RETURNING id
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Accessory group not found" });
      }
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting accessory group:", error);
      res.status(500).json({ error: "Failed to delete accessory group" });
    }
  });

  // POST /api/accessory-groups/:id/duplicate - Duplikuj grupę wraz z akcesoriami
  app.post("/api/accessory-groups/:id/duplicate", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { id } = req.params;
      const { name } = req.body;
      
      if (!name || name.trim() === '') {
        return res.status(400).json({ error: "Group name is required" });
      }
      
      // Rozpocznij transakcję
      await client.query('BEGIN');
      
      // Pobierz oryginalną grupę
      const originalGroupResult = await client.query(`
        SELECT * FROM catalog.accessory_groups WHERE id = $1
      `, [id]);
      
      if (originalGroupResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Accessory group not found" });
      }
      
      const originalGroup = originalGroupResult.rows[0];
      
      // Wygeneruj unikalny kod bazując na nowej nazwie
      // Normalizuj polskie znaki przed usunięciem
      const normalizedName = name
        .normalize('NFD') // Decompose accented characters
        .replace(/[\u0300-\u036f]/g, '') // Remove diacritics
        .toLowerCase();
      
      const baseCode = normalizedName.replace(/\s+/g, '_').replace(/[^a-z0-9_]/g, '') || 'group';
      let newCode = baseCode;
      let counter = 1;
      
      console.log(`🔄 Duplikacja grupy ${id}: "${name}" → baseCode: "${baseCode}"`);
      
      // Sprawdź czy kod jest unikalny, jeśli nie - dodaj licznik
      while (true) {
        const codeCheck = await client.query(`
          SELECT id, name FROM catalog.accessory_groups WHERE code = $1
        `, [newCode]);
        
        if (codeCheck.rows.length === 0) {
          console.log(`✅ Kod "${newCode}" jest unikalny`);
          break;
        }
        
        console.log(`⚠️  Kod "${newCode}" już istnieje (grupa: "${codeCheck.rows[0].name}"), próbuję ${baseCode}_${counter}...`);
        newCode = `${baseCode}_${counter}`;
        counter++;
        
        // Safety: max 100 iteracji
        if (counter > 100) {
          console.error(`❌ Przekroczono limit 100 prób dla baseCode: ${baseCode}`);
          throw new Error('Could not generate unique code after 100 attempts');
        }
      }
      
      // Znajdź najwyższy display_order
      const maxOrderResult = await client.query(`
        SELECT COALESCE(MAX(display_order), 0) as max_order
        FROM catalog.accessory_groups
      `);
      
      const newDisplayOrder = maxOrderResult.rows[0].max_order + 1;
      
      // Utwórz nową grupę
      const newGroupResult = await client.query(`
        INSERT INTO catalog.accessory_groups (
          name,
          code,
          category,
          description,
          display_order,
          is_active
        ) VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING 
          id,
          name,
          code,
          category,
          description,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        name,
        newCode,
        originalGroup.category,
        originalGroup.description,
        newDisplayOrder,
        originalGroup.is_active
      ]);
      
      const newGroup = newGroupResult.rows[0];
      
      // Pobierz wszystkie akcesoria z oryginalnej grupy
      const accessoriesResult = await client.query(`
        SELECT * FROM catalog.accessories WHERE group_id = $1 ORDER BY display_order ASC
      `, [id]);
      
      let copiedCount = 0;
      
      // Skopiuj każde akcesorium do nowej grupy
      for (const accessory of accessoriesResult.rows) {
        // Wygeneruj unikalny kod dla akcesorium
        let accessoryCode = accessory.code;
        let accessoryCounter = 1;
        
        // Sprawdź czy kod akcesorium jest unikalny
        while (true) {
          const accessoryCodeCheck = await client.query(`
            SELECT id FROM catalog.accessories WHERE code = $1
          `, [accessoryCode]);
          
          if (accessoryCodeCheck.rows.length === 0) {
            break;
          }
          
          // Jeśli kod istnieje, dodaj licznik
          accessoryCode = `${accessory.code}_copy${accessoryCounter}`;
          accessoryCounter++;
          
          // Safety: max 100 iteracji
          if (accessoryCounter > 100) {
            throw new Error(`Could not generate unique code for accessory ${accessory.name}`);
          }
        }
        
        await client.query(`
          INSERT INTO catalog.accessories (
            group_id,
            name,
            code,
            alpma_code,
            description,
            image_url,
            display_order,
            is_active
          ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        `, [
          newGroup.id,
          accessory.name,
          accessoryCode,  // ✅ Używamy unikalnego kodu
          accessory.alpma_code,
          accessory.description,
          accessory.image_url,
          accessory.display_order,
          accessory.is_active
        ]);
        copiedCount++;
      }
      
      // Zatwierdź transakcję
      await client.query('COMMIT');
      
      console.log(`✅ Duplicated accessory group ${id} as new group ${newGroup.id} with ${copiedCount} accessories`);
      
      res.json({
        ...newGroup,
        copiedAccessoriesCount: copiedCount
      });
    } catch (error: any) {
      // Wycofaj transakcję w przypadku błędu
      await client.query('ROLLBACK');
      
      if (error.code === '23505') {
        return res.status(400).json({ error: "Accessory group with this code already exists" });
      }
      console.error("❌ Error duplicating accessory group:", error);
      res.status(500).json({ error: "Failed to duplicate accessory group" });
    } finally {
      // Zawsze zwolnij klienta do puli
      client.release();
    }
  });

  // GET /api/accessories - Lista wszystkich dodatków (z opcjonalnym filtrowaniem po grupie)
  app.get("/api/accessories", isAuthenticated, async (req, res) => {
    try {
      const { groupId, showInactive } = req.query;
      
      let query = `
        SELECT 
          a.id,
          a.group_id AS "groupId",
          a.warehouse_material_id AS "warehouseMaterialId",
          a.name,
          a.code,
          a.alpma_code AS "alpmaCode",
          a.description,
          a.image_url AS "imageUrl",
          a.display_order AS "displayOrder",
          a.is_active AS "isActive",
          a.created_at AS "createdAt",
          a.updated_at AS "updatedAt",
          g.name AS "groupName",
          g.code AS "groupCode"
        FROM catalog.accessories a
        LEFT JOIN catalog.accessory_groups g ON a.group_id = g.id
        WHERE 1=1
      `;
      
      const params: any[] = [];
      
      // Filter by active status (default: show only active)
      if (showInactive !== 'true') {
        query += ` AND a.is_active = true`;
      }
      
      if (groupId) {
        params.push(groupId);
        query += ` AND a.group_id = $${params.length}`;
      }
      
      query += ` ORDER BY a.display_order ASC, a.name ASC`;
      
      const result = await pool.query(query, params);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching accessories:", error);
      res.status(500).json({ error: "Failed to fetch accessories" });
    }
  });

  // GET /api/accessories/:id - Pobierz konkretny dodatek
  app.get("/api/accessories/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT 
          a.id,
          a.group_id AS "groupId",
          a.warehouse_material_id AS "warehouseMaterialId",
          a.name,
          a.code,
          a.alpma_code AS "alpmaCode",
          a.description,
          a.image_url AS "imageUrl",
          a.display_order AS "displayOrder",
          a.is_active AS "isActive",
          a.created_at AS "createdAt",
          a.updated_at AS "updatedAt",
          g.name AS "groupName",
          g.code AS "groupCode"
        FROM catalog.accessories a
        LEFT JOIN catalog.accessory_groups g ON a.group_id = g.id
        WHERE a.id = $1
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Accessory not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error fetching accessory:", error);
      res.status(500).json({ error: "Failed to fetch accessory" });
    }
  });

  // POST /api/upload/accessory-image - Upload zdjęcia akcesorium
  app.post("/api/upload/accessory-image", isAuthenticated, uploadAccessoryImage.single('image'), async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ error: "No image file provided" });
      }

      // Generate filename and upload through adapter
      const ext = path.extname(req.file.originalname);
      const filename = generateAccessoryImageFilename(ext);
      
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      const imageUrl = await adapter.upload({
        filename,
        buffer: req.file.buffer,
        mimetype: req.file.mimetype,
        subfolder: 'accessories/images'
      });
      
      res.json({ 
        url: imageUrl,
        filename 
      });
    } catch (error: any) {
      console.error("❌ Error uploading accessory image:", error);
      res.status(500).json({ error: "Failed to upload image" });
    }
  });

  // POST /api/accessories - Utwórz nowy dodatek
  app.post("/api/accessories", isAuthenticated, async (req, res) => {
    try {
      const { groupId, warehouseMaterialId, name, code, alpmaCode, description, imageUrl,
        unit, displayOrder, price } = req.body;
      
      if (!name || !code) {
        return res.status(400).json({ error: "Name and code are required" });
      }
      
      const result = await pool.query(`
        INSERT INTO catalog.accessories (group_id, warehouse_material_id, name, code, alpma_code, description, image_url, display_order, price)
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, true, $10)
        RETURNING 
          id,
          group_id AS "groupId",
          warehouse_material_id AS "warehouseMaterialId",
          name,
          code,
          alpma_code AS "alpmaCode",
          description,
          image_url AS "imageUrl",
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [groupId, warehouseMaterialId, name, code, alpmaCode, description, imageUrl,
        unit, displayOrder || 0, price]);
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Accessory with this code already exists" });
      }
      console.error("❌ Error creating accessory:", error);
      res.status(500).json({ error: "Failed to create accessory" });
    }
  });

  // PUT /api/accessories/:id - Aktualizuj dodatek
  app.put("/api/accessories/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { groupId, warehouseMaterialId, name, code, alpmaCode, description, imageUrl,
        unit, displayOrder, isActive, price } = req.body;
      
      const result = await pool.query(`
        UPDATE catalog.accessories 
        SET 
          group_id = COALESCE($1, group_id),
          warehouse_material_id = $2,
          name = COALESCE($3, name),
          code = COALESCE($4, code),
          alpma_code = COALESCE($5, alpma_code),
          description = COALESCE($6, description),
          image_url = COALESCE($7, image_url),
          display_order = COALESCE($8, display_order),
          is_active = COALESCE($9, is_active),
          price = $10,
          updated_at = CURRENT_TIMESTAMP
        WHERE id = $11
        RETURNING 
          id,
          group_id AS "groupId",
          warehouse_material_id AS "warehouseMaterialId",
          name,
          code,
          alpma_code AS "alpmaCode",
          description,
          image_url AS "imageUrl",
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [groupId, warehouseMaterialId, name, code, alpmaCode, description, imageUrl,
        unit, displayOrder, isActive, price, id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Accessory not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Accessory with this code already exists" });
      }
      console.error("❌ Error updating accessory:", error);
      res.status(500).json({ error: "Failed to update accessory" });
    }
  });

  // DELETE /api/accessories/:id - Usuń dodatek
  app.delete("/api/accessories/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        DELETE FROM catalog.accessories WHERE id = $1 RETURNING id
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Accessory not found" });
      }
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting accessory:", error);
      res.status(500).json({ error: "Failed to delete accessory" });
    }
  });

  // POST /api/accessories/:id/duplicate - Duplikuj dodatek do innej grupy
  app.post("/api/accessories/:id/duplicate", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { targetGroupId } = req.body;
      
      if (!targetGroupId) {
        return res.status(400).json({ error: "Target group ID is required" });
      }
      
      // Pobierz oryginalne akcesorium
      const originalResult = await pool.query(`
        SELECT * FROM catalog.accessories WHERE id = $1
      `, [id]);
      
      if (originalResult.rows.length === 0) {
        return res.status(404).json({ error: "Accessory not found" });
      }
      
      const original = originalResult.rows[0];
      
      // Sprawdź czy docelowa grupa istnieje
      const groupCheck = await pool.query(`
        SELECT id FROM catalog.accessory_groups WHERE id = $1
      `, [targetGroupId]);
      
      if (groupCheck.rows.length === 0) {
        return res.status(404).json({ error: "Target group not found" });
      }
      
      // Wygeneruj globalnie unikalny kod (constraint jest na całej tabeli, nie per grupa)
      let newCode = original.code;
      let counter = 1;
      
      // Sprawdź czy kod jest unikalny globalnie, jeśli nie - dodaj licznik
      while (true) {
        const codeCheck = await pool.query(`
          SELECT id FROM catalog.accessories 
          WHERE code = $1
        `, [newCode]);
        
        if (codeCheck.rows.length === 0) {
          break;
        }
        
        // Dodaj lub zwiększ licznik
        newCode = `${original.code}_${counter}`;
        counter++;
      }
      
      // Znajdź najwyższy display_order w docelowej grupie
      const maxOrderResult = await pool.query(`
        SELECT COALESCE(MAX(display_order), 0) as max_order
        FROM catalog.accessories
        WHERE group_id = $1
      `, [targetGroupId]);
      
      const newDisplayOrder = maxOrderResult.rows[0].max_order + 1;
      
      // Skopiuj zdjęcie jeśli istnieje (możemy użyć tego samego URL)
      const imageUrl = original.image_url;
      
      // Utwórz duplikat z nowym groupId
      const duplicateResult = await pool.query(`
        INSERT INTO catalog.accessories (
          group_id,
          name,
          code,
          alpma_code,
          description,
          image_url,
          display_order,
          is_active
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
        RETURNING 
          id,
          group_id AS "groupId",
          name,
          code,
          alpma_code AS "alpmaCode",
          description,
          image_url AS "imageUrl",
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        targetGroupId,
        original.name,
        newCode,
        original.alpma_code,
        original.description,
        imageUrl,
        newDisplayOrder,
        original.is_active
      ]);
      
      console.log(`✅ Duplicated accessory ${id} to group ${targetGroupId} as new ID ${duplicateResult.rows[0].id} with code ${newCode}`);
      
      res.json(duplicateResult.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Accessory with this code already exists in target group" });
      }
      console.error("❌ Error duplicating accessory:", error);
      res.status(500).json({ error: "Failed to duplicate accessory" });
    }
  });

  // POST /api/accessories/bulk-delete - Usuń wiele dodatków
  app.post("/api/accessories/bulk-delete", isAuthenticated, async (req, res) => {
    try {
      const { accessoryIds } = req.body;

      if (!accessoryIds || !Array.isArray(accessoryIds) || accessoryIds.length === 0) {
        return res.status(400).json({ error: "Accessory IDs array is required" });
      }

      const result = await pool.query(`
        DELETE FROM catalog.accessories WHERE id = ANY($1) RETURNING id
      `, [accessoryIds]);

      console.log(`✅ Bulk deleted ${result.rowCount} accessories`);

      res.json({ 
        success: true, 
        deletedCount: result.rowCount,
        accessoryIds: result.rows.map(r => r.id)
      });
    } catch (error) {
      console.error("❌ Error bulk deleting accessories:", error);
      res.status(500).json({ error: "Failed to bulk delete accessories" });
    }
  });

  // POST /api/accessories/bulk-edit - Edytuj wiele dodatków
  app.post("/api/accessories/bulk-edit", isAuthenticated, async (req, res) => {
    try {
      const { accessoryIds, updates } = req.body;

      if (!accessoryIds || !Array.isArray(accessoryIds) || accessoryIds.length === 0) {
        return res.status(400).json({ error: "Accessory IDs array is required" });
      }

      if (!updates || typeof updates !== 'object') {
        return res.status(400).json({ error: "Updates object is required" });
      }

      // Build dynamic UPDATE query based on provided fields
      const updateFields: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      if (updates.isActive !== undefined) {
        updateFields.push(`is_active = $${paramIndex}`);
        values.push(updates.isActive);
        paramIndex++;
      }

      if (updates.description !== undefined) {
        updateFields.push(`description = $${paramIndex}`);
        values.push(updates.description);
        paramIndex++;
      }

      if (updateFields.length === 0) {
        return res.status(400).json({ error: "No valid update fields provided" });
      }

      // Add updated_at
      updateFields.push(`updated_at = NOW()`);

      // Add accessoryIds to values
      values.push(accessoryIds);

      const query = `
        UPDATE catalog.accessories 
        SET ${updateFields.join(', ')}
        WHERE id = ANY($${paramIndex})
        RETURNING id
      `;

      const result = await pool.query(query, values);

      console.log(`✅ Bulk updated ${result.rowCount} accessories with:`, updates);

      res.json({ 
        success: true, 
        updatedCount: result.rowCount,
        accessoryIds: result.rows.map(r => r.id)
      });
    } catch (error) {
      console.error("❌ Error bulk editing accessories:", error);
      res.status(500).json({ error: "Failed to bulk edit accessories" });
    }
  });

  // POST /api/accessories/bulk-import - Bulk import zdjęć akcesoriów
  app.post("/api/accessories/bulk-import", isAuthenticated, uploadAccessoryImage.array('images', 50), async (req, res) => {
    const client = await pool.connect();
    
    try {
      const files = req.files as Express.Multer.File[];
      const { groupId } = req.body;
      
      if (!files || files.length === 0) {
        return res.status(400).json({ error: "No images provided" });
      }

      // Get adapter instance
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();

      await client.query('BEGIN');
      
      const created = [];
      const errors = [];

      for (const file of files) {
        try {
          // Pobierz nazwę pliku bez rozszerzenia
          const fileName = path.basename(file.originalname, path.extname(file.originalname));
          
          // Generate filename and upload through adapter
          const ext = path.extname(file.originalname);
          const filename = generateAccessoryImageFilename(ext);
          
          const imageUrl = await adapter.upload({
            filename,
            buffer: file.buffer,
            mimetype: file.mimetype,
            subfolder: 'accessories/images'
          });
          
          // Generuj unikalny kod z nazwy pliku (zamień spacje i znaki specjalne na dash)
          const code = fileName
            .toLowerCase()
            .replace(/[^a-z0-9]+/g, '-')
            .replace(/^-+|-+$/g, '');
          
          // Sprawdź czy kod już istnieje, jeśli tak dodaj suffix
          let uniqueCode = code;
          let suffix = 1;
          while (true) {
            const existing = await client.query(
              'SELECT id FROM catalog.accessories WHERE code = $1',
              [uniqueCode]
            );
            if (existing.rows.length === 0) break;
            uniqueCode = `${code}-${suffix}`;
            suffix++;
          }
          
          const result = await client.query(`
            INSERT INTO catalog.accessories (group_id, name, code, image_url, display_order)
            VALUES ($1, $2, $3, $4, 0)
            RETURNING 
              id,
              group_id AS "groupId",
              name,
              code,
              image_url AS "imageUrl",
              display_order AS "displayOrder"
          `, [groupId || null, fileName, uniqueCode, imageUrl]);
          
          created.push(result.rows[0]);
        } catch (error: any) {
          errors.push({
            fileName: file.originalname,
            error: error.message
          });
        }
      }
      
      await client.query('COMMIT');
      
      res.json({
        success: true,
        created: created.length,
        errors: errors.length,
        items: created,
        errorDetails: errors
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error in bulk import:", error);
      res.status(500).json({ error: "Failed to import accessories" });
    } finally {
      client.release();
    }
  });

  // POST /api/accessories/import-csv - Import akcesoriów z CSV
  app.post("/api/accessories/import-csv", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { csvData } = req.body;
      
      if (!csvData || typeof csvData !== 'string') {
        return res.status(400).json({ error: "CSV data is required" });
      }

      // Parse CSV using PapaParse
      const parseResult = Papa.parse(csvData, {
        header: true,
        skipEmptyLines: true,
        transformHeader: (header: string) => header.trim(),
      });

      if (parseResult.errors.length > 0) {
        return res.status(400).json({ 
          error: "CSV parsing failed",
          details: parseResult.errors.map(e => e.message)
        });
      }

      const rows = parseResult.data as any[];
      
      if (rows.length === 0) {
        return res.status(400).json({ error: "CSV file is empty" });
      }

      // Validate required columns
      const requiredColumns = ['name', 'code'];
      const firstRow = rows[0];
      const missingColumns = requiredColumns.filter(col => !(col in firstRow));
      if (missingColumns.length > 0) {
        return res.status(400).json({ 
          error: `Missing required columns: ${missingColumns.join(', ')}` 
        });
      }

      // Fetch all groups for lookup
      const groupsResult = await pool.query(`
        SELECT id, code FROM catalog.accessory_groups
      `);
      const groupsMap = new Map<string, number>();
      groupsResult.rows.forEach((g: any) => {
        groupsMap.set(g.code, g.id);
      });

      const results = {
        success: 0,
        failed: 0,
        errors: [] as Array<{ row: number; error: string; data?: any }>
      };

      // Start transaction
      await client.query('BEGIN');

      // Process each row
      for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const rowNumber = i + 2; // +2 because: +1 for 1-indexing, +1 for header row

        try {
          // Validate name and code
          if (!row.name || row.name.trim() === '') {
            results.errors.push({
              row: rowNumber,
              error: 'Name is required',
              data: row
            });
            results.failed++;
            continue;
          }

          if (!row.code || row.code.trim() === '') {
            results.errors.push({
              row: rowNumber,
              error: 'Code is required',
              data: row
            });
            results.failed++;
            continue;
          }

          // Resolve group_id from group_code
          let groupId = null;
          if (row.group_code && row.group_code.trim() !== '') {
            groupId = groupsMap.get(row.group_code.trim());
            if (groupId === undefined) {
              results.errors.push({
                row: rowNumber,
                error: `Group code not found: ${row.group_code}`,
                data: row
              });
              results.failed++;
              continue;
            }
          }

          // Parse display_order
          const displayOrder = row.display_order ? parseInt(row.display_order, 10) : 0;

          // Insert into database
          await client.query(`
            INSERT INTO catalog.accessories (
              group_id, name, code, description, image_url, display_order
            )
            VALUES ($1, $2, $3, $4, $5, $6)
          `, [
            groupId,
            row.name.trim(),
            row.code.trim(),
            row.description?.trim() || null,
            row.image_url?.trim() || null,
            displayOrder
          ]);

          results.success++;

          results.success++;
        } catch (error: any) {
          console.error(`❌ Error importing row ${rowNumber}:`, error);
          
          // Handle unique constraint violation
          if (error.code === '23505') {
            results.errors.push({
              row: rowNumber,
              error: 'Accessory with this code already exists',
              data: row
            });
          } else {
            results.errors.push({
              row: rowNumber,
              error: error instanceof Error ? error.message : 'Unknown error',
              data: row
            });
          }
          results.failed++;
        }
      }

      // Rollback if any errors occurred, otherwise commit
      if (results.failed > 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({
          error: "Import failed - no changes were made to the database",
          ...results
        });
      }
      
      // Commit transaction if all rows succeeded
      await client.query('COMMIT');

      res.json(results);
    } catch (error) {
      // Rollback transaction on error
      await client.query('ROLLBACK');
      console.error("❌ Error importing CSV:", error);
      res.status(500).json({ error: "Failed to import CSV" });
    } finally {
      client.release();
    }
  });

  // NOTE: Matrix accessories endpoints removed - accessories are now assigned via template tags only
  // Accessories are dynamically extracted from template HTML using {{akcesorium-CODE}} syntax
  // See buildTemplateContext() and extractAccessoryTags() functions for implementation

  // Recent updates checker removed - Allegro now uses incremental sync

  // ===== WAREHOUSE (MAGAZYN) =====
  
  // GET /api/warehouse/material-groups - Lista wszystkich grup materiałów
  app.get("/api/warehouse/material-groups", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          id,
          name,
          code,
          category,
          description,
          color,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
        FROM warehouse.material_groups
        WHERE is_active = true
        ORDER BY 
          CASE WHEN category IS NULL THEN 1 ELSE 0 END,
          category ASC,
          display_order ASC,
          name ASC
      `);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching material groups:", error);
      res.status(500).json({ error: "Failed to fetch material groups" });
    }
  });
  
  // POST /api/warehouse/material-groups - Utwórz nową grupę materiałów
  app.post("/api/warehouse/material-groups", isAuthenticated, async (req, res) => {
    try {
      const { name, code, category, description, displayOrder, color } = req.body;
      
      if (!name || !code) {
        return res.status(400).json({ error: "Name and code are required" });
      }
      
      const result = await pool.query(`
        INSERT INTO warehouse.material_groups (name, code, category, description, display_order, color)
        VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING 
          id,
          name,
          code,
          category,
          description,
          color,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [name, code, category || null, description, displayOrder || 0, color || null]);
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Material group with this code already exists" });
      }
      console.error("❌ Error creating material group:", error);
      res.status(500).json({ error: "Failed to create material group" });
    }
  });
  
  // PATCH /api/warehouse/material-groups/:id - Aktualizuj grupę materiałów
  app.patch("/api/warehouse/material-groups/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { name, code, category, description, displayOrder, isActive, color } = req.body;
      
      const result = await pool.query(`
        UPDATE warehouse.material_groups
        SET 
          name = COALESCE($1, name),
          code = COALESCE($2, code),
          category = COALESCE($3, category),
          description = COALESCE($4, description),
          display_order = COALESCE($5, display_order),
          is_active = COALESCE($6, is_active),
          color = COALESCE($7, color),
          updated_at = CURRENT_TIMESTAMP
        WHERE id = $8
        RETURNING 
          id,
          name,
          code,
          category,
          description,
          color,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [name, code, category, description, displayOrder, isActive, color, id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Material group not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Material group with this code already exists" });
      }
      console.error("❌ Error updating material group:", error);
      res.status(500).json({ error: "Failed to update material group" });
    }
  });
  
  // DELETE /api/warehouse/material-groups/:id - Usuń grupę materiałów
  app.delete("/api/warehouse/material-groups/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      await pool.query(
        'DELETE FROM warehouse.material_groups WHERE id = $1',
        [id]
      );
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting material group:", error);
      res.status(500).json({ error: "Failed to delete material group" });
    }
  });

  // ===== WAREHOUSE CATEGORIES =====

  // GET /api/warehouse/categories - Lista wszystkich kategorii magazynu
  app.get("/api/warehouse/categories", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          id,
          name,
          code,
          icon,
          icon_color AS "iconColor",
          description,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
        FROM warehouse.categories
        WHERE is_active = true
        ORDER BY display_order ASC, name ASC
      `);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching warehouse categories:", error);
      res.status(500).json({ error: "Failed to fetch warehouse categories" });
    }
  });

  // POST /api/warehouse/categories - Utwórz nową kategorię
  app.post("/api/warehouse/categories", requirePermission("manage_settings"), async (req, res) => {
    try {
      const { name, code, icon, iconColor, description, displayOrder } = req.body;
      
      if (!name || !code) {
        return res.status(400).json({ error: "Name and code are required" });
      }
      
      const result = await pool.query(`
        INSERT INTO warehouse.categories (name, code, icon, icon_color, description, display_order)
        VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING 
          id, name, code, icon, icon_color AS "iconColor", description,
          display_order AS "displayOrder", is_active AS "isActive",
          created_at AS "createdAt", updated_at AS "updatedAt"
      `, [name, code, icon || 'Box', iconColor || 'text-gray-600/70', description, displayOrder || 0]);
      
      res.status(201).json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Category with this code already exists" });
      }
      console.error("❌ Error creating warehouse category:", error);
      res.status(500).json({ error: "Failed to create warehouse category" });
    }
  });

  // PATCH /api/warehouse/categories/reorder - Zmień kolejność kategorii
  // MUSI BYĆ PRZED /:id aby routing działał poprawnie
  app.patch("/api/warehouse/categories/reorder", requirePermission("manage_settings"), async (req, res) => {
    try {
      const { categoryIds } = req.body; // Array of category IDs in new order
      
      if (!Array.isArray(categoryIds)) {
        return res.status(400).json({ error: "categoryIds must be an array" });
      }
      
      // Update display_order for each category
      const client = await pool.connect();
      try {
        await client.query('BEGIN');
        
        for (let i = 0; i < categoryIds.length; i++) {
          await client.query(
            'UPDATE warehouse.categories SET display_order = $1, updated_at = NOW() WHERE id = $2',
            [i + 1, categoryIds[i]]
          );
        }
        
        await client.query('COMMIT');
        
        // Fetch updated categories
        const result = await client.query(`
          SELECT 
            id, name, code, icon, icon_color AS "iconColor", description,
            display_order AS "displayOrder", is_active AS "isActive",
            created_at AS "createdAt", updated_at AS "updatedAt"
          FROM warehouse.categories
          WHERE is_active = true
          ORDER BY display_order ASC
        `);
        
        res.json(result.rows);
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error) {
      console.error("❌ Error reordering warehouse categories:", error);
      res.status(500).json({ error: "Failed to reorder warehouse categories" });
    }
  });

  // PATCH /api/warehouse/categories/:id - Aktualizuj kategorię
  app.patch("/api/warehouse/categories/:id", requirePermission("manage_settings"), async (req, res) => {
    try {
      const { id } = req.params;
      const { name, code, icon, iconColor, description, displayOrder, isActive } = req.body;
      
      const result = await pool.query(`
        UPDATE warehouse.categories
        SET 
          name = COALESCE($1, name),
          code = COALESCE($2, code),
          icon = COALESCE($3, icon),
          icon_color = COALESCE($4, icon_color),
          description = COALESCE($5, description),
          display_order = COALESCE($6, display_order),
          is_active = COALESCE($7, is_active),
          updated_at = NOW()
        WHERE id = $8
        RETURNING 
          id, name, code, icon, icon_color AS "iconColor", description,
          display_order AS "displayOrder", is_active AS "isActive",
          created_at AS "createdAt", updated_at AS "updatedAt"
      `, [name, code, icon, iconColor, description, displayOrder, isActive, id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Category not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Category with this code already exists" });
      }
      console.error("❌ Error updating warehouse category:", error);
      res.status(500).json({ error: "Failed to update warehouse category" });
    }
  });

  // DELETE /api/warehouse/categories/:id - Usuń kategorię
  app.delete("/api/warehouse/categories/:id", requirePermission("manage_settings"), async (req, res) => {
    try {
      const { id } = req.params;
      
      // Sprawdź czy kategoria ma przypisane grupy
      const checkGroups = await pool.query(
        'SELECT COUNT(*) as count FROM warehouse.material_groups WHERE category = (SELECT code FROM warehouse.categories WHERE id = $1)',
        [id]
      );
      
      if (parseInt(checkGroups.rows[0].count, 10) > 0) {
        return res.status(400).json({ 
          error: "Nie można usunąć kategorii, która ma przypisane grupy materiałów",
          groupCount: parseInt(checkGroups.rows[0].count, 10)
        });
      }
      
      await pool.query('DELETE FROM warehouse.categories WHERE id = $1', [id]);
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting warehouse category:", error);
      res.status(500).json({ error: "Failed to delete warehouse category" });
    }
  });

  // GET /api/warehouse/categories/by-code/:code - Pobierz kategorię po kodzie
  app.get("/api/warehouse/categories/by-code/:code", isAuthenticated, async (req, res) => {
    try {
      const { code } = req.params;
      
      const result = await pool.query(`
        SELECT 
          id, name, code, icon, icon_color AS "iconColor", description,
          display_order AS "displayOrder", is_active AS "isActive",
          created_at AS "createdAt", updated_at AS "updatedAt"
        FROM warehouse.categories
        WHERE code = $1
      `, [code]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Category not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error fetching warehouse category by code:", error);
      res.status(500).json({ error: "Failed to fetch warehouse category" });
    }
  });

  // GET /api/warehouse/categories/:categoryId/groups - Pobierz grupy dla kategorii
  app.get("/api/warehouse/categories/:categoryId/groups", requirePermission("manage_settings"), async (req, res) => {
    try {
      const { categoryId } = req.params;
      
      // Najpierw pobierz kod kategorii
      const categoryResult = await pool.query(
        'SELECT code FROM warehouse.categories WHERE id = $1',
        [categoryId]
      );
      
      if (categoryResult.rows.length === 0) {
        return res.status(404).json({ error: "Category not found" });
      }
      
      const categoryCode = categoryResult.rows[0].code;
      
      // Pobierz grupy dla tej kategorii
      const result = await pool.query(`
        SELECT 
          id,
          name,
          code,
          category,
          $1::integer AS "categoryId",
          description,
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
        FROM warehouse.material_groups
        WHERE category = $2
        ORDER BY display_order ASC, name ASC
      `, [categoryId, categoryCode]);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching warehouse category groups:", error);
      res.status(500).json({ error: "Failed to fetch warehouse category groups" });
    }
  });

  // POST /api/warehouse/categories/:categoryId/groups - Utwórz nową grupę dla kategorii
  app.post("/api/warehouse/categories/:categoryId/groups", requirePermission("manage_settings"), async (req, res) => {
    try {
      const { categoryId } = req.params;
      const { name, description } = req.body;
      
      // Pobierz kod kategorii
      const categoryResult = await pool.query(
        'SELECT code FROM warehouse.categories WHERE id = $1',
        [categoryId]
      );
      
      if (categoryResult.rows.length === 0) {
        return res.status(404).json({ error: "Category not found" });
      }
      
      const categoryCode = categoryResult.rows[0].code;
      
      // Wygeneruj unikalny kod dla grupy
      const codeBase = name.toLowerCase()
        .replace(/\s+/g, '_')
        .replace(/[^a-z0-9_]/g, '');
      
      let code = codeBase;
      let counter = 1;
      
      while (true) {
        const existing = await pool.query(
          'SELECT id FROM warehouse.material_groups WHERE code = $1',
          [code]
        );
        
        if (existing.rows.length === 0) break;
        code = `${codeBase}_${counter}`;
        counter++;
      }
      
      // Utwórz grupę
      const result = await pool.query(`
        INSERT INTO warehouse.material_groups (name, code, category, description)
        VALUES ($1, $2, $3, $4)
        RETURNING 
          id, name, code, category, $5::integer AS "categoryId", description,
          display_order AS "displayOrder", is_active AS "isActive",
          created_at AS "createdAt", updated_at AS "updatedAt"
      `, [name, code, categoryCode, description || null, categoryId]);
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Group with this code already exists" });
      }
      console.error("❌ Error creating warehouse group:", error);
      res.status(500).json({ error: "Failed to create warehouse group" });
    }
  });

  // PATCH /api/warehouse/groups/:id - Aktualizuj grupę
  app.patch("/api/warehouse/groups/:id", requirePermission("manage_settings"), async (req, res) => {
    try {
      const { id } = req.params;
      const { name, description } = req.body;
      
      const result = await pool.query(`
        UPDATE warehouse.material_groups
        SET 
          name = COALESCE($1, name),
          description = COALESCE($2, description),
          updated_at = NOW()
        WHERE id = $3
        RETURNING 
          id, name, code, category, description,
          display_order AS "displayOrder", is_active AS "isActive",
          created_at AS "createdAt", updated_at AS "updatedAt"
      `, [name, description, id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Group not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating warehouse group:", error);
      res.status(500).json({ error: "Failed to update warehouse group" });
    }
  });

  // DELETE /api/warehouse/groups/:id - Usuń grupę
  app.delete("/api/warehouse/groups/:id", requirePermission("manage_settings"), async (req, res) => {
    try {
      const { id } = req.params;
      
      // Sprawdź czy grupa ma przypisane materiały
      const checkMaterials = await pool.query(
        'SELECT COUNT(*) as count FROM warehouse.materials WHERE group_id = $1',
        [id]
      );
      
      if (parseInt(checkMaterials.rows[0].count, 10) > 0) {
        return res.status(400).json({ 
          error: "Nie można usunąć grupy, która ma przypisane materiały",
          materialCount: parseInt(checkMaterials.rows[0].count, 10)
        });
      }
      
      const result = await pool.query(
        'DELETE FROM warehouse.material_groups WHERE id = $1 RETURNING id',
        [id]
      );
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Group not found" });
      }
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting warehouse group:", error);
      res.status(500).json({ error: "Failed to delete warehouse group" });
    }
  });
  
  // GET /api/warehouse/materials/all - All materials for combobox (no category filter)
  app.get("/api/warehouse/materials/all", isAuthenticated, async (req, res) => {
    try {
      const query = `
        SELECT 
          m.id,
          m.name,
          m.internal_code AS "internalCode",
          NULL AS "supplierCode",
          m.description,
          m.primary_image_url AS "primaryImageUrl",
          m.gallery,
          mg.category,
          mg.name AS "groupName"
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        WHERE m.is_active = true
        ORDER BY m.name ASC
      `;
      
      const result = await pool.query(query);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching all materials:", error);
      res.status(500).json({ error: "Failed to fetch materials" });
    }
  });

  // GET /api/warehouse/plyty - Lista płyt meblowych do wyboru w combobox
  app.get("/api/warehouse/plyty", isAuthenticated, async (req, res) => {
    try {
      const query = `
        SELECT 
          m.id,
          m.name,
          m.internal_code AS "internalCode",
          NULL AS "supplierCode",
          mg.name AS "groupName",
          m.specifications->>'thickness' AS "thickness"
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        WHERE mg.category = 'plyty' AND m.is_active = true
        ORDER BY m.name ASC
      `;
      
      const result = await pool.query(query);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching plyty materials:", error);
      res.status(500).json({ error: "Failed to fetch plyty materials" });
    }
  });

  // GET /api/warehouse/obrzeza - Lista obrzeży do wyboru w combobox
  app.get("/api/warehouse/obrzeza", isAuthenticated, async (req, res) => {
    try {
      const query = `
        SELECT 
          m.id,
          m.name,
          m.internal_code AS "internalCode",
          NULL AS "supplierCode",
          mg.name AS "groupName",
          m.specifications->>'width' AS "width",
          m.specifications->>'thickness' AS "thickness"
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        WHERE mg.category = 'obrzeza' AND m.is_active = true
        ORDER BY m.name ASC
      `;
      
      const result = await pool.query(query);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching obrzeza materials:", error);
      res.status(500).json({ error: "Failed to fetch obrzeza materials" });
    }
  });

  // GET /api/warehouse/materials/search - Zaawansowane wyszukiwanie z paginacją server-side
  app.get("/api/warehouse/materials/search", isAuthenticated, async (req, res) => {
    try {
      const {
        category,
        groupCode,
        search = '',
        page = '1',
        limit = '25',
        sortBy = 'name',
        sortOrder = 'ASC',
        groupBy = null,
        archived,
        colors,
        excludeColors,
        lengths,
        excludeLengths,
        widths,
        excludeWidths,
        stockFilter,
        cz1Codes,
        excludeCz1Codes,
        cz2Codes,
        excludeCz2Codes,
        furnitureTypes,
        sources,
        locationIds,
        excludeLocationIds,
        carrierIds,
        productTypes,
        excludeProductTypes,
        excludeCarrierIds
      } = req.query;

      const pageNum = parseInt(page as string, 10);
      const limitNum = parseInt(limit as string, 10);
      const offset = (pageNum - 1) * limitNum;

      // Parse search query - split by ; for AND, by , for OR
      const searchTerms: string[] = [];
      if (search) {
        const andTerms = (search as string).split(';').map(t => t.trim()).filter(Boolean);
        searchTerms.push(...andTerms);
      }

      // Special handling for produkty-spakowane category - use packed_products table
      if (category === 'produkty-spakowane') {
        const showArchived = archived === 'true';
        let whereClause = showArchived ? 'pp.is_archived = true' : 'pp.is_active = true AND pp.is_archived = false';
        const params: any[] = [];
        let paramIndex = 1;
        
        if (groupCode) {
          whereClause += ` AND mg.code = $${paramIndex}`;
          params.push(groupCode);
          paramIndex++;
        }

        if (searchTerms.length > 0) {
          const searchConditions = searchTerms.map(() => {
            const condition = `(
              LOWER(pp.product_name) LIKE $${paramIndex} OR
              LOWER(pp.product_sku) LIKE $${paramIndex} OR
              LOWER(pp.notes) LIKE $${paramIndex} OR
              LOWER(mg.name) LIKE $${paramIndex}
            )`;
            paramIndex++;
            return condition;
          });
          whereClause += ` AND ${searchConditions.join(' AND ')}`;
          searchTerms.forEach(term => {
            params.push(`%${term.toLowerCase()}%`);
          });
        }

        // Filter by groupCode if provided
        if (groupCode) {
          whereClause += ` AND mg.code = $${paramIndex}`;
          params.push(groupCode);
          paramIndex++;
        }

        // Filter by product types (include)
        if (productTypes) {
          const typeList = (productTypes as string).split(",").map(t => t.trim()).filter(Boolean);
          if (typeList.length > 0) {
            whereClause += ` AND pp.product_type IN (${typeList.map(() => `$${paramIndex++}`).join(", ")})`;
            typeList.forEach(t => params.push(t));
          }
        }

        // Filter by product types (exclude)
        if (excludeProductTypes) {
          const excludeTypeList = (excludeProductTypes as string).split(",").map(t => t.trim()).filter(Boolean);
          if (excludeTypeList.length > 0) {
            whereClause += ` AND (pp.product_type IS NULL OR pp.product_type NOT IN (${excludeTypeList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeTypeList.forEach(t => params.push(t));
          }
        }

        // Filter by colors (include)
        if (colors) {
          const colorList = (colors as string).split(",").map(c => c.trim()).filter(Boolean);
          if (colorList.length > 0) {
            whereClause += ` AND cp.color IN (${colorList.map(() => `$${paramIndex++}`).join(", ")})`;
            colorList.forEach(c => params.push(c));
          }
        }

        // Filter by colors (exclude)
        if (excludeColors) {
          const excludeColorList = (excludeColors as string).split(",").map(c => c.trim()).filter(Boolean);
          if (excludeColorList.length > 0) {
            whereClause += ` AND (cp.color IS NULL OR cp.color NOT IN (${excludeColorList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeColorList.forEach(c => params.push(c));
          }
        }

        // Filter by locations (include)
        if (locationIds) {
          const locList = (locationIds as string).split(",").map(l => parseInt(l.trim())).filter(l => !isNaN(l));
          if (locList.length > 0) {
            whereClause += ` AND pp.location_id IN (${locList.map(() => `$${paramIndex++}`).join(", ")})`;
            locList.forEach(l => params.push(l));
          }
        }

        // Filter by locations (exclude)
        if (excludeLocationIds) {
          const excludeLocList = (excludeLocationIds as string).split(",").map(l => parseInt(l.trim())).filter(l => !isNaN(l));
          if (excludeLocList.length > 0) {
            whereClause += ` AND (pp.location_id IS NULL OR pp.location_id NOT IN (${excludeLocList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeLocList.forEach(l => params.push(l));
          }
        }

        // Filter by carriers (include)
        if (carrierIds) {
          const carrList = (carrierIds as string).split(",").map(c => parseInt(c.trim())).filter(c => !isNaN(c));
          if (carrList.length > 0) {
            whereClause += ` AND pp.carrier_id IN (${carrList.map(() => `$${paramIndex++}`).join(", ")})`;
            carrList.forEach(c => params.push(c));
          }
        }

        // Filter by carriers (exclude)
        if (excludeCarrierIds) {
          const excludeCarrList = (excludeCarrierIds as string).split(",").map(c => parseInt(c.trim())).filter(c => !isNaN(c));
          if (excludeCarrList.length > 0) {
            whereClause += ` AND (pp.carrier_id IS NULL OR pp.carrier_id NOT IN (${excludeCarrList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeCarrList.forEach(c => params.push(c));
          }
        }

        // Stock filter
        if (stockFilter === "inStock") {
          whereClause += ` AND pp.quantity > 0`;
        } else if (stockFilter === "outOfStock") {
          whereClause += ` AND pp.quantity = 0`;
        }

        const allowedSortColumns: Record<string, string> = {
          name: 'pp.product_name',
          internalCode: 'pp.product_sku',
          supplierCode: 'pp.product_sku',
          groupName: 'mg.name',
          createdAt: 'pp.created_at',
          updatedAt: 'pp.updated_at',
          catalogProductType: 'pp.product_type',
          catalogLength: 'cp.length',
          catalogWidth: 'cp.width',
          catalogColor: 'cp.color',
          catalogLegs: 'cp.legs',
          catalogDoors: 'cp.doors',
          quantity: 'pp.quantity'
        };

        const sortColumn = allowedSortColumns[sortBy as string] || 'pp.product_name';
        const sortDirection = (sortOrder as string).toUpperCase() === 'DESC' ? 'DESC' : 'ASC';

        const countQuery = `
          SELECT COUNT(*)::int as total
          FROM warehouse.packed_products pp
          LEFT JOIN warehouse.material_groups mg ON pp.group_id = mg.id
          LEFT JOIN catalog.products cp ON pp.catalog_product_id = cp.id
          WHERE ${whereClause}
        `;
        const countResult = await pool.query(countQuery, params);
        const total = countResult.rows[0].total;

        const dataQuery = `
          SELECT 
            pp.id,
            pp.group_id AS "groupId",
            pp.product_name AS name,
            pp.product_sku AS "internalCode",
            pp.product_sku AS "supplierCode",
            pp.notes AS description,
            pp.notes,
            jsonb_build_object(
              'product_type', pp.product_type,
              'catalog_product_id', pp.catalog_product_id,
              'catalog_set_id', pp.catalog_set_id
            ) AS specifications,
            'szt' AS "unitOfMeasure",
            COALESCE((SELECT COUNT(*) FROM warehouse.packed_product_items ppi WHERE ppi.packed_product_id = pp.id AND ppi.status IN ('available', 'reserved')), pp.quantity) AS quantity,
            COALESCE((SELECT COUNT(*) FROM warehouse.packed_product_items ppi WHERE ppi.packed_product_id = pp.id AND ppi.status = 'reserved'), pp.reserved_quantity) AS "quantityReserved",
            COALESCE((SELECT COUNT(*) FROM warehouse.packed_product_items ppi WHERE ppi.packed_product_id = pp.id AND ppi.status = 'available'), pp.quantity - pp.reserved_quantity) AS "availableQuantity",
            CASE WHEN pp.image_url IS NOT NULL THEN jsonb_build_array(pp.image_url) ELSE '[]'::jsonb END AS gallery,
            pp.image_url AS "primaryImageUrl",
            0 AS "displayOrder",
            pp.is_active AS "isActive",
            pp.location_id AS "locationId",
            pp.carrier_id AS "carrierId",
            pp.created_at AS "createdAt",
            pp.updated_at AS "updatedAt",
            mg.name AS "groupName",
            mg.code AS "groupCode",
            'produkty-spakowane' AS "groupCategory",
            pl.name AS "locationName",
            NULL AS "price",
            pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs"
          FROM warehouse.packed_products pp
          LEFT JOIN warehouse.material_groups mg ON pp.group_id = mg.id
          LEFT JOIN production.production_locations pl ON pp.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON pp.carrier_id = pc.id
          LEFT JOIN catalog.products cp ON pp.catalog_product_id = cp.id
          WHERE ${whereClause}
          ORDER BY ${sortColumn} ${sortDirection}
          LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
        `;

        const dataResult = await pool.query(dataQuery, [...params, limitNum, offset]);

        return res.json({
          materials: dataResult.rows,
          pagination: {
            page: pageNum,
            limit: limitNum,
            total,
            totalPages: Math.ceil(total / limitNum),
            offset: offset
          }
        });
      }

      // Special handling for opakowania category - use packaging_materials table
      if (category === 'opakowania') {
        let whereClause = 'pm.is_active = true';
        const params: any[] = [];
        let paramIndex = 1;
        
        // Filter by material group category if group exists, 
        // OR map ungrouped materials by their pm.category to high-level categories:
        // - pm.category='karton' → 'opakowania'
        // - other pm.category values → 'produkty-spakowane'
        whereClause += ` AND (
          mg.category = $${paramIndex} 
          OR (
            pm.group_id IS NULL 
            AND CASE 
              WHEN pm.category = 'karton' THEN 'opakowania' 
              ELSE 'produkty-spakowane' 
            END = $${paramIndex}
          )
        )`;
        params.push(category);
        paramIndex++;

        if (groupCode) {
          whereClause += ` AND mg.code = $${paramIndex}`;
          params.push(groupCode);
          paramIndex++;
        }

        if (searchTerms.length > 0) {
          const searchConditions = searchTerms.map(() => {
            const condition = `(
              LOWER(pm.name) LIKE $${paramIndex} OR
              LOWER(pm.supplier_code) LIKE $${paramIndex} OR
              LOWER(pm.description) LIKE $${paramIndex} OR
              LOWER(mg.name) LIKE $${paramIndex}
            )`;
            paramIndex++;
            return condition;
          });
          whereClause += ` AND ${searchConditions.join(' AND ')}`;
          searchTerms.forEach(term => {
            params.push(`%${term.toLowerCase()}%`);
          });
        }

        // Filter by groupCode if provided
        if (groupCode) {
          whereClause += ` AND mg.code = $${paramIndex}`;
          params.push(groupCode);
          paramIndex++;
        }

        const allowedSortColumns: Record<string, string> = {
          name: 'pm.name',
          internalCode: 'pm.supplier_code',
          supplierCode: 'pm.supplier_code',
          groupName: 'mg.name',
          createdAt: 'pm.created_at',
          updatedAt: 'pm.updated_at'
        };

        const sortColumn = allowedSortColumns[sortBy as string] || 'pm.name';
        const sortDirection = (sortOrder as string).toUpperCase() === 'DESC' ? 'DESC' : 'ASC';

        const countQuery = `
          SELECT COUNT(*)::int as total
          FROM warehouse.packaging_materials pm
          LEFT JOIN warehouse.material_groups mg ON pm.group_id = mg.id
          WHERE ${whereClause}
        `;
        const countResult = await pool.query(countQuery, params);
        const total = countResult.rows[0].total;

        const dataQuery = `
          SELECT 
            pm.id,
            pm.group_id AS "groupId",
            pm.name,
            pm.supplier_code AS "internalCode",
            pNULL AS "supplierCode",
            pm.description,
            pm.notes,
            jsonb_build_object(
              'category', pm.category,
              'length', pm.length,
              'width', pm.width,
              'height', pm.height,
              'supplierCode', pm.supplier_code,
              'notes', pm.notes
            ) AS specifications,
            pm.unit AS "unitOfMeasure",
            pm.quantity,
            CASE WHEN pm.image_url IS NOT NULL THEN jsonb_build_array(pm.image_url) ELSE '[]'::jsonb END AS gallery,
            pm.image_url AS "primaryImageUrl",
            0 AS "displayOrder",
            pm.is_active AS "isActive",
            pm.location_id AS "locationId",
            pm.carrier_id AS "carrierId",
            pm.created_at AS "createdAt",
            pm.updated_at AS "updatedAt",
            mg.name AS "groupName",
            mg.code AS "groupCode",
            mg.category AS "groupCategory",
            pl.name AS "locationName",
            NULL AS "price",
            pc.name AS "carrierName"
          FROM warehouse.packaging_materials pm
          LEFT JOIN warehouse.material_groups mg ON pm.group_id = mg.id
          LEFT JOIN production.production_locations pl ON pm.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON pm.carrier_id = pc.id
          WHERE ${whereClause}
          ORDER BY ${sortColumn} ${sortDirection}
          LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
        `;

        const dataResult = await pool.query(dataQuery, [...params, limitNum, offset]);

        return res.json({
          materials: dataResult.rows,
          pagination: {
            page: pageNum,
            limit: limitNum,
            total,
            totalPages: Math.ceil(total / limitNum),
            offset: offset
          }
        });
      }

      // Special handling for formatki category
      // Special handling for formatki category - all formatki are now in stock_panels
      // Otherwise, search in stock_panels (auto-generated formatki from panels)
      if (category === 'formatki') {
        let whereClause = 'sp.is_active = true AND (sp.is_archived = false OR sp.is_archived IS NULL)';
        const params: any[] = [];
        let paramIndex = 1;

        if (searchTerms.length > 0) {
          const searchConditions = searchTerms.map(() => {
            const condition = `(
              LOWER(sp.generated_name) LIKE $${paramIndex} OR
              LOWER(sp.cz1) LIKE $${paramIndex} OR
              LOWER(sp.cz2) LIKE $${paramIndex} OR
              LOWER(sp.board_code) LIKE $${paramIndex} OR
              LOWER(sp.color_code) LIKE $${paramIndex} OR
              LOWER(mg.name) LIKE $${paramIndex}
            )`;
            paramIndex++;
            return condition;
          });
          whereClause += ` AND ${searchConditions.join(' AND ')}`;
          searchTerms.forEach(term => {
            params.push(`%${term.toLowerCase()}%`);
          });
        }

        // Filter by groupCode if provided
        if (groupCode) {
          whereClause += ` AND mg.code = $${paramIndex}`;
          params.push(groupCode);
          paramIndex++;
        }

        // Filter by colors if provided (include)
        if (colors) {
          const colorList = (colors as string).split(",").map(c => c.trim()).filter(Boolean);
          if (colorList.length > 0) {
            whereClause += ` AND sp.color_code IN (${colorList.map(() => `$${paramIndex++}`).join(", ")})`;
            colorList.forEach(c => params.push(c));
          }
        }

        // Filter by excludeColors if provided (exclude)
        if (excludeColors) {
          const excludeColorList = (excludeColors as string).split(",").map(c => c.trim()).filter(Boolean);
          if (excludeColorList.length > 0) {
            whereClause += ` AND (sp.color_code IS NULL OR sp.color_code NOT IN (${excludeColorList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeColorList.forEach(c => params.push(c));
          }
        }

        // Filter by lengths if provided
        if (lengths) {
          const lengthList = (lengths as string).split(",").map(l => parseInt(l.trim())).filter(l => !isNaN(l));
          if (lengthList.length > 0) {
            whereClause += ` AND sp.length IN (${lengthList.map(() => `$${paramIndex++}`).join(", ")})`;
            lengthList.forEach(l => params.push(l));
          }
        }

        // Filter by excludeLengths if provided (exclude)
        if (excludeLengths) {
          const excludeLengthList = (excludeLengths as string).split(",").map(l => parseInt(l.trim())).filter(l => !isNaN(l));
          if (excludeLengthList.length > 0) {
            whereClause += ` AND (sp.length IS NULL OR sp.length NOT IN (${excludeLengthList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeLengthList.forEach(l => params.push(l));
          }
        }

        // Filter by widths if provided
        if (widths) {
          const widthList = (widths as string).split(",").map(w => parseInt(w.trim())).filter(w => !isNaN(w));
          if (widthList.length > 0) {
            whereClause += ` AND sp.width IN (${widthList.map(() => `$${paramIndex++}`).join(", ")})`;
            widthList.forEach(w => params.push(w));
          }
        }


        // Filter by excludeWidths if provided (exclude)
        if (excludeWidths) {
          const excludeWidthList = (excludeWidths as string).split(",").map(w => parseInt(w.trim())).filter(w => !isNaN(w));
          if (excludeWidthList.length > 0) {
            whereClause += ` AND (sp.width IS NULL OR sp.width NOT IN (${excludeWidthList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeWidthList.forEach(w => params.push(w));
          }
        }

        // Filter by CZ1 codes if provided
        if (cz1Codes) {
          const cz1List = (cz1Codes as string).split(",").map(c => c.trim()).filter(Boolean);
          if (cz1List.length > 0) {
            whereClause += ` AND sp.cz1 IN (${cz1List.map(() => `$${paramIndex++}`).join(", ")})`;
            cz1List.forEach(c => params.push(c));
          }
        }

        // Filter by excludeCz1Codes if provided (exclude)
        if (excludeCz1Codes) {
          const excludeCz1List = (excludeCz1Codes as string).split(",").map(c => c.trim()).filter(Boolean);
          if (excludeCz1List.length > 0) {
            whereClause += ` AND (sp.cz1 IS NULL OR sp.cz1 NOT IN (${excludeCz1List.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeCz1List.forEach(c => params.push(c));
          }
        }

        // Filter by CZ2 codes if provided
        if (cz2Codes) {
          const cz2List = (cz2Codes as string).split(",").map(c => c.trim()).filter(Boolean);
          if (cz2List.length > 0) {
            whereClause += ` AND sp.cz2 IN (${cz2List.map(() => `$${paramIndex++}`).join(", ")})`;
            cz2List.forEach(c => params.push(c));
          }
        }

        // Filter by excludeCz2Codes if provided (exclude)
        if (excludeCz2Codes) {
          const excludeCz2List = (excludeCz2Codes as string).split(",").map(c => c.trim()).filter(Boolean);
          if (excludeCz2List.length > 0) {
            whereClause += ` AND (sp.cz2 IS NULL OR sp.cz2 NOT IN (${excludeCz2List.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeCz2List.forEach(c => params.push(c));
          }
        }

        // Filter by furniture types if provided
        if (furnitureTypes) {
          const typeList = (furnitureTypes as string).split(",").map(t => t.trim()).filter(Boolean);
          if (typeList.length > 0) {
            whereClause += ` AND sp.furniture_type IN (${typeList.map(() => `$${paramIndex++}`).join(", ")})`;
            typeList.forEach(t => params.push(t));
          }
        }

        // Filter by sources if provided
        if (sources) {
          const sourceList = (sources as string).split(",").map(s => s.trim()).filter(Boolean);
          if (sourceList.length > 0) {
            whereClause += ` AND sp.source IN (${sourceList.map(() => `$${paramIndex++}`).join(", ")})`;
            sourceList.forEach(s => params.push(s));
          }
        }

        // Filter by location IDs if provided
        if (locationIds) {
          const locList = (locationIds as string).split(",").map(l => parseInt(l.trim())).filter(l => !isNaN(l));
          if (locList.length > 0) {
            whereClause += ` AND sp.location_id IN (${locList.map(() => `$${paramIndex++}`).join(", ")})`;
            locList.forEach(l => params.push(l));
          }
        }

        // Filter by excludeLocationIds if provided (exclude)
        if (excludeLocationIds) {
          const excludeLocList = (excludeLocationIds as string).split(",").map(l => parseInt(l.trim())).filter(l => !isNaN(l));
          if (excludeLocList.length > 0) {
            whereClause += ` AND (sp.location_id IS NULL OR sp.location_id NOT IN (${excludeLocList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeLocList.forEach(l => params.push(l));
          }
        }

        // Filter by carrier IDs if provided (include)
        if (carrierIds) {
          const carList = (carrierIds as string).split(",").map(c => parseInt(c.trim())).filter(c => !isNaN(c));
          if (carList.length > 0) {
            whereClause += ` AND sp.carrier_id IN (${carList.map(() => `$${paramIndex++}`).join(", ")})`;
            carList.forEach(c => params.push(c));
          }
        }

        // Filter by excludeCarrierIds if provided (exclude)
        if (excludeCarrierIds) {
          const excludeCarList = (excludeCarrierIds as string).split(",").map(c => parseInt(c.trim())).filter(c => !isNaN(c));
          if (excludeCarList.length > 0) {
            whereClause += ` AND (sp.carrier_id IS NULL OR sp.carrier_id NOT IN (${excludeCarList.map(() => `$${paramIndex++}`).join(", ")}))`;
            excludeCarList.forEach(c => params.push(c));
          }
        }

        // Filter by stock quantity
        if (stockFilter === "inStock") {
          whereClause += ` AND sp.quantity > 0`;
        } else if (stockFilter === "outOfStock") {
          whereClause += ` AND sp.quantity = 0`;
        }

        const allowedSortColumns: Record<string, string> = {
          name: 'sp.generated_name',
          internalCode: 'sp.generated_name',
          supplierCode: 'sp.generated_name',
          groupName: 'mg.name',
          createdAt: 'sp.created_at',
          updatedAt: 'sp.updated_at',
          colorCode: 'sp.color_code',
          length: 'sp.length',
          width: 'sp.width',
          thickness: 'sp.thickness',
          quantity: 'sp.quantity',
          reservedQuantity: 'sp.reserved_quantity',
          availableQuantity: '(sp.quantity - COALESCE(sp.reserved_quantity, 0))',
          boardCode: 'sp.board_code',
          edgingCode: 'sp.edging_code',
          originalSymbol: 'sp.original_symbol',
          price: 'sp.price'
        };

        const sortColumn = allowedSortColumns[sortBy as string] || 'sp.generated_name';
        const sortDirection = (sortOrder as string).toUpperCase() === 'DESC' ? 'DESC' : 'ASC';

        const countQuery = `
          SELECT COUNT(*)::int as total
          FROM warehouse.stock_panels sp
          LEFT JOIN warehouse.material_groups mg ON sp.group_id = mg.id
          WHERE ${whereClause}
        `;
        const countResult = await pool.query(countQuery, params);
        const total = countResult.rows[0].total;

        const dataQuery = `
          WITH duplicates_cte AS (
            SELECT 
              generated_name,
              COUNT(*) as dup_count,
              ARRAY_AGG(id ORDER BY id) as dup_ids
            FROM warehouse.stock_panels
            WHERE is_active = true AND (is_archived = false OR is_archived IS NULL)
            GROUP BY generated_name
            HAVING COUNT(*) > 1
          )
          SELECT 
            sp.id,
            sp.group_id AS "groupId",
            sp.generated_name AS name,
            sp.generated_name AS "internalCode",
            NULL AS "supplierCode",
            sp.original_symbol AS "originalSymbol",
            CONCAT(
              sp.length, '×', sp.width, '×', sp.thickness, 'mm',
              CASE WHEN sp.color_code IS NOT NULL THEN ' ' || sp.color_code ELSE '' END,
              CASE WHEN sp.source != 'stock' THEN ' (' || sp.source || ')' END
            ) AS description,
            jsonb_build_object(
              'category', 'formatki',
              'length', sp.length,
              'width', sp.width,
              'thickness', sp.thickness,
              'boardCode', sp.board_code,
              'edgingCode', sp.edging_code,
              'colorCode', sp.color_code,
              'color', sp.color_code,
              'edge1', sp.edge1,
              'edge2', sp.edge2,
              'edge3', sp.edge3,
              'edge4', sp.edge4,
              'isDrilled', sp.is_drilled,
              'isEdged', sp.is_edged,
              'source', sp.source,
              'sourceReference', sp.source_reference,
              'cz1', sp.cz1,
              'cz2', sp.cz2,
              'furnitureType', sp.furniture_type,
              'materialCost', sp.material_cost,
              'calculatedBoardCost', CASE WHEN board.price IS NOT NULL AND sp.length IS NOT NULL AND sp.width IS NOT NULL THEN ROUND((board.price::numeric * (sp.length::numeric * sp.width::numeric / 1000000)), 4) ELSE NULL END,
              'calculatedEdgingCost', CASE WHEN edging.price IS NOT NULL AND sp.length IS NOT NULL AND sp.width IS NOT NULL AND sp.edge1 = 'T' AND sp.edge2 = 'T' AND sp.edge3 = 'T' AND sp.edge4 = 'T' THEN ROUND((edging.price::numeric * 2 * (sp.length::numeric + sp.width::numeric) / 1000), 4) ELSE NULL END,
              'calculatedMaterialCost', CASE WHEN board.price IS NOT NULL AND sp.length IS NOT NULL AND sp.width IS NOT NULL THEN ROUND((board.price::numeric * (sp.length::numeric * sp.width::numeric / 1000000)), 4) + COALESCE(CASE WHEN edging.price IS NOT NULL AND sp.edge1 = 'T' AND sp.edge2 = 'T' AND sp.edge3 = 'T' AND sp.edge4 = 'T' THEN ROUND((edging.price::numeric * 2 * (sp.length::numeric + sp.width::numeric) / 1000), 4) ELSE 0 END, 0) ELSE NULL END
            ) AS specifications,
            sp.unit AS "unitOfMeasure",
            sp.quantity,
            COALESCE((
              SELECT SUM(CAST(pbr.quantity_reserved AS decimal) - COALESCE(CAST(pbr.quantity_consumed AS decimal), 0))
              FROM production.production_buffer_reservations pbr
              WHERE pbr.product_sku = sp.generated_name AND pbr.status = 'ACTIVE'
            ), 0)::int AS "quantityReserved",
            CASE WHEN sp.image_url IS NOT NULL THEN jsonb_build_array(sp.image_url) ELSE '[]'::jsonb END AS gallery,
            sp.image_url AS "primaryImageUrl",
            0 AS "displayOrder",
            sp.is_active AS "isActive",
            sp.location_id AS "locationId",
            sp.carrier_id AS "carrierId",
          sp.group_id AS "groupId",
            sp.created_at AS "createdAt",
            sp.updated_at AS "updatedAt",
            mg.name AS "groupName",
            mg.code AS "groupCode",
            'formatki' AS "groupCategory",
            pl.name AS "locationName",
            NULL AS "price",
            pc.name AS "carrierName",
            COALESCE(dup.dup_count, 0)::int AS "duplicateCount",
            COALESCE(dup.dup_ids, ARRAY[]::int[]) AS "duplicateIds"
          FROM warehouse.stock_panels sp
          LEFT JOIN warehouse.material_groups mg ON sp.group_id = mg.id
          LEFT JOIN production.production_locations pl ON sp.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON sp.carrier_id = pc.id
          LEFT JOIN warehouse.materials board ON board.internal_code = sp.board_code
          LEFT JOIN warehouse.materials edging ON edging.internal_code = sp.edging_code
          LEFT JOIN duplicates_cte dup ON dup.generated_name = sp.generated_name
          WHERE ${whereClause}
          ORDER BY ${sortColumn} ${sortDirection}
          LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
        `;

        const dataResult = await pool.query(dataQuery, [...params, limitNum, offset]);


        // Calculate filter counts (excluding current filter from its own count)
        const buildBaseWhere = (excludeGroup: string | null = null) => {
          let baseWhere = "sp.is_active = true AND (sp.is_archived = false OR sp.is_archived IS NULL)";
          const baseParams: any[] = [];
          let baseParamIndex = 1;

          // Search terms
          if (searchTerms.length > 0) {
            const searchConditions = searchTerms.map(() => {
              const condition = `(
                LOWER(sp.generated_name) LIKE $${baseParamIndex} OR
                LOWER(sp.cz1) LIKE $${baseParamIndex} OR
                LOWER(sp.cz2) LIKE $${baseParamIndex} OR
                LOWER(sp.board_code) LIKE $${baseParamIndex} OR
                LOWER(sp.color_code) LIKE $${baseParamIndex} OR
                LOWER(mg.name) LIKE $${baseParamIndex}
              )`;
              baseParamIndex++;
              return condition;
            });
            baseWhere += ` AND ${searchConditions.join(" AND ")}`;
            searchTerms.forEach(term => {
              baseParams.push(`%${term.toLowerCase()}%`);
            });
          }

          // Group filter
          if (groupCode) {
            baseWhere += ` AND mg.code = $${baseParamIndex}`;
            baseParams.push(groupCode);
            baseParamIndex++;
          }

          // Colors (exclude if this is the group being counted)
          if (colors && excludeGroup !== "colors") {
            const colorList = (colors as string).split(",").map(c => c.trim()).filter(Boolean);
            if (colorList.length > 0) {
              baseWhere += ` AND sp.color_code IN (${colorList.map(() => `$${baseParamIndex++}`).join(", ")})`;
              colorList.forEach(c => baseParams.push(c));
            }
          }

          // Lengths (exclude if this is the group being counted)
          if (lengths && excludeGroup !== "lengths") {
            const lengthList = (lengths as string).split(",").map(l => parseInt(l.trim())).filter(l => !isNaN(l));
            if (lengthList.length > 0) {
              baseWhere += ` AND sp.length IN (${lengthList.map(() => `$${baseParamIndex++}`).join(", ")})`;
              lengthList.forEach(l => baseParams.push(l));
            }
          }

          // Widths (exclude if this is the group being counted)
          if (widths && excludeGroup !== "widths") {
            const widthList = (widths as string).split(",").map(w => parseInt(w.trim())).filter(w => !isNaN(w));
            if (widthList.length > 0) {
              baseWhere += ` AND sp.width IN (${widthList.map(() => `$${baseParamIndex++}`).join(", ")})`;
              widthList.forEach(w => baseParams.push(w));
            }
          }

          // Stock filter (exclude if this is the group being counted)
          if (stockFilter && excludeGroup !== "stock") {
            if (stockFilter === "inStock") {
              baseWhere += ` AND sp.quantity > 0`;
            } else if (stockFilter === "outOfStock") {
              baseWhere += ` AND sp.quantity = 0`;
            }
          }

          return { baseWhere, baseParams };
        };

        // Get color counts
        const { baseWhere: colorBaseWhere, baseParams: colorBaseParams } = buildBaseWhere("colors");
        const colorCountsQuery = `
          SELECT sp.color_code as code, COUNT(*)::int as count
          FROM warehouse.stock_panels sp
          LEFT JOIN warehouse.material_groups mg ON sp.group_id = mg.id
          WHERE ${colorBaseWhere} AND sp.color_code IS NOT NULL
          GROUP BY sp.color_code
        `;
        const colorCountsResult = await pool.query(colorCountsQuery, colorBaseParams);
        const colorCounts: Record<string, number> = {};
        colorCountsResult.rows.forEach((row: any) => {
          colorCounts[row.code] = row.count;
        });

        // Get length counts
        const { baseWhere: lengthBaseWhere, baseParams: lengthBaseParams } = buildBaseWhere("lengths");
        const lengthCountsQuery = `
          SELECT sp.length::int::text as code, COUNT(*)::int as count
          FROM warehouse.stock_panels sp
          LEFT JOIN warehouse.material_groups mg ON sp.group_id = mg.id
          WHERE ${lengthBaseWhere} AND sp.length IS NOT NULL
          GROUP BY sp.length::int
        `;
        const lengthCountsResult = await pool.query(lengthCountsQuery, lengthBaseParams);
        const lengthCounts: Record<string, number> = {};
        lengthCountsResult.rows.forEach((row: any) => {
          lengthCounts[row.code] = row.count;
        });

        // Get width counts
        const { baseWhere: widthBaseWhere, baseParams: widthBaseParams } = buildBaseWhere("widths");
        const widthCountsQuery = `
          SELECT sp.width::int::text as code, COUNT(*)::int as count
          FROM warehouse.stock_panels sp
          LEFT JOIN warehouse.material_groups mg ON sp.group_id = mg.id
          WHERE ${widthBaseWhere} AND sp.width IS NOT NULL
          GROUP BY sp.width::int
        `;
        const widthCountsResult = await pool.query(widthCountsQuery, widthBaseParams);
        const widthCounts: Record<string, number> = {};
        widthCountsResult.rows.forEach((row: any) => {
          widthCounts[row.code] = row.count;
        });

        // Get stock counts
        const { baseWhere: stockBaseWhere, baseParams: stockBaseParams } = buildBaseWhere("stock");
        const stockCountsQuery = `
          SELECT 
            SUM(CASE WHEN sp.quantity > 0 THEN 1 ELSE 0 END)::int as "inStock",
            SUM(CASE WHEN sp.quantity = 0 THEN 1 ELSE 0 END)::int as "outOfStock"
          FROM warehouse.stock_panels sp
          LEFT JOIN warehouse.material_groups mg ON sp.group_id = mg.id
          WHERE ${stockBaseWhere}
        `;
        const stockCountsResult = await pool.query(stockCountsQuery, stockBaseParams);
        const stockCounts = stockCountsResult.rows[0] || { inStock: 0, outOfStock: 0 };

        return res.json({
          filterCounts: {
            colors: colorCounts,
            lengths: lengthCounts,
            widths: widthCounts,
            stock: stockCounts
          },
          materials: dataResult.rows,
          pagination: {
            page: pageNum,
            limit: limitNum,
            total,
            totalPages: Math.ceil(total / limitNum),
            offset: offset
          }
        });
      }

      // Default handling for warehouse.materials table
      let whereClause = 'm.is_active = true AND (m.is_archived = false OR m.is_archived IS NULL)';
      const params: any[] = [];
      let paramIndex = 1;

      if (category) {
        whereClause += ` AND (mg.category = $${paramIndex} OR m.specifications->>'category' = $${paramIndex} OR m.group_id IS NULL)`;
        params.push(category);
        paramIndex++;
      }

      if (groupCode) {
        whereClause += ` AND mg.code = $${paramIndex}`;
        params.push(groupCode);
        paramIndex++;
      }

      if (searchTerms.length > 0) {
        const searchConditions = searchTerms.map(() => {
          const condition = `(
            LOWER(m.name) LIKE $${paramIndex} OR
            LOWER(m.internal_code) LIKE $${paramIndex} OR
            LOWER(m.supplier_code) LIKE $${paramIndex} OR
            LOWER(m.description) LIKE $${paramIndex} OR
            LOWER(mg.name) LIKE $${paramIndex}
          )`;
          paramIndex++;
          return condition;
        });
        whereClause += ` AND ${searchConditions.join(' AND ')}`;
        searchTerms.forEach(term => {
          params.push(`%${term.toLowerCase()}%`);
        });
      }

        // Filter by groupCode if provided
        if (groupCode) {
          whereClause += ` AND mg.code = $${paramIndex}`;
          params.push(groupCode);
          paramIndex++;
        }

      const allowedSortColumns: Record<string, string> = {
        name: 'm.name',
        internalCode: 'm.internal_code',
        supplierCode: 'm.supplier_code',
        groupName: 'mg.name',
        createdAt: 'm.created_at',
        updatedAt: 'm.updated_at'
      };

      const sortColumn = allowedSortColumns[sortBy as string] || 'm.name';
      const sortDirection = (sortOrder as string).toUpperCase() === 'DESC' ? 'DESC' : 'ASC';

      const countQuery = `
        SELECT COUNT(*)::int as total
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        WHERE ${whereClause}
      `;
      const countResult = await pool.query(countQuery, params);
      const total = countResult.rows[0].total;

      const dataQuery = `
        SELECT 
          m.id,
          m.group_id AS "groupId",
          m.name,
          m.internal_code AS "internalCode",
          NULL AS "supplierCode",
          m.description,
          m.specifications,
          m.unit_of_measure AS "unitOfMeasure",
          m.quantity,
          COALESCE(SUM(CASE 
            WHEN r.status = 'ACTIVE' 
            THEN r.quantity_reserved - r.quantity_consumed 
            ELSE 0 
          END), 0) AS "quantityReserved",
          m.gallery,
          m.primary_image_url AS "primaryImageUrl",
          m.display_order AS "displayOrder",
          m.is_active AS "isActive",
          m.location_id AS "locationId",
          m.carrier_id AS "carrierId",
          m.created_at AS "createdAt",
          m.updated_at AS "updatedAt",
          mg.name AS "groupName",
          mg.code AS "groupCode",
          mg.category AS "groupCategory",
          pl.name AS "locationName",
          pc.name AS "carrierName"
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        LEFT JOIN production.production_locations pl ON m.location_id = pl.id
        LEFT JOIN production.production_carriers pc ON m.carrier_id = pc.id
        LEFT JOIN production.production_buffer_reservations r 
          ON r.product_sku = m.internal_code AND r.status = 'ACTIVE'
        WHERE ${whereClause}
        GROUP BY m.id, m.name, m.internal_code, m.supplier_code, m.description, 
                 m.specifications, m.unit_of_measure, m.quantity, m.price, m.gallery,
                 m.primary_image_url, m.display_order, m.is_active, m.location_id, 
                 m.carrier_id, m.created_at, m.updated_at, mg.name, mg.code, 
                 mg.category, pl.name, pc.name, m.group_id
        ORDER BY ${sortColumn} ${sortDirection}
        LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
      `;
      
      const dataResult = await pool.query(dataQuery, [...params, limitNum, offset]);

      res.json({
        materials: dataResult.rows,
        pagination: {
          page: pageNum,
          limit: limitNum,
          total,
          totalPages: Math.ceil(total / limitNum),
          offset: offset
        }
      });
    } catch (error) {
      console.error("❌ Error searching materials:", error);
      res.status(500).json({ error: "Failed to search materials" });
    }
  });

  // GET /api/warehouse/materials/:id - Get single material by ID
  app.get("/api/warehouse/materials/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { category } = req.query;
      
      // Special handling for produkty-spakowane category - use packed_products table
      if (category === 'produkty-spakowane') {
        const query = `
          SELECT 
            pp.id,
            pp.product_name AS name,
            pp.product_sku AS "internalCode",
            pp.product_sku AS "supplierCode",
            pp.quantity,
            pp.reserved_quantity AS "reservedQuantity",
            (pp.quantity - pp.reserved_quantity) AS "availableQuantity",
            pp.notes AS description,
            pp.notes,
            jsonb_build_object(
              'category', 'produkty-spakowane',
              'productType', pp.product_type,
              'catalogProductId', pp.catalog_product_id,
              'notes', pp.notes
            ) AS specifications,
            'szt' AS "unitOfMeasure",
            CASE WHEN pp.image_url IS NOT NULL THEN jsonb_build_array(pp.image_url) ELSE '[]'::jsonb END AS gallery,
            pp.image_url AS "primaryImageUrl",
            pp.is_active AS "isActive",
            pp.location_id AS "locationId",
            pp.carrier_id AS "carrierId",
            'produkty-spakowane' AS "categoryName",
            mg.name AS "groupName",
            pl.code AS "locationCode",
            pl.name AS "locationName",
            NULL AS "price",
            pl.path AS "locationPath",
            pc.code AS "carrierCode",
            pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
            pp.catalog_product_id AS "catalogProductId"
          FROM warehouse.packed_products pp
          LEFT JOIN warehouse.material_groups mg ON pp.group_id = mg.id
          LEFT JOIN production.production_locations pl ON pp.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON pp.carrier_id = pc.id
          LEFT JOIN catalog.products cp ON pp.catalog_product_id = cp.id
          WHERE pp.id = $1
        `;
        
        const result = await pool.query(query, [id]);
        
        if (result.rows.length === 0) {
          return res.status(404).json({ error: "Packed product not found" });
        }
        
        return res.json(result.rows[0]);
      }
      
      // Special handling for opakowania category - use packaging_materials table
      if (category === 'opakowania') {
        const query = `
          SELECT 
            pm.id,
            pm.name,
            pm.supplier_code AS "internalCode",
            pNULL AS "supplierCode",
            pm.quantity,
            pm.description,
            pm.notes,
            jsonb_build_object(
              'category', pm.category,
              'length', pm.length,
              'width', pm.width,
              'height', pm.height,
              'supplierCode', pm.supplier_code,
              'notes', pm.notes
            ) AS specifications,
            pm.unit AS "unitOfMeasure",
            CASE WHEN pm.image_url IS NOT NULL THEN jsonb_build_array(pm.image_url) ELSE '[]'::jsonb END AS gallery,
            pm.image_url AS "primaryImageUrl",
            pm.is_active AS "isActive",
            pm.location_id AS "locationId",
            pm.carrier_id AS "carrierId",
            pm.category AS "categoryName",
            mg.name AS "groupName",
            pl.code AS "locationCode",
            pl.name AS "locationName",
            NULL AS "price",
            pl.path AS "locationPath",
            pc.code AS "carrierCode",
            pc.name AS "carrierName"
          FROM warehouse.packaging_materials pm
          LEFT JOIN warehouse.material_groups mg ON pm.group_id = mg.id
          LEFT JOIN production.production_locations pl ON pm.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON pm.carrier_id = pc.id
          WHERE pm.id = $1
        `;
        
        const result = await pool.query(query, [id]);
        
        if (result.rows.length === 0) {
          return res.status(404).json({ error: "Packaging material not found" });
        }
        
        return res.json(result.rows[0]);
      }
      
      // Special handling for formatki category - use stock_panels table
      if (category === 'formatki') {
        const query = `
          SELECT 
            sp.id,
            sp.generated_name AS name,
            sp.generated_name AS "internalCode",
            NULL AS "supplierCode",
            sp.original_symbol AS "originalSymbol",
            sp.quantity,
            sp.unit AS "unitOfMeasure",
            CONCAT(
              sp.length, '×', sp.width, '×', sp.thickness, 'mm',
              CASE WHEN sp.color_code IS NOT NULL THEN ' ' || sp.color_code ELSE '' END,
              CASE WHEN sp.source != 'stock' THEN ' (' || sp.source || ')' END
            ) AS description,
            jsonb_build_object(
              'category', 'formatki',
              'length', sp.length,
              'width', sp.width,
              'thickness', sp.thickness,
              'boardCode', sp.board_code,
              'edgingCode', sp.edging_code,
              'colorCode', sp.color_code,
              'edge1', sp.edge1,
              'edge2', sp.edge2,
              'edge3', sp.edge3,
              'edge4', sp.edge4,
              'isDrilled', sp.is_drilled,
              'isEdged', sp.is_edged,
              'source', sp.source,
              'sourceReference', sp.source_reference,
              'cz1', sp.cz1,
              'cz2', sp.cz2,
              'furnitureType', sp.furniture_type,
              'materialCost', sp.material_cost,
              'boardId', board.id,
              'boardName', board.name,
              'edgingId', edging.id,
              'edgingName', edging.name,
              'calculatedBoardCost', CASE WHEN board.price IS NOT NULL AND sp.length IS NOT NULL AND sp.width IS NOT NULL THEN ROUND((board.price::numeric * (sp.length::numeric * sp.width::numeric / 1000000)), 4) ELSE NULL END,
              'calculatedEdgingCost', CASE 
                WHEN edging.price IS NOT NULL 
                  AND sp.length IS NOT NULL 
                  AND sp.width IS NOT NULL 
                  AND sp.edge1 = 'T' AND sp.edge2 = 'T' AND sp.edge3 = 'T' AND sp.edge4 = 'T'
                THEN ROUND((edging.price::numeric * 2 * (sp.length::numeric + sp.width::numeric) / 1000), 4)
                ELSE NULL
              END,
              'calculatedMaterialCost', CASE 
                WHEN board.price IS NOT NULL AND sp.length IS NOT NULL AND sp.width IS NOT NULL 
                THEN ROUND((board.price::numeric * (sp.length::numeric * sp.width::numeric / 1000000)), 4)
                  + COALESCE(
                      CASE 
                        WHEN edging.price IS NOT NULL 
                          AND sp.edge1 = 'T' AND sp.edge2 = 'T' AND sp.edge3 = 'T' AND sp.edge4 = 'T'
                        THEN ROUND((edging.price::numeric * 2 * (sp.length::numeric + sp.width::numeric) / 1000), 4)
                        ELSE 0
                      END, 0)
                ELSE NULL
              END
            ) AS specifications,
            CASE WHEN sp.image_url IS NOT NULL THEN jsonb_build_array(sp.image_url) ELSE '[]'::jsonb END AS gallery,
            sp.image_url AS "primaryImageUrl",
            sp.is_active AS "isActive",
            sp.location_id AS "locationId",
            sp.carrier_id AS "carrierId",
            sp.group_id AS "groupId",
            sp.notes,
            'formatki' AS "categoryName",
            mg.name AS "groupName",
            pl.code AS "locationCode",
            pl.name AS "locationName",
            NULL AS "price",
            pl.path AS "locationPath",
            pc.code AS "carrierCode",
            pc.name AS "carrierName"
          FROM warehouse.stock_panels sp
          LEFT JOIN warehouse.material_groups mg ON sp.group_id = mg.id
          LEFT JOIN production.production_locations pl ON sp.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON sp.carrier_id = pc.id
          LEFT JOIN warehouse.materials board ON board.internal_code = sp.board_code
          LEFT JOIN warehouse.materials edging ON edging.internal_code = sp.edging_code
          WHERE sp.id = \$1
        `;
        
        const result = await pool.query(query, [id]);
        
        if (result.rows.length === 0) {
          return res.status(404).json({ error: "Material not found" });
        }
        
        return res.json(result.rows[0]);
      }
      
      // Default handling for warehouse.materials table
      const query = `
        SELECT 
          m.id,
          m.name,
          m.internal_code AS "internalCode",
          NULL AS "supplierCode",
          m.quantity,
          m.description,
          m.specifications,
          m.unit_of_measure AS "unitOfMeasure",
          m.primary_image_url AS "primaryImageUrl",
          m.gallery,
          m.is_active AS "isActive",
          m.location_id AS "locationId",
          m.carrier_id AS "carrierId",
          mg.category AS "categoryName",
          mg.name AS "groupName",
          pl.code AS "locationCode",
          pl.name AS "locationName",
          pl.path AS "locationPath",
          pc.code AS "carrierCode",
          pc.name AS "carrierName"
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        LEFT JOIN production.production_locations pl ON m.location_id = pl.id
        LEFT JOIN production.production_carriers pc ON m.carrier_id = pc.id
        WHERE m.id = $1
      `;
      
      const result = await pool.query(query, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error fetching material:", error);
      res.status(500).json({ error: "Failed to fetch material" });
    }
  });

  // GET /api/warehouse/materials/:id/usage - Get material usage history
  app.get("/api/warehouse/materials/:id/usage", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { category } = req.query;
      
      // Special handling for produkty-spakowane - return plan line reservations
      if (category === 'produkty-spakowane') {
        // Get catalog_product_id from packed_products
        const packedProductCheck = await pool.query(
          'SELECT catalog_product_id, product_sku FROM warehouse.packed_products WHERE id = $1',
          [id]
        );
        
        if (packedProductCheck.rows.length === 0) {
          return res.status(404).json({ error: "Packed product not found" });
        }
        
        const catalogProductId = packedProductCheck.rows[0].catalog_product_id;
        const productSku = packedProductCheck.rows[0].product_sku;
        
        if (!catalogProductId) {
          // No catalog link - return empty usage
          return res.json([]);
        }
        
        // Get plan line reservations
        const reservationsQuery = `
          SELECT 
            ppl.id,
            ppl.plan_id AS "planId",
            ppl.quantity AS "quantityReserved",
            ppl.status AS "lineStatus",
            ppl.created_at AS "reservedAt",
            ppl.updated_at AS "updatedAt",
            ppl.notes,
            pp.plan_number AS "planNumber",
            pp.name AS "planName",
            pp.status AS "planStatus",
            pp.priority AS "planPriority"
          FROM production.production_plan_lines ppl
          INNER JOIN production.production_plans pp ON ppl.plan_id = pp.id
          WHERE ppl.product_id = $1
          ORDER BY ppl.created_at DESC
          LIMIT 100
        `;
        
        const reservationsResult = await pool.query(reservationsQuery, [catalogProductId]);
        
        // Transform to usage format with reservation info
        const usageData = reservationsResult.rows.map(row => ({
          id: row.id,
          type: 'plan_reservation',
          usedAt: row.reservedAt,
          quantityUsed: row.quantityReserved,
          planId: row.planId,
          planNumber: row.planNumber,
          planName: row.planName,
          planStatus: row.planStatus,
          planPriority: row.planPriority,
          lineStatus: row.lineStatus,
          notes: row.notes,
          updatedAt: row.updatedAt
        }));
        
        return res.json(usageData);
      }
      
      // Special handling for formatki - return empty usage for now
      if (category === 'formatki') {
        // Verify formatka exists in stock_panels
        const formatkiCheck = await pool.query(
          'SELECT id FROM warehouse.stock_panels WHERE id = $1',
          [id]
        );
        
        if (formatkiCheck.rows.length === 0) {
          return res.status(404).json({ error: "Formatka not found" });
        }
        
        // TODO: Implement formatki usage tracking if needed
        return res.json([]);
      }
      
      // First get the material's internal code
      const materialResult = await pool.query(
        'SELECT internal_code FROM warehouse.materials WHERE id = $1',
        [id]
      );
      
      if (materialResult.rows.length === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      const internalCode = materialResult.rows[0].internal_code;
      
      // Get usage history from production_material_movements
      const usageQuery = `
        SELECT 
          pmm.id,
          pmm.movement_date AS "usedAt",
          pmm.quantity AS "quantityUsed",
          pmm.notes,
          pmm.production_order_id AS "productionOrderId",
          po.order_number AS "productionOrderNumber"
        FROM production.production_material_movements pmm
        LEFT JOIN production.production_orders po ON po.id = pmm.production_order_id
        WHERE pmm.material_code = $1
          AND pmm.status = 'completed'
        ORDER BY pmm.movement_date DESC
        LIMIT 100
      `;
      
      const usageResult = await pool.query(usageQuery, [internalCode]);
      res.json(usageResult.rows);
    } catch (error) {
      console.error("❌ Error fetching material usage:", error);
      res.status(500).json({ error: "Failed to fetch material usage" });
    }
  });

  app.get("/api/warehouse/materials", isAuthenticated, async (req, res) => {
    try {
      const { category, groupId, groupCode, locationId, carrierId } = req.query;
      
      let query = `
        SELECT 
          m.id,
          m.group_id AS "groupId",
          m.name,
          m.internal_code AS "internalCode",
          NULL AS "supplierCode",
          m.description,
          m.specifications,
          m.unit_of_measure AS "unitOfMeasure",
          m.quantity,
          m.gallery,
          m.primary_image_url AS "primaryImageUrl",
          m.display_order AS "displayOrder",
          m.is_active AS "isActive",
          m.location_id AS "locationId",
          m.carrier_id AS "carrierId",
          m.created_at AS "createdAt",
          m.updated_at AS "updatedAt",
          mg.name AS "groupName",
          mg.code AS "groupCode",
          mg.category AS "groupCategory",
          mg.category AS "categoryName",
          pl.name AS "locationName",
          pc.name AS "carrierName"
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        LEFT JOIN production.production_locations pl ON m.location_id = pl.id
        LEFT JOIN production.production_carriers pc ON m.carrier_id = pc.id
        WHERE m.is_active = true
      `;
      
      const params: any[] = [];
      let paramIndex = 1;
      
      if (category) {
        query += ` AND mg.category = $${paramIndex}`;
        params.push(category);
        paramIndex++;
      }
      
      if (groupId) {
        query += ` AND m.group_id = $${paramIndex}`;
        params.push(parseInt(groupId as string, 10));
        paramIndex++;
      }
      
      if (groupCode) {
        query += ` AND mg.code = $${paramIndex}`;
        params.push(groupCode);
        paramIndex++;
      }
      
      if (locationId) {
        query += ` AND m.location_id = $${paramIndex}`;
        params.push(parseInt(locationId as string, 10));
        paramIndex++;
      }
      
      if (carrierId) {
        query += ` AND m.carrier_id = $${paramIndex}`;
        params.push(parseInt(carrierId as string, 10));
        paramIndex++;
      }
      
      query += ` ORDER BY m.display_order ASC, m.name ASC`;
      
      const result = await pool.query(query, params);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching materials:", error);
      res.status(500).json({ error: "Failed to fetch materials" });
    }
  });

  // GET /api/warehouse/inventory-items - Lista pozycji do spisu inwentaryzacyjnego (materiały/formatki/opakowania)
  app.get("/api/warehouse/inventory-items", isAuthenticated, async (req, res) => {
    try {
      const { category, locationId, carrierId } = req.query;
      
      console.log('🔍 [INVENTORY-ITEMS] Request params:', { category, locationId, carrierId });
      
      // Determine which table to query based on category
      let items = [];
      
      if (category === 'formatki') {
        // Fetch stock panels (formatki)
        let query = `
          SELECT 
            sp.id,
            sp.generated_name AS "name",
            sp.generated_name AS "internalCode",
            sp.quantity,
          sp.unit AS "unitOfMeasure",
            'szt' AS "unitOfMeasure",
            'Formatki' AS "categoryName",
            'Formatki' AS "groupName",
            CONCAT(sp.length, 'x', sp.width, 'x', sp.thickness) AS "dimensions",
            sp.color_code AS "colorCode",
            sp.location_id AS "locationId",
            sp.carrier_id AS "carrierId",
          sp.group_id AS "groupId",
            pl.name AS "locationName",
            NULL AS "price",
            pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
            'stock_panel' AS "itemType"
          FROM warehouse.stock_panels sp
          LEFT JOIN production.production_locations pl ON sp.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON sp.carrier_id = pc.id
          WHERE sp.is_active = true AND (sp.is_archived = false OR sp.is_archived IS NULL)
        `;
        
        const params: any[] = [];
        let paramIndex = 1;
        
        if (locationId) {
          query += ` AND sp.location_id = $${paramIndex}`;
          params.push(parseInt(locationId as string, 10));
          paramIndex++;
        }
        
        if (carrierId) {
          query += ` AND sp.carrier_id = $${paramIndex}`;
          params.push(parseInt(carrierId as string, 10));
          paramIndex++;
        }
        
        query += ` ORDER BY sp.generated_name ASC`;
        
        const result = await pool.query(query, params);
        items = result.rows;
      } else if (category === 'opakowania') {
        // Fetch packaging materials (opakowania)
        let query = `
          SELECT 
            pm.id,
            pm.name,
            pm.supplier_code AS "internalCode",
            pm.quantity,
            pm.unit AS "unitOfMeasure",
            'Opakowania' AS "categoryName",
            'Opakowania' AS "groupName",
            pm.location_id AS "locationId",
            pm.carrier_id AS "carrierId",
            pl.name AS "locationName",
            NULL AS "price",
            pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
            'packaging_material' AS "itemType"
          FROM warehouse.packaging_materials pm
          LEFT JOIN production.production_locations pl ON pm.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON pm.carrier_id = pc.id
          WHERE pm.is_active = true
        `;
        
        const params: any[] = [];
        let paramIndex = 1;
        
        if (locationId) {
          query += ` AND pm.location_id = $${paramIndex}`;
          params.push(parseInt(locationId as string, 10));
          paramIndex++;
        }
        
        if (carrierId) {
          query += ` AND pm.carrier_id = $${paramIndex}`;
          params.push(parseInt(carrierId as string, 10));
          paramIndex++;
        }
        
        query += ` ORDER BY pm.name ASC`;
        
        const result = await pool.query(query, params);
        items = result.rows;
      } else {
        // Fetch regular materials (tkaniny, pianki, plyty, obrzeza, okucia, sruby or all)
        let query = `
          SELECT 
            m.id,
            m.name,
            m.internal_code AS "internalCode",
            m.quantity,
            m.unit_of_measure AS "unitOfMeasure",
            mg.category AS "categoryName",
            mg.name AS "groupName",
            m.location_id AS "locationId",
            m.carrier_id AS "carrierId",
            pl.name AS "locationName",
            NULL AS "price",
            pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
            'material' AS "itemType"
          FROM warehouse.materials m
          LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
          LEFT JOIN production.production_locations pl ON m.location_id = pl.id
          LEFT JOIN production.production_carriers pc ON m.carrier_id = pc.id
          WHERE m.is_active = true
        `;
        
        const params: any[] = [];
        let paramIndex = 1;
        
        if (category && category !== 'all') {
          query += ` AND mg.category = $${paramIndex}`;
          params.push(category);
          paramIndex++;
        }
        
        if (locationId) {
          query += ` AND m.location_id = $${paramIndex}`;
          params.push(parseInt(locationId as string, 10));
          paramIndex++;
        }
        
        if (carrierId) {
          query += ` AND m.carrier_id = $${paramIndex}`;
          params.push(parseInt(carrierId as string, 10));
          paramIndex++;
        }
        
        query += ` ORDER BY m.display_order ASC, m.name ASC`;
        
        const result = await pool.query(query, params);
        items = result.rows;
      }
      
      console.log('🔍 [INVENTORY-ITEMS] Returning items:', { category, count: items.length });
      res.json(items);
    } catch (error) {
      console.error("❌ Error fetching inventory items:", error);
      res.status(500).json({ error: "Failed to fetch inventory items" });
    }
  });
  
  // POST /api/warehouse/materials - Utwórz nowy materiał
  app.post("/api/warehouse/materials", isAuthenticated, async (req, res) => {
    try {
      const {
        groupId,
        name,
        internalCode,
        supplierCode,
        description,
        specifications,
        unitOfMeasure,
        gallery,
        primaryImageUrl,
        displayOrder
      } = req.body;
      
      if (!name || !internalCode) {
        return res.status(400).json({ error: "Name and internal code are required" });
      }
      
      const result = await pool.query(`
        INSERT INTO warehouse.materials (
          group_id, name, internal_code, supplier_code, description,
          specifications, unit_of_measure, price, gallery, primary_image_url, display_order
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11)
        RETURNING 
          id,
          group_id AS "groupId",
          name,
          internal_code AS "internalCode",
          supplier_code AS "supplierCode",
          description,
          specifications,
          unit_of_measure AS "unitOfMeasure",
          gallery,
          primary_image_url AS "primaryImageUrl",
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        groupId || null,
        name,
        internalCode,
        supplierCode || null,
        description || null,
        specifications || null,
        unitOfMeasure || 'szt',
        price || null,
        gallery || [],
        primaryImageUrl || null,
        displayOrder || 0
      ]);
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Material with this internal code already exists" });
      }
      console.error("❌ Error creating material:", error);
      res.status(500).json({ error: "Failed to create material" });
    }
  });
  
  // PATCH /api/warehouse/materials/:id - Aktualizuj materiał
  app.patch("/api/warehouse/materials/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { category } = req.query;
      const {
        groupId,
        name,
        internalCode,
        supplierCode,
        description,
        specifications,
        unitOfMeasure,
        quantity,
        gallery,
        primaryImageUrl,
        displayOrder,
        isActive,
        locationId,
        carrierId,
        price,
      } = req.body;
      
      // Special handling for formatki category - update stock_panels table
      if (category === 'formatki') {
        const updates: string[] = [];
        const values: any[] = [];
        let paramIndex = 1;
        
        // Only allow updating specific fields for formatki
        if (quantity !== undefined) {
          updates.push(`quantity = $${paramIndex++}`);
          values.push(quantity);
        }
        if (unitOfMeasure !== undefined) {
          updates.push(`unit = $${paramIndex++}`);
          values.push(unitOfMeasure);
        }
        if (isActive !== undefined) {
          updates.push(`is_active = $${paramIndex++}`);
          values.push(isActive);
        }
        if (locationId !== undefined) {
          updates.push(`location_id = $${paramIndex++}`);
          values.push(locationId);
        }
        if (carrierId !== undefined) {
          updates.push(`carrier_id = $${paramIndex++}`);
          values.push(carrierId);
        }
        
        if (updates.length === 0) {
          return res.status(400).json({ error: "No fields to update" });
        }
        
        values.push(id);
        
        const result = await pool.query(`
          UPDATE warehouse.stock_panels
          SET ${updates.join(', ')}
          WHERE id = $${paramIndex}
          RETURNING 
            id,
            generated_name AS name,
            generated_name AS "internalCode",
            quantity,
            unit AS "unitOfMeasure",
            is_active AS "isActive",
            location_id AS "locationId",
            carrier_id AS "carrierId"
        `, values);
        
        if (result.rows.length === 0) {
          return res.status(404).json({ error: "Formatka not found" });
        }
        
        return res.json(result.rows[0]);
      }
      
      // Special handling for opakowania and produkty-spakowane categories - update packaging_materials table
      if (category === 'opakowania' || category === 'produkty-spakowane') {
        const updates: string[] = [];
        const values: any[] = [];
        let paramIndex = 1;
        
        if (groupId !== undefined) {
          updates.push(`group_id = $${paramIndex++}`);
          values.push(groupId);
        }
        if (name !== undefined) {
          updates.push(`name = $${paramIndex++}`);
          values.push(name);
        }
        // For packaging materials, use internalCode as the primary code
        // (packaging_materials.supplier_code is used as internal code in UI)
        if (internalCode !== undefined) {
          updates.push(`supplier_code = $${paramIndex++}`);
          values.push(internalCode);
        }
        if (description !== undefined) {
          updates.push(`description = $${paramIndex++}`);
          values.push(description);
        }
        if (unitOfMeasure !== undefined) {
          updates.push(`unit = $${paramIndex++}`);
          values.push(unitOfMeasure);
        }
        if (quantity !== undefined) {
          updates.push(`quantity = $${paramIndex++}`);
          values.push(quantity);
        }
        if (isActive !== undefined) {
          updates.push(`is_active = $${paramIndex++}`);
          values.push(isActive);
        }
        if (locationId !== undefined) {
          updates.push(`location_id = $${paramIndex++}`);
          values.push(locationId);
        }
        if (carrierId !== undefined) {
          updates.push(`carrier_id = $${paramIndex++}`);
          values.push(carrierId);
        }
        
        if (updates.length === 0) {
          return res.status(400).json({ error: "No fields to update" });
        }
        
        updates.push('updated_at = CURRENT_TIMESTAMP');
        values.push(id);
        
        const result = await pool.query(`
          UPDATE warehouse.packaging_materials
          SET ${updates.join(', ')}
          WHERE id = $${paramIndex}
          RETURNING 
            id,
            group_id AS "groupId",
            name,
            supplier_code AS "internalCode",
            supplier_code AS "supplierCode",
            description,
            unit AS "unitOfMeasure",
            quantity,
            location_id AS "locationId",
            carrier_id AS "carrierId",
            is_active AS "isActive",
            created_at AS "createdAt",
            updated_at AS "updatedAt"
        `, values);
        
        if (result.rows.length === 0) {
          return res.status(404).json({ error: "Packaging material not found" });
        }
        
        return res.json(result.rows[0]);
      }
      
      // Build dynamic update query based on provided fields
      const updates: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;
      
      if (groupId !== undefined) {
        updates.push(`group_id = $${paramIndex++}`);
        values.push(groupId);
      }
      if (name !== undefined) {
        updates.push(`name = $${paramIndex++}`);
        values.push(name);
      }
      if (internalCode !== undefined) {
        updates.push(`internal_code = $${paramIndex++}`);
        values.push(internalCode);
      }
      if (supplierCode !== undefined) {
        updates.push(`supplier_code = $${paramIndex++}`);
        values.push(supplierCode);
      }
      if (description !== undefined) {
        updates.push(`description = $${paramIndex++}`);
        values.push(description);
      }
      if (specifications !== undefined) {
        updates.push(`specifications = $${paramIndex++}`);
        values.push(specifications);
      }
      if (unitOfMeasure !== undefined) {
        updates.push(`unit_of_measure = $${paramIndex++}`);
        values.push(unitOfMeasure);
      }
      if (price !== undefined) {
        updates.push(`price = $${paramIndex++}`);
        values.push(price);
      }
      if (gallery !== undefined) {
        updates.push(`gallery = $${paramIndex++}`);
        values.push(gallery);
      }
      if (primaryImageUrl !== undefined) {
        updates.push(`primary_image_url = $${paramIndex++}`);
        values.push(primaryImageUrl);
      }
      if (displayOrder !== undefined) {
        updates.push(`display_order = $${paramIndex++}`);
        values.push(displayOrder);
      }
      if (isActive !== undefined) {
        updates.push(`is_active = $${paramIndex++}`);
        values.push(isActive);
      }
      if (locationId !== undefined) {
        updates.push(`location_id = $${paramIndex++}`);
        values.push(locationId);
      }
      if (carrierId !== undefined) {
        updates.push(`carrier_id = $${paramIndex++}`);
        values.push(carrierId);
      }
      
      if (updates.length === 0) {
        return res.status(400).json({ error: "No fields to update" });
      }
      
      updates.push('updated_at = CURRENT_TIMESTAMP');
      values.push(id);
      
      const result = await pool.query(`
        UPDATE warehouse.materials
        SET ${updates.join(', ')}
        WHERE id = $${paramIndex}
        RETURNING 
          id,
          group_id AS "groupId",
          name,
          internal_code AS "internalCode",
          supplier_code AS "supplierCode",
          description,
          specifications,
          unit_of_measure AS "unitOfMeasure",
          quantity,
          location_id AS "locationId",
          carrier_id AS "carrierId",
          gallery,
          primary_image_url AS "primaryImageUrl",
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, values);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "Material with this internal code already exists" });
      }
      console.error("❌ Error updating material:", error);
      res.status(500).json({ error: "Failed to update material" });
    }
  });
  
  // DELETE /api/warehouse/materials/:id - Usuń materiał
  app.delete("/api/warehouse/materials/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { category } = req.query;
      
      // Special handling for formatki category - delete from stock_panels
      if (category === 'formatki') {
        await pool.query(
          'DELETE FROM warehouse.stock_panels WHERE id = $1',
          [id]
        );
        
        return res.json({ success: true });
      }
      
      await pool.query(
        'DELETE FROM warehouse.materials WHERE id = $1',
        [id]
      );
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting material:", error);
      res.status(500).json({ error: "Failed to delete material" });
    }
  });

  // GET /api/warehouse/formatki/:id/reservation-history - Pobierz historię rezerwacji formatek
  app.get("/api/warehouse/formatki/:id/reservation-history", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { limit = '50', offset = '0' } = req.query;
      
      const formatkiCheck = await pool.query(
        'SELECT id, generated_name as "generatedName" FROM warehouse.stock_panels WHERE id = $1',
        [id]
      );
      
      if (formatkiCheck.rows.length === 0) {
        return res.status(404).json({ error: "Formatka not found" });
      }
      
      const generatedName = formatkiCheck.rows[0].generatedName;
      
      const countResult = await pool.query(`
        SELECT COUNT(*) as total
        FROM production.production_buffer_reservations
        WHERE product_sku = $1
      `, [generatedName]);
      
      const total = parseInt(countResult.rows[0].total);
      
      const reservationsResult = await pool.query(`
        SELECT 
          r.id,
          r.product_sku as "productSku",
          r.quantity_reserved as "quantityReserved",
          r.quantity_consumed as "quantityConsumed",
          r.status,
          r.reserved_at as "reservedAt",
          r.consumed_at as "consumedAt",
          r.cancelled_at as "cancelledAt",
          r.notes,
          pp.plan_number as "planNumber",
          pp.name as "planName",
          r.zlp_id as "planId",
          r.zlp_item_id as "planLineId",
          u.username as "reservedBy"
        FROM production.production_buffer_reservations r
        LEFT JOIN production.production_plans pp ON pp.id = r.zlp_id
        LEFT JOIN users u ON u.id = r.reserved_by
        WHERE r.product_sku = $1
        ORDER BY r.reserved_at DESC
        LIMIT $2 OFFSET $3
      `, [generatedName, parseInt(limit as string), parseInt(offset as string)]);
      
      const history = reservationsResult.rows.map((row: any) => ({
        id: row.id,
        materialId: parseInt(id as string),
        operationType: row.status === 'ACTIVE' ? 'reservation' : 
                      row.status === 'CONSUMED' ? 'consumption' : 
                      row.status === 'CANCELLED' ? 'cancellation' : 'reservation',
        quantityChange: row.status === 'CONSUMED' ? `-${row.quantityConsumed || row.quantityReserved}` : 
                       row.status === 'ACTIVE' ? `-${row.quantityReserved}` : 
                       `+${row.quantityReserved}`,
        quantityBefore: null,
        quantityAfter: null,
        productionPlanId: row.planId,
        notes: row.notes || `${row.status === 'ACTIVE' ? 'Rezerwacja' : row.status === 'CONSUMED' ? 'Wydanie' : 'Anulowanie'} dla planu ${row.planNumber || row.planId}`,
        documentNumber: row.planNumber,
        performedBy: row.reservedBy,
        createdAt: row.status === 'CONSUMED' ? row.consumedAt : 
                  row.status === 'CANCELLED' ? row.cancelledAt : row.reservedAt,
        originalStatus: row.status,
        quantityReserved: row.quantityReserved,
        quantityConsumed: row.quantityConsumed
      }));
      
      res.json({
        items: history,
        total,
        limit: parseInt(limit as string),
        offset: parseInt(offset as string)
      });
    } catch (error) {
      console.error("Error fetching formatka reservation history:", error);
      res.status(500).json({ error: "Failed to fetch reservation history" });
    }
  });

  // GET /api/warehouse/materials/:id/inventory-history - Pobierz historię stanów magazynowych
  app.get("/api/warehouse/materials/:id/inventory-history", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { limit = '100', offset = '0', category } = req.query;
      
      // Special handling for formatki - return empty history for now
      if (category === 'formatki') {
        // Verify formatka exists in stock_panels
        const formatkiCheck = await pool.query(
          'SELECT id FROM warehouse.stock_panels WHERE id = $1',
          [id]
        );
        
        if (formatkiCheck.rows.length === 0) {
          return res.status(404).json({ error: "Formatka not found" });
        }
        
        // TODO: Implement formatki inventory history tracking if needed
        return res.json([]);
      }
      
      const result = await pool.query(`
        SELECT 
          id,
          material_id AS "materialId",
          operation_type AS "operationType",
          quantity_change AS "quantityChange",
          quantity_before AS "quantityBefore",
          quantity_after AS "quantityAfter",
          production_order_id AS "productionOrderId",
          production_plan_id AS "productionPlanId",
          notes,
          document_number AS "documentNumber",
          performed_by AS "performedBy",
          created_at AS "createdAt"
        FROM warehouse.inventory_history
        WHERE material_id = $1
        ORDER BY created_at DESC
        LIMIT $2 OFFSET $3
      `, [id, limit, offset]);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching inventory history:", error);
      res.status(500).json({ error: "Failed to fetch inventory history" });
    }
  });

  // POST /api/warehouse/materials/:id/inventory-adjustment - Dodaj wpis inwentaryzacyjny
  app.post("/api/warehouse/materials/:id/inventory-adjustment", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { id } = req.params;
      const { category } = req.query;
      const {
        operationType,
        quantityChange,
        notes,
        documentNumber,
        productionOrderId,
        productionPlanId
      } = req.body;
      
      if (!operationType || quantityChange === undefined) {
        return res.status(400).json({ error: "Operation type and quantity change are required" });
      }
      
      await client.query('BEGIN');
      
      // Special handling for formatki category - update stock_panels
      if (category === 'formatki') {
        const formatkiResult = await client.query(
          'SELECT quantity, generated_name FROM warehouse.stock_panels WHERE id = $1 FOR UPDATE',
          [id]
        );
        
        if (formatkiResult.rows.length === 0) {
          await client.query('ROLLBACK');
          return res.status(404).json({ error: "Formatka not found" });
        }
        
        const currentQuantity = safeParseNumber(formatkiResult.rows[0].quantity);
        const change = safeParseNumber(quantityChange);
        const newQuantity = currentQuantity + change;
        
        // Don't allow negative quantities
        if (newQuantity < 0) {
          await client.query('ROLLBACK');
          return res.status(400).json({ error: "Insufficient quantity. Cannot reduce below zero." });
        }
        
        // Update formatka quantity
        await client.query(
          'UPDATE warehouse.stock_panels SET quantity = $1 WHERE id = $2',
          [newQuantity, id]
        );
        
        await client.query('COMMIT');
        
        return res.json({
          history: null, // No history tracking for formatki yet
          newQuantity
        });
      }
      
      // Get current material quantity
      const materialResult = await client.query(
        'SELECT quantity, name FROM warehouse.materials WHERE id = $1 FOR UPDATE',
        [id]
      );
      
      if (materialResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Material not found" });
      }
      
      const currentQuantity = safeParseNumber(materialResult.rows[0].quantity);
      const change = safeParseNumber(quantityChange);
      const newQuantity = currentQuantity + change;
      
      // Don't allow negative quantities
      if (newQuantity < 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Insufficient quantity. Cannot reduce below zero." });
      }
      
      // Update material quantity
      await client.query(
        'UPDATE warehouse.materials SET quantity = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
        [newQuantity, id]
      );
      
      // Create inventory history record
      const username = (req.user as any)?.username || 'system';
      const historyResult = await client.query(`
        INSERT INTO warehouse.inventory_history (
          material_id, operation_type, quantity_change, quantity_before, quantity_after,
          production_order_id, production_plan_id, notes, document_number, performed_by
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING 
          id,
          material_id AS "materialId",
          operation_type AS "operationType",
          quantity_change AS "quantityChange",
          quantity_before AS "quantityBefore",
          quantity_after AS "quantityAfter",
          production_order_id AS "productionOrderId",
          production_plan_id AS "productionPlanId",
          notes,
          document_number AS "documentNumber",
          performed_by AS "performedBy",
          created_at AS "createdAt"
      `, [
        id,
        operationType,
        change,
        currentQuantity,
        newQuantity,
        productionOrderId || null,
        productionPlanId || null,
        notes || null,
        documentNumber || null,
        username
      ]);
      
      await client.query('COMMIT');
      
      res.json({
        history: historyResult.rows[0],
        newQuantity
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error creating inventory adjustment:", error);
      res.status(500).json({ error: "Failed to create inventory adjustment" });
    } finally {
      client.release();
    }
  });
  
  // GET /api/warehouse/materials/:id/relationships - Pobierz relacje materiału (spisy, produkty)
  app.get("/api/warehouse/materials/:id/relationships", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Get inventory counts containing this material
      const countsResult = await pool.query(`
        SELECT DISTINCT
          ic.id,
          ic.name,
          ic.status,
          ic.created_at AS "createdAt",
          ic.finalized_at AS "finalizedAt"
        FROM warehouse.inventory_counts ic
        INNER JOIN warehouse.inventory_count_items ici ON ic.id = ici.inventory_count_id
        WHERE ici.material_id = $1 OR ici.stock_panel_id = $1 OR ici.packaging_material_id = $1
        ORDER BY ic.created_at DESC
      `, [id]);
      
      // TODO: Get products linked to this material (requires warehouse_material_links table)
      // For now, return empty array
      
      res.json({
        inventoryCounts: countsResult.rows,
        linkedProducts: []
      });
    } catch (error) {
      console.error("❌ Error fetching material relationships:", error);
      res.status(500).json({ error: "Failed to fetch material relationships" });
    }
  });


  // GET /api/warehouse/materials/:id/inventory-counts - Get inventory counts containing this material or packed product
  app.get("/api/warehouse/materials/:id/inventory-counts", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { category } = req.query;
      
      // Special handling for produkty-spakowane category
      if (category === 'produkty-spakowane') {
        const packedProductCheck = await pool.query(
          'SELECT id FROM warehouse.packed_products WHERE id = $1',
          [id]
        );
        
        if (packedProductCheck.rows.length === 0) {
          return res.status(404).json({ error: "Packed product not found" });
        }
        
        const countsQuery = `
          SELECT 
            ic.id AS "countId",
            ic.name AS "countName",
            ic.status AS "countStatus",
            ic.created_at AS "countCreatedAt",
            ic.finalized_at AS "countFinalizedAt",
            ici.system_quantity AS "expectedQuantity",
            ici.counted_quantity AS "countedQuantity",
            ici.difference
          FROM warehouse.inventory_count_items ici
          INNER JOIN warehouse.inventory_counts ic ON ic.id = ici.inventory_count_id
          WHERE ici.packed_product_id = $1
          ORDER BY ic.created_at DESC
        `;
        
        const result = await pool.query(countsQuery, [id]);
        return res.json(result.rows);
      }
      
      // First check if the material exists
      const materialCheck = await pool.query(
        'SELECT id FROM warehouse.materials WHERE id = $1',
        [id]
      );
      
      if (materialCheck.rows.length === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      // Get all inventory counts that include this material
      const countsQuery = `
        SELECT 
          ic.id AS "countId",
          ic.name AS "countName",
          ic.status AS "countStatus",
          ic.created_at AS "countCreatedAt",
          ic.finalized_at AS "countFinalizedAt",
          ici.system_quantity AS "expectedQuantity",
          ici.counted_quantity AS "countedQuantity",
          ici.difference
        FROM warehouse.inventory_count_items ici
        INNER JOIN warehouse.inventory_counts ic ON ic.id = ici.inventory_count_id
        WHERE ici.material_id = $1
        ORDER BY ic.created_at DESC
      `;
      
      const result = await pool.query(countsQuery, [id]);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching material inventory counts:", error);
      res.status(500).json({ error: "Failed to fetch inventory counts" });
    }
  });
  // POST /api/warehouse/materials/:id/duplicate - Duplikuj materiał do innej grupy
  app.post("/api/warehouse/materials/:id/duplicate", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { id } = req.params;
      const { targetGroupId } = req.body;
      
      if (!targetGroupId) {
        return res.status(400).json({ error: "Target group ID is required" });
      }
      
      await client.query('BEGIN');
      
      // Get original material
      const materialResult = await client.query(
        'SELECT * FROM warehouse.materials WHERE id = $1',
        [id]
      );
      
      if (materialResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Material not found" });
      }
      
      const material = materialResult.rows[0];
      
      // Generate unique internal code
      const baseCode = material.internal_code;
      let newCode = `${baseCode}_COPY`;
      let counter = 1;
      
      while (true) {
        const checkResult = await client.query(
          'SELECT id FROM warehouse.materials WHERE internal_code = $1',
          [newCode]
        );
        
        if (checkResult.rows.length === 0) break;
        
        newCode = `${baseCode}_COPY${counter}`;
        counter++;
      }
      
      // Create duplicate
      const duplicateResult = await client.query(`
        INSERT INTO warehouse.materials (
          group_id, name, internal_code, supplier_code, description,
          specifications, unit_of_measure, gallery, primary_image_url, display_order
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING 
          id,
          group_id AS "groupId",
          name,
          internal_code AS "internalCode",
          supplier_code AS "supplierCode",
          description,
          specifications,
          unit_of_measure AS "unitOfMeasure",
          gallery,
          primary_image_url AS "primaryImageUrl",
          display_order AS "displayOrder",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        targetGroupId,
        `${material.name} (kopia)`,
        newCode,
        material.supplier_code,
        material.description,
        material.specifications,
        material.unit_of_measure,
        material.gallery,
        material.primary_image_url,
        material.display_order
      ]);
      
      await client.query('COMMIT');
      res.json(duplicateResult.rows[0]);
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error duplicating material:", error);
      res.status(500).json({ error: "Failed to duplicate material" });
    } finally {
      client.release();
    }
  });

  // POST /api/warehouse/materials/:id/images - Upload zdjęcia materiału
  app.post("/api/warehouse/materials/:id/images", isAuthenticated, uploadWarehouseMaterialImage.single('image'), async (req, res) => {
    try {
      const { id } = req.params;
      
      if (!req.file) {
        return res.status(400).json({ error: "No image file provided" });
      }
      
      // Check if material exists
      const materialCheck = await pool.query(
        'SELECT id, gallery FROM warehouse.materials WHERE id = $1',
        [id]
      );
      
      if (materialCheck.rows.length === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      // Get file storage adapter
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();
      
      // Generate filename and upload through adapter
      const ext = path.extname(req.file.originalname);
      const filename = generateWarehouseMaterialImageFilename(ext);
      
      const imageUrl = await adapter.upload({
        filename,
        buffer: req.file.buffer,
        mimetype: req.file.mimetype,
        subfolder: 'warehouse/materials',
      });
      
      // Add to gallery
      const currentGallery = materialCheck.rows[0].gallery || [];
      const updatedGallery = [...currentGallery, imageUrl];
      
      // Update material with new image in gallery
      const result = await pool.query(`
        UPDATE warehouse.materials
        SET gallery = $1, updated_at = CURRENT_TIMESTAMP
        WHERE id = $2
        RETURNING 
          id,
          gallery,
          primary_image_url AS "primaryImageUrl"
      `, [updatedGallery, id]);
      
      res.json({
        success: true,
        url: imageUrl,
        material: result.rows[0]
      });
    } catch (error: any) {
      console.error("❌ Error uploading material image:", error);
      res.status(500).json({ error: "Failed to upload image", details: error.message });
    }
  });

  // DELETE /api/warehouse/materials/:id/images - Usuń zdjęcie materiału
  app.delete("/api/warehouse/materials/:id/images", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { imageUrl } = req.body;
      
      if (!imageUrl) {
        return res.status(400).json({ error: "Image URL is required" });
      }
      
      // Get current gallery
      const materialResult = await pool.query(
        'SELECT gallery, primary_image_url FROM warehouse.materials WHERE id = $1',
        [id]
      );
      
      if (materialResult.rows.length === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      const currentGallery = materialResult.rows[0].gallery || [];
      const primaryImageUrl = materialResult.rows[0].primary_image_url;
      
      // Remove image from gallery
      const updatedGallery = currentGallery.filter((url: string) => url !== imageUrl);
      
      // If removed image was primary, set primary to null or first image
      let newPrimaryImageUrl = primaryImageUrl;
      if (primaryImageUrl === imageUrl) {
        newPrimaryImageUrl = updatedGallery.length > 0 ? updatedGallery[0] : null;
      }
      
      // Update material
      const result = await pool.query(`
        UPDATE warehouse.materials
        SET gallery = $1, primary_image_url = $2, updated_at = CURRENT_TIMESTAMP
        WHERE id = $3
        RETURNING 
          id,
          gallery,
          primary_image_url AS "primaryImageUrl"
      `, [updatedGallery, newPrimaryImageUrl, id]);
      
      // Try to delete file from storage
      try {
        // File deletion handled by storage system
        // await adapter.delete(imageUrl);
      } catch (deleteError) {
        console.error("⚠️  Failed to delete file from storage:", deleteError);
        // Continue anyway - database is updated
      }
      
      res.json({
        success: true,
        material: result.rows[0]
      });
    } catch (error) {
      console.error("❌ Error deleting material image:", error);
      res.status(500).json({ error: "Failed to delete image" });
    }
  });

  // POST /api/warehouse/materials/:id/primary-image - Ustaw główne zdjęcie materiału
  app.post("/api/warehouse/materials/:id/primary-image", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { imageUrl } = req.body;
      
      if (!imageUrl) {
        return res.status(400).json({ error: "Image URL is required" });
      }
      
      // Check if image exists in gallery
      const materialResult = await pool.query(
        'SELECT gallery FROM warehouse.materials WHERE id = $1',
        [id]
      );
      
      if (materialResult.rows.length === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      const currentGallery = materialResult.rows[0].gallery || [];
      
      // Verify image is in gallery
      if (!currentGallery.includes(imageUrl)) {
        return res.status(400).json({ error: "Image not in gallery" });
      }
      
      // Update primary image
      const result = await pool.query(`
        UPDATE warehouse.materials
        SET primary_image_url = $1, updated_at = CURRENT_TIMESTAMP
        WHERE id = $2
        RETURNING 
          id,
          primary_image_url AS "primaryImageUrl"
      `, [imageUrl, id]);
      
      res.json({
        success: true,
        material: result.rows[0]
      });
    } catch (error) {
      console.error("❌ Error setting primary image:", error);
      res.status(500).json({ error: "Failed to set primary image" });
    }
  });

  // POST /api/warehouse/materials/bulk-import - Bulk import zdjęć materiałów
  app.post("/api/warehouse/materials/bulk-import", isAuthenticated, uploadWarehouseMaterialImage.array('images', 50), async (req, res) => {
    const client = await pool.connect();
    
    try {
      const files = req.files as Express.Multer.File[];
      const { groupId, category, sessionId } = req.body;
      
      if (!files || files.length === 0) {
        return res.status(400).json({ error: "No images provided" });
      }

      const totalFiles = files.length;
      emitGenerationLog(sessionId, `🚀 Rozpoczęto import ${totalFiles} plików...`, 'info');

      // Get adapter instance
      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();

      await client.query('BEGIN');
      
      const created = [];
      const errors = [];

      for (let i = 0; i < files.length; i++) {
        const file = files[i];
        const progress = Math.round(((i + 1) / totalFiles) * 100);
        
        try {
          // Pobierz nazwę pliku bez rozszerzenia
          const fileName = path.basename(file.originalname, path.extname(file.originalname));
          
          emitGenerationLog(sessionId, `📦 [${i + 1}/${totalFiles}] Przetwarzanie: ${fileName}...`, 'info');
          
          // Generate filename and upload through adapter
          const ext = path.extname(file.originalname);
          const filename = generateWarehouseMaterialImageFilename(ext);
          
          const imageUrl = await adapter.upload({
            filename,
            buffer: file.buffer,
            mimetype: file.mimetype,
            subfolder: 'warehouse/materials'
          });
          
          // Generuj unikalny kod z nazwy pliku (zamień spacje i znaki specjalne na dash)
          const baseCode = fileName
            .toLowerCase()
            .replace(/[^a-z0-9]+/g, '-')
            .replace(/^-+|-+$/g, '');
          
          // Sprawdź czy kod już istnieje, jeśli tak dodaj suffix
          let uniqueCode = baseCode;
          let suffix = 1;
          while (true) {
            const existing = await client.query(
              'SELECT id FROM warehouse.materials WHERE internal_code = $1',
              [uniqueCode]
            );
            if (existing.rows.length === 0) break;
            uniqueCode = `${baseCode}-${suffix}`;
            suffix++;
          }
          
          const result = await client.query(`
            INSERT INTO warehouse.materials (
              group_id,
              name,
              internal_code,
              gallery,
              primary_image_url,
              display_order
            )
            VALUES ($1, $2, $3, $4, $5, 0)
            RETURNING 
              id,
              group_id AS "groupId",
              name,
              internal_code AS "internalCode",
              gallery,
              primary_image_url AS "primaryImageUrl",
              display_order AS "displayOrder"
          `, [groupId || null, fileName, uniqueCode, [imageUrl], imageUrl]);
          
          created.push(result.rows[0]);
          emitGenerationLog(sessionId, `✅ [${i + 1}/${totalFiles}] Utworzono: ${fileName} (kod: ${uniqueCode}) - ${progress}%`, 'success');
        } catch (error: any) {
          errors.push({
            fileName: file.originalname,
            error: error.message
          });
          emitGenerationLog(sessionId, `❌ [${i + 1}/${totalFiles}] Błąd: ${file.originalname} - ${error.message}`, 'error');
        }
      }
      
      await client.query('COMMIT');
      
      emitGenerationLog(sessionId, `✅ Import zakończony! Utworzono: ${created.length}, Błędów: ${errors.length}`, created.length > 0 ? 'success' : 'warning');
      console.log(`✅ Bulk imported ${created.length} warehouse materials (${errors.length} errors)`);
      
      res.json({
        success: true,
        created: created.length,
        errors: errors.length,
        items: created,
        errorDetails: errors
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error in warehouse materials bulk import:", error);
      res.status(500).json({ error: "Failed to import warehouse materials" });
    } finally {
      client.release();
    }
  });

  // PATCH /api/warehouse/materials/:id/primary-image - Ustaw główne zdjęcie materiału
  app.patch("/api/warehouse/materials/:id/primary-image", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { imageUrl } = req.body;
      
      if (!imageUrl) {
        return res.status(400).json({ error: "Image URL is required" });
      }
      
      // Verify material exists and image is in gallery
      const materialResult = await pool.query(
        'SELECT gallery FROM warehouse.materials WHERE id = $1',
        [id]
      );
      
      if (materialResult.rows.length === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      const gallery = materialResult.rows[0].gallery || [];
      if (!gallery.includes(imageUrl)) {
        return res.status(400).json({ error: "Image not found in material gallery" });
      }
      
      // Update primary image
      const result = await pool.query(`
        UPDATE warehouse.materials
        SET primary_image_url = $1, updated_at = CURRENT_TIMESTAMP
        WHERE id = $2
        RETURNING 
          id,
          gallery,
          primary_image_url AS "primaryImageUrl"
      `, [imageUrl,
        unit, id]);
      
      res.json({
        success: true,
        material: result.rows[0]
      });
    } catch (error) {
      console.error("❌ Error setting primary image:", error);
      res.status(500).json({ error: "Failed to set primary image" });
    }
  });

  // POST /api/warehouse/materials/bulk-delete - Grupowe usuwanie materiałów
  app.post("/api/warehouse/materials/bulk-delete", isAuthenticated, async (req, res) => {
    try {
      const { materialIds } = req.body;

      if (!Array.isArray(materialIds) || materialIds.length === 0) {
        return res.status(400).json({ error: "Material IDs array is required" });
      }

      const placeholders = materialIds.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(
        `DELETE FROM warehouse.materials WHERE id IN (${placeholders}) RETURNING id`,
        materialIds
      );

      res.json({
        success: true,
        deletedCount: result.rowCount,
        deletedIds: result.rows.map(r => r.id)
      });
    } catch (error) {
      console.error("❌ Error bulk deleting materials:", error);
      res.status(500).json({ error: "Failed to bulk delete materials" });
    }
  });


  // ==================== ARCHIVE ENDPOINTS ====================

  // PATCH /api/warehouse/materials/:id/archive - Archiwizuj materiał
  app.patch("/api/warehouse/materials/:id/archive", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(`
        UPDATE warehouse.materials
        SET is_archived = true, archived_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
        WHERE id = $1
        RETURNING id, name, is_archived AS "isArchived", archived_at AS "archivedAt"
      `, [id]);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      res.json({ success: true, material: result.rows[0] });
    } catch (error) {
      console.error("❌ Error archiving material:", error);
      res.status(500).json({ error: "Failed to archive material" });
    }
  });

  // PATCH /api/warehouse/materials/:id/restore - Przywróć materiał z archiwum
  app.patch("/api/warehouse/materials/:id/restore", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(`
        UPDATE warehouse.materials
        SET is_archived = false, archived_at = NULL, updated_at = CURRENT_TIMESTAMP
        WHERE id = $1
        RETURNING id, name, is_archived AS "isArchived"
      `, [id]);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ error: "Material not found" });
      }
      
      res.json({ success: true, material: result.rows[0] });
    } catch (error) {
      console.error("❌ Error restoring material:", error);
      res.status(500).json({ error: "Failed to restore material" });
    }
  });

  // POST /api/warehouse/materials/bulk-archive - Grupowa archiwizacja materiałów
  app.post("/api/warehouse/materials/bulk-archive", isAuthenticated, async (req, res) => {
    try {
      const { materialIds } = req.body;

      if (!Array.isArray(materialIds) || materialIds.length === 0) {
        return res.status(400).json({ error: "Material IDs array is required" });
      }

      const placeholders = materialIds.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(`
        UPDATE warehouse.materials 
        SET is_archived = true, archived_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
        WHERE id IN (${placeholders})
        RETURNING id, name
      `, materialIds);

      res.json({
        success: true,
        archivedCount: result.rowCount,
        archivedMaterials: result.rows
      });
    } catch (error) {
      console.error("❌ Error bulk archiving materials:", error);
      res.status(500).json({ error: "Failed to bulk archive materials" });
    }
  });

  // POST /api/warehouse/materials/bulk-restore - Grupowe przywracanie materiałów
  app.post("/api/warehouse/materials/bulk-restore", isAuthenticated, async (req, res) => {
    try {
      const { materialIds } = req.body;

      if (!Array.isArray(materialIds) || materialIds.length === 0) {
        return res.status(400).json({ error: "Material IDs array is required" });
      }

      const placeholders = materialIds.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(`
        UPDATE warehouse.materials 
        SET is_archived = false, archived_at = NULL, updated_at = CURRENT_TIMESTAMP
        WHERE id IN (${placeholders})
        RETURNING id, name
      `, materialIds);

      res.json({
        success: true,
        restoredCount: result.rowCount,
        restoredMaterials: result.rows
      });
    } catch (error) {
      console.error("❌ Error bulk restoring materials:", error);
      res.status(500).json({ error: "Failed to bulk restore materials" });
    }
  });

  // GET /api/warehouse/archive - Pobierz wszystkie zarchiwizowane materiały
  app.get("/api/warehouse/archive", isAuthenticated, async (req, res) => {
    try {
      const { category, search } = req.query;
      
      let whereClause = 'm.is_archived = true';
      const params: any[] = [];
      let paramIndex = 1;

      if (category && category !== 'all') {
        whereClause += ` AND (mg.category = $${paramIndex} OR m.specifications->>'category' = $${paramIndex})`;
        params.push(category);
        paramIndex++;
      }

      if (search) {
        whereClause += ` AND (LOWER(m.name) LIKE $${paramIndex} OR LOWER(m.internal_code) LIKE $${paramIndex})`;
        params.push(`%${(search as string).toLowerCase()}%`);
        paramIndex++;
      }

      const result = await pool.query(`
        SELECT 
          m.id,
          m.name,
          m.internal_code AS "internalCode",
          NULL AS "supplierCode",
          m.quantity,
          m.unit_of_measure AS "unitOfMeasure",
          m.archived_at AS "archivedAt",
          m.group_id AS "groupId",
          mg.name AS "groupName",
          mg.category,
          COALESCE(mg.category, m.specifications->>'category', 'other') AS "materialCategory"
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        WHERE ${whereClause}
        ORDER BY m.archived_at DESC
      `, params);

      // Get archived stock panels (formatki)
      const stockPanelsResult = await pool.query(`
        SELECT 
          sp.id,
          sp.name,
          sp.internal_code AS "internalCode",
          sp.supplier_code AS "supplierCode",
          sp.quantity,
          sp.archived_at AS "archivedAt",
          NULL AS "groupId",
          NULL AS "groupName",
          'formatki' AS category,
          'formatki' AS "materialCategory"
        FROM warehouse.stock_panels sp
        WHERE sp.is_archived = true
        ${search ? `AND (LOWER(sp.name) LIKE $1 OR LOWER(sp.internal_code) LIKE $1)` : ''}
        ${category && category !== 'all' && category !== 'formatki' ? 'AND 1=0' : ''}
        ORDER BY sp.archived_at DESC
      `, search ? [`%${(search as string).toLowerCase()}%`] : []);

      // Get archived packed products
      const packedProductsResult = await pool.query(`
        SELECT 
          pp.id,
          cp.name AS name,
          pp.sku AS "internalCode",
          pp.sku AS "supplierCode",
          pp.quantity,
          pp.archived_at AS "archivedAt",
          NULL AS "groupId",
          NULL AS "groupName",
          'produkty-spakowane' AS category,
          'produkty-spakowane' AS "materialCategory"
        FROM warehouse.packed_products pp
        LEFT JOIN catalog.products cp ON pp.catalog_product_id = cp.id
        WHERE pp.is_archived = true
        ${search ? `AND (LOWER(cp.name) LIKE $1 OR LOWER(pp.sku) LIKE $1)` : ''}
        ${category && category !== 'all' && category !== 'produkty-spakowane' ? 'AND 1=0' : ''}
        ORDER BY pp.archived_at DESC
      `, search ? [`%${(search as string).toLowerCase()}%`] : []);

      // Combine all archived items if category is 'all' or not specified
      let allItems = [...result.rows];
      
      if (!category || category === 'all' || category === 'formatki') {
        allItems = [...allItems, ...stockPanelsResult.rows];
      }
      if (!category || category === 'all' || category === 'produkty-spakowane') {
        allItems = [...allItems, ...packedProductsResult.rows];
      }

      // Sort by archivedAt desc
      allItems.sort((a, b) => new Date(b.archivedAt).getTime() - new Date(a.archivedAt).getTime());

      // Count by category
      const categoryCounts: Record<string, number> = {};
      allItems.forEach(item => {
        const cat = item.materialCategory || 'other';
        categoryCounts[cat] = (categoryCounts[cat] || 0) + 1;
      });

      res.json({
        materials: allItems,
        totalCount: allItems.length,
        categoryCounts
      });
    } catch (error) {
      console.error("❌ Error fetching archived materials:", error);
      res.status(500).json({ error: "Failed to fetch archived materials" });
    }
  });

  // PATCH /api/warehouse/stock-panels/:id/archive - Archiwizuj formatkę
  app.patch("/api/warehouse/stock-panels/:id/archive", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(`
        UPDATE warehouse.stock_panels
        SET is_archived = true, archived_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
        WHERE id = $1
        RETURNING id, name, is_archived AS "isArchived", archived_at AS "archivedAt"
      `, [id]);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ error: "Stock panel not found" });
      }
      
      res.json({ success: true, stockPanel: result.rows[0] });
    } catch (error) {
      console.error("❌ Error archiving stock panel:", error);
      res.status(500).json({ error: "Failed to archive stock panel" });
    }
  });

  // PATCH /api/warehouse/stock-panels/:id/restore - Przywróć formatkę z archiwum
  app.patch("/api/warehouse/stock-panels/:id/restore", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(`
        UPDATE warehouse.stock_panels
        SET is_archived = false, archived_at = NULL, updated_at = CURRENT_TIMESTAMP
        WHERE id = $1
        RETURNING id, name, is_archived AS "isArchived"
      `, [id]);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ error: "Stock panel not found" });
      }
      
      res.json({ success: true, stockPanel: result.rows[0] });
    } catch (error) {
      console.error("❌ Error restoring stock panel:", error);
      res.status(500).json({ error: "Failed to restore stock panel" });
    }
  });

  // POST /api/warehouse/stock-panels/bulk-archive - Grupowa archiwizacja formatek
  app.post("/api/warehouse/stock-panels/bulk-archive", isAuthenticated, async (req, res) => {
    try {
      const { ids } = req.body;

      if (!Array.isArray(ids) || ids.length === 0) {
        return res.status(400).json({ error: "IDs array is required" });
      }

      const placeholders = ids.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(`
        UPDATE warehouse.stock_panels 
        SET is_archived = true, archived_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
        WHERE id IN (${placeholders})
        RETURNING id, name
      `, ids);

      res.json({
        success: true,
        archivedCount: result.rowCount,
        archivedItems: result.rows
      });
    } catch (error) {
      console.error("❌ Error bulk archiving stock panels:", error);
      res.status(500).json({ error: "Failed to bulk archive stock panels" });
    }
  });

  // POST /api/warehouse/stock-panels/bulk-restore - Grupowe przywracanie formatek
  app.post("/api/warehouse/stock-panels/bulk-restore", isAuthenticated, async (req, res) => {
    try {
      const { ids } = req.body;

      if (!Array.isArray(ids) || ids.length === 0) {
        return res.status(400).json({ error: "IDs array is required" });
      }

      const placeholders = ids.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(`
        UPDATE warehouse.stock_panels 
        SET is_archived = false, archived_at = NULL, updated_at = CURRENT_TIMESTAMP
        WHERE id IN (${placeholders})
        RETURNING id, name
      `, ids);

      res.json({
        success: true,
        restoredCount: result.rowCount,
        restoredItems: result.rows
      });
    } catch (error) {
      console.error("❌ Error bulk restoring stock panels:", error);
      res.status(500).json({ error: "Failed to bulk restore stock panels" });
    }
  });

  // PATCH /api/warehouse/packed-products/:id/archive - Archiwizuj produkt spakowany
  app.patch("/api/warehouse/packed-products/:id/archive", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(`
        UPDATE warehouse.packed_products
        SET is_archived = true, archived_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
        WHERE id = $1
        RETURNING id, sku, is_archived AS "isArchived", archived_at AS "archivedAt"
      `, [id]);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ error: "Packed product not found" });
      }
      
      res.json({ success: true, packedProduct: result.rows[0] });
    } catch (error) {
      console.error("❌ Error archiving packed product:", error);
      res.status(500).json({ error: "Failed to archive packed product" });
    }
  });

  // PATCH /api/warehouse/packed-products/:id/restore - Przywróć produkt spakowany z archiwum
  app.patch("/api/warehouse/packed-products/:id/restore", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const result = await pool.query(`
        UPDATE warehouse.packed_products
        SET is_archived = false, archived_at = NULL, updated_at = CURRENT_TIMESTAMP
        WHERE id = $1
        RETURNING id, sku, is_archived AS "isArchived"
      `, [id]);
      
      if (result.rowCount === 0) {
        return res.status(404).json({ error: "Packed product not found" });
      }
      
      res.json({ success: true, packedProduct: result.rows[0] });
    } catch (error) {
      console.error("❌ Error restoring packed product:", error);
      res.status(500).json({ error: "Failed to restore packed product" });
    }
  });

  // POST /api/warehouse/packed-products/bulk-archive - Grupowa archiwizacja produktów spakowanych
  app.post("/api/warehouse/packed-products/bulk-archive", isAuthenticated, async (req, res) => {
    try {
      const { ids } = req.body;

      if (!Array.isArray(ids) || ids.length === 0) {
        return res.status(400).json({ error: "IDs array is required" });
      }

      const placeholders = ids.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(`
        UPDATE warehouse.packed_products 
        SET is_archived = true, archived_at = CURRENT_TIMESTAMP, updated_at = CURRENT_TIMESTAMP
        WHERE id IN (${placeholders})
        RETURNING id, sku
      `, ids);

      res.json({
        success: true,
        archivedCount: result.rowCount,
        archivedItems: result.rows
      });
    } catch (error) {
      console.error("❌ Error bulk archiving packed products:", error);
      res.status(500).json({ error: "Failed to bulk archive packed products" });
    }
  });

  // POST /api/warehouse/packed-products/bulk-restore - Grupowe przywracanie produktów spakowanych
  app.post("/api/warehouse/packed-products/bulk-restore", isAuthenticated, async (req, res) => {
    try {
      const { ids } = req.body;

      if (!Array.isArray(ids) || ids.length === 0) {
        return res.status(400).json({ error: "IDs array is required" });
      }

      const placeholders = ids.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(`
        UPDATE warehouse.packed_products 
        SET is_archived = false, archived_at = NULL, updated_at = CURRENT_TIMESTAMP
        WHERE id IN (${placeholders})
        RETURNING id, sku
      `, ids);

      res.json({
        success: true,
        restoredCount: result.rowCount,
        restoredItems: result.rows
      });
    } catch (error) {
      console.error("❌ Error bulk restoring packed products:", error);
      res.status(500).json({ error: "Failed to bulk restore packed products" });
    }
  });

  // ==================== END ARCHIVE ENDPOINTS ====================

  // POST /api/warehouse/materials/bulk-update - Grupowa aktualizacja materiałów
  app.post("/api/warehouse/materials/bulk-update", isAuthenticated, async (req, res) => {
    try {
      const { materialIds, updates } = req.body;
      const username = (req as any).user?.username || 'system';

      if (!Array.isArray(materialIds) || materialIds.length === 0) {
        return res.status(400).json({ error: "Material IDs array is required" });
      }

      if (!updates || typeof updates !== 'object') {
        return res.status(400).json({ error: "Updates object is required" });
      }

      // Build UPDATE query dynamically based on provided fields
      const allowedFields = ['price', 'unit_of_measure', 'carrier_id', 'location_id', 'group_id'];
      const updatePairs: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      for (const field of allowedFields) {
        if (updates[field] !== undefined) {
          // Allow null values for carrier_id, location_id, and group_id (to remove assignment)
          if (updates[field] === null && (field === 'carrier_id' || field === 'location_id' || field === 'group_id')) {
            updatePairs.push(`${field} = NULL`);
          } else if (updates[field] !== null) {
            updatePairs.push(`${field} = $${paramIndex}`);
            values.push(updates[field]);
            paramIndex++;
          }
        }
      }

      if (updatePairs.length === 0) {
        return res.status(400).json({ error: "No valid fields to update" });
      }

      // Add username parameter
      const usernameParamIndex = paramIndex;
      values.push(username);
      paramIndex++;

      // Add materialIds to params
      const placeholders = materialIds.map((_: any, i: number) => `$${paramIndex + i}`).join(',');
      values.push(...materialIds);

      const query = `
        UPDATE warehouse.materials
        SET ${updatePairs.join(', ')}, updated_at = CURRENT_TIMESTAMP, updated_by = $${usernameParamIndex}
        WHERE id IN (${placeholders})
        RETURNING id
      `;

      const result = await pool.query(query, values);

      res.json({
        success: true,
        updatedCount: result.rowCount,
        updatedIds: result.rows.map((r: any) => r.id)
      });
    } catch (error) {
      console.error("❌ Error bulk updating materials:", error);
      res.status(500).json({ error: "Failed to bulk update materials" });
    }
  });
  // POST /api/warehouse/materials/export - Eksport materiałów do CSV
  app.post("/api/warehouse/materials/export", isAuthenticated, async (req, res) => {
    try {
      const { materialIds, format = 'csv' } = req.body;

      if (!Array.isArray(materialIds) || materialIds.length === 0) {
        return res.status(400).json({ error: "Material IDs array is required" });
      }

      if (format !== 'csv') {
        return res.status(400).json({ error: "Only CSV format is currently supported" });
      }

      // Fetch materials data
      const placeholders = materialIds.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(`
        SELECT 
          m.id,
          mg.name AS "Grupa",
          m.name AS "Nazwa",
          m.internal_code AS "Kod wewnętrzny",
          m.supplier_code AS "Kod dostawcy",
          m.description AS "Opis",
          m.unit_of_measure AS "Jednostka",
          m.price AS "Cena",
          m.created_at AS "Data utworzenia"
        FROM warehouse.materials m
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        WHERE m.id IN (${placeholders})
        ORDER BY mg.name, m.name
      `, materialIds);

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "No materials found" });
      }

      // Generate CSV
      const headers = ['Grupa', 'Nazwa', 'Kod wewnętrzny', 'Kod dostawcy', 'Opis', 'Jednostka', 'Cena', 'Data utworzenia'];
      const csvRows = [headers.join(',')];
      
      result.rows.forEach(row => {
        const values = [
          row.Grupa || '',
          row.Nazwa || '',
          row['Kod wewnętrzny'] || '',
          row['Kod dostawcy'] || '',
          (row.Opis || '').replace(/,/g, ';').replace(/\n/g, ' '),
          row.Jednostka || '',
          row.Cena || '',
          row['Data utworzenia'] ? new Date(row['Data utworzenia']).toLocaleDateString('pl-PL') : ''
        ];
        csvRows.push(values.map(v => `"${v}"`).join(','));
      });

      const csv = csvRows.join('\n');
      const filename = `materialy-${new Date().toISOString().split('T')[0]}.csv`;

      res.setHeader('Content-Type', 'text/csv; charset=utf-8');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      res.send('\ufeff' + csv); // Add BOM for Excel UTF-8 support
    } catch (error) {
      console.error("❌ Error exporting materials:", error);
      res.status(500).json({ error: "Failed to export materials" });
    }
  });

  // POST /api/warehouse/materials/import-csv - Import materiałów z CSV
  app.post("/api/warehouse/materials/import-csv", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { csvData } = req.body;
      
      if (!csvData || typeof csvData !== 'string') {
        return res.status(400).json({ error: "CSV data is required" });
      }

      // Parse CSV using PapaParse
      const parseResult = Papa.parse(csvData, {
        header: true,
        skipEmptyLines: true,
        transformHeader: (header: string) => header.trim(),
      });

      if (parseResult.errors.length > 0) {
        return res.status(400).json({ 
          error: "CSV parsing failed",
          details: parseResult.errors.map(e => e.message)
        });
      }

      const rows = parseResult.data as any[];
      
      if (rows.length === 0) {
        return res.status(400).json({ error: "CSV file is empty" });
      }

      // Validate required columns
      const requiredColumns = ['name', 'internal_code'];
      const firstRow = rows[0];
      const missingColumns = requiredColumns.filter(col => !(col in firstRow));
      if (missingColumns.length > 0) {
        return res.status(400).json({ 
          error: `Missing required columns: ${missingColumns.join(', ')}` 
        });
      }

      // Fetch all groups for lookup
      const groupsResult = await pool.query(`
        SELECT id, code FROM warehouse.material_groups
      `);
      const groupsMap = new Map<string, number>();
      groupsResult.rows.forEach((g: any) => {
        groupsMap.set(g.code, g.id);
      });

      const results = {
        success: 0,
        failed: 0,
        errors: [] as Array<{ row: number; error: string; data?: any }>
      };

      // Start transaction
      await client.query('BEGIN');

      // Process each row
      for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const rowNumber = i + 2; // +2 because: +1 for 1-indexing, +1 for header row

        try {
          // Validate name and internal_code
          if (!row.name || row.name.trim() === '') {
            results.errors.push({
              row: rowNumber,
              error: 'Name is required',
              data: row
            });
            results.failed++;
            continue;
          }

          if (!row.internal_code || row.internal_code.trim() === '') {
            results.errors.push({
              row: rowNumber,
              error: 'Internal code is required',
              data: row
            });
            results.failed++;
            continue;
          }

          // Resolve group_id from group_code
          let groupId = null;
          if (row.group_code && row.group_code.trim() !== '') {
            groupId = groupsMap.get(row.group_code.trim());
            if (groupId === undefined) {
              results.errors.push({
                row: rowNumber,
                error: `Group code not found: ${row.group_code}`,
                data: row
              });
              results.failed++;
              continue;
            }
          }

          // Parse display_order
          const displayOrder = row.display_order ? parseInt(row.display_order, 10) : 0;
          
          // Parse price
          const price = row.price ? parseFloat(row.price) : null;

          // Insert or update (upsert) into database
          await client.query(`
            INSERT INTO warehouse.materials (
              group_id, name, internal_code, supplier_code, description, unit_of_measure, price, display_order
            )
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8)
            ON CONFLICT (internal_code) 
            DO UPDATE SET
              group_id = EXCLUDED.group_id,
              name = EXCLUDED.name,
              supplier_code = EXCLUDED.supplier_code,
              description = EXCLUDED.description,
              unit_of_measure = EXCLUDED.unit_of_measure,
              price = COALESCE(EXCLUDED.price, warehouse.materials.price),
              display_order = EXCLUDED.display_order,
              is_active = true,
              updated_at = CURRENT_TIMESTAMP
          `, [
            groupId,
            row.name.trim(),
            row.internal_code.trim(),
            row.supplier_code?.trim() || null,
            row.description?.trim() || null,
            row.unit_of_measure?.trim() || 'szt',
            displayOrder
          ]);

          results.success++;

          results.success++;
        } catch (error: any) {
          console.error(`❌ Error importing row ${rowNumber}:`, error);
          
          results.errors.push({
            row: rowNumber,
            error: error instanceof Error ? error.message : 'Unknown error',
            data: row
          });
          results.failed++;
        }
      }

      // Always commit - upsert handles conflicts gracefully
      await client.query('COMMIT');

      // Return results with appropriate status
      if (results.failed > 0 && results.success === 0) {
        return res.status(400).json({
          error: "Import failed - all rows had errors",
          ...results
        });
      }

      res.json({
        message: results.failed > 0 
          ? `Import completed with warnings: ${results.success} succeeded, ${results.failed} failed`
          : `Import completed successfully: ${results.success} materials imported`,
        ...results
      });
    } catch (error) {
      // Rollback transaction on error
      await client.query('ROLLBACK');
      console.error("❌ Error importing CSV:", error);
      res.status(500).json({ error: "Failed to import CSV" });
    } finally {
      client.release();
    }
  });

  // POST /api/warehouse/materials/import-with-mapping - Import materiałów z mapowaniem pól
  app.post("/api/warehouse/materials/import-with-mapping", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { rows, mapping, category } = req.body;
      const username = (req.user as any)?.username || "system";
      
      if (!rows || !Array.isArray(rows) || rows.length === 0) {
        return res.status(400).json({ error: "No data to import" });
      }

      // Debug: log first row to see mapped fields
      console.log("📦 [IMPORT] Category:", category);
      console.log("📦 [IMPORT] Mapping:", JSON.stringify(mapping));
      console.log("📦 [IMPORT] First row:", JSON.stringify(rows[0]));
      console.log("📦 [IMPORT] First row price:", rows[0]?.price, "type:", typeof rows[0]?.price);
      // Get group for category or default
      let groupId: number | null = null;
      if (category) {
        const groupResult = await pool.query(
          `SELECT id FROM warehouse.material_groups WHERE category = $1 LIMIT 1`,
          [category]
        );
        if (groupResult.rows.length > 0) {
          groupId = groupResult.rows[0].id;
        }
      }

      // Get all groups for lookup if group_code is in mapping
      const groupsResult = await pool.query(`SELECT id, code, name FROM warehouse.material_groups`);
      const groupsMap = new Map<string, number>();
      groupsResult.rows.forEach((g: any) => {
        groupsMap.set(g.code, g.id);
        groupsMap.set(g.name.toLowerCase(), g.id);
      });

      const results = {
        success: 0,
        failed: 0,
        errors: [] as Array<{ row: number; error: string; data?: any }>
      };

      await client.query("BEGIN");

      for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const rowNumber = i + 2;

        try {
          const name = row.name?.toString().trim() || null;
          const internalCode = row.internal_code?.toString().trim() || null;
          const supplierCode = row.supplier_code?.toString().trim() || null;
          const priceValue = row.price ? parseFloat(row.price) : null;
          
          // If we have supplier_code, try to update existing material first
          if (supplierCode) {
            const updateResult = await client.query(`
              UPDATE warehouse.materials SET
                name = COALESCE($1, name),
                description = COALESCE($2, description),
                unit_of_measure = COALESCE($3, unit_of_measure),
                price = COALESCE($4, price),
                updated_by = $6,
                is_active = true,
                updated_at = CURRENT_TIMESTAMP
              WHERE supplier_code = $5
              RETURNING id, name
            `, [
              name,
              row.description?.toString().trim() || null,
              row.unit_of_measure?.toString().trim() || null,
              priceValue,
              supplierCode,
              username
            ]);

          results.success++;
            
            if (updateResult.rowCount && updateResult.rowCount > 0) {
              console.log(`📦 [IMPORT] Updated ${updateResult.rowCount} material(s) by supplier_code: ${supplierCode}, price: ${priceValue}`);
              results.success++;
              continue; // Skip to next row - update was successful
            }
          }
          
          // For INSERT - require name and internal_code
          if (!name || !internalCode) {
            results.errors.push({
              row: rowNumber,
              error: supplierCode 
                ? `Materiał z kodem dostawcy "${supplierCode}" nie istnieje. Dla nowych wymagana jest Nazwa i Kod wewnętrzny.`
                : "Nazwa i kod wewnętrzny są wymagane",
              data: row
            });
            results.failed++;
            continue;
          }

          let finalGroupId = groupId;
          if (row.group_code) {
            const lookupId = groupsMap.get(row.group_code.toLowerCase()) || groupsMap.get(row.group_code);
            if (lookupId) {
              finalGroupId = lookupId;
            }
          }

          const specifications: Record<string, any> = {};
          if (row.thickness) specifications.thickness = row.thickness;
          if (row.width) specifications.width = row.width;
          if (row.length) specifications.length = row.length;
          if (row.color) specifications.color = row.color;
          if (row.material_type) specifications.material_type = row.material_type;

          // INSERT new material with ON CONFLICT on internal_code
          await client.query(`
            INSERT INTO warehouse.materials 
              (group_id, name, internal_code, supplier_code, description, unit_of_measure, price, specifications, display_order, is_active, updated_by)
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, true, $10)
            ON CONFLICT (internal_code) DO UPDATE SET
              group_id = COALESCE(EXCLUDED.group_id, warehouse.materials.group_id),
              name = EXCLUDED.name,
              supplier_code = EXCLUDED.supplier_code,
              description = EXCLUDED.description,
              unit_of_measure = EXCLUDED.unit_of_measure,
              price = COALESCE(EXCLUDED.price, warehouse.materials.price),
              specifications = EXCLUDED.specifications,
              display_order = EXCLUDED.display_order,
              is_active = true,
              updated_at = CURRENT_TIMESTAMP
          `, [
            finalGroupId,
            name,
            internalCode,
            supplierCode,
            row.description?.toString().trim() || null,
            row.unit_of_measure?.toString().trim() || "szt",
            priceValue,
            Object.keys(specifications).length > 0 ? JSON.stringify(specifications) : null,
            row.display_order ? parseInt(row.display_order) : 0,
            username
          ]);

          results.success++;

        } catch (error: any) {
          console.error(`Error importing row ${rowNumber}:`, error);
          results.errors.push({
            row: rowNumber,
            error: error instanceof Error ? error.message : "Unknown error",
            data: row
          });
          results.failed++;
        }
      }

      await client.query("COMMIT");

      res.json({
        message: results.failed > 0 
          ? `Import zakończony: ${results.success} sukces, ${results.failed} błędów`
          : `Import zakończony pomyślnie: ${results.success} materiałów`,
        ...results
      });
    } catch (error) {
      await client.query("ROLLBACK");
      console.error("Error importing with mapping:", error);
      res.status(500).json({ error: "Failed to import" });
    } finally {
      client.release();
    }
  });
  

  // POST /api/warehouse/stock-panels/import-with-mapping - Import formatek do stock_panels
  app.post("/api/warehouse/stock-panels/import-with-mapping", isAuthenticated, async (req, res) => {
    try {
      const { rows, mapping, groupId: providedGroupId } = req.body;
      
      if (!rows || !Array.isArray(rows) || rows.length === 0) {
        return res.status(400).json({ error: "No data to import" });
      }

      // Get default group for formatki category or use provided groupId
      let defaultGroupId: number | null = providedGroupId || null;
      if (!defaultGroupId) {
        const groupResult = await pool.query(
          `SELECT id FROM warehouse.material_groups WHERE category = 'formatki' LIMIT 1`
        );
        if (groupResult.rows.length > 0) {
          defaultGroupId = groupResult.rows[0].id;
        }
      }

      // Get all groups for lookup if group_code is in mapping
      const groupsResult = await pool.query(
        `SELECT id, code, name FROM warehouse.material_groups WHERE category = 'formatki'`
      );
      const groupsMap = new Map<string, number>();
      groupsResult.rows.forEach((g: any) => {
        groupsMap.set(g.code, g.id);
        groupsMap.set(g.name.toLowerCase(), g.id);
        groupsMap.set(g.name, g.id);
      });

      const results = {
        success: 0,
        failed: 0,
        errors: [] as Array<{ row: number; error: string; data?: any }>
      };

      for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const rowNumber = i + 2;

        try {
          const name = row.name?.toString().trim();
          const internalCode = row.internal_code?.toString().trim();
          
          if (!name && !internalCode) {
            results.errors.push({
              row: rowNumber,
              error: "Nazwa lub kod wewnętrzny jest wymagany",
              data: row
            });
            results.failed++;
            continue;
          }

          // Determine group
          let finalGroupId = defaultGroupId;
          if (row.group_code) {
            const lookupId = groupsMap.get(row.group_code.toLowerCase()) || groupsMap.get(row.group_code);
            if (lookupId) {
              finalGroupId = lookupId;
            }
          }

          // Parse dimensions
          const length = row.length ? parseFloat(row.length) : null;
          const width = row.width ? parseFloat(row.width) : null;
          const thickness = row.thickness ? parseFloat(row.thickness) : 18;
          
          // Parse color - use mapped field
          let colorCode = row.color?.toString().trim() || null;
          
          // Parse quantity
          const quantity = row.quantity ? parseInt(row.quantity) : 0;
          
          // Parse price
          const price = row.price ? parseFloat(row.price) : null;

          // Generate name if not provided
          const generatedName = name || `${length}x${width}-${colorCode || 'SUROWA'}`;

          // Use internal_code as original_symbol (oryginalny symbol z Nexo)
          const originalSymbol = internalCode || null;
          
          // board_code (kod płyty) jest osobnym polem - może być z mapowania lub null
          const boardCode = row.board_code?.toString().trim() || null;
          
          // edging_code (kod obrzeża) - może być z mapowania lub null  
          const edgingCode = row.edging_code?.toString().trim() || null;

          // Check if record with same original_symbol exists - update it, otherwise insert
          if (originalSymbol) {
            const existingResult = await pool.query(
              `SELECT id FROM warehouse.stock_panels WHERE original_symbol = $1 LIMIT 1`,
              [originalSymbol]
            );
            
            if (existingResult.rows.length > 0) {
              // Update existing record
              await pool.query(`
                UPDATE warehouse.stock_panels SET
                  generated_name = $1,
                  length = COALESCE($2, length),
                  width = COALESCE($3, width),
                  thickness = COALESCE($4, thickness),
                  color_code = COALESCE($5, color_code),
                  quantity = $6,
                  price = $7,
                  group_id = COALESCE($8, group_id),
                  board_code = COALESCE($9, board_code),
                  edging_code = COALESCE($10, edging_code),
                  is_active = true,
                  updated_at = CURRENT_TIMESTAMP
                WHERE id = $11
              `, [
                generatedName,
                length,
                width,
                thickness,
                colorCode,
                quantity,
                finalGroupId,
                boardCode,
                edgingCode,
                existingResult.rows[0].id
              ]);

          results.success++;
            } else {
              // Insert new record
              await pool.query(`
                INSERT INTO warehouse.stock_panels 
                  (generated_name, length, width, thickness, color_code, original_symbol, board_code, edging_code, quantity, price, group_id, is_active, source, source_reference)
                VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, true, 'import', 'CSV/Excel import')
              `, [
                generatedName,
                length,
                width,
                thickness,
                colorCode,
                originalSymbol,
                boardCode,
                edgingCode,
                quantity,
                finalGroupId
              ]);

          results.success++;
            }
          } else {
            // No original_symbol - just insert without matching
            await pool.query(`
              INSERT INTO warehouse.stock_panels 
                (generated_name, length, width, thickness, color_code, board_code, edging_code, quantity, price, group_id, is_active, source, source_reference)
              VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, true, 'import', 'CSV/Excel import')
            `, [
              generatedName,
              length,
              width,
              thickness,
              colorCode,
              boardCode,
              edgingCode,
              quantity,
              finalGroupId
            ]);

          results.success++;
          }

          results.success++;
        } catch (error: any) {
          console.error(`Error importing stock panel row ${rowNumber}:`, error);
          results.errors.push({
            row: rowNumber,
            error: error instanceof Error ? error.message : "Unknown error",
            data: row
          });
          results.failed++;
        }
      }

      res.json({
        message: results.failed > 0 
          ? `Import zakonczony: ${results.success} sukces, ${results.failed} bledow`
          : `Pomyslnie zaimportowano ${results.success} rekordow.`,
        ...results
      });
    } catch (error) {
      console.error("Error importing stock panels with mapping:", error);
      res.status(500).json({ error: "Failed to import stock panels" });
    }
  });
  // POST /api/warehouse/packed-products/import-with-mapping - Import produktow spakowanych
  app.post("/api/warehouse/packed-products/import-with-mapping", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { rows, mapping, groupId: providedGroupId } = req.body;
      
      if (!rows || !Array.isArray(rows) || rows.length === 0) {
        return res.status(400).json({ error: "No data to import" });
      }

      // Get default group for produkty-spakowane category or use provided groupId
      let defaultGroupId: number | null = providedGroupId || null;
      if (!defaultGroupId) {
        const groupResult = await pool.query(
          `SELECT id FROM warehouse.material_groups WHERE category = 'produkty-spakowane' LIMIT 1`
        );
        if (groupResult.rows.length > 0) {
          defaultGroupId = groupResult.rows[0].id;
        }
      }

      // Get all groups for lookup if group_code is in mapping
      const groupsResult = await pool.query(
        `SELECT id, code, name FROM warehouse.material_groups WHERE category = 'produkty-spakowane'`
      );
      const groupsMap = new Map<string, number>();
      groupsResult.rows.forEach((g: any) => {
        groupsMap.set(g.code, g.id);
        groupsMap.set(g.name.toLowerCase(), g.id);
        groupsMap.set(g.name, g.id);
      });

      const results = {
        success: 0,
        failed: 0,
        errors: [] as Array<{ row: number; error: string; data?: any }>
      };

      await client.query("BEGIN");

      for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const rowNumber = i + 2;

        try {
          const productName = row.product_name?.toString().trim() || row.name?.toString().trim();
          const productSku = row.product_sku?.toString().trim() || row.internal_code?.toString().trim() || row.sku?.toString().trim();
          
          if (!productName && !productSku) {
            results.errors.push({
              row: rowNumber,
              error: "Nazwa produktu lub SKU jest wymagany",
              data: row
            });
            results.failed++;
            continue;
          }

          // Determine group
          let finalGroupId = defaultGroupId;
          if (row.group_code) {
            const lookupId = groupsMap.get(row.group_code.toLowerCase()) || groupsMap.get(row.group_code);
            if (lookupId) {
              finalGroupId = lookupId;
            }
          }

          // Parse quantity
          const quantity = row.quantity ? parseInt(row.quantity) : 0;
          
          // Product type
          const productType = row.product_type?.toString().trim() || 'product';

          // Try to find catalog_product_id by SKU
          let catalogProductId: number | null = null;
          if (productSku) {
            const catalogResult = await client.query(
              `SELECT id FROM catalog.products WHERE sku = $1 LIMIT 1`,
              [productSku]
            );
            if (catalogResult.rows.length > 0) {
              catalogProductId = catalogResult.rows[0].id;
            }
          }

          await client.query(`
            INSERT INTO warehouse.packed_products 
              (product_name, product_sku, product_type, quantity, group_id, catalog_product_id, is_active, notes)
            VALUES ($1, $2, $3, $4, $5, $6, true, $7)
            ON CONFLICT (product_sku) WHERE product_sku IS NOT NULL DO UPDATE SET
              product_name = COALESCE(EXCLUDED.product_name, warehouse.packed_products.product_name),
              product_type = COALESCE(EXCLUDED.product_type, warehouse.packed_products.product_type),
              quantity = EXCLUDED.quantity,
              group_id = COALESCE(EXCLUDED.group_id, warehouse.packed_products.group_id),
              catalog_product_id = COALESCE(EXCLUDED.catalog_product_id, warehouse.packed_products.catalog_product_id),
              is_active = true,
              updated_at = CURRENT_TIMESTAMP
          `, [
            productName,
            productSku,
            productType,
            quantity,
            finalGroupId,
            catalogProductId,
            row.notes?.toString().trim() || null
          ]);

          results.success++;

          results.success++;
        } catch (error: any) {
          console.error(`Error importing packed product row ${rowNumber}:`, error);
          results.errors.push({
            row: rowNumber,
            error: error instanceof Error ? error.message : "Unknown error",
            data: row
          });
          results.failed++;
        }
      }

      await client.query("COMMIT");

      res.json({
        message: results.failed > 0 
          ? `Import zakonczony: ${results.success} sukces, ${results.failed} bledow`
          : `Pomyslnie zaimportowano ${results.success} rekordow.`,
        ...results
      });
    } catch (error) {
      await client.query("ROLLBACK");
      console.error("Error importing packed products with mapping:", error);
      res.status(500).json({ error: "Failed to import packed products" });
    } finally {
      client.release();
    }
  });

  // GET /api/warehouse/material-accessory-links - Lista powiązań materiałów z akcesoriami
  app.get("/api/warehouse/material-accessory-links", isAuthenticated, async (req, res) => {
    try {
      const { materialId, accessoryId } = req.query;
      
      let query = `
        SELECT 
          mal.id,
          mal.material_id AS "materialId",
          mal.accessory_id AS "accessoryId",
          mal.quantity,
          mal.notes,
          mal.created_at AS "createdAt",
          m.name AS "materialName",
          m.internal_code AS "materialCode",
          a.name AS "accessoryName",
          a.code AS "accessoryCode"
        FROM warehouse.material_accessory_links mal
        LEFT JOIN warehouse.materials m ON mal.material_id = m.id
        LEFT JOIN catalog.accessories a ON mal.accessory_id = a.id
        WHERE 1=1
      `;
      
      const params: any[] = [];
      
      if (materialId) {
        params.push(materialId);
        query += ` AND mal.material_id = $${params.length}`;
      }
      
      if (accessoryId) {
        params.push(accessoryId);
        query += ` AND mal.accessory_id = $${params.length}`;
      }
      
      query += ` ORDER BY mal.created_at DESC`;
      
      const result = await pool.query(query, params);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching material-accessory links:", error);
      res.status(500).json({ error: "Failed to fetch material-accessory links" });
    }
  });
  
  // POST /api/warehouse/material-accessory-links - Utwórz powiązanie materiału z akcesorium
  app.post("/api/warehouse/material-accessory-links", isAuthenticated, async (req, res) => {
    try {
      const { materialId, accessoryId, quantity, notes } = req.body;
      
      if (!materialId || !accessoryId) {
        return res.status(400).json({ error: "Material ID and accessory ID are required" });
      }
      
      const result = await pool.query(`
        INSERT INTO warehouse.material_accessory_links (material_id, accessory_id, quantity, notes)
        VALUES ($1, $2, $3, $4)
        RETURNING 
          id,
          material_id AS "materialId",
          accessory_id AS "accessoryId",
          quantity,
          notes,
          created_at AS "createdAt"
      `, [materialId, accessoryId, quantity || 1, notes || null]);
      
      res.json(result.rows[0]);
    } catch (error: any) {
      if (error.code === '23505') {
        return res.status(400).json({ error: "This link already exists" });
      }
      console.error("❌ Error creating material-accessory link:", error);
      res.status(500).json({ error: "Failed to create link" });
    }
  });
  
  // DELETE /api/warehouse/material-accessory-links/:id - Usuń powiązanie
  app.delete("/api/warehouse/material-accessory-links/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      await pool.query(
        'DELETE FROM warehouse.material_accessory_links WHERE id = $1',
        [id]
      );
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting material-accessory link:", error);
      res.status(500).json({ error: "Failed to delete link" });
    }
  });

  // ===== WAREHOUSE INVENTORY COUNTS (SPISY INWENTARYZACYJNE) =====
  
  // POST /api/warehouse/inventory-counts - Utwórz nowy spis inwentaryzacyjny z materiałami, formatkami, opakowaniami lub produktami spakowanymi
  app.post("/api/warehouse/inventory-counts", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { name, materialIds, panelIds, packagingMaterialIds, packedProductIds, notes } = req.body;
      const username = (req.user as any)?.username || 'system';
      
      // Validate - must have either materialIds, panelIds, packagingMaterialIds or packedProductIds (not multiple, not none)
      const hasMaterials = materialIds && Array.isArray(materialIds) && materialIds.length > 0;
      const hasPanels = panelIds && Array.isArray(panelIds) && panelIds.length > 0;
      const hasPackaging = packagingMaterialIds && Array.isArray(packagingMaterialIds) && packagingMaterialIds.length > 0;
      const hasPackedProducts = packedProductIds && Array.isArray(packedProductIds) && packedProductIds.length > 0;
      
      if (!name) {
        return res.status(400).json({ error: "Name is required" });
      }
      
      if (!hasMaterials && !hasPanels && !hasPackaging && !hasPackedProducts) {
        return res.status(400).json({ error: "Either materialIds, panelIds, packagingMaterialIds or packedProductIds are required" });
      }
      
      const typesCount = [hasMaterials, hasPanels, hasPackaging, hasPackedProducts].filter(Boolean).length;
      if (typesCount > 1) {
        return res.status(400).json({ error: "Cannot mix different types (materials, panels, packaging, packed products) in the same inventory count" });
      }
      
      await client.query('BEGIN');
      
      // Create inventory count
      const countResult = await client.query(`
        INSERT INTO warehouse.inventory_counts (name, status, notes, created_by)
        VALUES ($1, 'draft', $2, $3)
        RETURNING 
          id,
          name,
          status,
          notes,
          created_by AS "createdBy",
          created_at AS "createdAt",
          finalized_by AS "finalizedBy",
          finalized_at AS "finalizedAt"
      `, [name, notes || null, username]);
      
      const inventoryCount = countResult.rows[0];
      
      // Add items to the count (with current system quantities)
      const items = [];
      
      if (hasMaterials) {
        // Process materials
        for (const materialId of materialIds) {
          const materialResult = await client.query(
            'SELECT id, quantity FROM warehouse.materials WHERE id = $1',
            [materialId]
          );
          
          if (materialResult.rows.length === 0) {
            await client.query('ROLLBACK');
            return res.status(404).json({ error: `Material with ID ${materialId} not found` });
          }
          
          const systemQuantity = safeParseNumber(materialResult.rows[0].quantity);
          
          const itemResult = await client.query(`
            INSERT INTO warehouse.inventory_count_items (inventory_count_id, material_id, system_quantity)
            VALUES ($1, $2, $3)
            RETURNING 
              id,
              inventory_count_id AS "inventoryCountId",
              material_id AS "materialId",
              stock_panel_id AS "stockPanelId",
              system_quantity AS "systemQuantity",
              counted_quantity AS "countedQuantity",
              difference,
              notes,
              created_at AS "createdAt",
              updated_at AS "updatedAt"
          `, [inventoryCount.id, materialId, systemQuantity]);
          
          items.push(itemResult.rows[0]);
        }
      } else if (hasPanels) {
        // Process stock panels (formatki)
        for (const panelId of panelIds) {
          const panelResult = await client.query(
            'SELECT id, quantity FROM warehouse.stock_panels WHERE id = $1',
            [panelId]
          );
          
          if (panelResult.rows.length === 0) {
            await client.query('ROLLBACK');
            return res.status(404).json({ error: `Stock panel with ID ${panelId} not found` });
          }
          
          const systemQuantity = Math.floor(safeParseNumber(panelResult.rows[0].quantity));
          
          const itemResult = await client.query(`
            INSERT INTO warehouse.inventory_count_items (inventory_count_id, stock_panel_id, system_quantity)
            VALUES ($1, $2, $3)
            RETURNING 
              id,
              inventory_count_id AS "inventoryCountId",
              material_id AS "materialId",
              stock_panel_id AS "stockPanelId",
              packaging_material_id AS "packagingMaterialId",
              system_quantity AS "systemQuantity",
              counted_quantity AS "countedQuantity",
              difference,
              notes,
              created_at AS "createdAt",
              updated_at AS "updatedAt"
          `, [inventoryCount.id, panelId, systemQuantity]);
          
          items.push(itemResult.rows[0]);
        }
      } else if (hasPackaging) {
        // Process packaging materials (opakowania)
        for (const packagingId of packagingMaterialIds) {
          const packagingResult = await client.query(
            'SELECT id, quantity FROM warehouse.packaging_materials WHERE id = $1',
            [packagingId]
          );
          
          if (packagingResult.rows.length === 0) {
            await client.query('ROLLBACK');
            return res.status(404).json({ error: `Packaging material with ID ${packagingId} not found` });
          }
          
          const systemQuantity = safeParseNumber(packagingResult.rows[0].quantity);
          
          const itemResult = await client.query(`
            INSERT INTO warehouse.inventory_count_items (inventory_count_id, packaging_material_id, system_quantity)
            VALUES ($1, $2, $3)
            RETURNING 
              id,
              inventory_count_id AS "inventoryCountId",
              material_id AS "materialId",
              stock_panel_id AS "stockPanelId",
              packaging_material_id AS "packagingMaterialId",
              system_quantity AS "systemQuantity",
              counted_quantity AS "countedQuantity",
              difference,
              notes,
              created_at AS "createdAt",
              updated_at AS "updatedAt"
          `, [inventoryCount.id, packagingId, systemQuantity]);
          
          items.push(itemResult.rows[0]);
        }
      } else {
        // Process packed products (produkty spakowane)
        for (const packedProductId of packedProductIds) {
          const productResult = await client.query(
            'SELECT id, quantity FROM warehouse.packed_products WHERE id = $1',
            [packedProductId]
          );
          
          if (productResult.rows.length === 0) {
            await client.query('ROLLBACK');
            return res.status(404).json({ error: `Packed product with ID ${packedProductId} not found` });
          }
          
          const systemQuantity = Math.floor(safeParseNumber(productResult.rows[0].quantity));
          
          const itemResult = await client.query(`
            INSERT INTO warehouse.inventory_count_items (inventory_count_id, packed_product_id, system_quantity)
            VALUES ($1, $2, $3)
            RETURNING 
              id,
              inventory_count_id AS "inventoryCountId",
              material_id AS "materialId",
              stock_panel_id AS "stockPanelId",
              packaging_material_id AS "packagingMaterialId",
              packed_product_id AS "packedProductId",
              system_quantity AS "systemQuantity",
              counted_quantity AS "countedQuantity",
              difference,
              notes,
              created_at AS "createdAt",
              updated_at AS "updatedAt"
          `, [inventoryCount.id, packedProductId, systemQuantity]);
          
          items.push(itemResult.rows[0]);
        }
      }
      
      await client.query('COMMIT');
      
      res.json({
        inventoryCount,
        items
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error creating inventory count:", error);
      res.status(500).json({ error: "Failed to create inventory count" });
    } finally {
      client.release();
    }
  });

  // POST /api/warehouse/inventory-counts/:id/add-items - Dodaj materiały do istniejącego spisu
  app.post("/api/warehouse/inventory-counts/:id/add-items", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { id } = req.params;
      const { materialIds, panelIds, packagingMaterialIds } = req.body;
      
      // Validate - must have exactly one type
      const hasMaterials = materialIds && Array.isArray(materialIds) && materialIds.length > 0;
      const hasPanels = panelIds && Array.isArray(panelIds) && panelIds.length > 0;
      const hasPackaging = packagingMaterialIds && Array.isArray(packagingMaterialIds) && packagingMaterialIds.length > 0;
      
      if (!hasMaterials && !hasPanels && !hasPackaging) {
        return res.status(400).json({ error: "Either materialIds, panelIds or packagingMaterialIds are required" });
      }
      
      const typesCount = [hasMaterials, hasPanels, hasPackaging].filter(Boolean).length;
      if (typesCount > 1) {
        return res.status(400).json({ error: "Cannot mix different types in the same request" });
      }
      
      await client.query('BEGIN');
      
      // Check if inventory count exists and is in draft status
      const countCheck = await client.query(
        'SELECT id, status FROM warehouse.inventory_counts WHERE id = $1',
        [id]
      );
      
      if (countCheck.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      if (countCheck.rows[0].status !== 'draft') {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Cannot add items to finalized inventory count" });
      }
      
      const addedItems = [];
      
      if (hasMaterials) {
        for (const materialId of materialIds) {
          // Check if already exists
          const existsCheck = await client.query(
            'SELECT id FROM warehouse.inventory_count_items WHERE inventory_count_id = $1 AND material_id = $2',
            [id, materialId]
          );
          
          if (existsCheck.rows.length > 0) {
            continue; // Skip if already in count
          }
          
          const materialResult = await client.query(
            'SELECT id, quantity FROM warehouse.materials WHERE id = $1',
            [materialId]
          );
          
          if (materialResult.rows.length === 0) {
            continue; // Skip if material not found
          }
          
          const systemQuantity = safeParseNumber(materialResult.rows[0].quantity);
          
          const itemResult = await client.query(`
            INSERT INTO warehouse.inventory_count_items (inventory_count_id, material_id, system_quantity)
            VALUES ($1, $2, $3)
            RETURNING id
          `, [id, materialId, systemQuantity]);
          
          addedItems.push(itemResult.rows[0]);
        }
      } else if (hasPanels) {
        for (const panelId of panelIds) {
          const existsCheck = await client.query(
            'SELECT id FROM warehouse.inventory_count_items WHERE inventory_count_id = $1 AND stock_panel_id = $2',
            [id, panelId]
          );
          
          if (existsCheck.rows.length > 0) {
            continue;
          }
          
          const panelResult = await client.query(
            'SELECT id, quantity FROM warehouse.stock_panels WHERE id = $1',
            [panelId]
          );
          
          if (panelResult.rows.length === 0) {
            continue;
          }
          
          const systemQuantity = Math.floor(safeParseNumber(panelResult.rows[0].quantity));
          
          const itemResult = await client.query(`
            INSERT INTO warehouse.inventory_count_items (inventory_count_id, stock_panel_id, system_quantity)
            VALUES ($1, $2, $3)
            RETURNING id
          `, [id, panelId, systemQuantity]);
          
          addedItems.push(itemResult.rows[0]);
        }
      } else if (hasPackaging) {
        for (const packagingId of packagingMaterialIds) {
          const existsCheck = await client.query(
            'SELECT id FROM warehouse.inventory_count_items WHERE inventory_count_id = $1 AND packaging_material_id = $2',
            [id, packagingId]
          );
          
          if (existsCheck.rows.length > 0) {
            continue;
          }
          
          const packagingResult = await client.query(
            'SELECT id, quantity FROM warehouse.packaging_materials WHERE id = $1',
            [packagingId]
          );
          
          if (packagingResult.rows.length === 0) {
            continue;
          }
          
          const systemQuantity = safeParseNumber(packagingResult.rows[0].quantity);
          
          const itemResult = await client.query(`
            INSERT INTO warehouse.inventory_count_items (inventory_count_id, packaging_material_id, system_quantity)
            VALUES ($1, $2, $3)
            RETURNING id
          `, [id, packagingId, systemQuantity]);
          
          addedItems.push(itemResult.rows[0]);
        }
      }
      
      await client.query('COMMIT');
      
      res.json({
        success: true,
        addedCount: addedItems.length
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error adding items to inventory count:", error);
      res.status(500).json({ error: "Failed to add items to inventory count" });
    } finally {
      client.release();
    }
  });

  // GET /api/warehouse/inventory-counts - Lista spisów inwentaryzacyjnych
  app.get("/api/warehouse/inventory-counts", isAuthenticated, async (req, res) => {
    try {
      const { status } = req.query;
      
      let query = `
        SELECT 
          ic.id,
          ic.name,
          ic.status,
          ic.notes,
          ic.created_by AS "createdBy",
          ic.created_at AS "createdAt",
          ic.finalized_by AS "finalizedBy",
          ic.finalized_at AS "finalizedAt",
          COUNT(ici.id) AS "itemCount",
          COUNT(ici.counted_quantity) AS "countedItemCount"
        FROM warehouse.inventory_counts ic
        LEFT JOIN warehouse.inventory_count_items ici ON ic.id = ici.inventory_count_id
      `;
      
      const params: any[] = [];
      
      if (status) {
        query += ' WHERE ic.status = $1';
        params.push(status);
      }
      
      query += ' GROUP BY ic.id ORDER BY ic.created_at DESC';
      
      const result = await pool.query(query, params);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching inventory counts:", error);
      res.status(500).json({ error: "Failed to fetch inventory counts" });
    }
  });

  // GET /api/warehouse/inventory-counts/:id/export/nexo - Eksport spisu do XML (Nexo)
  app.get("/api/warehouse/inventory-counts/:id/export/nexo", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Helper function to escape XML special characters
      const escapeXml = (text: string | null | undefined): string => {
        if (!text) return '';
        return String(text)
          .replace(/&/g, '&amp;')
          .replace(/</g, '&lt;')
          .replace(/>/g, '&gt;')
          .replace(/"/g, '&quot;')
          .replace(/'/g, '&apos;');
      };
      
      // Get inventory count with creator username
      const countResult = await pool.query(`
        SELECT 
          ic.id,
          ic.name,
          ic.status,
          ic.notes,
          ic.created_by AS "createdBy",
          ic.created_at AS "createdAt",
          ic.finalized_at AS "finalizedAt",
          u.username AS "creatorUsername"
        FROM warehouse.inventory_counts ic
        LEFT JOIN users u ON ic.created_by = CAST(u.id AS TEXT)
        WHERE ic.id = $1
      `, [id]);
      
      if (countResult.rows.length === 0) {
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      const inventoryCount = countResult.rows[0];
      
      // Only allow export of finalized counts
      if (inventoryCount.status !== 'finalized') {
        return res.status(400).json({ error: "Only finalized inventory counts can be exported" });
      }
      
      // Get items with details - using originalSymbol for stock_panels, internal_code for materials
      const itemsResult = await pool.query(`
        SELECT 
          ici.counted_quantity AS "countedQuantity",
          ici.system_quantity AS "systemQuantity",
          'material' AS "itemType",
          m.name AS "itemName",
          COALESCE(m.supplier_code, m.internal_code) AS "symbolAsortymentu",
          m.unit_of_measure AS "unitOfMeasure",
          COALESCE(m.price, 0) AS "price"
        FROM warehouse.inventory_count_items ici
        JOIN warehouse.materials m ON ici.material_id = m.id
        WHERE ici.inventory_count_id = $1 AND ici.material_id IS NOT NULL
        
        UNION ALL
        
        SELECT 
          ici.counted_quantity AS "countedQuantity",
          ici.system_quantity AS "systemQuantity",
          'stock_panel' AS "itemType",
          sp.generated_name AS "itemName",
          COALESCE(sp.original_symbol, CAST(sp.id AS TEXT)) AS "symbolAsortymentu",
          'szt' AS "unitOfMeasure",
          COALESCE(sp.price, 0) AS "price"
        FROM warehouse.inventory_count_items ici
        JOIN warehouse.stock_panels sp ON ici.stock_panel_id = sp.id
        WHERE ici.inventory_count_id = $1 AND ici.stock_panel_id IS NOT NULL
        
        UNION ALL
        
        SELECT 
          ici.counted_quantity AS "countedQuantity",
          ici.system_quantity AS "systemQuantity",
          'packaging_material' AS "itemType",
          pm.name AS "itemName",
          COALESCE(pm.supplier_code, CAST(pm.id AS TEXT)) AS "symbolAsortymentu",
          pm.unit AS "unitOfMeasure",
          0 AS "price"
        FROM warehouse.inventory_count_items ici
        JOIN warehouse.packaging_materials pm ON ici.packaging_material_id = pm.id
        WHERE ici.inventory_count_id = $1 AND ici.packaging_material_id IS NOT NULL
        
        ORDER BY "itemName"
      `, [id]);
      
      // Generate XML in Subiekt Nexo format
      const createdDate = new Date(inventoryCount.createdAt).toISOString().split('T')[0];
      const creatorName = (req.user as any)?.username || 'System Export';
      
      let xml = `<?xml version="1.0" encoding="utf-8"?>\n`;
      xml += `<SpisInwentaryzacyjny xmlns="http://schemas.insert.com.pl/2017/SpisInwentaryzacyjny.xsd">\n`;
      xml += `    <Naglowek>\n`;
      xml += `        <Wykonujacy>${escapeXml(creatorName)}</Wykonujacy>\n`;
      xml += `        <DataWykonania>${escapeXml(createdDate)}</DataWykonania>\n`;
      xml += `        <SymbolMagazynu>MAG</SymbolMagazynu>\n`;
      xml += `    </Naglowek>\n`;
      xml += `    <Pozycje>\n`;
      
      itemsResult.rows.forEach((item) => {
        const quantity = item.countedQuantity || item.systemQuantity || 0;
        const price = parseFloat(item.price || 0).toFixed(2);
        xml += `        <Pozycja>\n`;
        xml += `            <KodKreskowy></KodKreskowy>\n`;
        xml += `            <Ilosc>${parseFloat(quantity).toFixed(0)}</Ilosc>\n`;
        xml += `            <SymbolAsortymentu>${escapeXml(item.symbolAsortymentu)}</SymbolAsortymentu>\n`;
        xml += `            <Cena>${price}</Cena>\n`;
        xml += `        </Pozycja>\n`;
      });
      
      xml += `    </Pozycje>\n`;
      xml += `</SpisInwentaryzacyjny>`;
      
      // Set headers for file download
      const filename = `SpisInwentaryzacyjny_${String(inventoryCount.id).padStart(4, '0')}_${createdDate}.xml`;
      res.setHeader('Content-Type', 'application/xml; charset=utf-8');
      res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
      res.send(xml);
    } catch (error) {
      console.error("❌ Error exporting inventory count to Nexo XML:", error);
      res.status(500).json({ error: "Failed to export inventory count" });
    }
  });

  // GET /api/warehouse/inventory-counts/:id - Szczegóły spisu inwentaryzacyjnego
  app.get("/api/warehouse/inventory-counts/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Get inventory count
      const countResult = await pool.query(`
        SELECT 
          id,
          name,
          status,
          notes,
          created_by AS "createdBy",
          created_at AS "createdAt",
          finalized_by AS "finalizedBy",
          finalized_at AS "finalizedAt"
        FROM warehouse.inventory_counts
        WHERE id = $1
      `, [id]);
      
      if (countResult.rows.length === 0) {
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      // Get items with details from materials, stock panels, packaging materials, and packed products
      const itemsResult = await pool.query(`
        SELECT 
          ici.id,
          ici.inventory_count_id AS "inventoryCountId",
          ici.system_quantity AS "systemQuantity",
          ici.counted_quantity AS "countedQuantity",
          ici.difference,
          ici.notes,
          ici.created_at AS "createdAt",
          ici.updated_at AS "updatedAt",
          'material' AS "itemType",
          ici.material_id AS "itemId",
          m.name AS "itemName",
          m.internal_code AS "itemCode",
          m.unit_of_measure AS "unitOfMeasure",
          mg.name AS "groupName",
          mg.category AS "itemCategory",
          NULL AS "dimensions",
          NULL AS "colorCode",
          NULL AS "supplierCode",
          NULL AS "originalSymbol"
        FROM warehouse.inventory_count_items ici
        JOIN warehouse.materials m ON ici.material_id = m.id
        LEFT JOIN warehouse.material_groups mg ON m.group_id = mg.id
        WHERE ici.inventory_count_id = $1 AND ici.material_id IS NOT NULL
        
        UNION ALL
        
        SELECT 
          ici.id,
          ici.inventory_count_id AS "inventoryCountId",
          ici.system_quantity AS "systemQuantity",
          ici.counted_quantity AS "countedQuantity",
          ici.difference,
          ici.notes,
          ici.created_at AS "createdAt",
          ici.updated_at AS "updatedAt",
          'stock_panel' AS "itemType",
          ici.stock_panel_id AS "itemId",
          sp.generated_name AS "itemName",
          NULL AS "itemCode",
          'szt' AS "unitOfMeasure",
          'Formatki' AS "groupName",
          'formatki' AS "itemCategory",
          CONCAT(sp.length, 'x', sp.width, 'x', sp.thickness) AS "dimensions",
          sp.color_code AS "colorCode",
          NULL AS "supplierCode",
          sp.original_symbol AS "originalSymbol"
        FROM warehouse.inventory_count_items ici
        JOIN warehouse.stock_panels sp ON ici.stock_panel_id = sp.id
        WHERE ici.inventory_count_id = $1 AND ici.stock_panel_id IS NOT NULL
        
        UNION ALL
        
        SELECT 
          ici.id,
          ici.inventory_count_id AS "inventoryCountId",
          ici.system_quantity AS "systemQuantity",
          ici.counted_quantity AS "countedQuantity",
          ici.difference,
          ici.notes,
          ici.created_at AS "createdAt",
          ici.updated_at AS "updatedAt",
          'packaging_material' AS "itemType",
          ici.packaging_material_id AS "itemId",
          pm.name AS "itemName",
          pm.supplier_code AS "itemCode",
          pm.unit AS "unitOfMeasure",
          'Opakowania' AS "groupName",
          'opakowania' AS "itemCategory",
          NULL AS "dimensions",
          NULL AS "colorCode",
          pNULL AS "supplierCode",
          NULL AS "originalSymbol"
        FROM warehouse.inventory_count_items ici
        JOIN warehouse.packaging_materials pm ON ici.packaging_material_id = pm.id
        WHERE ici.inventory_count_id = $1 AND ici.packaging_material_id IS NOT NULL
        
        UNION ALL
        
        SELECT 
          ici.id,
          ici.inventory_count_id AS "inventoryCountId",
          ici.system_quantity AS "systemQuantity",
          ici.counted_quantity AS "countedQuantity",
          ici.difference,
          ici.notes,
          ici.created_at AS "createdAt",
          ici.updated_at AS "updatedAt",
          'packed_product' AS "itemType",
          ici.packed_product_id AS "itemId",
          pp.product_name AS "itemName",
          pp.product_sku AS "itemCode",
          'szt' AS "unitOfMeasure",
          'Produkty spakowane' AS "groupName",
          'produkty-spakowane' AS "itemCategory",
          NULL AS "dimensions",
          NULL AS "colorCode",
          NULL AS "supplierCode",
          NULL AS "originalSymbol"
        FROM warehouse.inventory_count_items ici
        JOIN warehouse.packed_products pp ON ici.packed_product_id = pp.id
        WHERE ici.inventory_count_id = $1 AND ici.packed_product_id IS NOT NULL
        
        ORDER BY "groupName", "itemName"
      `, [id]);
      
      res.json({
        inventoryCount: countResult.rows[0],
        items: itemsResult.rows
      });
    } catch (error) {
      console.error("❌ Error fetching inventory count details:", error);
      res.status(500).json({ error: "Failed to fetch inventory count details" });
    }
  });

  // PATCH /api/warehouse/inventory-counts/:countId/items/:itemId - Aktualizuj ilość spisaną dla pozycji
  app.patch("/api/warehouse/inventory-counts/:countId/items/:itemId", isAuthenticated, async (req, res) => {
    try {
      const { countId, itemId } = req.params;
      const { countedQuantity, notes } = req.body;
      
      if (countedQuantity === undefined || countedQuantity === null) {
        return res.status(400).json({ error: "Counted quantity is required" });
      }
      
      // Validate quantity is non-negative number
      const counted = safeParseNumber(countedQuantity, NaN);
      if (isNaN(counted) || counted < 0) {
        return res.status(400).json({ error: "Counted quantity must be a non-negative number" });
      }
      
      // Check if count is still draft
      const countCheck = await pool.query(
        'SELECT status FROM warehouse.inventory_counts WHERE id = $1',
        [countId]
      );
      
      if (countCheck.rows.length === 0) {
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      if (countCheck.rows[0].status !== 'draft') {
        return res.status(400).json({ error: "Cannot update finalized inventory count" });
      }
      
      // Get item with system quantity
      const itemCheck = await pool.query(
        'SELECT system_quantity FROM warehouse.inventory_count_items WHERE id = $1 AND inventory_count_id = $2',
        [itemId, countId]
      );
      
      if (itemCheck.rows.length === 0) {
        return res.status(404).json({ error: "Inventory count item not found" });
      }
      
      const systemQuantity = safeParseNumber(itemCheck.rows[0].system_quantity);
      const difference = counted - systemQuantity;
      
      // Update item
      const result = await pool.query(`
        UPDATE warehouse.inventory_count_items
        SET 
          counted_quantity = $1,
          difference = $2,
          notes = $3,
          updated_at = CURRENT_TIMESTAMP
        WHERE id = $4 AND inventory_count_id = $5
        RETURNING 
          id,
          inventory_count_id AS "inventoryCountId",
          material_id AS "materialId",
          system_quantity AS "systemQuantity",
          counted_quantity AS "countedQuantity",
          difference,
          notes,
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [counted, difference, notes || null, itemId, countId]);
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating inventory count item:", error);
      res.status(500).json({ error: "Failed to update inventory count item" });
    }
  });

  // DELETE /api/warehouse/inventory-counts/:countId/items/:itemId - Usuń pozycję ze spisu
  app.delete("/api/warehouse/inventory-counts/:countId/items/:itemId", isAuthenticated, async (req, res) => {
    try {
      const { countId, itemId } = req.params;
      
      // Check if count is still draft
      const countCheck = await pool.query(
        'SELECT status FROM warehouse.inventory_counts WHERE id = $1',
        [countId]
      );
      
      if (countCheck.rows.length === 0) {
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      if (countCheck.rows[0].status !== 'draft') {
        return res.status(400).json({ error: "Cannot delete items from finalized inventory count" });
      }
      
      // Delete the item
      const result = await pool.query(
        'DELETE FROM warehouse.inventory_count_items WHERE id = $1 AND inventory_count_id = $2 RETURNING id',
        [itemId, countId]
      );
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Inventory count item not found" });
      }
      
      res.json({ success: true, message: "Item deleted successfully" });
    } catch (error) {
      console.error("❌ Error deleting inventory count item:", error);
      res.status(500).json({ error: "Failed to delete inventory count item" });
    }
  });

  // POST /api/warehouse/inventory-counts/:id/finalize - Przelicz magazyn (finalizacja)
  app.post("/api/warehouse/inventory-counts/:id/finalize", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { id } = req.params;
      const username = (req.user as any)?.username || 'system';
      
      await client.query('BEGIN');
      
      // Check if count is draft
      const countCheck = await client.query(
        'SELECT status, name FROM warehouse.inventory_counts WHERE id = $1',
        [id]
      );
      
      if (countCheck.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      if (countCheck.rows[0].status !== 'draft') {
        await client.query('ROLLBACK');
        return res.status(400).json({ error: "Inventory count is already finalized" });
      }
      
      const countName = countCheck.rows[0].name;
      
      // Get all items (materials, stock panels, packaging materials, packed products) with counted quantities
      const itemsResult = await client.query(`
        SELECT 
          ici.id,
          ici.material_id,
          ici.stock_panel_id,
          ici.packaging_material_id,
          ici.packed_product_id,
          ici.system_quantity,
          ici.counted_quantity,
          ici.difference,
          COALESCE(m.name, sp.generated_name, pm.name, pp.product_name) AS item_name,
          CASE 
            WHEN ici.material_id IS NOT NULL THEN m.quantity::text
            WHEN ici.stock_panel_id IS NOT NULL THEN sp.quantity::text
            WHEN ici.packaging_material_id IS NOT NULL THEN pm.quantity::text
            WHEN ici.packed_product_id IS NOT NULL THEN pp.quantity::text
          END AS current_quantity
        FROM warehouse.inventory_count_items ici
        LEFT JOIN warehouse.materials m ON ici.material_id = m.id
        LEFT JOIN warehouse.stock_panels sp ON ici.stock_panel_id = sp.id
        LEFT JOIN warehouse.packaging_materials pm ON ici.packaging_material_id = pm.id
        LEFT JOIN warehouse.packed_products pp ON ici.packed_product_id = pp.id
        WHERE ici.inventory_count_id = $1
      `, [id]);
      
      const items = itemsResult.rows;
      
      // Check if all items have counted quantities
      const uncountedItems = items.filter(item => item.counted_quantity === null);
      if (uncountedItems.length > 0) {
        await client.query('ROLLBACK');
        return res.status(400).json({ 
          error: `${uncountedItems.length} items have not been counted yet`,
          uncountedItems: uncountedItems.map(item => ({
            id: item.id,
            name: item.item_name
          }))
        });
      }
      
      // Update quantities and create inventory history entries
      let updatedMaterials = 0;
      let updatedStockPanels = 0;
      let updatedPackagingMaterials = 0;
      let updatedPackedProducts = 0;
      
      for (const item of items) {
        const difference = safeParseNumber(item.difference, 0);
        
        if (!isNaN(difference) && difference !== 0) {
          const currentQty = safeParseNumber(item.current_quantity, 0);
          const newQty = currentQty + difference;
          
          // Update material quantity
          if (item.material_id) {
            await client.query(
              'UPDATE warehouse.materials SET quantity = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
              [newQty, item.material_id]
            );
            
            // Create inventory history entry
            await client.query(`
              INSERT INTO warehouse.inventory_history (
                material_id, operation_type, quantity_change, quantity_before, quantity_after,
                notes, performed_by
              )
              VALUES ($1, 'inventory_count', $2, $3, $4, $5, $6)
            `, [
              item.material_id,
              difference,
              currentQty,
              newQty,
              `Spis inwentaryzacyjny: ${countName}`,
              username
            ]);

          results.success++;
            updatedMaterials++;
          }
          
          // Update stock panel quantity
          if (item.stock_panel_id) {
            await client.query(
              'UPDATE warehouse.stock_panels SET quantity = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
              [newQty, item.stock_panel_id]
            );
            updatedStockPanels++;
          }
          
          // Update packaging material quantity
          if (item.packaging_material_id) {
            await client.query(
              'UPDATE warehouse.packaging_materials SET quantity = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
              [newQty, item.packaging_material_id]
            );
            updatedPackagingMaterials++;
          }
          
          // Update packed product - create or remove items in packed_product_items
          if (item.packed_product_id) {
            if (difference > 0) {
              // Add new items (positive difference = counted more than system)
              const productResult = await client.query(
                'SELECT catalog_product_id, catalog_set_id, product_sku FROM warehouse.packed_products WHERE id = $1',
                [item.packed_product_id]
              );
              
              if (productResult.rows.length > 0) {
                const product = productResult.rows[0];
                const today = new Date().toISOString().slice(0, 10).replace(/-/g, '');
                const productIdStr = product.catalog_product_id?.toString() || '0';
                
                // Get next sequence number
                const seqResult = await client.query(`
                  SELECT COALESCE(MAX(CAST(SPLIT_PART(serial_number, '-', 3) AS INTEGER)), 0) + 1 as next_seq
                  FROM warehouse.packed_product_items
                  WHERE serial_number LIKE $1 || '-' || $2 || '-%'
                `, [productIdStr, today]);
                
                let nextSeq = seqResult.rows[0].next_seq;
                
                for (let i = 0; i < difference; i++) {
                  const serialNumber = `${productIdStr}-${today}-${String(nextSeq++).padStart(5, '0')}`;
                  await client.query(`
                    INSERT INTO warehouse.packed_product_items (
                      packed_product_id, catalog_product_id, catalog_set_id, product_sku,
                      serial_number, status, source_type, source_id,
                      packed_at, created_at, updated_at
                    ) VALUES ($1, $2, $3, $4, $5, 'available', 'inventory_count', $6, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)
                  `, [
                    item.packed_product_id,
                    product.catalog_product_id,
                    product.catalog_set_id,
                    product.product_sku,
                    serialNumber,
                    parseInt(id, 10)
                  ]);

          results.success++;
                }
              }
            } else if (difference < 0) {
              // Mark excess items as scrapped (negative difference = counted less than system)
              // Mark available items first (FIFO), keeping history
              const itemsToRemove = Math.abs(difference);
              
              // Check if we have enough available items
              const availableCountResult = await client.query(`
                SELECT COUNT(*)::int as count FROM warehouse.packed_product_items
                WHERE packed_product_id = $1 AND status = 'available'
              `, [item.packed_product_id]);
              
              const availableCount = availableCountResult.rows[0].count;
              const actualToRemove = Math.min(itemsToRemove, availableCount);
              
              if (actualToRemove > 0) {
                // Use CTE with FOR UPDATE SKIP LOCKED for concurrent-safe FIFO selection
                await client.query(`
                  WITH items_to_scrap AS (
                    SELECT id FROM warehouse.packed_product_items
                    WHERE packed_product_id = $1 AND status = 'available'
                    ORDER BY packed_at ASC
                    LIMIT $2
                    FOR UPDATE SKIP LOCKED
                  )
                  UPDATE warehouse.packed_product_items
                  SET status = 'scrapped',
                      source_type = 'inventory_count',
                      source_id = $3,
                      updated_at = CURRENT_TIMESTAMP
                  WHERE id IN (SELECT id FROM items_to_scrap)
                `, [item.packed_product_id, actualToRemove, parseInt(id, 10)]);
              }
              
              // If we still have a deficit (fewer available than needed), log a warning
              if (actualToRemove < itemsToRemove) {
                console.warn(`[INVENTORY] Warning: Only ${actualToRemove} items available to scrap for packed_product_id ${item.packed_product_id}, needed ${itemsToRemove}`);
              }
            }
            
            // Update summary quantity from actual items count
            await client.query(`
              UPDATE warehouse.packed_products pp
              SET 
                quantity = (SELECT COUNT(*) FROM warehouse.packed_product_items ppi WHERE ppi.packed_product_id = pp.id AND ppi.status IN ('available', 'reserved')),
                reserved_quantity = (SELECT COUNT(*) FROM warehouse.packed_product_items ppi WHERE ppi.packed_product_id = pp.id AND ppi.status = 'reserved'),
                updated_at = CURRENT_TIMESTAMP
              WHERE pp.id = $1
            `, [item.packed_product_id]);
            
            updatedPackedProducts++;
          }
        }
      }
      
      // Mark count as finalized
      const finalizeResult = await client.query(`
        UPDATE warehouse.inventory_counts
        SET 
          status = 'finalized',
          finalized_by = $1,
          finalized_at = CURRENT_TIMESTAMP
        WHERE id = $2
        RETURNING 
          id,
          name,
          status,
          notes,
          created_by AS "createdBy",
          created_at AS "createdAt",
          finalized_by AS "finalizedBy",
          finalized_at AS "finalizedAt"
      `, [username, id]);
      
      await client.query('COMMIT');
      
      res.json({
        inventoryCount: finalizeResult.rows[0],
        updatedItems: items.filter(item => {
          const diff = safeParseNumber(item.difference, 0);
          return !isNaN(diff) && diff !== 0;
        }).length,
        totalItems: items.length,
        stats: {
          materials: updatedMaterials,
          stockPanels: updatedStockPanels,
          packagingMaterials: updatedPackagingMaterials,
          packedProducts: updatedPackedProducts
        }
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error finalizing inventory count:", error);
      res.status(500).json({ error: "Failed to finalize inventory count" });
    } finally {
      client.release();
    }
  });

  // POST /api/warehouse/inventory-counts/:id/duplicate - Duplikuj spis (tylko pozycje, bez danych)
  app.post("/api/warehouse/inventory-counts/:id/duplicate", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { id } = req.params;
      const { name } = req.body;
      const username = (req.user as any)?.username || 'system';
      
      await client.query('BEGIN');
      
      // Get original count
      const originalResult = await client.query(
        'SELECT name FROM warehouse.inventory_counts WHERE id = $1',
        [id]
      );
      
      if (originalResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      const newName = name || `${originalResult.rows[0].name} (kopia)`;
      
      // Create new count
      const newCountResult = await client.query(`
        INSERT INTO warehouse.inventory_counts (name, status, created_by)
        VALUES ($1, 'draft', $2)
        RETURNING 
          id,
          name,
          status,
          notes,
          created_by AS "createdBy",
          created_at AS "createdAt",
          finalized_by AS "finalizedBy",
          finalized_at AS "finalizedAt"
      `, [newName, username]);
      
      const newCount = newCountResult.rows[0];
      
      // Copy items (support materials, stock panels, packaging materials, packed products)
      const itemsResult = await client.query(`
        SELECT 
          material_id,
          stock_panel_id,
          packaging_material_id,
          packed_product_id
        FROM warehouse.inventory_count_items 
        WHERE inventory_count_id = $1
      `, [id]);
      
      const items = [];
      for (const item of itemsResult.rows) {
        let systemQuantity = 0;
        
        // Get current system quantity based on item type
        if (item.material_id) {
          const materialResult = await client.query(
            'SELECT quantity FROM warehouse.materials WHERE id = $1',
            [item.material_id]
          );
          if (materialResult.rows.length > 0) {
            systemQuantity = safeParseNumber(materialResult.rows[0].quantity, 0);
          }
        } else if (item.stock_panel_id) {
          const stockPanelResult = await client.query(
            'SELECT quantity FROM warehouse.stock_panels WHERE id = $1',
            [item.stock_panel_id]
          );
          if (stockPanelResult.rows.length > 0) {
            systemQuantity = safeParseNumber(stockPanelResult.rows[0].quantity, 0);
          }
        } else if (item.packaging_material_id) {
          const packagingResult = await client.query(
            'SELECT quantity FROM warehouse.packaging_materials WHERE id = $1',
            [item.packaging_material_id]
          );
          if (packagingResult.rows.length > 0) {
            systemQuantity = safeParseNumber(packagingResult.rows[0].quantity, 0);
          }
        } else if (item.packed_product_id) {
          const packedProductResult = await client.query(
            'SELECT quantity FROM warehouse.packed_products WHERE id = $1',
            [item.packed_product_id]
          );
          if (packedProductResult.rows.length > 0) {
            systemQuantity = safeParseNumber(packedProductResult.rows[0].quantity, 0);
          }
        }
        
        const newItemResult = await client.query(`
          INSERT INTO warehouse.inventory_count_items (
            inventory_count_id, 
            material_id, 
            stock_panel_id,
            packaging_material_id,
            packed_product_id,
            system_quantity
          )
          VALUES ($1, $2, $3, $4, $5, $6)
          RETURNING 
            id,
            inventory_count_id AS "inventoryCountId",
            material_id AS "materialId",
            stock_panel_id AS "stockPanelId",
            packaging_material_id AS "packagingMaterialId",
            packed_product_id AS "packedProductId",
            system_quantity AS "systemQuantity",
            counted_quantity AS "countedQuantity",
            difference,
            notes,
            created_at AS "createdAt",
            updated_at AS "updatedAt"
        `, [
          newCount.id, 
          item.material_id || null, 
          item.stock_panel_id || null,
          item.packaging_material_id || null,
          item.packed_product_id || null,
          systemQuantity
        ]);
        
        items.push(newItemResult.rows[0]);
      }
      
      await client.query('COMMIT');
      
      res.json({
        inventoryCount: newCount,
        items
      });
    } catch (error) {
      await client.query('ROLLBACK');
      console.error("❌ Error duplicating inventory count:", error);
      res.status(500).json({ error: "Failed to duplicate inventory count" });
    } finally {
      client.release();
    }
  });

  // DELETE /api/warehouse/inventory-counts/:id - Usuń spis inwentaryzacyjny
  app.delete("/api/warehouse/inventory-counts/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Check if count is finalized
      const countCheck = await pool.query(
        'SELECT status FROM warehouse.inventory_counts WHERE id = $1',
        [id]
      );
      
      if (countCheck.rows.length === 0) {
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      if (countCheck.rows[0].status === 'finalized') {
        return res.status(400).json({ error: "Cannot delete finalized inventory count" });
      }
      
      // Delete count (items will be cascade deleted)
      await pool.query('DELETE FROM warehouse.inventory_counts WHERE id = $1', [id]);
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting inventory count:", error);
      res.status(500).json({ error: "Failed to delete inventory count" });
    }
  });

  // PATCH /api/warehouse/inventory-counts/:id - Zmień status spisu (archiwizacja)
  app.patch("/api/warehouse/inventory-counts/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { status } = req.body;
      
      // Validate status
      const allowedStatuses = ['draft', 'finalized', 'archived'];
      if (!status || !allowedStatuses.includes(status)) {
        return res.status(400).json({ error: "Invalid status. Allowed: draft, finalized, archived" });
      }
      
      // Check if count exists
      const countCheck = await pool.query(
        'SELECT id, status, name FROM warehouse.inventory_counts WHERE id = $1',
        [id]
      );
      
      if (countCheck.rows.length === 0) {
        return res.status(404).json({ error: "Inventory count not found" });
      }
      
      // Update status
      const result = await pool.query(
        `UPDATE warehouse.inventory_counts 
         SET status = $1 
         WHERE id = $2 
         RETURNING *`,
        [status, id]
      );
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating inventory count status:", error);
      res.status(500).json({ error: "Failed to update inventory count status" });
    }
  });

  // POST /api/warehouse/bulk-import-from-catalog - Bulk import produktów z katalogu do magazynu produkty-spakowane
  app.post("/api/warehouse/bulk-import-from-catalog", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { name, products, groupId, notes } = req.body;
      const username = (req.user as any)?.username || 'system';
      
      // Validate inputs
      if (!name || typeof name !== 'string') {
        return res.status(400).json({ error: "Name is required" });
      }
      
      if (!Array.isArray(products) || products.length === 0) {
        return res.status(400).json({ error: "Products array is required and must not be empty" });
      }
      
      // Validate each product has catalogProductId and optional countedQuantity
      for (const product of products) {
        if (!product.catalogProductId || typeof product.catalogProductId !== 'number') {
          return res.status(400).json({ error: "Each product must have a valid catalogProductId" });
        }
        if (product.countedQuantity !== undefined && typeof product.countedQuantity !== 'number') {
          return res.status(400).json({ error: "countedQuantity must be a number if provided" });
        }
      }
      
      await client.query('BEGIN');
      
      // Find or create default group for produkty-spakowane category
      let targetGroupId = groupId;
      if (!targetGroupId) {
        const defaultGroupResult = await client.query(`
          SELECT id FROM warehouse.material_groups
          WHERE category = 'produkty-spakowane' AND is_active = true
          ORDER BY id ASC
          LIMIT 1
        `);
        
        if (defaultGroupResult.rows.length > 0) {
          targetGroupId = defaultGroupResult.rows[0].id;
        } else {
          // Create default group for produkty-spakowane
          const newGroupResult = await client.query(`
            INSERT INTO warehouse.material_groups (name, code, category, description, is_active)
            VALUES ($1, $2, $3, $4, $5)
            RETURNING id
          `, [
            'Produkty z katalogu',
            'produkty-katalogowe',
            'produkty-spakowane',
            'Automatycznie utworzona grupa dla produktów z katalogu',
            true
          ]);

          results.success++;
          targetGroupId = newGroupResult.rows[0].id;
        }
      }
      
      // Verify all catalog products exist and get primary image
      const catalogProductIds = products.map((p: any) => p.catalogProductId);
      const catalogProductsResult = await client.query(`
        SELECT 
          p.id, 
          p.sku, 
          p.title, 
          p.length, 
          p.width, 
          p.height, 
          p.color,
          MAX(CASE WHEN pi.is_primary = true THEN pi.url END) AS "primaryImageUrl",
          MAX(CASE WHEN pi.is_primary = true THEN pi.thumbnail_url END) AS "primaryImageThumbnailUrl"
        FROM catalog.products p
        LEFT JOIN catalog.product_images pi ON p.id = pi.product_id
        WHERE p.id = ANY($1::int[])
        GROUP BY p.id, p.sku, p.title, p.length, p.width, p.height, p.color
      `, [catalogProductIds]);
      
      if (catalogProductsResult.rows.length !== products.length) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Some catalog products not found" });
      }
      
      const catalogProductsMap = new Map(
        catalogProductsResult.rows.map((p: any) => [p.id, p])
      );
      
      // Create inventory count
      const countResult = await client.query(`
        INSERT INTO warehouse.inventory_counts (name, status, notes, created_by)
        VALUES ($1, 'draft', $2, $3)
        RETURNING 
          id,
          name,
          status,
          notes,
          created_by AS "createdBy",
          created_at AS "createdAt",
          finalized_by AS "finalizedBy",
          finalized_at AS "finalizedAt"
      `, [name, notes || null, username]);
      
      const inventoryCount = countResult.rows[0];
      
      // Process each product
      const items = [];
      const createdMaterials = [];
      
      for (const productReq of products) {
        const catalogProduct = catalogProductsMap.get(productReq.catalogProductId);
        
        if (!catalogProduct) {
          continue; // Skip if not found (shouldn't happen due to earlier validation)
        }
        
        // Check if packed product already exists with this catalog_product_id
        const existingPackedProductResult = await client.query(`
          SELECT id, quantity, product_name, product_sku
          FROM warehouse.packed_products
          WHERE catalog_product_id = $1
        `, [catalogProduct.id]);
        
        let packedProductId: number;
        let systemQuantity: number;
        
        if (existingPackedProductResult.rows.length > 0) {
          // Packed product exists - use it
          const existingProduct = existingPackedProductResult.rows[0];
          packedProductId = existingProduct.id;
          systemQuantity = safeParseNumber(existingProduct.quantity);
        } else {
          // Create new packed product in warehouse.packed_products
          const productSku = catalogProduct.sku || `CAT-${catalogProduct.id}`;
          const productName = catalogProduct.title || `Produkt ${catalogProduct.sku}`;
          
          // Get primary image URL from catalog product
          const primaryImageUrl = catalogProduct.primaryImageUrl || null;
          
          const packedProductResult = await client.query(`
            INSERT INTO warehouse.packed_products (
              group_id,
              catalog_product_id,
              product_sku,
              product_name,
              product_type,
              quantity,
              image_url,
              notes,
              is_active
            )
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, true, $10)
            RETURNING id, quantity, product_name, product_sku
          `, [
            targetGroupId || null,
            catalogProduct.id,
            productSku,
            productName,
            'catalog_product',
            0,
            primaryImageUrl,
            `Produkt katalogowy: ${catalogProduct.title}`,
            true
          ]);

          results.success++;
          
          const newPackedProduct = packedProductResult.rows[0];
          packedProductId = newPackedProduct.id;
          systemQuantity = 0;
          
          createdMaterials.push({
            id: newPackedProduct.id,
            name: newPackedProduct.product_name,
            internalCode: newPackedProduct.product_sku,
            catalogProductId: catalogProduct.id
          });
        }
        
        // Add item to inventory count
        const countedQuantity = productReq.countedQuantity !== undefined ? productReq.countedQuantity : null;
        
        // Calculate difference if countedQuantity is provided
        let difference = null;
        if (countedQuantity !== null) {
          difference = countedQuantity - systemQuantity;
        }
        
        const itemResult = await client.query(`
          INSERT INTO warehouse.inventory_count_items (
            inventory_count_id,
            packed_product_id,
            system_quantity,
            counted_quantity,
            difference
          )
          VALUES ($1, $2, $3, $4, $5)
          RETURNING 
            id,
            inventory_count_id AS "inventoryCountId",
            material_id AS "materialId",
            stock_panel_id AS "stockPanelId",
            packaging_material_id AS "packagingMaterialId",
            packed_product_id AS "packedProductId",
            system_quantity AS "systemQuantity",
            counted_quantity AS "countedQuantity",
            difference,
            notes,
            created_at AS "createdAt",
            updated_at AS "updatedAt"
        `, [inventoryCount.id, packedProductId, systemQuantity, countedQuantity, difference]);
        
        items.push(itemResult.rows[0]);
      }
      
      await client.query('COMMIT');
      
      res.json({
        inventoryCount,
        items,
        createdMaterials,
        message: `Successfully created inventory count with ${items.length} items. ${createdMaterials.length} new materials created.`
      });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("❌ Error in bulk import from catalog:", error);
      
      // Handle unique constraint violation
      if (error.code === '23505' && error.constraint === 'packed_products_product_sku_key') {
        return res.status(409).json({ 
          error: "SKU conflict. A packed product with this SKU already exists.",
          details: error.detail
        });
      }
      
      res.status(500).json({ error: "Failed to import products from catalog" });
    } finally {
      client.release();
    }
  });

  // ===== WAREHOUSE FORMATKI (STOCK PANELS) =====
  
  // GET /api/warehouse/stock-panels - Lista formatek z filtrowaniem i paginacją
  app.get("/api/warehouse/stock-panels", isAuthenticated, async (req, res) => {
    try {
      const { 
        search, 
        boardCode, 
        edgingCode, 
        colorCode, 
        source, 
        isDrilled, 
        isEdged,
        minLength,
        maxLength,
        minWidth,
        maxWidth,
        page = 1, 
        limit = 50,
        sortBy = 'created_at',
        sortOrder = 'DESC'
      } = req.query;
      
      const offset = (Number(page) - 1) * Number(limit);
      
      // Build WHERE clauses
      const whereClauses: string[] = ['sp.is_active = true AND (sp.is_archived = false OR sp.is_archived IS NULL)'];
      const params: any[] = [];
      let paramIndex = 1;
      
      if (search && typeof search === 'string') {
        // Split search by semicolons to support multiple filters (AND logic)
        const searchTerms = search
          .split(';')
          .map((term: string) => term.trim())
          .filter((term: string) => term.length > 0);
        
        // Each term must match at least one field (OR within term)
        // All terms must match (AND between terms)
        const termClauses = searchTerms.map((term: string) => {
          const clause = `(
            sp.generated_name ILIKE $${paramIndex} OR
            sp.cz1 ILIKE $${paramIndex} OR
            sp.cz2 ILIKE $${paramIndex} OR
            sp.furniture_type ILIKE $${paramIndex} OR
            sp.board_code ILIKE $${paramIndex} OR
            sp.edging_code ILIKE $${paramIndex} OR
            sp.color_code ILIKE $${paramIndex} OR
            sp.notes ILIKE $${paramIndex}
          )`;
          params.push(`%${term}%`);
          paramIndex++;
          return clause;
        });
        
        if (termClauses.length > 0) {
          whereClauses.push(`(${termClauses.join(' AND ')})`);
        }
      }
      
      if (boardCode) {
        whereClauses.push(`sp.board_code = $${paramIndex}`);
        params.push(boardCode);
        paramIndex++;
      }
      
      if (edgingCode) {
        whereClauses.push(`sp.edging_code = $${paramIndex}`);
        params.push(edgingCode);
        paramIndex++;
      }
      
      if (colorCode) {
        whereClauses.push(`sp.color_code = $${paramIndex}`);
        params.push(colorCode);
        paramIndex++;
      }
      
      if (source) {
        whereClauses.push(`sp.source = $${paramIndex}`);
        params.push(source);
        paramIndex++;
      }
      
      if (isDrilled !== undefined) {
        whereClauses.push(`sp.is_drilled = $${paramIndex}`);
        params.push(isDrilled === 'true');
        paramIndex++;
      }
      
      if (isEdged !== undefined) {
        whereClauses.push(`sp.is_edged = $${paramIndex}`);
        params.push(isEdged === 'true');
        paramIndex++;
      }
      
      if (minLength) {
        whereClauses.push(`sp.length >= $${paramIndex}`);
        params.push(minLength);
        paramIndex++;
      }
      
      if (maxLength) {
        whereClauses.push(`sp.length <= $${paramIndex}`);
        params.push(maxLength);
        paramIndex++;
      }
      
      if (minWidth) {
        whereClauses.push(`sp.width >= $${paramIndex}`);
        params.push(minWidth);
        paramIndex++;
      }
      
      if (maxWidth) {
        whereClauses.push(`sp.width <= $${paramIndex}`);
        params.push(maxWidth);
        paramIndex++;
      }
      
      const whereClause = whereClauses.length > 0 ? 'WHERE ' + whereClauses.join(' AND ') : '';
      
      // Allowed sort columns
      const allowedSortBy = ['generated_name', 'length', 'width', 'thickness', 'quantity', 'created_at', 'source'];
      const sortColumn = allowedSortBy.includes(sortBy as string) ? sortBy : 'created_at';
      const sortDirection = sortOrder === 'ASC' ? 'ASC' : 'DESC';
      
      // Get total count
      const countResult = await pool.query(`
        SELECT COUNT(*) as total
        FROM warehouse.stock_panels sp
        ${whereClause}
      `, params);
      
      const total = parseInt(countResult.rows[0].total, 10);
      
      // Get paginated results with dictionary joins
      const result = await pool.query(`
        SELECT 
          sp.id,
          sp.generated_name AS "generatedName",
          sp.cz1,
          sp.cz2,
          sp.furniture_type AS "furnitureType",
          sp.length,
          sp.width,
          sp.thickness,
          sp.board_code AS "boardCode",
          sp.edging_code AS "edgingCode",
          sp.color_code AS "colorCode",
          sp.edge1,
          sp.edge2,
          sp.edge3,
          sp.edge4,
          sp.is_drilled AS "isDrilled",
          sp.is_edged AS "isEdged",
          sp.source,
          sp.source_reference AS "sourceReference",
          sp.quantity,
          sp.unit AS "unitOfMeasure",
          sp.location,
          sp.notes,
          sp.image_url AS "imageUrl",
          sp.is_active AS "isActive",
          sp.created_at AS "createdAt",
          sp.updated_at AS "updatedAt",
          dc.name AS "colorName",
          dc.color AS "colorHex"
        FROM warehouse.stock_panels sp
        LEFT JOIN product_creator.dictionaries dc ON dc.code = sp.color_code AND dc.dictionary_type = 'color' AND dc.is_active = true
        ${whereClause}
        ORDER BY sp.${sortColumn} ${sortDirection}
        LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
      `, [...params, Number(limit), offset]);
      
      res.json({
        data: result.rows,
        pagination: {
          total,
          page: Number(page),
          limit: Number(limit),
          totalPages: Math.ceil(total / Number(limit))
        }
      });
    } catch (error) {
      console.error("❌ Error fetching stock panels:", error);
      res.status(500).json({ error: "Failed to fetch stock panels" });
    }
  });
  
  // GET /api/warehouse/stock-panels/:id - Pojedyncza formatka
  app.get("/api/warehouse/stock-panels/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT 
          sp.id,
          sp.generated_name AS "generatedName",
          sp.cz1,
          sp.cz2,
          sp.furniture_type AS "furnitureType",
          sp.length,
          sp.width,
          sp.thickness,
          sp.board_code AS "boardCode",
          sp.edging_code AS "edgingCode",
          sp.color_code AS "colorCode",
          sp.edge1,
          sp.edge2,
          sp.edge3,
          sp.edge4,
          sp.is_drilled AS "isDrilled",
          sp.is_edged AS "isEdged",
          sp.source,
          sp.source_reference AS "sourceReference",
          sp.quantity,
          sp.unit AS "unitOfMeasure",
          sp.material_cost AS "materialCost",
          sp.location,
          sp.location_id AS "locationId",
          sp.carrier_id AS "carrierId",
          sp.group_id AS "groupId",
          sp.notes,
          sp.image_url AS "imageUrl",
          sp.is_active AS "isActive",
          sp.created_at AS "createdAt",
          sp.updated_at AS "updatedAt",
          dc.name AS "colorName",
          dc.color AS "colorHex",
          pl.name AS "locationName",
          pc.name AS "carrierName",
          -- Dane płyty do wyliczenia kosztu materiału
          board.id AS "boardId",
          board.name AS "boardName",
          board.price AS "boardPrice",
          CASE 
            WHEN board.price IS NOT NULL AND sp.length IS NOT NULL AND sp.width IS NOT NULL 
            THEN ROUND((board.price::numeric * (sp.length::numeric * sp.width::numeric / 1000000)), 4)
            ELSE NULL
          END AS "calculatedBoardCost",
          -- Dane obrzeża do wyliczenia kosztu
          edging.id AS "edgingId",
          edging.name AS "edgingName",
          edging.price AS "edgingPrice",
          -- Koszt obrzeża: jeśli wszystkie 4 krawędzie oklejone, obwód × cena za mb
          CASE 
            WHEN edging.price IS NOT NULL 
              AND sp.length IS NOT NULL 
              AND sp.width IS NOT NULL 
              AND sp.edge1 = 'T' AND sp.edge2 = 'T' AND sp.edge3 = 'T' AND sp.edge4 = 'T'
            THEN ROUND((edging.price::numeric * 2 * (sp.length::numeric + sp.width::numeric) / 1000), 4)
            ELSE NULL
          END AS "calculatedEdgingCost",
          -- Całkowity koszt materiału (płyta + obrzeże)
          CASE 
            WHEN board.price IS NOT NULL AND sp.length IS NOT NULL AND sp.width IS NOT NULL 
            THEN ROUND((board.price::numeric * (sp.length::numeric * sp.width::numeric / 1000000)), 4)
              + COALESCE(
                  CASE 
                    WHEN edging.price IS NOT NULL 
                      AND sp.edge1 = 'T' AND sp.edge2 = 'T' AND sp.edge3 = 'T' AND sp.edge4 = 'T'
                    THEN ROUND((edging.price::numeric * 2 * (sp.length::numeric + sp.width::numeric) / 1000), 4)
                    ELSE 0
                  END, 0)
            ELSE NULL
          END AS "calculatedMaterialCost"
        FROM warehouse.stock_panels sp
        LEFT JOIN product_creator.dictionaries dc ON dc.code = sp.color_code AND dc.dictionary_type = 'color' AND dc.is_active = true
        LEFT JOIN production.production_locations pl ON sp.location_id = pl.id
        LEFT JOIN production.production_carriers pc ON sp.carrier_id = pc.id
        LEFT JOIN warehouse.materials board ON board.internal_code = sp.board_code
        LEFT JOIN warehouse.materials edging ON edging.internal_code = sp.edging_code
        WHERE sp.id = \$1
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Stock panel not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error fetching stock panel:", error);
      res.status(500).json({ error: "Failed to fetch stock panel" });
      res.status(500).json({ error: "Failed to fetch stock panel" });
    }
  });

  // GET /api/warehouse/stock-panels/:id/usage - Get stock panel usage history
  app.get("/api/warehouse/stock-panels/:id/usage", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // First get the panel's generated name (used as material_code in movements)
      const panelResult = await pool.query(
        'SELECT generated_name FROM warehouse.stock_panels WHERE id = $1',
        [id]
      );
      
      if (panelResult.rows.length === 0) {
        return res.status(404).json({ error: "Stock panel not found" });
      }
      
      const generatedName = panelResult.rows[0].generated_name;
      
      // Get usage history from production_material_movements or stock_panel_events
      const usageQuery = `
        SELECT 
          spe.id,
          spe.event_date AS "usedAt",
          spe.quantity_change AS "quantityUsed",
          spe.notes,
          spe.production_order_id AS "productionOrderId",
          po.order_number AS "productionOrderNumber"
        FROM warehouse.stock_panel_events spe
        LEFT JOIN production.production_orders po ON po.id = spe.production_order_id
        WHERE spe.panel_id = $1
          AND spe.event_type = 'usage'
        ORDER BY spe.event_date DESC
        LIMIT 100
      `;
      
      const usageResult = await pool.query(usageQuery, [id]);
      res.json(usageResult.rows);
    } catch (error) {
      console.error("❌ Error fetching stock panel usage:", error);
      res.status(500).json({ error: "Failed to fetch stock panel usage" });
    }
  });


  // GET /api/warehouse/stock-panels/:id/usage-check - Check if stock panel is used in BOM or production plans
  app.get("/api/warehouse/stock-panels/:id/usage-check", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Get the panel's generated name
      const panelResult = await pool.query(
        'SELECT generated_name FROM warehouse.stock_panels WHERE id = $1',
        [id]
      );
      
      if (panelResult.rows.length === 0) {
        return res.status(404).json({ error: "Stock panel not found" });
      }
      
      const generatedName = panelResult.rows[0].generated_name;
      
      // Check BOM usage - formatki in product_components
      const bomQuery = `
        SELECT COUNT(*) as count
        FROM bom.product_components pc
        WHERE pc.generated_name = $1
      `;
      const bomResult = await pool.query(bomQuery, [generatedName]);
      const bomCount = parseInt(bomResult.rows[0].count || '0');
      
      // Check production plan usage - formatki in production_plan_lines
      const productionQuery = `
        SELECT COUNT(*) as count
        FROM production.production_plan_lines ppl
        JOIN catalog.products cp ON cp.id = ppl.product_id
        WHERE cp.sku = $1 OR cp.name = $1
      `;
      const productionResult = await pool.query(productionQuery, [generatedName]);
      const productionCount = parseInt(productionResult.rows[0].count || '0');
      
      res.json({
        isUsed: bomCount > 0 || productionCount > 0,
        bomCount,
        productionCount
      });
    } catch (error) {
      console.error("❌ Error checking stock panel usage:", error);
      res.status(500).json({ error: "Failed to check stock panel usage" });
    }
  });

  // GET /api/warehouse/stock-panels/:id/inventory-counts - Get inventory counts containing this stock panel
  app.get("/api/warehouse/stock-panels/:id/inventory-counts", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // First check if the panel exists
      const panelCheck = await pool.query(
        'SELECT id FROM warehouse.stock_panels WHERE id = $1',
        [id]
      );
      
      if (panelCheck.rows.length === 0) {
        return res.status(404).json({ error: "Stock panel not found" });
      }
      
      // Get all inventory counts that include this stock panel
      const countsQuery = `
        SELECT 
          ic.id AS "countId",
          ic.name AS "countName",
          ic.status AS "countStatus",
          ic.created_at AS "countCreatedAt",
          ic.finalized_at AS "countFinalizedAt",
          ici.system_quantity AS "expectedQuantity",
          ici.counted_quantity AS "countedQuantity",
          ici.difference
        FROM warehouse.inventory_count_items ici
        INNER JOIN warehouse.inventory_counts ic ON ic.id = ici.inventory_count_id
        WHERE ici.stock_panel_id = $1
        ORDER BY ic.created_at DESC
      `;
      
      const result = await pool.query(countsQuery, [id]);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching stock panel inventory counts:", error);
      res.status(500).json({ error: "Failed to fetch inventory counts" });
    }
  });
  
  // POST /api/warehouse/stock-panels - Utwórz nową formatkę
  app.post("/api/warehouse/stock-panels", isAuthenticated, async (req, res) => {
    try {
      const {
        generatedName,
        cz1,
        cz2,
        furnitureType,
        length,
        width,
        thickness,
        boardCode,
        edgingCode,
        colorCode,
        edge1,
        edge2,
        edge3,
        edge4,
        isDrilled,
        isEdged,
        source,
        sourceReference,
        quantity,
        location,
        notes,
        imageUrl
      } = req.body;
      
      if (!generatedName || !length || !width) {
        return res.status(400).json({ error: "Generated name, length and width are required" });
      }
      
      const result = await pool.query(`
        INSERT INTO warehouse.stock_panels (
          generated_name, cz1, cz2, furniture_type, length, width, thickness,
          board_code, edging_code, color_code, edge1, edge2, edge3, edge4,
          is_drilled, is_edged, source, source_reference, quantity, location, notes, image_url
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21, $22)
        RETURNING 
          id,
          generated_name AS "generatedName",
          cz1,
          cz2,
          furniture_type AS "furnitureType",
          length,
          width,
          thickness,
          board_code AS "boardCode",
          edging_code AS "edgingCode",
          color_code AS "colorCode",
          edge1,
          edge2,
          edge3,
          edge4,
          is_drilled AS "isDrilled",
          is_edged AS "isEdged",
          source,
          source_reference AS "sourceReference",
          quantity,
          location,
          notes,
          image_url AS "imageUrl",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        generatedName, cz1 || null, cz2 || null, furnitureType || null,
        length, width, thickness || 18,
        boardCode || null, edgingCode || null, colorCode || null,
        edge1 || false, edge2 || false, edge3 || false, edge4 || false,
        isDrilled || false, isEdged || false,
        source || 'stock', sourceReference || null,
        quantity || 0, location || null, notes || null, imageUrl || null
      ]);
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating stock panel:", error);
      res.status(500).json({ error: "Failed to create stock panel" });
    }
  });
  
  // PATCH /api/warehouse/stock-panels/:id - Aktualizuj formatkę
  app.patch("/api/warehouse/stock-panels/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const {
        generatedName,
        cz1,
        cz2,
        furnitureType,
        length,
        width,
        thickness,
        boardCode,
        edgingCode,
        colorCode,
        edge1,
        edge2,
        edge3,
        edge4,
        isDrilled,
        isEdged,
        source,
        sourceReference,
        quantity,
        location,
        locationId,
        carrierId,
        groupId,
        notes,
        imageUrl,
        isActive
      } = req.body;
      
      const result = await pool.query(`
        UPDATE warehouse.stock_panels
        SET 
          generated_name = COALESCE($1, generated_name),
          cz1 = COALESCE($2, cz1),
          cz2 = COALESCE($3, cz2),
          furniture_type = COALESCE($4, furniture_type),
          length = COALESCE($5, length),
          width = COALESCE($6, width),
          thickness = COALESCE($7, thickness),
          board_code = COALESCE($8, board_code),
          edging_code = COALESCE($9, edging_code),
          color_code = COALESCE($10, color_code),
          edge1 = COALESCE($11, edge1),
          edge2 = COALESCE($12, edge2),
          edge3 = COALESCE($13, edge3),
          edge4 = COALESCE($14, edge4),
          is_drilled = COALESCE($15, is_drilled),
          is_edged = COALESCE($16, is_edged),
          source = COALESCE($17, source),
          source_reference = COALESCE($18, source_reference),
          quantity = COALESCE($19, quantity),
          location = COALESCE($20, location),
          location_id = CASE WHEN $21::INTEGER IS NULL THEN NULL ELSE $21 END,
          carrier_id = CASE WHEN $22::INTEGER IS NULL THEN NULL ELSE $22 END,
          group_id = CASE WHEN $23::INTEGER IS NULL THEN NULL ELSE $23 END,
          notes = COALESCE($24, notes),
          image_url = COALESCE($25, image_url),
          unit = COALESCE($26, unit),
          price = COALESCE($27, price),
          is_active = COALESCE($28, is_active),
          updated_at = NOW()
        WHERE id = $29
        RETURNING 
          id,
          generated_name AS "generatedName",
          cz1,
          cz2,
          furniture_type AS "furnitureType",
          length,
          width,
          thickness,
          board_code AS "boardCode",
          edging_code AS "edgingCode",
          color_code AS "colorCode",
          edge1,
          edge2,
          edge3,
          edge4,
          is_drilled AS "isDrilled",
          is_edged AS "isEdged",
          source,
          source_reference AS "sourceReference",
          quantity,
          location,
          location_id AS "locationId",
          carrier_id AS "carrierId",
          group_id AS "groupId",
          notes,
          image_url AS "imageUrl",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        generatedName, cz1, cz2, furnitureType,
        length, width, thickness,
        boardCode, edgingCode, colorCode,
        edge1, edge2, edge3, edge4,
        isDrilled, isEdged,
        source, sourceReference,
        quantity, location, locationId, carrierId, groupId, notes, imageUrl,
        isActive,
        id
      ]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Stock panel not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating stock panel:", error);
      res.status(500).json({ error: "Failed to update stock panel" });
    }
  });
  
  // PATCH /api/warehouse/stock-panels/:id/quantity - Aktualizuj tylko ilość (szybka operacja magazynowa)
  app.patch("/api/warehouse/stock-panels/:id/quantity", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { quantity } = req.body;
      
      if (quantity === undefined || quantity === null) {
        return res.status(400).json({ error: "Quantity is required" });
      }
      
      const result = await pool.query(`
        UPDATE warehouse.stock_panels
        SET 
          quantity = $1,
          updated_at = NOW()
        WHERE id = $2
        RETURNING quantity, updated_at AS "updatedAt"
      `, [quantity, id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Stock panel not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating stock panel quantity:", error);
      res.status(500).json({ error: "Failed to update quantity" });
    }
  });
  
  // DELETE /api/warehouse/stock-panels/:id - Usuń formatkę
  app.delete("/api/warehouse/stock-panels/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      await pool.query(
        'DELETE FROM warehouse.stock_panels WHERE id = $1',
        [id]
      );
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting stock panel:", error);
      res.status(500).json({ error: "Failed to delete stock panel" });
    }
  });

  // POST /api/warehouse/stock-panels/bulk-delete - Grupowe usuwanie formatek
  app.post("/api/warehouse/stock-panels/bulk-delete", isAuthenticated, async (req, res) => {
    try {
      const { panelIds } = req.body;

      if (!Array.isArray(panelIds) || panelIds.length === 0) {
        return res.status(400).json({ error: "Panel IDs array is required" });
      }

      const placeholders = panelIds.map((_, i) => `$${i + 1}`).join(',');
      const result = await pool.query(
        `DELETE FROM warehouse.stock_panels WHERE id IN (${placeholders}) RETURNING id`,
        panelIds
      );

      res.json({
        success: true,
        deletedCount: result.rowCount,
        deletedIds: result.rows.map(r => r.id)
      });
    } catch (error) {
      console.error("❌ Error bulk deleting stock panels:", error);
      res.status(500).json({ error: "Failed to bulk delete stock panels" });
    }
  });

  // POST /api/warehouse/stock-panels/bulk-update - Grupowa aktualizacja formatek
  app.post("/api/warehouse/stock-panels/bulk-update", isAuthenticated, async (req, res) => {
    try {
      const { panelIds, updates, skipLockedPanels = true } = req.body;
      const username = (req as any).user?.username || 'system';

      if (!Array.isArray(panelIds) || panelIds.length === 0) {
        return res.status(400).json({ error: "Panel IDs array is required" });
      }

      if (!updates || typeof updates !== 'object') {
        return res.status(400).json({ error: "Updates object is required" });
      }

      // All selected panels can be updated
      const lockedPanelIds: number[] = [];
      const updateablePanelIds = panelIds;

      // Build UPDATE query dynamically based on provided fields
      // Extended list of allowed fields for formatki
      const allowedFields = [
        'carrier_id', 'location_id', 'location', 'source', 'notes', 'status',
        'cz1', 'cz2', 'color_code', 'furniture_type',
        'board_code', 'edging_code', 'thickness', 
        'is_drilled', 'is_complete', 'group_id'
      ];
      
      // Field name mapping from camelCase to snake_case
      const fieldMapping: Record<string, string> = {
        carrierId: 'carrier_id',
        locationId: 'location_id',
        furnitureType: 'furniture_type',
        plateType: 'plate_type',
        boardCode: 'board_code',
        edgingCode: 'edging_code',
        isDrilled: 'is_drilled',
        isComplete: 'is_complete',
        color: 'color_code',
        groupId: 'group_id'
      };
      
      const updatePairs: string[] = [];
      const values: any[] = [];
      let paramIndex = 1;

      // Process updates (handle both camelCase and snake_case keys)
      for (const [key, value] of Object.entries(updates)) {
        const dbField = fieldMapping[key] || key;
        
        if (!allowedFields.includes(dbField)) continue;
        
        if (value === null) {
          // Allow null values for optional fields
          updatePairs.push(`${dbField} = NULL`);
        } else if (value !== undefined) {
          updatePairs.push(`${dbField} = $${paramIndex}`);
          values.push(value);
          paramIndex++;
        }
      }

      if (updatePairs.length === 0) {
        return res.status(400).json({ error: "No valid fields to update" });
      }

      // Add username parameter
      const usernameParamIndex = paramIndex;
      values.push(username);
      paramIndex++;

      // Add panelIds to params
      const placeholders = updateablePanelIds.map((_: any, i: number) => `$${paramIndex + i}`).join(',');
      values.push(...updateablePanelIds);

      const query = `
        UPDATE warehouse.stock_panels
        SET ${updatePairs.join(', ')}, updated_at = CURRENT_TIMESTAMP, updated_by = $${usernameParamIndex}
        WHERE id IN (${placeholders})
        RETURNING id, generated_name AS "name"
      `;

      const result = await pool.query(query, values);

      res.json({
        success: true,
        updatedCount: result.rowCount,
        updatedIds: result.rows.map((row: any) => row.id),
        updatedItems: result.rows,
        skippedCount: lockedPanelIds.length,
        skippedIds: lockedPanelIds
      });
    } catch (error) {
      console.error("❌ Error bulk updating stock panels:", error);
      res.status(500).json({ error: "Failed to bulk update stock panels" });
    }
  });

  // ===== WAREHOUSE PACKAGING MATERIALS (OPAKOWANIA) =====
  
  // GET /api/warehouse/packaging-materials - Lista opakowań z filtrowaniem i paginacją
  app.get("/api/warehouse/packaging-materials", isAuthenticated, async (req, res) => {
    try {
      const { 
        search, 
        category,
        locationId,
        carrierId,
        page = 1, 
        limit = 50,
        sortBy = 'created_at',
        sortOrder = 'DESC'
      } = req.query;
      
      const offset = (Number(page) - 1) * Number(limit);
      
      // Build WHERE clauses
      const whereClauses: string[] = ['pm.is_active = true'];
      const params: any[] = [];
      let paramIndex = 1;
      
      if (search) {
        whereClauses.push(`(
          pm.name ILIKE $${paramIndex} OR
          pm.description ILIKE $${paramIndex} OR
          pm.supplier_code ILIKE $${paramIndex} OR
          pm.notes ILIKE $${paramIndex}
        )`);
        params.push(`%${search}%`);
        paramIndex++;
      }
      
      if (category) {
        whereClauses.push(`pm.category = $${paramIndex}`);
        params.push(category);
        paramIndex++;
      }
      
      if (locationId) {
        whereClauses.push(`pm.location_id = $${paramIndex}`);
        params.push(locationId);
        paramIndex++;
      }
      
      if (carrierId) {
        whereClauses.push(`pm.carrier_id = $${paramIndex}`);
        params.push(carrierId);
        paramIndex++;
      }
      
      const whereClause = whereClauses.length > 0 ? 'WHERE ' + whereClauses.join(' AND ') : '';
      
      // Allowed sort columns
      const allowedSortBy = ['name', 'category', 'quantity', 'created_at'];
      const sortColumn = allowedSortBy.includes(sortBy as string) ? sortBy : 'created_at';
      const sortDirection = sortOrder === 'ASC' ? 'ASC' : 'DESC';
      
      // Get total count
      const countResult = await pool.query(`
        SELECT COUNT(*) as total
        FROM warehouse.packaging_materials pm
        ${whereClause}
      `, params);
      
      const total = parseInt(countResult.rows[0].total, 10);
      
      // Get paginated results with location and carrier joins
      const result = await pool.query(`
        SELECT 
          pm.id,
          pm.category,
          pm.name,
          pm.description,
          pm.length,
          pm.width,
          pm.height,
          pm.quantity,
          pm.group_id AS "groupId",
          pm.location_id AS "locationId",
          pm.carrier_id AS "carrierId",
          pNULL AS "supplierCode",
          pm.notes,
          pm.image_url AS "imageUrl",
          pm.is_active AS "isActive",
          pm.created_at AS "createdAt",
          pm.updated_at AS "updatedAt",
          pl.name AS "locationName",
          pl.code AS "locationCode",
          pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
          pc.code AS "carrierCode"
        FROM warehouse.packaging_materials pm
        LEFT JOIN production.production_locations pl ON pl.id = pm.location_id
        LEFT JOIN production.production_carriers pc ON pc.id = pm.carrier_id
        ${whereClause}
        ORDER BY pm.${sortColumn} ${sortDirection}
        LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
      `, [...params, Number(limit), offset]);
      
      res.json({
        data: result.rows,
        pagination: {
          total,
          page: Number(page),
          limit: Number(limit),
          totalPages: Math.ceil(total / Number(limit))
        }
      });
    } catch (error) {
      console.error("❌ Error fetching packaging materials:", error);
      res.status(500).json({ error: "Failed to fetch packaging materials" });
    }
  });
  
  // GET /api/warehouse/packaging-materials/:id - Pojedyncze opakowanie
  app.get("/api/warehouse/packaging-materials/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT 
          pm.id,
          pm.category,
          pm.name,
          pm.description,
          pm.length,
          pm.width,
          pm.height,
          pm.quantity,
          pm.location_id AS "locationId",
          pm.carrier_id AS "carrierId",
          pm.group_id AS "groupId",
          pNULL AS "supplierCode",
          pm.notes,
          pm.image_url AS "imageUrl",
          pm.is_active AS "isActive",
          pm.created_at AS "createdAt",
          pm.updated_at AS "updatedAt",
          pl.name AS "locationName",
          pl.code AS "locationCode",
          pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
          pc.code AS "carrierCode",
          mg.name AS "groupName",
          mg.code AS "groupCode"
        FROM warehouse.packaging_materials pm
        LEFT JOIN production.production_locations pl ON pm.location_id = pl.id
        LEFT JOIN production.production_carriers pc ON pm.carrier_id = pc.id
        LEFT JOIN warehouse.material_groups mg ON pm.group_id = mg.id
        WHERE pm.id = $1
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Packaging material not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error fetching packaging material:", error);
      res.status(500).json({ error: "Failed to fetch packaging material" });
    }
  });
  
  // POST /api/warehouse/packaging-materials - Utwórz nowe opakowanie
  app.post("/api/warehouse/packaging-materials", isAuthenticated, async (req, res) => {
    try {
      // Preprocess decimal fields - convert numbers to strings for Drizzle decimal columns
      const preprocessedBody = {
        ...req.body,
        quantity: req.body.quantity !== undefined && req.body.quantity !== null ? String(req.body.quantity) : req.body.quantity,
        length: req.body.length !== undefined && req.body.length !== null ? String(req.body.length) : req.body.length,
        width: req.body.width !== undefined && req.body.width !== null ? String(req.body.width) : req.body.width,
        height: req.body.height !== undefined && req.body.height !== null ? String(req.body.height) : req.body.height,
      };
      
      const validation = insertPackagingMaterialSchema.safeParse(preprocessedBody);
      if (!validation.success) {
        return res.status(400).json({ error: "Invalid data", details: validation.error });
      }
      
      const {
        category,
        name,
        description,
        length,
        width,
        height,
        quantity,
        locationId,
        carrierId,
        supplierCode,
        notes,
        imageUrl
      } = validation.data;
      
      const result = await pool.query(`
        INSERT INTO warehouse.packaging_materials (
          category, name, description, length, width, height,
          quantity, unit, location_id, carrier_id, supplier_code, notes, image_url
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
        RETURNING 
          id,
          category,
          name,
          description,
          length,
          width,
          height,
          quantity,
          location_id AS "locationId",
          carrier_id AS "carrierId",
          supplier_code AS "supplierCode",
          notes,
          image_url AS "imageUrl",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        category, name, description || null,
        length || null, width || null, height || null,
        quantity || 0, unit || 'szt',
        locationId || null, carrierId || null,
        supplierCode || null, notes || null, imageUrl || null
      ]);
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating packaging material:", error);
      res.status(500).json({ error: "Failed to create packaging material" });
    }
  });
  
  // PATCH /api/warehouse/packaging-materials/:id - Aktualizuj opakowanie
  app.patch("/api/warehouse/packaging-materials/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      // Preprocess decimal fields - convert numbers to strings for Drizzle decimal columns
      const preprocessedBody = {
        ...req.body,
        quantity: req.body.quantity !== undefined && req.body.quantity !== null ? String(req.body.quantity) : req.body.quantity,
        length: req.body.length !== undefined && req.body.length !== null ? String(req.body.length) : req.body.length,
        width: req.body.width !== undefined && req.body.width !== null ? String(req.body.width) : req.body.width,
        height: req.body.height !== undefined && req.body.height !== null ? String(req.body.height) : req.body.height,
      };
      
      const validation = insertPackagingMaterialSchema.partial().safeParse(preprocessedBody);
      if (!validation.success) {
        return res.status(400).json({ error: "Invalid data", details: validation.error });
      }
      
      const {
        category,
        name,
        description,
        length,
        width,
        height,
        quantity,
        locationId,
        carrierId,
        groupId,
        supplierCode,
        notes,
        imageUrl,
        isActive
      } = validation.data;
      
      const result = await pool.query(`
        UPDATE warehouse.packaging_materials
        SET 
          category = COALESCE($1, category),
          name = COALESCE($2, name),
          description = COALESCE($3, description),
          length = COALESCE($4, length),
          width = COALESCE($5, width),
          height = COALESCE($6, height),
          quantity = COALESCE($7, quantity),
          unit = COALESCE($8, unit),
          location_id = CASE WHEN $9::INTEGER IS NULL THEN NULL ELSE $9 END,
          carrier_id = CASE WHEN $10::INTEGER IS NULL THEN NULL ELSE $10 END,
          group_id = CASE WHEN $11::INTEGER IS NULL THEN NULL ELSE $11 END,
          supplier_code = COALESCE($12, supplier_code),
          notes = COALESCE($13, notes),
          image_url = COALESCE($14, image_url),
          is_active = COALESCE($15, is_active),
          updated_at = NOW()
        WHERE id = $16
        RETURNING 
          id,
          category,
          name,
          description,
          length,
          width,
          height,
          quantity,
          location_id AS "locationId",
          carrier_id AS "carrierId",
          group_id AS "groupId",
          supplier_code AS "supplierCode",
          notes,
          image_url AS "imageUrl",
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        category, name, description,
        length, width, height,
        locationId, carrierId, groupId,
        supplierCode, notes, imageUrl,
        isActive,
        id
      ]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Packaging material not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating packaging material:", error);
      res.status(500).json({ error: "Failed to update packaging material" });
    }
  });
  
  // DELETE /api/warehouse/packaging-materials/:id - Usuń opakowanie
  app.delete("/api/warehouse/packaging-materials/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      await pool.query(
        'DELETE FROM warehouse.packaging_materials WHERE id = $1',
        [id]
      );
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting packaging material:", error);
      res.status(500).json({ error: "Failed to delete packaging material" });
    }
  });

  // POST /api/warehouse/packaging-materials/csv-import - Import opakowań z CSV
  app.post("/api/warehouse/packaging-materials/csv-import", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    
    try {
      const { csvData } = req.body;
      
      if (!csvData || typeof csvData !== 'string') {
        return res.status(400).json({ error: "CSV data is required" });
      }

      // Parse CSV using PapaParse
      const parseResult = Papa.parse(csvData, {
        header: true,
        skipEmptyLines: true,
        transformHeader: (header: string) => header.trim(),
      });

      if (parseResult.errors.length > 0) {
        return res.status(400).json({ 
          error: "CSV parsing failed",
          details: parseResult.errors.map(e => e.message)
        });
      }

      const rows = parseResult.data as any[];
      
      if (rows.length === 0) {
        return res.status(400).json({ error: "CSV file is empty" });
      }

      // Validate required columns
      const requiredColumns = ['category', 'name', 'unit'];
      const firstRow = rows[0];
      const missingColumns = requiredColumns.filter(col => !(col in firstRow));
      if (missingColumns.length > 0) {
        return res.status(400).json({ 
          error: `Missing required columns: ${missingColumns.join(', ')}` 
        });
      }

      // Fetch all locations for lookup
      const locationsResult = await pool.query(`
        SELECT id, code FROM production.production_locations
      `);
      const locationsMap = new Map<string, number>();
      locationsResult.rows.forEach((loc: any) => {
        locationsMap.set(loc.code, loc.id);
      });

      // Fetch all carriers for lookup
      const carriersResult = await pool.query(`
        SELECT id, code FROM production.production_carriers
      `);
      const carriersMap = new Map<string, number>();
      carriersResult.rows.forEach((car: any) => {
        carriersMap.set(car.code, car.id);
      });

      const results = {
        success: 0,
        failed: 0,
        errors: [] as Array<{ row: number; error: string; data?: any }>
      };

      // Start transaction
      await client.query('BEGIN');

      // Process each row
      for (let i = 0; i < rows.length; i++) {
        const row = rows[i];
        const rowNumber = i + 2; // +2 because: +1 for 1-indexing, +1 for header row

        try {
          // Validate required fields
          if (!row.category || row.category.trim() === '') {
            results.errors.push({
              row: rowNumber,
              error: 'Category is required',
              data: row
            });
            results.failed++;
            continue;
          }

          if (!row.name || row.name.trim() === '') {
            results.errors.push({
              row: rowNumber,
              error: 'Name is required',
              data: row
            });
            results.failed++;
            continue;
          }

          if (!row.unit || row.unit.trim() === '') {
            results.errors.push({
              row: rowNumber,
              error: 'Unit is required',
              data: row
            });
            results.failed++;
            continue;
          }

          // Validate category
          const validCategories = ['karton', 'wypełniacz', 'taśma', 'stretch', 'inne'];
          const category = row.category.trim().toLowerCase();
          if (!validCategories.includes(category)) {
            results.errors.push({
              row: rowNumber,
              error: `Invalid category: ${row.category}. Must be one of: ${validCategories.join(', ')}`,
              data: row
            });
            results.failed++;
            continue;
          }

          // Resolve location_id from location_code
          let locationId = null;
          if (row.location_code && row.location_code.trim() !== '') {
            locationId = locationsMap.get(row.location_code.trim());
            if (locationId === undefined) {
              results.errors.push({
                row: rowNumber,
                error: `Location code not found: ${row.location_code}`,
                data: row
              });
              results.failed++;
              continue;
            }
          }

          // Resolve carrier_id from carrier_code
          let carrierId = null;
          if (row.carrier_code && row.carrier_code.trim() !== '') {
            carrierId = carriersMap.get(row.carrier_code.trim());
            if (carrierId === undefined) {
              results.errors.push({
                row: rowNumber,
                error: `Carrier code not found: ${row.carrier_code}`,
                data: row
              });
              results.failed++;
              continue;
            }
          }

          // Decimal preprocessing (convert number to string for Drizzle decimal columns)
          const length = row.length && row.length !== '' ? String(safeParseNumber(row.length)) : null;
          const width = row.width && row.width !== '' ? String(safeParseNumber(row.width)) : null;
          const height = row.height && row.height !== '' ? String(safeParseNumber(row.height)) : null;
          const quantity = row.quantity && row.quantity !== '' ? String(safeParseNumber(row.quantity)) : '0';

          // Insert or update (upsert) based on category + name
          await client.query(`
            INSERT INTO warehouse.packaging_materials (
              category, name, description,
              length, width, height,
              location_id, carrier_id,
              supplier_code, notes, image_url,
              is_active
            )
            VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14)
            ON CONFLICT (category, name) 
            DO UPDATE SET
              description = EXCLUDED.description,
              length = EXCLUDED.length,
              width = EXCLUDED.width,
              height = EXCLUDED.height,
              quantity = EXCLUDED.quantity,
              location_id = EXCLUDED.location_id,
              carrier_id = EXCLUDED.carrier_id,
              supplier_code = EXCLUDED.supplier_code,
              notes = EXCLUDED.notes,
              image_url = EXCLUDED.image_url,
              is_active = EXCLUDED.is_active,
              updated_at = CURRENT_TIMESTAMP
          `, [
            category,
            row.name.trim(),
            row.description?.trim() || null,
            length,
            width,
            height,
            quantity,
            row.unit.trim(),
            locationId,
            carrierId,
            row.supplier_code?.trim() || null,
            row.notes?.trim() || null,
            row.image_url?.trim() || null,
            row.is_active !== undefined ? (row.is_active === 'true' || row.is_active === '1' || row.is_active === true) : true
          ]);

          results.success++;

          results.success++;
        } catch (error: any) {
          console.error(`❌ Error importing row ${rowNumber}:`, error);
          
          results.errors.push({
            row: rowNumber,
            error: error instanceof Error ? error.message : 'Unknown error',
            data: row
          });
          results.failed++;
        }
      }

      // Always commit - upsert handles conflicts gracefully
      await client.query('COMMIT');

      // Return results with appropriate status
      if (results.failed > 0 && results.success === 0) {
        return res.status(400).json({
          error: "Import failed - all rows had errors",
          ...results
        });
      }

      res.json({
        message: results.failed > 0 
          ? `Import completed with warnings: ${results.success} succeeded, ${results.failed} failed`
          : `Import completed successfully: ${results.success} packaging materials imported`,
        ...results
      });
    } catch (error) {
      // Rollback transaction on error
      await client.query('ROLLBACK');
      console.error("❌ Error importing CSV:", error);
      res.status(500).json({ error: "Failed to import CSV" });
    } finally {
      client.release();
    }
  });

  // ===== WAREHOUSE PACKED PRODUCTS =====
  
  // GET /api/warehouse/packed-products - Lista produktów spakowanych
  app.get("/api/warehouse/packed-products", isAuthenticated, async (req, res) => {
    try {
      const { 
        search, 
        productType,
        locationId,
        carrierId,
        page = 1, 
        limit = 50,
        sortBy = 'created_at',
        sortOrder = 'DESC',
        include,
        archived
      } = req.query;
      
      const includeBom = include && (include === 'bom' || include === 'all');
      
      const offset = (Number(page) - 1) * Number(limit);
      
      // Build WHERE clauses
      const showArchived = archived === 'true';
      const whereClauses: string[] = showArchived ? ['pp.is_archived = true'] : ['pp.is_active = true', 'pp.is_archived = false'];
      const params: any[] = [];
      let paramIndex = 1;
      
      if (search) {
        whereClauses.push(`(
          pp.product_sku ILIKE $${paramIndex} OR
          pp.product_name ILIKE $${paramIndex} OR
          pp.notes ILIKE $${paramIndex}
        )`);
        params.push(`%${search}%`);
        paramIndex++;
      }
      
      if (productType) {
        whereClauses.push(`pp.product_type = $${paramIndex}`);
        params.push(productType);
        paramIndex++;
      }
      
      if (locationId) {
        whereClauses.push(`pp.location_id = $${paramIndex}`);
        params.push(locationId);
        paramIndex++;
      }
      
      if (carrierId) {
        whereClauses.push(`pp.carrier_id = $${paramIndex}`);
        params.push(carrierId);
        paramIndex++;
      }
      
      const whereClause = whereClauses.length > 0 ? 'WHERE ' + whereClauses.join(' AND ') : '';
      
      // Allowed sort columns
      const allowedSortBy = ['product_sku', 'product_name', 'quantity', 'created_at'];
      const sortColumn = allowedSortBy.includes(sortBy as string) ? sortBy : 'created_at';
      const sortDirection = sortOrder === 'ASC' ? 'ASC' : 'DESC';
      
      // Get total count
      const countResult = await pool.query(`
        SELECT COUNT(*) as total
        FROM warehouse.packed_products pp
        ${whereClause}
      `, params);
      
      const total = parseInt(countResult.rows[0].total, 10);
      
      // Get paginated results with location and carrier names
      const result = await pool.query(`
        SELECT 
          pp.id,
          pp.catalog_product_id AS "catalogProductId",
          pp.catalog_set_id AS "catalogSetId",
          pp.product_sku AS "productSku",
          pp.product_name AS "productName",
          pp.product_type AS "productType",
          pp.quantity,
            pp.reserved_quantity AS "reservedQuantity",
            (pp.quantity - pp.reserved_quantity) AS "availableQuantity",
          pp.location_id AS "locationId",
          pp.carrier_id AS "carrierId",
          pp.image_url AS "imageUrl",
          pp.notes,
          pp.is_active AS "isActive",
          pp.is_archived AS "isArchived",
          pp.created_at AS "createdAt",
          pp.updated_at AS "updatedAt",
          pl.name AS "locationName",
          pl.code AS "locationCode",
          pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
          pc.code AS "carrierCode",
          (
            SELECT COUNT(*)
            FROM warehouse.packed_product_packages ppp
            WHERE ppp.packed_product_id = pp.id
          ) AS "packageCount"
          ${includeBom ? `, (
            SELECT COUNT(*)
            FROM bom.product_components pcomp
            INNER JOIN bom.product_boms pb ON pcomp.product_bom_id = pb.id
            WHERE pb.product_id = pp.catalog_product_id
          ) AS "bomComponentsCount"` : ''}
        FROM warehouse.packed_products pp
        LEFT JOIN production.production_locations pl ON pl.id = pp.location_id
        LEFT JOIN production.production_carriers pc ON pc.id = pp.carrier_id
        ${whereClause}
        ORDER BY pp.${sortColumn} ${sortDirection}
        LIMIT $${paramIndex} OFFSET $${paramIndex + 1}
      `, [...params, Number(limit), offset]);
      
      res.json({
        data: result.rows,
        pagination: {
          total,
          page: Number(page),
          limit: Number(limit),
          totalPages: Math.ceil(total / Number(limit))
        }
      });
    } catch (error) {
      console.error("❌ Error fetching packed products:", error);
      res.status(500).json({ error: "Failed to fetch packed products" });
    }
  });
  
  // GET /api/warehouse/packed-products/:id - Pojedynczy produkt spakowany
  app.get("/api/warehouse/packed-products/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { include } = req.query;
      
      const includeBom = include && (include === 'bom' || include === 'all');
      
      const result = await pool.query(`
        SELECT 
          pp.id,
          pp.catalog_product_id AS "catalogProductId",
          pp.catalog_set_id AS "catalogSetId",
          pp.product_sku AS "productSku",
          pp.product_name AS "productName",
          pp.product_type AS "productType",
          pp.quantity,
            pp.reserved_quantity AS "reservedQuantity",
            (pp.quantity - pp.reserved_quantity) AS "availableQuantity",
          pp.location_id AS "locationId",
          pp.carrier_id AS "carrierId",
          pp.image_url AS "imageUrl",
          pp.notes,
          pp.is_active AS "isActive",
          pp.is_archived AS "isArchived",
          pp.created_at AS "createdAt",
          pp.updated_at AS "updatedAt",
          pl.name AS "locationName",
          pl.code AS "locationCode",
          pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
          pc.code AS "carrierCode",
          COALESCE(cp.gallery, '[]'::jsonb) AS "catalogGallery",
          cp.length AS "catalogLength",
          cp.width AS "catalogWidth",
          cp.height AS "catalogHeight",
          cp.weight AS "catalogWeight",
          cp.color AS "catalogColor",
          COALESCE(cp.color_options, ARRAY[]::text[]) AS "catalogColorOptions",
          cp.material AS "catalogMaterial",
          cp.product_type AS "catalogProductType",
          cp.product_group AS "catalogProductGroup",
          cp.doors AS "catalogDoors",
          cp.legs AS "catalogLegs",
          COALESCE(cs.images, ARRAY[]::text[]) AS "catalogSetImages",
          cs.color AS "catalogSetColor",
          COALESCE(cs.color_options, ARRAY[]::text[]) AS "catalogSetColorOptions",
          cs.depth AS "catalogSetDepth",
          cs.panel_count AS "catalogSetPanelCount",
          cs.hook_length AS "catalogSetHookLength"
        FROM warehouse.packed_products pp
        LEFT JOIN production.production_locations pl ON pl.id = pp.location_id
        LEFT JOIN production.production_carriers pc ON pc.id = pp.carrier_id
        LEFT JOIN catalog.products cp ON pp.catalog_product_id = cp.id
        LEFT JOIN product_creator.product_sets cs ON pp.catalog_set_id = cs.id
        WHERE pp.id = $1
      `, [id]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Packed product not found" });
      }
      
      const product = result.rows[0];
      
      // Fetch marketplace platform links for either catalogProductId or catalogSetId
      if (product.catalogProductId) {
        const platformLinksResult = await pool.query(`
          SELECT 
            ppd.id,
            ppd.platform,
            ppd.link_type AS "linkType",
            ppd.set_id AS "setId",
            ppd.custom_title AS "customTitle",
            ppd.category_id AS "categoryId",
            ppd.category_name AS "categoryName",
            ppd.external_id AS "externalId"
          FROM catalog.product_platform_data ppd
          WHERE ppd.product_id = $1 AND ppd.link_type = 'product'
          ORDER BY ppd.platform
        `, [product.catalogProductId]);
        
        product.platformLinks = platformLinksResult.rows;
      } else if (product.catalogSetId) {
        const platformLinksResult = await pool.query(`
          SELECT 
            ppd.id,
            ppd.platform,
            ppd.link_type AS "linkType",
            ppd.set_id AS "setId",
            ppd.custom_title AS "customTitle",
            ppd.category_id AS "categoryId",
            ppd.category_name AS "categoryName",
            ppd.external_id AS "externalId"
          FROM catalog.product_platform_data ppd
          WHERE ppd.set_id = $1 AND ppd.link_type = 'set'
          ORDER BY ppd.platform
        `, [product.catalogSetId]);
        
        product.platformLinks = platformLinksResult.rows;
      } else {
        product.platformLinks = [];
      }
      
      // Fetch BOM components if requested and catalogProductId exists
      if (includeBom && product.catalogProductId) {
        const bomResult = await pool.query(`
          SELECT 
            pcomp.id,
            pb.product_id AS "productId",
            pcomp.generated_name AS "generatedName",
            pcomp.component_template_id AS "templateId",
            pcomp.calculated_length AS "calculatedLength",
            pcomp.calculated_width AS "calculatedWidth",
            pcomp.thickness,
            pcomp.quantity,
            pcomp.color AS "colorCode",
            pcomp.notes,
            pcomp.edging_pattern AS "edgingPattern",
            pcomp.edging_material AS "edgingMaterial",
            pcomp.position_in_bom AS "positionInBom",
            ct.furniture_type AS "furnitureType",
            ct.component_type AS "componentType"
          FROM bom.product_components pcomp
          INNER JOIN bom.product_boms pb ON pcomp.product_bom_id = pb.id
          LEFT JOIN bom.component_templates ct ON pcomp.component_template_id = ct.id
          WHERE pb.product_id = $1
          ORDER BY pcomp.position_in_bom
        `, [product.catalogProductId]);
        
        product.bomComponents = bomResult.rows;
      }
      
      res.json(product);
    } catch (error) {
      console.error("❌ Error fetching packed product:", error);
      res.status(500).json({ error: "Failed to fetch packed product" });
    }
  });
  
  // POST /api/warehouse/packed-products - Utwórz nowy produkt spakowany
  app.post("/api/warehouse/packed-products", isAuthenticated, async (req, res) => {
    try {
      const {
        catalogProductId,
        catalogSetId,
        productSku,
        productName,
        productType,
        quantity,
        locationId,
        carrierId,
        imageUrl,
        notes
      } = req.body;
      
      if (!productSku || !productName) {
        return res.status(400).json({ error: "Product SKU and name are required" });
      }
      
      // Ensure only one of catalogProductId or catalogSetId is set
      if (catalogProductId && catalogSetId) {
        return res.status(400).json({ error: "Cannot link to both catalog product and set" });
      }
      
      if (!catalogProductId && !catalogSetId) {
        return res.status(400).json({ error: "Must link to either catalog product or set" });
      }
      
      const result = await pool.query(`
        INSERT INTO warehouse.packed_products (
          catalog_product_id, catalog_set_id, product_sku, product_name, product_type,
          quantity, location_id, carrier_id, image_url, notes
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING 
          id,
          catalog_product_id AS "catalogProductId",
          catalog_set_id AS "catalogSetId",
          product_sku AS "productSku",
          product_name AS "productName",
          product_type AS "productType",
          quantity,
          location_id AS "locationId",
          carrier_id AS "carrierId",
          image_url AS "imageUrl",
          notes,
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        catalogProductId || null,
        catalogSetId || null,
        productSku,
        productName,
        productType || 'product',
        quantity || 0,
        locationId || null,
        carrierId || null,
        imageUrl || null,
        notes || null
      ]);
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating packed product:", error);
      res.status(500).json({ error: "Failed to create packed product" });
    }
  });
  
  // PATCH /api/warehouse/packed-products/:id - Aktualizuj produkt spakowany
  app.patch("/api/warehouse/packed-products/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const {
        productSku,
        productName,
        quantity,
        locationId,
        carrierId,
        groupId,
        imageUrl,
        notes,
        isActive,
      } = req.body;
      const result = await pool.query(`
        UPDATE warehouse.packed_products
        SET 
          product_sku = COALESCE($1, product_sku),
          product_name = COALESCE($2, product_name),
          quantity = COALESCE($3, quantity),
          location_id = COALESCE($4, location_id),
          carrier_id = COALESCE($5, carrier_id),
          group_id = CASE WHEN $6::INTEGER IS NULL THEN NULL ELSE $6 END,
          image_url = COALESCE($7, image_url),
          notes = COALESCE($8, notes),
          is_active = COALESCE($9, is_active),
          external_symbol = COALESCE($10, external_symbol),
          updated_at = NOW()
        WHERE id = $11
        RETURNING 
          id,
          catalog_product_id AS "catalogProductId",
          catalog_set_id AS "catalogSetId",
          product_sku AS "productSku",
          product_name AS "productName",
          product_type AS "productType",
          quantity,
          location_id AS "locationId",
          carrier_id AS "carrierId",
          group_id AS "groupId",
          image_url AS "imageUrl",
          notes,
          is_active AS "isActive",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        productSku,
        productName,
        quantity,
        locationId,
        carrierId,
        groupId,
        imageUrl,
        notes,
        isActive,
        id
      ]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Packed product not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating packed product:", error);
      res.status(500).json({ error: "Failed to update packed product" });
    }
  });
  
  // DELETE /api/warehouse/packed-products/:id - Usuń produkt spakowany
  app.delete("/api/warehouse/packed-products/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      await pool.query(
        'DELETE FROM warehouse.packed_products WHERE id = $1',
        [id]
      );
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting packed product:", error);
      res.status(500).json({ error: "Failed to delete packed product" });
    }
  });
  
  // POST /api/warehouse/packed-products/bulk-update - Bulk update ilości produktów spakowanych
  app.post("/api/warehouse/packed-products/bulk-update", isAuthenticated, async (req, res) => {
    try {
      const { ids, mode, quantity } = req.body;
      
      if (!ids || !Array.isArray(ids) || ids.length === 0) {
        return res.status(400).json({ error: "IDs array is required" });
      }
      
      if (!mode || !['set', 'increment', 'decrement'].includes(mode)) {
        return res.status(400).json({ error: "Mode must be 'set', 'increment', or 'decrement'" });
      }
      
      const qty = parseFloat(quantity);
      if (isNaN(qty)) {
        return res.status(400).json({ error: "Quantity must be a valid number" });
      }
      
      // Build SQL based on mode
      let updateQuery = '';
      if (mode === 'set') {
        updateQuery = `
          UPDATE warehouse.packed_products 
          SET quantity = $1, updated_at = CURRENT_TIMESTAMP
          WHERE id = ANY($2::int[])
          RETURNING id, product_sku AS "productSku", quantity
        `;
      } else if (mode === 'increment') {
        updateQuery = `
          UPDATE warehouse.packed_products 
          SET quantity = quantity + $1, updated_at = CURRENT_TIMESTAMP
          WHERE id = ANY($2::int[])
          RETURNING id, product_sku AS "productSku", quantity
        `;
      } else if (mode === 'decrement') {
        updateQuery = `
          UPDATE warehouse.packed_products 
          SET quantity = GREATEST(0, quantity - $1), updated_at = CURRENT_TIMESTAMP
          WHERE id = ANY($2::int[])
          RETURNING id, product_sku AS "productSku", quantity
        `;
      }
      
      const result = await pool.query(updateQuery, [qty, ids]);
      
      res.json({ 
        success: true, 
        updated: result.rows.length,
        products: result.rows
      });
    } catch (error) {
      console.error("❌ Error bulk updating packed products:", error);
      res.status(500).json({ error: "Failed to bulk update packed products" });
    }
  });
  
  // GET /api/warehouse/packed-products/:id/packages - Lista paczek produktu spakowanego
  app.get("/api/warehouse/packed-products/:id/packages", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      const result = await pool.query(`
        SELECT 
          ppp.id,
          ppp.packed_product_id AS "packedProductId",
          ppp.package_number AS "packageNumber",
          ppp.package_name AS "packageName",
          ppp.length,
          ppp.width,
          ppp.height,
          ppp.weight,
          ppp.location_id AS "locationId",
          ppp.carrier_id AS "carrierId",
          ppp.notes,
          ppp.created_at AS "createdAt",
          ppp.updated_at AS "updatedAt",
          pl.name AS "locationName",
          pl.code AS "locationCode",
          pc.name AS "carrierName",
            pp.catalog_product_id AS "catalogProductId",
            cp.sku AS "catalogProductSku",
            cp.length AS "catalogLength",
            cp.width AS "catalogWidth",
            cp.color AS "catalogColor",
            cp.product_type AS "catalogProductType",
            cp.doors AS "catalogDoors",
            cp.legs AS "catalogLegs",
          pc.code AS "carrierCode"
        FROM warehouse.packed_product_packages ppp
        LEFT JOIN production.production_locations pl ON pl.id = ppp.location_id
        LEFT JOIN production.production_carriers pc ON pc.id = ppp.carrier_id
        WHERE ppp.packed_product_id = $1
        ORDER BY ppp.package_number
      `, [id]);
      
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching packed product packages:", error);
      res.status(500).json({ error: "Failed to fetch packages" });
    }
  });
  
  // POST /api/warehouse/packed-products/:id/packages - Dodaj paczkę do produktu spakowanego
  app.post("/api/warehouse/packed-products/:id/packages", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const {
        packageNumber,
        packageName,
        length,
        width,
        height,
        weight,
        locationId,
        carrierId,
        notes
      } = req.body;
      
      if (!packageNumber) {
        return res.status(400).json({ error: "Package number is required" });
      }
      
      const result = await pool.query(`
        INSERT INTO warehouse.packed_product_packages (
          packed_product_id, package_number, package_name,
          length, width, height, weight,
          location_id, carrier_id, notes
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING 
          id,
          packed_product_id AS "packedProductId",
          package_number AS "packageNumber",
          package_name AS "packageName",
          length,
          width,
          height,
          weight,
          location_id AS "locationId",
          carrier_id AS "carrierId",
          notes,
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        id,
        packageNumber,
        packageName || null,
        length || null,
        width || null,
        height || null,
        weight || null,
        locationId || null,
        carrierId || null,
        notes || null
      ]);
      
      res.json(result.rows[0]);
    } catch (error: any) {
      console.error("❌ Error creating package:", error);
      
      // Check for unique constraint violation
      if (error.code === '23505' && error.constraint === 'idx_packed_product_packages_unique') {
        return res.status(400).json({ error: "Package with this number already exists for this product" });
      }
      
      res.status(500).json({ error: "Failed to create package" });
    }
  });
  
  // PATCH /api/warehouse/packed-product-packages/:id - Aktualizuj paczkę
  app.patch("/api/warehouse/packed-product-packages/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const {
        packageNumber,
        packageName,
        length,
        width,
        height,
        weight,
        locationId,
        carrierId,
        notes
      } = req.body;
      
      const result = await pool.query(`
        UPDATE warehouse.packed_product_packages
        SET 
          package_number = COALESCE($1, package_number),
          package_name = COALESCE($2, package_name),
          length = COALESCE($3, length),
          width = COALESCE($4, width),
          height = COALESCE($5, height),
          weight = COALESCE($6, weight),
          location_id = COALESCE($7, location_id),
          carrier_id = COALESCE($8, carrier_id),
          notes = COALESCE($9, notes),
          updated_at = NOW()
        WHERE id = $11
        RETURNING 
          id,
          packed_product_id AS "packedProductId",
          package_number AS "packageNumber",
          package_name AS "packageName",
          length,
          width,
          height,
          weight,
          location_id AS "locationId",
          carrier_id AS "carrierId",
          notes,
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        packageNumber,
        packageName,
        length,
        width,
        height,
        weight,
        locationId,
        carrierId,
        notes,
        id
      ]);
      
      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Package not found" });
      }
      
      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error updating package:", error);
      res.status(500).json({ error: "Failed to update package" });
    }
  });
  
  // DELETE /api/warehouse/packed-product-packages/:id - Usuń paczkę
  app.delete("/api/warehouse/packed-product-packages/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      
      await pool.query(
        'DELETE FROM warehouse.packed_product_packages WHERE id = $1',
        [id]
      );
      
      res.json({ success: true });
    } catch (error) {
      console.error("❌ Error deleting package:", error);
      res.status(500).json({ error: "Failed to delete package" });
    }
  });

  // ===== PRODUCTION LOCATIONS & CARRIERS =====
  
  // GET /api/production/locations - Lista lokalizacji produkcyjnych (dla warehouse location selector)
  app.get("/api/production/locations", isAuthenticated, async (req, res) => {
    try {
      const { parentId, level } = req.query;
      
      let query = `
        SELECT 
          pl.id,
          pl.location_group_id AS "locationGroupId",
          pl.parent_id AS "parentId",
          pl.code,
          pl.name,
          pl.path,
          pl.level,
          pl.barcode,
          pl.capacity,
          pl.capacity_unit AS "capacityUnit",
          pl.current_load AS "currentLoad",
          pl.dimensions,
          pl.status,
          pl.allows_storage AS "allowsStorage",
          pl.is_active AS "isActive",
          pl.notes,
          plg.name AS "groupName",
          plg.location_type AS "locationType"
        FROM production.production_locations pl
        LEFT JOIN production.production_location_groups plg ON pl.location_group_id = plg.id
        WHERE pl.is_active = true
      `;
      
      const params: any[] = [];
      let paramIndex = 1;
      
      if (parentId !== undefined) {
        if (parentId === 'null' || parentId === '') {
          query += ' AND pl.parent_id IS NULL';
        } else {
          query += ` AND pl.parent_id = $${paramIndex}`;
          params.push(parentId);
          paramIndex++;
        }
      }
      
      if (level !== undefined) {
        query += ` AND pl.level = $${paramIndex}`;
        params.push(level);
        paramIndex++;
      }
      
      query += ' ORDER BY pl.path, pl.code';
      
      const result = await pool.query(query, params);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching production locations:", error);
      res.status(500).json({ error: "Failed to fetch production locations" });
    }
  });
  
  // GET /api/production/carriers - Lista nośników produkcyjnych (Regały, Wózki, Palety)
  app.get("/api/production/carriers", isAuthenticated, async (req, res) => {
    try {
      const query = `
        SELECT 
          pc.id,
          pc.carrier_group_id AS "carrierGroupId",
          pc.code,
          pc.name,
          pc.barcode,
          pc.status,
          pc.capacity,
          pc.capacity_unit AS "capacityUnit",
          pc.current_load AS "currentLoad",
          pc.dimensions,
          pc.weight,
          pc.current_location_id AS "currentLocationId",
          pc.default_location_id AS "defaultLocationId",
          pc.last_maintenance_date AS "lastMaintenanceDate",
          pc.next_maintenance_date AS "nextMaintenanceDate",
          pc.notes,
          pc.is_active AS "isActive",
          pcg.name AS "groupName",
          pcg.code AS "groupCode",
          pl.name AS "currentLocationName",
          pl.code AS "currentLocationCode",
          CASE WHEN EXISTS (
            SELECT 1 FROM production.production_order_pallets pop
            WHERE pop.carrier_id = pc.id 
            AND pop.status NOT IN ('completed', 'cancelled')
          ) THEN true ELSE false END AS "isOccupied",
          (
            SELECT po.order_number 
            FROM production.production_order_pallets pop
            JOIN production.production_orders po ON pop.production_order_id = po.id
            WHERE pop.carrier_id = pc.id 
            AND pop.status NOT IN ('completed', 'cancelled')
            LIMIT 1
          ) AS "occupiedByZlp"
        FROM production.production_carriers pc
        LEFT JOIN production.production_carrier_groups pcg ON pc.carrier_group_id = pcg.id
        LEFT JOIN production.production_locations pl ON pc.current_location_id = pl.id
        WHERE pc.is_active = true
        ORDER BY pcg.name, pc.code
      `;
      
      const result = await pool.query(query);
      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching production carriers:", error);
      res.status(500).json({ error: "Failed to fetch production carriers" });
    }
  });
  
  // GET /api/warehouse/location-hierarchy - Hierarchia lokalizacji dla warehouse UI
  app.get("/api/warehouse/location-hierarchy", isAuthenticated, async (req, res) => {
    try {
      // Get halls (level 0 locations)
      const hallsResult = await pool.query(`
        SELECT 
          pl.id,
          pl.code,
          pl.name,
          pl.path,
          plg.location_type AS "locationType"
        FROM production.production_locations pl
        LEFT JOIN production.production_location_groups plg ON pl.location_group_id = plg.id
        WHERE pl.is_active = true AND pl.level = 0
        ORDER BY pl.code
      `);
      
      // Get all carriers
      const carriersResult = await pool.query(`
        SELECT 
          pc.id,
          pc.code,
          pc.name,
          pc.current_location_id AS "currentLocationId",
          pcg.name AS "type"
        FROM production.production_carriers pc
        LEFT JOIN production.production_carrier_groups pcg ON pc.carrier_group_id = pcg.id
        WHERE pc.is_active = true AND pc.status = 'available'
        ORDER BY pcg.name, pc.code
      `);
      
      // Get all child locations (slots, shelves)
      const slotsResult = await pool.query(`
        SELECT 
          pl.id,
          pl.parent_id AS "parentId",
          pl.code,
          pl.name,
          pl.path,
          pl.level,
          plg.location_type AS "locationType"
        FROM production.production_locations pl
        LEFT JOIN production.production_location_groups plg ON pl.location_group_id = plg.id
        WHERE pl.is_active = true AND pl.level > 0
        ORDER BY pl.path, pl.code
      `);
      
      res.json({
        halls: hallsResult.rows,
        carriers: carriersResult.rows,
        slots: slotsResult.rows
      });
    } catch (error) {
      console.error("❌ Error fetching location hierarchy:", error);
      res.status(500).json({ error: "Failed to fetch location hierarchy" });
    }
  });

  // POST /api/production/locations - Utwórz nową lokalizację
  app.post("/api/production/locations", isAuthenticated, async (req, res) => {
    try {
      const {
        locationGroupId,
        parentId,
        code,
        name,
        barcode,
        capacity,
        capacityUnit,
        dimensions,
        status,
        allowsStorage,
        notes
      } = req.body;

      if (!code || !name || !locationGroupId) {
        return res.status(400).json({ error: "Code, name, and location group ID are required" });
      }

      // Calculate path and level based on parent
      let path = code;
      let level = 0;

      if (parentId) {
        const parentResult = await pool.query(
          'SELECT path, level FROM production.production_locations WHERE id = $1',
          [parentId]
        );
        
        if (parentResult.rows.length === 0) {
          return res.status(404).json({ error: "Parent location not found" });
        }
        
        const parent = parentResult.rows[0];
        path = `${parent.path}/${code}`;
        level = parent.level + 1;
      }

      const result = await pool.query(`
        INSERT INTO production.production_locations (
          location_group_id, parent_id, code, name, path, level,
          barcode, capacity, capacity_unit, dimensions, status, allows_storage, notes
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13)
        RETURNING 
          id,
          location_group_id AS "locationGroupId",
          parent_id AS "parentId",
          code,
          name,
          path,
          level,
          barcode,
          capacity,
          capacity_unit AS "capacityUnit",
          dimensions,
          status,
          allows_storage AS "allowsStorage",
          is_active AS "isActive",
          notes,
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        locationGroupId,
        parentId || null,
        code,
        name,
        path,
        level,
        barcode || null,
        capacity || null,
        capacityUnit || null,
        dimensions || null,
        status || 'active',
        allowsStorage !== false,
        notes || null
      ]);

      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating location:", error);
      res.status(500).json({ error: "Failed to create location" });
    }
  });

  // POST /api/production/carriers - Utwórz nowy nośnik
  app.post("/api/production/carriers", isAuthenticated, async (req, res) => {
    try {
      const {
        carrierGroupId,
        code,
        name,
        barcode,
        status,
        capacity,
        capacityUnit,
        dimensions,
        weight,
        currentLocationId,
        defaultLocationId,
        notes
      } = req.body;

      if (!code || !name || !carrierGroupId) {
        return res.status(400).json({ error: "Code, name, and carrier group ID are required" });
      }

      const result = await pool.query(`
        INSERT INTO production.production_carriers (
          carrier_group_id, code, name, barcode, status,
          capacity, capacity_unit, dimensions, weight,
          current_location_id, default_location_id, notes
        )
        VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12)
        RETURNING 
          id,
          carrier_group_id AS "carrierGroupId",
          code,
          name,
          barcode,
          status,
          capacity,
          capacity_unit AS "capacityUnit",
          current_load AS "currentLoad",
          dimensions,
          weight,
          current_location_id AS "currentLocationId",
          default_location_id AS "defaultLocationId",
          is_active AS "isActive",
          notes,
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        carrierGroupId,
        code,
        name,
        barcode || null,
        status || 'available',
        capacity || null,
        capacityUnit || null,
        dimensions || null,
        weight || null,
        currentLocationId || null,
        defaultLocationId || null,
        notes || null
      ]);

      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating carrier:", error);
      res.status(500).json({ error: "Failed to create carrier" });
    }
  });

  // POST /api/production/location-groups - Utwórz nową grupę lokalizacji
  app.post("/api/production/location-groups", isAuthenticated, async (req, res) => {
    try {
      const {
        code,
        name,
        description,
        locationType,
        allowsSublocations,
        sortOrder
      } = req.body;

      if (!code || !name || !locationType) {
        return res.status(400).json({ error: "Code, name, and location type are required" });
      }

      const result = await pool.query(`
        INSERT INTO production.production_location_groups (
          code, name, description, location_type, allows_sublocations, sort_order
        )
        VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING 
          id,
          code,
          name,
          description,
          location_type AS "locationType",
          allows_sublocations AS "allowsSublocations",
          is_active AS "isActive",
          sort_order AS "sortOrder",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        code,
        name,
        description || null,
        locationType,
        allowsSublocations !== false,
        sortOrder || 100
      ]);

      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating location group:", error);
      res.status(500).json({ error: "Failed to create location group" });
    }
  });

  // POST /api/production/carrier-groups - Utwórz nową grupę nośników
  // Alias: POST /api/production/carriers/groups (dla kompatybilności z frontendem)
  const createCarrierGroupHandler = async (req: any, res: any) => {
    try {
      const {
        code,
        name,
        description,
        defaultCapacity,
        capacityUnit,
        sortOrder
      } = req.body;

      if (!code || !name) {
        return res.status(400).json({ error: "Code and name are required" });
      }

      const result = await pool.query(`
        INSERT INTO production.production_carrier_groups (
          code, name, description, default_capacity, capacity_unit, sort_order
        )
        VALUES ($1, $2, $3, $4, $5, $6)
        RETURNING 
          id,
          code,
          name,
          description,
          default_capacity AS "defaultCapacity",
          capacity_unit AS "capacityUnit",
          is_active AS "isActive",
          sort_order AS "sortOrder",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
      `, [
        code,
        name,
        description || null,
        defaultCapacity || null,
        capacityUnit || null,
        sortOrder || 100
      ]);

      res.json(result.rows[0]);
    } catch (error) {
      console.error("❌ Error creating carrier group:", error);
      res.status(500).json({ error: "Failed to create carrier group" });
    }
  };
  
  app.post("/api/production/carrier-groups", isAuthenticated, createCarrierGroupHandler);
  app.post("/api/production/carriers/groups", isAuthenticated, createCarrierGroupHandler);

  // POST /api/production/orders/:orderId/pallets - Dodaj nową paletę do zlecenia produkcyjnego (ZLP)
  app.post("/api/production/orders/:orderId/pallets", isAuthenticated, async (req, res) => {
    try {
      const orderId = parseInt(req.params.orderId);
      const { flowCode, carrierId, notes, splitAfterOperation = 'edging' } = req.body;

      if (!orderId || isNaN(orderId)) {
        return res.status(400).json({ error: "Wymagane ID zlecenia (orderId)" });
      }

      if (!flowCode) {
        return res.status(400).json({ error: "Wymagany kod przepływu (flowCode)" });
      }

      const orderCheck = await pool.query(
        `SELECT id, order_number FROM production.production_orders WHERE id = $1`,
        [orderId]
      );

      if (orderCheck.rows.length === 0) {
        return res.status(404).json({ error: "Nie znaleziono zlecenia o podanym ID" });
      }

      const zlpNumber = orderCheck.rows[0].order_number;

      const seqResult = await pool.query(
        `SELECT COALESCE(MAX(sequence), 0) + 1 as next_seq FROM production.production_order_pallets WHERE production_order_id = $1`,
        [orderId]
      );
      const nextSeq = seqResult.rows[0].next_seq;

      const palletLabel = `${zlpNumber}-${flowCode}-${nextSeq}`;

      const result = await pool.query(`
        INSERT INTO production.production_order_pallets (
          production_order_id, pallet_label, sequence, flow_code,
          carrier_id, status, split_after_operation, notes, created_at, updated_at
        )
        VALUES ($1, $2, $3, $4, $5, 'pending', $6, $7, NOW(), NOW())
        RETURNING 
          id, production_order_id AS "productionOrderId", pallet_label AS "palletLabel",
          sequence, flow_code AS "flowCode", carrier_id AS "carrierId", status,
          split_after_operation AS "splitAfterOperation", notes, created_at AS "createdAt"
      `, [orderId, palletLabel, nextSeq, flowCode, carrierId || null, splitAfterOperation, notes || null]);

      res.json(result.rows[0]);
    } catch (error) {
      console.error("[POST /api/production/orders/:orderId/pallets] Error:", error);
      res.status(500).json({ error: "Błąd podczas tworzenia palety" });
    }
  });

  // GET /api/production/orders/:orderId/pallets - Pobierz palety zlecenia
  app.get("/api/production/orders/:orderId/pallets", isAuthenticated, async (req, res) => {
    try {
      const orderId = parseInt(req.params.orderId);

      if (!orderId || isNaN(orderId)) {
        return res.status(400).json({ error: "Wymagane ID zlecenia (orderId)" });
      }

      const result = await pool.query(`
        SELECT 
          pop.id, pop.production_order_id AS "productionOrderId",
          pop.pallet_label AS "palletLabel", pop.sequence,
          pop.flow_code AS "flowCode", pop.carrier_id AS "carrierId",
          pop.status, pop.split_after_operation AS "splitAfterOperation",
          pop.component_count AS "componentCount", pop.notes,
          pop.created_at AS "createdAt",
          pc.code AS "carrierCode", pc.name AS "carrierName", pc.barcode AS "carrierBarcode"
        FROM production.production_order_pallets pop
        LEFT JOIN production.production_carriers pc ON pc.id = pop.carrier_id
        WHERE pop.production_order_id = $1
        ORDER BY pop.sequence
      `, [orderId]);

      res.json(result.rows);
    } catch (error) {
      console.error("[GET /api/production/orders/:orderId/pallets] Error:", error);
      res.status(500).json({ error: "Błąd podczas pobierania palet" });
    }
  });

  // GET /api/production/location-groups - Lista grup lokalizacji
  app.get("/api/production/location-groups", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT 
          id,
          code,
          name,
          description,
          location_type AS "locationType",
          allows_sublocations AS "allowsSublocations",
          is_active AS "isActive",
          sort_order AS "sortOrder",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
        FROM production.production_location_groups
        WHERE is_active = true
        ORDER BY name
      `);

      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching location groups:", error);
      res.status(500).json({ error: "Failed to fetch location groups" });
    }
  });

  // GET /api/production/carrier-groups - Lista grup nośników
  // Alias: GET /api/production/carriers/groups (dla kompatybilności z frontendem)
  const getCarrierGroupsHandler = async (req: any, res: any) => {
    try {
      const result = await pool.query(`
        SELECT 
          id,
          code,
          name,
          description,
          default_capacity AS "defaultCapacity",
          capacity_unit AS "capacityUnit",
          is_active AS "isActive",
          sort_order AS "sortOrder",
          created_at AS "createdAt",
          updated_at AS "updatedAt"
        FROM production.production_carrier_groups
        WHERE is_active = true
        ORDER BY name
      `);

      res.json(result.rows);
    } catch (error) {
      console.error("❌ Error fetching carrier groups:", error);
      res.status(500).json({ error: "Failed to fetch carrier groups" });
    }
  };
  
  app.get("/api/production/carrier-groups", isAuthenticated, getCarrierGroupsHandler);
  app.get("/api/production/carriers/groups", isAuthenticated, getCarrierGroupsHandler);

  // ===== FILE STORAGE SETTINGS =====
  
  // GET /api/file-storage/settings - Get file storage configuration
  app.get("/api/file-storage/settings", requireAdmin, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT id, provider, host, port, username, base_url, path_prefix, 
               is_active, last_connection_test, connection_status, last_error,
               created_at, updated_at
        FROM file_storage_settings
        ORDER BY created_at DESC
        LIMIT 1
      `);

      if (result.rows.length === 0) {
        // Return default configuration
        return res.json({
          provider: 'local',
          isActive: true,
        });
      }

      res.json(result.rows[0]);
    } catch (error) {
      console.error("Get file storage settings error:", error);
      res.status(500).json({ error: "Failed to fetch file storage settings" });
    }
  });

  // POST /api/file-storage/settings - Create or update file storage configuration
  app.post("/api/file-storage/settings", requireAdmin, async (req, res) => {
    try {
      const { provider, host, port, username, baseUrl, pathPrefix, isActive } = req.body;

      // Check if configuration already exists
      const existingResult = await pool.query(
        'SELECT id FROM file_storage_settings LIMIT 1'
      );

      if (existingResult.rows.length > 0) {
        // Normalize pathPrefix to remove leading slashes before saving
        const normalizedPathPrefix = (pathPrefix || 'OMS').replace(/^\/+/, '');
        
        // Update existing configuration
        const result = await pool.query(`
          UPDATE file_storage_settings
          SET provider = $1, host = $2, port = $3, username = $4,
              base_url = $5, path_prefix = $6, is_active = $7,
              updated_at = NOW()
          WHERE id = $8
          RETURNING *
        `, [provider, host, port || 22, username, baseUrl, normalizedPathPrefix, isActive, existingResult.rows[0].id]);

        // Reset adapter cache when configuration changes
        const { resetFileStorageAdapterCache } = await import('./file-storage-adapter.js');
        resetFileStorageAdapterCache();

        res.json(result.rows[0]);
      } else {
        // Normalize pathPrefix to remove leading slashes before saving
        const normalizedPathPrefix = (pathPrefix || 'OMS').replace(/^\/+/, '');
        
        // Insert new configuration
        const result = await pool.query(`
          INSERT INTO file_storage_settings (
            provider, host, port, username, base_url, path_prefix, is_active
          )
          VALUES ($1, $2, $3, $4, $5, $6, $7)
          RETURNING *
        `, [provider, host, port || 22, username, baseUrl, normalizedPathPrefix, isActive]);

        // Reset adapter cache when configuration changes
        const { resetFileStorageAdapterCache } = await import('./file-storage-adapter.js');
        resetFileStorageAdapterCache();

        res.json(result.rows[0]);
      }
    } catch (error) {
      console.error("Save file storage settings error:", error);
      res.status(500).json({ error: "Failed to save file storage settings" });
    }
  });

  // POST /api/file-storage/test - Test connection to external file storage
  app.post("/api/file-storage/test", requireAdmin, async (req, res) => {
    try {
      const { host, port, username, pathPrefix } = req.body;
      const password = process.env.FILE_STORAGE_PASSWORD || '';

      if (!host || !username || !password) {
        return res.status(400).json({ 
          error: "Missing required fields (host, username, or FILE_STORAGE_PASSWORD environment variable)" 
        });
      }

      // Test SFTP connection
      const SftpClient = (await import('ssh2-sftp-client')).default;
      const sftp = new SftpClient();

      try {
        await sftp.connect({
          host: host,
          port: port || 22,
          username: username,
          password: password,
        });

        // Test creating a directory - remove leading slash to make it relative to home directory
        const normalizedPrefix = (pathPrefix || 'OMS').replace(/^\/+/, '');
        const testPath = `${normalizedPrefix}/test`;
        console.log('🔍 SFTP Test - Attempting to create directory:', testPath);
        await sftp.mkdir(testPath, true);

        await sftp.end();

        // Update connection status in database
        await pool.query(`
          UPDATE file_storage_settings
          SET last_connection_test = NOW(),
              connection_status = 'connected',
              last_error = NULL
          WHERE host = $1
        `, [host]);

        res.json({ 
          success: true, 
          message: "Connection successful! Created test directory: /OMS/test" 
        });
      } catch (error: any) {
        await sftp.end().catch(() => {});

        // Update connection status in database
        await pool.query(`
          UPDATE file_storage_settings
          SET last_connection_test = NOW(),
              connection_status = 'failed',
              last_error = $1
          WHERE host = $2
        `, [error.message, host]);

        res.status(500).json({ 
          error: "Connection failed", 
          details: error.message 
        });
      }
    } catch (error: any) {
      console.error("Test connection error:", error);
      res.status(500).json({ error: "Failed to test connection", details: error.message });
    }
  });

  // POST /api/file-storage/migrate-database-urls - Migrate database URLs for matrices and catalog products
  app.post("/api/file-storage/migrate-database-urls", requireAdmin, async (req, res) => {
    try {
      const { getFileStorageAdapter, resetFileStorageAdapterCache } = await import('./file-storage-adapter.js');
      resetFileStorageAdapterCache();
      const adapter = await getFileStorageAdapter();
      
      const results = {
        matrices: { total: 0, updated: 0, failed: 0 },
        catalogImages: { total: 0, updated: 0, failed: 0 },
        errors: [] as Array<{ type: string; id: string | number; error: string }>,
      };

      console.log("\n🔄 Starting database URL migration...\n");

      // 1. Migrate Product Matrices colorImages
      console.log("📋 Migrating Product Matrices colorImages...");
      try {
        const matricesResult = await pool.query(`
          SELECT id, name, color_images
          FROM product_creator.product_matrices
          WHERE color_images IS NOT NULL AND color_images::text != '{}'
        `);

        for (const matrix of matricesResult.rows) {
          results.matrices.total++;
          try {
            const colorImages = matrix.color_images || {};
            let hasChanges = false;
            const updatedColorImages: Record<string, string[]> = {};

            for (const [colorName, urls] of Object.entries(colorImages)) {
              if (Array.isArray(urls)) {
                updatedColorImages[colorName] = [];
                
                for (const url of urls as string[]) {
                  // Skip if already an SFTP URL
                  if (url.startsWith('http')) {
                    updatedColorImages[colorName].push(url);
                    continue;
                  }

                  // Extract filename and migrate
                  const filename = url.split('/').pop();
                  if (!filename) {
                    updatedColorImages[colorName].push(url);
                    continue;
                  }

                  // Check if file exists locally
                  const localPath = path.join(process.cwd(), url.startsWith('/') ? url.substring(1) : url);
                  const fileExists = await fs.access(localPath).then(() => true).catch(() => false);

                  if (fileExists) {
                    try {
                      // Upload to SFTP
                      const fileBuffer = await fs.readFile(localPath);
                      const sftpUrl = await adapter.upload({
                        filename,
                        buffer: fileBuffer,
                        mimetype: 'image/jpeg',
                        subfolder: 'products/images'
                      });
                      
                      updatedColorImages[colorName].push(sftpUrl);
                      hasChanges = true;
                      console.log(`  ✅ Migrated: ${filename} for color "${colorName}"`);
                    } catch (uploadError: any) {
                      console.error(`  ❌ Upload failed for ${filename}:`, uploadError.message);
                      updatedColorImages[colorName].push(url); // Keep old URL
                    }
                  } else {
                    // File doesn't exist locally, convert URL anyway
                    const sftpUrl = adapter.getUrl(`products/images/${filename}`);
                    updatedColorImages[colorName].push(sftpUrl);
                    hasChanges = true;
                    console.log(`  ⚠️  File not found locally, converted URL: ${filename}`);
                  }
                }
              }
            }

            // Update database if changes were made
            if (hasChanges) {
              await pool.query(
                'UPDATE product_creator.product_matrices SET color_images = $1, updated_at = CURRENT_TIMESTAMP WHERE id = $2',
                [JSON.stringify(updatedColorImages), matrix.id]
              );
              results.matrices.updated++;
              console.log(`  ✅ Updated matrix "${matrix.name}" (ID: ${matrix.id})`);
            }
          } catch (error: any) {
            results.matrices.failed++;
            results.errors.push({
              type: 'matrix',
              id: matrix.id,
              error: error.message
            });
            console.error(`  ❌ Failed to migrate matrix ${matrix.id}:`, error.message);
          }
        }
      } catch (error: any) {
        console.error("❌ Error querying matrices:", error.message);
      }

      // 2. Migrate Catalog Product Images
      console.log("\n📸 Migrating Catalog Product Images...");
      try {
        const imagesResult = await pool.query(`
          SELECT id, product_id, url, filename
          FROM catalog.product_images
          WHERE url NOT LIKE 'http%'
        `);

        for (const image of imagesResult.rows) {
          results.catalogImages.total++;
          try {
            const filename = image.filename || image.url.split('/').pop();
            if (!filename) {
              results.catalogImages.failed++;
              continue;
            }

            // Check if file exists locally
            const localPath = path.join(process.cwd(), image.url.startsWith('/') ? image.url.substring(1) : image.url);
            const fileExists = await fs.access(localPath).then(() => true).catch(() => false);

            let sftpUrl: string;

            if (fileExists) {
              try {
                // Upload to SFTP
                const fileBuffer = await fs.readFile(localPath);
                sftpUrl = await adapter.upload({
                  filename,
                  buffer: fileBuffer,
                  mimetype: 'image/jpeg',
                  subfolder: 'products/images'
                });
                console.log(`  ✅ Uploaded: ${filename}`);
              } catch (uploadError: any) {
                console.error(`  ❌ Upload failed for ${filename}:`, uploadError.message);
                sftpUrl = adapter.getUrl(`products/images/${filename}`);
              }
            } else {
              // File doesn't exist locally, convert URL anyway
              sftpUrl = adapter.getUrl(`products/images/${filename}`);
              console.log(`  ⚠️  File not found locally, converted URL: ${filename}`);
            }

            // Update database
            await pool.query(
              'UPDATE catalog.product_images SET url = $1, updated_at = NOW() WHERE id = $2',
              [sftpUrl, image.id]
            );
            results.catalogImages.updated++;
          } catch (error: any) {
            results.catalogImages.failed++;
            results.errors.push({
              type: 'catalog_image',
              id: image.id,
              error: error.message
            });
            console.error(`  ❌ Failed to migrate image ${image.id}:`, error.message);
          }
        }
      } catch (error: any) {
        console.error("❌ Error querying catalog images:", error.message);
      }

      console.log("\n📊 Migration Summary:");
      console.log(`   Product Matrices:`);
      console.log(`     Total: ${results.matrices.total}`);
      console.log(`     ✅ Updated: ${results.matrices.updated}`);
      console.log(`     ❌ Failed: ${results.matrices.failed}`);
      console.log(`   Catalog Images:`);
      console.log(`     Total: ${results.catalogImages.total}`);
      console.log(`     ✅ Updated: ${results.catalogImages.updated}`);
      console.log(`     ❌ Failed: ${results.catalogImages.failed}`);

      res.json({
        success: true,
        message: "Database URL migration completed",
        results,
      });
    } catch (error: any) {
      console.error("Migration error:", error);
      res.status(500).json({ 
        error: "Migration failed", 
        details: error.message 
      });
    }
  });

  // POST /api/file-storage/migrate - Migrate existing files to SFTP
  app.post("/api/file-storage/migrate", requireAdmin, async (req, res) => {
    try {
      // CRITICAL: Reset adapter cache to ensure we use fresh configuration from database
      const { getFileStorageAdapter, resetFileStorageAdapterCache } = await import('./file-storage-adapter.js');
      resetFileStorageAdapterCache();
      const adapter = await getFileStorageAdapter();
      
      const results = {
        total: 0,
        success: 0,
        failed: 0,
        errors: [] as Array<{ file: string; error: string }>,
      };

      // Helper to migrate a single file
      const migrateFile = async (localPath: string, subfolder: string) => {
        try {
          const fileBuffer = await fs.readFile(localPath);
          const filename = path.basename(localPath);
          
          await adapter.upload({
            filename,
            buffer: fileBuffer,
            mimetype: 'image/jpeg', // Assuming all are JPEGs
            subfolder,
          });
          
          results.success++;
          console.log(`✅ Migrated: ${localPath} → ${subfolder}/${filename}`);
        } catch (error: any) {
          results.failed++;
          results.errors.push({ 
            file: localPath, 
            error: error.message 
          });
          console.error(`❌ Failed to migrate ${localPath}:`, error.message);
        }
      };

      // 1. Migrate uploads/products/* → products/images/
      console.log("📁 Scanning uploads/products/...");
      try {
        const uploadsProducts = await fs.readdir(path.join(process.cwd(), 'uploads/products'));
        for (const file of uploadsProducts) {
          if (file.match(/\.(jpg|jpeg|png|webp)$/i)) {
            results.total++;
            await migrateFile(
              path.join(process.cwd(), 'uploads/products', file),
              'products/images'
            );
          }
        }
      } catch (error: any) {
        console.log("⚠️  uploads/products/ not found or empty");
      }

      // 2. Migrate attached_assets/products/* → products/images/
      console.log("📁 Scanning attached_assets/products/...");
      try {
        const attachedProducts = await fs.readdir(path.join(process.cwd(), 'attached_assets/products'));
        for (const file of attachedProducts) {
          if (file.match(/\.(jpg|jpeg|png|webp)$/i)) {
            results.total++;
            await migrateFile(
              path.join(process.cwd(), 'attached_assets/products', file),
              'products/images'
            );
          }
        }
      } catch (error: any) {
        console.log("⚠️  attached_assets/products/ not found or empty");
      }

      // 3. Migrate uploads/accessories/* → accessories/images/
      console.log("📁 Scanning uploads/accessories/...");
      try {
        const uploadsAccessories = await fs.readdir(path.join(process.cwd(), 'uploads/accessories'));
        for (const file of uploadsAccessories) {
          if (file.match(/\.(jpg|jpeg|png|webp)$/i)) {
            results.total++;
            await migrateFile(
              path.join(process.cwd(), 'uploads/accessories', file),
              'accessories/images'
            );
          }
        }
      } catch (error: any) {
        console.log("⚠️  uploads/accessories/ not found or empty");
      }

      console.log("\n📊 Migration Summary:");
      console.log(`   Total: ${results.total}`);
      console.log(`   ✅ Success: ${results.success}`);
      console.log(`   ❌ Failed: ${results.failed}`);

      res.json({
        success: true,
        message: "Migration completed",
        results,
      });
    } catch (error: any) {
      console.error("Migration error:", error);
      res.status(500).json({ 
        error: "Migration failed", 
        details: error.message 
      });
    }
  });

  // POST /api/file-storage/upload - Generic file upload endpoint using adapter
  app.post("/api/file-storage/upload", isAuthenticated, upload.single('file'), async (req, res) => {
    try {
      if (!req.file) {
        return res.status(400).json({ error: "No file provided" });
      }

      const { category, metadata } = req.body;
      const validCategories = ['product-image', 'export-pdf', 'export-csv', 'ai-cache'];

      if (!validCategories.includes(category)) {
        return res.status(400).json({ 
          error: `Invalid category. Must be one of: ${validCategories.join(', ')}` 
        });
      }

      const { getFileStorageAdapter } = await import('./file-storage-adapter.js');
      const adapter = await getFileStorageAdapter();

      // Map category to subfolder
      const categoryToSubfolder: Record<string, string> = {
        'product-image': 'products/images',
        'export-pdf': 'exports/pdf',
        'export-csv': 'exports/csv',
        'ai-cache': 'ai-cache',
      };

      const url = await adapter.upload({
        filename: req.file.originalname,
        buffer: req.file.buffer,
        mimetype: req.file.mimetype,
        subfolder: categoryToSubfolder[category],
        metadata: metadata ? JSON.parse(metadata) : {},
      });

      res.json({ 
        success: true,
        url: url,
        message: "File uploaded successfully"
      });
    } catch (error: any) {
      console.error("File upload error:", error);
      res.status(500).json({ error: "Failed to upload file", details: error.message });
    }
  });

  // ==================== BACKUP MANAGEMENT ====================

  // GET /api/backup/list - List all backups (local + SFTP)
  app.get("/api/backup/list", isAuthenticated, async (req, res) => {
    try {
      const { listAllBackups } = await import('./backup-manager.js');
      const backups = await listAllBackups();
      res.json(backups);
    } catch (error: any) {
      console.error("List backups error:", error);
      res.status(500).json({ error: "Failed to list backups", details: error.message });
    }
  });

  // POST /api/backup/database - Create database backup
  app.post("/api/backup/database", isAuthenticated, async (req, res) => {
    try {
      const { exportDatabaseToSQL } = await import('./backup-manager.js');
      const filepath = await exportDatabaseToSQL();
      const filename = path.basename(filepath);
      
      res.json({ 
        success: true,
        message: "Database backup created successfully",
        filename,
        downloadUrl: `/api/backup/download/${filename}`
      });
    } catch (error: any) {
      console.error("Database backup error:", error);
      res.status(500).json({ error: "Failed to create database backup", details: error.message });
    }
  });

  // POST /api/backup/files - Create files backup
  app.post("/api/backup/files", isAuthenticated, async (req, res) => {
    try {
      const { createFilesBackup } = await import('./backup-manager.js');
      const filepath = await createFilesBackup();
      const filename = path.basename(filepath);
      
      res.json({ 
        success: true,
        message: "Files backup created successfully",
        filename,
        downloadUrl: `/api/backup/download/${filename}`
      });
    } catch (error: any) {
      console.error("Files backup error:", error);
      res.status(500).json({ error: "Failed to create files backup", details: error.message });
    }
  });

  // POST /api/backup/full - Create full backup (database + files)
  app.post("/api/backup/full", isAuthenticated, async (req, res) => {
    try {
      const { createFullBackup } = await import('./backup-manager.js');
      const { database, files } = await createFullBackup();
      
      res.json({ 
        success: true,
        message: "Full backup created successfully",
        database: {
          filename: path.basename(database),
          downloadUrl: `/api/backup/download/${path.basename(database)}`
        },
        files: {
          filename: path.basename(files),
          downloadUrl: `/api/backup/download/${path.basename(files)}`
        }
      });
    } catch (error: any) {
      console.error("Full backup error:", error);
      res.status(500).json({ error: "Failed to create full backup", details: error.message });
    }
  });

  // GET /api/backup/download/:filename - Download backup file (local or SFTP)
  app.get("/api/backup/download/:filename", isAuthenticated, async (req, res) => {
    try {
      const { filename } = req.params;
      
      // Security: prevent path traversal
      if (filename.includes('..') || filename.includes('/')) {
        return res.status(400).json({ error: "Invalid filename" });
      }

      const { getBackupPath, downloadBackupFromSFTP } = await import('./backup-manager.js');
      const filepath = getBackupPath(filename);

      // Try to serve from local file first
      const fs = await import('fs/promises');
      try {
        await fs.access(filepath);
        return res.download(filepath, filename);
      } catch (localError) {
        // File not found locally, try SFTP
        console.log(`📥 File not found locally, trying SFTP: ${filename}`);
        const buffer = await downloadBackupFromSFTP(filename);
        
        if (!buffer) {
          return res.status(404).json({ error: "Backup file not found locally or on SFTP" });
        }

        console.log(`✅ Downloaded from SFTP: ${filename}, size: ${buffer.length} bytes`);

        // Determine content type
        const contentType = filename.endsWith('.sql') 
          ? 'application/sql' 
          : filename.endsWith('.zip') 
            ? 'application/zip' 
            : 'application/octet-stream';

        res.setHeader('Content-Disposition', `attachment; filename="${filename}"`);
        res.setHeader('Content-Type', contentType);
        res.setHeader('Content-Length', buffer.length.toString());
        res.send(buffer);
      }
    } catch (error: any) {
      console.error("Download backup error:", error);
      res.status(500).json({ error: "Failed to download backup", details: error.message });
    }
  });

  // DELETE /api/backup/:filename - Delete backup
  app.delete("/api/backup/:filename", isAuthenticated, async (req, res) => {
    try {
      const { filename } = req.params;
      const { getBackupPath } = await import('./backup-manager.js');
      
      // Security: prevent path traversal
      if (filename.includes('..') || filename.includes('/')) {
        return res.status(400).json({ error: "Invalid filename" });
      }

      const filepath = getBackupPath(filename);
      const fs = await import('fs/promises');
      await fs.unlink(filepath);

      res.json({ success: true, message: "Backup deleted successfully" });
    } catch (error: any) {
      console.error("Delete backup error:", error);
      res.status(500).json({ error: "Failed to delete backup", details: error.message });
    }
  });

  // POST /api/backup/cleanup - Cleanup old backups
  app.post("/api/backup/cleanup", isAuthenticated, async (req, res) => {
    try {
      const { keepCount = 10 } = req.body;
      const { cleanupOldBackups } = await import('./backup-manager.js');
      const deletedCount = await cleanupOldBackups(keepCount);

      res.json({ 
        success: true,
        message: `Cleaned up ${deletedCount} old backup(s)`,
        deletedCount
      });
    } catch (error: any) {
      console.error("Cleanup backups error:", error);
      res.status(500).json({ error: "Failed to cleanup backups", details: error.message });
    }
  });

  // ============================================================================
  // BOM (BILL OF MATERIALS) MANAGEMENT
  // ============================================================================

  // POST /api/component-templates/import-csv - Import matrycy formatek z CSV (Airtable)
  app.post("/api/component-templates/import-csv", isAuthenticated, async (req, res) => {
    try {
      const { csvData } = req.body;

      if (!csvData || typeof csvData !== 'string') {
        return res.status(400).json({ error: "CSV data is required" });
      }

      console.log("📥 Starting import of component templates from CSV...");

      // Parse CSV
      const parsedCsv = Papa.parse(csvData, {
        header: true,
        skipEmptyLines: true,
        transformHeader: (header) => header.trim()
      });

      if (parsedCsv.errors.length > 0) {
        console.error("CSV parsing errors:", parsedCsv.errors);
        return res.status(400).json({ 
          error: "Failed to parse CSV", 
          details: parsedCsv.errors 
        });
      }

      const rows = parsedCsv.data as any[];
      console.log(`📊 Parsed ${rows.length} rows from CSV`);

      // Start transaction
      const client = await pool.connect();
      try {
        await client.query('BEGIN');

        let importedCount = 0;
        let skippedCount = 0;
        const errors: string[] = [];

        for (const row of rows) {
          try {
            // Map CSV columns to database fields
            const template = {
              cz1: row.cz1 || null,
              cz2: row.cz2 || null,
              furniture_type: row.Rodzaj || row.rodzajText || null,
              base_length: row['Długość 50'] ? parseFloat(row['Długość 50']) : null,
              base_width: row['Szerokość 30'] ? parseFloat(row['Szerokość 30']) : null,
              thickness: row['Grubość'] ? parseFloat(row['Grubość']) : null,
              furniture_length_condition: row['Długość Mebla'] ? parseInt(row['Długość Mebla'], 10) : null,
              furniture_width_condition: row['Szerokość Mebla'] ? parseInt(row['Szerokość Mebla'], 10) : null,
              alternative_lengths: [
                row['Długość 30'] ? parseInt(row['Długość 30'], 10) : null,
                row['Długość 40'] ? parseInt(row['Długość 40'], 10) : null,
                row['Długość 60'] ? parseInt(row['Długość 60'], 10) : null,
                row['Długość 80'] ? parseInt(row['Długość 80'], 10) : null,
                row['Długość 100'] ? parseInt(row['Długość 100'], 10) : null,
                row['Długość 120'] ? parseInt(row['Długość 120'], 10) : null,
              ].filter(v => v !== null),
              alternative_widths: [
                row['Szerokość 36'] ? parseInt(row['Szerokość 36'], 10) : null,
                row['Szerokość 55'] ? parseInt(row['Szerokość 55'], 10) : null,
                row['Szerokość 65'] ? parseInt(row['Szerokość 65'], 10) : null,
              ].filter(v => v !== null),
              plate_type: row.plyty || null,
              color: row.Kolor || null,
              edging_pattern: row.Oklejanie || null,
              edging_material: row.Obrzeze1 || null,
              drilling_required: row.Wiercenie === 'checked',
              no_color_change: row['Brak zmiany koloru'] === 'checked',
              exclude_from_cutting: row['Wyklucz z cięcia'] === 'checked',
              apply_by_length: row['Stosuj po Długości'] === 'checked',
              apply_by_width: row['Stosuj po Szerokości'] === 'checked',
              half_plate: row['Płyta na pół'] === 'checked',
              production_category: row.rodzajText || null,
              unit_price: row['Cena wiercenia (from Wiercenie Cennik)'] ? parseFloat(row['Cena wiercenia (from Wiercenie Cennik)']) : null,
              airtable_record_id: row['numer FKol'] || null,
              is_active: true,
            };

            // Skip empty rows
            if (!template.cz1 || !template.furniture_type) {
              skippedCount++;
              continue;
            }

            // Insert template
            await client.query(`
              INSERT INTO bom.component_templates (
                cz1, cz2, furniture_type, 
                base_length, base_width, thickness,
                furniture_length_condition, furniture_width_condition,
                alternative_lengths, alternative_widths,
                plate_type, color,
                edging_pattern, edging_material,
                drilling_required, no_color_change, exclude_from_cutting,
                apply_by_length, apply_by_width, half_plate,
                production_category, unit_price, airtable_record_id,
                is_active
              ) VALUES (
                $1, $2, $3, 
                $4, $5, $6,
                $7, $8,
                $9::integer[], $10::integer[],
                $11, $12,
                $13, $14,
                $15, $16, $17,
                $18, $19, $20,
                $21, $22, $23,
                $24
              )
              ON CONFLICT (furniture_type, cz1, cz2, base_length, base_width) 
              DO UPDATE SET
                thickness = EXCLUDED.thickness,
                furniture_length_condition = EXCLUDED.furniture_length_condition,
                furniture_width_condition = EXCLUDED.furniture_width_condition,
                alternative_lengths = EXCLUDED.alternative_lengths,
                alternative_widths = EXCLUDED.alternative_widths,
                plate_type = EXCLUDED.plate_type,
                color = EXCLUDED.color,
                edging_pattern = EXCLUDED.edging_pattern,
                edging_material = EXCLUDED.edging_material,
                drilling_required = EXCLUDED.drilling_required,
                no_color_change = EXCLUDED.no_color_change,
                exclude_from_cutting = EXCLUDED.exclude_from_cutting,
                apply_by_length = EXCLUDED.apply_by_length,
                apply_by_width = EXCLUDED.apply_by_width,
                half_plate = EXCLUDED.half_plate,
                production_category = EXCLUDED.production_category,
                updated_at = NOW()
            `, [
              template.cz1, template.cz2, template.furniture_type,
              template.base_length, template.base_width, template.thickness,
              template.furniture_length_condition, template.furniture_width_condition,
              template.alternative_lengths, template.alternative_widths,
              template.plate_type, template.color,
              template.edging_pattern, template.edging_material,
              template.drilling_required, template.no_color_change, template.exclude_from_cutting,
              template.apply_by_length, template.apply_by_width, template.half_plate,
              template.production_category, template.unit_price, template.airtable_record_id,
              template.is_active
            ]);

          results.success++;

            importedCount++;
          } catch (rowError: any) {
            errors.push(`Row error: ${rowError.message}`);
            console.error("Row import error:", rowError);
          }
        }

        await client.query('COMMIT');

        console.log(`✅ Import completed: ${importedCount} templates imported, ${skippedCount} skipped`);

        res.json({
          success: true,
          imported: importedCount,
          skipped: skippedCount,
          errors: errors.length > 0 ? errors : undefined
        });

      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }

    } catch (error: any) {
      console.error("Import component templates error:", error);
      res.status(500).json({ 
        error: "Failed to import component templates", 
        details: error.message 
      });
    }
  });

  // GET /api/component-templates - Lista szablonów formatek
  app.get("/api/component-templates", isAuthenticated, async (req, res) => {
    try {
      const { furnitureType, cz1, search, limit = 100, offset = 0 } = req.query;

      let query = `
        SELECT * FROM bom.component_templates
        WHERE is_active = true
      `;
      const params: any[] = [];
      let paramIndex = 1;

      if (furnitureType) {
        query += ` AND furniture_type = $${paramIndex}`;
        params.push(furnitureType);
        paramIndex++;
      }

      if (cz1) {
        query += ` AND cz1 = $${paramIndex}`;
        params.push(cz1);
        paramIndex++;
      }

      if (search) {
        query += ` AND (cz1 ILIKE $${paramIndex} OR cz2 ILIKE $${paramIndex} OR production_category ILIKE $${paramIndex})`;
        params.push(`%${search}%`);
        paramIndex++;
      }

      query += ` ORDER BY furniture_type, cz1, cz2 LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`;
      params.push(limit, offset);

      const result = await pool.query(query, params);

      // Count total
      let countQuery = `SELECT COUNT(*) FROM bom.component_templates WHERE is_active = true`;
      const countParams: any[] = [];
      let countParamIndex = 1;

      if (furnitureType) {
        countQuery += ` AND furniture_type = $${countParamIndex}`;
        countParams.push(furnitureType);
        countParamIndex++;
      }

      if (cz1) {
        countQuery += ` AND cz1 = $${countParamIndex}`;
        countParams.push(cz1);
        countParamIndex++;
      }

      if (search) {
        countQuery += ` AND (cz1 ILIKE $${countParamIndex} OR cz2 ILIKE $${countParamIndex} OR production_category ILIKE $${countParamIndex})`;
        countParams.push(`%${search}%`);
      }

      const countResult = await pool.query(countQuery, countParams);

      // Map snake_case parent_id to camelCase parentId for frontend
      const templates = result.rows.map(({ parent_id, ...row }) => ({
        ...row,
        parentId: parent_id
      }));

      res.json({
        templates,
        total: parseInt(countResult.rows[0].count, 10),
        limit: parseInt(limit as string, 10),
        offset: parseInt(offset as string, 10)
      });

    } catch (error: any) {
      console.error("Get component templates error:", error);
      res.status(500).json({ error: "Failed to get component templates", details: error.message });
    }
  });

  // GET /api/component-templates/furniture-types - Lista dostępnych typów mebli
  app.get("/api/component-templates/furniture-types", isAuthenticated, async (req, res) => {
    try {
      const result = await pool.query(`
        SELECT DISTINCT furniture_type, COUNT(*) as template_count
        FROM bom.component_templates
        WHERE is_active = true
        GROUP BY furniture_type
        ORDER BY furniture_type
      `);

      res.json(result.rows);
    } catch (error: any) {
      console.error("Get furniture types error:", error);
      res.status(500).json({ error: "Failed to get furniture types", details: error.message });
    }
  });

  // POST /api/component-templates - Dodaj nowy szablon formatki
  app.post("/api/component-templates", isAuthenticated, async (req, res) => {
    try {
      const template = req.body;

      // Walidacja: nie można zaznaczyć obu radio buttons jednocześnie
      if (template.apply_by_length && template.apply_by_width) {
        return res.status(400).json({ 
          error: "Błąd walidacji", 
          details: "Można zaznaczyć tylko 'Dł.' lub 'Szer.', nie oba jednocześnie" 
        });
      }

      const result = await pool.query(`
        INSERT INTO bom.component_templates (
          furniture_type, cz1, cz2,
          base_length, base_width, thickness,
          color, edging_pattern, edge1, edge2, edge3, edge4, plate_type, edging_material,
          furniture_length_condition, furniture_width_condition,
          alternative_lengths, alternative_widths,
          drilling_required, no_color_change, exclude_from_cutting,
          apply_by_length, apply_by_width, half_plate,
          production_category, unit_price, is_active,
          door, leg, parent_id
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14,
          $15, $16, $17::jsonb, $18::jsonb,
          $19, $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $30
        ) RETURNING *
      `, [
        template.furniture_type, template.cz1, template.cz2,
        template.base_length, template.base_width, template.thickness,
        template.color, template.edging_pattern, 
        template.edge1 || false, template.edge2 || false, template.edge3 || false, template.edge4 || false,
        template.plate_type, template.edging_material,
        template.furniture_length_condition, template.furniture_width_condition,
        JSON.stringify(template.alternative_lengths || []), 
        JSON.stringify(template.alternative_widths || []),
        template.drilling_required, template.no_color_change, template.exclude_from_cutting,
        template.apply_by_length, template.apply_by_width, template.half_plate,
        template.production_category, template.unit_price, template.is_active,
        template.door || null, template.leg || null, template.parentId || null
      ]);

      // Map snake_case parent_id to camelCase parentId for frontend
      const { parent_id, ...row } = result.rows[0];
      res.json({ ...row, parentId: parent_id });
    } catch (error: any) {
      console.error("Create component template error:", error);
      
      // Check for unique constraint violation
      if (error.code === '23505') {
        return res.status(409).json({ 
          error: "Duplikat formatki", 
          details: `Formatka o takiej kombinacji (Rodzaj, Cz1, Cz2, Długość, Szerokość) już istnieje` 
        });
      }
      
      res.status(500).json({ error: "Failed to create template", details: error.message });
    }
  });

  // PUT /api/component-templates/:id - Aktualizuj szablon formatki
  app.put("/api/component-templates/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const template = req.body;

      // Walidacja: nie można zaznaczyć obu radio buttons jednocześnie
      if (template.apply_by_length && template.apply_by_width) {
        return res.status(400).json({ 
          error: "Błąd walidacji", 
          details: "Można zaznaczyć tylko 'Dł.' lub 'Szer.', nie oba jednocześnie" 
        });
      }

      const result = await pool.query(`
        UPDATE bom.component_templates SET
          furniture_type = $1, cz1 = $2, cz2 = $3,
          base_length = $4, base_width = $5, thickness = $6,
          color = $7, edging_pattern = $8, edge1 = $9, edge2 = $10, edge3 = $11, edge4 = $12,
          plate_type = $13, edging_material = $14,
          furniture_length_condition = $15, furniture_width_condition = $16,
          alternative_lengths = $17::jsonb, alternative_widths = $18::jsonb,
          drilling_required = $19, no_color_change = $20, exclude_from_cutting = $21,
          apply_by_length = $22, apply_by_width = $23, half_plate = $24,
          production_category = $25, unit_price = $26, is_active = $27,
          door = $28, leg = $29, parent_id = $30,
          updated_at = NOW()
        WHERE id = $31
        RETURNING *
      `, [
        template.furniture_type, template.cz1, template.cz2,
        template.base_length, template.base_width, template.thickness,
        template.color, template.edging_pattern,
        template.edge1 || false, template.edge2 || false, template.edge3 || false, template.edge4 || false,
        template.plate_type, template.edging_material,
        template.furniture_length_condition, template.furniture_width_condition,
        JSON.stringify(template.alternative_lengths || []), 
        JSON.stringify(template.alternative_widths || []),
        template.drilling_required, template.no_color_change, template.exclude_from_cutting,
        template.apply_by_length, template.apply_by_width, template.half_plate,
        template.production_category, template.unit_price, template.is_active,
        template.door || null, template.leg || null, template.parentId || null,
        id
      ]);

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Template not found" });
      }

      // Map snake_case parent_id to camelCase parentId for frontend
      const { parent_id, ...row } = result.rows[0];
      res.json({ ...row, parentId: parent_id });
    } catch (error: any) {
      console.error("Update component template error:", error);
      
      // Check for unique constraint violation
      if (error.code === '23505') {
        return res.status(409).json({ 
          error: "Duplikat formatki", 
          details: `Formatka o takiej kombinacji (Rodzaj, Cz1, Cz2, Długość, Szerokość) już istnieje` 
        });
      }
      
      res.status(500).json({ error: "Failed to update template", details: error.message });
    }
  });

  // DELETE /api/component-templates/:id - Usuń szablon formatki
  app.delete("/api/component-templates/:id", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;

      const result = await pool.query(`
        DELETE FROM bom.component_templates
        WHERE id = $1
        RETURNING id
      `, [id]);

      if (result.rows.length === 0) {
        return res.status(404).json({ error: "Template not found" });
      }

      res.json({ success: true, id: parseInt(id, 10) });
    } catch (error: any) {
      console.error("Delete component template error:", error);
      
      // Check if it's a foreign key constraint violation
      if (error.code === '23503') {
        return res.status(400).json({ 
          error: "Nie można usunąć formatki", 
          details: "Ta formatka jest używana w wygenerowanych komponentach produktów. Usuń najpierw produkty używające tej formatki." 
        });
      }
      
      res.status(500).json({ error: "Failed to delete template", details: error.message });
    }
  });

  // POST /api/component-templates/:id/variant - Utwórz wariant formatki
  app.post("/api/component-templates/:id/variant", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;
      const { cz2, alternative_lengths, alternative_widths } = req.body;

      if (!cz2) {
        return res.status(400).json({ error: "Pole cz2 jest wymagane dla wariantu" });
      }

      // Pobierz oryginalną formatkę
      const originalResult = await pool.query(`
        SELECT * FROM bom.component_templates WHERE id = $1
      `, [id]);

      if (originalResult.rows.length === 0) {
        return res.status(404).json({ error: "Formatka nie została znaleziona" });
      }

      const original = originalResult.rows[0];

      // Użyj wymiarów z requestu jeśli są dostarczone, w przeciwnym razie skopiuj z oryginału
      const variantLengths = alternative_lengths !== undefined ? alternative_lengths : (original.alternative_lengths || []);
      const variantWidths = alternative_widths !== undefined ? alternative_widths : (original.alternative_widths || []);

      // Utwórz wariant - skopiuj wszystkie pola oprócz id, created_at, updated_at i cz2
      const result = await pool.query(`
        INSERT INTO bom.component_templates (
          furniture_type, cz1, cz2,
          base_length, base_width, thickness,
          color, edging_pattern, edge1, edge2, edge3, edge4,
          plate_type, edging_material,
          furniture_length_condition, furniture_width_condition,
          alternative_lengths, alternative_widths,
          drilling_required, no_color_change, exclude_from_cutting,
          apply_by_length, apply_by_width, half_plate,
          production_category, unit_price, is_active
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17::jsonb, $18::jsonb, $19, $20, $21, $22, $23, $24, $25, $26, $27
        )
        RETURNING *
      `, [
        original.furniture_type, original.cz1, cz2,
        original.base_length, original.base_width, original.thickness,
        original.color, original.edging_pattern, 
        original.edge1, original.edge2, original.edge3, original.edge4,
        original.plate_type, original.edging_material,
        original.furniture_length_condition, original.furniture_width_condition,
        JSON.stringify(variantLengths), 
        JSON.stringify(variantWidths),
        original.drilling_required, original.no_color_change, original.exclude_from_cutting,
        original.apply_by_length, original.apply_by_width, original.half_plate,
        original.production_category, original.unit_price, original.is_active
      ]);

      res.json(result.rows[0]);
    } catch (error: any) {
      console.error("Create variant error:", error);
      
      // Check for unique constraint violation
      if (error.code === '23505') {
        return res.status(409).json({ 
          error: "Duplikat formatki", 
          details: `Formatka o takiej kombinacji (Rodzaj, Cz1, Cz2, Długość, Szerokość) już istnieje` 
        });
      }
      
      res.status(500).json({ error: "Failed to create variant", details: error.message });
    }
  });

  // POST /api/component-templates/:id/duplicate-as-branch - Zduplikuj formatkę jako branch (z parent_id)
  app.post("/api/component-templates/:id/duplicate-as-branch", isAuthenticated, async (req, res) => {
    try {
      const { id } = req.params;

      // Pobierz oryginalną formatkę
      const originalResult = await pool.query(`
        SELECT * FROM bom.component_templates WHERE id = $1
      `, [id]);

      if (originalResult.rows.length === 0) {
        return res.status(404).json({ error: "Formatka nie została znaleziona" });
      }

      const original = originalResult.rows[0];

      // Logika parent_id:
      // - Jeśli źródłowa formatka JUŻ JEST branchem (ma parent_id) → użyj tego samego parent_id (zostaje w rodzinie)
      // - Jeśli źródłowa formatka jest rodzicem (parent_id=NULL) → ustaw parent_id na jej ID (nowy branch tego rodzica)
      const newParentId = original.parent_id !== null ? original.parent_id : original.id;

      // Skopiuj wszystkie pola + ustaw parent_id
      const result = await pool.query(`
        INSERT INTO bom.component_templates (
          parent_id, furniture_type, cz1, cz2, door, leg,
          base_length, base_width, thickness,
          color, edging_pattern, edge1, edge2, edge3, edge4,
          plate_type, edging_material,
          furniture_length_condition, furniture_width_condition,
          alternative_lengths, alternative_widths,
          drilling_required, no_color_change, exclude_from_cutting,
          apply_by_length, apply_by_width, half_plate,
          production_category, unit_price, is_active
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20::jsonb, $21::jsonb, $22, $23, $24, $25, $26, $27, $28, $29, $30
        )
        RETURNING *
      `, [
        newParentId, // parent_id = parent źródłowej formatki (lub jej ID jeśli jest rodzicem)
        original.furniture_type, original.cz1, original.cz2, original.door, original.leg,
        original.base_length, original.base_width, original.thickness,
        original.color, original.edging_pattern, 
        original.edge1, original.edge2, original.edge3, original.edge4,
        original.plate_type, original.edging_material,
        original.furniture_length_condition, original.furniture_width_condition,
        JSON.stringify(original.alternative_lengths || []), 
        JSON.stringify(original.alternative_widths || []),
        original.drilling_required, original.no_color_change, original.exclude_from_cutting,
        original.apply_by_length, original.apply_by_width, original.half_plate,
        original.production_category, original.unit_price, original.is_active
      ]);

      // Mapuj parent_id -> parentId dla frontendu
      const row = result.rows[0];
      const responseTemplate = {
        ...row,
        parentId: row.parent_id,
        parent_id: undefined
      };
      delete responseTemplate.parent_id;

      console.log(`✅ Template duplicated as branch: ${original.furniture_type} ${original.cz1} (ID: ${original.id} -> ${row.id}, parent_id=${row.parent_id})`);
      res.json(responseTemplate);
    } catch (error: any) {
      console.error("Duplicate as branch error:", error);
      res.status(500).json({ error: "Failed to duplicate as branch", details: error.message });
    }
  });

  // POST /api/component-templates/:id/duplicate - Zduplikuj formatkę (natychmiastowe zapisanie kopii)
  // LOGIKA:
  // - Jeśli formatka ma parent_id (jest potomkiem) → skopiuj z tym samym parent_id (pozostaje w rodzinie)
  // - Jeśli formatka NIE ma parent_id (jest rodzicem) → zduplikuj całą rodzinę (rodzic + wszyscy potomkowie)
  app.post("/api/component-templates/:id/duplicate", isAuthenticated, async (req, res) => {
    const client = await pool.connect();
    try {
      await client.query('BEGIN');
      
      const { id } = req.params;

      // Pobierz oryginalną formatkę
      const originalResult = await client.query(`
        SELECT * FROM bom.component_templates WHERE id = $1
      `, [id]);

      if (originalResult.rows.length === 0) {
        await client.query('ROLLBACK');
        return res.status(404).json({ error: "Formatka nie została znaleziona" });
      }

      const original = originalResult.rows[0];

      // PRZYPADEK 1: To jest potomek (branch) - skopiuj z tym samym parent_id
      if (original.parent_id !== null) {
        const result = await client.query(`
          INSERT INTO bom.component_templates (
            furniture_type, cz1, cz2,
            base_length, base_width, thickness,
            color, edging_pattern, edge1, edge2, edge3, edge4,
            plate_type, edging_material,
            furniture_length_condition, furniture_width_condition,
            alternative_lengths, alternative_widths,
            drilling_required, no_color_change, exclude_from_cutting,
            apply_by_length, apply_by_width, half_plate,
            production_category, unit_price, is_active,
            parent_id, door, leg
          ) VALUES (
            $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, 
            $17::jsonb, $18::jsonb, $19, $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $30
          )
          RETURNING *
        `, [
          original.furniture_type, original.cz1, original.cz2,
          original.base_length, original.base_width, original.thickness,
          original.color, original.edging_pattern, 
          original.edge1, original.edge2, original.edge3, original.edge4,
          original.plate_type, original.edging_material,
          original.furniture_length_condition, original.furniture_width_condition,
          JSON.stringify(original.alternative_lengths || []), 
          JSON.stringify(original.alternative_widths || []),
          original.drilling_required, original.no_color_change, original.exclude_from_cutting,
          original.apply_by_length, original.apply_by_width, original.half_plate,
          original.production_category, original.unit_price, original.is_active,
          original.parent_id, original.door, original.leg
        ]);

        await client.query('COMMIT');
        console.log(`✅ Template duplicated (branch): ${original.furniture_type} ${original.cz1} (ID: ${original.id} -> ${result.rows[0].id}, parent_id=${original.parent_id})`);
        
        const { parent_id, ...row } = result.rows[0];
        res.json({ ...row, parentId: parent_id });
        return;
      }

      // PRZYPADEK 2: To jest rodzic (parent_id=NULL) - zduplikuj całą rodzinę
      // Najpierw skopiuj rodzica
      const parentResult = await client.query(`
        INSERT INTO bom.component_templates (
          furniture_type, cz1, cz2,
          base_length, base_width, thickness,
          color, edging_pattern, edge1, edge2, edge3, edge4,
          plate_type, edging_material,
          furniture_length_condition, furniture_width_condition,
          alternative_lengths, alternative_widths,
          drilling_required, no_color_change, exclude_from_cutting,
          apply_by_length, apply_by_width, half_plate,
          production_category, unit_price, is_active
        ) VALUES (
          $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, 
          $17::jsonb, $18::jsonb, $19, $20, $21, $22, $23, $24, $25, $26, $27
        )
        RETURNING *
      `, [
        original.furniture_type, original.cz1, original.cz2,
        original.base_length, original.base_width, original.thickness,
        original.color, original.edging_pattern, 
        original.edge1, original.edge2, original.edge3, original.edge4,
        original.plate_type, original.edging_material,
        original.furniture_length_condition, original.furniture_width_condition,
        JSON.stringify(original.alternative_lengths || []), 
        JSON.stringify(original.alternative_widths || []),
        original.drilling_required, original.no_color_change, original.exclude_from_cutting,
        original.apply_by_length, original.apply_by_width, original.half_plate,
        original.production_category, original.unit_price, original.is_active
      ]);

      const newParentId = parentResult.rows[0].id;

      // Teraz skopiuj wszystkich potomków (branches)
      const childrenResult = await client.query(`
        SELECT * FROM bom.component_templates WHERE parent_id = $1
      `, [id]);

      const copiedChildren = [];
      for (const child of childrenResult.rows) {
        const childCopy = await client.query(`
          INSERT INTO bom.component_templates (
            furniture_type, cz1, cz2,
            base_length, base_width, thickness,
            color, edging_pattern, edge1, edge2, edge3, edge4,
            plate_type, edging_material,
            furniture_length_condition, furniture_width_condition,
            alternative_lengths, alternative_widths,
            drilling_required, no_color_change, exclude_from_cutting,
            apply_by_length, apply_by_width, half_plate,
            production_category, unit_price, is_active,
            parent_id, door, leg
          ) VALUES (
            $1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, 
            $17::jsonb, $18::jsonb, $19, $20, $21, $22, $23, $24, $25, $26, $27, $28, $29, $30
          )
          RETURNING *
        `, [
          child.furniture_type, child.cz1, child.cz2,
          child.base_length, child.base_width, child.thickness,
          child.color, child.edging_pattern, 
          child.edge1, child.edge2, child.edge3, child.edge4,
          child.plate_type, child.edging_material,
          child.furniture_length_condition, child.furniture_width_condition,
          JSON.stringify(child.alternative_lengths || []), 
          JSON.stringify(child.alternative_widths || []),
          child.drilling_required, child.no_color_change, child.exclude_from_cutting,
          child.apply_by_length, child.apply_by_width, child.half_plate,
          child.production_category, child.unit_price, child.is_active,
          newParentId, child.door, child.leg
        ]);
        copiedChildren.push(childCopy.rows[0]);
      }

      await client.query('COMMIT');
      console.log(`✅ Template family duplicated: ${original.furniture_type} ${original.cz1} (Parent: ${original.id} -> ${newParentId}, ${copiedChildren.length} children copied)`);
      
      const { parent_id, ...row } = parentResult.rows[0];
      res.json({ ...row, parentId: parent_id });
    } catch (error: any) {
      await client.query('ROLLBACK');
      console.error("Duplicate template error:", error);
      
      // Check for unique constraint violation
      if (error.code === '23505') {
        return res.status(409).json({ 
          error: "Duplikat formatki", 
          details: `Formatka o takiej kombinacji (Rodzaj, Cz1, Cz2, Długość, Szerokość) już istnieje` 
        });
      }
      
      res.status(500).json({ error: "Failed to duplicate template", details: error.message });
    } finally {
      client.release();
    }
  });

  // POST /api/component-templates/bulk-update - Aktualizuj wiele formatek jednocześnie
  app.post("/api/component-templates/bulk-update", isAuthenticated, async (req, res) => {
    try {
      const { templates } = req.body;

      if (!Array.isArray(templates) || templates.length === 0) {
        return res.status(400).json({ error: "Templates array is required and cannot be empty" });
      }

      const results = [];
      const client = await pool.connect();

      try {
        await client.query('BEGIN');

        for (let i = 0; i < templates.length; i++) {
          const template = templates[i];
          const savepointName = `sp_${i}`;
          
          try {
            // Utwórz savepoint dla tego rekordu
            await client.query(`SAVEPOINT ${savepointName}`);

            // Walidacja: nie można zaznaczyć obu radio buttons jednocześnie
            if (template.apply_by_length && template.apply_by_width) {
              results.push({
                id: template.id,
                success: false,
                error: "Można zaznaczyć tylko 'Dł.' lub 'Szer.', nie oba jednocześnie"
              });
              await client.query(`ROLLBACK TO SAVEPOINT ${savepointName}`);
              continue;
            }

            const result = await client.query(`
              UPDATE bom.component_templates SET
                furniture_type = $1, cz1 = $2, cz2 = $3,
                base_length = $4, base_width = $5, thickness = $6,
                color = $7, edging_pattern = $8, edge1 = $9, edge2 = $10, edge3 = $11, edge4 = $12,
                plate_type = $13, edging_material = $14,
                furniture_length_condition = $15, furniture_width_condition = $16,
                alternative_lengths = $17::jsonb, alternative_widths = $18::jsonb,
                drilling_required = $19, no_color_change = $20, exclude_from_cutting = $21,
                apply_by_length = $22, apply_by_width = $23, half_plate = $24,
                production_category = $25, unit_price = $26, is_active = $27,
                door = $28, leg = $29, parent_id = $30,
                updated_at = NOW()
              WHERE id = $31
              RETURNING *
            `, [
              template.furniture_type, template.cz1, template.cz2,
              template.base_length, template.base_width, template.thickness,
              template.color, template.edging_pattern,
              template.edge1 || false, template.edge2 || false, template.edge3 || false, template.edge4 || false,
              template.plate_type, template.edging_material,
              template.furniture_length_condition, template.furniture_width_condition,
              JSON.stringify(template.alternative_lengths || []), 
              JSON.stringify(template.alternative_widths || []),
              template.drilling_required, template.no_color_change, template.exclude_from_cutting,
              template.apply_by_length, template.apply_by_width, template.half_plate,
              template.production_category, template.unit_price, template.is_active,
              template.door || null, template.leg || null, template.parentId || null,
              template.id
            ]);

          results.success++;

            if (result.rows.length === 0) {
              results.push({
                id: template.id,
                success: false,
                error: "Template not found"
              });
              await client.query(`ROLLBACK TO SAVEPOINT ${savepointName}`);
            } else {
              results.push({
                id: template.id,
                success: true,
                data: result.rows[0]
              });
              // Savepoint udany, nie trzeba go rollback'ować
            }
          } catch (error: any) {
            // Wycofaj tylko ten rekord
            await client.query(`ROLLBACK TO SAVEPOINT ${savepointName}`);
            
            // Obsługa błędów dla pojedynczego rekordu
            let errorMessage = error.message;
            if (error.code === '23505') {
              errorMessage = "Duplikat formatki - taka kombinacja już istnieje";
            }
            results.push({
              id: template.id,
              success: false,
              error: errorMessage
            });
          }
        }

        // Zatwierdź wszystkie pomyślne zmiany
        await client.query('COMMIT');

        const successCount = results.filter(r => r.success).length;
        const failureCount = results.filter(r => !r.success).length;

        res.json({
          results,
          summary: {
            total: templates.length,
            success: successCount,
            failed: failureCount
          }
        });
      } catch (error) {
        await client.query('ROLLBACK');
        throw error;
      } finally {
        client.release();
      }
    } catch (error: any) {
      console.error("Bulk update component templates error:", error);
      res.status(500).json({ error: "Failed to bulk update templates", details: error.message });
    }
  });

  // POST /api/component-templates/duplicate-furniture-type - Duplikuj wszystkie formatki danego rodzaju
  app.post("/api/component-templates/duplicate-furniture-type", isAuthenticated, async (req, res) => {
    try {
      const { sourceFurnitureType, targetFurnitureType } = req.body;

      if (!sourceFurnitureType || !targetFurnitureType) {
        return res.status(400).json({ error: "Source and target furniture types are required" });
      }

      console.log(`🔄 Duplicating furniture type: ${sourceFurnitureType} -> ${targetFurnitureType}`);

      // Get all templates for source furniture type
      const sourceTemplates = await pool.query(`
        SELECT * FROM bom.component_templates
        WHERE furniture_type = $1 AND is_active = true
      `, [sourceFurnitureType]);

      if (sourceTemplates.rows.length === 0) {
        return res.status(404).json({ error: "No templates found for source furniture type" });
      }

      // Duplicate each template with new furniture_type
      let duplicatedCount = 0;
      for (const template of sourceTemplates.rows) {
        await pool.query(`
          INSERT INTO bom.component_templates (
            furniture_type, cz1, cz2,
            base_length, base_width, thickness,
            color, edging_pattern, plate_type, edging_material,
            furniture_length_condition, furniture_width_condition,
            alternative_lengths, alternative_widths,
            drilling_required, no_color_change, exclude_from_cutting,
            apply_by_length, apply_by_width, half_plate,
            production_category, unit_price, is_active
          ) VALUES (
            $1, $2, $3, $4, $5, $6, $7, $8, $9, $10,
            $11, $12, $13::integer[], $14::integer[],
            $15, $16, $17, $18, $19, $20, $21, $22, $23
          )
          ON CONFLICT (furniture_type, cz1, cz2, base_length, base_width) DO NOTHING
        `, [
          targetFurnitureType, template.cz1, template.cz2,
          template.base_length, template.base_width, template.thickness,
          template.color, template.edging_pattern, template.plate_type, template.edging_material,
          template.furniture_length_condition, template.furniture_width_condition,
          template.alternative_lengths, template.alternative_widths,
          template.drilling_required, template.no_color_change, template.exclude_from_cutting,
          template.apply_by_length, template.apply_by_width, template.half_plate,
          targetFurnitureType, // production_category = target furniture type
          template.unit_price, template.is_active
        ]);
        duplicatedCount++;
      }

      console.log(`✅ Duplicated ${duplicatedCount} templates to ${targetFurnitureType}`);

      res.json({ 
        success: true, 
        duplicated: duplicatedCount, 
        sourceFurnitureType, 
        targetFurnitureType 
      });
    } catch (error: any) {
      console.error("Duplicate furniture type error:", error);
      res.status(500).json({ error: "Failed to duplicate furniture type", details: error.message });
    }
  });

  // ============================================================================
  // BOM GENERATION - Matching Algorithm & Metrics
  // ============================================================================

  // Helper function: Calculate and persist component metrics (board area, edging, glue, costs)
  async function persistComponentMe