/**
 * System profesjonalnej numeracji zamówień
 * Oparty na najlepszych praktykach ERP i systemów finansowych
 * 
 * Kluczowe zasady:
 * 1. Numer nadawany TYLKO po pomyślnym zapisie zamówienia (nie przez DEFAULT nextval)
 * 2. Atomowość - SELECT FOR UPDATE zapewnia brak duplikatów i luk
 * 3. Audyt - każde nadanie numeru jest logowane
 * 4. Wykrywanie luk - możliwość sprawdzenia ciągłości numeracji
 */

import { PoolClient } from 'pg';
import { pool } from './postgres';

/**
 * Nadaje kolejny numer zamówienia w sposób atomowy i bezpieczny
 * Format: "00001", "00002", ... "00065" (5 cyfr z wiodącymi zerami)
 * 
 * @param client - Klient PostgreSQL (transakcja)
 * @param orderId - ID zamówienia w tabeli commerce.orders
 * @param source - Źródło zamówienia (ALLEGRO/SHOPER)
 * @param sourceOrderId - ID zamówienia w systemie źródłowym
 * @returns Nowy, unikalny numer zamówienia w formacie "00001"
 */
export async function getNextOrderNumber(
  client: PoolClient,
  orderId: number,
  source: string,
  sourceOrderId: string
): Promise<string> {
  try {
    // KROK 1: Pobierz i zablokuj sekwencję (SELECT FOR UPDATE)
    // Dzięki temu tylko jedna transakcja może nadać numer w danym momencie
    const sequenceResult = await client.query(`
      SELECT current_value 
      FROM commerce.order_sequences 
      WHERE sequence_name = 'main_order_sequence'
      FOR UPDATE
    `);

    if (sequenceResult.rows.length === 0) {
      throw new Error('❌ Brak sekwencji main_order_sequence w tabeli order_sequences');
    }

    const currentValue = sequenceResult.rows[0].current_value;
    const nextValue = currentValue + 1;
    
    // Formatuj numer z wiodącymi zerami (5 cyfr)
    const nextOrderNumber = String(nextValue).padStart(5, '0');

    // KROK 2: Zaktualizuj sekwencję (increment)
    await client.query(`
      UPDATE commerce.order_sequences 
      SET current_value = $1, last_updated = NOW()
      WHERE sequence_name = 'main_order_sequence'
    `, [nextValue]);

    // KROK 3: Zapisz w audyt logu
    await client.query(`
      INSERT INTO commerce.order_number_audit (
        order_id, order_number, source, source_order_id, assigned_at, assigned_by, notes
      ) VALUES ($1, $2, $3, $4, NOW(), 'SYSTEM', $5)
    `, [
      orderId,
      nextOrderNumber,
      source,
      sourceOrderId,
      `Numer nadany automatycznie podczas synchronizacji ${source}`
    ]);

    console.log(`🔢 [ORDER NUMBERING] Nadano numer #${nextOrderNumber} dla zamówienia ${source}-${sourceOrderId} (commerce.orders.id=${orderId})`);

    return nextOrderNumber;
  } catch (error) {
    console.error('❌ [ORDER NUMBERING] Błąd podczas nadawania numeru zamówienia:', error);
    throw error;
  }
}

/**
 * Sprawdza czy są luki w numeracji zamówień
 * 
 * @returns Lista luk w numeracji (np. [{gap_start: "00005", gap_end: "00007", missing_count: 2}])
 */
export async function checkOrderNumberGaps(): Promise<Array<{
  gap_start: string;
  gap_end: string;
  missing_count: number;
}>> {
  const client = await pool.connect();
  try {
    const result = await client.query(`
      WITH numbered_orders AS (
        SELECT 
          order_number::INTEGER as num,
          LEAD(order_number::INTEGER) OVER (ORDER BY order_number::INTEGER) as next_num
        FROM commerce.orders
        WHERE order_number IS NOT NULL
        ORDER BY order_number::INTEGER
      )
      SELECT 
        LPAD((num + 1)::TEXT, 5, '0') as gap_start,
        LPAD((next_num - 1)::TEXT, 5, '0') as gap_end,
        next_num - num - 1 as missing_count
      FROM numbered_orders
      WHERE next_num - num > 1
      ORDER BY num
    `);

    if (result.rows.length === 0) {
      console.log('✅ [ORDER NUMBERING] Brak luk w numeracji zamówień');
      return [];
    }

    console.log(`⚠️ [ORDER NUMBERING] Wykryto ${result.rows.length} luk(i) w numeracji:`);
    result.rows.forEach((gap: { gap_start: string; gap_end: string; missing_count: number }) => {
      console.log(`   - Luka: numery ${gap.gap_start}-${gap.gap_end} (${gap.missing_count} brakujących)`);
    });

    return result.rows;
  } finally {
    client.release();
  }
}

/**
 * Pobiera statystyki numeracji zamówień
 */
export async function getOrderNumberingStats(): Promise<{
  total_orders: number;
  current_sequence_value: number;
  lowest_number: string;
  highest_number: string;
  gaps_count: number;
}> {
  const client = await pool.connect();
  try {
    // Pobierz statystyki
    const statsResult = await client.query(`
      SELECT 
        COUNT(*) FILTER (WHERE order_number IS NOT NULL) as total_orders,
        MIN(order_number) as lowest_number,
        MAX(order_number) as highest_number
      FROM commerce.orders
    `);

    const sequenceResult = await client.query(`
      SELECT current_value 
      FROM commerce.order_sequences 
      WHERE sequence_name = 'main_order_sequence'
    `);

    const gaps = await checkOrderNumberGaps();

    return {
      total_orders: parseInt(statsResult.rows[0].total_orders || '0'),
      current_sequence_value: sequenceResult.rows[0]?.current_value || 0,
      lowest_number: statsResult.rows[0].lowest_number || '00000',
      highest_number: statsResult.rows[0].highest_number || '00000',
      gaps_count: gaps.length
    };
  } finally {
    client.release();
  }
}

/**
 * Naprawia numer zamówienia (tylko dla administracji!)
 * UWAGA: Używać tylko w wyjątkowych sytuacjach, po konsultacji
 * 
 * @param orderId - ID zamówienia w commerce.orders
 * @param newOrderNumber - Nowy numer do nadania (format "00001")
 * @param reason - Powód zmiany (do audytu)
 */
export async function fixOrderNumber(
  orderId: number,
  newOrderNumber: string,
  reason: string
): Promise<void> {
  const client = await pool.connect();
  try {
    await client.query('BEGIN');

    // Sprawdź czy numer już nie istnieje
    const existingCheck = await client.query(`
      SELECT id FROM commerce.orders WHERE order_number = $1
    `, [newOrderNumber]);

    if (existingCheck.rows.length > 0 && existingCheck.rows[0].id !== orderId) {
      throw new Error(`❌ Numer ${newOrderNumber} jest już przypisany do innego zamówienia`);
    }

    // Pobierz dane zamówienia
    const orderData = await client.query(`
      SELECT source, source_order_id, order_number as old_number
      FROM commerce.orders
      WHERE id = $1
    `, [orderId]);

    if (orderData.rows.length === 0) {
      throw new Error(`❌ Nie znaleziono zamówienia o id=${orderId}`);
    }

    const { source, source_order_id, old_number } = orderData.rows[0];

    // Zaktualizuj numer
    await client.query(`
      UPDATE commerce.orders 
      SET order_number = $1, updated_at = NOW()
      WHERE id = $2
    `, [newOrderNumber, orderId]);

    // Zapisz w audyt logu
    await client.query(`
      INSERT INTO commerce.order_number_audit (
        order_id, order_number, source, source_order_id, assigned_at, assigned_by, notes
      ) VALUES ($1, $2, $3, $4, NOW(), 'ADMIN_FIX', $5)
    `, [
      orderId,
      newOrderNumber,
      source,
      source_order_id,
      `RĘCZNA KOREKTA: zmiana ${old_number} → ${newOrderNumber}. Powód: ${reason}`
    ]);

    await client.query('COMMIT');

    console.log(`🔧 [ORDER NUMBERING] Naprawiono numer zamówienia id=${orderId}: ${old_number} → ${newOrderNumber}`);
  } catch (error) {
    await client.query('ROLLBACK');
    console.error('❌ [ORDER NUMBERING] Błąd podczas naprawy numeru:', error);
    throw error;
  } finally {
    client.release();
  }
}
