import { useState, useEffect, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Checkbox } from "@/components/ui/checkbox";
import { Badge } from "@/components/ui/badge";
import { 
  Select, 
  SelectContent, 
  SelectItem, 
  SelectTrigger, 
  SelectValue 
} from "@/components/ui/select";
import { 
  FileCheck, 
  Package, 
  MapPin, 
  Building2,
  Truck,
  ArrowLeft
} from "lucide-react";
import { useToast } from "@/hooks/use-toast";
import { apiRequest, queryClient } from "@/lib/queryClient";

interface Material {
  id: number;
  name: string;
  internalCode: string;
  quantity: string;
  unitOfMeasure: string;
  categoryName: string;
  groupName?: string;
  locationName?: string;
  carrierName?: string;
  locationId?: number;
  carrierId?: number;
  itemType: 'material' | 'stock_panel' | 'packaging_material';
  dimensions?: string;
  colorCode?: string;
}

interface Location {
  id: number;
  name: string;
  level: number;
  parentId: number | null;
}

interface Carrier {
  id: number;
  name: string;
  type: string;
}

export default function WarehouseInventoryCountNewPage() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  // Form state
  const [countName, setCountName] = useState(() => {
    // Generate default inventory name: "Spis inwentaryzacyjny DD-MM-YYYY"
    const today = new Date();
    const day = String(today.getDate()).padStart(2, '0');
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const year = today.getFullYear();
    return `Spis inwentaryzacyjny ${day}-${month}-${year}`;
  });
  const [isCountNameDirty, setIsCountNameDirty] = useState(false);
  const [notes, setNotes] = useState("");
  const [selectedCategory, setSelectedCategory] = useState<string>("all");
  const [selectedLocationId, setSelectedLocationId] = useState<string>("all");
  const [selectedCarrierId, setSelectedCarrierId] = useState<string>("all");
  const [selectedMaterialIds, setSelectedMaterialIds] = useState<Set<number>>(new Set());

  const categories = [
    { value: "all", label: "Wszystkie kategorie" },
    { value: "tkaniny", label: "Tkaniny" },
    { value: "pianki", label: "Wypełnienia" },
    { value: "plyty", label: "Płyty meblowe" },
    { value: "obrzeza", label: "Obrzeża" },
    { value: "okucia", label: "Okucia" },
    { value: "sruby", label: "Śruby i łączniki" },
    { value: "formatki", label: "Formatki" },
    { value: "opakowania", label: "Opakowania" },
  ];

  // Fetch locations (halls)
  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
  });

  // Fetch carriers
  const { data: carriers = [] } = useQuery<Carrier[]>({
    queryKey: ["/api/production/carriers"],
  });

  // Fetch inventory items (materials/formatki/opakowania) with filters
  const { data: materials = [], isLoading } = useQuery<Material[]>({
    queryKey: [
      "/api/warehouse/inventory-items",
      { 
        category: selectedCategory === "all" ? undefined : selectedCategory,
        locationId: selectedLocationId === "all" ? undefined : selectedLocationId,
        carrierId: selectedCarrierId === "all" ? undefined : selectedCarrierId,
      }
    ],
  });

  // Debug logging
  useEffect(() => {
    console.log('🔍 [FRONTEND] Materials data:', { 
      selectedCategory, 
      materialsCount: materials.length, 
      isLoading,
      materials: materials.slice(0, 3)
    });
  }, [materials, selectedCategory, isLoading]);

  // Stable projection of selected material IDs (sorted array for dependency tracking)
  const selectedMaterialIdsArray = useMemo(() => 
    Array.from(selectedMaterialIds).sort((a, b) => a - b),
    [selectedMaterialIds]
  );

  // Auto-generate inventory count name when category or selected materials change (only if user hasn't manually edited)
  // Uses same logic as warehouse-category.tsx: DD-MM-YYYY - group1-group2
  useEffect(() => {
    // Don't auto-update if user manually edited the name
    if (isCountNameDirty) return;
    
    const today = new Date();
    const day = String(today.getDate()).padStart(2, '0');
    const month = String(today.getMonth() + 1).padStart(2, '0');
    const year = today.getFullYear();
    const dateStr = `${day}-${month}-${year}`;
    
    // Extract unique group names from selected materials (same as warehouse-category.tsx line 2553)
    const selectedMaterialsData = materials.filter(m => selectedMaterialIds.has(m.id));
    const uniqueGroups = Array.from(new Set(selectedMaterialsData.map(m => m.groupName).filter((g): g is string => Boolean(g))));
    const groupsStr = uniqueGroups.join('-');
    
    const generatedName = `Spis inwentaryzacyjny ${dateStr}${groupsStr ? ' - ' + groupsStr : ''}`;
    setCountName(generatedName);
  }, [selectedCategory, selectedLocationId, selectedCarrierId, selectedMaterialIdsArray, materials, isCountNameDirty]);

  const createCountMutation = useMutation({
    mutationFn: async (data: {
      name: string;
      materialIds?: number[];
      panelIds?: number[];
      packagingMaterialIds?: number[];
      notes?: string;
    }) => {
      const res = await apiRequest("POST", "/api/warehouse/inventory-counts", data);
      return await res.json() as {
        inventoryCount: { id: number };
        items: any[];
      };
    },
    onSuccess: (response) => {
      queryClient.invalidateQueries({ queryKey: ["/api/warehouse/inventory-counts"] });
      toast({
        title: "Sukces",
        description: "Spis inwentaryzacyjny został utworzony",
      });
      setLocation(`/warehouse/inventory-counts/${response.inventoryCount.id}`);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć spisu",
        variant: "destructive",
      });
    },
  });

  const handleToggleMaterial = (materialId: number) => {
    const newSelected = new Set(selectedMaterialIds);
    if (newSelected.has(materialId)) {
      newSelected.delete(materialId);
    } else {
      newSelected.add(materialId);
    }
    setSelectedMaterialIds(newSelected);
  };

  const handleSelectAll = () => {
    if (selectedMaterialIds.size === materials.length) {
      setSelectedMaterialIds(new Set());
    } else {
      setSelectedMaterialIds(new Set(materials.map(m => m.id)));
    }
  };

  const handleCreateCount = () => {
    if (!countName.trim()) {
      toast({
        title: "Błąd",
        description: "Podaj nazwę spisu",
        variant: "destructive",
      });
      return;
    }

    if (selectedMaterialIds.size === 0) {
      toast({
        title: "Błąd",
        description: "Zaznacz przynajmniej jeden materiał",
        variant: "destructive",
      });
      return;
    }

    // Determine item type from selected materials
    const selectedItems = materials.filter(m => selectedMaterialIds.has(m.id));
    if (selectedItems.length === 0) {
      toast({
        title: "Błąd",
        description: "Nie wybrano żadnych pozycji",
        variant: "destructive",
      });
      return;
    }

    const itemType = selectedItems[0].itemType;
    
    // Prepare data based on item type
    const data: any = {
      name: countName,
      notes: notes || undefined,
    };

    if (itemType === 'stock_panel') {
      data.panelIds = Array.from(selectedMaterialIds);
    } else if (itemType === 'packaging_material') {
      data.packagingMaterialIds = Array.from(selectedMaterialIds);
    } else {
      data.materialIds = Array.from(selectedMaterialIds);
    }

    createCountMutation.mutate(data);
  };

  // Get hall locations (level 1)
  const hallLocations = locations.filter(loc => loc.level === 1);

  return (
    <WarehouseLayout category="all">
      <div className="space-y-6 p-6">
        {/* Header */}
        <div className="flex items-center gap-4">
          <Button
            variant="ghost"
            size="icon"
            onClick={() => setLocation("/warehouse/inventory-counts")}
            data-testid="button-back"
          >
            <ArrowLeft className="h-4 w-4" />
          </Button>
          <div>
            <h1 className="text-3xl font-bold">Nowy spis inwentaryzacyjny</h1>
            <p className="text-sm text-muted-foreground mt-1">
              Wybierz materiały do spisania, filtruj po lokalizacji
            </p>
          </div>
        </div>

        {/* Configuration Card */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <FileCheck className="h-5 w-5" />
              Konfiguracja spisu
            </CardTitle>
            <CardDescription>
              Zdefiniuj nazwę i zakres spisu
            </CardDescription>
          </CardHeader>
          <CardContent className="space-y-4">
            <div className="grid gap-4 md:grid-cols-2">
              <div>
                <Label htmlFor="count-name">Nazwa spisu *</Label>
                <Input
                  id="count-name"
                  placeholder="np. Spis Okuć Hala1 - Grudzień 2024"
                  value={countName}
                  onChange={(e) => {
                    setCountName(e.target.value);
                    setIsCountNameDirty(true);
                  }}
                  data-testid="input-count-name"
                />
              </div>
              <div>
                <Label htmlFor="notes">Notatki</Label>
                <Input
                  id="notes"
                  placeholder="Dodatkowe informacje"
                  value={notes}
                  onChange={(e) => setNotes(e.target.value)}
                  data-testid="input-notes"
                />
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Filters Card */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <MapPin className="h-5 w-5" />
              Filtrowanie materiałów
            </CardTitle>
            <CardDescription>
              Użyj filtrów aby wybrać konkretną lokalizację lub kategorię
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="grid gap-4 md:grid-cols-3">
              {/* Category filter */}
              <div>
                <Label htmlFor="category-filter">Kategoria</Label>
                <Select value={selectedCategory} onValueChange={setSelectedCategory}>
                  <SelectTrigger id="category-filter" data-testid="select-category">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    {categories.map((cat) => (
                      <SelectItem key={cat.value} value={cat.value}>
                        {cat.label}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {/* Location filter */}
              <div>
                <Label htmlFor="location-filter" className="flex items-center gap-1">
                  <Building2 className="h-4 w-4" />
                  Hala
                </Label>
                <Select value={selectedLocationId} onValueChange={setSelectedLocationId}>
                  <SelectTrigger id="location-filter" data-testid="select-location">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie lokalizacje</SelectItem>
                    {hallLocations.map((loc) => (
                      <SelectItem key={loc.id} value={loc.id.toString()}>
                        {loc.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              {/* Carrier filter */}
              <div>
                <Label htmlFor="carrier-filter" className="flex items-center gap-1">
                  <Truck className="h-4 w-4" />
                  Nośnik
                </Label>
                <Select value={selectedCarrierId} onValueChange={setSelectedCarrierId}>
                  <SelectTrigger id="carrier-filter" data-testid="select-carrier">
                    <SelectValue />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie nośniki</SelectItem>
                    {carriers.map((carrier) => (
                      <SelectItem key={carrier.id} value={carrier.id.toString()}>
                        {carrier.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Materials Selection */}
        <Card>
          <CardHeader>
            <div className="flex items-center justify-between">
              <div>
                <CardTitle className="flex items-center gap-2">
                  <Package className="h-5 w-5" />
                  Materiały do spisania
                </CardTitle>
                <CardDescription className="mt-1">
                  Zaznacz materiały które mają zostać uwzględnione w spisie
                </CardDescription>
              </div>
              <div className="flex items-center gap-3">
                <Badge variant="outline">
                  Zaznaczono: {selectedMaterialIds.size} / {materials.length}
                </Badge>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={handleSelectAll}
                  data-testid="button-select-all"
                >
                  {selectedMaterialIds.size === materials.length ? "Odznacz wszystko" : "Zaznacz wszystko"}
                </Button>
              </div>
            </div>
          </CardHeader>
          <CardContent>
            {isLoading ? (
              <div className="text-center py-8 text-muted-foreground">Ładowanie materiałów...</div>
            ) : materials.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground">
                Brak materiałów dla wybranych filtrów
              </div>
            ) : (
              <div className="space-y-2 max-h-96 overflow-y-auto">
                {materials.map((material) => (
                  <div
                    key={material.id}
                    className="flex items-center gap-3 p-3 border rounded-md hover-elevate"
                    data-testid={`material-item-${material.id}`}
                  >
                    <Checkbox
                      checked={selectedMaterialIds.has(material.id)}
                      onCheckedChange={() => handleToggleMaterial(material.id)}
                      data-testid={`checkbox-material-${material.id}`}
                    />
                    <div className="flex-1 min-w-0">
                      <div className="flex items-center gap-2 flex-wrap">
                        <p className="font-medium truncate">{material.name}</p>
                        <Badge variant="secondary" className="text-xs">
                          {material.internalCode}
                        </Badge>
                        {material.locationName && (
                          <Badge variant="outline" className="text-xs">
                            <Building2 className="h-3 w-3 mr-1" />
                            {material.locationName}
                          </Badge>
                        )}
                        {material.carrierName && (
                          <Badge variant="outline" className="text-xs">
                            <Truck className="h-3 w-3 mr-1" />
                            {material.carrierName}
                          </Badge>
                        )}
                      </div>
                      <p className="text-sm text-muted-foreground">
                        Stan: {material.quantity} {material.unitOfMeasure}
                      </p>
                    </div>
                  </div>
                ))}
              </div>
            )}
          </CardContent>
        </Card>

        {/* Actions */}
        <div className="flex items-center justify-between">
          <Button
            variant="outline"
            onClick={() => setLocation("/warehouse/inventory-counts")}
            data-testid="button-cancel"
          >
            Anuluj
          </Button>
          <Button
            onClick={handleCreateCount}
            disabled={createCountMutation.isPending || selectedMaterialIds.size === 0 || !countName.trim()}
            data-testid="button-create-count"
          >
            <FileCheck className="h-4 w-4 mr-2" />
            {createCountMutation.isPending ? "Tworzenie..." : `Utwórz spis (${selectedMaterialIds.size} pozycji)`}
          </Button>
        </div>
      </div>
    </WarehouseLayout>
  );
}
