import { useQuery, useMutation } from "@tanstack/react-query";
import { WarehouseLayout } from "@/features/warehouse/warehouse-layout";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { FileText, Plus, Trash2, CheckCircle, Ban, Printer, Eye, Filter, X } from "lucide-react";
import { Link, useLocation } from "wouter";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { useState } from "react";
import { Combobox, ComboboxOption } from "@/components/ui/combobox";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";

interface WarehouseDocument {
  id: number;
  docNumber: string;
  docType: string;
  status: string;
  planId: number | null;
  planName: string | null;
  targetDepartment: string | null;
  targetLocationId: number | null;
  targetLocationName: string | null;
  issuedAt: string | null;
  issuedBy: number | null;
  issuedByName: string | null;
  confirmedAt: string | null;
  confirmedByName: string | null;
  completedAt: string | null;
  completedByName: string | null;
  remarks: string | null;
  totalLines: number;
  totalQuantity: number;
  createdAt: string;
  updatedAt: string;
}

interface DocumentsResponse {
  documents: WarehouseDocument[];
  pagination: {
    limit: number;
    offset: number;
    total: number;
  };
  stats: {
    byType: Record<string, number>;
    byStatus: Record<string, number>;
    total: number;
  };
}

const DOCUMENT_TYPE_LABELS: Record<string, string> = {
  'WZ-SPAK': 'Produkty spakowane',
  'WZ-TAP': 'Tapicernia',
  'WZ-FORM': 'Formatki',
  'WZ-OKUC': 'Okucia',
  'WZ-OPAK': 'Opakowania',
};

const STATUS_LABELS: Record<string, string> = {
  'draft': 'Szkic',
  'confirmed': 'Potwierdzony',
  'printed': 'Wydrukowany',
  'completed': 'Zrealizowany',
  'cancelled': 'Anulowany',
};

const STATUS_VARIANTS: Record<string, "secondary" | "default" | "outline" | "destructive"> = {
  'draft': 'secondary',
  'confirmed': 'default',
  'printed': 'default',
  'completed': 'outline',
  'cancelled': 'destructive',
};

export default function WarehouseDocumentsPage() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [selectedDocType, setSelectedDocType] = useState<string>('');
  const [selectedStatus, setSelectedStatus] = useState<string>('');
  const [deleteDialogOpen, setDeleteDialogOpen] = useState(false);
  const [cancelDialogOpen, setCancelDialogOpen] = useState(false);
  const [confirmDialogOpen, setConfirmDialogOpen] = useState(false);
  const [completeDialogOpen, setCompleteDialogOpen] = useState(false);
  const [selectedDoc, setSelectedDoc] = useState<WarehouseDocument | null>(null);
  const [activeTab, setActiveTab] = useState<string>('all');

  const queryParams = new URLSearchParams();
  if (selectedDocType) queryParams.append('docType', selectedDocType);
  if (selectedStatus) queryParams.append('status', selectedStatus);
  if (activeTab && activeTab !== 'all') queryParams.append('statusTab', activeTab);
  const queryString = queryParams.toString();

  const { data, isLoading } = useQuery<DocumentsResponse>({
    queryKey: ['/api/warehouse/documents', { docType: selectedDocType, status: selectedStatus, statusTab: activeTab }],
    queryFn: async () => {
      const url = queryString ? `/api/warehouse/documents?${queryString}` : '/api/warehouse/documents';
      const response = await fetch(url, { credentials: 'include' });
      if (!response.ok) throw new Error('Failed to fetch documents');
      return response.json();
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('DELETE', `/api/warehouse/documents/${id}`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Dokument został usunięty" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents'] });
      setDeleteDialogOpen(false);
      setSelectedDoc(null);
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się usunąć dokumentu", variant: "destructive" });
    },
  });

  const cancelMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('POST', `/api/warehouse/documents/${id}/cancel`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Dokument został anulowany" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents'] });
      setCancelDialogOpen(false);
      setSelectedDoc(null);
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się anulować dokumentu", variant: "destructive" });
    },
  });

  const confirmMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('POST', `/api/warehouse/documents/${id}/confirm`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Dokument został potwierdzony" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents'] });
      setConfirmDialogOpen(false);
      setSelectedDoc(null);
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się potwierdzić dokumentu", variant: "destructive" });
    },
  });

  const completeMutation = useMutation({
    mutationFn: async (id: number) => {
      return await apiRequest('POST', `/api/warehouse/documents/${id}/complete`);
    },
    onSuccess: () => {
      toast({ title: "Sukces", description: "Dokument został zrealizowany" });
      queryClient.invalidateQueries({ queryKey: ['/api/warehouse/documents'] });
      setCompleteDialogOpen(false);
      setSelectedDoc(null);
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message || "Nie udało się zrealizować dokumentu", variant: "destructive" });
    },
  });

  const documents = data?.documents || [];
  const stats = data?.stats || { byType: {}, byStatus: {}, total: 0 };

  const docTypeOptions: ComboboxOption[] = [
    { value: '', label: 'Wszystkie typy' },
    ...Object.entries(DOCUMENT_TYPE_LABELS).map(([value, label]) => ({
      value,
      label: `${value} - ${label}`,
    })),
  ];

  const statusOptions: ComboboxOption[] = [
    { value: '', label: 'Wszystkie statusy' },
    ...Object.entries(STATUS_LABELS).map(([value, label]) => ({ value, label })),
  ];

  const handleDelete = (doc: WarehouseDocument) => {
    setSelectedDoc(doc);
    setDeleteDialogOpen(true);
  };

  const handleCancel = (doc: WarehouseDocument) => {
    setSelectedDoc(doc);
    setCancelDialogOpen(true);
  };

  const handleConfirm = (doc: WarehouseDocument) => {
    setSelectedDoc(doc);
    setConfirmDialogOpen(true);
  };

  const handleComplete = (doc: WarehouseDocument) => {
    setSelectedDoc(doc);
    setCompleteDialogOpen(true);
  };

  const clearFilters = () => {
    setSelectedDocType('');
    setSelectedStatus('');
  };

  const formatDate = (dateStr: string | null) => {
    if (!dateStr) return '-';
    return new Date(dateStr).toLocaleDateString('pl-PL', {
      year: 'numeric',
      month: '2-digit',
      day: '2-digit',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const getStatusBadge = (status: string) => {
    return (
      <Badge variant={STATUS_VARIANTS[status] || 'secondary'} data-testid={`badge-status-${status}`}>
        {STATUS_LABELS[status] || status}
      </Badge>
    );
  };

  const getDocTypeBadge = (docType: string) => {
    return (
      <Badge variant="outline" className="font-mono text-xs" data-testid={`badge-doctype-${docType}`}>
        {docType}
      </Badge>
    );
  };

  const activeCount = (stats.byStatus['draft'] || 0) + (stats.byStatus['confirmed'] || 0) + (stats.byStatus['printed'] || 0);
  const completedCount = stats.byStatus['completed'] || 0;
  const cancelledCount = stats.byStatus['cancelled'] || 0;

  if (isLoading) {
    return (
      <WarehouseLayout category="all" hideSubmenu={true}>
        <div className="p-6">Ładowanie...</div>
      </WarehouseLayout>
    );
  }

  return (
    <WarehouseLayout category="all" hideSubmenu={true}>
      <div className="p-6 space-y-6">
        <div className="flex items-center justify-between gap-4">
          <div className="flex items-center gap-3">
            <FileText className="w-6 h-6 text-muted-foreground" />
            <div>
              <h1 className="text-2xl font-bold">Dokumenty magazynowe</h1>
              <p className="text-sm text-muted-foreground">Wydania materiałów z magazynu</p>
            </div>
          </div>
          <Link href="/warehouse/documents/new">
            <Button data-testid="button-create-document">
              <Plus className="w-4 h-4 mr-2" />
              Nowy dokument
            </Button>
          </Link>
        </div>

        <div className="grid grid-cols-2 md:grid-cols-5 gap-3">
          {Object.entries(DOCUMENT_TYPE_LABELS).map(([type, label]) => (
            <Card
              key={type}
              className={`cursor-pointer transition-colors ${selectedDocType === type ? 'ring-2 ring-primary' : 'hover-elevate'}`}
              onClick={() => setSelectedDocType(selectedDocType === type ? '' : type)}
              data-testid={`card-doctype-${type}`}
            >
              <CardHeader className="p-3 pb-1">
                <CardDescription className="text-xs font-mono">{type}</CardDescription>
              </CardHeader>
              <CardContent className="p-3 pt-0">
                <div className="text-2xl font-bold">{stats.byType[type] || 0}</div>
                <p className="text-xs text-muted-foreground">{label}</p>
              </CardContent>
            </Card>
          ))}
        </div>

        <div className="flex flex-wrap items-center gap-3">
          <div className="flex items-center gap-2">
            <Filter className="w-4 h-4 text-muted-foreground" />
            <span className="text-sm text-muted-foreground">Filtry:</span>
          </div>
          <div className="w-48">
            <Combobox
              options={docTypeOptions}
              value={selectedDocType}
              onChange={setSelectedDocType}
              placeholder="Typ dokumentu"
            />
          </div>
          <div className="w-40">
            <Combobox
              options={statusOptions}
              value={selectedStatus}
              onChange={setSelectedStatus}
              placeholder="Status"
            />
          </div>
          {(selectedDocType || selectedStatus) && (
            <Button variant="ghost" size="sm" onClick={clearFilters} data-testid="button-clear-filters">
              <X className="w-4 h-4 mr-1" />
              Wyczyść
            </Button>
          )}
        </div>

        <Tabs value={activeTab} onValueChange={setActiveTab}>
          <TabsList>
            <TabsTrigger value="all" data-testid="tab-all">
              Wszystkie ({stats.total})
            </TabsTrigger>
            <TabsTrigger value="active" data-testid="tab-active">
              Aktywne ({activeCount})
            </TabsTrigger>
            <TabsTrigger value="completed" data-testid="tab-completed">
              Zrealizowane ({completedCount})
            </TabsTrigger>
            <TabsTrigger value="cancelled" data-testid="tab-cancelled">
              Anulowane ({cancelledCount})
            </TabsTrigger>
          </TabsList>

          <TabsContent value={activeTab} className="mt-4">
            {documents.length === 0 ? (
              <div className="text-center py-12 text-muted-foreground">
                <FileText className="w-12 h-12 mx-auto mb-4 opacity-50" />
                <p>Brak dokumentów do wyświetlenia</p>
              </div>
            ) : (
              <div className="border rounded-md">
                <Table>
                  <TableHeader>
                    <TableRow className="bg-muted/50">
                      <TableHead className="w-[4%] text-center">Lp.</TableHead>
                      <TableHead className="w-[14%]">Numer dokumentu</TableHead>
                      <TableHead className="w-[10%]">Status</TableHead>
                      <TableHead className="w-[15%]">Plan</TableHead>
                      <TableHead className="w-[12%]">Cel</TableHead>
                      <TableHead className="w-[8%] text-center">Poz.</TableHead>
                      <TableHead className="w-[8%] text-center">Ilość</TableHead>
                      <TableHead className="w-[12%]">Data</TableHead>
                      <TableHead className="w-[13%] text-right">Akcje</TableHead>
                    </TableRow>
                  </TableHeader>
                  <TableBody>
                    {documents.map((doc, index) => (
                      <TableRow key={doc.id} data-testid={`row-document-${doc.id}`}>
                        <TableCell className="text-center text-muted-foreground font-mono text-sm">
                          {index + 1}
                        </TableCell>
                        <TableCell>
                          <Link href={`/warehouse/documents/${doc.id}`}>
                            <span className="font-mono text-sm hover:underline cursor-pointer" data-testid={`link-document-${doc.id}`}>
                              <Badge variant="outline" className="mr-2 font-mono text-xs">{doc.docType}</Badge>
                              {doc.docNumber}
                            </span>
                          </Link>
                        </TableCell>
                        <TableCell>{getStatusBadge(doc.status)}</TableCell>
                        <TableCell>
                          {doc.planName ? (
                            <Link href={`/production/plans/${doc.planId}`}>
                              <span className="text-sm hover:underline cursor-pointer">{doc.planName}</span>
                            </Link>
                          ) : (
                            <span className="text-muted-foreground">-</span>
                          )}
                        </TableCell>
                        <TableCell>
                          <span className="text-sm">{doc.targetDepartment || doc.targetLocationName || '-'}</span>
                        </TableCell>
                        <TableCell className="text-center">
                          <span className="font-mono">{doc.totalLines}</span>
                        </TableCell>
                        <TableCell className="text-center">
                          <span className="font-mono">{doc.totalQuantity}</span>
                        </TableCell>
                        <TableCell>
                          <span className="text-sm text-muted-foreground">{formatDate(doc.createdAt)}</span>
                        </TableCell>
                        <TableCell className="text-right">
                          <div className="flex items-center justify-end gap-1">
                            <Link href={`/warehouse/documents/${doc.id}`}>
                              <Button variant="ghost" size="icon" data-testid={`button-view-${doc.id}`}>
                                <Eye className="w-4 h-4" />
                              </Button>
                            </Link>
                            {doc.status === 'draft' && (
                              <>
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  onClick={() => handleConfirm(doc)}
                                  data-testid={`button-confirm-${doc.id}`}
                                >
                                  <CheckCircle className="w-4 h-4 text-green-600" />
                                </Button>
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  onClick={() => handleDelete(doc)}
                                  data-testid={`button-delete-${doc.id}`}
                                >
                                  <Trash2 className="w-4 h-4 text-destructive" />
                                </Button>
                              </>
                            )}
                            {(doc.status === 'confirmed' || doc.status === 'printed') && (
                              <>
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  onClick={() => window.open(`/api/warehouse/documents/${doc.id}/pdf`, '_blank')}
                                  data-testid={`button-print-${doc.id}`}
                                  title="Drukuj PDF"
                                >
                                  <Printer className="w-4 h-4 text-blue-600" />
                                </Button>
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  onClick={() => handleComplete(doc)}
                                  data-testid={`button-complete-${doc.id}`}
                                >
                                  <CheckCircle className="w-4 h-4 text-green-600" />
                                </Button>
                                <Button
                                  variant="ghost"
                                  size="icon"
                                  onClick={() => handleCancel(doc)}
                                  data-testid={`button-cancel-${doc.id}`}
                                >
                                  <Ban className="w-4 h-4 text-destructive" />
                                </Button>
                              </>
                            )}
                          </div>
                        </TableCell>
                      </TableRow>
                    ))}
                  </TableBody>
                </Table>
              </div>
            )}
          </TabsContent>
        </Tabs>

        <AlertDialog open={deleteDialogOpen} onOpenChange={setDeleteDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Usuń dokument</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz usunąć dokument {selectedDoc?.docNumber}? Ta operacja jest nieodwracalna.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Anuluj</AlertDialogCancel>
              <AlertDialogAction
                onClick={() => selectedDoc && deleteMutation.mutate(selectedDoc.id)}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              >
                Usuń
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        <AlertDialog open={cancelDialogOpen} onOpenChange={setCancelDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Anuluj dokument</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz anulować dokument {selectedDoc?.docNumber}?
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Nie</AlertDialogCancel>
              <AlertDialogAction
                onClick={() => selectedDoc && cancelMutation.mutate(selectedDoc.id)}
                className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
              >
                Anuluj dokument
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        <AlertDialog open={confirmDialogOpen} onOpenChange={setConfirmDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Potwierdź dokument</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz potwierdzić dokument {selectedDoc?.docNumber}? Po potwierdzeniu dokument nie będzie mógł być edytowany.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Anuluj</AlertDialogCancel>
              <AlertDialogAction onClick={() => selectedDoc && confirmMutation.mutate(selectedDoc.id)}>
                Potwierdź
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>

        <AlertDialog open={completeDialogOpen} onOpenChange={setCompleteDialogOpen}>
          <AlertDialogContent>
            <AlertDialogHeader>
              <AlertDialogTitle>Zrealizuj dokument</AlertDialogTitle>
              <AlertDialogDescription>
                Czy na pewno chcesz oznaczyć dokument {selectedDoc?.docNumber} jako zrealizowany? Ta operacja potwierdzi wydanie materiałów z magazynu.
              </AlertDialogDescription>
            </AlertDialogHeader>
            <AlertDialogFooter>
              <AlertDialogCancel>Anuluj</AlertDialogCancel>
              <AlertDialogAction onClick={() => selectedDoc && completeMutation.mutate(selectedDoc.id)}>
                Zrealizuj
              </AlertDialogAction>
            </AlertDialogFooter>
          </AlertDialogContent>
        </AlertDialog>
      </div>
    </WarehouseLayout>
  );
}
