import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Link, useLocation } from "wouter";
import { Plus, Edit, Trash2, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import type { ProductionRouting } from "@shared/schema";

export default function ProductionRoutingsList() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();

  const { data: routings = [], isLoading } = useQuery<ProductionRouting[]>({
    queryKey: ["routings"],
    queryFn: async () => {
      const res = await fetch("/api/production/routings");
      if (!res.ok) throw new Error("Failed to fetch routings");
      return res.json();
    },
  });

  const deleteRoutingMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/production/routings/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["routings"] });
      toast({ title: "Marszruta została usunięta" });
    },
    onError: (error: any) => {
      toast({ 
        title: "Błąd", 
        description: error.message || "Nie udało się usunąć marszruty",
        variant: "destructive" 
      });
    },
  });

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-96">
        <Loader2 className="h-8 w-8 animate-spin" />
      </div>
    );
  }

  return (
    <div className="container mx-auto p-6 space-y-4">
      <div className="flex items-center justify-between">
        <h1 className="text-2xl font-bold">Marszruty Produkcyjne</h1>
        <Button onClick={() => setLocation("/production/routings/new")}>
          <Plus className="mr-2 h-4 w-4" />
          Nowa Marszruta
        </Button>
      </div>

      <Card>
        <CardHeader className="pb-3">
          <CardTitle className="text-lg">Lista Marszrut</CardTitle>
        </CardHeader>
        <CardContent className="p-0">
          <div className="border-t">
            <table className="w-full">
              <thead>
                <tr className="border-b bg-muted/50 text-xs">
                  <th className="text-left py-2 px-3 font-medium">Kod</th>
                  <th className="text-left py-2 px-3 font-medium">Nazwa</th>
                  <th className="text-left py-2 px-3 font-medium">Operacje</th>
                  <th className="text-left py-2 px-3 font-medium">Status</th>
                  <th className="text-right py-2 px-3 font-medium">Akcje</th>
                </tr>
              </thead>
              <tbody>
                {routings.length === 0 ? (
                  <tr>
                    <td colSpan={5} className="text-center py-8 text-muted-foreground text-sm">
                      Brak marszrut. Kliknij "Nowa Marszruta" aby dodać.
                    </td>
                  </tr>
                ) : (
                  routings.map((routing) => (
                    <tr 
                      key={routing.id} 
                      className="border-b hover:bg-muted/50 cursor-pointer h-[30px]"
                      onClick={() => setLocation(`/production/routings/${routing.id}`)}
                    >
                      <td className="py-1 px-3">
                        <span className="font-mono text-sm text-primary font-medium">
                          {routing.code}
                        </span>
                      </td>
                      <td className="py-1 px-3 text-sm">{routing.name}</td>
                      <td className="py-1 px-3 text-sm text-muted-foreground">
                        {routing.totalEstimatedTime ? `${routing.totalEstimatedTime} min` : "-"}
                      </td>
                      <td className="py-1 px-3">
                        {routing.isActive ? (
                          <Badge variant="default" className="h-5 text-xs">Aktywna</Badge>
                        ) : (
                          <Badge variant="secondary" className="h-5 text-xs">Nieaktywna</Badge>
                        )}
                      </td>
                      <td className="py-1 px-3 text-right">
                        <div className="flex gap-1 justify-end" onClick={(e) => e.stopPropagation()}>
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-7 w-7"
                            onClick={() => setLocation(`/production/routings/${routing.id}`)}
                          >
                            <Edit className="h-3.5 w-3.5" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-7 w-7"
                            onClick={() => {
                              if (confirm(`Czy na pewno chcesz usunąć marszrutę ${routing.code}?`)) {
                                deleteRoutingMutation.mutate(routing.id);
                              }
                            }}
                          >
                            <Trash2 className="h-3.5 w-3.5" />
                          </Button>
                        </div>
                      </td>
                    </tr>
                  ))
                )}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
