import { useState } from "react";
import { useLocation } from "wouter";
import { ArrowLeft, Loader2 } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { useToast } from "@/hooks/use-toast";
import { apiRequest } from "@/lib/queryClient";

export default function ProductionRoutingNew() {
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [isSubmitting, setIsSubmitting] = useState(false);
  
  const [form, setForm] = useState({
    name: "",
    description: "",
    isActive: true,
  });

  // Auto-generate code from name
  const generateCode = (name: string): string => {
    if (!name) return "";
    
    return "marsz_" + name
      .toLowerCase()
      .normalize("NFD")
      .replace(/[\u0300-\u036f]/g, "") // Remove diacritics
      .replace(/ą/g, "a")
      .replace(/ć/g, "c")
      .replace(/ę/g, "e")
      .replace(/ł/g, "l")
      .replace(/ń/g, "n")
      .replace(/ó/g, "o")
      .replace(/ś/g, "s")
      .replace(/ź/g, "z")
      .replace(/ż/g, "z")
      .replace(/[^a-z0-9]+/g, "_") // Replace non-alphanumeric with underscore
      .replace(/^_+|_+$/g, "")     // Remove leading/trailing underscores
      .replace(/_+/g, "_");         // Replace multiple underscores with single
  };

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    
    if (!form.name) {
      toast({ title: "Wypełnij nazwę marszruty", variant: "destructive" });
      return;
    }

    const generatedCode = generateCode(form.name);
    if (!generatedCode) {
      toast({ title: "Nie można wygenerować kodu z podanej nazwy", variant: "destructive" });
      return;
    }

    setIsSubmitting(true);
    try {
      const res = await apiRequest("POST", "/api/production/routings", {
        code: generatedCode,
        name: form.name,
        description: form.description || null,
        isActive: form.isActive,
      });
      const result: { id: number } = await res.json();

      toast({ title: "Utworzono marszrutę" });
      setLocation(`/production/routings/${result.id}`);
    } catch (error: any) {
      toast({ 
        title: error.message || "Błąd tworzenia marszruty", 
        variant: "destructive" 
      });
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <div className="container mx-auto p-6 max-w-2xl space-y-4">
      <div className="flex items-center gap-3">
        <Button variant="ghost" size="icon" onClick={() => setLocation("/production/routings")}>
          <ArrowLeft className="h-4 w-4" />
        </Button>
        <h1 className="text-2xl font-bold">Nowa marszruta produkcyjna</h1>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane marszruty</CardTitle>
        </CardHeader>
        <CardContent>
          <form onSubmit={handleSubmit} className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="name">
                Nazwa <span className="text-destructive">*</span>
              </Label>
              <Input
                id="name"
                value={form.name}
                onChange={(e) => setForm({ ...form, name: e.target.value })}
                placeholder="np. SIEDZISKA TAPICEROWANE"
                maxLength={255}
                required
                data-testid="input-routing-name"
              />
              {form.name && (
                <p className="text-xs text-muted-foreground">
                  Kod: <span className="font-mono text-primary">{generateCode(form.name)}</span>
                </p>
              )}
            </div>

            <div className="space-y-2">
              <Label htmlFor="description">Opis</Label>
              <Textarea
                id="description"
                value={form.description}
                onChange={(e) => setForm({ ...form, description: e.target.value })}
                placeholder="Opcjonalny opis marszruty..."
                rows={3}
                data-testid="input-routing-description"
              />
            </div>

            <div className="flex items-center justify-between py-2">
              <div className="space-y-0.5">
                <Label htmlFor="isActive">Aktywna</Label>
                <p className="text-xs text-muted-foreground">
                  Czy marszruta jest aktywna i dostępna do użycia
                </p>
              </div>
              <Switch
                id="isActive"
                checked={form.isActive}
                onCheckedChange={(checked) => setForm({ ...form, isActive: checked })}
                data-testid="switch-routing-active"
              />
            </div>

            <div className="flex gap-2 justify-end pt-4">
              <Button
                type="button"
                variant="outline"
                onClick={() => setLocation("/production/routings")}
                disabled={isSubmitting}
                data-testid="button-cancel"
              >
                Anuluj
              </Button>
              <Button
                type="submit"
                disabled={isSubmitting}
                data-testid="button-save-routing"
              >
                {isSubmitting && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                Utwórz marszrutę
              </Button>
            </div>
          </form>
        </CardContent>
      </Card>
    </div>
  );
}
