import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useParams, useLocation } from "wouter";
import { ArrowLeft, Plus, Trash2, Loader2, ArrowUp, ArrowDown, Save, Edit, X, Check, ChevronsUpDown, Box, Scissors, Hammer, Paintbrush, Ruler, Drill, Cog, CircleDot, Square, Star, Zap, Truck, Timer, Wrench, Settings, Package, Layers, Factory, Palette, LucideIcon } from "lucide-react";
import * as LucideIcons from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from "@/components/ui/command";
import { useToast } from "@/hooks/use-toast";
import { cn } from "@/lib/utils";
import { apiRequest } from "@/lib/queryClient";
import type { ProductionRouting, ProductionRoutingOperation, ProductionWorkCenter, ProductCreatorDictionary, ProductionLocation } from "@shared/schema";

const AVAILABLE_ICONS = [
  { name: "Scissors", label: "Cięcie", icon: Scissors },
  { name: "Hammer", label: "Montaż", icon: Hammer },
  { name: "Paintbrush", label: "Malowanie", icon: Paintbrush },
  { name: "Ruler", label: "Pomiar", icon: Ruler },
  { name: "Drill", label: "Wiercenie", icon: Drill },
  { name: "Cog", label: "Obróbka", icon: Cog },
  { name: "Wrench", label: "Naprawa", icon: Wrench },
  { name: "Settings", label: "Ustawienia", icon: Settings },
  { name: "Package", label: "Pakowanie", icon: Package },
  { name: "Layers", label: "Warstwy", icon: Layers },
  { name: "Factory", label: "Produkcja", icon: Factory },
  { name: "Palette", label: "Kolor", icon: Palette },
  { name: "Truck", label: "Transport", icon: Truck },
  { name: "Timer", label: "Czas", icon: Timer },
  { name: "Star", label: "Jakość", icon: Star },
  { name: "Zap", label: "Szybka op.", icon: Zap },
  { name: "Box", label: "Magazyn", icon: Box },
];

const AVAILABLE_COLORS = [
  { name: "#3B82F6", label: "Niebieski" },
  { name: "#10B981", label: "Zielony" },
  { name: "#F59E0B", label: "Pomarańczowy" },
  { name: "#EF4444", label: "Czerwony" },
  { name: "#8B5CF6", label: "Fioletowy" },
  { name: "#EC4899", label: "Różowy" },
  { name: "#6B7280", label: "Szary" },
  { name: "#14B8A6", label: "Turkusowy" },
  { name: "#84CC16", label: "Limonka" },
  { name: "#F97316", label: "Łososiowy" },
];

function getIconComponent(iconName: string | null | undefined): LucideIcon | null {
  if (!iconName) return null;
  const found = AVAILABLE_ICONS.find(i => i.name === iconName);
  return found ? found.icon : null;
}

export default function ProductionRoutingDetail() {
  const { id } = useParams();
  const [, setLocation] = useLocation();
  const { toast } = useToast();
  const [newOp, setNewOp] = useState<{ code: string; name: string; estimatedTime: string; workCenterId: string; bufferBeforeId: string; bufferAfterId: string } | null>(null);
  const [editingId, setEditingId] = useState<number | null>(null);
  const [editForm, setEditForm] = useState<{ estimatedTime: string; workCenterId: string; bufferBeforeId: string; bufferAfterId: string; icon: string; color: string }>({ estimatedTime: "", workCenterId: "", bufferBeforeId: "", bufferAfterId: "", icon: "", color: "" });
  
  const [isEditingHeader, setIsEditingHeader] = useState(false);
  const [headerForm, setHeaderForm] = useState({ name: "", code: "" });
  
  const [operationOpen, setOperationOpen] = useState(false);
  const [workCenterOpen, setWorkCenterOpen] = useState(false);
  const [editWorkCenterOpen, setEditWorkCenterOpen] = useState(false);
  const [bufferBeforeOpen, setBufferBeforeOpen] = useState(false);
  const [bufferAfterOpen, setBufferAfterOpen] = useState(false);
  const [editBufferBeforeOpen, setEditBufferBeforeOpen] = useState(false);
  const [editBufferAfterOpen, setEditBufferAfterOpen] = useState(false);
  const [editIconOpen, setEditIconOpen] = useState(false);
  const [editColorOpen, setEditColorOpen] = useState(false);

  const { data: routing, isLoading: isLoadingRouting, refetch: refetchRouting } = useQuery<ProductionRouting>({
    queryKey: ["routing-detail", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/routings/${id}`);
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
  });

  const { data: operations = [], refetch: refetchOps } = useQuery<ProductionRoutingOperation[]>({
    queryKey: ["routing-ops", id],
    queryFn: async () => {
      const res = await fetch(`/api/production/routings/${id}/operations`);
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
    enabled: !!id,
  });

  const { data: workCenters = [] } = useQuery<ProductionWorkCenter[]>({
    queryKey: ["wcs"],
    queryFn: async () => {
      const res = await fetch("/api/production/work-centers");
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
  });

  const { data: dictOps = [] } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["dict-ops"],
    queryFn: async () => {
      const res = await fetch("/api/dictionaries?type=production_operation");
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
  });

  const { data: locations = [] } = useQuery<ProductionLocation[]>({
    queryKey: ["locations"],
    queryFn: async () => {
      const res = await fetch("/api/production/locations");
      if (!res.ok) throw new Error("Failed");
      return res.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: (data: any) => apiRequest("POST", `/api/production/routings/${id}/operations`, data),
    onSuccess: () => {
      refetchOps();
      setNewOp(null);
      toast({ title: "Dodano operację" });
    },
    onError: () => toast({ title: "Błąd", variant: "destructive" }),
  });

  const deleteMutation = useMutation({
    mutationFn: (opId: number) => apiRequest("DELETE", `/api/production/routings/${id}/operations/${opId}`),
    onSuccess: () => {
      refetchOps();
      toast({ title: "Usunięto operację" });
    },
    onError: () => toast({ title: "Błąd usuwania", variant: "destructive" }),
  });

  const updateMutation = useMutation({
    mutationFn: ({ opId, data }: { opId: number; data: any }) => 
      apiRequest("PATCH", `/api/production/routings/${id}/operations/${opId}`, data),
    onSuccess: () => {
      refetchOps();
      setEditingId(null);
      toast({ title: "Zaktualizowano operację" });
    },
    onError: () => toast({ title: "Błąd aktualizacji", variant: "destructive" }),
  });

  const reseqMutation = useMutation({
    mutationFn: (orderedIds: number[]) => apiRequest("POST", `/api/production/routings/${id}/operations/resequence`, { orderedIds }),
    onSuccess: () => {
      refetchOps();
      toast({ title: "Zmieniono kolejność" });
    },
    onError: () => toast({ title: "Błąd kolejności", variant: "destructive" }),
  });

  const updateRoutingMutation = useMutation({
    mutationFn: (data: { name: string; code: string }) => 
      apiRequest("PATCH", `/api/production/routings/${id}`, data),
    onSuccess: () => {
      refetchRouting();
      setIsEditingHeader(false);
      toast({ title: "Zaktualizowano marszrutę" });
    },
    onError: () => toast({ title: "Błąd aktualizacji", variant: "destructive" }),
  });

  const handleStartEditHeader = () => {
    if (routing) {
      setHeaderForm({ name: routing.name, code: routing.code });
      setIsEditingHeader(true);
    }
  };

  const handleSaveHeader = () => {
    if (!headerForm.name.trim() || !headerForm.code.trim()) {
      toast({ title: "Nazwa i kod są wymagane", variant: "destructive" });
      return;
    }
    updateRoutingMutation.mutate(headerForm);
  };

  const handleMove = (opId: number, direction: 'up' | 'down') => {
    const idx = operations.findIndex(op => op.id === opId);
    if (idx === -1) return;
    if (direction === 'up' && idx === 0) return;
    if (direction === 'down' && idx === operations.length - 1) return;

    const newArr = [...operations];
    const targetIdx = direction === 'up' ? idx - 1 : idx + 1;
    [newArr[idx], newArr[targetIdx]] = [newArr[targetIdx], newArr[idx]];
    
    reseqMutation.mutate(newArr.map(op => op.id));
  };

  const handleSaveNew = () => {
    if (!newOp?.code || !newOp?.name) {
      toast({ title: "Wybierz operację ze słownika", variant: "destructive" });
      return;
    }

    createMutation.mutate({
      routingId: Number(id),
      code: newOp.code,
      name: newOp.name,
      estimatedTime: newOp.estimatedTime || "0",
      setupTime: "0",
      workCenterId: newOp.workCenterId && newOp.workCenterId !== "" ? Number(newOp.workCenterId) : null,
      bufferBeforeId: newOp.bufferBeforeId && newOp.bufferBeforeId !== "" ? Number(newOp.bufferBeforeId) : null,
      bufferAfterId: newOp.bufferAfterId && newOp.bufferAfterId !== "" ? Number(newOp.bufferAfterId) : null,
      isActive: true,
    });
  };

  const handleSelectDict = (dictId: string) => {
    const dict = dictOps.find(d => d.id === Number(dictId));
    if (dict) {
      setNewOp(prev => ({
        ...prev!,
        code: dict.code,
        name: dict.name,
      }));
    }
  };

  if (isLoadingRouting) {
    return <div className="flex items-center justify-center h-96"><Loader2 className="h-8 w-8 animate-spin" /></div>;
  }

  if (!routing) {
    return <div className="container mx-auto p-6"><div className="text-center py-8">Nie znaleziono</div></div>;
  }

  return (
    <div className="container mx-auto p-6 space-y-4">
      <div className="flex items-center justify-between">
        <div className="flex items-center gap-3">
          <Button variant="ghost" size="icon" onClick={() => setLocation("/production/routings")}>
            <ArrowLeft className="h-4 w-4" />
          </Button>
          {isEditingHeader ? (
            <div className="flex items-center gap-3">
              <div className="space-y-1">
                <Input
                  value={headerForm.name}
                  onChange={(e) => setHeaderForm(p => ({ ...p, name: e.target.value }))}
                  placeholder="Nazwa marszruty"
                  className="h-9 text-lg font-bold"
                  data-testid="input-routing-name"
                />
                <Input
                  value={headerForm.code}
                  onChange={(e) => setHeaderForm(p => ({ ...p, code: e.target.value }))}
                  placeholder="Kod marszruty"
                  className="h-7 text-sm font-mono"
                  data-testid="input-routing-code"
                />
              </div>
              <div className="flex gap-1">
                <Button 
                  size="icon" 
                  onClick={handleSaveHeader}
                  disabled={updateRoutingMutation.isPending}
                  data-testid="button-save-header"
                >
                  <Check className="h-4 w-4" />
                </Button>
                <Button 
                  size="icon" 
                  variant="ghost" 
                  onClick={() => setIsEditingHeader(false)}
                  data-testid="button-cancel-header"
                >
                  <X className="h-4 w-4" />
                </Button>
              </div>
            </div>
          ) : (
            <div 
              className="cursor-pointer hover:bg-muted/50 rounded-md p-2 -m-2 transition-colors group"
              onClick={handleStartEditHeader}
              data-testid="header-edit-trigger"
            >
              <div className="flex items-center gap-2">
                <h1 className="text-2xl font-bold">{routing.name}</h1>
                <Edit className="h-4 w-4 opacity-0 group-hover:opacity-50 transition-opacity" />
              </div>
              <p className="text-sm text-muted-foreground font-mono">{routing.code}</p>
            </div>
          )}
        </div>
        <Badge variant={routing.isActive ? "default" : "secondary"}>
          {routing.isActive ? "Aktywna" : "Nieaktywna"}
        </Badge>
      </div>

      <Card>
        <CardHeader className="pb-3">
          <div className="flex items-center justify-between">
            <CardTitle className="text-lg">Operacje</CardTitle>
            {!newOp && (
              <Button size="sm" onClick={() => setNewOp({ code: "", name: "", estimatedTime: "0", workCenterId: "", bufferBeforeId: "", bufferAfterId: "" })}>
                <Plus className="h-4 w-4 mr-1" />
                Dodaj
              </Button>
            )}
          </div>
        </CardHeader>
        <CardContent className="p-0">
          <div className="border-t">
            <table className="w-full">
              <thead>
                <tr className="border-b bg-muted/50 text-xs">
                  <th className="text-left py-2 px-3 font-medium w-14">Kol.</th>
                  <th className="text-left py-2 px-3 font-medium">Operacja</th>
                  <th className="text-left py-2 px-3 font-medium w-28">Kod</th>
                  <th className="text-left py-2 px-3 font-medium">Gniazdo</th>
                  <th className="text-left py-2 px-3 font-medium">Bufor przed</th>
                  <th className="text-left py-2 px-3 font-medium">Bufor po</th>
                  <th className="text-left py-2 px-3 font-medium w-24">Ikona</th>
                  <th className="text-left py-2 px-3 font-medium w-20">Kolor</th>
                  <th className="text-left py-2 px-3 font-medium w-20">Czas</th>
                  <th className="text-right py-2 px-3 font-medium w-28">Akcje</th>
                </tr>
              </thead>
              <tbody>
                {newOp && (
                  <tr className="border-b bg-accent/20 h-[30px]">
                    <td className="py-1 px-3 text-sm text-muted-foreground">Nowa</td>
                    <td className="py-1 px-3">
                      <Popover open={operationOpen} onOpenChange={setOperationOpen}>
                        <PopoverTrigger asChild>
                          <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                            <span className="truncate">{newOp.name || "Wybierz operację..."}</span>
                            <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                          </Button>
                        </PopoverTrigger>
                        <PopoverContent className="w-[250px] p-0 z-[10001]" align="start">
                          <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                            <CommandInput placeholder="Szukaj operacji..." className="h-8 text-xs" />
                            <CommandList>
                              <CommandEmpty>Brak wyników</CommandEmpty>
                              <CommandGroup>
                                {dictOps.map(d => (
                                  <CommandItem
                                    key={d.id}
                                    value={d.name}
                                    onSelect={() => {
                                      handleSelectDict(String(d.id));
                                      setOperationOpen(false);
                                    }}
                                    className="text-xs"
                                  >
                                    <Check className={cn("mr-2 h-3 w-3", newOp.code === d.code ? "opacity-100" : "opacity-0")} />
                                    {d.name}
                                  </CommandItem>
                                ))}
                              </CommandGroup>
                            </CommandList>
                          </Command>
                        </PopoverContent>
                      </Popover>
                    </td>
                    <td className="py-1 px-3">
                      <Input
                        value={newOp.code || ""}
                        placeholder="Automatycznie"
                        className="h-7 text-xs bg-muted/50 cursor-not-allowed font-mono"
                        disabled
                      />
                    </td>
                    <td className="py-1 px-3">
                      <Popover open={workCenterOpen} onOpenChange={setWorkCenterOpen}>
                        <PopoverTrigger asChild>
                          <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                            <span className="truncate">
                              {newOp.workCenterId ? workCenters.find(wc => wc.id === Number(newOp.workCenterId))?.name || "Brak" : "Brak"}
                            </span>
                            <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                          </Button>
                        </PopoverTrigger>
                        <PopoverContent className="w-[250px] p-0 z-[10001]" align="start">
                          <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                            <CommandInput placeholder="Szukaj gniazda..." className="h-8 text-xs" />
                            <CommandList>
                              <CommandEmpty>Brak wyników</CommandEmpty>
                              <CommandGroup>
                                <CommandItem
                                  value="Brak"
                                  onSelect={() => {
                                    setNewOp({ ...newOp, workCenterId: "" });
                                    setWorkCenterOpen(false);
                                  }}
                                  className="text-xs"
                                >
                                  <Check className={cn("mr-2 h-3 w-3", !newOp.workCenterId ? "opacity-100" : "opacity-0")} />
                                  Brak
                                </CommandItem>
                                {workCenters.map(wc => (
                                  <CommandItem
                                    key={wc.id}
                                    value={wc.name}
                                    onSelect={() => {
                                      setNewOp({ ...newOp, workCenterId: String(wc.id) });
                                      setWorkCenterOpen(false);
                                    }}
                                    className="text-xs"
                                  >
                                    <Check className={cn("mr-2 h-3 w-3", newOp.workCenterId === String(wc.id) ? "opacity-100" : "opacity-0")} />
                                    {wc.name}
                                  </CommandItem>
                                ))}
                              </CommandGroup>
                            </CommandList>
                          </Command>
                        </PopoverContent>
                      </Popover>
                    </td>
                    <td className="py-1 px-3">
                      <Popover open={bufferBeforeOpen} onOpenChange={setBufferBeforeOpen}>
                        <PopoverTrigger asChild>
                          <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                            <span className="truncate">
                              {newOp.bufferBeforeId ? locations.find(l => l.id === Number(newOp.bufferBeforeId))?.name || "Brak" : "Brak"}
                            </span>
                            <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                          </Button>
                        </PopoverTrigger>
                        <PopoverContent className="w-[200px] p-0 z-[10001]" align="start">
                          <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                            <CommandInput placeholder="Szukaj bufora..." className="h-8 text-xs" />
                            <CommandList>
                              <CommandEmpty>Brak wyników</CommandEmpty>
                              <CommandGroup>
                                <CommandItem
                                  value="Brak"
                                  onSelect={() => {
                                    setNewOp({ ...newOp, bufferBeforeId: "" });
                                    setBufferBeforeOpen(false);
                                  }}
                                  className="text-xs"
                                >
                                  <Check className={cn("mr-2 h-3 w-3", !newOp.bufferBeforeId ? "opacity-100" : "opacity-0")} />
                                  Brak
                                </CommandItem>
                                {locations.map(loc => (
                                  <CommandItem
                                    key={loc.id}
                                    value={loc.name}
                                    onSelect={() => {
                                      setNewOp({ ...newOp, bufferBeforeId: String(loc.id) });
                                      setBufferBeforeOpen(false);
                                    }}
                                    className="text-xs"
                                  >
                                    <Check className={cn("mr-2 h-3 w-3", newOp.bufferBeforeId === String(loc.id) ? "opacity-100" : "opacity-0")} />
                                    {loc.name}
                                  </CommandItem>
                                ))}
                              </CommandGroup>
                            </CommandList>
                          </Command>
                        </PopoverContent>
                      </Popover>
                    </td>
                    <td className="py-1 px-3">
                      <Popover open={bufferAfterOpen} onOpenChange={setBufferAfterOpen}>
                        <PopoverTrigger asChild>
                          <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                            <span className="truncate">
                              {newOp.bufferAfterId ? locations.find(l => l.id === Number(newOp.bufferAfterId))?.name || "Brak" : "Brak"}
                            </span>
                            <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                          </Button>
                        </PopoverTrigger>
                        <PopoverContent className="w-[200px] p-0 z-[10001]" align="start">
                          <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                            <CommandInput placeholder="Szukaj bufora..." className="h-8 text-xs" />
                            <CommandList>
                              <CommandEmpty>Brak wyników</CommandEmpty>
                              <CommandGroup>
                                <CommandItem
                                  value="Brak"
                                  onSelect={() => {
                                    setNewOp({ ...newOp, bufferAfterId: "" });
                                    setBufferAfterOpen(false);
                                  }}
                                  className="text-xs"
                                >
                                  <Check className={cn("mr-2 h-3 w-3", !newOp.bufferAfterId ? "opacity-100" : "opacity-0")} />
                                  Brak
                                </CommandItem>
                                {locations.map(loc => (
                                  <CommandItem
                                    key={loc.id}
                                    value={loc.name}
                                    onSelect={() => {
                                      setNewOp({ ...newOp, bufferAfterId: String(loc.id) });
                                      setBufferAfterOpen(false);
                                    }}
                                    className="text-xs"
                                  >
                                    <Check className={cn("mr-2 h-3 w-3", newOp.bufferAfterId === String(loc.id) ? "opacity-100" : "opacity-0")} />
                                    {loc.name}
                                  </CommandItem>
                                ))}
                              </CommandGroup>
                            </CommandList>
                          </Command>
                        </PopoverContent>
                      </Popover>
                    </td>
                    <td className="py-1 px-3 text-xs text-muted-foreground">-</td>
                    <td className="py-1 px-3 text-xs text-muted-foreground">-</td>
                    <td className="py-1 px-3">
                      <Input
                        type="number"
                        value={newOp.estimatedTime || ""}
                        onChange={(e) => setNewOp({ ...newOp, estimatedTime: e.target.value })}
                        placeholder="min"
                        className="h-7 text-xs w-16"
                      />
                    </td>
                    <td className="py-1 px-3 text-right">
                      <div className="flex gap-1 justify-end">
                        <Button size="sm" className="h-7 text-xs" onClick={handleSaveNew} disabled={createMutation.isPending || !newOp.code}>
                          <Save className="h-3 w-3 mr-1" />
                          Zapisz
                        </Button>
                        <Button size="sm" variant="ghost" className="h-7 text-xs" onClick={() => setNewOp(null)}>
                          Anuluj
                        </Button>
                      </div>
                    </td>
                  </tr>
                )}

                {operations.length === 0 && !newOp ? (
                  <tr>
                    <td colSpan={10} className="text-center py-8 text-sm text-muted-foreground">
                      Brak operacji. Kliknij "Dodaj" aby rozpocząć.
                    </td>
                  </tr>
                ) : (
                  operations.map((op, idx) => {
                    const isEditing = editingId === op.id;
                    
                    if (isEditing) {
                      return (
                        <tr key={op.id} className="border-b bg-accent/10 h-[30px]">
                          <td className="py-1 px-3">
                            <Badge variant="outline" className="h-5 text-xs">{op.sequence}</Badge>
                          </td>
                          <td className="py-1 px-3 text-sm">{op.name}</td>
                          <td className="py-1 px-3 text-sm font-mono text-muted-foreground">{op.code}</td>
                          <td className="py-1 px-3">
                            <Popover open={editWorkCenterOpen} onOpenChange={setEditWorkCenterOpen}>
                              <PopoverTrigger asChild>
                                <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                                  <span className="truncate">
                                    {editForm.workCenterId ? workCenters.find(wc => wc.id === Number(editForm.workCenterId))?.name || "Brak" : "Brak"}
                                  </span>
                                  <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-[250px] p-0 z-[10001]" align="start">
                                <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                                  <CommandInput placeholder="Szukaj gniazda..." className="h-8 text-xs" />
                                  <CommandList>
                                    <CommandEmpty>Brak wyników</CommandEmpty>
                                    <CommandGroup>
                                      <CommandItem
                                        value="Brak"
                                        onSelect={() => {
                                          setEditForm({ ...editForm, workCenterId: "" });
                                          setEditWorkCenterOpen(false);
                                        }}
                                        className="text-xs"
                                      >
                                        <Check className={cn("mr-2 h-3 w-3", !editForm.workCenterId ? "opacity-100" : "opacity-0")} />
                                        Brak
                                      </CommandItem>
                                      {workCenters.map(wc => (
                                        <CommandItem
                                          key={wc.id}
                                          value={wc.name}
                                          onSelect={() => {
                                            setEditForm({ ...editForm, workCenterId: String(wc.id) });
                                            setEditWorkCenterOpen(false);
                                          }}
                                          className="text-xs"
                                        >
                                          <Check className={cn("mr-2 h-3 w-3", editForm.workCenterId === String(wc.id) ? "opacity-100" : "opacity-0")} />
                                          {wc.name}
                                        </CommandItem>
                                      ))}
                                    </CommandGroup>
                                  </CommandList>
                                </Command>
                              </PopoverContent>
                            </Popover>
                          </td>
                          <td className="py-1 px-3">
                            <Popover open={editBufferBeforeOpen} onOpenChange={setEditBufferBeforeOpen}>
                              <PopoverTrigger asChild>
                                <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                                  <span className="truncate">
                                    {editForm.bufferBeforeId ? locations.find(l => l.id === Number(editForm.bufferBeforeId))?.name || "Brak" : "Brak"}
                                  </span>
                                  <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-[200px] p-0 z-[10001]" align="start">
                                <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                                  <CommandInput placeholder="Szukaj bufora..." className="h-8 text-xs" />
                                  <CommandList>
                                    <CommandEmpty>Brak wyników</CommandEmpty>
                                    <CommandGroup>
                                      <CommandItem
                                        value="Brak"
                                        onSelect={() => {
                                          setEditForm({ ...editForm, bufferBeforeId: "" });
                                          setEditBufferBeforeOpen(false);
                                        }}
                                        className="text-xs"
                                      >
                                        <Check className={cn("mr-2 h-3 w-3", !editForm.bufferBeforeId ? "opacity-100" : "opacity-0")} />
                                        Brak
                                      </CommandItem>
                                      {locations.map(loc => (
                                        <CommandItem
                                          key={loc.id}
                                          value={loc.name}
                                          onSelect={() => {
                                            setEditForm({ ...editForm, bufferBeforeId: String(loc.id) });
                                            setEditBufferBeforeOpen(false);
                                          }}
                                          className="text-xs"
                                        >
                                          <Check className={cn("mr-2 h-3 w-3", editForm.bufferBeforeId === String(loc.id) ? "opacity-100" : "opacity-0")} />
                                          {loc.name}
                                        </CommandItem>
                                      ))}
                                    </CommandGroup>
                                  </CommandList>
                                </Command>
                              </PopoverContent>
                            </Popover>
                          </td>
                          <td className="py-1 px-3">
                            <Popover open={editBufferAfterOpen} onOpenChange={setEditBufferAfterOpen}>
                              <PopoverTrigger asChild>
                                <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                                  <span className="truncate">
                                    {editForm.bufferAfterId ? locations.find(l => l.id === Number(editForm.bufferAfterId))?.name || "Brak" : "Brak"}
                                  </span>
                                  <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-[200px] p-0 z-[10001]" align="start">
                                <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                                  <CommandInput placeholder="Szukaj bufora..." className="h-8 text-xs" />
                                  <CommandList>
                                    <CommandEmpty>Brak wyników</CommandEmpty>
                                    <CommandGroup>
                                      <CommandItem
                                        value="Brak"
                                        onSelect={() => {
                                          setEditForm({ ...editForm, bufferAfterId: "" });
                                          setEditBufferAfterOpen(false);
                                        }}
                                        className="text-xs"
                                      >
                                        <Check className={cn("mr-2 h-3 w-3", !editForm.bufferAfterId ? "opacity-100" : "opacity-0")} />
                                        Brak
                                      </CommandItem>
                                      {locations.map(loc => (
                                        <CommandItem
                                          key={loc.id}
                                          value={loc.name}
                                          onSelect={() => {
                                            setEditForm({ ...editForm, bufferAfterId: String(loc.id) });
                                            setEditBufferAfterOpen(false);
                                          }}
                                          className="text-xs"
                                        >
                                          <Check className={cn("mr-2 h-3 w-3", editForm.bufferAfterId === String(loc.id) ? "opacity-100" : "opacity-0")} />
                                          {loc.name}
                                        </CommandItem>
                                      ))}
                                    </CommandGroup>
                                  </CommandList>
                                </Command>
                              </PopoverContent>
                            </Popover>
                          </td>
                          <td className="py-1 px-3">
                            <Popover open={editIconOpen} onOpenChange={setEditIconOpen}>
                              <PopoverTrigger asChild>
                                <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                                  {editForm.icon ? (
                                    <span className="flex items-center gap-1">
                                      {(() => {
                                        const IconComp = getIconComponent(editForm.icon);
                                        return IconComp ? <IconComp className="h-3 w-3" /> : null;
                                      })()}
                                      <span className="truncate">{AVAILABLE_ICONS.find(i => i.name === editForm.icon)?.label || editForm.icon}</span>
                                    </span>
                                  ) : (
                                    <span className="text-muted-foreground">Brak</span>
                                  )}
                                  <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-[200px] p-0 z-[10001]" align="start">
                                <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                                  <CommandInput placeholder="Szukaj ikony..." className="h-8 text-xs" />
                                  <CommandList>
                                    <CommandEmpty>Brak wyników</CommandEmpty>
                                    <CommandGroup>
                                      <CommandItem
                                        value="Brak"
                                        onSelect={() => {
                                          setEditForm({ ...editForm, icon: "" });
                                          setEditIconOpen(false);
                                        }}
                                        className="text-xs"
                                      >
                                        <Check className={cn("mr-2 h-3 w-3", !editForm.icon ? "opacity-100" : "opacity-0")} />
                                        Brak
                                      </CommandItem>
                                      {AVAILABLE_ICONS.map(iconItem => (
                                        <CommandItem
                                          key={iconItem.name}
                                          value={iconItem.label}
                                          onSelect={() => {
                                            setEditForm({ ...editForm, icon: iconItem.name });
                                            setEditIconOpen(false);
                                          }}
                                          className="text-xs"
                                        >
                                          <Check className={cn("mr-2 h-3 w-3", editForm.icon === iconItem.name ? "opacity-100" : "opacity-0")} />
                                          <iconItem.icon className="mr-2 h-3 w-3" />
                                          {iconItem.label}
                                        </CommandItem>
                                      ))}
                                    </CommandGroup>
                                  </CommandList>
                                </Command>
                              </PopoverContent>
                            </Popover>
                          </td>
                          <td className="py-1 px-3">
                            <Popover open={editColorOpen} onOpenChange={setEditColorOpen}>
                              <PopoverTrigger asChild>
                                <Button variant="outline" role="combobox" className="h-7 text-xs w-full justify-between">
                                  {editForm.color ? (
                                    <span className="flex items-center gap-1">
                                      <span 
                                        className="w-3 h-3 rounded-sm border" 
                                        style={{ backgroundColor: editForm.color }}
                                      />
                                      <span className="truncate">{AVAILABLE_COLORS.find(c => c.name === editForm.color)?.label || editForm.color}</span>
                                    </span>
                                  ) : (
                                    <span className="text-muted-foreground">Brak</span>
                                  )}
                                  <ChevronsUpDown className="ml-1 h-3 w-3 shrink-0 opacity-50" />
                                </Button>
                              </PopoverTrigger>
                              <PopoverContent className="w-[180px] p-0 z-[10001]" align="start">
                                <Command filter={(value, search) => value.toLowerCase().includes(search.toLowerCase()) ? 1 : 0}>
                                  <CommandInput placeholder="Szukaj koloru..." className="h-8 text-xs" />
                                  <CommandList>
                                    <CommandEmpty>Brak wyników</CommandEmpty>
                                    <CommandGroup>
                                      <CommandItem
                                        value="Brak"
                                        onSelect={() => {
                                          setEditForm({ ...editForm, color: "" });
                                          setEditColorOpen(false);
                                        }}
                                        className="text-xs"
                                      >
                                        <Check className={cn("mr-2 h-3 w-3", !editForm.color ? "opacity-100" : "opacity-0")} />
                                        Brak
                                      </CommandItem>
                                      {AVAILABLE_COLORS.map(colorItem => (
                                        <CommandItem
                                          key={colorItem.name}
                                          value={colorItem.label}
                                          onSelect={() => {
                                            setEditForm({ ...editForm, color: colorItem.name });
                                            setEditColorOpen(false);
                                          }}
                                          className="text-xs"
                                        >
                                          <Check className={cn("mr-2 h-3 w-3", editForm.color === colorItem.name ? "opacity-100" : "opacity-0")} />
                                          <span 
                                            className="mr-2 w-3 h-3 rounded-sm border" 
                                            style={{ backgroundColor: colorItem.name }}
                                          />
                                          {colorItem.label}
                                        </CommandItem>
                                      ))}
                                    </CommandGroup>
                                  </CommandList>
                                </Command>
                              </PopoverContent>
                            </Popover>
                          </td>
                          <td className="py-1 px-3">
                            <Input
                              type="number"
                              value={editForm.estimatedTime}
                              onChange={(e) => setEditForm({ ...editForm, estimatedTime: e.target.value })}
                              className="h-7 text-xs w-16"
                            />
                          </td>
                          <td className="py-1 px-3 text-right">
                            <div className="flex gap-1 justify-end">
                              <Button
                                size="sm"
                                className="h-7 text-xs"
                                onClick={() => {
                                  updateMutation.mutate({
                                    opId: op.id,
                                    data: {
                                      estimatedTime: editForm.estimatedTime || "0",
                                      workCenterId: editForm.workCenterId && editForm.workCenterId !== "" ? Number(editForm.workCenterId) : null,
                                      bufferBeforeId: editForm.bufferBeforeId && editForm.bufferBeforeId !== "" ? Number(editForm.bufferBeforeId) : null,
                                      bufferAfterId: editForm.bufferAfterId && editForm.bufferAfterId !== "" ? Number(editForm.bufferAfterId) : null,
                                      icon: editForm.icon || null,
                                      color: editForm.color || null,
                                    }
                                  });
                                }}
                                disabled={updateMutation.isPending}
                              >
                                <Save className="h-3 w-3 mr-1" />
                                Zapisz
                              </Button>
                              <Button
                                size="sm"
                                variant="ghost"
                                className="h-7 text-xs"
                                onClick={() => setEditingId(null)}
                              >
                                Anuluj
                              </Button>
                            </div>
                          </td>
                        </tr>
                      );
                    }
                    
                    return (
                      <tr 
                        key={op.id} 
                        className="border-b hover:bg-muted/30 h-[30px] cursor-pointer"
                        onClick={() => {
                          setEditingId(op.id);
                          setEditForm({
                            estimatedTime: op.estimatedTime || "0",
                            workCenterId: op.workCenterId ? String(op.workCenterId) : "",
                            bufferBeforeId: op.bufferBeforeId ? String(op.bufferBeforeId) : "",
                            bufferAfterId: op.bufferAfterId ? String(op.bufferAfterId) : "",
                            icon: op.icon || "",
                            color: op.color || "",
                          });
                        }}
                      >
                        <td className="py-1 px-3">
                          <Badge variant="outline" className="h-5 text-xs">{op.sequence}</Badge>
                        </td>
                        <td className="py-1 px-3 text-sm">{op.name}</td>
                        <td className="py-1 px-3 text-sm font-mono text-muted-foreground">{op.code}</td>
                        <td className="py-1 px-3 text-xs text-muted-foreground">
                          {op.workCenterId ? workCenters.find(wc => wc.id === op.workCenterId)?.name || "-" : "-"}
                        </td>
                        <td className="py-1 px-3 text-xs text-muted-foreground">
                          {op.bufferBeforeId ? locations.find(l => l.id === op.bufferBeforeId)?.name || "-" : "-"}
                        </td>
                        <td className="py-1 px-3 text-xs text-muted-foreground">
                          {op.bufferAfterId ? locations.find(l => l.id === op.bufferAfterId)?.name || "-" : "-"}
                        </td>
                        <td className="py-1 px-3">
                          {op.icon ? (
                            <span className="flex items-center gap-1">
                              {(() => {
                                const IconComp = getIconComponent(op.icon);
                                return IconComp ? <IconComp className="h-3.5 w-3.5" style={{ color: op.color || undefined }} /> : null;
                              })()}
                              <span className="text-xs text-muted-foreground">{AVAILABLE_ICONS.find(i => i.name === op.icon)?.label || op.icon}</span>
                            </span>
                          ) : (
                            <span className="text-xs text-muted-foreground">-</span>
                          )}
                        </td>
                        <td className="py-1 px-3">
                          {op.color ? (
                            <span className="flex items-center gap-1">
                              <span 
                                className="w-3 h-3 rounded-sm border" 
                                style={{ backgroundColor: op.color }}
                              />
                              <span className="text-xs text-muted-foreground">{AVAILABLE_COLORS.find(c => c.name === op.color)?.label || ""}</span>
                            </span>
                          ) : (
                            <span className="text-xs text-muted-foreground">-</span>
                          )}
                        </td>
                        <td className="py-1 px-3 text-sm text-muted-foreground">
                          {op.estimatedTime ? `${op.estimatedTime} min` : "-"}
                        </td>
                        <td className="py-1 px-3 text-right" onClick={(e) => e.stopPropagation()}>
                          <div className="flex gap-1 justify-end">
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => handleMove(op.id, 'up')}
                              disabled={idx === 0 || reseqMutation.isPending}
                            >
                              <ArrowUp className="h-3.5 w-3.5" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => handleMove(op.id, 'down')}
                              disabled={idx === operations.length - 1 || reseqMutation.isPending}
                            >
                              <ArrowDown className="h-3.5 w-3.5" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              className="h-7 w-7"
                              onClick={() => {
                                if (confirm("Czy na pewno chcesz usunąć?")) {
                                  deleteMutation.mutate(op.id);
                                }
                              }}
                            >
                              <Trash2 className="h-3.5 w-3.5" />
                            </Button>
                          </div>
                        </td>
                      </tr>
                    );
                  })
                )}
              </tbody>
            </table>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
