import { useState, useEffect } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { Link, useLocation } from "wouter";
import { Plus, Edit, Trash2, Loader2, Calendar, Package2, FileText } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient } from "@/lib/queryClient";
import type { ProductionPlan } from "@shared/schema";

interface PlanWithStats extends ProductionPlan {
  productsCount: number;
  ordersCount: number;
  packedCount: number;
  totalQuantity: number;
  reservedFormatki: number;
  totalFormatki: number;
}
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { z } from "zod";
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Textarea } from "@/components/ui/textarea";

const statusColors: Record<string, string> = {
  draft: "bg-gray-500",
  approved: "bg-blue-500",
  in_progress: "bg-yellow-500",
  completed: "bg-green-500",
  cancelled: "bg-red-500",
};

const statusLabels: Record<string, string> = {
  draft: "Projekt",
  approved: "Zatwierdzony",
  in_progress: "W realizacji",
  completed: "Zakończony",
  cancelled: "Anulowany",
};

const priorityColors: Record<string, string> = {
  low: "bg-gray-400",
  normal: "bg-blue-400",
  high: "bg-orange-400",
  urgent: "bg-red-500",
};

const priorityLabels: Record<string, string> = {
  low: "Niski",
  normal: "Normalny",
  high: "Wysoki",
  urgent: "Pilny",
};

const planFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().optional(),
  plannedStartDate: z.string().optional(),
  plannedEndDate: z.string().optional(),
  status: z.enum(['draft', 'approved', 'in_progress', 'completed', 'cancelled']).optional(),
  priority: z.enum(['low', 'normal', 'high', 'urgent']).optional(),
  notes: z.string().optional(),
});

type PlanFormValues = z.infer<typeof planFormSchema>;

export default function ProductionPlansPage() {
  const { toast } = useToast();
  const [location, setLocation] = useLocation();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingPlan, setEditingPlan] = useState<ProductionPlan | null>(null);
  
  // Parse URL params
  const urlParams = new URLSearchParams(window.location.search);
  
  // Filters state (initialized from URL)
  const [statusFilter, setStatusFilter] = useState<string>(urlParams.get("status") || "all");
  const [priorityFilter, setPriorityFilter] = useState<string>(urlParams.get("priority") || "all");
  const [searchFilter, setSearchFilter] = useState<string>(urlParams.get("search") || "");

  // Sync filters to URL
  useEffect(() => {
    const params = new URLSearchParams();
    if (statusFilter && statusFilter !== "all") params.set("status", statusFilter);
    if (priorityFilter && priorityFilter !== "all") params.set("priority", priorityFilter);
    if (searchFilter) params.set("search", searchFilter);
    
    const newSearch = params.toString();
    const currentSearch = window.location.search.slice(1);
    
    if (newSearch !== currentSearch) {
      setLocation(`/production/plans${newSearch ? `?${newSearch}` : ""}`, { replace: true });
    }
  }, [statusFilter, priorityFilter, searchFilter, setLocation]);

  // Build query params
  const queryParams = new URLSearchParams();
  if (statusFilter && statusFilter !== "all") queryParams.append("status", statusFilter);
  if (priorityFilter && priorityFilter !== "all") queryParams.append("priority", priorityFilter);
  if (searchFilter) queryParams.append("search", searchFilter);
  const queryString = queryParams.toString();

  // Fetch plans with filters
  const { data: plans = [], isLoading } = useQuery<PlanWithStats[]>({
    queryKey: ["/api/production/planning/plans", queryString],
    queryFn: async () => {
      const url = queryString ? `/api/production/planning/plans?${queryString}` : "/api/production/planning/plans";
      const response = await fetch(url, { credentials: "include" });
      if (!response.ok) throw new Error("Failed to fetch plans");
      return response.json();
    },
  });

  // Form
  const form = useForm<PlanFormValues>({
    resolver: zodResolver(planFormSchema),
    defaultValues: {
      name: "",
      description: "",
      status: "draft",
      priority: "normal",
      notes: "",
    },
  });

  // Create/Update mutation
  const saveMutation = useMutation({
    mutationFn: async (data: PlanFormValues) => {
      if (editingPlan) {
        const response = await fetch(`/api/production/planning/plans/${editingPlan.id}`, {
          method: "PATCH",
          headers: { "Content-Type": "application/json" },
          credentials: "include",
          body: JSON.stringify(data),
        });
        if (!response.ok) {
          const error = await response.json().catch(() => ({ message: "Failed to update plan" }));
          throw new Error(error.message);
        }
        return await response.json();
      } else {
        const response = await fetch("/api/production/planning/plans", {
          method: "POST",
          headers: { "Content-Type": "application/json" },
          credentials: "include",
          body: JSON.stringify(data),
        });
        if (!response.ok) {
          const error = await response.json().catch(() => ({ message: "Failed to create plan" }));
          throw new Error(error.message);
        }
        return await response.json();
      }
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/planning/plans"] });
      toast({
        title: "Sukces",
        description: editingPlan ? "Plan zaktualizowany" : "Plan utworzony",
      });
      handleCloseDialog();
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message,
      });
    },
  });

  // Delete mutation
  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      const response = await fetch(`/api/production/planning/plans/${id}`, {
        method: "DELETE",
        credentials: "include",
      });
      if (!response.ok) throw new Error("Failed to delete plan");
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/planning/plans"] });
      toast({
        title: "Sukces",
        description: "Plan usunięty",
      });
    },
    onError: (error: Error) => {
      toast({
        variant: "destructive",
        title: "Błąd",
        description: error.message,
      });
    },
  });

  const handleOpenDialog = (plan?: ProductionPlan) => {
    setEditingPlan(plan || null);
    if (plan) {
      form.reset({
        name: plan.name,
        description: plan.description || "",
        plannedStartDate: plan.plannedStartDate ? format(plan.plannedStartDate, "yyyy-MM-dd") : "",
        plannedEndDate: plan.plannedEndDate ? format(plan.plannedEndDate, "yyyy-MM-dd") : "",
        status: plan.status as any,
        priority: plan.priority as any || "normal",
        notes: plan.notes || "",
      });
    } else {
      form.reset({
        name: "",
        description: "",
        status: "draft",
        priority: "normal",
        notes: "",
      });
    }
    setIsDialogOpen(true);
  };

  const handleCloseDialog = () => {
    setIsDialogOpen(false);
    setEditingPlan(null);
    form.reset();
  };

  const handleSubmit = (data: PlanFormValues) => {
    saveMutation.mutate(data);
  };

  const handleDelete = (id: number) => {
    if (confirm("Czy na pewno chcesz usunąć ten plan?")) {
      deleteMutation.mutate(id);
    }
  };

  return (
    <div className="container mx-auto p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold" data-testid="text-page-title">Plany Produkcji</h1>
          <p className="text-muted-foreground">
            Zarządzaj planami produkcji i liniami planowania
          </p>
        </div>
        <Link href="/production/plans/new">
          <Button data-testid="button-create-plan">
            <Plus className="h-4 w-4 mr-2" />
            Nowy Plan
          </Button>
        </Link>
      </div>

      {/* Filters */}
      <Card>
        <CardHeader>
          <CardTitle>Filtry</CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
            <div className="space-y-2">
              <Label htmlFor="search">Szukaj</Label>
              <Input
                id="search"
                placeholder="Numer lub nazwa planu..."
                value={searchFilter}
                onChange={(e) => setSearchFilter(e.target.value)}
                data-testid="input-search"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="status-filter">Status</Label>
              <Select value={statusFilter} onValueChange={setStatusFilter}>
                <SelectTrigger id="status-filter" data-testid="select-status">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="draft">Projekt</SelectItem>
                  <SelectItem value="approved">Zatwierdzony</SelectItem>
                  <SelectItem value="in_progress">W realizacji</SelectItem>
                  <SelectItem value="completed">Zakończony</SelectItem>
                  <SelectItem value="cancelled">Anulowany</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="space-y-2">
              <Label htmlFor="priority-filter">Priorytet</Label>
              <Select value={priorityFilter} onValueChange={setPriorityFilter}>
                <SelectTrigger id="priority-filter" data-testid="select-priority">
                  <SelectValue placeholder="Wszystkie" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">Wszystkie</SelectItem>
                  <SelectItem value="low">Niski</SelectItem>
                  <SelectItem value="normal">Normalny</SelectItem>
                  <SelectItem value="high">Wysoki</SelectItem>
                  <SelectItem value="urgent">Pilny</SelectItem>
                </SelectContent>
              </Select>
            </div>

            <div className="flex items-end">
              <Button 
                variant="outline" 
                onClick={() => {
                  setStatusFilter("all");
                  setPriorityFilter("all");
                  setSearchFilter("");
                }}
                data-testid="button-clear-filters"
              >
                Wyczyść filtry
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Plans Table */}
      <Card>
        <CardContent className="pt-6">
          {isLoading ? (
            <div className="flex justify-center p-8">
              <Loader2 className="h-8 w-8 animate-spin" />
            </div>
          ) : plans.length === 0 ? (
            <div className="text-center p-8 text-muted-foreground">
              Brak planów produkcji
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead>Numer</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Status</TableHead>
                  <TableHead>Priorytet</TableHead>
                  <TableHead className="text-center">Zamówienia</TableHead>
                  <TableHead className="text-center">Produkty</TableHead>
                  <TableHead className="text-center">Pakowane</TableHead>
                  <TableHead className="text-center">Formatki</TableHead>
                  <TableHead>Data rozp.</TableHead>
                  <TableHead>Utworzono</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {plans.map((plan) => {
                  const packingPercent = plan.productsCount > 0 
                    ? Math.round((plan.packedCount / plan.productsCount) * 100) 
                    : 0;
                  
                  return (
                    <TableRow key={plan.id} data-testid={`row-plan-${plan.id}`}>
                      <TableCell>
                        <Link 
                          href={`/production/plans/${plan.id}`}
                          data-testid={`link-plan-${plan.id}`}
                        >
                          <Badge 
                            variant="outline" 
                            className="text-sm font-bold px-2 py-1 bg-purple-600 text-white border-purple-600 hover:bg-purple-700"
                          >
                            {plan.planNumber}
                          </Badge>
                        </Link>
                      </TableCell>
                      <TableCell className="max-w-[200px] truncate">{plan.name}</TableCell>
                      <TableCell>
                        <Badge className={statusColors[plan.status]} data-testid={`badge-status-${plan.id}`}>
                          {statusLabels[plan.status]}
                        </Badge>
                      </TableCell>
                      <TableCell>
                        <Badge className={priorityColors[plan.priority || 'normal']} data-testid={`badge-priority-${plan.id}`}>
                          {priorityLabels[plan.priority || 'normal']}
                        </Badge>
                      </TableCell>
                      <TableCell className="text-center">
                        <span className="font-medium" data-testid={`orders-count-${plan.id}`}>
                          {plan.ordersCount || 0}
                        </span>
                      </TableCell>
                      <TableCell className="text-center">
                        <span className="font-medium" data-testid={`products-count-${plan.id}`}>
                          {plan.productsCount || 0}
                        </span>
                      </TableCell>
                      <TableCell className="text-center">
                        <div className="flex flex-col items-center gap-0.5" data-testid={`packed-stats-${plan.id}`}>
                          <span className="font-medium text-sm">
                            {plan.packedCount || 0}/{plan.productsCount || 0}
                          </span>
                          <span className={`text-xs ${packingPercent >= 100 ? 'text-green-500' : packingPercent > 0 ? 'text-yellow-500' : 'text-muted-foreground'}`}>
                            {packingPercent}%
                          </span>
                        </div>
                      </TableCell>
                      <TableCell className="text-center">
                        <div className="flex flex-col items-center gap-0.5" data-testid={`formatki-stats-${plan.id}`}>
                          <span className="font-medium text-sm">
                            {String(plan.reservedFormatki || 0).padStart(4, '0')}/{String(plan.totalFormatki || 0).padStart(4, '0')}
                          </span>
                        </div>
                      </TableCell>
                      <TableCell>
                        {plan.plannedStartDate ? format(plan.plannedStartDate, "dd.MM", { locale: pl }) : "-"}
                      </TableCell>
                      <TableCell className="text-xs text-muted-foreground">
                        {plan.createdAt ? format(plan.createdAt, "dd.MM.yy", { locale: pl }) : "-"}
                      </TableCell>
                      <TableCell className="text-right space-x-1">
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleOpenDialog(plan)}
                          data-testid={`button-edit-${plan.id}`}
                        >
                          <Edit className="h-4 w-4" />
                        </Button>
                        <Button
                          variant="ghost"
                          size="icon"
                          onClick={() => handleDelete(plan.id)}
                          disabled={deleteMutation.isPending}
                          data-testid={`button-delete-${plan.id}`}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </TableCell>
                    </TableRow>
                  );
                })}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      {/* Create/Edit Dialog */}
      <Dialog open={isDialogOpen} onOpenChange={handleCloseDialog}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>{editingPlan ? "Edytuj plan" : "Nowy plan produkcji"}</DialogTitle>
            <DialogDescription>
              {editingPlan ? "Zaktualizuj informacje o planie" : "Utwórz nowy plan produkcji"}
            </DialogDescription>
          </DialogHeader>
          
          <Form {...form}>
            <form onSubmit={form.handleSubmit(handleSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa planu *</FormLabel>
                    <FormControl>
                      <Input {...field} placeholder="Plan produkcji styczeń 2025" data-testid="input-name" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="Opcjonalny opis planu..." data-testid="input-description" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="plannedStartDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Data rozpoczęcia</FormLabel>
                      <FormControl>
                        <Input type="date" {...field} data-testid="input-start-date" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="plannedEndDate"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Data zakończenia</FormLabel>
                      <FormControl>
                        <Input type="date" {...field} data-testid="input-end-date" />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="status"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Status</FormLabel>
                      <Select value={field.value} onValueChange={field.onChange}>
                        <FormControl>
                          <SelectTrigger data-testid="select-form-status">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="draft">Projekt</SelectItem>
                          <SelectItem value="approved">Zatwierdzony</SelectItem>
                          <SelectItem value="in_progress">W realizacji</SelectItem>
                          <SelectItem value="completed">Zakończony</SelectItem>
                          <SelectItem value="cancelled">Anulowany</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="priority"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Priorytet</FormLabel>
                      <Select value={field.value} onValueChange={field.onChange}>
                        <FormControl>
                          <SelectTrigger data-testid="select-form-priority">
                            <SelectValue />
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="low">Niski</SelectItem>
                          <SelectItem value="normal">Normalny</SelectItem>
                          <SelectItem value="high">Wysoki</SelectItem>
                          <SelectItem value="urgent">Pilny</SelectItem>
                        </SelectContent>
                      </Select>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="notes"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Notatki</FormLabel>
                    <FormControl>
                      <Textarea {...field} placeholder="Dodatkowe notatki..." data-testid="input-notes" />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="flex justify-end gap-2">
                <Button 
                  type="button" 
                  variant="outline" 
                  onClick={handleCloseDialog}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button 
                  type="submit" 
                  disabled={saveMutation.isPending}
                  data-testid="button-save"
                >
                  {saveMutation.isPending && <Loader2 className="h-4 w-4 mr-2 animate-spin" />}
                  {editingPlan ? "Zaktualizuj" : "Utwórz"}
                </Button>
              </div>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
