import { useState, useCallback } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useRoute, Link } from "wouter";
import { ArrowLeft, Scissors, Truck, Loader2, GitMerge, Users, Route, Factory, Network, Lock, Unlock, X, AlertTriangle } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { ScrollArea, ScrollBar } from "@/components/ui/scroll-area";
import { CuttingSequencePanel } from "@/features/production-plans/CuttingSequencePanel";
import { TransportBatchesPanel } from "@/features/production-plans/TransportBatchesPanel";
import { MergePointsPanel } from "@/features/production-plans/MergePointsPanel";
import { OperatorsAssignmentPanel } from "@/features/production-plans/OperatorsAssignmentPanel";
import { PlanRoutingAssignmentsPanel } from "@/features/production-plans/PlanRoutingAssignmentsPanel";
import { ZlpGenerationProgressDialog } from "@/components/zlp-generation-progress-dialog";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface ProductionPlan {
  id: number;
  planNumber: string;
  name: string;
  status: string;
  priority: string;
  zlpLocked?: boolean;
}

interface PlanItem {
  id: number;
  [key: string]: any;
}

interface ZlpDashboard {
  totalZlps: number;
}

export default function ProductionPlanRouting() {
  const [, params] = useRoute("/production/plans/:id/routing");
  const planId = params?.id ? parseInt(params.id, 10) : 0;
  const [activeTab, setActiveTab] = useState<string>("routing-assignments");
  const { toast } = useToast();

  const [zlpProgressDialogOpen, setZlpProgressDialogOpen] = useState(false);
  const [zlpSessionId, setZlpSessionId] = useState<string | null>(null);
  const [showUnlockConfirm, setShowUnlockConfirm] = useState(false);
  const [showLockConfirm, setShowLockConfirm] = useState(false);

  const { data: plan, isLoading: planLoading } = useQuery<ProductionPlan>({
    queryKey: [`/api/production/planning/plans/${planId}`],
    enabled: planId > 0,
  });

  const { data: planItems } = useQuery<PlanItem[]>({
    queryKey: [`/api/production/planning/plans/${planId}/items`],
    enabled: planId > 0,
  });

  const { data: zlpDashboard } = useQuery<ZlpDashboard>({
    queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`],
    enabled: planId > 0,
  });

  const planLinesCount = planItems?.length || 0;
  const hasZlps = (zlpDashboard?.totalZlps || 0) > 0;
  const zlpLocked = plan?.zlpLocked || false;

  const generateOrdersMutation = useMutation({
    mutationFn: async () => {
      const response = await apiRequest(
        "POST",
        `/api/production/planning/plans/${planId}/generate-orders`
      );
      return await response.json();
    },
    onSuccess: (data: any) => {
      if (data.sessionId) {
        setZlpSessionId(data.sessionId);
        setZlpProgressDialogOpen(true);
      }
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd generowania ZLP",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const lockZlpMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/production/planning/plans/${planId}/lock-zlp`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}`] });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`] });
      toast({ title: "Zlecenia zatwierdzone", description: "Plan produkcyjny został zablokowany do edycji" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const unlockZlpMutation = useMutation({
    mutationFn: async () => {
      return apiRequest("POST", `/api/production/planning/plans/${planId}/unlock-zlp`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}`] });
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`] });
      toast({ title: "Blokada zdjęta", description: "Plan produkcyjny można teraz edytować" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const handleZlpGenerationComplete = useCallback((result: any) => {
    queryClient.invalidateQueries({
      queryKey: [`/api/production/planning/plans/${planId}`]
    });
    queryClient.invalidateQueries({
      queryKey: [`/api/production/planning/plans/${planId}/items`]
    });
    queryClient.invalidateQueries({
      queryKey: [`/api/production/planning/plans/${planId}/zlp-dashboard`]
    });
    
    if (result.success) {
      toast({
        title: "Zlecenia wygenerowane",
        description: `Utworzono ${result.workOrdersCount} zleceń ZLP`,
      });
    }
  }, [planId, toast]);

  const isGenerating = generateOrdersMutation.isPending;

  if (planLoading) {
    return (
      <div className="flex items-center justify-center h-full">
        <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
      </div>
    );
  }

  if (!plan) {
    return (
      <div className="flex flex-col items-center justify-center h-full gap-4">
        <p className="text-muted-foreground">Nie znaleziono planu produkcyjnego</p>
        <Link href="/production/plans">
          <Button variant="outline" className="gap-2">
            <ArrowLeft className="h-4 w-4" />
            Wróć do listy planów
          </Button>
        </Link>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-full">
      <div className="border-b bg-background">
        <div className="flex items-center justify-between p-4 gap-2 flex-wrap">
          <div className="flex items-center gap-4">
            <Link href={`/production/plans/${planId}`}>
              <Button variant="ghost" size="icon" data-testid="button-back">
                <ArrowLeft className="h-5 w-5" />
              </Button>
            </Link>
            <div>
              <h1 className="text-lg font-semibold">Ustawienia produkcji</h1>
              <p className="text-sm text-muted-foreground">
                {plan.planNumber} - {plan.name}
              </p>
            </div>
          </div>

          <div className="flex items-center gap-2 flex-wrap">
            {!zlpLocked && (
              <Tooltip>
                <TooltipTrigger asChild>
                  <span>
                    <Button
                      onClick={() => generateOrdersMutation.mutate()}
                      disabled={isGenerating || planLinesCount === 0}
                      size="sm"
                      variant="default"
                      data-testid="button-generate-zlp"
                      className="gap-1.5"
                    >
                      {isGenerating ? (
                        <>
                          <Loader2 className="h-4 w-4 animate-spin" />
                          <span className="hidden sm:inline">Generowanie...</span>
                        </>
                      ) : planLinesCount === 0 ? (
                        <>
                          <AlertTriangle className="h-4 w-4" />
                          <span className="hidden sm:inline">{hasZlps ? "Regeneruj ZLP" : "Generuj ZLP"}</span>
                          <span className="sm:hidden">ZLP</span>
                        </>
                      ) : (
                        <>
                          <Factory className="h-4 w-4" />
                          <span className="hidden sm:inline">{hasZlps ? "Regeneruj ZLP" : "Generuj ZLP"}</span>
                          <span className="sm:hidden">ZLP</span>
                        </>
                      )}
                    </Button>
                  </span>
                </TooltipTrigger>
                {planLinesCount === 0 && !isGenerating && (
                  <TooltipContent>
                    <p>Dodaj produkty do planu przed generowaniem ZLP</p>
                  </TooltipContent>
                )}
              </Tooltip>
            )}

            {hasZlps && (
              zlpLocked ? (
                <div className="flex items-center gap-1">
                  {!showUnlockConfirm ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Badge 
                          variant="outline" 
                          className="cursor-pointer hover-elevate gap-1 px-2 py-1"
                          onClick={() => setShowUnlockConfirm(true)}
                          data-testid="badge-unlock-zlp-step1"
                        >
                          <Lock className="h-3 w-3" />
                          <span className="text-xs">Zablokowany</span>
                        </Badge>
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Kliknij aby odblokować plan</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <div className="flex items-center gap-1 animate-in fade-in slide-in-from-left-2 duration-200">
                      <Button
                        onClick={() => {
                          unlockZlpMutation.mutate();
                          setShowUnlockConfirm(false);
                        }}
                        disabled={unlockZlpMutation.isPending}
                        size="sm"
                        variant="outline"
                        data-testid="button-unlock-zlp"
                        className="gap-1.5"
                      >
                        {unlockZlpMutation.isPending ? (
                          <Loader2 className="h-4 w-4 animate-spin" />
                        ) : (
                          <Unlock className="h-4 w-4" />
                        )}
                        <span>Odblokuj</span>
                      </Button>
                      <Button
                        onClick={() => setShowUnlockConfirm(false)}
                        size="icon"
                        variant="ghost"
                        className="h-8 w-8"
                        data-testid="button-cancel-unlock"
                      >
                        <X className="h-4 w-4" />
                      </Button>
                    </div>
                  )}
                </div>
              ) : (
                <div className="flex items-center gap-1">
                  {!showLockConfirm ? (
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Badge 
                          variant="secondary" 
                          className="cursor-pointer hover-elevate gap-1 px-2 py-1 bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400"
                          onClick={() => setShowLockConfirm(true)}
                          data-testid="badge-lock-zlp-step1"
                        >
                          <Lock className="h-3 w-3" />
                          <span className="text-xs">Zatwierdź...</span>
                        </Badge>
                      </TooltipTrigger>
                      <TooltipContent>
                        <p>Kliknij aby zatwierdzić i zablokować plan</p>
                      </TooltipContent>
                    </Tooltip>
                  ) : (
                    <div className="flex items-center gap-1 animate-in fade-in slide-in-from-left-2 duration-200">
                      <Button
                        onClick={() => {
                          lockZlpMutation.mutate();
                          setShowLockConfirm(false);
                        }}
                        disabled={lockZlpMutation.isPending}
                        size="sm"
                        variant="default"
                        data-testid="button-lock-zlp"
                        className="gap-1.5"
                      >
                        {lockZlpMutation.isPending ? (
                          <Loader2 className="h-4 w-4 animate-spin" />
                        ) : (
                          <Lock className="h-4 w-4" />
                        )}
                        <span>Zatwierdź</span>
                      </Button>
                      <Button
                        onClick={() => setShowLockConfirm(false)}
                        size="icon"
                        variant="ghost"
                        className="h-8 w-8"
                        data-testid="button-cancel-lock"
                      >
                        <X className="h-4 w-4" />
                      </Button>
                    </div>
                  )}
                </div>
              )
            )}
          </div>
        </div>
      </div>

      <div className="flex-1 overflow-hidden">
        <Tabs value={activeTab} onValueChange={setActiveTab} className="h-full flex flex-col">
          <div className="border-b bg-muted/20">
            <ScrollArea className="w-full">
              <TabsList className="h-12 bg-transparent inline-flex w-max min-w-full px-4">
                <TabsTrigger 
                  value="routing-assignments" 
                  className="gap-2 data-[state=active]:bg-background whitespace-nowrap"
                  data-testid="tab-routing-assignments"
                >
                  <Route className="h-4 w-4" />
                  <span className="hidden sm:inline">Przypisanie marszrut</span>
                  <span className="sm:hidden">Marszruty</span>
                </TabsTrigger>
                <TabsTrigger 
                  value="cutting" 
                  className="gap-2 data-[state=active]:bg-background whitespace-nowrap"
                  data-testid="tab-cutting"
                >
                  <Scissors className="h-4 w-4" />
                  <span className="hidden sm:inline">Kolejność cięcia</span>
                  <span className="sm:hidden">Cięcie</span>
                </TabsTrigger>
                <TabsTrigger 
                  value="transport" 
                  className="gap-2 data-[state=active]:bg-background whitespace-nowrap"
                  data-testid="tab-transport"
                >
                  <Truck className="h-4 w-4" />
                  <span className="hidden sm:inline">Partie transportowe</span>
                  <span className="sm:hidden">Transport</span>
                </TabsTrigger>
                <TabsTrigger 
                  value="merge-points" 
                  className="gap-2 data-[state=active]:bg-background whitespace-nowrap"
                  data-testid="tab-merge-points"
                >
                  <GitMerge className="h-4 w-4" />
                  <span className="hidden sm:inline">Punkty zbieżności</span>
                  <span className="sm:hidden">Zbieżność</span>
                </TabsTrigger>
                <TabsTrigger 
                  value="operators" 
                  className="gap-2 data-[state=active]:bg-background whitespace-nowrap"
                  data-testid="tab-operators"
                >
                  <Users className="h-4 w-4" />
                  <span className="hidden sm:inline">Operatorzy</span>
                  <span className="sm:hidden">Operatorzy</span>
                </TabsTrigger>
                <Link 
                  href={`/production/plans/${planId}/flow-tree`}
                  className="inline-flex items-center justify-center gap-2 px-3 py-1.5 text-sm font-medium rounded-md whitespace-nowrap ring-offset-background transition-all focus-visible:outline-none focus-visible:ring-2 focus-visible:ring-ring focus-visible:ring-offset-2 hover:bg-muted/50 text-muted-foreground hover:text-foreground"
                  data-testid="tab-flow-tree"
                >
                  <Network className="h-4 w-4" />
                  <span className="hidden sm:inline">Drzewo przepływu</span>
                  <span className="sm:hidden">Przepływ</span>
                </Link>
              </TabsList>
              <ScrollBar orientation="horizontal" />
            </ScrollArea>
          </div>

          <TabsContent value="routing-assignments" className="flex-1 overflow-auto m-0 p-4" data-testid="content-routing-assignments">
            <PlanRoutingAssignmentsPanel planId={planId} zlpLocked={plan.zlpLocked} hasZlps={hasZlps} />
          </TabsContent>

          <TabsContent value="cutting" className="flex-1 overflow-auto m-0 p-4" data-testid="content-cutting">
            <CuttingSequencePanel planId={planId} />
          </TabsContent>

          <TabsContent value="transport" className="flex-1 overflow-auto m-0 p-4" data-testid="content-transport">
            <TransportBatchesPanel planId={planId} />
          </TabsContent>

          <TabsContent value="merge-points" className="flex-1 overflow-auto m-0 p-4" data-testid="content-merge-points">
            <MergePointsPanel planId={planId} />
          </TabsContent>

          <TabsContent value="operators" className="flex-1 overflow-auto m-0 p-4" data-testid="content-operators">
            <OperatorsAssignmentPanel planId={planId} />
          </TabsContent>
        </Tabs>
      </div>

      <ZlpGenerationProgressDialog
        sessionId={zlpSessionId}
        open={zlpProgressDialogOpen}
        onOpenChange={setZlpProgressDialogOpen}
        planId={planId}
        onComplete={handleZlpGenerationComplete}
      />
    </div>
  );
}
