import { useCallback, useMemo, useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useRoute, useLocation, Link } from "wouter";
import { queryClient, apiRequest } from "@/lib/queryClient";
import {
  ReactFlow,
  Node,
  Edge,
  Controls,
  Background,
  BackgroundVariant,
  MiniMap,
  useNodesState,
  useEdgesState,
  MarkerType,
  Handle,
  Position,
} from "@xyflow/react";
import dagre from "dagre";
import "@xyflow/react/dist/style.css";
import {
  ArrowLeft,
  Loader2,
  Play,
  CheckCircle,
  Clock,
  Package,
  Truck,
  GitMerge,
  CircleDot,
  Flag,
  AlertTriangle,
  Box,
  Route,
  User,
  CalendarClock,
  ChevronDown,
  ChevronRight,
  Layers,
  FileText,
  Settings,
  Trash2,
  Plus,
  Pencil,
  Factory,
  PanelRightOpen,
  PanelRightClose,
  Scissors,
  Palette,
  Split,
  ArrowRight,
} from "lucide-react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Tooltip,
  TooltipContent,
  TooltipTrigger,
} from "@/components/ui/tooltip";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Checkbox } from "@/components/ui/checkbox";
import { Label } from "@/components/ui/label";
import { ScrollArea } from "@/components/ui/scroll-area";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Switch } from "@/components/ui/switch";
import { Combobox } from "@/components/ui/combobox";
import { useToast } from "@/hooks/use-toast";
import { Star } from "lucide-react";
import type {
  FlowNode,
  FlowEdge,
  FlowNodeType,
  FlowNodeStatus,
  ComponentFamily,
  ProductionPlanFlowGraph,
  MergePointProgress,
} from "@shared/schema";
import { Progress } from "@/components/ui/progress";
import { getColorHex, getTextColor } from "@/features/production-plans/color-helpers";
import { ZlpGenerationProgressDialog } from "@/components/zlp-generation-progress-dialog";

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

interface ProductionOperator {
  id: number;
  full_name: string;
  short_code: string | null;
  work_center_id: number | null;
  is_active: boolean;
}

interface SelectedNodeInfo {
  workOrderIds: number[];
  selectedWorkOrderId: number;
  label: string;
  currentOperatorId?: number | null;
  currentOperatorName?: string | null;
}

interface PalletFlowInfo {
  flowCode: string;
  color: string;
  palletCount: number;
  operations: { name: string; operationName: string; isSplitPoint: boolean; isShared: boolean }[];
}

interface OperationNode {
  id: string;
  operationCode: string;
  operationName: string;
  flowCodes: string[];
  isSplitPoint: boolean;
}

function PalletTreePanel({ 
  flowGraph, 
  isOpen, 
  onToggle 
}: { 
  flowGraph: ProductionPlanFlowGraph | undefined;
  isOpen: boolean;
  onToggle: () => void;
}) {
  const { operationNodes, palletFlows, splitPointNode, sharedOps, postSplitFlows } = useMemo(() => {
    if (!flowGraph?.nodes || !flowGraph?.edges) {
      return { operationNodes: [], palletFlows: [], splitPointNode: null, sharedOps: [], postSplitFlows: new Map() };
    }

    const ops: OperationNode[] = [];
    let splitNode: OperationNode | null = null;

    flowGraph.nodes.forEach(node => {
      if (node.type === 'operation' && node.data.operationCode) {
        const opNode: OperationNode = {
          id: node.id,
          operationCode: node.data.operationCode,
          operationName: node.data.operationName || node.data.operationCode,
          flowCodes: node.data.flowCodes || [],
          isSplitPoint: node.data.isSplitPoint || false,
        };
        ops.push(opNode);
        if (node.data.isSplitPoint) {
          splitNode = opNode;
        }
      }
    });

    const edgeMap = new Map<string, string[]>();
    flowGraph.edges.forEach(edge => {
      if (!edgeMap.has(edge.source)) edgeMap.set(edge.source, []);
      edgeMap.get(edge.source)!.push(edge.target);
    });

    const reverseEdgeMap = new Map<string, string[]>();
    flowGraph.edges.forEach(edge => {
      if (!reverseEdgeMap.has(edge.target)) reverseEdgeMap.set(edge.target, []);
      reverseEdgeMap.get(edge.target)!.push(edge.source);
    });

    const opById = new Map<string, OperationNode>();
    ops.forEach(op => opById.set(op.id, op));

    const orderedOps: OperationNode[] = [];
    const visited = new Set<string>();

    const startNodes = ops.filter(op => {
      const sources = reverseEdgeMap.get(op.id) || [];
      return sources.every(s => !opById.has(s));
    });

    function dfs(nodeId: string) {
      if (visited.has(nodeId)) return;
      visited.add(nodeId);
      
      const op = opById.get(nodeId);
      if (op) orderedOps.push(op);
      
      const targets = edgeMap.get(nodeId) || [];
      for (const target of targets) {
        if (opById.has(target)) {
          dfs(target);
        } else {
          const nextTargets = edgeMap.get(target) || [];
          nextTargets.forEach(t => dfs(t));
        }
      }
    }

    startNodes.forEach(start => dfs(start.id));
    ops.forEach(op => {
      if (!visited.has(op.id)) dfs(op.id);
    });

    const shared: OperationNode[] = [];
    const postSplit = new Map<string, OperationNode[]>();
    let foundSplit = false;

    orderedOps.forEach(op => {
      if (op.isSplitPoint) {
        shared.push(op);
        foundSplit = true;
      } else if (!foundSplit) {
        if (op.flowCodes.length > 1) {
          shared.push(op);
        } else {
          op.flowCodes.forEach(fc => {
            if (!postSplit.has(fc)) postSplit.set(fc, []);
            postSplit.get(fc)!.push(op);
          });
        }
      } else {
        op.flowCodes.forEach(fc => {
          if (!postSplit.has(fc)) postSplit.set(fc, []);
          postSplit.get(fc)!.push(op);
        });
      }
    });

    const allFlowCodes = new Set<string>();
    ops.forEach(op => op.flowCodes.forEach(fc => allFlowCodes.add(fc)));

    const flows: PalletFlowInfo[] = Array.from(allFlowCodes).map(flowCode => {
      const color = flowCode === 'CO' ? 'bg-blue-500' :
                    flowCode === 'COW' ? 'bg-purple-500' :
                    flowCode === 'CW' ? 'bg-orange-500' : 'bg-gray-500';

      const flowOps = orderedOps
        .filter(op => op.flowCodes.includes(flowCode))
        .map(op => ({
          name: op.operationCode,
          operationName: op.operationName,
          isSplitPoint: op.isSplitPoint,
          isShared: op.flowCodes.length > 1,
        }));

      const palletNode = flowGraph.nodes.find(n => 
        n.data.flowCodes?.includes(flowCode) && n.data.palletCount
      );

      return {
        flowCode,
        color,
        palletCount: palletNode?.data.palletCount || 0,
        operations: flowOps,
      };
    });

    return {
      operationNodes: orderedOps,
      palletFlows: flows,
      splitPointNode: splitNode,
      sharedOps: shared,
      postSplitFlows: postSplit,
    };
  }, [flowGraph]);

  if (!isOpen) {
    return (
      <Button
        variant="outline"
        size="icon"
        className="fixed right-4 top-1/2 -translate-y-1/2 z-50 h-10 w-10 rounded-full shadow-lg"
        onClick={onToggle}
        data-testid="button-open-pallet-tree"
      >
        <PanelRightOpen className="h-5 w-5" />
      </Button>
    );
  }

  return (
    <div className="fixed right-0 top-0 h-full w-80 bg-background border-l shadow-xl z-40 flex flex-col">
      <div className="flex items-center justify-between p-3 border-b">
        <div className="flex items-center gap-2">
          <GitMerge className="h-5 w-5 text-amber-500" />
          <h3 className="font-semibold">Drzewo palet</h3>
        </div>
        <Button variant="ghost" size="icon" onClick={onToggle} data-testid="button-close-pallet-tree">
          <PanelRightClose className="h-5 w-5" />
        </Button>
      </div>
      
      <ScrollArea className="flex-1 p-3">
        {palletFlows.length === 0 ? (
          <div className="text-center py-8 text-muted-foreground">
            <Package className="h-12 w-12 mx-auto mb-2 opacity-30" />
            <p className="text-sm">Brak danych o paletach.</p>
            <p className="text-xs mt-1">Wygeneruj ZLP aby zobaczyć przepływy palet.</p>
          </div>
        ) : (
          <div className="space-y-4">
            {sharedOps.length > 0 && (
              <Card className="border-dashed border-green-500/50">
                <CardHeader className="p-3 pb-2">
                  <CardTitle className="text-sm flex items-center gap-2">
                    <Scissors className="h-4 w-4 text-green-600" />
                    Wspólny etap produkcji
                  </CardTitle>
                  <p className="text-[10px] text-muted-foreground">
                    Wszystkie przepływy idą razem
                  </p>
                </CardHeader>
                <CardContent className="p-3 pt-0">
                  <div className="space-y-1.5">
                    {sharedOps.map((op, idx) => (
                      <div key={idx} className="flex items-center gap-2">
                        <div className={`w-5 h-5 rounded flex items-center justify-center text-[9px] font-bold ${
                          op.isSplitPoint ? 'bg-amber-500 text-white' : 'bg-green-100 dark:bg-green-900/40 text-green-700 dark:text-green-300'
                        }`}>
                          {idx + 1}
                        </div>
                        <div className="flex-1">
                          <span className="text-xs font-medium">{op.operationName}</span>
                          {op.isSplitPoint && (
                            <div className="flex items-center gap-1 text-[9px] text-amber-600 dark:text-amber-400">
                              <Split className="h-2.5 w-2.5" />
                              <span>punkt podziału</span>
                            </div>
                          )}
                        </div>
                        <div className="flex gap-0.5">
                          {op.flowCodes.map(fc => (
                            <span
                              key={fc}
                              className={`text-[8px] px-1 py-0.5 rounded text-white ${
                                fc === 'CO' ? 'bg-blue-500' :
                                fc === 'COW' ? 'bg-purple-500' :
                                fc === 'CW' ? 'bg-orange-500' : 'bg-gray-500'
                              }`}
                            >
                              {fc}
                            </span>
                          ))}
                        </div>
                      </div>
                    ))}
                  </div>
                </CardContent>
              </Card>
            )}

            {splitPointNode && (
              <Card className="border-amber-500/50 bg-amber-50/50 dark:bg-amber-900/20">
                <CardContent className="p-3">
                  <div className="flex items-center gap-2">
                    <Split className="h-5 w-5 text-amber-600" />
                    <div className="flex-1">
                      <p className="text-xs font-medium text-amber-700 dark:text-amber-400">Punkt podziału palet</p>
                      <p className="text-sm font-semibold">{splitPointNode.operationName}</p>
                    </div>
                  </div>
                  <div className="flex flex-wrap gap-1 mt-2">
                    {splitPointNode.flowCodes.map(fc => (
                      <Badge
                        key={fc}
                        className={`text-white text-[10px] ${
                          fc === 'CO' ? 'bg-blue-500' :
                          fc === 'COW' ? 'bg-purple-500' :
                          fc === 'CW' ? 'bg-orange-500' : 'bg-gray-500'
                        }`}
                      >
                        {fc}
                      </Badge>
                    ))}
                    <ArrowRight className="h-4 w-4 text-amber-600 mx-1" />
                    <span className="text-[10px] text-muted-foreground">rozdzielają się</span>
                  </div>
                </CardContent>
              </Card>
            )}

            <div className="space-y-2">
              <h4 className="text-xs font-medium text-muted-foreground uppercase tracking-wide flex items-center gap-2">
                <Route className="h-3 w-3" />
                Ścieżki przepływów po podziale
              </h4>
              {palletFlows.map((flow) => {
                const postSplitOps = flow.operations.filter(op => !op.isShared && !op.isSplitPoint);
                return (
                  <Card key={flow.flowCode}>
                    <CardContent className="p-3">
                      <div className="flex items-center justify-between mb-2">
                        <Badge className={`${flow.color} text-white`}>
                          {flow.flowCode}
                        </Badge>
                        <span className="text-xs text-muted-foreground">
                          {flow.palletCount > 0 ? `${flow.palletCount} palet` : ''}
                        </span>
                      </div>
                      
                      <div className="text-[10px] text-muted-foreground mb-2">
                        Pełna ścieżka: {flow.operations.map(op => op.operationName).join(' → ')}
                      </div>
                      
                      {postSplitOps.length > 0 ? (
                        <div className="space-y-1 border-t pt-2 mt-2">
                          <div className="text-[9px] text-muted-foreground uppercase">Po podziale:</div>
                          {postSplitOps.map((op, idx) => (
                            <div key={idx} className="flex items-center gap-2 text-xs">
                              <div className={`w-4 h-4 rounded flex items-center justify-center text-[8px] ${flow.color} text-white`}>
                                {idx + 1}
                              </div>
                              <span>{op.operationName}</span>
                            </div>
                          ))}
                        </div>
                      ) : (
                        <div className="text-[10px] text-muted-foreground italic">
                          Kończy po podziale (brak dodatkowych operacji)
                        </div>
                      )}
                    </CardContent>
                  </Card>
                );
              })}
            </div>

            <Card className="bg-muted/50">
              <CardContent className="p-3">
                <h4 className="text-xs font-medium mb-2">Legenda przepływów</h4>
                <div className="space-y-1.5 text-[10px]">
                  <div className="flex items-center gap-2">
                    <Badge className="bg-blue-500 text-white h-4 px-1.5">CO</Badge>
                    <span>Cięcie + Oklejanie</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Badge className="bg-purple-500 text-white h-4 px-1.5">COW</Badge>
                    <span>Cięcie + Oklejanie + Wiercenie</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <Badge className="bg-orange-500 text-white h-4 px-1.5">CW</Badge>
                    <span>Cięcie + Wiercenie</span>
                  </div>
                </div>
                <div className="border-t mt-2 pt-2 space-y-1 text-[10px]">
                  <div className="flex items-center gap-2">
                    <div className="w-4 h-4 rounded bg-green-100 dark:bg-green-900/40 flex items-center justify-center text-green-700 dark:text-green-300 text-[8px] font-bold">1</div>
                    <span>Wspólna operacja</span>
                  </div>
                  <div className="flex items-center gap-2">
                    <div className="w-4 h-4 rounded bg-amber-500 flex items-center justify-center text-white text-[8px] font-bold">2</div>
                    <span>Punkt podziału palet</span>
                  </div>
                </div>
              </CardContent>
            </Card>
          </div>
        )}
      </ScrollArea>
    </div>
  );
}

const nodeWidth = 220;
const nodeHeight = 140;

const getLayoutedElements = (
  nodes: Node[],
  edges: Edge[],
  direction = "TB"
) => {
  const dagreGraph = new dagre.graphlib.Graph();
  dagreGraph.setDefaultEdgeLabel(() => ({}));
  dagreGraph.setGraph({ rankdir: direction, nodesep: 120, ranksep: 140 });

  nodes.forEach((node) => {
    const height = node.data?.isSplitPoint ? nodeHeight + 30 : nodeHeight;
    dagreGraph.setNode(node.id, { width: nodeWidth, height });
  });

  edges.forEach((edge) => {
    dagreGraph.setEdge(edge.source, edge.target);
  });

  dagre.layout(dagreGraph);

  const layoutedNodes = nodes.map((node) => {
    const nodeWithPosition = dagreGraph.node(node.id);
    const height = node.data?.isSplitPoint ? nodeHeight + 30 : nodeHeight;
    return {
      ...node,
      position: {
        x: nodeWithPosition.x - nodeWidth / 2,
        y: nodeWithPosition.y - height / 2,
      },
    };
  });

  return { nodes: layoutedNodes, edges };
};

// Transform flow graph to split nodes after split point into separate nodes per flow
function splitNodesAfterSplitPoint(
  flowGraph: ProductionPlanFlowGraph
): { nodes: FlowNode[]; edges: FlowEdge[] } {
  if (!flowGraph.nodes || !flowGraph.edges) {
    return { nodes: [], edges: [] };
  }

  const nodes = [...flowGraph.nodes];
  const edges = [...flowGraph.edges];
  
  // Find split point node
  const splitPointNode = nodes.find(n => n.data.isSplitPoint);
  if (!splitPointNode) {
    return { nodes, edges };
  }

  // Find all nodes that come after split point (by traversing edges)
  const nodesAfterSplit = new Set<string>();
  const visited = new Set<string>();
  
  function findNodesAfter(nodeId: string) {
    if (visited.has(nodeId)) return;
    visited.add(nodeId);
    
    const outEdges = edges.filter(e => e.source === nodeId);
    for (const edge of outEdges) {
      if (edge.target !== splitPointNode.id) {
        nodesAfterSplit.add(edge.target);
        findNodesAfter(edge.target);
      }
    }
  }
  
  findNodesAfter(splitPointNode.id);

  // Split nodes that have multiple flowCodes into separate nodes per flow
  const newNodes: FlowNode[] = [];
  const newEdges: FlowEdge[] = [];
  const nodeIdMapping = new Map<string, Map<string, string>>(); // originalId -> flowCode -> newId

  for (const node of nodes) {
    const flowCodes = node.data.flowCodes || [];
    
    // If node is after split point and has multiple flow codes, split it
    if (nodesAfterSplit.has(node.id) && flowCodes.length > 1) {
      for (const flowCode of flowCodes) {
        const newId = `${node.id}-${flowCode}`;
        
        // Count pallets for this specific flow
        const palletsForFlow = (node.data.pallets || []).filter(
          (p: { flowCode: string }) => p.flowCode === flowCode
        );
        
        const newNode: FlowNode = {
          ...node,
          id: newId,
          label: `${node.label} (${flowCode})`,
          data: {
            ...node.data,
            flowCodes: [flowCode],
            palletCount: palletsForFlow.length,
            pallets: palletsForFlow,
          },
        };
        newNodes.push(newNode);
        
        if (!nodeIdMapping.has(node.id)) {
          nodeIdMapping.set(node.id, new Map());
        }
        nodeIdMapping.get(node.id)!.set(flowCode, newId);
      }
    } else {
      newNodes.push(node);
    }
  }

  // Rebuild edges with new node IDs
  for (const edge of edges) {
    const sourceMapping = nodeIdMapping.get(edge.source);
    const targetMapping = nodeIdMapping.get(edge.target);
    
    if (sourceMapping && targetMapping) {
      // Both source and target were split - create edge per flow
      Array.from(sourceMapping.entries()).forEach(([flowCode, sourceId]) => {
        const targetId = targetMapping.get(flowCode);
        if (targetId) {
          newEdges.push({
            ...edge,
            id: `${sourceId}-${targetId}`,
            source: sourceId,
            target: targetId,
          });
        }
      });
    } else if (sourceMapping) {
      // Only source was split - connect each new source to original target
      // but only if target's flowCodes include this flow
      const targetNode = nodes.find(n => n.id === edge.target);
      const targetFlows = targetNode?.data.flowCodes || [];
      
      Array.from(sourceMapping.entries()).forEach(([flowCode, sourceId]) => {
        // Only create edge if target has no flowCodes (like merge/end nodes) or includes this flow
        if (targetFlows.length === 0 || targetFlows.includes(flowCode)) {
          newEdges.push({
            ...edge,
            id: `${sourceId}-${edge.target}`,
            source: sourceId,
            target: edge.target,
          });
        }
      });
    } else if (targetMapping) {
      // Only target was split - connect original source to each new target
      // But we need to figure out which flows the source has
      const sourceNode = nodes.find(n => n.id === edge.source);
      const sourceFlows = sourceNode?.data.flowCodes || [];
      
      if (sourceFlows.length > 0) {
        for (const flowCode of sourceFlows) {
          const targetId = targetMapping.get(flowCode);
          if (targetId) {
            newEdges.push({
              ...edge,
              id: `${edge.source}-${targetId}`,
              source: edge.source,
              target: targetId,
            });
          }
        }
      } else {
        // Source has no flow info, connect to all targets
        Array.from(targetMapping.values()).forEach((targetId) => {
          newEdges.push({
            ...edge,
            id: `${edge.source}-${targetId}`,
            source: edge.source,
            target: targetId,
          });
        });
      }
    } else {
      // Neither was split - keep original edge
      newEdges.push(edge);
    }
  }

  return { nodes: newNodes, edges: newEdges };
}

const statusColors: Record<FlowNodeStatus, { bg: string; border: string; text: string }> = {
  pending: { bg: "bg-gray-100 dark:bg-gray-800", border: "border-gray-300 dark:border-gray-600", text: "text-gray-600 dark:text-gray-400" },
  ready: { bg: "bg-blue-100 dark:bg-blue-900/40", border: "border-blue-400 dark:border-blue-600", text: "text-blue-700 dark:text-blue-300" },
  in_progress: { bg: "bg-yellow-100 dark:bg-yellow-900/40", border: "border-yellow-400 dark:border-yellow-600", text: "text-yellow-700 dark:text-yellow-300" },
  completed: { bg: "bg-green-100 dark:bg-green-900/40", border: "border-green-400 dark:border-green-600", text: "text-green-700 dark:text-green-300" },
  blocked: { bg: "bg-red-100 dark:bg-red-900/40", border: "border-red-400 dark:border-red-600", text: "text-red-700 dark:text-red-300" },
};

const familyLabels: Record<ComponentFamily, string> = {
  formatki_hdf: "HDF",
  formatki_biale: "Białe",
  formatki_kolor: "Kolorowe",
  tapicerowane: "Tapicerowane",
  siedziska: "Siedziska",
  akcesoria: "Akcesoria",
  mixed: "Mieszane",
};

const familyColors: Record<ComponentFamily, string> = {
  formatki_hdf: "bg-amber-500",
  formatki_biale: "bg-slate-100 dark:bg-slate-700 text-slate-900 dark:text-slate-100",
  formatki_kolor: "bg-purple-500",
  tapicerowane: "bg-rose-500",
  siedziska: "bg-cyan-500",
  akcesoria: "bg-emerald-500",
  mixed: "bg-gray-500",
};

const nodeTypeIcons: Record<FlowNodeType, typeof Play> = {
  operation: Play,
  buffer: Box,
  transport: Truck,
  merge: GitMerge,
  split: GitMerge,
  start: CircleDot,
  end: Flag,
  warehouse_packed: Package,
  buffer_formatki: Layers,
};

function getMaterialColorStyle(colorCode: string, colors: DictionaryItem[]): { bg: string; text: string; hex?: string } {
  const hexColor = getColorHex(colorCode, colors);
  if (hexColor) {
    const textColor = getTextColor(hexColor);
    return { bg: '', text: '', hex: hexColor };
  }
  
  const code = (colorCode || '').toUpperCase();
  if (code.startsWith('HDF')) return { bg: 'bg-amber-600', text: 'text-white' };
  if (code.includes('BIALY') || code.includes('BIAŁY') || code === 'BIALY') return { bg: 'bg-slate-100 dark:bg-slate-300', text: 'text-slate-900' };
  if (code.includes('SONOMA')) return { bg: 'bg-yellow-700', text: 'text-white' };
  if (code.includes('ORZECH')) return { bg: 'bg-amber-900', text: 'text-white' };
  if (code.includes('DAB') || code.includes('DĄB')) return { bg: 'bg-amber-800', text: 'text-white' };
  if (code.includes('CZARNY') || code.includes('BLACK')) return { bg: 'bg-gray-900', text: 'text-white' };
  if (code.includes('SZARY') || code.includes('GREY')) return { bg: 'bg-gray-500', text: 'text-white' };
  if (code.includes('SUROWA')) return { bg: 'bg-stone-400', text: 'text-stone-900' };
  if (code.includes('tapicerowane') || code.includes('TAPIC')) return { bg: 'bg-rose-500', text: 'text-white' };
  if (code.includes('siedziska') || code.includes('SIEDZISKO')) return { bg: 'bg-cyan-500', text: 'text-white' };
  if (code.includes('akcesoria')) return { bg: 'bg-emerald-500', text: 'text-white' };
  return { bg: 'bg-purple-500', text: 'text-white' };
}

function formatDateTime(isoString: string | undefined): string | null {
  if (!isoString) return null;
  try {
    const date = new Date(isoString);
    return date.toLocaleString('pl-PL', { 
      day: '2-digit', 
      month: '2-digit', 
      hour: '2-digit', 
      minute: '2-digit' 
    });
  } catch {
    return null;
  }
}

interface OperationItem {
  id: number;
  componentName: string;
  componentType: string;
  quantity: number;
  colorCode: string;
  length: number | null;
  width: number | null;
  thickness: number | null;
  isDamaged: boolean;
  itemStatus: string;
  zlpNumber: string;
  sourceFurnitureReference: string | null;
}

function ElementsSection({ 
  planId, 
  zlpIds, 
  materialColor, 
  operationCode,
  itemCount 
}: { 
  planId: number; 
  zlpIds: number[]; 
  materialColor?: string; 
  operationCode?: string;
  itemCount: number;
}) {
  const [isExpanded, setIsExpanded] = useState(false);
  
  const queryParamsObj: Record<string, string> = {
    zlpIds: zlpIds.join(','),
  };
  if (materialColor) queryParamsObj.materialColor = materialColor;
  if (operationCode) queryParamsObj.operationCode = operationCode;
  
  const { data: items, isLoading, error } = useQuery<OperationItem[]>({
    queryKey: [`/api/production/planning/${planId}/operation-items`, queryParamsObj],
    enabled: isExpanded && zlpIds.length > 0,
  });

  if (itemCount === 0) return null;

  return (
    <div className="border-t border-dashed pt-1 mt-1">
      <button
        onClick={(e) => {
          e.stopPropagation();
          setIsExpanded(!isExpanded);
        }}
        className="flex items-center gap-1 text-[9px] text-muted-foreground hover:text-foreground transition-colors w-full"
        data-testid="button-toggle-elements"
      >
        {isExpanded ? (
          <ChevronDown className="h-2.5 w-2.5" />
        ) : (
          <ChevronRight className="h-2.5 w-2.5" />
        )}
        <Layers className="h-2.5 w-2.5" />
        <span>Elementy ({itemCount})</span>
      </button>
      
      {isExpanded && (
        <div className="mt-1 max-h-32 overflow-y-auto">
          {isLoading ? (
            <div className="flex items-center justify-center py-2">
              <Loader2 className="h-3 w-3 animate-spin text-muted-foreground" />
            </div>
          ) : error ? (
            <div className="text-[9px] text-red-500 py-1">Błąd ładowania</div>
          ) : items && items.length > 0 ? (
            <div className="space-y-0.5">
              {items.slice(0, 10).map((item) => (
                <div 
                  key={item.id} 
                  className={`text-[8px] px-1 py-0.5 rounded bg-muted/50 flex items-center gap-1 ${item.isDamaged ? 'border-l-2 border-red-500' : ''}`}
                >
                  <span className="font-medium truncate max-w-[80px]">{item.componentName}</span>
                  {item.length && item.width && (
                    <span className="text-muted-foreground">{item.length}x{item.width}</span>
                  )}
                  {item.quantity > 1 && (
                    <span className="text-muted-foreground">x{item.quantity}</span>
                  )}
                </div>
              ))}
              {items.length > 10 && (
                <div className="text-[8px] text-muted-foreground text-center py-0.5">
                  +{items.length - 10} więcej...
                </div>
              )}
            </div>
          ) : (
            <div className="text-[9px] text-muted-foreground py-1">Brak elementów</div>
          )}
        </div>
      )}
    </div>
  );
}

function CustomNode({ data, type, id }: { data: FlowNode["data"] & { label: string; status: FlowNodeStatus; nodeType: FlowNodeType; colorDictionary?: DictionaryItem[]; planId?: number }; type: string; id: string }) {
  const colors = statusColors[data.status] || statusColors.pending;
  const Icon = nodeTypeIcons[data.nodeType] || Play;
  const materialColor = data.materialColor;
  const colorDict = data.colorDictionary || [];
  const isInProgress = data.status === 'in_progress';
  
  const startTime = formatDateTime(data.actualStartTime);
  const endTime = formatDateTime(data.actualEndTime);
  
  return (
    <div className={`px-3 py-2 rounded-lg border-2 ${colors.bg} ${isInProgress ? 'animate-glow-pulse' : colors.border} min-w-[160px] shadow-sm`}>
      <Handle type="target" position={Position.Top} className="!bg-muted-foreground" />
      
      {/* Node ID, routing operation ID and work order IDs from database */}
      <div className="text-[8px] text-muted-foreground font-mono mb-0.5 opacity-60">
        {[
          data.routingOperationId ? `RO:${data.routingOperationId}` : null,
          data.workOrderIds && data.workOrderIds.length > 0 ? `WO:${data.workOrderIds.slice(0, 3).join(',')}` : null,
          `ID:${id}`
        ].filter(Boolean).join(' | ')}
      </div>
      
      <div className="flex items-center gap-2 mb-1">
        <Icon className={`h-4 w-4 ${colors.text}`} />
        <span className={`text-sm font-medium ${colors.text} truncate`}>{data.label}</span>
      </div>
      
      {data.workCenterName && (
        <div className="text-xs text-muted-foreground truncate mb-1">
          {data.workCenterName}
        </div>
      )}
      
      {materialColor && (() => {
        const style = getMaterialColorStyle(materialColor, colorDict);
        return (
          <div className="flex flex-wrap gap-1 mb-1">
            {style.hex ? (
              <span 
                className="text-[10px] px-1.5 py-0.5 rounded"
                style={{ backgroundColor: style.hex, color: getTextColor(style.hex) }}
              >
                {materialColor}
              </span>
            ) : (
              <span className={`text-[10px] px-1.5 py-0.5 rounded ${style.bg} ${style.text}`}>
                {materialColor}
              </span>
            )}
          </div>
        );
      })()}
      
      {/* ZLP numbers */}
      {data.zlpNumbers && data.zlpNumbers.length > 0 && data.zlpNumbers.length <= 3 && (
        <div className="flex flex-wrap items-center gap-1 mb-1">
          <FileText className="h-3 w-3 text-muted-foreground" />
          {data.zlpNumbers.filter((n: string) => n).slice(0, 3).map((zlpNum: string, idx: number) => (
            <span key={idx} className="text-[9px] px-1 py-0.5 rounded bg-muted font-mono">
              {zlpNum}
            </span>
          ))}
        </div>
      )}

      {/* Flow codes (CO, COW, CW) - pallet routing paths */}
      {data.flowCodes && data.flowCodes.length > 0 && (
        <div className="flex flex-wrap items-center gap-1 mb-1">
          <Route className="h-3 w-3 text-muted-foreground" />
          {data.flowCodes.map((flowCode: string) => (
            <span
              key={flowCode}
              className={`text-[9px] px-1.5 py-0.5 rounded font-medium ${
                flowCode.includes('CO') && !flowCode.includes('COW') ? 'bg-blue-500 text-white' :
                flowCode.includes('COW') ? 'bg-purple-500 text-white' :
                flowCode.includes('CW') ? 'bg-orange-500 text-white' :
                'bg-gray-500 text-white'
              }`}
            >
              {flowCode.replace('PLAN-marsz_formatki_', '').toUpperCase()}
            </span>
          ))}
        </div>
      )}

      {/* Pallets with carrier names */}
      {(data.pallets && data.pallets.length > 0) || data.zlpIds?.length > 0 ? (
        <div className="flex flex-col gap-0.5 mb-1 border-t border-dashed pt-1 mt-1">
          <div className="flex items-center justify-between text-[9px] text-muted-foreground">
            <div className="flex items-center gap-1">
              <Package className="h-3 w-3" />
              <span>Palety ({data.pallets?.length || 0}):</span>
            </div>
            {data.zlpIds?.length >= 1 && data.flowCodes?.length > 0 && (
              <Tooltip>
                <TooltipTrigger asChild>
                  <button
                    onClick={(e) => {
                      e.stopPropagation();
                      if (data.onAddPallet) {
                        data.onAddPallet(data.zlpIds[0], data.flowCodes[0]);
                      }
                    }}
                    className="h-4 w-4 rounded flex items-center justify-center hover:bg-muted transition-colors"
                    data-testid={`button-add-pallet-${data.zlpIds?.[0]}`}
                  >
                    <Plus className="h-3 w-3" />
                  </button>
                </TooltipTrigger>
                <TooltipContent side="top" className="text-xs">
                  Dodaj paletę
                </TooltipContent>
              </Tooltip>
            )}
          </div>
          {data.pallets && data.pallets.length > 0 && (
            <div className="flex flex-wrap gap-1">
              {data.pallets.slice(0, 4).map((pallet: { id: number; label: string; carrierName?: string; zlpNumber?: string }) => (
                <Tooltip key={pallet.id}>
                  <TooltipTrigger asChild>
                    <span className="text-[8px] px-1.5 py-0.5 rounded bg-amber-100 dark:bg-amber-900/40 text-amber-700 dark:text-amber-300 cursor-help">
                      #{pallet.id} {pallet.carrierName || pallet.label}
                    </span>
                  </TooltipTrigger>
                  <TooltipContent side="bottom" className="text-xs">
                    <p><strong>ID palety:</strong> {pallet.id}</p>
                    <p><strong>Paleta:</strong> {pallet.label}</p>
                    {pallet.carrierName && <p><strong>Nośnik:</strong> {pallet.carrierName}</p>}
                    {pallet.zlpNumber && <p><strong>ZLP:</strong> {pallet.zlpNumber}</p>}
                  </TooltipContent>
                </Tooltip>
              ))}
              {data.pallets.length > 4 && (
                <span className="text-[8px] text-muted-foreground">+{data.pallets.length - 4}</span>
              )}
            </div>
          )}
        </div>
      ) : null}
      
      {/* Split point indicator - where pallets diverge */}
      {data.isSplitPoint && (
        <div className="flex items-center gap-1 text-[9px] text-amber-600 dark:text-amber-400 mb-1 bg-amber-100 dark:bg-amber-900/30 px-1.5 py-0.5 rounded">
          <GitMerge className="h-3 w-3 rotate-180" />
          <span className="font-medium">Punkt podziału palet</span>
        </div>
      )}
      
      {/* Operator and time info */}
      {(data.operatorName || startTime) && (
        <div className="flex flex-col gap-0.5 text-[9px] text-muted-foreground mb-1 border-t border-dashed pt-1 mt-1">
          {data.operatorName && (
            <div className="flex items-center gap-1 truncate">
              <User className="h-2.5 w-2.5 flex-shrink-0" />
              <span className="truncate">{data.operatorName}</span>
            </div>
          )}
          {startTime && (
            <div className="flex items-center gap-1">
              <CalendarClock className="h-2.5 w-2.5 flex-shrink-0" />
              <span>{startTime}{endTime && ` - ${endTime}`}</span>
            </div>
          )}
        </div>
      )}
      
      <div className="flex items-center justify-between text-[10px] text-muted-foreground">
        {data.itemCount !== undefined && (
          <span>{data.completedCount || 0}/{data.itemCount} szt.</span>
        )}
        {data.zlpNumbers && data.zlpNumbers.length > 0 && (
          <span>{data.zlpNumbers.length} ZLP</span>
        )}
        {data.damagedCount !== undefined && data.damagedCount > 0 && (
          <span className="text-red-500 flex items-center gap-0.5">
            <AlertTriangle className="h-3 w-3" />
            {data.damagedCount}
          </span>
        )}
      </div>
      
      {/* Elements section with lazy loading */}
      {data.planId && data.zlpIds && data.zlpIds.length > 0 && data.itemCount !== undefined && data.itemCount > 0 && (
        <ElementsSection
          planId={data.planId}
          zlpIds={data.zlpIds}
          materialColor={data.materialColor}
          operationCode={data.operationCode}
          itemCount={data.itemCount}
        />
      )}
      
      <Handle type="source" position={Position.Bottom} className="!bg-muted-foreground" />
    </div>
  );
}

function StartNode({ data, id }: { data: FlowNode["data"] & { label: string }; id: string }) {
  return (
    <div className="px-4 py-3 rounded-full bg-green-500 text-white shadow-lg">
      <Handle type="source" position={Position.Bottom} className="!bg-white" />
      <div className="text-[8px] font-mono opacity-70 mb-0.5">ID: {id}</div>
      <div className="flex items-center gap-2">
        <CircleDot className="h-4 w-4" />
        <span className="text-sm font-medium">{data.label}</span>
      </div>
      {data.zlpNumbers && (
        <div className="text-xs mt-1 opacity-80">
          {data.zlpNumbers.length} ZLP | {data.itemCount} szt.
        </div>
      )}
    </div>
  );
}

function EndNode({ data, id }: { data: FlowNode["data"] & { label: string; status: FlowNodeStatus }; id: string }) {
  const isCompleted = data.status === "completed";
  return (
    <div className={`px-4 py-3 rounded-full shadow-lg ${isCompleted ? "bg-green-500 text-white" : "bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-300"}`}>
      <Handle type="target" position={Position.Top} className={isCompleted ? "!bg-white" : "!bg-gray-500"} />
      <div className="text-[8px] font-mono opacity-70 mb-0.5">ID: {id}</div>
      <div className="flex items-center gap-2">
        <Flag className="h-4 w-4" />
        <span className="text-sm font-medium">{data.label}</span>
      </div>
    </div>
  );
}

function BufferNode({ data, id }: { data: FlowNode["data"] & { label: string; status: FlowNodeStatus; colorDictionary?: DictionaryItem[] }; id: string }) {
  const colors = statusColors[data.status] || statusColors.ready;
  const isMerge = data.isMergePoint;
  const allColors = data.materialColors || [];
  const activeColors = new Set(data.activeColors || allColors);
  const colorDict = data.colorDictionary || [];
  const isInProgress = data.status === 'in_progress';
  
  return (
    <div className={`px-3 py-2 rounded-md border-2 ${isMerge ? 'border-solid' : 'border-dashed'} ${colors.bg} ${isInProgress ? 'animate-glow-pulse' : colors.border} min-w-[140px] shadow-sm`}>
      <Handle type="target" position={Position.Top} className="!bg-muted-foreground" />
      <div className="text-[8px] text-muted-foreground font-mono mb-0.5 opacity-60">ID: {id}</div>
      <div className="flex items-center gap-2">
        {isMerge ? (
          <GitMerge className={`h-4 w-4 ${colors.text}`} />
        ) : (
          <Box className={`h-4 w-4 ${colors.text}`} />
        )}
        <span className={`text-sm ${colors.text} truncate`}>{data.label}</span>
      </div>
      {isMerge && allColors.length > 0 && (
        <div className="flex flex-wrap gap-1 mt-1">
          {allColors.map((color) => {
            const style = getMaterialColorStyle(color, colorDict);
            const isActive = activeColors.has(color);
            const opacityClass = isActive ? '' : 'opacity-30 grayscale';
            
            return style.hex ? (
              <span
                key={color}
                className={`text-[10px] px-1.5 py-0.5 rounded ${opacityClass}`}
                style={{ backgroundColor: style.hex, color: getTextColor(style.hex) }}
              >
                {color}
              </span>
            ) : (
              <span
                key={color}
                className={`text-[10px] px-1.5 py-0.5 rounded ${style.bg} ${style.text} ${opacityClass}`}
              >
                {color}
              </span>
            );
          })}
        </div>
      )}
      {data.bufferUsed !== undefined && data.bufferCapacity !== undefined && (
        <div className="text-xs text-muted-foreground mt-1">
          {data.bufferUsed}/{data.bufferCapacity}
        </div>
      )}
      <Handle type="source" position={Position.Bottom} className="!bg-muted-foreground" />
    </div>
  );
}

function MergeNode({ data, id }: { data: FlowNode["data"] & { label: string; status: FlowNodeStatus; colorDictionary?: DictionaryItem[] }; id: string }) {
  const mergeData = data.mergePointData;
  const materialColors = data.materialColors || [];
  const colorDict = data.colorDictionary || [];
  
  const getMergeStatusColor = () => {
    if (!mergeData) return statusColors.pending;
    switch (mergeData.status) {
      case 'released': return statusColors.completed;
      case 'ready': return statusColors.ready;
      case 'partial': return statusColors.in_progress;
      default: return statusColors.pending;
    }
  };
  
  const colors = getMergeStatusColor();
  const arrivedCount = mergeData?.arrivedFamilies?.length || 0;
  const requiredCount = mergeData?.requiredFamilies?.length || 0;
  const progressPercent = requiredCount > 0 ? (arrivedCount / requiredCount) * 100 : 0;
  
  return (
    <div className={`px-3 py-2 rounded-lg border-2 border-solid ${colors.bg} ${colors.border} min-w-[180px] shadow-md`}>
      <Handle type="target" position={Position.Top} className="!bg-muted-foreground !w-3 !h-3" />
      
      <div className="text-[8px] text-muted-foreground font-mono mb-0.5 opacity-60">ID: {id}</div>
      <div className="flex items-center gap-2 mb-1">
        <GitMerge className={`h-5 w-5 ${colors.text}`} />
        <span className={`text-sm font-semibold ${colors.text}`}>{data.label}</span>
      </div>
      
      {mergeData && (
        <>
          <div className="text-[10px] text-muted-foreground mb-1">
            Etap: {mergeData.mergeStage}
          </div>
          
          <div className="flex items-center gap-2 mb-1">
            <Progress value={progressPercent} className="h-1.5 flex-1" />
            <span className={`text-[11px] font-medium ${colors.text}`}>
              {arrivedCount}/{requiredCount}
            </span>
          </div>
          
          <div className="flex flex-wrap gap-1 mt-1">
            {mergeData.requiredFamilies.map((family) => {
              const arrived = mergeData.arrivedFamilies?.includes(family);
              const style = getMaterialColorStyle(family, colorDict);
              const arrivedClasses = style.hex 
                ? '' 
                : `${style.bg} ${style.text}`;
              return style.hex && arrived ? (
                <span
                  key={family}
                  className="text-[9px] px-1.5 py-0.5 rounded flex items-center gap-0.5"
                  style={{ backgroundColor: style.hex, color: getTextColor(style.hex) }}
                >
                  <CheckCircle className="h-2.5 w-2.5" />
                  {familyLabels[family as ComponentFamily] || family}
                </span>
              ) : (
                <span
                  key={family}
                  className={`text-[9px] px-1.5 py-0.5 rounded flex items-center gap-0.5
                    ${arrived ? arrivedClasses : 'bg-gray-300 dark:bg-gray-600 text-gray-600 dark:text-gray-300 opacity-60'}`}
                >
                  {arrived && <CheckCircle className="h-2.5 w-2.5" />}
                  {familyLabels[family as ComponentFamily] || family}
                </span>
              );
            })}
          </div>
          
          {mergeData.manualRelease && mergeData.status === 'ready' && (
            <div className="mt-1 text-[10px] text-amber-600 dark:text-amber-400 flex items-center gap-1">
              <AlertTriangle className="h-3 w-3" />
              Wymaga zatwierdzenia
            </div>
          )}
          
          {mergeData.status === 'released' && (
            <div className="mt-1 text-[10px] text-green-600 dark:text-green-400 flex items-center gap-1">
              <CheckCircle className="h-3 w-3" />
              Zwolniono
            </div>
          )}
        </>
      )}
      
      {!mergeData && materialColors.length > 0 && (
        <div className="flex flex-wrap gap-1 mt-1">
          {materialColors.map((color) => {
            const style = getMaterialColorStyle(color, colorDict);
            return style.hex ? (
              <span
                key={color}
                className="text-[10px] px-1.5 py-0.5 rounded"
                style={{ backgroundColor: style.hex, color: getTextColor(style.hex) }}
              >
                {color}
              </span>
            ) : (
              <span
                key={color}
                className={`text-[10px] px-1.5 py-0.5 rounded ${style.bg} ${style.text}`}
              >
                {color}
              </span>
            );
          })}
        </div>
      )}
      
      <Handle type="source" position={Position.Bottom} className="!bg-muted-foreground !w-3 !h-3" />
    </div>
  );
}

function WarehousePackedNode({ data }: { data: FlowNode["data"] & { label: string; status: FlowNodeStatus } }) {
  const colors = statusColors[data.status] || statusColors.ready;
  const items = data.packedProductsItems || [];
  const allPicked = items.every(i => i.isPicked);
  const pickedCount = items.filter(i => i.isPicked).length;
  
  return (
    <div className={`px-3 py-2 rounded-lg border-2 ${colors.bg} ${colors.border} min-w-[180px] shadow-md bg-gradient-to-br from-emerald-50 to-emerald-100 dark:from-emerald-900/30 dark:to-emerald-800/30`}>
      <Handle type="target" position={Position.Top} className="!bg-emerald-500" />
      
      <div className="flex items-center gap-2 mb-1">
        <Package className="h-5 w-5 text-emerald-600 dark:text-emerald-400" />
        <span className="text-sm font-semibold text-emerald-800 dark:text-emerald-200">{data.label}</span>
      </div>
      
      {data.warehouseDocumentNumber && (
        <div className="text-[10px] text-emerald-700 dark:text-emerald-300 mb-1">
          Dok: {data.warehouseDocumentNumber}
        </div>
      )}
      
      <div className="flex items-center gap-2 mb-1">
        <Progress 
          value={items.length > 0 ? (pickedCount / items.length) * 100 : 0} 
          className="h-1.5 flex-1 bg-emerald-200 dark:bg-emerald-700" 
        />
        <span className="text-[11px] font-medium text-emerald-700 dark:text-emerald-300">
          {pickedCount}/{items.length}
        </span>
      </div>
      
      <div className="text-[10px] text-emerald-600 dark:text-emerald-400">
        {data.packedProductsCount || 0} szt produktów spakowanych
      </div>
      
      {items.length > 0 && items.length <= 5 && (
        <div className="mt-1 space-y-0.5 max-h-[60px] overflow-y-auto">
          {items.slice(0, 5).map((item, idx) => (
            <div 
              key={idx} 
              className={`text-[9px] px-1 py-0.5 rounded flex items-center gap-1 ${
                item.isPicked 
                  ? 'bg-emerald-200 dark:bg-emerald-700 line-through opacity-60' 
                  : 'bg-white dark:bg-emerald-800'
              }`}
            >
              {item.isPicked && <CheckCircle className="h-2.5 w-2.5 text-emerald-600" />}
              <span className="truncate">{item.productName}</span>
              <span className="text-muted-foreground">x{item.quantity}</span>
            </div>
          ))}
        </div>
      )}
      
      {allPicked && (
        <div className="mt-1 text-[10px] text-emerald-600 dark:text-emerald-400 flex items-center gap-1">
          <CheckCircle className="h-3 w-3" />
          Wszystko pobrane
        </div>
      )}
      
      <Handle type="source" position={Position.Bottom} className="!bg-emerald-500" />
    </div>
  );
}

function BufferFormatkiNode({ data }: { data: FlowNode["data"] & { label: string; status: FlowNodeStatus } }) {
  const colors = statusColors[data.status] || statusColors.ready;
  const items = data.reservationItems || [];
  const totalReserved = data.reservedQuantity || 0;
  const totalConsumed = data.consumedQuantity || 0;
  const progressPercent = totalReserved > 0 ? (totalConsumed / totalReserved) * 100 : 0;
  const hasWzForm = !!data.warehouseDocumentNumber;
  
  return (
    <div className={`px-3 py-2 rounded-lg border-2 ${colors.bg} ${colors.border} min-w-[180px] shadow-md bg-gradient-to-br from-amber-50 to-amber-100 dark:from-amber-900/30 dark:to-amber-800/30`}>
      <Handle type="target" position={Position.Top} className="!bg-amber-500" />
      
      <div className="flex items-center gap-2 mb-1">
        <Layers className="h-5 w-5 text-amber-600 dark:text-amber-400" />
        <span className="text-sm font-semibold text-amber-800 dark:text-amber-200">{data.label}</span>
      </div>
      
      {hasWzForm && (
        <div className="text-[10px] bg-amber-200/60 dark:bg-amber-700/60 rounded px-1.5 py-0.5 mb-1 flex items-center gap-1">
          <FileText className="h-3 w-3 text-amber-700 dark:text-amber-300" />
          <span className="font-medium text-amber-800 dark:text-amber-200">WZ-FORM: {data.warehouseDocumentNumber}</span>
          <Badge 
            variant="outline" 
            className="text-[8px] px-1 py-0 h-3.5 border-amber-500 text-amber-700 dark:text-amber-300"
          >
            {data.warehouseDocumentStatus === 'confirmed' ? 'Potw.' : 
             data.warehouseDocumentStatus === 'completed' ? 'Zakończone' : 'Robocze'}
          </Badge>
        </div>
      )}
      
      <div className="text-[10px] text-amber-700 dark:text-amber-300 mb-1">
        {data.reservationsCount || 0} rezerwacji
      </div>
      
      <div className="flex items-center gap-2 mb-1">
        <Progress 
          value={progressPercent} 
          className="h-1.5 flex-1 bg-amber-200 dark:bg-amber-700" 
        />
        <span className="text-[11px] font-medium text-amber-700 dark:text-amber-300">
          {Math.round(totalConsumed)}/{Math.round(totalReserved)}
        </span>
      </div>
      
      <div className="text-[10px] text-amber-600 dark:text-amber-400">
        {Math.round(totalReserved)} szt formatek zarezerwowanych
      </div>
      
      {items.length > 0 && items.length <= 5 && (
        <div className="mt-1 space-y-0.5 max-h-[60px] overflow-y-auto">
          {items.slice(0, 5).map((item, idx) => (
            <div 
              key={idx} 
              className={`text-[9px] px-1 py-0.5 rounded flex items-center gap-1 ${
                item.status === 'CONSUMED' 
                  ? 'bg-amber-200 dark:bg-amber-700 line-through opacity-60' 
                  : 'bg-white dark:bg-amber-800'
              }`}
            >
              {item.status === 'CONSUMED' && <CheckCircle className="h-2.5 w-2.5 text-amber-600" />}
              <span className="truncate">{item.productName}</span>
              <span className="text-muted-foreground">
                {Math.round(item.quantityConsumed)}/{Math.round(item.quantityReserved)}
              </span>
            </div>
          ))}
        </div>
      )}
      
      {progressPercent >= 100 && (
        <div className="mt-1 text-[10px] text-amber-600 dark:text-amber-400 flex items-center gap-1">
          <CheckCircle className="h-3 w-3" />
          Wszystko pobrane
        </div>
      )}
      
      <Handle type="source" position={Position.Bottom} className="!bg-amber-500" />
    </div>
  );
}

const nodeTypes = {
  operation: CustomNode,
  merge: MergeNode,
  split: CustomNode,
  buffer: BufferNode,
  transport: CustomNode,
  start: StartNode,
  end: EndNode,
  warehouse_packed: WarehousePackedNode,
  buffer_formatki: BufferFormatkiNode,
};

interface RoutingRule {
  id: number;
  planId: number | null;
  planNumber?: string;
  planName?: string;
  productType: string | null;
  completionStatus: 'incomplete' | 'complete' | 'any';
  destination: string;
  destinationName?: string;
  priority: number;
  isActive: boolean;
  name?: string;
  description?: string;
}

export default function ProductionPlanFlowTree() {
  const [, params] = useRoute("/production/plans/:id/flow-tree");
  const [, navigate] = useLocation();
  const planId = params?.id ? parseInt(params.id) : null;
  const { toast } = useToast();

  const [operatorDialogOpen, setOperatorDialogOpen] = useState(false);
  const [selectedNode, setSelectedNode] = useState<SelectedNodeInfo | null>(null);
  const [selectedOperatorIds, setSelectedOperatorIds] = useState<number[]>([]);
  const [primaryOperatorId, setPrimaryOperatorId] = useState<number | null>(null);
  const [routingDialogOpen, setRoutingDialogOpen] = useState(false);
  const [editingRule, setEditingRule] = useState<RoutingRule | null>(null);
  const [editForm, setEditForm] = useState({
    name: '',
    completionStatus: 'incomplete' as 'incomplete' | 'complete' | 'any',
    destination: 'packing',
    priority: 0,
    isActive: true,
  });
  const [zlpProgressDialogOpen, setZlpProgressDialogOpen] = useState(false);
  const [zlpSessionId, setZlpSessionId] = useState<string | null>(null);
  const [palletTreeOpen, setPalletTreeOpen] = useState(false);
  const [addPalletDialog, setAddPalletDialog] = useState<{ orderId: number; flowCode: string } | null>(null);
  const [newPalletCarrierId, setNewPalletCarrierId] = useState<string>("");
  const [showNewCarrierForm, setShowNewCarrierForm] = useState(false);
  const [newCarrierName, setNewCarrierName] = useState("");
  const [newCarrierCode, setNewCarrierCode] = useState("");
  const [newCarrierGroupId, setNewCarrierGroupId] = useState<string>("");

  const { data: flowGraph, isLoading, error } = useQuery<ProductionPlanFlowGraph>({
    queryKey: [`/api/production/planning/plans/${planId}/flow`],
    enabled: !!planId,
  });

  const { data: colorDictionary = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });

  const { data: carriersData = [], refetch: refetchCarriers } = useQuery<{ id: number; code: string; name: string; isOccupied?: boolean; occupiedByZlp?: string }[]>({
    queryKey: ["/api/production/carriers"],
  });

  const { data: carrierGroupsData = [] } = useQuery<{ id: number; code: string; name: string }[]>({
    queryKey: ["/api/production/carriers/groups"],
  });

  const carrierOptions = useMemo(() => {
    const options = [{ value: "none", label: "Brak nośnika" }];
    carriersData.forEach((c) => {
      const occupied = c.isOccupied ? ` [zajęty: ${c.occupiedByZlp}]` : '';
      options.push({ value: c.id.toString(), label: `${c.name} (${c.code})${occupied}` });
    });
    return options;
  }, [carriersData]);

  const nextCarrierNumber = useMemo(() => {
    let maxNum = 0;
    carriersData.forEach((c) => {
      const match = c.code.match(/PAL-(\d+)/i);
      if (match) {
        const num = parseInt(match[1], 10);
        if (num > maxNum) maxNum = num;
      }
    });
    return maxNum + 1;
  }, [carriersData]);

  const { data: operatorsData } = useQuery<{ operators: ProductionOperator[] }>({
    queryKey: ["/api/production/operators", { is_active: "true" }],
  });
  const operators = operatorsData?.operators || [];
  
  const { data: routingRulesData } = useQuery<{ rules: RoutingRule[] }>({
    queryKey: ["/api/production/planning/routing-rules", { planId: planId?.toString() || "all" }],
    enabled: !!planId,
  });
  const routingRules = routingRulesData?.rules || [];

  const generateOrdersMutation = useMutation({
    mutationFn: async () => {
      const response = await apiRequest(
        "POST",
        `/api/production/planning/plans/${planId}/generate-orders`
      );
      return await response.json();
    },
    onSuccess: (data: any) => {
      if (data.sessionId) {
        setZlpSessionId(data.sessionId);
        setZlpProgressDialogOpen(true);
      }
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd generowania ZLP",
        description: error.message,
        variant: "destructive",
      });
    },
  });

  const handleZlpGenerationComplete = useCallback((result: any) => {
    queryClient.invalidateQueries({
      queryKey: [`/api/production/planning/plans/${planId}/flow`]
    });
    
    if (result.success) {
      toast({
        title: "Zlecenia wygenerowane",
        description: `Utworzono ${result.workOrdersCount} zleceń ZLP`,
      });
    }
  }, [planId, toast]);

  const updateOperatorsMutation = useMutation({
    mutationFn: ({ workOrderId, operatorIds, primaryOperatorId }: { 
      workOrderId: number; 
      operatorIds: number[];
      primaryOperatorId: number | null;
    }) =>
      apiRequest("PUT", `/api/production/work-orders/${workOrderId}/operators`, { 
        operatorIds,
        primaryOperatorId,
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/flow`] });
      setOperatorDialogOpen(false);
      setSelectedNode(null);
      toast({ title: "Operatorzy zostali przypisani" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się przypisać operatorów",
        variant: "destructive",
      });
    },
  });

  const addPalletMutation = useMutation({
    mutationFn: ({ orderId, flowCode, carrierId }: { orderId: number; flowCode: string; carrierId?: number }) =>
      apiRequest("POST", `/api/production/orders/${orderId}/pallets`, { flowCode, carrierId }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/planning/plans/${planId}/flow`] });
      refetchCarriers();
      setAddPalletDialog(null);
      setNewPalletCarrierId("");
      setShowNewCarrierForm(false);
      toast({ title: "Paleta została dodana" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się dodać palety",
        variant: "destructive",
      });
    },
  });

  const createCarrierMutation = useMutation({
    mutationFn: async (data: { name: string; code: string; carrierGroupId: number }) => {
      const response = await apiRequest("POST", "/api/production/carriers", data);
      return response.json();
    },
    onSuccess: (newCarrier: { id: number }) => {
      refetchCarriers();
      setNewPalletCarrierId(newCarrier.id.toString());
      setShowNewCarrierForm(false);
      setNewCarrierName("");
      setNewCarrierCode("");
      setNewCarrierGroupId("");
      toast({ title: "Nośnik został utworzony" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się utworzyć nośnika",
        variant: "destructive",
      });
    },
  });

  const handleAddPallet = useCallback((orderId: number, flowCode: string) => {
    setAddPalletDialog({ orderId, flowCode });
  }, []);

  const handleNodeClick = useCallback((event: any, node: Node) => {
    const data = node.data as any;
    const workOrderIds = data?.workOrderIds as number[] | undefined;
    if (node.type === "operation" && workOrderIds && workOrderIds.length > 0) {
      setSelectedNode({
        workOrderIds,
        selectedWorkOrderId: workOrderIds[0],
        label: (data.label as string) || "Operacja",
        currentOperatorId: data.operatorId as number | null | undefined,
        currentOperatorName: data.operatorName as string | null | undefined,
      });
      
      // Ustaw obecnych operatorów z assignedOperators (junction table) lub fallback do legacy
      const assignedOperators = data.assignedOperators as Array<{ operatorId: number; operatorName: string; isPrimary: boolean }> | undefined;
      if (assignedOperators && assignedOperators.length > 0) {
        setSelectedOperatorIds(assignedOperators.map(op => op.operatorId));
        const primaryOp = assignedOperators.find(op => op.isPrimary);
        setPrimaryOperatorId(primaryOp?.operatorId || assignedOperators[0]?.operatorId || null);
      } else {
        // Fallback do legacy pola operatorId
        const currentOpId = data.operatorId as number | null | undefined;
        if (currentOpId) {
          setSelectedOperatorIds([currentOpId]);
          setPrimaryOperatorId(currentOpId);
        } else {
          setSelectedOperatorIds([]);
          setPrimaryOperatorId(null);
        }
      }
      setOperatorDialogOpen(true);
    }
  }, []);

  const handleOperatorToggle = (operatorId: number, checked: boolean) => {
    if (checked) {
      setSelectedOperatorIds(prev => [...prev, operatorId]);
      // Automatycznie ustaw pierwszego dodanego jako głównego
      if (selectedOperatorIds.length === 0) {
        setPrimaryOperatorId(operatorId);
      }
    } else {
      setSelectedOperatorIds(prev => prev.filter(id => id !== operatorId));
      // Jeśli usuwamy głównego, ustaw następnego jako głównego
      if (primaryOperatorId === operatorId) {
        const remaining = selectedOperatorIds.filter(id => id !== operatorId);
        setPrimaryOperatorId(remaining.length > 0 ? remaining[0] : null);
      }
    }
  };

  const handlePrimaryToggle = (operatorId: number) => {
    if (selectedOperatorIds.includes(operatorId)) {
      setPrimaryOperatorId(operatorId);
    }
  };

  const handleSaveOperators = () => {
    if (!selectedNode) return;
    updateOperatorsMutation.mutate({ 
      workOrderId: selectedNode.selectedWorkOrderId, 
      operatorIds: selectedOperatorIds,
      primaryOperatorId,
    });
  };

  const { nodes: layoutedNodes, edges: layoutedEdges } = useMemo(() => {
    if (!flowGraph || !flowGraph.nodes || !flowGraph.nodes.length) {
      return { nodes: [], edges: [] };
    }

    // Split nodes after split point into separate nodes per flow
    const { nodes: splitNodes, edges: splitEdges } = splitNodesAfterSplitPoint(flowGraph);

    const rfNodes: Node[] = splitNodes.map((node) => ({
      id: node.id,
      type: node.type,
      position: { x: 0, y: 0 },
      data: {
        ...node.data,
        label: node.label,
        status: node.status,
        nodeType: node.type,
        colorDictionary,
        planId: planId,
        onAddPallet: handleAddPallet,
      },
    }));

    const rfEdges: Edge[] = splitEdges.map((edge) => ({
      id: edge.id,
      source: edge.source,
      target: edge.target,
      animated: edge.animated,
      markerEnd: { type: MarkerType.ArrowClosed },
      style: {
        stroke: edge.animated ? "#f59e0b" : "#6b7280",
        strokeWidth: 2,
      },
    }));

    return getLayoutedElements(rfNodes, rfEdges, "TB");
  }, [flowGraph, colorDictionary, planId, handleAddPallet]);

  const [nodes, setNodes, onNodesChange] = useNodesState(layoutedNodes);
  const [edges, setEdges, onEdgesChange] = useEdgesState(layoutedEdges);

  useMemo(() => {
    if (layoutedNodes.length > 0) {
      setNodes(layoutedNodes);
      setEdges(layoutedEdges);
    }
  }, [layoutedNodes, layoutedEdges, setNodes, setEdges]);

  if (!planId) {
    return (
      <div className="flex items-center justify-center h-screen">
        <p className="text-muted-foreground">Nieprawidłowy ID planu</p>
      </div>
    );
  }

  if (isLoading) {
    return (
      <div className="flex items-center justify-center h-screen">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </div>
    );
  }

  if (error || !flowGraph || !flowGraph.metadata) {
    return (
      <div className="flex flex-col items-center justify-center h-screen gap-4">
        <p className="text-destructive">Nie udało się wczytać danych przepływu</p>
        <Button variant="outline" onClick={() => navigate(`/production/plans/${planId}`)}>
          <ArrowLeft className="h-4 w-4 mr-2" />
          Powrót do planu
        </Button>
      </div>
    );
  }

  return (
    <div className="flex flex-col h-screen">
      <div className="flex-shrink-0 border-b bg-background p-4">
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-4">
            <Button variant="ghost" size="sm" asChild>
              <Link href={`/production/plans/${planId}`}>
                <ArrowLeft className="h-4 w-4 mr-2" />
                Powrót
              </Link>
            </Button>
            <div>
              <h1 className="text-xl font-semibold" data-testid="text-flow-tree-title">
                Drzewo przepływu: {flowGraph.planName}
              </h1>
              <div className="flex items-center gap-4 mt-1 text-sm text-muted-foreground">
                <span>{flowGraph.metadata.totalZlps} ZLP</span>
                <span>{flowGraph.metadata.totalItems} elementów</span>
                <span className="text-green-600">
                  <CheckCircle className="h-3 w-3 inline mr-1" />
                  {flowGraph.metadata.completedItems} ukończonych
                </span>
                {flowGraph.metadata.damagedItems > 0 && (
                  <span className="text-red-600">
                    <AlertTriangle className="h-3 w-3 inline mr-1" />
                    {flowGraph.metadata.damagedItems} uszkodzonych
                  </span>
                )}
              </div>
            </div>
          </div>
          
          <div className="flex items-center gap-2">
            {planId && (
              <Button
                onClick={() => generateOrdersMutation.mutate()}
                disabled={generateOrdersMutation.isPending}
                size="sm"
                variant="default"
                data-testid="button-regenerate-zlp"
                className="gap-1.5"
              >
                {generateOrdersMutation.isPending ? (
                  <>
                    <Loader2 className="h-4 w-4 animate-spin" />
                    <span className="hidden sm:inline">Generowanie...</span>
                  </>
                ) : (
                  <>
                    <Factory className="h-4 w-4" />
                    <span className="hidden sm:inline">Regeneruj ZLP</span>
                  </>
                )}
              </Button>
            )}
            {planId && (
              <Button
                variant="outline"
                size="sm"
                onClick={() => setRoutingDialogOpen(true)}
                data-testid="button-routing-rules"
              >
                <Settings className="h-4 w-4 mr-1.5" />
                Reguły routingu
                {routingRules.length > 0 && (
                  <Badge variant="secondary" className="ml-1.5 h-5 px-1.5 text-[10px]">
                    {routingRules.length}
                  </Badge>
                )}
              </Button>
            )}
            {planId && (
              <Button
                variant="outline"
                size="sm"
                asChild
                data-testid="button-routing-settings-page"
              >
                <Link href={`/production/plans/${planId}/routing`}>
                  <Route className="h-4 w-4 mr-1.5" />
                  <span className="hidden sm:inline">Ustawienia routingu</span>
                </Link>
              </Button>
            )}
            {(flowGraph.metadata.componentFamilies || []).map((family) => (
              <Badge key={family} variant="outline" className={`${familyColors[family]} text-white border-0`}>
                {familyLabels[family]}
              </Badge>
            ))}
          </div>
        </div>
      </div>

      <div className="flex-1">
        <ReactFlow
          nodes={nodes}
          edges={edges}
          onNodesChange={onNodesChange}
          onEdgesChange={onEdgesChange}
          onNodeClick={handleNodeClick}
          nodeTypes={nodeTypes}
          fitView
          fitViewOptions={{ padding: 0.2 }}
          minZoom={0.1}
          maxZoom={2}
          attributionPosition="bottom-left"
        >
          <Background variant={BackgroundVariant.Dots} gap={20} size={1} />
          <Controls position="bottom-right" />
          <MiniMap
            nodeStrokeWidth={3}
            zoomable
            pannable
            position="top-right"
            className="!bg-background border rounded-lg"
          />
        </ReactFlow>
      </div>

      <div className="flex-shrink-0 border-t bg-background p-2">
        <div className="flex items-center justify-center gap-6 text-xs text-muted-foreground">
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full bg-green-500" />
            <span>Zakończone</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full bg-yellow-500" />
            <span>W trakcie</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full bg-blue-500" />
            <span>Gotowe</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full bg-gray-400" />
            <span>Oczekuje</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 border-2 border-dashed border-gray-400 rounded" />
            <span>Bufor</span>
          </div>
        </div>
      </div>

      <Dialog open={operatorDialogOpen} onOpenChange={setOperatorDialogOpen}>
        <DialogContent className="max-w-md" data-testid="dialog-assign-operators">
          <DialogHeader>
            <DialogTitle>Przypisz operatorów</DialogTitle>
            <DialogDescription>
              {selectedNode?.label}
              {selectedNode && selectedNode.workOrderIds.length > 1 && (
                <span className="block mt-1 text-xs">
                  Ta operacja zawiera {selectedNode.workOrderIds.length} zleceń - operatorzy zostaną przypisani do pierwszego.
                </span>
              )}
            </DialogDescription>
          </DialogHeader>
          <div className="py-4">
            <p className="text-sm text-muted-foreground mb-3">
              Wybierz operatorów (kliknij gwiazdkę aby oznaczyć głównego):
            </p>
            <ScrollArea className="h-[250px] border rounded-md p-2">
              <div className="space-y-2">
                {operators.map((op) => {
                  const isSelected = selectedOperatorIds.includes(op.id);
                  const isPrimary = primaryOperatorId === op.id;
                  return (
                    <div
                      key={op.id}
                      className={`flex items-center gap-3 p-2 rounded-md hover-elevate ${
                        isSelected ? "bg-accent/50" : ""
                      }`}
                      data-testid={`operator-row-${op.id}`}
                    >
                      <Checkbox
                        id={`operator-${op.id}`}
                        checked={isSelected}
                        onCheckedChange={(checked) => handleOperatorToggle(op.id, !!checked)}
                        data-testid={`checkbox-operator-${op.id}`}
                      />
                      <Label
                        htmlFor={`operator-${op.id}`}
                        className="flex-1 cursor-pointer text-sm"
                      >
                        {op.full_name} {op.short_code && `(${op.short_code})`}
                      </Label>
                      {isSelected && (
                        <Button
                          type="button"
                          variant="ghost"
                          size="icon"
                          className={`h-7 w-7 ${isPrimary ? "text-yellow-500" : "text-muted-foreground"}`}
                          onClick={() => handlePrimaryToggle(op.id)}
                          title={isPrimary ? "Główny operator" : "Ustaw jako głównego"}
                          data-testid={`button-primary-${op.id}`}
                        >
                          <Star className={`h-4 w-4 ${isPrimary ? "fill-current" : ""}`} />
                        </Button>
                      )}
                    </div>
                  );
                })}
              </div>
            </ScrollArea>
            {selectedOperatorIds.length > 0 && (
              <p className="text-xs text-muted-foreground mt-2">
                Wybrano: {selectedOperatorIds.length} operator{selectedOperatorIds.length === 1 ? "a" : "ów"}
                {primaryOperatorId && (
                  <>, główny: {operators.find(o => o.id === primaryOperatorId)?.full_name}</>
                )}
              </p>
            )}
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setOperatorDialogOpen(false)}
              data-testid="button-cancel-operators"
            >
              Anuluj
            </Button>
            <Button
              onClick={handleSaveOperators}
              disabled={updateOperatorsMutation.isPending}
              data-testid="button-save-operators"
            >
              {updateOperatorsMutation.isPending ? (
                <>
                  <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  Zapisywanie...
                </>
              ) : (
                "Zapisz"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={routingDialogOpen} onOpenChange={setRoutingDialogOpen}>
        <DialogContent className="max-w-2xl max-h-[80vh]" data-testid="dialog-routing-rules">
          <DialogHeader>
            <DialogTitle>Reguły routingu magazynowego</DialogTitle>
            <DialogDescription>
              Skonfiguruj gdzie produkty z magazynu powinny trafiać w zależności od ich statusu kompletności.
            </DialogDescription>
          </DialogHeader>
          <ScrollArea className="max-h-[50vh] pr-4">
            <div className="space-y-3">
              {routingRules.length === 0 ? (
                <div className="text-center py-8 text-muted-foreground">
                  <Settings className="h-12 w-12 mx-auto mb-2 opacity-30" />
                  <p>Brak skonfigurowanych reguł routingu.</p>
                  <p className="text-sm">Produkty będą kierowane domyślnie do pakowania.</p>
                </div>
              ) : (
                routingRules.map((rule) => (
                  <Card key={rule.id} className={`${rule.isActive ? '' : 'opacity-50'}`}>
                    <CardContent className="p-3">
                      <div className="flex items-start justify-between gap-2">
                        <div className="flex-1 min-w-0">
                          <div className="flex items-center gap-2 mb-1">
                            <Badge 
                              variant={rule.isActive ? 'default' : 'secondary'}
                              className="text-[10px] h-4"
                            >
                              {rule.isActive ? 'Aktywna' : 'Nieaktywna'}
                            </Badge>
                            <Badge variant="outline" className="text-[10px] h-4">
                              Priorytet: {rule.priority}
                            </Badge>
                            {rule.planId ? (
                              <Badge variant="outline" className="text-[10px] h-4 bg-blue-50 dark:bg-blue-900/30">
                                Plan: {rule.planNumber || rule.planId}
                              </Badge>
                            ) : (
                              <Badge variant="outline" className="text-[10px] h-4 bg-green-50 dark:bg-green-900/30">
                                Globalna
                              </Badge>
                            )}
                          </div>
                          <p className="font-medium text-sm truncate">
                            {rule.name || `Reguła #${rule.id}`}
                          </p>
                          <div className="flex flex-wrap items-center gap-x-3 gap-y-1 text-xs text-muted-foreground mt-1">
                            <span>
                              Status: {
                                rule.completionStatus === 'incomplete' ? 'Niekompletne' :
                                rule.completionStatus === 'complete' ? 'Kompletne' : 'Dowolny'
                              }
                            </span>
                            <span>→</span>
                            <span className="font-medium text-foreground">
                              {rule.destination === 'packing' ? 'Pakowanie' :
                               rule.destination === 'shipping_buffer' ? 'Bufor wysyłkowy' :
                               rule.destination === 'assembly' ? 'Montaż' :
                               rule.destinationName || rule.destination}
                            </span>
                          </div>
                          {rule.description && (
                            <p className="text-xs text-muted-foreground mt-1 truncate">
                              {rule.description}
                            </p>
                          )}
                        </div>
                        <div className="flex gap-1">
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-7 w-7"
                            onClick={() => {
                              setEditingRule(rule);
                              setEditForm({
                                name: rule.name || '',
                                completionStatus: rule.completionStatus,
                                destination: rule.destination,
                                priority: rule.priority,
                                isActive: rule.isActive,
                              });
                            }}
                            data-testid={`button-edit-rule-${rule.id}`}
                          >
                            <Pencil className="h-4 w-4" />
                          </Button>
                          <Button
                            variant="ghost"
                            size="icon"
                            className="h-7 w-7 text-destructive hover:text-destructive"
                            onClick={() => {
                              if (confirm('Czy na pewno usunąć tę regułę?')) {
                                apiRequest("DELETE", `/api/production/planning/routing-rules/${rule.id}`)
                                  .then(() => {
                                    queryClient.invalidateQueries({ 
                                      queryKey: ["/api/production/planning/routing-rules"] 
                                    });
                                    queryClient.invalidateQueries({ 
                                      queryKey: [`/api/production/planning/plans/${planId}/flow`] 
                                    });
                                    toast({ title: "Reguła usunięta" });
                                  })
                                  .catch(() => {
                                    toast({ 
                                      title: "Błąd", 
                                      description: "Nie udało się usunąć reguły",
                                      variant: "destructive" 
                                    });
                                  });
                              }
                            }}
                            data-testid={`button-delete-rule-${rule.id}`}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </div>
                      </div>
                    </CardContent>
                  </Card>
                ))
              )}
            </div>
          </ScrollArea>
          <DialogFooter className="flex-col sm:flex-row gap-2">
            <Button
              variant="outline"
              className="w-full sm:w-auto"
              onClick={() => {
                const newRule = {
                  planId: planId,
                  completionStatus: 'incomplete' as const,
                  destination: 'packing',
                  priority: 0,
                  isActive: true,
                  name: `Nowa reguła dla planu ${flowGraph?.planName || planId}`,
                };
                apiRequest("POST", "/api/production/planning/routing-rules", newRule)
                  .then(() => {
                    queryClient.invalidateQueries({ 
                      queryKey: ["/api/production/planning/routing-rules"] 
                    });
                    queryClient.invalidateQueries({ 
                      queryKey: [`/api/production/planning/plans/${planId}/flow`] 
                    });
                    toast({ title: "Reguła dodana" });
                  })
                  .catch(() => {
                    toast({ 
                      title: "Błąd", 
                      description: "Nie udało się dodać reguły",
                      variant: "destructive" 
                    });
                  });
              }}
              data-testid="button-add-rule"
            >
              <Plus className="h-4 w-4 mr-1.5" />
              Dodaj regułę dla tego planu
            </Button>
            <Button
              onClick={() => setRoutingDialogOpen(false)}
              data-testid="button-close-routing"
            >
              Zamknij
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <Dialog open={!!editingRule} onOpenChange={(open) => !open && setEditingRule(null)}>
        <DialogContent className="max-w-md" data-testid="dialog-edit-rule">
          <DialogHeader>
            <DialogTitle>Edytuj regułę routingu</DialogTitle>
            <DialogDescription>
              Zmień parametry reguły routingu magazynowego.
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4">
            <div className="space-y-2">
              <Label htmlFor="rule-name">Nazwa reguły</Label>
              <Input
                id="rule-name"
                value={editForm.name}
                onChange={(e) => setEditForm(prev => ({ ...prev, name: e.target.value }))}
                placeholder="Nazwa reguły"
                data-testid="input-rule-name"
              />
            </div>
            
            <div className="space-y-2">
              <Label>Status kompletności</Label>
              <Select
                value={editForm.completionStatus}
                onValueChange={(value: 'incomplete' | 'complete' | 'any') => 
                  setEditForm(prev => ({ ...prev, completionStatus: value }))
                }
              >
                <SelectTrigger data-testid="select-completion-status">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="incomplete">Niekompletne</SelectItem>
                  <SelectItem value="complete">Kompletne</SelectItem>
                  <SelectItem value="any">Dowolny</SelectItem>
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <Label>Destynacja</Label>
              <Select
                value={editForm.destination}
                onValueChange={(value) => setEditForm(prev => ({ ...prev, destination: value }))}
              >
                <SelectTrigger data-testid="select-destination">
                  <SelectValue />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="packing">Pakowanie</SelectItem>
                  <SelectItem value="shipping_buffer">Bufor wysyłkowy</SelectItem>
                  <SelectItem value="assembly">Montaż</SelectItem>
                </SelectContent>
              </Select>
            </div>
            
            <div className="space-y-2">
              <Label htmlFor="rule-priority">Priorytet</Label>
              <Input
                id="rule-priority"
                type="number"
                value={editForm.priority}
                onChange={(e) => setEditForm(prev => ({ ...prev, priority: parseInt(e.target.value) || 0 }))}
                data-testid="input-rule-priority"
              />
            </div>
            
            <div className="flex items-center gap-2">
              <Switch
                id="rule-active"
                checked={editForm.isActive}
                onCheckedChange={(checked) => setEditForm(prev => ({ ...prev, isActive: checked }))}
                data-testid="switch-rule-active"
              />
              <Label htmlFor="rule-active">Aktywna</Label>
            </div>
          </div>
          <DialogFooter>
            <Button
              variant="outline"
              onClick={() => setEditingRule(null)}
              data-testid="button-cancel-edit"
            >
              Anuluj
            </Button>
            <Button
              onClick={() => {
                if (!editingRule) return;
                apiRequest("PATCH", `/api/production/planning/routing-rules/${editingRule.id}`, editForm)
                  .then(() => {
                    queryClient.invalidateQueries({ 
                      queryKey: ["/api/production/planning/routing-rules"] 
                    });
                    queryClient.invalidateQueries({ 
                      queryKey: [`/api/production/planning/plans/${planId}/flow`] 
                    });
                    setEditingRule(null);
                    toast({ title: "Reguła zaktualizowana" });
                  })
                  .catch(() => {
                    toast({ 
                      title: "Błąd", 
                      description: "Nie udało się zaktualizować reguły",
                      variant: "destructive" 
                    });
                  });
              }}
              data-testid="button-save-rule"
            >
              Zapisz
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <ZlpGenerationProgressDialog
        sessionId={zlpSessionId}
        open={zlpProgressDialogOpen}
        onOpenChange={setZlpProgressDialogOpen}
        planId={planId || 0}
        onComplete={handleZlpGenerationComplete}
      />

      <Dialog open={!!addPalletDialog} onOpenChange={(open) => {
        if (!open) {
          setAddPalletDialog(null);
          setShowNewCarrierForm(false);
          setNewCarrierName("");
          setNewCarrierCode("");
          setNewCarrierGroupId("");
        }
      }}>
        <DialogContent className="max-w-lg overflow-visible">
          <DialogHeader>
            <DialogTitle>Dodaj nową paletę</DialogTitle>
            <DialogDescription>
              Dodaj paletę do zlecenia ZLP z przepływem {addPalletDialog?.flowCode}
            </DialogDescription>
          </DialogHeader>
          <div className="space-y-4 py-4">
            {!showNewCarrierForm ? (
              <div className="space-y-2">
                <Label>Nośnik (opcjonalnie)</Label>
                <div className="flex gap-2">
                  <Combobox
                    options={carrierOptions}
                    value={newPalletCarrierId}
                    onChange={setNewPalletCarrierId}
                    placeholder="Wybierz nośnik..."
                    searchPlaceholder="Szukaj nośnika..."
                    className="flex-1"
                  />
                  <Button
                    type="button"
                    size="icon"
                    variant="outline"
                    onClick={() => {
                      const nextNum = nextCarrierNumber;
                      setNewCarrierCode(`PAL-${String(nextNum).padStart(3, '0')}`);
                      setNewCarrierName(`Paleta ${nextNum}`);
                      setShowNewCarrierForm(true);
                    }}
                    title="Dodaj nowy nośnik"
                    data-testid="button-show-new-carrier-form"
                  >
                    <Plus className="h-4 w-4" />
                  </Button>
                </div>
              </div>
            ) : (
              <div className="space-y-3 p-3 border rounded-md bg-muted/30">
                <div className="flex items-center justify-between">
                  <Label className="text-sm font-medium">Nowy nośnik</Label>
                  <Button
                    type="button"
                    size="sm"
                    variant="ghost"
                    onClick={() => {
                      setShowNewCarrierForm(false);
                      setNewCarrierName("");
                      setNewCarrierCode("");
                      setNewCarrierGroupId("");
                    }}
                    data-testid="button-cancel-new-carrier"
                  >
                    Anuluj
                  </Button>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="carrier-group" className="text-xs">Grupa nośników</Label>
                  <Select value={newCarrierGroupId} onValueChange={setNewCarrierGroupId}>
                    <SelectTrigger data-testid="select-carrier-group">
                      <SelectValue placeholder="Wybierz grupę..." />
                    </SelectTrigger>
                    <SelectContent>
                      {carrierGroupsData.map((group) => (
                        <SelectItem key={group.id} value={group.id.toString()}>
                          {group.name} ({group.code})
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="grid grid-cols-2 gap-2">
                  <div className="space-y-1">
                    <Label htmlFor="carrier-code" className="text-xs">Kod</Label>
                    <Input
                      id="carrier-code"
                      value={newCarrierCode}
                      onChange={(e) => setNewCarrierCode(e.target.value)}
                      placeholder="np. PAL-001"
                      data-testid="input-carrier-code"
                    />
                  </div>
                  <div className="space-y-1">
                    <Label htmlFor="carrier-name" className="text-xs">Nazwa</Label>
                    <Input
                      id="carrier-name"
                      value={newCarrierName}
                      onChange={(e) => setNewCarrierName(e.target.value)}
                      placeholder="np. Paleta 1"
                      data-testid="input-carrier-name"
                    />
                  </div>
                </div>
                <Button
                  type="button"
                  size="sm"
                  className="w-full"
                  onClick={() => {
                    if (newCarrierGroupId && newCarrierCode && newCarrierName) {
                      createCarrierMutation.mutate({
                        carrierGroupId: parseInt(newCarrierGroupId),
                        code: newCarrierCode,
                        name: newCarrierName,
                      });
                    }
                  }}
                  disabled={!newCarrierGroupId || !newCarrierCode || !newCarrierName || createCarrierMutation.isPending}
                  data-testid="button-create-carrier"
                >
                  {createCarrierMutation.isPending ? (
                    <>
                      <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                      Tworzenie...
                    </>
                  ) : (
                    "Utwórz nośnik"
                  )}
                </Button>
              </div>
            )}
          </div>
          <DialogFooter>
            <Button variant="outline" onClick={() => setAddPalletDialog(null)} data-testid="button-cancel-add-pallet">
              Anuluj
            </Button>
            <Button
              onClick={() => {
                if (addPalletDialog) {
                  addPalletMutation.mutate({
                    orderId: addPalletDialog.orderId,
                    flowCode: addPalletDialog.flowCode,
                    carrierId: newPalletCarrierId && newPalletCarrierId !== "none" ? parseInt(newPalletCarrierId) : undefined,
                  });
                }
              }}
              disabled={addPalletMutation.isPending || showNewCarrierForm}
              data-testid="button-confirm-add-pallet"
            >
              {addPalletMutation.isPending ? (
                <>
                  <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                  Dodawanie...
                </>
              ) : (
                "Dodaj paletę"
              )}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <PalletTreePanel
        flowGraph={flowGraph}
        isOpen={palletTreeOpen}
        onToggle={() => setPalletTreeOpen(!palletTreeOpen)}
      />
    </div>
  );
}
