import { useState } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { useLocation } from "wouter";
import { Plus, Edit, Trash2, Loader2, Wrench, Scissors, Hammer, Paintbrush, Ruler, Drill, Cog, Settings, Package, Layers, Factory, Palette, Truck, Timer, Star, Zap, Box, LucideIcon } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Dialog, DialogContent, DialogDescription, DialogFooter, DialogHeader, DialogTitle } from "@/components/ui/dialog";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage, FormDescription } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Badge } from "@/components/ui/badge";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { z } from "zod";
import type { ProductCreatorDictionary } from "@shared/schema";

const AVAILABLE_ICONS = [
  { name: "Scissors", label: "Cięcie", icon: Scissors },
  { name: "Hammer", label: "Montaż", icon: Hammer },
  { name: "Paintbrush", label: "Malowanie", icon: Paintbrush },
  { name: "Ruler", label: "Pomiar", icon: Ruler },
  { name: "Drill", label: "Wiercenie", icon: Drill },
  { name: "Cog", label: "Obróbka", icon: Cog },
  { name: "Wrench", label: "Naprawa", icon: Wrench },
  { name: "Settings", label: "Ustawienia", icon: Settings },
  { name: "Package", label: "Pakowanie", icon: Package },
  { name: "Layers", label: "Warstwy", icon: Layers },
  { name: "Factory", label: "Produkcja", icon: Factory },
  { name: "Palette", label: "Kolor", icon: Palette },
  { name: "Truck", label: "Transport", icon: Truck },
  { name: "Timer", label: "Czas", icon: Timer },
  { name: "Star", label: "Jakość", icon: Star },
  { name: "Zap", label: "Szybka op.", icon: Zap },
  { name: "Box", label: "Magazyn", icon: Box },
];

const AVAILABLE_COLORS = [
  { name: "#3B82F6", label: "Niebieski" },
  { name: "#10B981", label: "Zielony" },
  { name: "#F59E0B", label: "Pomarańczowy" },
  { name: "#EF4444", label: "Czerwony" },
  { name: "#8B5CF6", label: "Fioletowy" },
  { name: "#EC4899", label: "Różowy" },
  { name: "#6B7280", label: "Szary" },
  { name: "#14B8A6", label: "Turkusowy" },
  { name: "#84CC16", label: "Limonka" },
  { name: "#F97316", label: "Łososiowy" },
];

const ICON_MAP: Record<string, LucideIcon> = {
  Scissors, Hammer, Paintbrush, Ruler, Drill, Cog, 
  Wrench, Settings, Package, Layers, Factory, Palette, 
  Truck, Timer, Star, Zap, Box
};

function getIconComponent(iconName: string | null | undefined): LucideIcon | null {
  if (!iconName) return null;
  return ICON_MAP[iconName] || null;
}

const operationFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().optional(),
  sortOrder: z.coerce.number().optional(),
  icon: z.string().optional(),
  color: z.string().optional(),
});

type OperationFormData = z.infer<typeof operationFormSchema>;

function generateOperationCode(name: string): string {
  if (!name) return "";
  
  return name
    .toLowerCase()
    .normalize("NFD")
    .replace(/[\u0300-\u036f]/g, "")
    .replace(/ą/g, "a")
    .replace(/ć/g, "c")
    .replace(/ę/g, "e")
    .replace(/ł/g, "l")
    .replace(/ń/g, "n")
    .replace(/ó/g, "o")
    .replace(/ś/g, "s")
    .replace(/ź/g, "z")
    .replace(/ż/g, "z")
    .replace(/[^a-z0-9]+/g, "_")
    .replace(/^_+|_+$/g, "")
    .replace(/_+/g, "_");
}

export default function ProductionOperations() {
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [, navigate] = useLocation();
  const { toast } = useToast();

  const form = useForm<OperationFormData>({
    resolver: zodResolver(operationFormSchema),
    defaultValues: {
      name: "",
      description: "",
      sortOrder: 0,
      icon: "",
      color: "",
    },
  });

  const { data: operations = [], isLoading } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries", { type: "production_operation" }],
    queryFn: async () => {
      const response = await fetch("/api/dictionaries?type=production_operation");
      if (!response.ok) throw new Error("Failed to fetch operations");
      return response.json();
    },
  });

  const createMutation = useMutation({
    mutationFn: (data: any) =>
      apiRequest("POST", "/api/dictionaries", {
        ...data,
        dictionaryType: "production_operation",
      }),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      setIsDialogOpen(false);
      form.reset();
      toast({ title: "Operacja została utworzona" });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd tworzenia operacji",
        description: error.message || "Nieznany błąd",
        variant: "destructive",
      });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: (id: number) => apiRequest("DELETE", `/api/dictionaries/${id}`),
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      toast({ title: "Operacja została usunięta" });
    },
  });

  const onSubmit = (data: OperationFormData) => {
    const generatedCode = generateOperationCode(data.name);
    
    const normalizedData = {
      code: generatedCode,
      ...data,
      description: data.description || undefined,
      sortOrder: data.sortOrder || 0,
      icon: data.icon || undefined,
      color: data.color || undefined,
    };

    createMutation.mutate(normalizedData);
  };

  const handleEdit = (operation: ProductCreatorDictionary) => {
    navigate(`/production/operations/${operation.id}/edit`);
  };

  const handleDelete = (id: number) => {
    if (confirm("Czy na pewno chcesz usunąć tę operację?")) {
      deleteMutation.mutate(id);
    }
  };

  const handleAddNew = () => {
    form.reset({
      name: "",
      description: "",
      sortOrder: 0,
      icon: "",
      color: "",
    });
    setIsDialogOpen(true);
  };

  const sortedOperations = [...operations].sort((a, b) => {
    const orderA = a.sortOrder ?? 0;
    const orderB = b.sortOrder ?? 0;
    return orderA - orderB;
  });

  const watchedIcon = form.watch("icon");
  const SelectedIconComp = getIconComponent(watchedIcon);

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold">Operacje Produkcyjne</h1>
          <p className="text-muted-foreground">Zarządzaj słownikiem operacji produkcyjnych</p>
        </div>
        <Button onClick={handleAddNew} data-testid="button-add-operation">
          <Plus className="w-4 h-4 mr-2" />
          Dodaj Operację
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Wrench className="w-5 h-5" />
            Lista Operacji ({operations.length})
          </CardTitle>
        </CardHeader>
        <CardContent>
          {isLoading ? (
            <div className="flex items-center justify-center py-8">
              <Loader2 className="w-6 h-6 animate-spin" />
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-[100px]">Kolejność</TableHead>
                  <TableHead className="w-[150px]">Kod</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead className="w-[120px]">Ikona</TableHead>
                  <TableHead className="w-[120px]">Kolor</TableHead>
                  <TableHead>Opis</TableHead>
                  <TableHead className="w-[120px]">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {sortedOperations.length === 0 ? (
                  <TableRow>
                    <TableCell colSpan={7} className="text-center text-muted-foreground py-8">
                      Brak operacji. Dodaj pierwszą operację.
                    </TableCell>
                  </TableRow>
                ) : (
                  sortedOperations.map((operation) => {
                    const IconComp = getIconComponent(operation.icon);
                    return (
                      <TableRow 
                        key={operation.id} 
                        data-testid={`row-operation-${operation.id}`}
                        className="cursor-pointer hover-elevate"
                        onClick={() => handleEdit(operation)}
                      >
                        <TableCell>
                          <Badge variant="outline">{operation.sortOrder ?? 0}</Badge>
                        </TableCell>
                        <TableCell>
                          <code className="text-sm font-mono bg-muted px-2 py-1 rounded">
                            {operation.code}
                          </code>
                        </TableCell>
                        <TableCell className="font-medium">{operation.name}</TableCell>
                        <TableCell>
                          {IconComp ? (
                            <div className="flex items-center gap-1">
                              <IconComp className="w-4 h-4" />
                              <span className="text-xs text-muted-foreground">
                                {AVAILABLE_ICONS.find(i => i.name === operation.icon)?.label}
                              </span>
                            </div>
                          ) : (
                            <span className="text-muted-foreground">-</span>
                          )}
                        </TableCell>
                        <TableCell>
                          {operation.color ? (
                            <div className="flex items-center gap-2">
                              <div 
                                className="w-5 h-5 rounded border" 
                                style={{ backgroundColor: operation.color }}
                              />
                              <span className="text-xs text-muted-foreground">
                                {AVAILABLE_COLORS.find(c => c.name === operation.color)?.label || operation.color}
                              </span>
                            </div>
                          ) : (
                            <span className="text-muted-foreground">-</span>
                          )}
                        </TableCell>
                        <TableCell className="text-muted-foreground">
                          {operation.description || "-"}
                        </TableCell>
                        <TableCell>
                          <div className="flex gap-2">
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={(e) => {
                                e.stopPropagation();
                                handleEdit(operation);
                              }}
                              data-testid={`button-edit-operation-${operation.id}`}
                            >
                              <Edit className="w-4 h-4" />
                            </Button>
                            <Button
                              variant="ghost"
                              size="icon"
                              onClick={(e) => {
                                e.stopPropagation();
                                handleDelete(operation.id);
                              }}
                              data-testid={`button-delete-operation-${operation.id}`}
                            >
                              <Trash2 className="w-4 h-4" />
                            </Button>
                          </div>
                        </TableCell>
                      </TableRow>
                    );
                  })
                )}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>Dodaj Operację</DialogTitle>
            <DialogDescription>
              Dodaj nową operację do słownika
            </DialogDescription>
          </DialogHeader>

          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
              <FormField
                control={form.control}
                name="name"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Nazwa *</FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        placeholder="np. Cięcie płyt"
                        data-testid="input-operation-name"
                      />
                    </FormControl>
                    {field.value && (
                      <p className="text-xs text-muted-foreground">
                        Kod: <span className="font-mono text-primary">{generateOperationCode(field.value)}</span>
                      </p>
                    )}
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="sortOrder"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Kolejność</FormLabel>
                    <FormControl>
                      <Input
                        {...field}
                        type="number"
                        placeholder="0"
                        data-testid="input-operation-sort"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea
                        {...field}
                        placeholder="Szczegółowy opis operacji..."
                        rows={3}
                        data-testid="input-operation-description"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-2 gap-4">
                <FormField
                  control={form.control}
                  name="icon"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Ikona</FormLabel>
                      <Select 
                        value={field.value || "none"} 
                        onValueChange={(val) => field.onChange(val === "none" ? "" : val)}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-icon">
                            <SelectValue placeholder="Wybierz ikonę...">
                              {field.value && SelectedIconComp ? (
                                <span className="flex items-center gap-2">
                                  <SelectedIconComp className="w-4 h-4" />
                                  <span>{AVAILABLE_ICONS.find(i => i.name === field.value)?.label}</span>
                                </span>
                              ) : (
                                <span className="text-muted-foreground">Brak</span>
                              )}
                            </SelectValue>
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="none">
                            <span className="text-muted-foreground">Brak</span>
                          </SelectItem>
                          {AVAILABLE_ICONS.map(iconItem => {
                            const IconComp = iconItem.icon;
                            return (
                              <SelectItem key={iconItem.name} value={iconItem.name}>
                                <span className="flex items-center gap-2">
                                  <IconComp className="w-4 h-4" />
                                  <span>{iconItem.label}</span>
                                </span>
                              </SelectItem>
                            );
                          })}
                        </SelectContent>
                      </Select>
                      <FormDescription>Ikona wyświetlana przy operacji</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="color"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kolor</FormLabel>
                      <Select 
                        value={field.value || "none"} 
                        onValueChange={(val) => field.onChange(val === "none" ? "" : val)}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-color">
                            <SelectValue placeholder="Wybierz kolor...">
                              {field.value ? (
                                <span className="flex items-center gap-2">
                                  <div 
                                    className="w-4 h-4 rounded border" 
                                    style={{ backgroundColor: field.value }}
                                  />
                                  <span>{AVAILABLE_COLORS.find(c => c.name === field.value)?.label}</span>
                                </span>
                              ) : (
                                <span className="text-muted-foreground">Brak</span>
                              )}
                            </SelectValue>
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="none">
                            <span className="text-muted-foreground">Brak</span>
                          </SelectItem>
                          {AVAILABLE_COLORS.map(colorItem => (
                            <SelectItem key={colorItem.name} value={colorItem.name}>
                              <span className="flex items-center gap-2">
                                <div 
                                  className="w-4 h-4 rounded border" 
                                  style={{ backgroundColor: colorItem.name }}
                                />
                                <span>{colorItem.label}</span>
                              </span>
                            </SelectItem>
                          ))}
                        </SelectContent>
                      </Select>
                      <FormDescription>Kolor wyróżniający operację</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <DialogFooter>
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => setIsDialogOpen(false)}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={createMutation.isPending}
                  data-testid="button-submit-operation"
                >
                  {createMutation.isPending && (
                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  )}
                  Utwórz
                </Button>
              </DialogFooter>
            </form>
          </Form>
        </DialogContent>
      </Dialog>
    </div>
  );
}
