import { useEffect } from "react";
import { useParams, useLocation } from "wouter";
import { useQuery, useMutation } from "@tanstack/react-query";
import { useForm } from "react-hook-form";
import { zodResolver } from "@hookform/resolvers/zod";
import { ArrowLeft, Loader2, Save, Scissors, Hammer, Paintbrush, Ruler, Drill, Cog, Wrench, Settings, Package, Layers, Factory, Palette, Truck, Timer, Star, Zap, Box, X, LucideIcon } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage, FormDescription } from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { z } from "zod";
import type { ProductCreatorDictionary } from "@shared/schema";

const AVAILABLE_ICONS = [
  { name: "Scissors", label: "Cięcie", icon: Scissors },
  { name: "Hammer", label: "Montaż", icon: Hammer },
  { name: "Paintbrush", label: "Malowanie", icon: Paintbrush },
  { name: "Ruler", label: "Pomiar", icon: Ruler },
  { name: "Drill", label: "Wiercenie", icon: Drill },
  { name: "Cog", label: "Obróbka", icon: Cog },
  { name: "Wrench", label: "Naprawa", icon: Wrench },
  { name: "Settings", label: "Ustawienia", icon: Settings },
  { name: "Package", label: "Pakowanie", icon: Package },
  { name: "Layers", label: "Warstwy", icon: Layers },
  { name: "Factory", label: "Produkcja", icon: Factory },
  { name: "Palette", label: "Kolor", icon: Palette },
  { name: "Truck", label: "Transport", icon: Truck },
  { name: "Timer", label: "Czas", icon: Timer },
  { name: "Star", label: "Jakość", icon: Star },
  { name: "Zap", label: "Szybka op.", icon: Zap },
  { name: "Box", label: "Magazyn", icon: Box },
];

const AVAILABLE_COLORS = [
  { name: "#3B82F6", label: "Niebieski" },
  { name: "#10B981", label: "Zielony" },
  { name: "#F59E0B", label: "Pomarańczowy" },
  { name: "#EF4444", label: "Czerwony" },
  { name: "#8B5CF6", label: "Fioletowy" },
  { name: "#EC4899", label: "Różowy" },
  { name: "#6B7280", label: "Szary" },
  { name: "#14B8A6", label: "Turkusowy" },
  { name: "#84CC16", label: "Limonka" },
  { name: "#F97316", label: "Łososiowy" },
];

const ICON_MAP: Record<string, LucideIcon> = {
  Scissors, Hammer, Paintbrush, Ruler, Drill, Cog, 
  Wrench, Settings, Package, Layers, Factory, Palette, 
  Truck, Timer, Star, Zap, Box
};

function getIconComponent(iconName: string | null | undefined): LucideIcon | null {
  if (!iconName) return null;
  return ICON_MAP[iconName] || null;
}

const operationFormSchema = z.object({
  name: z.string().min(1, "Nazwa jest wymagana"),
  description: z.string().optional(),
  sortOrder: z.coerce.number().optional(),
  icon: z.string().optional(),
  color: z.string().optional(),
});

type OperationFormData = z.infer<typeof operationFormSchema>;

export default function ProductionOperationEdit() {
  const params = useParams<{ id: string }>();
  const [, navigate] = useLocation();
  const { toast } = useToast();
  const operationId = params.id ? parseInt(params.id, 10) : null;

  const form = useForm<OperationFormData>({
    resolver: zodResolver(operationFormSchema),
    defaultValues: {
      name: "",
      description: "",
      sortOrder: 0,
      icon: "",
      color: "",
    },
  });

  const { data: operation, isLoading } = useQuery<ProductCreatorDictionary>({
    queryKey: ["/api/dictionaries", operationId],
    queryFn: async () => {
      const response = await fetch(`/api/dictionaries/${operationId}`);
      if (!response.ok) throw new Error("Failed to fetch operation");
      return response.json();
    },
    enabled: !!operationId,
  });

  useEffect(() => {
    if (operation) {
      form.reset({
        name: operation.name || "",
        description: operation.description || "",
        sortOrder: operation.sortOrder ?? 0,
        icon: operation.icon || "",
        color: operation.color || "",
      });
    }
  }, [operation, form]);

  const updateMutation = useMutation({
    mutationFn: async (data: OperationFormData) => {
      return apiRequest("PATCH", `/api/dictionaries/${operationId}`, {
        name: data.name,
        description: data.description || null,
        sortOrder: data.sortOrder ?? 0,
        icon: data.icon || null,
        color: data.color || null,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/dictionaries"] });
      toast({
        title: "Sukces",
        description: "Operacja została zaktualizowana",
      });
      navigate("/production/operations");
    },
    onError: (error: Error) => {
      toast({
        title: "Błąd",
        description: error.message || "Nie udało się zaktualizować operacji",
        variant: "destructive",
      });
    },
  });

  const onSubmit = (data: OperationFormData) => {
    updateMutation.mutate(data);
  };

  if (isLoading) {
    return (
      <div className="p-6 flex items-center justify-center min-h-[400px]">
        <Loader2 className="w-8 h-8 animate-spin text-muted-foreground" />
      </div>
    );
  }

  if (!operation) {
    return (
      <div className="p-6">
        <Card>
          <CardContent className="p-6 text-center text-muted-foreground">
            Nie znaleziono operacji
          </CardContent>
        </Card>
      </div>
    );
  }

  const selectedIcon = form.watch("icon");
  const SelectedIconComp = getIconComponent(selectedIcon);

  return (
    <div className="p-6 space-y-6">
      <div className="flex items-center gap-4">
        <Button 
          variant="ghost" 
          size="icon"
          onClick={() => navigate("/production/operations")}
          data-testid="button-back"
        >
          <ArrowLeft className="w-5 h-5" />
        </Button>
        <div>
          <h1 className="text-2xl font-bold">Edycja Operacji</h1>
          <p className="text-muted-foreground">
            Kod: <code className="text-sm font-mono bg-muted px-2 py-0.5 rounded">{operation.code}</code>
          </p>
        </div>
      </div>

      <Card>
        <CardHeader>
          <CardTitle>Dane Operacji</CardTitle>
          <CardDescription>Zaktualizuj informacje o operacji produkcyjnej</CardDescription>
        </CardHeader>
        <CardContent>
          <Form {...form}>
            <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <FormField
                  control={form.control}
                  name="name"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Nazwa *</FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          placeholder="np. Cięcie płyt"
                          data-testid="input-operation-name"
                        />
                      </FormControl>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="sortOrder"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kolejność</FormLabel>
                      <FormControl>
                        <Input
                          {...field}
                          type="number"
                          placeholder="0"
                          data-testid="input-operation-sort"
                        />
                      </FormControl>
                      <FormDescription>Kolejność wyświetlania na liście</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <FormField
                control={form.control}
                name="description"
                render={({ field }) => (
                  <FormItem>
                    <FormLabel>Opis</FormLabel>
                    <FormControl>
                      <Textarea
                        {...field}
                        placeholder="Szczegółowy opis operacji..."
                        rows={3}
                        data-testid="input-operation-description"
                      />
                    </FormControl>
                    <FormMessage />
                  </FormItem>
                )}
              />

              <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                <FormField
                  control={form.control}
                  name="icon"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Ikona</FormLabel>
                      <Select 
                        value={field.value || "none"} 
                        onValueChange={(val) => field.onChange(val === "none" ? "" : val)}
                      >
                        <FormControl>
                          <SelectTrigger data-testid="select-icon">
                            <SelectValue placeholder="Wybierz ikonę...">
                              {field.value && SelectedIconComp ? (
                                <span className="flex items-center gap-2">
                                  <SelectedIconComp className="w-4 h-4" />
                                  <span>{AVAILABLE_ICONS.find(i => i.name === field.value)?.label}</span>
                                </span>
                              ) : (
                                <span className="text-muted-foreground">Brak</span>
                              )}
                            </SelectValue>
                          </SelectTrigger>
                        </FormControl>
                        <SelectContent>
                          <SelectItem value="none">
                            <span className="text-muted-foreground">Brak</span>
                          </SelectItem>
                          {AVAILABLE_ICONS.map(iconItem => {
                            const IconComp = iconItem.icon;
                            return (
                              <SelectItem key={iconItem.name} value={iconItem.name}>
                                <span className="flex items-center gap-2">
                                  <IconComp className="w-4 h-4" />
                                  <span>{iconItem.label}</span>
                                </span>
                              </SelectItem>
                            );
                          })}
                        </SelectContent>
                      </Select>
                      <FormDescription>Ikona wyświetlana przy operacji</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />

                <FormField
                  control={form.control}
                  name="color"
                  render={({ field }) => (
                    <FormItem>
                      <FormLabel>Kolor</FormLabel>
                      <FormControl>
                        <div className="space-y-3">
                          <div className="flex flex-wrap gap-2" data-testid="color-palette">
                            <button
                              type="button"
                              onClick={() => field.onChange("")}
                              className={`w-8 h-8 rounded-md border-2 flex items-center justify-center transition-all ${
                                !field.value 
                                  ? "border-primary ring-2 ring-primary/30" 
                                  : "border-muted hover:border-muted-foreground"
                              }`}
                              title="Brak koloru"
                            >
                              <X className="w-4 h-4 text-muted-foreground" />
                            </button>
                            {AVAILABLE_COLORS.map(colorItem => (
                              <button
                                key={colorItem.name}
                                type="button"
                                onClick={() => field.onChange(colorItem.name)}
                                className={`w-8 h-8 rounded-md border-2 transition-all ${
                                  field.value === colorItem.name 
                                    ? "border-primary ring-2 ring-primary/30 scale-110" 
                                    : "border-transparent hover:scale-105"
                                }`}
                                style={{ backgroundColor: colorItem.name }}
                                title={colorItem.label}
                              />
                            ))}
                          </div>
                          {field.value && (
                            <div className="flex items-center gap-2 text-sm">
                              <div 
                                className="w-4 h-4 rounded border" 
                                style={{ backgroundColor: field.value }}
                              />
                              <span className="text-muted-foreground">
                                Wybrany: {AVAILABLE_COLORS.find(c => c.name === field.value)?.label || field.value}
                              </span>
                            </div>
                          )}
                        </div>
                      </FormControl>
                      <FormDescription>Kolor wyróżniający operację w wizualizacji</FormDescription>
                      <FormMessage />
                    </FormItem>
                  )}
                />
              </div>

              <div className="flex justify-end gap-3 pt-4 border-t">
                <Button
                  type="button"
                  variant="outline"
                  onClick={() => navigate("/production/operations")}
                  data-testid="button-cancel"
                >
                  Anuluj
                </Button>
                <Button
                  type="submit"
                  disabled={updateMutation.isPending}
                  data-testid="button-save"
                >
                  {updateMutation.isPending && (
                    <Loader2 className="w-4 h-4 mr-2 animate-spin" />
                  )}
                  <Save className="w-4 h-4 mr-2" />
                  Zapisz zmiany
                </Button>
              </div>
            </form>
          </Form>
        </CardContent>
      </Card>
    </div>
  );
}
