import { useState, useEffect, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Switch } from "@/components/ui/switch";
import { Combobox } from "@/components/ui/combobox";
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogFooter,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogCancel,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from "@/components/ui/alert-dialog";
import { Plus, Pencil, Trash2, FlaskConical, Settings2, Route, MapPin } from "lucide-react";

interface ComponentRoutingTemplate {
  id: number;
  name: string;
  description: string | null;
  componentPattern: string | null;
  materialType: string | null;
  colorCode: string | null;
  defaultRoutingId: number;
  defaultRoutingCode: string;
  defaultRoutingName: string;
  defaultWorkCenterId: number | null;
  defaultWorkCenterCode: string | null;
  defaultWorkCenterName: string | null;
  locationId: number | null;
  locationCode: string | null;
  locationName: string | null;
  priority: number;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
}

interface Routing {
  id: number;
  code: string;
  name: string;
}

interface WorkCenter {
  id: number;
  code: string;
  name: string;
}

interface Location {
  id: number;
  code: string;
  name: string;
}

interface Dictionary {
  id: number;
  code: string;
  name: string;
  dictionaryType: string;
}

const MATERIAL_TYPES = [
  { value: "plyta_18mm", label: "Płyta 18mm" },
  { value: "plyta_25mm", label: "Płyta 25mm" },
  { value: "hdf_3mm", label: "HDF 3mm" },
  { value: "hdf_8mm", label: "HDF 8mm" },
];

export default function ComponentRoutingTemplatesPage() {
  const { toast } = useToast();
  const [isDialogOpen, setIsDialogOpen] = useState(false);
  const [editingTemplate, setEditingTemplate] = useState<ComponentRoutingTemplate | null>(null);
  const [deleteConfirmId, setDeleteConfirmId] = useState<number | null>(null);
  const [testPattern, setTestPattern] = useState("");
  const [testComponentName, setTestComponentName] = useState("");
  const [testResult, setTestResult] = useState<boolean | null>(null);

  const [formData, setFormData] = useState({
    name: "",
    description: "",
    componentPattern: "",
    materialType: "",
    colorCode: "",
    defaultRoutingId: 0,
    defaultWorkCenterId: 0,
    locationId: 0,
    priority: 100,
    isActive: true,
  });

  const { data: templates = [], isLoading } = useQuery<ComponentRoutingTemplate[]>({
    queryKey: ["/api/production/component-routing-templates"],
  });

  const { data: routings = [] } = useQuery<Routing[]>({
    queryKey: ["/api/production/routings"],
  });

  const { data: workCenters = [] } = useQuery<WorkCenter[]>({
    queryKey: ["/api/production/work-centers"],
  });

  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
  });

  const { data: dictionaries = [] } = useQuery<Dictionary[]>({
    queryKey: ["/api/dictionaries"],
  });

  const colors = useMemo(() => 
    dictionaries.filter((d) => d.dictionaryType === "color"),
    [dictionaries]
  );

  const componentCz1 = useMemo(() => 
    dictionaries.filter((d) => d.dictionaryType === "component_cz1"),
    [dictionaries]
  );

  const colorOptions = useMemo(() => 
    colors.map((c) => ({ 
      value: c.code, 
      label: c.name === c.code ? c.name : `${c.name} (${c.code})` 
    })),
    [colors]
  );

  const componentPatternOptions = useMemo(() => 
    componentCz1.map((c) => ({ 
      value: c.code, 
      label: c.name === c.code ? c.name : `${c.name} (${c.code})` 
    })),
    [componentCz1]
  );

  const routingOptions = useMemo(() => 
    routings.map((r) => ({ value: r.id.toString(), label: `${r.name} (${r.code})` })),
    [routings]
  );

  const workCenterOptions = useMemo(() => 
    workCenters.map((wc) => ({ value: wc.id.toString(), label: `${wc.name} (${wc.code})` })),
    [workCenters]
  );

  const locationOptions = useMemo(() => 
    locations.map((loc) => ({ value: loc.id.toString(), label: `${loc.name} (${loc.code})` })),
    [locations]
  );

  const generateAutoName = useMemo(() => {
    const parts: string[] = [];
    
    if (formData.componentPattern) {
      const patterns = formData.componentPattern.split(/[|\n]/).map(p => p.trim()).filter(p => p);
      if (patterns.length === 1) {
        parts.push(patterns[0]);
      } else if (patterns.length > 1) {
        parts.push(`${patterns[0]} +${patterns.length - 1}`);
      }
    }
    
    const conditions: string[] = [];
    if (formData.materialType) {
      const mt = MATERIAL_TYPES.find(t => t.value === formData.materialType);
      if (mt) conditions.push(mt.label);
    }
    if (formData.colorCode) {
      conditions.push(formData.colorCode);
    }
    if (conditions.length > 0) {
      parts.push(`[${conditions.join(", ")}]`);
    }
    
    const routing = routings.find(r => r.id === formData.defaultRoutingId);
    if (routing) {
      parts.push("→");
      parts.push(routing.code);
    }
    
    return parts.join(" ") || "";
  }, [formData.componentPattern, formData.materialType, formData.colorCode, formData.defaultRoutingId, routings]);

  useEffect(() => {
    if (generateAutoName) {
      setFormData(prev => ({ ...prev, name: generateAutoName }));
    }
  }, [generateAutoName]);

  const createMutation = useMutation({
    mutationFn: async (data: typeof formData) => {
      return apiRequest("POST", "/api/production/component-routing-templates", {
        ...data,
        componentPattern: data.componentPattern || null,
        materialType: data.materialType || null,
        colorCode: data.colorCode || null,
        defaultWorkCenterId: data.defaultWorkCenterId || null,
        locationId: data.locationId || null,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/component-routing-templates"] });
      setIsDialogOpen(false);
      resetForm();
      toast({ title: "Szablon utworzony" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const updateMutation = useMutation({
    mutationFn: async ({ id, data }: { id: number; data: typeof formData }) => {
      return apiRequest("PUT", `/api/production/component-routing-templates/${id}`, {
        ...data,
        componentPattern: data.componentPattern || null,
        materialType: data.materialType || null,
        colorCode: data.colorCode || null,
        defaultWorkCenterId: data.defaultWorkCenterId || null,
        locationId: data.locationId || null,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/component-routing-templates"] });
      setIsDialogOpen(false);
      setEditingTemplate(null);
      resetForm();
      toast({ title: "Szablon zaktualizowany" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/component-routing-templates/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/production/component-routing-templates"] });
      setDeleteConfirmId(null);
      toast({ title: "Szablon usunięty" });
    },
    onError: (error: any) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const testPatternMutation = useMutation({
    mutationFn: async ({ componentName, pattern }: { componentName: string; pattern: string }) => {
      const response = await apiRequest("POST", "/api/production/component-routing-templates/test-pattern", { componentName, pattern });
      return response;
    },
    onSuccess: (data: any) => {
      setTestResult(data.matches);
    },
  });

  const resetForm = () => {
    setFormData({
      name: "",
      description: "",
      componentPattern: "",
      materialType: "",
      colorCode: "",
      defaultRoutingId: 0,
      defaultWorkCenterId: 0,
      locationId: 0,
      priority: 100,
      isActive: true,
    });
  };

  const openEditDialog = (template: ComponentRoutingTemplate) => {
    setEditingTemplate(template);
    setFormData({
      name: template.name,
      description: template.description || "",
      componentPattern: template.componentPattern || "",
      materialType: template.materialType || "",
      colorCode: template.colorCode || "",
      defaultRoutingId: template.defaultRoutingId,
      defaultWorkCenterId: template.defaultWorkCenterId || 0,
      locationId: template.locationId || 0,
      priority: template.priority,
      isActive: template.isActive,
    });
    setIsDialogOpen(true);
  };

  const openCreateDialog = () => {
    setEditingTemplate(null);
    resetForm();
    setIsDialogOpen(true);
  };

  const handleSubmit = () => {
    if (editingTemplate) {
      updateMutation.mutate({ id: editingTemplate.id, data: formData });
    } else {
      createMutation.mutate(formData);
    }
  };

  const handleTestPattern = () => {
    if (testPattern && testComponentName) {
      testPatternMutation.mutate({ componentName: testComponentName, pattern: testPattern });
    }
  };

  return (
    <div className="container mx-auto p-4 space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-2xl font-bold flex items-center gap-2">
            <Settings2 className="h-6 w-6" />
            Szablony Marszrut Komponentów
          </h1>
          <p className="text-muted-foreground">
            Globalne szablony definiujące domyślne marszruty dla wzorców komponentów
          </p>
        </div>
        <Button onClick={openCreateDialog} data-testid="button-create-template">
          <Plus className="h-4 w-4 mr-2" />
          Nowy szablon
        </Button>
      </div>

      <Card>
        <CardHeader>
          <CardTitle className="text-base flex items-center gap-2">
            <FlaskConical className="h-4 w-4" />
            Test wzorca
          </CardTitle>
          <CardDescription>Sprawdź czy nazwa komponentu pasuje do wzorca SQL LIKE</CardDescription>
        </CardHeader>
        <CardContent>
          <div className="flex items-end gap-4">
            <div className="flex-1">
              <Label htmlFor="testComponentName">Nazwa komponentu</Label>
              <Input
                id="testComponentName"
                value={testComponentName}
                onChange={(e) => setTestComponentName(e.target.value)}
                placeholder="np. BOCZKI-COW-VB-390x280-WOTAN"
                data-testid="input-test-component-name"
              />
            </div>
            <div className="flex-1">
              <Label htmlFor="testPattern">Wzorzec (SQL LIKE)</Label>
              <Input
                id="testPattern"
                value={testPattern}
                onChange={(e) => setTestPattern(e.target.value)}
                placeholder="np. BOCZKI-COW-%"
                data-testid="input-test-pattern"
              />
            </div>
            <Button onClick={handleTestPattern} variant="outline" data-testid="button-test-pattern">
              Testuj
            </Button>
            {testResult !== null && (
              <Badge variant={testResult ? "default" : "destructive"}>
                {testResult ? "Pasuje" : "Nie pasuje"}
              </Badge>
            )}
          </div>
        </CardContent>
      </Card>

      <Card>
        <CardContent className="p-0">
          {isLoading ? (
            <div className="p-8 text-center text-muted-foreground">Ładowanie...</div>
          ) : templates.length === 0 ? (
            <div className="p-8 text-center text-muted-foreground">
              Brak szablonów. Kliknij "Nowy szablon" aby utworzyć pierwszy.
            </div>
          ) : (
            <Table>
              <TableHeader>
                <TableRow>
                  <TableHead className="w-[140px]">ID</TableHead>
                  <TableHead>Nazwa</TableHead>
                  <TableHead>Wzorzec</TableHead>
                  <TableHead>Materiał / Kolor</TableHead>
                  <TableHead>Domyślna marszruta</TableHead>
                  <TableHead>Lokalizacja</TableHead>
                  <TableHead className="text-center">Priorytet</TableHead>
                  <TableHead className="text-center">Status</TableHead>
                  <TableHead className="text-right">Akcje</TableHead>
                </TableRow>
              </TableHeader>
              <TableBody>
                {templates.map((template) => (
                  <TableRow key={template.id} data-testid={`row-template-${template.id}`}>
                    <TableCell>
                      <code className="text-xs font-mono bg-muted px-1.5 py-0.5 rounded">
                        TEMPL-ROUT-{String(template.id).padStart(3, '0')}
                      </code>
                    </TableCell>
                    <TableCell className="font-medium">{template.name}</TableCell>
                    <TableCell>
                      <div className="flex flex-wrap gap-1">
                        {template.componentPattern ? (
                          template.componentPattern.split(/[|\n]/).filter(p => p.trim()).map((pattern, idx) => (
                            <code key={idx} className="text-xs bg-muted px-1 py-0.5 rounded whitespace-nowrap">
                              {pattern.trim()}
                            </code>
                          ))
                        ) : (
                          <code className="text-xs bg-muted px-1 py-0.5 rounded">*</code>
                        )}
                      </div>
                    </TableCell>
                    <TableCell>
                      <div className="flex flex-col gap-1">
                        {template.materialType && (
                          <Badge variant="outline" className="text-xs w-fit">
                            {MATERIAL_TYPES.find((m) => m.value === template.materialType)?.label ||
                              template.materialType}
                          </Badge>
                        )}
                        {template.colorCode && (
                          <Badge variant="secondary" className="text-xs w-fit">
                            {template.colorCode}
                          </Badge>
                        )}
                        {!template.materialType && !template.colorCode && (
                          <span className="text-muted-foreground text-xs">wszystkie</span>
                        )}
                      </div>
                    </TableCell>
                    <TableCell>
                      <div className="flex items-center gap-1">
                        <Route className="h-3 w-3 text-muted-foreground" />
                        <span className="text-sm">{template.defaultRoutingName}</span>
                      </div>
                    </TableCell>
                    <TableCell>
                      {template.locationName ? (
                        <div className="flex items-center gap-1">
                          <MapPin className="h-3 w-3 text-muted-foreground" />
                          <span className="text-sm">{template.locationName}</span>
                        </div>
                      ) : (
                        <span className="text-muted-foreground text-xs">-</span>
                      )}
                    </TableCell>
                    <TableCell className="text-center">{template.priority}</TableCell>
                    <TableCell className="text-center">
                      <Badge variant={template.isActive ? "default" : "secondary"}>
                        {template.isActive ? "Aktywny" : "Nieaktywny"}
                      </Badge>
                    </TableCell>
                    <TableCell className="text-right">
                      <div className="flex items-center justify-end gap-1">
                        <Button
                          size="icon"
                          variant="ghost"
                          onClick={() => openEditDialog(template)}
                          data-testid={`button-edit-template-${template.id}`}
                        >
                          <Pencil className="h-4 w-4" />
                        </Button>
                        <Button
                          size="icon"
                          variant="ghost"
                          onClick={() => setDeleteConfirmId(template.id)}
                          data-testid={`button-delete-template-${template.id}`}
                        >
                          <Trash2 className="h-4 w-4" />
                        </Button>
                      </div>
                    </TableCell>
                  </TableRow>
                ))}
              </TableBody>
            </Table>
          )}
        </CardContent>
      </Card>

      <Dialog open={isDialogOpen} onOpenChange={setIsDialogOpen}>
        <DialogContent className="max-w-4xl">
          <DialogHeader>
            <DialogTitle>
              {editingTemplate ? "Edytuj szablon" : "Nowy szablon marszruty"}
            </DialogTitle>
            <DialogDescription>
              Zdefiniuj domyślną marszrutę dla wzorca komponentów. Będzie używana we wszystkich planach.
            </DialogDescription>
          </DialogHeader>

          <div className="grid gap-4 py-4">
            <div className="space-y-2">
              <Label>Nazwa szablonu (generowana automatycznie)</Label>
              <div 
                className="flex items-center min-h-9 px-3 py-2 rounded-md border bg-muted/50 text-sm"
                data-testid="display-template-name"
              >
                {formData.name || <span className="text-muted-foreground italic">Wypełnij pola poniżej...</span>}
              </div>
            </div>

            <div className="space-y-2">
              <Label htmlFor="priority">Priorytet</Label>
              <Input
                id="priority"
                type="number"
                value={formData.priority}
                onChange={(e) => setFormData({ ...formData, priority: parseInt(e.target.value) || 0 })}
                data-testid="input-template-priority"
                className="w-32"
              />
            </div>

            <div className="space-y-2">
              <Label htmlFor="description">Opis</Label>
              <Textarea
                id="description"
                value={formData.description}
                onChange={(e) => setFormData({ ...formData, description: e.target.value })}
                placeholder="Opcjonalny opis szablonu..."
                rows={2}
                data-testid="input-template-description"
              />
            </div>

            <div className="border-t pt-4">
              <h4 className="font-medium mb-3">Warunki dopasowania</h4>
              <div className="grid grid-cols-3 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="componentPattern">Wzorzec nazwy (SQL LIKE)</Label>
                  <div className="flex gap-2">
                    <Textarea
                      id="componentPattern"
                      value={formData.componentPattern}
                      onChange={(e) => setFormData({ ...formData, componentPattern: e.target.value })}
                      placeholder="np. BOCZKI%&#10;DNO%&#10;SIEDZISKO%"
                      data-testid="input-template-component-pattern"
                      className="flex-1 min-h-[70px]"
                      rows={3}
                    />
                    <Combobox
                      options={componentPatternOptions}
                      value=""
                      onChange={(value) => {
                        if (value) {
                          const newPattern = `${value}%`;
                          const current = formData.componentPattern.trim();
                          if (current) {
                            setFormData({ ...formData, componentPattern: `${current}\n${newPattern}` });
                          } else {
                            setFormData({ ...formData, componentPattern: newPattern });
                          }
                        }
                      }}
                      placeholder="Ze słownika"
                      emptyText="Brak"
                      searchPlaceholder="Szukaj..."
                      className="w-[140px]"
                    />
                  </div>
                  <p className="text-xs text-muted-foreground">Wiele wzorców w osobnych liniach. % = dowolne znaki, _ = jeden znak.</p>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="materialType">Typ materiału</Label>
                  <Select
                    value={formData.materialType || "__all__"}
                    onValueChange={(value) => setFormData({ ...formData, materialType: value === "__all__" ? "" : value })}
                  >
                    <SelectTrigger data-testid="select-template-material-type">
                      <SelectValue placeholder="Wszystkie" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="__all__">Wszystkie</SelectItem>
                      {MATERIAL_TYPES.map((type) => (
                        <SelectItem key={type.value} value={type.value}>
                          {type.label}
                        </SelectItem>
                      ))}
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label htmlFor="colorCode">Kolor</Label>
                  <Combobox
                    options={colorOptions}
                    value={formData.colorCode}
                    onChange={(value) => setFormData({ ...formData, colorCode: value })}
                    placeholder="Wybierz kolor"
                    emptyText="Brak kolorów"
                    searchPlaceholder="Szukaj koloru..."
                    className="w-full"
                  />
                </div>
              </div>
            </div>

            <div className="border-t pt-4">
              <h4 className="font-medium mb-3">Domyślna marszruta i lokalizacja</h4>
              <div className="grid grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label htmlFor="defaultRoutingId">Marszruta *</Label>
                  <Combobox
                    options={routingOptions}
                    value={formData.defaultRoutingId ? formData.defaultRoutingId.toString() : ""}
                    onChange={(value) => setFormData({ ...formData, defaultRoutingId: value ? parseInt(value) : 0 })}
                    placeholder="Wybierz marszrutę"
                    emptyText="Brak marszrut"
                    searchPlaceholder="Szukaj marszruty..."
                    className="w-full"
                  />
                </div>
                <div className="space-y-2">
                  <Label htmlFor="locationId">Lokalizacja</Label>
                  <Combobox
                    options={locationOptions}
                    value={formData.locationId ? formData.locationId.toString() : ""}
                    onChange={(value) => setFormData({ ...formData, locationId: value ? parseInt(value) : 0 })}
                    placeholder="Wybierz lokalizację"
                    emptyText="Brak lokalizacji"
                    searchPlaceholder="Szukaj lokalizacji..."
                    className="w-full"
                  />
                </div>
              </div>
            </div>

            <div className="flex items-center gap-2 pt-4">
              <Switch
                id="isActive"
                checked={formData.isActive}
                onCheckedChange={(checked) => setFormData({ ...formData, isActive: checked })}
                data-testid="switch-template-active"
              />
              <Label htmlFor="isActive">Szablon aktywny</Label>
            </div>
          </div>

          <DialogFooter>
            <Button variant="outline" onClick={() => setIsDialogOpen(false)}>
              Anuluj
            </Button>
            <Button
              onClick={handleSubmit}
              disabled={!formData.name || !formData.defaultRoutingId}
              data-testid="button-save-template"
            >
              {editingTemplate ? "Zapisz" : "Utwórz"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>

      <AlertDialog open={deleteConfirmId !== null} onOpenChange={() => setDeleteConfirmId(null)}>
        <AlertDialogContent>
          <AlertDialogHeader>
            <AlertDialogTitle>Czy na pewno chcesz usunąć ten szablon?</AlertDialogTitle>
            <AlertDialogDescription>
              Tej operacji nie można cofnąć. Szablon zostanie trwale usunięty.
            </AlertDialogDescription>
          </AlertDialogHeader>
          <AlertDialogFooter>
            <AlertDialogCancel>Anuluj</AlertDialogCancel>
            <AlertDialogAction
              onClick={() => deleteConfirmId && deleteMutation.mutate(deleteConfirmId)}
              className="bg-destructive text-destructive-foreground hover:bg-destructive/90"
            >
              Usuń
            </AlertDialogAction>
          </AlertDialogFooter>
        </AlertDialogContent>
      </AlertDialog>
    </div>
  );
}
