import { useState, useEffect, useRef, useCallback } from "react";
import { useQuery } from "@tanstack/react-query";
import cytoscape, { Core, NodeSingular } from "cytoscape";
import dagre from "cytoscape-dagre";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Skeleton } from "@/components/ui/skeleton";
import { ScrollArea } from "@/components/ui/scroll-area";
import {
  BarChart3,
  TrendingUp,
  Package,
  ClipboardList,
  Factory,
  CheckCircle2,
  Clock,
  Pause,
  XCircle,
  FileBox,
  Boxes,
  Network,
  Search,
  RefreshCw,
  ZoomIn,
  ZoomOut,
  Maximize2,
  Layers,
  ShoppingCart,
  Wrench,
  Calendar,
  Star,
  Component,
  Activity,
} from "lucide-react";
import { format } from "date-fns";
import { pl } from "date-fns/locale";
import { Progress } from "@/components/ui/progress";

cytoscape.use(dagre);

interface ZlpStatusSummary {
  status: string;
  count: number;
  itemCount: number;
}

interface PlanSummary {
  id: number;
  name: string;
  status: string;
  zlpCount: number;
  itemCount: number;
  completedCount: number;
}

interface WorkCenterStats {
  id: number;
  name: string;
  code: string;
  totalOperations: number;
  completedOperations: number;
  inProgressOperations: number;
  pendingOperations: number;
}

interface DailyStats {
  date: string;
  completedZlp: number;
  createdZlp: number;
}

interface TopProduct {
  id: number;
  name: string;
  sku: string;
  zlpCount: number;
  totalQuantity: number;
}

interface RecentZlp {
  id: number;
  name: string;
  status: string;
  itemCount: number;
  createdAt: string;
}

interface AnalyticsDashboard {
  zlpByStatus: ZlpStatusSummary[];
  totalZlp: number;
  totalItems: number;
  completedItems: number;
  pendingItems: number;
  inProgressItems: number;
  plans: PlanSummary[];
  recentPlans: PlanSummary[];
  workCenterStats: WorkCenterStats[];
  dailyStats: DailyStats[];
  topProducts: TopProduct[];
  recentZlps: RecentZlp[];
  bomStats: {
    totalBoms: number;
    bomsWithFormatki: number;
    averageFormatkiPerBom: number;
  };
  formatkiStats: {
    totalFormatki: number;
    uniqueFormatki: number;
    formatkiInProduction: number;
  };
}

interface CytoscapeNode {
  data: {
    id: string;
    label: string;
    type: "product" | "set" | "formatka" | "bom" | "order" | "zlp";
    sku?: string;
    color?: string;
    status?: string;
  };
}

interface CytoscapeEdge {
  data: {
    id: string;
    source: string;
    target: string;
    label: string;
  };
}

interface CytoscapeGraph {
  nodes: CytoscapeNode[];
  edges: CytoscapeEdge[];
  stats: {
    totalProducts: number;
    totalSets: number;
    totalBoms: number;
    totalZlps: number;
    totalFormatki: number;
  };
}

const statusLabels: Record<string, string> = {
  draft: "Szkic",
  pending: "Oczekujące",
  in_progress: "W trakcie",
  ready: "Gotowe",
  completed: "Zakończone",
  cancelled: "Anulowane",
  on_hold: "Wstrzymane",
  approved: "Zatwierdzone",
  generated: "Wygenerowane",
};

const statusIcons: Record<string, typeof Clock> = {
  draft: FileBox,
  pending: Clock,
  in_progress: Factory,
  ready: CheckCircle2,
  completed: CheckCircle2,
  cancelled: XCircle,
  on_hold: Pause,
  approved: CheckCircle2,
  generated: Package,
};

const statusColors: Record<string, string> = {
  draft: "bg-slate-500",
  pending: "bg-amber-500",
  in_progress: "bg-blue-500",
  ready: "bg-emerald-500",
  completed: "bg-green-600",
  cancelled: "bg-red-500",
  on_hold: "bg-orange-500",
  approved: "bg-teal-500",
  generated: "bg-violet-500",
};

const nodeTypeColors: Record<string, string> = {
  product: "#3b82f6",
  set: "#a855f7",
  formatka: "#f59e0b",
  bom: "#22c55e",
  order: "#f43f5e",
  zlp: "#06b6d4",
};

const nodeColors: Record<string, { bg: string; border: string; text: string }> = {
  product: { bg: "bg-blue-500/20", border: "border-blue-500", text: "text-blue-700 dark:text-blue-300" },
  set: { bg: "bg-purple-500/20", border: "border-purple-500", text: "text-purple-700 dark:text-purple-300" },
  formatka: { bg: "bg-amber-500/20", border: "border-amber-500", text: "text-amber-700 dark:text-amber-300" },
  bom: { bg: "bg-green-500/20", border: "border-green-500", text: "text-green-700 dark:text-green-300" },
  order: { bg: "bg-rose-500/20", border: "border-rose-500", text: "text-rose-700 dark:text-rose-300" },
  zlp: { bg: "bg-cyan-500/20", border: "border-cyan-500", text: "text-cyan-700 dark:text-cyan-300" },
};

const nodeIcons: Record<string, typeof Package> = {
  product: Package,
  set: Boxes,
  formatka: Layers,
  bom: FileBox,
  order: ShoppingCart,
  zlp: ClipboardList,
};

function StatCard({
  title,
  value,
  icon: Icon,
  description,
  trend,
  color = "text-primary",
}: {
  title: string;
  value: number | string;
  icon: typeof Package;
  description?: string;
  trend?: { value: number; label: string };
  color?: string;
}) {
  return (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2 gap-2">
        <CardTitle className="text-sm font-medium">{title}</CardTitle>
        <Icon className={`h-4 w-4 ${color}`} />
      </CardHeader>
      <CardContent>
        <div className="text-2xl font-bold">{value}</div>
        {description && (
          <p className="text-xs text-muted-foreground mt-1">{description}</p>
        )}
        {trend && (
          <div className="flex items-center gap-1 mt-1">
            <TrendingUp className="h-3 w-3 text-green-500" />
            <span className="text-xs text-green-500">+{trend.value}%</span>
            <span className="text-xs text-muted-foreground">{trend.label}</span>
          </div>
        )}
      </CardContent>
    </Card>
  );
}

function StatusDistribution({ data }: { data: ZlpStatusSummary[] }) {
  const total = data.reduce((sum, item) => sum + item.count, 0);

  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base flex items-center gap-2">
          <BarChart3 className="h-4 w-4" />
          Rozkład statusów ZLP
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-3">
        {data.map((item) => {
          const percentage = total > 0 ? (item.count / total) * 100 : 0;
          const Icon = statusIcons[item.status] || Clock;
          return (
            <div key={item.status} className="space-y-1">
              <div className="flex items-center justify-between text-sm">
                <div className="flex items-center gap-2">
                  <Icon className="h-3.5 w-3.5 text-muted-foreground" />
                  <span>{statusLabels[item.status] || item.status}</span>
                </div>
                <div className="flex items-center gap-2">
                  <Badge variant="secondary" className="text-xs">
                    {item.count} ZLP
                  </Badge>
                  <span className="text-muted-foreground text-xs w-12 text-right">
                    {percentage.toFixed(1)}%
                  </span>
                </div>
              </div>
              <div className="h-2 bg-muted rounded-full overflow-hidden">
                <div
                  className={`h-full ${statusColors[item.status] || "bg-slate-500"} transition-all`}
                  style={{ width: `${percentage}%` }}
                />
              </div>
            </div>
          );
        })}
      </CardContent>
    </Card>
  );
}

function PlansOverview({ plans }: { plans: PlanSummary[] }) {
  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base flex items-center gap-2">
          <Factory className="h-4 w-4" />
          Plany produkcyjne
        </CardTitle>
        <CardDescription>Ostatnie aktywne plany</CardDescription>
      </CardHeader>
      <CardContent>
        <ScrollArea className="h-[280px]">
          <div className="space-y-2">
            {plans.map((plan) => {
              const progress = plan.itemCount > 0 
                ? Math.round((plan.completedCount / plan.itemCount) * 100) 
                : 0;
              return (
                <div
                  key={plan.id}
                  className="p-3 rounded-lg border bg-card hover-elevate"
                >
                  <div className="flex items-start justify-between gap-2">
                    <div className="min-w-0 flex-1">
                      <div className="font-medium text-sm truncate">{plan.name}</div>
                      <div className="text-xs text-muted-foreground mt-0.5">
                        {plan.zlpCount} ZLP · {plan.itemCount} pozycji
                      </div>
                    </div>
                    <Badge
                      variant="secondary"
                      className={`text-xs ${statusColors[plan.status]} text-white`}
                    >
                      {statusLabels[plan.status] || plan.status}
                    </Badge>
                  </div>
                  <div className="mt-2">
                    <div className="flex items-center justify-between text-xs mb-1">
                      <span className="text-muted-foreground">Postęp</span>
                      <span className="font-medium">{progress}%</span>
                    </div>
                    <div className="h-1.5 bg-muted rounded-full overflow-hidden">
                      <div
                        className="h-full bg-primary transition-all"
                        style={{ width: `${progress}%` }}
                      />
                    </div>
                  </div>
                </div>
              );
            })}
            {plans.length === 0 && (
              <div className="text-center py-8 text-muted-foreground text-sm">
                Brak aktywnych planów
              </div>
            )}
          </div>
        </ScrollArea>
      </CardContent>
    </Card>
  );
}

function WorkCentersOverview({ data }: { data: WorkCenterStats[] }) {
  const safeData = data || [];
  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base flex items-center gap-2">
          <Wrench className="h-4 w-4" />
          Stanowiska robocze
        </CardTitle>
        <CardDescription>Obciążenie stanowisk</CardDescription>
      </CardHeader>
      <CardContent>
        <ScrollArea className="h-[280px]">
          <div className="space-y-3">
            {safeData.map((wc) => {
              const total = wc.totalOperations;
              const completedPct = total > 0 ? (wc.completedOperations / total) * 100 : 0;
              const inProgressPct = total > 0 ? (wc.inProgressOperations / total) * 100 : 0;
              return (
                <div key={wc.id} className="space-y-2">
                  <div className="flex items-center justify-between">
                    <div>
                      <div className="font-medium text-sm">{wc.name}</div>
                      <div className="text-xs text-muted-foreground">{wc.code}</div>
                    </div>
                    <div className="text-right">
                      <div className="text-sm font-medium">{wc.totalOperations}</div>
                      <div className="text-xs text-muted-foreground">operacji</div>
                    </div>
                  </div>
                  <div className="flex h-2 gap-0.5 rounded-full overflow-hidden bg-muted">
                    <div 
                      className="bg-green-500 transition-all" 
                      style={{ width: `${completedPct}%` }} 
                    />
                    <div 
                      className="bg-blue-500 transition-all" 
                      style={{ width: `${inProgressPct}%` }} 
                    />
                  </div>
                  <div className="flex justify-between text-xs text-muted-foreground">
                    <span className="flex items-center gap-1">
                      <span className="w-2 h-2 rounded-full bg-green-500" />
                      {wc.completedOperations} gotowe
                    </span>
                    <span className="flex items-center gap-1">
                      <span className="w-2 h-2 rounded-full bg-blue-500" />
                      {wc.inProgressOperations} w trakcie
                    </span>
                  </div>
                </div>
              );
            })}
            {safeData.length === 0 && (
              <div className="text-center py-8 text-muted-foreground text-sm">
                Brak danych stanowisk
              </div>
            )}
          </div>
        </ScrollArea>
      </CardContent>
    </Card>
  );
}

function DailyStatsChart({ data }: { data: DailyStats[] }) {
  if (!data || data.length === 0) {
    return (
      <Card>
        <CardHeader>
          <CardTitle className="text-base flex items-center gap-2">
            <Calendar className="h-4 w-4" />
            Aktywność dzienna
          </CardTitle>
          <CardDescription>Ostatnie 14 dni</CardDescription>
        </CardHeader>
        <CardContent className="flex items-center justify-center h-[220px] text-muted-foreground text-sm">
          Brak danych aktywności
        </CardContent>
      </Card>
    );
  }
  
  const allValues = data.flatMap(d => [d.createdZlp || 0, d.completedZlp || 0]);
  const maxValue = allValues.length > 0 ? Math.max(...allValues, 1) : 1;
  
  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base flex items-center gap-2">
          <Calendar className="h-4 w-4" />
          Aktywność dzienna
        </CardTitle>
        <CardDescription>Ostatnie 14 dni</CardDescription>
      </CardHeader>
      <CardContent>
        <div className="flex items-end gap-1 h-[180px]">
          {data.map((day, i) => {
            const createdHeight = (day.createdZlp / maxValue) * 100;
            const completedHeight = (day.completedZlp / maxValue) * 100;
            const dayLabel = format(new Date(day.date), "dd", { locale: pl });
            return (
              <div key={i} className="flex-1 flex flex-col items-center gap-1">
                <div className="flex-1 w-full flex items-end gap-0.5">
                  <div 
                    className="flex-1 bg-blue-500/80 rounded-t transition-all"
                    style={{ height: `${createdHeight}%`, minHeight: day.createdZlp > 0 ? 4 : 0 }}
                    title={`Utworzone: ${day.createdZlp}`}
                  />
                  <div 
                    className="flex-1 bg-green-500/80 rounded-t transition-all"
                    style={{ height: `${completedHeight}%`, minHeight: day.completedZlp > 0 ? 4 : 0 }}
                    title={`Zakończone: ${day.completedZlp}`}
                  />
                </div>
                <span className="text-[10px] text-muted-foreground">{dayLabel}</span>
              </div>
            );
          })}
        </div>
        <div className="flex justify-center gap-6 mt-3 text-xs text-muted-foreground">
          <span className="flex items-center gap-1">
            <span className="w-3 h-3 rounded bg-blue-500/80" />
            Utworzone
          </span>
          <span className="flex items-center gap-1">
            <span className="w-3 h-3 rounded bg-green-500/80" />
            Zakończone
          </span>
        </div>
      </CardContent>
    </Card>
  );
}

function TopProductsList({ data }: { data: TopProduct[] }) {
  const safeData = data || [];
  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base flex items-center gap-2">
          <Star className="h-4 w-4" />
          Top produkty
        </CardTitle>
        <CardDescription>Najczęściej produkowane</CardDescription>
      </CardHeader>
      <CardContent>
        <ScrollArea className="h-[280px]">
          <div className="space-y-2">
            {safeData.map((product, i) => (
              <div 
                key={product.id}
                className="flex items-center gap-3 p-2 rounded-lg border bg-card hover-elevate"
              >
                <div className="w-6 h-6 rounded-full bg-primary/10 flex items-center justify-center text-xs font-medium text-primary">
                  {i + 1}
                </div>
                <div className="min-w-0 flex-1">
                  <div className="font-medium text-sm truncate">{product.name}</div>
                  <div className="text-xs text-muted-foreground">{product.sku || "—"}</div>
                </div>
                <div className="text-right">
                  <div className="font-medium text-sm">{product.zlpCount || 0}</div>
                  <div className="text-xs text-muted-foreground">ZLP</div>
                </div>
              </div>
            ))}
            {safeData.length === 0 && (
              <div className="text-center py-8 text-muted-foreground text-sm">
                Brak danych produktów
              </div>
            )}
          </div>
        </ScrollArea>
      </CardContent>
    </Card>
  );
}

function RecentZlpsList({ data }: { data: RecentZlp[] }) {
  const safeData = data || [];
  
  const formatDate = (dateStr: string) => {
    try {
      const date = new Date(dateStr);
      if (isNaN(date.getTime())) return "—";
      return format(date, "dd MMM yyyy", { locale: pl });
    } catch {
      return "—";
    }
  };
  
  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base flex items-center gap-2">
          <Activity className="h-4 w-4" />
          Ostatnie ZLP
        </CardTitle>
        <CardDescription>Najnowsze zlecenia produkcyjne</CardDescription>
      </CardHeader>
      <CardContent>
        <ScrollArea className="h-[280px]">
          <div className="space-y-2">
            {safeData.map((zlp) => (
              <div 
                key={zlp.id}
                className="flex items-center gap-3 p-2 rounded-lg border bg-card hover-elevate"
              >
                <div className="min-w-0 flex-1">
                  <div className="font-medium text-sm truncate">{zlp.name}</div>
                  <div className="text-xs text-muted-foreground">
                    {zlp.itemCount || 0} pozycji · {formatDate(zlp.createdAt)}
                  </div>
                </div>
                <Badge
                  variant="secondary"
                  className={`text-xs ${statusColors[zlp.status] || "bg-slate-500"} text-white`}
                >
                  {statusLabels[zlp.status] || zlp.status}
                </Badge>
              </div>
            ))}
            {safeData.length === 0 && (
              <div className="text-center py-8 text-muted-foreground text-sm">
                Brak zleceń
              </div>
            )}
          </div>
        </ScrollArea>
      </CardContent>
    </Card>
  );
}

function BomFormatkiStats({ bomStats, formatkiStats }: { 
  bomStats: AnalyticsDashboard['bomStats']; 
  formatkiStats: AnalyticsDashboard['formatkiStats'];
}) {
  const safeBom = bomStats || { totalBoms: 0, bomsWithFormatki: 0, averageFormatkiPerBom: 0 };
  const safeFormatki = formatkiStats || { totalFormatki: 0, uniqueFormatki: 0, formatkiInProduction: 0 };
  
  return (
    <Card>
      <CardHeader>
        <CardTitle className="text-base flex items-center gap-2">
          <Component className="h-4 w-4" />
          Struktura produktów
        </CardTitle>
        <CardDescription>BOM i formatki</CardDescription>
      </CardHeader>
      <CardContent className="space-y-4">
        <div className="grid grid-cols-2 gap-4">
          <div className="space-y-1">
            <div className="text-2xl font-bold">{safeBom.totalBoms || 0}</div>
            <div className="text-xs text-muted-foreground">Wszystkie BOM</div>
          </div>
          <div className="space-y-1">
            <div className="text-2xl font-bold">{safeBom.bomsWithFormatki || 0}</div>
            <div className="text-xs text-muted-foreground">BOM z formatkami</div>
          </div>
        </div>
        <div className="pt-2 border-t space-y-3">
          <div className="flex justify-between items-center">
            <span className="text-sm text-muted-foreground">Średnio formatek/BOM</span>
            <span className="font-medium">{(safeBom.averageFormatkiPerBom || 0).toFixed(1)}</span>
          </div>
          <div className="flex justify-between items-center">
            <span className="text-sm text-muted-foreground">Wszystkie formatki</span>
            <span className="font-medium">{(safeFormatki.totalFormatki || 0).toLocaleString()}</span>
          </div>
          <div className="flex justify-between items-center">
            <span className="text-sm text-muted-foreground">Używane w BOM</span>
            <span className="font-medium">{safeFormatki.uniqueFormatki || 0}</span>
          </div>
          <div className="flex justify-between items-center">
            <span className="text-sm text-muted-foreground">W aktywnej produkcji</span>
            <Badge variant="secondary" className="bg-blue-500 text-white">
              {safeFormatki.formatkiInProduction || 0}
            </Badge>
          </div>
        </div>
      </CardContent>
    </Card>
  );
}

type LayoutType = "horizontal" | "vertical" | "radial" | "grid";

const layoutConfigs: Record<LayoutType, any> = {
  horizontal: {
    name: "dagre",
    rankDir: "LR",
    nodeSep: 120,
    rankSep: 200,
    ranker: "tight-tree",
    animate: false,
  },
  vertical: {
    name: "dagre",
    rankDir: "TB",
    nodeSep: 80,
    rankSep: 120,
    ranker: "tight-tree",
    animate: false,
  },
  radial: {
    name: "concentric",
    concentric: (node: any) => {
      const type = node.data("type");
      if (type === "order") return 5;
      if (type === "zlp") return 4;
      if (type === "product" || type === "set") return 3;
      if (type === "bom") return 2;
      return 1;
    },
    levelWidth: () => 3,
    minNodeSpacing: 80,
    animate: false,
  },
  grid: {
    name: "grid",
    rows: undefined,
    cols: undefined,
    fit: true,
    padding: 50,
    avoidOverlap: true,
    avoidOverlapPadding: 30,
    nodeDimensionsIncludeLabels: true,
    spacingFactor: 1.5,
    animate: false,
  },
};

function ProductConnectionGraph() {
  const [searchQuery, setSearchQuery] = useState("");
  const [debouncedSearch, setDebouncedSearch] = useState("");
  const [nodeTypeFilter, setNodeTypeFilter] = useState<string>("all");
  const [layoutType, setLayoutType] = useState<LayoutType>("horizontal");
  const [selectedNode, setSelectedNode] = useState<CytoscapeNode["data"] | null>(null);
  const containerRef = useRef<HTMLDivElement>(null);
  const cyRef = useRef<Core | null>(null);

  useEffect(() => {
    const timer = setTimeout(() => {
      setDebouncedSearch(searchQuery);
    }, 500);
    return () => clearTimeout(timer);
  }, [searchQuery]);

  const { data: graphData, isLoading, refetch } = useQuery<CytoscapeGraph>({
    queryKey: ["/api/production/analytics/connection-graph", { search: debouncedSearch, type: nodeTypeFilter }],
    enabled: true,
  });

  const initCytoscape = useCallback(() => {
    if (!containerRef.current || !graphData) return;

    if (cyRef.current) {
      cyRef.current.destroy();
    }

    const isDark = document.documentElement.classList.contains("dark");
    const textColor = isDark ? "#e2e8f0" : "#1e293b";
    const bgColor = isDark ? "#020817" : "#ffffff";

    const cy = cytoscape({
      container: containerRef.current,
      elements: [...graphData.nodes, ...graphData.edges],
      style: [
        {
          selector: "node",
          style: {
            "background-color": (ele: NodeSingular) => nodeTypeColors[ele.data("type")] || "#64748b",
            label: "data(label)",
            color: textColor,
            "text-valign": "bottom",
            "text-halign": "center",
            "font-size": "10px",
            "text-margin-y": 5,
            width: 40,
            height: 40,
            "border-width": 2,
            "border-color": (ele: NodeSingular) => nodeTypeColors[ele.data("type")] || "#64748b",
            "text-wrap": "ellipsis",
            "text-max-width": "100px",
          },
        },
        {
          selector: "node:selected",
          style: {
            "border-width": 4,
            "border-color": "#f97316",
            "background-color": "#f97316",
          },
        },
        {
          selector: "edge",
          style: {
            width: 1.5,
            "line-color": isDark ? "#475569" : "#94a3b8",
            "target-arrow-color": isDark ? "#475569" : "#94a3b8",
            "target-arrow-shape": "triangle",
            "curve-style": "bezier",
            label: "data(label)",
            "font-size": "8px",
            color: isDark ? "#94a3b8" : "#64748b",
            "text-rotation": "autorotate",
            "text-margin-y": -10,
          },
        },
        {
          selector: "edge:selected",
          style: {
            width: 2.5,
            "line-color": "#f97316",
            "target-arrow-color": "#f97316",
          },
        },
        {
          selector: "node.faded",
          style: {
            opacity: 0.15,
            "text-opacity": 0.15,
          },
        },
        {
          selector: "edge.faded",
          style: {
            opacity: 0.1,
          },
        },
        {
          selector: "node.highlighted",
          style: {
            "border-width": 3,
            "border-color": "#f97316",
            "z-index": 10,
          },
        },
        {
          selector: "edge.highlighted",
          style: {
            width: 2.5,
            "line-color": "#f97316",
            "target-arrow-color": "#f97316",
            opacity: 1,
            "z-index": 10,
          },
        },
      ],
      layout: layoutConfigs[layoutType] as any,
      minZoom: 0.1,
      maxZoom: 3,
      wheelSensitivity: 0.3,
    });

    cy.on("tap", "node", (evt) => {
      const node = evt.target;
      setSelectedNode(node.data());
      
      // Reset all styles first
      cy.elements().removeClass("faded highlighted");
      
      // Get connected nodes and edges
      const neighborhood = node.neighborhood();
      const connectedNodes = neighborhood.nodes();
      const connectedEdges = neighborhood.edges();
      
      // Fade all elements
      cy.elements().addClass("faded");
      
      // Highlight clicked node and its neighbors
      node.removeClass("faded").addClass("highlighted");
      connectedNodes.removeClass("faded").addClass("highlighted");
      connectedEdges.removeClass("faded").addClass("highlighted");
    });

    cy.on("tap", (evt) => {
      if (evt.target === cy) {
        setSelectedNode(null);
        // Reset all styles when clicking on background
        cy.elements().removeClass("faded highlighted");
      }
    });

    cyRef.current = cy;
  }, [graphData, layoutType]);

  useEffect(() => {
    initCytoscape();
    return () => {
      if (cyRef.current) {
        cyRef.current.destroy();
        cyRef.current = null;
      }
    };
  }, [initCytoscape]);

  const handleLayoutChange = (newLayout: LayoutType) => {
    setLayoutType(newLayout);
    if (cyRef.current) {
      const layout = cyRef.current.layout(layoutConfigs[newLayout]);
      layout.run();
    }
  };

  const handleZoomIn = () => {
    if (cyRef.current) {
      cyRef.current.zoom(cyRef.current.zoom() * 1.2);
    }
  };

  const handleZoomOut = () => {
    if (cyRef.current) {
      cyRef.current.zoom(cyRef.current.zoom() / 1.2);
    }
  };

  const handleFit = () => {
    if (cyRef.current) {
      cyRef.current.fit(undefined, 50);
    }
  };

  if (isLoading) {
    return (
      <div className="space-y-4">
        <div className="flex items-center gap-4">
          <Skeleton className="h-10 flex-1" />
          <Skeleton className="h-10 w-40" />
          <Skeleton className="h-10 w-10" />
        </div>
        <Skeleton className="h-[600px] w-full" />
      </div>
    );
  }

  return (
    <div className="space-y-4">
      <div className="flex items-center gap-4 flex-wrap">
        <div className="relative flex-1 min-w-[200px]">
          <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
          <Input
            placeholder="Szukaj produktu, SKU, zestawu..."
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            className="pl-9"
            data-testid="input-graph-search"
          />
        </div>
        <Select value={nodeTypeFilter} onValueChange={setNodeTypeFilter}>
          <SelectTrigger className="w-[180px]" data-testid="select-node-type">
            <SelectValue placeholder="Typ węzła" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="all">Wszystkie typy</SelectItem>
            <SelectItem value="product">Produkty</SelectItem>
            <SelectItem value="set">Zestawy</SelectItem>
            <SelectItem value="formatka">Formatki</SelectItem>
            <SelectItem value="bom">BOM</SelectItem>
            <SelectItem value="zlp">ZLP</SelectItem>
          </SelectContent>
        </Select>
        <Select value={layoutType} onValueChange={(v) => handleLayoutChange(v as LayoutType)}>
          <SelectTrigger className="w-[160px]" data-testid="select-layout-type">
            <Layers className="h-4 w-4 mr-2" />
            <SelectValue placeholder="Układ" />
          </SelectTrigger>
          <SelectContent>
            <SelectItem value="horizontal">Horyzontalny</SelectItem>
            <SelectItem value="vertical">Wertykalny</SelectItem>
            <SelectItem value="radial">Radialny</SelectItem>
            <SelectItem value="grid">Siatka</SelectItem>
          </SelectContent>
        </Select>
        <Button
          variant="outline"
          size="icon"
          onClick={() => refetch()}
          data-testid="button-refresh-graph"
        >
          <RefreshCw className="h-4 w-4" />
        </Button>
      </div>

      {graphData?.stats && (
        <div className="flex gap-2 flex-wrap">
          <Badge variant="outline" className={`${nodeColors.product.bg} ${nodeColors.product.border} ${nodeColors.product.text}`}>
            <Package className="h-3 w-3 mr-1" />
            {graphData.stats.totalProducts} produktów
          </Badge>
          <Badge variant="outline" className={`${nodeColors.set.bg} ${nodeColors.set.border} ${nodeColors.set.text}`}>
            <Boxes className="h-3 w-3 mr-1" />
            {graphData.stats.totalSets} zestawów
          </Badge>
          <Badge variant="outline" className={`${nodeColors.bom.bg} ${nodeColors.bom.border} ${nodeColors.bom.text}`}>
            <FileBox className="h-3 w-3 mr-1" />
            {graphData.stats.totalBoms} BOM
          </Badge>
          <Badge variant="outline" className={`${nodeColors.formatka.bg} ${nodeColors.formatka.border} ${nodeColors.formatka.text}`}>
            <Layers className="h-3 w-3 mr-1" />
            {graphData.stats.totalFormatki} formatek
          </Badge>
          <Badge variant="outline" className={`${nodeColors.zlp.bg} ${nodeColors.zlp.border} ${nodeColors.zlp.text}`}>
            <ClipboardList className="h-3 w-3 mr-1" />
            {graphData.stats.totalZlps} ZLP
          </Badge>
        </div>
      )}

      <Card className="overflow-hidden">
        <div className="relative">
          <div
            ref={containerRef}
            className="h-[600px] w-full bg-background"
            data-testid="cytoscape-container"
          />
          <div className="absolute top-4 right-4 flex flex-col gap-1">
            <Button
              variant="secondary"
              size="icon"
              onClick={handleZoomIn}
              className="h-8 w-8"
              data-testid="button-zoom-in"
            >
              <ZoomIn className="h-4 w-4" />
            </Button>
            <Button
              variant="secondary"
              size="icon"
              onClick={handleZoomOut}
              className="h-8 w-8"
              data-testid="button-zoom-out"
            >
              <ZoomOut className="h-4 w-4" />
            </Button>
            <Button
              variant="secondary"
              size="icon"
              onClick={handleFit}
              className="h-8 w-8"
              data-testid="button-fit"
            >
              <Maximize2 className="h-4 w-4" />
            </Button>
          </div>
        </div>
      </Card>

      {selectedNode && (
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm flex items-center gap-2">
              {(() => {
                const Icon = nodeIcons[selectedNode.type] || Package;
                const colors = nodeColors[selectedNode.type];
                return (
                  <>
                    <div className={`p-1.5 rounded ${colors.bg} ${colors.border} border`}>
                      <Icon className={`h-4 w-4 ${colors.text}`} />
                    </div>
                    <span>Szczegóły węzła</span>
                  </>
                );
              })()}
            </CardTitle>
          </CardHeader>
          <CardContent>
            <div className="space-y-2">
              <div>
                <div className="text-sm font-medium">{selectedNode.label}</div>
                {selectedNode.sku && (
                  <div className="text-xs text-muted-foreground">SKU: {selectedNode.sku}</div>
                )}
                {selectedNode.color && (
                  <div className="text-xs text-muted-foreground">Kolor: {selectedNode.color}</div>
                )}
                {selectedNode.status && (
                  <Badge variant="secondary" className="text-xs mt-1">
                    {statusLabels[selectedNode.status] || selectedNode.status}
                  </Badge>
                )}
              </div>
              <div className="text-xs text-muted-foreground">
                ID: {selectedNode.id}
              </div>
            </div>
          </CardContent>
        </Card>
      )}

      {(!graphData || graphData.nodes.length === 0) && !isLoading && (
        <Card>
          <CardContent className="py-12 text-center">
            <Network className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
            <div className="text-lg font-medium mb-1">Brak danych do wyświetlenia</div>
            <p className="text-sm text-muted-foreground">
              Wyszukaj produkt lub zmień filtry aby zobaczyć mapę połączeń
            </p>
          </CardContent>
        </Card>
      )}
    </div>
  );
}

export default function ProductionAnalytics() {
  const [activeTab, setActiveTab] = useState("dashboard");

  const { data: dashboardData, isLoading } = useQuery<AnalyticsDashboard>({
    queryKey: ["/api/production/analytics/dashboard"],
  });

  return (
    <div className="w-full px-4 md:px-6 lg:px-8 py-6 space-y-6">
      <div className="flex flex-col sm:flex-row sm:items-center justify-between gap-4">
        <div>
          <h1 className="text-xl md:text-2xl font-bold flex items-center gap-2">
            <BarChart3 className="h-5 w-5 md:h-6 md:w-6" />
            Analityka Produkcji
          </h1>
          <p className="text-sm text-muted-foreground">
            Podsumowania, statystyki i mapa połączeń produktów
          </p>
        </div>
      </div>

      <Tabs value={activeTab} onValueChange={setActiveTab}>
        <TabsList className="w-full sm:w-auto">
          <TabsTrigger value="dashboard" className="gap-2 flex-1 sm:flex-none" data-testid="tab-dashboard">
            <BarChart3 className="h-4 w-4" />
            <span className="hidden sm:inline">Dashboard</span>
            <span className="sm:hidden">Dash</span>
          </TabsTrigger>
          <TabsTrigger value="connections" className="gap-2 flex-1 sm:flex-none" data-testid="tab-connections">
            <Network className="h-4 w-4" />
            <span className="hidden sm:inline">Mapa połączeń</span>
            <span className="sm:hidden">Mapa</span>
          </TabsTrigger>
        </TabsList>

        <TabsContent value="dashboard" className="space-y-6 mt-6">
          {isLoading ? (
            <div className="space-y-6">
              <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-3 md:gap-4">
                {[...Array(6)].map((_, i) => (
                  <Skeleton key={i} className="h-28" />
                ))}
              </div>
              <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4 md:gap-6">
                {[...Array(6)].map((_, i) => (
                  <Skeleton key={i} className="h-[340px]" />
                ))}
              </div>
            </div>
          ) : dashboardData ? (
            <>
              {/* Top Stats Row - 6 columns on large screens */}
              <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-6 gap-3 md:gap-4">
                <StatCard
                  title="Wszystkie ZLP"
                  value={dashboardData.totalZlp}
                  icon={ClipboardList}
                  description="Łączna liczba"
                  color="text-cyan-500"
                />
                <StatCard
                  title="Pozycje"
                  value={dashboardData.totalItems}
                  icon={Package}
                  description={`${dashboardData.completedItems} ukończ.`}
                  color="text-blue-500"
                />
                <StatCard
                  title="W trakcie"
                  value={dashboardData.inProgressItems}
                  icon={Factory}
                  description="Realizowane"
                  color="text-amber-500"
                />
                <StatCard
                  title="Oczekujące"
                  value={dashboardData.pendingItems}
                  icon={Clock}
                  description="Do realizacji"
                  color="text-slate-500"
                />
                <StatCard
                  title="BOM"
                  value={dashboardData.bomStats.totalBoms}
                  icon={FileBox}
                  description={`${dashboardData.bomStats.bomsWithFormatki} z form.`}
                  color="text-purple-500"
                />
                <StatCard
                  title="Formatki"
                  value={dashboardData.formatkiStats.totalFormatki}
                  icon={Layers}
                  description={`${dashboardData.formatkiStats.formatkiInProduction} aktyw.`}
                  color="text-teal-500"
                />
              </div>

              {/* Charts and Details - responsive grid */}
              <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4 md:gap-6">
                <StatusDistribution data={dashboardData.zlpByStatus} />
                <DailyStatsChart data={dashboardData.dailyStats} />
                <BomFormatkiStats 
                  bomStats={dashboardData.bomStats} 
                  formatkiStats={dashboardData.formatkiStats} 
                />
              </div>

              {/* Bottom Row - 4 columns on XL */}
              <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-4 gap-4 md:gap-6">
                <PlansOverview plans={dashboardData.plans} />
                <WorkCentersOverview data={dashboardData.workCenterStats} />
                <TopProductsList data={dashboardData.topProducts} />
                <RecentZlpsList data={dashboardData.recentZlps} />
              </div>
            </>
          ) : (
            <Card>
              <CardContent className="py-12 text-center">
                <BarChart3 className="h-12 w-12 mx-auto text-muted-foreground mb-4" />
                <div className="text-lg font-medium mb-1">Brak danych</div>
                <p className="text-sm text-muted-foreground">
                  Dane analityczne są niedostępne
                </p>
              </CardContent>
            </Card>
          )}
        </TabsContent>

        <TabsContent value="connections" className="mt-6">
          <ProductConnectionGraph />
        </TabsContent>
      </Tabs>
    </div>
  );
}
