import { useMemo, useEffect } from "react";
import { useQuery } from "@tanstack/react-query";
import { useLocation } from "wouter";
import { Search, ChevronLeft, ChevronRight, RotateCcw, Calendar as CalendarIcon } from "lucide-react";
import { cn } from "@/lib/utils";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Popover, PopoverContent, PopoverTrigger } from "@/components/ui/popover";
import { Calendar } from "@/components/ui/calendar";
import { Badge } from "@/components/ui/badge";
import { OrdersTable } from "@/components/orders-table";
import { PaymentSummaryBar } from "@/components/payment-summary-bar";
import { ThemeToggle } from "@/components/theme-toggle";
import { useDashboardSettings } from "@/hooks/use-dashboard-settings";
import type { AllegroOrder } from "@shared/schema";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface FilterButtonGroupProps {
  label: string;
  value: string;
  options: Array<{ value: string; label: string; testId: string }>;
  onChange: (value: string) => void;
}

function FilterButtonGroup({ label, value, options, onChange }: FilterButtonGroupProps) {
  return (
    <div className="flex items-center gap-2">
      <span className="text-sm text-muted-foreground">{label}:</span>
      {options.map(opt => (
        <Button
          key={opt.value}
          variant={value === opt.value ? "default" : "outline"}
          size="sm"
          onClick={() => onChange(opt.value)}
          data-testid={opt.testId}
        >
          {opt.label}
        </Button>
      ))}
    </div>
  );
}

export default function Orders() {
  const { settings, updateSettings, resetSettings } = useDashboardSettings();
  const [location, setLocation] = useLocation();

  const {
    searchQuery,
    paymentFilter,
    fulfillmentFilter,
    invoiceFilter,
    paymentTypeFilter,
    sourceFilter,
    dateRangeType,
    customDateFrom,
    customDateTo,
    currentPage,
    itemsPerPage,
    sortBy,
    sortOrder,
  } = settings;

  // Odczytaj filtry z URL przy montowaniu
  useEffect(() => {
    const params = new URLSearchParams(window.location.search);
    const updates: any = {};
    
    if (params.has('search')) updates.searchQuery = params.get('search') || '';
    if (params.has('source')) updates.sourceFilter = params.get('source') || 'all';
    if (params.has('dateRange')) updates.dateRangeType = params.get('dateRange') || 'all';
    if (params.has('dateFrom')) updates.customDateFrom = params.get('dateFrom');
    if (params.has('dateTo')) updates.customDateTo = params.get('dateTo');
    if (params.has('payment')) updates.paymentFilter = params.get('payment') || 'all';
    if (params.has('fulfillment')) updates.fulfillmentFilter = params.get('fulfillment') || 'all';
    if (params.has('invoice')) updates.invoiceFilter = params.get('invoice') || 'all';
    if (params.has('paymentType')) updates.paymentTypeFilter = params.get('paymentType') || 'all';
    if (params.has('page')) updates.currentPage = parseInt(params.get('page') || '1');
    if (params.has('perPage')) updates.itemsPerPage = parseInt(params.get('perPage') || '25');
    if (params.has('sortBy')) updates.sortBy = params.get('sortBy') || 'order_number';
    if (params.has('sortOrder')) updates.sortOrder = (params.get('sortOrder') === 'ASC' ? 'ASC' : 'DESC') as 'ASC' | 'DESC';
    
    if (Object.keys(updates).length > 0) {
      updateSettings(updates);
    }
  }, []);

  // Aktualizuj URL przy każdej zmianie filtrów
  useEffect(() => {
    const params = new URLSearchParams();
    
    if (searchQuery) params.set('search', searchQuery);
    if (sourceFilter !== 'all') params.set('source', sourceFilter);
    if (dateRangeType !== 'all') params.set('dateRange', dateRangeType);
    if (dateRangeType === 'custom' && customDateFrom) params.set('dateFrom', customDateFrom);
    if (dateRangeType === 'custom' && customDateTo) params.set('dateTo', customDateTo);
    if (paymentFilter !== 'all') params.set('payment', paymentFilter);
    if (fulfillmentFilter !== 'all') params.set('fulfillment', fulfillmentFilter);
    if (invoiceFilter !== 'all') params.set('invoice', invoiceFilter);
    if (paymentTypeFilter !== 'all') params.set('paymentType', paymentTypeFilter);
    if (currentPage !== 1) params.set('page', currentPage.toString());
    if (itemsPerPage !== 25) params.set('perPage', itemsPerPage.toString());
    if (sortBy !== 'order_number') params.set('sortBy', sortBy);
    if (sortOrder !== 'DESC') params.set('sortOrder', sortOrder);
    
    const newSearch = params.toString();
    const newUrl = newSearch ? `/orders?${newSearch}` : '/orders';
    
    if (window.location.pathname + window.location.search !== newUrl) {
      window.history.pushState({}, '', newUrl);
    }
  }, [
    searchQuery,
    sourceFilter,
    dateRangeType,
    customDateFrom,
    customDateTo,
    paymentFilter,
    fulfillmentFilter,
    invoiceFilter,
    paymentTypeFilter,
    currentPage,
    itemsPerPage,
    sortBy,
    sortOrder,
  ]);

  const hasAnyFilters = useMemo(
    () =>
      searchQuery !== "" ||
      paymentFilter !== "all" ||
      fulfillmentFilter !== "all" ||
      invoiceFilter !== "all" ||
      paymentTypeFilter !== "all" ||
      sourceFilter !== "all" ||
      dateRangeType !== "all",
    [searchQuery, paymentFilter, fulfillmentFilter, invoiceFilter, paymentTypeFilter, sourceFilter, dateRangeType]
  );

  const buildOrdersUrl = () => {
    const params = new URLSearchParams();
    params.set('page', currentPage.toString());
    params.set('limit', itemsPerPage.toString());
    params.set('sortBy', sortBy);
    params.set('sortOrder', sortOrder);

    if (searchQuery) {
      params.set('search', searchQuery);
    }
    if (sourceFilter && sourceFilter !== 'all') {
      params.set('sourceFilter', sourceFilter);
    }
    if (dateRangeType && dateRangeType !== 'all') {
      params.set('dateRangeType', dateRangeType);
      if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
        params.set('customDateFrom', customDateFrom);
        params.set('customDateTo', customDateTo);
      }
    }
    if (paymentFilter && paymentFilter !== 'all') {
      params.set('paymentFilter', paymentFilter);
    }
    if (fulfillmentFilter && fulfillmentFilter !== 'all') {
      params.set('fulfillmentFilter', fulfillmentFilter);
    }
    if (invoiceFilter && invoiceFilter !== 'all') {
      params.set('invoiceFilter', invoiceFilter);
    }
    if (paymentTypeFilter && paymentTypeFilter !== 'all') {
      params.set('paymentTypeFilter', paymentTypeFilter);
    }
    return `/api/orders?${params.toString()}`;
  };

  const { data: orders = [], isLoading } = useQuery<AllegroOrder[]>({
    queryKey: [buildOrdersUrl()],
  });

  const buildCountUrl = () => {
    const params = new URLSearchParams();
    if (searchQuery) {
      params.set('search', searchQuery);
    }
    if (sourceFilter && sourceFilter !== 'all') {
      params.set('sourceFilter', sourceFilter);
    }
    if (dateRangeType && dateRangeType !== 'all') {
      params.set('dateRangeType', dateRangeType);
      if (dateRangeType === 'custom' && customDateFrom && customDateTo) {
        params.set('customDateFrom', customDateFrom);
        params.set('customDateTo', customDateTo);
      }
    }
    if (paymentFilter && paymentFilter !== 'all') {
      params.set('paymentFilter', paymentFilter);
    }
    if (fulfillmentFilter && fulfillmentFilter !== 'all') {
      params.set('fulfillmentFilter', fulfillmentFilter);
    }
    if (invoiceFilter && invoiceFilter !== 'all') {
      params.set('invoiceFilter', invoiceFilter);
    }
    if (paymentTypeFilter && paymentTypeFilter !== 'all') {
      params.set('paymentTypeFilter', paymentTypeFilter);
    }
    return `/api/orders/count?${params.toString()}`;
  };

  const { data: totalCountData } = useQuery<{ count: number }>({
    queryKey: [buildCountUrl()],
  });

  const { data: statistics } = useQuery<{
    totalOrders: number;
    paidOrders: number;
    unpaidOrders: number;
    totalRevenue: number;
  }>({
    queryKey: ["/api/orders/statistics"],
    refetchInterval: 5000,
  });

  const totalCount = totalCountData?.count || 0;
  const totalPages = Math.ceil(totalCount / itemsPerPage);

  // Używamy globalnych statystyk (nie filtrowanych)
  const totalOrders = statistics?.totalOrders || 0;

  return (
    <div className="flex-1 overflow-y-auto">
      <div className="w-full max-w-full px-4 md:px-6 py-6 md:py-8">
        <div className="flex flex-col md:flex-row items-start md:items-center justify-between gap-4 mb-6">
          <div className="flex flex-col sm:flex-row items-start sm:items-center gap-3 flex-wrap w-full md:w-auto">
            <h1 className="text-2xl font-semibold tracking-tight">Zamówienia</h1>
          </div>
          <ThemeToggle />
        </div>

        <div className="mt-6 max-w-full overflow-hidden">
          <div className="sticky top-0 z-10 bg-background/95 backdrop-blur supports-[backdrop-filter]:bg-background/60 pt-6 pb-4 space-y-4">
            {/* Search bar - full width on all devices */}
            <div className="relative w-full">
              <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
              <Input
                placeholder="Szukaj po ID, imię, nazwisko, email..."
                value={searchQuery}
                onChange={(e) => updateSettings({ searchQuery: e.target.value })}
                className="pl-9 w-full"
                data-testid="input-search-orders"
              />
            </div>

            {/* Filters container */}
            <div>
              {/* Desktop: Button groups */}
              <div className="hidden md:flex items-center gap-3 flex-wrap">
                <FilterButtonGroup
                  label="Źródło"
                  value={sourceFilter}
                  options={[
                    { value: "all", label: "Wszystkie", testId: "button-filter-source-all" },
                    { value: "ALLEGRO", label: "Allegro", testId: "button-filter-source-allegro" },
                    { value: "SHOPER", label: "Shoper", testId: "button-filter-source-shoper" },
                  ]}
                  onChange={(val) => updateSettings({ sourceFilter: val })}
                />

                <FilterButtonGroup
                  label="Płatność"
                  value={paymentFilter}
                  options={[
                    { value: "all", label: "Wszystkie", testId: "button-filter-payment-all" },
                    { value: "PAID", label: "Opłacone", testId: "button-filter-payment-paid" },
                    { value: "UNPAID", label: "Nieopłacone", testId: "button-filter-payment-unpaid" },
                  ]}
                  onChange={(val) => updateSettings({ paymentFilter: val })}
                />

                <div className="flex items-center gap-2 flex-wrap">
                  <span className="text-sm text-muted-foreground">Okres:</span>
                  {[
                    { value: "all", label: "Wszystkie" },
                    { value: "today", label: "Dzisiaj" },
                    { value: "yesterday", label: "Wczoraj" },
                    { value: "7days", label: "7 dni" },
                    { value: "30days", label: "30 dni" },
                  ].map(opt => (
                    <Button
                      key={opt.value}
                      variant={dateRangeType === opt.value ? "default" : "outline"}
                      size="sm"
                      onClick={() => updateSettings({ dateRangeType: opt.value })}
                      data-testid={`button-filter-date-${opt.value}`}
                    >
                      {opt.label}
                    </Button>
                  ))}
                  <Popover>
                    <PopoverTrigger asChild>
                      <Button
                        variant={dateRangeType === "custom" ? "default" : "outline"}
                        size="sm"
                        data-testid="button-filter-date-custom"
                      >
                        <CalendarIcon className="h-4 w-4 mr-2" />
                        {dateRangeType === "custom" && customDateFrom && customDateTo
                          ? `${format(new Date(customDateFrom), "dd.MM", { locale: pl })} - ${format(new Date(customDateTo), "dd.MM", { locale: pl })}`
                          : "Własny zakres"}
                      </Button>
                    </PopoverTrigger>
                    <PopoverContent className="w-auto p-0" align="start">
                      <div className="p-3 space-y-3">
                        <div className="space-y-2">
                          <div className="text-sm font-medium">Data od:</div>
                          <Calendar
                            mode="single"
                            selected={customDateFrom ? new Date(customDateFrom) : undefined}
                            onSelect={(date) => {
                              if (date) {
                                const dateStr = date.toISOString().split('T')[0];
                                updateSettings({ customDateFrom: dateStr, dateRangeType: 'custom' });
                              }
                            }}
                            locale={pl}
                            className="rounded-md border"
                          />
                        </div>
                        <div className="space-y-2">
                          <div className="text-sm font-medium">Data do:</div>
                          <Calendar
                            mode="single"
                            selected={customDateTo ? new Date(customDateTo) : undefined}
                            onSelect={(date) => {
                              if (date) {
                                const dateStr = date.toISOString().split('T')[0];
                                updateSettings({ customDateTo: dateStr, dateRangeType: 'custom' });
                              }
                            }}
                            locale={pl}
                            disabled={(date) => {
                              if (!customDateFrom) return false;
                              return date < new Date(customDateFrom);
                            }}
                            className="rounded-md border"
                          />
                        </div>
                      </div>
                    </PopoverContent>
                  </Popover>
                </div>

                <FilterButtonGroup
                  label="Realizacja"
                  value={fulfillmentFilter}
                  options={[
                    { value: "all", label: "Wszystkie", testId: "button-filter-fulfillment-all" },
                    { value: "NEW", label: "Nowe", testId: "button-filter-fulfillment-new" },
                    { value: "PROCESSING", label: "W real.", testId: "button-filter-fulfillment-processing" },
                    { value: "SENT", label: "Wysłane", testId: "button-filter-fulfillment-sent" },
                    { value: "CANCELLED", label: "Anulowane", testId: "button-filter-fulfillment-cancelled" },
                  ]}
                  onChange={(val) => updateSettings({ fulfillmentFilter: val })}
                />

                <FilterButtonGroup
                  label="Typ płatności"
                  value={paymentTypeFilter}
                  options={[
                    { value: "all", label: "Wszystkie", testId: "button-filter-payment-type-all" },
                    { value: "CASH_ON_DELIVERY", label: "Pobranie", testId: "button-filter-payment-type-cod" },
                    { value: "ONLINE", label: "Online", testId: "button-filter-payment-type-online" },
                  ]}
                  onChange={(val) => updateSettings({ paymentTypeFilter: val })}
                />

                <FilterButtonGroup
                  label="Faktura"
                  value={invoiceFilter}
                  options={[
                    { value: "all", label: "Wszystkie", testId: "button-filter-invoice-all" },
                    { value: "yes", label: "Z fakturą", testId: "button-filter-invoice-yes" },
                    { value: "no", label: "Bez faktury", testId: "button-filter-invoice-no" },
                  ]}
                  onChange={(val) => updateSettings({ invoiceFilter: val })}
                />
              </div>

              {/* Mobile: Select dropdowns */}
              <div className="flex md:hidden flex-col gap-2">
                <Select value={sourceFilter} onValueChange={(value) => updateSettings({ sourceFilter: value })}>
                  <SelectTrigger data-testid="select-source-filter-mobile">
                    <SelectValue placeholder="Źródło" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie</SelectItem>
                    <SelectItem value="ALLEGRO">Allegro</SelectItem>
                    <SelectItem value="SHOPER">Shoper</SelectItem>
                  </SelectContent>
                </Select>

                <Select value={paymentFilter} onValueChange={(value) => updateSettings({ paymentFilter: value })}>
                  <SelectTrigger data-testid="select-payment-filter-mobile">
                    <SelectValue placeholder="Płatność" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie</SelectItem>
                    <SelectItem value="PAID">Opłacone</SelectItem>
                    <SelectItem value="UNPAID">Nieopłacone</SelectItem>
                  </SelectContent>
                </Select>

                <Select value={dateRangeType} onValueChange={(value) => updateSettings({ dateRangeType: value })}>
                  <SelectTrigger data-testid="select-date-filter-mobile">
                    <SelectValue placeholder="Okres" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie</SelectItem>
                    <SelectItem value="today">Dzisiaj</SelectItem>
                    <SelectItem value="yesterday">Wczoraj</SelectItem>
                    <SelectItem value="7days">7 dni</SelectItem>
                    <SelectItem value="30days">30 dni</SelectItem>
                  </SelectContent>
                </Select>

                <Select value={fulfillmentFilter} onValueChange={(value) => updateSettings({ fulfillmentFilter: value })}>
                  <SelectTrigger data-testid="select-fulfillment-filter">
                    <SelectValue placeholder="Realizacja" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie</SelectItem>
                    <SelectItem value="NEW">Nowe</SelectItem>
                    <SelectItem value="PROCESSING">W realizacji</SelectItem>
                    <SelectItem value="SENT">Wysłane</SelectItem>
                    <SelectItem value="CANCELLED">Anulowane</SelectItem>
                  </SelectContent>
                </Select>

                <Select value={paymentTypeFilter} onValueChange={(value) => updateSettings({ paymentTypeFilter: value })}>
                  <SelectTrigger data-testid="select-payment-type-filter">
                    <SelectValue placeholder="Typ płatności" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie</SelectItem>
                    <SelectItem value="CASH_ON_DELIVERY">Pobranie</SelectItem>
                    <SelectItem value="ONLINE">Online</SelectItem>
                  </SelectContent>
                </Select>

                <Select value={invoiceFilter} onValueChange={(value) => updateSettings({ invoiceFilter: value })}>
                  <SelectTrigger data-testid="select-invoice-filter">
                    <SelectValue placeholder="Faktura" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="all">Wszystkie</SelectItem>
                    <SelectItem value="yes">Z fakturą</SelectItem>
                    <SelectItem value="no">Bez faktury</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            <div className="flex flex-col sm:flex-row items-stretch sm:items-center justify-between gap-3">
              <div className="flex items-center gap-3 flex-wrap">
                <Badge variant="secondary" className="text-base px-3 py-1" data-testid="badge-results-count">
                  Wyniki: {totalCount.toLocaleString('pl-PL')}
                </Badge>
                <Button
                  variant="outline"
                  size="default"
                  onClick={resetSettings}
                  data-testid="button-reset-filters"
                >
                  <RotateCcw className="h-4 w-4 mr-2" />
                  Resetuj
                </Button>
              </div>

              <div className="flex flex-col sm:flex-row items-stretch sm:items-center gap-2">
                <div className="text-sm text-muted-foreground hidden md:block">
                  {orders.length > 0 ? (currentPage - 1) * itemsPerPage + 1 : 0} - {Math.min(currentPage * itemsPerPage, totalCount)} z {totalCount}
                  {hasAnyFilters && totalCount !== totalOrders && ` (filtr. z ${totalOrders})`}
                </div>
                <div className="flex items-center gap-2">
                  <Select
                    value={itemsPerPage.toString()}
                    onValueChange={(value) => {
                      updateSettings({ itemsPerPage: Number(value), currentPage: 1 });
                    }}
                  >
                    <SelectTrigger className="w-[90px]" data-testid="select-items-per-page">
                      <SelectValue />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="10">10</SelectItem>
                      <SelectItem value="25">25</SelectItem>
                      <SelectItem value="50">50</SelectItem>
                      <SelectItem value="100">100</SelectItem>
                      <SelectItem value="250">250</SelectItem>
                      <SelectItem value="500">500</SelectItem>
                    </SelectContent>
                  </Select>
                  <div className="flex items-center gap-1">
                    <Button
                      variant="outline"
                      size="icon"
                      onClick={() => updateSettings({ currentPage: Math.max(1, currentPage - 1) })}
                      disabled={currentPage === 1}
                      data-testid="button-prev-page"
                    >
                      <ChevronLeft className="h-4 w-4" />
                    </Button>
                    <div className="text-sm px-2">
                      {currentPage} / {totalPages || 1}
                    </div>
                    <Button
                      variant="outline"
                      size="icon"
                      onClick={() => updateSettings({ currentPage: Math.min(totalPages, currentPage + 1) })}
                      disabled={currentPage === totalPages || totalPages === 0}
                      data-testid="button-next-page"
                    >
                      <ChevronRight className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              </div>
            </div>

            <PaymentSummaryBar
              searchQuery={searchQuery}
              paymentFilter={paymentFilter}
              fulfillmentFilter={fulfillmentFilter}
              invoiceFilter={invoiceFilter}
              paymentTypeFilter={paymentTypeFilter}
              sourceFilter={sourceFilter}
              dateRangeType={dateRangeType}
              customDateFrom={customDateFrom || undefined}
              customDateTo={customDateTo || undefined}
            />

            <div className="w-full overflow-x-auto">
              <OrdersTable 
                orders={orders} 
                isLoading={isLoading}
                sortBy={sortBy}
                sortOrder={sortOrder}
                onSort={(column) => {
                  if (sortBy === column) {
                    updateSettings({ sortOrder: sortOrder === 'ASC' ? 'DESC' : 'ASC' });
                  } else {
                    updateSettings({ sortBy: column, sortOrder: 'DESC' });
                  }
                }}
              />
            </div>

            <PaymentSummaryBar
              searchQuery={searchQuery}
              paymentFilter={paymentFilter}
              fulfillmentFilter={fulfillmentFilter}
              invoiceFilter={invoiceFilter}
              paymentTypeFilter={paymentTypeFilter}
              sourceFilter={sourceFilter}
              dateRangeType={dateRangeType}
              customDateFrom={customDateFrom || undefined}
              customDateTo={customDateTo || undefined}
            />
          </div>
        </div>
      </div>
    </div>
  );
}
