import { useQuery, useMutation } from "@tanstack/react-query";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Package, Plus, Search, Edit, Trash2, Check, ChevronDown, ChevronUp, X, ChevronLeft, ChevronRight, ChevronsLeft, ChevronsRight, ArrowUpDown, Image as ImageIcon } from "lucide-react";
import { useState, useEffect } from "react";
import { Skeleton } from "@/components/ui/skeleton";
import { useLocation, Link } from "wouter";
import { Checkbox } from "@/components/ui/checkbox";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from "@/components/ui/dialog";
import { useToast } from "@/hooks/use-toast";
import { queryClient, apiRequest } from "@/lib/queryClient";
import { DictionaryCombobox } from "@/components/dictionary-combobox";
import { Switch } from "@/components/ui/switch";
import { ConnectionStatusIcon } from "@/components/connection-status-icon";

interface ProductSet {
  id: number;
  setMatrixId: number | null;
  sku: string;
  title: string;
  shortDescription: string | null;
  fullDescription: string | null;
  color: string | null;
  colorOptions: string[] | null;
  depth: string | null;
  panelCount: number | null;
  hookLength: string | null;
  basePrice: string | null;
  calculatedPrice: string | null;
  finalPrice: string | null;
  modifierType: string | null;
  modifierOperation: string | null;
  modifierValue: string | null;
  images: string[];
  generatedFromMatrix: boolean;
  combinationKey: string | null;
  isActive: boolean;
  createdAt: string;
  updatedAt: string;
  marketplaceProductId: number | null;
  marketplaceExternalId: string | null;
  marketplacePlatform: 'allegro' | 'shoper' | null;
  isLinked: boolean;
  matrixLength: string | null;
  matrixWidth: string | null;
  matrixHeight: string | null;
  matrixDoors: string | null;
  matrixLegs: string | null;
  matrixProductGroup: string | null;
  matrixColorOptions: Record<string, string[]> | null;
  matrixComponentOverrides: Record<string, Record<string, {productId: number}>> | null;
  primaryImage: {
    url: string;
    thumbnailUrl: string;
    mediumUrl: string;
  } | null;
}

interface ProductCreatorDictionary {
  id: number;
  dictionaryType: string;
  code: string;
  name: string;
  readableName: string | null;
  color: string | null;
  isActive: boolean;
}

const STORAGE_KEY = 'catalog-sets-filters';

// Helper function to determine text color based on background luminance
function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}

// Helper function to determine if color needs visible border
function needsVisibleBorder(hexColor: string | null | undefined): boolean {
  if (!hexColor || !hexColor.startsWith('#')) return false;
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance < 30 || luminance > 230;
}

export default function CatalogSetsPage() {
  const { toast } = useToast();
  const [, setLocation] = useLocation();
  
  // Load filters from localStorage
  const loadFilters = () => {
    try {
      const saved = localStorage.getItem(STORAGE_KEY);
      if (saved) {
        return JSON.parse(saved);
      }
    } catch (error) {
      console.error('Error loading filters from localStorage:', error);
    }
    return {};
  };

  const savedFilters = loadFilters();
  
  const [searchQuery, setSearchQuery] = useState(savedFilters.searchQuery || "");
  const [selectedSets, setSelectedSets] = useState<number[]>([]);
  const [isBulkDeleteOpen, setIsBulkDeleteOpen] = useState(false);
  const [filtersExpanded, setFiltersExpanded] = useState(false);

  // Pagination and sorting
  const [page, setPage] = useState(savedFilters.page || 1);
  const [limit, setLimit] = useState(savedFilters.limit || 25);
  const [sortBy, setSortBy] = useState(savedFilters.sortBy || 'createdAt');
  const [sortOrder, setSortOrder] = useState<'ASC' | 'DESC'>(savedFilters.sortOrder || 'DESC');
  const [compactMode, setCompactMode] = useState(savedFilters.compactMode ?? true);
  const [groupBy, setGroupBy] = useState<string | null>(savedFilters.groupBy || null);
  const [expandedGroups, setExpandedGroups] = useState<Set<string>>(
    new Set(savedFilters.expandedGroups || [])
  );

  // Filter states
  const [filterColor, setFilterColor] = useState<string>(savedFilters.filterColor || "");
  const [filterPanelCount, setFilterPanelCount] = useState<string>(savedFilters.filterPanelCount || "");

  // Save filters to localStorage whenever they change
  useEffect(() => {
    const filters = {
      searchQuery,
      filterColor,
      filterPanelCount,
      page,
      limit,
      sortBy,
      sortOrder,
      compactMode,
      groupBy,
      expandedGroups: Array.from(expandedGroups),
    };
    try {
      localStorage.setItem(STORAGE_KEY, JSON.stringify(filters));
    } catch (error) {
      console.error('Error saving filters to localStorage:', error);
    }
  }, [searchQuery, filterColor, filterPanelCount, page, limit, sortBy, sortOrder, compactMode, groupBy, expandedGroups]);

  // Inline editing state
  const [editingCell, setEditingCell] = useState<{setId: number, field: string} | null>(null);
  const [editValue, setEditValue] = useState<string>("");

  // Get all sets
  const { data: sets = [], isLoading } = useQuery<ProductSet[]>({
    queryKey: ["/api/product-sets"],
  });

  const { data: colors } = useQuery<ProductCreatorDictionary[]>({
    queryKey: ["/api/dictionaries?type=color"],
  });

  // Auto-expand groups when grouping is first enabled or changed
  useEffect(() => {
    if (groupBy && sets.length > 0) {
      const groupedData = getGroupedSets();
      if (groupedData && expandedGroups.size === 0) {
        setExpandedGroups(new Set(groupedData.map(({ key }) => key)));
      }
    }
  }, [groupBy, sets.length]);

  // Parse search query into individual search phrases
  const getSearchPhrases = () => {
    if (!searchQuery) return [];
    const phrases: string[] = [];
    searchQuery.split(/[;,]/).forEach((phrase: string) => {
      const trimmed = phrase.trim();
      if (trimmed) phrases.push(trimmed);
    });
    return phrases;
  };

  const searchPhrases = getSearchPhrases();

  // Remove a specific search phrase
  const removeSearchPhrase = (phraseToRemove: string) => {
    const updatedPhrases = searchPhrases.filter(p => p !== phraseToRemove);
    setSearchQuery(updatedPhrases.join('; '));
  };

  // Smart search - split by semicolon (AND) and comma (OR)
  const filteredSets = sets.filter(set => {
    // Text search - supports AND (;) and OR (,)
    if (searchQuery) {
      const terms = searchQuery.split(';').map((t: string) => t.trim()).filter(Boolean);
      const matches = terms.every((term: string) => {
        const orTerms = term.split(',').map((t: string) => t.trim().toLowerCase()).filter(Boolean);
        return orTerms.some((orTerm: string) => {
          return (
            set.sku?.toLowerCase().includes(orTerm) ||
            set.title?.toLowerCase().includes(orTerm) ||
            set.color?.toLowerCase().includes(orTerm)
          );
        });
      });
      if (!matches) return false;
    }

    // Attribute filters
    if (filterColor && set.color !== filterColor) return false;
    if (filterPanelCount && Number(set.panelCount) !== Number(filterPanelCount)) return false;

    return true;
  });

  // Apply sorting
  const sortedSets = [...filteredSets].sort((a, b) => {
    let aValue: any = a[sortBy as keyof ProductSet];
    let bValue: any = b[sortBy as keyof ProductSet];

    if (aValue === null || aValue === undefined) return sortOrder === 'ASC' ? 1 : -1;
    if (bValue === null || bValue === undefined) return sortOrder === 'ASC' ? -1 : 1;

    if (typeof aValue === 'string') aValue = aValue.toLowerCase();
    if (typeof bValue === 'string') bValue = bValue.toLowerCase();

    if (aValue < bValue) return sortOrder === 'ASC' ? -1 : 1;
    if (aValue > bValue) return sortOrder === 'ASC' ? 1 : -1;
    return 0;
  });

  // Pagination
  const paginatedSets = sortedSets.slice((page - 1) * limit, page * limit);
  const totalPages = Math.ceil(sortedSets.length / limit);

  const activeFiltersCount = [
    ...searchPhrases,
    filterColor,
    filterPanelCount,
  ].filter(Boolean).length;

  const clearAllFilters = () => {
    setFilterColor("");
    setFilterPanelCount("");
    setSearchQuery("");
  };

  // Bulk delete mutation
  const bulkDeleteMutation = useMutation({
    mutationFn: async () => {
      await Promise.all(selectedSets.map(id =>
        apiRequest("DELETE", `/api/product-sets/${id}`, {})
      ));
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/product-sets"] });
      toast({
        title: "Zestawy usunięte",
        description: `Pomyślnie usunięto ${selectedSets.length} zestawów.`,
      });
      setSelectedSets([]);
      setIsBulkDeleteOpen(false);
    },
    onError: (error: any) => {
      toast({
        title: "Błąd usuwania",
        description: error.message || "Nie udało się usunąć zestawów.",
        variant: "destructive",
      });
    },
  });

  // Inline edit mutation
  const inlineUpdateMutation = useMutation({
    mutationFn: async ({ setId, field, value }: { setId: number, field: string, value: any }) => {
      const updates: any = {};
      
      if (field === 'sku') updates.sku = value;
      if (field === 'basePrice') updates.basePrice = value;
      if (field === 'calculatedPrice') updates.calculatedPrice = value;
      if (field === 'isActive') updates.isActive = value;
      
      return apiRequest("PUT", `/api/product-sets/${setId}`, updates);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: ["/api/product-sets"] });
      setEditingCell(null);
      setEditValue("");
      toast({
        title: "Zapisano",
        description: "Zestaw został zaktualizowany.",
      });
    },
    onError: (error: any) => {
      toast({
        title: "Błąd zapisu",
        description: error.message || "Nie udało się zapisać zmiany.",
        variant: "destructive",
      });
    },
  });

  const toggleSet = (setId: number) => {
    setSelectedSets(prev =>
      prev.includes(setId)
        ? prev.filter(id => id !== setId)
        : [...prev, setId]
    );
  };

  const toggleAll = () => {
    if (selectedSets.length === filteredSets.length) {
      setSelectedSets([]);
    } else {
      setSelectedSets(filteredSets.map(s => s.id));
    }
  };

  // Inline editing helpers
  const startEditing = (setId: number, field: string, currentValue: any) => {
    setEditingCell({ setId, field });
    setEditValue(currentValue?.toString() || "");
  };

  const cancelEditing = () => {
    setEditingCell(null);
    setEditValue("");
  };

  const saveEdit = (setId: number, field: string) => {
    let value: any = editValue;
    
    if (field === 'basePrice' || field === 'calculatedPrice') {
      value = editValue ? parseFloat(editValue) : null;
    }
    
    inlineUpdateMutation.mutate({ setId, field, value });
  };

  const toggleActive = (setId: number, currentValue: boolean) => {
    inlineUpdateMutation.mutate({ 
      setId, 
      field: 'isActive', 
      value: !currentValue 
    });
  };

  // Grouping functions
  const getGroupKey = (set: ProductSet, field: string): string => {
    const value = set[field as keyof ProductSet];
    if (value === null || value === undefined) return '(puste)';
    
    // For color, find readable name
    if (field === 'color' && colors) {
      const color = colors.find(c => c.name === value || c.code === value);
      return color?.name || String(value);
    }
    
    // For basePrice/calculatedPrice, format as PLN
    if (field === 'basePrice' || field === 'calculatedPrice') {
      return `${Number(value).toFixed(2)} PLN`;
    }
    
    return String(value);
  };

  const getGroupedSets = () => {
    if (!groupBy) return null;

    const groups = new Map<string, ProductSet[]>();
    
    filteredSets.forEach(set => {
      const key = getGroupKey(set, groupBy);
      if (!groups.has(key)) {
        groups.set(key, []);
      }
      groups.get(key)!.push(set);
    });

    return Array.from(groups.entries())
      .sort(([a], [b]) => a.localeCompare(b, 'pl'))
      .map(([key, sets]) => ({ key, sets }));
  };

  const toggleGroup = (groupKey: string) => {
    setExpandedGroups(prev => {
      const newSet = new Set(prev);
      if (newSet.has(groupKey)) {
        newSet.delete(groupKey);
      } else {
        newSet.add(groupKey);
      }
      return newSet;
    });
  };

  const collapseAllGroups = () => {
    setExpandedGroups(new Set());
  };

  const expandAllGroups = () => {
    const groupedData = getGroupedSets();
    if (groupedData) {
      setExpandedGroups(new Set(groupedData.map(({ key }) => key)));
    }
  };

  const renderSetRow = (set: ProductSet, index: number) => {
    const colorDict = colors?.find(c => c.name === set.color || c.code === set.color);
    const colorHex = colorDict?.color;
    
    // Get color options for this set (from the set itself, not matrix)
    const colorOptions = set.colorOptions || [];
    
    // Get component overrides for this set's color
    const hasOverrides = set.matrixComponentOverrides && set.color
      ? Object.keys(set.matrixComponentOverrides[set.color] || {}).length > 0
      : false;
    
    return (
      <TableRow 
        key={set.id} 
        data-testid={`row-set-${set.id}`}
        className="hover-elevate"
      >
        {/* Checkbox */}
        <TableCell className="w-8" onClick={(e) => e.stopPropagation()}>
          <Checkbox
            checked={selectedSets.includes(set.id)}
            onCheckedChange={() => toggleSet(set.id)}
            data-testid={`checkbox-set-${set.id}`}
          />
        </TableCell>

        {/* Row Number */}
        <TableCell className="text-center text-xs text-muted-foreground" data-testid={`text-row-${set.id}`}>
          {index + 1}
        </TableCell>

        {/* ID */}
        <TableCell className="text-center font-mono text-xs" data-testid={`text-id-${set.id}`}>
          {set.id}
        </TableCell>

        {/* Image */}
        <TableCell className="w-16">
          <Link href={`/catalog-sets/${set.id}`}>
            {set.primaryImage ? (
              <img
                src={set.primaryImage.thumbnailUrl}
                alt={set.title}
                className="w-12 h-12 object-cover rounded cursor-pointer"
                data-testid={`img-set-${set.id}`}
              />
            ) : set.images && set.images.length > 0 ? (
              <img
                src={set.images[0]}
                alt={set.title}
                className="w-12 h-12 object-cover rounded cursor-pointer"
                data-testid={`img-set-${set.id}`}
              />
            ) : (
              <div className="w-12 h-12 bg-muted flex items-center justify-center rounded cursor-pointer" data-testid={`img-placeholder-${set.id}`}>
                <ImageIcon className="w-6 h-6 text-muted-foreground" />
              </div>
            )}
          </Link>
        </TableCell>

        {/* SKU */}
        <TableCell className="font-mono text-xs" data-testid={`text-sku-${set.id}`} onClick={(e) => e.stopPropagation()}>
          {editingCell?.setId === set.id && editingCell.field === 'sku' ? (
            <div className="flex gap-1">
              <Input
                value={editValue}
                onChange={(e) => setEditValue(e.target.value)}
                className="h-6 text-xs"
                autoFocus
                data-testid={`input-sku-${set.id}`}
              />
              <Button size="sm" className="h-6 w-6 p-0" onClick={() => saveEdit(set.id, 'sku')} data-testid={`button-save-sku-${set.id}`}>
                <Check className="h-3 w-3" />
              </Button>
              <Button size="sm" variant="ghost" className="h-6 w-6 p-0" onClick={cancelEditing} data-testid={`button-cancel-sku-${set.id}`}>
                <X className="h-3 w-3" />
              </Button>
            </div>
          ) : (
            <span onClick={() => startEditing(set.id, 'sku', set.sku)} className="cursor-pointer hover-elevate px-1">
              {set.sku}
            </span>
          )}
        </TableCell>

        {/* Title */}
        <TableCell data-testid={`text-title-${set.id}`}>
          <Link href={`/catalog-sets/${set.id}`}>
            {compactMode ? (
              <div className="text-xs hover-elevate cursor-pointer px-1">{set.title}</div>
            ) : (
              <div className="hover-elevate cursor-pointer px-1">
                <div className="font-medium">{set.title}</div>
                {set.shortDescription && (
                  <div className="text-xs text-muted-foreground">{set.shortDescription}</div>
                )}
              </div>
            )}
          </Link>
        </TableCell>

        {/* Length */}
        <TableCell className="text-center text-xs" data-testid={`text-length-${set.id}`}>
          {set.matrixLength ? `${Math.round(Number(set.matrixLength) / 10)} cm` : '-'}
        </TableCell>

        {/* Width */}
        <TableCell className="text-center text-xs" data-testid={`text-width-${set.id}`}>
          {set.matrixWidth ? `${Math.round(Number(set.matrixWidth) / 10)} cm` : '-'}
        </TableCell>

        {/* Height */}
        <TableCell className="text-center text-xs" data-testid={`text-height-${set.id}`}>
          {set.matrixHeight ? `${Math.round(Number(set.matrixHeight) / 10)} cm` : '-'}
        </TableCell>

        {/* Color */}
        <TableCell data-testid={`badge-color-${set.id}`}>
          {set.color && colorDict ? (
            <Badge
              className="font-mono text-xs"
              style={{
                backgroundColor: colorHex || undefined,
                color: colorHex ? getTextColorForBackground(colorHex) : undefined,
                border: colorHex && needsVisibleBorder(colorHex) ? '1px solid hsl(var(--border))' : undefined
              }}
            >
              {set.color}
            </Badge>
          ) : (
            set.color && <Badge className="font-mono text-xs" variant="outline">{set.color}</Badge>
          )}
        </TableCell>

        {/* Color Options */}
        <TableCell data-testid={`text-color-options-${set.id}`}>
          {colorOptions.length > 0 ? (
            <div className="flex flex-wrap gap-1">
              {colorOptions.map((option, idx) => (
                <Badge key={idx} variant="secondary" className="text-xs">
                  {option}
                </Badge>
              ))}
            </div>
          ) : (
            <span className="text-xs text-muted-foreground">-</span>
          )}
        </TableCell>

        {/* Component Overrides */}
        <TableCell data-testid={`text-overrides-${set.id}`}>
          {hasOverrides ? (
            <Badge variant="outline" className="text-xs">
              Nadpisane
            </Badge>
          ) : (
            <span className="text-xs text-muted-foreground">-</span>
          )}
        </TableCell>

        {/* Doors */}
        <TableCell className="text-center text-xs" data-testid={`text-doors-${set.id}`}>
          {set.matrixDoors || '-'}
        </TableCell>

        {/* Legs */}
        <TableCell className="text-center text-xs" data-testid={`text-legs-${set.id}`}>
          {set.matrixLegs || '-'}
        </TableCell>

        {/* Product Group */}
        <TableCell className="text-xs" data-testid={`text-group-${set.id}`}>
          {set.matrixProductGroup || '-'}
        </TableCell>

        {/* Panel Count */}
        <TableCell className="text-center" data-testid={`text-panel-count-${set.id}`}>
          {set.panelCount || '-'}
        </TableCell>

        {/* Calculated Price (Components Total) */}
        <TableCell data-testid={`text-calculated-price-${set.id}`}>
          <span className="text-xs text-muted-foreground">
            {set.calculatedPrice ? `${Number(set.calculatedPrice).toFixed(2)} PLN` : '-'}
          </span>
        </TableCell>

        {/* Final Price (After Modifier) */}
        <TableCell data-testid={`text-final-price-${set.id}`}>
          <div className="flex items-center gap-2">
            <span className="font-medium">
              {set.finalPrice ? `${Number(set.finalPrice).toFixed(2)} PLN` : 
               set.calculatedPrice ? `${Number(set.calculatedPrice).toFixed(2)} PLN` : '-'}
            </span>
            {set.modifierValue && parseFloat(set.modifierValue) !== 0 && (
              <Badge variant="outline" className="text-xs">
                {set.modifierOperation === 'subtract' ? '-' : '+'}
                {set.modifierType === 'percentage' 
                  ? `${set.modifierValue}%` 
                  : `${parseFloat(set.modifierValue).toFixed(0)} PLN`
                }
              </Badge>
            )}
          </div>
        </TableCell>

        {/* Active */}
        <TableCell className="text-center" data-testid={`switch-active-${set.id}`}>
          <Switch
            checked={set.isActive}
            onCheckedChange={() => toggleActive(set.id, set.isActive)}
          />
        </TableCell>

        {/* Connection Status */}
        <TableCell className="w-12 text-center" onClick={(e) => e.stopPropagation()}>
          <ConnectionStatusIcon
            isLinked={set.isLinked}
            targetPath={
              set.marketplaceProductId && set.marketplaceExternalId
                ? `/product/${set.marketplaceExternalId}`
                : null
            }
            targetSku={set.marketplaceExternalId}
            productId={set.id}
            data-testid={`connection-status-${set.id}`}
          />
        </TableCell>

        {/* Actions */}
        <TableCell onClick={(e) => e.stopPropagation()}>
          <div className="flex gap-1">
            <Button
              size="sm"
              variant="ghost"
              className="h-6 w-6 p-0 text-destructive"
              onClick={() => {
                setSelectedSets([set.id]);
                setIsBulkDeleteOpen(true);
              }}
              data-testid={`button-delete-${set.id}`}
            >
              <Trash2 className="h-3 w-3" />
            </Button>
          </div>
        </TableCell>
      </TableRow>
    );
  };

  const renderTable = (setsToRender: ProductSet[]) => (
    <Table>
      <TableHeader>
        <TableRow>
          <TableHead className="w-8">
            <Checkbox
              checked={selectedSets.length === filteredSets.length && filteredSets.length > 0}
              onCheckedChange={toggleAll}
              data-testid="checkbox-select-all"
            />
          </TableHead>
          <TableHead className="w-12">#</TableHead>
          <TableHead className="w-16 cursor-pointer" onClick={() => {
            setSortBy('id');
            setSortOrder(sortBy === 'id' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            ID {sortBy === 'id' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="w-16">Zdjęcie</TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('sku');
            setSortOrder(sortBy === 'sku' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            SKU {sortBy === 'sku' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('title');
            setSortOrder(sortBy === 'title' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Nazwa {sortBy === 'title' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('matrixLength');
            setSortOrder(sortBy === 'matrixLength' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Długość {sortBy === 'matrixLength' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('matrixWidth');
            setSortOrder(sortBy === 'matrixWidth' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Szerokość {sortBy === 'matrixWidth' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead>Wysokość</TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('color');
            setSortOrder(sortBy === 'color' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Kolor {sortBy === 'color' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead>Opcje</TableHead>
          <TableHead>Nadpisania</TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('matrixDoors');
            setSortOrder(sortBy === 'matrixDoors' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Drzwi {sortBy === 'matrixDoors' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('matrixLegs');
            setSortOrder(sortBy === 'matrixLegs' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Nogi {sortBy === 'matrixLegs' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('matrixProductGroup');
            setSortOrder(sortBy === 'matrixProductGroup' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Grupa {sortBy === 'matrixProductGroup' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="text-center cursor-pointer" onClick={() => {
            setSortBy('panelCount');
            setSortOrder(sortBy === 'panelCount' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Panele {sortBy === 'panelCount' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('calculatedPrice');
            setSortOrder(sortBy === 'calculatedPrice' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Komponenty {sortBy === 'calculatedPrice' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="cursor-pointer" onClick={() => {
            setSortBy('finalPrice');
            setSortOrder(sortBy === 'finalPrice' && sortOrder === 'ASC' ? 'DESC' : 'ASC');
          }}>
            Cena końcowa {sortBy === 'finalPrice' && <ArrowUpDown className="inline h-3 w-3" />}
          </TableHead>
          <TableHead className="text-center">Aktywny</TableHead>
          <TableHead className="w-12 text-center">Połączenie</TableHead>
          <TableHead className="w-20">Akcje</TableHead>
        </TableRow>
      </TableHeader>
      <TableBody>
        {setsToRender.map((set, index) => renderSetRow(set, index))}
      </TableBody>
    </Table>
  );

  const groupedData = getGroupedSets();

  return (
    <div className="p-4 space-y-4">
      <Card>
        <CardHeader className="flex flex-row items-center justify-between gap-2 space-y-0 pb-4">
          <CardTitle className="flex items-center gap-2">
            <Package className="h-5 w-5" />
            Katalog Zestawów
            <Badge variant="outline">{filteredSets.length}</Badge>
          </CardTitle>
          <div className="flex gap-2">
            {selectedSets.length > 0 && (
              <Button
                size="sm"
                variant="destructive"
                onClick={() => setIsBulkDeleteOpen(true)}
                data-testid="button-bulk-delete"
              >
                <Trash2 className="h-4 w-4 mr-1" />
                Usuń ({selectedSets.length})
              </Button>
            )}
          </div>
        </CardHeader>
        <CardContent className="space-y-4">
          {/* Search and Filters */}
          <div className="space-y-2">
            <div className="flex gap-2">
              <div className="flex-1 relative">
                <Search className="absolute left-2 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input
                  placeholder="Szukaj zestawów (SKU, nazwa, kolor)... Użyj ; dla AND, , dla OR"
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="pl-8"
                  data-testid="input-search"
                />
              </div>
              <Button
                variant="outline"
                size="sm"
                onClick={() => setFiltersExpanded(!filtersExpanded)}
                data-testid="button-toggle-filters"
              >
                Filtry {activeFiltersCount > 0 && `(${activeFiltersCount})`}
                {filtersExpanded ? <ChevronUp className="ml-1 h-4 w-4" /> : <ChevronDown className="ml-1 h-4 w-4" />}
              </Button>
            </div>

            {/* Search phrases badges */}
            {searchPhrases.length > 0 && (
              <div className="flex flex-wrap gap-1">
                {searchPhrases.map((phrase, idx) => (
                  <Badge key={idx} variant="secondary" className="gap-1" data-testid={`badge-search-phrase-${idx}`}>
                    {phrase}
                    <X
                      className="h-3 w-3 cursor-pointer hover-elevate"
                      onClick={() => removeSearchPhrase(phrase)}
                    />
                  </Badge>
                ))}
              </div>
            )}

            {/* Expanded filters */}
            {filtersExpanded && (
              <div className="grid grid-cols-1 md:grid-cols-3 gap-2 p-3 border rounded-md bg-muted/50">
                <div className="space-y-1">
                  <Label className="text-xs">Kolor</Label>
                  <DictionaryCombobox
                    items={colors}
                    value={filterColor}
                    onChange={setFilterColor}
                    placeholder="Wybierz kolor"
                    testId="select-filter-color"
                  />
                </div>

                <div className="space-y-1">
                  <Label className="text-xs">Liczba paneli</Label>
                  <Input
                    type="number"
                    placeholder="np. 3"
                    value={filterPanelCount}
                    onChange={(e) => setFilterPanelCount(e.target.value)}
                    className="h-8"
                    data-testid="input-filter-panel-count"
                  />
                </div>

                <div className="flex items-end">
                  <Button
                    variant="outline"
                    size="sm"
                    onClick={clearAllFilters}
                    className="w-full"
                    data-testid="button-clear-filters"
                  >
                    <X className="h-4 w-4 mr-1" />
                    Wyczyść filtry
                  </Button>
                </div>
              </div>
            )}
          </div>

          {/* Grouping and View Options */}
          <div className="flex gap-2 items-center">
            <Label className="text-xs text-muted-foreground">Grupuj po:</Label>
            <Select value={groupBy || "none"} onValueChange={(val) => setGroupBy(val === "none" ? null : val)}>
              <SelectTrigger className="w-40 h-7 text-xs" data-testid="select-group-by">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="none">Brak grupowania</SelectItem>
                <SelectItem value="color">Kolor</SelectItem>
                <SelectItem value="basePrice">Cena bazowa</SelectItem>
                <SelectItem value="calculatedPrice">Cena obliczona</SelectItem>
              </SelectContent>
            </Select>

            {groupBy && groupedData && (
              <>
                <Button size="sm" variant="outline" onClick={expandAllGroups} className="h-7 text-xs" data-testid="button-expand-all">
                  Rozwiń wszystkie
                </Button>
                <Button size="sm" variant="outline" onClick={collapseAllGroups} className="h-7 text-xs" data-testid="button-collapse-all">
                  Zwiń wszystkie
                </Button>
              </>
            )}

            <div className="flex-1" />

            <div className="flex items-center gap-2">
              <Label className="text-xs text-muted-foreground">Tryb kompaktowy</Label>
              <Switch
                checked={compactMode}
                onCheckedChange={setCompactMode}
                data-testid="switch-compact-mode"
              />
            </div>

            <Select value={limit.toString()} onValueChange={(val) => {
              setLimit(parseInt(val));
              setPage(1);
            }}>
              <SelectTrigger className="w-24 h-7 text-xs" data-testid="select-limit">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="10">10</SelectItem>
                <SelectItem value="25">25</SelectItem>
                <SelectItem value="50">50</SelectItem>
                <SelectItem value="100">100</SelectItem>
                <SelectItem value="250">250</SelectItem>
              </SelectContent>
            </Select>
          </div>

          {/* Loading State */}
          {isLoading && (
            <div className="space-y-2">
              {[...Array(5)].map((_, i) => (
                <Skeleton key={i} className="h-16 w-full" />
              ))}
            </div>
          )}

          {/* Empty State */}
          {!isLoading && filteredSets.length === 0 && (
            <div className="text-center py-12">
              <Package className="h-12 w-12 text-muted-foreground mx-auto mb-4" />
              <p className="text-muted-foreground">Nie znaleziono zestawów</p>
            </div>
          )}

          {/* Table - Grouped or Normal */}
          {!isLoading && filteredSets.length > 0 && (
            <>
              {groupBy && groupedData ? (
                <div className="space-y-2">
                  {groupedData.map(({ key, sets: groupSets }) => {
                    const isExpanded = expandedGroups.has(key);
                    return (
                      <div key={key} className="border rounded-md">
                        <button
                          onClick={() => toggleGroup(key)}
                          className="w-full px-3 py-2 flex items-center justify-between hover-elevate bg-muted/50"
                          data-testid={`button-group-${key}`}
                        >
                          <div className="flex items-center gap-2">
                            {isExpanded ? <ChevronDown className="h-4 w-4" /> : <ChevronRight className="h-4 w-4" />}
                            <span className="font-medium">{key}</span>
                            <Badge variant="outline">{groupSets.length}</Badge>
                          </div>
                        </button>
                        {isExpanded && (
                          <div className="border-t">
                            {renderTable(groupSets)}
                          </div>
                        )}
                      </div>
                    );
                  })}
                </div>
              ) : (
                renderTable(paginatedSets)
              )}
            </>
          )}

          {/* Pagination */}
          {!groupBy && filteredSets.length > 0 && totalPages > 1 && (
            <div className="flex items-center justify-between">
              <div className="text-xs text-muted-foreground">
                Strona {page} z {totalPages} ({filteredSets.length} zestawów)
              </div>
              <div className="flex gap-1">
                <Button
                  size="sm"
                  variant="outline"
                  onClick={() => setPage(1)}
                  disabled={page === 1}
                  className="h-7 w-7 p-0"
                  data-testid="button-first-page"
                >
                  <ChevronsLeft className="h-4 w-4" />
                </Button>
                <Button
                  size="sm"
                  variant="outline"
                  onClick={() => setPage((p: number) => Math.max(1, p - 1))}
                  disabled={page === 1}
                  className="h-7 w-7 p-0"
                  data-testid="button-prev-page"
                >
                  <ChevronLeft className="h-4 w-4" />
                </Button>
                <Button
                  size="sm"
                  variant="outline"
                  onClick={() => setPage((p: number) => Math.min(totalPages, p + 1))}
                  disabled={page === totalPages}
                  className="h-7 w-7 p-0"
                  data-testid="button-next-page"
                >
                  <ChevronRight className="h-4 w-4" />
                </Button>
                <Button
                  size="sm"
                  variant="outline"
                  onClick={() => setPage(totalPages)}
                  disabled={page === totalPages}
                  className="h-7 w-7 p-0"
                  data-testid="button-last-page"
                >
                  <ChevronsRight className="h-4 w-4" />
                </Button>
              </div>
            </div>
          )}
        </CardContent>
      </Card>

      {/* Bulk Delete Dialog */}
      <Dialog open={isBulkDeleteOpen} onOpenChange={setIsBulkDeleteOpen}>
        <DialogContent>
          <DialogHeader>
            <DialogTitle>Potwierdź usunięcie</DialogTitle>
            <DialogDescription>
              Czy na pewno chcesz usunąć {selectedSets.length} {selectedSets.length === 1 ? 'zestaw' : 'zestawów'}?
              Ta operacja jest nieodwracalna.
            </DialogDescription>
          </DialogHeader>
          <DialogFooter>
            <Button variant="outline" onClick={() => setIsBulkDeleteOpen(false)} data-testid="button-cancel-delete">
              Anuluj
            </Button>
            <Button
              variant="destructive"
              onClick={() => bulkDeleteMutation.mutate()}
              disabled={bulkDeleteMutation.isPending}
              data-testid="button-confirm-delete"
            >
              {bulkDeleteMutation.isPending ? "Usuwanie..." : "Usuń"}
            </Button>
          </DialogFooter>
        </DialogContent>
      </Dialog>
    </div>
  );
}
