import { useState, useEffect } from "react";
import { Link } from "wouter";
import { X, ChevronRight, FileText, Package, ExternalLink } from "lucide-react";
import { Button } from "@/components/ui/button";
import { cn } from "@/lib/utils";
import { Badge } from "@/components/ui/badge";

const SIDEBAR_STORAGE_KEY = 'rightSidebarCollapsed';

interface RightSidebarProps {
  children: React.ReactNode;
  defaultCollapsed?: boolean;
}

export function RightSidebar({ children, defaultCollapsed = false }: RightSidebarProps) {
  const [isCollapsed, setIsCollapsed] = useState(() => {
    if (typeof window === 'undefined') return defaultCollapsed;
    const saved = localStorage.getItem(SIDEBAR_STORAGE_KEY);
    return saved !== null ? saved === 'true' : defaultCollapsed;
  });

  useEffect(() => {
    if (typeof window !== 'undefined') {
      localStorage.setItem(SIDEBAR_STORAGE_KEY, String(isCollapsed));
    }
  }, [isCollapsed]);

  if (isCollapsed) {
    return (
      <div className="hidden md:flex w-12 border-l bg-card flex-shrink-0 flex-col items-center py-4">
        <Button
          variant="ghost"
          size="icon"
          onClick={() => setIsCollapsed(false)}
          data-testid="button-expand-right-sidebar"
          className="h-8 w-8"
        >
          <ChevronRight className="h-4 w-4 rotate-180" />
        </Button>
      </div>
    );
  }

  return (
    <div className="hidden md:flex w-64 border-l bg-card flex-shrink-0 overflow-y-auto flex-col">
      {/* Header */}
      <div className="border-b px-4 py-4 flex items-center justify-between gap-2 flex-shrink-0">
        <h2 className="text-lg font-semibold flex-1">Powiązania</h2>
        <Button
          variant="ghost"
          size="icon"
          className="h-8 w-8 flex-shrink-0"
          onClick={() => setIsCollapsed(true)}
          data-testid="button-collapse-right-sidebar"
        >
          <X className="h-4 w-4" />
        </Button>
      </div>

      {/* Content */}
      <div className="flex-1 overflow-y-auto">
        {children}
      </div>
    </div>
  );
}

interface RelationshipSectionProps {
  title: string;
  icon: React.ReactNode;
  count?: number;
  children: React.ReactNode;
}

export function RelationshipSection({ title, icon, count, children }: RelationshipSectionProps) {
  return (
    <div className="border-b last:border-b-0">
      <div className="px-4 py-3 flex items-center gap-2 bg-muted/30">
        {icon}
        <h3 className="font-medium text-sm flex-1">{title}</h3>
        {count !== undefined && (
          <Badge variant="secondary" className="text-xs">
            {count}
          </Badge>
        )}
      </div>
      <div className="p-2">
        {children}
      </div>
    </div>
  );
}

interface RelationshipItemProps {
  href: string;
  title: string;
  subtitle?: string;
  badge?: string;
  badgeVariant?: "default" | "secondary" | "outline" | "destructive";
}

export function RelationshipItem({ 
  href, 
  title, 
  subtitle, 
  badge,
  badgeVariant = "secondary" 
}: RelationshipItemProps) {
  return (
    <Link href={href}>
      <div
        className="flex items-start gap-2 px-3 py-2 rounded-md text-sm transition-colors hover-elevate active-elevate-2 group"
        data-testid={`link-relationship-${href}`}
      >
        <ExternalLink className="h-3.5 w-3.5 mt-0.5 flex-shrink-0 text-muted-foreground group-hover:text-foreground transition-colors" />
        <div className="flex-1 min-w-0">
          <div className="font-medium truncate">{title}</div>
          {subtitle && (
            <div className="text-xs text-muted-foreground truncate mt-0.5">{subtitle}</div>
          )}
        </div>
        {badge && (
          <Badge variant={badgeVariant} className="text-xs flex-shrink-0">
            {badge}
          </Badge>
        )}
      </div>
    </Link>
  );
}

export function EmptyRelationship({ message }: { message: string }) {
  return (
    <div className="px-3 py-4 text-sm text-muted-foreground text-center">
      {message}
    </div>
  );
}
