import { useMemo, useState } from "react";
import { ChevronRight, ChevronDown, Package, Layers } from "lucide-react";
import { Badge } from "@/components/ui/badge";
import { Checkbox } from "@/components/ui/checkbox";
import { DraggableProductItem, type MarketplaceOrderItem, type SetComponent } from "./DraggableProductItem";
import { SetComponentsModal } from "./SetComponentsModal";

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
  shortName: string | null;
  color: string | null;
  isActive: boolean;
}

function getTextColorForBackground(hexColor: string | null | undefined): string {
  if (!hexColor || !hexColor.startsWith('#')) return 'black';
  
  const r = parseInt(hexColor.slice(1, 3), 16);
  const g = parseInt(hexColor.slice(3, 5), 16);
  const b = parseInt(hexColor.slice(5, 7), 16);
  
  const luminance = 0.299 * r + 0.587 * g + 0.114 * b;
  
  return luminance > 128 ? 'black' : 'white';
}


interface MarketplaceOrder {
  order_id: number;
  order_number: string;
  marketplace: string;
  buyer_first_name: string;
  buyer_last_name: string;
  buyer_email: string;
  order_date: string;
  payment_status: string;
  total_to_pay_amount: string;
  currency: string;
  items: MarketplaceOrderItem[];
}

interface UnpackedProduct {
  item_id: number;
  name: string;
  product_sku: string | null;
  product_color: string | null;
  product_type: string | null;
  product_length: string | null;
  product_width: string | null;
  product_height: string | null;
  product_doors: string | null;
  product_legs: string | null;
  image_url: string | null;
  quantity: number;
  catalog_product_id: number | null;
  catalog_set_id: number | null;
  is_in_plan: boolean;
  in_plan_number: string | null;
  in_plan_id: number | null;
  in_current_plan: boolean;
  orderNumber: string;
  buyerName: string;
  marketplace: string;
  orderDate: string;
  orderTone: 0 | 1;
  parentSetSku?: string | null;
  parentSetTitle?: string | null;
  parentSetImage?: string | null;
  originalItem: MarketplaceOrderItem;
  siblingComponents?: SetComponent[];
}

function getOrderToneMap(orders: MarketplaceOrder[]): Map<number, 0 | 1> {
  const uniqueOrderIds = Array.from(new Set(orders.map(o => o.order_id))).sort((a, b) => a - b);
  const toneMap = new Map<number, 0 | 1>();
  
  uniqueOrderIds.forEach((orderId, index) => {
    toneMap.set(orderId, (index % 2) as 0 | 1);
  });
  
  return toneMap;
}

interface ProductsOnlyListProps {
  orders: MarketplaceOrder[];
  colors?: DictionaryItem[];
  productTypes?: DictionaryItem[];
  dimensionLengths?: DictionaryItem[];
  dimensionWidths?: DictionaryItem[];
  onAddItem?: (item: MarketplaceOrderItem, orderNumber: string, buyerName: string, marketplace: string, setId?: number, componentId?: number) => void;
  isAddingItem?: boolean;
  recentlyAddedItems?: Set<number>;
  selectedItemIds?: Set<string>;
  onToggleItemSelection?: (itemId: string) => void;
  expandedColorGroups?: Set<string>;
  setExpandedColorGroups?: (value: Set<string> | ((prev: Set<string>) => Set<string>)) => void;
  expandedTypeGroups?: Set<string>;
  setExpandedTypeGroups?: (value: Set<string> | ((prev: Set<string>) => Set<string>)) => void;
  onHighlightInPlan?: (productId: number, orderNumber: string) => void;
  isLocked?: boolean;
}

export function ProductsOnlyList({ 
  orders, 
  colors,
  productTypes = [],
  dimensionLengths = [],
  dimensionWidths = [],
  onAddItem, 
  isAddingItem, 
  recentlyAddedItems = new Set(), 
  selectedItemIds = new Set(), 
  onToggleItemSelection,
  expandedColorGroups: propExpandedColorGroups,
  setExpandedColorGroups: propSetExpandedColorGroups,
  expandedTypeGroups: propExpandedTypeGroups,
  setExpandedTypeGroups: propSetExpandedTypeGroups,
  onHighlightInPlan,
  isLocked = false
}: ProductsOnlyListProps) {
  const [localExpandedColorGroups, setLocalExpandedColorGroups] = useState<Set<string>>(new Set());
  const [localExpandedTypeGroups, setLocalExpandedTypeGroups] = useState<Set<string>>(new Set());
  const [selectedProductForModal, setSelectedProductForModal] = useState<UnpackedProduct | null>(null);
  
  const expandedColorGroups = propExpandedColorGroups ?? localExpandedColorGroups;
  const setExpandedColorGroups = propSetExpandedColorGroups ?? setLocalExpandedColorGroups;
  const expandedTypeGroups = propExpandedTypeGroups ?? localExpandedTypeGroups;
  const setExpandedTypeGroups = propSetExpandedTypeGroups ?? setLocalExpandedTypeGroups;

  const orderToneMap = useMemo(() => getOrderToneMap(orders), [orders]);
  
  const unpackedProducts = useMemo(() => {
    const products: UnpackedProduct[] = [];

    orders.forEach(order => {
      const buyerName = `${order.buyer_first_name} ${order.buyer_last_name}`;
      const tone = orderToneMap.get(order.order_id) ?? 0;
      
      order.items.forEach(item => {
        const hasSetComponents = item.set_components && item.set_components.length > 0;
        
        if (hasSetComponents) {
          item.set_components!.forEach((component: SetComponent) => {
            products.push({
              item_id: component.component_id,
              name: component.component_title,
              product_sku: component.component_sku,
              product_color: component.component_color,
              product_type: component.component_product_type,
              product_length: component.component_length,
              product_width: component.component_width,
              product_height: component.component_height,
              product_doors: component.component_doors,
              product_legs: component.component_legs,
              image_url: component.primary_image_url,
              quantity: component.quantity,
              catalog_product_id: component.component_id,
              catalog_set_id: item.catalog_set_id,
              is_in_plan: component.is_in_any_plan,
              in_plan_number: component.in_plan_number,
              in_plan_id: component.in_plan_id,
              in_current_plan: component.is_in_current_plan,
              orderNumber: order.order_number,
              buyerName,
              marketplace: order.marketplace,
              orderDate: order.order_date,
              orderTone: tone,
              parentSetSku: item.catalog_set_sku || item.product_sku,
              parentSetTitle: item.catalog_set_title || item.name,
              parentSetImage: item.image_url,
              originalItem: item,
              siblingComponents: item.set_components!,
            });
          });
        } else {
          products.push({
            item_id: item.item_id,
            name: item.name,
            product_sku: item.product_sku,
            product_color: item.product_color,
            product_type: item.product_type,
            product_length: item.product_length,
            product_width: item.product_width,
            product_height: item.product_height,
            product_doors: item.product_doors,
            product_legs: item.product_legs,
            image_url: item.image_url,
            quantity: item.quantity,
            catalog_product_id: item.catalog_product_id,
            catalog_set_id: null,
            is_in_plan: item.is_in_plan,
            in_plan_number: item.in_plan_number,
            in_plan_id: item.in_plan_id,
            in_current_plan: item.in_current_plan,
            orderNumber: order.order_number,
            buyerName,
            marketplace: order.marketplace,
            orderDate: order.order_date,
            orderTone: tone,
            originalItem: item,
          });
        }
      });
    });

    return products;
  }, [orders, orderToneMap]);

  const groupedData = useMemo(() => {
    const groups = new Map<string, Map<string, UnpackedProduct[]>>();

    unpackedProducts.forEach(product => {
      const colorKey = product.product_color || 'Bez koloru';
      const typeKey = product.product_type || 'Bez typu';
      
      if (!groups.has(colorKey)) {
        groups.set(colorKey, new Map());
      }
      
      const colorGroup = groups.get(colorKey)!;
      if (!colorGroup.has(typeKey)) {
        colorGroup.set(typeKey, []);
      }
      
      colorGroup.get(typeKey)!.push(product);
    });

    return groups;
  }, [unpackedProducts]);

  const getColorGroupItemIds = (colorKey: string): string[] => {
    const typeGroups = groupedData.get(colorKey);
    if (!typeGroups) return [];
    const ids: string[] = [];
    typeGroups.forEach(items => {
      items.forEach(item => {
        if (item.catalog_set_id) {
          ids.push(`component-${item.orderNumber}-${item.item_id}`);
        } else {
          ids.push(`item-${item.item_id}`);
        }
      });
    });
    return ids;
  };

  const getTypeGroupItemIds = (colorKey: string, typeKey: string): string[] => {
    const typeGroups = groupedData.get(colorKey);
    if (!typeGroups) return [];
    const items = typeGroups.get(typeKey);
    return items ? items.map(item => {
      if (item.catalog_set_id) {
        return `component-${item.orderNumber}-${item.item_id}`;
      }
      return `item-${item.item_id}`;
    }) : [];
  };

  const areAllColorItemsSelected = (colorKey: string): boolean => {
    const ids = getColorGroupItemIds(colorKey);
    return ids.length > 0 && ids.every(id => selectedItemIds.has(id));
  };

  const areAllTypeItemsSelected = (colorKey: string, typeKey: string): boolean => {
    const ids = getTypeGroupItemIds(colorKey, typeKey);
    return ids.length > 0 && ids.every(id => selectedItemIds.has(id));
  };

  const toggleColorGroupSelection = (colorKey: string) => {
    if (!onToggleItemSelection) return;
    const ids = getColorGroupItemIds(colorKey);
    const allSelected = areAllColorItemsSelected(colorKey);
    
    ids.forEach(id => {
      if (allSelected) {
        if (selectedItemIds.has(id)) {
          onToggleItemSelection(id);
        }
      } else {
        if (!selectedItemIds.has(id)) {
          onToggleItemSelection(id);
        }
      }
    });
  };

  const toggleTypeGroupSelection = (colorKey: string, typeKey: string) => {
    if (!onToggleItemSelection) return;
    const ids = getTypeGroupItemIds(colorKey, typeKey);
    const allSelected = areAllTypeItemsSelected(colorKey, typeKey);
    
    ids.forEach(id => {
      if (allSelected) {
        if (selectedItemIds.has(id)) {
          onToggleItemSelection(id);
        }
      } else {
        if (!selectedItemIds.has(id)) {
          onToggleItemSelection(id);
        }
      }
    });
  };

  const toggleColorGroup = (colorKey: string) => {
    setExpandedColorGroups(prev => {
      const newSet = new Set(prev);
      
      if (newSet.has(colorKey)) {
        newSet.delete(colorKey);
        setExpandedTypeGroups(typeSet => {
          const newTypeSet = new Set(typeSet);
          const typeGroups = groupedData.get(colorKey);
          if (typeGroups) {
            typeGroups.forEach((_, typeKey) => {
              newTypeSet.delete(`${colorKey}-${typeKey}`);
            });
          }
          return newTypeSet;
        });
      } else {
        newSet.add(colorKey);
        setExpandedTypeGroups(typeSet => {
          const newTypeSet = new Set(typeSet);
          const typeGroups = groupedData.get(colorKey);
          if (typeGroups) {
            typeGroups.forEach((_, typeKey) => {
              newTypeSet.add(`${colorKey}-${typeKey}`);
            });
          }
          return newTypeSet;
        });
      }
      return newSet;
    });
  };

  const toggleTypeGroup = (typeKey: string) => {
    setExpandedTypeGroups(prev => {
      const newSet = new Set(prev);
      if (newSet.has(typeKey)) {
        newSet.delete(typeKey);
      } else {
        newSet.add(typeKey);
      }
      return newSet;
    });
  };

  if (groupedData.size === 0) {
    return (
      <div className="flex flex-col items-center justify-center p-8 text-center text-muted-foreground">
        <Package className="h-12 w-12 mb-4 opacity-50" />
        <p>Brak produktów spełniających kryteria</p>
        <p className="text-sm">Zmień filtry aby zobaczyć więcej produktów</p>
      </div>
    );
  }

  return (
    <div className="space-y-0" data-testid="products-only-list">
      {Array.from(groupedData.entries()).map(([colorKey, typeGroups]) => {
        const isColorExpanded = expandedColorGroups.has(colorKey);
        const colorDict = colors?.find(c => c.code === colorKey);
        const bgColor = colorDict?.color || '#94A3B8';
        const textColor = getTextColorForBackground(bgColor);
        
        let totalItems = 0;
        typeGroups.forEach(items => { totalItems += items.length; });

        return (
          <div key={colorKey} className="border border-border rounded-none mt-4 first:mt-0">
            <div
              className="flex items-center gap-2 h-[50px] px-3 hover-elevate font-medium bg-background"
              data-testid={`color-group-${colorKey}`}
              style={{ borderLeft: `4px solid ${bgColor}` }}
            >
              {onToggleItemSelection && (
                <Checkbox
                  checked={areAllColorItemsSelected(colorKey)}
                  onCheckedChange={() => toggleColorGroupSelection(colorKey)}
                  onClick={(e) => e.stopPropagation()}
                  className="flex-shrink-0"
                  data-testid={`checkbox-color-${colorKey}`}
                />
              )}
              
              <div 
                className="flex items-center gap-2 flex-1 cursor-pointer"
                onClick={() => toggleColorGroup(colorKey)}
              >
                {isColorExpanded ? (
                  <ChevronDown className="h-3 w-3 flex-shrink-0" />
                ) : (
                  <ChevronRight className="h-3 w-3 flex-shrink-0" />
                )}
                
                <Badge 
                  variant="outline" 
                  className="h-4 px-2 text-[10px] border-transparent"
                  style={{ backgroundColor: bgColor, color: textColor }}
                >
                  {colorKey}
                </Badge>
                
                <span className="text-xs text-muted-foreground">
                  {totalItems} {totalItems === 1 ? 'produkt' : 'produktów'}
                </span>
              </div>
            </div>

            {isColorExpanded && (
              <div>
                {Array.from(typeGroups.entries()).map(([typeKey, items], index) => {
                  const isTypeExpanded = expandedTypeGroups.has(`${colorKey}-${typeKey}`);
                  const isFirstType = index === 0;
                  
                  return (
                    <div key={`${colorKey}-${typeKey}`} className={isFirstType ? '' : 'border-t border-border/50'}>
                      <div
                        className="flex items-center gap-2 h-[40px] pl-6 pr-2 hover-elevate bg-background/80"
                        data-testid={`type-group-${colorKey}-${typeKey}`}
                      >
                        {onToggleItemSelection && (
                          <Checkbox
                            checked={areAllTypeItemsSelected(colorKey, typeKey)}
                            onCheckedChange={() => toggleTypeGroupSelection(colorKey, typeKey)}
                            onClick={(e) => e.stopPropagation()}
                            className="flex-shrink-0"
                            data-testid={`checkbox-type-${colorKey}-${typeKey}`}
                          />
                        )}
                        
                        <div 
                          className="flex items-center gap-2 flex-1 cursor-pointer"
                          onClick={() => toggleTypeGroup(`${colorKey}-${typeKey}`)}
                        >
                          {isTypeExpanded ? (
                            <ChevronDown className="h-3 w-3 flex-shrink-0" />
                          ) : (
                            <ChevronRight className="h-3 w-3 flex-shrink-0" />
                          )}
                          
                          <Badge variant="secondary" className="h-4 px-1 text-[9px]">
                            {typeKey}
                          </Badge>
                          
                          <span className="text-[10px] text-muted-foreground">
                            {items.length} szt.
                          </span>
                        </div>
                      </div>

                      {isTypeExpanded && (
                        <div className="pl-3 space-y-0 pb-2 bg-muted/20">
                          {items.map((product) => {
                            const itemKey = product.catalog_set_id 
                              ? `component-${product.orderNumber}-${product.item_id}`
                              : `item-${product.item_id}`;
                            
                            const draggableItem: MarketplaceOrderItem = {
                              ...product.originalItem,
                              item_id: product.item_id,
                              name: product.name,
                              product_sku: product.product_sku,
                              product_color: product.product_color,
                              product_type: product.product_type,
                              product_length: product.product_length,
                              product_width: product.product_width,
                              product_height: product.product_height,
                              product_doors: product.product_doors,
                              product_legs: product.product_legs,
                              image_url: product.image_url,
                              catalog_product_id: product.catalog_product_id,
                              catalog_set_id: product.catalog_set_id,
                              is_in_plan: product.is_in_plan,
                              in_plan_number: product.in_plan_number,
                              in_plan_id: product.in_plan_id,
                              in_current_plan: product.in_current_plan,
                              quantity: product.quantity,
                              set_components: null,
                            };

                            const hasSiblings = product.siblingComponents && product.siblingComponents.length > 0;
                            const siblingsInPlan = hasSiblings 
                              ? product.siblingComponents!.filter(c => c.is_in_any_plan).length 
                              : 0;
                            const totalSiblings = hasSiblings ? product.siblingComponents!.length : 0;
                            const allSiblingsInPlan = siblingsInPlan === totalSiblings && totalSiblings > 0;

                            const setIndicator = hasSiblings ? (
                              <button
                                onClick={(e) => {
                                  e.stopPropagation();
                                  setSelectedProductForModal(product);
                                }}
                                className={`
                                  flex items-center gap-0.5 px-1.5 py-0.5 rounded text-[10px] font-medium
                                  transition-colors cursor-pointer
                                  ${allSiblingsInPlan 
                                    ? 'bg-green-500/20 text-green-400 hover:bg-green-500/30 border border-green-500/40' 
                                    : 'bg-purple-500/20 text-purple-400 hover:bg-purple-500/30 border border-purple-500/40'
                                  }
                                `}
                                title={`Zestaw: ${siblingsInPlan}/${totalSiblings} w planach`}
                                data-testid={`set-indicator-${product.item_id}`}
                              >
                                <Layers className="h-3 w-3" />
                                <span>{siblingsInPlan}/{totalSiblings}</span>
                              </button>
                            ) : null;

                            return (
                              <div key={`${product.orderNumber}-${product.item_id}-${product.catalog_set_id || ''}`} className="relative flex items-stretch">
                                <div className="flex-shrink-0 w-6" />
                                <div className="flex-1">
                                  <DraggableProductItem
                                    item={draggableItem}
                                    orderNumber={product.orderNumber}
                                    buyerName={product.buyerName}
                                    marketplace={product.marketplace}
                                    orderDate={product.orderDate}
                                    colors={colors}
                                    productTypes={productTypes}
                                    dimensionLengths={dimensionLengths}
                                    dimensionWidths={dimensionWidths}
                                    onAddClick={onAddItem}
                                    setId={product.catalog_set_id ?? undefined}
                                    componentId={product.catalog_set_id ? product.item_id : undefined}
                                    isAdding={isAddingItem}
                                    isRecentlyAdded={recentlyAddedItems.has(product.item_id)}
                                    rowHeight="48px"
                                    showOrderInfo={true}
                                    isSelected={selectedItemIds?.has(itemKey) ?? false}
                                    onToggleSelect={onToggleItemSelection ? () => onToggleItemSelection(itemKey) : undefined}
                                    orderTone={product.orderTone}
                                    isSetChild={Boolean(product.catalog_set_id)}
                                    onHighlightInPlan={onHighlightInPlan}
                                    setIndicatorContent={setIndicator}
                                    isLocked={isLocked}
                                  />
                                </div>
                              </div>
                            );
                          })}
                        </div>
                      )}
                    </div>
                  );
                })}
              </div>
            )}
          </div>
        );
      })}

      {selectedProductForModal && selectedProductForModal.siblingComponents && onAddItem && (
        <SetComponentsModal
          open={!!selectedProductForModal}
          onOpenChange={(open) => !open && setSelectedProductForModal(null)}
          setComponents={selectedProductForModal.siblingComponents}
          parentSetSku={selectedProductForModal.parentSetSku || null}
          parentSetTitle={selectedProductForModal.parentSetTitle || null}
          parentSetImage={selectedProductForModal.parentSetImage}
          orderNumber={selectedProductForModal.orderNumber}
          buyerName={selectedProductForModal.buyerName}
          marketplace={selectedProductForModal.marketplace}
          currentProductId={selectedProductForModal.item_id}
          onAddComponent={onAddItem}
          isAdding={isAddingItem}
          catalogSetId={selectedProductForModal.catalog_set_id}
          originalItem={selectedProductForModal.originalItem}
          onHighlightInPlan={onHighlightInPlan}
        />
      )}
    </div>
  );
}
