import { useState, useMemo } from "react";
import { useQuery, useMutation } from "@tanstack/react-query";
import { Route, Plus, Trash2, Factory, Palette, Package, Settings2, Wrench, Layers, Cog } from "lucide-react";
import { OperationOverridesDialog } from "./OperationOverridesDialog";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from "@/components/ui/dialog";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Combobox } from "@/components/ui/combobox";
import { Label } from "@/components/ui/label";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Separator } from "@/components/ui/separator";
import { ScrollArea } from "@/components/ui/scroll-area";
import { apiRequest, queryClient } from "@/lib/queryClient";
import { useToast } from "@/hooks/use-toast";

interface RoutingAssignment {
  id: number;
  planId: number;
  materialType: string | null;
  colorCode: string | null;
  componentPattern: string | null;
  routingId: number;
  routingCode?: string;
  routingName?: string;
  locationId: number | null;
  locationCode?: string;
  locationName?: string;
  overrideWorkCenterId: number | null;
  workCenterCode?: string;
  workCenterName?: string;
  templateId: number | null;
  templateName?: string;
  priority: number;
  notes: string | null;
}

interface Routing {
  id: number;
  code: string;
  name: string;
}

interface Location {
  id: number;
  code: string;
  name: string;
}

interface WorkCenter {
  id: number;
  code: string;
  name: string;
}

interface DictionaryItem {
  id: number;
  code: string;
  name: string | null;
}

interface ComponentRoutingTemplate {
  id: number;
  name: string;
  materialType: string | null;
  colorCode: string | null;
  componentPattern: string | null;
  defaultRoutingId: number;
  defaultWorkCenterId: number | null;
  locationId: number | null;
  priority: number;
  routingCode?: string;
  routingName?: string;
  workCenterCode?: string;
  workCenterName?: string;
  locationCode?: string;
  locationName?: string;
}

interface PlanRoutingAssignmentsDialogProps {
  planId: number;
}

export function PlanRoutingAssignmentsDialog({ planId }: PlanRoutingAssignmentsDialogProps) {
  const [open, setOpen] = useState(false);
  const { toast } = useToast();

  const [newAssignment, setNewAssignment] = useState({
    materialType: "",
    colorCode: "",
    componentPattern: "",
    routingId: 0,
    locationId: 0,
    overrideWorkCenterId: 0,
    templateId: 0,
    priority: 10,
    notes: "",
  });

  const { data: assignments = [], isLoading: assignmentsLoading } = useQuery<RoutingAssignment[]>({
    queryKey: [`/api/production/plans/${planId}/routing-assignments`],
    enabled: open,
  });

  const { data: routings = [] } = useQuery<Routing[]>({
    queryKey: ["/api/production/routings"],
    enabled: open,
  });

  const { data: locations = [] } = useQuery<Location[]>({
    queryKey: ["/api/production/locations"],
    enabled: open,
  });

  const { data: workCenters = [] } = useQuery<WorkCenter[]>({
    queryKey: ["/api/production/work-centers"],
    enabled: open,
  });

  const { data: templates = [] } = useQuery<ComponentRoutingTemplate[]>({
    queryKey: ["/api/production/component-routing-templates"],
    enabled: open,
  });

  const { data: colors = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries", { type: "color" }],
    queryFn: () => fetch("/api/dictionaries?type=color").then(r => r.json()),
    enabled: open,
  });

  const { data: componentCz1 = [] } = useQuery<DictionaryItem[]>({
    queryKey: ["/api/dictionaries", { type: "component_cz1" }],
    queryFn: () => fetch("/api/dictionaries?type=component_cz1").then(r => r.json()),
    enabled: open,
  });

  const { data: materialTypes = [] } = useQuery<string[]>({
    queryKey: ["/api/production/material-types"],
    enabled: open,
  });

  const createMutation = useMutation({
    mutationFn: async (data: typeof newAssignment) => {
      return apiRequest("POST", `/api/production/plans/${planId}/routing-assignments`, {
        materialType: data.materialType || null,
        colorCode: data.colorCode || null,
        componentPattern: data.componentPattern || null,
        routingId: data.routingId,
        locationId: data.locationId || null,
        overrideWorkCenterId: data.overrideWorkCenterId || null,
        templateId: data.templateId || null,
        priority: data.priority,
        notes: data.notes || null,
      });
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/plans/${planId}/routing-assignments`] });
      toast({ title: "Przypisanie dodane" });
      setNewAssignment({
        materialType: "",
        colorCode: "",
        componentPattern: "",
        routingId: 0,
        locationId: 0,
        overrideWorkCenterId: 0,
        templateId: 0,
        priority: 10,
        notes: "",
      });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const deleteMutation = useMutation({
    mutationFn: async (id: number) => {
      return apiRequest("DELETE", `/api/production/plan-routing-assignments/${id}`);
    },
    onSuccess: () => {
      queryClient.invalidateQueries({ queryKey: [`/api/production/plans/${planId}/routing-assignments`] });
      toast({ title: "Przypisanie usunięte" });
    },
    onError: (error: Error) => {
      toast({ title: "Błąd", description: error.message, variant: "destructive" });
    },
  });

  const handleApplyTemplate = (template: ComponentRoutingTemplate) => {
    setNewAssignment({
      materialType: template.materialType || "",
      colorCode: template.colorCode || "",
      componentPattern: template.componentPattern || "",
      routingId: template.defaultRoutingId,
      locationId: template.locationId || 0,
      overrideWorkCenterId: template.defaultWorkCenterId || 0,
      templateId: template.id,
      priority: template.priority,
      notes: `Z szablonu: ${template.name}`,
    });
    toast({ title: "Szablon załadowany", description: `Zastosowano szablon: ${template.name}` });
  };

  const canSubmit = newAssignment.routingId > 0;

  const componentPatternOptions = useMemo(() => 
    componentCz1.map((c) => ({ 
      value: `${c.code}%`, 
      label: c.name === c.code || !c.name ? c.code : `${c.name} (${c.code})` 
    })),
    [componentCz1]
  );

  const groupedAssignments = useMemo(() => {
    const groups: Record<string, RoutingAssignment[]> = {};
    assignments.forEach(a => {
      const key = a.materialType || "Wszystkie materiały";
      if (!groups[key]) groups[key] = [];
      groups[key].push(a);
    });
    return groups;
  }, [assignments]);

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>
        <Button variant="outline" size="sm" className="gap-1.5" data-testid="button-routing-assignments">
          <Route className="h-4 w-4" />
          <span className="hidden sm:inline">Marszruty</span>
        </Button>
      </DialogTrigger>
      <DialogContent className="max-w-4xl h-[100vh] md:h-[90vh] max-h-[100vh] md:max-h-[90vh]">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Route className="h-5 w-5 text-primary" />
            Przypisanie marszrut do planu
          </DialogTitle>
        </DialogHeader>

        <div className="space-y-4">
          {templates.length > 0 && (
            <div className="rounded-md border p-3 bg-violet-500/10">
              <h3 className="text-sm font-medium mb-2 flex items-center gap-2">
                <Layers className="h-4 w-4" />
                Szablony komponentów (kliknij aby załadować)
              </h3>
              <div className="flex flex-wrap gap-2">
                {templates.slice(0, 8).map((template) => (
                  <Badge
                    key={template.id}
                    variant="outline"
                    className="cursor-pointer hover-elevate gap-1"
                    onClick={() => handleApplyTemplate(template)}
                    data-testid={`template-badge-${template.id}`}
                  >
                    <Layers className="h-3 w-3" />
                    {template.name}
                  </Badge>
                ))}
                {templates.length > 8 && (
                  <Badge variant="secondary" className="text-xs">
                    +{templates.length - 8} więcej
                  </Badge>
                )}
              </div>
            </div>
          )}

          <div className="rounded-md border p-4 bg-muted/30">
            <h3 className="text-sm font-medium mb-3 flex items-center gap-2">
              <Plus className="h-4 w-4" />
              Nowe przypisanie
            </h3>
            
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-3">
              <div className="space-y-1.5">
                <Label className="text-xs flex items-center gap-1">
                  <Package className="h-3 w-3" />
                  Typ materiału
                </Label>
                <Select
                  value={newAssignment.materialType}
                  onValueChange={(v) => setNewAssignment(p => ({ ...p, materialType: v === "any" ? "" : v }))}
                >
                  <SelectTrigger className="h-8" data-testid="select-material-type">
                    <SelectValue placeholder="Dowolny" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="any">Dowolny</SelectItem>
                    {materialTypes.map((type) => (
                      <SelectItem key={type} value={type}>{type}</SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-1.5">
                <Label className="text-xs flex items-center gap-1">
                  <Palette className="h-3 w-3" />
                  Kolor
                </Label>
                <Select
                  value={newAssignment.colorCode}
                  onValueChange={(v) => setNewAssignment(p => ({ ...p, colorCode: v === "any" ? "" : v }))}
                >
                  <SelectTrigger className="h-8" data-testid="select-color">
                    <SelectValue placeholder="Dowolny" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="any">Dowolny</SelectItem>
                    {colors.map((color) => (
                      <SelectItem key={color.code} value={color.code}>
                        {color.name || color.code}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-1.5">
                <Label className="text-xs">Wzorzec komponentu</Label>
                <div className="flex gap-1">
                  <Textarea
                    value={newAssignment.componentPattern}
                    onChange={(e) => setNewAssignment(p => ({ ...p, componentPattern: e.target.value }))}
                    placeholder="np. BOCZKI%&#10;DNO%"
                    className="flex-1 min-h-[60px] text-xs"
                    rows={2}
                    data-testid="input-component-pattern"
                  />
                  <Combobox
                    options={componentPatternOptions}
                    value=""
                    onChange={(v) => {
                      if (v) {
                        const current = newAssignment.componentPattern.trim();
                        if (current) {
                          setNewAssignment(p => ({ ...p, componentPattern: `${current}\n${v}` }));
                        } else {
                          setNewAssignment(p => ({ ...p, componentPattern: v }));
                        }
                      }
                    }}
                    placeholder="Dodaj"
                    emptyText="Brak"
                    searchPlaceholder="Szukaj..."
                    className="w-[90px] h-8"
                  />
                </div>
                <p className="text-[10px] text-muted-foreground">Wiele wzorców w osobnych liniach</p>
              </div>

              <div className="space-y-1.5">
                <Label className="text-xs flex items-center gap-1">
                  <Route className="h-3 w-3" />
                  Marszruta *
                </Label>
                <Select
                  value={newAssignment.routingId ? String(newAssignment.routingId) : ""}
                  onValueChange={(v) => setNewAssignment(p => ({ ...p, routingId: parseInt(v) }))}
                >
                  <SelectTrigger className="h-8" data-testid="select-routing">
                    <SelectValue placeholder="Wybierz marszrutę" />
                  </SelectTrigger>
                  <SelectContent>
                    {routings.map((routing) => (
                      <SelectItem key={routing.id} value={String(routing.id)}>
                        {routing.code} - {routing.name}
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-1.5">
                <Label className="text-xs flex items-center gap-1">
                  <Factory className="h-3 w-3" />
                  Lokalizacja
                </Label>
                <Select
                  value={newAssignment.locationId ? String(newAssignment.locationId) : ""}
                  onValueChange={(v) => setNewAssignment(p => ({ ...p, locationId: v && v !== "any" ? parseInt(v) : 0 }))}
                >
                  <SelectTrigger className="h-8" data-testid="select-location">
                    <SelectValue placeholder="Dowolna" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="any">Dowolna</SelectItem>
                    {locations.map((loc) => (
                      <SelectItem key={loc.id} value={String(loc.id)}>
                        {loc.name} ({loc.code})
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-1.5">
                <Label className="text-xs flex items-center gap-1">
                  <Wrench className="h-3 w-3" />
                  Maszyna (override)
                </Label>
                <Select
                  value={newAssignment.overrideWorkCenterId ? String(newAssignment.overrideWorkCenterId) : ""}
                  onValueChange={(v) => setNewAssignment(p => ({ ...p, overrideWorkCenterId: v && v !== "any" ? parseInt(v) : 0 }))}
                >
                  <SelectTrigger className="h-8" data-testid="select-work-center">
                    <SelectValue placeholder="Domyślna z marszruty" />
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="any">Domyślna z marszruty</SelectItem>
                    {workCenters.map((wc) => (
                      <SelectItem key={wc.id} value={String(wc.id)}>
                        {wc.name} ({wc.code})
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>

              <div className="space-y-1.5">
                <Label className="text-xs">Priorytet</Label>
                <Input
                  type="number"
                  value={newAssignment.priority}
                  onChange={(e) => setNewAssignment(p => ({ ...p, priority: parseInt(e.target.value) || 0 }))}
                  className="h-8"
                  data-testid="input-priority"
                />
              </div>

              <div className="space-y-1.5">
                <Label className="text-xs">Notatki</Label>
                <Input
                  value={newAssignment.notes}
                  onChange={(e) => setNewAssignment(p => ({ ...p, notes: e.target.value }))}
                  placeholder="Opcjonalne notatki"
                  className="h-8"
                  data-testid="input-notes"
                />
              </div>
            </div>

            <div className="mt-3 flex justify-end">
              <Button
                size="sm"
                onClick={() => createMutation.mutate(newAssignment)}
                disabled={!canSubmit || createMutation.isPending}
                data-testid="button-add-assignment"
              >
                <Plus className="h-4 w-4 mr-1" />
                Dodaj przypisanie
              </Button>
            </div>
          </div>

          <Separator />

          <div>
            <h3 className="text-sm font-medium mb-3 flex items-center gap-2">
              <Settings2 className="h-4 w-4" />
              Aktualne przypisania ({assignments.length})
            </h3>

            {assignmentsLoading ? (
              <div className="text-center py-8 text-muted-foreground">Ładowanie...</div>
            ) : assignments.length === 0 ? (
              <div className="text-center py-8 text-muted-foreground border rounded-md bg-muted/20">
                <Route className="h-8 w-8 mx-auto mb-2 opacity-50" />
                <p>Brak przypisań marszrut dla tego planu.</p>
                <p className="text-xs mt-1">System użyje szablonów komponentów lub domyślnych reguł wariantów.</p>
              </div>
            ) : (
              <ScrollArea className="h-[280px]">
                <div className="space-y-4 pr-4">
                  {Object.entries(groupedAssignments).map(([materialType, group]) => (
                    <div key={materialType} className="space-y-2">
                      <div className="text-xs font-medium text-muted-foreground uppercase tracking-wide flex items-center gap-1.5">
                        <Package className="h-3 w-3" />
                        {materialType}
                      </div>
                      
                      <div className="space-y-2">
                        {group.sort((a, b) => b.priority - a.priority).map((assignment) => (
                          <div
                            key={assignment.id}
                            className="flex items-center gap-2 p-2 rounded-md border bg-card hover-elevate"
                            data-testid={`assignment-row-${assignment.id}`}
                          >
                            <div className="flex-1 flex flex-wrap items-center gap-2">
                              {assignment.colorCode && (
                                <Badge variant="outline" className="gap-1">
                                  <Palette className="h-3 w-3" />
                                  {assignment.colorCode}
                                </Badge>
                              )}
                              
                              {assignment.componentPattern && (
                                assignment.componentPattern.split(/[|\n]/).filter(p => p.trim()).map((pattern, idx) => (
                                  <Badge key={idx} variant="secondary" className="font-mono text-xs">
                                    {pattern.trim()}
                                  </Badge>
                                ))
                              )}
                              
                              <span className="text-muted-foreground">→</span>
                              
                              <Badge variant="default" className="gap-1">
                                <Route className="h-3 w-3" />
                                {assignment.routingCode || assignment.routingName}
                              </Badge>
                              
                              {assignment.locationName && (
                                <Badge variant="outline" className="gap-1">
                                  <Factory className="h-3 w-3" />
                                  {assignment.locationName}
                                </Badge>
                              )}
                              
                              {assignment.workCenterName && (
                                <Badge variant="outline" className="gap-1 bg-amber-500/10 border-amber-500/30">
                                  <Wrench className="h-3 w-3" />
                                  {assignment.workCenterName}
                                </Badge>
                              )}
                              
                              {assignment.templateName && (
                                <Badge variant="secondary" className="gap-1 text-xs">
                                  <Layers className="h-3 w-3" />
                                  {assignment.templateName}
                                </Badge>
                              )}
                              
                              <span className="text-xs text-muted-foreground">
                                (p: {assignment.priority})
                              </span>
                            </div>

                            <div className="flex items-center gap-1">
                              <OperationOverridesDialog 
                                assignmentId={assignment.id}
                                assignmentLabel={`${assignment.materialType || 'Wszystkie'} / ${assignment.colorCode || 'Wszystkie'} / ${assignment.componentPattern || '*'}`}
                              />
                              <Button
                                size="icon"
                                variant="ghost"
                                onClick={() => deleteMutation.mutate(assignment.id)}
                                disabled={deleteMutation.isPending}
                                className="h-7 w-7 text-destructive"
                                data-testid={`button-delete-assignment-${assignment.id}`}
                              >
                                <Trash2 className="h-4 w-4" />
                              </Button>
                            </div>
                          </div>
                        ))}
                      </div>
                    </div>
                  ))}
                </div>
              </ScrollArea>
            )}
          </div>

          <div className="text-xs text-muted-foreground bg-muted/30 p-3 rounded-md">
            <p className="font-medium mb-1">Hierarchia rozwiązywania marszrut:</p>
            <ol className="list-decimal list-inside space-y-0.5">
              <li><strong>Przypisania planu</strong> (ta lista) - najwyższy priorytet</li>
              <li><strong>Szablony komponentów</strong> - globalne domyślne marszruty</li>
              <li><strong>Reguły wariantów</strong> - fallback dla nieprzypisanych</li>
            </ol>
            <p className="mt-2 text-muted-foreground/80">
              Maszyna (override) pozwala zmienić domyślne gniazdo produkcyjne z marszruty dla tego planu.
            </p>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
