import { useState, useEffect } from "react";
import { ArrowLeft, Calendar, Hash, Factory, Loader2, Route, BarChart3, AlertTriangle, Lock, LockOpen, X, GitBranch, Scissors, Truck } from "lucide-react";
import { Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { Tooltip, TooltipContent, TooltipTrigger } from "@/components/ui/tooltip";
import { format } from "date-fns";
import { pl } from "date-fns/locale";

interface ProductionPlan {
  id: number;
  planNumber: string;
  name: string;
  description?: string;
  plannedStartDate?: string;
  plannedEndDate?: string;
  status: string;
  priority: string;
  notes?: string;
  createdAt: string;
  autoAssignRoutings?: boolean;
}

interface PlanHeaderProps {
  plan: ProductionPlan;
  onBack: () => void;
  onGenerateOrders?: () => void;
  isGenerating?: boolean;
  onOpenRoutingPanel?: () => void;
  onOpenZlpDashboard?: () => void;
  onOpenCuttingSequence?: () => void;
  onOpenTransportBatches?: () => void;
  hasZlps?: boolean;
  hasRoutingGaps?: boolean;
  planLinesCount?: number;
  zlpLocked?: boolean;
  onLockZlp?: () => void;
  onUnlockZlp?: () => void;
  isLockPending?: boolean;
  isUnlockPending?: boolean;
}

const statusColors: Record<string, string> = {
  draft: "bg-gray-500",
  approved: "bg-blue-500",
  generated: "bg-purple-500",
  in_progress: "bg-yellow-500",
  completed: "bg-green-500",
  cancelled: "bg-red-500",
};

const statusLabels: Record<string, string> = {
  draft: "Projekt",
  approved: "Zatwierdzony",
  generated: "Wygenerowany",
  in_progress: "W realizacji",
  completed: "Zakończony",
  cancelled: "Anulowany",
};

const priorityColors: Record<string, string> = {
  low: "bg-gray-400",
  normal: "bg-blue-400",
  high: "bg-orange-400",
  urgent: "bg-red-500",
};

const priorityLabels: Record<string, string> = {
  low: "Niski",
  normal: "Normalny",
  high: "Wysoki",
  urgent: "Pilny",
};

export function PlanHeader({ plan, onBack, onGenerateOrders, isGenerating, onOpenRoutingPanel, onOpenZlpDashboard, onOpenCuttingSequence, onOpenTransportBatches, hasZlps, hasRoutingGaps, planLinesCount, zlpLocked, onLockZlp, onUnlockZlp, isLockPending, isUnlockPending }: PlanHeaderProps) {
  // Two-step confirmation states for Lock/Unlock buttons
  const [showLockConfirm, setShowLockConfirm] = useState(false);
  const [showUnlockConfirm, setShowUnlockConfirm] = useState(false);

  // Auto-close confirmation when operation completes (zlpLocked state changes)
  useEffect(() => {
    if (!isLockPending && showLockConfirm) {
      setShowLockConfirm(false);
    }
  }, [isLockPending, zlpLocked]);

  useEffect(() => {
    if (!isUnlockPending && showUnlockConfirm) {
      setShowUnlockConfirm(false);
    }
  }, [isUnlockPending, zlpLocked]);

  // Can generate orders if plan is not cancelled/completed and has items
  const hasPlanLines = (planLinesCount || 0) > 0;
  const allowedStatuses = ['draft', 'approved', 'in_progress', 'generated'];
  const canGenerateOrders = allowedStatuses.includes(plan.status) && onGenerateOrders && hasPlanLines && !zlpLocked;
  
  // Disable generation if routing gaps exist
  const isGenerateDisabled = isGenerating || hasRoutingGaps || !hasPlanLines;
  
  // Show lock/unlock button when ZLPs exist (keep visible during pending operations)
  const canLock = hasZlps && !zlpLocked && onLockZlp && !isUnlockPending;
  const canUnlock = hasZlps && zlpLocked && onUnlockZlp && !isLockPending;

  return (
    <div 
      className="min-h-[60px] py-2 border-b bg-card flex items-center px-4"
    >
      <div className="flex items-center gap-2 w-full flex-wrap">
        {/* Back Button */}
        <Button
          variant="ghost"
          size="icon"
          onClick={onBack}
          data-testid="button-back"
          className="flex-shrink-0 h-9 w-9"
        >
          <ArrowLeft className="h-5 w-5" />
        </Button>

        <div>
          <h1 className="text-base font-bold leading-tight" data-testid="title-plan-name">
            {plan.name}
          </h1>
        </div>

        <Separator orientation="vertical" className="hidden lg:block h-8" />

        {/* Plan Number with Icon */}
        <div className="flex items-center gap-1.5">
          <Hash className="text-primary h-4 w-4" />
          <span className="font-mono text-primary font-semibold text-sm" data-testid="text-plan-number">
            {plan.planNumber}
          </span>
        </div>

        {/* Start Date */}
        {plan.plannedStartDate && (
          <>
            <Separator orientation="vertical" className="hidden lg:block h-8" />
            <div className="flex items-center gap-1.5">
              <Calendar className="text-muted-foreground h-4 w-4" />
              <span className="text-muted-foreground text-xs hidden sm:inline">Start:</span>
              <span className="font-medium text-sm" data-testid="text-planned-start">
                {format(new Date(plan.plannedStartDate), "dd.MM.yyyy", { locale: pl })}
              </span>
            </div>
          </>
        )}

        {/* End Date */}
        {plan.plannedEndDate && (
          <>
            <Separator orientation="vertical" className="hidden lg:block h-8" />
            <div className="flex items-center gap-1.5">
              <Calendar className="text-muted-foreground h-4 w-4" />
              <span className="text-muted-foreground text-xs hidden sm:inline">Koniec:</span>
              <span className="font-medium text-sm" data-testid="text-planned-end">
                {format(new Date(plan.plannedEndDate), "dd.MM.yyyy", { locale: pl })}
              </span>
            </div>
          </>
        )}

        {/* Spacer */}
        <div className="flex-1 min-w-[20px]" />

        {/* Routing Panel Button - always show */}
        {onOpenRoutingPanel && (
          <Button 
            variant="outline" 
            size="sm" 
            className="gap-1.5 h-8" 
            onClick={onOpenRoutingPanel}
            data-testid="button-routing-assignments"
          >
            <Route className="h-4 w-4" />
            <span className="hidden sm:inline">Marszruty</span>
          </Button>
        )}

        {/* ZLP Dashboard Button - show when ZLPs exist */}
        {hasZlps && onOpenZlpDashboard && (
          <Button 
            variant="outline" 
            size="sm" 
            className="gap-1.5 h-8" 
            onClick={onOpenZlpDashboard}
            data-testid="button-zlp-dashboard"
          >
            <BarChart3 className="h-4 w-4" />
            <span className="hidden sm:inline">Dashboard ZLP</span>
          </Button>
        )}

        {/* Flow Tree Button - show when ZLPs exist */}
        {hasZlps && (
          <Button 
            variant="outline" 
            size="sm" 
            className="gap-1.5 h-8" 
            asChild
            data-testid="button-flow-tree"
          >
            <Link href={`/production/plans/${plan.id}/flow-tree`}>
              <GitBranch className="h-4 w-4" />
              <span className="hidden sm:inline">Drzewo przepływu</span>
            </Link>
          </Button>
        )}

        {/* Routing Settings Page Button - show when ZLPs exist */}
        {hasZlps && (
          <Button 
            variant="outline" 
            size="sm" 
            className="gap-1.5 h-8" 
            asChild
            data-testid="button-routing-settings-page"
          >
            <Link href={`/production/plans/${plan.id}/routing`}>
              <Route className="h-4 w-4" />
              <span className="hidden sm:inline">Ustawienia routingu</span>
            </Link>
          </Button>
        )}

        {/* Cutting Sequence Button - show when ZLPs exist */}
        {hasZlps && onOpenCuttingSequence && (
          <Button 
            variant="outline" 
            size="sm" 
            className="gap-1.5 h-8" 
            onClick={onOpenCuttingSequence}
            data-testid="button-cutting-sequence"
          >
            <Scissors className="h-4 w-4" />
            <span className="hidden sm:inline">Kolejność cięcia</span>
          </Button>
        )}

        {/* Transport Batches Button - show when ZLPs exist */}
        {hasZlps && onOpenTransportBatches && (
          <Button 
            variant="outline" 
            size="sm" 
            className="gap-1.5 h-8" 
            onClick={onOpenTransportBatches}
            data-testid="button-transport-batches"
          >
            <Truck className="h-4 w-4" />
            <span className="hidden sm:inline">Partie transportowe</span>
          </Button>
        )}

        {/* Generate Orders Button */}
        {canGenerateOrders && (
          <>
            <Tooltip>
              <TooltipTrigger asChild>
                <span>
                  <Button
                    onClick={onGenerateOrders}
                    disabled={isGenerateDisabled}
                    size="sm"
                    variant="default"
                    data-testid="button-generate-orders"
                    className={`h-8 gap-1.5 ${isGenerateDisabled && !isGenerating ? 'opacity-50' : ''}`}
                  >
                    {isGenerating ? (
                      <>
                        <Loader2 className="h-4 w-4 animate-spin" />
                        <span className="hidden sm:inline">Generowanie...</span>
                      </>
                    ) : hasRoutingGaps ? (
                      <>
                        <AlertTriangle className="h-4 w-4" />
                        <span className="hidden sm:inline">{hasZlps ? "Regeneruj ZLP" : "Generuj ZLP"}</span>
                      </>
                    ) : !hasPlanLines ? (
                      <>
                        <AlertTriangle className="h-4 w-4" />
                        <span className="hidden sm:inline">{hasZlps ? "Regeneruj ZLP" : "Generuj ZLP"}</span>
                      </>
                    ) : (
                      <>
                        <Factory className="h-4 w-4" />
                        <span className="hidden sm:inline">{hasZlps ? "Regeneruj ZLP" : "Generuj ZLP"}</span>
                      </>
                    )}
                  </Button>
                </span>
              </TooltipTrigger>
              {isGenerateDisabled && !isGenerating && (
                <TooltipContent>
                  {!hasPlanLines ? (
                    <p>Dodaj produkty do planu przed generowaniem ZLP</p>
                  ) : hasRoutingGaps ? (
                    <p>Przypisz marszruty do wszystkich produktów przed generowaniem ZLP</p>
                  ) : null}
                </TooltipContent>
              )}
            </Tooltip>
            <Separator orientation="vertical" className="hidden lg:block h-8" />
          </>
        )}

        {/* Lock/Unlock ZLP Button - Two-step confirmation */}
        {canLock && (
          <div className="flex items-center gap-1">
            {!showLockConfirm ? (
              <Tooltip>
                <TooltipTrigger asChild>
                  <Badge 
                    variant="secondary" 
                    className="cursor-pointer hover-elevate gap-1 px-2 py-1 h-8 bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400"
                    onClick={() => setShowLockConfirm(true)}
                    data-testid="badge-lock-zlp-step1"
                  >
                    <Lock className="h-3 w-3" />
                    <span className="text-xs hidden sm:inline">Zatwierdź...</span>
                  </Badge>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Kliknij aby zatwierdzić i zablokować plan</p>
                </TooltipContent>
              </Tooltip>
            ) : (
              <div className="flex items-center gap-1 animate-in fade-in slide-in-from-left-2 duration-200">
                <Button
                  onClick={() => onLockZlp?.()}
                  disabled={isLockPending}
                  size="sm"
                  variant="default"
                  className="h-8 gap-1.5 bg-green-600 hover:bg-green-700"
                  data-testid="button-lock-zlp"
                >
                  {isLockPending ? (
                    <Loader2 className="h-4 w-4 animate-spin" />
                  ) : (
                    <Lock className="h-4 w-4" />
                  )}
                  <span className="hidden sm:inline">{isLockPending ? "Zatwierdzanie..." : "Zatwierdź ZLP"}</span>
                </Button>
                {!isLockPending && (
                  <Button
                    onClick={() => setShowLockConfirm(false)}
                    size="icon"
                    variant="ghost"
                    className="h-8 w-8"
                    data-testid="button-cancel-lock"
                  >
                    <X className="h-4 w-4" />
                  </Button>
                )}
              </div>
            )}
          </div>
        )}
        
        {canUnlock && (
          <div className="flex items-center gap-1">
            {!showUnlockConfirm ? (
              <Tooltip>
                <TooltipTrigger asChild>
                  <Badge 
                    variant="outline" 
                    className="cursor-pointer hover-elevate gap-1 px-2 py-1 h-8"
                    onClick={() => setShowUnlockConfirm(true)}
                    data-testid="badge-unlock-zlp-step1"
                  >
                    <Lock className="h-3 w-3" />
                    <span className="text-xs hidden sm:inline">Zablokowany</span>
                  </Badge>
                </TooltipTrigger>
                <TooltipContent>
                  <p>Kliknij aby odblokować plan</p>
                </TooltipContent>
              </Tooltip>
            ) : (
              <div className="flex items-center gap-1 animate-in fade-in slide-in-from-left-2 duration-200">
                <Button
                  onClick={() => onUnlockZlp?.()}
                  disabled={isUnlockPending}
                  size="sm"
                  variant="outline"
                  className="h-8 gap-1.5"
                  data-testid="button-unlock-zlp"
                >
                  {isUnlockPending ? (
                    <Loader2 className="h-4 w-4 animate-spin" />
                  ) : (
                    <LockOpen className="h-4 w-4" />
                  )}
                  <span className="hidden sm:inline">{isUnlockPending ? "Odblokowywanie..." : "Odblokuj ZLP"}</span>
                </Button>
                {!isUnlockPending && (
                  <Button
                    onClick={() => setShowUnlockConfirm(false)}
                    size="icon"
                    variant="ghost"
                    className="h-8 w-8"
                    data-testid="button-cancel-unlock"
                  >
                    <X className="h-4 w-4" />
                  </Button>
                )}
              </div>
            )}
          </div>
        )}

        {/* Status Badge */}
        <Badge
          className={`${statusColors[plan.status] || 'bg-gray-500'} text-white h-6 px-2 text-xs`}
          data-testid={`badge-status-${plan.status}`}
        >
          {statusLabels[plan.status] || plan.status}
        </Badge>

        {/* Priority Badge */}
        <Badge
          className={`${priorityColors[plan.priority]} text-white h-6 px-2 text-xs`}
          data-testid={`badge-priority-${plan.priority}`}
        >
          {priorityLabels[plan.priority]}
        </Badge>
      </div>
    </div>
  );
}
